""" ANUGA models the effect of tsunamis and flooding upon a terrain mesh.
    In typical usage, a Domain class is created for a particular piece of
    terrain. Boundary conditions are specified for the domain, such as inflow
    and outflow, and then the simulation is run.

    This is the public API to ANUGA. It provides a toolkit of often-used
    modules, which can be used directly by including the following line in
    the user's code:

    import anuga
        
    This usage pattern abstracts away the internal heirarchy of the ANUGA
    system, allowing the user to concentrate on writing simulations without
    searching through the ANUGA source tree for the functions that they need.
    
    Also, it isolates the user from "under-the-hood" refactorings.
"""

#-----------------------------------------------------
# Make selected classes available directly
#-----------------------------------------------------

from numpy.testing import Tester
test = Tester().test


# PEP0440 compatible formatted version, see:
# https://www.python.org/dev/peps/pep-0440/
#
# Generic release markers:
# X.Y
# X.Y.Z # For bugfix releases
#
# Admissible pre-release markers:
# X.YaN # Alpha release
# X.YbN # Beta release
# X.YrcN # Release Candidate
# X.Y # Final release
#
# Dev branch marker is: 'X.Y.dev' or 'X.Y.devN' where N is an integer.
#
__version__ = '1.3.5'

__svn_revision__ = filter(str.isdigit, "$Revision: 9641 $")

#from anuga.__metadata__ import  __date__, __author__

#from .version import git_revision as __git_revision__
#from .version import svn_revision as __svn_revision__





# We first need to detect if we're being called as part of the anuga setup
# procedure itself in a reliable manner.
try:
    __ANUGA_SETUP__
except NameError:
    __ANUGA_SETUP__ = False

    
if __ANUGA_SETUP__:
    import sys as _sys
    _sys.stderr.write('Running from anuga source directory.\n')
    del _sys
else:

    try:
        from anuga.__config__ import show as show_config
    except ImportError:
        msg = """Error importing anuga: you should not try to import anuga from
        its source directory; please exit the anuga source tree, and relaunch
        your python interpreter from there."""
        raise ImportError(msg)
       
    #--------------------------------
    # Important basic classes
    #--------------------------------
    from anuga.shallow_water.shallow_water_domain import Domain
    from anuga.abstract_2d_finite_volumes.quantity import Quantity
    from anuga.abstract_2d_finite_volumes.region import Region
    from anuga.geospatial_data.geospatial_data import Geospatial_data
    from anuga.operators.base_operator import Operator
    from anuga.structures.structure_operator import Structure_operator


    from anuga.abstract_2d_finite_volumes.generic_domain import Generic_Domain
    from anuga.abstract_2d_finite_volumes.neighbour_mesh import Mesh
    #------------------------------------------------------------------------------ 
    # Miscellaneous
    #------------------------------------------------------------------------------ 
    from anuga.abstract_2d_finite_volumes.util import file_function, \
                                            sww2timeseries, sww2csv_gauges, \
                                            csv2timeseries_graphs

    from anuga.abstract_2d_finite_volumes.mesh_factory import rectangular_cross, \
                                                        rectangular

    from anuga.file.csv_file import load_csv_as_building_polygons,  \
                                    load_csv_as_polygons

    from anuga.file.sts import create_sts_boundary

    from anuga.file.ungenerate import load_ungenerate

    from anuga.geometry.polygon import read_polygon
    from anuga.geometry.polygon import plot_polygons
    from anuga.geometry.polygon import inside_polygon
    from anuga.geometry.polygon import polygon_area
    from anuga.geometry.polygon_function import Polygon_function

    from anuga.abstract_2d_finite_volumes.pmesh2domain import \
                                                pmesh_to_domain_instance

    from anuga.utilities.system_tools import file_length
    from anuga.utilities.sww_merge import sww_merge_parallel as sww_merge
    from anuga.utilities.file_utils import copy_code_files
    from anuga.utilities.numerical_tools import safe_acos as acos
    import anuga.utilities.plot_utils as plot_utils


    from anuga.caching import cache
    from os.path import join
    from anuga.config import indent



    #----------------------------
    # Parallel api 
    #----------------------------
    ## from anuga_parallel.parallel_api import distribute
    ## from anuga_parallel.parallel_api import myid, numprocs, get_processor_name
    ## from anuga_parallel.parallel_api import send, receive
    ## from anuga_parallel.parallel_api import pypar_available, barrier, finalize

    ## if pypar_available:
    ##     from anuga_parallel.parallel_api import sequential_distribute_dump
    ##     from anuga_parallel.parallel_api import sequential_distribute_load

    from anuga.parallel.parallel_api import distribute
    from anuga.parallel.parallel_api import myid, numprocs, get_processor_name
    from anuga.parallel.parallel_api import send, receive
    from anuga.parallel.parallel_api import pypar_available, barrier, finalize

    if pypar_available:
        from anuga.parallel.parallel_api import sequential_distribute_dump
        from anuga.parallel.parallel_api import sequential_distribute_load


    #-----------------------------
    # Checkpointing
    #-----------------------------
    from anuga.shallow_water.checkpoint import load_checkpoint_file


    #-----------------------------
    # SwW Standard Boundaries
    #-----------------------------
    from anuga.shallow_water.boundaries import File_boundary
    from anuga.shallow_water.boundaries import Reflective_boundary
    from anuga.shallow_water.boundaries import Field_boundary
    from anuga.shallow_water.boundaries import \
                        Time_stage_zero_momentum_boundary
    from anuga.shallow_water.boundaries import \
                        Transmissive_stage_zero_momentum_boundary
    from anuga.shallow_water.boundaries import \
                        Transmissive_momentum_set_stage_boundary
    from anuga.shallow_water.boundaries import \
                        Transmissive_n_momentum_zero_t_momentum_set_stage_boundary
    from anuga.shallow_water.boundaries import \
                        Flather_external_stage_zero_velocity_boundary
    from anuga.abstract_2d_finite_volumes.generic_boundary_conditions import \
                        Compute_fluxes_boundary


    #-----------------------------
    # General Boundaries
    #-----------------------------
    from anuga.abstract_2d_finite_volumes.generic_boundary_conditions \
                                import Dirichlet_boundary
    from anuga.abstract_2d_finite_volumes.generic_boundary_conditions \
                                import Time_boundary
    from anuga.abstract_2d_finite_volumes.generic_boundary_conditions \
                                import Time_space_boundary
    from anuga.abstract_2d_finite_volumes.generic_boundary_conditions \
                                import Transmissive_boundary



    #-----------------------------
    # Shallow Water Tsunamis
    #-----------------------------

    from anuga.tsunami_source.smf import slide_tsunami, slump_tsunami



    #-----------------------------
    # Forcing
    # These are old, should use operators
    #-----------------------------
    from anuga.shallow_water.forcing import Inflow, Rainfall, Wind_stress



    #-----------------------------
    # File conversion utilities
    #-----------------------------
    from anuga.file_conversion.file_conversion import sww2obj, \
                        timefile2netcdf, tsh2sww
    from anuga.file_conversion.urs2nc import urs2nc
    from anuga.file_conversion.urs2sww import urs2sww  
    from anuga.file_conversion.urs2sts import urs2sts
    from anuga.file_conversion.dem2pts import dem2pts                    
    from anuga.file_conversion.esri2sww import esri2sww   
    from anuga.file_conversion.sww2dem import sww2dem, sww2dem_batch 
    from anuga.file_conversion.asc2dem import asc2dem
    from anuga.file_conversion.xya2pts import xya2pts     
    from anuga.file_conversion.ferret2sww import ferret2sww     
    from anuga.file_conversion.dem2dem import dem2dem
    from anuga.file_conversion.sww2array import sww2array

    #-----------------------------
    # Parsing arguments
    #-----------------------------
    from anuga.utilities.argparsing import create_standard_parser
    from anuga.utilities.argparsing import parse_standard_args


    def get_args():
        """ Explicitly parse the argument list using standard anuga arguments

        Don't use this if you want to setup your own parser
        """
        parser = create_standard_parser()
        return parser.parse_args()


    #-----------------------------
    # Running Script
    #-----------------------------
    from anuga.utilities.run_anuga_script import run_script as run_anuga_script


    #-----------------------------
    # Mesh API
    #-----------------------------
    from anuga.pmesh.mesh_interface import create_mesh_from_regions

    #-----------------------------
    # SWW file access
    #-----------------------------
    from anuga.shallow_water.sww_interrogate import get_flow_through_cross_section

    #---------------------------
    # Operators
    #---------------------------
    from anuga.operators.kinematic_viscosity_operator import Kinematic_viscosity_operator

    from anuga.operators.rate_operators import Rate_operator
    from anuga.operators.set_friction_operators import Depth_friction_operator 

    from anuga.operators.set_elevation_operator import Set_elevation_operator
    from anuga.operators.set_quantity_operator import Set_quantity_operator
    from anuga.operators.set_stage_operator import Set_stage_operator

    from anuga.operators.set_elevation import Set_elevation
    from anuga.operators.set_quantity import Set_quantity

    from anuga.operators.erosion_operators import Bed_shear_erosion_operator
    from anuga.operators.erosion_operators import Flat_slice_erosion_operator
    from anuga.operators.erosion_operators import Flat_fill_slice_erosion_operator

    #---------------------------
    # Structure Operators
    #---------------------------


    if pypar_available:
        from anuga.parallel.parallel_operator_factory import Inlet_operator
        from anuga.parallel.parallel_operator_factory import Boyd_box_operator
        from anuga.parallel.parallel_operator_factory import Boyd_pipe_operator
        from anuga.parallel.parallel_operator_factory import Weir_orifice_trapezoid_operator
        from anuga.parallel.parallel_operator_factory import Internal_boundary_operator
    else:
        from anuga.structures.inlet_operator import Inlet_operator
        from anuga.structures.boyd_box_operator import Boyd_box_operator
        from anuga.structures.boyd_pipe_operator import Boyd_pipe_operator
        from anuga.structures.weir_orifice_trapezoid_operator import Weir_orifice_trapezoid_operator
        from anuga.structures.internal_boundary_operator import Internal_boundary_operator


    #----------------------------
    # Parallel distribute
    #----------------------------


    #----------------------------
    # 
    #Added by Petar Milevski 10/09/2013
    #import time, os

    from anuga.utilities.model_tools import get_polygon_from_single_file
    from anuga.utilities.model_tools import get_polygons_from_Mid_Mif
    from anuga.utilities.model_tools import get_polygon_list_from_files
    from anuga.utilities.model_tools import get_polygon_dictionary
    from anuga.utilities.model_tools import get_polygon_value_list
    from anuga.utilities.model_tools import read_polygon_dir
    from anuga.utilities.model_tools import read_hole_dir_multi_files_with_single_poly
    from anuga.utilities.model_tools import read_multi_poly_file
    from anuga.utilities.model_tools import read_hole_dir_single_file_with_multi_poly
    from anuga.utilities.model_tools import read_multi_poly_file_value
    from anuga.utilities.model_tools import Create_culvert_bridge_Operator


    #---------------------------
    # User Access Functions
    #---------------------------

    from anuga.utilities.system_tools import get_user_name, get_host_name, \
        get_revision_number
    from anuga.utilities.mem_time_equation import estimate_time_mem


    #-------------------------
    # create domain functions
    #-------------------------
    from anuga.extras import create_domain_from_regions
    from anuga.extras import create_domain_from_file
    from anuga.extras import rectangular_cross_domain

    
    #import logging as log
    from anuga.utilities import log

    from anuga.config import g
    from anuga.config import velocity_protection
    
## if use_psyco:
##     # try using psyco if available
##     try:
##         import psyco
##     except:
##         import os
##         import sys
##         if os.name == 'posix' and os.uname()[4] in ['x86_64', 'ia64']:
##             pass
##             # Psyco isn't supported on 64 bit systems, but it doesn't matter
##         elif sys.version[:3] == '2.7' :
##             pass
##             # Psyco isn't available for python 2.7 (16/05/2011)
##         else:
##             log.critical('WARNING: psyco (speedup) could not be imported, '
##                          'you may want to consider installing it')
##     else:
##         psyco.full() # aggressively compile everything
##         #psyco.background() # attempt to profile code - only compile most used
        




