# -*- coding: utf-8 -*-
#
# boing/gesture/rubine.py -
#
# Author: Paolo Olivo (paolo.olivo@inria.fr)
#
# Copyright © INRIA
#
# See the file LICENSE for information on usage and redistribution of
# this file, and for a DISCLAIMER OF ALL WARRANTIES.

"""The module :mod:`boing.gesture.rubine` provides an implementation
of the Rubine's gesture recognition algorithm.

"""

import collections
import datetime
import math

import numpy

from boing.gesture.utils import boundingBox
from boing.utils import quickdict

def computeFeatures(stroke):
    """Compute and return the 13 Rubine's features. *stroke* must
    composed by at least three points."""
    if len(stroke)<3: raise ValueError(
        "stroke must be composed by at least three points.")
    features = [0.0] * 13
    # f1, f2
    x0, x2 = stroke[0].x, stroke[2].x
    y0, y2 = stroke[0].y, stroke[2].y
    div = math.sqrt((x2-x0)**2 + (y2-y0)**2)
    if div==0: f1 = f2 = 0
    else:
        f1 = (x2-x0)/div
        f2 = (y2-y0)/div
    features[0] = f1
    features[1] = f2
    # f3
    minx, miny, maxx, maxy = boundingBox(stroke)
    features[2] = math.sqrt((maxx-minx)**2 + (maxy-miny)**2)
    # f4
    features[3] = math.atan2(maxy-miny, maxx-minx)
    # f5
    xp_1, yp_1 = stroke[-1].x, stroke[-1].y
    f5 = math.sqrt((xp_1-x0)**2 + (yp_1-y0)**2)
    features[4] = f5
    # f6, f7
    if f5==0: f6 = f7 = 0
    else:
        f6 = (xp_1-x0)/f5
        f7 = (yp_1-y0)/f5
    features[5] = f6
    features[6] = f7
    # f8, f9, f10, f11, f12
    Dx0 = stroke[1].x-stroke[0].x
    Dy0 = stroke[1].y-stroke[0].y
    tdelta = stroke[1].t-stroke[0].t
    Dt0 = tdelta.seconds * 1000 + tdelta.microseconds * 0.001
    f8 = math.sqrt(Dx0**2 + Dy0**2)
    f9 = f10 = f11 = 0
    f12 = (Dx0**2+Dy0**2)/(Dt0**2)
    for p in range(1, len(stroke) - 2):
        Dxp = stroke[p+1].x - stroke[p].x
        Dyp = stroke[p+1].y - stroke[p].y
        Dxp_1 = stroke[p].x - stroke[p-1].x
        Dyp_1 = stroke[p].y - stroke[p-1].y
        Tp = math.atan2(Dxp*Dyp_1 - Dxp_1*Dyp, Dxp*Dxp_1 + Dyp*Dyp_1)
        tdelta = stroke[p+1].t-stroke[p].t
        Dtp = tdelta.seconds * 1000 + tdelta.microseconds * 0.001
        f8 += math.sqrt(Dxp**2+Dyp**2)
        f9 += Tp
        f10 += math.fabs(Tp)
        f11 += Tp**2
        val = (Dxp**2 + Dyp**2) / (Dtp**2)
        if val > f12: f12 = val
    features[7] = f8
    features[8] = f9
    features[9] = f10
    features[10] = f11
    features[11] = f12
    # f13
    tdelta = stroke[-1].t-stroke[0].t
    features[12] = tdelta.seconds * 1000 + tdelta.microseconds * 0.001
    return features

# ---------------------------------------------------------------------

class RubineRecognizer:
    """Recognizer using the Rubine's recognition algorithm."""

    def __init__(self):
        self._classes = []
        self._templates = []
        self._weights = []

    def reset(self):
        """Reset the classifier."""
        self._classes = []
        self._templates = []
        self._weights = []

    def classes(self):
        """Return a copy of the current recognized classes."""
        return self._classes[:]

    def isValid(self, stroke):
        """Return whether *stroke* is valid instance for recognition
        or as training instance. *stroke* is valid if it is a container
        and it has at least 3 items."""
        return isinstance(stroke, collections.Container) and len(stroke)>3

    def getProperties(self):
        return {"dimensions":"xyt", "multistroke":False}

    def loadTestTemplates(self):
        """Load the test training set."""
        instances = list((name, normalize(stroke)) \
                             for name, stroke in testInstances())
        self.buildRecognizer(instances)

    def _addTemplate(self, name, stroke):
        """Add a new template *strokes* for the class *name*."""
        if not self.isValid(stroke): rvalue = False
        elif name in self._classes:
            self._templates[self._classes.index(name)].append(stroke)
        else:
            self._classes.append(name)
            self._templates.append([stroke])
            self._weights = [0.0]*len(self._classes)

    def buildRecognizer(self, instances):
        """Train the recognizer using *instances* as training set.
        *instances* must be a list of tuples (class, stroke)."""
        self.reset()
        # Add templates
        for name, stroke in instances:
            self._addTemplate(name, stroke)
        # do training
        class_fmean = [None] * len(self._classes)
        class_cov = numpy.empty((len(self._classes), 13, 13), numpy.float32)
        t_num = 0
        for c, t_set in enumerate(self._templates):
            t_num += len(t_set)
            # 1 - get features for each training instance
            f_set = numpy.empty((len(t_set), 13), numpy.float32)
            for i in range(len(t_set)):
                f_set[i] = computeFeatures(t_set[i])
            # 2 - get class' mean feature vector
            f_mean = numpy.mean(f_set, axis=0)
            class_fmean[c] = f_mean
            # 3 - get class' covariance matrix
            cov = numpy.empty((13, 13), numpy.float32)
            for i in range(13):
                for j in range(13):
                    summ = 0
                    for e in range(len(t_set)):
                        summ += (f_set[e][i] - f_mean[i]) * (f_set[e][j] - f_mean[j])
                    cov[i][j] = summ
            class_cov[c] = cov
        # 4 - get covariance average matrix
        cov_mean = numpy.matrix(numpy.sum(class_cov, axis=0) \
                                    / (-len(self._classes)+t_num))
        # 5 - get the inverse of the covariance average matrix
        # TODO:  Handle singular matrix
        # Use determinant to understand if matrix is singular
        # det = linalg.det(cov_mean)
        inverse = cov_mean.I
        # 6 - get class weights
        for c, f_mean in enumerate(class_fmean):
            wc = numpy.zeros(14)
            for j in range(13):
                for i in range(13):
                    wc[j+1] += inverse[i,j] * f_mean[i]
            for i in range(13):
                wc[0] += wc[i+1] * f_mean[i]
            wc[0] *= -0.5
            self._weights[c] = wc

    def recognize(self, stroke):
        """Return a dictionary {"cls": name}, where 'name' is the
        class that best fits *stroke*; return None if the recognizer has no
        classes or *stroke* is not a valid instance."""
        if not self._weights or not self.isValid(stroke): rvalue = None
        else:
            features = computeFeatures(stroke)
            maximum = float("-inf")
            best = None
            for c, weights in enumerate(self._weights):
                summ = weights[0]
                for i in range(13):
                    summ += weights[i+1]*features[i]
                if summ>maximum:
                    maximum = summ
                    best = c
            rvalue = dict(cls=self._classes[best])
        return rvalue

# ---------------------------------------------------------------------

_test_instances = [
 ("circle",[(1158465,141,145),(1158471,139,142),(1158479,135,141),(1158484,133,139),(1158490,131,139),(1158496,126,136),(1158502,123,135),(1158508,117,135),(1158515,114,136),(1158516,111,136),(1158524,105,139),(1158532,100,143),(1158539,94,148),(1158547,90,153),(1158555,86,158),(1158563,82,165),(1158572,79,172),(1158580,76,179),(1158589,74,186),(1158598,72,192),(1158608,71,202),(1158617,72,210),(1158627,73,215),(1158628,74,218),(1158640,79,225),(1158650,82,230),(1158651,84,232),(1158664,90,237),(1158675,98,241),(1158687,106,244),(1158699,112,245),(1158700,115,245),(1158713,125,244),(1158725,136,242),(1158738,146,236),(1158750,157,229),(1158751,160,227),(1158766,170,220),(1158779,180,208),(1158793,185,198),(1158806,187,188),(1158807,187,184),(1158824,184,171),(1158838,179,160),(1158852,171,148),(1158868,162,140),(1158883,154,138),(1158884,151,137),(1158901,140,140),(1158916,137,142)]),
 ("x",[(1120247,86,150),(1120258,85,152),(1120269,87,154),(1120277,88,157),(1120286,90,160),(1120293,91,163),(1120299,93,164),(1120307,99,170),(1120314,103,173),(1120322,106,176),(1120323,108,179),(1120333,112,183),(1120341,117,188),(1120343,119,190),(1120352,123,194),(1120360,126,200),(1120369,130,204),(1120377,133,208),(1120385,137,211),(1120396,140,214),(1120405,143,217),(1120406,146,220),(1120417,149,221),(1120427,152,222),(1120437,154,224),(1120447,156,225),(1120458,158,225),(1120482,160,223),(1120493,160,221),(1120504,160,213),(1120515,159,205),(1120527,157,197),(1120539,155,186),(1120551,154,174),(1120564,153,163),(1120576,153,151),(1120589,154,143),(1120602,155,136),(1120615,156,130),(1120627,157,128),(1120662,155,128),(1120675,152,130),(1120689,147,136),(1120703,140,144),(1120719,131,154),(1120734,121,168),(1120749,113,180),(1120750,110,183),(1120768,102,198),(1120783,96,211),(1120799,93,218),(1120815,90,223),(1120878,91,221)]),
 ("rectangle",[(913821,71,146),(913865,71,149),(913873,72,151),(913881,72,154),(913886,72,156),(913893,73,159),(913899,73,161),(913906,73,166),(913913,73,168),(913914,73,170),(913922,73,175),(913929,73,180),(913937,73,185),(913945,73,188),(913946,73,190),(913955,73,196),(913964,73,202),(913973,73,206),(913974,73,208),(913984,74,213),(913993,74,216),(913994,74,219),(914004,74,224),(914006,74,226),(914016,75,229),(914017,75,231),(914030,75,236),(914040,75,239),(914042,75,241),(914053,75,244),(914070,75,246),(914102,75,248),(914183,77,248),(914208,79,246),(914224,81,246),(914236,84,246),(914248,87,246),(914261,89,245),(914274,94,246),(914287,97,246),(914300,104,247),(914314,111,247),(914328,119,249),(914345,127,250),(914359,133,251),(914361,135,252),(914380,146,252),(914396,152,251),(914397,154,252),(914413,160,251),(914429,168,252),(914446,174,253),(914462,179,253),(914478,183,254),(914494,185,255),(914510,187,255),(914544,189,257),(914604,189,255),(914625,188,252),(914642,188,250),(914660,187,245),(914678,187,240),(914696,185,230),(914714,185,221),(914733,184,213),(914752,184,205),(914771,183,194),(914791,181,184),(914811,181,173),(914830,180,165),(914850,180,157),(914870,180,151),(914890,179,149),(914910,179,146),(914933,177,142),(914954,177,139),(914975,175,136),(914996,172,132),(915017,169,130),(915038,165,129),(915060,161,130),(915082,153,130),(915104,146,130),(915126,138,132),(915150,130,133),(915172,124,134),(915173,121,134),(915199,112,136),(915223,104,136),(915247,96,134),(915271,89,135),(915295,85,136),(915319,83,136),(915344,81,136),(915368,79,135)]),
 ("left_curly_brace",[(1404198,137,126),(1404206,134,126),(1404214,132,127),(1404221,129,127),(1404230,126,128),(1404238,123,131),(1404246,121,132),(1404254,119,133),(1404262,117,136),(1404270,115,139),(1404278,114,142),(1404290,113,144),(1404298,112,147),(1404318,114,150),(1404331,117,152),(1404340,119,152),(1404349,122,153),(1404358,124,155),(1404367,127,157),(1404377,130,157),(1404387,133,159),(1404388,136,161),(1404400,138,163),(1404410,140,165),(1404423,140,168),(1404435,138,170),(1404446,135,172),(1404457,132,173),(1404469,126,174),(1404480,123,174),(1404492,119,175),(1404507,117,175),(1404571,119,176),(1404584,121,175),(1404596,124,177),(1404609,126,177),(1404623,128,178),(1404624,130,178),(1404643,132,180),(1404657,132,183),(1404671,131,186),(1404685,128,192),(1404700,125,198),(1404716,124,202),(1404731,122,205),(1404732,121,208),(1404752,120,212),(1404767,120,215),(1404782,121,217),(1404798,125,220),(1404815,130,221),(1404831,136,222),(1404849,142,221)]),
 ("left_curly_brace",[(1402972,142,123),(1402983,140,122),(1402995,137,122),(1403003,134,124),(1403012,131,123),(1403020,128,124),(1403028,125,125),(1403036,123,128),(1403044,120,131),(1403052,118,133),(1403060,116,136),(1403068,115,139),(1403076,114,141),(1403088,114,143),(1403113,117,145),(1403121,118,147),(1403129,120,148),(1403138,123,148),(1403146,125,149),(1403155,128,150),(1403164,131,151),(1403173,134,151),(1403183,136,153),(1403193,139,154),(1403203,141,155),(1403213,143,158),(1403229,142,161),(1403241,139,163),(1403252,136,164),(1403263,133,165),(1403274,127,167),(1403286,125,167),(1403298,122,167),(1403309,120,167),(1403369,122,165),(1403382,124,166),(1403394,127,166),(1403407,129,166),(1403422,131,167),(1403442,133,170),(1403456,133,172),(1403469,131,177),(1403483,130,180),(1403497,128,185),(1403512,127,188),(1403526,127,191),(1403541,126,195),(1403557,126,198),(1403559,126,200),(1403575,126,202),(1403591,125,207),(1403606,126,213),(1403623,126,218),(1403639,128,224),(1403655,129,226),(1403672,134,229),(1403689,136,229)]),
 ("left_sq_bracket",[(977374,137,113),(977393,135,113),(977405,133,113),(977417,130,113),(977428,128,114),(977437,126,115),(977445,123,116),(977453,121,115),(977461,119,115),(977469,116,115),(977477,114,116),(977485,111,116),(977493,109,117),(977502,107,116),(977510,104,116),(977521,102,115),(977529,100,116),(977538,98,117),(977549,95,117),(977562,93,118),(977574,91,118),(977585,89,120),(977602,87,121),(977618,85,122),(977642,83,123),(977666,81,125),(977686,80,127),(977710,80,129),(977722,80,132),(977734,80,134),(977746,81,137),(977758,81,139),(977771,82,145),(977783,82,149),(977796,82,155),(977809,82,162),(977811,83,164),(977825,83,168),(977839,83,176),(977853,83,184),(977867,84,190),(977882,84,196),(977896,84,202),(977911,84,207),(977926,85,215),(977942,85,221),(977960,85,223),(977975,86,228),(977991,86,230),(978007,86,232),(978023,87,235),(978039,87,237),(978127,87,239),(978179,87,241),(978297,89,239),(978318,91,238),(978335,93,239),(978353,96,239),(978355,99,239),(978373,103,240),(978392,108,240),(978410,117,239),(978430,122,240),(978449,126,239),(978451,128,239),(978471,131,239),(978490,133,239),(978509,135,239)]),
 ("pigtail",[(1069523,69,233),(1069531,70,236),(1069539,72,236),(1069548,75,236),(1069555,78,235),(1069563,81,235),(1069572,85,233),(1069580,89,231),(1069588,91,230),(1069589,93,228),(1069599,98,226),(1069607,102,224),(1069609,104,223),(1069619,108,221),(1069628,114,216),(1069638,118,213),(1069647,122,210),(1069657,125,207),(1069667,128,204),(1069669,131,201),(1069681,134,198),(1069692,139,192),(1069703,141,188),(1069714,142,185),(1069726,144,178),(1069737,145,173),(1069749,145,169),(1069761,143,163),(1069773,142,161),(1069787,138,155),(1069799,136,153),(1069801,134,151),(1069815,132,150),(1069829,127,148),(1069843,121,150),(1069858,116,152),(1069872,112,156),(1069888,107,162),(1069903,105,167),(1069917,102,175),(1069934,101,181),(1069951,100,189),(1069968,104,201),(1069985,107,205),(1070004,114,212),(1070021,120,216),(1070039,127,220),(1070057,139,223),(1070076,147,226),(1070077,149,226),(1070097,157,226),(1070116,168,224),(1070134,175,220),(1070153,178,218)]),
 ("pigtail",[(1067055,69,226),(1067065,72,229),(1067073,75,230),(1067081,79,231),(1067090,83,230),(1067097,85,229),(1067103,91,226),(1067111,96,225),(1067118,99,225),(1067119,102,224),(1067129,108,220),(1067137,114,217),(1067139,117,215),(1067149,122,212),(1067157,128,209),(1067159,130,207),(1067170,135,203),(1067171,137,203),(1067181,142,199),(1067191,148,192),(1067200,150,190),(1067211,156,183),(1067221,157,181),(1067231,160,174),(1067242,160,168),(1067253,160,166),(1067265,157,161),(1067276,156,158),(1067288,151,154),(1067299,149,154),(1067311,142,154),(1067323,139,154),(1067336,133,156),(1067348,130,157),(1067361,124,162),(1067375,121,166),(1067388,117,173),(1067401,115,179),(1067403,115,181),(1067419,113,188),(1067433,113,195),(1067448,113,202),(1067462,115,209),(1067477,116,212),(1067492,120,218),(1067509,125,222),(1067524,132,225),(1067540,139,227),(1067556,145,230),(1067572,152,231),(1067588,158,231),(1067605,163,229),(1067622,167,226),(1067639,169,226)]),
 ("check",[(842650,76,190),(842678,77,188),(842694,79,186),(842705,81,184),(842717,83,183),(842729,86,184),(842737,88,185),(842745,91,187),(842753,94,190),(842761,97,194),(842769,100,197),(842777,103,201),(842785,105,203),(842794,108,208),(842802,110,211),(842809,112,214),(842817,113,217),(842826,114,219),(842837,116,221),(842865,118,221),(842889,119,218),(842899,119,215),(842909,118,212),(842920,118,207),(842931,119,199),(842942,121,191),(842953,122,186),(842965,126,179),(842966,127,176),(842980,131,170),(842993,138,161),(842994,140,159),(843008,146,151),(843010,148,148),(843023,151,144),(843035,156,137),(843047,158,133),(843060,160,128),(843073,161,126),(843074,163,123),(843089,164,120),(843090,165,117),(843106,165,115)]),
 ("triangle",[(1300756,133,143),(1300767,131,146),(1300775,129,149),(1300783,127,152),(1300791,124,156),(1300797,123,159),(1300803,120,163),(1300810,118,165),(1300817,115,170),(1300824,111,176),(1300833,107,181),(1300840,103,187),(1300848,99,192),(1300856,95,198),(1300864,91,203),(1300872,87,209),(1300881,83,213),(1300890,80,217),(1300898,78,220),(1300899,76,223),(1300910,73,227),(1300920,70,230),(1300929,68,232),(1300939,66,234),(1300948,64,236),(1300968,62,238),(1301021,64,240),(1301031,66,240),(1301042,69,240),(1301053,72,240),(1301054,76,240),(1301066,82,239),(1301078,90,238),(1301090,98,236),(1301103,106,234),(1301115,115,233),(1301128,124,230),(1301141,135,231),(1301154,143,231),(1301168,150,232),(1301182,158,233),(1301195,162,233),(1301209,168,235),(1301223,174,237),(1301237,177,239),(1301238,179,239),(1301254,184,239),(1301269,186,239),(1301284,189,238),(1301299,191,236),(1301314,192,233),(1301335,191,229),(1301351,188,223),(1301367,185,216),(1301384,181,208),(1301401,176,200),(1301418,170,191),(1301436,165,185),(1301454,160,180),(1301471,153,170),(1301488,149,164),(1301490,148,162),(1301508,144,157),(1301527,141,152),(1301546,139,147),(1301564,137,145)]),
 ("star",[(672186,75,260),(672214,76,258),(672234,78,256),(672247,80,254),(672259,81,251),(672267,83,249),(672275,84,246),(672283,85,244),(672291,86,241),(672298,87,238),(672300,88,236),(672309,89,233),(672318,90,228),(672326,91,224),(672328,92,222),(672337,93,217),(672346,94,213),(672348,94,211),(672358,95,205),(672360,96,203),(672372,97,195),(672382,98,191),(672392,98,185),(672403,100,176),(672414,101,172),(672425,101,166),(672437,102,161),(672448,102,151),(672460,103,146),(672472,103,140),(672484,105,134),(672496,105,132),(672509,106,127),(672521,107,124),(672534,107,122),(672547,108,119),(672572,110,118),(672600,112,118),(672620,114,120),(672636,116,123),(672650,118,126),(672664,121,132),(672679,123,138),(672681,124,140),(672697,127,147),(672713,132,157),(672729,138,169),(672747,142,178),(672749,143,181),(672767,148,192),(672769,149,195),(672786,153,203),(672803,158,212),(672805,160,215),(672824,165,225),(672842,168,233),(672860,170,237),(672861,171,240),(672881,173,246),(672899,175,251),(672917,176,256),(672936,176,258),(672973,174,260),(672993,172,260),(673012,170,258),(673032,165,254),(673052,157,248),(673072,145,240),(673074,143,238),(673098,130,231),(673119,119,225),(673141,108,219),(673142,105,218),(673166,92,209),(673188,83,204),(673211,73,196),(673233,64,191),(673255,56,183),(673277,52,179),(673302,48,175),(673362,50,175),(673385,54,175),(673408,63,176),(673433,74,175),(673457,88,174),(673482,103,173),(673483,105,172),(673510,122,170),(673511,124,170),(673537,140,169),(673563,157,169),(673589,167,168),(673614,179,168),(673640,186,168),(673666,193,168),(673726,193,170),(673752,188,173),(673778,177,179),(673805,166,187),(673833,151,196),(673861,136,208),(673891,122,218),(673920,107,229),(673948,99,238),(673977,89,247),(674005,83,255),(674034,79,259),(674062,77,261)]),
 ("star",[(683110,80,261),(683126,80,259),(683138,80,257),(683147,80,254),(683159,81,252),(683168,82,250),(683176,83,246),(683184,84,243),(683192,84,239),(683200,85,236),(683208,85,232),(683215,85,230),(683216,85,228),(683225,85,226),(683232,85,221),(683240,86,218),(683248,86,213),(683257,86,211),(683265,87,205),(683274,87,202),(683283,87,196),(683284,87,194),(683295,87,190),(683305,87,183),(683314,87,181),(683324,87,174),(683334,87,172),(683345,87,165),(683355,88,161),(683366,89,158),(683377,91,149),(683389,93,144),(683400,94,139),(683412,95,136),(683425,96,131),(683437,96,128),(683449,96,122),(683462,96,119),(683475,97,115),(683487,97,113),(683489,97,111),(683504,98,109),(683518,99,107),(683531,100,104),(683572,102,105),(683588,103,108),(683603,104,110),(683618,106,115),(683633,110,122),(683649,114,132),(683665,118,142),(683681,122,152),(683697,126,162),(683713,131,171),(683730,136,181),(683746,140,189),(683763,145,197),(683764,148,201),(683784,153,211),(683785,154,213),(683805,159,223),(683823,163,229),(683841,165,235),(683859,168,239),(683877,168,241),(683901,168,244),(683919,168,246),(683944,166,247),(683964,163,246),(683984,161,245),(684004,158,244),(684025,154,244),(684046,149,240),(684067,140,233),(684089,128,223),(684112,117,214),(684134,105,203),(684156,94,194),(684179,85,187),(684202,73,180),(684225,62,173),(684248,56,168),(684249,53,167),(684274,46,161),(684299,44,159),(684323,41,157),(684364,44,159),(684389,49,159),(684414,62,161),(684440,77,163),(684466,92,162),(684493,109,164),(684519,124,165),(684545,141,166),(684573,157,166),(684600,172,168),(684626,181,170),(684653,185,169),(684681,188,168),(684716,185,170),(684743,180,173),(684770,172,177),(684797,160,186),(684826,149,194),(684854,140,202),(684882,130,211),(684910,124,219),(684939,115,226),(684968,107,234),(684997,101,242),(685027,93,249),(685056,88,255),(685085,83,261),(685115,80,262),(685144,79,264),(685176,79,262)]),
 ("right_curly_brace",[(1201443,122,144),(1201459,122,142),(1201483,124,141),(1201496,127,141),(1201504,129,141),(1201512,131,141),(1201520,134,141),(1201528,136,140),(1201536,139,140),(1201544,142,142),(1201552,144,143),(1201561,147,145),(1201568,149,146),(1201580,152,149),(1201596,154,152),(1201607,154,154),(1201620,154,156),(1201629,153,159),(1201638,151,161),(1201647,149,162),(1201658,147,164),(1201668,141,169),(1201678,139,171),(1201689,136,173),(1201699,134,174),(1201700,132,174),(1201712,130,176),(1201723,128,178),(1201724,126,180),(1201737,124,182),(1201748,122,185),(1201763,120,189),(1201775,119,192),(1201776,119,194),(1201792,119,196),(1201805,120,198),(1201819,122,201),(1201833,126,201),(1201847,131,202),(1201862,135,202),(1201876,140,201),(1201891,142,201),(1201906,144,200),(1201922,149,198),(1201944,150,195),(1201980,148,196),(1201996,145,196),(1202012,141,197),(1202028,137,199),(1202045,134,200),(1202062,130,202),(1202079,128,203),(1202143,128,205),(1202160,130,206),(1202178,135,207),(1202197,140,209),(1202216,145,213),(1202236,150,220),(1202254,151,222),(1202255,151,225),(1202274,151,231),(1202294,148,238),(1202314,144,243),(1202335,135,249),(1202357,126,250),(1202378,117,249),(1202379,114,250),(1202399,108,252),(1202420,104,254)]),
 ("v",[(1020136,79,167),(1020144,78,165),(1020153,77,163),(1020160,76,161),(1020173,76,159),(1020185,76,156),(1020226,78,158),(1020234,79,161),(1020242,81,165),(1020249,81,168),(1020250,83,171),(1020260,85,177),(1020268,88,184),(1020276,91,190),(1020285,94,196),(1020294,97,202),(1020295,99,204),(1020305,101,210),(1020307,103,213),(1020318,107,218),(1020328,109,223),(1020337,111,227),(1020347,113,230),(1020357,115,232),(1020367,116,234),(1020383,118,236),(1020411,120,235),(1020431,122,232),(1020442,123,230),(1020453,124,228),(1020464,126,223),(1020476,127,220),(1020488,130,212),(1020500,132,203),(1020513,135,195),(1020526,137,185),(1020540,140,169),(1020553,143,159),(1020566,144,154),(1020568,145,151),(1020583,147,148),(1020584,148,145),(1020616,149,143)]),
 ("right_sq_bracket",[(799964,118,134),(800008,120,134),(800017,122,133),(800025,124,133),(800033,126,133),(800041,129,133),(800049,131,134),(800057,133,134),(800065,135,133),(800074,138,134),(800082,140,134),(800095,142,135),(800108,145,135),(800120,147,135),(800145,149,136),(800176,151,138),(800184,151,140),(800193,151,143),(800203,151,147),(800205,151,150),(800215,151,154),(800217,151,157),(800228,151,162),(800229,151,165),(800240,151,172),(800241,152,175),(800253,152,181),(800265,154,191),(800276,155,198),(800288,155,200),(800289,156,204),(800303,157,208),(800315,157,213),(800328,157,217),(800329,157,219),(800342,156,221),(800356,156,224),(800380,153,225),(800396,151,226),(800409,148,228),(800423,146,229),(800437,141,230),(800451,136,230),(800466,130,231),(800481,124,231),(800496,117,232),(800511,115,233),(800513,113,234),(800530,110,236),(800545,108,236),(800561,106,235)]),
 ("rectangle",[(907625,75,145),(907649,75,147),(907657,75,150),(907664,75,153),(907673,75,157),(907681,76,161),(907687,76,164),(907694,77,169),(907701,77,171),(907702,77,174),(907709,78,179),(907717,78,183),(907718,78,186),(907727,78,191),(907735,78,197),(907743,78,202),(907751,79,207),(907760,79,213),(907761,79,215),(907770,79,220),(907780,79,224),(907792,79,229),(907801,80,234),(907803,80,236),(907815,80,241),(907826,81,243),(907842,81,245),(907909,83,248),(907950,85,249),(907971,88,249),(907983,90,248),(907995,92,248),(908007,95,248),(908019,98,247),(908031,104,247),(908044,107,246),(908056,114,246),(908069,117,245),(908082,124,246),(908095,131,246),(908108,135,247),(908122,142,246),(908137,150,245),(908151,154,245),(908152,158,245),(908168,166,244),(908182,169,244),(908183,172,245),(908201,178,245),(908219,183,245),(908235,187,246),(908250,190,246),(908266,193,246),(908267,196,246),(908284,198,246),(908300,200,246),(908318,202,246),(908344,204,247),(908411,204,244),(908431,203,241),(908448,203,238),(908466,203,232),(908484,203,225),(908503,202,218),(908504,202,216),(908525,201,206),(908544,200,194),(908564,200,178),(908584,200,166),(908604,199,156),(908624,199,152),(908644,198,147),(908664,198,143),(908684,198,137),(908705,197,133),(908725,197,130),(908756,195,128),(908789,193,128),(908810,191,128),(908831,188,128),(908853,184,130),(908875,179,129),(908897,174,129),(908919,167,129),(908942,159,129),(908965,153,129),(908966,150,129),(908990,141,129),(909013,131,128),(909037,125,128),(909038,122,128),(909063,113,129),(909088,105,130),(909113,98,128),(909138,91,131),(909164,83,134),(909189,77,135),(909193,74,136),(909220,70,136),(909246,69,134)]),
 ("triangle",[(1308827,143,128),(1308838,141,130),(1308843,140,132),(1308850,138,135),(1308858,136,138),(1308866,134,141),(1308872,132,143),(1308878,130,147),(1308886,127,150),(1308893,125,153),(1308900,122,158),(1308907,118,162),(1308914,114,167),(1308922,110,172),(1308930,108,175),(1308931,106,178),(1308941,101,183),(1308949,96,189),(1308950,94,192),(1308961,89,197),(1308962,87,199),(1308974,83,203),(1308975,81,206),(1308986,75,211),(1308996,72,214),(1309006,67,219),(1309016,65,221),(1309027,62,223),(1309037,61,226),(1309038,59,228),(1309054,57,229),(1309082,57,232),(1309098,60,233),(1309111,62,232),(1309123,65,232),(1309135,70,232),(1309147,76,232),(1309161,82,232),(1309173,89,232),(1309175,91,231),(1309190,101,231),(1309204,109,230),(1309219,120,232),(1309233,127,233),(1309248,138,233),(1309263,148,233),(1309278,155,235),(1309279,157,235),(1309296,165,236),(1309311,171,236),(1309327,175,238),(1309343,180,239),(1309359,184,238),(1309375,186,237),(1309401,188,234),(1309418,187,231),(1309435,186,226),(1309452,184,222),(1309470,180,214),(1309488,175,203),(1309506,171,194),(1309524,166,186),(1309542,161,177),(1309561,156,168),(1309579,152,159),(1309598,149,151),(1309617,146,144)]),
 ("circle",[(1154653,126,140),(1154660,122,138),(1154666,120,138),(1154673,115,139),(1154679,113,139),(1154681,110,138),(1154689,105,139),(1154696,102,141),(1154697,100,142),(1154706,96,145),(1154714,92,151),(1154722,89,157),(1154730,86,164),(1154739,84,172),(1154748,83,181),(1154749,83,186),(1154758,83,195),(1154768,85,208),(1154778,87,216),(1154788,93,225),(1154798,98,230),(1154809,107,234),(1154819,113,234),(1154830,124,234),(1154841,134,230),(1154852,140,226),(1154853,143,224),(1154866,151,216),(1154878,158,207),(1154891,163,197),(1154903,167,185),(1154915,168,172),(1154916,167,168),(1154929,164,155),(1154943,159,145),(1154956,151,135),(1154970,145,132),(1154983,139,132),(1154985,136,133),(1155001,132,135)]),
 ("v",[(1028940,85,147),(1028996,85,150),(1029008,87,152),(1029017,87,155),(1029024,88,158),(1029033,89,162),(1029040,89,164),(1029047,90,166),(1029055,91,173),(1029063,92,176),(1029071,94,181),(1029079,96,187),(1029088,98,191),(1029097,100,196),(1029098,101,198),(1029110,103,205),(1029119,105,210),(1029129,107,214),(1029130,108,217),(1029142,110,220),(1029152,112,223),(1029162,115,228),(1029173,116,230),(1029185,119,232),(1029210,121,234),(1029250,123,235),(1029278,123,237),(1029318,124,235),(1029330,124,231),(1029343,125,223),(1029355,126,216),(1029369,127,207),(1029383,130,197),(1029396,133,190),(1029411,139,178),(1029426,143,171),(1029427,144,169),(1029444,148,162),(1029459,152,155),(1029473,154,152),(1029475,156,150),(1029495,157,148)]),
 ("right_curly_brace",[(1204820,128,136),(1204894,130,136),(1204907,132,136),(1204915,134,135),(1204927,137,135),(1204935,140,135),(1204943,142,136),(1204951,144,137),(1204959,146,138),(1204971,148,140),(1204987,150,142),(1204995,150,145),(1205003,150,147),(1205011,150,151),(1205019,149,153),(1205028,148,155),(1205036,147,159),(1205045,146,161),(1205055,144,165),(1205056,142,167),(1205067,140,169),(1205077,138,172),(1205087,135,174),(1205098,133,175),(1205109,128,180),(1205120,125,183),(1205131,121,187),(1205142,120,190),(1205154,118,192),(1205166,117,195),(1205178,116,197),(1205191,117,200),(1205203,120,201),(1205216,124,202),(1205229,127,202),(1205244,134,200),(1205257,140,200),(1205271,147,201),(1205285,149,200),(1205300,154,200),(1205315,156,199),(1205329,158,199),(1205376,156,198),(1205391,154,198),(1205407,149,199),(1205423,144,201),(1205439,139,202),(1205455,133,204),(1205472,129,204),(1205489,127,204),(1205507,125,206),(1205532,123,208),(1205549,124,210),(1205567,126,211),(1205584,127,213),(1205602,132,217),(1205621,137,220),(1205639,141,223),(1205658,145,227),(1205677,147,233),(1205697,150,240),(1205716,152,244),(1205736,151,248),(1205756,147,252),(1205757,145,253),(1205777,135,257),(1205798,127,258),(1205821,118,259),(1205845,113,261),(1205866,111,262),(1205885,108,261)]),
 ("check",[(845080,81,190),(845119,83,188),(845131,85,185),(845143,87,184),(845159,90,183),(845175,92,183),(845191,94,184),(845208,97,186),(845216,98,188),(845228,100,191),(845236,102,194),(845248,104,196),(845256,106,198),(845265,108,201),(845272,110,203),(845280,112,206),(845288,114,208),(845297,115,211),(845306,116,213),(845316,118,215),(845317,119,219),(845329,120,221),(845340,123,223),(845361,124,225),(845416,123,223),(845427,123,220),(845438,122,218),(845449,121,215),(845461,120,208),(845473,121,202),(845485,121,196),(845498,123,189),(845511,125,182),(845524,128,173),(845538,131,165),(845551,134,158),(845565,140,148),(845579,145,141),(845593,151,131),(845607,153,127),(845609,155,124),(845627,157,121),(845641,158,119),(845656,159,116),(845680,161,113),(845700,164,113)]),
 ("arrow",[(1359271,71,215),(1359286,74,213),(1359294,77,211),(1359300,79,211),(1359306,83,209),(1359313,85,207),(1359320,90,204),(1359327,96,200),(1359335,103,197),(1359342,109,194),(1359350,116,189),(1359359,122,186),(1359367,128,182),(1359376,134,179),(1359385,140,175),(1359394,145,172),(1359395,148,170),(1359408,156,167),(1359421,165,163),(1359431,170,161),(1359441,174,158),(1359451,177,156),(1359461,183,154),(1359472,185,153),(1359483,188,151),(1359493,190,148),(1359526,188,146),(1359537,185,145),(1359549,182,144),(1359551,178,144),(1359565,171,145),(1359577,164,148),(1359589,158,148),(1359601,152,150),(1359613,148,151),(1359614,144,152),(1359633,139,154),(1359646,136,154),(1359658,134,155),(1359699,136,153),(1359712,139,152),(1359726,145,151),(1359740,153,147),(1359755,161,144),(1359770,168,142),(1359771,170,141),(1359788,177,140),(1359806,183,140),(1359821,185,140),(1359836,188,140),(1359851,190,142),(1359871,189,144),(1359887,188,147),(1359903,186,153),(1359920,182,159),(1359937,177,165),(1359954,172,172),(1359971,165,181),(1359988,160,187),(1360006,158,191),(1360023,156,193)]),
 ("caret",[(1256745,86,238),(1256757,87,235),(1256765,89,233),(1256774,90,228),(1256779,91,226),(1256785,92,224),(1256786,94,222),(1256795,96,218),(1256801,97,215),(1256802,98,212),(1256811,101,206),(1256819,104,200),(1256826,106,193),(1256833,108,189),(1256835,109,185),(1256845,112,178),(1256853,114,170),(1256861,116,164),(1256862,117,160),(1256873,118,153),(1256874,119,150),(1256885,119,147),(1256887,120,143),(1256898,121,140),(1256908,122,138),(1256918,122,135),(1256934,124,133),(1256954,127,133),(1256965,128,135),(1256976,129,139),(1256987,134,145),(1256999,137,156),(1257011,141,166),(1257024,145,179),(1257036,148,190),(1257049,151,202),(1257062,156,218),(1257075,159,228),(1257089,163,237),(1257103,168,246),(1257116,170,251),(1257130,172,253),(1257144,174,255)]),
 ("right_curly_brace",[(1203145,125,134),(1203168,123,132),(1203188,122,130),(1203238,125,132),(1203246,127,132),(1203254,129,133),(1203262,132,133),(1203270,134,134),(1203278,137,134),(1203286,139,136),(1203294,142,136),(1203303,144,138),(1203311,145,140),(1203320,147,142),(1203328,148,144),(1203337,148,148),(1203346,149,150),(1203355,149,153),(1203365,149,157),(1203374,149,160),(1203384,148,162),(1203394,147,165),(1203404,144,168),(1203414,142,169),(1203425,140,170),(1203436,137,172),(1203447,134,173),(1203458,132,174),(1203470,129,174),(1203481,127,177),(1203496,125,179),(1203512,124,181),(1203524,124,184),(1203537,124,186),(1203550,127,188),(1203565,131,190),(1203578,136,191),(1203592,140,191),(1203593,143,190),(1203607,145,189),(1203608,148,187),(1203625,153,187),(1203639,155,185),(1203654,157,186),(1203701,155,185),(1203717,153,186),(1203733,148,188),(1203749,144,191),(1203766,139,193),(1203782,137,194),(1203799,134,195),(1203816,132,196),(1203853,132,198),(1203877,134,199),(1203894,136,200),(1203912,138,203),(1203913,139,205),(1203932,141,206),(1203951,145,212),(1203970,149,216),(1203990,152,224),(1204010,152,231),(1204030,151,236),(1204050,145,245),(1204070,139,248),(1204091,130,251),(1204111,124,251),(1204134,116,252),(1204155,112,253),(1204176,110,254)]),
 ("right_sq_bracket",[(802001,118,126),(802028,120,127),(802040,123,127),(802053,126,129),(802060,128,129),(802068,131,129),(802077,134,129),(802085,137,129),(802093,139,129),(802105,142,130),(802113,144,130),(802125,146,130),(802137,149,131),(802150,151,130),(802162,154,129),(802175,156,128),(802204,158,129),(802220,158,132),(802229,157,134),(802238,156,137),(802248,155,141),(802258,155,145),(802268,154,151),(802279,153,155),(802290,153,162),(802301,153,166),(802311,154,171),(802313,154,174),(802325,155,182),(802337,156,189),(802349,157,196),(802361,157,204),(802374,158,210),(802386,158,215),(802399,158,218),(802412,157,221),(802413,157,224),(802436,156,226),(802488,155,229),(802508,153,231),(802536,151,232),(802550,148,231),(802565,143,231),(802579,140,231),(802594,133,230),(802609,128,229),(802624,124,229),(802641,121,228),(802656,119,228)]),
 ("caret",[(1257506,78,248),(1257513,79,246),(1257521,80,244),(1257530,82,241),(1257538,83,238),(1257544,84,236),(1257550,85,233),(1257551,86,231),(1257559,89,226),(1257566,92,221),(1257573,94,219),(1257574,95,216),(1257584,98,210),(1257592,101,203),(1257600,103,198),(1257609,106,191),(1257617,109,184),(1257618,110,181),(1257629,112,175),(1257638,116,164),(1257649,118,158),(1257659,121,152),(1257668,123,147),(1257678,124,143),(1257688,126,139),(1257697,127,137),(1257708,129,134),(1257718,130,132),(1257738,132,131),(1257754,134,132),(1257766,136,133),(1257777,138,135),(1257778,139,137),(1257793,142,145),(1257805,146,154),(1257806,147,158),(1257821,151,172),(1257822,152,177),(1257838,158,197),(1257852,161,211),(1257865,165,224),(1257866,167,228),(1257883,171,242),(1257898,174,250),(1257899,175,252),(1257916,177,260),(1257930,179,262)]),
 ("x",[(1121187,82,147),(1121206,83,149),(1121218,85,152),(1121226,87,154),(1121234,89,158),(1121242,91,161),(1121248,93,163),(1121255,96,168),(1121262,97,170),(1121263,99,172),(1121272,102,177),(1121279,106,183),(1121286,110,190),(1121294,112,193),(1121295,114,196),(1121304,119,202),(1121313,123,207),(1121321,125,209),(1121322,128,212),(1121333,132,216),(1121343,135,220),(1121352,139,222),(1121361,141,224),(1121362,144,226),(1121374,146,227),(1121384,147,230),(1121394,150,231),(1121406,152,233),(1121430,154,233),(1121451,154,229),(1121462,154,227),(1121463,154,225),(1121476,152,217),(1121488,150,209),(1121501,148,200),(1121514,147,188),(1121527,148,175),(1121541,149,165),(1121554,151,156),(1121555,152,154),(1121572,154,143),(1121586,155,138),(1121599,157,132),(1121613,157,130),(1121627,157,128),(1121641,155,130),(1121657,150,134),(1121672,144,141),(1121687,139,148),(1121688,137,150),(1121706,128,159),(1121722,119,171),(1121723,117,174),(1121742,109,186),(1121743,107,189),(1121763,100,200),(1121764,99,203),(1121783,94,212),(1121801,91,217),(1121818,89,220),(1121874,91,221),(1121895,93,222)]),
 ("star",[(665902,83,251),(665935,84,248),(665947,85,245),(665955,86,242),(665962,87,240),(665970,88,237),(665978,89,235),(665986,90,232),(665994,92,229),(666002,93,225),(666010,95,222),(666018,96,219),(666026,96,216),(666035,98,211),(666044,99,207),(666053,100,202),(666062,101,197),(666063,102,195),(666074,104,187),(666084,105,182),(666094,106,179),(666095,106,176),(666105,107,173),(666107,108,171),(666118,109,166),(666129,110,163),(666140,112,155),(666152,113,150),(666163,114,144),(666175,114,139),(666187,114,134),(666199,115,129),(666212,115,126),(666225,116,123),(666237,118,121),(666278,120,123),(666290,120,126),(666303,122,129),(666316,124,134),(666330,127,141),(666345,130,148),(666359,134,159),(666375,138,170),(666391,143,180),(666406,147,192),(666422,150,201),(666423,151,203),(666441,156,215),(666457,159,225),(666459,160,227),(666476,162,234),(666495,166,244),(666512,168,249),(666528,169,252),(666586,167,252),(666603,164,250),(666621,159,248),(666639,154,245),(666657,144,242),(666675,133,237),(666694,121,230),(666714,112,224),(666715,109,223),(666736,96,217),(666756,85,210),(666777,74,203),(666797,65,195),(666799,63,194),(666821,58,190),(666841,50,187),(666862,46,185),(666882,44,182),(666906,43,180),(666927,45,178),(666948,52,178),(666970,63,178),(666993,75,175),(667015,89,174),(667038,104,172),(667061,118,171),(667063,121,171),(667087,137,170),(667112,152,169),(667136,165,169),(667160,175,168),(667183,184,168),(667207,191,168),(667231,196,170),(667255,200,171),(667278,201,173),(667306,199,173),(667331,195,177),(667356,187,181),(667382,176,186),(667383,174,187),(667415,157,197),(667443,142,207),(667470,128,216),(667471,126,217),(667501,114,224),(667528,101,233),(667555,93,238),(667583,86,244),(667611,80,250),(667638,78,251),(667679,79,249)]),
 ("question_mark",[(758943,112,146),(759010,111,144),(759027,111,142),(759039,111,140),(759056,112,137),(759068,113,135),(759076,113,133),(759088,115,131),(759104,117,129),(759116,120,128),(759129,122,126),(759141,125,125),(759153,127,123),(759165,130,123),(759177,132,123),(759190,135,124),(759203,137,125),(759216,140,127),(759227,142,128),(759240,144,131),(759252,147,132),(759262,148,135),(759272,150,137),(759282,151,139),(759293,151,141),(759305,152,144),(759318,153,146),(759329,154,149),(759343,154,153),(759354,154,157),(759367,154,159),(759379,153,161),(759380,153,164),(759393,152,166),(759406,151,168),(759420,149,170),(759433,147,172),(759446,144,173),(759459,141,173),(759473,138,172),(759487,134,172),(759501,131,172),(759516,129,173),(759530,126,174),(759545,122,176),(759560,120,178),(759575,119,181),(759591,118,183),(759606,118,186),(759622,118,190),(759638,118,193),(759654,119,197),(759671,120,199),(759687,120,202),(759689,120,204),(759724,121,206)]),
 ("v",[(1025521,81,155),(1025562,81,157),(1025571,82,159),(1025583,84,163),(1025591,86,165),(1025599,87,169),(1025607,89,171),(1025615,91,174),(1025623,93,177),(1025631,95,181),(1025639,97,184),(1025647,100,187),(1025656,102,191),(1025665,103,194),(1025674,107,200),(1025676,108,202),(1025686,111,206),(1025695,113,209),(1025704,116,215),(1025714,118,219),(1025724,120,222),(1025734,123,226),(1025744,124,230),(1025755,126,232),(1025766,126,234),(1025776,127,236),(1025787,129,239),(1025799,129,241),(1025892,131,239),(1025904,131,237),(1025916,130,234),(1025928,130,230),(1025941,130,223),(1025954,131,215),(1025968,132,206),(1025969,133,202),(1025984,135,191),(1025998,138,182),(1026012,142,173),(1026029,145,167),(1026043,147,162),(1026057,148,157),(1026072,150,152),(1026092,151,150),(1026132,149,148)]),
 ("left_sq_bracket",[(980508,139,119),(980528,136,119),(980544,134,119),(980556,132,119),(980569,130,119),(980581,127,118),(980593,124,118),(980605,122,117),(980617,119,116),(980629,117,115),(980641,114,117),(980653,112,116),(980665,110,115),(980681,107,116),(980701,105,117),(980722,103,118),(980747,102,120),(980771,99,121),(980783,98,123),(980795,98,126),(980805,97,128),(980815,97,131),(980825,97,134),(980836,97,140),(980847,97,145),(980848,97,148),(980859,97,152),(980860,97,155),(980873,97,163),(980884,97,172),(980896,97,177),(980909,97,185),(980922,98,192),(980934,99,199),(980936,99,201),(980949,99,208),(980962,100,215),(980976,102,221),(980990,103,226),(981005,104,232),(981019,105,239),(981034,105,242),(981048,106,247),(981083,106,249),(981103,106,251),(981188,109,249),(981204,111,249),(981219,113,249),(981235,116,249),(981237,118,249),(981253,123,249),(981269,128,249),(981286,133,247),(981303,138,247),(981322,142,247),(981339,144,247),(981356,147,247),(981358,149,246),(981378,151,246)]),
 ("v",[(1022219,92,155),(1022269,94,158),(1022277,95,160),(1022286,96,163),(1022292,97,165),(1022299,99,169),(1022305,100,171),(1022312,101,177),(1022319,103,183),(1022326,105,189),(1022334,106,193),(1022335,107,195),(1022343,108,197),(1022351,110,201),(1022359,113,207),(1022368,114,210),(1022377,116,213),(1022378,118,216),(1022388,119,218),(1022397,121,221),(1022399,122,224),(1022409,124,227),(1022419,125,229),(1022429,126,231),(1022440,127,234),(1022451,127,236),(1022524,129,234),(1022536,130,232),(1022548,130,229),(1022549,131,226),(1022564,132,218),(1022578,133,210),(1022591,135,201),(1022592,135,197),(1022606,137,186),(1022621,140,171),(1022635,142,163),(1022636,143,161),(1022653,145,157),(1022666,147,154),(1022713,148,152)]),
 ("delete_mark",[(1438941,115,130),(1438949,116,133),(1438957,118,137),(1438963,120,140),(1438970,123,146),(1438977,125,150),(1438978,127,152),(1438987,132,159),(1438995,137,168),(1439003,142,177),(1439011,147,183),(1439019,152,190),(1439028,157,196),(1439036,162,201),(1439038,164,204),(1439048,167,209),(1439049,168,212),(1439060,169,214),(1439062,171,216),(1439073,172,218),(1439105,170,220),(1439114,168,218),(1439124,165,218),(1439134,159,215),(1439145,154,215),(1439146,151,214),(1439160,143,214),(1439171,134,214),(1439182,125,214),(1439194,116,216),(1439205,109,217),(1439217,105,218),(1439229,101,220),(1439230,98,221),(1439244,95,222),(1439257,93,222),(1439289,93,220),(1439301,95,217),(1439314,97,215),(1439328,103,208),(1439342,111,201),(1439356,117,194),(1439370,127,182),(1439385,137,170),(1439400,144,159),(1439415,152,150),(1439430,158,143),(1439445,162,138),(1439460,165,135),(1439476,167,133),(1439492,170,129)]),
 ("rectangle",[(902582,78,149),(902631,78,153),(902639,78,157),(902647,78,160),(902653,79,162),(902659,79,164),(902660,79,167),(902666,79,169),(902673,79,173),(902681,79,178),(902688,79,183),(902695,80,189),(902703,80,193),(902711,80,198),(902719,80,202),(902728,81,208),(902736,81,210),(902745,81,216),(902755,82,222),(902764,82,224),(902774,82,227),(902784,83,229),(902843,83,231),(902887,85,230),(902904,88,232),(902915,90,233),(902926,92,232),(902938,94,233),(902950,99,232),(902962,102,233),(902963,106,233),(902976,109,234),(902990,117,235),(903003,123,236),(903004,126,236),(903019,135,237),(903034,142,238),(903036,145,238),(903050,152,238),(903052,154,239),(903067,165,238),(903083,174,237),(903099,179,236),(903115,186,235),(903131,191,235),(903147,195,233),(903163,197,233),(903179,200,233),(903196,201,235),(903272,201,233),(903288,199,231),(903305,198,226),(903323,198,220),(903341,196,207),(903361,195,195),(903380,195,181),(903398,195,173),(903417,195,163),(903437,194,155),(903457,192,145),(903475,192,143),(903496,192,138),(903515,191,135),(903516,191,133),(903536,191,130),(903556,190,128),(903576,188,129),(903595,186,129),(903616,181,132),(903638,173,131),(903660,162,131),(903683,151,132),(903685,149,132),(903707,138,132),(903709,136,132),(903736,122,131),(903737,120,131),(903760,109,130),(903761,107,130),(903793,90,132),(903816,81,133),(903840,76,133)]),
 ("circle",[(1155361,144,142),(1155366,142,140),(1155371,141,138),(1155377,137,135),(1155382,135,134),(1155388,129,132),(1155394,127,131),(1155401,121,131),(1155407,118,130),(1155414,112,132),(1155421,107,136),(1155428,104,138),(1155429,102,141),(1155438,98,147),(1155446,94,153),(1155454,91,160),(1155463,89,167),(1155465,88,170),(1155475,86,179),(1155476,86,183),(1155488,85,190),(1155489,86,194),(1155499,87,200),(1155500,88,203),(1155512,91,212),(1155522,95,217),(1155533,101,221),(1155543,106,224),(1155544,109,225),(1155558,117,230),(1155570,125,230),(1155581,134,230),(1155593,142,227),(1155607,150,221),(1155619,158,215),(1155620,160,213),(1155635,167,205),(1155636,169,202),(1155652,173,194),(1155653,174,191),(1155669,174,176),(1155683,171,164),(1155684,169,160),(1155701,162,149),(1155715,156,145),(1155716,154,143),(1155733,146,143),(1155748,143,145)]),
 ("right_curly_brace",[(1211448,126,138),(1211472,128,136),(1211480,130,134),(1211488,132,134),(1211496,135,132),(1211504,137,131),(1211512,140,131),(1211520,143,129),(1211528,145,129),(1211540,148,130),(1211552,150,130),(1211564,152,131),(1211573,154,132),(1211581,157,133),(1211589,159,134),(1211598,161,136),(1211607,163,138),(1211616,165,139),(1211625,166,141),(1211634,167,143),(1211644,168,146),(1211655,168,149),(1211665,168,154),(1211676,167,156),(1211686,166,158),(1211697,166,161),(1211720,163,163),(1211733,161,163),(1211745,158,164),(1211757,156,165),(1211769,153,164),(1211782,149,166),(1211794,146,167),(1211807,144,169),(1211820,142,169),(1211833,140,171),(1211846,138,173),(1211861,136,174),(1211875,134,177),(1211894,132,179),(1211908,132,181),(1211930,133,183),(1211946,136,184),(1211961,138,185),(1211977,143,186),(1211992,148,187),(1212008,153,188),(1212025,159,187),(1212041,164,186),(1212057,166,187),(1212074,168,187),(1212131,165,187),(1212149,160,186),(1212166,155,188),(1212184,149,189),(1212203,143,191),(1212204,140,191),(1212222,136,192),(1212243,131,193),(1212262,129,193),(1212281,129,195),(1212305,130,197),(1212325,133,199),(1212326,135,201),(1212346,138,205),(1212366,145,208),(1212386,150,211),(1212408,157,217),(1212429,160,222),(1212430,161,224),(1212451,162,228),(1212472,161,233),(1212494,156,236),(1212519,148,240),(1212543,138,243),(1212568,125,245),(1212591,118,247),(1212614,116,247)]),
 ("question_mark",[(753107,111,149),(753131,110,147),(753143,110,144),(753167,109,142),(753182,109,139),(753190,108,137),(753198,108,135),(753210,109,132),(753226,110,130),(753234,111,127),(753250,113,125),(753266,115,123),(753279,118,121),(753291,121,120),(753300,123,119),(753308,125,118),(753317,128,117),(753327,130,117),(753337,133,117),(753348,137,117),(753357,139,118),(753367,142,119),(753377,144,120),(753388,146,121),(753400,148,122),(753413,151,124),(753426,155,129),(753437,157,132),(753448,159,134),(753461,161,139),(753475,163,141),(753488,165,147),(753500,166,149),(753513,166,152),(753526,166,154),(753541,165,159),(753554,164,161),(753569,163,164),(753584,158,167),(753598,156,169),(753614,151,170),(753630,146,171),(753646,141,173),(753663,136,176),(753682,132,177),(753684,130,177),(753700,128,178),(753714,126,178),(753729,123,180),(753744,121,181),(753763,121,184),(753783,121,187),(753799,121,189),(753800,121,191),(753817,121,196),(753833,121,202),(753850,122,206),(753866,123,209),(753867,123,211),(753887,124,213),(753904,124,215),(753921,123,217)]),
 ("question_mark",[(755937,110,147),(755958,108,145),(755974,106,143),(755985,105,140),(756001,104,137),(756013,103,134),(756029,103,132),(756038,103,129),(756053,103,127),(756069,105,124),(756081,107,122),(756093,109,120),(756105,111,119),(756117,114,118),(756130,116,118),(756143,119,117),(756156,121,119),(756168,123,120),(756176,125,121),(756188,128,120),(756198,130,121),(756208,132,123),(756221,134,123),(756232,137,123),(756245,139,124),(756261,141,127),(756273,143,128),(756285,145,130),(756297,147,132),(756313,149,134),(756325,150,136),(756338,151,138),(756349,151,141),(756362,151,143),(756374,151,145),(756387,150,147),(756400,150,149),(756401,149,152),(756418,148,154),(756432,147,156),(756446,144,161),(756460,143,163),(756474,141,165),(756489,138,166),(756490,136,167),(756507,131,169),(756522,127,170),(756537,124,171),(756553,121,172),(756570,118,176),(756585,115,177),(756602,113,181),(756618,112,184),(756635,112,186),(756652,112,190),(756669,112,196),(756686,113,201),(756704,113,207),(756723,114,212),(756741,115,214),(756788,115,216),(756833,115,214)]),
 ("v",[(1030316,89,164),(1030341,90,162),(1030377,92,162),(1030402,94,164),(1030414,95,166),(1030421,96,169),(1030429,97,171),(1030437,99,175),(1030445,101,178),(1030453,103,182),(1030461,106,189),(1030469,108,194),(1030478,111,199),(1030487,114,204),(1030496,117,209),(1030505,119,214),(1030515,122,218),(1030524,124,222),(1030526,126,225),(1030536,128,228),(1030547,130,229),(1030558,133,233),(1030568,134,236),(1030581,136,239),(1030601,138,240),(1030618,139,242),(1030642,140,244),(1030683,142,242),(1030699,142,240),(1030715,142,237),(1030727,143,235),(1030740,143,233),(1030752,145,229),(1030766,146,226),(1030780,148,217),(1030794,149,208),(1030796,149,205),(1030810,151,196),(1030812,151,193),(1030828,153,182),(1030844,155,172),(1030860,157,165),(1030876,159,160),(1030892,162,155),(1030908,164,150),(1030923,165,148),(1030999,166,146)]),
 ("caret",[(1261521,79,245),(1261537,79,242),(1261549,79,239),(1261561,80,237),(1261569,80,234),(1261575,81,232),(1261582,82,230),(1261589,84,224),(1261597,86,220),(1261604,86,218),(1261606,87,216),(1261616,88,213),(1261624,90,207),(1261632,91,202),(1261634,92,200),(1261645,93,194),(1261646,94,192),(1261656,96,189),(1261658,97,186),(1261670,100,179),(1261680,102,173),(1261690,105,165),(1261701,107,160),(1261702,109,158),(1261716,112,151),(1261727,115,144),(1261738,117,139),(1261749,119,136),(1261761,119,134),(1261772,120,132),(1261774,121,129),(1261788,122,127),(1261805,124,125),(1261825,126,124),(1261842,129,125),(1261855,131,127),(1261868,132,130),(1261881,136,139),(1261895,141,154),(1261909,145,166),(1261924,151,182),(1261938,156,193),(1261940,157,196),(1261959,161,209),(1261960,162,211),(1261980,167,223),(1261995,169,229),(1261996,170,231),(1262014,173,237),(1262030,176,242),(1262032,177,244),(1262050,179,250),(1262070,181,255),(1262086,182,257)]),
 ("pigtail",[(1070730,75,218),(1070750,78,219),(1070762,80,220),(1070771,83,221),(1070778,86,221),(1070786,89,219),(1070795,92,218),(1070802,94,218),(1070810,98,216),(1070817,102,215),(1070825,105,213),(1070833,107,212),(1070842,111,211),(1070851,117,207),(1070852,119,206),(1070862,123,203),(1070864,125,201),(1070874,130,198),(1070876,132,196),(1070887,138,190),(1070899,142,185),(1070900,144,183),(1070912,149,176),(1070924,153,168),(1070937,155,160),(1070949,156,152),(1070962,156,144),(1070975,155,139),(1070976,154,137),(1070990,152,132),(1071004,148,129),(1071018,142,128),(1071031,135,128),(1071044,132,130),(1071059,126,136),(1071074,121,141),(1071089,117,149),(1071104,113,155),(1071119,112,161),(1071135,111,169),(1071136,111,171),(1071152,113,179),(1071169,115,188),(1071185,120,196),(1071202,125,203),(1071219,134,209),(1071236,141,212),(1071253,149,213),(1071271,157,214),(1071290,164,212),(1071311,172,209),(1071330,177,206)]),
 ("pigtail",[(1073134,86,219),(1073145,88,220),(1073154,91,219),(1073162,93,220),(1073171,97,219),(1073177,99,218),(1073183,103,216),(1073190,105,215),(1073197,109,214),(1073204,113,211),(1073211,117,209),(1073219,119,209),(1073226,122,205),(1073234,126,203),(1073242,129,199),(1073251,132,197),(1073259,136,191),(1073267,139,188),(1073276,141,184),(1073285,143,180),(1073294,144,178),(1073296,145,174),(1073307,147,170),(1073317,148,167),(1073327,149,161),(1073338,148,158),(1073349,147,153),(1073360,146,148),(1073372,144,146),(1073385,141,144),(1073402,139,142),(1073415,137,141),(1073427,135,141),(1073439,132,141),(1073451,130,143),(1073464,127,145),(1073476,124,149),(1073489,122,155),(1073503,121,160),(1073519,120,168),(1073532,120,173),(1073546,121,178),(1073561,123,183),(1073575,125,188),(1073590,128,194),(1073604,132,200),(1073619,135,202),(1073621,137,204),(1073639,143,208),(1073654,150,211),(1073670,156,212),(1073686,163,212),(1073703,170,212),(1073723,176,211),(1073740,180,210),(1073756,183,208),(1073773,185,206)]),
 ("left_curly_brace",[(1400764,135,125),(1400776,133,125),(1400784,130,127),(1400793,127,128),(1400800,123,130),(1400808,120,133),(1400817,117,137),(1400825,114,140),(1400833,113,142),(1400841,111,145),(1400849,110,148),(1400870,109,150),(1400890,112,152),(1400898,114,153),(1400907,116,154),(1400916,119,156),(1400925,122,156),(1400934,125,156),(1400943,128,158),(1400953,130,161),(1400962,133,164),(1400972,135,165),(1400987,137,168),(1400998,136,170),(1401011,134,173),(1401022,131,175),(1401033,129,175),(1401044,124,178),(1401056,122,178),(1401067,120,178),(1401079,118,179),(1401100,116,178),(1401141,118,179),(1401154,120,180),(1401167,125,181),(1401180,127,181),(1401193,130,181),(1401206,132,183),(1401220,133,186),(1401222,132,189),(1401236,131,194),(1401252,127,204),(1401266,123,213),(1401280,120,219),(1401281,120,221),(1401298,118,226),(1401312,116,229),(1401326,115,231),(1401344,115,233),(1401359,117,234),(1401374,123,236),(1401390,131,236),(1401405,138,239)]),
 ("caret",[(1263322,74,246),(1263338,75,243),(1263346,76,241),(1263354,78,238),(1263359,79,236),(1263367,82,233),(1263374,84,228),(1263381,86,226),(1263387,88,222),(1263395,92,217),(1263402,93,214),(1263403,95,211),(1263412,98,205),(1263420,101,199),(1263428,104,192),(1263436,106,184),(1263444,108,177),(1263452,109,170),(1263461,111,163),(1263470,112,157),(1263471,112,154),(1263482,115,148),(1263491,116,144),(1263501,118,140),(1263511,120,134),(1263521,122,129),(1263531,124,127),(1263541,125,125),(1263542,126,123),(1263566,128,122),(1263586,130,123),(1263598,131,125),(1263599,131,128),(1263613,133,136),(1263626,136,146),(1263627,137,149),(1263642,142,163),(1263657,146,174),(1263659,148,178),(1263675,154,195),(1263689,158,206),(1263691,160,209),(1263707,166,224),(1263723,172,237),(1263738,176,248),(1263754,179,253),(1263755,179,255),(1263771,180,258),(1263787,181,260)]),
 ("left_sq_bracket",[(972514,138,124),(972552,136,123),(972573,134,125),(972585,131,126),(972598,128,126),(972606,126,128),(972614,123,129),(972622,121,129),(972630,118,128),(972638,116,128),(972646,113,128),(972654,110,128),(972662,107,127),(972671,104,127),(972679,102,126),(972687,99,125),(972695,97,124),(972704,94,124),(972713,92,125),(972723,90,124),(972733,88,125),(972747,86,126),(972782,84,126),(972806,83,129),(972822,83,131),(972834,83,133),(972850,83,136),(972862,84,139),(972873,84,141),(972885,85,144),(972897,86,148),(972909,86,151),(972911,87,154),(972924,87,159),(972937,88,162),(972950,89,169),(972963,89,175),(972976,90,178),(972990,90,182),(973004,90,189),(973018,91,193),(973033,91,198),(973047,92,205),(973064,92,208),(973065,92,211),(973080,92,217),(973096,93,222),(973111,93,227),(973127,93,229),(973143,93,235),(973159,94,241),(973176,94,246),(973193,94,248),(973215,94,251),(973231,94,253),(973252,94,255),(973337,96,253),(973357,99,252),(973374,101,252),(973392,104,251),(973394,106,251),(973413,111,251),(973432,116,252),(973451,124,252),(973471,129,252),(973490,133,252),(973510,136,252),(973529,138,252),(973549,140,251)]),
 ("left_sq_bracket",[(970990,141,126),(971034,139,125),(971054,137,125),(971066,135,125),(971078,132,124),(971086,130,124),(971094,128,124),(971102,126,124),(971110,124,124),(971118,122,124),(971130,119,123),(971142,117,123),(971155,115,123),(971167,112,124),(971179,110,123),(971196,108,124),(971212,106,124),(971228,104,124),(971268,102,125),(971284,102,128),(971294,102,130),(971304,101,132),(971317,101,136),(971328,101,139),(971339,101,142),(971351,101,148),(971364,102,156),(971366,102,158),(971378,102,163),(971392,103,168),(971405,103,174),(971406,103,176),(971419,103,183),(971433,103,191),(971448,104,197),(971462,104,204),(971477,105,212),(971492,105,217),(971507,105,223),(971522,106,228),(971538,106,230),(971554,106,235),(971570,107,239),(971586,107,241),(971603,107,243),(971622,107,245),(971666,107,247),(971742,109,246),(971758,111,247),(971774,113,247),(971791,118,247),(971810,122,248),(971828,127,249),(971848,135,249),(971867,139,248),(971885,142,248),(971902,144,247),(971922,146,247),(971957,146,249)]),
 ("right_curly_brace",[(1208182,134,141),(1208230,133,139),(1208262,135,138),(1208279,137,138),(1208287,139,138),(1208294,141,138),(1208302,144,138),(1208310,146,140),(1208318,149,141),(1208326,152,142),(1208334,155,144),(1208342,157,146),(1208350,159,148),(1208359,161,150),(1208366,162,153),(1208374,162,155),(1208382,163,158),(1208391,162,160),(1208399,161,163),(1208408,160,165),(1208417,158,167),(1208426,156,170),(1208436,154,172),(1208445,151,173),(1208446,149,174),(1208457,146,175),(1208468,143,177),(1208479,138,180),(1208489,136,181),(1208500,133,183),(1208511,129,187),(1208523,127,190),(1208535,125,193),(1208547,125,196),(1208560,126,198),(1208572,128,200),(1208584,130,201),(1208597,134,201),(1208610,137,201),(1208623,143,201),(1208636,148,199),(1208649,151,199),(1208664,156,199),(1208677,158,198),(1208691,161,197),(1208705,163,196),(1208739,160,196),(1208754,158,197),(1208768,153,197),(1208784,148,198),(1208799,143,200),(1208815,139,201),(1208830,137,203),(1208831,134,204),(1208848,132,205),(1208912,133,208),(1208929,136,208),(1208945,140,210),(1208962,145,212),(1208979,151,214),(1208981,154,216),(1208999,159,220),(1209017,164,223),(1209035,166,230),(1209053,166,236),(1209072,163,240),(1209091,159,244),(1209110,147,246),(1209131,135,248),(1209151,125,251),(1209171,119,252),(1209172,117,253),(1209192,115,254),(1209215,113,255)]),
 ("right_sq_bracket",[(799027,121,127),(799059,123,127),(799072,126,126),(799079,129,126),(799088,131,126),(799096,134,126),(799104,137,125),(799112,139,125),(799120,141,125),(799128,144,125),(799136,146,124),(799145,148,124),(799154,150,124),(799163,152,125),(799174,155,124),(799186,157,125),(799202,159,127),(799218,160,129),(799227,160,134),(799236,160,136),(799247,160,141),(799258,160,148),(799269,160,153),(799270,160,155),(799281,161,160),(799283,161,163),(799295,161,171),(799309,161,180),(799323,161,191),(799337,162,200),(799351,163,211),(799365,163,218),(799367,163,221),(799380,163,227),(799395,162,232),(799410,161,238),(799422,159,240),(799442,157,241),(799454,155,239),(799468,152,238),(799483,147,237),(799501,141,236),(799517,135,235),(799534,130,235),(799552,124,236),(799569,120,237),(799571,117,238),(799587,115,238),(799603,110,238)]),
 ("circle",[(1156082,135,138),(1156089,132,135),(1156097,129,131),(1156105,125,127),(1156110,123,127),(1156116,121,127),(1156117,118,128),(1156125,114,129),(1156132,111,131),(1156139,106,134),(1156146,101,138),(1156153,97,144),(1156161,94,148),(1156162,92,152),(1156172,89,158),(1156180,86,165),(1156188,83,172),(1156196,81,180),(1156197,80,183),(1156208,79,191),(1156209,79,194),(1156221,79,200),(1156222,79,203),(1156232,80,209),(1156234,81,212),(1156246,85,220),(1156256,86,222),(1156257,89,225),(1156270,95,231),(1156281,102,235),(1156293,108,235),(1156294,111,236),(1156310,121,237),(1156311,124,237),(1156325,136,235),(1156326,139,234),(1156341,150,230),(1156354,162,224),(1156367,171,217),(1156381,178,210),(1156395,186,198),(1156411,188,184),(1156427,186,171),(1156442,180,161),(1156443,177,158),(1156459,167,149),(1156475,155,143),(1156490,147,142),(1156491,144,142),(1156508,133,145),(1156526,131,146)]),
 ("right_sq_bracket",[(803196,118,126),(803224,120,127),(803236,123,127),(803244,125,127),(803253,129,127),(803260,132,128),(803268,135,128),(803276,138,129),(803284,140,128),(803292,143,128),(803300,145,128),(803308,147,129),(803316,149,130),(803329,152,131),(803341,154,132),(803362,156,131),(803391,157,134),(803403,157,136),(803419,156,138),(803428,156,140),(803437,156,143),(803447,156,145),(803448,156,148),(803458,156,152),(803460,156,154),(803470,156,158),(803481,155,166),(803492,155,174),(803502,155,178),(803504,155,181),(803516,155,190),(803528,154,198),(803540,155,203),(803551,155,208),(803563,156,214),(803575,156,217),(803588,156,220),(803600,156,224),(803616,156,227),(803632,156,230),(803668,153,229),(803681,151,229),(803695,149,228),(803709,145,229),(803723,142,229),(803737,136,228),(803751,130,227),(803766,124,229),(803784,118,229),(803799,112,230),(803815,107,230),(803830,105,230)]),
 ("caret",[(1259867,81,248),(1259876,82,245),(1259888,85,244),(1259893,86,241),(1259900,87,237),(1259908,89,234),(1259914,91,232),(1259920,94,228),(1259927,95,226),(1259934,98,222),(1259941,101,216),(1259947,103,214),(1259949,104,211),(1259958,107,205),(1259966,109,198),(1259974,112,192),(1259982,113,185),(1259991,114,178),(1260000,115,171),(1260001,116,169),(1260012,118,159),(1260021,119,155),(1260031,120,149),(1260032,121,147),(1260045,122,142),(1260055,124,139),(1260056,125,136),(1260069,126,134),(1260080,129,129),(1260109,131,128),(1260133,132,130),(1260144,133,132),(1260146,134,135),(1260160,136,143),(1260172,139,152),(1260173,140,156),(1260188,144,167),(1260190,146,170),(1260205,150,180),(1260206,151,183),(1260224,155,196),(1260238,159,211),(1260253,162,221),(1260254,163,225),(1260270,168,236),(1260285,172,242),(1260300,173,244),(1260301,175,246),(1260322,177,247),(1260346,178,244)]),
 ("triangle",[(1302122,142,125),(1302151,140,127),(1302159,139,130),(1302167,137,133),(1302175,135,137),(1302181,134,139),(1302188,132,143),(1302194,131,145),(1302195,129,148),(1302204,126,152),(1302212,122,158),(1302219,120,160),(1302220,118,163),(1302229,113,170),(1302237,108,175),(1302245,103,181),(1302254,98,186),(1302262,93,192),(1302263,90,194),(1302274,86,199),(1302283,80,205),(1302293,76,208),(1302303,73,213),(1302313,70,216),(1302322,68,219),(1302323,66,221),(1302335,63,224),(1302351,61,226),(1302379,60,229),(1302419,62,226),(1302431,64,226),(1302442,66,225),(1302454,69,225),(1302466,75,225),(1302467,77,225),(1302481,83,225),(1302494,90,223),(1302507,99,222),(1302508,101,222),(1302524,113,220),(1302538,122,220),(1302552,133,219),(1302566,142,218),(1302568,144,217),(1302584,155,217),(1302599,166,216),(1302614,173,216),(1302615,176,216),(1302632,185,215),(1302647,194,218),(1302664,199,218),(1302680,205,220),(1302696,210,220),(1302723,212,220),(1302739,211,217),(1302756,208,214),(1302772,206,211),(1302789,201,208),(1302806,195,204),(1302807,192,201),(1302827,186,194),(1302845,180,187),(1302864,172,175),(1302882,169,167),(1302901,162,154),(1302920,156,144),(1302940,151,136),(1302959,147,131),(1302960,145,129),(1302981,143,128)]),
 ("left_sq_bracket",[(975830,141,122),(975850,139,121),(975862,137,120),(975874,135,120),(975886,133,119),(975899,130,119),(975907,128,120),(975915,125,119),(975923,123,119),(975931,121,118),(975939,119,118),(975947,116,117),(975959,114,116),(975972,111,115),(975980,109,115),(975989,107,115),(976002,105,114),(976013,102,114),(976025,100,113),(976041,97,113),(976061,95,114),(976081,93,116),(976097,92,118),(976110,92,121),(976121,91,123),(976132,91,127),(976143,91,131),(976155,91,137),(976167,91,141),(976178,91,148),(976191,90,153),(976203,90,161),(976215,91,167),(976228,91,170),(976241,91,179),(976254,92,185),(976255,92,187),(976269,93,192),(976282,94,199),(976296,94,203),(976311,94,208),(976325,95,213),(976340,96,219),(976354,96,221),(976369,97,227),(976384,97,230),(976399,97,235),(976415,97,238),(976443,97,240),(976503,98,242),(976588,100,242),(976608,102,242),(976626,104,240),(976643,106,239),(976660,111,240),(976661,113,241),(976680,118,241),(976698,123,241),(976716,127,241),(976717,130,241),(976735,132,242),(976753,134,241),(976771,136,241),(976790,138,241)]),
 ("check",[(851543,90,192),(851575,91,189),(851592,94,188),(851607,96,187),(851620,98,189),(851628,100,189),(851636,102,190),(851644,104,190),(851656,107,193),(851668,109,195),(851680,112,198),(851692,114,200),(851700,115,203),(851709,117,205),(851716,118,208),(851724,119,210),(851733,120,212),(851741,121,214),(851756,123,217),(851765,124,219),(851776,126,221),(851805,128,223),(851878,129,221),(851889,129,218),(851900,129,216),(851911,129,214),(851922,130,206),(851933,131,200),(851945,133,192),(851958,135,185),(851970,138,176),(851983,141,168),(851996,145,161),(852009,150,154),(852022,158,145),(852035,161,141),(852049,167,133),(852062,171,127),(852076,172,124),(852092,174,122),(852106,174,120),(852120,175,117)]),
 ("rectangle",[(909763,79,148),(909779,79,150),(909787,78,152),(909795,78,154),(909804,78,156),(909810,79,158),(909816,79,161),(909817,79,163),(909824,79,167),(909832,79,172),(909839,80,174),(909840,80,176),(909849,80,178),(909857,80,186),(909865,80,192),(909873,80,198),(909882,80,203),(909890,80,208),(909899,81,212),(909900,81,214),(909912,81,218),(909913,82,220),(909925,82,225),(909935,83,228),(909936,83,230),(909948,84,235),(909960,84,238),(909970,85,241),(909981,85,244),(909992,85,246),(909993,85,249),(910006,86,251),(910017,86,253),(910029,87,255),(910040,87,257),(910124,89,258),(910148,91,257),(910167,93,257),(910179,95,257),(910192,97,258),(910205,99,258),(910219,104,258),(910233,109,257),(910247,112,257),(910248,115,257),(910264,121,256),(910280,128,255),(910296,136,256),(910311,140,255),(910312,143,255),(910332,152,255),(910348,160,254),(910364,168,253),(910381,175,253),(910398,183,251),(910414,189,251),(910431,195,251),(910448,200,251),(910465,205,251),(910482,207,252),(910484,210,252),(910505,211,254),(910522,213,255),(910605,212,253),(910623,212,251),(910642,210,242),(910661,208,232),(910680,207,225),(910700,206,214),(910701,206,212),(910724,205,201),(910744,204,189),(910745,204,187),(910768,203,178),(910770,203,174),(910792,202,165),(910814,201,157),(910835,201,150),(910857,200,145),(910878,198,141),(910900,196,136),(910924,195,134),(910946,190,131),(910968,186,129),(910969,184,128),(910995,176,129),(911019,167,130),(911042,158,129),(911066,150,128),(911090,142,126),(911114,135,126),(911139,128,126),(911163,121,127),(911187,116,127),(911212,110,128),(911237,102,130),(911263,95,132),(911288,91,131),(911314,86,131),(911339,84,131),(911364,82,130)]),
 ("arrow",[(1358081,77,220),(1358096,80,220),(1358104,83,218),(1358109,84,216),(1358115,86,214),(1358121,90,211),(1358127,93,210),(1358128,96,208),(1358137,101,205),(1358144,104,203),(1358145,107,201),(1358155,113,198),(1358163,120,194),(1358171,125,190),(1358179,131,187),(1358187,136,185),(1358195,141,181),(1358196,144,179),(1358207,149,177),(1358208,151,175),(1358220,156,173),(1358221,158,172),(1358234,164,168),(1358244,168,165),(1358254,172,164),(1358264,178,158),(1358275,180,158),(1358285,183,156),(1358305,185,154),(1358330,184,152),(1358342,181,152),(1358353,178,152),(1358365,173,152),(1358377,170,152),(1358378,166,152),(1358391,161,153),(1358404,157,152),(1358417,152,153),(1358429,145,155),(1358443,139,157),(1358456,137,157),(1358469,135,158),(1358518,138,158),(1358532,141,157),(1358546,148,154),(1358561,156,153),(1358563,159,153),(1358580,168,150),(1358596,176,148),(1358611,182,149),(1358629,188,148),(1358630,190,148),(1358649,192,148),(1358666,193,150),(1358682,191,156),(1358699,185,163),(1358716,178,172),(1358733,173,179),(1358734,171,181),(1358755,165,191),(1358773,161,196),(1358791,156,203),(1358808,155,205)]),
 ("right_sq_bracket",[(805602,127,125),(805645,129,125),(805657,131,126),(805669,134,127),(805680,136,127),(805689,139,127),(805697,141,127),(805705,143,128),(805713,145,129),(805721,147,129),(805729,150,128),(805741,152,127),(805753,154,126),(805774,157,125),(805823,158,127),(805836,158,130),(805844,158,132),(805852,158,134),(805862,158,137),(805873,158,144),(805884,157,150),(805885,157,153),(805895,157,159),(805897,157,162),(805908,157,173),(805919,158,180),(805921,158,183),(805933,159,193),(805945,159,201),(805957,159,209),(805970,159,215),(805982,160,221),(805994,160,223),(806006,160,226),(806019,160,229),(806032,159,231),(806088,157,232),(806100,156,230),(806114,153,229),(806128,148,228),(806129,146,228),(806144,140,226),(806145,138,226),(806163,129,229),(806164,127,230),(806184,117,232),(806203,112,234),(806219,105,235),(806235,101,237)]),
 ("triangle",[(1307537,151,127),(1307558,149,128),(1307570,146,129),(1307578,144,131),(1307586,142,134),(1307592,140,136),(1307598,137,140),(1307604,136,142),(1307611,132,146),(1307618,130,148),(1307619,128,150),(1307626,126,152),(1307627,123,155),(1307635,118,160),(1307643,113,165),(1307651,108,171),(1307659,103,176),(1307668,97,182),(1307676,91,187),(1307685,86,193),(1307686,83,196),(1307696,78,201),(1307706,73,206),(1307707,71,208),(1307717,66,212),(1307727,60,217),(1307738,56,221),(1307739,54,223),(1307749,52,225),(1307750,51,227),(1307762,48,229),(1307773,45,231),(1307790,43,233),(1307818,44,235),(1307835,46,235),(1307848,48,234),(1307861,54,234),(1307875,62,232),(1307889,68,231),(1307904,77,231),(1307918,85,230),(1307920,87,230),(1307936,99,229),(1307950,108,228),(1307965,120,227),(1307979,132,226),(1307994,140,227),(1308009,152,228),(1308025,162,228),(1308044,174,227),(1308059,181,226),(1308074,184,227),(1308076,187,227),(1308093,189,227),(1308109,191,228),(1308156,192,225),(1308172,190,224),(1308189,186,217),(1308207,182,212),(1308224,175,199),(1308243,171,188),(1308261,165,175),(1308279,160,165),(1308280,159,163),(1308301,154,154),(1308319,152,149),(1308320,150,145),(1308341,147,140),(1308361,145,134)]),
 ("question_mark",[(748827,106,153),(748852,106,151),(748869,105,149),(748900,105,146),(748915,106,144),(748928,107,142),(748936,108,140),(748948,111,137),(748956,112,135),(748968,115,133),(748980,118,131),(748988,119,129),(749000,122,127),(749012,125,125),(749021,127,124),(749029,129,124),(749039,132,123),(749052,134,123),(749065,137,123),(749074,139,124),(749084,142,125),(749094,144,125),(749104,146,126),(749115,148,128),(749126,152,130),(749137,154,132),(749148,156,134),(749159,157,137),(749171,159,139),(749182,160,143),(749194,161,149),(749207,162,151),(749219,162,157),(749231,162,159),(749245,162,166),(749257,161,168),(749271,161,170),(749284,160,172),(749285,158,174),(749300,156,177),(749315,154,177),(749329,150,178),(749343,145,178),(749358,140,179),(749373,135,180),(749391,130,182),(749406,126,183),(749421,123,184),(749437,121,185),(749453,119,186),(749469,117,189),(749485,115,191),(749501,113,196),(749518,113,198),(749535,112,204),(749552,112,209),(749570,112,213),(749589,112,216),(749620,112,218),(749680,112,215)]),
 ("left_curly_brace",[(1401845,138,129),(1401858,136,128),(1401865,134,128),(1401878,131,127),(1401886,129,130),(1401894,127,132),(1401902,125,134),(1401910,123,137),(1401917,122,139),(1401924,120,141),(1401931,119,144),(1401938,118,146),(1401946,117,149),(1401954,116,151),(1401963,116,154),(1401984,118,156),(1401997,120,157),(1402005,122,157),(1402014,125,159),(1402023,127,159),(1402033,130,159),(1402043,132,159),(1402054,137,160),(1402065,139,160),(1402081,140,163),(1402093,139,165),(1402104,139,167),(1402116,137,169),(1402118,135,171),(1402131,132,173),(1402142,127,176),(1402153,123,177),(1402165,118,179),(1402177,115,181),(1402189,113,181),(1402246,116,179),(1402258,119,178),(1402272,121,177),(1402286,125,177),(1402300,128,177),(1402314,130,179),(1402329,132,180),(1402344,134,181),(1402346,136,183),(1402363,136,185),(1402378,135,190),(1402392,132,196),(1402393,131,199),(1402411,127,207),(1402426,124,213),(1402442,123,218),(1402458,123,220),(1402474,125,226),(1402490,127,227),(1402506,131,230),(1402522,136,230),(1402539,141,232),(1402557,145,231)]),
 ("caret",[(1260664,89,251),(1260671,89,249),(1260683,91,246),(1260690,92,243),(1260703,94,241),(1260710,96,238),(1260718,98,234),(1260726,100,230),(1260734,102,226),(1260741,103,223),(1260748,106,217),(1260756,109,211),(1260763,112,205),(1260771,114,199),(1260779,116,194),(1260788,118,189),(1260796,120,184),(1260805,121,178),(1260814,123,169),(1260823,124,164),(1260833,126,157),(1260843,129,148),(1260853,130,143),(1260863,133,135),(1260873,134,132),(1260883,135,130),(1260893,137,127),(1260906,139,125),(1260922,141,126),(1260937,143,128),(1260949,146,130),(1260960,147,133),(1260973,150,139),(1260985,153,144),(1260986,154,148),(1261001,159,161),(1261014,162,172),(1261028,166,185),(1261041,170,204),(1261055,174,216),(1261069,178,229),(1261070,179,234),(1261087,185,247),(1261102,189,256),(1261116,190,259),(1261117,192,261),(1261135,193,263)]),
 ("check",[(848745,89,183),(848838,92,182),(848858,94,181),(848867,95,179),(848878,98,177),(848890,100,176),(848902,102,175),(848914,104,174),(848930,106,174),(848946,109,174),(848958,111,175),(848970,113,176),(848978,114,178),(848987,115,180),(848998,117,183),(849006,118,186),(849014,119,189),(849026,121,192),(849034,122,194),(849044,123,197),(849054,124,199),(849055,124,202),(849065,126,205),(849076,127,208),(849086,128,210),(849097,130,213),(849109,132,218),(849122,134,220),(849134,135,222),(849154,135,225),(849194,135,222),(849210,134,219),(849223,133,216),(849235,133,212),(849248,133,208),(849261,133,200),(849263,133,197),(849276,133,191),(849291,136,179),(849305,139,173),(849321,142,167),(849323,145,163),(849339,150,155),(849353,152,151),(849367,157,144),(849381,162,140),(849395,165,134),(849410,169,127),(849427,172,122),(849442,175,117),(849458,176,114),(849473,177,112),(849474,178,109),(849510,178,107),(849535,178,109)]),
 ("v",[(1026683,77,148),(1026723,78,151),(1026731,79,153),(1026739,81,156),(1026747,82,159),(1026754,84,162),(1026763,85,165),(1026775,88,169),(1026782,88,171),(1026784,89,173),(1026794,91,177),(1026802,92,179),(1026804,94,184),(1026813,96,189),(1026822,98,194),(1026824,99,197),(1026835,101,201),(1026836,101,203),(1026866,108,217),(1026876,111,223),(1026887,112,227),(1026898,114,229),(1026909,116,234),(1026919,117,237),(1026947,118,239),(1026982,119,241),(1026999,119,244),(1027092,119,241),(1027104,119,238),(1027116,119,233),(1027129,120,227),(1027142,121,220),(1027156,122,214),(1027170,125,204),(1027184,129,191),(1027199,133,182),(1027213,136,175),(1027228,141,167),(1027243,145,160),(1027257,148,155),(1027272,151,152),(1027286,152,150),(1027302,154,147),(1027317,154,145)]),
 ("left_sq_bracket",[(979061,141,125),(979089,139,124),(979101,137,123),(979114,135,124),(979126,132,123),(979138,130,123),(979146,128,124),(979154,126,124),(979162,123,124),(979170,121,124),(979178,119,123),(979186,117,123),(979194,115,123),(979202,113,122),(979215,110,121),(979228,108,122),(979240,106,123),(979256,104,123),(979277,102,123),(979301,100,125),(979362,98,127),(979378,98,129),(979388,98,131),(979407,97,133),(979419,97,136),(979430,97,139),(979441,97,141),(979452,97,147),(979464,97,153),(979476,97,160),(979488,98,170),(979500,98,176),(979513,100,184),(979526,101,192),(979528,101,195),(979542,101,200),(979556,102,211),(979569,102,218),(979583,102,226),(979597,102,230),(979611,103,236),(979628,103,238),(979736,103,240),(979788,106,240),(979804,108,240),(979819,111,240),(979835,115,240),(979850,120,240),(979866,125,241),(979882,130,242),(979898,132,242),(979914,134,241),(979931,136,240)]),
 ("v",[(1021121,89,159),(1021134,89,157),(1021195,91,159),(1021202,92,162),(1021210,93,166),(1021216,93,168),(1021223,94,173),(1021229,95,176),(1021231,96,178),(1021238,96,180),(1021239,97,182),(1021246,99,188),(1021253,100,191),(1021255,101,193),(1021263,102,199),(1021271,104,203),(1021280,106,208),(1021288,107,213),(1021297,108,215),(1021306,109,221),(1021315,111,224),(1021325,112,226),(1021334,113,230),(1021335,114,233),(1021358,115,235),(1021411,116,237),(1021435,117,239),(1021479,119,240),(1021507,121,239),(1021519,123,235),(1021531,125,228),(1021544,127,223),(1021556,130,213),(1021569,132,203),(1021582,136,193),(1021584,137,190),(1021598,140,183),(1021612,144,175),(1021626,148,166),(1021640,150,163),(1021655,153,159),(1021669,154,157),(1021685,155,155)]),
 ("right_sq_bracket",[(804415,121,126),(804475,123,124),(804490,126,123),(804503,129,121),(804516,131,121),(804523,134,119),(804531,136,118),(804539,138,117),(804547,141,117),(804555,143,117),(804563,145,117),(804571,147,116),(804584,149,117),(804596,151,117),(804617,154,119),(804630,154,121),(804646,155,123),(804654,155,126),(804663,155,129),(804672,155,131),(804681,154,134),(804682,154,139),(804695,153,146),(804705,153,150),(804715,152,159),(804726,152,163),(804727,152,167),(804739,152,176),(804750,152,181),(804751,152,185),(804763,153,193),(804775,154,200),(804787,155,208),(804799,155,212),(804811,155,215),(804823,155,218),(804836,155,220),(804849,154,222),(804866,153,224),(804900,152,226),(804928,151,228),(804944,149,230),(804957,146,230),(804972,140,230),(804986,133,229),(805001,127,226),(805015,125,226),(805017,122,226),(805033,118,228),(805048,116,228)]),
 ("left_sq_bracket",[(974232,140,124),(974268,138,123),(974284,135,122),(974297,133,123),(974309,130,123),(974322,128,124),(974334,125,125),(974346,122,124),(974354,120,124),(974362,118,124),(974370,116,125),(974378,113,125),(974386,111,125),(974395,108,124),(974402,106,125),(974410,104,125),(974419,102,124),(974428,100,123),(974437,98,123),(974447,95,124),(974457,93,123),(974466,90,124),(974478,88,124),(974490,85,125),(974506,83,126),(974526,81,127),(974590,81,129),(974601,82,131),(974613,82,134),(974625,83,138),(974637,84,141),(974649,84,144),(974651,85,148),(974664,85,151),(974677,86,156),(974679,86,160),(974693,86,164),(974695,86,168),(974709,87,171),(974711,87,175),(974725,87,179),(974727,87,182),(974741,87,186),(974743,88,188),(974759,88,195),(974773,88,198),(974775,88,201),(974791,88,207),(974809,89,211),(974811,89,213),(974828,89,217),(974844,89,222),(974860,88,225),(974877,88,229),(974902,88,231),(974919,88,233),(974936,88,235),(974954,89,237),(974972,89,240),(974998,89,242),(975039,91,241),(975059,94,241),(975077,96,240),(975096,98,239),(975115,105,240),(975135,109,240),(975154,113,239),(975156,116,240),(975177,121,239),(975197,130,240),(975218,136,237),(975220,139,237),(975241,144,238),(975266,151,237),(975287,157,236),(975308,159,237)]),
 ("v",[(1027826,85,151),(1027841,85,153),(1027850,86,155),(1027858,86,158),(1027866,87,160),(1027874,89,164),(1027880,89,166),(1027887,90,169),(1027894,91,171),(1027901,93,174),(1027908,94,177),(1027915,96,180),(1027923,98,183),(1027931,99,185),(1027939,101,189),(1027948,104,195),(1027957,106,197),(1027967,110,205),(1027976,112,208),(1027985,114,212),(1027987,115,214),(1027996,116,216),(1028007,120,222),(1028017,121,225),(1028019,122,227),(1028030,124,231),(1028041,125,233),(1028053,128,240),(1028065,129,242),(1028077,130,247),(1028089,131,249),(1028111,132,252),(1028143,134,249),(1028159,134,246),(1028172,134,244),(1028186,134,238),(1028201,135,231),(1028216,136,223),(1028231,138,215),(1028246,139,203),(1028261,141,192),(1028277,143,179),(1028293,145,169),(1028310,146,163),(1028326,147,158),(1028340,148,156),(1028356,149,154),(1028384,150,151),(1028400,150,149),(1028429,150,147)]),
 ("right_curly_brace",[(1206502,126,138),(1206562,128,137),(1206574,131,136),(1206587,133,137),(1206595,135,137),(1206603,137,137),(1206611,139,139),(1206619,141,140),(1206627,143,141),(1206636,145,143),(1206643,147,145),(1206655,150,148),(1206664,152,149),(1206676,154,151),(1206692,156,153),(1206701,156,155),(1206713,156,157),(1206725,154,160),(1206737,152,162),(1206747,150,166),(1206756,148,168),(1206766,146,170),(1206776,144,173),(1206786,139,177),(1206796,137,179),(1206807,134,180),(1206817,130,184),(1206829,128,185),(1206840,125,190),(1206852,124,192),(1206864,123,195),(1206876,123,199),(1206888,123,202),(1206901,126,203),(1206913,131,204),(1206927,133,203),(1206939,139,201),(1206952,141,201),(1206954,144,201),(1206967,146,200),(1206980,151,200),(1206994,155,200),(1207009,158,198),(1207024,160,198),(1207042,162,198),(1207089,159,197),(1207105,157,197),(1207121,154,198),(1207122,152,199),(1207138,148,199),(1207154,144,199),(1207170,140,200),(1207185,137,203),(1207201,135,203),(1207218,133,205),(1207249,131,207),(1207277,133,208),(1207294,135,210),(1207311,137,211),(1207328,139,214),(1207346,143,218),(1207363,145,220),(1207381,149,225),(1207400,152,229),(1207418,153,234),(1207436,152,239),(1207455,148,243),(1207474,142,248),(1207494,133,254),(1207514,123,260),(1207533,117,263),(1207553,112,263),(1207573,110,263)]),
 ("star",[(663464,75,250),(663484,75,247),(663500,77,244),(663511,78,242),(663520,79,239),(663528,80,237),(663537,82,234),(663544,82,232),(663551,84,229),(663559,85,225),(663566,87,222),(663573,88,219),(663581,89,216),(663589,91,212),(663598,92,208),(663607,94,204),(663615,95,201),(663625,96,196),(663626,97,194),(663637,98,191),(663647,100,185),(663658,102,178),(663668,104,173),(663670,104,171),(663681,105,164),(663692,106,158),(663694,107,156),(663707,107,152),(663719,108,145),(663730,109,141),(663742,110,139),(663754,112,133),(663767,113,131),(663779,116,127),(663792,117,125),(663808,119,122),(663821,121,121),(663834,123,120),(663853,125,122),(663867,125,125),(663881,127,130),(663882,128,133),(663898,131,143),(663914,136,153),(663929,140,163),(663945,144,172),(663946,145,175),(663967,151,189),(663983,156,201),(663999,161,213),(664016,166,225),(664033,169,233),(664034,171,236),(664053,174,243),(664071,177,247),(664088,178,249),(664105,179,251),(664125,180,253),(664145,180,255),(664166,179,257),(664190,177,257),(664208,174,255),(664227,169,250),(664246,164,247),(664247,160,245),(664269,149,238),(664289,138,230),(664310,127,221),(664311,124,220),(664334,112,212),(664335,110,210),(664358,96,201),(664380,84,195),(664402,74,190),(664425,64,182),(664446,55,175),(664468,51,172),(664490,49,170),(664538,51,169),(664560,56,169),(664583,66,169),(664606,78,168),(664630,92,166),(664655,107,164),(664679,123,161),(664704,140,162),(664728,156,162),(664753,171,160),(664755,173,160),(664781,186,160),(664805,195,160),(664807,198,161),(664833,203,163),(664858,208,163),(664893,206,164),(664918,200,167),(664945,187,172),(664973,174,179),(664974,172,181),(665006,153,192),(665034,137,201),(665062,123,211),(665090,112,220),(665118,99,229),(665146,90,237),(665174,80,244),(665203,73,250),(665231,69,254),(665274,69,252)]),
 ("arrow",[(1351945,54,239),(1351952,56,238),(1351960,59,236),(1351968,62,235),(1351974,64,234),(1351980,66,233),(1351981,68,231),(1351989,73,229),(1351995,76,229),(1351997,79,228),(1352005,85,224),(1352013,91,220),(1352020,97,218),(1352028,100,216),(1352029,103,214),(1352039,109,211),(1352047,115,207),(1352055,122,203),(1352056,125,202),(1352067,131,199),(1352068,134,196),(1352081,144,191),(1352091,149,189),(1352101,157,183),(1352112,162,180),(1352113,165,179),(1352125,171,175),(1352136,174,172),(1352147,177,171),(1352158,179,169),(1352169,181,166),(1352192,178,165),(1352204,176,165),(1352216,173,166),(1352217,169,166),(1352233,161,165),(1352246,154,165),(1352259,147,167),(1352261,145,168),(1352277,135,171),(1352290,131,173),(1352304,125,176),(1352318,122,177),(1352331,120,177),(1352364,123,176),(1352378,126,176),(1352394,136,173),(1352409,146,170),(1352425,156,167),(1352441,167,163),(1352457,177,161),(1352476,186,158),(1352492,192,157),(1352508,196,157),(1352524,198,156),(1352555,198,158),(1352572,196,160),(1352589,190,167),(1352606,182,174),(1352624,176,180),(1352642,170,188),(1352643,168,191),(1352663,163,198),(1352682,159,204),(1352701,156,208),(1352719,155,210),(1352763,157,211)]),
 ("v",[(1024440,89,154),(1024497,91,156),(1024509,93,158),(1024517,94,161),(1024525,95,164),(1024533,97,168),(1024541,98,171),(1024548,99,174),(1024550,100,176),(1024558,102,182),(1024567,104,187),(1024575,106,192),(1024584,108,196),(1024592,111,200),(1024594,112,203),(1024604,113,205),(1024613,117,214),(1024622,118,218),(1024631,120,222),(1024641,121,225),(1024652,122,228),(1024654,123,231),(1024666,124,234),(1024677,125,236),(1024690,126,239),(1024706,126,241),(1024763,127,239),(1024775,128,237),(1024787,128,234),(1024799,129,232),(1024811,130,229),(1024824,132,222),(1024838,134,216),(1024851,136,204),(1024866,139,194),(1024880,143,182),(1024895,146,169),(1024910,149,162),(1024926,152,157),(1024940,153,155),(1024956,155,153)]),
 ("delete_mark",[(1446906,123,129),(1446922,123,131),(1446930,124,133),(1446938,125,136),(1446945,127,140),(1446952,129,142),(1446958,133,148),(1446966,137,154),(1446973,143,158),(1446981,145,161),(1446982,148,164),(1446992,153,170),(1447000,158,176),(1447002,160,178),(1447011,164,183),(1447019,168,188),(1447027,171,191),(1447036,175,196),(1447045,178,200),(1447048,180,202),(1447050,181,205),(1447059,184,208),(1447069,186,210),(1447079,187,213),(1447107,188,215),(1447119,186,212),(1447130,183,211),(1447141,177,208),(1447152,169,206),(1447164,162,205),(1447175,154,207),(1447187,145,209),(1447200,137,210),(1447212,129,214),(1447225,122,217),(1447238,118,218),(1447250,111,221),(1447263,109,222),(1447300,110,219),(1447313,112,217),(1447327,118,209),(1447329,120,207),(1447345,128,196),(1447359,135,187),(1447360,138,183),(1447377,148,167),(1447392,157,153),(1447408,163,145),(1447409,165,142),(1447426,172,133),(1447442,177,127),(1447458,179,127),(1447473,180,125)]),
 ("delete_mark",[(1445962,106,143),(1445970,106,146),(1445978,106,148),(1445988,107,152),(1445994,108,154),(1446000,111,160),(1446006,113,163),(1446012,117,170),(1446019,119,173),(1446020,122,176),(1446028,126,182),(1446036,131,187),(1446044,136,193),(1446051,138,196),(1446052,140,198),(1446062,142,200),(1446070,147,206),(1446079,151,210),(1446080,153,212),(1446090,156,215),(1446100,159,218),(1446109,162,221),(1446118,165,222),(1446128,167,224),(1446137,169,224),(1446178,167,223),(1446190,164,221),(1446201,163,219),(1446211,160,218),(1446222,158,216),(1446233,155,215),(1446234,151,215),(1446250,142,212),(1446263,134,212),(1446276,126,211),(1446288,117,213),(1446302,110,213),(1446303,108,213),(1446317,101,214),(1446330,96,214),(1446344,90,214),(1446357,88,214),(1446371,88,211),(1446386,88,208),(1446387,90,205),(1446404,93,200),(1446419,103,190),(1446434,110,181),(1446435,112,179),(1446452,123,168),(1446468,133,155),(1446484,143,142),(1446501,152,131),(1446517,157,126),(1446533,159,125)]),
 ("triangle",[(1304905,147,121),(1304916,145,122),(1304925,143,125),(1304932,142,127),(1304940,139,131),(1304948,137,135),(1304955,136,138),(1304961,133,143),(1304968,131,145),(1304969,129,148),(1304976,128,150),(1304977,126,153),(1304985,122,157),(1304993,118,162),(1305000,114,167),(1305009,109,171),(1305017,105,176),(1305025,100,180),(1305034,95,184),(1305043,90,188),(1305044,88,191),(1305054,84,194),(1305063,80,197),(1305064,78,199),(1305075,76,201),(1305086,69,204),(1305097,66,207),(1305108,63,209),(1305109,61,211),(1305120,59,212),(1305131,56,215),(1305143,53,219),(1305161,52,221),(1305200,54,223),(1305212,56,222),(1305224,58,222),(1305225,61,222),(1305238,64,221),(1305251,71,221),(1305264,79,220),(1305277,86,219),(1305291,93,218),(1305292,96,217),(1305309,106,218),(1305323,115,219),(1305338,127,220),(1305353,138,222),(1305368,146,223),(1305369,149,223),(1305386,159,224),(1305403,168,225),(1305419,176,227),(1305434,182,227),(1305450,188,226),(1305466,194,225),(1305483,198,225),(1305499,200,226),(1305532,201,224),(1305549,200,221),(1305566,196,215),(1305584,190,206),(1305603,183,195),(1305621,175,182),(1305639,169,173),(1305640,167,170),(1305661,162,163),(1305681,155,153),(1305700,150,143),(1305719,147,137),(1305721,145,134),(1305746,141,126)]),
 ("arrow",[(1354426,50,236),(1354441,53,235),(1354449,55,232),(1354458,58,231),(1354465,62,229),(1354471,64,228),(1354477,66,227),(1354478,69,225),(1354486,74,223),(1354493,80,221),(1354500,83,219),(1354501,85,217),(1354511,91,214),(1354519,96,211),(1354527,101,209),(1354535,107,206),(1354544,112,203),(1354553,117,200),(1354554,120,199),(1354565,125,197),(1354566,128,195),(1354578,135,192),(1354588,140,190),(1354598,147,185),(1354608,151,184),(1354618,155,183),(1354628,158,181),(1354629,162,179),(1354643,165,177),(1354654,170,173),(1354666,172,172),(1354677,176,168),(1354690,179,166),(1354702,180,164),(1354713,180,161),(1354725,180,159),(1354738,177,158),(1354750,175,157),(1354764,170,157),(1354777,164,158),(1354791,161,159),(1354804,154,163),(1354819,147,166),(1354833,144,167),(1354834,141,168),(1354851,139,169),(1354867,137,169),(1354906,140,170),(1354921,143,169),(1354922,146,167),(1354941,153,166),(1354957,161,164),(1354958,163,163),(1354977,172,160),(1354978,174,160),(1354998,181,159),(1355015,188,158),(1355033,193,159),(1355050,197,160),(1355075,199,163),(1355092,197,165),(1355110,194,169),(1355128,189,174),(1355147,185,179),(1355166,180,183),(1355185,172,193),(1355204,168,199),(1355223,166,202),(1355243,163,206),(1355312,163,208)]),
 ("question_mark",[(751496,104,145),(751516,103,142),(751553,103,140),(751566,103,138),(751573,103,135),(751581,104,133),(751598,105,131),(751606,106,128),(751614,107,125),(751622,108,123),(751634,111,121),(751646,113,118),(751654,115,116),(751666,117,116),(751677,119,116),(751685,121,115),(751698,124,116),(751706,126,115),(751715,128,114),(751724,130,115),(751734,133,116),(751744,135,117),(751755,140,120),(751765,142,121),(751776,144,123),(751787,146,125),(751798,149,127),(751809,150,129),(751821,152,130),(751832,154,132),(751844,156,134),(751857,158,137),(751869,159,139),(751881,160,141),(751882,160,143),(751896,160,146),(751909,160,149),(751922,159,153),(751936,158,155),(751950,157,157),(751964,155,159),(751978,153,161),(751993,151,163),(752010,146,167),(752025,142,170),(752041,138,172),(752056,134,173),(752072,132,175),(752087,127,175),(752103,124,175),(752121,122,176),(752142,120,178),(752182,119,180),(752199,119,183),(752215,119,185),(752233,120,190),(752250,121,194),(752268,122,200),(752288,123,205),(752306,123,211),(752325,124,215),(752343,124,223),(752362,124,225)]),
 ("right_sq_bracket",[(800952,116,135),(800976,119,134),(800988,121,132),(800997,123,130),(801005,125,129),(801013,128,128),(801021,130,129),(801029,132,128),(801037,135,127),(801045,137,128),(801053,139,128),(801065,142,129),(801073,144,129),(801085,146,130),(801098,149,130),(801110,151,132),(801130,153,133),(801143,153,136),(801152,153,138),(801161,153,140),(801171,153,145),(801182,154,149),(801192,154,156),(801202,154,161),(801204,154,164),(801214,155,168),(801216,155,171),(801228,156,179),(801240,156,187),(801253,157,194),(801265,157,201),(801277,156,208),(801290,155,213),(801302,155,218),(801304,155,221),(801317,155,224),(801330,155,229),(801344,154,231),(801357,154,233),(801394,152,235),(801420,150,236),(801436,147,235),(801451,145,234),(801465,139,233),(801480,133,230),(801495,127,229),(801511,120,228),(801526,115,230),(801544,110,231)]),
 ("star",[(680406,83,257),(680447,83,255),(680459,84,253),(680470,85,251),(680478,85,249),(680486,86,247),(680498,86,243),(680506,87,241),(680514,87,238),(680521,88,236),(680528,88,234),(680535,88,231),(680542,89,228),(680543,89,226),(680553,89,222),(680561,90,220),(680569,91,217),(680577,92,211),(680578,92,209),(680589,94,205),(680590,94,203),(680601,94,201),(680603,95,197),(680614,96,195),(680615,96,191),(680627,97,185),(680637,98,180),(680638,98,178),(680651,99,172),(680662,100,170),(680673,101,166),(680684,102,161),(680696,104,155),(680707,105,151),(680721,106,144),(680733,107,139),(680745,108,136),(680746,108,134),(680761,108,131),(680787,108,128),(680800,109,126),(680813,110,123),(680827,112,120),(680841,112,118),(680884,114,118),(680900,115,120),(680915,115,123),(680930,117,128),(680945,120,138),(680960,124,149),(680976,127,157),(680977,128,160),(680995,133,172),(680996,135,176),(681015,139,186),(681032,144,197),(681033,145,200),(681052,151,213),(681070,156,222),(681087,160,230),(681088,160,232),(681108,165,241),(681126,168,246),(681144,170,251),(681162,171,253),(681181,171,255),(681208,169,256),(681227,167,253),(681246,160,248),(681266,155,244),(681285,144,237),(681306,134,231),(681327,122,220),(681348,110,213),(681349,108,212),(681374,95,205),(681395,84,200),(681397,82,198),(681423,73,193),(681445,63,187),(681467,56,180),(681489,51,177),(681511,49,177),(681533,47,175),(681556,49,174),(681579,52,175),(681581,54,176),(681606,63,177),(681629,75,176),(681654,89,175),(681679,104,174),(681703,120,173),(681705,123,172),(681732,140,171),(681758,153,170),(681783,163,170),(681785,167,169),(681812,176,170),(681838,183,169),(681865,187,170),(681901,186,172),(681927,183,175),(681954,176,178),(681981,168,185),(682008,155,193),(682009,153,195),(682040,138,206),(682069,122,218),(682070,120,220),(682101,107,230),(682129,95,240),(682158,88,246),(682186,81,251),(682216,77,256),(682244,73,260),(682273,71,263)]),
 ("rectangle",[(911765,89,141),(911810,89,143),(911817,89,145),(911825,89,149),(911833,89,152),(911841,90,156),(911847,90,158),(911854,91,163),(911860,91,165),(911861,91,168),(911869,92,172),(911877,92,174),(911878,92,177),(911886,92,179),(911894,93,184),(911903,94,190),(911911,94,195),(911920,94,201),(911929,96,208),(911938,96,212),(911948,97,217),(911958,98,224),(911967,98,227),(911977,99,233),(911987,100,236),(911998,100,238),(912009,101,241),(912029,101,243),(912049,102,245),(912073,102,247),(912154,105,247),(912174,107,247),(912191,109,247),(912203,112,247),(912216,114,248),(912228,117,248),(912241,121,247),(912253,124,246),(912266,131,247),(912280,136,247),(912293,142,247),(912307,149,246),(912323,156,247),(912337,164,246),(912352,172,243),(912366,177,242),(912381,184,241),(912396,191,242),(912411,194,240),(912412,197,240),(912430,203,238),(912446,208,238),(912462,210,238),(912478,212,238),(912496,214,237),(912531,214,239),(912563,213,237),(912579,211,235),(912596,210,232),(912614,209,228),(912632,208,222),(912650,206,213),(912668,205,204),(912687,205,192),(912706,204,185),(912725,204,176),(912744,203,164),(912767,203,158),(912768,203,155),(912788,203,150),(912808,202,146),(912828,202,143),(912847,202,141),(912868,200,139),(912888,198,137),(912908,197,135),(912929,195,134),(912950,190,130),(912972,185,129),(912993,179,128),(913015,172,127),(913038,168,127),(913060,160,127),(913102,147,127),(913124,138,129),(913147,129,131),(913170,122,131),(913194,115,130),(913218,111,130),(913242,106,131),(913265,104,132),(913289,102,131),(913319,100,132)]),
 ("caret",[(1258277,77,245),(1258286,78,242),(1258294,80,239),(1258302,82,235),(1258308,83,232),(1258313,85,229),(1258315,86,227),(1258323,87,224),(1258329,91,220),(1258331,92,217),(1258339,96,211),(1258347,99,204),(1258355,103,196),(1258362,106,188),(1258371,109,179),(1258379,113,171),(1258387,116,164),(1258396,119,158),(1258405,121,152),(1258406,122,149),(1258418,125,143),(1258419,126,141),(1258430,129,134),(1258441,131,129),(1258451,133,125),(1258461,135,122),(1258471,136,120),(1258482,138,120),(1258506,141,119),(1258518,142,121),(1258531,144,126),(1258543,146,136),(1258555,149,146),(1258568,152,159),(1258580,155,172),(1258593,157,184),(1258606,160,195),(1258607,161,199),(1258623,165,215),(1258637,168,225),(1258650,171,234),(1258651,172,237),(1258668,174,245),(1258682,175,248),(1258696,177,253)]),
 ("x",[(1123177,95,144),(1123188,95,146),(1123200,97,150),(1123208,98,152),(1123215,101,156),(1123221,102,158),(1123227,104,160),(1123228,105,163),(1123236,109,167),(1123243,111,170),(1123244,113,173),(1123253,118,178),(1123260,122,184),(1123268,125,189),(1123276,129,194),(1123285,132,199),(1123293,133,201),(1123302,136,205),(1123311,140,210),(1123322,142,213),(1123323,145,215),(1123335,147,218),(1123344,150,219),(1123355,152,221),(1123376,154,222),(1123404,156,220),(1123414,157,217),(1123425,157,213),(1123436,158,207),(1123437,158,204),(1123450,157,194),(1123461,157,184),(1123473,157,173),(1123486,158,161),(1123498,159,151),(1123511,160,142),(1123524,162,133),(1123525,162,130),(1123539,163,123),(1123553,164,120),(1123577,161,119),(1123590,159,122),(1123604,156,125),(1123618,149,137),(1123633,141,147),(1123648,134,159),(1123663,124,174),(1123678,114,190),(1123694,104,203),(1123711,97,215),(1123727,92,221),(1123742,90,224),(1123758,88,226),(1123774,88,228),(1123799,89,226),(1123816,91,224)]),
 ("arrow",[(1355765,59,236),(1355776,62,234),(1355784,65,233),(1355792,68,230),(1355798,71,228),(1355805,76,226),(1355812,81,223),(1355819,84,222),(1355821,87,220),(1355831,93,216),(1355839,99,212),(1355847,106,208),(1355849,109,206),(1355859,115,203),(1355867,122,199),(1355875,127,195),(1355883,133,193),(1355891,138,190),(1355893,141,188),(1355905,149,183),(1355914,153,180),(1355924,160,177),(1355934,165,175),(1355944,167,174),(1355945,170,171),(1355957,173,170),(1355968,175,168),(1355978,177,166),(1356000,177,164),(1356012,175,163),(1356025,173,162),(1356037,167,162),(1356050,162,160),(1356063,157,160),(1356077,151,161),(1356090,144,163),(1356104,141,164),(1356105,138,164),(1356120,135,166),(1356135,133,167),(1356169,132,169),(1356184,135,169),(1356199,141,168),(1356216,151,164),(1356218,153,163),(1356237,162,160),(1356238,164,159),(1356258,176,156),(1356277,184,155),(1356292,189,153),(1356308,192,152),(1356323,194,152),(1356338,195,154),(1356354,194,159),(1356371,190,165),(1356388,184,174),(1356405,176,183),(1356423,171,190),(1356440,166,197),(1356441,165,199),(1356461,160,203),(1356480,158,206),(1356510,157,208)]),
 ("left_curly_brace",[(1407529,144,123),(1407544,142,121),(1407552,139,122),(1407560,136,121),(1407568,133,121),(1407575,129,123),(1407584,125,125),(1407592,122,129),(1407600,118,132),(1407608,116,135),(1407616,113,137),(1407624,111,140),(1407639,109,143),(1407667,110,145),(1407683,112,147),(1407692,115,147),(1407701,117,148),(1407710,120,148),(1407720,123,149),(1407730,125,150),(1407740,131,152),(1407750,134,154),(1407760,136,156),(1407770,137,158),(1407781,138,161),(1407793,137,165),(1407804,135,168),(1407815,130,170),(1407827,127,172),(1407828,125,173),(1407842,121,173),(1407854,115,174),(1407866,113,175),(1407879,111,174),(1407932,113,173),(1407945,115,173),(1407958,120,172),(1407972,123,173),(1407973,126,171),(1407989,132,171),(1408003,134,171),(1408018,139,172),(1408032,140,174),(1408047,140,176),(1408062,138,180),(1408078,135,185),(1408093,134,188),(1408109,131,195),(1408124,129,200),(1408140,127,205),(1408157,127,212),(1408173,128,214),(1408190,131,219),(1408207,136,222),(1408224,142,224),(1408242,148,224)]),
 ("left_curly_brace",[(1397615,139,124),(1397622,137,125),(1397630,134,125),(1397638,131,125),(1397646,127,127),(1397655,124,129),(1397663,121,131),(1397671,118,134),(1397679,115,137),(1397687,112,139),(1397695,109,142),(1397703,108,144),(1397711,107,147),(1397720,106,150),(1397729,105,152),(1397738,105,154),(1397747,107,158),(1397759,110,160),(1397768,113,160),(1397777,116,161),(1397787,119,164),(1397797,122,164),(1397808,126,169),(1397819,128,171),(1397831,128,174),(1397842,126,177),(1397853,124,179),(1397866,117,183),(1397878,110,186),(1397890,103,188),(1397903,100,188),(1397904,97,189),(1397919,95,189),(1397959,97,189),(1397972,99,190),(1397986,106,189),(1398000,112,189),(1398014,114,190),(1398028,117,190),(1398043,118,192),(1398057,117,194),(1398073,111,203),(1398088,107,207),(1398104,104,214),(1398120,102,223),(1398135,103,227),(1398154,106,233),(1398170,114,234),(1398187,125,236),(1398204,127,236)]),
 ("arrow",[(1350727,45,236),(1350738,48,234),(1350747,51,231),(1350753,53,230),(1350759,57,228),(1350765,60,227),(1350772,66,223),(1350779,69,221),(1350780,72,219),(1350787,75,218),(1350788,78,217),(1350796,85,214),(1350804,92,210),(1350812,98,205),(1350820,105,202),(1350829,111,198),(1350839,118,195),(1350840,121,193),(1350849,128,189),(1350859,134,185),(1350860,138,183),(1350870,144,180),(1350880,153,176),(1350890,158,173),(1350892,160,172),(1350902,164,169),(1350903,167,168),(1350915,170,166),(1350927,173,164),(1350928,176,163),(1350940,178,160),(1350951,180,159),(1350968,180,157),(1350980,179,155),(1350993,177,154),(1351006,174,154),(1351020,168,154),(1351022,166,153),(1351039,158,155),(1351052,151,156),(1351066,143,159),(1351080,139,159),(1351094,134,162),(1351108,132,163),(1351122,130,164),(1351146,132,163),(1351161,134,163),(1351176,140,163),(1351192,151,160),(1351209,159,158),(1351225,166,156),(1351241,174,154),(1351257,180,154),(1351258,182,154),(1351276,185,153),(1351292,187,155),(1351309,188,160),(1351326,186,166),(1351344,181,173),(1351362,176,178),(1351363,173,181),(1351384,167,187),(1351402,161,191),(1351403,159,195),(1351424,156,197),(1351442,154,199)]),
 ("star",[(677461,78,261),(677481,78,259),(677505,79,257),(677518,80,255),(677531,82,252),(677538,83,249),(677546,84,246),(677554,85,243),(677562,86,240),(677570,87,236),(677578,88,233),(677586,89,230),(677594,90,225),(677603,91,221),(677612,93,216),(677620,93,213),(677630,95,207),(677639,96,202),(677640,96,200),(677650,97,195),(677660,97,191),(677670,98,188),(677680,99,180),(677691,100,176),(677702,100,170),(677713,101,163),(677724,102,157),(677736,103,152),(677747,105,146),(677759,105,144),(677760,106,141),(677774,106,138),(677786,106,132),(677799,107,127),(677813,108,124),(677826,108,119),(677840,110,113),(677853,111,111),(677871,112,109),(677903,113,112),(677917,114,114),(677931,115,119),(677946,116,122),(677961,118,129),(677976,121,137),(677992,125,147),(678007,130,156),(678023,134,165),(678040,138,175),(678056,143,184),(678072,147,192),(678089,150,199),(678109,157,211),(678126,160,219),(678128,161,221),(678147,166,229),(678148,166,232),(678166,170,238),(678184,173,247),(678202,176,252),(678221,177,258),(678239,177,260),(678297,175,258),(678316,173,256),(678335,168,252),(678355,161,246),(678376,153,240),(678396,143,232),(678419,130,224),(678441,119,215),(678462,108,207),(678463,106,206),(678486,95,196),(678513,82,188),(678535,75,182),(678556,66,174),(678578,58,169),(678600,51,165),(678623,48,163),(678689,50,163),(678711,56,164),(678734,65,164),(678758,77,163),(678782,92,163),(678806,109,161),(678831,127,158),(678856,144,158),(678858,146,158),(678883,161,157),(678912,173,157),(678936,182,158),(678938,185,158),(678964,189,158),(678989,192,160),(679029,190,162),(679055,188,164),(679082,179,171),(679108,171,176),(679133,162,185),(679160,151,194),(679161,148,196),(679190,135,207),(679218,123,217),(679245,108,230),(679273,100,237),(679300,93,245),(679301,91,245),(679331,85,251),(679359,83,253),(679386,81,256),(679414,79,259),(679444,78,261)]),
 ("triangle",[(1306199,149,123),(1306237,147,125),(1306244,146,127),(1306255,144,130),(1306261,143,132),(1306268,142,134),(1306276,140,137),(1306283,136,144),(1306291,134,147),(1306292,133,149),(1306302,129,155),(1306310,125,160),(1306312,123,162),(1306321,118,167),(1306329,113,173),(1306337,109,177),(1306346,104,180),(1306354,100,183),(1306363,93,189),(1306372,89,192),(1306381,87,194),(1306391,81,199),(1306392,80,201),(1306403,76,204),(1306413,73,207),(1306423,67,212),(1306433,64,214),(1306444,61,218),(1306455,59,220),(1306467,56,220),(1306478,54,222),(1306492,52,224),(1306528,54,224),(1306540,57,224),(1306552,59,223),(1306565,66,221),(1306577,71,221),(1306590,78,219),(1306603,85,218),(1306616,95,214),(1306630,104,213),(1306644,112,212),(1306645,115,212),(1306661,126,211),(1306675,133,213),(1306677,136,212),(1306694,147,213),(1306710,157,216),(1306725,167,218),(1306743,176,219),(1306745,178,220),(1306763,184,221),(1306779,190,222),(1306780,193,222),(1306799,197,222),(1306815,199,221),(1306833,203,219),(1306859,204,217),(1306884,203,214),(1306901,199,210),(1306919,196,205),(1306920,194,202),(1306941,188,193),(1306959,182,184),(1306960,181,182),(1306982,175,173),(1307001,166,161),(1307021,160,151),(1307041,156,141),(1307060,152,135),(1307079,150,133),(1307081,147,132)]),
 ("x",[(1118371,82,144),(1118386,82,146),(1118394,83,149),(1118402,84,151),(1118410,86,153),(1118418,88,157),(1118424,90,159),(1118431,93,163),(1118437,95,166),(1118444,98,170),(1118451,102,175),(1118458,104,178),(1118459,105,180),(1118468,107,182),(1118476,113,188),(1118483,115,190),(1118492,120,196),(1118500,123,199),(1118508,125,201),(1118517,128,204),(1118526,132,207),(1118527,135,210),(1118539,139,215),(1118549,142,217),(1118559,144,219),(1118570,145,222),(1118571,147,221),(1118590,150,222),(1118606,151,220),(1118617,151,217),(1118629,151,213),(1118641,151,206),(1118653,150,197),(1118654,150,194),(1118669,149,185),(1118682,149,175),(1118683,148,171),(1118698,148,161),(1118699,148,157),(1118713,149,146),(1118714,149,143),(1118730,151,132),(1118744,152,127),(1118758,153,122),(1118791,151,124),(1118806,147,128),(1118807,146,130),(1118825,139,140),(1118841,130,151),(1118857,119,164),(1118859,117,167),(1118876,105,183),(1118893,95,198),(1118910,87,210),(1118911,85,212),(1118929,80,220),(1118930,79,222),(1118949,77,225),(1118966,75,226),(1119006,77,224)]),
 ("rectangle",[(905830,83,147),(905861,84,145),(905889,84,148),(905897,84,152),(905905,84,154),(905913,85,158),(905919,85,162),(905926,85,166),(905933,85,170),(905940,85,173),(905941,85,175),(905949,85,177),(905950,85,179),(905958,85,184),(905966,86,189),(905974,86,194),(905984,86,199),(905992,87,203),(905994,87,206),(906003,88,211),(906014,88,218),(906023,89,220),(906034,89,228),(906044,89,230),(906054,89,233),(906065,89,235),(906076,89,237),(906140,91,239),(906168,94,239),(906184,96,239),(906196,98,239),(906208,100,239),(906221,103,240),(906234,108,240),(906247,112,241),(906261,119,240),(906275,127,239),(906277,130,239),(906292,136,239),(906293,139,240),(906312,150,241),(906328,159,242),(906343,166,242),(906345,168,242),(906361,174,241),(906377,182,240),(906393,189,240),(906410,196,239),(906413,199,239),(906431,204,240),(906448,209,239),(906465,213,238),(906483,216,237),(906505,218,237),(906553,219,235),(906586,218,233),(906603,217,230),(906623,215,219),(906642,214,208),(906663,213,195),(906683,212,181),(906703,211,171),(906723,210,162),(906744,210,150),(906765,209,144),(906785,209,139),(906806,209,136),(906825,209,134),(906864,207,133),(906883,205,133),(906902,200,135),(906923,195,134),(906924,192,134),(906947,184,133),(906968,171,131),(906989,159,130),(907011,143,131),(907033,131,132),(907055,119,131),(907056,117,131),(907085,105,128),(907107,94,124),(907130,85,123),(907152,80,122),(907175,78,121)]),
 ("star",[(669554,79,256),(669580,80,254),(669597,82,251),(669605,84,249),(669618,86,247),(669630,88,244),(669637,88,242),(669644,90,238),(669650,91,235),(669658,92,232),(669666,94,228),(669674,95,225),(669682,97,220),(669691,98,216),(669699,99,211),(669708,101,207),(669717,103,203),(669719,103,201),(669729,105,197),(669739,107,191),(669749,108,188),(669759,110,182),(669769,112,178),(669780,115,170),(669791,116,163),(669802,117,158),(669813,118,155),(669825,118,146),(669836,118,140),(669848,119,134),(669860,119,128),(669872,120,125),(669885,121,119),(669898,122,114),(669911,122,112),(669932,123,110),(669956,123,112),(669971,124,115),(669985,125,120),(669998,126,123),(670000,128,127),(670015,131,136),(670030,134,144),(670046,138,156),(670061,142,167),(670079,145,177),(670080,146,179),(670098,149,190),(670114,152,201),(670130,155,210),(670131,156,212),(670150,159,220),(670152,159,222),(670169,161,227),(670187,164,236),(670204,165,240),(670222,166,242),(670239,166,246),(670257,167,249),(670274,167,252),(670292,166,254),(670335,164,255),(670353,163,253),(670372,159,249),(670392,151,242),(670413,143,233),(670433,133,225),(670454,125,218),(670456,121,215),(670477,111,207),(670498,101,198),(670519,88,190),(670540,82,185),(670562,73,181),(670583,65,176),(670605,60,173),(670626,57,169),(670648,55,168),(670670,54,166),(670692,56,165),(670715,62,166),(670738,71,167),(670761,82,166),(670786,95,165),(670810,107,163),(670834,120,161),(670836,123,161),(670861,136,161),(670886,149,159),(670887,151,159),(670913,161,159),(670938,171,159),(670964,179,160),(670990,185,160),(671015,189,159),(671075,188,162),(671101,185,164),(671128,177,169),(671154,168,176),(671156,164,178),(671184,154,185),(671212,142,195),(671239,127,207),(671267,116,216),(671296,107,224),(671324,97,233),(671352,92,240),(671382,87,246),(671410,84,250),(671439,83,252),(671440,81,254),(671471,80,256),(671501,78,258)]),
 ("left_curly_brace",[(1405269,142,118),(1405280,139,117),(1405288,136,117),(1405297,133,118),(1405304,130,118),(1405313,126,120),(1405321,123,122),(1405329,120,125),(1405337,118,127),(1405345,116,129),(1405353,114,132),(1405361,112,134),(1405369,111,137),(1405377,110,140),(1405390,111,142),(1405406,113,143),(1405415,115,144),(1405423,118,145),(1405432,121,145),(1405441,124,146),(1405450,127,146),(1405459,131,148),(1405468,134,150),(1405478,136,151),(1405488,139,153),(1405498,139,155),(1405508,139,157),(1405519,137,160),(1405529,135,163),(1405540,130,167),(1405551,127,167),(1405562,124,168),(1405563,122,168),(1405577,120,168),(1405589,118,168),(1405601,116,169),(1405644,118,167),(1405656,121,168),(1405669,123,168),(1405682,125,168),(1405695,130,169),(1405709,132,170),(1405722,134,170),(1405736,138,173),(1405751,140,176),(1405766,140,181),(1405780,137,187),(1405795,134,190),(1405809,131,195),(1405825,128,201),(1405840,126,203),(1405841,126,206),(1405858,125,208),(1405874,125,213),(1405890,128,216),(1405907,133,218),(1405924,139,219),(1405941,146,219),(1405958,149,219)]),
 ("delete_mark",[(1442373,122,130),(1442384,123,133),(1442392,124,137),(1442398,125,139),(1442405,128,144),(1442412,129,147),(1442413,131,150),(1442421,134,156),(1442429,138,162),(1442437,141,168),(1442444,144,173),(1442452,147,178),(1442461,150,184),(1442468,152,188),(1442477,154,192),(1442485,156,195),(1442493,158,199),(1442502,159,201),(1442511,160,203),(1442521,161,205),(1442532,162,208),(1442565,160,209),(1442577,157,207),(1442587,154,206),(1442598,151,205),(1442609,143,204),(1442621,139,205),(1442632,133,207),(1442634,130,208),(1442648,123,210),(1442660,116,214),(1442673,109,217),(1442674,107,218),(1442692,98,221),(1442707,91,224),(1442721,85,226),(1442736,83,227),(1442750,80,226),(1442769,80,224),(1442784,81,222),(1442800,88,214),(1442815,95,203),(1442832,105,191),(1442834,108,188),(1442852,119,174),(1442872,135,160),(1442890,150,146),(1442909,161,135),(1442927,166,130),(1442946,170,125)]),
 ("x",[(1119331,87,142),(1119341,89,145),(1119349,91,148),(1119357,93,151),(1119365,96,155),(1119371,98,157),(1119376,100,160),(1119378,102,162),(1119386,106,167),(1119393,108,169),(1119394,110,171),(1119403,115,177),(1119411,119,183),(1119418,123,189),(1119426,127,193),(1119434,129,196),(1119443,133,200),(1119452,137,206),(1119460,140,209),(1119472,143,212),(1119481,146,215),(1119490,151,220),(1119500,153,222),(1119509,155,223),(1119519,157,225),(1119565,158,223),(1119576,157,218),(1119588,155,211),(1119590,154,208),(1119603,152,200),(1119614,150,189),(1119626,148,179),(1119638,147,170),(1119650,147,158),(1119662,147,148),(1119677,147,141),(1119689,147,136),(1119712,144,135),(1119725,142,137),(1119737,140,139),(1119751,135,145),(1119765,131,152),(1119779,124,163),(1119793,116,177),(1119808,108,191),(1119825,100,206),(1119840,94,217),(1119855,91,222),(1119870,89,225),(1119885,87,226),(1119928,87,224)]),
 ("arrow",[(1360517,81,219),(1360533,83,218),(1360542,85,218),(1360550,87,216),(1360558,90,213),(1360566,94,211),(1360572,96,210),(1360579,101,208),(1360586,103,206),(1360587,106,204),(1360594,108,202),(1360595,111,200),(1360603,117,196),(1360611,123,193),(1360619,129,190),(1360627,135,186),(1360636,141,183),(1360645,147,179),(1360646,149,177),(1360656,155,174),(1360666,160,172),(1360667,163,170),(1360677,168,167),(1360689,175,162),(1360700,182,157),(1360711,185,154),(1360723,190,152),(1360739,192,151),(1360751,192,149),(1360767,190,148),(1360779,187,147),(1360793,182,146),(1360806,178,146),(1360820,168,147),(1360834,162,147),(1360836,160,148),(1360852,152,148),(1360867,148,148),(1360868,145,149),(1360883,141,150),(1360896,138,152),(1360950,140,153),(1360964,143,152),(1360979,153,151),(1360993,159,151),(1360995,162,151),(1361011,171,148),(1361027,180,144),(1361042,187,141),(1361058,190,141),(1361059,193,140),(1361077,195,140),(1361093,197,139),(1361115,197,141),(1361131,196,144),(1361148,194,149),(1361165,189,154),(1361183,182,166),(1361201,177,175),(1361219,173,183),(1361237,170,193),(1361255,169,195),(1361273,168,202),(1361295,169,204),(1361313,171,204),(1361332,173,202)]),
 ("circle",[(1160137,133,141),(1160144,130,139),(1160152,126,137),(1160159,123,135),(1160168,119,133),(1160174,117,133),(1160180,115,134),(1160181,113,134),(1160189,108,134),(1160196,105,136),(1160203,103,139),(1160210,99,142),(1160218,96,147),(1160225,93,152),(1160233,91,158),(1160241,89,164),(1160250,88,170),(1160258,87,177),(1160267,87,184),(1160276,87,193),(1160285,88,199),(1160294,89,202),(1160304,92,208),(1160314,96,215),(1160324,99,220),(1160334,105,223),(1160344,110,227),(1160355,115,229),(1160366,124,231),(1160377,132,233),(1160389,141,232),(1160401,150,230),(1160413,158,225),(1160425,165,220),(1160437,172,212),(1160449,178,204),(1160462,181,196),(1160475,184,186),(1160488,183,173),(1160502,180,164),(1160515,176,155),(1160516,175,153),(1160533,166,146),(1160547,158,144),(1160548,156,144),(1160565,144,144),(1160580,139,145)]),
 ("pigtail",[(1074364,81,219),(1074396,84,218),(1074409,86,220),(1074417,88,220),(1074425,90,220),(1074434,92,219),(1074442,95,220),(1074450,97,219),(1074458,99,220),(1074466,102,218),(1074474,105,217),(1074482,107,216),(1074490,110,216),(1074499,113,214),(1074507,116,212),(1074514,118,210),(1074523,121,208),(1074531,124,205),(1074539,126,202),(1074549,129,199),(1074559,132,196),(1074569,136,191),(1074579,139,187),(1074589,142,182),(1074590,144,179),(1074603,146,174),(1074613,148,170),(1074614,149,168),(1074627,151,162),(1074639,152,160),(1074650,152,157),(1074651,152,155),(1074664,152,151),(1074676,152,149),(1074688,152,146),(1074701,149,142),(1074713,148,139),(1074726,145,137),(1074739,141,135),(1074752,139,135),(1074766,134,136),(1074779,130,140),(1074793,128,142),(1074794,126,145),(1074811,122,150),(1074826,119,158),(1074841,117,163),(1074856,115,170),(1074872,114,175),(1074888,117,184),(1074907,120,190),(1074923,125,199),(1074939,129,203),(1074955,133,208),(1074972,138,213),(1074989,145,215),(1075007,155,218),(1075025,164,219),(1075026,166,219),(1075047,177,219),(1075065,182,218),(1075083,192,216),(1075101,196,213),(1075120,199,212),(1075139,201,211)]),
 ("check",[(846301,88,192),(846341,89,189),(846358,90,187),(846366,92,185),(846382,94,184),(846395,96,184),(846407,99,184),(846419,101,187),(846427,103,189),(846435,105,191),(846443,106,193),(846454,108,197),(846462,109,199),(846475,112,202),(846483,113,206),(846492,114,208),(846501,115,210),(846509,117,214),(846519,118,216),(846529,119,219),(846541,121,222),(846557,123,224),(846618,124,222),(846630,124,219),(846641,124,217),(846651,124,214),(846663,124,207),(846674,125,199),(846687,126,193),(846699,127,185),(846711,130,176),(846723,133,169),(846736,137,161),(846749,141,154),(846762,146,146),(846775,151,139),(846789,153,135),(846803,158,128),(846816,161,123),(846830,164,121),(846844,165,119),(846858,167,116),(846859,169,114),(846876,172,109),(846891,174,107)]),
 ("right_curly_brace",[(1213143,125,149),(1213199,128,147),(1213207,130,147),(1213215,133,146),(1213224,136,145),(1213232,139,144),(1213240,141,145),(1213248,144,145),(1213256,147,145),(1213264,149,145),(1213272,151,147),(1213281,153,149),(1213288,155,150),(1213297,156,153),(1213305,157,155),(1213313,158,157),(1213322,158,160),(1213331,157,162),(1213340,157,164),(1213350,156,167),(1213360,154,169),(1213369,152,171),(1213380,150,173),(1213390,147,175),(1213401,142,176),(1213412,139,177),(1213423,137,178),(1213424,134,178),(1213436,132,178),(1213448,130,180),(1213449,128,181),(1213462,125,182),(1213475,123,184),(1213493,122,187),(1213506,122,189),(1213519,123,192),(1213533,125,194),(1213547,128,196),(1213561,130,196),(1213562,132,196),(1213576,135,197),(1213591,141,196),(1213607,146,194),(1213624,151,194),(1213640,156,193),(1213656,161,191),(1213677,163,191),(1213705,161,191),(1213721,159,189),(1213738,157,190),(1213755,152,190),(1213773,148,193),(1213791,143,195),(1213808,141,195),(1213826,139,196),(1213853,137,196),(1213909,139,199),(1213928,141,201),(1213946,143,200),(1213966,149,205),(1213985,153,208),(1214005,158,213),(1214006,159,215),(1214026,162,218),(1214046,163,226),(1214066,162,229),(1214087,160,235),(1214110,154,241),(1214131,145,246),(1214155,132,247),(1214178,123,249),(1214200,117,249),(1214226,115,250)]),
 ("triangle",[(1303515,141,128),(1303526,140,130),(1303539,139,132),(1303548,137,136),(1303554,137,138),(1303560,136,140),(1303566,134,143),(1303572,132,148),(1303579,130,150),(1303580,129,153),(1303589,126,159),(1303597,122,164),(1303604,118,170),(1303612,115,173),(1303613,113,176),(1303623,108,182),(1303631,103,188),(1303639,98,193),(1303640,96,196),(1303651,90,202),(1303652,88,204),(1303665,81,211),(1303675,77,215),(1303685,70,220),(1303695,67,223),(1303697,65,225),(1303709,62,227),(1303719,60,230),(1303720,58,232),(1303733,55,235),(1303744,54,237),(1303756,52,238),(1303791,54,239),(1303803,56,240),(1303816,59,240),(1303829,61,240),(1303842,66,240),(1303855,72,239),(1303868,78,240),(1303869,80,240),(1303884,89,239),(1303898,97,238),(1303912,107,236),(1303927,115,234),(1303941,124,234),(1303956,132,233),(1303957,135,233),(1303974,145,231),(1303990,155,231),(1304005,164,231),(1304024,172,233),(1304025,174,233),(1304043,181,235),(1304059,186,236),(1304076,191,237),(1304093,196,238),(1304110,198,239),(1304128,200,238),(1304153,200,235),(1304170,198,230),(1304188,196,223),(1304206,192,216),(1304224,185,205),(1304243,177,196),(1304262,169,188),(1304280,164,179),(1304281,163,177),(1304303,157,165),(1304322,151,154),(1304342,146,146),(1304362,142,141),(1304381,139,139)]),
 ("check",[(850162,91,185),(850236,93,185),(850252,95,185),(850264,97,185),(850276,100,188),(850285,102,189),(850293,104,190),(850301,106,193),(850309,108,195),(850317,110,198),(850325,112,201),(850337,114,204),(850345,115,207),(850353,117,210),(850362,118,212),(850370,120,214),(850378,121,217),(850387,122,219),(850396,123,222),(850406,124,224),(850416,126,226),(850426,127,229),(850437,129,231),(850481,130,233),(850526,129,231),(850538,129,228),(850550,129,226),(850561,129,224),(850573,129,221),(850575,129,218),(850587,129,212),(850599,129,208),(850611,130,198),(850623,132,189),(850635,134,182),(850648,137,173),(850663,143,164),(850676,147,157),(850689,151,151),(850702,155,144),(850715,161,137),(850729,165,131),(850743,171,122),(850757,174,118),(850771,176,114),(850786,177,112),(850807,177,114),(850827,175,116),(850843,173,118)]),
 ("check",[(843768,81,194),(843779,82,192),(843800,84,191),(843811,86,190),(843823,88,190),(843831,90,192),(843839,93,193),(843847,95,193),(843855,98,195),(843863,100,198),(843871,102,201),(843879,104,202),(843886,105,204),(843894,107,208),(843903,110,211),(843912,111,214),(843921,113,216),(843929,114,219),(843938,116,222),(843946,117,224),(843955,118,227),(843964,119,229),(844026,120,227),(844035,120,224),(844045,120,221),(844055,119,218),(844066,119,214),(844067,118,211),(844080,118,203),(844091,119,196),(844102,121,190),(844114,123,184),(844115,124,181),(844129,128,174),(844141,133,168),(844142,134,166),(844157,139,159),(844170,144,153),(844183,150,146),(844196,153,143),(844209,157,137),(844222,161,133),(844237,162,130),(844251,162,128),(844282,164,126)]),
 ("x",[(1116505,84,162),(1116537,86,163),(1116545,88,166),(1116553,90,169),(1116559,91,171),(1116566,94,175),(1116572,96,178),(1116580,100,184),(1116587,103,189),(1116594,107,195),(1116601,111,200),(1116609,113,203),(1116610,114,206),(1116618,116,208),(1116626,122,214),(1116634,125,219),(1116642,128,223),(1116651,131,226),(1116659,133,228),(1116668,135,231),(1116677,137,233),(1116686,139,235),(1116713,142,233),(1116722,142,231),(1116732,142,229),(1116733,142,225),(1116743,142,222),(1116754,142,213),(1116765,141,204),(1116776,140,197),(1116777,140,193),(1116789,141,182),(1116801,141,173),(1116814,143,163),(1116826,145,153),(1116838,147,146),(1116851,148,141),(1116863,148,139),(1116876,148,137),(1116890,146,136),(1116904,144,138),(1116918,138,145),(1116933,135,148),(1116934,132,153),(1116951,123,165),(1116966,113,178),(1116982,104,193),(1116999,97,209),(1117015,90,222),(1117032,85,233),(1117048,81,239),(1117084,81,236)]),
 ("delete_mark",[(1440612,125,139),(1440624,128,142),(1440633,130,145),(1440639,131,147),(1440641,133,149),(1440648,136,154),(1440654,138,157),(1440661,143,162),(1440667,145,165),(1440674,150,172),(1440681,154,176),(1440688,158,182),(1440695,160,185),(1440696,161,187),(1440707,165,193),(1440715,167,197),(1440724,170,200),(1440734,172,204),(1440742,173,206),(1440751,175,209),(1440764,176,212),(1440785,176,214),(1440797,174,214),(1440806,171,213),(1440817,169,212),(1440827,163,212),(1440838,157,211),(1440849,150,211),(1440861,144,211),(1440862,142,211),(1440874,133,211),(1440886,124,212),(1440898,115,214),(1440911,106,215),(1440923,97,220),(1440936,89,225),(1440949,82,229),(1440962,77,232),(1440993,76,230),(1441006,77,228),(1441020,83,223),(1441034,92,215),(1441049,100,208),(1441050,103,206),(1441066,115,197),(1441081,127,185),(1441099,139,174),(1441114,150,164),(1441130,161,154),(1441146,170,144),(1441161,173,140),(1441162,175,138),(1441180,178,136)]),
 ("pigtail",[(1077325,75,225),(1077356,77,227),(1077365,79,226),(1077372,81,225),(1077381,84,224),(1077389,87,224),(1077397,90,223),(1077405,94,222),(1077413,97,221),(1077421,100,220),(1077429,103,218),(1077437,107,215),(1077445,110,213),(1077454,113,211),(1077462,116,207),(1077470,120,203),(1077479,123,200),(1077487,127,195),(1077496,130,192),(1077505,133,189),(1077514,137,183),(1077523,139,179),(1077533,140,177),(1077534,142,175),(1077547,144,168),(1077557,145,164),(1077560,145,162),(1077573,146,155),(1077584,145,150),(1077595,144,144),(1077607,143,142),(1077618,142,139),(1077630,139,135),(1077651,137,132),(1077676,134,130),(1077693,132,131),(1077705,129,130),(1077718,126,131),(1077731,124,132),(1077744,120,137),(1077758,117,143),(1077772,114,149),(1077786,112,154),(1077801,111,158),(1077802,111,161),(1077818,110,168),(1077833,111,174),(1077848,112,182),(1077850,113,185),(1077867,115,192),(1077883,117,198),(1077899,121,204),(1077916,125,210),(1077932,131,214),(1077933,135,215),(1077953,144,218),(1077971,153,220),(1077989,163,220),(1078007,170,220),(1078025,176,219),(1078044,180,218),(1078061,182,218),(1078080,185,217)]),
 ("pigtail",[(1071946,72,237),(1071958,75,237),(1071966,77,237),(1071975,79,237),(1071983,82,237),(1071991,84,239),(1071999,87,237),(1072007,91,236),(1072015,94,235),(1072023,97,233),(1072031,101,231),(1072039,103,230),(1072048,106,227),(1072057,109,224),(1072066,112,220),(1072068,114,217),(1072078,116,214),(1072088,119,209),(1072098,121,206),(1072100,124,202),(1072111,126,197),(1072112,128,194),(1072125,131,189),(1072136,134,182),(1072148,136,175),(1072160,136,170),(1072172,136,167),(1072185,135,162),(1072195,132,159),(1072207,131,157),(1072221,127,156),(1072234,124,155),(1072248,119,154),(1072262,114,155),(1072277,109,157),(1072291,106,160),(1072292,104,162),(1072307,100,168),(1072322,98,171),(1072338,96,177),(1072340,96,181),(1072355,96,184),(1072356,96,188),(1072372,98,194),(1072389,101,201),(1072406,107,207),(1072408,110,209),(1072425,115,213),(1072446,124,220),(1072463,134,225),(1072483,144,227),(1072484,146,227),(1072502,155,226),(1072504,157,226),(1072524,165,222),(1072544,171,218),(1072564,176,215),(1072583,178,213)]),
 ("arrow",[(1349546,66,218),(1349579,68,217),(1349587,70,215),(1349596,73,215),(1349603,76,213),(1349610,78,211),(1349616,82,209),(1349623,84,208),(1349624,87,208),(1349632,91,206),(1349640,96,204),(1349649,102,201),(1349658,107,198),(1349667,113,194),(1349668,116,193),(1349680,126,188),(1349689,132,184),(1349700,142,180),(1349711,151,175),(1349722,157,171),(1349732,164,168),(1349744,171,164),(1349755,175,162),(1349766,178,160),(1349768,181,158),(1349780,183,158),(1349790,185,155),(1349804,185,153),(1349820,183,151),(1349831,181,150),(1349842,179,150),(1349854,173,149),(1349865,169,148),(1349877,162,149),(1349889,159,150),(1349901,152,151),(1349914,148,152),(1349927,143,152),(1349939,141,154),(1349984,143,153),(1349997,146,153),(1350011,153,153),(1350012,155,152),(1350029,164,150),(1350044,173,149),(1350059,180,148),(1350060,182,148),(1350078,187,146),(1350097,193,145),(1350112,195,145),(1350128,197,146),(1350143,196,149),(1350160,193,157),(1350176,189,162),(1350198,184,168),(1350214,180,175),(1350216,178,178),(1350236,174,183),(1350255,172,184)]),
 ("delete_mark",[(1444160,120,135),(1444170,121,138),(1444176,123,141),(1444184,125,145),(1444190,126,147),(1444197,130,154),(1444204,134,161),(1444211,136,164),(1444213,139,168),(1444222,144,175),(1444230,148,182),(1444238,152,189),(1444246,156,194),(1444254,159,199),(1444263,162,204),(1444272,163,207),(1444273,166,209),(1444284,169,214),(1444293,171,216),(1444303,173,218),(1444313,175,219),(1444323,177,220),(1444361,174,219),(1444371,172,219),(1444382,168,217),(1444394,160,216),(1444405,154,215),(1444417,147,214),(1444429,140,216),(1444441,132,216),(1444453,123,216),(1444466,114,220),(1444478,106,219),(1444491,98,221),(1444505,90,224),(1444506,88,224),(1444520,81,225),(1444533,78,226),(1444534,75,225),(1444548,73,226),(1444562,71,226),(1444585,71,224),(1444600,74,222),(1444614,80,217),(1444630,90,207),(1444646,101,199),(1444662,113,190),(1444678,125,177),(1444694,138,166),(1444711,149,155),(1444728,156,147),(1444744,165,138),(1444761,169,135),(1444778,171,133)]),
 ("star",[(686079,80,254),(686144,80,251),(686165,82,249),(686173,83,247),(686181,84,244),(686190,85,240),(686198,86,237),(686206,88,234),(686213,88,232),(686215,89,229),(686223,90,225),(686230,92,220),(686239,93,216),(686247,94,212),(686256,95,207),(686265,96,202),(686274,97,197),(686275,97,194),(686284,98,188),(686294,100,179),(686304,101,176),(686315,104,167),(686325,106,163),(686336,108,156),(686347,108,148),(686358,109,145),(686359,109,142),(686371,109,135),(686383,109,130),(686394,109,125),(686406,109,122),(686418,109,118),(686431,109,116),(686458,109,114),(686474,109,116),(686487,110,119),(686500,112,125),(686514,113,130),(686515,114,133),(686531,117,141),(686545,119,150),(686547,120,153),(686562,123,165),(686581,127,175),(686596,131,186),(686611,134,195),(686627,138,204),(686643,142,212),(686659,145,221),(686675,148,227),(686691,151,235),(686708,154,240),(686725,156,244),(686742,158,249),(686759,158,251),(686775,158,253),(686794,158,255),(686818,159,257),(686874,156,255),(686892,154,253),(686910,149,249),(686929,144,245),(686948,136,239),(686967,131,234),(686986,121,227),(687006,112,220),(687026,107,217),(687027,104,214),(687049,97,210),(687069,88,203),(687090,78,195),(687111,72,188),(687132,64,181),(687153,59,177),(687155,57,176),(687178,53,173),(687199,50,174),(687263,53,174),(687285,58,175),(687308,67,176),(687331,78,176),(687332,81,176),(687356,94,176),(687381,108,174),(687406,122,172),(687430,136,171),(687432,138,171),(687458,151,169),(687484,162,168),(687508,171,168),(687533,179,168),(687558,183,167),(687588,185,167),(687647,184,169),(687673,179,172),(687699,172,178),(687700,169,181),(687728,160,188),(687757,145,199),(687785,133,207),(687812,123,215),(687813,119,218),(687842,110,224),(687873,98,233),(687901,91,237),(687929,86,240),(687930,84,242),(687960,80,248),(687989,78,252),(688018,77,254)]),
 ("star",[(661037,72,252),(661102,74,250),(661123,76,247),(661131,77,245),(661143,78,243),(661150,79,240),(661157,81,235),(661164,82,232),(661171,85,227),(661179,86,225),(661180,87,223),(661190,89,217),(661198,91,212),(661200,92,209),(661209,95,204),(661218,97,198),(661227,100,190),(661236,102,183),(661246,104,177),(661248,105,175),(661258,107,169),(661260,107,165),(661272,110,155),(661282,112,151),(661284,113,148),(661296,115,140),(661307,118,133),(661319,119,131),(661320,121,127),(661333,123,123),(661345,126,116),(661357,129,111),(661369,131,109),(661382,133,106),(661395,135,105),(661411,137,105),(661423,139,107),(661437,140,112),(661451,141,117),(661452,141,120),(661466,143,126),(661468,143,129),(661484,146,141),(661499,150,154),(661515,154,167),(661531,157,176),(661532,158,179),(661550,162,188),(661566,165,200),(661568,166,202),(661585,169,215),(661602,171,224),(661604,172,227),(661624,176,237),(661641,178,245),(661658,179,250),(661660,180,253),(661679,181,257),(661715,180,259),(661767,178,257),(661785,177,255),(661803,173,249),(661822,168,245),(661824,165,243),(661844,156,235),(661864,145,228),(661884,134,220),(661905,123,213),(661926,110,204),(661928,108,203),(661950,95,195),(661973,81,182),(661994,70,174),(661995,68,173),(662019,61,168),(662020,57,167),(662044,48,160),(662066,43,157),(662089,41,156),(662127,43,155),(662150,49,155),(662173,62,154),(662196,75,155),(662220,91,154),(662244,107,154),(662269,123,153),(662293,140,152),(662318,155,150),(662344,171,150),(662369,182,149),(662394,192,149),(662418,200,148),(662420,202,148),(662446,204,147),(662476,203,150),(662501,199,152),(662528,192,159),(662554,183,166),(662583,170,174),(662585,167,177),(662613,153,187),(662641,139,197),(662669,124,207),(662697,111,218),(662725,101,225),(662753,90,235),(662781,83,241),(662810,80,245),(662838,77,249),(662867,75,254),(662896,71,258),(662925,70,256)]),
 ("question_mark",[(757269,110,147),(757297,109,145),(757313,108,142),(757342,107,140),(757350,107,138),(757369,107,136),(757377,107,133),(757393,107,131),(757405,108,129),(757417,109,126),(757425,111,124),(757442,113,121),(757454,115,121),(757463,117,120),(757472,119,118),(757482,122,118),(757490,124,117),(757499,126,116),(757510,128,116),(757522,131,118),(757534,133,120),(757546,135,122),(757556,136,124),(757567,139,126),(757577,140,128),(757589,141,131),(757600,143,132),(757611,145,135),(757623,147,137),(757634,148,139),(757647,150,144),(757663,151,147),(757675,151,149),(757688,151,151),(757700,151,153),(757713,149,159),(757727,148,161),(757740,144,166),(757754,142,167),(757768,138,171),(757784,134,173),(757798,130,173),(757813,126,174),(757828,124,175),(757842,122,174),(757857,120,176),(757872,118,178),(757893,118,181),(757909,118,183),(757924,118,185),(757926,118,187),(757942,118,190),(757958,119,195),(757974,120,200),(757990,121,202),(758007,121,207),(758023,122,211),(758040,123,214),(758070,123,216),(758102,121,218)]),
 ("rectangle",[(904245,84,147),(904282,84,150),(904290,84,153),(904299,84,156),(904307,84,159),(904313,85,161),(904319,85,166),(904326,85,168),(904333,86,172),(904340,86,177),(904347,86,180),(904348,87,182),(904357,87,184),(904364,88,192),(904372,88,198),(904380,88,203),(904388,89,208),(904397,89,214),(904406,90,219),(904414,90,221),(904415,91,224),(904426,92,228),(904427,92,230),(904439,93,234),(904448,94,236),(904476,94,238),(904556,96,237),(904576,98,235),(904596,100,236),(904608,103,235),(904620,105,236),(904632,108,236),(904633,111,236),(904646,114,236),(904659,121,237),(904661,123,237),(904676,133,236),(904690,140,236),(904704,147,236),(904705,150,237),(904721,159,238),(904736,168,237),(904751,174,239),(904766,179,237),(904781,186,237),(904796,192,237),(904814,196,237),(904829,198,237),(904857,200,237),(904922,198,235),(904938,197,233),(904954,196,228),(904971,196,222),(904988,195,214),(905005,193,203),(905023,192,191),(905040,190,180),(905059,189,169),(905077,188,160),(905095,187,153),(905113,187,147),(905132,186,139),(905150,186,137),(905151,185,134),(905178,185,132),(905210,183,131),(905230,180,131),(905250,178,131),(905269,173,132),(905289,168,132),(905291,164,132),(905313,154,132),(905338,140,133),(905339,137,133),(905362,119,136),(905384,106,135),(905406,96,135),(905428,89,135),(905450,82,135)]),
 ("rectangle",[(917691,88,145),(917714,88,147),(917726,87,150),(917734,88,154),(917740,88,156),(917746,88,159),(917754,89,163),(917760,89,166),(917767,89,169),(917773,89,171),(917774,90,174),(917782,89,179),(917790,90,184),(917798,90,189),(917807,91,193),(917815,91,199),(917824,92,204),(917833,92,210),(917842,93,215),(917843,93,217),(917852,94,220),(917864,94,224),(917873,95,227),(917875,95,231),(917885,96,234),(917895,96,236),(917906,96,238),(917917,96,240),(917946,97,242),(918040,99,243),(918055,101,243),(918068,103,244),(918080,105,245),(918093,107,245),(918105,112,246),(918118,115,245),(918119,118,245),(918132,121,246),(918146,128,245),(918159,134,246),(918173,142,246),(918187,147,245),(918201,155,245),(918216,163,245),(918230,169,244),(918231,171,243),(918246,176,243),(918261,183,242),(918276,188,242),(918292,193,243),(918307,195,243),(918323,197,244),(918339,199,244),(918354,201,244),(918375,203,246),(918412,205,248),(918516,206,246),(918533,206,244),(918550,206,239),(918568,205,234),(918585,205,225),(918603,204,219),(918604,204,217),(918623,203,207),(918624,203,204),(918644,202,197),(918645,202,193),(918664,201,186),(918683,200,178),(918703,199,171),(918704,199,167),(918725,199,157),(918745,199,148),(918765,199,140),(918786,199,136),(918806,199,134),(918829,198,132),(918857,196,131),(918878,194,132),(918899,192,133),(918920,189,134),(918942,185,132),(918964,180,132),(918986,173,131),(919009,165,131),(919032,156,130),(919055,146,130),(919079,136,131),(919103,127,132),(919127,117,133),(919151,109,132),(919176,102,131),(919200,96,132),(919225,91,133),(919249,89,134),(919274,87,133)]),
 ("left_sq_bracket",[(967627,134,125),(967682,132,126),(967706,130,125),(967722,128,124),(967734,125,124),(967747,123,124),(967755,121,124),(967767,118,123),(967779,115,123),(967795,113,124),(967815,111,125),(967836,109,124),(967859,107,123),(967879,105,123),(967900,102,122),(967917,100,122),(967934,98,123),(967958,96,125),(967994,94,126),(968013,93,128),(968038,93,130),(968048,93,133),(968058,93,135),(968069,94,137),(968080,94,143),(968091,94,149),(968102,94,153),(968113,94,157),(968125,94,160),(968127,94,163),(968139,94,166),(968151,94,171),(968164,94,178),(968177,94,184),(968189,94,190),(968191,94,192),(968205,94,199),(968218,94,205),(968232,94,211),(968247,94,216),(968260,94,219),(968275,93,224),(968289,93,226),(968306,93,228),(968322,93,231),(968337,92,233),(968352,92,235),(968367,92,239),(968383,91,241),(968399,92,243),(968427,91,245),(968452,91,247),(968524,94,246),(968541,96,247),(968557,98,247),(968574,101,248),(968591,106,249),(968593,109,250),(968612,115,251),(968630,121,251),(968651,131,251),(968669,137,253),(968688,145,253),(968707,147,253),(968726,149,253),(968745,151,253)]),
 ("pigtail",[(1066054,71,224),(1066075,71,226),(1066083,72,230),(1066096,75,231),(1066103,78,232),(1066112,81,232),(1066120,85,232),(1066127,87,232),(1066133,92,230),(1066141,97,228),(1066148,99,227),(1066149,102,225),(1066158,107,222),(1066166,113,218),(1066175,119,213),(1066183,125,209),(1066191,131,203),(1066193,134,200),(1066203,139,193),(1066212,145,185),(1066222,149,178),(1066231,151,172),(1066232,151,170),(1066244,151,161),(1066255,150,156),(1066265,148,152),(1066275,145,150),(1066276,141,148),(1066289,135,149),(1066300,129,152),(1066312,125,155),(1066313,123,157),(1066328,116,171),(1066341,113,183),(1066353,113,197),(1066366,115,210),(1066379,119,222),(1066393,129,233),(1066406,138,238),(1066420,154,239),(1066434,165,237),(1066449,176,232),(1066463,182,226),(1066464,183,224),(1066481,185,220)]),
 ("question_mark",[(750137,106,157),(750148,106,155),(750156,105,153),(750168,104,151),(750181,103,149),(750193,102,146),(750201,102,144),(750209,102,142),(750217,102,139),(750233,102,137),(750248,102,135),(750265,104,132),(750277,106,130),(750289,108,128),(750298,110,127),(750306,112,125),(750314,114,125),(750323,116,123),(750333,119,123),(750343,121,123),(750354,125,124),(750364,127,124),(750375,130,124),(750387,134,126),(750398,136,127),(750412,141,129),(750423,143,130),(750434,146,131),(750445,148,133),(750457,150,135),(750470,152,138),(750484,155,143),(750497,157,148),(750510,158,150),(750523,158,153),(750538,158,157),(750552,158,159),(750568,156,163),(750583,153,168),(750598,150,170),(750614,145,173),(750629,141,175),(750645,138,175),(750646,136,176),(750663,131,179),(750680,127,179),(750697,125,180),(750698,122,182),(750716,120,183),(750732,118,184),(750749,116,186),(750773,115,188),(750791,115,193),(750811,116,198),(750831,117,202),(750851,118,209),(750872,119,214),(750891,120,216),(750961,119,213)]),
 ("delete_mark",[(1445060,129,134),(1445068,129,137),(1445076,129,141),(1445081,130,143),(1445087,131,147),(1445088,132,149),(1445096,134,154),(1445103,136,157),(1445109,140,164),(1445116,141,167),(1445117,143,170),(1445126,147,176),(1445133,151,182),(1445141,154,187),(1445148,157,192),(1445157,159,197),(1445165,162,202),(1445173,164,206),(1445182,167,211),(1445191,169,214),(1445200,170,216),(1445201,171,218),(1445213,173,221),(1445232,174,223),(1445253,171,221),(1445264,169,219),(1445275,166,218),(1445286,159,215),(1445298,153,213),(1445310,146,215),(1445322,139,215),(1445334,130,216),(1445346,121,219),(1445359,113,220),(1445372,104,222),(1445385,96,226),(1445386,94,226),(1445399,88,229),(1445413,85,229),(1445414,82,229),(1445427,80,230),(1445449,79,228),(1445463,80,226),(1445478,84,220),(1445496,92,213),(1445512,101,204),(1445528,110,195),(1445530,113,193),(1445547,123,185),(1445565,137,176),(1445581,147,167),(1445597,157,159),(1445614,164,151),(1445630,170,145),(1445646,174,140),(1445662,175,137),(1445680,176,135)]),
 ("x",[(1122233,89,159),(1122244,92,162),(1122252,94,165),(1122261,96,168),(1122268,99,170),(1122274,101,172),(1122282,106,175),(1122289,111,178),(1122297,116,183),(1122305,121,187),(1122314,126,192),(1122323,131,196),(1122332,135,201),(1122333,137,203),(1122343,141,207),(1122345,143,209),(1122355,146,214),(1122363,149,216),(1122374,151,219),(1122383,153,221),(1122392,155,223),(1122405,157,224),(1122417,158,221),(1122426,159,219),(1122436,160,216),(1122447,160,211),(1122457,159,203),(1122468,158,197),(1122479,157,188),(1122490,157,178),(1122502,157,167),(1122514,157,156),(1122526,158,149),(1122538,159,141),(1122550,160,133),(1122562,161,127),(1122575,161,125),(1122589,160,123),(1122601,158,123),(1122614,155,125),(1122628,151,130),(1122629,150,132),(1122645,145,138),(1122646,143,141),(1122662,135,153),(1122677,125,167),(1122692,118,178),(1122693,116,182),(1122711,109,198),(1122728,104,211),(1122744,100,219),(1122759,98,222),(1122775,95,227),(1122797,94,229),(1122817,96,226),(1122833,98,225),(1122850,99,223)]),
 ("right_curly_brace",[(1209767,130,128),(1209809,132,127),(1209816,134,126),(1209825,136,125),(1209832,138,126),(1209840,140,125),(1209848,142,125),(1209860,145,125),(1209868,147,125),(1209880,150,125),(1209892,152,126),(1209901,154,128),(1209909,155,131),(1209917,157,133),(1209928,159,135),(1209937,159,139),(1209946,159,143),(1209955,160,145),(1209964,160,148),(1209974,159,150),(1209985,158,153),(1210009,156,154),(1210025,153,154),(1210037,151,155),(1210048,149,155),(1210059,147,156),(1210069,142,159),(1210081,140,160),(1210092,138,163),(1210103,135,166),(1210115,134,168),(1210127,132,170),(1210140,132,172),(1210153,132,174),(1210166,134,176),(1210180,136,176),(1210193,138,176),(1210207,143,177),(1210220,146,178),(1210221,149,177),(1210235,151,178),(1210249,157,179),(1210263,162,178),(1210278,166,178),(1210293,169,178),(1210309,171,177),(1210356,169,176),(1210372,167,176),(1210373,164,175),(1210389,159,177),(1210405,153,178),(1210420,148,180),(1210436,144,181),(1210452,142,182),(1210469,140,183),(1210485,138,184),(1210525,138,186),(1210541,140,188),(1210558,142,189),(1210575,144,190),(1210593,149,195),(1210611,153,198),(1210629,157,203),(1210648,161,209),(1210666,165,215),(1210684,168,220),(1210686,169,223),(1210705,169,225),(1210706,169,228),(1210725,168,230),(1210726,166,233),(1210746,161,235),(1210768,150,238),(1210789,140,240),(1210790,138,241),(1210811,129,246),(1210835,123,247),(1210856,121,248)]),
 ("delete_mark",[(1443240,124,137),(1443257,125,140),(1443264,126,144),(1443270,127,146),(1443277,129,148),(1443284,131,152),(1443285,133,156),(1443294,136,163),(1443301,140,170),(1443309,144,177),(1443316,147,184),(1443324,151,189),(1443333,155,194),(1443341,158,199),(1443350,161,203),(1443359,164,207),(1443371,167,213),(1443380,170,217),(1443389,172,221),(1443399,174,223),(1443408,176,225),(1443460,174,226),(1443470,172,225),(1443481,169,223),(1443491,167,223),(1443493,164,223),(1443505,158,222),(1443515,153,221),(1443516,151,221),(1443530,143,221),(1443541,135,220),(1443553,126,220),(1443565,118,220),(1443577,110,219),(1443589,102,218),(1443602,95,218),(1443615,89,217),(1443627,86,215),(1443628,83,215),(1443643,80,215),(1443657,78,214),(1443692,80,212),(1443706,83,211),(1443720,94,204),(1443736,104,199),(1443751,112,193),(1443766,125,183),(1443781,137,171),(1443797,149,159),(1443813,159,150),(1443829,167,143),(1443844,173,138),(1443860,175,137)]),
 ("arrow",[(1353192,52,237),(1353211,55,236),(1353219,58,234),(1353228,62,231),(1353234,64,230),(1353241,69,226),(1353249,75,223),(1353256,81,220),(1353265,88,217),(1353273,94,213),(1353281,100,210),(1353289,107,206),(1353298,113,202),(1353307,119,200),(1353309,122,198),(1353319,128,195),(1353321,131,194),(1353332,137,191),(1353333,140,189),(1353345,149,186),(1353354,155,183),(1353364,162,178),(1353374,167,175),(1353384,172,171),(1353394,175,169),(1353404,177,168),(1353415,179,165),(1353435,179,163),(1353451,176,161),(1353462,174,161),(1353474,172,160),(1353485,166,160),(1353497,160,160),(1353509,154,160),(1353522,148,162),(1353535,142,164),(1353536,140,165),(1353551,134,168),(1353564,132,169),(1353577,129,171),(1353620,132,170),(1353633,136,169),(1353647,143,166),(1353649,145,165),(1353665,155,162),(1353681,165,159),(1353697,175,157),(1353713,183,157),(1353731,188,156),(1353732,191,156),(1353751,194,156),(1353766,196,157),(1353782,197,159),(1353798,196,162),(1353815,191,167),(1353833,186,172),(1353850,181,178),(1353867,174,186),(1353885,169,190),(1353903,165,195),(1353921,160,199),(1353948,158,201),(1353988,160,201)]),
 ("left_sq_bracket",[(982138,143,128),(982158,140,127),(982170,138,127),(982177,136,125),(982186,133,125),(982194,131,125),(982203,128,124),(982211,125,123),(982219,123,123),(982227,120,123),(982235,117,123),(982243,114,122),(982251,112,122),(982260,109,122),(982268,107,122),(982280,105,121),(982292,102,121),(982305,100,121),(982324,98,122),(982349,97,124),(982374,97,127),(982394,98,129),(982406,98,132),(982416,99,135),(982427,100,139),(982438,100,145),(982449,101,150),(982451,101,153),(982463,101,160),(982475,101,167),(982487,102,173),(982499,102,179),(982512,102,183),(982524,102,190),(982537,102,194),(982539,102,197),(982552,102,201),(982566,103,207),(982579,103,212),(982593,103,217),(982607,103,222),(982623,103,227),(982638,104,230),(982652,104,232),(982667,104,234),(982682,104,236),(982697,104,238),(982762,105,240),(982798,107,242),(982818,109,242),(982834,111,241),(982850,114,240),(982867,118,240),(982883,123,240),(982900,129,241),(982918,134,241),(982919,137,242),(982938,143,242),(982957,145,241),(982959,147,241),(983043,149,241),(983068,151,240)]),
 ("question_mark",[(747439,112,147),(747459,110,146),(747475,108,144),(747487,107,142),(747500,105,140),(747516,105,137),(747528,104,135),(747540,104,132),(747557,105,129),(747569,107,127),(747577,108,125),(747589,111,122),(747597,113,120),(747605,115,119),(747614,117,118),(747622,119,118),(747633,121,118),(747645,124,117),(747658,127,118),(747667,129,118),(747676,131,118),(747686,133,119),(747697,135,121),(747707,139,121),(747718,141,123),(747729,143,125),(747731,145,126),(747743,148,128),(747754,150,130),(747765,152,132),(747776,154,134),(747787,156,138),(747799,157,140),(747811,159,143),(747823,161,148),(747835,161,150),(747847,162,154),(747860,161,159),(747873,161,161),(747886,160,166),(747899,159,171),(747913,157,174),(747926,156,176),(747940,154,178),(747957,149,180),(747971,144,182),(747985,139,184),(748000,134,185),(748015,129,185),(748030,125,187),(748046,123,187),(748062,119,190),(748082,116,190),(748098,115,193),(748114,115,195),(748130,116,198),(748147,117,201),(748163,118,206),(748180,118,210),(748198,119,216),(748215,119,219),(748232,119,221),(748250,119,225)]),
 ("circle",[(1156877,138,143),(1156883,136,140),(1156889,132,135),(1156895,129,134),(1156897,127,132),(1156904,122,129),(1156911,119,130),(1156913,117,130),(1156921,111,133),(1156929,105,137),(1156936,99,143),(1156944,94,149),(1156953,89,156),(1156961,86,163),(1156970,82,170),(1156979,80,177),(1156981,79,181),(1156990,78,189),(1157000,77,201),(1157010,78,207),(1157020,80,214),(1157021,82,218),(1157033,88,227),(1157044,92,231),(1157045,94,233),(1157058,102,239),(1157069,111,241),(1157081,121,244),(1157093,131,243),(1157105,141,241),(1157118,150,235),(1157130,160,229),(1157143,168,221),(1157156,179,208),(1157169,184,200),(1157183,187,189),(1157196,185,177),(1157210,180,167),(1157225,171,154),(1157239,163,147),(1157254,152,141),(1157268,140,140),(1157284,131,143),(1157285,129,143),(1157300,122,147)]),
 ("check",[(852806,93,181),(852839,95,180),(852851,97,182),(852864,99,183),(852876,102,184),(852884,103,186),(852897,105,188),(852909,106,191),(852917,107,193),(852925,108,195),(852933,109,198),(852941,111,201),(852949,112,203),(852958,114,206),(852965,116,209),(852973,118,212),(852981,120,215),(852989,122,216),(852999,124,219),(853008,126,221),(853018,127,223),(853033,129,225),(853052,130,227),(853105,130,229),(853129,129,227),(853153,129,225),(853165,129,223),(853177,129,221),(853189,129,219),(853201,130,214),(853213,130,211),(853214,130,207),(853226,130,202),(853239,131,193),(853253,133,185),(853254,134,183),(853269,138,172),(853284,141,165),(853298,148,156),(853312,152,149),(853326,159,141),(853341,164,135),(853355,168,129),(853370,171,125),(853387,172,123),(853469,173,121),(853485,171,122)]),
 ("right_sq_bracket",[(798098,115,133),(798121,117,131),(798137,119,130),(798150,122,131),(798158,124,129),(798166,126,131),(798174,129,131),(798182,132,130),(798190,135,130),(798198,137,129),(798206,140,130),(798214,142,130),(798222,144,130),(798231,146,129),(798238,148,130),(798250,151,130),(798262,153,131),(798274,156,131),(798291,158,133),(798303,158,135),(798313,159,137),(798323,159,139),(798335,158,147),(798347,159,154),(798360,159,163),(798372,160,174),(798385,162,186),(798398,164,196),(798410,166,204),(798412,166,207),(798426,167,214),(798440,167,219),(798454,168,226),(798468,168,228),(798482,169,233),(798516,167,232),(798530,165,232),(798545,160,229),(798561,151,227),(798578,142,228),(798595,133,228),(798611,125,230),(798628,119,231),(798645,114,232)]),
 ("left_curly_brace",[(1399726,139,123),(1399733,137,123),(1399741,134,123),(1399749,131,123),(1399757,127,123),(1399766,124,125),(1399774,120,127),(1399782,117,130),(1399788,115,132),(1399795,112,136),(1399803,108,139),(1399811,106,143),(1399819,104,147),(1399828,102,149),(1399835,101,151),(1399843,101,154),(1399855,101,156),(1399866,103,157),(1399874,105,158),(1399883,108,158),(1399893,111,157),(1399903,114,157),(1399912,117,156),(1399922,120,155),(1399923,123,155),(1399935,125,156),(1399945,128,158),(1399955,129,160),(1399966,129,164),(1399978,129,167),(1399989,128,170),(1400000,126,172),(1400012,122,177),(1400023,119,178),(1400035,114,179),(1400047,111,181),(1400059,109,181),(1400073,107,179),(1400086,106,181),(1400105,108,179),(1400118,111,180),(1400132,113,180),(1400145,119,181),(1400160,123,181),(1400174,125,182),(1400197,126,185),(1400212,125,188),(1400227,121,193),(1400242,116,199),(1400256,113,206),(1400272,110,211),(1400287,108,217),(1400303,109,222),(1400319,110,224),(1400335,115,227),(1400351,120,228),(1400367,126,230),(1400384,135,229),(1400401,139,228)]),
 ("v",[(1023267,87,156),(1023312,89,158),(1023320,91,161),(1023329,93,163),(1023337,94,165),(1023345,96,167),(1023353,98,170),(1023361,100,172),(1023369,102,175),(1023377,103,177),(1023385,106,181),(1023386,107,184),(1023396,110,190),(1023405,113,195),(1023406,115,197),(1023417,118,203),(1023418,120,206),(1023429,122,208),(1023430,125,212),(1023442,128,217),(1023452,131,221),(1023454,133,224),(1023465,135,229),(1023475,137,231),(1023486,139,237),(1023497,141,239),(1023508,142,241),(1023525,143,244),(1023639,144,242),(1023655,145,240),(1023667,145,238),(1023679,145,233),(1023692,146,231),(1023704,148,223),(1023718,149,216),(1023731,151,208),(1023732,151,205),(1023746,153,197),(1023748,153,194),(1023766,156,182),(1023768,157,180),(1023783,159,175),(1023784,160,173),(1023800,163,167),(1023815,165,164),(1023830,166,162),(1023832,167,159),(1023848,167,157),(1023863,168,154),(1023926,167,156)]),
 ("pigtail",[(1075836,84,226),(1075851,86,227),(1075864,89,228),(1075872,91,228),(1075880,94,228),(1075889,98,227),(1075897,101,226),(1075903,103,225),(1075910,107,223),(1075918,110,221),(1075925,112,219),(1075926,114,218),(1075936,118,216),(1075944,122,212),(1075952,126,208),(1075953,128,206),(1075964,132,202),(1075972,136,198),(1075974,138,196),(1075985,141,192),(1075994,147,187),(1076004,150,182),(1076005,151,179),(1076017,154,175),(1076027,156,172),(1076038,159,165),(1076048,159,161),(1076049,159,159),(1076061,159,156),(1076072,158,153),(1076073,157,151),(1076087,157,148),(1076099,154,142),(1076111,153,140),(1076124,149,139),(1076137,146,138),(1076138,143,138),(1076152,140,140),(1076153,137,141),(1076169,131,146),(1076183,129,149),(1076197,124,154),(1076211,120,162),(1076225,117,167),(1076240,116,172),(1076255,115,181),(1076270,115,187),(1076286,117,196),(1076301,118,199),(1076302,119,203),(1076322,122,209),(1076339,126,213),(1076355,130,216),(1076372,136,221),(1076388,142,224),(1076389,144,226),(1076410,151,229),(1076428,158,229),(1076446,167,228),(1076463,172,226),(1076481,174,226),(1076482,177,224),(1076503,179,224),(1076522,181,224)]),
 ("rectangle",[(915766,76,150),(915803,75,152),(915815,75,154),(915824,75,157),(915831,75,160),(915837,75,162),(915844,76,166),(915850,76,168),(915851,76,170),(915859,77,172),(915866,77,177),(915867,77,179),(915876,78,184),(915884,78,190),(915892,79,195),(915901,79,201),(915909,80,206),(915918,81,211),(915919,81,213),(915930,82,219),(915940,82,226),(915949,83,230),(915959,84,236),(915969,84,238),(915979,86,243),(915990,86,246),(916000,87,248),(916011,87,250),(916031,87,253),(916059,88,256),(916075,88,258),(916132,89,260),(916177,91,258),(916193,93,256),(916210,95,256),(916223,98,255),(916235,100,254),(916248,105,253),(916261,108,252),(916275,114,252),(916288,121,252),(916302,127,251),(916316,135,250),(916331,143,250),(916346,149,249),(916361,155,248),(916376,165,247),(916391,171,247),(916407,178,247),(916422,184,247),(916438,190,246),(916454,195,246),(916471,199,246),(916487,202,247),(916503,204,247),(916520,206,247),(916583,204,244),(916611,202,243),(916628,201,240),(916646,200,234),(916664,198,226),(916682,197,217),(916701,196,207),(916720,195,196),(916739,194,187),(916740,194,184),(916761,193,173),(916780,193,165),(916800,192,160),(916830,191,155),(916851,189,151),(916895,187,149),(916918,186,147),(916939,182,143),(916961,178,143),(916982,171,142),(917006,162,140),(917028,155,138),(917050,144,139),(917071,135,137),(917094,125,138),(917119,116,138),(917143,108,139),(917166,102,140),(917167,100,141),(917190,95,143),(917191,92,144),(917217,88,144),(917240,86,144),(917263,83,145)]),
 ("caret",[(1255877,77,241),(1255914,77,239),(1255930,78,236),(1255938,80,234),(1255947,81,230),(1255953,82,228),(1255960,83,226),(1255967,87,219),(1255974,88,216),(1255976,88,214),(1255984,91,210),(1255992,93,204),(1256001,94,198),(1256009,96,191),(1256018,98,185),(1256026,100,179),(1256027,101,176),(1256040,103,167),(1256049,105,160),(1256059,107,153),(1256069,109,148),(1256080,112,141),(1256090,114,138),(1256100,115,136),(1256110,116,134),(1256112,117,131),(1256139,120,133),(1256151,122,135),(1256163,123,138),(1256174,125,140),(1256176,127,145),(1256192,132,156),(1256205,137,167),(1256218,142,179),(1256231,148,192),(1256232,150,196),(1256250,158,210),(1256264,165,221),(1256277,169,224),(1256291,174,231),(1256305,177,233)]),
 ("right_sq_bracket",[(797161,116,137),(797196,118,136),(797208,121,135),(797215,123,135),(797224,126,134),(797232,129,133),(797240,132,132),(797248,135,130),(797256,138,129),(797264,140,129),(797272,142,129),(797281,144,127),(797289,147,127),(797300,149,127),(797312,151,127),(797325,153,129),(797337,155,131),(797345,156,133),(797354,157,136),(797363,157,139),(797372,158,143),(797382,158,150),(797391,159,156),(797401,159,162),(797402,159,165),(797413,159,174),(797424,159,181),(797434,159,191),(797445,160,201),(797456,160,207),(797457,161,209),(797471,162,218),(797483,163,225),(797494,164,230),(797507,165,237),(797525,165,240),(797537,165,242),(797561,162,240),(797574,160,239),(797587,155,235),(797600,152,234),(797614,141,233),(797628,134,232),(797642,125,234),(797656,122,235),(797657,119,236),(797674,113,238),(797689,111,239)]),
 ("right_sq_bracket",[(796184,112,138),(796279,112,136),(796291,115,136),(796303,118,137),(796311,120,136),(796320,123,136),(796328,125,136),(796336,128,136),(796344,131,136),(796352,134,135),(796360,137,135),(796368,140,134),(796376,143,133),(796385,145,132),(796395,147,132),(796408,149,132),(796428,152,132),(796445,153,134),(796457,154,137),(796466,155,141),(796475,156,144),(796487,157,152),(796500,158,161),(796511,160,170),(796523,162,182),(796535,164,192),(796548,166,200),(796560,167,209),(796573,168,214),(796574,168,216),(796586,169,221),(796598,169,223),(796611,169,228),(796633,169,231),(796658,166,233),(796671,164,234),(796684,161,235),(796698,155,236),(796712,147,235),(796726,140,233),(796740,131,233),(796755,124,233),(796770,117,235),(796784,114,238),(796799,112,238)]),
 ("x",[(1115538,85,140),(1115554,86,142),(1115570,88,145),(1115579,90,147),(1115586,93,151),(1115593,94,153),(1115600,98,159),(1115607,102,165),(1115615,106,172),(1115623,111,179),(1115631,116,187),(1115640,121,194),(1115649,125,200),(1115658,131,208),(1115667,135,213),(1115677,139,217),(1115687,142,221),(1115697,145,224),(1115707,151,227),(1115716,153,227),(1115727,155,229),(1115736,157,229),(1115755,159,227),(1115776,161,225),(1115792,162,222),(1115804,162,220),(1115805,162,217),(1115817,162,211),(1115829,161,205),(1115843,160,195),(1115856,159,182),(1115870,158,169),(1115884,156,152),(1115899,156,140),(1115913,156,130),(1115927,156,122),(1115941,156,120),(1115956,154,120),(1115969,152,121),(1115983,150,123),(1115997,144,132),(1116014,139,144),(1116031,131,156),(1116048,122,174),(1116068,114,189),(1116069,113,192),(1116089,105,208),(1116107,102,216),(1116125,100,221),(1116142,98,225),(1116158,98,228)]),
 ("circle",[(1157641,144,142),(1157649,140,139),(1157654,138,137),(1157660,136,136),(1157661,133,135),(1157668,130,134),(1157674,125,132),(1157680,122,131),(1157686,116,131),(1157693,110,133),(1157700,107,136),(1157707,102,140),(1157715,97,146),(1157722,93,152),(1157730,89,159),(1157738,85,166),(1157747,82,173),(1157755,79,181),(1157764,78,189),(1157773,77,199),(1157782,77,206),(1157792,79,211),(1157802,82,219),(1157812,85,223),(1157822,91,229),(1157832,95,232),(1157843,102,237),(1157854,109,239),(1157864,115,239),(1157865,117,239),(1157877,123,239),(1157878,126,239),(1157889,136,239),(1157902,146,234),(1157914,155,229),(1157926,164,224),(1157939,172,216),(1157952,179,207),(1157953,181,204),(1157967,185,195),(1157981,187,185),(1157982,187,181),(1157996,184,170),(1157997,183,166),(1158012,176,156),(1158013,174,153),(1158028,167,142),(1158029,165,140),(1158045,155,132),(1158062,145,130),(1158078,135,133),(1158094,128,137)]),
 ("circle",[(1161885,137,138),(1161890,134,137),(1161895,132,136),(1161901,126,136),(1161906,123,136),(1161912,121,136),(1161913,118,137),(1161921,113,141),(1161928,110,143),(1161929,107,145),(1161938,103,149),(1161945,99,154),(1161953,95,160),(1161960,92,166),(1161968,90,170),(1161969,89,174),(1161979,87,182),(1161989,85,191),(1161998,85,199),(1162007,86,206),(1162016,88,213),(1162017,89,216),(1162029,95,223),(1162038,100,228),(1162049,108,233),(1162059,114,235),(1162069,124,238),(1162081,135,238),(1162091,143,236),(1162092,146,236),(1162106,156,233),(1162118,165,227),(1162129,173,222),(1162141,180,214),(1162154,185,206),(1162166,188,198),(1162179,189,187),(1162193,186,172),(1162206,181,159),(1162219,175,149),(1162220,173,145),(1162236,165,137),(1162251,158,134),(1162265,147,133),(1162280,139,135),(1162281,136,136),(1162298,130,138)]),
 ("question_mark",[(745782,108,148),(745821,107,146),(745850,106,144),(745890,104,141),(745903,104,139),(745910,104,136),(745923,104,134),(745940,105,131),(745948,106,129),(745964,108,127),(745972,110,125),(745984,112,123),(745996,114,121),(746008,117,121),(746017,119,119),(746025,121,119),(746033,123,118),(746042,125,118),(746051,128,118),(746061,130,118),(746071,133,118),(746081,138,119),(746091,140,119),(746102,142,120),(746112,146,123),(746123,148,125),(746134,151,127),(746145,153,129),(746156,156,130),(746168,157,132),(746180,159,135),(746195,161,137),(746207,161,139),(746240,161,141),(746253,161,144),(746266,159,146),(746279,158,149),(746292,156,154),(746306,154,156),(746320,153,158),(746321,151,160),(746337,149,161),(746351,147,164),(746366,145,166),(746381,140,171),(746396,138,172),(746411,135,173),(746426,133,173),(746442,129,176),(746458,127,177),(746474,125,177),(746490,123,178),(746507,121,181),(746533,119,183),(746550,119,185),(746567,120,189),(746584,120,195),(746602,121,199),(746620,122,204),(746638,123,213),(746656,123,217),(746675,123,220)]),
 ("triangle",[(1299383,137,139),(1299398,135,141),(1299406,133,144),(1299415,132,146),(1299423,130,149),(1299430,128,151),(1299437,126,155),(1299444,123,160),(1299452,120,166),(1299460,116,171),(1299469,112,177),(1299477,107,183),(1299486,102,188),(1299488,100,191),(1299497,95,195),(1299506,90,199),(1299514,86,203),(1299522,82,206),(1299523,80,209),(1299534,75,213),(1299535,73,213),(1299546,70,216),(1299556,67,219),(1299565,64,221),(1299575,61,223),(1299585,60,225),(1299621,62,226),(1299631,65,225),(1299642,67,226),(1299654,74,226),(1299666,77,227),(1299678,85,229),(1299689,91,230),(1299702,99,231),(1299714,108,232),(1299726,116,233),(1299739,125,233),(1299752,134,234),(1299765,145,233),(1299779,153,232),(1299792,160,233),(1299806,170,234),(1299820,177,235),(1299821,179,236),(1299838,186,237),(1299852,193,238),(1299867,198,239),(1299882,200,237),(1299897,202,239),(1299912,204,238),(1299930,206,234),(1299946,205,230),(1299962,202,222),(1299978,197,216),(1299994,192,207),(1300012,186,198),(1300029,179,189),(1300046,174,183),(1300063,170,178),(1300081,164,171),(1300082,161,168),(1300101,154,160),(1300119,148,155),(1300138,143,150),(1300157,138,148),(1300175,136,148)]),
 ("caret",[(1262460,79,247),(1262468,80,244),(1262477,81,242),(1262484,82,239),(1262493,84,236),(1262499,85,233),(1262506,87,229),(1262515,89,224),(1262522,91,219),(1262529,93,217),(1262530,94,213),(1262537,97,206),(1262545,100,200),(1262553,102,193),(1262561,104,187),(1262570,106,181),(1262578,108,176),(1262587,110,170),(1262596,112,164),(1262597,113,161),(1262607,114,156),(1262618,117,150),(1262628,119,145),(1262639,121,140),(1262650,125,133),(1262661,127,130),(1262672,129,129),(1262690,131,126),(1262722,133,128),(1262738,135,128),(1262750,137,130),(1262762,139,132),(1262775,141,135),(1262789,144,144),(1262791,146,148),(1262808,151,169),(1262823,158,191),(1262839,164,211),(1262854,170,224),(1262871,174,235),(1262887,176,241),(1262902,177,244),(1262918,178,246),(1262946,179,244),(1262962,178,242),(1262963,178,239)]),
 ("right_curly_brace",[(1199784,117,132),(1199814,115,132),(1199845,115,129),(1199869,117,129),(1199881,119,128),(1199893,122,127),(1199901,125,127),(1199909,127,127),(1199917,130,127),(1199925,133,129),(1199934,136,129),(1199941,138,130),(1199950,140,131),(1199963,143,134),(1199975,144,136),(1199983,145,139),(1199992,145,142),(1200001,145,145),(1200010,145,147),(1200020,145,149),(1200029,144,152),(1200040,142,157),(1200050,141,160),(1200060,139,163),(1200070,137,166),(1200072,135,167),(1200083,133,169),(1200094,131,172),(1200106,128,173),(1200118,126,176),(1200130,125,178),(1200142,125,180),(1200154,125,182),(1200156,126,184),(1200170,128,187),(1200183,130,187),(1200184,132,188),(1200197,135,189),(1200212,140,189),(1200226,145,189),(1200241,150,187),(1200256,155,186),(1200270,157,185),(1200287,159,184),(1200339,156,185),(1200354,154,185),(1200370,149,185),(1200387,145,187),(1200403,141,188),(1200420,136,191),(1200437,134,191),(1200454,131,192),(1200475,129,193),(1200495,129,195),(1200512,129,197),(1200531,131,200),(1200549,133,202),(1200568,136,206),(1200587,139,211),(1200606,142,215),(1200626,145,220),(1200646,147,225),(1200666,148,231),(1200687,147,239),(1200707,144,244),(1200728,139,248),(1200749,134,250),(1200772,126,253),(1200793,119,253),(1200815,115,253)]),
 ("caret",[(1259060,74,245),(1259071,76,242),(1259079,78,240),(1259087,79,238),(1259092,80,236),(1259098,82,234),(1259104,85,230),(1259111,86,227),(1259112,88,225),(1259120,91,221),(1259127,94,217),(1259135,96,214),(1259136,97,211),(1259146,101,204),(1259147,103,200),(1259157,106,193),(1259166,110,185),(1259174,113,177),(1259176,114,173),(1259188,117,162),(1259197,120,154),(1259206,121,149),(1259207,122,147),(1259219,125,139),(1259230,126,137),(1259240,129,128),(1259250,131,124),(1259261,132,122),(1259272,135,117),(1259283,137,115),(1259296,139,114),(1259307,141,114),(1259320,143,117),(1259331,145,120),(1259344,148,128),(1259356,150,138),(1259369,153,149),(1259382,156,163),(1259383,157,168),(1259398,160,181),(1259399,161,186),(1259416,167,203),(1259430,170,216),(1259431,171,219),(1259448,176,232),(1259463,179,239),(1259464,179,241),(1259482,182,248),(1259496,183,250)]),
 ("circle",[(1159312,127,141),(1159318,124,140),(1159324,120,139),(1159330,118,139),(1159332,116,139),(1159339,111,140),(1159346,109,141),(1159353,104,144),(1159361,100,147),(1159368,96,152),(1159376,93,157),(1159384,90,163),(1159393,87,169),(1159401,85,175),(1159410,83,181),(1159420,82,190),(1159429,82,195),(1159437,83,200),(1159446,84,205),(1159456,88,213),(1159465,91,216),(1159474,96,219),(1159485,103,222),(1159494,108,224),(1159496,111,224),(1159508,120,224),(1159526,133,223),(1159537,142,222),(1159549,152,218),(1159561,160,214),(1159573,167,210),(1159586,173,204),(1159598,178,198),(1159600,179,196),(1159614,182,188),(1159628,182,177),(1159642,178,167),(1159656,170,150),(1159670,163,138),(1159684,152,130),(1159699,143,129),(1159700,140,131),(1159717,129,136),(1159731,126,139)]),
 ("circle",[(1160987,128,148),(1160992,126,146),(1160997,124,146),(1161003,120,145),(1161008,118,145),(1161014,113,143),(1161020,111,142),(1161027,106,142),(1161034,102,141),(1161041,100,142),(1161048,97,146),(1161055,94,149),(1161063,91,154),(1161070,88,160),(1161078,86,163),(1161079,85,166),(1161089,83,174),(1161090,82,178),(1161101,81,185),(1161109,81,194),(1161111,81,197),(1161121,82,205),(1161131,84,215),(1161141,86,220),(1161151,89,227),(1161161,92,230),(1161162,94,232),(1161175,99,237),(1161185,103,239),(1161186,106,240),(1161200,114,242),(1161211,122,244),(1161223,132,243),(1161236,141,240),(1161248,151,235),(1161261,160,229),(1161262,163,227),(1161277,171,218),(1161278,173,216),(1161294,179,206),(1161295,180,203),(1161311,183,189),(1161325,181,177),(1161339,173,160),(1161353,166,147),(1161354,164,144),(1161372,153,137),(1161387,142,136),(1161402,130,139),(1161418,125,143)]),
 ("left_curly_brace",[(1398660,131,127),(1398667,129,126),(1398675,126,126),(1398683,124,127),(1398691,120,128),(1398699,116,131),(1398707,113,134),(1398714,111,136),(1398722,108,140),(1398730,106,143),(1398738,105,145),(1398739,104,148),(1398749,103,150),(1398757,102,153),(1398765,102,155),(1398778,104,158),(1398787,106,159),(1398795,109,159),(1398805,112,159),(1398814,116,159),(1398825,119,160),(1398836,124,161),(1398847,127,162),(1398859,129,164),(1398871,128,166),(1398883,127,169),(1398894,125,172),(1398907,123,174),(1398908,120,177),(1398922,117,179),(1398935,112,183),(1398949,107,185),(1398962,105,187),(1398976,103,187),(1398989,101,187),(1399015,101,185),(1399027,103,184),(1399041,105,184),(1399057,109,183),(1399074,114,183),(1399090,118,184),(1399103,121,185),(1399118,121,188),(1399133,122,190),(1399150,121,194),(1399167,119,199),(1399184,117,205),(1399202,114,211),(1399203,112,214),(1399222,109,220),(1399241,107,224),(1399260,106,230),(1399278,107,232),(1399297,112,235),(1399317,123,236),(1399336,128,236)]),
 ("triangle",[(1310095,149,130),(1310123,147,129),(1310131,145,132),(1310138,143,133),(1310144,142,135),(1310151,140,140),(1310159,138,142),(1310166,133,148),(1310174,132,151),(1310175,130,154),(1310184,126,158),(1310193,122,164),(1310195,120,167),(1310204,115,173),(1310212,110,177),(1310219,106,182),(1310228,102,186),(1310236,97,190),(1310245,93,195),(1310246,91,197),(1310257,89,198),(1310258,86,202),(1310270,82,205),(1310279,80,207),(1310289,75,212),(1310299,72,215),(1310309,70,218),(1310311,68,220),(1310323,66,222),(1310334,64,224),(1310354,62,224),(1310403,65,225),(1310415,68,225),(1310427,70,225),(1310439,75,224),(1310451,82,225),(1310464,89,225),(1310477,96,226),(1310490,103,226),(1310504,113,227),(1310517,121,226),(1310531,131,226),(1310545,139,226),(1310560,150,226),(1310574,158,226),(1310589,169,227),(1310604,179,230),(1310621,188,231),(1310636,195,234),(1310652,201,236),(1310667,203,237),(1310682,205,237),(1310699,207,236),(1310719,207,234),(1310736,205,232),(1310752,202,227),(1310769,199,221),(1310786,196,215),(1310804,189,202),(1310822,184,193),(1310840,177,182),(1310858,172,173),(1310876,165,164),(1310894,161,157),(1310895,158,153),(1310916,153,144),(1310935,150,135),(1310954,149,130),(1310973,149,128)]),
 ("check",[(847491,88,188),(847563,90,186),(847575,92,184),(847588,94,182),(847600,97,180),(847612,100,180),(847624,102,179),(847637,105,180),(847649,107,181),(847661,109,183),(847673,111,185),(847681,112,188),(847698,114,191),(847706,115,194),(847715,116,197),(847726,117,200),(847735,118,203),(847743,119,205),(847752,120,208),(847761,120,210),(847771,123,215),(847781,124,218),(847792,126,220),(847802,126,222),(847812,127,225),(847835,128,228),(847876,128,226),(847888,127,223),(847900,127,221),(847911,126,217),(847913,126,214),(847926,126,210),(847938,125,202),(847950,126,194),(847962,127,189),(847975,130,182),(847988,132,175),(847989,133,172),(848004,138,163),(848017,141,156),(848031,143,152),(848045,149,142),(848058,151,139),(848072,156,130),(848086,158,127),(848101,163,121),(848116,166,116),(848117,168,113),(848133,171,111),(848149,172,109),(848177,171,111),(848192,170,114)]),
 ("triangle",[(1311673,135,133),(1311685,132,135),(1311693,130,138),(1311701,127,142),(1311707,126,144),(1311714,123,147),(1311721,122,150),(1311729,118,155),(1311736,115,161),(1311745,110,166),(1311753,106,172),(1311755,103,174),(1311764,99,180),(1311772,94,185),(1311781,89,191),(1311790,81,200),(1311799,77,205),(1311809,72,210),(1311819,66,216),(1311828,63,219),(1311838,60,224),(1311848,57,226),(1311859,53,230),(1311878,51,232),(1311954,53,232),(1311965,55,233),(1311976,57,233),(1311987,60,233),(1311999,66,231),(1312011,71,231),(1312023,78,231),(1312035,85,230),(1312048,94,229),(1312061,107,230),(1312074,117,229),(1312088,127,229),(1312102,140,230),(1312116,149,230),(1312130,160,230),(1312144,166,229),(1312146,168,230),(1312160,171,229),(1312162,175,229),(1312179,180,229),(1312194,185,227),(1312209,187,227),(1312224,189,226),(1312239,189,224),(1312255,188,222),(1312271,183,216),(1312287,179,209),(1312304,172,201),(1312321,166,191),(1312338,159,182),(1312355,152,174),(1312373,147,166),(1312391,142,159),(1312409,135,150),(1312427,132,145),(1312445,130,143)]),
 ("check",[(854494,89,193),(854546,92,192),(854562,94,191),(854578,96,190),(854593,98,190),(854606,100,192),(854618,103,195),(854626,105,197),(854634,107,197),(854642,109,199),(854650,111,202),(854658,113,204),(854666,115,207),(854675,117,209),(854682,118,212),(854693,121,214),(854701,122,216),(854710,123,219),(854719,125,222),(854728,126,224),(854737,127,226),(854747,128,228),(854758,129,232),(854777,130,234),(854806,130,236),(854834,129,234),(854854,128,232),(854865,128,229),(854877,127,225),(854879,127,223),(854891,128,217),(854903,128,209),(854915,129,202),(854927,130,192),(854940,132,182),(854952,134,172),(854965,138,163),(854978,144,151),(854991,150,143),(855005,156,136),(855019,164,126),(855032,167,121),(855046,173,116),(855062,175,113),(855076,177,111),(855107,178,109)]),
 ("rectangle",[(901156,84,151),(901201,85,153),(901209,86,157),(901218,87,161),(901224,87,164),(901231,88,171),(901239,89,177),(901246,89,184),(901253,90,188),(901254,90,191),(901263,91,199),(901271,91,205),(901279,92,210),(901288,92,215),(901297,92,220),(901298,92,223),(901308,92,227),(901317,92,230),(901327,92,232),(901337,92,234),(901351,92,236),(901403,95,234),(901415,98,233),(901425,100,233),(901436,103,232),(901448,110,230),(901460,118,231),(901472,127,231),(901485,136,231),(901497,145,230),(901510,154,231),(901523,161,231),(901524,163,231),(901538,170,231),(901551,178,230),(901566,181,231),(901580,186,234),(901594,191,234),(901608,193,234),(901623,198,235),(901673,198,233),(901687,198,231),(901701,195,226),(901716,194,221),(901717,193,219),(901737,192,208),(901753,190,199),(901769,188,187),(901785,187,174),(901801,185,162),(901818,184,152),(901834,184,149),(901850,183,143),(901867,183,141),(901884,183,139),(901901,183,137),(901917,182,134),(901934,182,132),(901952,181,129),(901969,179,128),(901987,177,127),(901989,174,127),(902010,168,128),(902029,156,130),(902049,143,131),(902069,128,134),(902090,115,138),(902110,103,139),(902130,97,138),(902149,92,139),(902169,90,137)]),
 ("right_curly_brace",[(1214792,130,155),(1214817,128,154),(1214828,128,152),(1214852,129,150),(1214869,131,149),(1214881,133,149),(1214893,135,149),(1214901,137,149),(1214909,140,149),(1214917,142,150),(1214925,145,151),(1214934,148,151),(1214942,151,152),(1214951,153,154),(1214959,155,156),(1214968,156,159),(1214979,158,161),(1214988,158,163),(1214997,157,165),(1215007,157,168),(1215016,155,170),(1215026,154,173),(1215027,152,175),(1215039,150,177),(1215051,147,179),(1215062,144,180),(1215073,142,182),(1215085,140,184),(1215097,135,187),(1215109,133,189),(1215121,131,190),(1215133,130,192),(1215146,129,195),(1215158,129,197),(1215172,130,200),(1215184,132,202),(1215198,135,202),(1215200,137,202),(1215214,139,203),(1215228,145,202),(1215244,150,201),(1215259,155,200),(1215274,157,199),(1215275,160,198),(1215304,162,198),(1215357,160,199),(1215373,157,199),(1215389,155,199),(1215390,152,198),(1215407,148,198),(1215424,144,198),(1215440,142,198),(1215442,140,198),(1215460,137,200),(1215478,135,201),(1215542,138,201),(1215560,140,201),(1215579,142,202),(1215598,147,203),(1215617,149,205),(1215619,151,205),(1215639,154,207),(1215661,159,211),(1215681,160,218),(1215701,159,224),(1215703,158,227),(1215722,155,235),(1215743,147,243),(1215765,138,248),(1215789,125,248),(1215790,123,248),(1215811,118,247),(1215833,113,247)]),
 ("x",[(1117410,83,153),(1117423,84,155),(1117435,86,158),(1117440,87,160),(1117446,88,162),(1117452,92,167),(1117458,94,169),(1117464,98,175),(1117471,101,177),(1117472,103,180),(1117479,106,183),(1117480,108,186),(1117487,114,192),(1117495,119,199),(1117503,122,202),(1117504,124,205),(1117512,129,209),(1117521,133,213),(1117530,136,217),(1117541,141,222),(1117550,144,223),(1117560,147,225),(1117569,149,228),(1117580,151,230),(1117635,150,227),(1117647,148,225),(1117658,148,222),(1117670,146,216),(1117682,145,207),(1117694,143,198),(1117696,143,195),(1117712,141,181),(1117727,140,170),(1117728,140,167),(1117741,140,157),(1117754,141,149),(1117755,141,147),(1117769,140,143),(1117783,140,141),(1117784,140,138),(1117803,138,137),(1117817,136,139),(1117832,132,143),(1117846,129,146),(1117848,127,149),(1117864,120,159),(1117880,111,172),(1117896,102,186),(1117913,94,200),(1117930,88,213),(1117947,84,223),(1117964,83,228),(1117980,83,230),(1118012,83,228),(1118030,84,225)]),
 ("pigtail",[(1068229,69,230),(1068241,71,229),(1068249,73,229),(1068257,76,230),(1068265,79,229),(1068272,82,227),(1068280,86,226),(1068288,89,226),(1068295,91,226),(1068303,95,225),(1068311,98,223),(1068313,100,221),(1068322,103,220),(1068331,106,219),(1068340,110,216),(1068349,116,211),(1068359,118,208),(1068361,121,207),(1068371,125,203),(1068373,127,201),(1068385,132,195),(1068396,137,189),(1068407,140,184),(1068409,141,182),(1068421,145,176),(1068432,146,173),(1068445,148,167),(1068456,149,164),(1068469,148,157),(1068481,147,155),(1068495,144,150),(1068509,139,150),(1068522,137,149),(1068536,131,151),(1068551,125,154),(1068566,119,160),(1068581,113,168),(1068597,108,177),(1068612,105,186),(1068628,103,195),(1068645,103,205),(1068662,105,214),(1068679,108,219),(1068694,113,225),(1068713,122,232),(1068729,130,234),(1068748,140,235),(1068765,145,233),(1068784,152,230),(1068785,156,227)]),
 ("delete_mark",[(1441471,114,141),(1441479,115,144),(1441488,117,147),(1441495,119,150),(1441501,121,153),(1441506,122,156),(1441507,124,159),(1441516,128,166),(1441522,131,169),(1441523,133,172),(1441533,139,180),(1441540,144,186),(1441548,149,192),(1441556,153,198),(1441564,157,202),(1441572,161,207),(1441580,164,211),(1441589,165,213),(1441600,169,217),(1441610,171,219),(1441623,172,222),(1441650,171,224),(1441659,169,222),(1441669,166,221),(1441680,161,221),(1441690,155,220),(1441701,150,218),(1441702,147,217),(1441715,140,217),(1441726,131,216),(1441737,126,216),(1441739,123,216),(1441754,115,216),(1441768,105,217),(1441781,96,217),(1441793,89,219),(1441805,86,218),(1441806,83,219),(1441820,81,220),(1441857,81,218),(1441870,83,216),(1441884,88,212),(1441898,96,204),(1441912,103,199),(1441913,106,196),(1441932,116,188),(1441947,128,178),(1441964,140,168),(1441980,152,155),(1441996,163,145),(1442012,172,137),(1442028,178,131),(1442043,180,129),(1442060,182,127)]),
 ("question_mark",[(754482,109,151),(754498,108,149),(754522,106,148),(754539,105,146),(754555,103,144),(754564,103,142),(754580,102,140),(754592,102,138),(754600,103,135),(754612,103,133),(754624,104,130),(754632,105,127),(754644,107,126),(754656,110,124),(754668,112,122),(754681,115,121),(754692,117,118),(754704,120,117),(754716,122,116),(754728,125,116),(754738,127,115),(754748,129,115),(754758,131,114),(754768,133,114),(754779,136,116),(754781,138,115),(754792,140,117),(754803,142,119),(754813,144,120),(754824,146,122),(754835,148,124),(754847,151,128),(754859,152,130),(754871,154,133),(754883,157,136),(754895,159,142),(754907,161,144),(754909,162,146),(754923,163,149),(754924,164,152),(754947,165,158),(754948,165,161),(754965,166,166),(754979,166,168),(754993,164,174),(755008,161,176),(755023,159,178),(755037,157,179),(755053,153,180),(755068,150,181),(755084,146,180),(755100,141,181),(755116,137,182),(755132,132,183),(755149,128,184),(755165,125,185),(755182,123,186),(755199,121,188),(755217,119,191),(755234,118,194),(755252,118,198),(755253,119,200),(755273,120,205),(755291,121,207),(755292,122,209),(755313,122,214),(755333,123,216),(755351,123,219),(755428,123,217)]),
 ("left_curly_brace",[(1406343,150,116),(1406358,147,117),(1406366,145,116),(1406374,142,116),(1406383,139,117),(1406391,136,117),(1406399,133,118),(1406407,129,121),(1406415,126,122),(1406423,123,123),(1406431,120,125),(1406439,118,127),(1406447,115,128),(1406459,113,129),(1406475,112,131),(1406504,113,134),(1406513,115,134),(1406521,117,135),(1406530,120,135),(1406539,123,137),(1406549,126,138),(1406559,129,140),(1406569,135,143),(1406580,137,144),(1406590,139,147),(1406605,141,149),(1406616,140,152),(1406628,139,155),(1406639,134,159),(1406651,131,161),(1406662,124,166),(1406674,121,166),(1406687,117,166),(1406699,114,167),(1406721,112,166),(1406754,114,164),(1406767,116,163),(1406783,118,163),(1406798,120,162),(1406811,122,163),(1406826,125,164),(1406843,127,165),(1406858,129,166),(1406874,130,168),(1406889,129,171),(1406904,127,175),(1406919,125,179),(1406935,123,184),(1406950,121,190),(1406966,120,194),(1406983,119,199),(1406999,120,202),(1407016,123,207),(1407033,127,211),(1407050,133,215),(1407068,142,219),(1407085,148,220),(1407102,151,221)]),
 ("left_sq_bracket",[(969347,135,128),(969372,132,128),(969392,130,127),(969408,128,127),(969419,125,127),(969428,123,127),(969436,121,128),(969444,118,126),(969452,116,126),(969460,114,127),(969468,111,126),(969476,109,126),(969488,107,126),(969500,105,126),(969512,103,126),(969529,101,126),(969546,98,127),(969565,96,128),(969586,94,128),(969610,92,129),(969635,90,129),(969687,89,131),(969703,88,133),(969716,88,135),(969727,88,138),(969738,88,140),(969749,88,142),(969760,88,144),(969761,88,146),(969773,88,152),(969784,88,155),(969796,89,161),(969809,89,165),(969821,89,172),(969834,89,176),(969847,89,181),(969861,89,188),(969874,90,190),(969887,91,196),(969901,91,202),(969915,92,206),(969930,92,212),(969945,92,220),(969959,93,222),(969961,93,225),(969976,93,231),(969991,94,233),(969993,94,236),(970009,94,238),(970025,94,241),(970041,94,243),(970060,94,247),(970088,93,249),(970181,96,248),(970198,98,249),(970216,100,248),(970235,104,247),(970256,111,248),(970276,116,248),(970297,123,248),(970318,127,248),(970339,132,248),(970358,134,250),(970382,136,252),(970402,138,252),(970430,140,253)]),
 ("star",[(674689,89,257),(674714,89,255),(674746,90,253),(674758,91,250),(674767,92,248),(674774,93,246),(674781,94,243),(674789,95,241),(674797,95,237),(674804,96,235),(674812,97,232),(674820,98,229),(674822,98,227),(674831,99,225),(674839,100,221),(674848,102,215),(674850,103,213),(674860,104,209),(674870,106,200),(674880,107,196),(674890,108,189),(674900,108,184),(674902,108,182),(674913,109,176),(674924,110,170),(674935,110,165),(674946,111,157),(674958,111,152),(674970,112,145),(674982,112,138),(674994,113,133),(675007,114,127),(675019,115,122),(675032,115,119),(675045,116,117),(675046,116,115),(675098,118,117),(675111,118,119),(675125,120,125),(675127,120,127),(675141,121,132),(675143,121,136),(675157,122,140),(675159,123,144),(675175,126,153),(675191,129,162),(675209,132,172),(675211,133,174),(675229,136,184),(675246,139,196),(675247,140,199),(675266,143,210),(675267,144,212),(675286,148,225),(675304,151,234),(675323,154,241),(675340,156,246),(675358,158,253),(675377,158,256),(675406,158,258),(675443,155,257),(675461,153,255),(675480,148,251),(675500,141,244),(675519,135,238),(675538,129,232),(675558,119,224),(675578,110,217),(675598,100,210),(675600,98,208),(675622,89,200),(675643,80,195),(675664,70,187),(675686,62,183),(675707,53,175),(675729,45,170),(675753,41,167),(675774,39,165),(675815,39,163),(675837,44,165),(675860,53,165),(675884,66,166),(675908,81,166),(675932,97,164),(675956,114,162),(675980,132,159),(676005,148,158),(676030,164,158),(676032,166,158),(676057,180,158),(676082,189,159),(676107,200,159),(676132,204,159),(676157,206,158),(676199,204,160),(676225,200,162),(676251,189,169),(676278,180,175),(676305,164,185),(676333,150,194),(676361,134,206),(676388,123,213),(676417,110,224),(676445,102,234),(676473,93,243),(676502,87,250),(676504,86,252),(676533,83,257),(676562,81,258),(676591,80,258)]),
 ("arrow",[(1356897,68,222),(1356908,70,220),(1356917,73,218),(1356923,75,217),(1356929,77,215),(1356930,80,213),(1356936,82,212),(1356937,84,210),(1356944,87,209),(1356945,89,208),(1356953,92,206),(1356954,95,204),(1356962,101,201),(1356970,106,198),(1356979,112,194),(1356988,118,191),(1356997,124,187),(1356998,127,186),(1357007,132,183),(1357017,138,181),(1357018,141,180),(1357028,146,178),(1357038,154,173),(1357048,159,171),(1357049,161,170),(1357060,166,167),(1357061,168,167),(1357073,171,166),(1357085,174,164),(1357086,177,162),(1357098,180,160),(1357110,182,158),(1357133,183,156),(1357145,181,154),(1357158,178,153),(1357172,171,153),(1357185,164,153),(1357199,160,153),(1357214,150,154),(1357228,147,155),(1357243,141,157),(1357258,137,158),(1357272,135,158),(1357314,137,158),(1357329,140,157),(1357331,143,156),(1357348,151,154),(1357363,160,152),(1357379,170,149),(1357395,179,147),(1357411,185,145),(1357427,192,144),(1357443,196,144),(1357460,198,144),(1357475,200,144),(1357492,201,147),(1357509,199,149),(1357529,194,157),(1357530,191,160),(1357551,186,167),(1357569,180,176),(1357570,177,179),(1357590,171,187),(1357609,169,189),(1357627,165,194),(1357646,164,196)]),
 ("x",[(1124198,99,154),(1124203,99,156),(1124209,100,158),(1124215,102,160),(1124222,103,163),(1124231,105,167),(1124239,108,171),(1124246,109,173),(1124248,111,175),(1124256,115,181),(1124263,119,186),(1124270,121,189),(1124272,123,191),(1124281,128,197),(1124289,132,202),(1124297,137,205),(1124306,141,209),(1124314,144,212),(1124323,147,214),(1124332,150,216),(1124343,153,217),(1124352,155,219),(1124386,156,216),(1124396,156,214),(1124405,155,211),(1124416,153,205),(1124426,151,197),(1124437,150,191),(1124438,150,188),(1124453,149,177),(1124465,149,166),(1124476,149,157),(1124489,150,148),(1124501,151,138),(1124513,152,131),(1124514,152,129),(1124529,153,126),(1124542,154,124),(1124543,155,122),(1124579,154,124),(1124592,152,127),(1124606,148,134),(1124620,142,143),(1124634,137,152),(1124648,130,162),(1124663,122,172),(1124678,115,182),(1124679,113,184),(1124697,105,196),(1124713,100,208),(1124728,97,216),(1124744,95,222),(1124760,95,224),(1124796,97,222),(1124812,97,220),(1124829,99,214)]),
 ("delete_mark",[(1439775,121,140),(1439780,121,142),(1439786,122,145),(1439791,124,148),(1439797,125,150),(1439803,127,154),(1439804,129,157),(1439812,133,165),(1439819,135,168),(1439820,138,171),(1439828,143,178),(1439836,148,185),(1439843,153,191),(1439851,156,193),(1439852,158,196),(1439862,162,201),(1439870,165,206),(1439878,168,210),(1439879,168,212),(1439892,171,214),(1439907,171,216),(1439935,169,215),(1439944,167,214),(1439954,164,212),(1439964,158,211),(1439975,154,210),(1439976,151,210),(1439986,146,213),(1439987,143,213),(1440000,134,214),(1440014,125,217),(1440028,113,221),(1440040,104,224),(1440053,96,226),(1440065,89,230),(1440078,84,232),(1440091,81,233),(1440104,79,233),(1440119,78,231),(1440135,80,227),(1440150,81,224),(1440165,89,216),(1440182,96,207),(1440184,99,206),(1440200,108,195),(1440215,117,186),(1440216,120,182),(1440233,131,169),(1440248,142,158),(1440264,151,150),(1440280,156,144),(1440295,162,139),(1440311,169,134),(1440327,175,128)])
]

def normalize(track, delta=3):
    """Return a copy of *track* after having removed all points that
    are distant from the previous one less than *delta* on x and y
    coordinates."""
    t = track[:]
    drop = lambda p1, p2: abs(p1.x-p2.x)<delta and abs(p1.y-p2.y)<delta
    i = 0
    while i+1<len(t):
        if drop(t[i], t[i+1]): del t[i+1]
        else: i += 1
    return t

def testInstances():
    """Return a copy of the default training set for the RubineRecognizer."""
    adapter = lambda txy: quickdict(x=txy[1], y=txy[2],
                                    t=datetime.timedelta(milliseconds=txy[0]))
    data = _test_instances[:]
    for i in range(len(data)):
        name, stroke = data[i]
        data[i] = name, list(map(adapter, stroke))
    return data
