import mock
import time

import psycopg2
import redis

from cliquet.storage import exceptions
from cliquet.cache import CacheBase, SessionCache
from cliquet.cache import (postgresql as postgresql_backend,
                           redis as redis_backend)

from .support import unittest


class CacheBaseTest(unittest.TestCase):
    def setUp(self):
        self.cache = CacheBase()

    def test_mandatory_overrides(self):
        calls = [
            (self.cache.flush,),
            (self.cache.ping,),
            (self.cache.ttl, ''),
            (self.cache.expire, '', ''),
            (self.cache.get, ''),
            (self.cache.set, '', ''),
            (self.cache.delete, ''),
        ]
        for call in calls:
            self.assertRaises(NotImplementedError, *call)


class BaseTestCache(object):
    backend = None

    settings = {
        'cliquet.cache_url': ''
    }

    def __init__(self, *args, **kwargs):
        super(BaseTestCache, self).__init__(*args, **kwargs)
        self.cache = self.backend.load_from_config(self._get_config())
        self.client_error_patcher = None

    def _get_config(self, settings=None):
        """Mock Pyramid config object.
        """
        if settings is None:
            settings = self.settings
        return mock.Mock(registry=mock.Mock(settings=settings))

    def tearDown(self):
        mock.patch.stopall()
        super(BaseTestCache, self).tearDown()
        self.cache.flush()

    def test_backend_error_is_raised_anywhere(self):
        self.client_error_patcher.start()
        calls = [
            (self.cache.flush,),
            (self.cache.ttl, ''),
            (self.cache.expire, '', 0),
            (self.cache.get, ''),
            (self.cache.set, '', ''),
            (self.cache.delete, ''),
        ]
        for call in calls:
            self.assertRaises(exceptions.BackendError, *call)

    def test_ping_returns_an_error_if_unavailable(self):
        self.client_error_patcher.start()
        self.assertFalse(self.cache.ping())

    def test_ping_returns_true_if_available(self):
        self.assertTrue(self.cache.ping())

    def test_set_adds_the_record(self):
        stored = 'toto'
        self.cache.set('foobar', stored)
        retrieved = self.cache.get('foobar')
        self.assertEquals(retrieved, stored)

    def test_delete_removes_the_record(self):
        self.cache.set('foobar', 'toto')
        self.cache.delete('foobar')
        retrieved = self.cache.get('foobar')
        self.assertIsNone(retrieved)

    def test_expire_expires_the_value(self):
        self.cache.set('foobar', 'toto')
        self.cache.expire('foobar', 0.05)
        time.sleep(0.1)
        retrieved = self.cache.get('foobar')
        self.assertIsNone(retrieved)

    def test_set_with_ttl_expires_the_value(self):
        self.cache.set('foobar', 'toto', 0.05)
        time.sleep(0.1)
        retrieved = self.cache.get('foobar')
        self.assertIsNone(retrieved)

    def test_ttl_return_the_time_to_live(self):
        self.cache.set('foobar', 'toto')
        self.cache.expire('foobar', 10)
        ttl = self.cache.ttl('foobar')
        self.assertGreater(ttl, 0)
        self.assertLessEqual(ttl, 10)


class RedisCacheTest(BaseTestCache, unittest.TestCase):
    backend = redis_backend

    def __init__(self, *args, **kwargs):
        super(RedisCacheTest, self).__init__(*args, **kwargs)
        self.client_error_patcher = mock.patch.object(
            self.cache._client,
            'execute_command',
            side_effect=redis.RedisError)


class PostgreSQLCacheTest(BaseTestCache, unittest.TestCase):
    backend = postgresql_backend

    settings = {
        'cliquet.cache_url':
            'postgres://postgres:postgres@localhost:5432/testdb'
    }

    def __init__(self, *args, **kwargs):
        super(PostgreSQLCacheTest, self).__init__(*args, **kwargs)
        self.client_error_patcher = mock.patch(
            'cliquet.storage.postgresql.psycopg2.connect',
            side_effect=psycopg2.OperationalError)


class SessionCacheTest(unittest.TestCase):
    def setUp(self):
        self.cache = SessionCache(redis_backend.Redis(), 0.05)
        super(SessionCacheTest, self).setUp()

    def test_set_adds_the_record(self):
        stored = 'toto'
        self.cache.set('foobar', stored)
        retrieved = self.cache.get('foobar')
        self.assertEquals(retrieved, stored)

    def test_delete_removes_the_record(self):
        self.cache.set('foobar', 'toto')
        self.cache.delete('foobar')
        retrieved = self.cache.get('foobar')
        self.assertIsNone(retrieved)

    def test_set_expires_the_value(self):
        self.cache.set('foobar', 'toto')
        time.sleep(0.1)
        retrieved = self.cache.get('foobar')
        self.assertIsNone(retrieved)
