"""Convert HDF5 file with groups to/from directory tree of Numpy recarrays."""
# pylint: disable=W0212

import os
import logging  # diagnostics
import ast  # convert str(dtype) to list of tuples

import numpy as np
import tables as pt

from cgp.utils.load_memmap_offset import load
from cgp.utils.unstruct import unstruct

# Initialize logging. Keys refer to the dict made available by the logger.
keys = "asctime levelname name lineno process message"
fmt = "%(" + ")s\t%(".join(keys.split()) + ")s"
logging.basicConfig(level=logging.INFO, format=fmt)
hdflog = logging.getLogger('numpy_hdf')

def hdf2numpy(src, dst, where="/", ext=".npy"):
    """
    Create Numpy files from HDF5 file, with subdirectories for groups.
    
    >>> with Example() as src:
    ...     h5file = os.path.join(src, "test.h5")
    ...     numpy2hdf(src, h5file, where="/parent/group")
    ...     dst = os.path.join(src, "npy")
    ...     hdf2numpy(h5file, dst)
    ...     for root, dirs, files in os.walk(dst):
    ...         print ":", root.replace("\\\\", "/"), ":", files
    ...         for file in files:
    ...             print repr(np.load(os.path.join(root, file))[:])
    : .../npy : []
    : .../parent : []
    : .../parent/group : ['a.npy']
    array([(1, 1.5), (2, 2.5)], dtype=[('i', '|i1'), ('x', '<f8')])
    : .../parent/group/sub : ['b.npy']
    array([(1, 1.5), (2, 2.5)], dtype=[('i', '|i1'), ('x', '<f8')])
    """
    # standardize to forward slashes
    src = src.replace("\\", "/")
    dst = dst.replace("\\", "/")
    with pt.openFile(src) as f:
        for table in f.walkNodes(where, classname="Table"):
            relpath = os.path.relpath(table._v_pathname, "/")
            relpath = relpath.replace("\\", "/").replace("..", "")
            filename = os.path.join(dst, relpath + ext)
            dir_, _ = os.path.split(filename)
            if not os.path.exists(dir_):
                os.makedirs(dir_)
            np.save(filename, table[:])

def numpy2hdf(src, dst, where="/", ext=".npy", recursive=True):
    """
    Create HDF5 file from Numpy files, with groups for subdirectories.
    
    dst is the path to the destination HDF file. It will be created if required.
    src is the path to a directory that will correspond to the root group.
    Any .npy files in the directory tree are added as tables, and any 
    subdirectories are added as groups if recursive=True.
    
    >>> with Example() as src:
    ...     dst = os.path.join(src, "first.h5")
    ...     numpy2hdf(src, dst)
    ...     with pt.openFile(dst) as f:
    ...         f
    ...         a = np.load(os.path.join(src, "a.npy"))
    ...         print "Arrays equal:", all(f.root.a[:] == a)
    File(filename=...first.h5...)
    / (RootGroup) ''
    /a (Table(2,)) ''
      description := {
      "i": Int8Col(shape=(), dflt=0, pos=0),
      "x": Float64Col(shape=(), dflt=0.0, pos=1)}
      byteorder := 'little'
      chunkshape := (...,)
    /sub (Group) ''
    /sub/b (Table(2,)) ''...
    Arrays equal: True
    
    >>> with Example() as src:
    ...     dst = os.path.join(src, "second.h5")
    ...     numpy2hdf(src, dst, where="/parent/group")
    ...     with pt.openFile(dst) as f:
    ...         f
    File(filename=...second.h5...)...
    /parent/group/sub/b (Table(2,)) ''...
    
    An .npy file that contains a dict or list is treated like a subdirectory.
    Table or array names are taken from the dictionary keys or autogenerated 
    as (i0, i1, ...).
    
    >>> with Example() as src:
    ...     L = np.array([np.arange(1+i) for i in range(2)], dtype=object)
    ...     d = dict(("i%s" % k, v) for k, v in enumerate(L))
    ...     np.save(os.path.join(src, "list.npy"), L)
    ...     np.save(os.path.join(src, "dict.npy"), d)
    ...     dst = os.path.join(src, "third.h5")
    ...     numpy2hdf(src, dst, where="/parent/group")
    ...     with pt.openFile(dst) as f:
    ...         print ":", f
    ...     # Verify bugfix for case with default `where` argument
    ...     numpy2hdf(src, dst + ".nowhere")
    : ...third.h5...
    /parent/group/dict (Group) ''
    /parent/group/dict/i0 (Array(1,)) ''
    /parent/group/dict/i1 (Array(2,)) ''
    /parent/group/list (Group) ''
    /parent/group/list/k0 (Array(1,)) ''
    /parent/group/list/k1 (Array(2,)) ''...
    """
    # HDF5 path: / for root, otherwise /slash/at/start/but/not/end
    where = where.strip("/")  # will add leading slash later
    with pt.openFile(dst, "a") as f:
        for root, _dirs, files in os.walk(src):
            hdflog.debug("root: %s, files: %s", root, files)
            relpath = os.path.relpath(root, src)
            relpath = relpath.replace("\\", "/")
            group = where if relpath == "." else where + "/" + relpath
            group = "/" + group.strip("/")
            for file_ in files:
                name, extension = os.path.splitext(file_)
                if extension == ext:
                    filename = os.path.join(root, file_)
                    dictgroup = "/" + (group + "/" + name).strip("/")
                    try:
                        a = load(filename, mmap_mode="r")
                    except (IndexError, ValueError):
                        # Cannot memmap array with Python objects
                        a = np.load(filename)
                    if (a.dtype == object) and a.shape==():
                        dict2hdf(a.item(), f, dictgroup)
                    elif a.dtype.names:
                        if a.ndim > 1:
                            a = shoehorn_recarray(a)
                        f.createTable(group, name, np.atleast_1d(a), 
                                      createparents=True, 
                                      expectedrows=len(np.atleast_1d(a)))
                    elif a.dtype == object:
                        # Hope that v is a list
                        d = dict(("k%s" % k, v) for k, v in enumerate(a))
                        dict2hdf(d, f, dictgroup)
                    else:
                        f.createArray(group, name, a, createparents=True)
            if not recursive:
                break

def dict2hdf(d, f, where):
    """
    Walk dictionary and save leaves as tables or arrays in HDF5 file.
    
    >>> d = dict(array=np.arange(3), struct = 
    ...     np.rec.fromrecords([(1, 1.0), (2, 2.0)], 
    ...     dtype=[("a", int), ("b", float)]), nested=dict(c=np.arange(3.)))
    >>> with Example() as dir:
    ...     with pt.openFile("test.h5", "w") as f:
    ...         dict2hdf(d, f, "/walk")
    ...         print repr(f)
    File(filename=test.h5, title='', mode='w', rootUEP='/', filters=...)
    / (RootGroup) ''
    /walk (Group) ''
    /walk/array (Array(3,)) ''...
    /walk/struct (Table(2,)) ''
      description := {
      "a": Int...Col(shape=(), dflt=0, pos=0),
      "b": Float64Col(shape=(), dflt=0.0, pos=1)}...
    /walk/nested (Group) ''
    /walk/nested/c (Array(3,)) ''...
    """
    for k, v in d.items():
        kw = dict(where=where, name=k, createparents=True)
        hdflog.debug(kw)
        if hasattr(v, "items"):
            dict2hdf(v, f, where + "/" + k)
        else:
            try:
                v = np.asanyarray(v)
            except ValueError:
                # Hope that v is a list
                d_ = dict(("k%s" % k_, v_) for k_, v_ in enumerate(v))
                dict2hdf(d_, f, where + "/" + k)
            else:
                if v.dtype.names:
                    f.createTable(description=np.atleast_1d(v), 
                        expectedrows=len(np.atleast_1d(v)),
                        **kw)
                else:
                    f.createArray(object=v, **kw)

class Example(object):
    """
    Example for use in doctests, deleting temporary directory when done.
    
    >>> with Example() as src:
    ...     [files for root, dirs, files in os.walk(src)]
    [['a.npy'], ['b.npy']]
    >>> os.path.exists(src)
    False
    """
    def __enter__(self):
        import tempfile
        self.src = tempfile.mkdtemp()  # pylint: disable=W0201
        dtype = [("i", np.int8), ("x", np.float64)]
        a = np.array([(1, 1.5), (2, 2.5)], dtype=dtype)
        filename = os.path.join(self.src, "a.npy")
        np.save(filename, a)
        subdir = os.path.join(self.src, "sub")
        os.mkdir(subdir)
        filename = os.path.join(subdir, "b.npy")
        np.save(filename, a)
        return self.src
    
    def __exit__(self, exc_type, exc_value, traceback):
        import shutil
        shutil.rmtree(self.src)

def shoehorn_recarray(x, ndim=1):
    """
    Shoehorn multi-dimensional record array into HDF-compatible form.
    
    Pytables does not allow table rows to be arrays. Thus, a record array with 
    dimension > 1 cannot be converted to an HDF table. However, it will accept 
    a 1-D record array whose *fields* are arrays. This function rearranges the 
    dimensions of the entire array and its fields, retaining *ndim* dimensions 
    and pushing the rest into the fields.
    
    One dimension is suitable for creating a table, whereas zero dimensions 
    is suitable for a table row.
    
    >>> x = np.arange(24.0).view([("a", float), ("b", float)]).reshape(4, 3)
    >>> y = shoehorn_recarray(x)
    >>> np.testing.assert_equal(x["a"], y["a"])
    
    >>> x.shape
    (4, 3)
    >>> x.dtype
    dtype([('a', '<f8'), ('b', '<f8')])
    >>> y.shape
    (4,)
    >>> y.dtype
    dtype([('a', '<f8', (3,)), ('b', '<f8', (3,))])
    
    >>> z = shoehorn_recarray(x, ndim=0)
    >>> np.testing.assert_equal(x["a"], z["a"])
    >>> z.shape
    ()
    >>> z.dtype
    dtype([('a', '<f8', (4, 3)), ('b', '<f8', (4, 3))])
    """
    u = unstruct(x)
    # The shape of u is x.shape + (#fields,) + fieldshape
    # Roll the #fields dimension to position ndim (note zero-based indexing)
    r = np.rollaxis(u, len(x.shape), ndim)
    dtype = [i[:2] + (r.shape[1 + ndim:],) 
        for i in ast.literal_eval(str(x.dtype))]
    return r.flatten().view(dtype).squeeze()

def test_shoehorn_recarray():
    """Shoehorn multi-dimensional record array into HDF table."""
    x = np.arange(24.0).view([("a", float), ("b", float)]).reshape(4, 3)
    with Example() as src:
        np.save(os.path.join(src, "x.npy"), x)
        numpy2hdf(src, os.path.join(src, "dst.h5"))
        with pt.openFile(os.path.join(src, "dst.h5")) as f:
            np.testing.assert_equal(x["a"], f.root.x.cols.a[:])

if __name__ == "__main__":
    import doctest
    doctest.testmod(optionflags=doctest.ELLIPSIS|doctest.NORMALIZE_WHITESPACE)
