# -*- coding: utf-8 -*-
#
# Copyright (c) 2007 - 2009 -- Lars Heuer - Semagia <http://www.semagia.com/>.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials provided
#       with the distribution.
#
#     * Neither the name of the project nor the names of the contributors 
#       may be used to endorse or promote products derived from this 
#       software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
"""\
This module provides classes to read 
`XML Topic Maps (XTM) 2.0 <http://www.isotopicmaps.org/sam/sam-xtm/>`_

:author:       Lars Heuer (heuer[at]semagia.com)
:organization: Semagia - http://www.semagia.com/
:version:      $Rev: 168 $ - $Date: 2009-06-26 14:22:56 +0200 (Fr, 26 Jun 2009) $
:license:      BSD license
"""
from StringIO import StringIO
import xml.sax.handler as sax_handler
from xml.sax.saxutils import XMLGenerator
from tm import TMDM, XSD, mio
from tm.mio.deserializer import Context
from tm.mio.xmlutils import attributes
from tm.irilib import resolve_iri

__all__ = ['XTM20ContentHandler']

# XTM 2.0 namespace
NS_XTM = 'http://www.topicmaps.org/xtm/'

# Constants for XTM elements.
MERGE_MAP = 'mergeMap'
TOPIC_MAP = 'topicMap'
TOPIC = 'topic'
ASSOCIATION = 'association'
ROLE = 'role'
OCCURRENCE = 'occurrence'
NAME = 'name'
VARIANT = 'variant'

INSTANCE_OF = 'instanceOf'
TYPE = 'type'

VALUE = 'value'
RESOURCE_REF = 'resourceRef'
RESOURCE_DATA = 'resourceData'

SCOPE = 'scope'

TOPIC_REF = 'topicRef'

SUBJECT_IDENTIFIER = 'subjectIdentifier'
SUBJECT_LOCATOR = 'subjectLocator'
ITEM_IDENTITY = 'itemIdentity'

_VERSION = (None, 'version')
_REIFIER = (None, 'reifier')
_HREF = (None, 'href')
_ID = (None, 'id')
_DATATYPE = (None, 'datatype')

# States
_STATE_INITIAL = 1
_STATE_TOPIC = 2
_STATE_ASSOCIATION = 3
_STATE_ROLE = 4
_STATE_TYPE = 5
_STATE_INSTANCE_OF = 6
_STATE_SCOPE = 7
_STATE_OCCURRENCE = 8
_STATE_NAME = 9
_STATE_VARIANT = 10

#pylint: disable-msg=E1103
class XTM20ContentHandler(sax_handler.ContentHandler):
    """\
    Content handler for `XTM 2.0 <http://www.isotopicmaps.org/sam/sam-xtm/>`_
    topic maps.
    """

    # Default topic name type
    _TOPIC_NAME = mio.SUBJECT_IDENTIFIER, TMDM.topic_name

    def __init__(self, map_handler=None, locator=None):
        """\
        Initializes this handler with the specified input adapter `adapter`
        """
        self.map_handler = map_handler
        self.subordinate = False
        self.loaded = Context()
        self.reset(locator)

    def reset(self, locator=None):
        self._state = _STATE_INITIAL
        self._next_state = _STATE_INITIAL
        self._seen_type = False
        self._datatype = None
        self.doc_iri = locator
        self._accept_xml = False
        self._accept_content = False
        # The ``_content`` used to keep "ordinary" string content and it is
        # used by the XMLGenerator, so ``_content.truncate(0)`` effects also
        # the output of the XMLGenerator
        self._content = StringIO()
        self._xml_handler = XMLGenerator(self._content)

    def startDocument(self):
        pass

    def startPrefixMapping(self, prefix, uri):
        if self._accept_xml:
            self._xml_handler.startPrefixMapping(prefix, uri)

    def endPrefixMapping(self, prefix):
        if self._accept_xml:
            self._xml_handler.endPrefixMapping(prefix)

    def processingInstruction(self, target, data):
        if self._accept_xml:
            self._xml_handler.processingInstruction(target, data)

    def setDocumentLocator(self, locator):
        if not self._locator:
            self._locator = locator.getSystemId()

    def startElementNS(self, (uri, name), qname, attrs):
        attrs = attributes(attrs)
        if self._accept_xml:
            self._xml_handler.startElementNS((uri, name), qname, attrs)
            return
        handler = self.map_handler
        state = self._state
        href = self._href
        process_reifier = self._process_reifier
        create_locator = self._create_locator
        if TOPIC_REF == name:
            iid = href(attrs)
            if '#' not in iid:
                raise mio.MIOException('Invalid topic reference "%s". Does not contain a fragment identifier' % iid)
            ref = mio.ITEM_IDENTIFIER, iid
            if state == _STATE_INSTANCE_OF:
                handler.isa(ref)
            elif state == _STATE_ROLE:
                handler.player(ref)
            elif state == _STATE_SCOPE:
                handler.theme(ref)
            elif state == _STATE_TYPE:
                handler.type(ref)
                self._seen_type = True
            else:
                raise mio.MIOException('Unexpected "topicRef" element')
        elif TOPIC == name:
            handler.startTopic((mio.ITEM_IDENTIFIER, create_locator('#' + attrs.get(_ID))))
            self._state = _STATE_TOPIC
        elif INSTANCE_OF == name:
            if state != _STATE_TOPIC:
                raise mio.MIOException('Unexpected "instanceOf" element')
            self._state = _STATE_INSTANCE_OF
        elif TYPE == name:
            self._next_state, self._state = state, _STATE_TYPE 
        elif SUBJECT_IDENTIFIER == name:
            handler.subjectIdentifier(href(attrs))
        elif SUBJECT_LOCATOR == name:
            handler.subjectLocator(href(attrs))
        elif ITEM_IDENTITY == name:
            handler.itemIdentifier(href(attrs))
        elif ASSOCIATION == name:
            handler.startAssociation()
            process_reifier(attrs)
            self._state = _STATE_ASSOCIATION
        elif ROLE == name:
            handler.startRole()
            process_reifier(attrs)
            self._state = _STATE_ROLE
        elif VALUE == name:
            self._accept_content = True
            self._content.truncate(0)
        elif RESOURCE_DATA == name:
            self._datatype = attrs.get(_DATATYPE, XSD.string)
            self._accept_content = True
            self._content.truncate(0)
            self._accept_xml = self._datatype == XSD.anyType
        elif RESOURCE_REF == name:
            handler.value(href(attrs), XSD.anyURI)
        elif OCCURRENCE == name:
            handler.startOccurrence()
            process_reifier(attrs)
            self._state = _STATE_OCCURRENCE
        elif NAME == name:
            handler.startName()
            process_reifier(attrs)
            self._seen_type = False
            self._state = _STATE_NAME
        elif VARIANT == name:
            handler.startVariant()
            process_reifier(attrs)
            self._state = _STATE_VARIANT
        elif SCOPE == name:
            handler.startScope()
            self._next_state, self._state = state, _STATE_SCOPE
        elif TOPIC_MAP == name:
            version = attrs.get(_VERSION)
            if not '2.0' == version:
                raise mio.MIOException('Invalid XTM version. Expected "2.0", got: "%s"' % version)
            self._process_topicmap_reifier(attrs)
        elif MERGE_MAP == name:
            self._process_mergemap(href(attrs))
        else:
            raise mio.MIOException('Unknown start element %s' % name)

    def endElementNS(self, (uri, name), qname):
        if self._accept_xml:
            self._xml_handler.endElementNS((uri, name), qname)
            return
        handler = self.map_handler
        if TOPIC == name:
            state = _STATE_INITIAL
        elif name in (TOPIC_MAP, TOPIC_REF, RESOURCE_REF):
            return
        elif SCOPE == name:
            handler.endScope()
            self._state = self._next_state
        elif TYPE == name:
            self._state = self._next_state
        elif INSTANCE_OF == name:
            self._state = _STATE_TOPIC
        elif ASSOCIATION == name:
            handler.endAssociation()
            self._state = _STATE_INITIAL
        elif ROLE == name:
            handler.endRole()
            self._state = _STATE_ASSOCIATION
        elif OCCURRENCE == name:
            handler.endOccurrence()
            self._state = _STATE_TOPIC
        elif NAME == name:
            if not self._seen_type:
                handler.type(XTM20ContentHandler._TOPIC_NAME)
            handler.endName()
            self._state = _STATE_TOPIC
        elif VARIANT == name:
            handler.endVariant()
            self._state = _STATE_NAME
        elif RESOURCE_DATA == name:
            if self._accept_xml:
                self._content.flush()
                handler.value(self._content.getvalue(), XSD.anyType)
            elif XSD.anyURI == self._datatype:
                handler.value(self._create_locator(self._content.getvalue()), XSD.anyURI)
            else:
                handler.value(self._content.getvalue(), self._datatype)
            self._accept_content = False
            self._accept_xml = False
        elif VALUE == name:
            handler.value(self._content.getvalue())
            self._accept_content = False

    def endDocument(self):
        self.reset()

    def characters(self, content):
        if self._accept_xml:
            self._xml_handler.characters(content)
        elif self._accept_content:
            self._content.write(content)

    def _process_mergemap(self, iri):
        if iri in self.context.loaded:
            return
        self.context.add_loaded(iri)
        from mio.reader.xtm import create_deserializer
        deserializer = create_deserializer()
        deserializer.context = self.context
        deserializer.handler = self.map_handler
        deserializer.subordinate = True
        deserializer.parse(mio.Source(iri))

    def _process_topicmap_reifier(self, attrs):
        """\
        Reads the "reifier" attribute and reports the reifier iff it is not 
        ``None`` and if the content handler is not in ``subordinate`` mode.
        """
        reifier = attrs.get(_REIFIER, None)
        if reifier:
            topic_ref = mio.ITEM_IDENTIFIER, self._create_locator(reifier)
            if not self.subordinate:
                self.map_handler.reifier(topic_ref)
            else:
                self.map_handler.topic(topic_ref)

    def _process_reifier(self, attrs):
        """\
        If the ``reifier`` attribute is not ``None``, this method generates
        a ``startReifier``, ``topicRef`` and ``endReifier`` event.
        """
        reifier = attrs.get(_REIFIER, None)
        if reifier:
            self.map_handler.reifier((mio.ITEM_IDENTIFIER, self._create_locator(reifier)))

    def _href(self, attrs):
        """\
        Returns a locator from the attributes. Further, the locator is
        resolved against the base locator.
        """
        return self._create_locator(attrs.getValue(_HREF))

    def _create_locator(self, reference):
        """\
        Returns a locator where ``reference`` is resolved against the base
        locator.
        """
        return resolve_iri(self.doc_iri, reference)
