# -*- coding: utf-8 -*-
#
# Copyright (c) 2007 - 2009 -- Lars Heuer - Semagia <http://www.semagia.com/>.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials provided
#       with the distribution.
#
#     * Neither the name of the project nor the names of the contributors 
#       may be used to endorse or promote products derived from this 
#       software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
"""\
This module provides classes to read 
`XML Topic Maps (XTM) 1.0 <http://www.topicmaps.org/xtm/1.0/>`_

:author:       Lars Heuer (heuer[at]semagia.com)
:organization: Semagia - http://www.semagia.com/
:version:      $Rev: 168 $ - $Date: 2009-06-26 14:22:56 +0200 (Fr, 26 Jun 2009) $
:license:      BSD license
"""
import xml.sax.handler as sax_handler
from tm import TMDM, XSD, XTM_10, mio
from tm.mio.deserializer import Context
from tm.mio.xmlutils import attributes
from tm.irilib import resolve_iri

__all__ = ['XTM10ContentHandler']

# XML namespace
NS_XML = 'http://www.w3.org/XML/1998/namespace'

# XTM 1.0 namespace
NS_XTM = 'http://www.topicmaps.org/xtm/1.0/'

# XLink namespace
NS_XLINK = 'http://www.w3.org/1999/xlink'


# Constants for XTM elements.
MERGE_MAP = 'mergeMap'
TOPIC_MAP = 'topicMap'
TOPIC = 'topic'
ASSOCIATION = 'association'
MEMBER = 'member'
ROLE_SPEC = 'roleSpec'
OCCURRENCE = 'occurrence'
BASE_NAME = 'baseName'
BASE_NAME_STRING = 'baseNameString'
VARIANT = 'variant'
VARIANT_NAME = 'variantName'

INSTANCE_OF = 'instanceOf'

RESOURCE_REF = 'resourceRef'
RESOURCE_DATA = 'resourceData'

SCOPE = 'scope'
PARAMETERS = 'parameters'

TOPIC_REF = 'topicRef'

SUBJECT_IDENTITY = 'subjectIdentity'
SUBJECT_INDICATOR_REF = 'subjectIndicatorRef'

#pylint: disable-msg=E1103
class XTM10ContentHandler(object, sax_handler.ContentHandler):
    """\
    XTM 1.0 content handler.

    .. Note::
        
       This content handler does not support more than one topic map in
       an XTM file.
    """
    _ASSOCIATION = mio.SUBJECT_IDENTIFIER, XTM_10.DEFAULT_ASSOCIATION_TYPE
    _ROLE = mio.SUBJECT_IDENTIFIER, XTM_10.DEFAULT_ROLE_TYPE
    _TOPIC_NAME = mio.SUBJECT_IDENTIFIER, TMDM.topic_name
    _OCURRENCE = mio.SUBJECT_IDENTIFIER, XTM_10.DEFAULT_OCCURRENCE_TYPE

    def __init__(self, map_handler=None, locator=None):
        """\
        Initializes this handler with the specified input adapter `adapter`
        """
        self.map_handler = map_handler
        self.subordinate = False
        self.context = Context()
        self.reset(locator)

    def reset(self, locator=None):
        self._stack = []    # Stack of element names
        self._content = []
        self._accept_content = False
        self._bases = []    # Stack of base locators
        self._scope = []    # Scope which was generated by mergeMap
        self._seen_type = False
        self._seen_scope = False
        self._role_type = None
        self._variants = []
        if locator:
            self._bases.append(locator)

    def _get_doc_iri(self):
        if self._bases:
            return self._bases[-1]
        return None

    def _set_doc_iri(self, locator):
        self._bases.append(locator)

    def setDocumentLocator(self, locator):
        self._bases.append(locator.getSystemId())

    def startElementNS(self, (uri, name), qname, attrs):
        if uri != NS_XTM and uri != '':
            return
        attrs = attributes(attrs)
        handler = self.map_handler
        stack = self._stack
        href = self._href
        create_locator = self._create_locator
        add_scope = self._add_scope
        process_iid = self._process_iid
        base = attrs.get((None, 'base'), None) or attrs.get((NS_XML, 'base'), None)
        if base:
            base = create_locator(base)
        self._bases.append(base or self._bases[-1])
        if name in (INSTANCE_OF, SUBJECT_IDENTITY,
                    ROLE_SPEC, VARIANT_NAME, PARAMETERS):
            stack.append(name)
        elif TOPIC_REF == name:
            self._process_topic_ref(stack[-1], href(attrs))
        elif TOPIC == name:
            handler.startTopic((mio.ITEM_IDENTIFIER, create_locator('#' + attrs.get((None, 'id')))))
            stack.append(TOPIC)
        elif SUBJECT_INDICATOR_REF == name:
            self._process_sid(href(attrs))
        elif RESOURCE_REF == name:
            self._process_resource_ref(href(attrs))
        elif ASSOCIATION == name:
            handler.startAssociation()
            process_iid(attrs)
            stack.append(ASSOCIATION)
        elif MEMBER == name:
            self._role_type = None
            stack.append(MEMBER)
        elif OCCURRENCE == name:
            handler.startOccurrence()
            process_iid(attrs)
            stack.append(OCCURRENCE)
        elif BASE_NAME == name:
            handler.startName()
            process_iid(attrs)
            stack.append(BASE_NAME)
        elif name in (BASE_NAME_STRING, RESOURCE_DATA):
            self._content = []
            self._accept_content = True
        elif VARIANT == name:
            iid = attrs.get((None, 'id'), None)
            if iid:
                iid = create_locator('#' + iid)
            variant = Variant(iid)
            # Inherit the scope from the variant's parents
            for v in self._variants:
                variant.scope.extend(v.scope)
            self._variants.append(variant)
        elif SCOPE == name:
            handler.startScope()
            self._seen_scope = True
            self._process_mergemap_scope()
            stack.append(SCOPE)
        elif TOPIC_MAP == name:
            stack.append(TOPIC_MAP)
            process_iid(attrs)
        elif MERGE_MAP == name:
            stack.append(MERGE_MAP)
        else:
            raise mio.MIOException('Unknown start element "%s"' % name)

    def endElementNS(self, (uri, name), qname):
        if uri != NS_XTM and uri != '':
            return
        stack = self._stack
        handler = self.map_handler
        if name in (INSTANCE_OF, SUBJECT_IDENTITY, TOPIC_MAP,
                    VARIANT_NAME, PARAMETERS, ROLE_SPEC, MEMBER):
            stack.pop()
        elif TOPIC == name:
            handler.endTopic()
            stack.pop()
        elif ASSOCIATION == name:
            stack.pop()
            if not self._seen_type:
                handler.type(XTM10ContentHandler._ASSOCIATION)
            self._process_mergemap_scope()
            handler.endAssociation()
        elif OCCURRENCE == name:
            stack.pop()
            if not self._seen_type:
                handler.type(XTM10ContentHandler._OCURRENCE)
            self._process_mergemap_scope()
            handler.endOccurrence()
        elif BASE_NAME == name:
            stack.pop()
            if not self._seen_type:
                handler.type(XTM10ContentHandler._TOPIC_NAME)
            self._process_mergemap_scope()
            handler.endName()
        elif BASE_NAME_STRING == name:
            handler.value(''.join(self._content))
            self._accept_content = False
        elif RESOURCE_DATA == name:
            parent_el = stack[-1]
            value = ''.join(self._content), XSD.string
            if OCCURRENCE == parent_el:
                handler.value(*value)
            elif VARIANT_NAME == parent_el:
                self._variants[-1].literal = value
            else:
                raise mio.MIOException('Unexpected parent element "%s" while processing "resourceData"' % parent_el)
            self._accept_content = False
        elif VARIANT == name:
            variant = self._variants.pop()
            handler.startVariant()
            handler.startScope()
            for theme in variant.scope:
                handler.theme(theme)
            handler.endScope()
            if variant.iid:
                handler.itemIdentifier(variant.iid)
            handler.value(*variant.literal)
            handler.endVariant();
        elif MERGE_MAP == name:
            pass
        elif SCOPE == name:
            stack.pop()
            handler.endScope()
        elif name in (TOPIC_REF, RESOURCE_REF, SUBJECT_INDICATOR_REF):
            pass # noop.
        else:
            raise mio.MIOException('Unknown end element "%s"' % name)

    def startDocument(self):
        pass

    def endDocument(self):
        self.reset()

    def characters(self, content):
        if self._accept_content:
            self._content.append(content)

    def _href(self, attrs):
        """\
        Returns an absolute IRI from the attributes.
        """
        return self._create_locator(attrs.get((NS_XLINK, 'href')))

    def _process_iid(self, attrs):
        iid = attrs.get((None, 'id'), None)
        if iid:
            return self.map_handler.itemIdentifier(self._create_locator('#%s' % iid))

    def _create_locator(self, reference):
        """\
        Returns a locator where ``reference`` is resolved against the base
        locator.
        """
        return resolve_iri(self._bases[-1], reference)

    def _process_topic_ref(self, parent_el, ref):
        """\
        Processes a ``topicRef`` element in the context of ``parent_el``.
        
        `parent_el`
            The parent element.
        `ref`
            An IRI
        """
        handler = self.map_handler
        if SUBJECT_IDENTITY == parent_el:
            handler.itemIdentifier(ref)
            return
        ref = mio.ITEM_IDENTIFIER, ref
        if INSTANCE_OF == parent_el:
            stack = self._stack 
            if ASSOCIATION in stack \
                or OCCURRENCE in stack \
                or BASE_NAME in stack: # instanceOf in name is invalid but Ontopia uses it
                handler.type(ref)
                self._seen_type = True
            elif TOPIC in stack:
                handler.isa(ref)
            else:
                raise mio.MIOException('Unexpected "instanceOf" element')
        elif MEMBER == parent_el:
            if not self._role_type:
                self._role_type = ref
            handler.startRole(self._role_type)
            handler.player(ref)
            handler.endRole()
        elif SCOPE == parent_el:
            self._process_theme(ref)
        elif PARAMETERS == parent_el:
            self._variants[-1].scope.append(ref)
        elif ROLE_SPEC == parent_el:
            self._role_type = ref
        elif MERGE_MAP == parent_el:
            pass
        else:
            raise mio.MIOException('Unexpected parent element "%s" while processing topicRef' % parent_el)

    def _process_mergemap_scope(self):
        """\
        
        """

    def _process_role(self, player):
        """\
        Creates an association role with the specified player.
        """
        handler = self.map_handler
        handler.startRole(self._role_type or XTM10ContentHandler._ROLE)
        handler.player(player)
        handler.endRole()

    def _process_theme(self, ref):
        """\
        
        """
        stack = self._stack
        if ASSOCIATION in stack or OCCURRENCE in stack or BASE_NAME in stack:
            self.map_handler.theme(ref)
        else:
            raise mio.MIOException('Unexpected parent element while processing scope')

    def _process_sid(self, ref):
        """\
        
        """
        parent_el = self._stack[-1]
        if SUBJECT_IDENTITY == parent_el:
            self.map_handler.subjectIdentifier(ref)
        else:
            self._process_topic_ref(parent_el, ref)

    def _process_resource_ref(self, ref):
        """\
        
        """
        handler = self.map_handler
        parent_el = self._stack[-1]
        if SUBJECT_IDENTITY == parent_el:
            handler.subjectLocator(ref)
        elif OCCURRENCE == parent_el:
            handler.value(ref, XSD.anyURI)
        elif VARIANT_NAME == parent_el:
            self._variants[-1].literal = ref, XSD.anyURI
        else:
            slo = mio.SUBJECT_LOCATOR, ref
            if MEMBER == parent_el:
                self._process_role(slo)
            elif SCOPE == parent_el:
                self._process_theme(slo)
            elif MERGE_MAP == parent_el:
                pass
            else:
                raise mio.MIOException('Unexpected parent element "%s" while processing resourceRef' % parent_el)

    def _add_scope(self, scoped):
        """\
        Adds the scope of the ``mergeMap`` element.
        """
        add_theme = scoped.scope.append
        for theme in self._scope:
            add_theme(theme)

    
    doc_iri = property(_get_doc_iri, _set_doc_iri)

class Variant(object):
    """\
    Internally used object to keep track about variants.
    """
    __slots__ = ['iid', 'literal', 'scope']
    def __init__(self, iid=None):
        """\
        
        """
        self.iid = iid
        self.literal = None
        self.scope = []
