import pki
import os
import copy
from M2Crypto import X509, RSA

my_org = "My Organization"
my_location = "My City and Country"
my_certname = "My Certificate"
my_caname = "My Certificate Authority"

def passphrase_callback(*args):
    """this is a callback that will return the password for the ca
    when called. In real life you'll use an equivalent mechanism but
    will make sure you get the password from the user or from a secured
    file somewhere...
    """
    return 'pass'

def test_create_request():
    pubkey, privkey = pki.create_rsa_keyring(
            passphrase_callback=passphrase_callback)

    evp_pubkey = pki.as_evp_pkey(pubkey)
    evp_privkey = pki.as_evp_pkey(privkey)

    pki.create_cert_request(
            pubkey=evp_pubkey,
            privkey=evp_privkey,
            digest="md5",
            O=my_org,
            L=my_location,
            CN=my_certname)


def test_create_ca():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
            passphrase_callback=passphrase_callback,
            O=my_org,
            L=my_location,
            CN=my_caname)

    assert ca_cert != None
    assert pubkey != None
    assert privkey != None

def test_create_x509_certificate():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

def test_x509_verify():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    result = pki.verify_x509_certificate(ca_cert, cert)

    assert result == 1

def test_encrypt():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    result = pki.encrypt(cert.get_pubkey(), 'clear data')
    assert result != ''

def test_decrypt():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    clear_data = 'clear_data'
    k = pki.encrypt(cert.get_pubkey(), clear_data)
    p = pki.decrypt(priv, k)
    assert p == clear_data

def test_decrypt_using_files():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    # save the files to disk
    cert.save_pem('ca.cert')
    pub.save_key('pub')
    priv.save_key('priv', callback=passphrase_callback)
    
    # remove references from memory
    del cert
    del pub
    del priv

    # reload the certs from disk
    f = open('ca.cert', 'r')
    ca_cert_content = f.read()
    f.close()

    cert = X509.load_cert_string(ca_cert_content)
    pub = RSA.load_pub_key('pub')
    priv = RSA.load_key('priv', callback=passphrase_callback)

    # remove the files from disk
    os.unlink('ca.cert')
    os.unlink('pub')
    os.unlink('priv')

    # test some data encryption/decryption with the reloaded keys
    clear_data = 'clear_data'
    k = pki.encrypt(cert.get_pubkey(), clear_data)
    p = pki.decrypt(priv, k)
    assert p == clear_data

def test_sign_file():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    sig = pki.sign_file(priv, 'test/big.dat')
    assert sig != ''

def test_verify_file():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    sig = pki.sign_file(priv, 'test/big.dat')
    pubkey = cert.get_pubkey()
    verify = pki.verify_file(pubkey, 'test/big.dat', sig)
    assert verify == 1


def test_sign():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    sig = pki.sign(priv, 'blablabla')
    assert sig != ''

def test_sign_verify():
    ca_cert, pubkey, privkey = pki.create_certificate_authority(
        passphrase_callback=passphrase_callback,
        O=my_org,
        L=my_location,
        CN=my_caname)

    cert, pub, priv = pki.create_x509_certificate(ca_cert, privkey,
            cert_passphrase_callback=passphrase_callback,
            ca_passphrase_callback=passphrase_callback,
            O=my_org, L=my_location,
            CN=my_certname)

    sig = pki.sign(priv, 'blablabla')
    pubkey = cert.get_pubkey()
    verify = pki.verify(pubkey, 'blablabla', sig)
    assert verify == 1

