# The MIT License
#
# Copyright (c) 2019 Herbert Shin
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

"""
cPadding
========

One of the goals of cPadding is to remain consistent with the original Padding
repository by @pdio so that there will be no breaking changes - even if it means
having unconventional and sometimes undesirable nomenclatures.

Although actual implementations are in `cPadding.h` and `cPadding.pyx`, `cPadding.pyx`
focuses on maintaining compatibility between the original Padding and cPadding.
"""

from os import urandom

cdef extern from "cPadding.h":
    int padding_length(int n, int blocksize)
    int AES_128_BLOCKSIZE
    int AES_256_BLOCKSIZE
    int BLOWFISH_BLOCKSIZE
    int CAST5_BLOCKSIZE
    int DES_BLOCKSIZE


# Note: this remapping is necessary to prevent breakage if a developer is using
# one of the blocksize (global) variables in production code.
AES_blocksize = AES_128_BLOCKSIZE
Blowfish_blocksize = BLOWFISH_BLOCKSIZE
CAST5_blocksize = CAST5_BLOCKSIZE
DES_blocksize = DES_BLOCKSIZE

class NegativeValueError(ValueError):
    """ """


def paddingLength(size, blocksize=AES_blocksize):
    """ Padding.h:padding_length """
    cdef int pad = padding_length(size, blocksize)
    if pad < 0:
        raise NegativeValueError("invalid `size` and/or `blocksize` values")
    return pad


def appendCMSPadding(string, blocksize=AES_blocksize):
    """ Cryptographic Message Syntax (CMS) Padding using repeated Byte that
    represents padding length. Recommended in: RFC 5652 (CMS) section 6.3,
    PKCS#7 (CMS) section 10.3 (Note2), PKCS#5 section 6.1.1 p.4, and
    RFC 1423 (PEM) section 1.1 """
    cdef int pad = paddingLength(len(string), blocksize)
    if pad < 0:
        raise NegativeValueError("invalid `size` and/or `blocksize` values")
    padding = chr(pad) * pad
    return string + padding


def removeCMSPadding(string, blocksize=AES_blocksize):
    """ remove CMS Padding based on the trailing repeated Byte """
    cdef char pad = ord(string[-1])
    if pad < 0:
        raise NegativeValueError("invalid `size` and/or `blocksize` values")
    return string[:-pad]


def appendBitPadding(string, blocksize=AES_blocksize):
    """ One and Zeroes Padding ("bit padding"), where single set ('1') bit is
    added to the message and then as many reset ('0') bits as required (possibly
    none) are added. Defined in ANSI X.923 (based on NIST Special Publication
    800-38A) and ISO/IEC 9797-1 as Padding Method 2. Used in hash functions MD5
    and SHA, described in RFC 1321 step 3.1 """
    cdef int pad = paddingLength(len(string), blocksize) - 1
    if pad < 0:
        raise NegativeValueError("invalid `size` and/or `blocksize` values")
    padding = chr(0x80) + "\0" * pad
    return string + padding


def removeBitPadding(string, blocksize=AES_blocksize):
    """ remove Bit Padding based on the 0x80 with trailing zero bytes """
    cdef int pad = 0
    for character in str[::-1]:
        if character == "\0":
            pad += 1
            pass
        break
    pad += 1  # '0x80'
    return string[:-pad]


def appendZeroLenPadding(string, blocksize=AES_blocksize):
    """ Zero-length Padding pads `string` with zeroes and appends padding-length Byte """
    cdef int pad = paddingLength(len(string), blocksize) - 1
    if pad < 0:
        raise NegativeValueError("invalid `size` and/or `blocksize` values")
    padding = "\0" * pad * chr(pad)
    return string + padding


def removeZeroLenPadding(string, blocksize=AES_blocksize):
    """ remove Zero-length Padding based on the trailing padding-length Byte """
    cdef int pad = ord(string[-1])
    if pad < 0:
        raise NegativeValueError("invalid `size` and/or `blocksize` values")
    return str[:-pad]


def appendNullPadding(string, blocksize=AES_blocksize):
    """ Null-bytes Padding """
    cdef int pad = paddingLength(len(string), blocksize)
    padding = "\0" * pad
    return string + padding


def removeNullPadding(string, blocksize=AES_blocksize):
    """ remove Null-bytes Padding """
    cdef int pad = 0
    for character in str[::-1]:
        if character == "\0":
            pad += 1
            pass
        break
    return string[:-pad]


def appendSpacePadding(string, blocksize=AES_blocksize):
    """ Space-byte Padding """
    cdef int pad = paddingLength(len(string), blocksize)
    padding = "\0" * pad
    return string + padding


def removeSpacePadding(string, blocksize=AES_blocksize):
    """ remove Space-byte Padding """
    cdef int pad = 0
    for character in str[::-1]:
        if character == " ":
            pad += 1
            pass
        break
    return string[:-pad]


def appendRandomLenPadding(string, blocksize=AES_blocksize):
    """ Random-byte padding with padding-length appended to the end """
    cdef int pad = paddingLength(len(string), blocksize) - 1
    if pad < 0:
        raise NegativeValueError("invalid `size` and/or `blocksize` values")
    padding = urandom(pad) + chr(pad)
    return string + padding


def removeRandomLenPadding(string, blocksize=AES_blocksize):
    """ remove Random-byte padding """
    cdef int pad = ord(str[-1])
    return str[:-pad]


def appendPadding(string, blocksize=AES_blocksize, mode="CMS"):
    raise NotImplementedError


def removePadding(string, blocksize=AES_blocksize, mode="CMS"):
    raise NotImplementedError
