
------------------------------------------------------ temporal logic functions

/* codex_instant: Create a daterange object from one date
 * notes: - the upper bound of the range is excluded,
 *          ('[)') and set to the date plus one day
 */
CREATE OR REPLACE FUNCTION codex_instant (
    instant_date date
    ) RETURNS daterange AS $$
    SELECT daterange(instant_date, instant_date + 1)
    $$ LANGUAGE SQL IMMUTABLE;

/* codex_is_period: Test if a daterange object is an instant
 */
CREATE OR REPLACE FUNCTION codex_is_instant (
    instant_or_period daterange
    ) RETURNS boolean AS $$
    SELECT CASE upper(instant_or_period) - lower(instant_or_period)
           WHEN 1 THEN TRUE
           ELSE FALSE
           END;
    $$ LANGUAGE SQL IMMUTABLE;

/* codex_period: Create a daterange object from two dates
 * notes: - the upper bound of the range is excluded,
 *          and set to the end date plus one day
 */
CREATE OR REPLACE FUNCTION codex_period (
    start_date date,
    end_date date
    ) RETURNS daterange AS $$
    SELECT daterange(start_date, coalesce(end_date, start_date) + 1)
    $$ LANGUAGE SQL IMMUTABLE;

/* codex_is_period: Test if a daterange object is a period
 */
CREATE OR REPLACE FUNCTION codex_is_period (
    instant_or_period daterange
    ) RETURNS boolean AS $$
    SELECT NOT codex_is_instant(instant_or_period);
    $$ LANGUAGE SQL IMMUTABLE;

/* codex_contains: Check if a date is contained in an array of periods
 * example:
 *   <- SELECT codex_contains(
 *          '2008-05-15'::date,
 *          '{
 *          "[2008-04-01,2008-04-30)",
 *          "[2008-05-01,2008-05-30)",
 *          "[2008-06-01,2008-06-30)"
 *          }'::daterange[]);
 *   -> t
 * notes: - adapted from https://github.com/worden341/pgchronos
 */
CREATE OR REPLACE FUNCTION codex_contains (
    query_date date,
    periods daterange[]
    ) RETURNS boolean AS $$
    SELECT coalesce(
               (SELECT TRUE
                  FROM (SELECT unnest(periods) AS period)
                    AS period
                 WHERE query_date <@ period.period
                 LIMIT 1),
               FALSE);
    $$ LANGUAGE SQL IMMUTABLE;

CREATE OR REPLACE FUNCTION codex_contains (
    periods daterange[],
    query_date date
    ) RETURNS boolean AS $$
    SELECT codex_contains(query_date, periods);
    $$ LANGUAGE SQL IMMUTABLE;

/* codex_union: Return the union of a set of periods
 * example:
 *   <- SELECT codex_union('{
 *          "[2008-03-01,2008-03-02)",
 *          "[2008-03-02,2008-04-01)",
 *          "[2008-04-02,2008-05-01)",
 *          "[2008-04-02,2018-05-01)"
 *      }'::daterange[]);
 *   -> {"[2008-03-01,2008-04-01)","[2008-04-02,2018-05-01)"}
 * notes: - adapted from https://github.com/worden341/pgchronos
 */
CREATE OR REPLACE FUNCTION codex_union (
    periods daterange[]
    ) RETURNS daterange[] AS $$
    SELECT array_agg(merged_periods)
      FROM (
             SELECT daterange(start_date, min(end_date)) AS merged_periods
               FROM (
                    -- list lower bounds of each set of overlapping periods, A
                    SELECT DISTINCT lower(period) AS start_date
                      FROM unnest(periods) AS period
                     WHERE NOT codex_contains(lower(period)-1, periods)
                    )
                 AS t_in
               JOIN (
                    -- list upper bounds of each set of overlapping periods, B
                    SELECT upper(period) AS end_date
                      FROM unnest(periods) AS period
                     WHERE NOT codex_contains(upper(period), periods)
                    )
                 AS t_out
                    -- get the cartesian product of A and B,
                    -- with the constraint that Ai < Bi
                 ON (t_in.start_date < t_out.end_date)
           GROUP BY t_in.start_date
           ORDER BY t_in.start_date
           )
        AS results;
    $$ LANGUAGE SQL IMMUTABLE;

-------------------------------------------------- vocabulary-related functions

/* codex_concept_descendants: Return a subset of the concept table with
 * all the concepts descending from a query concept (including itself);
 * the descendants are sorted by increasing distance from the concept
 * example:
 *   <- SELECT concept_id, concept_name FROM codex_concept_descendants(436665);
 *   -> +------------+-----------------------------------------------------+
 *      | concept_id | concept_name                                        |
 *      +------------+-----------------------------------------------------+
 *      |    4307956 | Bipolar II                                          |
 *      |    4200385 | Severe bipolar disorder without psychotic features  |
 *      |    4220617 | Severe bipolar I disorder, single manic episode ... |
 *      |     432290 | Mild bipolar I disorder, single manic episode       |
 *      |     439248 | Mixed bipolar affective disorder, moderate          |
 *       ...
 */
CREATE OR REPLACE FUNCTION codex_concept_descendants (
    concept_id integer
    ) RETURNS SETOF concept AS $$
      SELECT concept.*
        FROM concept,
             concept_ancestor
       WHERE (concept_ancestor.ancestor_concept_id = $1)
         AND (concept.concept_id = concept_ancestor.descendant_concept_id)
    ORDER BY concept_ancestor.min_levels_of_separation;
    $$ LANGUAGE SQL STABLE;

/* codex_concept_descendants: Return a subset of the concept table with
 * all the concepts descending from a list of query concepts (including these);
 * the descendants are sorted by increasing distance from the query concepts
 * example:
 *   <- SELECT * FROM codex_concept_descendants(array[1234, 5678]);
 */
CREATE OR REPLACE FUNCTION codex_concept_descendants (
    concept_id integer[]
    ) RETURNS SETOF concept AS $$
      SELECT concept.*
        FROM concept,
             concept_ancestor
       WHERE (concept_ancestor.ancestor_concept_id IN (
              SELECT unnest($1) AS concept_ids))
         AND (concept.concept_id = concept_ancestor.descendant_concept_id)
    ORDER BY concept_ancestor.min_levels_of_separation;
    $$ LANGUAGE SQL STABLE;

/* codex_concept_ancestors: Return a subset of the concept table with
 * all the concepts leading to a query concept (including itself);
 * the ancestors are sorted by increasing distance from the concept
 * example:
 *   <- SELECT concept_id, concept_name FROM codex_concept_ancestors(436665);
 *   -> +------------+------------------+
 *      | concept_id |   concept_name   |
 *      +------------+------------------+
 *      |     432586 | Mental disorder  |
 *      |     436665 | Bipolar disorder |
 *      |     441840 | Clinical finding |
 *      |     444100 | Mood disorder    |
 *      |    4274025 | Disease          |
 *      +------------+------------------+
 */
CREATE OR REPLACE FUNCTION codex_concept_ancestors (
    concept_id integer
    ) RETURNS SETOF concept AS $$
      SELECT concept.*
        FROM concept,
             concept_ancestor
       WHERE (concept_ancestor.descendant_concept_id = $1)
         AND (concept.concept_id = concept_ancestor.ancestor_concept_id)
    ORDER BY concept_ancestor.min_levels_of_separation;
    $$ LANGUAGE SQL STABLE;

/* codex_concept_ancestors: Return a subset of the concept table with
 * all the concepts leading to a list of query concepts (including these);
 * the ancestors are sorted by increasing distance from the query concepts
 * example:
 *   <- SELECT * FROM codex_concept_ancestors(array[1234, 5678]);
 */
CREATE OR REPLACE FUNCTION codex_concept_ancestors (
    concept_id integer[]
    ) RETURNS SETOF concept AS $$
      SELECT concept.*
        FROM concept,
             concept_ancestor
       WHERE (concept_ancestor.descendant_concept_id IN (
              SELECT unnest($1) AS concept_id))
         AND (concept.concept_id = concept_ancestor.ancestor_concept_id)
    ORDER BY concept_ancestor.min_levels_of_separation;
    $$ LANGUAGE SQL STABLE;
