-- COmmon Data model Events eXtraction (CODEX) for CDMv4

DROP TABLE IF EXISTS events CASCADE;

CREATE TABLE events (
    person_id INT8 NOT NULL,  -- identifier of the person associated to this event
    event_class INT2 NOT NULL,  -- class of event, from a pre-established set of constants
    event_id INT8,  -- identifier of this event, in the original table
    event_concept_id INT4,  -- concept identifier for this event
    event_type_concept_id INT4,  -- concept identifier for the type of event
    event_visit_id INT8,  -- visit associated to this event
    event_start_date DATE NOT NULL ENCODE DELTA32K,  -- start date for this event
    event_duration INT2 NOT NULL,  -- duration of this event, in days
    event_result VARCHAR(255) ENCODE TEXT255  -- value(s) associated with this event
)
DISTSTYLE KEY
DISTKEY (person_id)
SORTKEY (person_id, event_start_date);

-- Birth events
INSERT INTO events
    SELECT
        person_id AS person_id,
        %(event_class)d AS event_class,
        NULL AS event_id,  -- birth records have no primary identifier
        NULL AS event_concept_id,
        NULL AS event_type_concept_id,
        NULL AS event_visit_id,
        TO_DATE(
            year_of_birth
            || '-' ||
            LPAD(COALESCE(month_of_birth, 1), 2, 0)
            || '-' ||
            LPAD(COALESCE(day_of_birth, 1), 2, 0),
            'YYYY-MM-DD'
        ) AS event_start_date,
        1 AS event_duration,
        (
            'gender_concept_id=' || gender_concept_id
            || ';' ||
            'race_concept_id=' ||
            DECODE(race_concept_id,
                NULL, '',
                0, '',
                race_concept_id::TEXT)
            || ';' ||
            'ethnicity_concept_id=' ||
            DECODE(ethnicity_concept_id,
                NULL, '',
                0, '',
                ethnicity_concept_id::TEXT)
        ) AS event_result
    FROM person;

-- Death events
INSERT INTO events
    SELECT
        person_id AS person_id,
        %(event_class)d AS event_class,
        NULL AS event_id,  -- entries in 'death' have no primary identifier
        death_type_concept_id AS event_concept_id,
        cause_of_death_concept_id AS event_type_concept_id,
        NULL AS event_visit_id,  -- entries in 'death' have no associated visit
        death_date AS event_start_date,
        1 AS event_duration,
        NULL AS event_result  -- entries in 'death' have no associated result
    FROM death;

-- Visit occurrences
INSERT INTO events
    SELECT
        person_id AS person_id,
        %(event_class)d AS event_class,
        visit_occurrence_id AS event_id,
        place_of_service_concept_id AS event_concept_id,
        NULL AS event_type_concept_id,  -- entries in 'visit_occurrence' have no associated type concept
        NULL AS event_visit_id,  -- entries in 'visit_occurrence' have no associated visit
        visit_start_date AS event_start_date,
        (CASE
            WHEN visit_end_date IS NOT NULL THEN
                DATEDIFF(DAY,
                    visit_start_date,
                    visit_end_date) + 1
            ELSE 1
        END) AS event_duration,
        NULL AS event_result  -- entries in 'visit_occurrence' have no associated result
    FROM visit_occurrence;

-- Drug exposures
INSERT INTO events
    SELECT
        person_id AS person_id,
        %(event_class)d AS event_class,
        drug_exposure_id AS event_id,
        drug_concept_id AS event_concept_id,
        drug_type_concept_id AS event_type_concept_id,
        visit_occurrence_id AS event_visit_id,
        drug_exposure_start_date AS event_start_date,
        (CASE
            WHEN days_supply IS NOT NULL THEN
                days_supply
            WHEN drug_exposure_end_date IS NOT NULL THEN
                DATEDIFF(DAY,
                    drug_exposure_start_date,
                    drug_exposure_end_date) + 1
            ELSE 1
        END) AS event_duration,
        (
            'refills=' ||
            DECODE(refills,
                NULL, '',
                0, '',
                refills::TEXT)
            || ';' ||
            'quantity=' ||
            DECODE(quantity,
                NULL, '',
                0, '',
                quantity::TEXT)
            || ';' ||
            'relevant_condition_concept_id=' ||
            DECODE(relevant_condition_concept_id,
                NULL, '',
                0, '',
                relevant_condition_concept_id::TEXT)
        ) AS event_result
    FROM drug_exposure;

-- Procedure occurrences
INSERT INTO events
    SELECT
        person_id AS person_id,
        %(event_class)d AS event_class,
        procedure_occurrence_id AS event_id,
        procedure_concept_id AS event_concept_id,
        procedure_type_concept_id AS event_type_concept_id,
        visit_occurrence_id AS event_visit_id,
        procedure_date AS event_start_date,
        1 AS event_duration,
        (
            'relevant_condition_concept_id=' ||
            DECODE(relevant_condition_concept_id,
                NULL, '',
                0, '',
                relevant_condition_concept_id::TEXT)
        ) AS event_result
    FROM procedure_occurrence;

-- Condition occurrences
INSERT INTO events
    SELECT
        person_id AS person_id,
        %(event_class)d AS event_class,
        condition_occurrence_id AS event_id,
        condition_concept_id AS event_concept_id,
        condition_type_concept_id AS event_type_concept_id,
        visit_occurrence_id AS event_visit_id,
        condition_start_date AS event_start_date,
        (CASE
            WHEN condition_end_date IS NOT NULL THEN
                DATEDIFF(DAY,
                    condition_start_date,
                    condition_end_date) + 1
            ELSE 1
        END) AS event_duration,
        NULL AS event_result  -- entries in 'condition_occurrence' have no associated result
    FROM condition_occurrence;

-- Observations
INSERT INTO events
    SELECT
        person_id AS person_id,
        %(event_class)d AS event_class,
        observation_id AS event_id,
        observation_concept_id AS event_concept_id,
        observation_type_concept_id AS event_type_concept_id,
        visit_occurrence_id AS event_visit_id,
        observation_date AS event_start_date,
        1 AS event_duration,
        (
            'value_as_number=' ||
            DECODE(value_as_number,
                NULL, '',
                value_as_number::TEXT)
            || ';' ||
            'value_as_string=' ||
            DECODE(value_as_string,
                NULL, '',
                value_as_string::TEXT)
            || ';' ||
            'value_as_concept_id=' ||
            DECODE(value_as_concept_id,
                NULL, '',
                0, '',
                value_as_concept_id::TEXT)
            || ';' ||
            'range_low=' || range_low
            || ';' ||
            'range_high=' || range_high
        ) AS event_result
    FROM observation;
