#The MIT License (MIT)
#Copyright (c) 2014 Microsoft Corporation

#Permission is hereby granted, free of charge, to any person obtaining a copy
#of this software and associated documentation files (the "Software"), to deal
#in the Software without restriction, including without limitation the rights
#to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
#copies of the Software, and to permit persons to whom the Software is
#furnished to do so, subject to the following conditions:

#The above copyright notice and this permission notice shall be included in all
#copies or substantial portions of the Software.

#THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
#AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
#LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
#OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
#SOFTWARE.

import pydocumentdb.base as base
from pydocumentdb.routing.collection_routing_map import _CollectionRoutingMap
import pydocumentdb.routing.routing_range as routing_range

class _PartitionKeyRangeCache(object):
    '''
    _PartitionKeyRangeCache provides list of effective partition key ranges for a collection.
    This implementation loads and caches the collection routing map per collection on demand.

    '''
    def __init__(self, client):
        '''
        Constructor
        '''
        
        self._documentClient = client
        
        # keeps the cached collection routing map by collection id
        self._collection_routing_map_by_item = {}
        
    def get_overlapping_ranges(self, collection_link, partition_key_ranges):
        '''
        Given a partition key range and a collection, 
        returns the list of overlapping partition key ranges
        
        :Parameters:
            `collection_link` (str): the name of the collection
            `partition_key_range` (list): List of partition key range
        
        :Returns:
            list. List of overlapping partition key ranges
        '''
        cl = self._documentClient
        
        collection_id = base.GetResourceIdOrFullNameFromLink(collection_link)
        
        collection_routing_map = self._collection_routing_map_by_item.get(collection_id)
        if collection_routing_map is None:
            partitionKeyRanges = list(cl._ReadPartitionKeyRanges(collection_link))
            collection_routing_map = _CollectionRoutingMap.CompleteRoutingMap([(r, True) for r in partitionKeyRanges], collection_id)
            self._collection_routing_map_by_item[collection_id] = collection_routing_map
        return collection_routing_map.get_overlapping_ranges(partition_key_ranges)

class _SmartRoutingMapProvider(_PartitionKeyRangeCache):
    """
    Efficiently uses PartitionKeyRangeCach and minimizes the unnecessary invocation of _CollectionRoutingMap.get_overlapping_ranges()
    """
    def __init__(self, client):
        super(self.__class__, self).__init__(client)

    
    def _second_range_is_after_first_range(self, range1, range2):
        if range1.max > range2.min:
            ##r.min < #previous_r.max
            return False
        else:
            if (range2.min == range2.min and range1.isMaxInclusive and range2.isMinInclusive):
                # the inclusive ending endpoint of previous_r is the same as the inclusive beginning endpoint of r
                return False
        
        return True    

    def _is_sorted_and_non_overlapping(self, ranges):
        for idx, r in list(enumerate(ranges))[1:]:
            previous_r = ranges[idx-1]
            if not self._second_range_is_after_first_range(previous_r, r):
                return False
        return True
            
    def _subtract_range(self, r, partition_key_range):
        """
        Evaluates and returns r - partition_key_range
        :Parameters:
            partition_key_range (dict): partition key range
            r (routing_range._Range): query range
        :Returns:
            (routing_range._Range): the subtract r - partition_key_range
        """
        
        left = max(partition_key_range[routing_range._PartitionKeyRange.MaxExclusive], r.min)

        if left == r.min:
            leftInclusive = r.isMinInclusive
        else:
            leftInclusive = False

        queryRange = routing_range._Range(left, r.max, leftInclusive,
                r.isMaxInclusive)
        return queryRange
            
    def get_overlapping_ranges(self, collection_link, sorted_ranges):
        '''
        Given the sorted ranges and a collection,
        Returns the list of overlapping partition key ranges
        
        :Parameters:
            - collection_link (str): the collection link
            - sorted_ranges (list of routing_range._Range) the sorted list of non-overlapping ranges
        :Returns:
            (list of dict): list of partition key ranges
        :Raises:
            ValueError if two ranges in sorted_ranges overlap or if the list is not sorted
        '''
        
        # validate if the list is non-overlapping and sorted
        if not self._is_sorted_and_non_overlapping(sorted_ranges):
            raise ValueError("the list of ranges is not a non-overlapping sorted ranges")
        
        target_partition_key_ranges = []
        
        it = iter(sorted_ranges)
        try:
            currentProvidedRange = next(it)
            while True:
                if (currentProvidedRange.isEmpty()):
                    # skip and go to the next item\
                    currentProvidedRange = next(it)
                    continue;
                
                if len(target_partition_key_ranges):
                    queryRange = self._subtract_range(currentProvidedRange, target_partition_key_ranges[-1])
                else:
                    queryRange = currentProvidedRange
    
                overlappingRanges = _PartitionKeyRangeCache.get_overlapping_ranges(self, collection_link, queryRange)
                assert len(overlappingRanges), ("code bug: returned overlapping ranges for queryRange {} is empty".format(queryRange))
                target_partition_key_ranges.extend(overlappingRanges)

                lastKnownTargetRange = routing_range._Range.PartitionKeyRangeToRange(target_partition_key_ranges[-1])
                
                # the overlapping ranges must contain the requested range
                assert currentProvidedRange.max <= lastKnownTargetRange.max, "code bug: returned overlapping ranges {} does not contain the requested range {}".format(overlappingRanges, queryRange)
                
                # the current range is contained in target_partition_key_ranges just move forward
                currentProvidedRange = next(it)
                
                while currentProvidedRange.max <= lastKnownTargetRange.max:
                    # the current range is covered too. just move forward
                    currentProvidedRange = next(it)
        except StopIteration:
            # when the iteration is exhausted we get here. There is nothing else to be done
            pass
        
        return target_partition_key_ranges
