# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.

from bfc.frozendict import FrozenDict

__all__ = ["Frozenable", "freeze"]

import collections, abc

class Freezable(object):
    """Abstract base class for classes that have a freeze() method,
    that return a frozen copy of self."""
    __slots__ = []

    __metaclass__ = abc.ABCMeta
    
    @classmethod
    def __subclasshook__(cls, C):
        if cls is Freezable:
            if any("freeze" in B.__dict__ for B in C.__mro__):
                return True
        return NotImplemented

def _is_freezable(t):
    return issubclass(t, Freezable)

def _is_hashable(t):
    return issubclass(t, collections.Hashable)

def _is_mapping(t):
    return issubclass(t, collections.Mapping)

def _is_set(t):
    return issubclass(t, collections.Set)

def _is_iterable(t):
    return issubclass(t, collections.Iterable)

class FreezeException(Exception): pass

def freeze(obj):
    """Freezes an object.

    If the object is hashable, it is returned as is:

    >>> freeze(5)
    5
    >>> freeze('bar')
    'bar'
    >>> freeze((1,'x'))
    (1, 'x')

    Dictionaries (and all mapping types) are converted to FrozenDicts,
    while their keys and values are freezed as well:
    
    >>> d = freeze({1:2,3:4})
    >>> d
    FrozenDict({1: 2, 3: 4})

    Same goes for sets:
    
    >>> freeze(set([1,2,3]))
    frozenset([1, 2, 3])

    All iterables become tuples, while their items are also freezed:
    
    >>> freeze([1,2,[3,4],[[5]]])
    (1, 2, (3, 4), ((5,),))

    Any other types that do not match the above, can implement methods to freeze:

    - return a frozen copy via frozen():

    >>> class Foo(object):
    ...     def __init__(self, x):
    ...         self.x = x
    ...     __hash__ = None
    ...     def freeze(self):
    ...         print('Foo.freeze was called!')
    ...         return freeze(self.x)
    ... 
    >>> freeze(Foo([1,2]))
    Foo.freeze was called!
    (1, 2)

    >>> class Bar(object):
    ...     def __hash__(self):
    ...         raise NotImplemented()
    >>> try:
    ...     freeze(Bar())
    ... except FreezeException:
    ...     pass
    ... except:
    ...     assert False
    ... else:
    ...     assert False

    """
    t = type(obj)

    if _is_freezable(t):
        # freeze API update: if you prefer to self-freeze yourself,
        # return None instead of a new object
        r = obj.freeze()
        return obj if r is None else r

    if _is_hashable(t):
        try:
            hash(obj)
        except: # although a class is hashable, its hash can return NotImplemented...
            pass
        else:
            return obj
    
    if _is_mapping(t):
        return FrozenDict((freeze(k), freeze(v)) for k, v in obj.iteritems())

    if _is_set(t):
        return frozenset(freeze(o) for o in obj)

    if _is_iterable(t):
        return tuple(freeze(o) for o in obj)

    raise FreezeException('I do not know how to freeze %r of type %s' % (obj, t))
