# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Diff-like functionality in pure Python

>>> d = Diff('abc', 'abc')
>>> d.are_same
True
>>> d.write_full_diff()
 a
 b
 c
>>> d = Diff('aAbc', 'abc')
>>> d.are_same
False
>>> d.write_full_diff()
 a
-A
 b
 c
>>> d = Diff('aAbc', 'abBc')
>>> d.are_same
False
>>> d.write_full_diff()
 a
-A
 b
+B
 c
>>> d = Diff('aAbc', 'aabBc')
>>> d.are_same
False
>>> d.write_full_diff()
 a
-A
+a
 b
+B
 c
>>> import re
>>> def re_eq(l, r): return re.match(r, l)
>>> d = Diff(['Hello', 'World'], ['H.*', 'W.*'], re_eq)
>>> d.are_same
True
>>> d.write_full_diff(on_equal_print_left = False)
 H.*
 W.*
>>> d = Diff(['Hello', 'world'], ['H.*', 'W.*'], re_eq)
>>> d.are_same
False
>>> d.write_full_diff(on_equal_print_left = False)
 H.*
-world
+W.*

"""

__all__ = ['Diff']

from cache.func import hashable_cached

import operator, itertools

class MatchingIndecies(object):
    __slots__ = ['left', 'right']
    def __init__(self, l, r):
        self.left = l
        self.right = r

class DiffPart(object):
    __slots__ = ['left_start', 'left_end',
                 'right_start', 'right_end',
                 ]
    def __init__(self, ls, le, rs, re):
        self.left_start = ls
        self.left_end = le
        self.right_start = rs
        self.right_end = re

def longest_common_subsequence(a, b, eq = operator.eq, hashable_elements = True):
    if hashable_elements:
        eq = hashable_cached(eq)
    result = []
    a_start = 0
    a_max = len(a)
    b_start = 0
    b_max = len(b)
    while a_start < a_max and b_start < b_max and eq(a[a_start], b[b_start]):
        result.append(MatchingIndecies(a_start, b_start))
        a_start = a_start + 1
        b_start = b_start + 1

    if a_start == a_max or b_start == b_max:
        return result

    lengths = [[0] * (b_max - b_start + 1) for i in xrange(a_max - a_start + 1)]

    for i in xrange(a_max - a_start):
        for j in xrange(b_max - b_start):
            if eq(a[a_start + i], b[b_start + j]):
                lengths[i + 1][j + 1] = lengths[i][j] + 1
            else:
                lengths[i + 1][j + 1] = max(lengths[i+1][j], lengths[i][j+1])

    result2 = []
    i = a_max - 1
    j = b_max - 1
    while i >= a_start and j >= b_start:
        if eq(a[i], b[j]):
            result2.append(MatchingIndecies(i, j))
            i = i - 1
            j = j - 1
        elif lengths[i - a_start + 1][j - b_start] > lengths[i - a_start][j - b_start + 1]:
            j = j - 1
        else:
            i = i - 1
    result2.reverse()
    return result + result2

def iterdiff(a, b, eq, hashable_elements):
    lcs = longest_common_subsequence(a, b, eq, hashable_elements)
    lcs.append(MatchingIndecies(len(a), len(b)))
    a_index = 0
    b_index = 0
    for mi in lcs:
        if a_index != mi.left or b_index != mi.right:
            yield DiffPart(a_index, mi.left, b_index, mi.right)
        a_index = mi.left + 1
        b_index = mi.right + 1

def nop(x):
    return x

class Diff(object):
    __slots__ = ['left',
                 'right',
                 'result',
                 ]
    def __init__(self, a, b, eq = operator.eq, hashable_elements = True):
        self.left = a
        self.right = b
        self.result = list(iterdiff(a, b, eq, hashable_elements))

    @property
    def are_same(self):
        return not self.result

    def write_full_diff(self,
                        on_equal_print_left = True,
                        left_to_string = nop,
                        right_to_string = nop
                        ):
        if self.are_same:
            if on_equal_print_left:
                for x in self.left:
                    print ' ' + left_to_string(x)
            else:
                for x in self.right:
                    print ' ' + right_to_string(x)
        else:
            index = 0
            result = itertools.chain(self.result, (DiffPart(len(self.left), len(self.left),
                                                            len(self.right), len(self.right)),))
            if on_equal_print_left:
                for dp in result:
                    while index < dp.left_start:
                        print ' ' + left_to_string(self.left[index])
                        index = index + 1
                    for index in xrange(dp.left_start, dp.left_end):
                        print '-' + left_to_string(self.left[index])
                    for index in xrange(dp.right_start, dp.right_end):
                        print '+' + right_to_string(self.right[index])
                    index = dp.left_end
            else:
                for dp in result:
                    while index < dp.right_start:
                        print ' ' + right_to_string(self.right[index])
                        index = index + 1
                    for index in xrange(dp.left_start, dp.left_end):
                        print '-' + left_to_string(self.left[index])
                    for index in xrange(dp.right_start, dp.right_end):
                        print '+' + right_to_string(self.right[index])
                    index = dp.right_end
        
