import time

from queue import Queue

import math

from lfc import Worker
from lfc.workers.impl.dedicatedworker import WorkerFunction


class LimitedDedicatedWorker(Worker):
    def __init__(self, f, limit, eventmanager, initf=None, initkwargs={}, temp=False, queue=None, workers=0, **kwargs):
        Worker.__init__(self, f=f, eventmanager=eventmanager, **kwargs)

        self.__initf = initf
        self.__initkwargs = initkwargs

        self.limit = limit

        self.__running = False

        self.__queue = queue

        self.workers = workers

        self.temp = temp
        if not self.temp:
            self.__queue = Queue()

    """
    Public function to submit work for this worker

    This does not fire that work to the thread pool but instead saves it 
       and fires its own temporary worker to the threadpool

    Args:
        result=False: Boolean check to see if results are expected back
        **kwargs: Takes in a variable amount of keyword arguments
    """

    def submit(self, result=False, timeout=None, *args, **kwargs):
        try:
            # Create a worker function that wraps the function and args
            wf = WorkerFunction(f=self._f, *args, **kwargs)

            # Adds a tuple of args to the __queue
            self.__queue.put(wf)

            # Check to see if there is already a worker function running
            # If not function is running this will start it
            # if not self.__running:
            self.__startworker()

            # Check to see if results are wanted back
            if result:

                # Start Time
                s = time.time()

                # Loop while not finished
                while not wf.finished:
                    time.sleep(0.1)

                    # Attempt to start a worker, this will fail if one is already running
                    self.__startworker()

                    # Check for timeout and handle timeout conditions
                    if timeout is not None:
                        c = time.time()
                        if c - s > timeout:
                            break

                # Loop has concluded so there is either a result or an exception
                # Raise the exception is there is one
                # Otherwise return results
                if wf.exception is not None:
                    raise wf.exception
                else:
                    return wf.result

        except Exception as e:
            self.log.error(e)

    def __startworker(self):
        if self.temp:
            return

        if self.__running:
            return

        if self.workers >= self.limit:
            return

        if self.workers >= math.sqrt(self.__queue.qsize()):
            return

        self.workers += 1

        copy = LimitedDedicatedWorker(f=self._f, limit=self.limit, eventmanager=self.eventManager, initf=self.__initf, initkwargs=self.__initkwargs, temp=True, queue=self.__queue, workers=self.workers)

        self.eventManager._submit(copy._worker)

    # Internal worker function made to process all work in the queue
    # This function doesn't care about anything besides there being WorkerFunctions in the queue to run
    def _worker(self):
        try:
            # Attempt to use the initializer with the given initargs
            if self.__initf:
                self.__initf(**self.__initkwargs)

            # Start the loop on the condition the queue is not empty
            while True:
                wf = self.__queue.get()
                if wf is not None:
                    wf.run()

                if self.__queue.qsize() == 0 or self.eventManager.exit:
                    self.workers -= 1
                    print(f"BREAKING Queue Size = {self.__queue.qsize()}, Exit = {self.eventManager.exit}")
                    break
        except Exception as e:
            self.log.error(e)
