import queue
import time

from lfc.workers.worker import Worker

"""
One further implementation of the Worker class which adds functionality

The goal of the DedicatedWorker is to loop through a lot of work on one thread

It does this by reserving a thread when work is submitted, and instead of work 
being sent to the threadpool it is sent to the private queue.
"""

class DedicatedWorker(Worker):
    """
    Dedicated Worker Constructor

    Args:
        f: Function that this worker will be based around
        eventManager: A reference to the EventManager
        initializer: A function to be run within the thread to initialize anything
        initargs: Preset arguments that the initializer would need
        terminate: A function to be run when the thread is finishing
        termargs: Preset arguments that the terminate function will run
        **kwargs: Any number of keyword arguments that are passed in
    """
    def __init__(self, f, eventmanager, initializer=None, initargs=[], terminate=None, termargs=[], **kwargs):
        Worker.__init__(self, f, eventmanager, **kwargs)

        # The internal queue which will be housing all the work for this do to at any point in time
        self.__queue = queue.Queue()

        # The initializer and initargs are saved so that when a thread is reserved
        # by the internal _worker function it knows what to do at the start
        self.__initializer = initializer
        self.__initargs = initargs

        # The terminate and termargs are saved for any concluding things that
        # need to happen when the work is done
        self.__terminate = terminate
        self.__termargs = termargs

        # Private boolean checek to see if this worker already has a thread reserved
        self.__running = False

    """
    Public function to submit work for this worker
    
    This does not fire that work to the thread pool but instead saves it 
       and fires its own temporary worker to the threadpool
       
    Args:
        result=False: Boolean check to see if results are expected back
        **kwargs: Takes in a variable amount of keyword arguments
    """
    def submit(self, result=False, timeout=None, *args, **kwargs):
        try:
            #Create a worker function that wraps the function and args
            wf = WorkerFunction(f=self._f, *args, **kwargs)

            #Adds a tuple of args to the __queue
            self.__queue.put(wf)

            # Check to see if there is already a worker function running
            # If not function is running this will start it
            #if not self.__running:
            self.__startworker()

            # Check to see if results are wanted back
            if result:

                # Start Time
                s = time.time()

                # Loop while not finished
                while not wf.finished:
                    time.sleep(0.1)

                    # Attempt to start a worker, this will fail if one is already running
                    self.__startworker()

                    # Check for timeout and handle timeout conditions
                    if timeout is not None:
                        c = time.time()
                        if c - s > timeout:
                            break

                # Loop has concluded so there is either a result or an exception
                # Raise the exception is there is one
                # Otherwise return results
                if wf.exception is not None:
                    raise wf.exception
                else:
                    return wf.result

        except Exception as e:
            self.log.error(e)

    # Private function used to standardize and simplify the method at which
    #   the worker gets put into the threadpool
    def __startworker(self):

        # Check to see if there is already a worker running so it doesn't make duplicates
        # OR too see if there is stuff in the queue
        if self.__running:
            return

        # Since no worker is running this indicates that it is currently running
        #   and will be handling the work in the queue
        self.__running = True

        self.eventManager._submit(f=self._worker)

    # Internal worker function made to process all work in the queue
    # This function doesn't care about anything besides there being WorkerFunctions in the queue to run
    def _worker(self):
        try:
            # Attempt to use the initializer with the given initargs
            if self.__initializer is not None:
                self.__initializer(*self.__initargs)

            # Start the loop on the condition the queue is not empty
            while True:
                wf = self.__queue.get()
                if wf is not None:
                    wf.run()

                if self.__queue.qsize() == 0 or self.eventManager.exit:
                    break

            #Attempt to use the terminate funcition with the given args
            if self.__terminate is not None:
                self.__terminate(*self.__termargs)
        except Exception as e:
            self.log.error(e)

        # Change the __running boolean to indicate that this worker has stopped and no worker is currently running
        self.__running = False

class WorkerFunction:
    def __init__(self, f, *args, **kwargs):
        self.__f = f
        self.__args = args
        self.__kwargs = kwargs

        self.result = None
        self.exception = None
        self.finished = False

    def run(self):
        try:
            self.result = self.__f(*self.__args, **self.__kwargs)
        except Exception as e:
            self.exception = e
        self.finished = True
