import logging
import time
import traceback
from concurrent import futures
from concurrent.futures.thread import ThreadPoolExecutor
from queue import Queue

from lfc.enums.verbosity import Verbosity
from lfc.event import Event
from lfc.events.exit import Exit
from lfc.events.start import Start


class EventManager:
    """
    A bare bones event framework using concurrent futures for threading

    Args:
        threads = 50: Specifies the amount of threads that will be used. Default is 50
        verbosity = 0: Specifies the verbosity that the program will have to the logger
        logger = None: Optional argument to pass in a custom logger, if nothing is passed in a custom one will be made
    """
    def __init__(self, threads=50, verbosity=0, logger=None, autoexit=True):
        self.threads = threads

        self.autoexit = autoexit

        #Pull a verbosity
        self.verbosity = Verbosity.get(verbosity)

        # Handler Dictionary
        # Specifically, a dicionary were event class objects are keys and then a list of handler functions are the item
        # This allows for quick access to a set of handlers for a specific event ensuring that no
        #   additional sorting or checks need to be done to fire a specific event
        self.handlers = {}

        # Futures Dictionary
        # This is a private dictionary used internally so that the main thread can manage and clean
        #   the running futures generated by the event system
        # Items are put into this dictionary through the protected _submit function which is the only semi-public gateway
        #   for classes to submit events to the ThreadPool
        # Outline:
        #   Key= Future Object
        #   Item=(fn, args, kwargs)
        self.__futures = {}


        self.__queue = Queue()

        # ThreadPool
        # The main manager for threads and workers
        self.__threadPool = ThreadPoolExecutor(max_workers=self.threads)

        self.log = logger
        # This checks ensures that if no logger was passed in a logger will be made so that
        #   error and debug messages as all levels will be reported to the user
        if logger is None:
            self.log = logging.getLogger()
            formatter = logging.Formatter('%(asctime)s %(levelname)-8s %(message)s')
            printhandler = logging.StreamHandler()
            printhandler.setFormatter(formatter)
            self.log.addHandler(printhandler)
            self.log.setLevel(logging.DEBUG)

        # This boolean is set so that after firing the start event on the main thread there
        #   are no other instances of it that could be created and thus no duplicate cleaners
        self.started = False

        # This boolean is set so that any dedicated while loops not made by the Thread Pool Executor
        #   will be able to check if the program is meant to end
        self.exit = False

    """
    Representing the handling of a single event and a single thread
    
    This is the basis for the entire event system in terms of handling events in order 
        and making sure the exit event ends the program after it runs 

    Args:
        event: An event object specifying what needs to be run from this class
               along with what data the handler will need
    """
    def __handle(self, event):
        eventName = event.__class__.__name__
        if eventName in self.handlers.keys():

            # First sort the handlers for a specific event by the priority value
            handlers = sorted(self.handlers[eventName], key=lambda h: h.priority.value)

            # DEBUG & VERBOSITY
            if self.verbosity == Verbosity.VERBOSE:
                self.log.info("{} firing".format(eventName))
            elif self.verbosity == Verbosity.VERY_VERBOSE:
                self.log.info("{} firing to {}".format(eventName, [h.__name__ for h in handlers]))
            elif self.verbosity == Verbosity.DEBUG:
                self.log.info("{} (args={}, kwargs={}) firing to {}".format(eventName, event.args, event.kwargs, [h.__name__ for h in handlers]))

            # Loop through the ordered list of handlers
            for h in handlers:
                try:
                    # Fire the handler with args and kwargs
                    h(*event.args, **event.kwargs)
                except Exception as e:
                    self.log.error(e)

        if eventName.lower() == 'exit':
            self.exit = True



    """
    Fires an event into the system triggering all handlers for that event
    
    This is implemented as the only methodology to put a function into the threadpool, 
        because it forces all futures generated by putting something in the threadpool 
        to be added to the futures list and cleaned by the main thread
    
    Args:
        event: An event class that might contain data
    """
    def _submit(self, f, *args, **kwargs):
        try:
            # DEBUG & VERBOSITY
            if self.verbosity == Verbosity.VERBOSE:
                self.log.info("Submitting {} to Thread Pool".format(f.__name__))
            elif self.verbosity == Verbosity.VERY_VERBOSE:
                self.log.debug("Submitting to Thread Pool (f='{}', args={}, kwargs={})".format(f.__name__, args, kwargs))
            elif self.verbosity == Verbosity.DEBUG:
                self.log.debug("Submitting (f='{}', args={}, kwargs={}) to Thread Pool (Pool Size = {}) ".format(f.__name__, args, kwargs, len(self.__futures)))



            # Submits the function and args to the threadpool and gets a future object back
            future = self.__threadPool.submit(f, *args, **kwargs)
            future.add_done_callback(self.__callback)
            # Adds that future to the private futures list so that it can be tracked and cleaned
            self.__futures[future] = (f, args, kwargs)

            # Returns the future in the case that thread locking needs to be dealt with
            return future
        except Exception as e:
            traceback.print_exc()
            self.log.error(e)

    """
    A filter method for events that checks to see if it needs to be sent to the Thread Pool
    This is not necessary but it helps improve performance by making sure a event will actually do something

    Args:
        event: An Event object
    """
    def fire(self, event):
        try:
            # First check to see if the object recieved as an argument is actually an instance of an Event object
            if not isinstance(event, Event):
                return

            eventName = event.__class__.__name__

            # Check to see if there are handlers ready to catch the event
            if not eventName in self.handlers.keys() and eventName.lower() != 'exit':
                self.log.error("No handler for {}".format(eventName))
                return

            # Check to see if the end boolean is true meaning nothing else should be happening
            if self.exit:
                if self.verbosity >= Verbosity.DEFAULT:
                    self.log.debug("Exiting, no more events can be fired")
                return

            self.__queue.put(event)
            #self._submit(f=self.__handle, event=event)
        except Exception as e:
            self.log.error(e)

    """
    This triggers the start of the entire event system by launching the first event (Start)
    After the start event is fired this then uses the main thread for cleaning the futures made by events being fired
    The while loop that looks for cleaning opportunities is conditional on the self.end variable, and when 
    """
    def start(self):
        # Check to make sure that we aren't making duplicate main threads
        if self.started:
            return

        # Mark Start Time
        s = time.time()

        # Fire the start event
        # Creates a single thread that should trigger all functionality of the program
        self._submit(f=self.__handle, event=Start())

        # While loop on the main thread which handles cleaning on the conditional that the exit boolean isn't true
        while not self.exit:

            # If there is room to put more futures in, do that else clean and put futures in
            if len(self.__futures) < self.threads - 1:
                self.__submitEvents()
            self.__clean()

            # Check if the program should automatically exit when there is nothing else to do
            if self.autoexit:
                # Is there nothing left in queue
                if self.__queue.qsize() == 0 and len(self.__futures) == 0:
                    self._submit(f=self.__handle, event=Exit())

        # Clean out all done futures, in the case that something was missed by a clean attempt
        self.__clean()

        # Final clean through all submitted funcitons
        for f in futures.as_completed(self.__futures.keys()):
            self.__futures.pop(f)

        # Log Runtime
        if self.verbosity > Verbosity.DEFAULT:
            self.log.info("Runtime = {}".format(time.time()-s))

        # Code that is designed to run after the main while loop has shut down
        # First closing the threadPool which joins all current threads and ensures they have shutdown
        self.__threadPool.shutdown()

        # Fires exit on the main thread finalizing shutdown
        exit()

    def __clean(self):
        for f in [f for f in list(self.__futures.keys()) if f.done()]:
            self.__futures.pop(f)
            self.__submitEvents()

    def __submitEvents(self):
        # Fire more events bc there is room to handle more
        if len(self.__futures) < self.threads - 1 and self.__queue.qsize() > 0:
            # Determine how many events should be fired
            x = self.__queue.qsize()
            if x > self.threads:
                x = self.threads - len(self.__futures) - 1

            # Fire x number of events
            for i in range(0, x):
                self._submit(f=self.__handle, event=self.__queue.get())

    # This is added to the end of every function sent into the thread pool
    # Which forces all futures to report their exceptions when they finish
    def __callback(self, f):
        if f.exception():
            self.log.error(f.exception())
