class EventHandler:
    """
    Constructor meant to be used only by the event handler

    When called the handler class attempts to look at itself to
    find what it will be handling so it can give that
    information to the event manager

    Additionally it takes keyword arguments are translated into
    variables so the internal methods can have access to external classes

    Args:
        eventManager: A reference to the main EventManager object
    """
    def __init__(self, eventManager, **kwargs):
        self.eventManager = eventManager
        self.kwargs = kwargs
        self.log = eventManager.log
        self.verbosity = eventManager.verbosity

        # This loop will go through all functions within this class and subclass to determine
        #   what functions have been decorated and can handle an event
        for f in dir(self):
            try:
                # A check to see if the function is private and or a build in function of a class
                if f.startswith("__"):
                    continue

                # Gets a member function/variable of a class
                func = getattr(self, f)

                # Check if the pulled atr is callable meaning it is a function
                if not callable(func):
                    continue

                # Since everything at this point we know is a function that is not private we check
                #   if the function object has the attributes event and priority
                if hasattr(func, "event") and hasattr(func, 'priority'):
                    event = func.event

                    # Check if the event is none, meaning the decorator was used incorrectly
                    if event is None:
                        continue

                    # DEBUG
                    if self.verbosity == 'DEBUG':
                        self.log.debug("{} handling {}".format(func.__name__, event))

                    # At this point it is sure that the member is a function and has used the decorator correctly
                    #   so we are putting it into the handler dictionary so it will be used for the correct event
                    if event in self.eventManager.handlers.keys():
                        # Append the handler function to the list of handlers under a certain event
                        self.eventManager.handlers[event].append(func)
                    else:
                        # Used when there is no handlers for an event and so there
                        #   is no event key in the handlers dictionary
                        # This inserts a new list as an item under the new event key
                        self.eventManager.handlers[event] = [func]
            except Exception as e:
                #Logs any potential areas
                self.log.error(e)

        #This takes all keyword arguments and turns them into member variables
        for kw, arg in kwargs.items():
            setattr(self, kw, arg)

    #Wrapper function for the eventmanager.fire function
    def fire(self, event):
        self.eventManager.fire(event)







