from functools import wraps

from lfc.enums.priority import Priority
from lfc.event import Event

"""
This is a decorator for adding member variables to a function that should be a handler of a specific event

Args:
    event: Takes in an event which should either be an event class object or string
    priority=Priority.MEDIUM: Takes in an optional priority object which defaults to MEDIUM 
                                so that other events can schedule around the default values
                                
Example:

class ExampleHandler(EventHandler):
    
    @Catch(Start)
    def handleStart():
        foo()
        
    @Catch("Start")
    def handleStartFromString():
        bar()
        
    @Catch(Exit, priority=Priority.CRITICAL)
    def handleExitWithCriticalPriority():
        pass
"""
def Catch(event, priority=Priority.MEDIUM):

    # Internal function which takes in the actual function that is being decorated
    def HandlingFunction(f):

        # Check if the event is not None meaning it should be decorated as a handler
        if event is not None:

            # Check if the passed in event is a string so it doesnt need to pull the name
            if isinstance(event, str):
                f.event = event

            # Check to see if the class passed in is a subclass of the Event class from lfc
            elif issubclass(event, Event):
                f.event = event.__name__

            f.priority = priority

        # Creates a warper for the function call with all arguments needed so the function can be called normally
        @wraps(f)
        def wrapper(*args, **kwargs):
            # Actual function call
            return f(*args, **kwargs)
        return wrapper
    return HandlingFunction