﻿# -*- coding: UTF-8 -*-

__all__ = [
    'hub',
    'Entry',
    'Comment',
    'Tag',
    'Review',
    'Subscription',
    'User',
    'Group',
    'Permission',
    'VisitIdentity'
]


#import logging

from datetime import datetime, date

from sqlobject import *

from turbogears import identity, config
from turbogears.database import PackageHub

from docutils.core import publish_parts
from textile import textile
from elementtree import ElementTree

from cblog.utils.misc import et_textlist

#log = logging.getLogger('cblog.model')

hub = PackageHub('cblog')
__connection__ = hub

def rest2html(text):
    """Convert text (unicode encoded) in ReST syntax to HTML (unicode)."""

    overrides = dict(input_encoding='unicode', output_encoding='unicode')
    return publish_parts(text, settings_overrides=overrides,
      writer_name='html')['html_body']

def textile2html(text):
    """Convert text (unicode encoded) in textile syntax to HTML.

    HTML code will be ASCII encoded with numerical character entities
    (Workaround for bug in unicode handling of textile module).
    """

    return textile(text.encode('ascii','xmlcharrefreplace'))


class Formattible(object):
    """Base class for SQLObjects with 'text' attribute, supporting on-the-fly
    HTML comversion.

    FIXME: it should be possible for subclasses to inherit the _get_html_text
    method from this class.
    """

    # function that does text->HTML conversion
    # needs to be wrapped with 'staticmethod', so that it does not receive
    # a 'self' argument.
    _formatter = staticmethod(rest2html)


class Entry(SQLObject, Formattible):
    """A blog entry."""

    class sqlmeta:
        defaultOrder = '-created'

    created = DateTimeCol(default=datetime.now)
    author = ForeignKey('User', notNull=True)
    title = UnicodeCol(length=255, notNull=True)
    text = UnicodeCol(notNull=True)
    html_text = UnicodeCol(default=None)
    comments = SQLMultipleJoin('Comment')
    tags = SQLRelatedJoin('Tag', orderBy='name')
    subscriptions = SQLMultipleJoin('Subscription', orderBy='email')

    def _get_month(self):
        """Get the date of the first of the month when the entry was created."""

        return date(self.created.year, self.created.month, 1)

    def _get_comment_count(self):
        """Get the number of comments for this entry."""

        return self.comments.count()

    def _get_tag_count(self):
        """Get the number of comments for this entry."""

        return self.tag.count()

    def _get_teaser(self):
        """Get first paragraph of text of this entry.

        If the first paragraph is a heading, enclose it and the
        first paragraph in a DIV and return that.
        """

        d = ElementTree.fromstring(self.html_text)
        children = d.getchildren()
        try:
            firstpar = children[0]
        except IndexError:
            return None
        # TODO: determine length of first paragraph with HTML tags stripped
        firstpar_html = ElementTree.tostring(firstpar)
        if firstpar.tag in ['h1', 'h2', 'h3', 'h4', 'h5', 'h6'] or \
          len(firstpar_html) <= config.get('cblog.teaser.minlength', 100):
            teaser = ElementTree.Element('div')
            for child in children[:2]:
                teaser.append(child)
        else:
            teaser = firstpar
        teaser.set('class', 'teaser')
        return teaser

    def _get_description(self):
        """Get description of entry.

        ATM the description is generated by converting the teaser to plain text.
        The teaser usually consists of the first paragraph of the entry text,
        (plus a heading if present).
        """

        teaser = self.teaser
        if teaser:
            return ''.join(et_textlist(teaser)).strip()

    def _get_html_text(self):
        """Return entry text as HTML formatted string."""

        html = self._SO_get_html_text()
        if html is None:
            try:
                html = self._formatter(self.text)
                self._SO_set_html_text(html)
            except Exception, exc:
                return _('Error: Could not format text: %s') % exc
        return html

    def _set_text(self, value):
        if value is not None:
            try:
                html = self._formatter(value)
                self.html_text = html
            except:
                self.html_text = None
        self._SO_set_text(value)


class Review(SQLObject, Formattible):
    """A review for a featured blog article."""

    class sqlmeta:
        defaultOrder = '-created'

    created = DateTimeCol(default=datetime.now)
    author = ForeignKey('User', notNull=True)
    text = UnicodeCol(notNull=True)
    html_text = UnicodeCol(default=None)
    entry = ForeignKey('Entry')
    active = BoolCol(default=True)

    def _get_html_text(self):
        """Return entry text as HTML formatted string."""

        html = self._SO_get_html_text()
        if html is None:
            try:
                html = self._formatter(self.text)
                self._SO_set_html_text(html)
            except Exception, exc:
                return _('Error: Could not format text: %s') % exc
        return html

    def _set_text(self, value):
        if value is not None:
            try:
                html = self._formatter(value)
                self.html_text = html
            except:
                self.html_text = None
        self._SO_set_text(value)


class Comment(SQLObject, Formattible):
    """A comment to a blog entry."""

    class sqlmeta:
        defaultOrder = 'created'

    _formatter = staticmethod(textile2html)

    created = DateTimeCol(default=datetime.now)
    author = UnicodeCol(length=50, notNull=True)
    email =  UnicodeCol(length=255, notNull=True)
    homepage = UnicodeCol(length=255)
    ipAddress = StringCol(length=15)
    text = UnicodeCol(notNull=True)
    html_text = UnicodeCol(default=None)
    entry = ForeignKey('Entry', notNull=True)

    def _get_html_text(self):
        """Return entry text as HTML formatted string."""

        html = self._SO_get_html_text()
        if html is None:
            try:
                html = self._formatter(self.text)
                self._SO_set_html_text(html)
            except Exception, exc:
                return _('Error: Could not format text: %s') % exc
        return html

    def _set_text(self, value):
        if value is not None:
            try:
                html = self._formatter(value)
                self.html_text = html
            except:
                self.html_text = None
        self._SO_set_text(value)


class Tag(SQLObject):
    """A tag (category) for a blog entry.

    Entries can have unlimited tags.
    """

    class sqlmeta:
        defaultOrder = 'name'

    name = UnicodeCol(alternateID=True, length=100, notNull=True)
    entries = SQLRelatedJoin('Entry', orderBy='-created')

    def _get_entry_count(self):
        """Get the number of comments for this entry."""

        return self.entries.count()

    @classmethod
    def byLabel(cls, value):
        """Enter method docstring here."""

        try:
            return cls.select(func.LOWER(cls.q.name) == value.lower())[0]
        except IndexError:
            raise SQLObjectNotFound(value)


class Subscription(SQLObject):
    """A subscription to be notified of changes/comments to a post."""

    entry = ForeignKey('Entry', notNone=True)
    email = UnicodeCol(length=255)
    type = EnumCol(enumValues=['update', 'comment'], notNone=True)


# identity model classes
class Visit(SQLObject):
    class sqlmeta:
        table = "visit"

    visit_key = StringCol(length=40, alternateID=True,
                          alternateMethodName="by_visit_key")
    created = DateTimeCol(default=datetime.now)
    expiry = DateTimeCol()

    def lookup_visit(cls, visit_key):
        try:
            return cls.by_visit_key(visit_key)
        except SQLObjectNotFound:
            return None
    lookup_visit = classmethod(lookup_visit)


class VisitIdentity(SQLObject):
    visit_key = StringCol(length=40, alternateID=True,
      alternateMethodName='by_visit_key')
    user_id = IntCol()


class Group(SQLObject):
    """An ultra-simple group definition."""

    # names like "Group", "Order" and "User" are reserved words in SQL
    # so we set the name to something safe for SQL
    class sqlmeta:
        table='tg_group'

    group_name = UnicodeCol(length=16, alternateID=True,
      alternateMethodName="by_group_name")
    display_name = UnicodeCol(length=255)
    created = DateTimeCol(default=datetime.now)

    # collection of all users belonging to this group
    users = RelatedJoin('User', intermediateTable='user_group',
      joinColumn='group_id', otherColumn='user_id')

    # collection of all permissions for this group
    permissions = RelatedJoin( 'Permission', joinColumn='group_id',
      intermediateTable='group_permission', otherColumn='permission_id')


class User(SQLObject):
    """Reasonably basic User definition. Probably would want additional attributes."""

    # names like "Group", "Order" and "User" are reserved words in SQL
    # so we set the name to something safe for SQL
    class sqlmeta:
        table='tg_user'

    user_name = UnicodeCol(length=16, alternateID=True,
      alternateMethodName='by_user_name')
    email_address = UnicodeCol(length=255, alternateID=True,
      alternateMethodName='by_email_address')
    display_name = UnicodeCol(length=255)
    password = UnicodeCol(length=40)
    created = DateTimeCol(default=datetime.now)

    # groups this user belongs to
    groups = RelatedJoin('Group', intermediateTable='user_group',
      joinColumn='user_id', otherColumn='group_id')

    def _get_permissions(self):
        perms = set()
        for g in self.groups:
            perms = perms | set(g.permissions)
        return perms

    def _set_password(self, cleartext_password):
        """Runs cleartext_password through the hash algorithm before saving."""

        hash = identity.encrypt_password(cleartext_password)
        self._SO_set_password(hash)

    def set_password_raw( self, password ):
        """Saves the password as-is to the database."""

        self._SO_set_password(password)


class Permission(SQLObject):
    permission_name = UnicodeCol(length=16, alternateID=True,
      alternateMethodName='by_permission_name')
    description = UnicodeCol(length=255)
    groups = RelatedJoin('Group', intermediateTable='group_permission',
      joinColumn='permission_id', otherColumn='group_id')
