# Author: Seamus Wassman
# URL: http://github.com/SiCKRAGETV/SickRage/
#
# This file is part of SickRage.
#
# SickRage is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# SickRage is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SickRage.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import unicode_literals

import re
import traceback

import requests

import sickrage
from core.caches import tv_cache
from core.exceptions import AuthException
from core.helpers import bs4_parser
from providers import TorrentProvider


class MoreThanTVProvider(TorrentProvider):
    def __init__(self):
        super(MoreThanTVProvider, self).__init__("MoreThanTV")

        self.supportsBacklog = True

        self._uid = None
        self._hash = None
        self.username = None
        self.password = None
        self.ratio = None
        self.minseed = None
        self.minleech = None
        # self.freeleech = False

        self.urls = {'base_url': 'https://www.morethan.tv/',
                     'login': 'https://www.morethan.tv/login.php',
                     'detail': 'https://www.morethan.tv/torrents.php?id=%s',
                     'search': 'https://www.morethan.tv/torrents.php?tags_type=1&order_by=time&order_way=desc&action=basic&searchsubmit=1&searchstr=%s',
                     'download': 'https://www.morethan.tv/torrents.php?action=download&id=%s'}

        self.url = self.urls['base_url']

        self.cookies = None

        self.proper_strings = ['PROPER', 'REPACK']

        self.cache = MoreThanTVCache(self)

    def _checkAuth(self):

        if not self.username or not self.password:
            raise AuthException("Your authentication credentials for " + self.name + " are missing, check your config.")

        return True

    def _doLogin(self):
        if any(requests.utils.dict_from_cookiejar(self.session.cookies).values()):
            return True

        if self._uid and self._hash:
            requests.utils.add_dict_to_cookiejar(self.session.cookies, self.cookies)
        else:
            login_params = {'username': self.username,
                            'password': self.password,
                            'login': 'Log in',
                            'keeplogged': '1'}

            response = self.getURL(self.urls['login'], post_data=login_params, timeout=30)
            if not response:
                sickrage.srLogger.warning("Unable to connect to provider")
                return False

            if re.search('Your username or password was incorrect.', response):
                sickrage.srLogger.warning("Invalid username or password. Check your settings")
                return False

            return True

    def _doSearch(self, search_params, search_mode='eponly', epcount=0, age=0, epObj=None):

        results = []
        items = {'Season': [], 'Episode': [], 'RSS': []}

        # freeleech = '3' if self.freeleech else '0'

        if not self._doLogin():
            return results

        for mode in search_params.keys():
            sickrage.srLogger.debug("Search Mode: %s" % mode)
            for search_string in search_params[mode]:

                if mode is not 'RSS':
                    sickrage.srLogger.debug("Search string: %s " % search_string)

                searchURL = self.urls['search'] % (search_string.replace('(', '').replace(')', ''))
                sickrage.srLogger.debug("Search URL: %s" % searchURL)

                # returns top 15 results by default, expandable in user profile to 100
                data = self.getURL(searchURL)
                if not data:
                    continue

                try:
                    with bs4_parser(data) as html:
                        torrent_table = html.find('table', attrs={'class': 'torrent_table'})
                        torrent_rows = torrent_table.findChildren('tr') if torrent_table else []

                        # Continue only if one Release is found
                        if len(torrent_rows) < 2:
                            sickrage.srLogger.debug("Data returned from provider does not contain any torrents")
                            continue

                        # skip colheader
                        for result in torrent_rows[1:]:
                            cells = result.findChildren('td')
                            link = cells[1].find('span', attrs={'title': 'Download'}).parent
                            title_anchor = cells[1].find('a', attrs = {'dir': 'ltr'})

                            # skip if torrent has been nuked due to poor quality
                            if cells[1].find('img', alt='Nuked') is not None:
                                continue

                            torrent_id_long = link[b'href'].replace('torrents.php?action=download&id=', '')

                            try:
                                if title_anchor.has_key('title'):
                                    title = cells[1].find('a', {'title': 'View torrent'}).contents[0].strip()
                                else:
                                    title = title_anchor.contents[0]
                                download_url = self.urls['download'] % torrent_id_long

                                seeders = cells[6].contents[0]

                                leechers = cells[7].contents[0]

                                size = -1
                                if re.match(r'\d+([,\.]\d+)?\s*[KkMmGgTt]?[Bb]', cells[4].contents[0]):
                                    size = self._convertSize(cells[4].text.strip())

                            except (AttributeError, TypeError):
                                continue

                            if not all([title, download_url]):
                                continue

                            # Filter unseeded torrent
                            if seeders < self.minseed or leechers < self.minleech:
                                if mode is not 'RSS':
                                    sickrage.srLogger.debug(
                                            "Discarding torrent because it doesn't meet the minimum seeders or leechers: {0} (S:{1} L:{2})".format(
                                                    title, seeders, leechers))
                                continue

                            item = title, download_url, size, seeders, leechers
                            if mode is not 'RSS':
                                sickrage.srLogger.debug("Found result: %s " % title)

                            items[mode].append(item)

                except Exception as e:
                    sickrage.srLogger.error("Failed parsing provider. Traceback: %s" % traceback.format_exc())

            # For each search mode sort all the items by seeders if available
            items[mode].sort(key=lambda tup: tup[3], reverse=True)

            results += items[mode]

        return results

    def seedRatio(self):
        return self.ratio

    @staticmethod
    def _convertSize(sizeString):
        size = sizeString[:-2].strip()
        modifier = sizeString[-2:].upper()
        try:
            size = float(size)
            if modifier in 'KB':
                size = size * 1024
            elif modifier in 'MB':
                size = size * 1024 ** 2
            elif modifier in 'GB':
                size = size * 1024 ** 3
            elif modifier in 'TB':
                size = size * 1024 ** 4
        except Exception:
            size = -1
        return int(size)


class MoreThanTVCache(tv_cache.TVCache):
    def __init__(self, provider_obj):
        tv_cache.TVCache.__init__(self, provider_obj)

        # poll delay in minutes
        self.minTime = 20

    def _getRSSData(self):
        search_params = {'RSS': ['']}
        return {'entries': self.provider._doSearch(search_params)}
