# -*- coding: utf-8 -*-
import threading
import time
import logging

from qpython import qconnection

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.ERROR)

class Server:
    def __init__(self):
        self.name = ""
        self.host = ""
        self.port = 0
        self.username = ""
        self.password = ""
        self.pandas = False

    def tostring(self):
        return self.name


class ReConnectThread(threading.Thread):
    def __init__(self, server):
        super(ReConnectThread, self).__init__()
        self.server = server
        self.conn = None

    def run(self):
        while True:
            if ConnectionPool.instance is not None:
                logger.info("thread[%s] reconnect to %s", self.getName(), self.server.tostring())
                self.conn = ConnectionPool.instance.createconnection(self.server)
                if self.conn is not None and self.conn.is_connected():
                    if not self.server.tostring() in ConnectionPool.instance.busyDict:
                        ConnectionPool.instance.busyDict[self.server.tostring()] = []

                    ConnectionPool.instance.busyDict[self.server.tostring()].append(self.conn)
                    logger.info("thread[%s] reconnect OK", self.getName())
                    break
                else:
                    time.sleep(5)

    def join(self, timeout=None):
        threading.Thread.join(self)
        return self.conn


class ConnectionPool:
    instance = None

    def __init__(self):
        self.freeDict = {}
        self.busyDict = {}

    def init(self):
        pass

    @staticmethod
    def getinstance():
        if ConnectionPool.instance is None:
            ConnectionPool.instance = ConnectionPool()
        return ConnectionPool.instance

    def startreconnect(self, server):
        thread = ReConnectThread(server)
        thread.start()
        return thread.join()

    def checkconnection(self, server, conn):
        if conn.is_connected():
            try:
                conn.sync("{til 1}[]")
                self.busyDict[server.tostring()].append(conn)
                return conn
            except Exception as e:
                if e.args[0] == 10054:
                    self.purge(server)
                    return self.startreconnect(server)
        else:
            self.purge(server)
            return self.startreconnect(server)

    def purge(self, server):
        l = None
        if server.tostring() in self.freeDict:
            l = self.freeDict[server.tostring()]

            if len(l) > 0:
                for conn in l:
                    conn.close()
                    logger.info("DB[%s] conn is closed OK", server.tostring())

        self.busyDict = {}

        if l is not None:
            del l[:]

        logger.info("purge [%s] OK!", server.tostring())

    def createconnection(self, server):
        conn = None
        try:
            conn = qconnection.QConnection(host=server.host, port=server.port,
                                           username=server.username, password=server.password, pandas=server.pandas)
            conn.open()
        except Exception as e:
            logger.error(str(e))

        return conn

    def leaseconnection(self, server):
        conn = None
        l = None
        dead = []

        if server.tostring() in self.freeDict:
            l = self.freeDict[server.tostring()]

        if l is not None:
            for conn in l:
                if not conn.is_connected():
                    conn = None
                else:
                    break
        else:
            self.freeDict[server.tostring()] = []

        if conn is None:
            conn = self.createconnection(server)
        else:
            l.remove(conn)

        if not server.tostring() in self.busyDict:
            self.busyDict[server.tostring()] = []

        return conn

    def freeconnection(self, server, conn):
        if conn is None:
            return

        l = None
        if server.tostring() in self.busyDict:
            l = self.busyDict[server.tostring()]

        if l is not None:
            try:
                l.remove(conn)
            except:
                conn.close()

        if conn.is_connected():
            l = self.freeDict[server.tostring()]
            if l is None:
                conn.close()
            else:
                l.append(conn)


class Query:
    def __init__(self, server):
        self.conn = None
        self.server = server

    def query_sync(self, query, *parameters, **options):
        try:
            self.conn = ConnectionPool.getinstance().leaseconnection(self.server)
            self.conn = ConnectionPool.getinstance().checkconnection(self.server, self.conn)
            logger.debug("lease DB[%s] free:%s, busy:%s",
                         self.server.tostring(),
                         str(len(ConnectionPool.getinstance().freeDict[self.server.tostring()])),
                         str(len(ConnectionPool.getinstance().busyDict[self.server.tostring()])))
            if self.conn is not None:
                result = self.conn.sync(query, *parameters, **options)
                return result
            else:
                return None
        except Exception as e:
            logger.error("DB[%s] error code: %s, msg: %s", self.server.tostring(), str(e.args[0]), str(e.message))
            logger.error("query: %s", query)
            logger.error("parameters: %s", parameters)
            logger.error("options: %s", options)
        finally:
            ConnectionPool.getinstance().freeconnection(self.server, self.conn)
            logger.debug("free DB[%s] free:%s, busy:%s",
                         self.server.tostring(),
                         str(len(ConnectionPool.getinstance().freeDict[self.server.tostring()])),
                         str(len(ConnectionPool.getinstance().busyDict[self.server.tostring()])))
