try:
	import Tkinter as tk
except:
	import tkinter as tk

'''
version of Michael Lange's very nice ToolTip widget
originally posted at http://tkinter.unpythonic.net/wiki/ToolTip
adapted for python 3.x and multiple monitor compatibility
'''
class ToolTip:

	def __init__(self, master, text='Your text here', delay=1500, **opts):
		self.master = master
		self._opts = {'anchor':'center', 'bd':1, 'bg':'lightyellow', 'delay':delay, 'fg':'black',\
					  'follow_mouse':0, 'font':None, 'justify':'left', 'padx':4, 'pady':2,\
					  'relief':'solid', 'state':'normal', 'text':text, 'textvariable':None,\
					  'width':0, 'wraplength':150}
		self.configure(**opts)
		self._tipwindow = None
		self._id = None
		self._id1 = self.master.bind("<Enter>", self.enter, '+')
		self._id2 = self.master.bind("<Leave>", self.leave, '+')
		self._id3 = self.master.bind("<ButtonPress>", self.leave, '+')
		self._follow_mouse = 0
		if self._opts['follow_mouse']:
			self._id4 = self.master.bind("<Motion>", self.motion, '+')
			self._follow_mouse = 1

	def configure(self, **opts):
		for key in opts:
			if key in self._opts:
				self._opts[key] = opts[key]
			else:
				KeyError = 'KeyError: Unknown option: "%s"' %key
				raise KeyError

	##----these methods handle the callbacks on "<Enter>", "<Leave>" and "<Motion>"---------------##
	##----events on the parent widget; override them if you want to change the widget's behavior--##

	def enter(self, event=None):
		self._schedule()
		
	def leave(self, event=None):
		self._unschedule()
		self._hide()

	def motion(self, event=None):
		if self._tipwindow and self._follow_mouse:
			x, y = self.coords()
			self._tipwindow.wm_geometry("+%d+%d" % (x, y))

	##------the methods that do the work:---------------------------------------------------------##

	def _schedule(self):
		self._unschedule()
		if self._opts['state'] == 'disabled':
			return
		self._id = self.master.after(self._opts['delay'], self._show)

	def _unschedule(self):
		id = self._id
		self._id = None
		if id:
			self.master.after_cancel(id)

	def _show(self):
		if self._opts['state'] == 'disabled':
			self._unschedule()
			return
		if not self._tipwindow:
			self._tipwindow = tw = tk.Toplevel(self.master)
			# hide the window until we know the geometry
			tw.withdraw()
			tw.wm_overrideredirect(1)

			if tw.tk.call("tk", "windowingsystem") == 'aqua':
				tw.tk.call("::tk::unsupported::MacWindowStyle", "style", tw._w, "help", "none")

			self.create_contents()
			tw.update_idletasks()
			x, y = self.coords()
			tw.wm_geometry("+%d+%d" % (x, y))
			tw.deiconify()

	def _hide(self):
		tw = self._tipwindow
		self._tipwindow = None
		if tw:
			tw.destroy()

	def _top_window(self):
		tw = self.master
		while(hasattr(tw, 'master') and tw.master is not None):
			tw = tw.master
		return tw
				
	##----these methods might be overridden in derived classes:----------------------------------##

	def coords(self):
		# The tip window must be completely outside the master widget;
		# otherwise when the mouse enters the tip window we get
		# a leave event and it disappears, and then we get an enter
		# event and it reappears, and so on forever :-(
		# or we take care that the mouse pointer is always outside the tipwindow :-)
		tw = self._tipwindow
		twx, twy = tw.winfo_reqwidth(), tw.winfo_reqheight()
		# w, h = tw.winfo_screenwidth(), tw.winfo_screenheight()
		(w, h) = self._top_window().wm_maxsize()
		# calculate the y coordinate:
		if self._follow_mouse:
			y = tw.winfo_pointery() + 20
			# make sure the tipwindow is never outside the screen:
			if y + twy > h:
				y = y - twy - 30
		else:
			y = self.master.winfo_rooty() + self.master.winfo_height() + 3
			if y + twy > h:
				y = self.master.winfo_rooty() - twy - 3
		# we can use the same x coord in both cases:
		x = tw.winfo_pointerx() - twx / 2
		if x < 0:
			x = 0
		elif x + twx > w:
			x = w - twx
		return x, y

	def create_contents(self):
		opts = self._opts.copy()
		for opt in ('delay', 'follow_mouse', 'state'):
			del opts[opt]
		label = tk.Label(self._tipwindow, **opts)
		label.pack()
