#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
    Module SHCS.main
"""

import os
import os.path
import sys
import json
import logging
import shutil
from optparse import OptionParser


from flask import Flask, abort, request, send_from_directory
from werkzeug.utils import secure_filename

import shcs
from shcs.util import Trace, LEVEL, ConfigShcs
from shcs.login import Login, login_required
from shcs.doc import Doc

app = Flask(__name__)
app.doc = Doc(app, "/ws").doc
app.trace = Trace(app).trace
app.login = Login(app, "/ws")
app.config = ConfigShcs(app.config)

def checkpath(path):
    return True

############################ Route #############################################

@app.route("/ws/version", methods = ['GET'])
@app.doc.doc()
@app.trace
def url_version():
    """ 
    list of version: shcs,
    """
    return json.dumps([ {'module' : 'shcs'     , 'version' : shcs.__version__},
                        {'module' : 'python', 'version' :  '.'.join([str(a) for a in sys.version_info[0:3]])} 
                         ])    

@app.route("/")
@app.trace
def index():
    return send_from_directory(os.path.join(os.path.dirname(os.path.abspath(__file__)),'templates'),'shcs.html')

@app.route("/<path:path>")
@app.trace
def get(path):
    path = os.path.abspath(os.path.join(os.path.abspath(app.config['SHCS_CONF_DIR']), path))
    if os.path.isfile(path): 
        return send_from_directory(*os.path.split(path))
    return abort(404)


@app.route("/ws/path", methods = ['POST'])
@login_required
@app.doc.doc()
@app.trace
def path():
    """ 
    return list of files, list of dirs and parents

    **param**:

    - path
    """
    try:
        path = os.path.abspath(os.path.join(os.path.abspath(app.config['SHCS_CONF_DIR']), json.loads(request.data.decode())['path']))
        if os.path.isdir(path) and checkpath(path):
            res = {'files':[], 'dirs':[], 'parent': '/'.join(json.loads(request.data.decode())['path'].split('/')[:-1])}
            for p in os.listdir(path):
                p = os.path.join(path, p) 
                if os.path.isdir(p):
                    res['dirs'].append({'name': os.path.split(p)[1], 
                                        'path': p[len(os.path.abspath(app.config['SHCS_CONF_DIR']))+1:]
                                        })
                else:
                    res['files'].append({'name': os.path.split(p)[1],
                                        'path': p[len(os.path.abspath(app.config['SHCS_CONF_DIR']))+1:],
                                        'ext' : p.split('.')[-1] 
                                        })
 
            return json.dumps(res)
        else:
            raise Exception('%s is not dir' % request.data.decode())
    except:
        return json.dumps({'files':[], 'dirs':[], 'parent':''})

@app.route("/ws/rm", methods = ['POST'])
@login_required
@app.doc.doc()
@app.trace
def remove():
    """ 
    remove path
    """
    try:
        path = os.path.abspath(os.path.join(os.path.abspath(app.config['SHCS_CONF_DIR']), json.loads(request.data.decode())['path']))
        if checkpath(path):
            if os.path.isdir(path):
                shutil.rmtree(path)
            else:
                os.remove(path)
            return json.dumps({'rm': True}) 
        else:
            raise Exception('%s is not path' % request.data.decode())
    except Exception as e:
        app.logger.error(e)
        return abort(418)

@app.route("/ws/mkdir", methods = ['POST'])
@login_required
@app.doc.doc()
@app.trace
def mkdir():
    """ 
    create directory
    """
    try:
        path = os.path.abspath(os.path.join(os.path.abspath(app.config['SHCS_CONF_DIR']), json.loads(request.data.decode())['path']))
        name = json.loads(request.data.decode())['name'] 
        if checkpath(path):
            if os.path.isdir(path):
                os.makedirs(os.path.join(path,name))
            else:
                raise Exception('%s is not dir' % request.data.decode())
            return json.dumps({'mkdir': True}) 
        else:
            raise Exception('%s is not path' % request.data.decode())
    except Exception as e:
        app.logger.error(e)
        return abort(418)

@app.route("/ws/uploadcli/<path:path>", methods = ['POST'])
@login_required
@app.doc.doc()
@app.trace
def url_uploadcli(path):
    """ 
    upload file in directory

    **param**:

    - file
    """
    bfile = request.data
    if not len(bfile) or not len(path):
        return abort(418)
    try:
        path = os.path.abspath(os.path.join(os.path.abspath(app.config['SHCS_CONF_DIR']), path))
        with open(path, 'wb') as file_:
            file_.write(bfile)
        return json.dumps({'upload' : True})
    except Exception as e:
        app.logger.error(e)
        return abort(418)

@app.route("/ws/upload/", methods=['GET', 'POST'])
@login_required
@app.trace
def url_upload_racine():
    return url_upload('')

@app.route("/ws/upload/<path>", methods=['GET', 'POST'])
@login_required
@app.doc.doc()
@app.trace
def url_upload(path):
    if 'file' not in request.files:
        print('nofile')
        return abort(418)
    file = request.files['file']
    if file.filename == '':
        print('filename')
        return abort(418)
    if file:
        filename = secure_filename(file.filename)
        file.save(os.path.join(app.config['SHCS_CONF_DIR'], path, filename))
        return json.dumps({'upload' : True})
    return abort(418)

############################ Load #############################################

def main():
    parser = OptionParser(version="shcs %s" % shcs.__version__,
                             usage= "usage: %prog [options] args" )
    parser.description= "run a api server for Self Hosted Cloud Storage"
    parser.epilog = "by Frederic Aoustin"
    parser.add_option("-H", "--host",
        dest = "host",
        help ="the hostname to listen on",
        type = "string",
        default = "")
    parser.add_option("-p", "--port",
        dest = "port",
        help ="the port of the webserver",
        type = "int")
    parser.add_option("-d", "--dir",
        dest = "dir_shcs",
        help = "dir of shcs",
        type = "string",
        default = "")
    parser.add_option("-l", "--log",
        dest = "level",
        help = "level of log: %s" % ','.join(LEVEL),
        type = "string",
        default = "")
    parser.add_option("-c", "--conf",
        dest = "conf",
        help = "file configuration",
        type = "string",
        default = "")
    (options, args) = parser.parse_args()
    try:
        #value default
        app.config['SHCS_HOST'] = '0.0.0.0' 
        app.config['SHCS_PORT'] = 5000
        app.config['SHCS_LEVEL_LOG'] = logging.DEBUG
        app.config['SHCS_CONF_DIR'] = os.path.abspath('.')
        #load config from .shcs/conf.py
        app.config.from_pyfile(os.path.join(os.path.expanduser("~"),'.shcs','conf.py'), silent=True)
        #load config from env
        app.config.from_env('SHCS_')
        if app.config['SHCS_LEVEL_LOG'] in LEVEL.keys():
            app.config['SHCS_LEVEL_LOG'] = LEVEL[app.config['SHCS_LEVEL_LOG']]
        #load config from options
        if options.host:
            app.config['SHCS_HOST'] = options.host
        if options.port:
            app.config['SHCS_PORT'] = options.port
        if options.dir_shcs:
            app.config['SHCS_CONF_DIR'] = os.path.abspath(options.dir_shcs)
        if options.conf:
            app.config.from_pyfile(os.path.abspath(options.conf), silent=False)
        if options.level:
            app.config['SHCS_LEVEL_LOG'] = LEVEL[options.level]
        app.config.complete()
        #run webserver
        app.logger.setLevel(app.config['SHCS_LEVEL_LOG'])
        stream_handler = logging.StreamHandler()
        stream_handler.setLevel(app.config['SHCS_LEVEL_LOG'])
        app.logger.addHandler(stream_handler)
        app.run(host = app.config['SHCS_HOST'],
                port = app.config['SHCS_PORT'],
                threaded=True)
    except Exception as e:
        print(parser.error(e))
        parser.print_help()
        sys.exit(1)

if __name__ == "__main__":
    main()
 
