# -*- coding: utf-8 -*-
#
#	Konga Client library
#
#		Main module
#

import sys
import traceback
import atexit


DEFAULT_DISCOVER_TIMEOUT	= 5000
DEFAULT_CONNECT_TIMEOUT		= 30000
DEFAULT_EXECUTE_TIMEOUT		= 10000

CRYPT_NONE					= 0
CRYPT_LOW					= 1
CRYPT_NORMAL				= 2
CRYPT_HIGH					= 3

PROGRESS_PROCESS			= 0
PROGRESS_SEND				= 1
PROGRESS_EXECUTE			= 2
PROGRESS_RECEIVE			= 3
PROGRESS_COMPLETE			= 4

NIC_UP						= 0x1
NIC_CAN_BROADCAST			= 0x2
NIC_CAN_MULTICAST			= 0x4
NIC_IS_LOOPBACK				= 0x8

PROGRESS_INDEFINITE			= -1.0

ROUND						= 1
FLOOR						= 2
CEIL						= 3

BACKUP_ON_COMPUTER			= 0x1
BACKUP_ON_CLOUD				= 0x2



class Log(object):
	"""La classe Log serve ad immagazzinare i messaggi prodotti dal server durante un'operazione; oggetti di questa classe
	vengono popolati durante le chiamate asincrone della classe :class:`~kongalib.Client`.
	"""
	
	INFO		= 0		#: Costante che identifica i messaggi di tipo informativo
	WARNING		= 1		#: Costante che identifica i messaggi di tipo avviso
	ERROR		= 2		#: Costante che identifica i messaggi di tipo errore
	
	def __init__(self, title=''):
		self.title = title
		self.messages = []
		self.clear()
		self.state_stack = []
	
	def save(self):
		"""Salva lo stato del log"""
		self.state_stack.append((self.errors, self.warnings, len(self.messages)))
	
	def restore(self):
		"""Ripristina il log allo stesso stato in cui si trovava prima dell'ultima chiamata al metodo :meth:`save`."""
		self.errors, self.warning, count = self.state_stack.pop()
		self.messages[count:] = []
	
	def clear(self):
		"""Elimina tutti i messaggi presenti sul log."""
		self.messages[:] = []
		self.errors = False
		self.warnings = False
	
	def add_message(self, type, msg, name=None, row=None, colname=None):
		"""Aggiunge un messaggio *msg* di tipo *type*."""
		if type < Log.INFO or type > Log.ERROR:
			type = Log.ERROR
		self.messages.append((type, msg, name, row, colname))
		if type == Log.WARNING:
			self.warnings = True
		elif type == Log.ERROR:
			self.errors = True
	
	def info(self, message, name=None, row=None, colname=None):
		"""Esattamente come chiamare ``add_message(Log.INFO, message)``."""
		self.add_message(Log.INFO, message, name, row, colname)
	
	def warning(self, message, name=None, row=None, colname=None):
		"""Esattamente come chiamare ``add_message(Log.WARNING, message)``."""
		self.add_message(Log.WARNING, message, name, row, colname)
	
	def error(self, message, name=None, row=None, colname=None):
		"""Esattamente come chiamare ``add_message(Log.ERROR, message)``."""
		self.add_message(Log.ERROR, message, name, row, colname)
	
	def has_messages(self):
		"""Controlla la presenza di messaggi di qualsiasi tipo sul log."""
		return len(self.messages) > 0
	
	def has_warnings(self):
		"""Controlla la presenza di messaggi di tipo :attr:`WARNING` sul log."""
		return self.warnings
	
	def has_errors(self):
		"""Controlla la presenza di messaggi di tipo :attr:`ERROR` sul log."""
		return self.errors
	
	def get_messages(self, type=None):
		"""Ottiene la lista dei messaggi sul log. Se *type* è ``None``, tutti i messaggi vengono restituiti, altrimenti solo i messaggi di
		un determinato tipo."""
		if type is None:
			return self.messages
		return [ msg for msg in self.messages if msg[0] == type ]
	
	def strip_html(self, html):
		"""Elimina tutto il codice HTML dalla stringa in *html*, lasciando solo le parti testuali."""
		from HTMLParser import HTMLParser
		class Stripper(HTMLParser):
			def __init__(self):
				self.reset()
				self.fed = []
				self.feed(html)
			def handle_starttag(self, tag, attrs):
				if tag == 'br':
					self.fed.append('\n')
			def handle_startendtag(self, tag, attrs):
				if tag == 'br':
					self.fed.append('\n')
			def handle_data(self, d):
				self.fed.append(d)
			def get_data(self):
				return ''.join(self.fed)
		return Stripper().get_data()
	
	def format_message(self, message):
		"""Ritorna *message* formattato correttamente per essere visualizzato. *message* deve essere una tupla di almeno due elementi:
		``(tipo, testo)``, dove ``tipo`` è uno tra :attr:`INFO`, :attr:`WARNING` ed :attr:`ERROR`, mentre ``testo`` è il corpo del messaggio,
		che può includere codice HTML (che verrà soppresso nella formattazione)."""
		type = ('INFO', 'WARNING', 'ERROR')
		msg = self.strip_html(message[1])
		if not isinstance(msg, unicode):
			msg = msg.decode('utf-8', 'replace')
		return u'[%s] %s' % (type[message[0]], msg)
	
	def dumps(self):
		"""Esegue un dump dei messaggi presenti sul log e restituisce un'unica stringa. Eventuale codice HTML presente nei messaggi è eliminato
		usando :meth:`strip_html`."""
		return u'\n'.join([ self.format_message(message) for message in self.messages ])
	
	def dump(self):
		"""Esattamente come :meth:`dumps` ma stampa il dump su ``stdout``."""
		print self.dumps()


class Error(Exception):
	"""Rappresenza un errore generico generato dal server Konga. Eccezioni di questo tipo hanno due attributi:
	
	.. attribute:: errno
		
		Codice identificativo dell'errore
	
	.. attribute:: error
		
		Stringa che descrive l'errore
	"""
	def __init__(self, errno, msg):
		self.errno = errno
		self.error = self.msg = msg
	def __str__(self):
		msg = self.msg and ('%s' % self.msg) or '(internal error)'
		if isinstance(msg, unicode):
			msg = msg.encode('utf-8')
		return msg
	def __unicode__(self):
		if isinstance(self.msg, unicode):
			error = self.msg
		else:
			error = unicode(self.msg, 'utf-8', 'replace')
		return error or u'(internal error)'


class ErrorList(Error):
	"""Rappresenza una lista di errori generati dal server Konga. Eccezioni di questo tipo hanno tre attributi:
	
	.. attribute:: errno
		
		Codice identificativo dell'ultimo errore
	
	.. attribute:: error
		
		Stringa che descrive l'ultimo errore
	
	.. attribute:: errors
	
		Lista degli errori, in cui ogni errore è una tupla nella forma ``(type, errno, prefix, error)``; *type* è uno tra :attr:`Log.INFO`,
		:attr:`Log.WARNING` e :attr:`Log.ERROR`; *errno* ed *error* sono il codice e la descrizione dell'errore, e *prefix* un eventuale
		prefisso che identifica il contesto dell'errore.
	"""
	
	PREPARE_CALLBACK = None

	def __init__(self, errors=None):
		self.errors = errors or []
		self.errno = OK
		self.error = 'No error'
		for type, errno, prefix, error in self.errors:
			if type == Log.ERROR:
				self.errno = errno
				self.error = error
				break
	
	def __str__(self):
		return '\n'.join([ str(e) for e in self.get_errors() ])
	
	def __unicode__(self):
		text = []
		for e in self.get_errors():
			if isinstance(e.error, unicode):
				error = e.error
			else:
				error = unicode(e.error, 'utf-8', 'replace')
			text.append(error or u'(internal error)')
		return u'\n'.join(text)
	
	def __iter__(self):
		return iter(self.errors)
	
	def get_errors(self):
		return [ Error(errno, error) for type, errno, prefix, error in self.errors ]
	
	def add_error(self, errno, error, prefix=''):
		"""Aggiunge un errore (tipo :attr:`Log.ERROR`) al log, assegnando il codice e la descrizione da *errno* ed *error*.
		*prefix* è una stringa di prefisso e può essere usato per dare un contesto all'errore.
		"""
		self.errors.append((Log.ERROR, errno, prefix, error))
		self.errno = errno
		self.error = error
	
	def prepare_log(self, log=None):
		"""Trasferisce la lista degli errori su *log* e restituisce il log aggiornato. *log* deve essere un oggetto di classe :class:`~kongalib.Log`;
		se ``None``, un nuovo oggetto :class:`~kongalib.Log` verrà creato e popolato con gli errori.
		"""
		if ErrorList.PREPARE_CALLBACK is not None:
			return ErrorList.PREPARE_CALLBACK(self, log)
		if log is None:
			log = Log()
		for type, errno, prefix, error in self.errors:
			message = '%s (%d)' % (error, errno)
			if prefix:
				message = '%s %s' % (prefix, message)
			log.add_message(type, message)
		return log
	
	@classmethod
	def from_error(cls, errno, error, prefix=''):
		"""Costruisce un nuovo oggetto :class:`ErrorList` contenente un singolo errore (tipo :attr:`Log.ERROR`), assegnando il codice e la
		descrizione da *errno* ed *error*. *prefix* è una stringa di prefisso e può essere usato per dare un contesto all'errore.
		"""
		res = ErrorList()
		res.add_error(errno, error, prefix)
		return res
	
	@classmethod
	def from_exception(cls, e=None):
		"""Costruisce un nuovo oggetto :class:`ErrorList` contenente un singolo errore (tipo :attr:`Log.ERROR`), a partire dall'eccezione *e*;
		se *e* è ``None``, viene assunta l'eccezione corrente. L'errore creato avrà un codice *errno* valido se *e* è un'eccezione di classe
		:exc:`Error`, altrimenti sarà ``-1``.
		"""
		if e is None:
			e = sys.exc_info()[1]
		if isinstance(e, ErrorList):
			return e
		elif isinstance(e, Error):
			errno = e.errno
			error = e.msg
		else:
			errno = -1
			error = str(e)
		return cls.from_error(errno, error)
	
	@classmethod
	def set_prepare_callback(cls, callback):
		cls.PREPARE_CALLBACK = callback


class JSONError(Exception):
	def __init__(self, msg):
		self.msg = msg
	def __str__(self):
		return self.msg
	def __unicode__(self):
		return unicode(self.msg and (self.msg.decode('utf-8', 'replace')))


class InterpreterTimeout(Exception):
	pass


from _kongalib import Decimal, Deferred, JSONEncoder, JSONDecoder, Interpreter, start_timer, hash_password, host_lookup, get_network_interfaces, get_machine_uuid, set_interpreter_timeout, _cleanup, lock, unlock, set_default_idle_callback, checksum
from .constants import *
from .client import *
from .expression import *
from .data_dictionary import *


class ErrorMessage(object):
	def __init__(self, errno, error):
		self.errno = errno
		self.error = error
		exc = sys.exc_info()
		if exc is None:
			self.traceback = None
		else:
			self.traceback = ''.join(traceback.format_exception(*exc))
	
	def __str__(self):
		return self.error if self.traceback is None else self.traceback



def round(number, ndigits=2):
	"""Arrotonda *number* a *ndigits* cifre decimali, arrotondando verso 0 se l'ultima cifra dopo l'arrotondamento è compresa tra 0
	e 5, altrimenti arrotonda allontanadosi da 0. Restituisce un oggetto :class:`~kongalib.Decimal`.
	"""
	return Decimal(number).round(10**(-ndigits))

def floor(number, ndigits=2):
	"""Arrotonda *number* a *ndigits* cifre decimali, arrotondando verso 0. Restituisce un oggetto :class:`~kongalib.Decimal`."""
	return Decimal(number).floor(10**(-ndigits))

def ceil(number, ndigits=2):
	"""Arrotonda *number* a *ndigits* cifre decimali, arrotondando allontanandosi da 0. Restituisce un oggetto :class:`~kongalib.Decimal`."""
	return Decimal(number).ceil(10**(-ndigits))

def multiply_and_round(x, y, ndigits=2):
	return Decimal(x).multiply(Decimal(y), 10**(-ndigits), ROUND)

def multiply_and_floor(x, y, ndigits=2):
	return Decimal(x).multiply(Decimal(y), 10**(-ndigits), FLOOR)

def multiply_and_ceil(x, y, ndigits=2):
	return Decimal(x).multiply(Decimal(y), 10**(-ndigits), CEIL)

def divide_and_round(x, y, ndigits=2):
	return Decimal(x).divide(Decimal(y), 10**(-ndigits), ROUND)

def divide_and_floor(x, y, ndigits=2):
	return Decimal(x).divide(Decimal(y), 10**(-ndigits), FLOOR)

def divide_and_ceil(x, y, ndigits=2):
	return Decimal(x).divide(Decimal(y), 10**(-ndigits), CEIL)

def escape(query):
	"""Sostituisce `'` con `''` per preparare una stringa all'inserimento SQL."""
	return query.replace("'", "''")


atexit.register(_cleanup)

