# Copyright (c) 2011, Roger Lew [see LICENSE.txt]
# This software is funded in part by NIH Grant P20 RR016454.

"""
Implementation of Gleason's (1999) non-iterative upper quantile
studentized range approximation.

According to Gleason this method should be more accurate than the
AS190 FORTRAN algorithm of Lund and Lund (1983) and works from .5
<= p <= .999 (The AS190 only works from .9 <= p <= .99).

It is more efficient then the Copenhaver & Holland (1988) algorithm
(used by the _qtukey_ R function) although it requires storing the A
table in memory. (q distribution) approximations in Python. 


see:
  Gleason, J. R. (1999). An accurate, non-iterative approximation
    for studentized range quantiles. Computational Statistics &
    Data Analysis, (31), 147-158.

  Gleason, J. R. (1998). A table of quantile points of the
    Studentized range distribution.
    http://www.stata.com/stb/stb46/dm64/sturng.pdf
"""

import math
import scipy.stats
import numpy as np

inf = float('inf')

__version__ = '0.1.0'

# Gleason's table was derived using least square estimation on the tabled
# r values for combinations of p and v. In total there are 206
# estimates over p-values of .5, .75, .9, .95, .975, .99, .995,
# and .999, and over v (degrees of freedom) of (1) - 20, 24, 30, 40,
# 60, 120, and inf. combinations with p < .95 don't have coefficients
# for v = 1. Hence the parentheses. These coefficients allow us to
# form f-hat. f-hat with the inverse t transform of tinv(p,v) yields
# a fairly accurate estimate of the studentized range distribution
# across a wide range of values. According to Gleason this method
# should be more accurate than algorithm AS190 of Lund and Lund (1983)
# and work across a wider range of values (The AS190 only works
# from .9 <= p <= .99). R's qtukey algorithm was used to add tables
# at .675, .8, and .85. These aid approximations when p < .9.
#
# The code that generated this table is appended to this module.
A = {(0.5, 2.0): [-0.88295934991583214, -0.10835767874249443, 0.0352149700495221, -0.0028576292579228012],
 (0.5, 3.0): [-0.89085828818676438, -0.10255696888792074, 0.033613640379713497, -0.0027101701857209976],
 (0.5, 4.0): [-0.89627344945045684, -0.099072529021258487, 0.032657776341753673, -0.0026219009362902787],
 (0.5, 5.0): [-0.89959144879756103, -0.097272844110187773, 0.032236190415860663, -0.002591155830057537],
 (0.5, 6.0): [-0.89959428376501982, -0.098176296834564428, 0.032590768610732956, -0.0026319891963598615],
 (0.5, 7.0): [-0.90131492091018683, -0.097135896562982382, 0.03230412115239191, -0.0026057961634804285],
 (0.5, 8.0): [-0.90292500944834952, -0.096047496546115954, 0.032030944920569127, -0.0025848746683383955],
 (0.5, 9.0): [-0.9038559928940928, -0.095390763565738529, 0.031832648234512949, -0.0025656057007047918],
 (0.5, 10.0): [-0.90562525379467984, -0.093954483221486543, 0.03141444938262776, -0.0025257832916653956],
 (0.5, 11.0): [-0.90420346257529105, -0.0958516693693829, 0.032115040285931255, -0.002605506159437763],
 (0.5, 12.0): [-0.90585972937460668, -0.094449311845996822, 0.031705947733453081, -0.0025673332066129847],
 (0.5, 13.0): [-0.90555438106628416, -0.094792978391019175, 0.031826590275101478, -0.0025807103785249579],
 (0.5, 14.0): [-0.90652757562369612, -0.093792145780375938, 0.031468962294904083, -0.0025395170861296228],
 (0.5, 15.0): [-0.90642322724066238, -0.094173028619748883, 0.031657521285272029, -0.0025659276118003083],
 (0.5, 16.0): [-0.90716338246671335, -0.093785182451676802, 0.03163009346924494, -0.002570146090170638],
 (0.5, 17.0): [-0.90790133946869811, -0.093001146270897533, 0.031376863494271288, -0.0025451435748183849],
 (0.5, 18.0): [-0.90774328292845974, -0.093343527295075376, 0.031518143450785116, -0.0025613910248375596],
 (0.5, 19.0): [-0.90789499879191526, -0.093169642608640649, 0.031440780374353244, -0.0025498351049578483],
 (0.5, 20.0): [-0.90842708327698896, -0.092696011032778858, 0.031296038357099071, -0.0025346961804811525],
 (0.5, 24.0): [-0.90832813147821712, -0.092959311916482185, 0.03146406454310606, -0.0025611385777973895],
 (0.5, 30.0): [-0.90857624905461332, -0.093043129195321664, 0.031578788012309143, -0.0025766591226119152],
 (0.5, 40.0): [-0.91034084696100859, -0.091978039564480391, 0.031451632302399363, -0.0025791419494332465],
 (0.5, 60.0): [-0.91084357127713189, -0.091452670579325893, 0.031333146248528364, -0.0025669785067454366],
 (0.5, 120.0): [-0.90963648183039658, -0.093414579110265739, 0.032215608327220022, -0.0026704030990264437],
 (0.5, inf): [-0.91077158925964963, -0.092899204484489142, 0.032230416912145475, -0.0026696936023914348],
 (0.675, 2.0): [-0.67231520904389186, -0.09708362532986603, 0.02799137933469889, -0.0021425184525484041],
 (0.675, 3.0): [-0.65661725056315867, -0.081471951380160404, 0.023457322947504779, -0.0017448568891613406],
 (0.675, 4.0): [-0.65045677770297317, -0.071419072406791545, 0.020741962182471374, -0.001517126209568785],
 (0.675, 5.0): [-0.64718875047209556, -0.064720614871626531, 0.01905345143571386, -0.0013836234271927847],
 (0.675, 6.0): [-0.64523003841198567, -0.059926312037388831, 0.01791899659057046, -0.0012992249623417871],
 (0.675, 7.0): [-0.64403312861734563, -0.056248194793861318, 0.017091447950672811, -0.0012406560066107941],
 (0.675, 8.0): [-0.64325095862596238, -0.05335254312565324, 0.016471879276660707, -0.001199183903262265],
 (0.675, 9.0): [-0.64271153028924755, -0.051023766428447609, 0.015997994860602181, -0.0011693636709498008],
 (0.675, 10.0): [-0.6423224451650702, -0.049118326290018666, 0.015629704569054163, -0.0011477775090329386],
 (0.675, 11.0): [-0.64203897671285504, -0.047524629974986764, 0.015334801952969377, -0.0011315058025702313],
 (0.675, 12.0): [-0.64180344408869994, -0.046205914049072018, 0.015108292888342614, -0.0011207367043959897],
 (0.675, 13.0): [-0.6416208639794071, -0.045076100000500885, 0.014922656767180032, -0.0011126140945521457],
 (0.675, 14.0): [-0.64146906563699835, -0.044108522452636773, 0.014772953791711617, -0.0011069708060179198],
 (0.675, 15.0): [-0.64133914897734845, -0.043273373950890412, 0.014651692699466836, -0.0011032217776929564],
 (0.675, 16.0): [-0.64123238139185579, -0.042538921484985538, 0.014549991204593729, -0.0011005632422753356],
 (0.675, 17.0): [-0.64113033413779319, -0.041905706783392137, 0.014470808198935406, -0.0010995289383542409],
 (0.675, 18.0): [-0.64104137135600303, -0.041343888468608654, 0.014404564687281501, -0.0010991305354429996],
 (0.675, 19.0): [-0.64096064971515487, -0.040845692105626136, 0.01435015942909861, -0.0010993656511852185],
 (0.675, 20.0): [-0.64088647523255626, -0.040402174481623386, 0.01430576926735631, -0.0011001304134790556],
 (0.675, 24.0): [-0.64063764736881346, -0.039034707353895179, 0.014196700609123885, -0.0011061958349350795],
 (0.675, 30.0): [-0.64034987535235388, -0.037749653033981534, 0.014147041611264444, -0.0011188251986068445],
 (0.675, 40.0): [-0.63999905616411024, -0.036583301923206794, 0.014172068626651695, -0.0011391001791816212],
 (0.675, 60.0): [-0.63955586826309208, -0.035576931594548521, 0.014287296487741139, -0.0011675808817846116],
 (0.675, 120.0): [-0.63899243116821458, -0.034763752559167679, 0.014500725190290368, -0.0012028489580529706],
 (0.675, inf): [-0.63832683323134509, -0.034101467570732166, 0.014780917649722329, -0.001236620023735713],
 (0.75, 2.0): [-0.60684073560670648, -0.096375192942729851, 0.026567529769184486, -0.0019963229292960898],
 (0.75, 3.0): [-0.57986144453003852, -0.078570293610093378, 0.021280638276795324, -0.001532930731551889],
 (0.75, 4.0): [-0.56820771869183506, -0.06681135436038517, 0.018065283350794117, -0.0012641484724346202],
 (0.75, 5.0): [-0.56175292871671112, -0.058864521794785157, 0.016046733170898681, -0.0011052558211829939],
 (0.75, 6.0): [-0.55773449268359931, -0.053136923459579252, 0.01468425824321499, -0.0010042826915115861],
 (0.75, 7.0): [-0.55509525149913208, -0.048752642898503604, 0.013696564383844536, -0.00093482185549827103],
 (0.75, 8.0): [-0.55324993355107743, -0.045305562587450929, 0.01295968338891324, -0.00088583557203332508],
 (0.75, 9.0): [-0.55189258709027345, -0.042539823948769581, 0.012398792565006282, -0.00085083978544979347],
 (0.75, 10.0): [-0.55085384603743504, -0.040281426378764043, 0.011964422654502024, -0.0008256032473864728],
 (0.75, 11.0): [-0.55003198360538852, -0.038410173133805292, 0.011623293182938985, -0.00080732963328378031],
 (0.75, 12.0): [-0.54936541592278409, -0.036838543228568187, 0.011351822600621295, -0.0007940703594567052],
 (0.75, 13.0): [-0.54881015906915254, -0.035506711348015002, 0.011134691554885176, -0.00078463602814659032],
 (0.75, 14.0): [-0.54834094541955181, -0.0343647882908239, 0.010958873087079176, -0.00077796635887029367],
 (0.75, 15.0): [-0.5479360241186555, -0.033379237544292575, 0.010816141034596395, -0.00077344175515294629],
 (0.75, 16.0): [-0.54758347502790272, -0.032520571322629303, 0.010699241179699753, -0.00077050856015551717],
 (0.75, 17.0): [-0.54727115829566308, -0.031769278839750816, 0.010603750365329739, -0.00076886430964610386],
 (0.75, 18.0): [-0.54699351985037137, -0.031105474259446469, 0.010524668405611162, -0.00076810649071267444],
 (0.75, 19.0): [-0.54674357756621605, -0.030516965658311333, 0.010459478262962181, -0.00076808646302480152],
 (0.75, 20.0): [-0.54651728308561454, -0.029992320080790526, 0.010405695329574007, -0.00076864176045635005],
 (0.75, 24.0): [-0.54578309720733786, -0.028372626485180134, 0.01026993883484459, -0.00077427361939660109],
 (0.75, 30.0): [-0.54501247137028497, -0.026834879822533398, 0.010195600459218277, -0.00078648584447463591],
 (0.75, 40.0): [-0.54418127923468218, -0.025413219300466416, 0.01019645344529874, -0.00080610767188247828],
 (0.75, 60.0): [-0.54326518518227263, -0.024141965564418756, 0.01028500258073293, -0.00083332210342951653],
 (0.75, 120.0): [-0.54224758681278973, -0.023039061869719091, 0.010463361748265541, -0.0008661278925891499],
 (0.75, inf): [-0.54114579054892464, -0.022065933875202247, 0.010703744011780116, -0.00089726596994886348],
 (0.8, 2.0): [-0.56895273802087132, -0.096326258000447806, 0.02581591636118712, -0.0019136562120801108],
 (0.8, 3.0): [-0.53360383830686464, -0.077585190807644297, 0.020184759204921889, -0.0014242745951284523],
 (0.8, 4.0): [-0.51780274805120896, -0.064987732511795404, 0.016713307701483146, -0.0011353796259826795],
 (0.8, 5.0): [-0.50894361625599438, -0.056379181772874092, 0.014511268573311477, -0.00096225584177557087],
 (0.8, 6.0): [-0.50335152880952017, -0.050168861993075561, 0.013028071538776014, -0.00085269819351505407],
 (0.8, 7.0): [-0.49960934359908316, -0.045417334012632218, 0.011955593404919803, -0.00077759606382047936],
 (0.8, 8.0): [-0.49694517907843788, -0.041689155526497927, 0.011158988125687103, -0.00072497446317474614],
 (0.8, 9.0): [-0.49495599752171698, -0.038702217018168346, 0.010554359942423179, -0.0006875213030982021],
 (0.8, 10.0): [-0.49341407687280098, -0.036266791358251657, 0.010087355365255875, -0.00066060845601547357],
 (0.8, 11.0): [-0.49218129404420446, -0.034252402629385587, 0.0097218581360419224, -0.00064123455594965314],
 (0.8, 12.0): [-0.49117223626959239, -0.032563273549993929, 0.0094318596712741863, -0.00062725268941850331],
 (0.8, 13.0): [-0.490327810543638, -0.031132496072522949, 0.0091999766328660637, -0.00061729447836928906],
 (0.8, 14.0): [-0.4896104951233789, -0.029906922605539881, 0.0090124523845055617, -0.00061026218126304015],
 (0.8, 15.0): [-0.48899070084628704, -0.028849606494080016, 0.0088602807673874729, -0.00060548977797508503],
 (0.8, 16.0): [-0.48844921207327591, -0.027929790198328447, 0.0087359926862880764, -0.00060242120349963599],
 (0.8, 17.0): [-0.48797119612361467, -0.027123635761742364, 0.0086338142989082541, -0.00060061825119387291],
 (0.8, 18.0): [-0.48754596902994451, -0.02641196818884043, 0.0085493194451390954, -0.0005997708057462898],
 (0.8, 19.0): [-0.48716341937360824, -0.025781420750942394, 0.0084796650749454656, -0.00059970026119227126],
 (0.8, 20.0): [-0.48681739257988271, -0.02521962921751, 0.0084221842171932262, -0.00060023210658953854],
 (0.8, 24.0): [-0.48570639832841372, -0.023480606371084227, 0.0082744896914012152, -0.00060568100822033311],
 (0.8, 30.0): [-0.48455866275780235, -0.021824664036191023, 0.0081888534411139677, -0.00061762161351218371],
 (0.8, 40.0): [-0.48335479228388684, -0.020279953234361256, 0.0081765075404572219, -0.00063657094446528944],
 (0.8, 60.0): [-0.48207352498555073, -0.018875338267721791, 0.0082473976370829368, -0.00066242456078474362],
 (0.8, 120.0): [-0.48070354161835677, -0.017621709922754979, 0.008400971928159787, -0.00069300471995309334],
 (0.8, inf): [-0.47926687785226879, -0.01647657473889709, 0.0086097057863877887, -0.00072160841835695714],
 (0.85, 2.0): [-0.53366806875301798, -0.098288179497027478, 0.026002333879111234, -0.0019567144739820797],
 (0.85, 3.0): [-0.48995918681033968, -0.077312729084000748, 0.019368987152806605, -0.0013449672721499548],
 (0.85, 4.0): [-0.46956079872009177, -0.063818510341980911, 0.015581606006215132, -0.0010264311872190911],
 (0.85, 5.0): [-0.45790853352523475, -0.054680516039663604, 0.013229854082846555, -0.00084248448517311566],
 (0.85, 6.0): [-0.45050708551107788, -0.048050935218101241, 0.011636407083823559, -0.00072491474693732623],
 (0.85, 7.0): [-0.44548336321091087, -0.042996625981049925, 0.01049305778549834, -0.00064528838486392756],
 (0.85, 8.0): [-0.44186624153233306, -0.039040014860078714, 0.0096479563088420848, -0.00058990910218007745],
 (0.85, 9.0): [-0.43914119265238044, -0.035875686164546013, 0.0090088779057859673, -0.00055071452047213522],
 (0.85, 10.0): [-0.43701255765849584, -0.03330099294556281, 0.0085172143098326027, -0.00052272752494016562],
 (0.85, 11.0): [-0.43530108791946065, -0.031174745222262719, 0.008133563129382744, -0.00050268366533676818],
 (0.85, 12.0): [-0.43389220124169653, -0.029396186049245557, 0.0078306272963703048, -0.00048836443076860755],
 (0.85, 13.0): [-0.43271026840758803, -0.027890760494750474, 0.0075886921462505891, -0.00047819344945002859],
 (0.85, 14.0): [-0.43170230556849887, -0.026604152715615115, 0.007393908783713138, -0.00047109983786457345],
 (0.85, 15.0): [-0.43083160317103264, -0.025494230501978722, 0.007235874417362574, -0.00046630683036032725],
 (0.85, 16.0): [-0.43006992861498106, -0.024529611938495417, 0.0071069224558510165, -0.00046323867057538394],
 (0.85, 17.0): [-0.42939734795165213, -0.02368502720291903, 0.0070011547278080108, -0.00046147961226557314],
 (0.85, 18.0): [-0.42879829238134015, -0.022940653412082609, 0.0069140055607698828, -0.00046070869047200601],
 (0.85, 19.0): [-0.42826119443597555, -0.022280181864448346, 0.006841774727680972, -0.00046066841691314308],
 (0.85, 20.0): [-0.42777654832295436, -0.02169090970096799, 0.0067817410839607645, -0.0004611862269586752],
 (0.85, 24.0): [-0.42622450209296969, -0.01986964468117897, 0.0066276792356543953, -0.00046668812618852728],
 (0.85, 30.0): [-0.42463810543383879, -0.018130113675808385, 0.006534460953697952, -0.00047835579725921098],
 (0.85, 40.0): [-0.42299917847098473, -0.016498222493123136, 0.0065120557129924777, -0.00049656042544224085],
 (0.85, 60.0): [-0.42129387422012782, -0.014992119853123369, 0.0065657790688290691, -0.00052069700146808355],
 (0.85, 120.0): [-0.41951580701490354, -0.013615720044120976, 0.0066923902987349234, -0.0005484690281610721],
 (0.85, inf): [-0.41768751536539883, -0.012327528404953755, 0.0068664932318025585, -0.00057403733243764372],
 (0.9, 1.0): [-0.6585106278675138, -0.1267162474820277, 0.036318803760278122, -0.0029012834201025805],
 (0.9, 2.0): [-0.50391945726172283, -0.096996103865096259, 0.024726436130790733, -0.0017901398273744127],
 (0.9, 3.0): [-0.4479979182291533, -0.077180370648644955, 0.018584042189222643, -0.001264703828331896],
 (0.9, 4.0): [-0.42164091596482073, -0.063427072932876666, 0.014732204463463419, -0.00094904182127271842],
 (0.9, 5.0): [-0.40686856330790044, -0.053361939062039433, 0.01204180170335639, -0.00072960194022120338],
 (0.9, 6.0): [-0.39669925789907978, -0.046951520199759011, 0.010546648205763893, -0.00062621209078883241],
 (0.9, 7.0): [-0.39006554081141048, -0.041694801540275557, 0.0093687530924632008, -0.0005464867872303214],
 (0.9, 8.0): [-0.38570204483793363, -0.037083917652691638, 0.0083233242879095868, -0.00047177614075825258],
 (0.9, 9.0): [-0.38190737000062414, -0.034004588739095211, 0.0077532002518532353, -0.00044306559389054831],
 (0.9, 10.0): [-0.37893272775008979, -0.031394679167590242, 0.0072596807861431325, -0.00041605194106405867],
 (0.9, 11.0): [-0.37692513186023358, -0.028780785149001637, 0.0066937878979664057, -0.00037419976277404334],
 (0.9, 12.0): [-0.37506345659830947, -0.026956477974734286, 0.0064147711745726659, -0.00036595362178730477],
 (0.9, 13.0): [-0.37339516946113238, -0.025439485602346219, 0.0061760621809841369, -0.00035678698617910956],
 (0.9, 14.0): [-0.37216979276677131, -0.023963483636767226, 0.0059263262706201756, -0.00034397876870089948],
 (0.9, 15.0): [-0.37120944888281743, -0.022696141900468508, 0.0057521710497353239, -0.00033961145983476843],
 (0.9, 16.0): [-0.3695892426019971, -0.022227886908940057, 0.0057691712301431467, -0.00035042768882046643],
 (0.9, 17.0): [-0.36884224592818288, -0.021146979315724644, 0.0055957933266519442, -0.00034283815875318804],
 (0.9, 18.0): [-0.36803086723343037, -0.020337736878415714, 0.0054655397430489385, -0.00033452988420175621],
 (0.9, 19.0): [-0.36767004001882442, -0.019370116500772715, 0.0053249299076292191, -0.00032975532554220332],
 (0.9, 20.0): [-0.36642277132758949, -0.01934424112480183, 0.0054454931133001558, -0.00034868069516253293],
 (0.9, 24.0): [-0.36450650468372503, -0.017284258633645015, 0.0052337510782599003, -0.00034898214364945947],
 (0.9, 30.0): [-0.36251869423341082, -0.015358554426158638, 0.0050914277370144277, -0.00035574502905126734],
 (0.9, 40.0): [-0.36008886992506056, -0.014016832622919514, 0.0051930826614040209, -0.00038798306911321553],
 (0.9, 60.0): [-0.35825590313866135, -0.011991573391724925, 0.0050632224756444145, -0.00039090217182869726],
 (0.9, 120.0): [-0.35543612222333298, -0.011074404122148444, 0.0053504571082762589, -0.00043647137714447593],
 (0.9, inf): [-0.35311806564024756, -0.0096253990414879986, 0.0054548579510655159, -0.00045343902755160632],
 (0.95, 1.0): [-0.65330318088973294, -0.12638310817332354, 0.035987535341051702, -0.0028562665706893522],
 (0.95, 2.0): [-0.47225160405564126, -0.10182570343658619, 0.025846563338429506, -0.0019096768798857868],
 (0.95, 3.0): [-0.40566355775271878, -0.077067170013778885, 0.017789908652442146, -0.001182961555182008],
 (0.95, 4.0): [-0.37041675300542959, -0.063815685577225778, 0.01411520966544547, -0.0008999609171063828],
 (0.95, 5.0): [-0.35152397646720185, -0.052156510852884537, 0.010753741222880133, -0.00059868455897273067],
 (0.95, 6.0): [-0.33806730027133081, -0.045668399349043841, 0.0093168896425132337, -0.0005136971608503643],
 (0.95, 7.0): [-0.32924040175872199, -0.040019612406715456, 0.0080051238183793429, -0.00042054579557102245],
 (0.95, 8.0): [-0.32289030039334032, -0.035575348405695657, 0.007050909775632554, -0.00035980782319226746],
 (0.95, 9.0): [-0.31767304710974376, -0.032464939881036499, 0.0064755928438992921, -0.00033166737793184506],
 (0.95, 10.0): [-0.31424317473498103, -0.029133468700706038, 0.0057437475336408351, -0.00027894281528771696],
 (0.95, 11.0): [-0.31113589283424969, -0.026851156315576278, 0.0053517918635722517, -0.0002615596872885862],
 (0.95, 12.0): [-0.30848982164871669, -0.025043254872384686, 0.005066173629734474, -0.00025017270080750052],
 (0.95, 13.0): [-0.30592128919884576, -0.023863876626390546, 0.0049618058400428926, -0.00025665434161387151],
 (0.95, 14.0): [-0.30449677518889301, -0.021983969423751429, 0.0045740487122021079, -0.00022881136392172898],
 (0.95, 15.0): [-0.30264907833832044, -0.021048808343562278, 0.0044866590250142759, -0.00023187608142614951],
 (0.95, 16.0): [-0.30118294553477187, -0.020160229997545364, 0.0044170776905826051, -0.00023733498050904596],
 (0.95, 17.0): [-0.3002001326635178, -0.018959272599067697, 0.0041925336493807098, -0.00022274029418265301],
 (0.95, 18.0): [-0.29857887814785344, -0.018664422630496814, 0.0042557733988477552, -0.00023758808740105848],
 (0.95, 19.0): [-0.29796289486160354, -0.017632215746469806, 0.0040792770147688728, -0.00022753260795166938],
 (0.95, 20.0): [-0.29681505970865046, -0.017302570025579684, 0.0041188450480143593, -0.00023913065417682309],
 (0.95, 24.0): [-0.29403147196811175, -0.01533232783984817, 0.0039292159517863709, -0.00024003434010390977],
 (0.95, 30.0): [-0.29080775484071275, -0.013844060389794662, 0.0039279170501101413, -0.00026085110459751987],
 (0.95, 40.0): [-0.28821582185342171, -0.01189469630990703, 0.0038202656927830898, -0.0002693336194350946],
 (0.95, 60.0): [-0.28525636744900812, -0.010235910426528022, 0.0038147029196624466, -0.00028598361422334375],
 (0.95, 120.0): [-0.28241065111205454, -0.0086103923634938337, 0.003845064339917686, -0.00030206086972587253],
 (0.95, inf): [-0.27885569704677726, -0.0078122495812381048, 0.0041798552067408932, -0.00034694964062345629],
 (0.975, 1.0): [-0.65203597885174513, -0.12608944774940967, 0.035710040554783627, -0.0028116026442772399],
 (0.975, 2.0): [-0.46371890879705741, -0.096954461010577905, 0.023958313423898623, -0.0017124566348133202],
 (0.975, 3.0): [-0.38265281990754796, -0.076782541335526089, 0.017405079474561064, -0.0011610854379924631],
 (0.975, 4.0): [-0.34051192565110938, -0.063652349598389016, 0.013528312782152246, -0.0008364473601537028],
 (0.975, 5.0): [-0.31777656417090877, -0.051694678774421153, 0.010115804329866336, -0.0005451743371032038],
 (0.975, 6.0): [-0.30177149938585757, -0.044806687159052408, 0.0084835481617903594, -0.00042827813493253377],
 (0.975, 7.0): [-0.29046971185906478, -0.039732835874670941, 0.0074353607765197532, -0.00037562981115711169],
 (0.975, 8.0): [-0.28309484926230921, -0.034764893891885848, 0.0062932473138266089, -0.00029339197684159529],
 (0.975, 9.0): [-0.27711709277868585, -0.031210449641623134, 0.0055576188443350509, -0.00024663735956551068],
 (0.975, 10.0): [-0.27249203211492262, -0.028259759372435966, 0.0049911212074455992, -0.00021535392834437782],
 (0.975, 11.0): [-0.26848515021691682, -0.026146713333257302, 0.0046557803568792987, -0.00020400669250621739],
 (0.975, 12.0): [-0.26499922437386386, -0.024522921053148971, 0.0044259590015614022, -0.0001985564740060658],
 (0.975, 13.0): [-0.26250236928546411, -0.0227858827447628, 0.0041502799226475554, -0.00018801252668909456],
 (0.975, 14.0): [-0.26038553174364137, -0.021303500607948053, 0.0039195574024041609, -0.00017826161154043979],
 (0.975, 15.0): [-0.25801242901932769, -0.020505531708817169, 0.0038754955472984015, -0.0001858900563428824],
 (0.975, 16.0): [-0.25685315511071366, -0.018888424866086957, 0.0035453116954393799, -0.00016235797886753067],
 (0.975, 17.0): [-0.25501132629145151, -0.018362947885688929, 0.0035653918669487847, -0.00017470337460033235],
 (0.975, 18.0): [-0.25325045222067688, -0.017993539495110827, 0.0036035875565163808, -0.00018635501447584882],
 (0.975, 19.0): [-0.25236899379300259, -0.016948922416341366, 0.0034138934710515553, -0.00017462256003169442],
 (0.975, 20.0): [-0.25134497508425463, -0.016249570380371448, 0.0033197304717984153, -0.00017098113848001579],
 (0.975, 24.0): [-0.24768691325998249, -0.014668154544120683, 0.0032850768733389598, -0.00019013455606734964],
 (0.975, 30.0): [-0.24420835183292702, -0.012911166166469038, 0.0031977656784336374, -0.00020114885727500788],
 (0.975, 40.0): [-0.24105725037869152, -0.010836529782364976, 0.0030231316977222225, -0.00020128711355471899],
 (0.975, 60.0): [-0.23732082641705521, -0.0095442733897155782, 0.0031432906761517856, -0.00023062226558621612],
 (0.975, 120.0): [-0.23358581852549984, -0.0081281263238318253, 0.0031877299925455261, -0.00024496231881361558],
 (0.975, inf): [-0.2300410452727577, -0.0067112650195882131, 0.0032760274712813515, -0.00026244026807958009],
 (0.99, 1.0): [-0.65154118444038034, -0.12666040452395502, 0.036074810416191719, -0.0028668117579749922],
 (0.99, 2.0): [-0.45463403183943174, -0.098701237908550024, 0.024412716371591064, -0.0017613773605811786],
 (0.99, 3.0): [-0.36402060651844093, -0.079244952201653646, 0.017838121515705886, -0.0011908008860085826],
 (0.99, 4.0): [-0.3190350522555564, -0.061060750300327753, 0.012093158368281065, -0.00067268384721496115],
 (0.99, 5.0): [-0.2891701470553773, -0.052940778659566257, 0.010231008630821594, -0.00057178333431846036],
 (0.99, 6.0): [-0.27283242596633522, -0.042505407008106504, 0.0072753298215092205, -0.0003131391977894847],
 (0.99, 7.0): [-0.25773972106118731, -0.039384174807843203, 0.0069120739700150972, -0.00032993909141332527],
 (0.99, 8.0): [-0.24913627483742828, -0.03383158855084735, 0.0055516322473688423, -0.00022570873702919568],
 (0.99, 9.0): [-0.24252379434233087, -0.02948829712754536, 0.0045215510436569517, -0.00014424614768008398],
 (0.99, 10.0): [-0.23654349656373497, -0.027056001320687164, 0.0041627253420002432, -0.00013804425491531003],
 (0.99, 11.0): [-0.23187404147826934, -0.024803671836710492, 0.0037885888188750492, -0.00012335039102508567],
 (0.99, 12.0): [-0.22749930066910126, -0.023655077225720755, 0.003784502256879711, -0.00014785682788739606],
 (0.99, 13.0): [-0.22458989558207271, -0.021688389929084013, 0.0034075276450255747, -0.00012436941490208086],
 (0.99, 14.0): [-0.22197622272448911, -0.02018884974283543, 0.0031648755231887017, -0.00011320818224181392],
 (0.99, 15.0): [-0.21939242470986425, -0.019327478103590881, 0.0031295486167590859, -0.00012373109137800846],
 (0.99, 16.0): [-0.21739438919373255, -0.01821583143541583, 0.0029638257424372447, -0.0001171457539698052],
 (0.99, 17.0): [-0.21548924828458427, -0.017447844900781943, 0.0028994885787259144, -0.00012001976168265449],
 (0.99, 18.0): [-0.21365014612020392, -0.016888694372679678, 0.0028778034238414363, -0.0001259120235739301],
 (0.99, 19.0): [-0.21236654263815041, -0.016057145728422536, 0.0027571448118507537, -0.0001204917337210947],
 (0.99, 20.0): [-0.21092692689706136, -0.01564171270017432, 0.0027766010641493131, -0.0001308493840065704],
 (0.99, 24.0): [-0.20681961439904428, -0.013804284968727574, 0.0026308229593800937, -0.00013550562338699381],
 (0.99, 30.0): [-0.20271691622279778, -0.012060947140135853, 0.0025426117358538906, -0.00014589024929035642],
 (0.99, 40.0): [-0.19833098265442628, -0.010714531680438524, 0.0025985984705170126, -0.00016882791246542901],
 (0.99, 60.0): [-0.19406769082084402, -0.0093297075702737098, 0.0026521507248569789, -0.0001888486171167949],
 (0.99, 120.0): [-0.19010212880907207, -0.0075958244755514213, 0.0025660837632140725, -0.00018906491837293758],
 (0.99, inf): [-0.186020706211493, -0.0062121114210353682, 0.0026328279166738158, -0.00020453351012805052],
 (0.995, 1.0): [-0.65135583635661531, -0.12668689866337121, 0.036067521660015332, -0.0028654516328321454],
 (0.995, 2.0): [-0.45229774089738833, -0.09869462860950691, 0.02438185825329264, -0.001759473415899723],
 (0.995, 3.0): [-0.35935765272048575, -0.076650407692885406, 0.016823026609689814, -0.0010835134135790991],
 (0.995, 4.0): [-0.30704474462192749, -0.063093050920705981, 0.012771684502808769, -0.00075852505399928531],
 (0.995, 5.0): [-0.27582553501688212, -0.052533332917035346, 0.0097775938120779598, -0.0005133795254888381],
 (0.995, 6.0): [-0.25657971831245591, -0.043424910145731375, 0.0074324128499724945, -0.0003410516652174486],
 (0.995, 7.0): [-0.24090407950569406, -0.039591603220324859, 0.0068848424141914801, -0.00034737125801538636],
 (0.995, 8.0): [-0.23089543133473453, -0.03435327791609611, 0.0056009425626644127, -0.00024389221811655951],
 (0.995, 9.0): [-0.22322696095411121, -0.03029475597889358, 0.0046751186411600979, -0.00017437419528546324],
 (0.995, 10.0): [-0.21722684592402627, -0.026993557697710533, 0.0039811570501716541, -0.00013135256074832147],
 (0.995, 11.0): [-0.21171635759993446, -0.025156195041448264, 0.0037507766633595352, -0.00012959846004950202],
 (0.995, 12.0): [-0.20745335474990634, -0.023318780797030911, 0.0034934880598020942, -0.00012642671304117684],
 (0.995, 13.0): [-0.20426054816256753, -0.021189793343734595, 0.0030314710877513166, -9.0497606103732332e-05],
 (0.995, 14.0): [-0.20113536690938813, -0.020011539296288954, 0.0029215890464524474, -9.5715380839748261e-05],
 (0.995, 15.0): [-0.19855602826840513, -0.01880851876017486, 0.0027608805660540251, -9.2472385927030383e-05],
 (0.995, 16.0): [-0.19619157638792725, -0.017970460601785341, 0.0027113715090171277, -9.9864824385381795e-05],
 (0.995, 17.0): [-0.1942801611162514, -0.01700975124956414, 0.002583334945238065, -9.6137100472196323e-05],
 (0.995, 18.0): [-0.19243178819784695, -0.016316189379426773, 0.0025227504927096455, -9.8068270916733366e-05],
 (0.995, 19.0): [-0.19061294475686408, -0.015862265092921587, 0.0025207001930445971, -0.00010466146665569867],
 (0.995, 20.0): [-0.18946302538863491, -0.014975798409772235, 0.002370051318992727, -9.5507852680227619e-05],
 (0.995, 24.0): [-0.18444251617679566, -0.013770953751498978, 0.0024579438035075233, -0.00012688394630866389],
 (0.995, 30.0): [-0.18009742605688861, -0.011831340627406827, 0.0022801120847115593, -0.00012536245143653706],
 (0.995, 40.0): [-0.17562721951011537, -0.010157141764190043, 0.0022121940481707198, -0.00013454261353727798],
 (0.995, 60.0): [-0.17084630920894936, -0.0090224935075483795, 0.0023435518384654127, -0.00016240293427831741],
 (0.995, 120.0): [-0.1664841387648312, -0.0074792186116807834, 0.002328459346007119, -0.00017116472617037583],
 (0.995, inf): [-0.16213921663793421, -0.0058986021437813247, 0.0022605827043969138, -0.00016896219634973171],
 (0.999, 1.0): [-0.65233994845136212, -0.12579426574988065, 0.035830574958983098, -0.0028470551894114204],
 (0.999, 2.0): [-0.4505016315750302, -0.098294817878711652, 0.024134468776708048, -0.0017269609379735316],
 (0.999, 3.0): [-0.351617421894931, -0.076801144245097527, 0.016695690191021122, -0.0010661118783643947],
 (0.999, 4.0): [-0.29398449377994895, -0.062773190276601401, 0.012454217482731917, -0.00072644137382976443],
 (0.999, 5.0): [-0.25725363009818014, -0.053463805400325699, 0.0099664299623761152, -0.00054866108914422161],
 (0.999, 6.0): [-0.23674223920005646, -0.040973177834396947, 0.0062599560114277161, -0.00021565822255671392],
 (0.999, 7.0): [-0.21840108442139808, -0.037037025394265038, 0.0055908082177147152, -0.0002023881124263067],
 (0.999, 8.0): [-0.20579643371299658, -0.032500933109930977, 0.0046441818358497157, -0.0001476978704608718],
 (0.999, 9.0): [-0.19604595769095531, -0.029166889186081884, 0.0040644209565527476, -0.00012853913179767718],
 (0.999, 10.0): [-0.18857330287676169, -0.026316690691780432, 0.0035897299020772738, -0.00011572226586430652],
 (0.999, 11.0): [-0.18207428360807623, -0.024201119347206972, 0.0031647510676023885, -8.114751511244715e-05],
 (0.999, 12.0): [-0.17796353178877308, -0.021054364758016638, 0.0023968298550549236, -1.5909545743532569e-05],
 (0.999, 13.0): [-0.1737196921766046, -0.019577124639238108, 0.0022391645055279195, -2.0611474068230431e-05],
 (0.999, 14.0): [-0.16905297826779728, -0.019671163365621237, 0.0026495221277000607, -9.1074422378289126e-05],
 (0.999, 15.0): [-0.16635661930258899, -0.017903774385241853, 0.0022301348217556049, -5.1957056069149958e-05],
 (0.999, 16.0): [-0.16388773834475404, -0.016671950759491717, 0.0020365404831623429, -4.359373668129763e-05],
 (0.999, 17.0): [-0.16131936283497747, -0.015998893308245964, 0.001999036315663478, -4.8175244370014342e-05],
 (0.999, 18.0): [-0.15880632960919236, -0.015830717127442558, 0.0021688413139927835, -8.0618345020946119e-05],
 (0.999, 19.0): [-0.15644842725924088, -0.015729355126082055, 0.0022981408855021618, -0.00010093633657934939],
 (0.999, 20.0): [-0.15516595469565647, -0.014725109021187939, 0.0021117163307118937, -8.880739145070134e-05],
 (0.999, 24.0): [-0.14997438028518495, -0.012755320160932797, 0.0018871639981093338, -8.089623992713669e-05],
 (0.999, 30.0): [-0.14459974717194707, -0.011247325724108963, 0.0018637407358929518, -9.6415397571235364e-05],
 (0.999, 40.0): [-0.13933286320563637, -0.0097151721743039439, 0.0018131234354607877, -0.00010452579215215284],
 (0.999, 60.0): [-0.13424555460061166, -0.0082163016415636291, 0.0017883424316324902, -0.00011415861605797041],
 (0.999, 120.0): [-0.12896119771867159, -0.0070426671839773653, 0.0018472353572520945, -0.00012862191125851183],
 (0.999, inf): [-0.12397213340441901, -0.0056901226377689519, 0.0018260697992466377, -0.00013263461877905376]}

# p values that are defined in the A table
p_keys = [.5,.675,.75,.8,.85,.9,.95,.975,.99,.995,.999]

# v values that are defined in the A table
v_keys = range(2, 21) + [24, 30, 40, 60, 120, inf]
    
def _phi(p):
    """returns the pth quantile inverse norm"""
    return scipy.stats.norm.isf(p)

def _ptransform(p):
    """function for p-value abcissa transformation"""
    return -1. / (1. + 1.5 * _phi((1. + p)/2.))

def _select_points(a, list_like):
    """
    returns one above a, one below a, and the third
    closest point to a sorted in ascending order
    for quadratic interpolation. Assumes that points
    above and below a exist.
    """
    foo = [x for x in list(list_like) if x-a <= 0]
    z = [min(foo, key=lambda x : abs(x-a))]
    
    foo = [x for x in list(list_like) if x-a > 0]
    z.append(min(foo, key=lambda x : abs(x-a)))
    
    foo = [x for x in list(list_like) if x not in z]
    z.append(min(foo, key=lambda x : abs(x-a)))
    
    return sorted(z)

def _func(a, p, r, v):
    """
    calculates f for the coefficients in a, probability p,
    sample mean difference r, and degrees of freedom v.
    """
    # eq. 2.3
    f = a[0]*np.log(r-1.) + \
        a[1]*np.log(r-1.)**2 + \
        a[2]*np.log(r-1.)**3 + \
        a[3]*np.log(r-1.)**4

    # eq. 2.7 and 2.8 corrections
    if r == 3:
        f += -0.002 / (1. + 12. * _phi(p)**2)

        if v <= 4.364:
            f += 1./517. - 1./(312.*(v,1e38)[v==inf])
        else:
            f += 1./(191.*(v,1e38)[v==inf])

    return -f

def _interpolate_p(p, r, v):
    """
    interpolates p based on the values in the A table for the
    scalar value of r and the scalar value of v
    """

    # interpolate p (v should be in table)
    # if .5 < p < .75 use linear interpolation in q
    # if p > .75 use quadratic interpolation in log(y + r/v)
    # by -1. / (1. + 1.5 * _phi((1. + p)/2.))
    
    # find the 3 closest v values        
    p0, p1, p2 = _select_points(p, p_keys)

    y0 = _func(A[(p0, v)], p0, r, v) + 1.
    y1 = _func(A[(p1, v)], p1, r, v) + 1.
    y2 = _func(A[(p2, v)], p2, r, v) + 1.
    
    y_log0 = math.log(y0 + float(r)/float(v))
    y_log1 = math.log(y1 + float(r)/float(v))
    y_log2 = math.log(y2 + float(r)/float(v))

    # If p < .85 apply only the ordinate transformation
    # if p > .85 apply the ordinate and the abcissa transformation
    # In both cases apply quadratic interpolation
    if p > .85:
        p_t  = _ptransform(p)
        p0_t = _ptransform(p0)
        p1_t = _ptransform(p1)
        p2_t = _ptransform(p2)

        # calculate derivatives for quadratic interpolation
        d2 = 2*((y_log2-y_log1)/(p2_t-p1_t) - \
                (y_log1-y_log0)/(p1_t-p0_t))/(p2_t-p0_t)
        if (p2+p0)>=(p1+p1):
            d1 = (y_log2-y_log1)/(p2_t-p1_t) - 0.5*d2*(p2_t-p1_t)
        else:
            d1 = (y_log1-y_log0)/(p1_t-p0_t) + 0.5*d2*(p1_t-p0_t)
        d0 = y_log1

        # interpolate value
        y_log = (d2/2.) * (p_t-p1_t)**2. + d1 * (p_t-p1_t) + d0

        # transform back to y
        y = math.exp(y_log) - float(r)/float(v)

    else:
        # calculate derivatives for quadratic interpolation
        d2 = 2*((y_log2-y_log1)/(p2-p1) - \
                (y_log1-y_log0)/(p1-p0))/(p2-p0)
        if (p2+p0)>=(p1+p1):
            d1 = (y_log2-y_log1)/(p2-p1) - 0.5*d2*(p2-p1)
        else:
            d1 = (y_log1-y_log0)/(p1-p0) + 0.5*d2*(p1-p0)
        d0 = y_log1

        # interpolate values
        y_log = (d2/2.) * (p-p1)**2. + d1 * (p-p1) + d0

        # transform back to y
        y = math.exp(y_log) - float(r)/float(v)
        
    return y

def _interpolate_v(p, r, v):

    """
    interpolates v based on the values in the A table for the
    scalar value of r and th
    """
    # interpolate v (p should be in table)
    # ordinate: y**2
    # abcissa:  1./v

    # find the 3 closest v values        
    v0, v1, v2 = _select_points(v, v_keys)

    # y = f - 1.
    y0 = _func(A[(p,v0)], p, r, v0) + 1.
    y1 = _func(A[(p,v1)], p, r, v1) + 1.
    y2 = _func(A[(p,v2)], p, r, v2) + 1.

    # if v2 is inf set to a big number so interpolation
    # calculations will work
    if v2 > 1e38: v2 = 1e38

    # calculate derivatives for quadratic interpolation
    d2 = 2.*((y2**2-y1**2)/(1./v2-1./v1) - \
             (y0**2-y1**2)/(1./v0-1./v1)) / (1./v2-1./v0)
    if (1./v2 + 1./v0) >= (1./v1+1./v1):
        d1 = (y2**2-y1**2) / (1./v2-1./v1) - 0.5*d2*(1./v2-1./v1)
    else:
        d1 = (y1**2-y0**2) / (1./v1-1./v0) + 0.5*d2*(1./v1-1./v0)
    d0 = y1**2

    # calculate y
    y = math.sqrt((d2/2.)*(1./v-1./v1)**2. + d1*(1./v-1./v1)+ d0)

    return y
    
def qsturng(p, r, v):
    """
    return an approximation of the p-th quantile of the studentized
    range distribution for r means and v degrees of freedom
    """
    # r is interpolated through the q to y here we only need to
    # account for when p and/or v are not found in the table.
    
    global A, p_keys, v_keys

    if p < .5 or p > .999:
        raise ValueError('p must be between .5 and .999')
        
    if v < 1:
        raise ValueError('v must be > 1')
    
    if A.has_key((p,v)):
        f = _func(A[(p,v)], p, r, v)
        y = f + 1.

    elif p not in p_keys and v not in v_keys:
        # apply bilinear (quadratic) interpolation
        #                 
        #   p0,v2 +        o         + p1,v2    + p2,v2
        #                    r2
        #
        # 1
        # -                 (p,v)
        # v                x 
        #
        #                    r1
        #   p0,v1 +        o         + p1,v1    + p2,v1
        #
        #
        #   p0,v0 +        o r0      + p1,v0    + p2,v0
        #
        #             _ptransform(p)
        #
        # (p1 and v1 may be below or above (p,v). The algorithm
        # works in both cases. For diagramatic simplicity it is
        # shown as above)
        #
        # 1. at v0, v1, and v2 use quadratic interpolation
        #    to find r0, r1, r2
        #
        # 2. use r0, r1, r2 and quadratic interpolaiton
        #    to find y and (p,v)

        # find the 3 closest v values        
        v0, v1, v2 = _select_points(v, v_keys)

        # find the 2 closest p values        
        p0, p1, p2 = _select_points(p, p_keys)

        r0 = _interpolate_p(p, r, v0)
        r1 = _interpolate_p(p, r, v1)
        r2 = _interpolate_p(p, r, v2)

        # calculate derivatives for quadratic interpolation
        d2 = 2.*((r2**2-r1**2)/(1./v2-1./v1) - \
                 (r0**2-r1**2)/(1./v0-1./v1)) / (1./v2-1./v0)
        if (1./v2 + 1./v0) >= (1./v1+1./v1):
            d1 = (r2**2-r1**2) / (1./v2-1./v1) - 0.5*d2*(1./v2-1./v1)
        else:
            d1 = (r1**2-r0**2) / (1./v1-1./v0) + 0.5*d2*(1./v1-1./v0)
        d0 = r1**2

        # calculate y
        y = math.sqrt((d2/2.)*(1./v-1./v1)**2. + d1*(1./v-1./v1)+ d0)
        
    elif v not in v_keys:
        y = _interpolate_v(p, r, v)
        
    elif p not in p_keys:
        y = _interpolate_p(p, r, v)
        
    return math.sqrt(2) * -y * scipy.stats.t.isf((1.+p)/2., (v,1e38)[v>1e38])

##p, r, v = .9, 10, 20
##print
##print 'p and v interpolation'
##print '\t20\t22\t24'
##print '.75',qsturng(.75, r, 20),qsturng(.75, r, 22),qsturng(.75, r, 24)
##print '.85',qsturng(.85, r, 20),qsturng(.85, r, 22),qsturng(.85, r, 24)
##print '.90',qsturng(.90, r, 20),qsturng(.90, r, 22),qsturng(.90, r, 24)
##print
##print 'p and v interpolation'
##print '\t120\t500\tinf'
##print '.950',qsturng(.95, r, 120),qsturng(.95, r, 500),qsturng(.95, r, inf)
##print '.960',qsturng(.96, r, 120),qsturng(.96, r, 500),qsturng(.96, r, inf)
##print '.975',qsturng(.975, r, 120),qsturng(.975, r, 500),qsturng(.975, r, inf)
##print
##print 'p and v interpolation'
##print '\t40\t50\t60'
##print '.950',qsturng(.95, r, 40),qsturng(.95, r, 50),qsturng(.95, r, 60)
##print '.960',qsturng(.96, r, 40),qsturng(.96, r, 50),qsturng(.96, r, 60)
##print '.975',qsturng(.975, r, 40),qsturng(.975, r, 50),qsturng(.975, r, 60)
##print
##print 'p and v interpolation'
##print '\t20\t22\t24'
##print '.50',qsturng(.5, r, 20),qsturng(.5, r, 22),qsturng(.5, r, 24)
##print '.60',qsturng(.6, r, 20),qsturng(.6, r, 22),qsturng(.6, r, 24)
##print '.75',qsturng(.75, r, 20),qsturng(.75, r, 22),qsturng(.75, r, 24)


##
##"""this script builds the A matrix for Gleason's upper
##   quantile stundentized range algorithm"""
##
##import math
##import scipy.stats
##from scipy.optimize import leastsq
##
##import numpy as np
##from pprint import pprint as pp
##from collections import OrderedDict
##from numpy.random import random
##
### These tables were pulled from:
###    http://www.stata.com/stb/stb46/dm64/sturng.pdf
###
### the table was programmed by Gleason and extends Harter's (1960) table
### using the Copenhaver & Holland (1988) algorithm (C&H). Gleason found
### that the 4th significant digit of the C&H differed from Harter's
### tables on about 20% of the values. Gleason states this was do to
### consevative rounding by Harter. In those event the table reflects
### Harter's orginal approximations.
##q0500 = """\
##2 1.155 1.908 2.377 2.713 2.973 3.184 3.361 3.513 3.645 3.762 3.867 3.963 4.049 4.129 4.203 4.271 4.335 4.394 4.451 4.878 5.165 5.549 5.810 6.006
##3 1.082 1.791 2.230 2.545 2.789 2.986 3.152 3.294 3.418 3.528 3.626 3.715 3.796 3.871 3.940 4.004 4.064 4.120 4.172 4.573 4.842 5.202 5.447 5.630
##4 1.048 1.736 2.163 2.468 2.704 2.895 3.055 3.193 3.313 3.419 3.515 3.601 3.680 3.752 3.819 3.881 3.939 3.993 4.044 4.432 4.693 5.043 5.279 5.457
##5 1.028 1.705 2.124 2.423 2.655 2.843 3.000 3.135 3.253 3.357 3.451 3.535 3.613 3.684 3.749 3.810 3.867 3.920 3.970 4.351 4.608 4.951 5.184 5.358
##6 1.015 1.684 2.098 2.394 2.623 2.808 2.964 3.097 3.213 3.317 3.409 3.493 3.569 3.639 3.704 3.764 3.820 3.873 3.922 4.299 4.552 4.891 5.121 5.294
##7 1.006 1.670 2.080 2.374 2.601 2.784 2.938 3.070 3.186 3.288 3.380 3.463 3.538 3.608 3.672 3.732 3.787 3.840 3.889 4.262 4.513 4.849 5.077 5.249
##8 .9990 1.659 2.067 2.359 2.584 2.767 2.919 3.051 3.165 3.267 3.358 3.440 3.515 3.584 3.648 3.708 3.763 3.815 3.863 4.234 4.484 4.818 5.045 5.215
##9 .9938 1.651 2.057 2.347 2.571 2.753 2.905 3.035 3.149 3.250 3.341 3.423 3.498 3.566 3.630 3.689 3.744 3.796 3.844 4.213 4.461 4.794 5.020 5.189
##10 .9897 1.645 2.049 2.338 2.561 2.742 2.893 3.023 3.137 3.237 3.328 3.409 3.484 3.552 3.615 3.674 3.729 3.780 3.829 4.196 4.443 4.775 5.000 5.168
##11 .9863 1.639 2.042 2.330 2.553 2.733 2.884 3.013 3.127 3.227 3.317 3.398 3.472 3.540 3.603 3.662 3.717 3.768 3.816 4.182 4.429 4.759 4.983 5.152
##12 .9836 1.635 2.037 2.324 2.546 2.726 2.876 3.005 3.118 3.218 3.308 3.389 3.463 3.531 3.594 3.652 3.706 3.757 3.805 4.171 4.417 4.746 4.970 5.138
##13 .9812 1.631 2.032 2.319 2.540 2.719 2.869 2.998 3.111 3.210 3.300 3.381 3.455 3.522 3.585 3.643 3.698 3.749 3.796 4.161 4.406 4.735 4.958 5.126
##14 .9792 1.628 2.028 2.314 2.535 2.714 2.864 2.992 3.105 3.204 3.293 3.374 3.448 3.515 3.578 3.636 3.690 3.741 3.789 4.153 4.397 4.726 4.948 5.115
##15 .9775 1.625 2.025 2.310 2.531 2.709 2.859 2.987 3.099 3.199 3.288 3.368 3.442 3.509 3.572 3.630 3.684 3.735 3.782 4.145 4.390 4.718 4.940 5.107
##16 .9760 1.623 2.022 2.307 2.527 2.705 2.855 2.983 3.095 3.194 3.283 3.363 3.436 3.504 3.566 3.624 3.678 3.729 3.776 4.139 4.383 4.710 4.932 5.099
##17 .9747 1.621 2.019 2.304 2.524 2.702 2.851 2.979 3.090 3.189 3.278 3.359 3.432 3.499 3.561 3.619 3.673 3.724 3.771 4.133 4.377 4.704 4.926 5.092
##18 .9735 1.619 2.017 2.301 2.521 2.699 2.848 2.975 3.087 3.186 3.274 3.354 3.428 3.495 3.557 3.615 3.669 3.719 3.767 4.128 4.372 4.698 4.920 5.086
##19 .9724 1.617 2.015 2.299 2.518 2.696 2.845 2.972 3.084 3.182 3.271 3.351 3.424 3.491 3.553 3.611 3.665 3.715 3.763 4.124 4.367 4.693 4.914 5.080
##20 .9715 1.616 2.013 2.297 2.516 2.693 2.842 2.969 3.081 3.179 3.268 3.348 3.421 3.488 3.550 3.607 3.661 3.712 3.759 4.120 4.363 4.688 4.909 5.075
##24 .9685 1.611 2.007 2.290 2.508 2.685 2.833 2.960 3.071 3.170 3.258 3.337 3.410 3.477 3.539 3.596 3.650 3.700 3.747 4.107 4.349 4.674 4.894 5.060
##30 .9656 1.606 2.001 2.283 2.501 2.677 2.825 2.951 3.062 3.160 3.248 3.327 3.400 3.466 3.528 3.585 3.638 3.688 3.735 4.094 4.335 4.659 4.878 5.043
##40 .9626 1.602 1.996 2.277 2.494 2.669 2.816 2.942 3.053 3.150 3.238 3.317 3.389 3.455 3.517 3.574 3.627 3.677 3.723 4.080 4.321 4.643 4.862 5.027
##60 .9597 1.597 1.990 2.270 2.486 2.661 2.808 2.933 3.043 3.140 3.227 3.306 3.378 3.444 3.505 3.562 3.615 3.665 3.711 4.067 4.306 4.627 4.845 5.009
##120 .9568 1.592 1.984 2.263 2.479 2.653 2.799 2.924 3.034 3.130 3.217 3.296 3.367 3.433 3.494 3.550 3.603 3.652 3.699 4.052 4.290 4.610 4.827 4.990
##1e38 .9539 1.588 1.978 2.257 2.472 2.645 2.791 2.915 3.024 3.121 3.207 3.285 3.356 3.422 3.482 3.538 3.591 3.640 3.686 4.037 4.274 4.591 4.806 4.968"""
##
##q0675 = """\
##2 1.829602 2.751705 3.332700 3.754119 4.082579 4.350351 4.575528 4.769258 4.938876 5.089456 5.22465 5.347168 5.459072 5.56197 5.657136 5.745596 5.828188 5.905606 5.978428 6.534036 6.908522 7.411898 7.753537 8.010516
##3 1.660743 2.469725 2.973973 3.338757 3.622958 3.854718 4.049715 4.217574 4.364624 4.495236 4.612559 4.718926 4.816117 4.905518 4.988228 5.065133 5.136955 5.204295 5.267653 5.751485 6.07799 6.517299 6.815682 7.040219
##4 1.585479 2.344680 2.814410 3.153343 3.417165 3.632254 3.813232 3.96905 4.105579 4.226877 4.335857 4.434684 4.525004 4.608102 4.684995 4.756504 4.823298 4.885932 4.944872 5.395226 5.699385 6.108899 6.387203 6.596702
##5 1.543029 2.27426 2.72431 3.048331 3.300303 3.505645 3.678397 3.827131 3.957462 4.073264 4.177319 4.27169 4.35795 4.437321 4.510774 4.579091 4.642911 4.702763 4.759089 5.189651 5.480611 5.872552 6.139026 6.339673
##6 1.515809 2.229127 2.666435 2.980707 3.224876 3.423769 3.591058 3.735076 3.861273 3.973403 4.074164 4.165554 4.249094 4.325968 4.397115 4.463293 4.525119 4.583105 4.637678 5.054965 5.337079 5.717251 5.975813 6.170546
##7 1.496881 2.197746 2.626119 2.933501 3.17212 3.366405 3.529778 3.670406 3.793624 3.903106 4.001488 4.090723 4.172295 4.247362 4.31684 4.381468 4.441849 4.498482 4.551786 4.959448 5.235148 5.606794 5.859629 6.050083
##8 1.482962 2.174667 2.596423 2.898666 3.133126 3.323942 3.484357 3.622418 3.743376 3.850846 3.947419 4.035012 4.115085 4.188774 4.256978 4.320423 4.379701 4.435301 4.487633 4.887938 5.158734 5.523864 5.772325 5.959514
##9 1.472298 2.156982 2.573637 2.871897 3.103116 3.29122 3.449316 3.585359 3.704538 3.810421 3.905564 3.991859 4.070746 4.143343 4.210539 4.273046 4.331448 4.386229 4.437792 4.832253 5.099153 5.459107 5.704096 5.888693
##10 1.463868 2.142999 2.555601 2.85068 3.079300 3.265222 3.421445 3.555857 3.673594 3.778189 3.872171 3.957411 4.035332 4.107041 4.173413 4.235156 4.292845 4.346957 4.39789 4.787576 5.05129 5.407011 5.649161 5.831642
##11 1.457037 2.131666 2.540969 2.833447 3.059936 3.244061 3.398740 3.531802 3.648345 3.751871 3.844888 3.929251 4.006369 4.077337 4.143024 4.204129 4.261223 4.314777 4.365186 4.750882 5.01193 5.36411 5.603886 5.784574
##12 1.451389 2.122295 2.528860 2.819172 3.043878 3.226497 3.379879 3.511805 3.627341 3.729965 3.822167 3.905787 3.982224 4.052565 4.117671 4.178235 4.234823 4.287903 4.337867 4.720168 4.978946 5.32811 5.565864 5.745066
##13 1.446642 2.114418 2.518673 2.807152 3.030346 3.211683 3.363958 3.494914 3.609588 3.71144 3.802942 3.885925 3.961777 4.031579 4.096185 4.156283 4.212436 4.265108 4.314686 4.694058 4.950875 5.297431 5.533435 5.711335
##14 1.442597 2.107703 2.509984 2.796892 3.018785 3.199019 3.350338 3.480454 3.594382 3.695564 3.786459 3.868888 3.944232 4.013564 4.077734 4.137427 4.1932 4.245515 4.294759 4.671571 4.926672 5.270944 5.505418 5.682176
##15 1.439108 2.101911 2.502485 2.788030 3.008793 3.188066 3.338551 3.467934 3.581209 3.681803 3.772166 3.854108 3.929005 3.997925 4.061712 4.121047 4.176486 4.228488 4.277436 4.651989 4.905573 5.247825 5.480945 5.656694
##16 1.436068 2.096865 2.495948 2.7803 3.000071 3.178498 3.328250 3.456985 3.569684 3.669759 3.75965 3.841162 3.915663 3.984216 4.047663 4.106681 4.161824 4.213546 4.262231 4.634774 4.887015 5.227455 5.459365 5.634213
##17 1.433397 2.092428 2.490198 2.773497 2.992391 3.170069 3.319169 3.447329 3.559514 3.659127 3.748598 3.829725 3.903873 3.972099 4.035242 4.093976 4.148852 4.200325 4.248775 4.619514 4.870529 5.209359 5.440181 5.614218
##18 1.431030 2.088497 2.485101 2.767464 2.985576 3.162585 3.311102 3.438749 3.550474 3.649671 3.738765 3.819547 3.893377 3.961308 4.024178 4.082656 4.137293 4.188541 4.236778 4.605888 4.855804 5.193166 5.423003 5.596306
##19 1.428918 2.08499 2.480552 2.762076 2.979488 3.155896 3.30389 3.431072 3.542383 3.641206 3.729960 3.810429 3.883971 3.951636 4.014258 4.072505 4.126925 4.177969 4.226014 4.593643 4.84256 5.178585 5.407524 5.580158
##20 1.427023 2.081842 2.476467 2.757236 2.974016 3.149882 3.297401 3.424164 3.535099 3.633583 3.722027 3.802213 3.875493 3.942916 4.005312 4.063349 4.117571 4.168429 4.216298 4.582577 4.830579 5.16538 5.393498 5.565519
##24 1.421053 2.071924 2.463589 2.741964 2.956732 3.130867 3.276871 3.402288 3.512015 3.609405 3.696852 3.776122 3.848556 3.915194 3.976858 4.03421 4.087789 4.138042 4.185340 4.547205 4.792208 5.122986 5.348394 5.518394
##30 1.415131 2.062082 2.450796 2.726770 2.939512 3.111895 3.256356 3.380399 3.488888 3.585153 3.67157 3.749892 3.821449 3.887270 3.948172 4.00481 4.057717 4.107336 4.154034 4.511241 4.753052 5.079524 5.302021 5.469846
##40 1.409257 2.052316 2.438086 2.711654 2.922351 3.092956 3.235846 3.358481 3.465697 3.5608 3.646150 3.723487 3.794128 3.859096 3.919198 3.975085 4.027284 4.076232 4.122296 4.474532 4.712898 5.034679 5.253982 5.419412
##60 1.40343 2.042626 2.425459 2.696611 2.905244 3.074043 3.215327 3.336516 3.442417 3.536316 3.620555 3.696861 3.766541 3.830609 3.889866 3.944956 3.996401 4.044636 4.09002 4.436878 4.671454 4.987998 5.203693 5.366394
##120 1.397651 2.033010 2.412913 2.681639 2.888185 3.055146 3.194785 3.314484 3.419022 3.511665 3.59474 3.66996 3.738623 3.801735 3.86009 3.914325 3.96496 4.012423 4.057072 4.398008 4.628308 4.938805 5.150236 5.309666
##1e38 1.391918 2.023469 2.400447 2.666735 2.871167 3.036254 3.174203 3.292360 3.395479 3.486805 3.568651 3.642718 3.710296 3.772381 3.829761 3.883069 3.93282 3.979437 4.023276 4.357546 4.582861 4.886029 5.092081 5.247256"""
##
##q0750 = """\
##2 2.267583 3.308014 3.969236 4.451126 4.82785 5.13561 5.394819 5.618097 5.813776 5.987632 6.143829 6.285461 6.41489 6.533954 6.644113 6.746546 6.842214 6.931913 7.01631 7.660853 8.09584 8.68119 9.0788 9.377929
##3 2.011896 2.883775 3.431223 3.829258 4.140443 4.394852 4.609323 4.794233 4.956425 5.10064 5.230299 5.347941 5.455509 5.554514 5.646158 5.73141 5.811064 5.885775 5.956093 6.493827 6.857365 7.3472 7.680302 7.931152
##4 1.901267 2.701018 3.198596 3.559322 3.841087 4.071417 4.265624 4.433118 4.580085 4.710812 4.828384 4.935098 5.032703 5.122566 5.205771 5.283192 5.355547 5.423427 5.48733 5.976418 6.307462 6.753955 7.057827 7.286775
##5 1.839820 2.599651 3.069171 3.40865 3.673526 3.889955 4.072422 4.229795 4.367901 4.490764 4.601285 4.701617 4.793402 4.877922 4.956192 5.029034 5.09712 5.161005 5.221154 5.681792 5.993844 6.415033 6.70187 6.918073
##6 1.800788 2.535293 2.986795 3.312495 3.566338 3.773641 3.948369 4.099056 4.231292 4.348941 4.45478 4.550869 4.638783 4.719746 4.794731 4.864523 4.929764 4.990987 5.048635 5.490302 5.789693 6.194025 6.469523 6.677248
##7 1.773818 2.490830 2.929770 3.245783 3.491823 3.692639 3.86185 4.007755 4.135786 4.249692 4.352165 4.445203 4.530329 4.608729 4.681345 4.748937 4.812126 4.871427 4.927269 5.35523 5.64547 6.037624 6.304935 6.506543
##8 1.754075 2.458283 2.887956 3.196776 3.436989 3.632943 3.798002 3.940301 4.065154 4.176226 4.276148 4.366871 4.44988 4.526332 4.597147 4.663065 4.724692 4.782528 4.836995 5.254505 5.537762 5.920623 6.18169 6.378634
##9 1.739001 2.433431 2.855984 3.159245 3.394934 3.587097 3.748912 3.888384 4.010742 4.119586 4.2175 4.306396 4.387735 4.462649 4.532041 4.596635 4.657026 4.713704 4.767081 5.176307 5.454025 5.829512 6.085625 6.27887
##10 1.727116 2.413835 2.830746 3.129578 3.361648 3.550769 3.709972 3.847165 3.967506 4.074547 4.170832 4.258248 4.33823 4.411895 4.480128 4.543645 4.60303 4.658765 4.711254 5.113722 5.386914 5.756374 6.008439 6.198662
##11 1.717506 2.397989 2.810315 3.105535 3.334641 3.521265 3.678317 3.813629 3.932304 4.037852 4.132788 4.218976 4.297831 4.370457 4.437728 4.500349 4.558895 4.613844 4.665594 5.062423 5.331834 5.696254 5.944934 6.13263
##12 1.709576 2.384911 2.793439 3.085654 3.312288 3.496821 3.65207 3.785802 3.903075 4.007365 4.101163 4.186312 4.264215 4.335963 4.402419 4.46428 4.522117 4.5764 4.627523 5.019561 5.285754 5.645883 5.891679 6.077222
##13 1.70292 2.373934 2.779263 3.068939 3.293478 3.476235 3.62995 3.762334 3.878409 3.981623 4.074447 4.158707 4.235793 4.306786 4.372542 4.433751 4.490977 4.544687 4.595269 4.983178 5.246593 5.603014 5.846316 6.029999
##14 1.697255 2.364590 2.767188 3.05469 3.27743 3.458659 3.611050 3.742271 3.85731 3.959593 4.051572 4.13506 4.211437 4.281774 4.346922 4.407563 4.464258 4.517468 4.567581 4.951886 5.212872 5.566049 5.807169 5.989222
##15 1.692374 2.356539 2.756778 3.042397 3.263576 3.443476 3.594714 3.724919 3.839053 3.940521 4.031759 4.114571 4.190326 4.260088 4.324701 4.384844 4.441071 4.493842 4.54354 4.92467 5.183511 5.533819 5.773008 5.95362
##16 1.688126 2.349531 2.747712 3.031684 3.251494 3.430227 3.580451 3.709761 3.823097 3.923845 4.01443 4.096644 4.171848 4.2411 4.305239 4.364939 4.420752 4.473133 4.522464 4.900769 5.1577 5.505449 5.742915 5.92224
##17 1.684395 2.343375 2.739744 3.022264 3.240865 3.418565 3.567889 3.696405 3.809031 3.909139 3.999142 4.080823 4.155535 4.224333 4.288048 4.347353 4.402795 4.454828 4.50383 4.879604 5.134819 5.48027 5.716186 5.894353
##18 1.681092 2.337926 2.732687 3.013916 3.23144 3.408218 3.55674 3.684546 3.796536 3.896071 3.985552 4.066754 4.141026 4.209415 4.27275 4.331699 4.386808 4.438527 4.487232 4.860723 5.114389 5.457759 5.692272 5.86939
##19 1.678147 2.333067 2.726393 3.006467 3.223026 3.398978 3.546777 3.673945 3.785363 3.884381 3.973391 4.054162 4.128035 4.196054 4.259045 4.317673 4.37248 4.423914 4.472351 4.843772 5.096029 5.437505 5.67074 5.846902
##20 1.675506 2.328708 2.720745 2.999780 3.215469 3.390674 3.537821 3.664411 3.775312 3.873861 3.962444 4.042823 4.116334 4.184018 4.246696 4.305031 4.359563 4.410739 4.458932 4.828464 5.079434 5.419178 5.651242 5.82653
##24 1.667194 2.314991 2.702957 2.978701 3.191627 3.364455 3.50952 3.63426 3.7435 3.840544 3.927753 4.006868 4.079211 4.14581 4.207477 4.264864 4.318505 4.368841 4.416241 4.779619 5.026378 5.360437 5.58865 5.761054
##30 1.658964 2.301406 2.68532 2.957771 3.167919 3.338345 3.481298 3.604155 3.711699 3.8072 3.892996 3.970809 4.041947 4.107423 4.168039 4.224442 4.277156 4.326617 4.373187 4.730101 4.9724 5.300398 5.524486 5.693793
##40 1.650814 2.28795 2.667830 2.936984 3.144337 3.312335 3.453142 3.574077 3.679882 3.773798 3.858136 3.934602 4.004488 4.068796 4.128318 4.183691 4.235434 4.283976 4.329675 4.679735 4.917252 5.238689 5.458277 5.62419
##60 1.642744 2.274622 2.650486 2.916339 3.120874 3.286413 3.425034 3.544004 3.648023 3.740302 3.823131 3.898196 3.966776 4.029861 4.088234 4.142523 4.19324 4.24081 4.285584 4.628295 4.860604 5.174794 5.389348 5.551435
##120 1.634753 2.261421 2.633285 2.895829 3.097525 3.260567 3.396959 3.513912 3.616089 3.706672 3.787929 3.861531 3.92874 3.990536 4.047692 4.10083 4.150455 4.196985 4.240768 4.575490 4.802013 5.107977 5.316696 5.474283
##1e38 1.626840 2.248346 2.616224 2.875451 3.074279 3.234786 3.368898 3.483775 3.584045 3.672862 3.752475 3.824535 3.890294 3.950721 4.006580 4.058483 4.106932 4.152338 4.195044 4.520933 4.740866 5.037152 5.238766 5.390726"""
##
##q0800 = """\
##2 2.666345 3.820436 4.558532 5.098158 5.520848 5.866626 6.158145 6.409446 6.62982 6.825717 7.001791 7.161505 7.307502 7.441845 7.566171 7.681802 7.789818 7.891113 7.986436 8.714887 9.206808 9.868718 10.31830 10.65683
##3 2.316120 3.245426 3.832597 4.261107 4.596942 4.871989 5.104169 5.304561 5.480484 5.637021 5.777843 5.905682 6.022626 6.130305 6.230013 6.322797 6.409513 6.49087 6.567462 7.153711 7.55053 8.085698 8.449862 8.724212
##4 2.168283 3.003795 3.52645 3.90676 4.204595 4.44853 4.654519 4.832388 4.988615 5.127694 5.25287 5.366554 5.470593 5.566425 5.655195 5.737827 5.815079 5.887577 5.955847 6.47896 6.833568 7.31242 7.638648 7.884592
##5 2.087215 2.871505 3.358337 3.711564 3.987876 4.214094 4.405111 4.57007 4.714986 4.844026 4.960193 5.065723 5.162321 5.25132 5.333779 5.410553 5.482342 5.549725 5.613191 6.099852 6.430105 6.876484 7.180827 7.410389
##6 2.036122 2.788188 3.252203 3.588013 3.850385 4.06507 4.246305 4.402806 4.540296 4.662734 4.772969 4.873124 4.964814 5.049304 5.127595 5.200498 5.268677 5.33268 5.392969 5.855517 6.169658 6.594568 6.884463 7.103222
##7 2.001005 2.730943 3.179141 3.502777 3.755348 3.961886 4.136188 4.286677 4.418877 4.536604 4.642603 4.738913 4.82709 4.908349 4.983652 5.05378 5.119368 5.180945 5.238953 5.684175 5.986732 6.39621 6.675724 6.886723
##8 1.975395 2.689205 3.125785 3.440421 3.685706 3.886163 4.055272 4.201248 4.329470 4.443647 4.546448 4.639854 4.725375 4.804189 4.87723 4.945254 5.008879 5.068616 5.124894 5.556957 5.850708 6.248453 6.520077 6.725179
##9 1.955898 2.657432 3.085114 3.392817 3.632464 3.828199 3.993263 4.135716 4.260824 4.37222 4.472512 4.563637 4.64707 4.723960 4.79522 4.861587 4.923664 4.981949 5.036862 5.458529 5.745314 6.133773 6.399153 6.59959
##10 1.940561 2.632439 3.053086 3.355281 3.590431 3.782386 3.944205 4.083824 4.206424 4.315576 4.413841 4.503121 4.584862 4.660193 4.730009 4.795032 4.855852 4.912959 4.966762 5.37997 5.661078 6.041965 6.302252 6.498885
##11 1.928182 2.612267 3.027211 3.324922 3.556399 3.745257 3.904411 4.041698 4.16223 4.269529 4.366119 4.453871 4.534212 4.608251 4.676869 4.740775 4.800552 4.85668 4.909561 5.315725 5.5921 5.966667 6.222702 6.41616
##12 1.917981 2.595645 3.005872 3.299861 3.528278 3.714552 3.871475 4.006806 4.125603 4.231343 4.326522 4.412988 4.492148 4.565096 4.632701 4.695665 4.754559 4.809858 4.861959 5.262152 5.534506 5.9037 6.156119 6.346875
##13 1.909431 2.581711 2.987972 3.278821 3.504650 3.688731 3.843759 3.977426 4.094742 4.199153 4.293127 4.378492 4.45664 4.528654 4.595391 4.657546 4.715682 4.770269 4.8217 5.216755 5.485643 5.850201 6.099497 6.287919
##14 1.90216 2.569864 2.972742 3.260906 3.484516 3.666713 3.82011 3.952342 4.068380 4.171642 4.264573 4.348985 4.426256 4.497459 4.563444 4.624895 4.682373 4.736342 4.787189 5.177769 5.443632 5.804138 6.050703 6.237084
##15 1.895903 2.559666 2.959626 3.245467 3.467154 3.647714 3.799691 3.930673 4.045596 4.147854 4.239872 4.32345 4.399954 4.470446 4.53577 4.596605 4.653505 4.706931 4.757265 5.143906 5.4071 5.764029 6.00818 6.192757
##16 1.89046 2.550797 2.948212 3.232024 3.452027 3.631152 3.781882 3.911763 4.025705 4.127077 4.218291 4.301132 4.376957 4.446821 4.511561 4.571849 4.628237 4.681181 4.731062 5.114203 5.375025 5.728765 5.970765 6.153733
##17 1.885683 2.543012 2.93819 3.220213 3.438729 3.616585 3.766210 3.895117 4.008187 4.108772 4.19927 4.281455 4.356676 4.42598 4.490198 4.549999 4.605930 4.658444 4.707919 5.08793 5.346623 5.697501 5.937567 6.119088
##18 1.881457 2.536125 2.929319 3.209754 3.426947 3.603672 3.752313 3.880348 3.992639 4.09252 4.182377 4.263975 4.338653 4.407454 4.471204 4.530568 4.586089 4.638215 4.687324 5.064516 5.321288 5.66958 5.907896 6.088107
##19 1.877691 2.529988 2.921412 3.200427 3.416435 3.592147 3.739903 3.867156 3.978745 4.077993 4.167272 4.24834 4.322529 4.390877 4.454204 4.513172 4.568321 4.620098 4.668877 5.043513 5.298541 5.644481 5.881205 6.060224
##20 1.874315 2.524485 2.914320 3.192058 3.407000 3.581797 3.728755 3.855300 3.966255 4.064929 4.153685 4.234272 4.308018 4.375954 4.438897 4.497506 4.552317 4.603776 4.652254 5.024562 5.277998 5.621789 5.857057 6.034984
##24 1.863701 2.507187 2.89201 3.165709 3.377268 3.549158 3.69357 3.817854 3.92678 4.023614 4.110690 4.189731 4.262048 4.328656 4.390359 4.447807 4.501527 4.551956 4.599461 4.964202 5.212441 5.549191 5.779675 5.954012
##30 1.853207 2.490080 2.869925 3.139592 3.347756 3.516716 3.658554 3.780544 3.887403 3.982357 4.067712 4.145167 4.216014 4.281252 4.341675 4.397921 4.450509 4.49987 4.546362 4.903188 5.145946 5.47522 5.700597 5.871091
##40 1.842829 2.473164 2.848060 3.113699 3.318456 3.484461 3.62369 3.743347 3.848096 3.941126 4.024712 4.100533 4.16986 4.233681 4.292775 4.34777 4.399178 4.447421 4.492854 4.841333 5.078248 5.399473 5.619306 5.785608
##60 1.832568 2.456435 2.826413 3.088026 3.289358 3.452379 3.588962 3.70624 3.808829 3.899882 3.981645 4.055774 4.123524 4.185868 4.243575 4.297261 4.34743 4.394499 4.438814 4.778404 5.009002 5.321406 5.535087 5.696698
##120 601.822421 2.439890 2.804980 3.062567 3.260456 3.420458 3.55435 3.669198 3.769570 3.858583 3.938458 4.010829 4.076934 4.137731 4.193979 4.246285 4.295145 4.340968 4.384094 4.714106 4.937761 5.24027 5.446912 5.603078
##1e38 1.812388 2.423529 2.783758 3.037317 3.231739 3.388684 3.519834 3.632192 3.73028 3.817183 3.895093 3.965627 4.030005 4.089173 4.143877 4.194716 4.242179 4.286668 4.328517 4.648069 4.863937 5.155024 5.353283 5.50281"""
##
##q0850 = """\
##2 3.226562 4.548022 5.398759 6.022701 6.512387 6.913502 7.251997 7.54401 7.800236 8.028116 8.233021 8.418953 8.588968 8.74545 8.890294 9.02503 9.150913 9.268977 9.380094 10.22972 10.80450 11.58094 12.11086 12.51097
##3 2.721399 3.731515 4.374509 4.845675 5.215912 5.5197 5.776502 5.998388 6.193356 6.366968 6.523249 6.665198 6.795111 6.914781 7.025634 7.128823 7.225292 7.315823 7.401073 8.054202 8.496827 9.094477 9.501702 9.808753
##4 2.514747 3.399285 3.956491 4.363675 4.68348 4.945965 5.16798 5.359938 5.52872 5.679113 5.814574 5.937683 6.050411 6.154302 6.25058 6.34024 6.424092 6.502812 6.576963 7.145835 7.532079 8.054293 8.410406 8.679063
##5 2.403262 3.220436 3.730867 4.102766 4.394545 4.633955 4.836465 5.011596 5.165628 5.302922 5.426626 5.539086 5.642096 5.737057 5.825086 5.907085 5.983793 6.055822 6.123687 6.644817 6.999123 7.478735 7.806159 8.05333
##6 2.333697 3.108965 3.589945 3.939419 4.213263 4.437836 4.627757 4.791998 4.936465 5.06525 5.181307 5.286833 5.38351 5.47265 5.555297 5.632296 5.704339 5.771998 5.835756 6.325681 6.659107 7.110867 7.419518 7.652631
##7 2.286206 3.032919 3.493643 3.827573 4.088914 4.303095 4.484169 4.640738 4.77845 4.901217 5.011858 5.112468 5.20465 5.289655 5.368478 5.441922 5.510646 5.575196 5.63603 6.103717 6.422253 6.85415 7.149428 7.372543
##8 2.251741 2.977758 3.423692 3.746199 3.998303 4.204778 4.379271 4.530117 4.662781 4.781044 4.887624 4.984545 5.073351 5.155248 5.231193 5.30196 5.368185 5.430392 5.489022 5.939926 6.24721 6.664096 6.949269 7.16483
##9 2.225598 2.935932 3.370588 3.684337 3.92933 4.12985 4.299243 4.445643 4.57438 4.68913 4.792542 4.88658 4.972745 5.052208 5.125899 5.194568 5.258832 5.319201 5.376101 5.81381 6.112237 6.517297 6.794508 7.004117
##10 2.205093 2.903132 3.328904 3.635722 3.875064 4.070838 4.236155 4.378996 4.504581 4.61651 4.717372 4.809088 4.893124 4.970624 5.042494 5.109469 5.172148 5.231029 5.28653 5.713546 6.004781 6.400234 6.670974 6.875744
##11 2.18858 2.876725 3.295316 3.596509 3.831250 4.023149 4.18513 4.325052 4.448048 4.557656 4.656418 4.74622 4.828499 4.904377 4.974743 5.040315 5.101683 5.159333 5.213674 5.631818 5.917077 6.304535 6.569888 6.770631
##12 2.174999 2.855009 3.267675 3.564211 3.795132 3.983803 4.143002 4.280484 4.401312 4.508975 4.605974 4.694168 4.77497 4.849483 4.918582 4.982973 5.043235 5.099847 5.153209 5.563854 5.84405 6.22473 6.485512 6.682838
##13 2.163633 2.836837 3.244531 3.537147 3.764842 3.950784 4.107624 4.243034 4.36202 4.468027 4.563524 4.650346 4.729887 4.803233 4.871249 4.93463 4.993946 5.049668 5.102191 5.5064 5.782244 6.157087 6.413931 6.608311
##14 2.153982 2.821408 3.224869 3.514139 3.739075 3.922677 4.077491 4.21112 4.328519 4.433097 4.527298 4.612934 4.691385 4.763723 4.830801 4.893306 4.951801 5.006752 5.058548 5.457164 5.729217 6.09897 6.352377 6.544185
##15 2.145684 2.808145 3.207959 3.494339 3.716887 3.898461 4.051515 4.183594 4.299611 4.402944 4.496014 4.580615 4.658112 4.729568 4.795825 4.857564 4.91534 4.969615 5.020773 5.414476 5.683193 6.048461 6.298836 6.488374
##16 2.138475 2.796621 3.193261 3.47712 3.697581 3.877377 4.028889 4.159607 4.274409 4.376647 4.468721 4.55241 4.629066 4.699743 4.765275 4.826337 4.883478 4.937154 4.987748 5.377097 5.642851 6.004129 6.251805 6.439321
##17 2.132153 2.786517 3.180367 3.462007 3.680628 3.858856 4.009003 4.138517 4.252242 4.353508 4.444698 4.527576 4.603485 4.673469 4.738356 4.798814 4.855389 4.908532 4.958622 5.344083 5.607184 5.964886 6.21014 6.395841
##18 2.126564 2.777584 3.168965 3.448637 3.665623 3.842455 3.991387 4.119827 4.232591 4.332989 4.423388 4.505541 4.58078 4.650143 4.714452 4.774369 4.830436 4.883101 4.932739 5.314701 5.575413 5.929888 6.172953 6.357013
##19 2.121587 2.769631 3.158810 3.436724 3.652248 3.82783 3.975673 4.103149 4.21505 4.314667 4.404354 4.485854 4.560491 4.629294 4.693081 4.752509 4.808118 4.860351 4.909581 5.288379 5.546924 5.898469 6.139544 6.322112
##20 2.117128 2.762505 3.149708 3.426043 3.640251 3.814707 3.961568 4.088173 4.199295 4.298206 4.387249 4.468158 4.542248 4.610545 4.673858 4.732844 4.788036 4.839877 4.888736 5.264656 5.521227 5.870097 6.109355 6.290558
##24 2.103128 2.740133 3.121118 3.392466 3.60251 3.773393 3.917129 4.040961 4.149593 4.246248 4.33323 4.412242 4.484578 4.551244 4.613036 4.670595 4.724445 4.775019 4.822681 5.189274 5.439419 5.779554 6.012859 6.189586
##30 2.089309 2.718054 3.092876 3.359261 3.56514 3.732436 3.873024 3.994053 4.10016 4.19452 4.2794 4.356475 4.427015 4.492009 4.552236 4.608325 4.660792 4.710058 4.756481 5.113372 5.356776 5.687685 5.914664 6.08662
##40 2.07567 2.696264 3.064978 3.326418 3.52813 3.691822 3.829233 3.947423 4.050965 4.142986 4.225718 4.300807 4.369502 4.432773 4.491385 4.545955 4.596987 4.644896 4.69003 5.036754 5.27302 5.594062 5.814221 5.981005
##60 2.062208 2.674759 3.037417 3.293931 3.491470 3.651535 3.785736 3.901046 4.001976 4.091607 4.172136 4.245183 4.311975 4.373463 4.430401 4.483391 4.532928 4.579419 4.623205 4.95919 5.187807 5.498134 5.710792 5.871841
##120 2.048920 2.653534 3.010189 3.261791 3.455148 3.611561 3.742514 3.854896 3.953159 4.040341 4.118605 4.189543 4.254363 4.314 4.369191 4.42053 4.4685 4.513501 4.555864 4.880396 5.100706 5.399172 5.60337 5.757859
##1e38 2.035805 2.632586 2.983286 3.229990 3.419154 3.571884 3.699544 3.808945 3.904479 3.989143 4.065068 4.133821 4.19659 4.254292 4.307653 4.357255 4.403572 4.446994 4.487848 4.800043 5.011193 5.296241 5.4906 5.637297"""
##
##q0900 = """\
##1 8.929 13.44 16.36 18.49 20.15 21.51 22.64 23.62 24.48 25.24 25.92 26.54 27.10 27.62 28.10 28.54 28.96 29.35 29.71 32.50 34.38 36.91 38.62 39.91
##2 4.129 5.733 6.773 7.538 8.139 8.633 9.049 9.409 9.725 10.01 10.26 10.49 10.70 10.89 11.07 11.24 11.39 11.54 11.68 12.73 13.44 14.40 15.04 15.54
##3 3.328 4.467 5.199 5.738 6.162 6.511 6.806 7.062 7.287 7.487 7.667 7.831 7.982 8.120 8.248 8.368 8.479 8.584 8.683 9.440 9.954 10.65 11.12 11.48
##4 3.015 3.976 4.586 5.035 5.388 5.679 5.926 6.139 6.327 6.494 6.645 6.783 6.909 7.025 7.132 7.233 7.326 7.414 7.497 8.135 8.569 9.156 9.558 9.861
##5 2.850 3.717 4.264 4.664 4.979 5.238 5.458 5.648 5.816 5.965 6.100 6.223 6.336 6.439 6.536 6.626 6.710 6.788 6.863 7.435 7.824 8.353 8.714 8.987
##6 2.748 3.558 4.065 4.435 4.726 4.966 5.168 5.344 5.499 5.637 5.762 5.875 5.979 6.075 6.164 6.247 6.325 6.398 6.466 6.996 7.358 7.848 8.184 8.438
##7 2.679 3.451 3.931 4.280 4.555 4.780 4.971 5.137 5.283 5.413 5.530 5.637 5.735 5.826 5.910 5.988 6.061 6.130 6.195 6.695 7.036 7.500 7.818 8.059
##8 2.630 3.374 3.834 4.169 4.431 4.646 4.829 4.987 5.126 5.250 5.362 5.464 5.558 5.644 5.724 5.799 5.869 5.935 5.997 6.475 6.801 7.245 7.550 7.780
##9 2.592 3.316 3.761 4.084 4.337 4.545 4.721 4.873 5.007 5.126 5.234 5.333 5.423 5.506 5.583 5.655 5.722 5.786 5.845 6.306 6.621 7.049 7.343 7.566
##10 2.563 3.270 3.704 4.018 4.264 4.465 4.636 4.783 4.913 5.029 5.134 5.229 5.316 5.397 5.472 5.542 5.607 5.668 5.726 6.173 6.478 6.894 7.180 7.396
##11 2.540 3.234 3.658 3.965 4.205 4.401 4.567 4.711 4.838 4.951 5.053 5.145 5.231 5.309 5.382 5.450 5.514 5.573 5.630 6.065 6.363 6.768 7.046 7.257
##12 2.521 3.204 3.621 3.921 4.156 4.349 4.511 4.652 4.776 4.886 4.986 5.076 5.160 5.236 5.308 5.374 5.436 5.495 5.550 5.975 6.267 6.663 6.936 7.142
##13 2.504 3.179 3.589 3.885 4.116 4.304 4.464 4.602 4.724 4.832 4.930 5.019 5.100 5.175 5.245 5.310 5.371 5.429 5.483 5.900 6.186 6.575 6.842 7.045
##14 2.491 3.158 3.563 3.854 4.081 4.267 4.424 4.560 4.679 4.786 4.882 4.969 5.050 5.124 5.192 5.256 5.316 5.372 5.426 5.836 6.116 6.499 6.762 6.961
##15 2.479 3.140 3.540 3.828 4.052 4.235 4.390 4.524 4.641 4.746 4.841 4.927 5.006 5.079 5.146 5.209 5.268 5.324 5.376 5.780 6.056 6.433 6.692 6.888
##16 2.469 3.124 3.520 3.804 4.026 4.207 4.360 4.492 4.608 4.712 4.805 4.890 4.968 5.040 5.106 5.169 5.227 5.282 5.333 5.731 6.004 6.376 6.631 6.825
##17 2.460 3.110 3.503 3.784 4.003 4.182 4.334 4.464 4.579 4.681 4.774 4.857 4.934 5.005 5.071 5.133 5.190 5.244 5.295 5.688 5.958 6.325 6.577 6.769
##18 2.452 3.098 3.487 3.766 3.984 4.161 4.310 4.440 4.553 4.654 4.746 4.829 4.905 4.975 5.040 5.101 5.158 5.211 5.262 5.650 5.917 6.280 6.529 6.718
##19 2.445 3.087 3.474 3.751 3.966 4.142 4.290 4.418 4.530 4.630 4.721 4.803 4.878 4.948 5.012 5.072 5.129 5.182 5.232 5.616 5.880 6.239 6.486 6.673
##20 2.439 3.077 3.462 3.736 3.950 4.124 4.271 4.398 4.510 4.609 4.699 4.780 4.855 4.923 4.987 5.047 5.103 5.155 5.205 5.586 5.847 6.202 6.447 6.633
##24 2.420 3.047 3.423 3.692 3.900 4.070 4.213 4.336 4.445 4.541 4.628 4.707 4.780 4.847 4.909 4.966 5.020 5.071 5.119 5.489 5.741 6.086 6.323 6.503
##30 2.400 3.017 3.386 3.648 3.851 4.016 4.155 4.275 4.381 4.474 4.559 4.635 4.706 4.770 4.830 4.886 4.939 4.988 5.034 5.391 5.636 5.969 6.198 6.372
##40 2.381 2.988 3.348 3.605 3.802 3.963 4.099 4.215 4.317 4.408 4.490 4.564 4.632 4.694 4.752 4.806 4.857 4.904 4.949 5.294 5.529 5.850 6.071 6.238
##60 2.363 2.959 3.312 3.562 3.755 3.911 4.042 4.155 4.254 4.342 4.421 4.493 4.558 4.619 4.675 4.727 4.775 4.821 4.864 5.196 5.422 5.730 5.941 6.101
##120 2.344 2.930 3.276 3.520 3.707 3.859 3.986 4.096 4.191 4.276 4.353 4.422 4.485 4.543 4.597 4.647 4.694 4.738 4.779 5.097 5.313 5.606 5.808 5.960
##1e38 2.326 2.902 3.240 3.478 3.661 3.808 3.931 4.037 4.129 4.211 4.285 4.351 4.412 4.468 4.519 4.568 4.612 4.654 4.694 4.997 5.202 5.480 5.669 5.812"""
##
##q0950 = """\
##1 17.97 26.98 32.82 37.08 40.41 43.12 45.40 47.36 49.07 50.59 51.96 53.20 54.33 55.36 56.32 57.22 58.04 58.83 59.56 65.15 68.92 73.97 77.40 79.98
##2 6.085 8.331 9.799 10.88 11.73 12.43 13.03 13.54 13.99 14.40 14.76 15.09 15.39 15.65 15.92 16.14 16.38 16.57 16.78 18.27 19.28 20.66 21.59 22.29
##3 4.501 5.910 6.825 7.502 8.037 8.478 8.852 9.177 9.462 9.717 9.946 10.15 10.35 10.52 10.69 10.84 10.98 11.11 11.24 12.21 12.86 13.76 14.36 14.82
##4 3.926 5.040 5.757 6.287 6.706 7.053 7.347 7.602 7.826 8.027 8.208 8.373 8.524 8.664 8.793 8.914 9.027 9.133 9.233 10.00 10.53 11.24 11.73 12.10
##5 3.635 4.602 5.218 5.673 6.033 6.330 6.582 6.801 6.995 7.167 7.323 7.466 7.596 7.716 7.828 7.932 8.030 8.122 8.208 8.875 9.330 9.949 10.37 10.69
##6 3.460 4.339 4.896 5.305 5.628 5.895 6.122 6.319 6.493 6.649 6.789 6.917 7.034 7.143 7.244 7.338 7.426 7.508 7.586 8.189 8.601 9.162 9.547 9.839
##7 3.344 4.165 4.681 5.060 5.359 5.606 5.815 5.997 6.158 6.302 6.431 6.550 6.658 6.759 6.852 6.939 7.020 7.097 7.169 7.727 8.110 8.631 8.989 9.260
##8 3.261 4.041 4.529 4.886 5.167 5.399 5.596 5.767 5.918 6.053 6.175 6.287 6.389 6.483 6.571 6.653 6.729 6.801 6.869 7.395 7.756 8.248 8.586 8.843
##9 3.199 3.948 4.415 4.755 5.024 5.244 5.432 5.595 5.738 5.867 5.983 6.089 6.186 6.276 6.359 6.437 6.510 6.579 6.643 7.144 7.488 7.958 8.281 8.526
##10 3.151 3.877 4.327 4.654 4.912 5.124 5.304 5.460 5.598 5.722 5.833 5.935 6.028 6.114 6.194 6.269 6.339 6.405 6.467 6.948 7.278 7.730 8.041 8.276
##11 3.113 3.820 4.256 4.574 4.823 5.028 5.202 5.353 5.486 5.605 5.713 5.811 5.901 5.984 6.062 6.134 6.202 6.265 6.325 6.790 7.109 7.546 7.847 8.075
##12 3.081 3.773 4.199 4.508 4.750 4.950 5.119 5.265 5.395 5.510 5.615 5.710 5.797 5.878 5.953 6.023 6.089 6.151 6.209 6.660 6.970 7.394 7.687 7.908
##13 3.055 3.734 4.151 4.453 4.690 4.884 5.049 5.192 5.318 5.431 5.533 5.625 5.711 5.789 5.862 5.931 5.995 6.055 6.112 6.551 6.853 7.267 7.552 7.769
##14 3.033 3.701 4.111 4.407 4.639 4.829 4.990 5.130 5.253 5.364 5.463 5.554 5.637 5.714 5.785 5.852 5.915 5.973 6.029 6.459 6.754 7.159 7.437 7.649
##15 3.014 3.673 4.076 4.367 4.595 4.782 4.940 5.077 5.198 5.306 5.403 5.492 5.574 5.649 5.719 5.785 5.846 5.904 5.958 6.379 6.669 7.065 7.338 7.546
##16 2.998 3.649 4.046 4.333 4.557 4.741 4.896 5.031 5.150 5.256 5.352 5.439 5.519 5.593 5.662 5.726 5.786 5.843 5.896 6.310 6.594 6.983 7.252 7.456
##17 2.984 3.628 4.020 4.303 4.524 4.705 4.858 4.991 5.108 5.212 5.306 5.392 5.471 5.544 5.612 5.675 5.734 5.790 5.842 6.249 6.529 6.912 7.176 7.377
##18 2.971 3.609 3.997 4.276 4.494 4.673 4.824 4.955 5.071 5.173 5.266 5.351 5.429 5.501 5.567 5.629 5.688 5.743 5.794 6.195 6.471 6.848 7.108 7.307
##19 2.960 3.593 3.977 4.253 4.468 4.645 4.794 4.924 5.037 5.139 5.231 5.314 5.391 5.462 5.528 5.589 5.647 5.701 5.752 6.147 6.419 6.791 7.048 7.244
##20 2.950 3.578 3.958 4.232 4.445 4.620 4.768 4.895 5.008 5.108 5.199 5.282 5.357 5.427 5.492 5.553 5.610 5.663 5.714 6.104 6.372 6.740 6.994 7.187
##24 2.919 3.532 3.901 4.166 4.373 4.541 4.684 4.807 4.915 5.012 5.099 5.179 5.251 5.319 5.381 5.439 5.494 5.545 5.594 5.968 6.226 6.578 6.822 7.007
##30 2.888 3.486 3.845 4.102 4.301 4.464 4.601 4.720 4.824 4.917 5.001 5.077 5.147 5.211 5.271 5.327 5.379 5.429 5.475 5.833 6.080 6.417 6.650 6.827
##40 2.858 3.442 3.791 4.039 4.232 4.388 4.521 4.634 4.735 4.824 4.904 4.977 5.044 5.106 5.163 5.216 5.266 5.313 5.358 5.700 5.934 6.255 6.477 6.645
##60 2.829 3.399 3.737 3.977 4.163 4.314 4.441 4.550 4.646 4.732 4.808 4.878 4.942 5.001 5.056 5.107 5.154 5.199 5.241 5.566 5.789 6.093 6.302 6.462
##120 2.800 3.356 3.685 3.917 4.096 4.241 4.363 4.468 4.560 4.641 4.714 4.781 4.842 4.898 4.950 4.998 5.043 5.086 5.126 5.434 5.644 5.929 6.126 6.275
##1e38 2.772 3.314 3.633 3.858 4.030 4.170 4.286 4.387 4.474 4.552 4.622 4.685 4.743 4.796 4.845 4.891 4.934 4.974 5.012 5.301 5.498 5.764 5.947 6.085"""
##
##q0975 = """\
##1 35.99 54.00 65.69 74.22 80.87 86.29 90.85 94.77 98.20 101.3 104.0 106.5 108.8 110.8 112.7 114.5 116.2 117.7 119.2 130.4 137.9 148.1 154.9 160.0
##2 8.776 11.94 14.02 15.54 16.75 17.74 18.58 19.31 19.95 20.52 21.03 21.49 21.91 22.30 22.67 23.01 23.32 23.62 23.89 26.03 27.47 29.42 30.74 31.74
##3 5.907 7.661 8.808 9.659 10.33 10.89 11.36 11.77 12.14 12.46 12.75 13.01 13.25 13.47 13.68 13.87 14.05 14.22 14.38 15.62 16.46 17.58 18.37 18.95
##4 4.943 6.244 7.088 7.715 8.213 8.625 8.975 9.279 9.548 9.788 10.00 10.20 10.38 10.55 10.71 10.85 10.99 11.11 11.23 12.16 12.78 13.65 14.23 14.68
##5 4.474 5.558 6.257 6.775 7.186 7.526 7.816 8.068 8.291 8.490 8.670 8.834 8.984 9.124 9.253 9.373 9.486 9.592 9.693 10.47 11.00 11.72 12.21 12.59
##6 4.198 5.158 5.772 6.226 6.586 6.884 7.138 7.359 7.554 7.729 7.887 8.031 8.163 8.285 8.399 8.505 8.605 8.698 8.787 9.469 9.937 10.57 11.01 11.34
##7 4.018 4.897 5.455 5.867 6.194 6.464 6.694 6.894 7.071 7.230 7.373 7.504 7.624 7.735 7.838 7.935 8.025 8.110 8.191 8.812 9.239 9.822 10.22 10.53
##8 3.891 4.714 5.233 5.616 5.919 6.169 6.382 6.567 6.731 6.878 7.011 7.132 7.244 7.347 7.442 7.532 7.616 7.694 7.769 8.346 8.743 9.286 9.660 9.944
##9 3.797 4.578 5.069 5.430 5.715 5.950 6.151 6.325 6.479 6.617 6.742 6.856 6.961 7.057 7.148 7.232 7.311 7.385 7.455 7.999 8.373 8.885 9.238 9.506
##10 3.725 4.474 4.943 5.286 5.558 5.782 5.972 6.138 6.284 6.415 6.534 6.642 6.742 6.834 6.920 7.000 7.075 7.145 7.212 7.729 8.085 8.574 8.910 9.166
##11 3.667 4.391 4.843 5.173 5.433 5.648 5.830 5.989 6.130 6.255 6.369 6.473 6.568 6.656 6.738 6.815 6.887 6.955 7.018 7.514 7.856 8.324 8.648 8.894
##12 3.620 4.324 4.761 5.080 5.332 5.539 5.715 5.868 6.004 6.125 6.235 6.335 6.426 6.511 6.591 6.664 6.734 6.799 6.860 7.338 7.668 8.120 8.433 8.670
##13 3.582 4.269 4.694 5.004 5.248 5.449 5.620 5.768 5.899 6.017 6.123 6.220 6.309 6.391 6.468 6.539 6.606 6.670 6.729 7.192 7.511 7.950 8.253 8.484
##14 3.549 4.222 4.638 4.940 5.178 5.373 5.540 5.684 5.812 5.926 6.029 6.123 6.210 6.290 6.364 6.434 6.499 6.560 6.618 7.068 7.379 7.806 8.100 8.325
##15 3.521 4.182 4.589 4.885 5.117 5.309 5.471 5.612 5.736 5.848 5.948 6.040 6.125 6.205 6.275 6.343 6.407 6.467 6.523 6.962 7.265 7.682 7.969 8.189
##16 3.497 4.148 4.548 4.838 5.065 5.253 5.412 5.550 5.671 5.780 5.879 5.969 6.051 6.128 6.199 6.265 6.327 6.386 6.441 6.870 7.167 7.574 7.856 8.070
##17 3.476 4.118 4.511 4.797 5.020 5.204 5.360 5.495 5.615 5.722 5.818 5.906 5.987 6.062 6.132 6.197 6.258 6.315 6.369 6.790 7.080 7.479 7.756 7.966
##18 3.458 4.091 4.479 4.760 4.980 5.161 5.315 5.448 5.565 5.670 5.765 5.851 5.931 6.004 6.073 6.137 6.196 6.253 6.306 6.719 7.004 7.396 7.667 7.874
##19 3.441 4.068 4.451 4.728 4.945 5.123 5.274 5.405 5.521 5.624 5.717 5.803 5.881 5.953 6.020 6.083 6.142 6.197 6.250 6.656 6.936 7.322 7.589 7.792
##20 3.427 4.047 4.426 4.699 4.914 5.089 5.238 5.367 5.481 5.583 5.675 5.759 5.836 5.907 5.974 6.035 6.093 6.148 6.199 6.599 6.876 7.255 7.518 7.718
##24 3.381 3.982 4.347 4.610 4.816 4.984 5.126 5.250 5.358 5.455 5.543 5.623 5.696 5.764 5.827 5.886 5.941 5.993 6.042 6.422 6.685 7.046 7.295 7.486
##30 3.337 3.919 4.271 4.523 4.720 4.881 5.017 5.134 5.238 5.330 5.414 5.490 5.560 5.624 5.684 5.740 5.792 5.841 5.888 6.248 6.497 6.838 7.075 7.255
##40 3.294 3.858 4.196 4.439 4.627 4.780 4.910 5.022 5.120 5.208 5.287 5.360 5.426 5.487 5.543 5.596 5.646 5.692 5.736 6.077 6.311 6.633 6.855 7.025
##60 3.251 3.798 4.124 4.356 4.536 4.682 4.806 4.912 5.006 5.089 5.164 5.232 5.295 5.352 5.406 5.456 5.502 5.546 5.588 5.908 6.127 6.428 6.636 6.795
##120 3.210 3.739 4.053 4.275 4.447 4.587 4.704 4.805 4.894 4.972 5.043 5.107 5.166 5.221 5.271 5.318 5.362 5.403 5.442 5.741 5.946 6.225 6.418 6.564
##1e38 3.170 3.682 3.984 4.197 4.361 4.494 4.605 4.700 4.784 4.858 4.925 4.985 5.041 5.092 5.139 5.183 5.224 5.262 5.299 5.577 5.766 6.023 6.199 6.333"""
##
##q0990 = """\
##1 90.02 135.0 164.3 185.6 202.2 215.8 227.2 237.0 245.6 253.2 260.0 266.2 271.8 277.0 281.8 286.3 290.4 294.3 298.0 326.0 344.8 370.1 387.3 400.1
##2 14.04 19.02 22.29 24.72 26.63 28.20 29.53 30.68 31.69 32.59 33.40 34.13 34.81 35.43 36.00 36.53 37.03 37.50 37.95 41.32 43.61 46.70 48.80 50.38
##3 8.260 10.62 12.17 13.32 14.24 15.00 15.65 16.21 16.69 17.13 17.53 17.89 18.22 18.52 18.81 19.07 19.32 19.55 19.77 21.44 22.59 24.13 25.19 25.99
##4 6.511 8.120 9.173 9.958 10.58 11.10 11.54 11.92 12.26 12.57 12.84 13.09 13.32 13.53 13.72 13.91 14.08 14.24 14.39 15.57 16.38 17.46 18.20 18.77
##5 5.702 6.976 7.804 8.421 8.913 9.321 9.669 9.971 10.24 10.48 10.70 10.89 11.08 11.24 11.40 11.55 11.68 11.81 11.93 12.87 13.51 14.39 14.99 15.45
##6 5.243 6.331 7.033 7.556 7.972 8.318 8.612 8.869 9.097 9.300 9.485 9.653 9.808 9.951 10.08 10.21 10.32 10.43 10.54 11.34 11.89 12.65 13.17 13.55
##7 4.949 5.919 6.542 7.005 7.373 7.678 7.939 8.166 8.367 8.548 8.711 8.860 8.997 9.124 9.242 9.353 9.456 9.553 9.645 10.36 10.85 11.52 11.98 12.34
##8 4.745 5.635 6.204 6.625 6.959 7.237 7.474 7.680 7.863 8.027 8.176 8.311 8.436 8.552 8.659 8.760 8.854 8.943 9.027 9.677 10.13 10.74 11.17 11.49
##9 4.596 5.428 5.957 6.347 6.657 6.915 7.134 7.325 7.494 7.646 7.784 7.910 8.025 8.132 8.232 8.325 8.412 8.495 8.573 9.177 9.594 10.17 10.56 10.86
##10 4.482 5.270 5.769 6.136 6.428 6.669 6.875 7.054 7.213 7.356 7.485 7.603 7.712 7.812 7.906 7.993 8.075 8.153 8.226 8.794 9.186 9.726 10.10 10.38
##11 4.392 5.146 5.621 5.970 6.247 6.476 6.671 6.841 6.992 7.127 7.250 7.362 7.464 7.560 7.648 7.731 7.809 7.883 7.952 8.491 8.864 9.377 9.732 10.00
##12 4.320 5.046 5.502 5.836 6.101 6.320 6.507 6.670 6.814 6.943 7.060 7.166 7.265 7.356 7.441 7.520 7.594 7.664 7.730 8.246 8.602 9.093 9.433 9.693
##13 4.260 4.964 5.404 5.726 5.981 6.192 6.372 6.528 6.666 6.791 6.903 7.006 7.100 7.188 7.269 7.345 7.417 7.484 7.548 8.043 8.386 8.859 9.186 9.436
##14 4.210 4.895 5.322 5.634 5.881 6.085 6.258 6.409 6.543 6.663 6.772 6.871 6.962 7.047 7.125 7.199 7.268 7.333 7.394 7.873 8.204 8.661 8.978 9.219
##15 4.167 4.836 5.252 5.556 5.796 5.994 6.162 6.309 6.438 6.555 6.660 6.756 6.845 6.927 7.003 7.074 7.141 7.204 7.264 7.727 8.049 8.492 8.800 9.034
##16 4.131 4.786 5.192 5.489 5.722 5.915 6.079 6.222 6.348 6.461 6.564 6.658 6.744 6.823 6.897 6.967 7.032 7.093 7.151 7.602 7.915 8.346 8.646 8.874
##17 4.099 4.742 5.140 5.430 5.659 5.847 6.007 6.147 6.270 6.380 6.480 6.572 6.656 6.733 6.806 6.873 6.937 6.997 7.053 7.493 7.798 8.219 8.511 8.734
##18 4.071 4.703 5.094 5.379 5.603 5.787 5.944 6.081 6.201 6.309 6.407 6.496 6.579 6.655 6.725 6.791 6.854 6.912 6.967 7.397 7.696 8.107 8.393 8.611
##19 4.046 4.669 5.054 5.334 5.553 5.735 5.889 6.022 6.141 6.246 6.342 6.430 6.510 6.585 6.654 6.719 6.780 6.837 6.891 7.312 7.605 8.008 8.288 8.501
##20 4.024 4.639 5.018 5.293 5.510 5.688 5.839 5.970 6.086 6.190 6.285 6.370 6.449 6.523 6.591 6.654 6.714 6.770 6.823 7.237 7.523 7.919 8.194 8.404
##24 3.955 4.546 4.907 5.168 5.373 5.542 5.685 5.809 5.919 6.017 6.105 6.186 6.261 6.330 6.394 6.453 6.510 6.562 6.612 7.001 7.270 7.641 7.900 8.097
##30 3.889 4.455 4.799 5.048 5.242 5.401 5.536 5.653 5.756 5.848 5.932 6.008 6.078 6.142 6.202 6.258 6.311 6.361 6.407 6.771 7.023 7.370 7.611 7.796
##40 3.825 4.367 4.695 4.931 5.114 5.265 5.392 5.502 5.599 5.685 5.764 5.835 5.900 5.961 6.017 6.069 6.118 6.165 6.208 6.547 6.781 7.104 7.328 7.499
##60 3.762 4.282 4.594 4.818 4.991 5.133 5.253 5.356 5.447 5.528 5.601 5.667 5.728 5.784 5.837 5.886 5.931 5.974 6.015 6.329 6.546 6.843 7.049 7.207
##120 3.702 4.200 4.497 4.708 4.872 5.005 5.118 5.214 5.299 5.375 5.443 5.505 5.561 5.614 5.662 5.708 5.750 5.790 5.827 6.117 6.316 6.588 6.776 6.919
##1e38 3.643 4.120 4.403 4.603 4.757 4.882 4.987 5.078 5.157 5.227 5.290 5.348 5.400 5.448 5.493 5.535 5.574 5.611 5.645 5.911 6.092 6.338 6.507 6.636"""
##
##q0995 = """\
##1 180.1 270.1 328.5 371.2 404.4 431.6 454.4 474.0 491.1 506.3 520.0 532.4 543.6 554.0 563.6 572.5 580.9 588.7 596.0 652.0 689.6 740.2 774.5 800.3
##2 19.92 26.97 31.60 35.02 37.73 39.95 41.83 43.46 44.89 46.16 47.31 48.35 49.30 50.17 50.99 51.74 52.45 53.12 53.74 58.52 61.76 66.13 69.10 71.35
##3 10.54 13.51 15.45 16.91 18.06 19.01 19.83 20.53 21.15 21.70 22.20 22.66 23.08 23.46 23.82 24.15 24.46 24.76 25.03 27.15 28.60 30.55 31.88 32.90
##4 7.916 9.813 11.06 11.99 12.74 13.35 13.88 14.33 14.74 15.10 15.42 15.72 15.99 16.24 16.47 16.70 16.90 17.09 17.28 18.68 19.63 20.93 21.83 22.50
##5 6.751 8.195 9.140 9.846 10.41 10.88 11.28 11.62 11.93 12.21 12.46 12.69 12.90 13.09 13.27 13.44 13.60 13.74 13.89 14.96 15.71 16.72 17.41 17.94
##6 6.105 7.306 8.087 8.670 9.135 9.522 9.852 10.14 10.39 10.62 10.83 11.02 11.19 11.35 11.50 11.64 11.78 11.90 12.02 12.92 13.54 14.40 14.98 15.43
##7 5.698 6.750 7.429 7.935 8.339 8.674 8.961 9.211 9.433 9.632 9.812 9.976 10.13 10.27 10.40 10.52 10.64 10.74 10.84 11.64 12.18 12.93 13.45 13.85
##8 5.420 6.370 6.981 7.435 7.796 8.097 8.354 8.578 8.777 8.955 9.117 9.265 9.401 9.527 9.644 9.754 9.856 9.953 10.04 10.76 11.25 11.92 12.39 12.75
##9 5.218 6.096 6.657 7.073 7.405 7.680 7.915 8.120 8.302 8.466 8.614 8.749 8.874 8.989 9.097 9.197 9.292 9.381 9.465 10.12 10.57 11.19 11.62 11.95
##10 5.065 5.888 6.412 6.800 7.109 7.365 7.584 7.775 7.944 8.096 8.233 8.359 8.475 8.583 8.683 8.777 8.864 8.947 9.025 9.635 10.06 10.64 11.04 11.35
##11 4.945 5.726 6.221 6.587 6.878 7.119 7.325 7.504 7.664 7.807 7.936 8.055 8.164 8.265 8.359 8.447 8.530 8.608 8.681 9.255 9.653 10.20 10.58 10.87
##12 4.849 5.596 6.068 6.416 6.693 6.922 7.117 7.288 7.439 7.574 7.697 7.810 7.913 8.009 8.099 8.182 8.261 8.335 8.405 8.950 9.328 9.850 10.21 10.49
##13 4.769 5.489 5.943 6.276 6.541 6.760 6.947 7.110 7.254 7.384 7.502 7.609 7.708 7.800 7.885 7.965 8.040 8.111 8.178 8.699 9.061 9.560 9.907 10.17
##14 4.703 5.401 5.838 6.160 6.414 6.625 6.805 6.962 7.101 7.225 7.338 7.442 7.537 7.625 7.707 7.784 7.856 7.924 7.988 8.489 8.837 9.317 9.651 9.906
##15 4.647 5.325 5.750 6.061 6.307 6.511 6.685 6.836 6.970 7.091 7.200 7.300 7.391 7.476 7.556 7.630 7.699 7.765 7.827 8.310 8.647 9.111 9.434 9.680
##16 4.599 5.261 5.674 5.976 6.216 6.413 6.582 6.729 6.859 6.975 7.081 7.178 7.267 7.349 7.426 7.498 7.565 7.629 7.689 8.157 8.483 8.933 9.246 9.486
##17 4.557 5.205 5.608 5.903 6.136 6.329 6.493 6.636 6.762 6.876 6.978 7.072 7.159 7.239 7.313 7.383 7.449 7.510 7.569 8.024 8.341 8.779 9.083 9.316
##18 4.521 5.156 5.550 5.839 6.067 6.255 6.415 6.554 6.677 6.788 6.888 6.980 7.064 7.142 7.215 7.283 7.347 7.407 7.464 7.908 8.216 8.643 8.940 9.167
##19 4.488 5.112 5.500 5.782 6.005 6.189 6.346 6.482 6.603 6.711 6.809 6.898 6.981 7.057 7.128 7.194 7.257 7.316 7.371 7.805 8.106 8.523 8.813 9.035
##20 4.460 5.074 5.455 5.732 5.951 6.131 6.285 6.418 6.536 6.642 6.738 6.826 6.906 6.981 7.051 7.116 7.177 7.234 7.289 7.713 8.008 8.416 8.700 8.917
##24 4.371 4.955 5.315 5.577 5.783 5.952 6.096 6.221 6.331 6.430 6.520 6.602 6.677 6.747 6.812 6.872 6.929 6.983 7.034 7.429 7.703 8.083 8.348 8.551
##30 4.285 4.841 5.181 5.428 5.621 5.779 5.914 6.031 6.134 6.226 6.310 6.386 6.456 6.521 6.581 6.638 6.691 6.740 6.787 7.154 7.409 7.760 8.005 8.193
##40 4.202 4.731 5.052 5.284 5.465 5.614 5.739 5.848 5.944 6.030 6.108 6.178 6.243 6.303 6.359 6.411 6.460 6.507 6.550 6.888 7.123 7.447 7.672 7.844
##60 4.122 4.625 4.928 5.146 5.316 5.454 5.571 5.673 5.762 5.841 5.913 5.979 6.039 6.094 6.146 6.194 6.239 6.281 6.321 6.632 6.846 7.142 7.347 7.504
##120 4.044 4.523 4.809 5.013 5.172 5.301 5.410 5.504 5.586 5.660 5.726 5.786  5.842 5.893 5.940 5.984 6.025 6.064 6.101 6.384 6.579 6.846 7.031 7.172
##1e38 3.970 4.424 4.694 4.886 5.033 5.154 5.255 5.341 5.418 5.485 5.546 5.602  5.652 5.699 5.742 5.783 5.820 5.856 5.889 6.146 6.322 6.561 6.725 6.850"""
##
##q0999 = """\
##1 900.3 1351. 1643. 1856. 2022. 2158. 2272. 2370. 2455. 2532. 2600. 2662. 2718. 2770. 2818. 2863. 2904. 2943. 2980. 3260. 3448. 3701. 3873. 4002.
##2 44.69 60.42 70.77 78.43 84.49 89.46 93.67 97.30 100.5 103.3 105.9 108.2 110.4 112.3 114.2 115.9 117.4 118.9 120.3 131.0 138.3 148.0 154.7 159.7
##3 18.28 23.32 26.65 29.13 31.11 32.74 34.12 35.33 36.39 37.34 38.20 38.98 39.69 40.35 40.97 41.54 42.07 42.58 43.05 46.68 49.16 52.51 54.81 56.53
##4 12.18 14.98 16.84 18.23 19.34 20.26 21.04 21.73 22.33 22.87 23.36 23.81 24.21 24.59 24.94 25.27 25.58 25.87 26.14 28.24 29.68 31.65 32.98 34.00
##5 9.714 11.67 12.96 13.93 14.71 15.35 15.91 16.39 16.82 17.18 17.53 17.85 18.13 18.41 18.66 18.89 19.10 19.31 19.51 21.01 22.03 23.45 24.41 25.15
##6 8.427 9.960 10.97 11.72 12.32 12.82 13.25 13.63 13.96 14.26 14.53 14.78 15.00 15.21 15.41 15.59 15.78 15.94 16.09 17.28 18.10 19.22 20.00 20.58
##7 7.648 8.930 9.768 10.40 10.90 11.32 11.67 11.99 12.27 12.52 12.74 12.95 13.14 13.32 13.48 13.64 13.78 13.92 14.05 15.06 15.74 16.69 17.35 17.85
##8 7.129 8.250 8.977 9.522 9.958 10.32 10.63 10.90 11.15 11.36 11.56 11.74 11.91 12.06 12.20 12.34 12.46 12.58 12.69 13.57 14.17 15.01 15.59 16.02
##9 6.761 7.768 8.419 8.906 9.295 9.619 9.896 10.14 10.35 10.55 10.72 10.89 11.03 11.17 11.30 11.42 11.53 11.64 11.74 12.52 13.07 13.82 14.34 14.74
##10 6.487 7.411 8.006 8.449 8.804 9.099 9.352 9.573 9.769 9.946 10.11 10.25 10.39 10.51 10.63 10.74 10.84 10.94 11.03 11.75 12.25 12.94 13.42 13.79
##11 6.275 7.135 7.687 8.098 8.426 8.699 8.933 9.137 9.319 9.482 9.630 9.766 9.891 10.01 10.12 10.22 10.31 10.40 10.49 11.15 11.61 12.25 12.70 13.03
##12 6.106 6.917 7.435 7.820 8.127 8.382 8.601 8.792 8.962 9.115 9.253 9.380 9.497 9.606 9.707 9.802 9.891 9.975 10.05 10.68 11.11 11.71 12.12 12.44
##13 5.969 6.740 7.231 7.595 7.885 8.126 8.332 8.513 8.673 8.817 8.948 9.068 9.178 9.280 9.376 9.465 9.549 9.629 9.704 10.29 10.70 11.27 11.66 11.96
##14 5.855 6.593 7.062 7.409 7.685 7.914 8.110 8.282 8.434 8.571 8.695 8.809 8.914 9.011 9.102 9.187 9.267 9.342 9.414 9.972 10.36 10.90 11.28 11.57
##15 5.760 6.470 6.920 7.252 7.517 7.736 7.924 8.088 8.234 8.364 8.483 8.592 8.692 8.785 8.872 8.953 9.030 9.102 9.170 9.703 10.08 10.59 10.95 11.23
##16 5.678 6.365 6.799 7.119 7.374 7.585 7.765 7.923 8.063 8.189 8.303 8.407 8.504 8.593 8.676 8.754 8.828 8.897 8.962 9.475 9.832 10.33 10.68 10.94
##17 5.608 6.274 6.695 7.004 7.250 7.454 7.629 7.781 7.916 8.037 8.147 8.248 8.341 8.427 8.507 8.583 8.653 8.720 8.783 9.277 9.623 10.10 10.44 10.69
##18 5.546 6.195 6.604 6.905 7.143 7.341 7.510 7.657 7.788 7.905 8.012 8.109 8.199 8.283 8.361 8.433 8.502 8.566 8.627 9.106 9.440 9.904 10.23 10.48
##19 5.492 6.126 6.524 6.817 7.049 7.241 7.405 7.549 7.676 7.790 7.893 7.988 8.075 8.156 8.232 8.302 8.369 8.431 8.491 8.955 9.279 9.729 10.04 10.29
##20 5.444 6.065 6.454 6.740 6.966 7.153 7.313 7.453 7.576 7.687 7.788 7.880 7.965 8.044 8.118 8.186 8.251 8.312 8.370 8.821 9.136 9.575 9.881 10.12
##24 5.297 5.877 6.238 6.502 6.711 6.884 7.031 7.159 7.272 7.374 7.467 7.551 7.629 7.701 7.768 7.831 7.890 7.946 7.999 8.411 8.699 9.100 9.380 9.595
##30 5.156 5.698 6.033 6.277 6.469 6.628 6.763 6.880 6.984 7.077 7.161 7.239 7.310 7.375 7.437 7.494 7.548 7.598 7.646 8.021 8.283 8.646 8.901 9.096
##40 5.022 5.527 5.838 6.063 6.240 6.385 6.509 6.616 6.710 6.795 6.872 6.942 7.007 7.066 7.122 7.174 7.223 7.268 7.312 7.651 7.887 8.214 8.442 8.618
##60 4.893 5.365 5.653 5.860 6.022 6.155 6.268 6.365 6.451 6.528 6.598 6.661 6.720 6.773 6.824 6.870 6.914 6.956 6.995 7.299 7.510 7.802 8.005 8.161
##120 4.771 5.211 5.476 5.667 5.815 5.937 6.039 6.128 6.206 6.275 6.338 6.395 6.448 6.496 6.541 6.583 6.623 6.660 6.695 6.966 7.153 7.410 7.589 7.726
##1e38 4.654 5.063 5.309 5.484 5.619 5.730 5.823 5.903 5.973 6.036 6.092 6.144 6.191 6.234 6.274 6.312 6.347 6.380 6.411 6.651 6.816 7.041 7.196 7.314"""
##
### Build the T+ 'matrix'
### T is a dict of dicts of lists
### Building them as OrderedDicts ensures that we can
### iterate over them in order
##
###                 [alpha keys]        [v keys]
###                   [table values as lists of floats]
##T = OrderedDict([(0.500, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0500.split('\n')])),
##                 (0.675, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0675.split('\n')])),
##                 (0.750, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0750.split('\n')])),
##                 (0.800, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0800.split('\n')])),
##                 (0.850, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0850.split('\n')])),
##                 (0.900, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0900.split('\n')])),
##                 (0.950, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0950.split('\n')])),
##                 (0.975, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0975.split('\n')])),
##                 (0.990, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0990.split('\n')])),
##                 (0.995, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0995.split('\n')])),
##                 (0.999, OrderedDict([(float(L.split()[0]),
##                    map(float, L.split()[1:])) for L in q0999.split('\n')]))])
## 
##### This dict maps r values to the correct list index
####R = OrderedDict(zip([2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,
####                     17,18,19,20,30,40,60,80,100], range(24)))
####
####inf = float('inf')
##### we will need a tinv function
####_tinv = lambda p, df : scipy.stats.t.isf(p, df)
####_phi = lambda p : scipy.stats.norm.isf(p)
####
##### Now we can build the A 'matrix'
####
##### these are for the least squares fitting
####def qhat(a, p, r, v):
####
####    # eq. 2.3
####    p_ = (1. + p) /2.
####    
####    f = a[0]*np.log(r-1.) + \
####        a[1]*np.log(r-1.)**2 + \
####        a[2]*np.log(r-1.)**3 + \
####        a[3]*np.log(r-1.)**4
####
####    # eq. 2.7 and 2.8 corrections
####    for i, r_ in enumerate(r):
####        if r_ == 3:
####            f[i] += -0.002 / (1. + 12. * _phi(p)**2)
####
####            if v <= 4.364:
####                f[i] += 1./517. - 1./(312.*v)
####            else:
####                f[i] += 1./(191.*v)
####
####    return math.sqrt(2) * (f - 1.) * _tinv(p_, v)
####
####errfunc = lambda a, p, r, v, q: qhat(a, p, r, v) - q
####
####A = {} # this is the error matrix
####for p in T:
####    for v in T[p]:
####        #eq. 2.4
####        a0 = random(4)
####        a1, success = leastsq(errfunc, a0,
####                              args=(p, np.array(R.keys()),
####                                    v, np.array(T[p][v])))
####
####        if v == 1e38:
####            A[(p,inf)] = list(a1)
####        else:
####            A[(p,v)] = list(a1)
####
##### repr-ize A
####pp(A, width=160)
