"""
Copyright 2012-2016 Ministerie van Sociale Zaken en Werkgelegenheid

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

import unittest

from qualitylib import metric, domain, metric_source, requirement


class FakeJira(object):
    """ Fake Jira. """

    @staticmethod
    def nr_open_bugs():
        """ Return a fake number of open bugs. """
        return 7

    @staticmethod
    def nr_open_bugs_url():
        """ Return a fake url for the nr of open bugs query. """
        return 'http://openbugs/'

    @staticmethod
    def nr_open_security_bugs():
        """ Return a fake number of open security bugs. """
        return 7

    @staticmethod
    def nr_open_security_bugs_url():
        """ Return a fake url for the nr of open security bugs query. """
        return 'http://opensecuritybugs/'

    @staticmethod
    def nr_technical_debt_issues_url():
        """ Return a fake url for the number of technical debt issues query. """
        return 'http://technicaldebtissues/'

    @staticmethod
    def nr_technical_debt_issues():
        """ Return the number of technical debt issues. """
        return 8


class OpenBugsTest(unittest.TestCase):
    """ Unit tests for the number of open bugs metric. """

    def setUp(self):
        self.__project = domain.Project(metric_sources={metric_source.Jira: FakeJira()},
                                        requirements=[requirement.TrackBugs])
        self.__metric = metric.OpenBugs(project=self.__project)

    def test_value(self):
        """ Test that the value is correct. """
        self.assertEqual(FakeJira.nr_open_bugs(), self.__metric.value())

    def test_url(self):
        """ Test that the url is correct. """
        self.assertEqual({'Jira': FakeJira.nr_open_bugs_url()}, self.__metric.url())

    def test_should_be_measured(self):
        """ Test that the metric should be measured when the project has the appropriate requirement. """
        self.assertTrue(metric.OpenBugs.should_be_measured(self.__project))


class OpenSecurityBugsTest(unittest.TestCase):
    """ Unit tests for the number of open security bugs metric. """

    def setUp(self):
        self.__project = domain.Project(metric_sources={metric_source.Jira: FakeJira()},
                                        requirements=[requirement.TrackBugs])
        self.__metric = metric.OpenSecurityBugs(project=self.__project)

    def test_value(self):
        """ Test that the value is correct. """
        self.assertEqual(FakeJira.nr_open_security_bugs(), self.__metric.value())

    def test_url(self):
        """ Test that the url is correct. """
        self.assertEqual({'Jira': FakeJira.nr_open_security_bugs_url()}, self.__metric.url())

    def test_should_be_measured(self):
        """ Test that the metric should be measured when the project has the appropriate requirement. """
        self.assertTrue(metric.OpenSecurityBugs.should_be_measured(self.__project))


class TechnicalDebtIssuesTest(unittest.TestCase):
    """ Unit tests for the number of technical debt issues metric. """

    def setUp(self):
        self.__project = domain.Project(metric_sources={metric_source.Jira: FakeJira()},
                                        requirements=[requirement.TrackTechnicalDebt])
        self.__metric = metric.TechnicalDebtIssues(project=self.__project)

    def test_value(self):
        """ Test that the value is correct. """
        self.assertEqual(FakeJira.nr_technical_debt_issues(), self.__metric.value())

    def test_url(self):
        """ Test that the url is correct. """
        self.assertEqual({'Jira': FakeJira.nr_technical_debt_issues_url()}, self.__metric.url())

    def test_should_be_measured(self):
        """ Test that the metric should be measured when the project has the appropriate requirement. """
        self.assertTrue(metric.TechnicalDebtIssues.should_be_measured(self.__project))
