from time import time
from collections import defaultdict
from collections import deque
from collections import namedtuple
from os.path import basename
from os.path import splitext
from logging import getLogger

from . import pathmaker
from .archive import ArchiveFile
from .archive import FileNotFoundError
from .exception import BadArchiveFile
from .exception import UnsupportedArchiveFile

logger = getLogger(__name__)


# XXX the i prefix here means archive internal, not for mapper.
FileEntry = namedtuple(
    'FileEntry', ['archive_path', 'ifilename', 'ifile_size'])


class DefaultMapper(object):
    """
    Mapper that tracks the nested structure within archive files.
    """

    def __init__(self, path=None, pathmaker_name='default', _pathmaker=None,
            overwrite=False, include_arcname=False, splitext_arcname=False):
        """
        Initialize the mapping, optionally with a path to an archive
        file.

        Mapping dict keys are names of file or directory, values are
        either a tuple that represent a file, or a dict to represent a
        directory.
        """

        self.include_arcname = include_arcname
        self.splitext_arcname = splitext_arcname
        self.overwrite = overwrite
        if _pathmaker:
            self.pathmaker = _pathmaker
        else:
            self.pathmaker = getattr(pathmaker, pathmaker_name)()

        # The actual filesystem mapping
        self.mapping = {}
        # a mapping with keys of generated paths against source archive.
        # its keys includes a map of directory to its source archive.
        self.reverse_mapping = defaultdict(deque)
        # Tracked file added timestamps (see _load_infolist)
        self.archives = {}
        # A flattened mapping of archive to its list of internal entries
        # including directory entries.
        self.archive_ifilenames = {}

        if path:
            self.load_archive(path)

    def mkdir(self, path_fragments):
        """
        Creates the dir entries identified by path if not already exists
        and return the complete directory.
        """

        # set current to root node
        current = self.mapping

        for c, frag in enumerate(path_fragments):
            if frag in current:
                current = current[frag]
                if not isinstance(current, dict):
                    raise ValueError(
                        'cannot create directory `%(filename)s` at '
                        '`%(path)s/`: file entry exists.' % {
                            'filename': frag,
                            'path': '/'.join(path_fragments[:c]),
                        }
                    )
            else:
                # create directory dict entry and set current.
                current[frag] = current = {}

        return current

    def traverse(self, path):
        """
        Traverse to path, or return the entry identified by path.
        """

        path_fragments = path and path.split('/') or []
        return self._traverse(path_fragments)

    def _traverse(self, path_fragments):
        current = self.mapping

        for frag in path_fragments:
            if not isinstance(current, dict) or frag not in current:
                # No such frag in dir.
                return None
            current = current[frag]

        return current

    def _load_infolist(self, archive_path, infolist):
        self.archives[archive_path] = time()
        archive_name = (
            splitext(basename(archive_path))[0]
            if self.splitext_arcname else
            basename(archive_path)
        ) + '/'
        self.archive_ifilenames[archive_path] = i_filenames = []

        for info in infolist:
            if self.include_arcname:
                raw_filename = archive_name + info.filename
            else:
                raw_filename = info.filename
            frags, filename = self.pathmaker(raw_filename)

            # the internal filename.
            ifilename = '/'.join([] + frags + [filename])

            fentry = FileEntry(archive_path, info.filename, info.file_size)

            # These may appear redundant compared to the ones below, but
            # do note these are always added despite conflict that will
            # appear at the filesystem presentation level.  This is used
            # for locating current zip files for regeneration if an
            # archive was removed from this mapping.
            self.reverse_mapping[ifilename].append(fentry)
            i_filenames.append(ifilename)

            try:
                target = self.mkdir(frags)
            except ValueError as e:
                # using info.filename rather than raw_filename because
                # the message is provided by the exception generated by
                # self.mkdir.
                logger.warning(
                    '`%s` could not be created: %s', info.filename, e.args[0])
                continue

            if not filename:
                # was a directory entry
                continue

            if filename in target:
                if not self.overwrite:
                    logger.info('`%s` already exists; ignoring', info.filename)
                    continue
            target[filename] = fentry

    def _unload_functions(self):
        """
        Return a tuple of needed functions for unloading an archive out
        of the mappings.
        """

        if self.overwrite:
            # check the rightmost (newest) item, pop from right
            return -1, deque.pop
        else:
            # check the leftmost (oldest) item, pop from left
            return 0, deque.popleft

    def _unload_infolist(self, archive_path):
        # pop this out right away to mark this as to be pruned off.
        ifilenames = self.archive_ifilenames.pop(archive_path)
        index, pop = self._unload_functions()
        all_frags = set()
        for ifilename in ifilenames:
            # lookup via the reverse mapping to see that this ifilename
            # is the active check that the current active
            fentries = self.reverse_mapping[ifilename]

            if fentries[index].archive_path != archive_path:
                # leave the reverse mapping in place.
                continue

            original_filename = fentries[index].ifilename

            archive_ifilenames = self.archive_ifilenames
            # We have a match, time to clean up to the latest fresh and
            # active entry that other previous calls to this function
            # have left behind.
            fentry_replacement = None
            try:
                while fentries[index].archive_path not in archive_ifilenames:
                    pop(fentries)
                else:
                    # Successfully left the loop, the fileentry is now
                    # the replacement.
                    fentry_replacement = fentries[index]
            except IndexError:
                # This no longer exists in any active archive.
                self.reverse_mapping.pop(ifilename)

            # We have an ifilename created in _load_infolist, invert
            # operation to derive filename and frags.
            raw = ifilename.split('/')
            filename = raw[-1]
            frags = raw[:-1]

            if frags:
                all_frags.add(tuple(frags))

            info = self._traverse(frags)
            if not info is None:
                # The file's directory may not have been added to
                # self.mapping, if its creation may have been blocked
                # by another file.
                if filename in info:
                    # Only remove, for the same reason as above.
                    if (original_filename.endswith('/') ==
                            isinstance(info[filename], dict)):
                        # Only if the types are the same type should
                        # this entry be removed (file for file, dir for
                        # dir).
                        info.pop(filename)

                    if fentry_replacement:
                        # Restore the fileentry with the replacement
                        # from above.
                        info[filename] = fentry_replacement

                # XXX filename could be empty, does it mean the
                # directory can be removed if empty?

            # if ifilename_paths is not empty, it may be possible to
            # restore that entry with the newer (or previous) "version"
            # from the archive, if it also resolves to a file, but this
            # will need more thought to do, given that files and dirs
            # are two different types.

        # finally, purge all empty directories.  Yes this includes
        # directories that may not be wholly owned by this archive.
        for frags in sorted(all_frags, key=lambda x: len(x)):
            target = self._traverse(frags[:-1])
            if target.get(frags[-1]) == {}:
                target.pop(frags[-1])

        # discard the date associated with this archive path too.
        self.archives.pop(archive_path)

    def load_archive(self, archive_path):
        """
        Load an archive file identified by archive_path into the
        mapping.
        """

        try:
            with ArchiveFile(archive_path) as af:
                self._load_infolist(archive_path, af.infolist())
            logger.info('loaded `%s`', archive_path)
            return True
        except BadArchiveFile:
            logger.warning(
                '`%s` appears to be an invalid archive file', archive_path)
        except UnsupportedArchiveFile as e:
            logger.warning('`%s` %s', archive_path, str(e))
        except FileNotFoundError:
            logger.warning(
                '`%s` does not exist.', archive_path)
        except:
            logger.exception('Exception')
        return False

    def unload_archive(self, archive_path):
        self._unload_infolist(archive_path)
        logger.info('unloaded `%s`', archive_path)

    def open(self, path):
        info = self.traverse(path)
        if info is None:
            return
        archive_path, filename, _ = info
        # it is possible to return those values, but given that the
        # underlying files can change, or that new stack comes in, it's
        # best not to directly expose this.
        try:
            with ArchiveFile(archive_path) as zf:
                # zf does remain open because of this.
                return (id(info), zf.open(filename))
        except BadArchiveFile:  # pragma: no cover
            logger.warning(
                '`%s` became an invalid archive file', archive_path)
        except FileNotFoundError:  # pragma: no cover
            # shoud unload the archive?
            logger.warning(
                '`%s` no longer exists.', archive_path)
        except:  # pragma: no cover
            logger.exception('Exception')
        return False

    def readfile(self, path):
        """
        Return the complete file with information contained in path.
        """

        # This is an alternative implementation that returns the entire
        # contents.  Should include similar exception handling as above,
        # but that pattern should really be generalized.

        info = self.traverse(path)

        archive_filename, filename, _ = info
        with ArchiveFile(archive_filename) as af:
            with af.open(filename) as f:
                return f.read()

    def readdir(self, path):
        """
        Return a listing of all files in a directory
        """

        info = self.traverse(path)
        if not isinstance(info, dict):
            return []
        return list(info.keys())
