#!/usr/bin/python
# -*- coding: utf-8 -*-

# Hive Appier Framework
# Copyright (C) 2008-2012 Hive Solutions Lda.
#
# This file is part of Hive Appier Framework.
#
# Hive Appier Framework is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Hive Appier Framework is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Hive Appier Framework. If not, see <http://www.gnu.org/licenses/>.

__author__ = "João Magalhães joamag@hive.pt>"
""" The author(s) of the module """

__version__ = "1.0.0"
""" The version of the module """

__revision__ = "$LastChangedRevision$"
""" The revision number of the module """

__date__ = "$LastChangedDate$"
""" The last change date of the module """

__copyright__ = "Copyright (c) 2008-2012 Hive Solutions Lda."
""" The copyright for the module """

__license__ = "GNU General Public License (GPL), Version 3"
""" The license for the module """

import time
import appier

import base

ADMIN_TYPE = 1

USER_TYPE = 2

ACCOUNT_S = {
    ADMIN_TYPE : "admin",
    USER_TYPE : "user"
}

class Account(base.Base):

    username = dict(
        index = True
    )

    email = dict(
        index = True,
        immutable = True
    )

    password = dict(
        private = True
    )

    type = dict(
        type = int,
        safe = True
    )

    last_login = dict(
        type = int,
        safe = True
    )

    @classmethod
    def setup(cls):
        super(Account, cls).setup()

        # tries to find the root account (default) in case it's not
        # found returns immediately nothing to be done
        root = cls.find(username = "root")
        if root: return

        # creates the structure to be used as the server description
        # using the values provided as parameters
        account = {
            "enabled" : True,
            "username" : "root",
            "email" : "root@root.com",
            "password" : "root",
            "type" : ADMIN_TYPE
        }
        collection = cls._collection()
        collection.save(account)

    @classmethod
    def login(cls, username, password):
        # verifies that both the provided username and password are valid
        # and that are correctly and properly defined (required for validation)
        if not username or not password:
            raise appier.OperationalError(
                message = "Both username and password must be provided",
                code = 400
            )

        # tries to retrieve the account with the provided username, so that
        # the other validation steps may be done as required by login operation
        account = cls.get(
            username = username,
            build = False,
            raise_e = False
        )
        if not account:
            raise appier.OperationalError(
                message = "No valid account found",
                code = 403
            )

        # verifies that the retrieved account is currently enabled, because
        # disabled accounts are not considered to be valid ones
        if not account.enabled:
            raise appier.OperationalError(
                message = "Account is not enabled",
                code = 403
            )

        # retrieves the value of the password for the stored account and then
        # verifies that the value matched the one that has been provided
        _password = account.password
        if not password == _password:
            raise appier.OperationalError(
                message = "Invalid or mismatch password",
                code = 403
            )

        # updates the last login of the account with the current timestamp
        # and saves the account so that this value is persisted, then returns
        # the account to the caller method so that it may be used
        account.last_login = time.time()
        account.save()
        return account

    def tokens(self):
        if self.type == ADMIN_TYPE:
            return ["*"]

        if self.type == USER_TYPE:
            return ["base", "user"]

    def type_s(self, capitalize = False):
        type_s = ACCOUNT_S.get(self.type, None)
        type_s = type_s.capitalize() if capitalize else type_s
        return type_s
