#!/usr/bin/python
__author__ = "Ronie Martinez"

import re


class NoCase(object):
    """
    NoCase is a class that automatically converts method calls from camelCase to snake_case and vice versa.
    Sample usage:

    class MyClass(NoCase):

        def myMethod(self):
            return "my method"

        def my_other_method(self):
            return "my other method"

    my_class = MyClass()
    print(my_class.my_method())  # prints "my method"
    print(my_class.myOtherMethod())  # prints "my other method"
    """

    _camel = re.compile("^[a-z0-9]+([A-Z][a-z0-9]+)+$")
    _snake = re.compile("^[a-z0-9]+(_[a-z0-9]+)+$")
    _camel_splitter = re.compile("([A-Z][a-z0-9]+)")

    def __getattr__(self, name):
        if NoCase._camel.match(name):
            return getattr(self, self._to_snake_case(name))
        elif NoCase._snake.match(name):
            return getattr(self, self._to_camel_case(name))
        raise AttributeError(name)

    def _to_snake_case(self, string):
        return "_".join([m.lower() for m in NoCase._camel_splitter.split(string) if m])

    def _to_camel_case(self, string):
        words = string.split("_")
        for i in range(1, len(words)):
            words[i] = words[i].title()
        return "".join(words)
