from numpy import atleast_2d
from flopy.mbase import Package

class ModflowGhb(Package):
    '''General head boundary package class
    layer_row_column_head_cond is a list of 2D arrays. Each array has 5 columns containing
    the layer #, row #, column #, head value, and conductance value of a ghb cell.
    Enter a separate array for each stress period. When fewer arrays are supplied than stress periods,
    the ghb cells of the last array are repeated.
    No parameters implemented'''
    def __init__(self, model, ighbcb=0, layer_row_column_head_cond=None, no_print=False, extension='ghb', unitnumber=23):
        Package.__init__(self, model, extension, 'GHB', unitnumber) # Call ancestor's init to set self.parent, extension, name and unit number
        self.heading = '# GHB for MODFLOW, generated by Flopy.'
        self.url = 'ghb.htm'
        self.ighbcb = ighbcb # no cell by cell terms are written
        self.mxactb, self.layer_row_column_head_cond = self.assign_layer_row_column_data(layer_row_column_head_cond, 5)
        self.no_print = no_print
        self.np = 0
        self.parent.add_package(self)
    def __repr__( self ):
        return 'GHB package class'
    def ncells( self):
        # Returns the  maximum number of cells that have a well (developped for MT3DMS SSM package)
        return self.mxactb
    def write_file(self):
        f_ghb = open(self.fn_path, 'w')
        f_ghb.write('%s\n' % self.heading)
        f_ghb.write('%10i%10i' % (self.mxactb, self.ighbcb))
        if self.no_print == True:
            f_ghb.write('  NOPRINT')
        f_ghb.write('\n')
        self.write_layer_row_column_data(f_ghb, self.layer_row_column_head_cond)
        f_ghb.close()
