"""This module generates rmaps from pre-cloned (or complete) references and an rmap stub.

e.g.  python -m crds.jwst.stub jwst_0013.pmap "(('META.INSTRUMENT.DETECTOR','META.INSTRUMENT.GRATING14'),)" *.fits

After using the tool,  inspect the .rmap,  copy the generated or renamed files to a test cache,  and perform
assignment tests.

IMPORTANT:
---------
There are other activities associated with adding a type unrelated to this tool:

1.  Add the type name,  short and long form, to either the project __init__ or to the project locate.py.

2.  Add a .tpn or equivalent for the type.

3.  Update the appropriate test or ops server with the new type names and parameter constraint information.

4.  Sync your local cache with --all to get the latest files,  mainly .imaps and .pmaps

Do these activities first, .tpn development can be omitted but file submissions will be unchecked.   

These activities are required to make types submissible to the server.

USING THIS TOOL:
----------------

This tool has a limited use case in the role of bootstrapping a new type with these properties:

1. It is used for new types when no .rmap currently exists.

2. It is used with a fully populated and expanded set of initial references,  i.e. it can do renaming but not cloning.

3. It derives higher level test contexts from the baseline context.   These contexts are likely to have,  but not
guaranteed to have,  unique official names.

4. It generates viable reference and mapping names using your current cache, server cache, and current directory 
to determine the first unused serial number of each file.  It generates but *does not reserve* these names.

5.  Generated files are all properly interlinked so they can be submitted without renaming to
the CRDS server or used for local testing.  If directly submitted, 4 submissions in dependency order are required:  
references, rmap, imap, and pmap.   This is both clunky and brittle from a naming perspective but barring interference 
from other file submissions,  works.   If interference does occur,  either regenerate or follow the approach below.

6. Incoming references may be renamed CRDS-style by this tool.   The original (generally more descriptive) names of 
references devised by the developer are effectively lost information.

Intervening file submissions to the server by other instruments or types may update the .pmap and .imap and consume the 
serial numbers assigned by this tool.  This may make re-running the tool necessary to avoid duplicate file names.

ALTERNATIVE APPROACH and/or FOLLOW-ON:
-------------------------------------

A more robust alternative which uses the server to assign official names is the following:

1. Run this tool to generate a new .rmap, .imap, .pmap,  and renamed references.   The serial numbers chosen by this
tool should be, at the time of generation,  unique and "official".  The are not however reserved names.

2. Copy the new files generated by this tool to your local cache and do tests.  These files are yours, not "official", 
so it's good to either use a throw-away cache or repair your cache using crds.sync when you're finished.   The sync 
tool can determine what's official and what isn't,  purge junk,  and re-download files as needed to get official 
versions.

3. Edit the new .rmap and delete all references,  rendering it empty.  It's basic form is known to be good,  and 
reference file insertion is known to work at this point.

4. Submit Mappings the now empty .rmap to the web site with renaming turned on,  and context generation turned on.  This
will add the empty type and generate a context including it.   Inspect the official .rmap carefully prior to confirming 
the submission.

5. Batch Submit the initial set of files to the web site with renaming turned on and context generation turned on,  
deriving from the new context.  This will rename the files to CRDS-style names,  but will record the original developer 
names in the file catalog.  It will add the renamed files to the empty rmap and generate a complete rmap,  then add 
that to a second generated context.     Inspect the resulting .rmap carefully prior to confirming the submission.

6.  The end result will be renamed references whose original names are known to the CRDS catalog.  The generated names
will be "official" by definition and guaranteed to be unique since they were assigned by the server.  The first new
context only contains the baseline empty .rmap.   The second context contains the .rmap updated with the new references,
and is the usable end result.   In the event of errors,  nothing precludes deriving from the baseline empty .rmap more
than once.
"""

import sys
import os.path
import glob

from astropy.io import fits as pyfits

from crds import rmap, timestamp, pysh, log, data_file, utils
from crds.client import api

def generate_rmaps_and_context(reference_context, spec, all_references):
    """Generate rmaps for complete (non-stub) references in `all_references` and then
    generate a higher level context derived from `reference_context` by inserting the
    new .rmaps.
    """
    reference_context = api.get_context_by_date(reference_context, "jwst")

    spec = get_stub_spec(spec)

    parkey_map = getattr(spec, "reference_to_dataset", None)

    pmap = rmap.get_cached_mapping(reference_context)
    rmaps = []
    last_added = None
    for instr in pmap.obs_package.INSTRUMENTS:
        # added_references = [ link_reference_to_crds_name(ref) for ref in all_references 
        #                     if instr.lower() in pmap.locate.get_file_properties(ref)[0]]
        added_references = [ ref for ref in all_references 
                             if instr.lower() in pmap.locate.get_file_properties(ref)[0]]
        if added_references:
            last_added = added_references
            path = generate_new_rmap(reference_context, spec.parkey, added_references, parkey_map=parkey_map) 
            path = insert_references(path, added_references)
            rmaps.append(path)
            pysh.sh("cp $path .", trace_commands=True, raise_on_error=True)

    # Save last version of populated .rmap
    final = os.path.basename(path)[:-len(".rmap")] + ".final.rmap"
    pysh.sh("cp $path $final", trace_commands=True, raise_on_error=True)
    
    # Save empty .rmap for submission to server.
    path = generate_new_rmap(reference_context, spec.parkey, last_added, parkey_map=parkey_map)
    empty = os.path.basename(path)[:-len(".rmap")] + ".empty.rmap"
    pysh.sh("cp $path $empty", trace_commands=True, raise_on_error=True)

    rmaps_str = " ".join([os.path.basename(mapping) for mapping in rmaps])
    pysh.sh("python -m crds.newcontext ${reference_context} ${rmaps_str} --verbose", trace_commands=True, raise_on_error=True)
    

RMAP_STUB = """
header = {
    'derived_from' : 'crds.jwst.stub tool',
    'filekind' : 'PHOTOM',
    'instrument' : 'MIRI',
    'mapping' : 'REFERENCE',
    'name' : 'jwst_miri_photom_0001.rmap',
    'observatory' : 'JWST',
    'parkey' : (('META.INSTRUMENT.DETECTOR', 'META.INSTRUMENT.FILTER'),),
    'sha1sum' : 'c36e180a97d16ff6a587500cdaaf85325442c080',
}

selector = Match({
})
"""

def generate_new_rmap(reference_context, parkey, new_references, parkey_map=None):
    """Create an entirely .rmap given a reference context and a list of new files
    of the same type.
    """    
    log.info("context:", reference_context, "parkey:", parkey, "parkey_map:", log.PP(parkey_map), 
             "references:", log.PP(new_references), sep="\n")

    pmap = rmap.get_cached_mapping(reference_context)

    old_instrument, old_filekind = None, None
    for ref in new_references:
        instrument, filekind = pmap.locate.get_file_properties(ref)          # works from filename
        assert not old_instrument or instrument == old_instrument, "Multiple instruments deteted at " + repr(ref)
        assert not old_filekind or filekind == old_filekind, "Multiple filekinds detected at " + repr(ref)
        old_instrument, old_filekind = instrument, filekind
        
        header = data_file.get_header(ref)
        assert header["REFTYPE"].upper() ==  filekind.upper()

    assert instrument in pmap.obs_package.INSTRUMENTS, "Invalid instrument " + repr(instrument)
    assert filekind in pmap.obs_package.FILEKINDS, "Invalid filekind at " + repr(filekind)

    name = "_".join(["jwst",instrument,filekind,"0000"]) + ".rmap"
    path = rmap.locate_mapping(name)
    
    new_rmap = rmap.ReferenceMapping.from_string(RMAP_STUB, name, ignore_checksum=True)
    new_rmap.header["instrument"] = instrument.upper()
    new_rmap.header["filekind"] = filekind.upper()
    new_rmap.header["parkey"] = eval(parkey.upper()) if parkey.strip() else ((),)
    if parkey_map:
        new_rmap.header["reference_to_dataset"] = parkey_map
    new_rmap.header["name"] = name
    new_rmap.header["observatory"] = pmap.observatory.upper()   
    new_rmap.write(path)

    new_rmap = rmap.load_mapping(path)
    new_rmap.validate_mapping()

    new_rmap.header["derived_from"] = 'generated as stub rmap on ' + timestamp.now().split(".")[0]
    new_rmap.write(path)

    new_rmap = rmap.load_mapping(path)
    new_rmap.validate_mapping()
    
    return path

def insert_references(rmap_path, new_references):
    
    files = " ".join(["./" + ref for ref in new_references])

    pysh.sh("python -m crds.refactor insert ${rmap_path} ${rmap_path} ${files} --verbose", 
            trace_commands=True, raise_on_error=True)
    
    return rmap_path

def link_reference_to_crds_name(reference):
    new = new_name(reference)
    log.info("Renaming reference", repr(reference), "to", repr(new))
    pysh.sh("ln -s ${reference} ${new}", raise_on_error=True)
    return new

def new_name(reference):
    """Generate a unique CRDS name for `reference`."""
    observatory = utils.file_to_observatory(reference)
    instrument, filekind = utils.get_file_properties(observatory, reference)
    rootname = "_".join([observatory, instrument, filekind])
    if rmap.is_mapping(reference):
        names = api.list_mappings(observatory, rootname+"*") + \
            rmap.list_mappings(rootname + "*", observatory) + \
            glob.glob(rootname + "*")
    else:
        names = api.list_references(observatory, rootname+"*") + \
            rmap.list_references(rootname + "*", observatory) + \
            glob.glob(rootname + "*")
    old = last_serial(names)
    new = rootname + "_{:04d}".format(old+1) + os.path.splitext(reference)[-1]
    return new

def last_serial(names):
    """Return the highest serial number taken from files in `names`."""
    last = -1
    for name in names:
        try:
            name_no = int(name.split("_")[-1].split(".")[0], 10)
        except:
            continue
        if last < name_no:
            last = name_no
    return last

def get_stub_spec(spec):
    if spec.startswith("@"):
        spec = utils.evalfile(spec[1:])
        spec["parkey"] = str(spec["parkey"])
    else:
        spec = {"parkey": spec }
    return utils.Struct(spec)

if __name__ == "__main__":
        
    pysh.usage("<reference_context> <comma-separated-parkeys> <new_references>", 2)
    
    reference_context = sys.argv[1]
    parkey = sys.argv[2]
    all_references = sys.argv[3:]
    
    generate_rmaps_and_context(reference_context, parkey, all_references)

