import numpy

from datascryer.default_methods.holt_winters import HoltWinters
from datascryer.methods.abc_anomaly import AnomalyMethod


class StddevAnomaly(AnomalyMethod):
    @staticmethod
    def sma(y, n):
        half = int(n / 2)
        result = []
        for i in range(half, len(y) - half):
            result.append(sum(y[i - half:i + half]) / n)
        return result

    @staticmethod
    def chop_for_sma(y, n):
        half = int(n / 2)
        del_index = [i for i in range(half)]
        for i in range(len(y) - half, len(y)):
            del_index.append(i)
        return numpy.delete(y, del_index, axis=0)

    @staticmethod
    def detect_anomaly(y, sma, sigma=2):
        stddev = numpy.std(y)
        errors = []
        e = sigma * stddev
        for i in range(len(y)):
            if abs(y[i] - sma[i]) > e:
                errors.append(y[i])
            else:
                errors.append(None)
        return errors

    @staticmethod
    def filter_errors(errors, hits=5, distance=10):
        real_errors = [None] * len(errors)
        for i in range(len(errors) - distance):
            if distance - errors[i:i + distance].count(None) >= hits:
                for j in range(distance):
                    real_errors[i + j] = errors[i + j]
        return real_errors

    def search_anomaly(self, options, lookback_range, lookback_data):
        raw_data = []
        for d in lookback_data:
            raw_data.append(d[1])

        start = lookback_data[0][0]
        end = lookback_data[len(lookback_data) - 1][0]

        sma = 10
        if 'sma' in options:
            sma = options['sma']

        if 'resolution' in options:
            raw_data = HoltWinters().trim_data(raw_data, options['resolution'])

        m = StddevAnomaly().sma(raw_data, sma)
        y = StddevAnomaly().chop_for_sma(raw_data, sma)

        sigma = 3
        if 'sigma' in options:
            sigma = options['sigma']
        e = StddevAnomaly().detect_anomaly(y, m, sigma)

        anomaly_per_ten = 1
        if 'errors_per_ten' in options:
            anomaly_per_ten = options['anomaly_per_ten']
        r = StddevAnomaly().filter_errors(e, hits=anomaly_per_ten)

        step = (end - start) / len(raw_data)
        x_forecast = [(start + int(sma / 2) * step) + y * step for y in range(len(r))]
        result = []
        for e in list(zip(x_forecast, r)):
            if e[1] is not None:
                result.append(e)
        return result


def less_data():
    series = [[1473847672000, 0.000292], [1473847732000, 0.000391], [1473847792000, 0.000437],
              [1473847852000, 0.000573], [1473847912000, 0.000382], [1473847972000, 0.000511],
              [1473848032000, 0.000509], [1473848092000, 0.00051], [1473848152000, 0.000598], [1473848212000, 0.000383],
              [1473848272000, 0.000348], [1473848332000, 0.000397], [1473848392000, 0.000516],
              [1473848452000, 0.000573], [1473848512000, 0.000746], [1473848572000, 0.000281],
              [1473848632000, 0.000385], [1473848692000, 0.000346], [1473848752000, 0.000568],
              [1473848812000, 0.000392], [1473848872000, 0.000648], [1473848932000, 0.000279],
              [1473848992000, 0.000417], [1473849052000, 0.000671], [1473849112000, 0.000654],
              [1473849172000, 0.000555], [1473849232000, 0.000367], [1473849292000, 0.000453],
              [1473849352000, 0.000364], [1473849412000, 0.000525], [1473849472000, 0.000535],
              [1473849532000, 0.000506], [1473849592000, 0.000296], [1473849652000, 0.000477],
              [1473849712000, 0.000523], [1473849772000, 0.00059], [1473849832000, 0.000769], [1473849892000, 0.000284],
              [1473849952000, 0.000719], [1473850012000, 0.000458], [1473850072000, 0.000394],
              [1473850132000, 0.000437], [1473850192000, 0.000801], [1473850252000, 0.000388],
              [1473850312000, 0.000459], [1473850372000, 0.000436], [1473850432000, 0.000673],
              [1473850492000, 0.000526], [1473850552000, 0.000813], [1473850612000, 0.000666],
              [1473850672000, 0.000338], [1473850732000, 0.000308], [1473850792000, 0.000512],
              [1473850852000, 0.000507], [1473850912000, 0.000527], [1473850972000, 0.000529], [1473851032000, 0.00053],
              [1473851092000, 0.000448], [1473851152000, 0.000381]]
    import matplotlib.pyplot as plt
    result = StddevAnomaly().search_anomaly({}, len(series), series)
    plt.plot(*zip(*series))
    plt.plot(*zip(*result), 'x')
    plt.show()


def more_data():
    series = [[1473845992000, 0.000536], [1473846052000, 0.000328], [1473846112000, 0.000527],
              [1473846172000, 0.000255], [1473846232000, 0.000471], [1473846292000, 0.000715],
              [1473846352000, 0.003529], [1473846412000, 0.000355], [1473846472000, 0.000454],
              [1473846532000, 0.000554], [1473846592000, 0.000445], [1473846652000, 0.000433],
              [1473846712000, 0.000389], [1473846772000, 0.000588], [1473846832000, 0.000361],
              [1473846892000, 0.000516], [1473846952000, 0.000381], [1473847012000, 0.000652],
              [1473847072000, 0.000534], [1473847132000, 0.000366], [1473847192000, 0.000639],
              [1473847252000, 0.000611], [1473847312000, 0.00053], [1473847372000, 0.000455], [1473847432000, 0.000377],
              [1473847492000, 0.000787], [1473847552000, 0.000348], [1473847612000, 0.000505],
              [1473847672000, 0.000292], [1473847732000, 0.000391], [1473847792000, 0.000437],
              [1473847852000, 0.000573], [1473847912000, 0.000382], [1473847972000, 0.000511],
              [1473848032000, 0.000509], [1473848092000, 0.00051], [1473848152000, 0.000598], [1473848212000, 0.000383],
              [1473848272000, 0.000348], [1473848332000, 0.000397], [1473848392000, 0.000516],
              [1473848452000, 0.000573], [1473848512000, 0.000746], [1473848572000, 0.000281],
              [1473848632000, 0.000385], [1473848692000, 0.000346], [1473848752000, 0.000568],
              [1473848812000, 0.000392], [1473848872000, 0.000648], [1473848932000, 0.000279],
              [1473848992000, 0.000417], [1473849052000, 0.000671], [1473849112000, 0.000654],
              [1473849172000, 0.000555], [1473849232000, 0.000367], [1473849292000, 0.000453],
              [1473849352000, 0.000364], [1473849412000, 0.000525], [1473849472000, 0.000535],
              [1473849532000, 0.000506], [1473849592000, 0.000296], [1473849652000, 0.000477],
              [1473849712000, 0.000523], [1473849772000, 0.00059], [1473849832000, 0.000769], [1473849892000, 0.000284],
              [1473849952000, 0.000719], [1473850012000, 0.000458], [1473850072000, 0.000394],
              [1473850132000, 0.000437], [1473850192000, 0.000801], [1473850252000, 0.000388],
              [1473850312000, 0.000459], [1473850372000, 0.000436], [1473850432000, 0.000673],
              [1473850492000, 0.000526], [1473850552000, 0.000813], [1473850612000, 0.000666],
              [1473850672000, 0.000338], [1473850732000, 0.000308], [1473850792000, 0.000512],
              [1473850852000, 0.000507], [1473850912000, 0.000527], [1473850972000, 0.000529], [1473851032000, 0.00053],
              [1473851092000, 0.000448], [1473851152000, 0.000381], [1473851212000, 0.00079], [1473851272000, 0.000684],
              [1473851332000, 0.000527], [1473851392000, 0.001077], [1473851452000, 0.000918],
              [1473851512000, 0.000506], [1473851572000, 0.000794], [1473851632000, 0.000356],
              [1473851692000, 0.000446], [1473851752000, 0.000836], [1473851812000, 0.000648],
              [1473851872000, 0.000617], [1473851932000, 0.000565], [1473851992000, 0.00044], [1473852052000, 0.000444],
              [1473852112000, 0.000989], [1473852172000, 0.00043], [1473852232000, 0.000624], [1473852292000, 0.000548],
              [1473852352000, 0.000509], [1473852412000, 0.000895], [1473852472000, 0.000628],
              [1473852532000, 0.000505], [1473852592000, 0.00082], [1473852652000, 0.00063], [1473852712000, 0.000549],
              [1473852772000, 0.000456], [1473852832000, 0.00036], [1473852892000, 0.00038], [1473852952000, 0.000448],
              [1473853012000, 0.000516], [1473853072000, 0.000367]]
    result = StddevAnomaly().search_anomaly({}, len(series), series)
    import matplotlib.pyplot as plt
    plt.plot(*zip(*series))
    plt.plot(*zip(*result), 'x')
    plt.show()


def ssh():
    from random import randint, seed
    import pandas as pd
    import matplotlib.pyplot as plt

    seed(1)
    df = pd.DataFrame(pd.read_csv('ssh.csv', sep=';'))[:20000]
    y = df.value.as_matrix()
    y_raw = numpy.flipud(y)
    y = numpy.append(y_raw, y_raw)
    y = numpy.append(y, y_raw)
    for i in range(len(y)):
        y[i] += randint(-10, 10)
    for i in range(46100, 46120):
        y[i] += 100
        y[i] *= 10

    x = [i for i in range(0, len(y) * 2, 2)]
    series = list(zip(x, y))
    result = StddevAnomaly().search_anomaly({}, len(series), series)
    print(result)
    plt.plot(*zip(*series))
    plt.plot(*zip(*result), 'x')
    plt.show()


if __name__ == '__main__':
    more_data()
