"""daytrades: Daytrades plugin for the trade module.

This plugin provides the Daytrade class, a subclass of Operation, and
the fetch_daytrades() task for the OperationContainer.

With this plugin the trade module can:
- Identify daytrades in a group of operations
- Separate daytrades from other operations on the OperationContainer

It provides:
- Daytrade, a subclass of operation.Operation
- the fetch_daytrades() task to the OperationContainer

Daytrades can be accumulated just like any other Operation object.
They will update the accumulator results, but will not change the
quantity or the price of the asset on the Portfolio.

-----------------------------------------------------------------------

Copyright (c) 2015 Rafael da Silva Rocha

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

from __future__ import absolute_import

from ..operation import Operation
from ..utils import average_price, same_sign


class Daytrade(Operation):
    """A daytrade operation.

    Daytrades are operations of purchase and sale of an Asset on
    the same date.

    Attributes:
        asset: An asset instance, the asset that is being traded.
        quantity: The traded quantity of the asset.
        purchase: A Operation object representing the purchase of the
            asset.
        sale: A Operation object representing the sale of the asset.
        update_position: Set to False, as daytrades don't change the
            portfolio position; they just create results.
    """

    update_position = False

    def __init__(self, date, asset, quantity, purchase_price, sale_price):
        """Creates the daytrade object.

        Base on the informed values this method creates 2 operations:
        - a purchase operation
        - a sale operation
        and them appends them to the Daytrade object operations list.

        Both operations can be treated like any other operation when it
        comes to taxes and the prorate of commissions.
        """
        self.date = date
        self.asset = asset
        self.quantity = quantity

        # Purchase is 0, Sale is 1
        self.operations = [
            Operation(
                date=date,
                asset=asset,
                quantity=quantity,
                price=purchase_price
            ),
            Operation(
                date=date,
                asset=asset,
                quantity=quantity*-1,
                price=sale_price
            )
        ]

        # FIXME this name and result property
        self.results = {'daytrades':self.result}

    @property
    def result(self):
        """Returns the profit or the loss generated by the daytrade."""
        return abs(self.operations[1].real_value) - \
                                        abs(self.operations[0].real_value)


def fetch_daytrades(container):
    """Fetch the daytrades from the OperationContainer operations.

    The daytrades are placed on the container positions under the
    'daytrades' key, inexed by the Daytrade asset's symbol.
    """
    for i, operation_a in enumerate(container.operations):
        for operation_b in \
                [
                    op for op in container.operations[i:] if daytrade_condition(
                                                        op, operation_a
                                                    )
                ]:
            if operation_b.quantity != 0 and operation_a.quantity != 0:
                extract_daytrade(container, operation_a, operation_b)


def daytrade_condition(operation_a, operation_b):
    """Checks if the operations are day trades."""
    return (
        operation_a.asset.symbol == operation_b.asset.symbol and
        not same_sign(operation_a.quantity, operation_b.quantity) and
        operation_a.quantity != 0 and
        operation_b.quantity != 0
    )


def extract_daytrade(container, operation_a, operation_b):
    """Extracts the daytrade part of two operations."""

    # Find what is the purchase and what is the sale
    purchase, sale = find_purchase_and_sale(operation_a, operation_b)

    # Find the daytraded quantity; the daytraded
    # quantity is always the smallest absolute quantity
    daytrade_quantity = min([abs(purchase.quantity), abs(sale.quantity)])

    # Update the operations that originated the
    # daytrade with the new quantity after the
    # daytraded part has been extracted; One of
    # the operations will always have zero
    # quantity after this, being fully consumed
    # by the daytrade. The other operation may or
    # may not end with zero quantity.
    purchase.quantity -= daytrade_quantity
    sale.quantity += daytrade_quantity

    # Now that we know everything we need to know
    # about the daytrade, we create the Daytrade object
    daytrade = Daytrade(
        container.date,
        purchase.asset,
        daytrade_quantity,
        purchase.price,
        sale.price
    )

    # If this container already have a Daytrade
    # with this asset, we merge this daytrade
    # with the daytrade in self.daytrades -
    # in the end, there is only one daytrade per
    # asset per OperationContainer.

    if 'daytrades' not in container.positions:
        container.positions['daytrades'] = {}
    if daytrade.asset.symbol in container.positions['daytrades']:
        container.merge_operations(
            container.positions['daytrades'][daytrade.asset.symbol].\
                operations[0],
            daytrade.operations[0]
        )
        container.merge_operations(
            container.positions['daytrades'][daytrade.asset.symbol].\
                operations[1],
            daytrade.operations[1]
        )
        container.positions['daytrades'][daytrade.asset.symbol].quantity += \
                                                            daytrade.quantity
    else:
        container.positions['daytrades'][daytrade.asset.symbol] = daytrade


def find_purchase_and_sale(operation_a, operation_b):
    """Find which is a purchase and which is a sale."""
    if same_sign(operation_a.quantity, operation_b.quantity):
        return None
    if operation_a.quantity > operation_b.quantity:
        return operation_a, operation_b
    return operation_b, operation_a
