# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
A thread-based pool of workers.

>>> from time import sleep
>>> from functools import partial
>>> def callback(secs, success, result):
...     print 'Done sleeping for %s seconds' % secs
... 
>>> tp = ThreadPool(2)
>>> for secs in xrange(3):
...     tp.apply_async(sleep, (secs,), {}, partial(callback, secs))
... 
>>> tp.is_busy
True
>>> tp.flush()
Done sleeping for 0 seconds
Done sleeping for 1 seconds
Done sleeping for 2 seconds
>>> tp.is_busy
False
>>> tp.apply(sleep, (2,))
>>> for secs in xrange(4):
...     tp.apply_async(sleep, (secs,), {}, partial(callback, secs))
... 
>>> tp.flush(timeout = 2.5)
Done sleeping for 0 seconds
Done sleeping for 1 seconds
Done sleeping for 2 seconds
>>> tp.join()
Done sleeping for 3 seconds
>>> tp = ThreadPool(2)
>>> for secs in xrange(3):
...     tp.apply_async(sleep, (secs,), {}, partial(callback, secs))
... 
>>> count = 0
>>> def stop_condition():
...     global count
...     if count >= 1: return True
...     count = count + 1
...     return False
>>> tp.flush(stop_condition = stop_condition)
Done sleeping for 0 seconds
Done sleeping for 1 seconds
>>> tp.join()
Done sleeping for 2 seconds
>>> count = 0
>>> tp = ThreadPool(2)
>>> for secs in xrange(3):
...     tp.apply_async(sleep, (secs,), {}, partial(callback, secs))
... 
>>> count = 0
>>> tp.flush(timeout = 3, stop_condition = stop_condition)
Done sleeping for 0 seconds
Done sleeping for 1 seconds
>>> tp.join()
Done sleeping for 2 seconds
>>> tp = ThreadPool(1)
>>> def sleepy_callback(success, result):
...     sleep(1)    
...     print 'Done'
>>> tp.apply_async(sleep, (1,), {}, sleepy_callback)
>>> tp.apply_async(sleep, (2,), {}, sleepy_callback)
>>> tp.flush(timeout = 1.5)
Done
>>> tp.join()
Done
>>> with ThreadPool(5) as tp:
...     for secs in xrange(3):
...         tp.apply_async(sleep, (secs,), {}, partial(callback, secs))
...
Done sleeping for 0 seconds
Done sleeping for 1 seconds
Done sleeping for 2 seconds

Note: The maximam number of workers should be at least the number of cores in your system.

"""
from .safeapply import safe_apply, unsafe

import Queue
from threading import Thread
import functools
from time import time as _time

__all__ = ['ThreadPool','Empty','ThreadPoolException']

class ThreadPoolException(Exception): pass

class Empty(ThreadPoolException): pass

class Job(object):
    __slots__ = ['func', 'args', 'kws', 'callback']

    def __init__(self, func, args, kws, callback):
        self.func = func
        self.args = args
        self.kws = kws
        self.callback = callback

    # this is called from the worker thread
    def run(self):
        success, result = safe_apply(self.func, *self.args, **self.kws)
        return self, success, result

class ThreadPool(object):
    __slots__ = ['_workers',
                 '_input_queue',
                 '_output_queue',
                 '_jobs_at_workers',
                 ]

    def __init__(self, workers = 1):
        self._input_queue = Queue.Queue()
        self._output_queue = Queue.Queue()
        self._workers = [Thread(target = self._worker) for i in range(workers)]
        self._jobs_at_workers = 0
        for w in self._workers:
            w.start()

    def _worker(self):
        while True:
            job = self._input_queue.get()
            try:
                if job is None:
                    break
                self._output_queue.put(job.run())
            finally:
                self._input_queue.task_done()

    def apply_async(self, func, args = (), kws = {}, callback = None):
        """Adds a job to the pool. The callback will be called with the result.
        callback shall be defined as def callback(success, result) where:
        - success is True and result is the return value of func(*args, **kws) if no exception was raised; or
        - success is False and result is sys.exc_info() of the exception moment.
        """
        self._input_queue.put(Job(func, args, kws, callback))
        self._jobs_at_workers += 1

    def _handle_single_job(self, block = True, timeout = None):
        try:
            job, success, result = self._output_queue.get(block, timeout)
        except Queue.Empty:
            raise Empty
        self._output_queue.task_done()
        self._jobs_at_workers -= 1
        job.callback(success, result)

    def flush(self, block = True, timeout = None, stop_condition = None):
        """
        Flushes jobs in the thread pool
        - if block is False, flush will only handle already ready jobs;
        - Unspecified timeout (is None) means unlimited time;
        - stop_condition, if provided, must be a callable instance without parameters
          that shall return True if flush is to be stopped in the middle.
        """
        if timeout is None or not block:
            while self._jobs_at_workers:
                self._handle_single_job(block)
                if stop_condition is not None and stop_condition():
                    break
        else:
            assert timeout >= 0.0
            endtime = _time() + timeout
            while self._jobs_at_workers:
                remaining = endtime - _time()
                if remaining <= 0.0:
                    break
                try:
                    self._handle_single_job(True, remaining)
                except Empty:
                    break
                if stop_condition is not None and stop_condition():
                    break

    @property
    def is_busy(self):
        """Returns whether the thread pool is busy."""
        return self._jobs_at_workers > 0

    def apply(self, func, args = (), kws = {}, timeout = None):
        """Synchronous applying, as if you called func(*args, **kws) yourself."""
        sr = []
        def callback(success, result):
            sr.extend((success, result))
        self.apply_async(func, args, kws, callback)

        self.flush(timeout = timeout, stop_condition = lambda : sr)

        assert sr, "callback was not called"

        return unsafe(*sr)

    def join(self, block = True, timeout = None):
        """
        Ends all the work of the ThreadPool.
        flush() is called with block and timeout parameters.
        """
        self.flush(block, timeout)
        for w in self._workers:
            self._input_queue.put(None)
        for w in self._workers:
            w.join()
        self._input_queue.join()
        self._output_queue.join()

    # context-manager:
    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, exc_traceback):
        self.join()
        return False
