# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.

from weakref import ref as wr

from bfc.views.wr import WrView
from bfc.orderedset import OrderedSet

__all__ = ['Graph', 'Vertex', 'DigraphException']

class DigraphException(Exception): pass

class EdgesView(object):
    __slots__ = ['__graph']
    def __init__(self, graph):
        self.__graph = graph

    def __iter__(self):
        for v in self.__graph._vertices:
            for w in v.next_vertices:
                yield (v,w)

    def __contains__(self, edge):
        v, w = edge
        return v in self.__graph._vertices and wr(w) in v._next_weakrefs

class Graph(object):
    """Represents a Directed Graph as a set of Vertices.

    To add a vertex, simply create a Vertex, and then add it to the Graph
    To remove a vertex, use Graph.remove(vertex) - this will make the vertex an orphan and not belong to any graph.
    While a vertex lives in a Graph, it cannot be __del__'ed.
    It will be __del__'ed only after:
    - your references are removed
    - Graph.remove(vertex), Graph.clear or Graph.__del__ are called.
    """
    __slots__ = ['_vertices', # all vertices (not weakrefs)
                 '_source_weakrefs', # weakrefs to source vertices
                 '_pit_weakrefs', # weakrefs to pit vertices
                 '__weakref__',
                 ]

    def __init__(self):
        object.__init__(self)
        self._source_weakrefs = OrderedSet()
        self._pit_weakrefs = OrderedSet()
        self._vertices = OrderedSet()

    def add(self, vertex):
        """Adds a vertex to the Graph. Vertices cannot be connected before being added to the same graph.

        >>> g = Graph()
        >>> s, t = Vertex(), Vertex()
        >>> g.add(s)
        >>> g.add(t)
        >>> Vertex.connect(s, t)

        """

        assert vertex._graph_weakref is None, 'The Vertex already belongs to some Graph'
        assert vertex not in self._vertices, 'The Vertex already belongs to this Graph'

        vertex._on_add_to_graph(self)
        self._vertices.add(vertex)

    def remove(self, vertex):
        """
        Removes a vertex from a graph. A removed vertex becomes orphan.
        
        >>> g = Graph()
        >>> s, t, w = Vertex(), Vertex(), Vertex()
        >>> for v in [s,t,w]: g.add(v)
        >>> Vertex.connect(s,t)
        >>> Vertex.connect(t,w)
        >>> t.is_orphan
        False
        >>> g.remove(t)
        >>> t.is_orphan
        True

        """

        assert vertex._graph_weakref() is self, 'The Vertex does not belong to this Graph'
        assert vertex in self._vertices, 'The Vertex does not belong to this Graph'
        
        vertex._on_remove_from_graph()
        self._vertices.remove(vertex)

    def pop(self, vertex):
        """
        Pops a vertex from a graph. A popped vertex becomes orphan.

        >>> g = Graph()
        >>> s, t, w = Vertex(), Vertex(), Vertex()
        >>> for v in [s,t,w]: g.add(v)
        >>> Vertex.connect(s,t)
        >>> Vertex.connect(t,w)
        >>> t.is_orphan
        False
        >>> t is g.pop(t)
        True
        >>> t.is_orphan
        True
        
        """
        self.remove(vertex)
        return vertex

    def clear(self):
        """
        Removes all vertices from the graph. They become orphan.
        
        >>> g = Graph()
        >>> s, t = Vertex(), Vertex()
        >>> for v in [s,t]: g.add(v)
        >>> s.graph is g
        True
        >>> t.graph is g
        True
        >>> Vertex.connect(s,t)
        >>> s.graph is g
        True
        >>> t.graph is g
        True
        >>> t.is_orphan
        False
        >>> g.clear()
        >>> t.is_orphan
        True

        """
        for v in self._vertices:
            v._on_remove_from_graph()

        self._vertices.clear()

    def __del__(self):
        for v in self._vertices:
            v._on_del_graph(self)

        self._vertices.clear()
        self._vertices = None

    @property
    def sources(self):
        """Returns a view of sources.

        Vertices can neighter be added nor become sources and source vertices
        cannot connect to other vertices while the view is iterating.

        >>> g = Graph()
        >>> s = Vertex()
        >>> g.add(s)
        >>> t = Vertex()
        >>> g.add(t)
        >>> t in g.sources
        True
        >>> Vertex.connect(s, t)
        >>> t in g.sources
        False

        """

        return WrView(self._source_weakrefs)

    @property
    def pits(self):
        """Returns a view of pits.

        Vertices can neighter be added nor become pits and pit vertices cannot
        connect from other vertices while the view is iterating."""
        return WrView(self._pit_weakrefs)

    @property
    def edges(self):
        """Returns a view of edges.

        Graph cannot be changed while the view is iterating.

        >>> g = Graph()
        >>> s = Vertex()
        >>> g.add(s)
        >>> t = Vertex()
        >>> g.add(t)
        >>> Vertex.connect(s, t)
        >>> list(g.edges) == [(s,t)]
        True
        >>> (s,t) in g.edges
        True
        >>> (t, s) in g.edges
        False
        
        """
        return EdgesView(self)

    __hash__ = None

class Vertex(object):
    """A vertex in a digraph.

    A Vertex can either belong to a graph, or not belong to one.
    When not belonging to a graph, the Vertex is always orphan and cannot be connected to other Vertices.
    When belonging to a graph, the Vertex can be connected to and disconnected from other Vertices of the same Graph only.
    """
    __slots__ = ['_next_weakrefs', # weakrefs to next vertices
                 '_prev_weakrefs', # weakrefs to prev vertices
                 '_graph_weakref', # weakref to containing Graph
                 '__weakref__',
                 ]

    def __init__(self):
        object.__init__(self)
        self._next_weakrefs = OrderedSet()
        self._prev_weakrefs = OrderedSet()
        self._graph_weakref = None

    @property
    def graph(self):
        """
        Returns the owner graph.
        Can only be used after adding to a graph.

        >>> g = Graph()
        >>> v = Vertex()
        >>> g.add(v)
        >>> v.graph is g
        True
        
        """
        assert self._graph_weakref is not None, 'Vertex does not belong to any graph'
        g = self._graph_weakref()
        assert g is not None
        return g

    def become_pit(self):
        """
        Removes all edges from the vertex.
        Can only be used after adding to a graph.

        >>> g = Graph()
        >>> v = Vertex()
        >>> w = Vertex()
        >>> g.add(v)
        >>> g.add(w)
        >>> Vertex.connect(v,w)
        >>> len(v.next_vertices)
        1
        >>> v.is_pit
        False
        >>> v.become_pit()
        >>> v.is_pit
        True
        >>> len(v.next_vertices)
        0
        
        """
        assert self._graph_weakref is not None, 'The vertex must be added to a graph first'

        next = self._next_weakrefs
        if not next: # already pit
            return

        graph = self._graph_weakref()
        assert graph is not None
        my_wr = wr(self)

        for dest in next:
            d = dest()
            d._prev_weakrefs.remove(my_wr)
            if not d._prev_weakrefs: # dest becomes a source
                graph._source_weakrefs.add(dest)

        next.clear()

        graph._pit_weakrefs.add(my_wr)

    @property
    def is_pit(self):
        """
        Returns whether the vertex is a pit or not.
        When not belonging to a Graph, it is always a pit.
        
        >>> g = Graph()
        >>> v = Vertex()
        >>> v.is_pit
        True
        >>> w = Vertex()
        >>> g.add(v)
        >>> v.is_pit
        True
        >>> g.add(w)
        >>> Vertex.connect(v,w)
        >>> v.is_pit
        False
        >>> Vertex.disconnect(v,w)
        >>> v.is_pit
        True
        
        """
        return not self._next_weakrefs

    def become_source(self):
        """
        Removes all edges to the vertex.
        Can only be used after adding to a graph.

        >>> g = Graph()
        >>> v = Vertex()
        >>> w = Vertex()
        >>> g.add(v)
        >>> g.add(w)
        >>> Vertex.connect(v,w)
        >>> len(w.prev_vertices)
        1
        >>> w.is_source
        False
        >>> w.become_source()
        >>> w.is_source
        True
        >>> len(w.prev_vertices)
        0
                                                                                                                
        """
        assert self._graph_weakref is not None, 'The vertex must be added to a graph first'

        prev = self._prev_weakrefs
        if not prev: # already source
            return
        my_wr = wr(self)
        graph = self._graph_weakref()

        for src in prev:
            s = src()
            s._next_weakrefs.remove(my_wr)
            if not s._next_weakrefs: # source becomes a pit
                graph._pit_weakrefs.add(s)

        prev.clear()

        graph._source_weakrefs.add(my_wr)

    @property
    def is_source(self):
        """
        Returns whether the vertex is a source in a graph.
        If it does not belong to a graph, it is a source.

        >>> g = Graph()
        >>> v = Vertex()
        >>> w = Vertex()
        >>> w.is_source
        True
        >>> g.add(v)
        >>> g.add(w)
        >>> w.is_source
        True
        >>> Vertex.connect(v,w)
        >>> w.is_source
        False
        >>> w.become_source()
        >>> w.is_source
        True
                                                                                                        
        """
        return not self._prev_weakrefs

    def become_orphan(self):
        """
        Disconnects all edges from the vertex.

        >>> g = Graph()
        >>> s, t, w = Vertex(), Vertex(), Vertex()
        >>> for v in [s,t,w]: g.add(v)
        >>> Vertex.connect(s,t)
        >>> Vertex.connect(t,w)
        >>> t.is_orphan
        False
        >>> len(t.next_vertices), len(t.prev_vertices)
        (1, 1)
        >>> t.become_orphan()
        >>> t.is_orphan
        True
        >>> len(t.next_vertices), len(t.next_vertices)
        (0, 0)
        
        """
        self.become_pit()
        self.become_source()

    @property
    def is_orphan(self):
        """
        Returns whether a Vertex is orphan.
        If it does not belong to a graph, it is orphan.

        >>> g = Graph()
        >>> s, t, w = Vertex(), Vertex(), Vertex()
        >>> t.is_orphan
        True
        >>> for v in [s,t,w]: g.add(v)
        >>> t.is_orphan
        True
        >>> Vertex.connect(s,t)
        >>> t.is_orphan
        False
        >>> Vertex.connect(t,w)
        >>> t.is_orphan
        False
        >>> t.become_orphan()
        >>> t.is_orphan
        True
        
        """
        return self.is_pit and self.is_source

    @property
    def next_vertices(self):
        """
        Returns a generator of vertices connected from the current one.

        >>> g = Graph()
        >>> s, t, w = Vertex(), Vertex(), Vertex()
        >>> set(s.next_vertices)
        set([])
        >>> for v in [s,t,w]: g.add(v)
        >>> Vertex.connect(s, t)
        >>> set(s.next_vertices) == set([t])
        True
        >>> Vertex.connect(s, w)
        >>> set(s.next_vertices) == set([t, w])
        True
        >>> Vertex.connect(t, s)
        >>> set(s.next_vertices) == set([t, w])
        True
        
        """
        return WrView(self._next_weakrefs)

    @property
    def prev_vertices(self):
        """
        Returns a generator of vertices connected to the current one.

        >>> g = Graph()
        >>> s, t, w = Vertex(), Vertex(), Vertex()
        >>> set(s.prev_vertices)
        set([])
        >>> for v in [s,t,w]: g.add(v)
        >>> Vertex.connect(t, s)
        >>> set(s.prev_vertices) == set([t])
        True
        >>> Vertex.connect(w, s)
        >>> set(s.prev_vertices) == set([t, w])
        True
        >>> Vertex.connect(s, t)
        >>> set(s.prev_vertices) == set([t, w])
        True
        
        """
        return WrView(self._prev_weakrefs)

    @classmethod
    def connect(cls, source, dest):
        """
        Creates an edge from a vertex to another one.
        Both vertices must be instances of the same Vertex-derived class, and belong to the same graph.

        >>> g = Graph()
        >>> s = Vertex()
        >>> t = Vertex()
        >>> g.add(s)
        >>> g.add(t)
        >>> Vertex.connect(s, t)
        >>> Vertex.connect(s, t)
        Traceback (most recent call last):
            ...
        DigraphException: Vertices already connected
        
        """

        assert isinstance(source, cls), '%r is not a %s' % (source, cls.__name__)
        assert isinstance(dest, cls), '%r is not a %s' % (dest, cls.__name__)
        assert source._graph_weakref is not None, 'Source vertex must belong to a graph'
        assert source._graph_weakref is dest._graph_weakref, 'Vertices belong to different graphs'

        source_wr = wr(source)
        dest_wr = wr(dest)

        if dest_wr in source._next_weakrefs:
            raise DigraphException('Vertices already connected')

        assert source_wr not in dest._prev_weakrefs, 'Vertices already connected, and found an inconsistency'

        source._next_weakrefs.add(dest_wr)
        dest._prev_weakrefs.add(source_wr)

        if 1 == len(source._next_weakrefs): # source was a pit
            source._graph_weakref()._pit_weakrefs.remove(source_wr)

        if 1 == len(dest._prev_weakrefs): # dest was a source
            dest._graph_weakref()._source_weakrefs.remove(dest_wr)

    @classmethod
    def disconnect(cls, source, dest):
        """
        Removes the edge from a vertex to another one.
        Both vertices must be instances of the same Vertex-derived class, and belong to the same graph.

        >>> g = Graph()
        >>> s = Vertex()
        >>> t = Vertex()
        >>> g.add(s)
        >>> g.add(t)
        >>> Vertex.connect(s, t)
        >>> Vertex.disconnect(s, t)
        >>> Vertex.disconnect(s, t)
        Traceback (most recent call last):
            ...
        DigraphException: Vertices are not connected

        """

        assert isinstance(source, cls), '%r is not a %s' % (source, cls.__name__)
        assert isinstance(dest, cls), '%r is not a %s' % (dest, cls.__name__)
        assert source._graph_weakref is not None, 'Source vertex must belong to a graph'
        assert source._graph_weakref is dest._graph_weakref, 'Vertices belong to different graphs'

        source_wr = wr(source)
        dest_wr = wr(dest)

        if dest_wr not in source._next_weakrefs:
            raise DigraphException('Vertices are not connected')
        assert source_wr in dest._prev_weakrefs, 'Vertices are not connected, and found an inconsistency'

        source._next_weakrefs.remove(dest_wr)
        dest._prev_weakrefs.remove(source_wr)
        
        if not source._next_weakrefs: # source becomes a pit
            source._graph_weakref()._pit_weakrefs.add(source_wr)

        if not dest._prev_weakrefs: # dest becomes a source
            dest._graph_weakref()._source_weakrefs.add(dest_wr)

    @classmethod
    def are_connected(cls, source, dest):
        """
        Returns whether a vertex is connected to another one.
        Both vertecies must be instances of the same Vertex-derived class, and belong to the same graph.

        >>> g = Graph()
        >>> s, t = Vertex(), Vertex()
        >>> for v in [s,t]: g.add(v)
        >>> Vertex.connect(s, t)
        >>> Vertex.are_connected(s, t), Vertex.are_connected(t, s)
        (True, False)
        >>> Vertex.connect(t, s)
        >>> Vertex.are_connected(s, t), Vertex.are_connected(t, s)
        (True, True)
        >>> Vertex.disconnect(s, t)
        >>> Vertex.are_connected(s, t), Vertex.are_connected(t, s)
        (False, True)

        """
        
        assert isinstance(source, cls), '%r is not a %s' % (source, cls.__name__)
        assert isinstance(dest, cls), '%r is not a %s' % (dest, cls.__name__)
        assert source._graph_weakref is not None, 'Source vertex must belong to a graph'
        assert source._graph_weakref is dest._graph_weakref, 'Vertices belong to different graphs'

        d_in_s_n = wr(dest) in source._next_weakrefs
        s_in_d_p = wr(source) in dest._prev_weakrefs

        assert d_in_s_n == s_in_d_p, 'Inconsistency found'

        return d_in_s_n

    # internal methods. Do not call directly.

    def _on_add_to_graph(self, graph):
        assert self._graph_weakref is None, 'The vertex already belongs to some graph'

        vertex_wr = wr(self)
        self._graph_weakref = wr(graph)
        graph._source_weakrefs.add(vertex_wr)
        graph._pit_weakrefs.add(vertex_wr)
        
    def _on_remove_from_graph(self):
        assert self._graph_weakref is not None, 'The vertex does not belong to any graph'

        self.become_orphan()

        graph = self._graph_weakref()
        if graph:
            my_wr = wr(self)
            graph._source_weakrefs.remove(my_wr)
            graph._pit_weakrefs.remove(my_wr)
        self._graph_weakref = None

    def _on_del_graph(self, graph):
        self._next_weakrefs.clear()
        self._prev_weakrefs.clear()

        self._graph_weakref = None
