# bloggerfs.py - blogger file system in userspace using FUSE
# Copyright (C) 2010 Marco Giusti
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
# MA 02111-1307, USA


# rfc3339.py -- Implementation of the majority of RFC 3339 for python.
# Copyright (c) 2008, 2009, 2010 LShift Ltd. <query@lshift.net>


# gdata - Python client library for Google data APIs
# Copyright (C) 2009 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import os
import stat
import errno
import time
import functools
import logging
import inspect

import fuse
import gdata.blogger.client
import atom.core
import atom.data
from zope.interface import interface, declarations, implements
from zope.interface import Interface, Attribute
from zope.interface.adapter import AdapterRegistry
import rfc3339

__author__ = 'Marco Giusti'
__copyright__ = 'Copyright 2010, Marco Giusti'
__license__ = 'LGPL'
__version__ = '0.2'
__email__ = 'marco.giusti@gmail.com'


class NullHandler(logging.Handler):

    def emit(self, record):
        pass


fuse.fuse_python_api = (0, 2)
logger = logging.getLogger('bloggerfs')
logger.setLevel(logging.WARNING)
debug = logger.debug
info = logger.info
warn = logger.warn
error = logger.error
critical = logger.critical


def debugfunc(func):
    @functools.wraps(func)
    def wrapper(*args, **kwds):
        debug(func.__name__)
        for i, arg in enumerate(args):
            debug('args[%s]: %s' % (i, arg))
        for k in kwds:
            debug('kwds[%s]: %s' % (k, kwds[k]))
        try:
            ret = func(*args, **kwds)
        except:
            logger.exception('')
            raise
        debug('return value: %s\n' % ret)
        return ret
    return wrapper

def getmembers(obj, predicate=None):
    """Return all members of an object as (name, value) pairs sorted by name.
    Optionally, only return members that satisfy a given predicate."""
    if inspect.isclass(obj):
        mro = (obj,) + inspect.getmro(obj)
    else:
        mro = (obj,)
    results = []
    for key in dir(obj):
        # First try to get the value via __dict__. Some descriptors don't
        # like calling their __get__ (see bug #1785).
        for base in mro:
            if key in base.__dict__:
                value = base.__dict__[key]
                break
        else:
            value = getattr(obj, key)
        if not predicate or predicate(value):
            results.append((key, value))
    results.sort()
    return results

inspect.getmembers = getmembers # monkey patch inspect


# Taken from Twisted

globalRegistry = AdapterRegistry()
ALLOW_DUPLICATES = False


def _registered(registry, required, provided):
    """
    Return the adapter factory for the given parameters in the given
    registry, or None if there is not one.
    """
    return registry.registered([required], provided)


def registerAdapter(adapterFactory, origInterface, *interfaceClasses):
    """Register an adapter class.

    An adapter class is expected to implement the given interface, by
    adapting instances implementing 'origInterface'. An adapter class's
    __init__ method should accept one parameter, an instance implementing
    'origInterface'.
    """
    self = globalRegistry
    assert interfaceClasses, "You need to pass an Interface"
    global ALLOW_DUPLICATES

    # deal with class->interface adapters:
    if not isinstance(origInterface, interface.InterfaceClass):
        origInterface = declarations.implementedBy(origInterface)

    for interfaceClass in interfaceClasses:
        factory = _registered(self, origInterface, interfaceClass)
        if factory is not None and not ALLOW_DUPLICATES:
            raise ValueError("an adapter (%s) was already registered." % (factory, ))
    for interfaceClass in interfaceClasses:
        self.register([origInterface], interfaceClass, '', adapterFactory)


# add global adapter lookup hook for our newly created registry
def _hook(iface, ob, lookup=globalRegistry.lookup1):
    factory = lookup(declarations.providedBy(ob), iface)
    if factory is None:
        return None
    else:
        return factory(ob)
interface.adapter_hooks.append(_hook)

# End taken


class INode(Interface):
    """"""


class IFileNode(INode):

    def open(path, flags):
        """File open operation"""

    def read(path, size, offset):
        """"""

    def truncate(path, size):
        """"""

    def write(path, buf, size):
        """"""

    def release(path, flags):
        """"""

    def utime(path, times):
        """"""

    def rename(path, pathto):
        """"""

    def fsync(path, isfsyncfile):
        """"""

    def flush(path):
        """"""

    def fgetattr(path):
        """Get file attributes.

        Similar to stat().  The 'st_dev' and 'st_blksize' fields are
        ignored.	 The 'st_ino' field is ignored except if the 'use_ino'
        mount option is given."""

    def lock(path):
        """"""

    def create(path, mode, file_info):
        """"""


class IDirNode(INode):

    def opendir(path):
        """"""

    def readdir(path):
        """"""

    def fsyncdir(path):
        """"""

    def releasedir(path):
        """"""


def getType(obj):
    if isinstance(obj, DirNode):
        return stat.S_IFDIR
    return stat.S_IFREG


class DirEntry(fuse.Direntry):

    def __repr__(self):
        type = self.type == stat.S_IFREG and 'file' or 'dir'
        return '<bloggerfs.DirEntry name=%s, type=%s>' % (self.name, type)


class IUpdated(Interface):

    _updated = Attribute('Modification time in seconds from epoch')


class IPublished(Interface):

    _published = Attribute('Creation time in seconds from epoch')


class GenericUpdated(object):
    implements(IUpdated)

    def __init__(self, original):
        self.original = original

    @property
    def updated(self):
        if not hasattr(self.original, '_updated'):
            self.original._updated = int(time.time())
        return self.original._updated

registerAdapter(GenericUpdated, INode, IUpdated)


class GenericPublished(object):
    implements(IPublished)

    def __init__(self, original):
        self.original = original

    @property
    def published(self):
        if not hasattr(self.original, '_published'):
            self.original._published = IUpdated(self.original).updated
        return self.original._published

registerAdapter(GenericPublished, INode, IPublished)


class Node(object):
    implements(INode)

    _mtime = None
    _children = None

    def __init__(self, original):
        self.original = original

    @property
    def mtime(self):
        return IUpdated(self).updated

    @property
    def ctime(self):
        return IPublished(self).published

    def getChildren(self):
        return {}

    @property
    def children(self):
        if self._children is None:
            self._children = self.getChildren()
        return self._children

    def locate(self, path):
        if path == []:
            return self
        name = path.pop(0)
        children = self.children
        if name in children:
            return children[name].locate(path)
        raise IOError(errno.ENOENT)


class FileNode(Node):
    implements(IFileNode)

    @property
    def text(self):
        return self.original

    def getattr(self, path):
        assert path == [], 'Inaspected path'
        size = len(self.text)
        st = fuse.Stat(st_mode=stat.S_IFREG|0440, st_nlink=1,
                       st_uid=os.getuid(), st_gid=os.getgid(), st_size=size,
                       st_ctime=self.ctime, st_mtime=self.mtime)
        return st

    def read(self, path, size, offset):
        return self.text[offset:offset+size]

registerAdapter(FileNode, str, IFileNode)


class TextNode(FileNode):

    @property
    def text(self):
        if self.original.text is not None:
            return self.original.text
        return ''

# registerAdapter(TextNode, atom.data.Text, IFileNode)


class DirNode(Node):
    implements(IDirNode)

    def readdir(self, path, offset):
        assert offset == 0, 'readdir, offset != 0'
        assert path == [], 'readdir, inaspected path %s' % path
        yield fuse.Direntry('.')
        yield fuse.Direntry('..')
        for name, node in self.children.items():
            yield fuse.Direntry(name, type=getType(node))

    def getattr(self, path):
        assert path == [], 'Inaspected path'
        st = fuse.Stat(st_mode=stat.S_IFDIR|0550, st_nlink=2,
                       st_uid=os.geteuid(), st_gid=os.getegid(), st_size=4096,
                       st_ctime=self.ctime, st_mtime=self.mtime)
        return st


class ListNode(DirNode):

    def getChildren(self):
        return dict((str(i), INode(el)) for i, el in enumerate(self.original))

registerAdapter(ListNode, list, IDirNode)


class ElementNode(DirNode):

    def getChildren(self):
        children = {}
        for name, member_type in self.original._members:
            if isinstance(member_type, tuple):
                continue

            element = getattr(self.original, name)
            node = INode(element, None)
            if node:
                children[name] = node
        if self.original.text:
            children['text'] = FileNode(self.original.text)
        return children

registerAdapter(ElementNode, atom.core.XmlElement, IDirNode)


class FeedEntryNode(ElementNode):
    implements(IUpdated)

    _updated = None

    @property
    def updated(self):
        if self._updated is None:
            if self.original.updated:
                try:
                    updated = rfc3339.parse_datetime(self.original.updated.text)
                    self._updated = int(time.mktime(updated.timetuple()))
                except:
                    logger.exception('')
                    self._updated = int(time.time())
            else:
                self._updated = int(time.time())
        return self._updated

    @property
    def blogId(self):
        return self.original.get_blog_id()


class EntryNode(FeedEntryNode):
    implements(IPublished)

    _published = None

    @property
    def published(self):
        if self._published is None:
            if self.original.published:
                try:
                    published = rfc3339.parse_datetime(self.original.published.text)
                    self._published = int(time.mktime(published.timetuple()))
                except:
                    logger.exception('')
                    self._published = self.mtime
            else:
                self._published = self.mtime
        return self._published

registerAdapter(EntryNode, atom.data.Entry, IDirNode)


class CommentNode(EntryNode):

    @property
    def id(self):
        return self.original.get_comment_id()

registerAdapter(CommentNode, gdata.blogger.data.Comment, INode)


class PostNode(EntryNode):

    def getChildren(self):
        children = EntryNode.getChildren(self)
        feed = client.get_post_comments(self.blogId, self.id)
        children['comments'] = ListNode(feed.entry)
        return children

    @property
    def id(self):
        return self.original.get_post_id()

registerAdapter(PostNode, gdata.blogger.data.BlogPost, INode)


class BlogNode(FeedEntryNode):

    @property
    def id(self):
        return self.original.get_blog_id()

    def getChildren(self):
        children = FeedEntryNode.getChildren(self)
        feed = client.get_posts(self.blogId)
        children['posts'] = ListNode(feed.entry)
        return children

registerAdapter(BlogNode, gdata.blogger.data.Blog, INode)


class AccountNode(FeedEntryNode):

    _original = None

    def __init__(self):
        pass

    @property
    def original(self):
        if self._original is None:
            self._original = client.get_blogs()
        return self._original

    def getChildren(self):
        return dict((b.get_blog_id(), INode(b)) for b in self.original.entry)

registerAdapter(AccountNode, gdata.blogger.data.BlogFeed, INode)


client = None


class BloggerFS(fuse.Fuse):

    def __init__(self, *args, **kwds):
        fuse.Fuse.__init__(self, *args, **kwds)
        global client
        client = gdata.blogger.client.BloggerClient()
        self._account = AccountNode()

    def main(self):
        if self.parser.fuse_args.mount_expected():
            options = self.cmdline[0]
            if options.debug:
                logger.setLevel(logging.DEBUG)
            if options.syslog:
                from logging import handlers
                handler = handlers.SysLogHandler()
            elif options.logfile:
                from logging import handlers
                handler = handlers.FileHandler(options.logfile)
            elif options.debug:
                handler = logging.StreamHandler()
            else:
                handler = NullHandler()
            formatter = logging.Formatter('[%(asctime)s] [%(levelname)s] '
                                          '[%(funcName)s]  %(message)s')
            handler.setFormatter(formatter)
            logger.addHandler(handler)

            if not options.email:
                from gdata.sample_util import authorize_client
                authorize_client(client, service='blogger',
                                 source='bloggerfs',
                                 scopes=['http://www.blogger.com/feeds/'])
            else:
                email = options.email
                if not options.password:
                    import getpass
                    password = getpass.getpass()
                else:
                    password = options.password
                client.client_login(email, password, source='bloggerfs',
                                    service='blogger')
        fuse.Fuse.main(self)

    def _splitpath(self, path):
        return (path.split('/')[1:], [])[path == '/']

    def getattr(self, path):
        parts = self._splitpath(path)
        return self._account.locate(parts).getattr(parts)

    def readdir(self, path, offset):
        parts = self._splitpath(path)
        return self._account.locate(parts).readdir(parts, offset)

    def read(self, path, size, offset):
        parts = self._splitpath(path)
        return self._account.locate(parts).read(parts, size, offset)


def main(argv=None):
    if argv is None:
        import sys
        argv = sys.argv[1:]

    server = BloggerFS()
    server.parser.add_option('-e', '--email', help='Google account email')
    server.parser.add_option('-p', '--password', help='Google account password')
    server.parser.add_option('-b', '--debug', help='Show debugging info',
                             action='store_true')
    server.parser.add_option('--syslog', action='store_true',
                             help='Log messages to syslog')
    server.parser.add_option('-l', '--logfile', help='Log messages to file')
    server.parse(args=argv, errex=1)
    server.main()


if __name__ == '__main__':
    main()

