import os
import re
import shutil
import sys

from six import print_, itervalues
from six.moves import input

import marathoner
from marathoner.commands import collect_commands
from marathoner.executor import Executor
from marathoner.project import Project, ConfigError


class UsageError(Exception):
    pass


def new_marathoner(args):
    if len(args) != 1:
        raise UsageError('Exactly one argument is required.')
    project_name = args[0]
    if not re.search(r'^[_a-zA-Z]\w*$', project_name):
        print_('Error: Project names must begin with a letter and '
               'contain only letters, numbers and underscores.')
        sys.exit(1)
    elif os.path.exists(project_name):
        print_('Error: directory "%s" already exists' % project_name)
        sys.exit(1)

    templates_path = os.path.join(marathoner.__path__[0], 'templates', 'project')
    shutil.copytree(templates_path, project_name)


def run_marathoner(args):
    project = Project(args[0] if args else '.')
    executor = Executor(project)
    commands = [cmd(project, executor) for cmd in itervalues(collect_commands())]

    print_('Welcome to Marathoner! Type "help" for any help.')
    print_('You are now working on project', project.project_name)
    while True:
        user_input = input('>>> ').strip()
        if not user_input:
            continue
        for cmd in commands:
            if cmd.is_match(user_input):
                cmd.handle(user_input)
                break
        else:
            print_('Unrecognized command. Type "help" to see the list of available commands.')


available_commands = {
    'new': (new_marathoner, '<project_name>', 'Create a new marathoner project.'),
    'run': (run_marathoner, '[path_to_project]', 'Run marathoner project.')
}


def print_usage(cmd_name, help):
    _, cmd_syntax, cmd_help = available_commands[cmd_name]
    cmd = '%s %s' % (cmd_name, cmd_syntax)
    if help:
        print_('  %-25s %s' % (cmd, cmd_help))
    else:
        print_('  %s' % cmd)


def print_help():
    print_('Marathoner %s\n' % marathoner.__version__)

    print_('Usage:')
    print_('  marathoner <command> [args]\n')
    print_('Available commands:')
    for cmd_name in available_commands:
        print_usage(cmd_name, help=True)


def execute(argv=None):
    if argv is None:
        argv = sys.argv
    if len(argv) <= 1:
        print_help()
        sys.exit(0)

    cmd_name, args = argv[1], argv[2:]
    if cmd_name not in available_commands:
        print_('Unknown command: %s\n' % cmd_name)
        print_('Use "marathoner" to see available commands')
        sys.exit(2)
    cmd_func = available_commands[cmd_name][0]

    try:
        cmd_func(args)
    except UsageError as e:
        print_('%s\nUsage:' % e)
        print_usage(cmd_name, help=False)
        sys.exit(2)
    except ConfigError as e:
        print_(e)
        sys.exit(2)

if __name__ == '__main__':
    execute()
