#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import print_function

import os
import sys
import os.path
import shutil
import tempfile
import getpass
import io

import github
import requests

import myterm.color
from myterm.parser import arg, verb, alias, VerbParser
from myterm.config import Config
from myterm.table import Table

from github_client import __version__

#for support of py2
import sys
if sys.version_info[0] > 2:
    # py3k
    import configparser
else:
    # py2
    import ConfigParser as configparser
    input = raw_input

DESCRIPTION = """client for github"""
AUTHOR = "Frederic Aoustin"
PROG = "github"
VERSION = __version__

GETBOOL = {'false': False, 'False': False, 
            'true': True, 'True': True}

CONFIG = {}

REPO_KEY = ['clone_url','created_at','default_branch',
            'description','downloads_url','etag','forks_url',
            'full_name','git_url','hooks_url','html_url','id','language',
            'last_modified','name','pushed_at','size','ssh_url','svn_url',
            'updated_at','url']

class Github(github.Github):

    def __init__(self, login_or_token='', password='', *args, **kw):
        if login_or_token == '':
            login_or_token = None
        if password == '':
            password = None
        github.Github.__init__(self, login_or_token, password, *args, **kw)


def search_repo(path):
    if os.path.isfile(os.path.join(path,'.git','config')):
        with open(os.path.join(path,'.git','config')) as f:
            data = ''
            for l in f.readlines():
                data = data + '\n' + l.strip()
        try:
            config = configparser.RawConfigParser(allow_no_value=True)
            try:
                #for py2
                config.readfp(io.BytesIO(data))
            except:
                config.read_string(data)
            return '/'.join(config.get('remote "origin"','url').split('/')[-2:])[:-4]
        except:
            return '' 
    parent = os.path.abspath(os.path.join(os.getcwd(), '..'))
    if parent != path:
        return search_repo(parent)
    return ''

def first_value(*args):
    for arg in args:
        if arg != None and len(arg):
            return arg
    return args[-1]

from functools import wraps    

def manage_error(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except github.GithubException as e:
            try:
                VerbParser().error('\n'.join([e.data[d] for d in e.data]))
            except:
                VerbParser().error(e)
        except Exception as e:
            VerbParser().error(e)
    return decorated_function

def check_login(verb, arg, value):
    user = ''
    if value != None and len(value):
        user = value.split(':')[0]
    else:
        user = first_value(value, CONFIG.get('GITHUB_USER',''))
    if not len(user):
        return ':'
    if ':' in value:
        return '%s:%s' % (user, value.split(':')[1])
    if len(CONFIG.get('GITHUB_PASSWORD','')):
        return '%s:%s' % (user, CONFIG['GITHUB_PASSWORD'])
    return '%s:%s' % (user, getpass.getpass('password for %s:' % user))
    

def arg_url(func):
    arg(func, '-H', '--url', dest='url', type=str, default='https://api.github.com',
        help="url of github.com api")
    arg(func, '-u', '--login', dest='login', type=str, default='', check=check_login,
        help="user:password on github.com")
    return func


@verb('version')
@arg_url
@manage_error
def version(url="", login=""):
    """get version of github"""
    versions = Table()
    versions.add_columns('module', 'version')
    try:
        r = requests.get(url)
        versions.append({'module':'github api', 'version': r.headers['X-GitHub-Media-Type']})
    except:
        versions.append({'module': 'github api', 'version': 'unknown'})
    versions.append({'module': 'github cli', 'version': VERSION})
    versions.append({'module' : 'python', 'version': '.'.join([str(a) for a in sys.version_info[0:3]])})
    versions.orderby('module')
    versions.pprint(noheader=True)

@verb('repos', usage="%s repos [owner]" % PROG)
@alias('ls')
@arg_url
@manage_error
def repos(owner="", url="", login=":"):
    """get information repository"""
    owner = first_value(owner, CONFIG.get('GITHUB_USER', None), '')
    g = Github(*login.split(':'))
    res = Table()
    res.add_columns('repos', 'desc')
    for repo in g.get_user(owner).get_repos():
        res.append({'repos': repo.name, 'desc': repo.description})
    res.orderby('repos')
    res.pprint(noheader=True)    

@verb('repo', usage="%s repos [owner/repos]" % PROG)
@arg_url
@manage_error
def repo(repo="", url="", login=":"):
    """get information repository"""
    owner = ''
    if not len(repo):
        repo = search_repo(os.getcwd())
    if len(repo) and "/" not in repo:
        owner = first_value(owner, login.split(':')[0], CONFIG.get('GITHUB_USER', None), '')
        repo = "%s/%s" % (owner, repo)
    g = Github(*login.split(":"))
    res = Table()
    res.add_columns('key', 'value')
    grepo = g.get_repo(repo)
    for key in REPO_KEY:
        res.append({'key': key, 'value': getattr(grepo, key)})
    if len(login) > 1:
        auth = grepo.clone_url.split('//')[0] + "//" + login + '@' + grepo.clone_url.split('//')[1]
        res.append({'key': 'clone_url_authentified', 'value': auth})
    res.orderby('key')
    res.pprint(noheader=True)    

@verb('branch', usage="%s branch [owner/repos]" % PROG)
@arg_url
@manage_error
def branch(repo="", url="", login=":"):
    """list branch"""
    owner = ''
    if not len(repo):
        repo = search_repo(os.getcwd())
    if len(repo) and "/" not in repo:
        owner = first_value(owner, login.split(':')[0], CONFIG.get('GITHUB_USER', None), '')
        repo = "%s/%s" % (owner, repo)
    g = Github(*login.split(":"))
    res = Table()
    res.add_columns('name')
    for br in g.get_repo(repo).get_branches():
        res.append({'name': br.name})
    res.orderby('name')
    res.pprint(noheader=True)    

@verb('create', usage="%s create [owner/repos]" % PROG)
@arg('-f', '--force', dest='force', default=False, action="store_true", help="use default answer")
@arg_url
@manage_error
def create(repo="", url="", login=":", force=False):
    """create repository"""
    if not len(repo):
        VerbParser().error('title of repository is null')
    if len(login) < 2:
        VerbParser().error('you must be authenticated')
    g = Github(*login.split(":"))
    if force:
        desc = repo
    else:
        desc = input("Description:")
    g.get_user().create_repo(repo, desc)
    print('%s is created' % repo, file=sys.stdinfo)


@verb('rm', usage="%s rm [owner/repos]" % PROG)
@arg_url
@manage_error
def rm(repo="", url="", login=":"):
    """delete repository"""
    owner = ''
    if not len(repo):
        repo = search_repo(os.getcwd())
    if len(repo) and "/" not in repo:
        owner = first_value(owner, login.split(':')[0], CONFIG.get('GITHUB_USER', None), '')
        repo = "%s/%s" % (owner, repo)
    g = Github(*login.split(":"))
    g.get_repo(repo).delete()
    print('%s is deleted' % repo, file=sys.stdinfo)

@verb('edit', usage="%s edit [owner/repos] [description|private|has_issues|has_wiki|has_downloads|default_branch] [value]" % PROG)
@arg_url
@manage_error
def edit(repo="", param= "", value="", url="", login=":"):
    """edit parameters of repository"""
    owner = ''
    if not len(repo):
        repo = search_repo(os.getcwd())
    if len(repo) and "/" not in repo:
        owner = first_value(owner, login.split(':')[0], CONFIG.get('GITHUB_USER', None), '')
        repo = "%s/%s" % (owner, repo)
    if not len(param) or param not in "description|private|has_issues|has_wiki|has_downloads|default_branch".split('|'):
        VerbParser().error("param not in list %s" % ",".join("description|private|has_issues|has_wiki|has_downloads|default_branch".split('|')))
    if not len(value):
        value = input("new value of %s: " % param)
    g = Github(*login.split(":"))
    repo = g.get_repo(repo)
    if value in GETBOOL.keys():
        value = GETBOOL[value]
    repo.edit(repo.name, **{param: value})
    print('%s is modified' % repo.name, file=sys.stdinfo)


def main():
    global CONFIG
    try:
        if not os.path.isfile(os.path.join(os.path.expanduser("~"),'.github','conf.py')):
            if not os.path.isdir(os.path.join(os.path.expanduser("~"),'.github')):
                os.mkdir(os.path.join(os.path.expanduser("~"),'.github'))
            shutil.copy(os.path.join(os.path.dirname(__file__),'conf.py'), os.path.join(os.path.expanduser("~"),'.github','conf.py'))
    except:
        pass
    CONFIG = Config()
    CONFIG.from_pyfile(os.path.join(os.path.expanduser("~"),'.github','conf.py'), silent=True)
    CONFIG.from_env('GITHUB_')
    parser = VerbParser()
    parser.prog = PROG
    parser.version = "%s %s" % (PROG, VERSION)
    parser.description= DESCRIPTION
    parser.epilog = AUTHOR
    parser.parse_args()

if __name__ == '__main__':
    main() 
