"""
mfriv module.  Contains the ModflowRiv class. Note that the user can access
the ModflowRiv class as `flopy.modflow.ModflowRiv`.

Additional information for this MODFLOW package can be found at the `Online
MODFLOW Guide
<http://water.usgs.gov/ogw/modflow/MODFLOW-2005-Guide/index.html?riv.htm>`_.

"""
import sys
import numpy as np
from flopy.mbase import Package
from flopy.utils.util_list import mflist

class ModflowRiv(Package):
    """
    MODFLOW River Package Class.

    Parameters
    ----------
    model : model object
        The model object (of type :class:`flopy.modflow.mf.Modflow`) to which
        this package will be added.
    irivcb : int
        is a flag and a unit number. (the default is 0).
    layer_row_column_data : list of records
        In its most general form, this is a triple list of river records  The
        innermost list is the layer, row, column, and flux rate for a single
        river.  Then for a stress period, there can be a list of rivers.  Then
        for a simulation, there can be a separate list for each stress period.
        This gives the form of
            lrcd = [
                     [  #stress period 1
                       [l1, r1, c1, stg1, cond1, rbot1],
                       [l2, r2, c2, stg2, cond2, rbot2],
                       [l3, r3, c3, stg3, cond3, rbot3],
                       ],
                     [  #stress period 2
                       [l1, r1, c1, stg1, cond1, rbot1],
                       [l2, r2, c2, stg2, cond2, rbot2],
                       [l3, r3, c3, stg3, cond3, rbot3],
                       ], ...
                     [  #stress period kper
                       [l1, r1, c1, stg1, cond1, rbot1],
                       [l2, r2, c2, stg2, cond2, rbot2],
                       [l3, r3, c3, stg3, cond3, rbot3],
                       ],
                    ]
        Note that if there are not records in layer_row_column_Q, then the last
        group of rivers will apply until the end of the simulation.
    options : list of strings
        Package options. (default is None).
    naux : int
        number of auxiliary variables
    extension : string
        Filename extension (default is 'riv')
    unitnumber : int
        File unit number (default is 18).

    Attributes
    ----------
    mxactr : int
        Maximum number of river cells for a stress period.  This is calculated
        automatically by FloPy based on the information in
        layer_row_column_data.

    Methods
    -------

    See Also
    --------

    Notes
    -----
    Parameters are not supported in FloPy.

    Examples
    --------

    >>> import flopy
    >>> m = flopy.modflow.Modflow()
    >>> lrcd = [[[2, 3, 4, 15.6, 1050., -4]]]  #this river boundary will be
    >>>                                        #applied to all stress periods
    >>> riv = flopy.modflow.ModflowRiv(m, layer_row_column_data=lrcd)

    """
    def __init__(self, model, ipakcb=0, stress_period_data=None,dtype=None,
                 extension ='riv', unitnumber=18, options=None,**kwargs):
        """
        Package constructor.

        """
        # Call parent init to set self.parent, extension, name and unit number
        Package.__init__(self, model, extension, 'RIV', unitnumber)
        self.heading = '# RIV for MODFLOW, generated by Flopy.'
        self.url = 'riv.htm'
        self.ipakcb = ipakcb
        self.mxactr = 0
        self.np = 0
        if options is None:
            options = []
        self.options = options
        if dtype is not None:
            self.dtype = dtype
        else:
            self.dtype = self.get_default_dtype()
        self.stress_period_data = mflist(model,self.dtype,stress_period_data)
        self.parent.add_package(self)

    def __repr__( self ):
        return 'River class'

    @staticmethod
    def get_empty(ncells=0,aux_names=None):
        #get an empty recarray that correponds to dtype
        dtype = ModflowRiv.get_default_dtype()
        if aux_names is not None:
            dtype = Package.add_to_dtype(dtype,aux_names,np.float32)
        d = np.zeros((ncells,len(dtype)),dtype=dtype)
        d[:,:] = -1.0E+10
        return np.core.records.fromarrays(d.transpose(),dtype=dtype)


    @staticmethod
    def get_default_dtype():
        dtype = np.dtype([("k",np.int),("i",np.int),\
                         ("j",np.int),("stage",np.float32),\
                        ("cond",np.float32),("rbot",np.float32)])
        return dtype

    def ncells( self):
        # Returns the  maximum number of cells that have river (developed for MT3DMS SSM package)
        return self.stress_period_data.mxact

    def write_file(self):
        """
        Write the file.

        """
        f_riv = open(self.fn_path, 'w')
        f_riv.write('{0}\n'.format(self.heading))
        line = '{0:10d}{1:10d}'.format(self.stress_period_data.mxact, self.ipakcb)
        for opt in self.options:
            line += ' ' + str(opt)
        line += '\n'
        f_riv.write(line)
        self.stress_period_data.write_transient(f_riv)
        f_riv.close()

    def add_record(self,kper,index,values):
        try:
            self.stress_period_data.add_record(kper,index,values)
        except Exception as e:
            raise Exception("mfriv error adding record to list: "+str(e))

    @staticmethod
    def load(f, model, nper=None, ext_unit_dict=None):
        """
        Load an existing package.

        Parameters
        ----------
        f : filename or file handle
            File to load.
        model : model object
            The model object (of type :class:`flopy.modflow.mf.Modflow`) to
            which this package will be added.
        nper : int
            The number of stress periods.  If nper is None, then nper will be
            obtained from the model object. (default is None).
        ext_unit_dict : dictionary, optional
            If the arrays in the file are specified using EXTERNAL,
            or older style array control records, then `f` should be a file
            handle.  In this case ext_unit_dict is required, which can be
            constructed using the function
            :class:`flopy.utils.mfreadnam.parsenamefile`.

        Returns
        -------
        rch : ModflowRiv object
            ModflowRiv object.

        Examples
        --------

        >>> import flopy
        >>> m = flopy.modflow.Modflow()
        >>> riv = flopy.modflow.ModflowRiv.load('test.riv', m)

        """

        if model.verbose:
            sys.stdout.write('loading riv package file...\n')

        return Package.load(model, ModflowRiv, f, nper)
