# -*- coding: utf-8 -*-
"""
Class SupplyUseTable. 
Version 1.1. Last change: May 18th, 2015.  
Check https://github.com/stefanpauliuk/pySUT for latest version.

Methods for efficient handling of supply and use tables (SUTs)

Created on Mon Jun 30 17:21:28 2014

@author: stefan pauliuk, NTNU Trondheim, Norway
Guillaume Majeau-Bettez, NTNU Trondheim, Norway

Checked for python 2.7 compatibility by Chris Mutel, PSI, Villingen, CH

standard abbreviation: SUT

dependencies:
    numpy >= 1.9
    scipy >= 0.14

Repository for this class, documentation, and tutorials: https://github.com/stefanpauliuk/pySUT

"""

from __future__ import division, print_function
import logging
import numpy as np
from scipy import sparse as sp
from scipy.sparse import linalg as sl


class SupplyUseTable(object):

    """ Class containing a complete supply and use table

    Attributes
    ----------
    V : product by industry supply table
    U : product by industry use table
    Y : final demand product by end-use category
    F : Extensions: type by industry
    FY: Extensions: type by end-use category
    TL: Trade link (for MRIO models)
    l_pro: Labels of products
    l_ind: Labels of industries
    l_ext: Labels of extensions


    unit : Unit for each row of V,U, and y (product unit)
    version : string
        This can be used as a version tracking system.
    year : string or int
        Baseyear of the IOSystem
    name : string, optional
        Name of the SUT, default is 'SUT'
    regions: Number of regions, for multiregional SUT

    E_bar : Mapping of primary production
        + product-by-industry matrix of 0 or 1
        + coefficient 1 to indicate primary product and 0 otherwise
        + can be automatically generated by self.build_E_bar for simple cases

    Xi : Matrix of substitutability between products
        + [product-by-product dimensions]
        + Xi[2,1] = 0.5 means that every secondary production of product 1
          displaces (pushes out) 0.5 units of product 2.
        + Can be automatically generated by self.build_mr_Xi for simple,
          multi-regional cases

    PHI : Partition coefficients for partition allocation or constructs
        + industry-by-product dimensions, with rows adding up to 1.0
        + PHI[2,1] = 0.3 means that the total supply of product 1 will be
          attributed 30% of requirements/emissions of industry 2
        + Can be generated on the fly based on self.PSI

    PSI: Matrix of intensive properties of products to guide partition
         allocation within each industry.
        + [product-by-industry dimensions]
        + If PSI[:, 2] contains energy densities, industry 2 will be
          partitioned based on the energy content of its supplies
        + Used in the calculation of partition coefficients PHI

    Gamma: Matrix of alternate activities
        + [industry-by-product dimensions]
        + Gamma[2, 1] = 1 means that the technology of the primary product of
          industry 2 will be assumed for all secondary productions of product 1
        + Can be automatically generated by self.build_mr_Gamma for simple,
          multi-regional cases

    """


    def __init__(self, V=None, U=None, Y=None, F=None, FY=None, TL=None,
                 unit=None, version=None, year=None, name='SUT', regions=1,
                 E_bar=None, Xi=None, PHI=None, PSI=None, Gamma=None):
        """ Basic initialisation and dimension check methods """

        self.V = V          # optional
        self.U = U          # optional
        self.Y = Y          # optional
        self.F = F          # optional
        self.FY = FY        # optional
        self.TL = TL        # optional
        self.l_pro = None
        self.l_ind = None
        self.l_ext = None

        self.name = name  # optional
        self.regions = regions # Number of regions, for multiregional SUT
        self.unit = unit  # optional
        self.year = year  # optional
        self.version = version  # optional

        self.E_bar = E_bar
        self.Xi = Xi
        self.PHI = PHI
        self.PSI = PSI
        self.Gamma = Gamma

    def return_version_info(self):
        return str('Class SupplyUseTable. Version 1.1. Last change: May 9th, 2015.  Check https://github.com/stefanpauliuk/pySUT for latest version.')

    def dimension_check(self):
        """ This method checks which variables are present and checks whether data types and dimensions match
        """
        # Compile a little report on the presence and dimensions of the elements in the SUT
        DimReport = str('<br><b> Checking dimensions of SUT structure</b><br>')
        if self.V is not None:
            DimReport += str('Supply table is present with ' + str(len(self.V)) +
                             ' rows (products) and ' + str(len(self.V[0])) + ' columns (industries).<br>')
        else:
            DimReport += str('Supply table is not present.<br>')
        if self.U is not None:
            DimReport += str('Use table is present with ' + str(len(self.U)) +
                             ' rows (products) and ' + str(len(self.U[0])) + ' columns (industries).<br>')
        else:
            DimReport += str('Use table is not present.<br>')
        if self.Y is not None:
            if len(self.Y.shape) == 1:  # if Y is a true vector
                DimReport += str('Final demand is present with ' + str(len(self.Y)) +
                                 ' rows (products) and 1 column (FD categories).<br>')
            else:
                DimReport += str('Final demand is present with ' + str(len(self.Y)) +
                                 ' rows (products) and ' + str(len(self.Y[0])) + ' columns (FD categories).<br>')
        else:
            DimReport += str('Final demand is not present.<br>')
        if self.F is not None:
            DimReport += str('Industry extensions are present with ' + str(len(self.F)) +
                             ' rows (stressors) and ' + str(len(self.F[0])) + ' columns (industries).<br>')
        else:
            DimReport += str('Industry extensions are not present.<br>')
        if self.FY is not None:
            DimReport += str('FD extensions are present with ' + str(len(self.FY)) +
                             ' rows (stressors) and ' + str(len(self.FY[0])) + ' columns (FD categories).<br>')
        else:
            DimReport += str('FD extensions are not present.<br>')
        if self.TL is not None:
            DimReport += str('Trade link is present with ' + str(len(self.TL)) +
                             ' rows (products) and ' + str(len(self.TL[0])) + ' columns (regions).<br>')
        else:
            DimReport += str('Trade link is not present.<br>')

        # for most operations, especially the constructs, U and V are required to
        # be present and have correct dimensions. We check for this:
        if self.U is not None:
            if self.V is not None:
                if len(self.V) == len(self.U):
                    if len(self.V[0]) == len(self.U[0]):
                        StatusFlag = 1  # V and U have proper dimensions
                    else:
                        StatusFlag = 0
                else:
                    StatusFlag = 0
            else:
                StatusFlag = 0
        else:
            StatusFlag = 0

        return DimReport, StatusFlag

    def compare_IndustrialUseAndSupply(self):
        """ This method computes total industrial supply and total industrial use, and compares the two
        ResultVector = U.e */ V.e
        """
        return self.U.sum(axis=1) / self.V.sum(axis=1)

    def supply_diag_check(self):
        """ to apply the BTC, we need to have a non-zero diagonal for each producing sector.
        Determine which sectors produce: """
        SupplySum_i = self.g
        SupplySum_p = self.q
        SupplyDiag = self.V.diagonal()
        SupplyDiag_Eval = np.zeros((self.V.shape[0], 7))
        for m in range(0, self.V.shape[0]):
            if SupplySum_p[m] != 0:
                if SupplySum_i[m] != 0:
                    if SupplyDiag[m] != 0:
                        SupplyDiag_Eval[m, 0] = 1  # Non-zero supply by main producer. Normal situation, OK
                    else:
                        SupplyDiag_Eval[m, 1] = 1  # No supply by apparent main producer, problem
                else:
                    # Product only produced by other sectors, this sector is empty
                    SupplyDiag_Eval[m, 2] = 1
            else:
                if SupplySum_i[m] != 0:
                    # product not produced, apparent main sector produces only other products
                    SupplyDiag_Eval[m, 3] = 1
                else:
                    SupplyDiag_Eval[m, 4] = 1  # Product not produced and main sector is empty
            SupplyDiag_Eval[m, 5] = SupplySum_p[m]
            SupplyDiag_Eval[m, 6] = SupplySum_i[m]

        return SupplyDiag_Eval

    def _check_secondary_prod(self, full_debug=False):
        """ Diagnostic of primary and secondary productions

        This simple method gives an overview of the number of exclusive
        secondary productions, off-diagonal primary productions, and primary
        productions that are of smaller magnitude than their associated
        secondary productions.

        Args:
        -----
        * full_debug: Output a table to further examine a summary of "strange"
          primary-secondary coproduction patterns

        Returns:
        -------
        * strange: None, or an array of secondary productions that surpass
          their associated primary production in magnitude.

        """

        # Default
        strange = None
        # check how many off diagonal primary poducts we have
        if self.E_bar.shape[0] == self.E_bar.shape[1]:
            offdiag = self.E_bar - ddiag(self.E_bar)

            offdiag_tot = np.sum(offdiag)
            if  offdiag_tot > 0:
                msg = "Found {} off-diagonal primary productions"
                logging.info(msg.format(offdiag_tot))

        # Check how many exclusive secondary products
        exclus = (self.E_bar.sum(1) == 0) & (self.q != 0)
        exclus_tot = np.sum(exclus)
        if exclus_tot > 0:
            msg = "Found {} exclusive secondary products."
            logging.info(msg.format(exclus_tot))

        # Check how many secondary products are produced in greater amount than
        # their associated primary product
        big_sec = np.max(self.V_bar, 0) < np.max(self.V_tild, 0)
        big_sec_tot = np.sum(big_sec)
        if big_sec_tot > 0:
            msg = ("Found {} secondary products that are produced in greater"
                   "amount than their primary product")
            logging.info(msg.format(big_sec_tot))

            if full_debug:
                bo_bar = np.argmax(self.V_bar[:, big_sec], axis=0)
                bo_all = np.argmax(self.V[:, big_sec], axis=0)
                header = np.array(["Country",
                                   "Industry",
                                   "main product",
                                   "amount",
                                   "max product",
                                   "amount"])
                strange = np.row_stack([header,
                                np.column_stack([self.l_ind[big_sec, 0:2],
                                                   self.l_pro[bo_bar,1],
                                                   self.V_bar[bo_bar, big_sec],
                                                   self.l_pro[bo_all,1],
                                                   self.V[bo_all, big_sec]])])

        return strange


            # q_bar_glo = aggregate_regions_vectorised(
            #                         self.V_bar(), regions=self.regions).sum(1)
            # q_glo = aggregate_regions_vectorised(
            #                         self.V, regions=self.regions).sum(1)
            # exclus_glo = (q_bar_glo == 0) & (q_glo != 0)
            # exclus_glo_tot = np.sum(exclus_glo)
            # if exclus_glo_tot > 0:
            #     msg = "Found {} globally exclusive secondary products."
            # else:
            #     msg = "But there are {} gobally exclusive secondary products."
            # logging.info(msg.format(exclus_glo_tot))




    """
    Basic computations, row sum, col sum, etc.
    """

    @property
    def q(self):
        """ Vector of total product output, calculate from V, as property"""
        return self.V.sum(axis=1)

    @property
    def g(self):
        """ Vector of total industry output, calculate from V, as property"""
        return self.V.sum(axis=0)

    @property
    def V_bar(self):
        """
        Table of primary production, calculated from V and E_bar, as property
        """
        if self.E_bar is None and (self.V.shape[0] == self.V.shape[1]):
            logging.warning("Assuming primary production is on diagonal")
            return ddiag(self.V)
        else:
            return self.V * self.E_bar

    @property
    def V_tild(self):
        """
        Table of secondary production, calculated from V, as property
        """
        return self.V - self.V_bar


    def g_V(self):
        """ Compute total industrial output g from supply table V."""
        logging.warning("Planned deprecation of g_V(), use property 'g' instead")
        return self.g

    def q_V(self):
        """ Compute total product output g from supply table V."""
        logging.warning("Planned deprecation of q_V(), use property 'q' instead")
        return self.q

    def return_diag_V(self):
        """ Returns the diagonal of the supply table in matrix form : V^              """
        if self.V.shape[0] != self.V.shape[1]:
            raise ValueError(
                'Error: Supply table is not square, there is no proper diagonal of that matrix.')
        else:
            Result_Array = np.zeros((self.V.shape[0], self.V.shape[0]))
            for m in range(0, self.V.shape[0]):
                Result_Array[m, m] = self.V[m, m]
            return Result_Array

    def return_offdiag_V(self):
        """   Returns the off-diagonal of the supply table in matrix form : V_offdiag              """
        if self.V.shape[0] != self.V.shape[1]:
            raise ValueError(
                'Error: Supply table is not square, there is no proper diagonal of that matrix.')
        else:
            Result_Array = self.V.copy()
            for m in range(0, self.V.shape[0]):
                Result_Array[m, m] = 0
            return Result_Array

    def market_balance(self):
        """ Returns the market balance of the SUT."""
        if self.Y is not None:
            if len(self.Y.shape) == 1:  # if Y is a true vector
                return self.V.sum(axis=1) - self.U.sum(axis=1) - self.Y
            else:  # if Y is an array
                return self.V.sum(axis=1) - self.U.sum(axis=1) - self.Y.sum(axis=1)
        else:
            raise ValueError(
                'Error: There is no final demand; the market balance cannot be computed.')

    """
    Aggregation, removal, and re-arrangement methods
    """
    def build_Aggregation_Matrix(self, Position_Vector): 
        """Turn a vector of target positions into a matrix that aggregates 
        or re-arranges rows of the table it is multiplied to from the left 
        (or columns, if multiplied transposed from the right)"""
        AM_length = Position_Vector.max() + 1 # Maximum row number of new matrix (+1 to get the right length)
        AM_width  = len(Position_Vector) # Number of rows of the to-be-aggregated matrix
        Rearrange_Matrix = np.zeros((AM_length,AM_width))
        for m in range(0,len(Position_Vector)):
            Rearrange_Matrix[Position_Vector[m].item(0),m] = 1 # place 1 in aggregation matrix at [PositionVector[m],m], so that column m is aggregated with Positionvector[m] in the aggregated matrix
        return Rearrange_Matrix

    def aggregate_rearrange_products(self, PA, PR):
        """ multiplies an aggregation matrix PA from the left to V, U, and Y, rearranges the rows in columns of V, U, and Y according to the sorting matrix PR
        Equations: X_aggregated = PA * X, where X = U, V, or Y (and also TL)
        X_new = PR * X_aggregated * PR', where X = U, V
        Y_new = PR * Y_aggregated (and also TL)
        """
        self.V = np.dot(PR, np.dot(np.dot(PA, self.V), PR.transpose()))
        self.U = np.dot(PR, np.dot(np.dot(PA, self.U), PR.transpose()))
        if self.Y is not None:
            self.Y = np.dot(PR, np.dot(PA, self.Y))
        if self.F is not None:
            self.F = np.dot(self.F, PR.transpose())
        # No changes apply to FY
        if self.TL is not None:
            self.TL = np.dot(PR, np.dot(PA, self.TL))

        return 'Products were aggregated. Products and industries were resorted successfully.'

    def aggregate_regions(self, AV):
        """ This method aggregates the supply and use table. The length of the vector AV sais how many regions there are in the model. The total number of products and industries must be a multiple of that number, else, an error is given.
        Then, the SUT is summed up according to the positions in AV. if AV[n] == x, then region n in the big SUT is aggregated into region x
        OBS: This method required the presence of U, V, Y, F, and FY. Only TL is optional.
        """
        # First, check whether the elements in AV are monotonically increasing, starting from 1:
        if (np.unique(AV) - np.arange(1, max(AV) + 1, 1)).sum() == 0:
            DR, StatusFlag = self.dimension_check()
            if StatusFlag == 1:  # Dimensions are OK, continue
                ProdsPerRegion = len(self.V) / len(AV)
                IndusPerRegion = len(self.V[0]) / len(AV)
                FDPerRegion = len(self.Y[0]) / len(AV)
                # if the number of products is a true multiple of the number of regions
                if int(ProdsPerRegion) == ProdsPerRegion:
                    # if the number of industries is a true multiple of the number of regions
                    if int(IndusPerRegion) == IndusPerRegion:
                        # if the number of final demand categories is a true multiple of the
                        # number of regions
                        if int(FDPerRegion) == FDPerRegion:
                            print('Everything has proper dimensions. Aggregating SUT.')
                            NewSupply = np.zeros(
                                (ProdsPerRegion * max(AV), IndusPerRegion * max(AV)))
                            NewUse = np.zeros((ProdsPerRegion * max(AV), IndusPerRegion * max(AV)))
                            NewF = np.zeros((len(self.F), IndusPerRegion * max(AV)))
                            NewY = np.zeros((ProdsPerRegion * max(AV), FDPerRegion * max(AV)))
                            NewFY = np.zeros((len(self.F), FDPerRegion * max(AV)))

                            SupplyIM = np.zeros((ProdsPerRegion * max(AV), len(self.V[0])))
                            UseIM = np.zeros((ProdsPerRegion * max(AV), len(self.U[0])))
                            YIM = np.zeros((ProdsPerRegion * max(AV), len(self.Y[0])))
                            for m in range(0, len(AV)):  # aggregate rows
                                SupplyIM[(AV[m] - 1) * ProdsPerRegion:(AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] = SupplyIM[(AV[m] - 1) * ProdsPerRegion:(
                                    AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] + self.V[m * ProdsPerRegion:m * ProdsPerRegion + ProdsPerRegion, :]
                                UseIM[(AV[m] - 1) * ProdsPerRegion:(AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] = UseIM[(AV[m] - 1) * ProdsPerRegion:(
                                    AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] + self.U[m * ProdsPerRegion:m * ProdsPerRegion + ProdsPerRegion, :]
                                YIM[(AV[m] - 1) * ProdsPerRegion:(AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] = YIM[(AV[m] - 1) * ProdsPerRegion:(
                                    AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] + self.Y[m * ProdsPerRegion:m * ProdsPerRegion + ProdsPerRegion, :]
                            for m in range(0, len(AV)):  # aggregate columns
                                NewSupply[:, (AV[m] - 1) * IndusPerRegion:(AV[m] - 1) * IndusPerRegion + IndusPerRegion] = NewSupply[:, (AV[m] - 1) * IndusPerRegion:(
                                    AV[m] - 1) * IndusPerRegion + IndusPerRegion] + SupplyIM[:, m * IndusPerRegion:m * IndusPerRegion + IndusPerRegion]
                                NewUse[:, (AV[m] - 1) * IndusPerRegion:(AV[m] - 1) * IndusPerRegion + IndusPerRegion] = NewUse[:, (AV[m] - 1) * IndusPerRegion:(
                                    AV[m] - 1) * IndusPerRegion + IndusPerRegion] + UseIM[:, m * IndusPerRegion:m * IndusPerRegion + IndusPerRegion]
                                NewY[:, (AV[m] - 1) * FDPerRegion:(AV[m] - 1) * FDPerRegion + FDPerRegion] = NewY[:, (AV[m] - 1) * FDPerRegion:(
                                    AV[m] - 1) * FDPerRegion + FDPerRegion] + YIM[:, m * FDPerRegion:m * FDPerRegion + FDPerRegion]
                                NewF[:, (AV[m] - 1) * IndusPerRegion:(AV[m] - 1) * IndusPerRegion + IndusPerRegion] = NewF[:, (AV[m] - 1) * IndusPerRegion:(
                                    AV[m] - 1) * IndusPerRegion + IndusPerRegion] + self.F[:, m * IndusPerRegion:m * IndusPerRegion + IndusPerRegion]
                                if self.FY is not None:  # if we have findal demand extensions
                                    NewFY[:, (AV[m] - 1) * FDPerRegion:(AV[m] - 1) * FDPerRegion + FDPerRegion] = NewFY[:, (AV[m] - 1) * FDPerRegion:(
                                        AV[m] - 1) * FDPerRegion + FDPerRegion] + self.FY[:, m * FDPerRegion:m * FDPerRegion + FDPerRegion]
                            # assign the new values to the object
                            self.V = NewSupply
                            self.U = NewUse
                            self.Y = NewY
                            self.F = NewF
                            self.FY = NewFY
                            if self.TL is not None:  # Special case: If a trade link is present:
                                NewTL = np.zeros((ProdsPerRegion * max(AV), max(AV)))
                                TL_IM = np.zeros((ProdsPerRegion * max(AV), len(AV)))
                                # First, aggregate the origin regions:
                                for m in range(0, len(AV)):
                                    TL_IM[(AV[m] - 1) * ProdsPerRegion:(AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] = TL_IM[(AV[m] - 1) * ProdsPerRegion:(
                                        AV[m] - 1) * ProdsPerRegion + ProdsPerRegion, :] + self.TL[m * ProdsPerRegion:m * ProdsPerRegion + ProdsPerRegion, :]
                                # Second, aggregate the destination regions:
                                for m in range(0, len(AV)):
                                    NewTL[
                                        :, int((AV[m] - 1))] = NewTL[:, int((AV[m] - 1))] + TL_IM[:, m]
                                self.TL = NewTL
                            ExitFlag = 1
                            ExitComment = 'Aggregation of regions went allright.'
                        else:
                            ExitFlag = 4
                            ExitComment = 'Total number of final demand categories is not a true multiple of the number of regions.'
                    else:
                        ExitFlag = 2
                        ExitComment = 'Total number of industries is not a true multiple of the number of regions.'
                else:
                    ExitFlag = 3
                    ExitComment = 'Total number of products is not a true multiple of the number of regions.'
            else:
                ExitFlag = 5
                ExitComment = 'Problem with the dimensions of the SUT.'
        else:
            ExitFlag = 0
            ExitComment = 'Problem with the sorting vector. It needs to contain all natural numbers from 1,2,3,... to its maximum value.'
        return ExitFlag, ExitComment

    def remove_products_industries(self, RPV, RIV):
        """ This method sets the products with the indices in the remove-product-vector RPV to zero.
        Likewise for the industries in the remove-industy-vector RIV
        """
        # First: remove products from U, V, and Y:
        for x in RPV:
            self.U[x, :] = 0
            self.V[x, :] = 0
            if self.Y is not None:
                self.Y[x, :] = 0
            # No changes to FY
            if self.TL is not None:
                # This might be problematic, since some methods require T to have at least
                # a 1 on the diagonal
                self.TL[x, :] = 0
        # Second: remove industries from U, V, and F:
        for x in RIV:
            self.U[:, x] = 0
            self.V[:, x] = 0
            self.F[:, x] = 0

        return 'Products and industries were removed successfully.'

    """
    Modify tables
    """

    def build_E_bar(self, prefer_exclusive=True, prefer_diag=True):
        """ Determine E_bar based on V, indentifying each primary supply flow

        Makes a best guess at the primary production flow of each industry. If
        a supply flow is found on the diagonal of the supply matrix (V) pick
        this one.  Otherwise, by default, give the preference to any product
        that is exclusive to this industry, i.e. a product that is not produced
        by any other industry.  For all the rest, pick the biggest supply flows
        of each industry.

        Args
        ----

        prefer_diag: If the supply table is square, any value on the diagonal
                     is taken to be primary production. Otherwise, pick the
                     largest supply flow.

        prefer_exclusive: Default True. When determining between off-diagonal
                          entries, give preference to a product that is not
                          produced by any other industry over a larger supply
                          flow.

        In other words, prefer_diag has precedence over prefer_exclusive, which
        has precedence over the general rule of the primary production being
        the largest production.

        """


        # Initialize zero arrays
        V_exclusive = np.zeros_like(self.V)
        V_exclusive_max = np.zeros_like(self.V)
        V_max = np.zeros_like(self.V)

        # column numbers
        cols = np.arange(self.V.shape[1])

        # If square, assume that diagonal is mainproduct whenever not null
        # Otherwise, don't assume anything
        if self.V.shape[0] == self.V.shape[1] and prefer_diag:
            done = np.array(np.diag(self.V), dtype=bool)
            E_bar = np.array(np.diag(done), dtype=int)
        else:
            E_bar = np.zeros_like(self.V, dtype=int)
            done = np.sum(E_bar, 0) != 0

        if prefer_exclusive:
            # For all other industries, if sole producer of product, make that
            # product the main product

            # Filters for exclusive products and exclusive productions of
            # interest
            V_binary = np.array(np.array(self.V, dtype=bool), dtype=int)
            exclusive_product = np.sum(V_binary, 1) == 1
            mask = np.outer(exclusive_product, ~done)

            V_exclusive[mask] = self.V[mask]
            max_in_column = np.argmax(np.abs(V_exclusive), axis=0)
            V_exclusive_max[max_in_column, cols] = V_exclusive[max_in_column, cols]

            E_bar[np.array(V_exclusive_max, dtype=bool)] = 1
            done = np.sum(E_bar, 0) != 0


        # For each column without a main product, chose the largest supply flow
        max_in_column = np.argmax(np.abs(self.V), axis=0)
        V_max[max_in_column, cols] = self.V[max_in_column, cols]
        E_bar[:, ~ done] = np.array(np.array(V_max[:, ~done], dtype=bool), dtype=int)
        self.E_bar = E_bar
        self._check_secondary_prod()

    def primary_market_shares_of_regions(self):
        """ Calculate a region's share in each product's global primary supply

        For each object type, calculate the fraction that each region
        represents in its primary supply (secondary supply, as defined by
        E_tild, is excluded)

        Dependencies:
        -------------
            self.V
            self.E_bar
            self.regions

        Returns
        -------
            D: region-per-product_type matrix, with market share coefficients
                + Each column must add up to 1 (or zero if no production)

        """


        # Aggregate primary supply within each region, across industries
        Vagg = aggregate_within_regions(self.V_bar, self.regions, axis=1)

        # Aggregate primary supply within each product group, across regions
        e = np.ones(self.regions, dtype=int)
        Vagg = aggregate_regions_vectorised(Vagg, e, axis=0)

        # world-wide primary production of each product
        q_bar = np.sum(Vagg, 1)

        # Normalize regional production relative to total world production
        D = Vagg.T * _one_over(q_bar) # keep zero where there is a 0, not inf

        return D

    def build_mr_Xi(self):
        """ Define Product substitutability matrix for multiregional system

        By default, products displace identical products produced in the same
        region. If all products were produced as primary product in all
        regions, the Xi matrix would be an identity matrix.

        Otherwise, if a product is only produced as a secondary product in a
        region, make this product substitute average primary production mix.

        Dependencies
        ------------
            self.E_bar to indicate primary production

        Returns
        -------
            self.Xi: a product*regions-by-product*region, square matrix

        """

        # By default secondary production substitutes identical product from
        # primary production in the same region
        e_bar = np.array(np.array(np.sum(self.E_bar, 1), bool), int)
        Xi = np.diag(e_bar)

        # When no local primary production to substitute, turn to global
        # primary mix
        D = self.primary_market_shares_of_regions()

        # Rearrange mix data in a region_product-by-product table
        global_mix = np.array([]).reshape(0, D.shape[1])
        for row in D:
            global_mix = np.vstack([global_mix, np.diag(row)])

        # Then tile to get same dimensions as Xi, and filter out columns where
        # Xi already has a coefficient. This gives the completementary
        # situations where the average global mix (rather than the local
        # production) get substituted
        Xi_glob = np.tile(global_mix, self.regions) * (1 - e_bar).T

        # TODO: check if there are economy-wide exclusive secondary products
        #
        # For now assume everything gets primarily produced by somebody
        # somewhere

        # Put all together and return to self
        self.Xi = Xi + Xi_glob

    def build_mr_Gamma(self, exclude_minority_prod=True):
        """ Autogenerate alternate activity matrix for multi-regional SUT

        """

        V_prim = self.V_bar.copy()
        if exclude_minority_prod:
            # Remove primary productions that are minority productions
            not_prim = V_prim.max(0) < self.V.max(0)
            V_prim[:, not_prim] = 0.0


        # Share of global primary production by held industries*countries
        # [rows] of product groups [columns]
        X = aggregate_regions_vectorised(V_prim.T, axis=1, regions=self.regions)
        D = X * _one_over(X.sum(axis=0))

        # Check that there is globally at least one eligible primary producer
        # for each product.
        excl_glo = D.sum(0) == 0
        if np.any(excl_glo):
            msg = ("There are {} products without an eligible primary producer"
                   " from which to assume a technology.")
            logging.warning(msg.format(np.sum(excl_glo)))

        # Use this mix for all exclusive secondary productions
        e_excl = np.array(V_prim.sum(1) == 0, int)
        Gamma_excl = np.kron(np.ones(self.regions), D) * e_excl

        # Otherwise, use local primary production mix (could be more than one
        # if multiple primary producers
        Gamma_prim = V_prim.T * _one_over(V_prim.sum(1))

        self.Gamma = Gamma_excl + Gamma_prim



    def add_ones_to_diagonal(self):
        """ This method adds ones where there is a zero on the diagonal of V. This is needed for simple applications of the BTC."""
        if self.V.shape[0] != self.V.shape[1]:
            return 'Error: Supply table is not square, there is no proper diagonal of that matrix.'
        else:
            for m in range(0, self.V.shape[0]):
                if self.V[m, m] == 0:
                    self.V[m, m] = 1

    def clear_non_diag_supply(self):
        """ This method allows for simple application of the BTC. It removes all sectors that do not produce their respective main product."""
        if self.V.shape[0] != self.V.shape[1]:
            raise ValueError(
                'Error: Supply table is not square, there is no proper diagonal of that matrix.')
        else:
            for m in range(0, self.V.shape[0]):
                if self.V[m, m] == 0:
                    self.V[:, m] = 0
                    self.U[:, m] = 0

    """
    Constructs. Below, it is always assumed that U and V are present. For the industrial stressorts, F must be present as well.
    """

    """ byproduct technology construct (BTC)"""

    def Build_BTC_A_matrix(self, Xi=None):
        """ Builds the A-matrix of the normal BTC construct, using Xi as mapping matrix
        returns: A matrix for BTC construct
        A_BTC = (U - Xi * V_offdiag)V_diag_inv """
        if Xi == None:
            Xi = np.ones((self.V.shape))
        self.A_BTC = np.dot(
            (self.U - Xi * self.return_offdiag_V()), np.linalg.inv(self.return_diag_V()))
        return self.A_BTC

    def Build_BTC_Am_matrix(self):
        """ returns use part of BTC construct: Am = UV^-1. Used to re-construct the SUT from the BTC-IO model """
        return np.dot(self.U, np.linalg.inv(self.return_diag_V()))

    def Build_BTC_Ab_matrix(self):
        """ returns use part of BTC construct: Ab = VoffdiagV^-1. Used to re-construct the SUT from the BTC-IO model """
        return np.dot(self.return_offdiag_V(), np.linalg.inv(self.return_diag_V()))

    def Build_BTC_S(self):
        """Returns stressor coefficient matrix for the BTC construct."""
        self.S_BTC = np.dot(self.F, np.linalg.inv(self.return_diag_V()))
        return self.S_BTC

    """ Commodity technology construct (CTC)"""

    def Build_CTC_A_matrix_ixi(self):
        """ Builds the A-matrix of the CTC construct, industry-by-industry
        return: A matrix for CTC construct
        Equation taken from Miller and Blair (2009), chapter 5, Equation 5.26a
        A_CTC_ixi = g^ * V^-1 * U * g^-1"""
        if self.V.shape[0] != self.V.shape[1]:
            raise ValueError('Error: Supply table V is not square, no matrix inversion possible.')
        else:
            try:
                V_inv = np.linalg.inv(self.V)
                try:
                    g_inv = np.linalg.inv(np.diag(self.g))
                    self.A_CTC_ixi = np.dot(
                        np.dot(np.diag(self.g), V_inv), np.dot(self.U, g_inv))
                except:
                    raise ValueError(
                        'Error: Diagonal of total industry output g cannot be inverted. Singular matrix.')
            except:
                raise ValueError('Error: Supply table V is square, but no inverse exists.')
        return self.A_CTC_ixi

    def Build_CTC_A_matrix_cxc(self):
        """ Builds the A-matrix of the CTC construct, commodity-by-commodity
        return: A matrix for CTC construct
        Equation taken from Miller and Blair (2009), chapter 5, Equation 5.26
        A_CTC_cxc = U * V^-1"""
        if self.V.shape[0] != self.V.shape[1]:
            raise ValueError('Error: Supply table V is not square, no matrix inversion possible.')
        else:
            try:
                V_inv = np.linalg.inv(self.V)
                self.A_CTC_cxc = np.dot(self.U, V_inv)
            except:
                raise ValueError('Error: Supply table V is square, but no inverse exists.')
        return self.A_CTC_cxc

    def Build_CTC_cxc_S(self):
        """Returns stressor coefficient matrix for the CTC cxc construct. S = F V^-1"""
        self.S_CTC_cxc = np.dot(self.F, np.linalg.inv(self.V))
        return self.S_CTC_cxc

    """ Industry technology construct (ITC)"""

    def Build_ITC_A_matrix_ixi(self):
        """ Builds the A-matrix of the ITC construct, industry-by-industry
        return: A matrix for ITC construct
        Equation taken from Miller and Blair (2009), chapter 5, Equation 5.27a
        A_ITC_ixi = V'*q^-1  *  U * g^-1"""

        try:
            self.g_hat_inv = np.linalg.inv(np.diag(self.g))
            self.q_hat_inv = np.linalg.inv(np.diag(self.q))
            self.A_ITC_ixi = np.dot(
                np.dot(self.V.transpose(), self.q_hat_inv), np.dot(self.U, self.g_hat_inv))
        except:
            raise ValueError('Error: Singular matrix.')

        return self.A_ITC_ixi

    def Build_ITC_A_matrix_cxc(self):
        """ Builds the A-matrix of the ITC construct, commodity-by-commodity
        return: A matrix for ITC construct
        Equation taken from Miller and Blair (2009), chapter 5, Equation 5.27
        A_ITC_cxc = U * g^-1  *  V'*q^-1 """

        try:
            self.g_hat_inv = np.linalg.inv(np.diag(self.g))
            self.q_hat_inv = np.linalg.inv(np.diag(self.q))
            self.A_ITC_cxc = np.dot(
                np.dot(self.U, self.g_hat_inv), np.dot(self.V.transpose(), self.q_hat_inv))
        except:
            raise ValueError('Error: Singular matrix.')

        return self.A_ITC_cxc

    def Build_ITC_cxc_S(self):
        """Returns stressor coefficient matrix for the ITC cxc construct."""
        self.S_ITC_cxc = np.dot(np.dot(self.F, np.linalg.inv(np.diag(self.g))), np.dot(
            self.V.transpose(), np.linalg.inv(np.diag(self.q))))
        return self.S_ITC_cxc


    """ Aggregation Constructs"""

    def pc_agg(self, keep_size=True, return_flows=True):
        """Performs Partition Aggregation Construct of SuUT inventory

        Parameters
        ----------
        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN because product not
                   produced and not consumed

        Returns
        --------
        A : Normalized technical requirements [com,com]
        S : Normalized, constructed emissions [ext, com]
        nn_in : filter to remove np.empty rows in A or Z [com]
        nn_out : filter to remove np.empty columns in A or Z [com]
        Z : constructed intermediate flow matrix [com,com]
        F_con : Constructed emissions [ext,com]

        Depends on
        ----------
        self.U :    Use table [com, ind]
        self.V :    Supply table [com, ind]
        self.PHI :  Partition coefficient [ind, com] (default=np.empty(0))
        OR
        self.PSI :  Properties table [com, properties]
        self.F :    Unallocated emissions [ext, ind] (default=np.empty(0))

        """
        # Default values
        F_con = np.empty(0)
        S = np.empty(0)

        # Partitioning properties and coefficients
        if self.PHI is None:
            self.__pa_coeff()

        # Partitioning of product flows
        Z = (self.__sU * self.__sPHI).toarray()  # <-- eq:PCagg

        # Partitioning of environmental extensions
        if self.F is not None:
            F_con = (self.__sF * self.__sPHI).toarray()  # <-- eq:PCEnvExt

        (A, S, nn_in, nn_out) = matrix_norm(Z, self.V, F_con, keep_size)

        if not return_flows:
            Z = np.empty(0)
            F_con = np.empty(0)

        return (A, S, nn_in, nn_out, Z, F_con)


    def psc_agg(self, keep_size=True, return_flows=False):
        """Performs Product Substitution aggregation Construct

        Args
        ----
        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN

        return_flows: Return unnormalized flows in addition to
                                   normalized coefficients.

        Returns
        --------
        A : Normalized technical requirements [com,com]
        A_main : Normalized technical input requirements [com, com]
        A_byprod : Normalized technical byproduct generation [com, com]
        A = A_main - A_byprod, where all elements in A_main and A_byprod are always nonnegative
        S : Normalized, constructed emissions [ext, com]
        nn_in : filter to remove np.empty rows in A or Z [com]
        nn_out : filter to remove np.empty columns in A or Z [com]

        If return_flows is True :
            Z : constructed intermediate flow matrix [com,com]
            F_con : Constructed emissions [ext,com]

        Depends on
        ----------
        self.U:     Use table [com, ind]
        self.V:     Supply table [com, ind]
        self.E_bar: 0 or 1 mapping of primary commodities to industries
                    [com,ind]
        self.Xi :   substitution table [com,com]
        self.F :    Unallocated emissions [ext, ind] (default=np.empty(0))
        """

        # Default values
        F_con = np.empty(0)
        S = np.empty(0)

        # Construction of Product Flows  # <-- eq:PSCagg
        # ------------- sparse matrix start ---------------------
        #        Z = ((self.__sU - self.__sXi * self.__sV_tild) * self.__sE_bar.T
        #            ).toarray()
            
        Z_main = (self.__sU * self.__sE_bar.T
            ).toarray()    
            
        Z_byprod = (self.__sXi * self.__sV_tild * self.__sE_bar.T
            ).toarray()     
            
        # ------------- sparse matrix end ---------------------            
            
        Z = Z_main - Z_byprod    


        # Allocation of Environmental Extensions
        if self.F is not None:
            F_con = (self.__sF * self.__sE_bar.T).toarray()  #eq:NonProdBalEnvExt

        # Normalizing
        (A, S, nn_in, nn_out)        = matrix_norm(Z, self.V_bar, F_con, keep_size)
        (A_main, S, nn_in, nn_out)   = matrix_norm(Z_main, self.V_bar, F_con, keep_size)
        (A_byprod, S, nn_in, nn_out) = matrix_norm(Z_byprod, self.V_bar, F_con, keep_size)

        # Return allocated values
        if return_flows:
            logging.warning("Unnormalized flows (Z, F_con) for this construct"
                            " may differ from calculated flows for a given"
                            " final demand")
        else:
            Z = np.empty(0)
            F_con = np.empty(0)

        return (A, A_main, A_byprod, S, nn_in, nn_out, Z, F_con)


    def aac_agg(self, nmax=np.Inf, res_tol=0, keep_size=True, return_flows=True):
        """ Alternative Activity aggregation Construct of SuUT inventory

        Args
        ----
        nmax : maximum number of iterative loops for defining A_gamma
               (default=Inf)
        res_tol: maximum residual acceptable in defining A_gamma (default 0)
        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN

        Returns
        --------
        A : Normalized technical requirements [com,com]
        S : Normalized, constructed emissions [ext, com]
        nn_in : filter to remove np.empty rows in A or Z [com]
        nn_out : filter to remove np.empty columns in A or Z [com]
        Z : constructed intermediate flow matrix [com,com]
        F_con : Constructed emissions [ext,com]

        Depends on
        ----------
        self.U : Use table [com, ind]
        self.V : Supply table [com, ind]
        self.E_bar : 0 or 1 mapping of primary commodities to industries
                     [com,ind]
        Gamma : mapping of alternate activity for each commodity
                [ind,com]
        self.F : Unallocated emissions [ext, ind] (default=np.empty(0))

        """
        # Default values
        F_con = np.empty(0)
        S = np.empty(0)

        # Basic variables

        # Calculate competing technology requirements
        A_gamma, F_gamma = self.__alternate_tech(nmax=nmax, res_tol=res_tol)

        # Allocation step
        sA_gamma = sp.csc_matrix(A_gamma)

        # Perform construct
            #------------------ matrix  notation start ------------------
        Z = ((self.__sU - sA_gamma * self.__sV_tild) * self.__sE_bar.T
            #------------------ matrix  notation end --------------------
            ).toarray() + A_gamma * self.V_tild.sum(1)  # <-- eq:AACagg


        # Partitioning of environmental extensions
        if self.F is not None:
            sF_gamma = sp.csc_matrix(F_gamma)
                    #------------------ matrix  notation start ---------------
            F_con = ((self.__sF - sF_gamma * self.__sV_tild) * self.__sE_bar.T
                    #------------------ matrix  notation end -----------------
                    ).toarray() + F_gamma * self.V_tild.sum(1)  # eq:AACEnvExt

        # Normalize and return
        (A, S, nn_in, nn_out) = matrix_norm(Z, self.V, F_con, keep_size)

        if not return_flows:
            Z = np.empty(0)
            F_con = np.empty(0)

        return (A, S, nn_in, nn_out, Z, F_con)

    def lsc(self, keep_size=True, return_flows=False):
        """ Performs Lump-sum aggregation Construct of SuUT inventory

        Args
        ----

        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN
        return_flows: Return unnormalized flows in addition to
                                   normalized coefficients.

        Returns
        --------
        A : Normalized technical requirements [com,com]
        S : Normalized, constructed emissions [ext, com]
        nn_in : filter to remove np.empty rows in A or Z [com]
        nn_out : filter to remove np.empty columns in A or Z [com]

        If return_flows is True :
            Z : constructed intermediate flow matrix [com,com]
            F_con : Constructed emissions [ext,com]

        Depends on
        ----------
        self.U : Use table [com, ind]
        self.V : Supply table [com, ind]
        self.E_bar : 0 or 1 mapping of primary commodities to industries
                     [com,ind]
        self.F : Unallocated emissions [ext, ind] (default=np.empty(0))

        """
        # Default values
        F_con = np.empty(0)
        S = np.empty(0)

        #------------start sparse matrix notation----------------------------
        # Allocation of Product Flows
        Z = (self.__sU * self.__sE_bar.T).toarray()  # <-- eq:LSCagg

        # Allocation of Environmental Extensions
        if self.F is not None:
            F_con = (self.__sF * self.__sE_bar.T).toarray()  # eq:NonProdBalEnvExt
        #------------ end sparse matrix notation----------------------------

        # Normalizing
        V_dd = self.E_bar * self.g  # <-- eq:LSCagg
        (A, S, nn_in, nn_out) = matrix_norm(Z, V_dd, F_con, keep_size)

        # Return allocated values
        if return_flows:
            logging.warning("Unnormalized flows (Z, F_con) for this construct"
                            " may differ from calculated flows for a given"
                            " final demand")
        else:
            Z = np.empty(0)
            F_con = np.empty(0)

        return (A, S, nn_in, nn_out, Z, F_con)

    def itc(self, keep_size=True, return_flows=True):
        """Performs Industry Technology Construct of SuUT inventory

        Args
        ----
        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN

        Returns
        --------
        A : Normalized technical requirements [com,com]
        S : Normalized, constructed emissions [ext, com]
        nn_in : filter to remove np.empty rows in A or Z [com]
        nn_out : filter to remove np.empty columns in A or Z [com]
        Z : constructed intermediate flow matrix [com,com]
        F_con : Constructed emissions [ext,com]

        Depends on
        ----------
        self.U : Use table [com, ind]
        self.V : Supply table [com, ind]
        self.F : Unallocated emissions [ext, ind] (default=np.empty(0))
        """
        # Default values
        F_con = np.empty(0)
        S = np.empty(0)

        # ------------- sparse matrix start ---------------------
        V = self.__sV
        Z = (sp.csc_matrix(self.U * _one_over(self.g)) * V.T).toarray()  # eq:itc

        if self.F is not None:
            F_con = (sp.csc_matrix(self.F * _one_over(self.g)) * V.T).toarray()
        # ------------- sparse matrix end ---------------------

        (A, S, nn_in, nn_out) = matrix_norm(Z, self.V, F_con, keep_size)

        if not return_flows:
            Z = np.empty(0)
            F_con = np.empty(0)

        return (A, S, nn_in, nn_out, Z, F_con)


    def esc(self, keep_size=True, return_flows=True):
        """ Performs European System Construct on SuUT inventory

        Args
        ----
        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN

        Depends on
        ----------
        self.U :     Use table [com, ind]
        self.V :     Supply table [com, ind]
        self.E_bar:  0 or 1 mapping of primary production [com,ind]
                     (if absent and system square, assume identity matrix)
        selfself..G :     Unallocated emissions [ext, ind] (default=np.empty(0))

        Returns
        --------
        A:      Normalized technical requirements [com,com]
        S:      Normalized, constructed emissions [ext, com]
        nn_in:  filter to remove np.empty rows in A or Z [com]
        nn_out: filter to remove np.empty columns in A or Z [com]
        Z:      constructed intermediate flow matrix [com,com]
        F_con:  Unnormalized, constructed emissions [ext,com]

        """
        # Default output
        F_con = np.empty(0)
        S = np.empty(0)

        # When no explicit designation of primary production, assume it is on
        # the diagonal if the supply table is square

        # ---------------start sparse matrix -----------------------------
        # Construct product flows
        Z = (self.__sU * self.__sE_bar.T).toarray()

        # Construct extension flows
        if self.F is not None:
            F_con = (self.__sF * self.__sE_bar.T).toarray()  # eq:ESCEnvExt

        # ---------------end sparse matrix -----------------------------

        # Normalize and return
        A, S, nn_in, nn_out = matrix_norm(Z, self.V, F_con, keep_size)

        if not return_flows:
            Z = np.empty(0)
            F_con = np.empty(0)

        return A, S, nn_in, nn_out, Z, F_con



    def ctc(self, return_flows=True):
        """Performs Commodity Technology Construct of SuUT inventory

        Args
        ----
        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN

        Returns
        --------
        A : Normalized technical requirements [com,com]
        S : Normalized, constructed emissions [ext, com]
        nn_in : filter to remove np.empty rows in A or Z [com]
        nn_out : filter to remove np.empty columns in A or Z [com]
        Z : constructed intermediate flow matrix [com,com]
        F_con : Constructed emissions [ext,com]

        Depends on
        ----------
        self.U : Use table [com, ind]
        self.V : Supply table [com, ind]
        self.F : Unallocated emissions [ext, ind] (default=np.empty(0))

        """
        # Default values
        F_con = np.empty(0)
        Z = np.empty(0)
        S = np.empty(0)

        inv_V = sl.inv(self.__sV)
        A = (self.__sU * inv_V).toarray()  # <-- eq:ctc

        if self.F is not None:
            S = (self.__sF * inv_V).toarray()


        if return_flows:
            Z = A * self.q
            F_con = S * self.q  # <--eq:CTCEnvExt

        # Just get filters
        __, __, nn_in, nn_out = matrix_norm(A, self.V, just_filters=True)


        return (A, S, nn_in, nn_out, Z, F_con)


    def btc(self, keep_size=True, return_flows=True):
        """Performs Byproduct Technology Construct of SuUT inventory

        Args
        ----
        keep_size: by default, keep all rows and columns in normalized A-matrix
                   even if some rows/columns should be NaN

        Returns
        --------
        A : Normalized technical requirements [com,com]
        S : Normalized, constructed emissions [ext, com]
        nn_in : filter to remove np.empty rows in A or Z [com]
        nn_out : filter to remove np.empty columns in A or Z [com]
        Z : constructed intermediate flow matrix [com,com]
        F_con : Constructed emissions [ext,com]

        Parameters
        ----------
        self.U : Use table [com, ind]
        self.V : Supply table [com, ind]
        self.E_bar : 0 or 1 mapping of primary commodities to industries
                     [com,ind]
        self.F : Unallocated emissions [ext, ind] (default=np.empty(0))

        """

        # Default values
        F_con = np.empty(0)
        S = np.empty(0)

        # -------------------sparse matrix ----------------------------
        # The construct
        Z = ((self.__sU - self.__sV_tild) * self.__sE_bar.T).toarray()

        if self.F is not None:
            # eq:NonProdBalEnvExt
            F_con = (self.__sF * self.__sE_bar.T).toarray()
        # -------------------sparse matrix ----------------------------

        (A, S, nn_in, nn_out) = matrix_norm(Z, self.V_bar, F_con, keep_size)

        if not return_flows:
            Z = np.empty(0)
            F_con = np.empty(0)

        return (A, S, nn_in, nn_out, Z, F_con)

    """ HELPER/HIDDEN METHODS"""

    def __pa_coeff(self):
        """Calculates partition coefficients from supply and properties table

        Parameters
        ----------
        self.V : Supply table [com, ind]
        self.PSI : Properties table [com, properties]

        Generates
        --------
        self.PHI : Partition coefficient [ind, com (default=np.empty(0))]
            Properties in self.PSI should be intensive properties (e.g. energy
            density, price etc., not extensive properties such as energy
            content, value, or mass

        """
        # Calculate total amount of the partition property that is output by
        # each industry (total mass output for all commodities supplied by
        # industry J)
        denominator = ddiag(self.V.T.dot(self.PSI))

        # Calculate the share of this total output of property that is mediated
        # by each output (share of total mass output by ind. J that happens via
        # commodity j.
        self.PHI = np.linalg.inv(denominator).dot(self.V.T * self.PSI.T)



    def __alternate_tech(self, nmax=np.Inf, lay=None, res_tol=1e-30):
        """Compilation of Alternate Technologies for use in AAA and AAC models

        Args
        ----
        nmax :      maximum number of iterations, as this search for
                    alternative technologies is not garanteed to suceed
        res_tol:    maximum residual acceptable in defining A_gamma (default 0)

        Generates
        ---------
        A_gamma :   the selected alternative technology that will be assumed
                    for each secondary production
        S_gamma :   the selected alternative technology (emissions) that will
                    be assumed for each secondary production

        Depends on
        ----------
        self.U :    Use table [com, ind]
        self.F :    Unallocated emissions [ext, ind] (default=np.empty(0))
        self.V :    Supply table [com, ind]
        self.E_bar: mapping of primary commodities to industries [com,ind]
        Gamma :     mapping of alternate producer for each commodity [ind,com]


        """

        # Basic variables
        com = self.V.shape[0]
        traceable = False
        org = 1 # check on this
        Gamma = self.Gamma

        # If a property layer is defined for Gamma, then evaluate unit
        # conversion
        if Gamma.ndim == 3:
            if lay is None:
                raise TypeError('expected a value for lay')
            s = Gamma.shape
            tmp = np.zeros((s[0], s[2]))
            for i in range(Gamma.shape[1]):
                tmp += self.__diaginv(lay[i, :].dot(self.E_bar)).dot(
                       Gamma[:, i, :]).dot(ddiag(lay[i, :]))
            Gamma = tmp

        so = np.array(np.sum(self.V != 0, 0) == 1, dtype=int)
        mo = np.array(np.sum(self.V != 0, 0) != 1, dtype=int)

        invg = _one_over(self.g)

        #================ Sparse Matrix Section ==========================
        M = sp.csc_matrix(self.V_tild * _one_over(self.V_bar.sum(0)))
        Gamma = sp.csc_matrix(Gamma)

        # Iteration 0: Prepare summation term used in definition of A_gamma
        n = 0
        tier = -1 * Gamma * M
        tier_n = sp.identity(tier.shape[0])  #=tier**n
        theSum = Gamma.copy()                #=identity*Gamma = tier_n * Gamma
        n = n + 1
        res = theSum.sum()

        # Iterations 1 to nmax
        while ((res > res_tol) or (res < 0)) and (n <= nmax):
            tier_n = tier_n * tier
            term = tier_n * Gamma
            theSum = theSum + term
            n += 1
            res = term.sum()
            print("residual: {:+.2e}".format(res))
        #theSum = theSum.toarray()
        print("number of iterations:{}".format(n))
        #================ Sparse Matrix Section ==========================

        def apply_to_requirements(X):
            """ Apply to X, representing either U or F """
            if not traceable:
                B_so = (X * invg) * so
                N_mo = (X * _one_over(self.V_bar.sum(0))) * mo
                #------------start sparse matrix----------
                requirements = sp.csc_matrix(B_so + N_mo)
                X_gamma = (requirements * theSum).toarray()
                #------------end sparse matrix----------
            else:
                X_gamma = np.zeros([org, com, com])
                for I in range(org):
                    Bo_so = (X[I, :, :] * invg) * so
                    No_mo = (X[I, :, :] * _one_over(self.V_bar.sum(0))) * mo
                    #------------start sparse matrix----------
                    requirements = sp.csc_matrix(Bo_so + No_mo)
                    X_gamma[I, :, :] = (requirements * theSum).toarray()
                    #------------start sparse matrix----------
            return X_gamma

        A_gamma = apply_to_requirements(self.U)
        if self.F is not None:
            S_gamma = apply_to_requirements(self.F)
        else:
            S_gamma = np.empty(0)

        return A_gamma, S_gamma

    @property
    def __sU(self):
        """ Returns sparse version of self.U """
        return sp.csc_matrix(self.U)

    @property
    def __sV(self):
        return sp.csc_matrix(self.V)
        """ Returns sparse version of self.V """
    @property
    def __sV_bar(self):
        """ Returns sparse version of self.V_bar """
        return sp.csc_matrix(self.V_bar)

    @property
    def __sV_tild(self):
        """ Returns sparse version of self.V_tild """
        return sp.csc_matrix(self.V_tild)

    @property
    def __sF(self):
        """ Returns sparse version of self.F """
        return sp.csc_matrix(self.F)

    @property
    def __sXi(self):
        """ Returns sparse version of self.Xi """
        return sp.csc_matrix(self.Xi)

    @property
    def __sPHI(self):
        """ Returns sparse version of self.PHI """
        return sp.csc_matrix(self.PHI)

    @property
    def __sE_bar(self):
        """ Returns sparse version of self.E_bar """
        if self.E_bar is None and (self.V.shape[0] == self.V.shape[1]):
            logging.warning("Assuming primary production is on diagonal")
            return sp.eye(self.V.shape[0])
        else:
            return sp.csc_matrix(self.E_bar)

    def __diaginv(self, x):
        """Diagonalizes a vector and inverses it, even if it contains zero values.

        * Element-wise divide a vector of ones by x
        * Replace any instance of Infinity by 0
        * Diagonalize the resulting vector

        Parameters
        ----------
        x : vector to be diagonalized

        Returns
        --------
        y : diagonalized and inversed vector
           Values on diagonal = 1/coefficient, or 0 if coefficient == 0

        """
        y = np.ones(len(x)) / x
        y[y == np.Inf] = 0
        return ddiag(y)
#############################################################################
# Helper functions outside object
def aggregate_regions_vectorised(X, AV=None, axis=None, regions=None):
    """ In an array X, aggregate regions, along either axis or both

    Args
    ----

    X:      Array to be aggregated.
    AV:     Aggregation vector, specifying which region to aggregate with which
                If AV is None and regions is not, assume that all regions are
                to be aggregated as one.
                AV[0] = 5 means that region 1 (index+1) should be aggregated
                        with region 5.
    axis:   axis [0 or 1] along which X should be aggregated. If None, aggregate
                along both.
    regions: Number of regions. Used to build an aggregation vector of ones if
                AV is None.

    Needless to say, the aggregated axis must be multiple of the number of
    regions.

    Returns
    ------
    X  (aggregated)

    """
    # Define aggregation vector if none specified. Assume all regions are to be
    # aggregated as one.
    if AV is None:
        AV = np.ones(regions, dtype=int)

    # Use local variables for this method
    # Generate region correspondence matrix for aggregation
    pos = np.zeros((len(AV), max(AV)), dtype=int)
    pos[np.arange(len(AV)), AV - 1] = 1

    # If somewhat sparse, treat as sparse matrix, otherwise stick with numpy
    if ((X == 0).sum() / X.size) > 0.50:
        sparse = True
        X = sp.csc_matrix(X)
    else:
        sparse = False

    if axis == 0 or axis is None:
        # Generate aggregation matrix
        entries_per_region = int(X.shape[0]/len(AV))
        agg = np.kron(pos, np.eye(entries_per_region, dtype=int))

        if sparse:
            #-----------start-sparse matrix----------------
            agg = sp.csc_matrix(agg)
            X = agg.T * X
            #-------------end-sparse matrix----------------
        else:
            # Aggregate rows
            X = agg.T.dot(X)

    if axis == 1 or axis is None:
        # Generate aggregation matrix
        entries_per_region = int(X.shape[1]/len(AV))
        agg = np.kron(pos, np.eye(entries_per_region, dtype=int))

        # Aggregate columns
        if sparse:
            agg = sp.csc_matrix(agg)
            X = X * agg
        else:
            X = X.dot(agg)

    if sparse:
        X = X.toarray()

    return X

def aggregate_within_regions(X, regions, axis=None):
    """ Aggregate the products or industries within each regions for mrSUT

    For multi-regional SUT, aggregate rows, columns or both such that
    there remains only one entry per region along chosen axis. The number
    of regions is specified by regions.

    Args
    ----
        X: a numpy array of appropriate dimensions
        regions: number of regions
        axis: 0 to aggregate rows, 1 for columns, None for all.
              + default: None
    Returns
    -------
        X (aggregated)

    """
    if axis == 0 or axis is None:
        # Generate aggregation matrix
        entries_per_regions = int(X.shape[0] / regions)
        e = np.ones((entries_per_regions, 1))
        agg = np.kron(np.eye(regions), e)

        # Aggregate rows to one entry per region
        X = agg.T.dot(X)

    if axis == 1 or axis is None:
        # Generate aggregation matrix
        entries_per_regions = int(X.shape[1] / regions)
        e = np.ones((entries_per_regions, 1))
        agg = np.kron(np.eye(regions), e)

        # Aggregate columns to one entry per region
        X = X.dot(agg)

    return X

def matrix_norm(Z, V, F_con=np.empty(0), keep_size=False, just_filters=False):
    """ Normalizes a flow matrices, even if some rows and columns are null

    Processes product flows (Z) and environmental extensions (F_con).

    Normalizes columns of products that are indeed supplied (non-null supply),
    as identified by nn_out. Remove all other columns.

    Keep only rows of products that are actually produced *OR* of products that
    are consumed (in spite of not being produced...), as identified by nn_in.
    Remove all other rows.

    Parameters
    ----------
    Z : Flow matrix to be normalized
        dimensions : [com, com] | [com, ind, com] | [ind,com,ind,com]
    V : Production volume with which flows are normalized
        [com, ind]

    F_con: Allocated or construced but unnormalized environmental extensions
           [str, com] | [str, ind, com]

    keep_size: Do not remove empty rows and columns from A, leave with zeros

    just_filters: Don't normalize anything, just return nn_in and nn_out, leave
                  rest empty


    Returns
    -------
    A : Normalized flow matrix, without null/nan rows and nan columns
    S : Normalized extensions, by default without nan columns
    nn_in : filter applied to rows (0 for removed rows, 1 for kept rows)
    nn_out : filter applied to cols (0 for removed cols, 1 for kept cols)

    """

    # Collapse dimensions
    def _collapse_dims(x, first2dimensions=False):
        """Collapse 3-d or 4-d array in two dimensions

        Parameters
        ----------
        x : 3d or 4d array to be collapsed

        first2dimensions : Boolean : For 3d array, should the last two dimensions
            be flattened together (default) or should the first two be
            flattened together instead (=true)?

        Returns
        --------
        z : Flatened 2d array

        """

        s = x.shape
        if x.ndim == 4:
            z = x.reshape((s[0] * s[1], s[2] * s[3]))
        elif x.ndim == 3:
            if first2dimensions:
                z = x.reshape((s[0] * s[1], s[2]))
            else:
                z = x.reshape((s[0], s[1] * s[2]))
        elif x.ndim == 2:
            print('Already in 2-dimensional, pass')
            z = x
        else:
            print('PROBLEM? ndim(Y) = {}'.format(x.ndim))
        return z

    if Z.ndim > 2:
        Z = _collapse_dims(Z)
    if F_con.ndim > 2:
        F_con = _collapse_dims(F_con)

    # Basic Variables
    com = np.size(V, 0)
    ind = np.size(V, 1)
    #com2 = np.size(Z, 0)

    # Total production (q, q_tr) and intermediate consumptin (u) vectors
    q = np.sum(V, 1)
    u = np.sum(Z, 1)
    if np.max(Z.shape) == com * ind:
        q_tr = np.zeros(ind * com)
        for i in range(ind):
            q_tr[i * com:(i + 1) * com] = V[:, i]

    # Column filtering: keep only commodities that are produced (cannot
    # produce a normalized recipe for something that has not production volume)
    if np.size(Z, 1) == com:
        nn_out = q != 0
    elif np.size(Z, 1) == com * ind:
        nn_out = q_tr != 0
    else:
        raise Exception("Mismatched columns between Z and V")

    # Row Filtering: Preserve only commodities if they are produced *OR* if
    # they are used (even if they are not produced, to get the recipe right).
    if np.size(Z, 0) == com:
        nn_in = (abs(q) + abs(u)) != 0
    elif np.size(Z, 0) == com * ind:
        nn_in = (abs(q_tr) + abs(u)) != 0
    else:
        raise Exception("Mismatched rows between Z and V")


    if just_filters:
        # just want filters, the rest empty
        A = np.empty(0)
        S = np.empty(0)
    else:
        # Apply filters and normalize

        # remove empty entried, diagonalize, inverse...
        if np.size(Z, 1) == com:
            q_inv = _one_over(q[nn_out])
        else:
            q_inv = _one_over(q_tr[nn_out])

        # and use to normalize product and stressor flows.
        A = Z[nn_in, :][:, nn_out] * q_inv
        if F_con.size:
            S = F_con[:, nn_out] * q_inv
        else:
            S = np.empty(0)

        # Restore size if need be
        if keep_size:
            A = restore_size(A, nn_in, nn_out)
            S = restore_size(S, nn_out=nn_out)

    # Return
    return (A, S, nn_in, nn_out)

def restore_size(X, nn_in=None, nn_out=None):
    """ Accepts an array and restores its size with empty rows and columns"""

    # Make sure we have somthing significant
    if not X.size:
        return X

    # Restore  rows
    if nn_in is not None:
        X0 = np.zeros((len(nn_in), X.shape[1]))
        X0[nn_in, :] = X
    else:
        X0 = X

    # Restore cols
    if nn_out is not None:
        X1 = np.zeros((X0.shape[0], len(nn_out)))
        X1[:, nn_out] = X0
    else:
        X1 = X0

    return X1



def _one_over(x):
    """Simple function to invert each element of vector. if 0, stays 0, not Inf

    * Element-wise divide a vector of ones by x
    * Replace any instance of Infinity by 0

    Parameters
    ----------
    x : vector to be diagonalized

    Returns
    --------
    y : inversed vector
       Values = 1/coefficient, or 0 if coefficient == 0

    """
    y = 1 / x
    y[y == np.Inf] = 0
    return y

def ddiag(a, nozero=False):
    """ Robust diagonalization : always put selected diagonal on a diagonal!

    This small function aims at getting a behaviour closer to the
    mathematical "hat", compared to what np.diag() can delivers.

    If applied to a vector or a 2d-matrix with one dimension of size 1, put
    the coefficients on the diagonal of a matrix with off-diagonal elements
    equal to zero.

    If applied to a 2d-matrix (with all dimensions of size > 1), replace
    all off-diagonal elements by zeros.

    Parameters
    ----------
    a : numpy matrix or vector to be diagonalized

    Returns
    --------
    b : Diagonalized vector

    Raises:
       ValueError if a is more than 2dimensional

    See Also
    --------
        diag
    """

    # If numpy vector
    if a.ndim == 1:
        b = np.diag(a)

    # If numpy 2d-array
    elif a.ndim == 2:

        #...but with dimension of magnitude 1
        if min(a.shape) == 1:
            b = np.diag(np.squeeze(a))

        # ... or a "true" 2-d matrix
        else:
            b = np.diag(np.diag(a))

    else:
        raise ValueError("Input must be 1- or 2-d")

    # Extreme case: a 1 element matrix/vector
    if b.ndim == 1 & b.size == 1:
        b = b.reshape((1, 1))

    if nozero:
        # Replace offdiagonal zeros by nan if desired
        c = np.empty_like(b) *  np.nan
        di = np.diag_indices_from(c)
        c[di] = b.diagonal()
        return c
    else:
        # A certainly diagonal vector is returned
        return b


"""
End of file
"""
