# Copyright (C) 2010 Marco Giusti

# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:

# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


"""Collect tests looking at Emacs local variables declaration format.

Look at the first lines (default to 2) of modules and collect test
cases. To add a test case just add a line in the format::

    # -*- test-case-name: test_file[,test_file...] -*-

You can add several test modules just splitting them by comas.
This plugin tries some euristic to find the test files so you can
express it as absolute paths::

    /foo/bar/tests/test_bar.py  # test module
    /foo/bar/tests              # test package

as paths relative to sys.path:

    bar/tests/test_bar.py
    bar/tests

or dotted strings::

    bar.tests.test_bar
    bar.tests


How to use it in vim::

    :autocmd FileType python map <F5> <Esc>:w<CR>:!py.test --testmodule %<CR>

This save the file and run the module specific tests.

PS. The parsing functions are taken from twisted trial. They are here
to not depends expressly on twisted.
"""

import sys
import os


def pytest_addoption(parser):
    group = parser.getgroup('collect')
    group.addoption('--testmodule',
                    dest='testmodule',
                    action='store_true',
                    default=False,
                    help="Grep files for test cases (-*- test-case-name)")


def pytest_configure(config):
    if config.getvalue('testmodule'):
        nargs = []
        for arg in config.args:
            nargs.extend(exists(path) for path in getTestModules(arg))
        nargs = filter(None, nargs)
        if nargs:
            config._setargs(nargs)


def exists(name):
    if name.startswith(os.path.sep):
        if os.path.exists(name):
            return name
    for path in sys.path:
        fullpath = os.path.join(path, name)
        if os.path.exists(fullpath):
            return fullpath
        elif os.path.exists(fullpath + '.py'):
            return fullpath + '.py'
        name = name.replace('.', os.path.sep)
        fullpath = os.path.join(path, name + '.py')
        if os.path.exists(fullpath):
            return fullpath


# The following lines are copyrighted by Twisted Matrix Laboratories.
# Copyright (c) 2001-2007 Twisted Matrix Laboratories.

def _parseLocalVariables(line):
    """Accepts a single line in Emacs local variable declaration format and
    returns a dict of all the variables {name: value}.
    Raises ValueError if 'line' is in the wrong format.

    See http://www.gnu.org/software/emacs/manual/html_node/File-Variables.html
    """
    paren = '-*-'
    start = line.find(paren) + len(paren)
    end = line.rfind(paren)
    if start == -1 or end == -1:
        raise ValueError("%r not a valid local variable declaration" % (line,))
    items = line[start:end].split(';')
    localVars = {}
    for item in items:
        if len(item.strip()) == 0:
            continue
        split = item.split(':')
        if len(split) != 2:
            raise ValueError("%r contains invalid declaration %r"
                             % (line, item))
        localVars[split[0].strip()] = split[1].strip()
    return localVars


def loadLocalVariables(filename):
    """Accepts a filename and attempts to load the Emacs variable declarations
    from that file, simulating what Emacs does.

    See http://www.gnu.org/software/emacs/manual/html_node/File-Variables.html
    """
    f = file(filename, "r")
    lines = [f.readline(), f.readline()]
    f.close()
    for line in lines:
        try:
            return _parseLocalVariables(line)
        except ValueError:
            pass
    return {}


def getTestModules(filename):
    testCaseVar = loadLocalVariables(filename).get('test-case-name', None)
    if testCaseVar is None:
        return []
    return testCaseVar.split(',')
