# -*- test-case-name: bloggerfs.test_bloggerfs -*-

# bloggerfs.py - blogger file system in userspace using FUSE
# Copyright (C) 2010 Marco Giusti
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
# MA 02111-1307, USA


# rfc3339.py -- Implementation of the majority of RFC 3339 for python.
# Copyright (c) 2008, 2009, 2010 LShift Ltd. <query@lshift.net>


# gdata - Python client library for Google data APIs
# Copyright (C) 2009 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import os
import stat
import errno
import time
import datetime
import logging
import inspect
import functools

import fuse
import gdata.blogger.client
import atom.core
import atom.data
from zope.interface import interface, declarations, implements
from zope.interface import Interface, Attribute
from zope.interface.adapter import AdapterRegistry

import rfc3339

__author__ = 'Marco Giusti'
__copyright__ = 'Copyright 2010, Marco Giusti'
__license__ = 'LGPL'
__version__ = '0.3'
__email__ = 'marco.giusti@gmail.com'


__all__ = ['ALLOW_DUPLICATES', 'AccountNode', 'BlogNode', 'BloggerFS',
           'CommentNode', 'DirEntry', 'DirNode', 'ElementNode', 'EntryNode',
           'FeedEntryNode', 'FileNode', 'DictNode', 'publishedFactory',
           'updatedFactory', 'registerAdapter2', 'registerAdapter',
           'IDirNode', 'IFileNode', 'INode', 'IPublished', 'IUpdated',
           'ListNode', 'Node', 'PostNode', 'TextNode', 'TimeOutMixin',
           'AddCommentNode', 'AddLabelNode',
           '__author__',
           '__copyright__', '__email__', '__license__', '__version__',
           'client', 'getType', 'main']


class NullHandler(logging.Handler):

    def emit(self, record):
        pass


fuse.fuse_python_api = (0, 2)
_logger = logging.getLogger('bloggerfs')
_logger.setLevel(logging.WARNING)
_debug = _logger.debug
_info = _logger.info
_warn = _logger.warn
_error = _logger.error
_critical = _logger.critical
_exception = _logger.exception
client = None


def getmembers(obj, predicate=None):
    """Return all members of an object as (name, value) pairs sorted by name.
    Optionally, only return members that satisfy a given predicate."""
    if inspect.isclass(obj):
        mro = (obj,) + inspect.getmro(obj)
    else:
        mro = (obj,)
    results = []
    for key in dir(obj):
        # First try to get the value via __dict__. Some descriptors don't
        # like calling their __get__ (see bug #1785).
        for base in mro:
            if key in base.__dict__:
                value = base.__dict__[key]
                break
        else:
            value = getattr(obj, key)
        if not predicate or predicate(value):
            results.append((key, value))
    results.sort()
    return results


inspect.getmembers = getmembers     # monkey patch inspect


# Taken from Twisted

globalRegistry = AdapterRegistry()
ALLOW_DUPLICATES = False


def _registered(registry, required, provided):
    """
    Return the adapter factory for the given parameters in the given
    registry, or None if there is not one.
    """
    return registry.registered([required], provided)


def registerAdapter(adapterFactory, origInterface, *interfaceClasses):
    """Register an adapter class.

    An adapter class is expected to implement the given interface, by
    adapting instances implementing 'origInterface'. An adapter class's
    __init__ method should accept one parameter, an instance implementing
    'origInterface'.
    """
    self = globalRegistry
    assert interfaceClasses, "You need to pass an Interface"
    global ALLOW_DUPLICATES

    # deal with class->interface adapters:
    if not isinstance(origInterface, interface.InterfaceClass):
        origInterface = declarations.implementedBy(origInterface)

    for interfaceClass in interfaceClasses:
        factory = _registered(self, origInterface, interfaceClass)
        if factory is not None and not ALLOW_DUPLICATES:
            raise ValueError("an adapter (%s) was already registered." %
                             (factory, ))
    for interfaceClass in interfaceClasses:
        self.register([origInterface], interfaceClass, '', adapterFactory)


# add global adapter lookup hook for our newly created registry
def _hook(iface, ob, lookup=globalRegistry.lookup1):
    factory = lookup(declarations.providedBy(ob), iface)
    if factory is None:
        return None
    else:
        return factory(ob)
interface.adapter_hooks.append(_hook)


def _registered2(registry, required1, required2, provided):
    """
    Return the adapter factory for the given parameters in the given
    registry, or None if there is not one.
    """
    return registry.registered([required1, required2], provided)


def registerAdapter2(adapterFactory, (iface1, iface2), *interfaceClasses):
    """Register an adapter class.

    An adapter class is expected to implement the given interfaces, by
    adapting instances implementing ('iface1', 'iface2'). An adapter class's
    __init__ method should accept two parameters, the instances implementing
    'iface1' and 'iface2'.
    """
    self = globalRegistry
    assert interfaceClasses, "You need to pass an Interface"
    global ALLOW_DUPLICATES

    # deal with class->interface adapters:
    if not isinstance(iface1, interface.InterfaceClass):
        iface1 = declarations.implementedBy(iface1)
    if not isinstance(iface2, interface.InterfaceClass):
        iface2 = declarations.implementedBy(iface2)

    for interfaceClass in interfaceClasses:
        factory = _registered2(self, iface1, iface2, interfaceClass)
        if factory is not None and not ALLOW_DUPLICATES:
            raise ValueError("an adapter (%s) was already registered." %
                             (factory, ))
    for interfaceClass in interfaceClasses:
        self.register([iface1, iface2], interfaceClass, '', adapterFactory)


# add global adapter lookup hook for our newly created registry
def _hook2(iface, objs, query=globalRegistry.queryMultiAdapter):
    if isinstance(objs, tuple) and len(objs) == 2:
        return query(objs, iface)
    return None
interface.adapter_hooks.append(_hook2)

# End taken


class INode(Interface):
    """"""


class IFileNode(INode):

    def open(flags):
        """File open operation"""

    def read(size, offset):
        """"""

    def truncate(size):
        """"""

    def write(buf, size):
        """"""

    def release(flags):
        """"""

    def utime(times):
        """"""

    def rename(pathto):
        """"""

    def fsync(isfsyncfile):
        """"""

    def flush():
        """"""

    def fgetattr():
        """Get file attributes.

        Similar to stat().  The 'st_dev' and 'st_blksize' fields are
        ignored.	 The 'st_ino' field is ignored except if the 'use_ino'
        mount option is given."""

    def lock():
        """"""

    def create(mode, file_info):
        """"""


class IDirNode(INode):

    def opendir():
        """"""

    def readdir():
        """"""

    def fsyncdir():
        """"""

    def releasedir():
        """"""


def getType(obj):
    if isinstance(obj, DirNode):
        return stat.S_IFDIR
    return stat.S_IFREG


class DirEntry(fuse.Direntry):

    def __eq__(self, other):
        if other.__class__ != DirEntry:
            return NotImplemented
        if self.name != other.name or self.offset != other.offset or \
           self.type != other.type or self.ino != other.ino:
            return False
        return True

    def __ne__(self, other):
        return not self.__eq__(other)

    def __repr__(self):
        type = self.type == stat.S_IFREG and 'file' or 'dir'
        return '<bloggerfs.DirEntry name="%s", type="%s">' % (self.name, type)


class IUpdated(Interface):

    updated = Attribute('Modification time in seconds from epoch')


class IPublished(Interface):

    published = Attribute('Creation time in seconds from epoch')


def updatedFactory(node):
    return IUpdated(node.parent)

registerAdapter(updatedFactory, INode, IUpdated)


def publishedFactory(node):
    return IPublished(node.parent)

registerAdapter(publishedFactory, INode, IPublished)


class Node(object):
    implements(INode)

    def __init__(self, original, parent):
        self.original = original
        self.parent = parent

    @property
    def mtime(self):
        return IUpdated(self).updated

    @property
    def ctime(self):
        return IPublished(self).published

    def locate(self, path):
        raise NotImplementedError('Locate method not implemented')


class FileNode(Node):
    implements(IFileNode)

    mode = 0440

    def locate(self, path):
        assert path == []
        return self

    @property
    def text(self):
        return self.original

    def getattr(self):
        size = len(self.text)
        st = fuse.Stat(st_mode=stat.S_IFREG | self.mode, st_nlink=1,
                       st_uid=os.getuid(), st_gid=os.getgid(), st_size=size,
                       st_ctime=self.ctime, st_mtime=self.mtime)
        return st

    def read(self, size, offset):
        return self.text[offset:offset+size]

    def write(self, buf, offset):
        # parent = IFileNode(self.parent, None)
        # if parent:
        #     return parent.write(buf, offset, fh=None)
        raise IOError(errno.EACCES, 'Permission denied')

    def open(self, flags):
        return 0

    def truncate(self, size):
        return 0

    def utime(self, times):
        return 0

    def rename(self, pathto):
        return 0

    def fsync(self, isfsyncfile):
        return 0

    def release(self, fh=None):
        return 0

registerAdapter2(FileNode, (str, INode), IFileNode)


class TextNode(FileNode):

    @property
    def text(self):
        if self.original.text is not None:
            return self.original.text
        return ''

registerAdapter2(TextNode, (atom.data.Text, INode), IFileNode)


class DirNode(Node):
    implements(IDirNode)

    mode = 0550
    _children = None

    def getChildren(self):
        return {}

    @property
    def children(self):
        if self._children is None:
            self._children = self.getChildren()
        return self._children

    def getChild(self, k):
        if k in self.children:
            return self.children[k]
        raise IOError(errno.ENOENT, 'No such file or directory')

    def deleteChild(self, k):
        if k not in self.children:
            raise IOError(errno.ENOENT, 'No such file or directory')
        del self.children[k]

    def locate(self, path):
        if path == []:
            return self
        name = path.pop(0)
        child = self.getChild(name)
        return child.locate(path)

    def readdir(self, offset):
        assert offset == 0, 'readdir, offset != 0'
        yield DirEntry('.', type=stat.S_IFDIR)
        yield DirEntry('..', type=stat.S_IFDIR)
        for name, node in self.children.items():
            yield DirEntry(name, type=getType(node))
            # yield IEntry((name, node))

    def getattr(self):
        st = fuse.Stat(st_mode=stat.S_IFDIR | self.mode, st_nlink=2,
                       st_uid=os.geteuid(), st_gid=os.getegid(), st_size=4096,
                       st_ctime=self.ctime, st_mtime=self.mtime)
        return st

    def rmdir(self):
        raise IOError(errno.EACCES, 'Permission denied')


class TimeOutMixin(object):

    seconds = 300
    delta = datetime.timedelta(seconds=seconds)

    def __init__(self, original, parent):
        super(TimeOutMixin, self).__init__(original, parent)
        self._tick = datetime.datetime.now()

    @property
    def children(self):
        now = datetime.datetime.now()
        expire = datetime.timedelta(seconds=self.seconds)
        if self._children is None or now - self._tick > expire:
            self._children = self.getChildren()
            self._tick = now
        return self._children


class ListNode(DirNode):

    def getChildren(self):
        return dict((str(i), INode((el, self)))
                    for i, el in enumerate(self.original))

registerAdapter2(ListNode, (list, INode), IDirNode)


class DictNode(DirNode):

    def getChildren(self):
        return dict((str(k), INode((v, self)))
                    for k, v in self.original.iteritems())

registerAdapter2(DictNode, (dict, INode), IDirNode)


class ElementNode(DirNode):

    def getChildren(self):
        children = {}
        for name, member_type in self.original._members:
            if isinstance(member_type, tuple):
                continue

            element = getattr(self.original, name)
            if element is not None:
                children[name] = INode((element, self))
        if self.original.text:
            children['text'] = INode((self.original.text, self))
        return children

registerAdapter2(ElementNode, (atom.core.XmlElement, INode), IDirNode)


class FeedEntryNode(ElementNode):
    implements(IUpdated)

    _updated = None

    @property
    def updated(self):
        if self._updated is None:
            if self.original.updated:
                try:
                    updated = self.original.updated.text
                    updated = rfc3339.parse_datetime(updated)
                    self._updated = int(time.mktime(updated.timetuple()))
                except:
                    _exception('')
                    self._updated = int(time.time())
            else:
                self._updated = int(time.time())
        return self._updated

    @property
    def blogId(self):
        return self.original.get_blog_id()


class EntryNode(FeedEntryNode):
    implements(IPublished)

    _published = None

    @property
    def published(self):
        if self._published is None:
            if self.original.published:
                try:
                    published = self.original.published.text
                    published = rfc3339.parse_datetime(published)
                    self._published = int(time.mktime(published.timetuple()))
                except:
                    _exception('')
                    self._published = self.mtime
            else:
                self._published = self.mtime
        return self._published

registerAdapter2(EntryNode, (atom.data.Entry, INode), IDirNode)


class CommentNode(EntryNode):

    @property
    def id(self):
        return self.original.get_comment_id()

    def rmdir(self):
        client.delete(self.original)
        self.parent.deleteChild(self.id)

registerAdapter2(CommentNode, (gdata.blogger.data.Comment, INode), INode)


class WriteOnlyMixin(object):

    mode = 0220

    def read(self, size, offset):
        raise IOError(errno.EINVAL, 'Invalid argument')


class AddCommentNode(WriteOnlyMixin, FileNode):

    def write(self, buf, offset):
        try:
            client.add_comment(self.parent.get_blog_id(),
                               self.parent.get_post_id(), buf,
                               body_type='text')
            return len(buf)
        except gdata.client.RequestError, e:
            raise IOError(errno.EIO, e.reason)


class AddLabelNode(WriteOnlyMixin, FileNode):

    def write(self, buf, offset):
        try:
            post = self.parent.original
            post.add_label(buf)
            client.update(post)
            return len(buf)
        except gdata.client.RequestError, e:
            raise IOError(errno.EIO, e.reason)


class PostNode(TimeOutMixin, EntryNode):

    def getChildren(self):
        children = EntryNode.getChildren(self)
        try:
            feed = client.get_post_comments(self.blogId, self.id)
        except gdata.client.RequestError:
            _exception('')
        else:
            comments = dict((e.get_comment_id(), e) for e in feed.entry)
            children['comments'] = INode((comments, self))
            children['add_comment'] = AddCommentNode('', self)
            children['add_label'] = AddLabelNode('', self)
        return children

    def get_post_id(self):
        return self.original.get_post_id()

    def get_blog_id(self):
        return self.original.get_blog_id()

    @property
    def id(self):
        return self.get_post_id()

    def rmdir(self):
        client.delete(self.original)
        self.parent.deleteChild(self.id)

registerAdapter2(PostNode, (gdata.blogger.data.BlogPost, INode), INode)


class BlogNode(TimeOutMixin, FeedEntryNode):

    @property
    def id(self):
        return self.original.get_blog_id()

    def getChildren(self):
        children = FeedEntryNode.getChildren(self)
        try:
            feed = client.get_posts(self.blogId)
        except gdata.client.RequestError:
            _exception('')
        else:
            posts = dict((e.get_post_id(), e) for e in feed.entry)
            children['posts'] = INode((posts, self))
        return children

registerAdapter2(BlogNode, (gdata.blogger.data.Blog, INode), INode)


class AccountNode(FeedEntryNode):
    implements(IPublished)

    _original = None
    published = time.time()

    def __init__(self, original, parent):
        pass

    @property
    def original(self):
        if self._original is None:
            try:
                self._original = client.get_blogs()
            except gdata.client.RequestError, e:
                raise IOError(errno.EIO, e.reason)
        return self._original

    @property
    def parent(self):
        return self

    def getChildren(self):
        return dict((e.get_blog_id(), INode((e, self)))
                    for e in self.original.entry)

registerAdapter2(AccountNode, (gdata.blogger.data.BlogFeed, INode), INode)


def catchException(f):
    @functools.wraps
    def wrapper(*args, **kwds):
        try:
            return f(*args, **kwds)
        except:
            _exception('Error in fuse api')
            raise
    return wrapper


class BloggerFS(fuse.Fuse):

    _fuse_api = set(['getattr', 'readdir', 'read', 'rmdir', 'write', 'open',
                     'release', 'truncate', 'fsync', 'utime', 'rename'])

    def __init__(self, *args, **kwds):
        fuse.Fuse.__init__(self, *args, **kwds)
        global client
        client = gdata.blogger.client.BloggerClient()
        self._account = AccountNode(None, None)

    def main(self):
        if self.parser.fuse_args.mount_expected():
            options = self.cmdline[0]
            if options.debug:
                _logger.setLevel(logging.DEBUG)
            if options.syslog:
                from logging import handlers
                handler = handlers.SysLogHandler()
            elif options.logfile:
                from logging import handlers
                handler = handlers.FileHandler(options.logfile)
            elif options.debug:
                handler = logging.StreamHandler()
            else:
                handler = NullHandler()
            formatter = logging.Formatter('[%(asctime)s] [%(levelname)s] '
                                          '[%(funcName)s]  %(message)s')
            handler.setFormatter(formatter)
            _logger.addHandler(handler)

            if not options.email:
                from gdata.sample_util import authorize_client
                authorize_client(client, service='blogger',
                                 source='bloggerfs',
                                 scopes=['http://www.blogger.com/feeds/'])
            else:
                email = options.email
                if not options.password:
                    import getpass
                    password = getpass.getpass()
                else:
                    password = options.password
                client.client_login(email, password, source='bloggerfs',
                                    service='blogger')
        fuse.Fuse.main(self)

    def _splitpath(self, path):
        return (path.split('/')[1:], [])[path == '/']

    def __getattr__(self, name):
        if name in self._fuse_api:
            def _(path, *args, **kwds):
                try:
                    parts = self._splitpath(path)
                    node = self._account.locate(parts)
                    func = getattr(node, name, None)
                    if func is None:
                        raise IOError(errno.ENOSYS, 'Function %s not '
                                      'implemented' % name)
                    return func(*args, **kwds)
                except:
                    _exception('Error in %s' % name)
                    raise
            return _
        raise AttributeError(name)


def main(argv=None):
    if argv is None:
        import sys
        argv = sys.argv[1:]

    server = BloggerFS()
    server.parser.add_option('-e', '--email', help='Google account email')
    server.parser.add_option('-p', '--password',
                             help='Google account password')
    server.parser.add_option('-b', '--debug', help='Show debugging info',
                             action='store_true')
    server.parser.add_option('--syslog', action='store_true',
                             help='Log messages to syslog')
    server.parser.add_option('-l', '--logfile', help='Log messages to file')
    server.parse(args=argv, errex=1)
    server.main()


if __name__ == '__main__':
    main()
