# Copyright (c) 2014, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

import stix
import stix.bindings.exploit_target as exploit_target_binding
from stix.common import DateTimeWithPrecision, StructuredText
from stix.common.related import GenericRelationshipList, RelatedObservable


class Vulnerability(stix.Entity):
    _binding = exploit_target_binding
    _binding_class = _binding.VulnerabilityType
    _namespace = "http://stix.mitre.org/ExploitTarget-1"

    def __init__(self, title=None, description=None, short_description=None):
        self.is_known = None
        self.is_publicly_acknowledged = None
        self.title = title
        self.description = description
        self.short_description = short_description
        self.cve_id = None
        self.osvdb_id = None
        self.source = None
        self.cvss_score = None
        self.discovered_datetime = None
        self.published_datetime = None
        self.affected_software = AffectedSoftware()
        # references

    @property
    def title(self):
        return self._title

    @title.setter
    def title(self, value):
        self._title = value

    @property
    def description(self):
        return self._description

    @description.setter
    def description(self, value):
        if value:
            if isinstance(value, StructuredText):
                self._description = value
            else:
                self._description = StructuredText(value=value)
        else:
            self._description = None

    @property
    def short_description(self):
        return self._short_description

    @short_description.setter
    def short_description(self, value):
        if value:
            if isinstance(value, StructuredText):
                self._short_description = value
            else:
                self._short_description = StructuredText(value=value)
        else:
            self._short_description = None

    @property
    def discovered_datetime(self):
        return self._discovered_datetime

    @discovered_datetime.setter
    def discovered_datetime(self, value):
        if value:
            if isinstance(value, DateTimeWithPrecision):
                self._discovered_datetime = value
            else:
                self._discovered_datetime = DateTimeWithPrecision(value=value)
        else:
            self._discovered_datetime = None

    def to_obj(self, return_obj=None):
        if not return_obj:
            return_obj = self._binding_class()

        if self.is_known is not None:
            return_obj.set_is_known(self.is_known)
        if self.is_publicly_acknowledged is not None:
            return_obj.set_is_publicly_acknowledged(self.is_publicly_acknowledged)
        return_obj.set_Title(self.title)
        if self.description:
            return_obj.set_Description(self.description.to_obj())
        if self.short_description:
            return_obj.set_Short_Description(self.short_description.to_obj())
        return_obj.set_CVE_ID(self.cve_id)
        return_obj.set_OSVDB_ID(self.osvdb_id)
        return_obj.set_Source(self.source)
        if self.cvss_score:
            return_obj.set_CVSS_Score(self.cvss_score.to_obj())
        if self.discovered_datetime:
            return_obj.set_Discovered_DateTime(self.discovered_datetime.to_obj())
        if self.published_datetime:
            return_obj.set_Published_DateTime(self.published_datetime.to_obj())
        if self.affected_software:
            return_obj.set_Affected_Software(self.affected_software.to_obj())

        return return_obj

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        if not obj:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.is_known = obj.get_is_known()
        return_obj.is_publicly_acknowledged = obj.get_is_publicly_acknowledged()
        return_obj.title = obj.get_Title()
        return_obj.description = StructuredText.from_obj(obj.get_Description())
        return_obj.short_description = StructuredText.from_obj(obj.get_Short_Description())
        return_obj.cve_id = obj.get_CVE_ID()
        return_obj.osvdb_id = obj.get_OSVDB_ID()
        return_obj.source = obj.get_Source()
        return_obj.cvss_score = CVSSVector.from_obj(obj.get_CVSS_Score())
        return_obj.discovered_datetime = DateTimeWithPrecision.from_obj(obj.get_Discovered_DateTime())
        return_obj.published_datetime = DateTimeWithPrecision.from_obj(obj.get_Published_DateTime())
        return_obj.affected_software = AffectedSoftware.from_obj(obj.get_Affected_Software())

        return return_obj

    def to_dict(self):
        d = {}

        if self.is_known is not None:
            d['is_known'] = self.is_known
        if self.is_publicly_acknowledged is not None:
            d['is_publicly_acknowledged'] = self.is_publicly_acknowledged
        if self.title:
            d['title'] = self.title
        if self.description:
            d['description'] = self.description.to_dict()
        if self.short_description:
            d['short_description'] = self.short_description.to_dict()
        if self.cve_id:
            d['cve_id'] = self.cve_id
        if self.osvdb_id:
            d['osvdb_id'] = self.osvdb_id
        if self.source:
            d['source'] = self.source
        if self.cvss_score:
            d['cvss_score'] = self.cvss_score.to_dict()
        if self.discovered_datetime:
            d['discovered_datetime'] = self.discovered_datetime.to_dict()
        if self.published_datetime:
            d['published_datetime'] = self.published_datetime.to_dict()
        if self.affected_software:
            d['affected_software'] = self.affected_software.to_dict()

        return d

    @classmethod
    def from_dict(cls, dict_repr, return_obj=None):
        if not dict_repr:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.is_known = dict_repr.get('is_known')
        return_obj.is_publicly_acknowledged = dict_repr.get('is_publicly_acknowledged')
        return_obj.title = dict_repr.get('title')
        return_obj.description = StructuredText.from_dict(dict_repr.get('description'))
        return_obj.short_description = StructuredText.from_dict(dict_repr.get('short_description'))
        return_obj.cve_id = dict_repr.get('cve_id')
        return_obj.osvdb_id = dict_repr.get('osvdb_id')
        return_obj.source = dict_repr.get('source')
        return_obj.cvss_score = CVSSVector.from_dict(dict_repr.get('cvss_score'))
        return_obj.discovered_datetime = DateTimeWithPrecision.from_dict(dict_repr.get('discovered_datetime'))
        return_obj.published_datetime = DateTimeWithPrecision.from_dict(dict_repr.get('published_datetime'))
        return_obj.affected_software = AffectedSoftware.from_dict(dict_repr.get('affected_software'))

        return return_obj


class CVSSVector(stix.Entity):
    _binding = exploit_target_binding
    _binding_class = exploit_target_binding.CVSSVectorType
    _namespace = "http://stix.mitre.org/ExploitTarget-1"

    def __init__(self):
        self.overall_score = None
        self.base_score = None
        self.base_vector = None
        self.temporal_score = None
        self.temporal_vector = None
        self.environmental_score = None
        self.environmental_vector = None

    def to_obj(self, return_obj=None):
        if not return_obj:
            return_obj = self._binding_class()

        return_obj.set_Overall_Score(self.overall_score)
        return_obj.set_Base_Score(self.base_score)
        return_obj.set_Base_Vector(self.base_vector)
        return_obj.set_Temporal_Score(self.temporal_score)
        return_obj.set_Temporal_Vector(self.temporal_vector)
        return_obj.set_Environmental_Score(self.environmental_score)
        return_obj.set_Environmental_Vector(self.environmental_vector)

        return return_obj

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        if not obj:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.overall_score = obj.get_Overall_Score()
        return_obj.base_score = obj.get_Base_Score()
        return_obj.base_vector = obj.get_Base_Vector()
        return_obj.temporal_score = obj.get_Temporal_Score()
        return_obj.temporal_vector = obj.get_Temporal_Vector()
        return_obj.environmental_score = obj.get_Environmental_Score()
        return_obj.environmental_vector = obj.get_Environmental_Vector()

        return return_obj

    def to_dict(self):
        d = {}

        if self.overall_score:
            d['overall_score'] = self.overall_score
        if self.base_score:
            d['base_score'] = self.base_score
        if self.base_vector:
            d['base_vector'] = self.base_vector
        if self.temporal_score:
            d['temporal_score'] = self.temporal_score
        if self.temporal_vector:
            d['temporal_vector'] = self.temporal_vector
        if self.environmental_score:
            d['environmental_score'] = self.environmental_score
        if self.environmental_vector:
            d['environmental_vector'] = self.environmental_vector

        return d

    @classmethod
    def from_dict(cls, dict_repr, return_obj=None):
        if not dict_repr:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.overall_score = dict_repr.get('overall_score')
        return_obj.base_score = dict_repr.get('base_score')
        return_obj.base_vector = dict_repr.get('base_vector')
        return_obj.temporal_score = dict_repr.get('temporal_score')
        return_obj.temporal_vector = dict_repr.get('temporal_vector')
        return_obj.environmental_score = dict_repr.get('environmental_score')
        return_obj.environmental_vector = dict_repr.get('environmental_vector')

        return return_obj


class AffectedSoftware(GenericRelationshipList):
    _binding = exploit_target_binding
    _binding_class = exploit_target_binding.AffectedSoftwareType
    _namespace = "http://stix.mitre.org/ExploitTarget-1"
    _binding_var = "Affected_Software"
    _contained_type = RelatedObservable
    _inner_name = "affected_software"
