"""
A directive for including a matplotlib plot in a Sphinx document.

This directive is based on the ``plot`` directive of matplotlib, with thanks.

By default, in HTML output, `nbplot` will include a .png file with a
link to a high-res .png and .pdf.  In LaTeX output, it will include a
.pdf.

The source code for the plot is **inline content** to the directive:.  It can
be without doctest syntax::

    .. nbplot::

        import matplotlib.pyplot as plt
        import matplotlib.image as mpimg
        import numpy as np
        img = mpimg.imread('_static/stinkbug.png')
        imgplot = plt.imshow(img)

or with **doctest** syntax::

    .. nbplot::

        A plotting example:
        >>> import matplotlib.pyplot as plt
        >>> plt.plot([1,2,3], [4,5,6])  #doctest: +ELLIPSIS
        [...]

Options
-------

The ``nbplot`` directive supports the following options:

    format : {'python', 'doctest'}
        Specify the format of the input.  If not specified, nbplot guesses the
        format from the content.

    include-source : bool
        Whether to display the source code. The default can be changed
        using the `nbplot_include_source` variable in conf.py

    encoding : str
        If this source file is in a non-UTF8 or non-ASCII encoding,
        the encoding must be specified using the `:encoding:` option.
        The encoding will not be inferred using the ``-*- coding -*-``
        metacomment.

    keepfigs: bool
        If specified, do not close figures from previous plot command. This
        allows you to build up a plot in stages, with the current nbplot
        command building on the figure generated by previous nbplot commands.

    nofigs : bool
        If specified, the code block will be run, but no figures will be
        inserted.  This is sometimes necessary when your code generates a plot,
        but you do not want to include it.  If the code does not generate a
        plot, this option is not necessary.

    raises : str
        String giving error class.  The code runner will assert this error was
        raised by the enclosed code, and suppress stderr.

The namespace of the nbplot command is reset to empty for each document.  The
code in each nbplot directive instance in a given document uses the namespace
generated by previous nbplot directive instances in the same document.

Additionally, this directive supports all of the options of the
`image` directive, except for `target` (since plot will add its own
target).  These include `alt`, `height`, `width`, `scale`, `align` and
`class`.

Configuration options
---------------------

The nbplot directive has the following configuration options:

    nbplot_include_source
        Default value for the include-source option

    nbplot_html_show_source_link
        Whether to show a link to the source in HTML.

    nbplot_pre_code
        Code that should be executed before each plot.

    nbplot_basedir
        Base directory, to which ``plot::`` file names are relative
        to.  (If None or empty, file names are relative to the
        directory where the file containing the directive is.)

    nbplot_formats
        File formats to generate. List of tuples or strings::

            [(suffix, dpi), suffix, ...]

        that determine the file format and the DPI. For entries whose
        DPI was omitted, sensible defaults are chosen. When passing from
        the command line through sphinx_build the list should be passed as
        suffix:dpi,suffix:dpi, ....

    nbplot_html_show_formats
        Whether to show links to the files in HTML.

    nbplot_rcparams
        A dictionary containing any non-standard rcParams that should
        be applied at the beginning of each document.

    nbplot_working_directory
        By default, the working directory will be changed to the directory of
        the example, so the code can get at its data files, if any.  Also its
        path will be added to `sys.path` so it can import any helper modules
        sitting beside it.  This configuration option can be used to specify
        a central directory (also added to `sys.path`) where data files and
        helper modules for all code are located.

    nbplot_template
        Provide a customized template for preparing restructured text.
"""
from __future__ import (absolute_import, division, print_function,
                        unicode_literals)

import six

def setup_module(module):
    # Prevent nosetests trying to run setup function
    pass

import sys, os, shutil, io, re, textwrap
from os.path import relpath
import traceback

from docutils.parsers.rst import directives
from docutils.parsers.rst.directives.images import Image
align = Image.align
from sphinx.util.compat import Directive

import jinja2
def format_template(template, **kw):
    return jinja2.Template(template).render(**kw)

import matplotlib
import matplotlib.cbook as cbook
matplotlib.use('Agg')
import matplotlib.pyplot as plt
from matplotlib._pylab_helpers import Gcf

__version__ = 2

# Options for NBPlotDirective

def _option_boolean(arg):
    if not arg or not arg.strip():
        # no argument given, assume used as a flag
        return True
    elif arg.strip().lower() in ('no', '0', 'false'):
        return False
    elif arg.strip().lower() in ('yes', '1', 'true'):
        return True
    else:
        raise ValueError('"%s" unknown boolean' % arg)


def _option_format(arg):
    return directives.choice(arg, ('python', 'doctest'))


def _option_align(arg):
    return directives.choice(
        arg,
        ("top", "middle", "bottom", "left", "center", "right"))


class NBPlotDirective(Directive):
    """ Implements nbplot directive

    """ + __doc__
    has_content = True
    required_arguments = 0
    optional_arguments = 0
    final_argument_whitespace = False
    option_spec = {'alt': directives.unchanged,
                   'height': directives.length_or_unitless,
                   'width': directives.length_or_percentage_or_unitless,
                   'scale': directives.nonnegative_int,
                   'align': _option_align,
                   'class': directives.class_option,
                   'include-source': _option_boolean,
                   'format': _option_format,
                   'keepfigs': directives.flag,
                   'nofigs': directives.flag,
                   'encoding': directives.encoding,
                   'raises': directives.unchanged,
                  }

    def run(self):
        document = self.state.document
        config = document.settings.env.config
        nofigs = 'nofigs' in self.options
        env = document.settings.env
        docname = env.docname

        self.options.setdefault('include-source', config.nbplot_include_source)
        # If this is the first directive in the document, clear context
        if env.nbplot_reset_markers.get(docname, False):
            context_reset = False
        else:
            env.nbplot_reset_markers[docname] = True
            context_reset = True
        close_figs = False if 'keepfigs' in self.options else True

        rst_file = document.attributes['source']
        rst_dir = os.path.dirname(rst_file)

        source_file_name = rst_file
        code = textwrap.dedent("\n".join(map(str, self.content)))
        counter = document.attributes.get('_plot_counter', 0) + 1
        document.attributes['_plot_counter'] = counter
        base, ext = os.path.splitext(os.path.basename(source_file_name))
        output_base = '%s-%d.py' % (base, counter)

        base, source_ext = os.path.splitext(output_base)
        if source_ext in ('.py', '.rst', '.txt'):
            output_base = base
        else:
            source_ext = ''

        # ensure that LaTeX includegraphics doesn't choke in foo.bar.pdf
        # filenames
        output_base = output_base.replace('.', '-')

        # is it in doctest format?
        is_doctest = contains_doctest(code)
        if 'format' in self.options:
            is_doctest = False if self.options['format'] == 'python' else True

        # should the code raise an exception?
        raises = (eval(self.options['raises']) if 'raises' in self.options
                  else None)

        # determine output directory name fragment
        source_rel_name = relpath(source_file_name, setup.confdir)
        source_rel_dir = os.path.dirname(source_rel_name)
        while source_rel_dir.startswith(os.path.sep):
            source_rel_dir = source_rel_dir[1:]

        # build_dir: where to place output files (temporarily)
        build_dir = os.path.join(os.path.dirname(setup.app.doctreedir),
                                 'nbplot_directive',
                                 source_rel_dir)
        # get rid of .. in paths, also changes pathsep
        # see note in Python docs for warning about symbolic links on Windows.
        # need to compare source and dest paths at end
        build_dir = os.path.normpath(build_dir)

        if not os.path.exists(build_dir):
            os.makedirs(build_dir)

        # output_dir: final location in the builder's directory
        dest_dir = os.path.abspath(os.path.join(setup.app.builder.outdir,
                                                source_rel_dir))
        if not os.path.exists(dest_dir):
            os.makedirs(dest_dir) # no problem here for me, but just use built-ins

        # how to link to files from the RST file
        dest_dir_link = os.path.join(relpath(setup.confdir, rst_dir),
                                     source_rel_dir).replace(os.path.sep, '/')
        build_dir_link = relpath(build_dir, rst_dir).replace(os.path.sep, '/')
        source_link = dest_dir_link + '/' + output_base + source_ext

        # make figures
        try:
            results = render_figures(code,
                                     source_file_name,
                                     build_dir,
                                     output_base,
                                     config=config,
                                     context = True,  # keep plot context
                                     function_name = None,
                                     context_reset=context_reset,
                                     close_figs=close_figs,
                                     raises=raises)
            errors = []
        except PlotError as err:
            reporter = self.state.memo.reporter
            sm = reporter.system_message(
                2,
                'Exception plotting {output_base}\n'
                'from: {source_file_name}\n'
                'with code:\n\n{code}\n\n'
                'Exception:\n{err}'.format(**locals()),
                line=self.lineno)
            results = [(code, [])]
            errors = [sm]

        # generate output restructuredtext
        total_lines = []
        for j, (code_piece, images) in enumerate(results):
            if is_doctest:
                lines = [''] + [row.rstrip() for row in code_piece.split('\n')]
            else:
                lines = ['.. code-block:: python', '']
                lines += ['    %s' % row.rstrip()
                        for row in code_piece.split('\n')]
            if self.options['include-source']:
                source_code = '\n'.join(lines)
            else:  # Doctest blocks still go into page, but hidden
                if is_doctest:
                    source_code = ('.. doctest::\n'
                                   '    :hide:\n'
                                   '    ' + '\n    '.join(lines))
                else:  # Non-doctest blocks get dropped from page output
                    source_code = ""

            if nofigs:
                images = []

            opts = [':%s: %s' % (key, val)
                    for key, val in self.options.items()
                    if key in ('alt', 'height', 'width', 'scale', 'align',
                               'class')]

            only_html = ".. only:: html"
            only_latex = ".. only:: latex"
            only_texinfo = ".. only:: texinfo"

            # Not-None src_link signals the need for a source link in the generated
            # html
            if j == 0 and config.nbplot_html_show_source_link:
                src_link = source_link
            else:
                src_link = None

            result = format_template(
                config.nbplot_template or TEMPLATE,
                dest_dir=dest_dir_link,
                build_dir=build_dir_link,
                source_link=src_link,
                multi_image=len(images) > 1,
                only_html=only_html,
                only_latex=only_latex,
                only_texinfo=only_texinfo,
                options=opts,
                images=images,
                source_code=source_code,
                html_show_formats=config.nbplot_html_show_formats and len(images),
                caption='')

            total_lines.extend(result.split("\n"))
            total_lines.extend("\n")

        if total_lines:
            self.state_machine.insert_input(
                total_lines, source=source_file_name)

        # copy image files to builder's output directory, if necessary
        if not os.path.exists(dest_dir):
            cbook.mkdirs(dest_dir)

        for code_piece, images in results:
            for img in images:
                for fn in img.filenames():
                    destimg = os.path.join(dest_dir, os.path.basename(fn))
                    if fn != destimg:
                        shutil.copyfile(fn, destimg)

        # copy script (if necessary)
        target_name = os.path.join(dest_dir, output_base + source_ext)
        with io.open(target_name, 'w', encoding="utf-8") as f:
            if source_file_name == rst_file:
                code_escaped = unescape_doctest(code)
            else:
                code_escaped = code
            f.write(code_escaped)

        return errors


def mark_plot_labels(app, document):
    """
    To make plots referenceable, we need to move the reference from
    the "htmlonly" (or "latexonly") node to the actual figure node
    itself.
    """
    for name, explicit in document.nametypes.items():
        if not explicit:
            continue
        labelid = document.nameids[name]
        if labelid is None:
            continue
        node = document.ids[labelid]
        if node.tagname in ('html_only', 'latex_only'):
            for n in node:
                if n.tagname == 'figure':
                    sectname = name
                    for c in n:
                        if c.tagname == 'caption':
                            sectname = c.astext()
                            break

                    node['ids'].remove(labelid)
                    node['names'].remove(name)
                    n['ids'].append(labelid)
                    n['names'].append(name)
                    document.settings.env.labels[name] = \
                        document.settings.env.docname, labelid, sectname
                    break


def clear_reset_marker(app, env, docname):
    """ Clear markers for whether `docname` has seen a plot context reset
    """
    if not hasattr(env, 'nbplot_reset_markers'):
        env.nbplot_reset_markers = {}
    env.nbplot_reset_markers[docname] = False


def setup(app):
    setup.app = app
    setup.config = app.config
    setup.confdir = app.confdir

    app.add_directive('nbplot', NBPlotDirective)
    pre_default = "import numpy as np\nfrom matplotlib import pyplot as plt\n"
    app.add_config_value('nbplot_pre_code', pre_default, True)
    app.add_config_value('nbplot_include_source', True, True)
    app.add_config_value('nbplot_html_show_source_link', False, True)
    app.add_config_value('nbplot_formats', ['png', 'hires.png', 'pdf'], True)
    app.add_config_value('nbplot_basedir', None, True)
    app.add_config_value('nbplot_html_show_formats', True, True)
    app.add_config_value('nbplot_rcparams', {}, True)
    app.add_config_value('nbplot_working_directory', None, True)
    app.add_config_value('nbplot_template', None, True)

    app.connect(str('env-purge-doc'), clear_reset_marker)
    app.connect(str('doctree-read'), mark_plot_labels)

#------------------------------------------------------------------------------
# Doctest handling
#------------------------------------------------------------------------------

def contains_doctest(text):
    try:
        # check if it's valid Python as-is
        compile(text, '<string>', 'exec')
        return False
    except SyntaxError:
        pass
    r = re.compile(r'^\s*>>>', re.M)
    m = r.search(text)
    return bool(m)


def unescape_doctest(text):
    """
    Extract code from a piece of text, which contains either Python code
    or doctests.

    """
    if not contains_doctest(text):
        return text

    code = ""
    for line in text.split("\n"):
        m = re.match(r'^\s*(>>>|\.\.\.) (.*)$', line)
        if m:
            code += m.group(2) + "\n"
        elif line.strip():
            code += "# " + line.strip() + "\n"
        else:
            code += "\n"
    return code


def split_code_at_show(text):
    """
    Split code at plt.show()

    """

    parts = []
    is_doctest = contains_doctest(text)

    part = []
    for line in text.split("\n"):
        if (not is_doctest and line.strip() == 'plt.show()') or \
               (is_doctest and line.strip() == '>>> plt.show()'):
            part.append(line)
            parts.append("\n".join(part))
            part = []
        else:
            part.append(line)
    if "\n".join(part).strip():
        parts.append("\n".join(part))
    return parts


def remove_coding(text):
    """
    Remove the coding comment, which six.exec_ doesn't like.
    """
    sub_re = re.compile("^#\s*-\*-\s*coding:\s*.*-\*-$", flags=re.MULTILINE)
    return sub_re.sub("", text)

#------------------------------------------------------------------------------
# Template
#------------------------------------------------------------------------------


TEMPLATE = """
{{ source_code }}

{{ only_html }}

   {% if source_link or (html_show_formats and not multi_image) %}
   (
   {%- if source_link -%}
   `Source code <{{ source_link }}>`__
   {%- endif -%}
   {%- if html_show_formats and not multi_image -%}
     {%- for img in images -%}
       {%- for fmt in img.formats -%}
         {%- if source_link or not loop.first -%}, {% endif -%}
         `{{ fmt }} <{{ dest_dir }}/{{ img.basename }}.{{ fmt }}>`__
       {%- endfor -%}
     {%- endfor -%}
   {%- endif -%}
   )
   {% endif %}

   {% for img in images %}
   .. figure:: {{ build_dir }}/{{ img.basename }}.png
      {% for option in options -%}
      {{ option }}
      {% endfor %}

      {% if html_show_formats and multi_image -%}
        (
        {%- for fmt in img.formats -%}
        {%- if not loop.first -%}, {% endif -%}
        `{{ fmt }} <{{ dest_dir }}/{{ img.basename }}.{{ fmt }}>`__
        {%- endfor -%}
        )
      {%- endif -%}

      {{ caption }}
   {% endfor %}

{{ only_latex }}

   {% for img in images %}
   {% if 'pdf' in img.formats -%}
   .. image:: {{ build_dir }}/{{ img.basename }}.pdf
   {% endif -%}
   {% endfor %}

{{ only_texinfo }}

   {% for img in images %}
   .. image:: {{ build_dir }}/{{ img.basename }}.png
      {% for option in options -%}
      {{ option }}
      {% endfor %}

   {% endfor %}

"""

exception_template = """
.. htmlonly::

   [`source code <%(linkdir)s/%(basename)s.py>`__]

Exception occurred rendering plot.

"""

# the context of the plot for all directives
plot_context = dict()


class ImageFile(object):
    def __init__(self, basename, dirname):
        self.basename = basename
        self.dirname = dirname
        self.formats = []

    def filename(self, format):
        return os.path.join(self.dirname, "%s.%s" % (self.basename, format))

    def filenames(self):
        return [self.filename(fmt) for fmt in self.formats]


class PlotError(RuntimeError):
    pass


def _check_wd(dirname):
    try:
        os.chdir(dirname)
    except OSError as err:
        raise OSError(str(err) + '\n`nbplot_working_directory` option in'
                        'Sphinx configuration file must be a valid '
                        'directory path')
    except TypeError as err:
        raise TypeError(str(err) + '\n`nbplot_working_directory` option in '
                        'Sphinx configuration file must be a string or '
                        'None')
    return dirname


def run_code(code, code_path=None, ns=None, function_name=None, workdir=None,
             pre_code=None, raises=None):
    """
    Run `code` from file at `code_path` in namespace `ns`

    Parameters
    ----------
    code : str
        code to run.
    code_path : str
        Filename containing the code.
    ns : None or dict, optional
        Python namespace in which to execute code.  If None, make fresh
        namespace.
    function_name : None or str, optional
        If non-empty string, name of function to execute after executing
        `code`.
    workdir : None or str, optional
        Working directory in which to run code.  Defaults to current working
        directory.
    pre_code : None or str, optional
        Any code to run before `code`.
    raises : None or Exception class
        An exception that the run code should raise.

    Returns
    -------
    ns : dict
        Namespace, filled from execution of `code`.
    """
    # Change the working directory to the directory of the example, so
    # it can get at its data files, if any.  Add its path to sys.path
    # so it can import any helper modules sitting beside it.
    if six.PY2:
        pwd = os.getcwdu()
    else:
        pwd = os.getcwd()
    old_sys_path = list(sys.path)
    workdir = os.getcwd() if workdir is None else workdir
    os.chdir(workdir)
    sys.path.insert(0, workdir)

    # Reset sys.argv
    old_sys_argv = sys.argv
    sys.argv = [code_path]

    # Redirect stdout
    stdout = sys.stdout
    sys.stdout = io.StringIO() if six.PY3 else io.BytesIO()

    # Assign a do-nothing print function to the namespace.  There
    # doesn't seem to be any other way to provide a way to (not) print
    # that works correctly across Python 2 and 3.
    def _dummy_print(*arg, **kwarg):
        pass

    ns = {} if ns is None else ns
    try:
        try:
            code = unescape_doctest(code)
            if pre_code and not ns:
                six.exec_(six.text_type(pre_code), ns)
            ns['print'] = _dummy_print
            if "__main__" in code:
                six.exec_("__name__ = '__main__'", ns)
            code = remove_coding(code)
            if raises is None:
                six.exec_(code, ns)
            else:  # Code should raise exception
                try:
                    six.exec_(code, ns)
                except raises:
                    pass
            if function_name:
                six.exec_(function_name + "()", ns)
        except (Exception, SystemExit):
            raise PlotError(traceback.format_exc())
    finally:
        os.chdir(pwd)
        sys.argv = old_sys_argv
        sys.path[:] = old_sys_path
        sys.stdout = stdout
    return ns


def render_figures(code, code_path, output_dir, output_base, config,
                   context=True, function_name=None, context_reset=False,
                   close_figs=False, raises=None):
    """ Run plot code and save the hi/low res PNGs, PDF in `output_dir`

    Save the images under `output_dir` with file names derived from
    `output_base`.

    Parameters
    ----------
    code : str
        String containing code to run.
    code_path : str
        Path of file containing code.  Usually path to ``.rst`` file.
    output_dir : str
        Path to which to write output images from plots.
    output_base : str
        Prefix for filename(s) for output image(s).
    config : instance
        Sphinx configuration instance.
    context : {True, False}, optional
        If True, use persistent context (workspace) for executing code.
        Otherwise create new empty context for executing code.
    function_name : None or str, optional
        If not-empty str, name of function to execute after executing `code`.
    context_reset : {False, True}, optional
        If True, clear persistent context (workspace) for code.
    close_figs : {False, True}, optional
        If True, close all figures generated before our `code` runs.  False can
        be useful when building up a plot with several `code` blocks.
    raises : None or Exception, optional
        Exception class that code should raise, or None, for no exception.
    """
    # -- Parse format list
    default_dpi = {'png': 80, 'hires.png': 200, 'pdf': 200}
    formats = []
    plot_formats = config.nbplot_formats
    if isinstance(plot_formats, six.string_types):
        # String Sphinx < 1.3, Split on , to mimic
        # Sphinx 1.3 and later. Sphinx 1.3 always
        # returns a list.
        plot_formats = plot_formats.split(',')
    for fmt in plot_formats:
        if isinstance(fmt, six.string_types):
            if ':' in fmt:
                suffix,dpi = fmt.split(':')
                formats.append((str(suffix), int(dpi)))
            else:
                formats.append((fmt, default_dpi.get(fmt, 80)))
        elif type(fmt) in (tuple, list) and len(fmt)==2:
            formats.append((str(fmt[0]), int(fmt[1])))
        else:
            raise PlotError('invalid image format "%r" in nbplot_formats' % fmt)

    # Build the output
    code_pieces = split_code_at_show(code)

    results = []
    if context:
        ns = plot_context
    else:
        ns = {}

    if context_reset:
        plt.close('all')
        matplotlib.rc_file_defaults()
        matplotlib.rcParams.update(config.nbplot_rcparams)
        plot_context.clear()

    close_figs = not context or close_figs

    # Get working directory for code execution
    if setup.config.nbplot_working_directory is not None:
        workdir = _check_wd(setup.config.nbplot_working_directory)
    elif code_path is not None:
        workdir = os.path.abspath(os.path.dirname(code_path))
    else:
        workdir = None

    for i, code_piece in enumerate(code_pieces):

        if close_figs:
            plt.close('all')

        run_code(code_piece, code_path, ns, function_name, workdir=workdir,
                 pre_code=setup.config.nbplot_pre_code, raises=raises)

        images = []
        fig_managers = Gcf.get_all_fig_managers()
        for j, figman in enumerate(fig_managers):
            if len(fig_managers) == 1 and len(code_pieces) == 1:
                img = ImageFile(output_base, output_dir)
            elif len(code_pieces) == 1:
                img = ImageFile("%s_%02d" % (output_base, j), output_dir)
            else:
                img = ImageFile("%s_%02d_%02d" % (output_base, i, j),
                                output_dir)
            images.append(img)
            for format, dpi in formats:
                try:
                    figman.canvas.figure.savefig(img.filename(format), dpi=dpi)
                except Exception:
                    raise PlotError(traceback.format_exc())
                img.formats.append(format)

        results.append((code_piece, images))

    return results
