import codecs
import json
from pathlib import Path

import config
from utils import nrange


def get(hit, meta, fragment):
    filename = Path(hit['filename']).name
    with codecs.open(
        config.ROOT_DIR + f'/corpus/perseus/text/{filename}', 'r', 'utf8'
    ) as fp:
        doc = json.load(fp)

    divs = meta['meta'].split('-')

    start = [
        int(v)
        for k, v in meta['start'].items()
        if k in divs
    ]
    end = [
        int(v)
        for k, v in meta['end'].items()
        if k in divs
    ]

    pre_start = start[:-1] + [(start[-1] - config.LINES_OF_CONTEXT),]
    pre_end = start[:-1] + [(start[-1] - 1),]
    pre = []
    for ref in nrange(pre_start, pre_end):
        text = doc['text']
        for div in ref:
            try:
                text = text[str(div)]
            except KeyError:
                text = None
                break
        if text:
            pre.append(f"<pre>{text}</pre>")

    match = []
    for ref in nrange(start, end):
        text = doc['text']

        for div in ref:
            text = text[str(div)]
        match.append(f"<match>{text}</match>")

    post_start = end[:-1] + [(end[-1] + 1), ]
    post_end = end[:-1] + [(end[-1] + config.LINES_OF_CONTEXT), ]
    post = []
    for ref in nrange(post_start, post_end):
        text = doc['text']
        for div in ref:
            try:
                text = text[str(div)]
            except KeyError:
                text = None
                break
        if text:
            post.append(f"<post>{text}</post>")

    if 'poem' in divs or divs[-1] == 'verse':
        joiner = '\n\n'
    else:
        joiner = ' '
    return f'{joiner}'.join([*pre, *match, *post])


index = {0: {'author': 'Ammianus Marcellinus',
             'work': {'title': 'Rerum Gestarum',
                      'meta': 'book-chapter-section',
                      'urn': 'urn:cts:latinLit:stoa0023.stoa001.perseus-lat1-simple'}},
         1: {'author': 'Apuleius',
             'work': {'title': 'Apologia',
                      'meta': 'section',
                      'urn': 'urn:cts:latinLit:phi1212.phi001.perseus-lat1-simple'}},
         2: {'author': 'Apuleius',
             'work': {'title': 'Florida',
                      'meta': 'section',
                      'urn': 'urn:cts:latinLit:phi1212.phi003.perseus-lat1-simple'}},
         3: {'author': 'Apuleius',
             'work': {'title': 'Metamorphoses',
                      'meta': 'book-chapteer',
                      'urn': 'urn:cts:latinLit:phi1212.phi002.perseus-lat1-simple'}},
         4: {'author': 'Ausonius, Decimus Magnus',
             'work': {'title': 'Bissula',
                      'meta': 'poem-line',
                      'urn': 'urn:cts:latinLit:stoa0045.stoa001.perseus-lat2-simple'}},
         5: {'author': 'Ausonius, Decimus Magnus',
             'work': {'title': 'Caesares',
                      'meta': 'poem-line',
                      'urn': 'urn:cts:latinLit:stoa0045.stoa002.perseus-lat2-simple'}},
         6: {'author': 'Ausonius, Decimus Magnus',
             'work': {'title': 'Commemoratio Professorum Burdigalensium',
                      'meta': 'poem-line',
                      'urn': 'urn:cts:latinLit:stoa0045.stoa004.perseus-lat2-simple'}},
         7: {'author': 'Ausonius, Decimus Magnus',
             'work': {'title': 'De Herediolo',
                      'meta': 'line',
                      'urn': 'urn:cts:latinLit:stoa0045.stoa006.perseus-lat2-simple'}},
         8: {'author': 'Ausonius, Decimus Magnus',
             'work': {'title': 'Eclogarum Liber',
                      'meta': 'poem-line',
                      'urn': 'urn:cts:latinLit:stoa0045.stoa007.perseus-lat2-simple'}},
         9: {'author': 'Ausonius, Decimus Magnus',
             'work': {'title': 'Ephemeris',
                      'meta': 'poem-line',
                      'urn': 'urn:cts:latinLit:stoa0045.stoa008.perseus-lat2-simple'}},
         10: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Epicedion in Patrem',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa009.perseus-lat2-simple'}},
         11: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Epigrammaton Liber',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa010.perseus-lat2-simple'}},
         12: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Epistulae',
                       'meta': 'letter-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa011.perseus-lat2-simple'}},
         13: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Epitaphia',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa012.perseus-lat2-simple'}},
         14: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Genethliacon ad Ausonium Nepotem',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa013.perseus-lat2-simple'}},
         15: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Gratiarum Actio',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa014.perseus-lat2-simple'}},
         16: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Griphus Ternarii Numeri',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa015.perseus-lat2-simple'}},
         17: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Liber Protrepticus ad Nepotem',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa016.perseus-lat2-simple'}},
         18: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Mosella',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa019.perseus-lat2-simple'}},
         19: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Oratio Versibus Rhopalicis',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa020.perseus-lat2-simple'}},
         20: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Ordo Urbium Nobilium',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa021.perseus-lat2-simple'}},
         21: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Parentalia',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa022.perseus-lat2-simple'}},
         22: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Praefatiunculae',
                       'meta': 'book-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa025.perseus-lat2-simple'}},
         23: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Precationes',
                       'meta': 'book-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa026.perseus-lat2-simple'}},
         24: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Technopaegnion',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa028.perseus-lat2-simple'}},
         25: {'author': 'Ausonius, Decimus Magnus',
              'work': {'title': 'Versus Paschales Prosodic',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:stoa0045.stoa027.perseus-lat2-simple'}},
         26: {'author': 'Boethius D. 524',
              'work': {'title': 'De consolatione philosophiae',
                       'meta': 'book-section',
                       'urn': 'urn:cts:latinLit:stoa0058.stoa001.perseus-lat2-simple'}},
         27: {'author': 'Boethius D. 524',
              'work': {'title': 'De Fide Catholica',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:stoa0058.stoa006.perseus-lat1-simple'}},
         28: {'author': 'Boethius D. 524',
              'work': {'title': 'Liber De Persona et Duabus Naturis Contra Eutychen Et Nestorium',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:stoa0058.stoa023.perseus-lat1-simple'}},
         29: {'author': 'Boethius D. 524',
              'work': {'title': 'Quomodo Substantiae in Eo Quod Sint Bonae Sint Cum Non Sint Substanialia Bona',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:stoa0058.stoa003.perseus-lat1-simple'}},
         30: {'author': 'Boethius D. 524',
              'work': {'title': 'Quomodo Trinitas Unus Deus Ac Non Tres Dii (De Trinitate)',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:stoa0058.stoa025.perseus-lat1-simple'}},
         31: {'author': 'Boethius D. 524',
              'work': {
                  'title': 'Utrum Pater Et Filius Ac Spiritus Sanctus De Divinitate Substantialiter Praedicentur Liber',
                  'meta': 'section',
                  'urn': 'urn:cts:latinLit:stoa0058.stoa028.perseus-lat1-simple'}},
         32: {'author': 'Caesar, Julius',
              'work': {'title': 'Gallic War',
                       'meta': 'Book-Chapter-Section',
                       'urn': 'urn:cts:latinLit:phi0448.phi001.perseus-lat2-simple'}},
         33: {'author': 'Celsus, Aulus Cornelius',
              'work': {'title': 'De Medicina',
                       'meta': 'book-chapter',
                       'urn': 'urn:cts:latinLit:phi0836.phi002.perseus-lat5-simple'}},
         34: {'author': 'Cicero',
              'work': {'title': 'Academica',
                       'meta': 'book-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi045.perseus-lat1-simple'}},
         35: {'author': 'Cicero',
              'work': {'title': 'Orationes de Lege Agraria',
                       'meta': 'chapter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi011.perseus-lat2-simple'}},
         36: {'author': 'Cicero',
              'work': {'title': 'Brutus',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi039.perseus-lat1-simple'}},
         37: {'author': 'Cicero',
              'work': {'title': 'De Amicitia',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi052.perseus-lat1-simple'}},
         38: {'author': 'Cicero',
              'work': {'title': 'De Divinatione',
                       'meta': 'book-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi053.perseus-lat1-simple'}},
         39: {'author': 'Cicero',
              'work': {'title': 'De Fato',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi054.perseus-lat1-simple'}},
         40: {'author': 'Cicero',
              'work': {'title': 'de Finibus Bonorum et Malorum',
                       'meta': 'book-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi048.perseus-lat1-simple'}},
         41: {'author': 'Cicero',
              'work': {'title': 'De Inventione',
                       'meta': 'chapter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi036.perseus-lat1-simple'}},
         42: {'author': 'Cicero',
              'work': {'title': 'de Natura Deorum',
                       'meta': 'chapter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi050.perseus-lat1-simple'}},
         43: {'author': 'Cicero',
              'work': {'title': 'De Officiis',
                       'meta': 'book-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi055.perseus-lat1-simple'}},
         44: {'author': 'Cicero',
              'work': {'title': 'De Optimo Genere Oratorum',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi041.perseus-lat1-simple'}},
         45: {'author': 'Cicero',
              'work': {'title': 'De Republica',
                       'meta': 'book-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi043.perseus-lat1-simple'}},
         46: {'author': 'Cicero',
              'work': {'title': 'De Senectute',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi051.perseus-lat1-simple'}},
         47: {'author': 'Cicero',
              'work': {'title': 'In Caecilium',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi004.perseus-lat2-simple'}},
         48: {'author': 'Cicero',
              'work': {'title': 'Pro Archia',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi016.perseus-lat2-simple'}},
         49: {'author': 'Cicero',
              'work': {'title': 'For Marcus Caelius',
                       'meta': 'unknown',
                       'urn': 'urn:cts:latinLit:phi0474.phi024.perseus-lat2-simple'}},
         50: {'author': 'Cicero',
              'work': {'title': 'Pro Fonteio',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi007.perseus-lat2-simple'}},
         51: {'author': 'Cicero',
              'work': {'title': 'Pro P. Quinctio',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi001.perseus-lat2-simple'}},
         52: {'author': 'Cicero',
              'work': {'title': 'Pro Roscio comoedo',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi003.perseus-lat2-simple'}},
         53: {'author': 'Cicero',
              'work': {'title': 'Pro S. Roscio Amerino',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi002.perseus-lat2-simple'}},
         54: {'author': 'Cicero',
              'work': {'title': 'Pro Sulla',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi015.perseus-lat2-simple'}},
         55: {'author': 'Cicero',
              'work': {'title': 'In Catilinam',
                       'meta': 'chapter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi013.perseus-lat2-simple'}},
         56: {'author': 'Cicero',
              'work': {'title': 'Pro Cluentio',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi010.perseus-lat2-simple'}},
         57: {'author': 'Cicero',
              'work': {'title': 'Pro C. Rabiro perduellionis reo',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi012.perseus-lat2-simple'}},
         58: {'author': 'Cicero',
              'work': {'title': 'Pro Murena',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi014.perseus-lat2-simple'}},
         59: {'author': 'Cicero',
              'work': {'title': 'Pro Flacco',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi017.perseus-lat2-simple'}},
         60: {'author': 'Cicero',
              'work': {'title': 'Post reditum in senatu',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi019.perseus-lat2-simple'}},
         61: {'author': 'Cicero',
              'work': {'title': 'Letters to Atticus',
                       'meta': 'book-letter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi057.perseus-lat1-simple'}},
         62: {'author': 'Cicero',
              'work': {'title': 'Letters to Brutus',
                       'meta': 'book-letter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi059.perseus-lat1-simple'}},
         63: {'author': 'Cicero',
              'work': {'title': 'Letters to his brother Quintus',
                       'meta': 'book-letter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi058.perseus-lat1-simple'}},
         64: {'author': 'Cicero',
              'work': {'title': 'Letters to his Friends',
                       'meta': 'book-letter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi056.perseus-lat1-simple'}},
         65: {'author': 'Cicero',
              'work': {'title': 'Lucullus',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi046.perseus-lat1-simple'}},
         66: {'author': 'Cicero',
              'work': {'title': 'Pro A. Caecina',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi008.perseus-lat2-simple'}},
         67: {'author': 'Cicero',
              'work': {'title': 'Pro Tullio',
                       'meta': 'chapter',
                       'urn': 'urn:cts:latinLit:phi0474.phi006.perseus-lat2-simple'}},
         68: {'author': 'Cicero',
              'work': {'title': 'On Oratory',
                       'meta': 'chapter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi037.perseus-lat1-simple'}},
         69: {'author': 'Cicero',
              'work': {'title': 'Pro lege manilia',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi009.perseus-lat2-simple'}},
         70: {'author': 'Cicero',
              'work': {'title': 'In Verrem',
                       'meta': 'actio-book-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi005.perseus-lat2-simple'}},
         71: {'author': 'Cicero',
              'work': {'title': 'Orator',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi040.perseus-lat1-simple'}},
         72: {'author': 'Cicero',
              'work': {'title': 'Paradoxa Stoicorum',
                       'meta': 'book-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi047.perseus-lat1-simple'}},
         73: {'author': 'Cicero',
              'work': {'title': 'Partitiones Oratoriae',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi038.perseus-lat1-simple'}},
         74: {'author': 'Cicero',
              'work': {'title': 'Timaeus',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi072.perseus-lat1-simple'}},
         75: {'author': 'Cicero',
              'work': {'title': 'Post reditum ad populum',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi018.perseus-lat2-simple'}},
         76: {'author': 'Cicero',
              'work': {'title': 'Topica',
                       'meta': 'section',
                       'urn': 'urn:cts:latinLit:phi0474.phi042.perseus-lat1-simple'}},
         77: {'author': 'Cicero',
              'work': {'title': 'Tusculanae Disputationes',
                       'meta': 'chapter-section',
                       'urn': 'urn:cts:latinLit:phi0474.phi049.perseus-lat1-simple'}},
         78: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Carminum minorum corpusculum',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa001.perseus-lat2-simple'}},
         79: {'author': 'Claudianus, Claudius',
              'work': {'title': 'de bello Gildonico',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa002.perseus-lat2-simple'}},
         80: {'author': 'Claudianus, Claudius',
              'work': {'title': 'de Bello Gothico',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa003.perseus-lat2-simple'}},
         81: {'author': 'Claudianus, Claudius',
              'work': {'title': 'de consulatu Stilichonis',
                       'meta': 'book-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa004.perseus-lat2-simple'}},
         82: {'author': 'Claudianus, Claudius',
              'work': {'title': 'de raptu Proserpinae',
                       'meta': 'book-poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa005.perseus-lat2-simple'}},
         83: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Epithalamium de nuptiis Honorii Augusti',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa006.perseus-lat2-simple'}},
         84: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Fescinnina de nuptiis Honorii Augusti',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa007.perseus-lat2-simple'}},
         85: {'author': 'Claudianus, Claudius',
              'work': {'title': 'In Eutropium',
                       'meta': 'book-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa008.perseus-lat2-simple'}},
         86: {'author': 'Claudianus, Claudius',
              'work': {'title': 'In Rufinum',
                       'meta': 'book-poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa009.perseus-lat2-simple'}},
         87: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Panegyricus de quarto consulatu Honorii Augusti',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa011.perseus-lat2-simple'}},
         88: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Panegyricus de sexto consulatu Honorii Augusti',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa012.perseus-lat2-simple'}},
         89: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Panegyricus de tertio consulatu Honorii Augusti',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa010.perseus-lat2-simple'}},
         90: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Panegyricus dictus Manlio Theodoro consuli',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa013.perseus-lat2-simple'}},
         91: {'author': 'Claudianus, Claudius',
              'work': {'title': 'Panegyricus dictus Probino et Olybrio consulibus',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:stoa0089.stoa014.perseus-lat2-simple'}},
         92: {'author': 'Columella, Lucius Junius Moderatus',
              'work': {'title': 'Res Rustica',
                       'meta': 'book-chapter-section',
                       'urn': 'urn:cts:latinLit:phi0845.phi002.perseus-lat3-simple'}},
         93: {'author': 'Curtius Rufus, Quintus',
              'work': {'title': 'Historiarum Alexandri Magni',
                       'meta': 'book-chapter-section',
                       'urn': 'urn:cts:latinLit:phi0860.phi001.perseus-lat2-simple'}},
         94: {'author': 'Florus, Lucius Annaeus',
              'work': {'title': 'Epitome Rerum Romanorum',
                       'meta': 'book-topic-chapter-section',
                       'urn': 'urn:cts:latinLit:phi1242.phi001.perseus-lat1-simple'}},
         95: {'author': 'Gellius, Aulus',
              'work': {'title': 'Noctes Atticae',
                       'meta': 'book-chapter-section',
                       'urn': 'urn:cts:latinLit:phi1254.phi001.perseus-lat1-simple'}},
         96: {'author': 'Horace',
              'work': {'title': 'Ars Poetica',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:phi0893.phi006.perseus-lat2-simple'}},
         97: {'author': 'Horace',
              'work': {'title': 'Carmen Saeculare',
                       'meta': 'line',
                       'urn': 'urn:cts:latinLit:phi0893.phi002.perseus-lat2-simple'}},
         98: {'author': 'Horace',
              'work': {'title': 'Epistulae',
                       'meta': 'book-poem-line',
                       'urn': 'urn:cts:latinLit:phi0893.phi005.perseus-lat2-simple'}},
         99: {'author': 'Horace',
              'work': {'title': 'Epodi',
                       'meta': 'poem-line',
                       'urn': 'urn:cts:latinLit:phi0893.phi003.perseus-lat2-simple'}},
         100: {'author': 'Horace',
               'work': {'title': 'Odes',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi0893.phi001.perseus-lat2-simple'}},
         101: {'author': 'Horace',
               'work': {'title': 'Satires',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi0893.phi004.perseus-lat2-simple'}},
         102: {'author': 'Jerome Saint D. 419 Or 20',
               'work': {'title': 'Epistolae',
                        'meta': 'letter-section',
                        'urn': 'urn:cts:latinLit:stoa0162.stoa004.perseus-lat2-simple'}},
         103: {'author': 'Juvenal',
               'work': {'title': 'Satires',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi1276.phi001.perseus-lat2-simple'}},
         104: {'author': 'Lucan',
               'work': {'title': 'Civil War',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi0917.phi001.perseus-lat2-simple'}},
         105: {'author': 'Lucretius',
               'work': {'title': 'De Rerum Natura',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi0550.phi001.perseus-lat1-simple'}},
         106: {'author': 'Martial',
               'work': {'title': 'Epigrammata',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi1294.phi002.perseus-lat2-simple'}},
         107: {'author': 'Minucius Felix, Marcus',
               'work': {'title': 'Octavius',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0203.stoa001.perseus-lat2-simple'}},
         108: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Agesilaus',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo017.perseus-lat2-simple'}},
         109: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Alcibiades',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo007.perseus-lat2-simple'}},
         110: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Aristides',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo003.perseus-lat2-simple'}},
         111: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Atticus',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo025.perseus-lat2-simple'}},
         112: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Cato',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo024.perseus-lat2-simple'}},
         113: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Chabrias',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo012.perseus-lat2-simple'}},
         114: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Cimon',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo005.perseus-lat2-simple'}},
         115: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Conon',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo009.perseus-lat2-simple'}},
         116: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Datames',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo014.perseus-lat2-simple'}},
         117: {'author': 'Nepos, Cornelius',
               'work': {'title': 'De Regibus',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo021.perseus-lat2-simple'}},
         118: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Dion',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo010.perseus-lat2-simple'}},
         119: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Epaminondas',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo015.perseus-lat2-simple'}},
         120: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Eumenes',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo018.perseus-lat2-simple'}},
         121: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Hamilcar',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo022.perseus-lat2-simple'}},
         122: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Hannibal',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo023.perseus-lat2-simple'}},
         123: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Iphicrates',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo011.perseus-lat2-simple'}},
         124: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Lysander',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo006.perseus-lat2-simple'}},
         125: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Miltiades',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo001.perseus-lat2-simple'}},
         126: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Pausanias',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo004.perseus-lat2-simple'}},
         127: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Pelopidas',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo016.perseus-lat2-simple'}},
         128: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Phocion',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo019.perseus-lat2-simple'}},
         129: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Themistocles',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo002.perseus-lat2-simple'}},
         130: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Thrasybulus',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo008.perseus-lat2-simple'}},
         131: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Timoleon',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo020.perseus-lat2-simple'}},
         132: {'author': 'Nepos, Cornelius',
               'work': {'title': 'Timotheus',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi0588.abo013.perseus-lat2-simple'}},
         133: {'author': 'Ovid',
               'work': {'title': 'Amores',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi0959.phi001.perseus-lat2-simple'}},
         134: {'author': 'Ovid',
               'work': {'title': 'Medicamina faciei femineae',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0959.phi003.perseus-lat2-simple'}},
         135: {'author': 'Ovid',
               'work': {'title': 'Ars Amatoria',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi0959.phi004.perseus-lat2-simple'}},
         136: {'author': 'Ovid',
               'work': {'title': 'Fasti',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi0959.phi007.perseus-lat2-simple'}},
         137: {'author': 'Ovid',
               'work': {'title': 'Epistulae',
                        'meta': 'poem-line',
                        'urn': 'urn:cts:latinLit:phi0959.phi002.perseus-lat2-simple'}},
         138: {'author': 'Ovid',
               'work': {'title': 'Ibis',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0959.phi010.perseus-lat2-simple'}},
         139: {'author': 'Ovid',
               'work': {'title': 'Ex Ponto',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi0959.phi009.perseus-lat2-simple'}},
         140: {'author': 'Ovid',
               'work': {'title': 'Metamorphoses',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi0959.phi006.perseus-lat2-simple'}},
         141: {'author': 'Ovid',
               'work': {'title': 'Remedia amoris',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0959.phi005.perseus-lat2-simple'}},
         142: {'author': 'Ovid',
               'work': {'title': 'Tristia',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi0959.phi008.perseus-lat2-simple'}},
         143: {'author': 'Paris, Julius',
               'work': {'title': 'Facta et Dicta Memorabilia',
                        'meta': 'book-chapter-section',
                        'urn': 'urn:cts:latinLit:phi1038.phi001.perseus-lat1-simple'}},
         144: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Asinaria',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi002.perseus-lat2-simple'}},
         145: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Aulularia',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi003.perseus-lat2-simple'}},
         146: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Bacchides',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi004.perseus-lat2-simple'}},
         147: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Captivi',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi005.perseus-lat2-simple'}},
         148: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Casina',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi006.perseus-lat2-simple'}},
         149: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Curculio',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi008.perseus-lat2-simple'}},
         150: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Epidicus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi009.perseus-lat2-simple'}},
         151: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Menaechmi',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi010.perseus-lat2-simple'}},
         152: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Mercator',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi011.perseus-lat2-simple'}},
         153: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Miles Gloriosus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi012.perseus-lat2-simple'}},
         154: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Mostellaria',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi013.perseus-lat2-simple'}},
         155: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Persa',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi014.perseus-lat2-simple'}},
         156: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Poenulus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi015.perseus-lat2-simple'}},
         157: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Rudens',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi017.perseus-lat2-simple'}},
         158: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Stichus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi018.perseus-lat2-simple'}},
         159: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Trinummus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi019.perseus-lat2-simple'}},
         160: {'author': 'Plautus, Titus Maccius',
               'work': {'title': 'Truculentus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0119.phi020.perseus-lat2-simple'}},
         161: {'author': 'Pliny, The Elder',
               'work': {'title': 'Naturalis Historia',
                        'meta': 'book-chapter',
                        'urn': 'urn:cts:latinLit:phi0978.phi001.perseus-lat2-simple'}},
         162: {'author': 'Pliny, The Younger',
               'work': {'title': 'Epistulae',
                        'meta': 'book-letter-section',
                        'urn': 'urn:cts:latinLit:phi1318.phi001.perseus-lat1-simple'}},
         163: {'author': 'Propertius, Sextus',
               'work': {'title': 'Elegies',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi0620.phi001.perseus-lat3-simple'}},
         164: {'author': 'Prudentius B. 348',
               'work': {'title': 'Apotheosis',
                        'meta': 'section-line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa005.perseus-lat2-simple'}},
         165: {'author': 'Prudentius B. 348',
               'work': {'title': 'Cathemerina',
                        'meta': 'poem-line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa004.perseus-lat2-simple'}},
         166: {'author': 'Prudentius B. 348',
               'work': {'title': 'Contra Orationem Symmachia',
                        'meta': 'book-section-line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa007.perseus-lat2-simple'}},
         167: {'author': 'Prudentius B. 348',
               'work': {'title': 'Dittochaeon',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa008.perseus-lat2-simple'}},
         168: {'author': 'Prudentius B. 348',
               'work': {'title': 'Epilogus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa009.perseus-lat2-simple'}},
         169: {'author': 'Prudentius B. 348',
               'work': {'title': 'Hamartigenia',
                        'meta': 'section-line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa006.perseus-lat2-simple'}},
         170: {'author': 'Prudentius B. 348',
               'work': {'title': 'Liber Peristephanon',
                        'meta': 'poem-line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa001.perseus-lat2-simple'}},
         171: {'author': 'Prudentius B. 348',
               'work': {'title': 'Praefetio',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa003.perseus-lat2-simple'}},
         172: {'author': 'Prudentius B. 348',
               'work': {'title': 'Psychomachia',
                        'meta': 'section-line',
                        'urn': 'urn:cts:latinLit:stoa0238.stoa002.perseus-lat2-simple'}},
         173: {'author': 'Quintus Tullius Cicero',
               'work': {'title': 'Commentariolum Petitionis',
                        'meta': 'section',
                        'urn': 'urn:cts:latinLit:phi0478.phi003.perseus-lat2-simple'}},
         174: {'author': 'Sallust',
               'work': {'title': 'Bellum Iugurthinum',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi0631.phi002.perseus-lat4-simple'}},
         175: {'author': 'Sallust',
               'work': {'title': 'Catilinae Coniuratio',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi0631.phi001.perseus-lat3-simple'}},
         176: {'author': 'Sallust',
               'work': {'title': 'Historiae',
                        'meta': 'book-section',
                        'urn': 'urn:cts:latinLit:phi0631.phi003.perseus-lat2-simple'}},
         177: {'author': 'Seneca, Lucius Annaeus, 55 B.C.-Ca. 39 A.D',
               'work': {'title': 'Controversiae',
                        'meta': 'book-chapter-section',
                        'urn': 'urn:cts:latinLit:phi1014.phi001.perseus-lat1-simple'}},
         178: {'author': 'Seneca, Lucius Annaeus, 55 B.C.-Ca. 39 A.D',
               'work': {'title': 'Excerpta Controversiae',
                        'meta': 'book-chapter',
                        'urn': 'urn:cts:latinLit:phi1014.phi002.perseus-lat1-simple'}},
         179: {'author': 'Seneca, Lucius Annaeus, 55 B.C.-Ca. 39 A.D',
               'work': {'title': 'Fragmenta',
                        'meta': 'fragment',
                        'urn': 'urn:cts:latinLit:phi1014.phi004.perseus-lat1-simple'}},
         180: {'author': 'Seneca, Lucius Annaeus, 55 B.C.-Ca. 39 A.D',
               'work': {'title': 'Suasoriae',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi1014.phi003.perseus-lat1-simple'}},
         181: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Agamemnon',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi007.perseus-lat2-simple'}},
         182: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Apocolocyntosis',
                        'meta': 'section',
                        'urn': 'urn:cts:latinLit:phi1017.phi011.perseus-lat2-simple'}},
         183: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'De Clementia',
                        'meta': 'book-chapter-section',
                        'urn': 'urn:cts:latinLit:phi1017.phi014.perseus-lat2-simple'}},
         184: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Hercules Furens',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi001.perseus-lat2-simple'}},
         185: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Hercules Oetaeus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi009.perseus-lat2-simple'}},
         186: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Medea',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi004.perseus-lat2-simple'}},
         187: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Octavia',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi010.perseus-lat2-simple'}},
         188: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Oedipus',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi006.perseus-lat2-simple'}},
         189: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Phaedra',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi005.perseus-lat2-simple'}},
         190: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Phoenissae',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi003.perseus-lat2-simple'}},
         191: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Thyestes',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi008.perseus-lat2-simple'}},
         192: {'author': 'Seneca, Lucius Annaeus (Plays)',
               'work': {'title': 'Troades Furens',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi1017.phi002.perseus-lat2-simple'}},
         193: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de Brevitate Vitae',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa004.perseus-lat2-simple'}},
         194: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de consolatione ad Helviam',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa006.perseus-lat2-simple'}},
         195: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de consolatione ad Marciam',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa007.perseus-lat2-simple'}},
         196: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de consolatione ad Polybium',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa008.perseus-lat2-simple'}},
         197: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de Constantia',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa009.perseus-lat2-simple'}},
         198: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de Ira',
                        'meta': 'book-chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa010.perseus-lat2-simple'}},
         199: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de Otio Sapientis',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa011.perseus-lat2-simple'}},
         200: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de Providentia',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa012.perseus-lat2-simple'}},
         201: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de Tranquilitate Animi',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa013.perseus-lat2-simple'}},
         202: {'author': 'Seneca, Lucius Annaeus',
               'work': {'title': 'de Vita Beata',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0255.stoa014.perseus-lat2-simple'}},
         203: {'author': 'Silius Italicus, Tiberius Catius',
               'work': {'title': 'Punica',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi1345.phi001.perseus-lat2-simple'}},
         204: {'author': 'Statius, P. Papinius (Publius Papinius)',
               'work': {'title': 'Achilleis',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi1020.phi003.perseus-lat2-simple'}},
         205: {'author': 'Statius, P. Papinius (Publius Papinius)',
               'work': {'title': 'Silvae',
                        'meta': 'book-poem-line',
                        'urn': 'urn:cts:latinLit:phi1020.phi002.perseus-lat2-simple'}},
         206: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Caligula',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo014.perseus-lat2-simple'}},
         207: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Divus Augustus',
                        'meta': 'unknown',
                        'urn': 'urn:cts:latinLit:phi1348.abo012.perseus-lat2-simple'}},
         208: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Divus Claudius',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo015.perseus-lat2-simple'}},
         209: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Divus Julius',
                        'meta': 'unknown',
                        'urn': 'urn:cts:latinLit:phi1348.abo011.perseus-lat2-simple'}},
         210: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Divus Titus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo021.perseus-lat2-simple'}},
         211: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Divus Vespasianus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo020.perseus-lat2-simple'}},
         212: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Domitianus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo022.perseus-lat2-simple'}},
         213: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Galba',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo017.perseus-lat2-simple'}},
         214: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Nero',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo016.perseus-lat2-simple'}},
         215: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Otho',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo018.perseus-lat2-simple'}},
         216: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Tiberius',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo013.perseus-lat2-simple'}},
         217: {'author': 'Suetonius Ca. 69-Ca. 122',
               'work': {'title': 'Vitellius',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi1348.abo019.perseus-lat2-simple'}},
         218: {'author': 'Tacitus, Cornelius',
               'work': {'title': 'Agricola',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi1351.phi001.perseus-lat1-simple'}},
         219: {'author': 'Tacitus, Cornelius',
               'work': {'title': 'Germania',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:phi1351.phi002.perseus-lat1-simple'}},
         220: {'author': 'Terence',
               'work': {'title': 'Andria',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0134.phi001.perseus-lat2-simple'}},
         221: {'author': 'Terence',
               'work': {'title': 'Phormio',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0134.phi004.perseus-lat2-simple'}},
         222: {'author': 'Terence',
               'work': {'title': 'The Brothers',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0134.phi006.perseus-lat2-simple'}},
         223: {'author': 'Terence',
               'work': {'title': 'The Eunuch',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0134.phi003.perseus-lat2-simple'}},
         224: {'author': 'Terence',
               'work': {'title': 'The Mother-in-Law',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0134.phi005.perseus-lat2-simple'}},
         225: {'author': 'Terence',
               'work': {'title': 'The Self-Tormenter',
                        'meta': 'line',
                        'urn': 'urn:cts:latinLit:phi0134.phi002.perseus-lat2-simple'}},
         226: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Ad Martyras',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa001.opp-lat1-simple'}},
         227: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Ad Nationes Libri Duo',
                        'meta': 'book-chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa002.opp-lat1-simple'}},
         228: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Ad Scapulam',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa003.opp-lat1-simple'}},
         229: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Ad Uxorem',
                        'meta': 'book-chapter',
                        'urn': 'urn:cts:latinLit:stoa0276.stoa002.opp-lat1-simple'}},
         230: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Adversus Hermogenem',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa004.opp-lat1-simple'}},
         231: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Adversus Judaeos Liber',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa005.opp-lat1-simple'}},
         232: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Adversus Marcionem',
                        'meta': 'book-chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa006.opp-lat1-simple'}},
         233: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Adversus Praxean',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa007.opp-lat1-simple'}},
         234: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Adversus Valentinianos',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa008.opp-lat1-simple'}},
         235: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Apologeticum',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa009.perseus-lat2-simple'}},
         236: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Anima',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa010.opp-lat1-simple'}},
         237: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Baptismo',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa011.opp-lat1-simple'}},
         238: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Carne Christi',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa012.opp-lat1-simple'}},
         239: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Carnis Resurrectione',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa026.opp-lat1-simple'}},
         240: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Corona',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa013.opp-lat1-simple'}},
         241: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Cultu Feminarum',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa014.opp-lat1-simple'}},
         242: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Exhortatione Castitatis Liber',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa015.opp-lat1-simple'}},
         243: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Fuga in Persecutione',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa016.opp-lat1-simple'}},
         244: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De idolatria',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa017.opp-lat1-simple'}},
         245: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De ieiunio adversus psychicos',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa018.opp-lat1-simple'}},
         246: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Monogamia',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa019.opp-lat1-simple'}},
         247: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Oratione',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa020.opp-lat1-simple'}},
         248: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Paenitentia',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa021.opp-lat1-simple'}},
         249: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Pallio',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa022.opp-lat1-simple'}},
         250: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Patientia',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa023.opp-lat1-simple'}},
         251: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Praescriptionibus Hereticorum',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa024.opp-lat1-simple'}},
         252: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Pudicitia',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa025.opp-lat1-simple'}},
         253: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Spectaculis',
                        'meta': 'chapter-section',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa027.perseus-lat2-simple'}},
         254: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Testimionio Animae',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa028.opp-lat1-simple'}},
         255: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'De Virginibus Velandis',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa029.opp-lat1-simple'}},
         256: {'author': 'Tertullian Ca. 160-Ca. 230',
               'work': {'title': 'Scorpiace',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:stoa0275.stoa030.opp-lat1-simple'}},
         257: {'author': 'Tibullus',
               'work': {'title': 'Elegiae',
                        'meta': 'poem-line',
                        'urn': 'urn:cts:latinLit:phi0660.phi003.perseus-lat2-simple'}},
         258: {'author': 'Valerius Flaccus, Gaius',
               'work': {'title': 'Argonautica',
                        'meta': 'book-line',
                        'urn': 'urn:cts:latinLit:phi1035.phi001.perseus-lat2-simple'}},
         259: {'author': 'Virgil',
               'work': {'title': 'Aeneid',
                        'meta': 'Book-line',
                        'urn': 'urn:cts:latinLit:phi0690.phi003.perseus-lat2-simple'}},
         260: {'author': 'Virgil',
               'work': {'title': 'Eclogues',
                        'meta': 'poem-line',
                        'urn': 'urn:cts:latinLit:phi0690.phi001.perseus-lat2-simple'}},
         261: {'author': 'Virgil',
               'work': {'title': 'Georgics',
                        'meta': 'poem-line',
                        'urn': 'urn:cts:latinLit:phi0690.phi002.perseus-lat2-simple'}},
         262: {'author': 'Vitruvius Pollio',
               'work': {'title': 'On Architecture',
                        'meta': 'book-chapter-section',
                        'urn': 'urn:cts:latinLit:phi1056.phi001.perseus-lat1-simple'}},
         263: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Alexander Severus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi018.perseus-lat2-simple'}},
         264: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Antoninus Caracalla',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi013.perseus-lat2-simple'}},
         265: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Antoninus Geta',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi014.perseus-lat2-simple'}},
         266: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Antoninus Heliogobalus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi017.perseus-lat2-simple'}},
         267: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Antoninus Pius',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi003.perseus-lat2-simple'}},
         268: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Avidius Casius',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi006.perseus-lat2-simple'}},
         269: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Carus et Carinus et Numerianus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi030.perseus-lat2-simple'}},
         270: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Clodinus Albinus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi012.perseus-lat2-simple'}},
         271: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Commodus Antoninus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi007.perseus-lat2-simple'}},
         272: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'De Vita Hadriani',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi001.perseus-lat2-simple'}},
         273: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Diadumenus Antoninus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi016.perseus-lat2-simple'}},
         274: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Didius Julianus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi009.perseus-lat2-simple'}},
         275: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Divus Aurelianus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi026.perseus-lat2-simple'}},
         276: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Divus Claudius',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi025.perseus-lat2-simple'}},
         277: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Firmus Saturninus, Proculus et Bonosus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi029.perseus-lat2-simple'}},
         278: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Gallieni Duo',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi023.perseus-lat2-simple'}},
         279: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Goridani Tres',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi020.perseus-lat2-simple'}},
         280: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Helius',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi002.perseus-lat2-simple'}},
         281: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Marcus Antoninus Philosophus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi004.perseus-lat2-simple'}},
         282: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Maximini Duo',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi019.perseus-lat2-simple'}},
         283: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Maximus et Balbinus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi021.perseus-lat2-simple'}},
         284: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Opilius Macrinus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi015.perseus-lat2-simple'}},
         285: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Pertinax',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi008.perseus-lat2-simple'}},
         286: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Pescennius Niger',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi011.perseus-lat2-simple'}},
         287: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Probus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi028.perseus-lat2-simple'}},
         288: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Severus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi010.perseus-lat2-simple'}},
         289: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Tacitus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi027.perseus-lat2-simple'}},
         290: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Tyranni Triginta',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi024.perseus-lat2-simple'}},
         291: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Valeriani Duo',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi022.perseus-lat2-simple'}},
         292: {'author': 'Vopiscus, Flavius Fl. 3./4. Jh',
               'work': {'title': 'Verus',
                        'meta': 'chapter',
                        'urn': 'urn:cts:latinLit:phi2331.phi005.perseus-lat2-simple'}}}

reverse_index = {'Ammianus Marcellinus': [{'title': 'Rerum Gestarum',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:stoa0023.stoa001.perseus-lat1-simple',
               'docnum': 0}],
             'Apuleius': [{'title': 'Apologia',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi1212.phi001.perseus-lat1-simple',
               'docnum': 1},
              {'title': 'Florida',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi1212.phi003.perseus-lat1-simple',
               'docnum': 2},
              {'title': 'Metamorphoses',
               'meta': 'book-chapteer',
               'urn': 'urn:cts:latinLit:phi1212.phi002.perseus-lat1-simple',
               'docnum': 3}],
             'Ausonius, Decimus Magnus': [{'title': 'Bissula',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa001.perseus-lat2-simple',
               'docnum': 4},
              {'title': 'Caesares',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa002.perseus-lat2-simple',
               'docnum': 5},
              {'title': 'Commemoratio Professorum Burdigalensium',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa004.perseus-lat2-simple',
               'docnum': 6},
              {'title': 'De Herediolo',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa006.perseus-lat2-simple',
               'docnum': 7},
              {'title': 'Eclogarum Liber',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa007.perseus-lat2-simple',
               'docnum': 8},
              {'title': 'Ephemeris',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa008.perseus-lat2-simple',
               'docnum': 9},
              {'title': 'Epicedion in Patrem',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa009.perseus-lat2-simple',
               'docnum': 10},
              {'title': 'Epigrammaton Liber',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa010.perseus-lat2-simple',
               'docnum': 11},
              {'title': 'Epistulae',
               'meta': 'letter-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa011.perseus-lat2-simple',
               'docnum': 12},
              {'title': 'Epitaphia',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa012.perseus-lat2-simple',
               'docnum': 13},
              {'title': 'Genethliacon ad Ausonium Nepotem',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa013.perseus-lat2-simple',
               'docnum': 14},
              {'title': 'Gratiarum Actio',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:stoa0045.stoa014.perseus-lat2-simple',
               'docnum': 15},
              {'title': 'Griphus Ternarii Numeri',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa015.perseus-lat2-simple',
               'docnum': 16},
              {'title': 'Liber Protrepticus ad Nepotem',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa016.perseus-lat2-simple',
               'docnum': 17},
              {'title': 'Mosella',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa019.perseus-lat2-simple',
               'docnum': 18},
              {'title': 'Oratio Versibus Rhopalicis',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa020.perseus-lat2-simple',
               'docnum': 19},
              {'title': 'Ordo Urbium Nobilium',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa021.perseus-lat2-simple',
               'docnum': 20},
              {'title': 'Parentalia',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa022.perseus-lat2-simple',
               'docnum': 21},
              {'title': 'Praefatiunculae',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa025.perseus-lat2-simple',
               'docnum': 22},
              {'title': 'Precationes',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa026.perseus-lat2-simple',
               'docnum': 23},
              {'title': 'Technopaegnion',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa028.perseus-lat2-simple',
               'docnum': 24},
              {'title': 'Versus Paschales Prosodic',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0045.stoa027.perseus-lat2-simple',
               'docnum': 25}],
             'Boethius D. 524': [{'title': 'De consolatione philosophiae',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:stoa0058.stoa001.perseus-lat2-simple',
               'docnum': 26},
              {'title': 'De Fide Catholica',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:stoa0058.stoa006.perseus-lat1-simple',
               'docnum': 27},
              {'title': 'Liber De Persona et Duabus Naturis Contra Eutychen Et Nestorium',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:stoa0058.stoa023.perseus-lat1-simple',
               'docnum': 28},
              {'title': 'Quomodo Substantiae in Eo Quod Sint Bonae Sint Cum Non Sint Substanialia Bona',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:stoa0058.stoa003.perseus-lat1-simple',
               'docnum': 29},
              {'title': 'Quomodo Trinitas Unus Deus Ac Non Tres Dii (De Trinitate)',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:stoa0058.stoa025.perseus-lat1-simple',
               'docnum': 30},
              {'title': 'Utrum Pater Et Filius Ac Spiritus Sanctus De Divinitate Substantialiter Praedicentur Liber',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:stoa0058.stoa028.perseus-lat1-simple',
               'docnum': 31}],
             'Caesar, Julius': [{'title': 'Gallic War',
               'meta': 'Book-Chapter-Section',
               'urn': 'urn:cts:latinLit:phi0448.phi001.perseus-lat2-simple',
               'docnum': 32}],
             'Celsus, Aulus Cornelius': [{'title': 'De Medicina',
               'meta': 'book-chapter',
               'urn': 'urn:cts:latinLit:phi0836.phi002.perseus-lat5-simple',
               'docnum': 33}],
             'Cicero': [{'title': 'Academica',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:phi0474.phi045.perseus-lat1-simple',
               'docnum': 34},
              {'title': 'Orationes de Lege Agraria',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi011.perseus-lat2-simple',
               'docnum': 35},
              {'title': 'Brutus',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi039.perseus-lat1-simple',
               'docnum': 36},
              {'title': 'De Amicitia',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi052.perseus-lat1-simple',
               'docnum': 37},
              {'title': 'De Divinatione',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:phi0474.phi053.perseus-lat1-simple',
               'docnum': 38},
              {'title': 'De Fato',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi054.perseus-lat1-simple',
               'docnum': 39},
              {'title': 'de Finibus Bonorum et Malorum',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:phi0474.phi048.perseus-lat1-simple',
               'docnum': 40},
              {'title': 'De Inventione',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi036.perseus-lat1-simple',
               'docnum': 41},
              {'title': 'de Natura Deorum',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi050.perseus-lat1-simple',
               'docnum': 42},
              {'title': 'De Officiis',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:phi0474.phi055.perseus-lat1-simple',
               'docnum': 43},
              {'title': 'De Optimo Genere Oratorum',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi041.perseus-lat1-simple',
               'docnum': 44},
              {'title': 'De Republica',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:phi0474.phi043.perseus-lat1-simple',
               'docnum': 45},
              {'title': 'De Senectute',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi051.perseus-lat1-simple',
               'docnum': 46},
              {'title': 'In Caecilium',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi004.perseus-lat2-simple',
               'docnum': 47},
              {'title': 'Pro Archia',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi016.perseus-lat2-simple',
               'docnum': 48},
              {'title': 'For Marcus Caelius',
               'meta': 'unknown',
               'urn': 'urn:cts:latinLit:phi0474.phi024.perseus-lat2-simple',
               'docnum': 49},
              {'title': 'Pro Fonteio',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi007.perseus-lat2-simple',
               'docnum': 50},
              {'title': 'Pro P. Quinctio',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi001.perseus-lat2-simple',
               'docnum': 51},
              {'title': 'Pro Roscio comoedo',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi003.perseus-lat2-simple',
               'docnum': 52},
              {'title': 'Pro S. Roscio Amerino',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi002.perseus-lat2-simple',
               'docnum': 53},
              {'title': 'Pro Sulla',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi015.perseus-lat2-simple',
               'docnum': 54},
              {'title': 'In Catilinam',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi013.perseus-lat2-simple',
               'docnum': 55},
              {'title': 'Pro Cluentio',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi010.perseus-lat2-simple',
               'docnum': 56},
              {'title': 'Pro C. Rabiro perduellionis reo',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi012.perseus-lat2-simple',
               'docnum': 57},
              {'title': 'Pro Murena',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi014.perseus-lat2-simple',
               'docnum': 58},
              {'title': 'Pro Flacco',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi017.perseus-lat2-simple',
               'docnum': 59},
              {'title': 'Post reditum in senatu',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi019.perseus-lat2-simple',
               'docnum': 60},
              {'title': 'Letters to Atticus',
               'meta': 'book-letter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi057.perseus-lat1-simple',
               'docnum': 61},
              {'title': 'Letters to Brutus',
               'meta': 'book-letter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi059.perseus-lat1-simple',
               'docnum': 62},
              {'title': 'Letters to his brother Quintus',
               'meta': 'book-letter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi058.perseus-lat1-simple',
               'docnum': 63},
              {'title': 'Letters to his Friends',
               'meta': 'book-letter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi056.perseus-lat1-simple',
               'docnum': 64},
              {'title': 'Lucullus',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi046.perseus-lat1-simple',
               'docnum': 65},
              {'title': 'Pro A. Caecina',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi008.perseus-lat2-simple',
               'docnum': 66},
              {'title': 'Pro Tullio',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi0474.phi006.perseus-lat2-simple',
               'docnum': 67},
              {'title': 'On Oratory',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi037.perseus-lat1-simple',
               'docnum': 68},
              {'title': 'Pro lege manilia',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi009.perseus-lat2-simple',
               'docnum': 69},
              {'title': 'In Verrem',
               'meta': 'actio-book-section',
               'urn': 'urn:cts:latinLit:phi0474.phi005.perseus-lat2-simple',
               'docnum': 70},
              {'title': 'Orator',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi040.perseus-lat1-simple',
               'docnum': 71},
              {'title': 'Paradoxa Stoicorum',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:phi0474.phi047.perseus-lat1-simple',
               'docnum': 72},
              {'title': 'Partitiones Oratoriae',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi038.perseus-lat1-simple',
               'docnum': 73},
              {'title': 'Timaeus',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi072.perseus-lat1-simple',
               'docnum': 74},
              {'title': 'Post reditum ad populum',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi018.perseus-lat2-simple',
               'docnum': 75},
              {'title': 'Topica',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0474.phi042.perseus-lat1-simple',
               'docnum': 76},
              {'title': 'Tusculanae Disputationes',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0474.phi049.perseus-lat1-simple',
               'docnum': 77}],
             'Claudianus, Claudius': [{'title': 'Carminum minorum corpusculum',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa001.perseus-lat2-simple',
               'docnum': 78},
              {'title': 'de bello Gildonico',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa002.perseus-lat2-simple',
               'docnum': 79},
              {'title': 'de Bello Gothico',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa003.perseus-lat2-simple',
               'docnum': 80},
              {'title': 'de consulatu Stilichonis',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa004.perseus-lat2-simple',
               'docnum': 81},
              {'title': 'de raptu Proserpinae',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa005.perseus-lat2-simple',
               'docnum': 82},
              {'title': 'Epithalamium de nuptiis Honorii Augusti',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa006.perseus-lat2-simple',
               'docnum': 83},
              {'title': 'Fescinnina de nuptiis Honorii Augusti',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa007.perseus-lat2-simple',
               'docnum': 84},
              {'title': 'In Eutropium',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa008.perseus-lat2-simple',
               'docnum': 85},
              {'title': 'In Rufinum',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa009.perseus-lat2-simple',
               'docnum': 86},
              {'title': 'Panegyricus de quarto consulatu Honorii Augusti',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa011.perseus-lat2-simple',
               'docnum': 87},
              {'title': 'Panegyricus de sexto consulatu Honorii Augusti',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa012.perseus-lat2-simple',
               'docnum': 88},
              {'title': 'Panegyricus de tertio consulatu Honorii Augusti',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa010.perseus-lat2-simple',
               'docnum': 89},
              {'title': 'Panegyricus dictus Manlio Theodoro consuli',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa013.perseus-lat2-simple',
               'docnum': 90},
              {'title': 'Panegyricus dictus Probino et Olybrio consulibus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0089.stoa014.perseus-lat2-simple',
               'docnum': 91}],
             'Columella, Lucius Junius Moderatus': [{'title': 'Res Rustica',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:phi0845.phi002.perseus-lat3-simple',
               'docnum': 92}],
             'Curtius Rufus, Quintus': [{'title': 'Historiarum Alexandri Magni',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:phi0860.phi001.perseus-lat2-simple',
               'docnum': 93}],
             'Florus, Lucius Annaeus': [{'title': 'Epitome Rerum Romanorum',
               'meta': 'book-topic-chapter-section',
               'urn': 'urn:cts:latinLit:phi1242.phi001.perseus-lat1-simple',
               'docnum': 94}],
             'Gellius, Aulus': [{'title': 'Noctes Atticae',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:phi1254.phi001.perseus-lat1-simple',
               'docnum': 95}],
             'Horace': [{'title': 'Ars Poetica',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0893.phi006.perseus-lat2-simple',
               'docnum': 96},
              {'title': 'Carmen Saeculare',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0893.phi002.perseus-lat2-simple',
               'docnum': 97},
              {'title': 'Epistulae',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi0893.phi005.perseus-lat2-simple',
               'docnum': 98},
              {'title': 'Epodi',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:phi0893.phi003.perseus-lat2-simple',
               'docnum': 99},
              {'title': 'Odes',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi0893.phi001.perseus-lat2-simple',
               'docnum': 100},
              {'title': 'Satires',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi0893.phi004.perseus-lat2-simple',
               'docnum': 101}],
             'Jerome Saint D. 419 Or 20': [{'title': 'Epistolae',
               'meta': 'letter-section',
               'urn': 'urn:cts:latinLit:stoa0162.stoa004.perseus-lat2-simple',
               'docnum': 102}],
             'Juvenal': [{'title': 'Satires',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi1276.phi001.perseus-lat2-simple',
               'docnum': 103}],
             'Lucan': [{'title': 'Civil War',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi0917.phi001.perseus-lat2-simple',
               'docnum': 104}],
             'Lucretius': [{'title': 'De Rerum Natura',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi0550.phi001.perseus-lat1-simple',
               'docnum': 105}],
             'Martial': [{'title': 'Epigrammata',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi1294.phi002.perseus-lat2-simple',
               'docnum': 106}],
             'Minucius Felix, Marcus': [{'title': 'Octavius',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0203.stoa001.perseus-lat2-simple',
               'docnum': 107}],
             'Nepos, Cornelius': [{'title': 'Agesilaus',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo017.perseus-lat2-simple',
               'docnum': 108},
              {'title': 'Alcibiades',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo007.perseus-lat2-simple',
               'docnum': 109},
              {'title': 'Aristides',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo003.perseus-lat2-simple',
               'docnum': 110},
              {'title': 'Atticus',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo025.perseus-lat2-simple',
               'docnum': 111},
              {'title': 'Cato',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo024.perseus-lat2-simple',
               'docnum': 112},
              {'title': 'Chabrias',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo012.perseus-lat2-simple',
               'docnum': 113},
              {'title': 'Cimon',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo005.perseus-lat2-simple',
               'docnum': 114},
              {'title': 'Conon',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo009.perseus-lat2-simple',
               'docnum': 115},
              {'title': 'Datames',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo014.perseus-lat2-simple',
               'docnum': 116},
              {'title': 'De Regibus',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo021.perseus-lat2-simple',
               'docnum': 117},
              {'title': 'Dion',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo010.perseus-lat2-simple',
               'docnum': 118},
              {'title': 'Epaminondas',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo015.perseus-lat2-simple',
               'docnum': 119},
              {'title': 'Eumenes',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo018.perseus-lat2-simple',
               'docnum': 120},
              {'title': 'Hamilcar',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo022.perseus-lat2-simple',
               'docnum': 121},
              {'title': 'Hannibal',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo023.perseus-lat2-simple',
               'docnum': 122},
              {'title': 'Iphicrates',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo011.perseus-lat2-simple',
               'docnum': 123},
              {'title': 'Lysander',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo006.perseus-lat2-simple',
               'docnum': 124},
              {'title': 'Miltiades',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo001.perseus-lat2-simple',
               'docnum': 125},
              {'title': 'Pausanias',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo004.perseus-lat2-simple',
               'docnum': 126},
              {'title': 'Pelopidas',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo016.perseus-lat2-simple',
               'docnum': 127},
              {'title': 'Phocion',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo019.perseus-lat2-simple',
               'docnum': 128},
              {'title': 'Themistocles',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo002.perseus-lat2-simple',
               'docnum': 129},
              {'title': 'Thrasybulus',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo008.perseus-lat2-simple',
               'docnum': 130},
              {'title': 'Timoleon',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo020.perseus-lat2-simple',
               'docnum': 131},
              {'title': 'Timotheus',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi0588.abo013.perseus-lat2-simple',
               'docnum': 132}],
             'Ovid': [{'title': 'Amores',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi0959.phi001.perseus-lat2-simple',
               'docnum': 133},
              {'title': 'Medicamina faciei femineae',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0959.phi003.perseus-lat2-simple',
               'docnum': 134},
              {'title': 'Ars Amatoria',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi0959.phi004.perseus-lat2-simple',
               'docnum': 135},
              {'title': 'Fasti',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi0959.phi007.perseus-lat2-simple',
               'docnum': 136},
              {'title': 'Epistulae',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:phi0959.phi002.perseus-lat2-simple',
               'docnum': 137},
              {'title': 'Ibis',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0959.phi010.perseus-lat2-simple',
               'docnum': 138},
              {'title': 'Ex Ponto',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi0959.phi009.perseus-lat2-simple',
               'docnum': 139},
              {'title': 'Metamorphoses',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi0959.phi006.perseus-lat2-simple',
               'docnum': 140},
              {'title': 'Remedia amoris',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0959.phi005.perseus-lat2-simple',
               'docnum': 141},
              {'title': 'Tristia',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi0959.phi008.perseus-lat2-simple',
               'docnum': 142}],
             'Paris, Julius': [{'title': 'Facta et Dicta Memorabilia',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:phi1038.phi001.perseus-lat1-simple',
               'docnum': 143}],
             'Plautus, Titus Maccius': [{'title': 'Asinaria',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi002.perseus-lat2-simple',
               'docnum': 144},
              {'title': 'Aulularia',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi003.perseus-lat2-simple',
               'docnum': 145},
              {'title': 'Bacchides',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi004.perseus-lat2-simple',
               'docnum': 146},
              {'title': 'Captivi',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi005.perseus-lat2-simple',
               'docnum': 147},
              {'title': 'Casina',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi006.perseus-lat2-simple',
               'docnum': 148},
              {'title': 'Curculio',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi008.perseus-lat2-simple',
               'docnum': 149},
              {'title': 'Epidicus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi009.perseus-lat2-simple',
               'docnum': 150},
              {'title': 'Menaechmi',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi010.perseus-lat2-simple',
               'docnum': 151},
              {'title': 'Mercator',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi011.perseus-lat2-simple',
               'docnum': 152},
              {'title': 'Miles Gloriosus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi012.perseus-lat2-simple',
               'docnum': 153},
              {'title': 'Mostellaria',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi013.perseus-lat2-simple',
               'docnum': 154},
              {'title': 'Persa',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi014.perseus-lat2-simple',
               'docnum': 155},
              {'title': 'Poenulus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi015.perseus-lat2-simple',
               'docnum': 156},
              {'title': 'Rudens',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi017.perseus-lat2-simple',
               'docnum': 157},
              {'title': 'Stichus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi018.perseus-lat2-simple',
               'docnum': 158},
              {'title': 'Trinummus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi019.perseus-lat2-simple',
               'docnum': 159},
              {'title': 'Truculentus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0119.phi020.perseus-lat2-simple',
               'docnum': 160}],
             'Pliny, The Elder': [{'title': 'Naturalis Historia',
               'meta': 'book-chapter',
               'urn': 'urn:cts:latinLit:phi0978.phi001.perseus-lat2-simple',
               'docnum': 161}],
             'Pliny, The Younger': [{'title': 'Epistulae',
               'meta': 'book-letter-section',
               'urn': 'urn:cts:latinLit:phi1318.phi001.perseus-lat1-simple',
               'docnum': 162}],
             'Propertius, Sextus': [{'title': 'Elegies',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi0620.phi001.perseus-lat3-simple',
               'docnum': 163}],
             'Prudentius B. 348': [{'title': 'Apotheosis',
               'meta': 'section-line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa005.perseus-lat2-simple',
               'docnum': 164},
              {'title': 'Cathemerina',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa004.perseus-lat2-simple',
               'docnum': 165},
              {'title': 'Contra Orationem Symmachia',
               'meta': 'book-section-line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa007.perseus-lat2-simple',
               'docnum': 166},
              {'title': 'Dittochaeon',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa008.perseus-lat2-simple',
               'docnum': 167},
              {'title': 'Epilogus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa009.perseus-lat2-simple',
               'docnum': 168},
              {'title': 'Hamartigenia',
               'meta': 'section-line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa006.perseus-lat2-simple',
               'docnum': 169},
              {'title': 'Liber Peristephanon',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa001.perseus-lat2-simple',
               'docnum': 170},
              {'title': 'Praefetio',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa003.perseus-lat2-simple',
               'docnum': 171},
              {'title': 'Psychomachia',
               'meta': 'section-line',
               'urn': 'urn:cts:latinLit:stoa0238.stoa002.perseus-lat2-simple',
               'docnum': 172}],
             'Quintus Tullius Cicero': [{'title': 'Commentariolum Petitionis',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi0478.phi003.perseus-lat2-simple',
               'docnum': 173}],
             'Sallust': [{'title': 'Bellum Iugurthinum',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi0631.phi002.perseus-lat4-simple',
               'docnum': 174},
              {'title': 'Catilinae Coniuratio',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi0631.phi001.perseus-lat3-simple',
               'docnum': 175},
              {'title': 'Historiae',
               'meta': 'book-section',
               'urn': 'urn:cts:latinLit:phi0631.phi003.perseus-lat2-simple',
               'docnum': 176}],
             'Seneca, Lucius Annaeus, 55 B.C.-Ca. 39 A.D': [{'title': 'Controversiae',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:phi1014.phi001.perseus-lat1-simple',
               'docnum': 177},
              {'title': 'Excerpta Controversiae',
               'meta': 'book-chapter',
               'urn': 'urn:cts:latinLit:phi1014.phi002.perseus-lat1-simple',
               'docnum': 178},
              {'title': 'Fragmenta',
               'meta': 'fragment',
               'urn': 'urn:cts:latinLit:phi1014.phi004.perseus-lat1-simple',
               'docnum': 179},
              {'title': 'Suasoriae',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi1014.phi003.perseus-lat1-simple',
               'docnum': 180}],
             'Seneca, Lucius Annaeus (Plays)': [{'title': 'Agamemnon',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi007.perseus-lat2-simple',
               'docnum': 181},
              {'title': 'Apocolocyntosis',
               'meta': 'section',
               'urn': 'urn:cts:latinLit:phi1017.phi011.perseus-lat2-simple',
               'docnum': 182},
              {'title': 'De Clementia',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:phi1017.phi014.perseus-lat2-simple',
               'docnum': 183},
              {'title': 'Hercules Furens',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi001.perseus-lat2-simple',
               'docnum': 184},
              {'title': 'Hercules Oetaeus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi009.perseus-lat2-simple',
               'docnum': 185},
              {'title': 'Medea',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi004.perseus-lat2-simple',
               'docnum': 186},
              {'title': 'Octavia',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi010.perseus-lat2-simple',
               'docnum': 187},
              {'title': 'Oedipus',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi006.perseus-lat2-simple',
               'docnum': 188},
              {'title': 'Phaedra',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi005.perseus-lat2-simple',
               'docnum': 189},
              {'title': 'Phoenissae',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi003.perseus-lat2-simple',
               'docnum': 190},
              {'title': 'Thyestes',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi008.perseus-lat2-simple',
               'docnum': 191},
              {'title': 'Troades Furens',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi1017.phi002.perseus-lat2-simple',
               'docnum': 192}],
             'Seneca, Lucius Annaeus': [{'title': 'de Brevitate Vitae',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa004.perseus-lat2-simple',
               'docnum': 193},
              {'title': 'de consolatione ad Helviam',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa006.perseus-lat2-simple',
               'docnum': 194},
              {'title': 'de consolatione ad Marciam',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa007.perseus-lat2-simple',
               'docnum': 195},
              {'title': 'de consolatione ad Polybium',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa008.perseus-lat2-simple',
               'docnum': 196},
              {'title': 'de Constantia',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa009.perseus-lat2-simple',
               'docnum': 197},
              {'title': 'de Ira',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa010.perseus-lat2-simple',
               'docnum': 198},
              {'title': 'de Otio Sapientis',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa011.perseus-lat2-simple',
               'docnum': 199},
              {'title': 'de Providentia',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa012.perseus-lat2-simple',
               'docnum': 200},
              {'title': 'de Tranquilitate Animi',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa013.perseus-lat2-simple',
               'docnum': 201},
              {'title': 'de Vita Beata',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0255.stoa014.perseus-lat2-simple',
               'docnum': 202}],
             'Silius Italicus, Tiberius Catius': [{'title': 'Punica',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi1345.phi001.perseus-lat2-simple',
               'docnum': 203}],
             'Statius, P. Papinius (Publius Papinius)': [{'title': 'Achilleis',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi1020.phi003.perseus-lat2-simple',
               'docnum': 204},
              {'title': 'Silvae',
               'meta': 'book-poem-line',
               'urn': 'urn:cts:latinLit:phi1020.phi002.perseus-lat2-simple',
               'docnum': 205}],
             'Suetonius Ca. 69-Ca. 122': [{'title': 'Caligula',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo014.perseus-lat2-simple',
               'docnum': 206},
              {'title': 'Divus Augustus',
               'meta': 'unknown',
               'urn': 'urn:cts:latinLit:phi1348.abo012.perseus-lat2-simple',
               'docnum': 207},
              {'title': 'Divus Claudius',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo015.perseus-lat2-simple',
               'docnum': 208},
              {'title': 'Divus Julius',
               'meta': 'unknown',
               'urn': 'urn:cts:latinLit:phi1348.abo011.perseus-lat2-simple',
               'docnum': 209},
              {'title': 'Divus Titus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo021.perseus-lat2-simple',
               'docnum': 210},
              {'title': 'Divus Vespasianus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo020.perseus-lat2-simple',
               'docnum': 211},
              {'title': 'Domitianus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo022.perseus-lat2-simple',
               'docnum': 212},
              {'title': 'Galba',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo017.perseus-lat2-simple',
               'docnum': 213},
              {'title': 'Nero',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo016.perseus-lat2-simple',
               'docnum': 214},
              {'title': 'Otho',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo018.perseus-lat2-simple',
               'docnum': 215},
              {'title': 'Tiberius',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo013.perseus-lat2-simple',
               'docnum': 216},
              {'title': 'Vitellius',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi1348.abo019.perseus-lat2-simple',
               'docnum': 217}],
             'Tacitus, Cornelius': [{'title': 'Agricola',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi1351.phi001.perseus-lat1-simple',
               'docnum': 218},
              {'title': 'Germania',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:phi1351.phi002.perseus-lat1-simple',
               'docnum': 219}],
             'Terence': [{'title': 'Andria',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0134.phi001.perseus-lat2-simple',
               'docnum': 220},
              {'title': 'Phormio',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0134.phi004.perseus-lat2-simple',
               'docnum': 221},
              {'title': 'The Brothers',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0134.phi006.perseus-lat2-simple',
               'docnum': 222},
              {'title': 'The Eunuch',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0134.phi003.perseus-lat2-simple',
               'docnum': 223},
              {'title': 'The Mother-in-Law',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0134.phi005.perseus-lat2-simple',
               'docnum': 224},
              {'title': 'The Self-Tormenter',
               'meta': 'line',
               'urn': 'urn:cts:latinLit:phi0134.phi002.perseus-lat2-simple',
               'docnum': 225}],
             'Tertullian Ca. 160-Ca. 230': [{'title': 'Ad Martyras',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa001.opp-lat1-simple',
               'docnum': 226},
              {'title': 'Ad Nationes Libri Duo',
               'meta': 'book-chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa002.opp-lat1-simple',
               'docnum': 227},
              {'title': 'Ad Scapulam',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa003.opp-lat1-simple',
               'docnum': 228},
              {'title': 'Ad Uxorem',
               'meta': 'book-chapter',
               'urn': 'urn:cts:latinLit:stoa0276.stoa002.opp-lat1-simple',
               'docnum': 229},
              {'title': 'Adversus Hermogenem',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa004.opp-lat1-simple',
               'docnum': 230},
              {'title': 'Adversus Judaeos Liber',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa005.opp-lat1-simple',
               'docnum': 231},
              {'title': 'Adversus Marcionem',
               'meta': 'book-chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa006.opp-lat1-simple',
               'docnum': 232},
              {'title': 'Adversus Praxean',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa007.opp-lat1-simple',
               'docnum': 233},
              {'title': 'Adversus Valentinianos',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa008.opp-lat1-simple',
               'docnum': 234},
              {'title': 'Apologeticum',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0275.stoa009.perseus-lat2-simple',
               'docnum': 235},
              {'title': 'De Anima',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa010.opp-lat1-simple',
               'docnum': 236},
              {'title': 'De Baptismo',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa011.opp-lat1-simple',
               'docnum': 237},
              {'title': 'De Carne Christi',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa012.opp-lat1-simple',
               'docnum': 238},
              {'title': 'De Carnis Resurrectione',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa026.opp-lat1-simple',
               'docnum': 239},
              {'title': 'De Corona',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa013.opp-lat1-simple',
               'docnum': 240},
              {'title': 'De Cultu Feminarum',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa014.opp-lat1-simple',
               'docnum': 241},
              {'title': 'De Exhortatione Castitatis Liber',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa015.opp-lat1-simple',
               'docnum': 242},
              {'title': 'De Fuga in Persecutione',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa016.opp-lat1-simple',
               'docnum': 243},
              {'title': 'De idolatria',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa017.opp-lat1-simple',
               'docnum': 244},
              {'title': 'De ieiunio adversus psychicos',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa018.opp-lat1-simple',
               'docnum': 245},
              {'title': 'De Monogamia',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa019.opp-lat1-simple',
               'docnum': 246},
              {'title': 'De Oratione',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa020.opp-lat1-simple',
               'docnum': 247},
              {'title': 'De Paenitentia',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa021.opp-lat1-simple',
               'docnum': 248},
              {'title': 'De Pallio',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa022.opp-lat1-simple',
               'docnum': 249},
              {'title': 'De Patientia',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa023.opp-lat1-simple',
               'docnum': 250},
              {'title': 'De Praescriptionibus Hereticorum',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa024.opp-lat1-simple',
               'docnum': 251},
              {'title': 'De Pudicitia',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa025.opp-lat1-simple',
               'docnum': 252},
              {'title': 'De Spectaculis',
               'meta': 'chapter-section',
               'urn': 'urn:cts:latinLit:stoa0275.stoa027.perseus-lat2-simple',
               'docnum': 253},
              {'title': 'De Testimionio Animae',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa028.opp-lat1-simple',
               'docnum': 254},
              {'title': 'De Virginibus Velandis',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa029.opp-lat1-simple',
               'docnum': 255},
              {'title': 'Scorpiace',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:stoa0275.stoa030.opp-lat1-simple',
               'docnum': 256}],
             'Tibullus': [{'title': 'Elegiae',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:phi0660.phi003.perseus-lat2-simple',
               'docnum': 257}],
             'Valerius Flaccus, Gaius': [{'title': 'Argonautica',
               'meta': 'book-line',
               'urn': 'urn:cts:latinLit:phi1035.phi001.perseus-lat2-simple',
               'docnum': 258}],
             'Virgil': [{'title': 'Aeneid',
               'meta': 'Book-line',
               'urn': 'urn:cts:latinLit:phi0690.phi003.perseus-lat2-simple',
               'docnum': 259},
              {'title': 'Eclogues',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:phi0690.phi001.perseus-lat2-simple',
               'docnum': 260},
              {'title': 'Georgics',
               'meta': 'poem-line',
               'urn': 'urn:cts:latinLit:phi0690.phi002.perseus-lat2-simple',
               'docnum': 261}],
             'Vitruvius Pollio': [{'title': 'On Architecture',
               'meta': 'book-chapter-section',
               'urn': 'urn:cts:latinLit:phi1056.phi001.perseus-lat1-simple',
               'docnum': 262}],
             'Vopiscus, Flavius Fl. 3./4. Jh': [{'title': 'Alexander Severus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi018.perseus-lat2-simple',
               'docnum': 263},
              {'title': 'Antoninus Caracalla',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi013.perseus-lat2-simple',
               'docnum': 264},
              {'title': 'Antoninus Geta',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi014.perseus-lat2-simple',
               'docnum': 265},
              {'title': 'Antoninus Heliogobalus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi017.perseus-lat2-simple',
               'docnum': 266},
              {'title': 'Antoninus Pius',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi003.perseus-lat2-simple',
               'docnum': 267},
              {'title': 'Avidius Casius',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi006.perseus-lat2-simple',
               'docnum': 268},
              {'title': 'Carus et Carinus et Numerianus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi030.perseus-lat2-simple',
               'docnum': 269},
              {'title': 'Clodinus Albinus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi012.perseus-lat2-simple',
               'docnum': 270},
              {'title': 'Commodus Antoninus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi007.perseus-lat2-simple',
               'docnum': 271},
              {'title': 'De Vita Hadriani',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi001.perseus-lat2-simple',
               'docnum': 272},
              {'title': 'Diadumenus Antoninus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi016.perseus-lat2-simple',
               'docnum': 273},
              {'title': 'Didius Julianus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi009.perseus-lat2-simple',
               'docnum': 274},
              {'title': 'Divus Aurelianus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi026.perseus-lat2-simple',
               'docnum': 275},
              {'title': 'Divus Claudius',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi025.perseus-lat2-simple',
               'docnum': 276},
              {'title': 'Firmus Saturninus, Proculus et Bonosus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi029.perseus-lat2-simple',
               'docnum': 277},
              {'title': 'Gallieni Duo',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi023.perseus-lat2-simple',
               'docnum': 278},
              {'title': 'Goridani Tres',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi020.perseus-lat2-simple',
               'docnum': 279},
              {'title': 'Helius',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi002.perseus-lat2-simple',
               'docnum': 280},
              {'title': 'Marcus Antoninus Philosophus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi004.perseus-lat2-simple',
               'docnum': 281},
              {'title': 'Maximini Duo',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi019.perseus-lat2-simple',
               'docnum': 282},
              {'title': 'Maximus et Balbinus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi021.perseus-lat2-simple',
               'docnum': 283},
              {'title': 'Opilius Macrinus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi015.perseus-lat2-simple',
               'docnum': 284},
              {'title': 'Pertinax',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi008.perseus-lat2-simple',
               'docnum': 285},
              {'title': 'Pescennius Niger',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi011.perseus-lat2-simple',
               'docnum': 286},
              {'title': 'Probus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi028.perseus-lat2-simple',
               'docnum': 287},
              {'title': 'Severus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi010.perseus-lat2-simple',
               'docnum': 288},
              {'title': 'Tacitus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi027.perseus-lat2-simple',
               'docnum': 289},
              {'title': 'Tyranni Triginta',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi024.perseus-lat2-simple',
               'docnum': 290},
              {'title': 'Valeriani Duo',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi022.perseus-lat2-simple',
               'docnum': 291},
              {'title': 'Verus',
               'meta': 'chapter',
               'urn': 'urn:cts:latinLit:phi2331.phi005.perseus-lat2-simple',
               'docnum': 292}]}
