from MyCapytain.common.constants import Mimetypes
from MyCapytain.resolvers.cts.api import HttpCtsResolver
from MyCapytain.retrievers.cts5 import HttpCtsRetriever as CTS
from requests.exceptions import HTTPError
import config
from utils import nrange


def get(hit, meta, fragment):
    resolver = HttpCtsResolver(CTS("http://scaife-cts.perseus.org/api/cts"))
    urn = hit.get('urn', None)

    divs = meta['meta'].split('-')

    start = [
        int(meta['start'][div])
        for div in divs
        if div != 'line'
    ]
    end = [
        int(meta['end'][div])
        for div in divs
        if div != 'line'
    ]

    pre = []
    pre_start = start[:-1] + [(start[-1] - config.LINES_OF_CONTEXT),]
    pre_end = start[:-1] + [(start[-1] - 1),]
    for ref in nrange(pre_start, pre_end):
        cite = '.'.join([str(i) for i in ref])
        try:
            passage = resolver.getTextualNode(urn, subreference=cite)
        except HTTPError:
            continue
        else:
            text = passage.export(Mimetypes.PLAINTEXT)
            pre.append(f"<pre>{text}</pre>")

    match = []
    for ref in nrange(start, end):
        cite = '.'.join([str(i) for i in ref])
        try:
            passage = resolver.getTextualNode(urn, subreference=cite)
        except HTTPError:
            text = "NULL"
        else:
            text = passage.export(Mimetypes.PLAINTEXT)
        match.append(f"<match>{text}</match>")

    post = []
    post_start = end[:-1] + [(end[-1] + 1),]  # to avoid overlap
    post_end = end[:-1] + [(end[-1] + config.LINES_OF_CONTEXT),]
    for ref in nrange(post_start, post_end):
        cite = '.'.join([str(i) for i in ref])
        try:
            passage = resolver.getTextualNode(urn, subreference=cite)
        except HTTPError:
            continue
        else:
            text = passage.export(Mimetypes.PLAINTEXT)
            post.append(f"<post>{text}</post>")

    if 'poem' in divs or divs[-1] == 'verse':
        joiner = '\n\n'
    else:
        joiner = ' '
    return f'{joiner}'.join([*pre, *match, *post])


subord = {"main clause": ["00"], "paratactic subjunctive": ["AA"], "paratactic subjunctive in interrogative clause": ["AB"], "ablative absolute": ["AD", "AE"], "infinitive clause": ["AG"], "AC": ["AK"], "ATQVE": ["AK"], "AN": ["AW"], "ANNE": ["BD"], "ANTEQVAM": ["BK"], "ANTEAQVAM": ["BK"], "adverbial or subordinating conjunction,": ["BL"], "CEV": ["BL"], "CVIVS": ["BM"], "CVIVSMODI": ["BM"], "CVM": ["BN"], "CVICVIMODI": ["BO"], "CVMCVMQVE": ["BS"], "interrogative": ["BX", "DG", "FN", "GA", "GF", "GK", "GS", "HG", "JG", "JJ", "JN", "JX", "KG", "KS", "LD", "LK", "PA", "PS", "RG", "RS", "WX", "YA"], "CVR": ["BX"], "DONEC": ["CA"], "DVM": ["CD"], "DVMMODO": ["CG"], "ECQVIS": ["CL"], "ECQVID": ["CL"], "ECQVISNAM": ["CM"], "ECQVIDNAM": ["CM"], "ETIAMSI": ["CN"], "ECQVANDO": ["CR"], "ETSI": ["CS"], "LICET": ["CX"], "MODO": ["DA"], "subordinating conjunction": ["DD", "HX", "KX", "LE", "MK", "NN", "SX", "TB", "WK", "XK", "ZE"], "NE": ["DD", "DG"], "NECNE": ["DP", "DS"], "NEC": ["DR"], "NECVBI": ["DT"], "NEDVM": ["DX"], "NEVE": ["EA"], "NEV": ["EA"], "NI": ["ED"], "NISI": ["EG"], "NONNE": ["EK"], "NVM": ["EN"], "NVMQVIS": ["EP"], "NVMQVID": ["EP"], "NVMQVANDO": ["EQ"], "POSTQVAM": ["ES"], "POSTEAQVAM": ["ES"], "PRAETERQVAM": ["EY"], "PRAEVT": ["EZ"], "PRIVSQVAM": ["FA"], "PROQVAM": ["FD"], "PROVT": ["FG"], "relative": ["FK", "FX", "GE", "GG", "GN", "HD", "JF", "JH", "JK", "JS", "KD", "KN", "LA", "LG", "NX", "PN", "RD", "RN", "SG", "WS", "XX"], "QVA": ["FK", "FN"], "QVACVMQVE": ["FS"], "QVALIS": ["FX", "GA"], "QVALISCVMQVE": ["GD"], "QVALITER": ["GE", "GF"], "QVAM": ["GG", "GK"], "QVAMDIV": ["GN", "GS"], "QVAMDVDVM": ["GX"], "QVAMLIBET": ["HA"], "QVAMOBREM": ["HD", "HG"], "QVAMQVAM": ["HK"], "QVAMVIS": ["HN"], "QVANAM": ["HP"], "interrogative adverb": ["HS", "MA", "SW", "WG", "XG"], "QVANDO": ["HS", "HX"], "QVANDOCVMQVE": ["JA"], "QVANDOQVE": ["JD"], "QVANDOQVIDEM": ["JE"], "QVANTILLVS": ["JF", "JG"], "QVANTO": ["JH", "JJ"], "QVANTOPERE": ["JK", "JN"], "QVANTVLVS": ["JS", "JX"], "QVANTVLVSLIBET": ["JZ"], "QVANTVLVSCVMQVE": ["KA"], "QVANTVS": ["KD", "KG"], "QVANTVM": ["KD", "KG"], "QVANTVSCVMQVE": ["KK"], "QVAPROPTER": ["KL"], "QVARE": ["KN", "KS"], "relative adverb": ["KW", "LX", "TA", "WD", "XD"], "QVASI": ["KW", "KX"], "QVATENVS": ["LA", "LD", "LE"], "QVEMADMODVM": ["LG", "LK"], "pronom relative": ["LN"], "QVI": ["LN", "LS", "LX", "MA"], "pronom interrogative": ["LS"], "QVIA": ["MD"], "QVICVMQVE": ["MG"], "QVIN": ["MK"], "quin": ["ML"], "NON": ["ML"], "QVIPPE": ["MN"], "QVIS": ["MS"], "QVID": ["MS"], "QVISNAM": ["MX"], "QVIDNAM": ["MX"], "QVISQVE": ["MY"], "QVISQVIS": ["NA"], "adv. lieu relative": ["ND"], "QVO": ["ND", "NG", "NK", "NN"], "adv. lieu interrogative": ["NG"], "abl. diff. mes.": ["NK"], "QVOAD": ["NX", "PA"], "QVOADVSQVE": ["PC"], "QVOCVMQVE": ["PD"], "QVOD": ["PG"], "QVOMINVS": ["PK"], "QVOMODO": ["PN", "PS"], "QVOMODOCVMQVE": ["PT"], "QVONAM": ["PU"], "QVONIAM": ["PX"], "QVOQVO": ["RA"], "QVORSVS": ["RB"], "QVOT": ["RD", "RG"], "QVOTCVMQVE": ["RK"], "QVOTIENS": ["RN", "RS"], "QVOTIENSCVMQVE": ["RX"], "QVOTQVOT": ["SA"], "QVOTVS": ["SD"], "QVOTVSCVMQVE": ["SG"], "QVOTVSQVISQVE": ["SK"], "QVOVSQVE": ["SN"], "SEV": ["SS"], "SI": ["SW", "SX"], "SICVBI": ["SY"], "SICVNDE": ["SZ"], "SICVT": ["TA", "TB"], "SICVTI": ["TA", "TB"], "SIMVL": ["TC"], "SIMVLAC": ["TD"], "SIMVLATQVE": ["TD"], "SIN": ["TG"], "SIQVIDEM": ["TK"], "SIVE": ["TN"], "TAMENETSI": ["TR"], "TAMETSI": ["TS"], "TAMQVAM": ["TX"], "VBI": ["WD", "WG", "WK"], "VBICVMQVE": ["WN"], "VBINAM": ["WP"], "VBIVBI": ["WQ"], "VNDE": ["WS", "WX"], "VNDECVMQVE": ["XA"], "VSQVEQVO": ["XC"], "VT": ["XD", "XG", "XK"], "VTI": ["XD", "XG", "XK"], "VTCVMQVE": ["XS"], "VTER": ["XX", "YA"], "VTERCVMQVE": ["YD"], "VTQVI": ["YS"], "VTRVM": ["YW"], "VTRVMNE": ["YZ"], "adverbial": ["ZD"], "VELVT": ["ZD", "ZE"], "VELVTI": ["ZD", "ZE"]}

subord_codes = {
    "00": (None, "main clause"),
    "AA": (None, "paratactic subjunctive"),
    "AB": (None, "paratactic subjunctive in interrogative clause"),
    "AD": (None, "ablative absolute"),
    "AE": (None, "ablative absolute"),
    "AG": (None, "infinitive clause"),
    "AK": (("AC", "ATQVE"), None),
    "AW": (("AN",), None),
    "BD": (("ANNE",), None),
    "BK": (("ANTEQVAM", "ANTEAQVAM"), None),
    "BL": (("CEV",), "adverbial or subordinating conjunction,"),
    "BM": (("CVIVS", "CVIVSMODI"), None),
    "BN": (("CVM",), None),
    "BO": (("CVICVIMODI",), None),
    "BS": (("CVMCVMQVE",), None),
    "BX": (("CVR",), "interrogative"),
    "CA": (("DONEC",), None),
    "CD": (("DVM",), None),
    "CG": (("DVMMODO",), None),
    "CL": (("ECQVIS", "ECQVID"), None),
    "CM": (("ECQVISNAM", "ECQVIDNAM"), None),
    "CN": (("ETIAMSI",), None),
    "CR": (("ECQVANDO",), None),
    "CS": (("ETSI",), None),
    "CX": (("LICET",), None),
    "DA": (("MODO",), None),
    "DD": (("NE",), "subordinating conjunction"),
    "DG": (("NE",), "interrogative"),
    "DP": (("NECNE",), None),
    "DR": (("NEC",), None),
    "DS": (("NECNE",), None),
    "DT": (("NECVBI",), None),
    "DX": (("NEDVM",), None),
    "EA": (("NEVE", "NEV"), None),
    "ED": (("NI",), None),
    "EG": (("NISI",), None),
    "EK": (("NONNE",), None),
    "EN": (("NVM",), None),
    "EP": (("NVMQVIS", "NVMQVID"), None),
    "EQ": (("NVMQVANDO",), None),
    "ES": (("POSTQVAM", "POSTEAQVAM"), None),
    "EY": (("PRAETERQVAM",), None),
    "EZ": (("PRAEVT",), None),
    "FA": (("PRIVSQVAM",), None),
    "FD": (("PROQVAM",), None),
    "FG": (("PROVT",), None),
    "FK": (("QVA",), "relative"),
    "FN": (("QVA",), "interrogative"),
    "FS": (("QVACVMQVE",), None),
    "FX": (("QVALIS",), "relative"),
    "GA": (("QVALIS",), "interrogative"),
    "GD": (("QVALISCVMQVE",), None),
    "GE": (("QVALITER",), "relative"),
    "GF": (("QVALITER",), "interrogative"),
    "GG": (("QVAM",), "relative"),
    "GK": (("QVAM",), "interrogative"),
    "GN": (("QVAMDIV",), "relative"),
    "GS": (("QVAMDIV",), "interrogative"),
    "GX": (("QVAMDVDVM",), None),
    "HA": (("QVAMLIBET",), None),
    "HD": (("QVAMOBREM",), "relative"),
    "HG": (("QVAMOBREM",), "interrogative"),
    "HK": (("QVAMQVAM",), None),
    "HN": (("QVAMVIS",), None),
    "HP": (("QVANAM",), None),
    "HS": (("QVANDO",), "interrogative adverb"),
    "HX": (("QVANDO",), "subordinating conjunction"),
    "JA": (("QVANDOCVMQVE",), None),
    "JD": (("QVANDOQVE",), None),
    "JE": (("QVANDOQVIDEM",), None),
    "JF": (("QVANTILLVS",), "relative"),
    "JG": (("QVANTILLVS",), "interrogative"),
    "JH": (("QVANTO",), "relative"),
    "JJ": (("QVANTO",), "interrogative"),
    "JK": (("QVANTOPERE",), "relative"),
    "JN": (("QVANTOPERE",), "interrogative"),
    "JS": (("QVANTVLVS",), "relative"),
    "JX": (("QVANTVLVS",), "interrogative"),
    "JZ": (("QVANTVLVSLIBET",), None),
    "KA": (("QVANTVLVSCVMQVE",), None),
    "KD": (("QVANTVS", "QVANTVM"), "relative"),
    "KG": (("QVANTVS", "QVANTVM"), "interrogative"),
    "KK": (("QVANTVSCVMQVE",), None),
    "KL": (("QVAPROPTER",), None),
    "KN": (("QVARE",), "relative"),
    "KS": (("QVARE",), "interrogative"),
    "KW": (("QVASI",), "relative adverb"),
    "KX": (("QVASI",), "subordinating conjunction"),
    "LA": (("QVATENVS",), "relative"),
    "LD": (("QVATENVS",), "interrogative"),
    "LE": (("QVATENVS",), "subordinating conjunction"),
    "LG": (("QVEMADMODVM",), "relative"),
    "LK": (("QVEMADMODVM",), "interrogative"),
    "LN": (("QVI",), "pronom relative"),
    "LS": (("QVI",), "pronom interrogative"),
    "LX": (("QVI",), "relative adverb"),
    "MA": (("QVI",), "interrogative adverb"),
    "MD": (("QVIA",), None),
    "MG": (("QVICVMQVE",), None),
    "MK": (("QVIN",), "subordinating conjunction"),
    "ML": (("NON",), "quin"),
    "MN": (("QVIPPE",), None),
    "MS": (("QVIS", "QVID"), None),
    "MX": (("QVISNAM", "QVIDNAM"), None),
    "MY": (("QVISQVE",), None),
    "NA": (("QVISQVIS",), None),
    "ND": (("QVO",), "adv. lieu relative"),
    "NG": (("QVO",), "adv. lieu interrogative"),
    "NK": (("QVO",), "abl. diff. mes."),
    "NN": (("QVO",), "subordinating conjunction"),
    "NX": (("QVOAD",), "relative"),
    "PA": (("QVOAD",), "interrogative"),
    "PC": (("QVOADVSQVE",), None),
    "PD": (("QVOCVMQVE",), None),
    "PG": (("QVOD",), None),
    "PK": (("QVOMINVS",), None),
    "PN": (("QVOMODO",), "relative"),
    "PS": (("QVOMODO",), "interrogative"),
    "PT": (("QVOMODOCVMQVE",), None),
    "PU": (("QVONAM",), None),
    "PX": (("QVONIAM",), None),
    "RA": (("QVOQVO",), None),
    "RB": (("QVORSVS",), None),
    "RD": (("QVOT",), "relative"),
    "RG": (("QVOT",), "interrogative"),
    "RK": (("QVOTCVMQVE",), None),
    "RN": (("QVOTIENS",), "relative"),
    "RS": (("QVOTIENS",), "interrogative"),
    "RX": (("QVOTIENSCVMQVE",), None),
    "SA": (("QVOTQVOT",), None),
    "SD": (("QVOTVS",), None),
    "SG": (("QVOTVSCVMQVE",), "relative"),
    "SK": (("QVOTVSQVISQVE",), None),
    "SN": (("QVOVSQVE",), None),
    "SS": (("SEV",), None),
    "SW": (("SI",), "interrogative adverb"),
    "SX": (("SI",), "subordinating conjunction"),
    "SY": (("SICVBI",), None),
    "SZ": (("SICVNDE",), None),
    "TA": (("SICVT", "SICVTI"), "relative adverb"),
    "TB": (("SICVT", "SICVTI"), "subordinating conjunction"),
    "TC": (("SIMVL",), None),
    "TD": (("SIMVLAC", "SIMVLATQVE"), None),
    "TG": (("SIN",), None),
    "TK": (("SIQVIDEM",), None),
    "TN": (("SIVE",), None),
    "TR": (("TAMENETSI",), None),
    "TS": (("TAMETSI",), None),
    "TX": (("TAMQVAM",), None),
    "WD": (("VBI",), "relative adverb"),
    "WG": (("VBI",), "interrogative adverb"),
    "WK": (("VBI",), "subordinating conjunction"),
    "WN": (("VBICVMQVE",), None),
    "WP": (("VBINAM",), None),
    "WQ": (("VBIVBI",), None),
    "WS": (("VNDE",), "relative"),
    "WX": (("VNDE",), "interrogative"),
    "XA": (("VNDECVMQVE",), None),
    "XC": (("VSQVEQVO",), None),
    "XD": (("VT", "VTI"), "relative adverb"),
    "XG": (("VT", "VTI"), "interrogative adverb"),
    "XK": (("VT", "VTI"), "subordinating conjunction"),
    "XS": (("VTCVMQVE",), None),
    "XX": (("VTER",), "relative"),
    "YA": (("VTER",), "interrogative"),
    "YD": (("VTERCVMQVE",), None),
    "YS": (("VTQVI",), None),
    "YW": (("VTRVM",), None),
    "YZ": (("VTRVMNE",), None),
    "ZD": (("VELVT", "VELVTI"), "adverbial"),
    "ZE": (("VELVT", "VELVTI"), "subordinating conjunction"),
}

MORPHS = [
    {  # Case
        "1": "n",
        "2": "v",
        "3": "a",
        "4": "g",
        "5": "d",
        "6": "b",
        "7": "l",
        "8": "-",
    },
    {"1": "s", "2": "p"},  # Number
    {"1": "p", "2": "c", "3": "s"},  # Degree
    {  # Mood
        "0": "-",  # Ignore (we had one for an agendis annotation)
        "1": "i",
        "3": "s",
        "2": "m",
        "4": "p",
        "7": "n",
        "5": "d",
        "6": "g",
        "8": "s",
        "9": "s",
    },
    {  # Tense
        "0": "-",  # Ignore (We had one for an agendis annotation)
        "1": "p",
        "2": "i",
        "3": "f",
        "4": "r",
        "5": "l",
        "6": "u",
        "7": "r",  # periphrasis?
        "8": "l",  # periphrasis?
        "9": "f",  # periphrasis?
    },
    {"1": "a", "2": "p", "3": "d", "4": "d"},  # Voice  # semi-deponent
    {"1": "1", "2": "2", "3": "3"},  # Person
    {
        # Gender
        "0": "-",  # Ignore (We had one for an agendis annotation)
        "1": "a",
        "2": "f",
        "3": "m/f",  # c
        "4": "m",
        "5": "m/n",  # c
        "6": "n",
    },
]

GROUP = {
    "A": {
        "1": { "group": "1", "stem": "-" },
        "2": { "group": "2", "stem": "-" },
        "3": { "group": "3", "stem": "-" },
        "4": { "group": "4", "stem": "-" },
        "5": { "group": "5", "stem": "-" },
        "6": { "group": "-", "stem": "-" },
        "7": { "group": "3", "stem": "g" }, # could be group 2 -os, -on or even group 1 -es/-as, -ae
    },
"B":
{
    "1": { "group": "1", "stem": "-" },
    "2": { "group": "2", "stem": "-" },
    "3": { "group": "3", "stem": "-" },
    "4": { "group": "4", "stem": "-" },
    "5": { "group": "3", "stem": "i" },
    "6": { "group": "3", "stem": "-" }, # irregulars are classed with group 3
},
"C": {
    "1": { "group": "1", "stem": "-" },
    "2": { "group": "3", "stem": "-" },
    "3": { "group": "3", "stem": "r" },
    "4": { "group": "3", "stem": "i" },
    "5": { "group": "3", "stem": "-" },
    "6": { "group": "-", "stem": "-" },  # irregulars
    "7": { "group": "1", "stem": "g" },  # greek decl in -os, -on
    " ": { "group": "-", "stem": "-" },  # unspecified 
},
"D": {
"1": { "group": "-", "stem": "-" },  # indecl except for unus, duo, tres
"2": { "group": "1", "stem": "-" },  # -us
"3": { "group": "1", "stem": "-" },  # -eni
"4": { "group": "3", "stem": "-" },  # -ex
"5": { "group": "-", "stem": "-" },  # adverbs
"6": { "group": "-", "stem": "-" }, 
},
}

POS_MAP = {
    "A": "n",
    "B": "v",
    "C": "a",
    "D": "a",
    "E": "n",
    "F": "n",
    "G": "n",
    "H": "n",
    "I": "n",
    "J": "n",
    "K": "n",
    "L": "n",
    "M": "r",
    "N": "r",
    "O": "r",
    "P": "r",
    "Q": "r",
    "R": "p",
    "S": "-",
    "T": "-",
    "U": "-",
    "#": "v",
    "0": "-",
    " ": "-",
}


def bpn2lwn(morph_code: str) -> str:
    """
    Convert a BPN code string to a Perseus-like morphological string

    :return: str
    """
    if len(morph_code) < 11:
        morph_code +=  ' ' * (11 - len(morph_code))
    pos = POS_MAP[morph_code[0]]
    morph = morph_code[2:]
    if morph_code[0] in 'ABCD':
        group = GROUP[morph_code[0]][morph_code[1]]
    else:
        group = { "group": "-", "stem": "-" }
    morpho = []
    for i, morph_char in enumerate(morph):
        if morph_char.strip():
            morpho.append(MORPHS[i].get(morph_char, "-"))
        else:
            morpho.append("-")
    if pos in "ar":  #aps---mn1-
        desc = f"{pos}{morpho[2]}{morpho[1]}{morpho[4]}{morpho[3]}{morpho[5]}{morpho[7]}{morpho[0]}{group['group']}{group['stem']}"
    else:
        desc = f"{pos}{morpho[6]}{morpho[1]}{morpho[4]}{morpho[3]}{morpho[5]}{morpho[7]}{morpho[0]}{group['group']}{group['stem']}"
    return desc


def parse_bpn(line: str) -> dict:
    """ Parse a single line of the LASLA corpus

    :param line: Input line
    :return: Dictionary
    """
    line = line.replace("\n", "")
    if not line.strip():
        return None

    author_code = line[0]
    work_code = line[1:3]
    form_code = line[3]
    sent_id = line[4:8]
    lemma, lemma_n = line[8:29].strip(), line[29]
    form = line[30:55].strip()
    refs = line[55:66].strip()
    ref_n = line[66]
    morpho = line[67:76]
    subord = line[76:78]
    if len(line) == 80:
        morpho += line[79]
    else:
        morpho += ' '
    return {
        "author_code": author_code,
        "work_code": work_code,
        "form_code": form_code,
        "sent_id": sent_id,
        "refs": refs,
        "ref_n": ref_n,
        "lemma": lemma,
        "lemma_n": lemma_n,
        "form": form,
        "morpho": morpho,
        "subord": subord,
    }


DICT_MORPH = {
    "-": { "-": { "-": "----------" } },
    "N": {
        "2": {
            "1": "aps---mn1-",
            "2": "aps---an3-",
            "3": "aps---mn1r",
            "4": "aps---cn3i",
            "5": "aps---an3-",
            "6": "aps---an--",
            "7": "aps---mn1g",
            "&": "acs---cn3-",
        },
        "1": {
            "1": "n-s----n1-",
            "2": "n-s----n2-",
            "3": "n-s----n3-",
            "4": "n-s----n4-",
            "5": "n-s----n5-",
            "6": "n-s----n--",
            "7": "n-s----n-g",
        },
    },
    "A": {
        "1": {
            "1": "n-s----n1-",
            "2": "n-s----n2-",
            "3": "n-s----n3-",
            "4": "n-s----n4-",
            "5": "n-s----n5-",
            "6": "n-s----n--",
            "7": "n-s----n-g",
        },
        "2": {
            "1": "aps---mn1-",
            "2": "aps---an3-",
            "3": "aps---mn1r",
            "4": "aps---cn3i",
            "5": "aps---an3-",
            "6": "aps---an--",
            "7": "aps---mn1g",
            "&": "acs---cn3-",
        },
    },
    # NOVNS
    "1": {
        "1": "n-s----n1-",
        "2": "n-s----n2-",
        "3": "n-s----n3-",
        "4": "n-s----n4-",
        "5": "n-s----n5-",
        "6": "n-s----n--",
        "7": "n-s----n-g",
    },
    # ADJECTIVES
    "2": {
        "1": "aps---mn1-",
        "2": "aps---an3-",
        "3": "aps---mn1r",
        "4": "aps---cn3i",
        "5": "aps---an3-",
        "6": "aps---an--",
        "7": "aps---mn1g",
        "&": "acs---cn3-",
        "G": "aps---mn1g",
    },
    # NVMERALS
    "3": {
        "1": "----------",
        "2": "aps---mn1-",
        "3": "rp--------",
        "4": "aps---an3-",
        "5": "rp--------",
        "6": "rp--------",
        "N": "rp--------",
        "K": "aps---mn1-",
        "E": "rc--------",
        "B": "acs---cn3-",
    },
    # PRONOVNS
    "4": {
        "1": "o-----an--",
        "2": "aps---mn1-",
        "3": "o-----an--",  # reflexive
        "4": "aps---mn1-",
        "5": "aps---mn1-",  # some pronominal declensions
        "6": "aps---mn1-",  # some pronominal, some indeclinable
        "7": "----------",  # interrogatives
        "8": "----------",  # indefinites
    },
    # VERBS
    "5": {
        "1": "v1spia--1-",
        "2": "v1spia--2-",
        "3": "v1spia--3-",
        "4": "v1spia--4-",
        "5": "v1spia--3i",
        "6": "v1spia--3-",  # irregulars
        "7": "----------",  # greek participles?
        "J": "v1spid--1-",  # deponent
        "K": "v1spid--2-",
        "L": "v1spid--3-",
        "M": "v1spid--4-",
        "N": "v1spid--3i",
        "S": "v1spia--2-",  # semi-deponents
        "T": "v1spia--3-",
    },
    # ADVERBS
    "6": {
        "-": "rp--------",
        "&": "rp--------",
        "0": "rp--------",
        "6": "r---------",
        "7": "r---------",
        "8": "r---------",
        "9": "----------",  # conjunctions
    },
    # PREPOSITIONS
    "7": {"-": "p---------", "0": "p---------"},
    # CONJVNCTION?
    "8": {"1": "----------", "2": "----------"},
    # INTERJECTION
    "9": {"0": "----------"},
    "E": {"6": "v1spia--3-"},  # auxiliary verbs
}

lexicon = {
    "AB": {"-": ["70"]},
    "ABACTVS": {"-": ["14"]},
    "ABACVS": {"-": ["12"]},
    "ABALIENO": {"-": ["51"]},
    "ABATOS": {"-": ["27"]},
    "ABAVVS": {"-": ["12"]},
    "ABBAS": {"-": ["13"]},
    "ABDICO": {"1": ["51"]},
    "ABDITVM": {"-": ["12"]},
    "ABDITVS": {"C": ["21"], "-": ["21"]},
    "ABDO": {"-": ["53"]},
    "ABDOMEN": {"-": ["13"]},
    "ABDVCO": {"-": ["53"]},
    "ABEO": {"1": ["56"]},
    "ABERRO": {"2": ["51"]},
    "ABHINC": {"-": ["60"]},
    "ABHORREO": {"-": ["52"]},
    "ABHORRIDVS": {"-": ["21"]},
    "ABICIO": {"-": ["55"]},
    "ABIECTE": {"-": ["6&", "66"]},
    "ABIECTVS": {"-": ["21"]},
    "ABIEGNEVS": {"-": ["21"]},
    "ABIEGNVS": {"-": ["21"]},
    "ABIES": {"-": ["13"]},
    "ABIGO": {"-": ["53"]},
    "ABITVS": {"-": ["14"]},
    "ABIVDICO": {"-": ["51"]},
    "ABIVNGO": {"-": ["53"]},
    "ABIVRO": {"-": ["51"]},
    "ABLAQVEO": {"1": ["51"]},
    "ABLEGO": {"1": ["51"]},
    "ABLEVO": {"1": ["51"]},
    "ABLVDO": {"-": ["53"]},
    "ABLVO": {"-": ["53"]},
    "ABNEGO": {"-": ["51"]},
    "ABNEPOS": {"-": ["13"]},
    "ABNOCTO": {"-": ["51"]},
    "ABNORMIS": {"-": ["24"]},
    "ABNVO": {"-": ["53"]},
    "ABOLEO": {"-": ["52"]},
    "ABOLESCO": {"-": ["53"]},
    "ABOLITIO": {"-": ["13"]},
    "ABOLLA": {"-": ["11"]},
    "ABOMINOR": {"-": ["5J"]},
    "ABORIOR": {"-": ["5M"]},
    "ABORISCOR": {"-": ["5L"]},
    "ABORTIVVS": {"-": ["21"]},
    "ABORTVS": {"-": ["14"]},
    "ABRADO": {"-": ["53"]},
    "ABRIPIO": {"-": ["55"]},
    "ABROGO": {"-": ["51"]},
    "ABROTONVM": {"-": ["12"]},
    "ABROTONVS": {"-": ["12"]},
    "ABRVMPO": {"-": ["53"]},
    "ABRVPTVM": {"-": ["12"]},
    "ABRVPTVS": {"-": ["21"]},
    "ABSCEDO": {"1": ["53"]},
    "ABSCESSVS": {"-": ["14"]},
    "ABSCIDO": {"2": ["53"]},
    "ABSCINDO": {"-": ["53"]},
    "ABSCISIO": {"-": ["13"]},
    "ABSCISVS": {"-": ["21"]},
    "ABSCONDITVS": {"C": ["21"], "-": ["21"]},
    "ABSCONDO": {"-": ["53"]},
    "ABSCONSVS": {"C": ["21"], "-": ["21"]},
    "ABSENS": {"-": ["25"]},
    "ABSENTIA": {"-": ["11"]},
    "ABSILIO": {"-": ["54"]},
    "ABSIMILIS": {"-": ["24"]},
    "ABSINTHIVM": {"-": ["12"]},
    "ABSISTO": {"-": ["53"]},
    "ABSOLVO": {"-": ["53"]},
    "ABSOLVTE": {"-": ["60"]},
    "ABSOLVTIO": {"-": ["13"]},
    "ABSOLVTVS": {"-": ["21"]},
    "ABSONVS": {"-": ["21"]},
    "ABSORBEO": {"-": ["52"]},
    "ABSQVE": {"1": ["60"], "2": ["70"]},
    "ABSTANTIA": {"-": ["11"]},
    "ABSTEMIVS": {"-": ["21"]},
    "ABSTERGEO": {"-": ["52"]},
    "ABSTERREO": {"-": ["52"]},
    "ABSTINENS": {"-": ["25"]},
    "ABSTINENTER": {"-": ["60"]},
    "ABSTINENTIA": {"-": ["11"]},
    "ABSTINEO": {"-": ["52"]},
    "ABSTRAHO": {"-": ["53"]},
    "ABSTRVDO": {"-": ["53"]},
    "ABSTRVSVS": {"-": ["21"]},
    "ABSVM": {"1": ["56"]},
    "ABSVMO": {"-": ["53"]},
    "ABSVRDE": {"-": ["60"]},
    "ABSVRDVS": {"-": ["21"]},
    "ABVNDANS": {"1": ["13"], "2": ["25"]},
    "ABVNDANTIA": {"-": ["11"]},
    "ABVNDE": {"-": ["60"]},
    "ABVNDO": {"-": ["51"]},
    "ABVNDVS": {"-": ["21"]},
    "ABVSQVE": {"-": ["70"]},
    "ABVTOR": {"-": ["5L"]},
    "AC": {"1": ["81"], "2": ["82"]},
    "ACADEMICI": {"N": ["12"]},
    "ACADEMICVS": {"N": ["21"], "A": ["21"]},
    "ACALANTHIS": {"-": ["17"]},
    "ACANTHVS": {"-": ["12"]},
    "ACCEDO": {"1": ["53"]},
    "ACCELERO": {"-": ["51"]},
    "ACCENDO": {"-": ["53"]},
    "ACCENSEO": {"-": ["52"]},
    "ACCENSI": {"-": ["12"]},
    "ACCENSVS": {"1": ["12"], "2": ["14"]},
    "ACCEPTATVS": {"-": ["21"]},
    "ACCEPTIO": {"-": ["13"]},
    "ACCEPTO": {"-": ["51"]},
    "ACCEPTVM": {"-": ["12"]},
    "ACCEPTVS": {"-": ["21"]},
    "ACCESSIO": {"-": ["13"]},
    "ACCESSITO": {"-": ["51"]},
    "ACCESSO": {"-": ["51"]},
    "ACCESSVS": {"-": ["14"]},
    "ACCIDENS": {"-": ["13"]},
    "ACCIDO": {"1": ["53"], "2": ["53"]},
    "ACCIEO": {"-": ["52"]},
    "ACCINGO": {"-": ["53"]},
    "ACCIO": {"-": ["54"]},
    "ACCIPIO": {"-": ["55"]},
    "ACCIPITER": {"-": ["13"]},
    "ACCITO": {"1": ["51"]},
    "ACCITVS": {"-": ["14"]},
    "ACCLAMATIO": {"-": ["13"]},
    "ACCLAMO": {"-": ["51"]},
    "ACCLINIS": {"-": ["24"]},
    "ACCLIVIS": {"-": ["24"]},
    "ACCLIVITAS": {"-": ["13"]},
    "ACCLIVVS": {"-": ["21"]},
    "ACCOGNOSCO": {"-": ["53"]},
    "ACCOLA": {"-": ["11"]},
    "ACCOLO": {"2": ["53"]},
    "ACCOMMODATIO": {"-": ["13"]},
    "ACCOMMODATVS": {"-": ["21"]},
    "ACCOMMODO": {"-": ["51"]},
    "ACCOMMODVS": {"-": ["21"]},
    "ACCREDO": {"-": ["53"]},
    "ACCREMENTVM": {"-": ["12"]},
    "ACCRESCO": {"-": ["53"]},
    "ACCVBITIO": {"-": ["13"]},
    "ACCVBO": {"-": ["51"]},
    "ACCVMBO": {"-": ["53"]},
    "ACCVMVLATOR": {"-": ["13"]},
    "ACCVMVLO": {"-": ["51"]},
    "ACCVRATE": {"-": ["60"]},
    "ACCVRATVS": {"-": ["21"]},
    "ACCVRO": {"-": ["51"]},
    "ACCVRRO": {"-": ["53"]},
    "ACCVRSVS": {"-": ["14"]},
    "ACCVSATIO": {"-": ["13"]},
    "ACCVSATOR": {"-": ["13"]},
    "ACCVSATORIE": {"-": ["60"]},
    "ACCVSATORIVS": {"-": ["21"]},
    "ACCVSO": {"-": ["51"]},
    "ACEO": {"-": ["52"]},
    "ACER": {"1": ["13"], "2": ["23"]},
    "ACERBE": {"-": ["60"]},
    "ACERBITAS": {"-": ["13"]},
    "ACERBVM": {"-": ["12"]},
    "ACERBVS": {"-": ["21"]},
    "ACERNVS": {"-": ["21"]},
    "ACERRA": {"-": ["11"]},
    "ACERSECOMES": {"-": ["17"]},
    "ACERVATIM": {"-": ["60"]},
    "ACERVO": {"-": ["51"]},
    "ACERVVS": {"-": ["12"]},
    "ACESCO": {"-": ["53"]},
    "ACESTENSES": {"N": ["13"]},
    "ACETABVLVM": {"-": ["12"]},
    "ACETVM": {"-": ["12"]},
    "ACHAEVS": {"N": ["21"], "A": ["21"]},
    "ACHAIA": {"N": ["11"]},
    "ACHAICVS": {"N": ["21"], "A": ["21"]},
    "ACHIVI": {"N": ["12"]},
    "ACHIVVS": {"N": ["21"], "A": ["21"]},
    "ACHRADINA": {"N": ["11"]},
    "ACIDVS": {"-": ["21"]},
    "ACIES": {"-": ["15"]},
    "ACILIVS": {"A": ["21"], "N": ["12"]},
    "ACINA": {"1": ["11"], "-": ["11"]},
    "ACINACES": {"-": ["13"]},
    "ACINVM": {"-": ["12"]},
    "ACINVS": {"-": ["12"]},
    "ACIPENSER": {"-": ["13"]},
    "ACOENONOETVS": {"-": ["12"]},
    "ACONITVM": {"-": ["12"]},
    "ACOSMOS": {"-": ["17"]},
    "ACQVIESCO": {"-": ["53"]},
    "ACQVIRO": {"-": ["53"]},
    "ACQVISITIO": {"-": ["13"]},
    "ACRIMONIA": {"-": ["11"]},
    "ACRITER": {"-": ["60"]},
    "ACRITVDO": {"-": ["13"]},
    "ACROAMA": {"-": ["17"]},
    "ACROASIS": {"-": ["17"]},
    "ACROBATICOS": {"-": ["27"]},
    "ACROLITHVS": {"-": ["27"]},
    "ACROTERIVM": {"-": ["12"]},
    "ACTA": {"-": ["11"]},
    "ACTIACVS": {"N": ["21"], "A": ["21"]},
    "ACTIO": {"-": ["13"]},
    "ACTITO": {"-": ["51"]},
    "ACTIVITAS": {"-": ["13"]},
    "ACTIVS": {"-": ["21"], "A": ["21"]},
    "ACTIVVS": {"-": ["21"]},
    "ACTOR": {"-": ["13"], "N": ["13"]},
    "ACTVARIA": {"-": ["11"]},
    "ACTVARIVS": {"1": ["12"], "2": ["21"]},
    "ACTVM": {"-": ["12"]},
    "ACTVOSVS": {"-": ["21"]},
    "ACTVS": {"1": ["14"], "2": ["21"]},
    "ACTVTVM": {"-": ["60"]},
    "ACVLEATVS": {"-": ["21"]},
    "ACVLEVS": {"-": ["12"]},
    "ACVMEN": {"-": ["13"]},
    "ACVO": {"-": ["53"]},
    "ACVS": {"1": ["12"], "2": ["13"], "3": ["14"]},
    "ACVTE": {"-": ["60"]},
    "ACVTVLVS": {"-": ["21"]},
    "ACVTVM": {"-": ["12"]},
    "ACVTVS": {"-": ["21"]},
    "AD": {"1": ["60"], "2": ["70"]},
    "ADACTVS": {"-": ["14"]},
    "ADAEQVO": {"-": ["51"]},
    "ADAGGERO": {"1": ["51"]},
    "ADAMANTINVS": {"-": ["21"]},
    "ADAMAS": {"-": ["17"]},
    "ADAMO": {"-": ["51"]},
    "ADAPERIO": {"-": ["54"]},
    "ADAREO": {"-": ["52"]},
    "ADAVCTVS": {"-": ["14"]},
    "ADAVGEO": {"-": ["52"]},
    "ADAVGESCO": {"-": ["53"]},
    "ADAVGMEN": {"-": ["13"]},
    "ADBIBO": {"2": ["53"]},
    "ADDECET": {"-": ["52"]},
    "ADDICO": {"2": ["53"]},
    "ADDICTIO": {"-": ["13"]},
    "ADDISCO": {"-": ["53"]},
    "ADDITAMENTVM": {"-": ["12"]},
    "ADDO": {"-": ["53"]},
    "ADDOCEO": {"-": ["52"]},
    "ADDVBITO": {"-": ["51"]},
    "ADDVCO": {"-": ["53"]},
    "ADDVCTE": {"-": ["6&"]},
    "ADDVCTVS": {"-": ["21"]},
    "ADEDO": {"2": ["56"]},
    "ADEMO": {"-": ["53"]},
    "ADEMPTIO": {"-": ["13"]},
    "ADEO": {"1": ["56"], "2": ["60"]},
    "ADEPS": {"-": ["13"]},
    "ADEQVITO": {"-": ["51"]},
    "ADHAEREO": {"-": ["52"]},
    "ADHAERESCO": {"-": ["53"]},
    "ADHAESVS": {"-": ["14"]},
    "ADHIBEO": {"-": ["52"]},
    "ADHINNIO": {"-": ["54"]},
    "ADHORTATIO": {"-": ["13"]},
    "ADHORTATOR": {"-": ["13"]},
    "ADHORTOR": {"-": ["5J"]},
    "ADHVC": {"-": ["60"]},
    "ADIACEO": {"-": ["52"]},
    "ADICIO": {"-": ["55"]},
    "ADICO": {"2": ["53"]},
    "ADIECTIO": {"-": ["13"]},
    "ADIECTVS": {"-": ["14"]},
    "ADIGO": {"-": ["53"]},
    "ADIMO": {"-": ["53"]},
    "ADINDO": {"-": ["53"]},
    "ADIPATVM": {"-": ["12"]},
    "ADIPES": {"-": ["13"]},
    "ADIPISCOR": {"-": ["5L"]},
    "ADITIALIS": {"-": ["24"]},
    "ADITVS": {"-": ["14"]},
    "ADIVDICO": {"-": ["51"]},
    "ADIVMENTVM": {"-": ["12"]},
    "ADIVNCTIO": {"-": ["13"]},
    "ADIVNCTVM": {"-": ["12"]},
    "ADIVNCTVS": {"-": ["21"]},
    "ADIVNGO": {"-": ["53"]},
    "ADIVRO": {"-": ["51"]},
    "ADIVTO": {"-": ["51"]},
    "ADIVTOR": {"1": ["13"], "2": ["5J"]},
    "ADIVTORIVM": {"-": ["12"]},
    "ADIVTRIX": {"-": ["13"], "N": ["13"]},
    "ADIVVO": {"-": ["51"]},
    "ADMATVRO": {"-": ["51"]},
    "ADMETIOR": {"-": ["5M"]},
    "ADMINICVLVM": {"-": ["12"]},
    "ADMINISTER": {"-": ["12"]},
    "ADMINISTRA": {"-": ["11"]},
    "ADMINISTRATIO": {"-": ["13"]},
    "ADMINISTRO": {"-": ["51"]},
    "ADMIRABILIS": {"-": ["24"]},
    "ADMIRANDVS": {"-": ["21"]},
    "ADMIRATIO": {"-": ["13"]},
    "ADMIRATOR": {"-": ["13"]},
    "ADMIROR": {"-": ["5J"]},
    "ADMISCEO": {"-": ["52"]},
    "ADMISSIO": {"-": ["13"]},
    "ADMISSVM": {"-": ["12"]},
    "ADMITTO": {"-": ["53"]},
    "ADMODERATE": {"-": ["60"]},
    "ADMODVM": {"-": ["60"]},
    "ADMONEO": {"-": ["52"]},
    "ADMONITIO": {"-": ["13"]},
    "ADMONITOR": {"-": ["13"]},
    "ADMONITVS": {"-": ["14"]},
    "ADMORDEO": {"-": ["52"]},
    "ADMOVEO": {"-": ["52"]},
    "ADMVGIO": {"-": ["54"]},
    "ADMVRMVRATIO": {"-": ["13"]},
    "ADMVRMVRO": {"-": ["51"]},
    "ADOLEO": {"-": ["52"]},
    "ADOLESCENS": {"1": ["13"], "2": ["25"]},
    "ADOLESCENTIA": {"-": ["11"]},
    "ADOLESCENTVLVS": {"-": ["12"]},
    "ADOLESCO": {"1": ["53"], "2": ["53"]},
    "ADOPERIO": {"-": ["54"]},
    "ADOPINOR": {"-": ["5J"]},
    "ADOPTATIO": {"-": ["13"]},
    "ADOPTIO": {"-": ["13"]},
    "ADOPTIVVS": {"-": ["21"]},
    "ADOPTO": {"-": ["51"]},
    "ADOR": {"-": ["13"]},
    "ADOREA": {"-": ["11"]},
    "ADOREVS": {"-": ["21"]},
    "ADORIOR": {"-": ["5M"]},
    "ADORNO": {"-": ["51"]},
    "ADORO": {"-": ["51"]},
    "ADVECTICIVS": {"-": ["21"]},
    "ADVECTO": {"-": ["51"]},
    "ADVECTOR": {"-": ["13"]},
    "ADVECTVS": {"-": ["14"]},
    "ADVEHO": {"-": ["53"]},
    "ADVEHOR": {"-": ["5L"]},
    "ADVELO": {"-": ["51"]},
    "ADVENA": {"-": ["11"]},
    "ADVENIO": {"-": ["54"]},
    "ADVENTICIVM": {"-": ["12"]},
    "ADVENTICIVS": {"-": ["21"]},
    "ADVENTO": {"-": ["51"]},
    "ADVENTVS": {"-": ["14"]},
    "ADVERRO": {"-": ["53"]},
    "ADVERSARIVS": {"1": ["12"], "2": ["21"]},
    "ADVERSO": {"-": ["51"]},
    "ADVERSOR": {"-": ["5J"]},
    "ADVERSVM": {"-": ["12"]},
    "ADVERSVS": {"1": ["21"], "2": ["60"], "-": ["70"], "3": ["70"]},
    "ADVERTO": {"-": ["33"]},
    "ADVESPERASCO": {"-": ["53"]},
    "ADVIGILO": {"-": ["51"]},
    "ADVIVO": {"-": ["53"]},
    "ADVLATIO": {"-": ["13"]},
    "ADVLATOR": {"-": ["13"]},
    "ADVLATORIVS": {"-": ["21"]},
    "ADVLESCENTIA": {"-": ["11"]},
    "ADVLO": {"-": ["51"]},
    "ADVLOR": {"-": ["5J"]},
    "ADVLTER": {"1": ["12"], "2": ["21"]},
    "ADVLTERA": {"-": ["11"]},
    "ADVLTERINVS": {"-": ["21"]},
    "ADVLTERIVM": {"-": ["12"]},
    "ADVLTERO": {"-": ["51"]},
    "ADVLTVS": {"-": ["21"]},
    "ADVMBRATIM": {"-": ["60"]},
    "ADVMBRATIO": {"-": ["13"]},
    "ADVMBRATVS": {"-": ["21"]},
    "ADVMBRO": {"-": ["51"]},
    "ADVNCVS": {"-": ["21"]},
    "ADVOCATIO": {"-": ["13"]},
    "ADVOCATVS": {"-": ["12"]},
    "ADVOCO": {"-": ["51"]},
    "ADVOLO": {"2": ["51"]},
    "ADVOLVO": {"-": ["53"]},
    "ADVRGEO": {"-": ["52"]},
    "ADVRO": {"-": ["53"]},
    "ADVSTVS": {"-": ["21"]},
    "ADYTVM": {"-": ["12"]},
    "AEBVTIVS": {"N": ["12"], "A": ["21"]},
    "AEDES": {"-": ["13"]},
    "AEDICVLA": {"-": ["11"]},
    "AEDIFICATIO": {"-": ["13"]},
    "AEDIFICATOR": {"-": ["13"]},
    "AEDIFICIVM": {"-": ["12"]},
    "AEDIFICO": {"-": ["51"]},
    "AEDILICIVS": {"1": ["12"], "2": ["21"]},
    "AEDILIS": {"-": ["13"]},
    "AEDILITAS": {"-": ["13"]},
    "AEDITVENS": {"-": ["13"]},
    "AEDITVMVS": {"-": ["12"]},
    "AEDITVVS": {"-": ["12"]},
    "AEDON": {"-": ["17"]},
    "AEGAEVS": {"-": ["21"], "A": ["21"]},
    "AEGER": {"1": ["12"], "2": ["21"]},
    "AEGINIVM": {"N": ["12"]},
    "AEGIS": {"-": ["17"], "N": ["13"]},
    "AEGRE": {"-": ["60"]},
    "AEGREO": {"-": ["52"]},
    "AEGRESCO": {"-": ["53"]},
    "AEGRIMONIA": {"-": ["11"]},
    "AEGRITOMARVS": {"N": ["12"]},
    "AEGRITVDO": {"-": ["13"]},
    "AEGROR": {"-": ["13"]},
    "AEGROTATIO": {"-": ["13"]},
    "AEGROTO": {"-": ["51"]},
    "AEGROTVS": {"-": ["21"]},
    "AEGRVM": {"C": ["12"], "-": ["12"]},
    "AEGYPTIVS": {"N": ["21"], "A": ["21"]},
    "AEGYPTVS": {"N": ["12"]},
    "AELINOS": {"-": ["17"]},
    "AELVRVS": {"-": ["12"]},
    "AEMILIANVS": {"N": ["21"], "A": ["21"]},
    "AEMILIVS": {"N": ["12"], "A": ["21"]},
    "AEMVLA": {"-": ["11"]},
    "AEMVLATIO": {"-": ["13"]},
    "AEMVLATOR": {"-": ["13"]},
    "AEMVLATVS": {"-": ["14"]},
    "AEMVLOR": {"-": ["5J"]},
    "AEMVLVS": {"1": ["12"], "2": ["21"]},
    "AENEAS": {"N": ["17"]},
    "AENEATOR": {"-": ["13"]},
    "AENEVM": {"-": ["12"]},
    "AENEVS": {"-": ["21"]},
    "AENIGMA": {"-": ["17"]},
    "AENIPES": {"-": ["25"]},
    "AENVM": {"-": ["12"]},
    "AENVS": {"-": ["21"], "N": ["12"]},
    "AEQVABILIS": {"-": ["24"]},
    "AEQVABILITER": {"-": ["60"]},
    "AEQVAEVVS": {"-": ["21"]},
    "AEQVALIS": {"1": ["13"], "2": ["24"]},
    "AEQVALITAS": {"-": ["13"]},
    "AEQVALITER": {"-": ["60"]},
    "AEQVANIMITAS": {"-": ["13"]},
    "AEQVE": {"-": ["60"]},
    "AEQVI": {"N": ["12"]},
    "AEQVICOLI": {"N": ["12"]},
    "AEQVICVS": {"N": ["21"]},
    "AEQVILATATIO": {"-": ["13"]},
    "AEQVINOCTIALIS": {"-": ["24"]},
    "AEQVINOCTIVM": {"-": ["12"]},
    "AEQVIPARO": {"-": ["51"]},
    "AEQVIPOLLENS": {"-": ["25"]},
    "AEQVIPONDIVM": {"-": ["12"]},
    "AEQVITAS": {"-": ["13"]},
    "AEQVO": {"-": ["51"]},
    "AEQVOR": {"-": ["13"], "N": ["13"]},
    "AEQVOREVS": {"-": ["21"]},
    "AEQVVM": {"-": ["12"]},
    "AEQVVS": {"-": ["21"], "N": ["21"]},
    "AER": {"-": ["17"]},
    "AERACEVS": {"-": ["21"]},
    "AERARIA": {"-": ["11"]},
    "AERARIVM": {"-": ["12"]},
    "AERARIVS": {"1": ["12"], "2": ["21"]},
    "AERATVS": {"-": ["21"]},
    "AEREVS": {"1": ["12"], "2": ["21"]},
    "AERIPES": {"-": ["25"]},
    "AERIVS": {"-": ["21"]},
    "AERO": {"-": ["13"]},
    "AERVCA": {"-": ["11"]},
    "AERVGINOSVS": {"-": ["21"]},
    "AERVGO": {"-": ["13"]},
    "AERVMNA": {"-": ["11"]},
    "AERVMNABILIS": {"-": ["24"]},
    "AERVMNOSVS": {"-": ["21"]},
    "AES": {"-": ["13"]},
    "AESCHRIO": {"N": ["17"]},
    "AESCHYLVS": {"N": ["12"]},
    "AESCVLAPIVS": {"N": ["12"]},
    "AESCVLETVM": {"-": ["12"]},
    "AESCVLINVS": {"-": ["21"]},
    "AESCVLVS": {"-": ["12"]},
    "AESERNINVS": {"C": ["12"], "N": ["12"]},
    "AESOPEVS": {"N": ["21"], "A": ["21"]},
    "AESTAS": {"-": ["13"]},
    "AESTIFER": {"-": ["21"]},
    "AESTIMATIO": {"-": ["13"]},
    "AESTIMATOR": {"-": ["13"]},
    "AESTIMO": {"-": ["51"]},
    "AESTIVA": {"-": ["12"]},
    "AESTIVVS": {"-": ["21"]},
    "AESTVARIVM": {"-": ["12"]},
    "AESTVO": {"-": ["51"]},
    "AESTVOSE": {"-": ["6-"]},
    "AESTVOSVS": {"-": ["21"]},
    "AESTVS": {"-": ["14"]},
    "AETAS": {"-": ["13"]},
    "AETERNITAS": {"-": ["13"]},
    "AETERNO": {"1": ["51"], "2": ["60"]},
    "AETERNVM": {"1": ["12"], "2": ["60"]},
    "AETERNVS": {"-": ["21"]},
    "AETHER": {"-": ["17"]},
    "AETHERIVS": {"-": ["21"]},
    "AETHRA": {"-": ["11"], "N": ["11"]},
    "AETIOLOGIA": {"-": ["17"]},
    "AETNA": {"1N": ["11"], "2N": ["11"], "N": ["11"]},
    "AETNAEVS": {"N": ["21"], "A": ["21"]},
    "AETNENSES": {"N": ["13"]},
    "AETNENSIS": {"N": ["24"], "A": ["24"]},
    "AETOLIA": {"N": ["11"]},
    "AETOLICVS": {"N": ["21"]},
    "AETOLVS": {"N": ["21"], "A": ["21"]},
    "AEVVM": {"-": ["12"]},
    "AEVVS": {"-": ["12"]},
    "AFFABILIS": {"-": ["24"]},
    "AFFABRE": {"-": ["60"]},
    "AFFATIM": {"-": ["60"]},
    "AFFATVS": {"-": ["14"]},
    "AFFECTATIO": {"-": ["13"]},
    "AFFECTATOR": {"-": ["13"]},
    "AFFECTIO": {"-": ["13"]},
    "AFFECTO": {"-": ["51"]},
    "AFFECTVS": {"1": ["14"], "2": ["21"]},
    "AFFERO": {"-": ["56"]},
    "AFFICIO": {"-": ["55"]},
    "AFFIGO": {"-": ["53"]},
    "AFFINGO": {"-": ["53"]},
    "AFFINIS": {"1": ["13"], "2": ["24"]},
    "AFFINITAS": {"-": ["13"]},
    "AFFIRMATIO": {"-": ["13"]},
    "AFFIRMO": {"-": ["51"]},
    "AFFLATVS": {"-": ["14"]},
    "AFFLICTATIO": {"-": ["13"]},
    "AFFLICTO": {"-": ["51"]},
    "AFFLICTVS": {"1": ["14"], "2": ["21"]},
    "AFFLIGO": {"-": ["53"]},
    "AFFLO": {"-": ["51"]},
    "AFFLVENS": {"-": ["25"]},
    "AFFLVENTER": {"-": ["60"]},
    "AFFLVENTIA": {"-": ["11"]},
    "AFFLVO": {"-": ["53"]},
    "AFFOR": {"-": ["5J"]},
    "AFFRICO": {"-": ["51"]},
    "AFFVLGEO": {"-": ["52"]},
    "AFFVNDO": {"2": ["53"]},
    "AFRICA": {"N": ["11"]},
    "AFRICANVS": {"C": ["12"], "N": ["21"], "A": ["21"]},
    "AFRICVS": {"A": ["21"], "N": ["12"]},
    "AGASO": {"-": ["13"]},
    "AGATHINVS": {"N": ["12"]},
    "AGATHOCLES": {"N": ["17"]},
    "AGE": {"-": ["90"]},
    "AGEDVM": {"-": ["90"]},
    "AGELLVS": {"-": ["12"]},
    "AGENS": {"1": ["13"], "2": ["25"]},
    "AGER": {"-": ["12"], "N": ["12"]},
    "AGGER": {"-": ["13"]},
    "AGGERATIO": {"-": ["13"]},
    "AGGERO": {"1": ["51"], "2": ["53"]},
    "AGGESTVS": {"-": ["14"]},
    "AGGLOMERO": {"-": ["51"]},
    "AGGLVTINO": {"-": ["51"]},
    "AGGRAVO": {"-": ["51"]},
    "AGGREDIOR": {"-": ["5N"]},
    "AGGREGO": {"-": ["51"]},
    "AGILIS": {"-": ["24"]},
    "AGILITAS": {"-": ["13"]},
    "AGITABILIS": {"-": ["24"]},
    "AGITATIO": {"-": ["13"]},
    "AGITATOR": {"-": ["13"]},
    "AGITATVS": {"1": ["14"], "2": ["21"]},
    "AGITE": {"-": ["90"]},
    "AGITO": {"-": ["51"]},
    "AGMEN": {"-": ["13"]},
    "AGNA": {"-": ["11"]},
    "AGNASCOR": {"-": ["5L"]},
    "AGNATVS": {"-": ["12"]},
    "AGNINA": {"-": ["11"]},
    "AGNINVS": {"-": ["21"]},
    "AGNOSCO": {"-": ["53"]},
    "AGNVS": {"-": ["12"]},
    "AGO": {"-": ["53"]},
    "AGON": {"-": ["17"]},
    "AGRAMMATVS": {"-": ["21"]},
    "AGRARII": {"-": ["12"]},
    "AGRARIVS": {"-": ["21"]},
    "AGRESTIS": {"1": ["13"], "2": ["24"], "N": ["13"]},
    "AGRICOLA": {"-": ["11"], "N": ["11"]},
    "AGRIGENTINI": {"N": ["12"]},
    "AGRIGENTINVS": {"N": ["21"], "A": ["21"]},
    "AGRIGENTVM": {"N": ["12"]},
    "AGRIPPA": {"N": ["11"]},
    "AGRO": {"-": ["51"]},
    "AGYRINENSES": {"N": ["13"]},
    "AGYRINENSIS": {"N": ["24"], "A": ["24"]},
    "AGYRIVM": {"N": ["12"]},
    "AH": {"-": ["90"]},
    "AIAX": {"N": ["13"]},
    "AIO": {"-": ["56"]},
    "ALA": {"-": ["11"]},
    "ALACER": {"-": ["23"]},
    "ALACRITAS": {"-": ["13"]},
    "ALACRITER": {"-": ["60"]},
    "ALAPA": {"-": ["11"]},
    "ALARII": {"-": ["12"]},
    "ALARIVS": {"-": ["21"]},
    "ALATVS": {"-": ["21"]},
    "ALBA": {"1N": ["11"], "2N": ["11"], "N": ["11"]},
    "ALBANI": {"N": ["12"]},
    "ALBANVS": {"N": ["21"], "A": ["21"]},
    "ALBARIVS": {"-": ["21"]},
    "ALBATVS": {"-": ["21"]},
    "ALBEO": {"-": ["52"]},
    "ALBESCO": {"-": ["53"]},
    "ALBICERIS": {"-": ["24"]},
    "ALBICO": {"-": ["51"]},
    "ALBIDVS": {"-": ["21"]},
    "ALBINVS": {"N": ["12"]},
    "ALBVLA": {"N": ["11"]},
    "ALBVLVS": {"-": ["21"]},
    "ALBVM": {"-": ["12"]},
    "ALBVS": {"-": ["21"], "A": ["21"]},
    "ALCES": {"-": ["13"]},
    "ALCYON": {"-": ["17"]},
    "ALCYONEVM": {"-": ["12"]},
    "ALEA": {"-": ["11"]},
    "ALEATOR": {"-": ["13"]},
    "ALEBRIA": {"-": ["13"]},
    "ALEO": {"-": ["13"]},
    "ALES": {"1": ["13"], "2": ["25"], "N": ["13"]},
    "ALESCO": {"-": ["53"]},
    "ALEXANDER": {"N": ["17"]},
    "ALEXANDRIA": {"N": ["11"]},
    "ALGA": {"-": ["11"]},
    "ALGEO": {"-": ["52"]},
    "ALGIDVS": {"-": ["21"], "N": ["12"], "A": ["21"]},
    "ALGOR": {"-": ["13"]},
    "ALGVS": {"-": ["14"]},
    "ALIA": {"-": ["60"]},
    "ALIAS": {"-": ["60"]},
    "ALIBI": {"-": ["60"]},
    "ALICA": {"-": ["11"]},
    "ALICVBI": {"-": ["60"]},
    "ALICVNDE": {"-": ["60"]},
    "ALIENATIO": {"-": ["13"]},
    "ALIENIGENA": {"-": ["11"]},
    "ALIENIGENVS": {"-": ["21"]},
    "ALIENO": {"-": ["51"]},
    "ALIENVM": {"C": ["12"], "-": ["12"]},
    "ALIENVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "ALIGER": {"-": ["21"]},
    "ALIMENTVM": {"-": ["12"]},
    "ALIMONIVM": {"-": ["12"]},
    "ALIO": {"-": ["60"]},
    "ALIOQVI": {"-": ["60"]},
    "ALIPES": {"-": ["25"]},
    "ALIPILVS": {"-": ["12"]},
    "ALIPTES": {"-": ["17"]},
    "ALIQVA": {"-": ["60"]},
    "ALIQVAM": {"-": ["60"]},
    "ALIQVAMDIV": {"-": ["60"]},
    "ALIQVAMMVLTVS": {"-": ["21"]},
    "ALIQVANDO": {"-": ["60"]},
    "ALIQVANTO": {"-": ["60"]},
    "ALIQVANTVLVM": {"1": ["48"], "2": ["60"]},
    "ALIQVANTVM": {"1": ["48"], "2": ["60"]},
    "ALIQVANTVS": {"-": ["48"]},
    "ALIQVATENVS": {"-": ["60"]},
    "ALIQVIS": {"-": ["48"]},
    "ALIQVO": {"-": ["60"]},
    "ALIQVOT": {"-": ["48"]},
    "ALIQVOTFARIAM": {"-": ["60"]},
    "ALIQVOTIENS": {"-": ["60"]},
    "ALIS": {"-": ["48"]},
    "ALITER": {"-": ["60"]},
    "ALIVBI": {"-": ["60"]},
    "ALIVNDE": {"-": ["60"]},
    "ALIVS": {"-": ["48"]},
    "ALIVSMODI": {"-": ["60"]},
    "ALLABOR": {"2": ["5L"]},
    "ALLABORO": {"-": ["51"]},
    "ALLAPSVS": {"-": ["14"]},
    "ALLAVDABILIS": {"-": ["24"]},
    "ALLEC": {"-": ["13"]},
    "ALLECTO": {"-": ["51"], "N": ["11"]},
    "ALLEGATIO": {"-": ["13"]},
    "ALLEGO": {"1": ["51"], "2": ["53"]},
    "ALLEVO": {"1": ["51"], "2": ["51"]},
    "ALLEX": {"-": ["13"]},
    "ALLICEFACIO": {"-": ["55"]},
    "ALLICIO": {"-": ["55"]},
    "ALLIDO": {"-": ["53"]},
    "ALLIGATIO": {"-": ["13"]},
    "ALLIGO": {"2": ["51"]},
    "ALLINO": {"-": ["53"]},
    "ALLIVM": {"-": ["12"]},
    "ALLOBROGES": {"N": ["17"]},
    "ALLOBROX": {"N": ["27"], "A": ["25"]},
    "ALLOCVTIO": {"-": ["13"]},
    "ALLOQVIVM": {"-": ["12"]},
    "ALLOQVOR": {"-": ["5L"]},
    "ALLVCEO": {"-": ["52"]},
    "ALLVDO": {"-": ["53"]},
    "ALLVO": {"-": ["53"]},
    "ALMVS": {"-": ["21"]},
    "ALNEVS": {"-": ["21"]},
    "ALNVS": {"-": ["12"]},
    "ALO": {"-": ["53"]},
    "ALOE": {"-": ["17"]},
    "ALPES": {"N": ["13"]},
    "ALPHA": {"-": ["16"]},
    "ALSIVS": {"-": ["21"]},
    "ALTAR": {"-": ["13"]},
    "ALTE": {"-": ["60"]},
    "ALTER": {"-": ["48"]},
    "ALTERAS": {"-": ["60"]},
    "ALTERCATIO": {"-": ["13"]},
    "ALTERCOR": {"-": ["5J"]},
    "ALTERNIS": {"-": ["60"]},
    "ALTERNO": {"-": ["51"]},
    "ALTERNVS": {"-": ["21"]},
    "ALTERO": {"-": ["51"]},
    "ALTERVTER": {"-": ["48"]},
    "ALTILIS": {"1": ["13"], "2": ["24"]},
    "ALTISONVS": {"-": ["21"]},
    "ALTITONANS": {"-": ["25"]},
    "ALTITVDO": {"-": ["13"]},
    "ALTIVOLANS": {"-": ["25"]},
    "A": {"-": ["13"]},
    "ALTRINSECVS": {"-": ["60"]},
    "ALTRIX": {"-": ["13"]},
    "ALTVM": {"-": ["12"]},
    "ALTVS": {"-": ["21"]},
    "ALVARIVM": {"-": ["12"]},
    "ALVCINATIO": {"-": ["13"]},
    "ALVEATVS": {"-": ["21"]},
    "ALVEOLATVS": {"-": ["21"]},
    "ALVEOLVS": {"-": ["12"]},
    "ALVEVS": {"-": ["12"]},
    "ALVMEN": {"-": ["13"]},
    "ALVMINOSVS": {"-": ["21"]},
    "ALVMNA": {"-": ["11"]},
    "ALVMNVS": {"1": ["12"], "2": ["21"]},
    "ALVNTINI": {"N": ["12"]},
    "ALVNTINVS": {"N": ["21"], "A": ["21"]},
    "ALVNTIVM": {"N": ["12"]},
    "ALVTA": {"1": ["11"], "2": ["11"]},
    "ALVVS": {"-": ["12"]},
    "AM": {"-": ["70"]},
    "AMABILIS": {"-": ["24"]},
    "AMABILITER": {"-": ["60"]},
    "AMANDO": {"2": ["51"], "-": ["51"]},
    "AMANS": {"1": ["13"], "2": ["25"]},
    "AMANTER": {"-": ["60"]},
    "AMARA": {"-": ["12"]},
    "AMARACINVM": {"-": ["12"]},
    "AMARACVS": {"-": ["12"]},
    "AMARANTVS": {"-": ["12"]},
    "AMARE": {"-": ["60"]},
    "AMARITAS": {"-": ["13"]},
    "AMARITIES": {"-": ["15"]},
    "AMARITVDO": {"-": ["13"]},
    "AMAROR": {"-": ["13"]},
    "AMARVS": {"-": ["21"]},
    "AMATOR": {"-": ["13"]},
    "AMATORIVM": {"-": ["12"]},
    "AMATORIVS": {"-": ["21"]},
    "AMBACTVS": {"-": ["12"]},
    "AMBAGES": {"-": ["13"]},
    "AMBARVALIS": {"-": ["24"]},
    "AMBEDO": {"2": ["56"]},
    "AMBEO": {"1": ["56"]},
    "AMBIGO": {"-": ["53"]},
    "AMBIGVE": {"-": ["60"]},
    "AMBIGVITAS": {"-": ["13"]},
    "AMBIGVVM": {"-": ["12"]},
    "AMBIGVVS": {"-": ["21"]},
    "AMBIO": {"-": ["56"]},
    "AMBITIO": {"-": ["13"], "N": ["13"]},
    "AMBITIOSE": {"-": ["60"]},
    "AMBITIOSVS": {"-": ["21"]},
    "AMBITVS": {"-": ["14"]},
    "AMBO": {"2": ["31"]},
    "AMBROSIA": {"-": ["11"]},
    "AMBROSIVS": {"-": ["21"], "N": ["12"]},
    "AMBVBAIA": {"-": ["11"]},
    "AMBVLATIO": {"-": ["13"]},
    "AMBVLATOR": {"-": ["13"]},
    "AMBVLATORIVS": {"-": ["21"]},
    "AMBVLATRIX": {"-": ["13"]},
    "AMBVLO": {"-": ["51"]},
    "AMBVRO": {"-": ["53"]},
    "AMELLVS": {"-": ["12"]},
    "AMENS": {"-": ["25"]},
    "AMENTIA": {"-": ["11"]},
    "AMENTVM": {"-": ["12"]},
    "AMERIOLA": {"N": ["11"]},
    "AMES": {"-": ["13"]},
    "AMESTRATINI": {"N": ["12"]},
    "AMESTRATINVS": {"N": ["21"], "A": ["21"]},
    "AMESTRATVS": {"N": ["12"]},
    "AMETHYSTINA": {"-": ["12"]},
    "AMETHYSTVS": {"-": ["12"]},
    "AMICA": {"-": ["11"]},
    "AMICE": {"-": ["60"]},
    "AMICIO": {"-": ["54"]},
    "AMICITIA": {"-": ["11"]},
    "AMICITIES": {"-": ["15"]},
    "AMICTVS": {"-": ["14"]},
    "AMICVLVM": {"-": ["12"]},
    "AMICVLVS": {"-": ["12"]},
    "AMICVS": {"1": ["12"], "2": ["21"]},
    "AMISSIO": {"-": ["13"]},
    "AMITA": {"-": ["11"]},
    "AMITTO": {"-": ["53"]},
    "AMMONIACVS": {"-": ["21"]},
    "AMNIS": {"-": ["13"]},
    "AMO": {"-": ["51"]},
    "AMOENITAS": {"-": ["13"]},
    "AMOENVS": {"-": ["21"]},
    "AMOLIOR": {"-": ["5M"]},
    "AMOMON": {"-": ["17"]},
    "AMOMVM": {"-": ["12"]},
    "AMOR": {"-": ["13"], "N": ["13"]},
    "AMOVEO": {"-": ["52"]},
    "AMPHERES": {"-": ["27"]},
    "AMPHILOCHI": {"N": ["12"]},
    "AMPHIO": {"-": ["17"]},
    "AMPHIPOLIS": {"N": ["17"]},
    "AMPHIPROSTYLOS": {"-": ["27"]},
    "AMPHITHALAMVS": {"-": ["17"]},
    "AMPHITHEATRVM": {"-": ["12"]},
    "AMPHORA": {"-": ["11"]},
    "AMPLA": {"-": ["11"]},
    "AMPLE": {"-": ["60"]},
    "AMPLECTOR": {"-": ["5L"]},
    "AMPLEXOR": {"-": ["5J"]},
    "AMPLEXVS": {"-": ["14"]},
    "AMPLIFICATIO": {"-": ["13"]},
    "AMPLIFICATOR": {"-": ["13"]},
    "AMPLIFICE": {"-": ["60"]},
    "AMPLIFICO": {"-": ["51"]},
    "AMPLIO": {"-": ["51"]},
    "AMPLITVDO": {"-": ["13"]},
    "AMPLIVS": {"-": ["13"]},
    "AMPLVS": {"-": ["21"]},
    "AMPVLLOR": {"-": ["5J"]},
    "AMPVTO": {"-": ["51"]},
    "AMVLIVS": {"N": ["12"]},
    "AMVRCA": {"-": ["11"]},
    "AMVRCARIVS": {"-": ["21"]},
    "AMVSOS": {"-": ["27"]},
    "AMVSSIVM": {"-": ["12"]},
    "AMYGDALVM": {"-": ["12"]},
    "AMYLVM": {"-": ["12"]},
    "AMYSTIS": {"-": ["17"]},
    "AN": {"-": ["67"]},
    "ANABATHRVM": {"-": ["12"]},
    "ANADEMA": {"-": ["17"]},
    "ANALECTA": {"-": ["11"]},
    "ANALEMMA": {"-": ["17"]},
    "ANALEMPTRIS": {"-": ["13"]},
    "ANALOGIA": {"-": ["17"]},
    "ANAPHORICVS": {"-": ["21"]},
    "ANATONVS": {"-": ["21"]},
    "ANCEPS": {"1": ["13"], "2": ["25"]},
    "ANCHISES": {"N": ["17"]},
    "ANCILE": {"-": ["16"]},
    "ANCILLA": {"-": ["11"]},
    "ANCILLARIOLVS": {"-": ["12"]},
    "ANCILLARIS": {"-": ["24"]},
    "ANCILLVLA": {"-": ["11"]},
    "ANCISVS": {"-": ["21"]},
    "ANCON": {"-": ["17"], "N": ["17"]},
    "ANCORA": {"-": ["11"]},
    "ANCVS": {"N": ["12"]},
    "ANDRO": {"N": ["17"]},
    "ANDRON": {"-": ["17"], "N": ["17"]},
    "ANDRONITIS": {"-": ["17"]},
    "ANDROSTHENES": {"N": ["13"]},
    "ANELLVS": {"-": ["12"]},
    "ANETHVM": {"-": ["12"]},
    "ANFRACTVS": {"1": ["14"], "2": ["21"]},
    "ANGELLVS": {"-": ["12"]},
    "ANGELVS": {"-": ["12"]},
    "ANGINA": {"-": ["11"]},
    "ANGIPORTVM": {"-": ["12"]},
    "ANGIPORTVS": {"-": ["14"]},
    "ANGO": {"-": ["53"]},
    "ANGOR": {"-": ["13"]},
    "ANGVBATAE": {"-": ["11"]},
    "ANGVICOMVS": {"-": ["21"]},
    "ANGVIFER": {"-": ["21"]},
    "ANGVILLA": {"-": ["11"]},
    "ANGVIMANVS": {"-": ["26"]},
    "ANGVINEVS": {"-": ["21"]},
    "ANGVINVS": {"-": ["21"]},
    "ANGVIS": {"-": ["13"], "N": ["13"]},
    "ANGVLARIS": {"-": ["24"]},
    "ANGVLVS": {"-": ["12"]},
    "ANGVSTE": {"-": ["60"]},
    "ANGVSTIA": {"-": ["11"]},
    "ANGVSTO": {"-": ["51"]},
    "ANGVSTVM": {"-": ["12"]},
    "ANGVSTVS": {"-": ["21"]},
    "ANHELITVS": {"-": ["14"]},
    "ANHELO": {"-": ["51"]},
    "ANHELVS": {"-": ["21"]},
    "ANIATROLOGETVS": {"-": ["21"]},
    "ANICVLA": {"-": ["11"]},
    "ANILIS": {"-": ["24"]},
    "ANILITAS": {"-": ["13"]},
    "ANILITER": {"-": ["60"]},
    "ANIMA": {"-": ["11"]},
    "ANIMADVERSIO": {"-": ["13"]},
    "ANIMADVERTO": {"-": ["53"]},
    "ANIMAL": {"-": ["13"]},
    "ANIMALIS": {"-": ["24"]},
    "ANIMANS": {"1": ["13"], "2": ["25"]},
    "ANIMATIO": {"-": ["13"]},
    "ANIMATVS": {"-": ["21"]},
    "ANIMO": {"-": ["51"]},
    "ANIMOSE": {"-": ["6&"]},
    "ANIMOSVS": {"-": ["21"]},
    "ANIMVS": {"-": ["12"]},
    "ANIO": {"N": ["16"]},
    "ANISOCYCLI": {"-": ["17"]},
    "ANISVM": {"-": ["12"]},
    "ANNAEA": {"N": ["11"]},
    "ANNAEVS": {"N": ["12"]},
    "ANNALIS": {"1": ["13"], "2": ["24"], "N": ["13"]},
    "ANNATO": {"-": ["51"]},
    "ANNE": {"-": ["67"]},
    "ANNECTO": {"-": ["53"]},
    "ANNIA": {"N": ["11"]},
    "ANNIANVS": {"A": ["21"]},
    "ANNICVLVS": {"-": ["21"]},
    "ANNITOR": {"2": ["5L"]},
    "ANNIVERSARIVS": {"-": ["21"]},
    "ANNIVS": {"N": ["12"]},
    "ANNO": {"1": ["51"], "2": ["51"]},
    "ANNONA": {"-": ["11"]},
    "ANNOSVS": {"-": ["21"]},
    "ANNOTINVS": {"-": ["21"]},
    "ANNOTO": {"-": ["51"]},
    "ANNVMERO": {"1": ["51"]},
    "ANNVNTIO": {"-": ["51"]},
    "ANNVO": {"-": ["53"]},
    "ANNVS": {"-": ["12"], "N": ["12"]},
    "ANNVVM": {"-": ["12"]},
    "ANNVVS": {"-": ["21"]},
    "ANQVIRO": {"-": ["53"]},
    "ANSA": {"-": ["11"]},
    "ANSER": {"-": ["13"], "N": ["12"]},
    "ANTAE": {"-": ["11"]},
    "ANTARIVS": {"-": ["21"]},
    "ANTE": {"1": ["60"], "2": ["70"]},
    "ANTEA": {"-": ["60"]},
    "ANTEACTVS": {"-": ["21"]},
    "ANTEAGO": {"-": ["53"]},
    "ANTEAQVAM": {"-": ["82"]},
    "ANTECAPIO": {"2": ["55"]},
    "ANTECEDENS": {"-": ["25"]},
    "ANTECEDENTIA": {"-": ["13"]},
    "ANTECEDO": {"1": ["53"]},
    "ANTECELLO": {"-": ["53"]},
    "ANTECESSOR": {"-": ["13"]},
    "ANTECHON": {"-": ["57"]},
    "ANTECVRRO": {"-": ["53"]},
    "ANTECVRSOR": {"-": ["13"]},
    "ANTEEO": {"1": ["56"]},
    "ANTEFERO": {"-": ["56"]},
    "ANTEFIXVM": {"-": ["12"]},
    "ANTEFIXVS": {"-": ["21"]},
    "ANTEHABEO": {"-": ["52"]},
    "ANTEHAC": {"-": ["60"]},
    "ANTELVCANVS": {"-": ["21"]},
    "ANTEMERIDIANVS": {"-": ["21"]},
    "ANTEMNATES": {"N": ["13"]},
    "ANTENNA": {"-": ["11"]},
    "ANTENOR": {"N": ["17"]},
    "ANTEPAGMENTVM": {"-": ["12"]},
    "ANTEPONO": {"-": ["53"]},
    "ANTEQVAM": {"-": ["82"], "1": ["6-"]},
    "ANTERIS": {"-": ["17"]},
    "ANTES": {"-": ["13"]},
    "ANTESIGNANVS": {"1": ["12"], "2": ["21"]},
    "ANTESTO": {"-": ["51"]},
    "ANTESTOR": {"-": ["5J"]},
    "ANTEVENIO": {"-": ["54"]},
    "ANTEVERTO": {"-": ["53"]},
    "ANTHROPINVS": {"N": ["12"]},
    "ANTIATES": {"N": ["13"]},
    "ANTIBASIS": {"-": ["17"]},
    "ANTIBOREVM": {"-": ["12"]},
    "ANTICIPO": {"-": ["51"]},
    "ANTIDOTVM": {"-": ["12"]},
    "ANTIGONVS": {"N": ["12"]},
    "ANTIOCHENSES": {"N": ["13"]},
    "ANTIOCHIA": {"N": ["11"]},
    "ANTIOCHVS": {"N": ["12"]},
    "ANTIPODES": {"-": ["17"]},
    "ANTIQVARIA": {"-": ["11"]},
    "ANTIQVARIVS": {"1": ["12"], "2": ["21"]},
    "ANTIQVE": {"-": ["60"]},
    "ANTIQVI": {"-": ["12"]},
    "ANTIQVITAS": {"-": ["13"]},
    "ANTIQVITVS": {"-": ["60"]},
    "ANTIQVVM": {"-": ["12"]},
    "ANTIQVVS": {"-": ["21"]},
    "ANTISTES": {"-": ["13"]},
    "ANTISTITA": {"-": ["11"]},
    "ANTISTIVS": {"N": ["12"]},
    "ANTITHETON": {"-": ["17"]},
    "ANTIVM": {"N": ["12"]},
    "ANTONIANI": {"N": ["12"]},
    "ANTONIANVS": {"N": ["21"], "A": ["21"]},
    "ANTONIVS": {"N": ["12"], "A": ["21"]},
    "ANTRVM": {"-": ["12"]},
    "ANVLARIVS": {"-": ["21"]},
    "ANVLVS": {"-": ["12"]},
    "ANVS": {"1": ["12"], "2": ["14"]},
    "ANXIE": {"-": ["60"]},
    "ANXIETAS": {"-": ["13"]},
    "ANXIVS": {"-": ["21"]},
    "AONIA": {"N": ["11"]},
    "APELLES": {"N": ["13"]},
    "APER": {"-": ["12"], "N": ["12"]},
    "APERIO": {"-": ["54"]},
    "APERTE": {"-": ["60"]},
    "APERTVM": {"-": ["12"]},
    "APERTVRA": {"-": ["11"]},
    "APERTVS": {"-": ["21"]},
    "APEX": {"-": ["13"]},
    "APIACVS": {"-": ["21"]},
    "APICIVS": {"-": ["21"], "N": ["12"]},
    "APIOLAE": {"N": ["11"]},
    "APIS": {"-": ["13"], "N": ["17"]},
    "APISCOR": {"-": ["5L"]},
    "APIVM": {"-": ["12"]},
    "APLVSTRE": {"-": ["16"]},
    "APOLLO": {"N": ["13"]},
    "APOLLODORVS": {"N": ["12"]},
    "APOLLONIA": {"N": ["11"]},
    "APOLLONIATAE": {"N": ["17"]},
    "APOLLONIENSIS": {"N": ["24"], "A": ["24"]},
    "APOLLONIVS": {"N": ["12"]},
    "APOLOGO": {"-": ["51"]},
    "APOPHYSIS": {"-": ["17"]},
    "APOTHECA": {"-": ["11"]},
    "APOTHESIS": {"-": ["17"]},
    "APPAGINECVLI": {"-": ["12"]},
    "APPARATIO": {"-": ["13"]},
    "APPARATVS": {"1": ["14"], "2": ["21"]},
    "APPAREO": {"-": ["52"]},
    "APPARIO": {"-": ["55"]},
    "APPARITIO": {"-": ["13"]},
    "APPARITOR": {"-": ["13"]},
    "APPARO": {"2": ["51"]},
    "APPELLATIO": {"-": ["13"]},
    "APPELLATOR": {"-": ["13"]},
    "APPELLITO": {"-": ["51"]},
    "APPELLO": {"1": ["51"], "2": ["53"]},
    "APPENDEO": {"-": ["52"]},
    "APPENDIX": {"-": ["13"]},
    "APPENDO": {"-": ["53"]},
    "APPERIOR": {"-": ["5M"]},
    "APPETENS": {"-": ["25"]},
    "APPETITIO": {"-": ["13"]},
    "APPETITVS": {"-": ["14"]},
    "APPETO": {"-": ["53"]},
    "APPIVS": {"N": ["12"], "A": ["21"]},
    "APPLAVDO": {"-": ["53"]},
    "APPLICATIO": {"-": ["13"]},
    "APPLICO": {"-": ["51"]},
    "APPLORO": {"-": ["51"]},
    "APPONO": {"-": ["53"]},
    "APPORTATIO": {"-": ["13"]},
    "APPORTO": {"-": ["51"]},
    "APPOSITVS": {"-": ["21"]},
    "APPRECOR": {"-": ["5J"]},
    "APPREHENDO": {"-": ["53"]},
    "APPREHENSIO": {"-": ["13"]},
    "APPRIME": {"-": ["60"]},
    "APPRIMO": {"-": ["53"]},
    "APPRIMVS": {"-": ["21"]},
    "APPROBATIO": {"-": ["13"]},
    "APPROBO": {"-": ["51"]},
    "APPROPERO": {"-": ["51"]},
    "APPROPINQVO": {"-": ["51"]},
    "APPVGNO": {"-": ["51"]},
    "APPVLSVS": {"-": ["14"]},
    "APRICVM": {"-": ["12"]},
    "APRICVS": {"-": ["21"]},
    "APRONIANVS": {"N": ["21"], "A": ["21"]},
    "APRONIVS": {"N": ["12"]},
    "APTE": {"-": ["60"]},
    "APTO": {"-": ["51"]},
    "APTVS": {"-": ["21"]},
    "APVD": {"-": ["70"]},
    "APYRENVM": {"-": ["12"]},
    "AQVA": {"-": ["11"]},
    "AQVALICVLVS": {"-": ["12"]},
    "AQVALIS": {"2": ["24"], "1": ["24"]},
    "AQVARIVM": {"-": ["12"]},
    "AQVARIVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "AQVATICVS": {"-": ["21"]},
    "AQVATIO": {"-": ["13"]},
    "AQVATOR": {"-": ["13"]},
    "AQVATVS": {"-": ["21"]},
    "AQVIFOLIVS": {"-": ["21"]},
    "AQVILA": {"-": ["11"], "N": ["11"]},
    "AQVILIFER": {"-": ["12"]},
    "AQVILIVS": {"N": ["12"], "A": ["21"]},
    "AQVILO": {"-": ["13"], "N": ["13"]},
    "AQVOR": {"-": ["5J"]},
    "AQVOSVS": {"-": ["21"]},
    "AQVVLA": {"-": ["11"]},
    "ARA": {"-": ["11"], "N": ["11"]},
    "ARACHNE": {"-": ["17"], "N": ["11"]},
    "ARAEOSTYLOS": {"-": ["27"]},
    "ARANEA": {"-": ["11"]},
    "ARANEOSVS": {"-": ["21"]},
    "ARANEVS": {"1": ["12"], "2": ["21"]},
    "ARATIO": {"-": ["13"]},
    "ARATOR": {"-": ["13"]},
    "ARATRVM": {"-": ["12"]},
    "ARBITER": {"-": ["12"]},
    "ARBITRA": {"-": ["11"]},
    "ARBITRATVS": {"-": ["14"]},
    "ARBITRIVM": {"-": ["12"]},
    "ARBITROR": {"-": ["5J"]},
    "ARBOR": {"-": ["13"]},
    "ARBORARIVS": {"-": ["21"]},
    "ARBOREVS": {"-": ["21"]},
    "ARBVSCVLA": {"-": ["11"], "N": ["11"]},
    "ARBVSTIVVS": {"-": ["21"]},
    "ARBVSTVM": {"-": ["12"]},
    "ARBVTEVS": {"-": ["21"]},
    "ARBVTVM": {"-": ["12"]},
    "ARBVTVS": {"-": ["12"]},
    "ARCA": {"-": ["11"]},
    "ARCADES": {"N": ["13"]},
    "ARCADIA": {"N": ["11"]},
    "ARCADICVS": {"N": ["21"], "A": ["21"]},
    "ARCANO": {"-": ["60"]},
    "ARCANVM": {"-": ["12"]},
    "ARCANVS": {"-": ["21"]},
    "ARCATVS": {"1": ["12"], "2": ["21"]},
    "ARCEO": {"-": ["52"]},
    "ARCESSITVS": {"1": ["14"], "2": ["21"]},
    "ARCESSO": {"-": ["53"]},
    "ARCHAGATHVS": {"N": ["12"]},
    "ARCHETYPVS": {"-": ["21"]},
    "ARCHIMAGIRVS": {"-": ["12"]},
    "ARCHIMEDES": {"N": ["17"]},
    "ARCHIPIRATA": {"-": ["11"]},
    "ARCHITECTO": {"-": ["51"]},
    "ARCHITECTONICVS": {"-": ["21"]},
    "ARCHITECTOR": {"1": ["13"], "2": ["5J"], "-": ["5J"]},
    "ARCHITECTVRA": {"-": ["11"]},
    "ARCHITECTVS": {"-": ["12"]},
    "ARCHONIDAS": {"N": ["17"]},
    "ARCITENENS": {"1": ["13"], "2": ["25"]},
    "ARCTOS": {"-": ["17"]},
    "ARCTOVS": {"-": ["21"], "A": ["21"]},
    "ARCVLA": {"-": ["11"]},
    "ARCVS": {"-": ["14"], "N": ["12"]},
    "ARDEA": {"-": ["11"], "N": ["11"]},
    "ARDENS": {"-": ["25"], "-": ["22"]},
    "ARDENTER": {"-": ["60"]},
    "ARDEO": {"-": ["52"]},
    "ARDESCO": {"-": ["53"]},
    "ARDOR": {"-": ["13"]},
    "ARDVVM": {"-": ["12"]},
    "ARDVVS": {"-": ["21"]},
    "AREA": {"-": ["11"]},
    "AREFACIO": {"-": ["55"]},
    "ARENA": {"-": ["11"]},
    "ARENARIVM": {"-": ["12"]},
    "ARENATIO": {"-": ["13"]},
    "ARENATVM": {"-": ["12"]},
    "ARENATVS": {"-": ["21"]},
    "ARENOSVS": {"-": ["21"]},
    "AREO": {"-": ["52"]},
    "ARESCO": {"-": ["53"]},
    "ARETALOGVS": {"-": ["12"]},
    "ARETHVSA": {"N": ["11"]},
    "AREVS": {"N": ["12"], "A": ["21"]},
    "ARGEI": {"N": ["12"]},
    "ARGENTARIA": {"-": ["11"]},
    "ARGENTARIVS": {"1": ["12"], "2": ["21"]},
    "ARGENTEVS": {"1": ["12"], "2": ["21"]},
    "ARGENTIFODINA": {"-": ["11"]},
    "ARGENTVM": {"-": ["12"]},
    "ARGILETVM": {"N": ["12"]},
    "ARGILLA": {"-": ["11"]},
    "ARGVMENTATIO": {"-": ["13"]},
    "ARGVMENTOR": {"-": ["5J"]},
    "ARGVMENTVM": {"-": ["12"]},
    "ARGVO": {"-": ["53"]},
    "ARGVTATIO": {"-": ["13"]},
    "ARGVTE": {"-": ["60"]},
    "ARGVTIA": {"-": ["11"]},
    "ARGVTO": {"-": ["51"]},
    "ARGVTVS": {"-": ["21"]},
    "ARICIA": {"N": ["11"]},
    "ARICINI": {"N": ["12"]},
    "ARICINVS": {"N": ["21"], "A": ["21"]},
    "ARIDA": {"-": ["11"]},
    "ARIDVLVS": {"-": ["21"]},
    "ARIDVM": {"-": ["12"]},
    "ARIDVS": {"-": ["21"]},
    "ARIES": {"-": ["13"], "N": ["13"]},
    "ARIETARIVS": {"-": ["21"]},
    "ARIETO": {"-": ["51"]},
    "ARIMINENSES": {"N": ["13"]},
    "ARIMINVM": {"N": ["12"]},
    "ARISTA": {"-": ["11"]},
    "ARISTAEVS": {"N": ["12"]},
    "ARISTEVS": {"N": ["12"]},
    "ARISTIDES": {"N": ["17"]},
    "ARISTODAMVS": {"N": ["12"]},
    "ARISTODEMVS": {"N": ["12"]},
    "ARISTOTELES": {"N": ["17"]},
    "ARISTVS": {"N": ["12"]},
    "ARITHMETICA": {"-": ["17"]},
    "ARMA": {"-": ["12"]},
    "ARMAMAXA": {"-": ["11"]},
    "ARMAMENTARIVM": {"-": ["12"]},
    "ARMAMENTVM": {"-": ["12"]},
    "ARMARIVM": {"-": ["12"]},
    "ARMATI": {"-": ["12"]},
    "ARMATVRA": {"-": ["11"]},
    "ARMATVS": {"1": ["14"], "2": ["21"]},
    "ARMENTARIVS": {"1": ["12"], "2": ["21"]},
    "ARMENTVM": {"-": ["12"]},
    "ARMIFER": {"-": ["21"]},
    "ARMIGER": {"1": ["12"], "2": ["21"]},
    "ARMILLA": {"-": ["11"]},
    "ARMIPOTENS": {"-": ["25"], "N": ["13"]},
    "ARMISONVS": {"-": ["21"]},
    "ARMO": {"-": ["51"]},
    "ARMVS": {"-": ["12"]},
    "ARO": {"-": ["51"]},
    "ARRECTARIVM": {"-": ["12"]},
    "ARRECTVS": {"-": ["21"]},
    "ARREPO": {"-": ["53"]},
    "ARRIDEO": {"-": ["52"]},
    "ARRIGO": {"-": ["53"]},
    "ARRIPIO": {"-": ["55"]},
    "ARRISOR": {"-": ["13"]},
    "ARRIVS": {"N": ["12"]},
    "ARRODO": {"-": ["53"]},
    "ARROGANS": {"A": ["25"], "-": ["25"]},
    "ARROGANTER": {"-": ["60"]},
    "ARROGANTIA": {"-": ["11"]},
    "ARROGO": {"-": ["51"]},
    "ARROSOR": {"-": ["13"]},
    "ARRVNS": {"N": ["13"]},
    "ARS": {"-": ["13"], "N": ["13"]},
    "ARSENICON": {"-": ["17"]},
    "ARSIA": {"N": ["11"]},
    "ARSINEVM": {"-": ["12"]},
    "ARSINOE": {"N": ["17"]},
    "ARTE": {"-": ["60"]},
    "ARTEMIDORVS": {"N": ["12"]},
    "ARTEMO": {"-": ["13"], "N": ["17"]},
    "ARTERIA": {"-": ["16"]},
    "ARTERIACE": {"-": ["17"]},
    "ARTICVLARIVS": {"-": ["21"]},
    "ARTICVLATIM": {"-": ["60"]},
    "ARTICVLO": {"-": ["51"]},
    "ARTICVLVS": {"-": ["12"]},
    "ARTIFEX": {"-": ["13"]},
    "ARTIFICIOSE": {"-": ["60"]},
    "ARTIFICIOSVS": {"-": ["21"]},
    "ARTIFICIVM": {"-": ["12"]},
    "ARTIO": {"-": ["54"]},
    "ARTO": {"-": ["51"]},
    "ARTOCREAS": {"-": ["17"]},
    "ARTOPTA": {"-": ["11"]},
    "ARTVM": {"-": ["12"]},
    "ARTVS": {"1": ["14"], "2": ["21"]},
    "ARVLA": {"-": ["11"]},
    "ARVNDINETVM": {"-": ["12"]},
    "ARVNDINOSVS": {"-": ["21"]},
    "ARVNDO": {"-": ["13"]},
    "ARVVM": {"-": ["12"]},
    "ARX": {"-": ["13"], "N": ["13"]},
    "AS": {"-": ["13"]},
    "ASCANIVS": {"N": ["12"]},
    "ASCENDO": {"-": ["53"]},
    "ASCENSIO": {"-": ["13"]},
    "ASCENSVS": {"-": ["14"]},
    "ASCIA": {"-": ["11"]},
    "ASCIO": {"1": ["51"], "2": ["54"]},
    "ASCISCO": {"-": ["53"]},
    "ASCITVS": {"1": ["14"], "2": ["21"]},
    "ASCRIBO": {"-": ["53"]},
    "ASCRIPTICIVS": {"-": ["21"]},
    "ASCRIPTIO": {"-": ["13"]},
    "ASELLA": {"-": ["11"]},
    "ASELLVS": {"-": ["12"], "N": ["12"]},
    "ASIA": {"N": ["11"]},
    "ASIATICVS": {"N": ["21"], "A": ["21"]},
    "ASILVS": {"-": ["12"]},
    "ASINARIVS": {"1": ["12"], "2": ["21"]},
    "ASININVS": {"-": ["21"]},
    "ASINIVS": {"N": ["12"]},
    "ASINVS": {"-": ["12"]},
    "ASOTVS": {"-": ["21"]},
    "ASPARAGVS": {"-": ["12"]},
    "ASPECTO": {"-": ["51"]},
    "ASPECTVS": {"-": ["14"]},
    "ASPENDIVS": {"N": ["21"], "A": ["21"]},
    "ASPENDVS": {"N": ["17"]},
    "ASPER": {"-": ["21"], "N": ["12"]},
    "ASPERA": {"-": ["12"]},
    "ASPERE": {"-": ["60"]},
    "ASPERGO": {"1": ["13"], "2": ["53"]},
    "ASPERITAS": {"-": ["13"]},
    "ASPERNATIO": {"-": ["13"]},
    "ASPERNOR": {"-": ["5J"]},
    "ASPERO": {"-": ["51"]},
    "ASPERVM": {"-": ["12"]},
    "ASPHALTITIS": {"-": ["27"]},
    "ASPICIO": {"-": ["55"]},
    "ASPIRO": {"-": ["51"]},
    "ASPIS": {"-": ["17"]},
    "ASPLENOS": {"-": ["17"]},
    "ASPORTATIO": {"-": ["13"]},
    "ASPORTO": {"-": ["51"]},
    "ASSARIVS": {"-": ["21"]},
    "ASSECLA": {"-": ["11"]},
    "ASSECTATOR": {"-": ["13"]},
    "ASSECTOR": {"2": ["5J"]},
    "ASSENSIO": {"-": ["13"]},
    "ASSENSOR": {"-": ["13"]},
    "ASSENSVS": {"-": ["14"]},
    "ASSENTATIO": {"-": ["13"]},
    "ASSENTATOR": {"-": ["13"]},
    "ASSENTIO": {"-": ["54"]},
    "ASSENTIOR": {"-": ["5M"]},
    "ASSENTOR": {"-": ["5J"]},
    "ASSEQVOR": {"-": ["5L"]},
    "ASSER": {"-": ["13"]},
    "ASSERCVLVM": {"-": ["12"]},
    "ASSERO": {"2": ["53"], "3": ["53"]},
    "ASSERTOR": {"-": ["13"]},
    "ASSERVO": {"-": ["51"]},
    "ASSESSOR": {"-": ["13"]},
    "ASSEVERATIO": {"-": ["13"]},
    "ASSEVERO": {"-": ["51"]},
    "ASSICCO": {"-": ["51"]},
    "ASSIDEO": {"-": ["52"]},
    "ASSIDO": {"-": ["53"]},
    "ASSIDVE": {"-": ["60"]},
    "ASSIDVITAS": {"-": ["13"]},
    "ASSIDVVS": {"-": ["21"]},
    "ASSIGNO": {"-": ["51"]},
    "ASSILIO": {"-": ["54"]},
    "ASSIMILIS": {"-": ["24"]},
    "ASSIMVLATIO": {"-": ["13"]},
    "ASSIMVLO": {"-": ["51"]},
    "ASSISTO": {"-": ["53"]},
    "ASSOLEO": {"-": ["5S"]},
    "ASSONO": {"1": ["51"]},
    "ASSORINI": {"N": ["12"]},
    "ASSORINVS": {"N": ["21"], "A": ["21"]},
    "ASSORVM": {"N": ["12"]},
    "ASSVEFACIO": {"-": ["55"]},
    "ASSVESCO": {"-": ["53"]},
    "ASSVETVDO": {"-": ["13"]},
    "ASSVETVS": {"-": ["21"]},
    "ASSVGO": {"-": ["53"]},
    "ASSVLA": {"-": ["11"]},
    "ASSVLTO": {"-": ["51"]},
    "ASSVLTVS": {"-": ["14"]},
    "ASSVM": {"1": ["56"], "2": ["12"]},
    "ASSVMO": {"-": ["53"]},
    "ASSVRGO": {"-": ["53"]},
    "ASSVS": {"-": ["21"]},
    "ASTO": {"-": ["51"]},
    "ASTRAGALVS": {"-": ["12"]},
    "ASTREPO": {"-": ["53"]},
    "ASTRICTE": {"-": ["6&"]},
    "ASTRICTVS": {"A": ["21"], "-": ["21"]},
    "ASTRINGO": {"-": ["53"]},
    "ASTROLOGIA": {"-": ["11"]},
    "ASTROLOGVS": {"-": ["12"]},
    "ASTRONOMIA": {"-": ["11"]},
    "ASTRVM": {"-": ["12"]},
    "ASTRVO": {"-": ["53"]},
    "ASTV": {"-": ["17"]},
    "ASTVPEO": {"-": ["52"]},
    "ASTVRCO": {"-": ["13"]},
    "ASTVS": {"-": ["14"]},
    "ASTVTE": {"-": ["60"]},
    "ASTVTIA": {"-": ["11"]},
    "ASTVTVS": {"-": ["21"]},
    "ASYLVM": {"-": ["12"]},
    "AT": {"1": ["60"], "2": ["81"]},
    "ATAT": {"-": ["90"]},
    "ATAVI": {"-": ["12"]},
    "ATAVVS": {"-": ["12"]},
    "ATER": {"-": ["21"]},
    "ATHENAE": {"N": ["11"]},
    "ATHENIENSES": {"N": ["13"]},
    "ATHENIENSIS": {"N": ["24"], "A": ["24"]},
    "ATHENIO": {"N": ["17"]},
    "ATHLETA": {"-": ["11"]},
    "ATIDIVS": {"N": ["12"]},
    "ATILIVS": {"N": ["12"]},
    "ATINIVS": {"A": ["21"], "N": ["12"]},
    "ATLANTES": {"-": ["17"]},
    "ATOMVS": {"-": ["12"]},
    "ATQVE": {"1": ["81"], "2": ["82"]},
    "ATQVI": {"-": ["81"]},
    "ATRAMENTVM": {"-": ["12"]},
    "ATRATVS": {"-": ["21"]},
    "ATRIENSIS": {"1": ["13"], "2": ["24"]},
    "ATRIVM": {"-": ["12"], "N": ["12"]},
    "ATROCITAS": {"-": ["13"]},
    "ATROCITER": {"-": ["60"]},
    "ATROX": {"-": ["25"]},
    "ATTAGEN": {"-": ["13"]},
    "ATTALICVS": {"N": ["21"], "A": ["21"]},
    "ATTALVS": {"N": ["12"]},
    "ATTAMEN": {"-": ["60"]},
    "ATTEGIA": {"-": ["11"]},
    "ATTEMPERO": {"-": ["51"]},
    "ATTENDO": {"-": ["53"]},
    "ATTENTE": {"-": ["6&"]},
    "ATTENTO": {"-": ["51"]},
    "ATTENTVS": {"1": ["14"], "2": ["21"]},
    "ATTENVATVS": {"-": ["21"]},
    "ATTENVO": {"-": ["51"]},
    "ATTERO": {"-": ["53"]},
    "ATTEXO": {"-": ["53"]},
    "ATTICA": {"N": ["11"]},
    "ATTICVS": {"C": ["12"], "N": ["12"], "A": ["21"]},
    "ATTINEO": {"-": ["52"]},
    "ATTINGO": {"-": ["53"]},
    "ATTIS": {"N": ["17"]},
    "ATTOLLO": {"-": ["53"]},
    "ATTONDEO": {"-": ["52"]},
    "ATTONITVS": {"1": ["14"], "2": ["21"]},
    "ATTONO": {"-": ["51"]},
    "ATTRACTIO": {"-": ["13"]},
    "ATTRACTVS": {"-": ["21"]},
    "ATTRAHO": {"-": ["53"]},
    "ATTRECTO": {"-": ["51"]},
    "ATTRIBVO": {"-": ["53"]},
    "ATTRITVS": {"1": ["14"], "2": ["21"]},
    "ATTVS": {"N": ["12"]},
    "AVARA": {"-": ["11"]},
    "AVARE": {"-": ["60"]},
    "AVARITER": {"-": ["60"]},
    "AVARITIA": {"-": ["11"], "N": ["11"]},
    "AVARITIES": {"-": ["15"]},
    "AVARVS": {"1": ["12"], "2": ["21"]},
    "AVCEPS": {"-": ["13"]},
    "AVCTE": {"-": ["6&"]},
    "AVCTIFICVS": {"-": ["21"]},
    "AVCTIO": {"-": ["13"]},
    "AVCTIONARIVS": {"-": ["21"]},
    "AVCTIONOR": {"-": ["5J"]},
    "AVCTITO": {"-": ["51"]},
    "AVCTO": {"-": ["51"]},
    "AVCTOR": {"-": ["13"]},
    "AVCTORAMENTVM": {"-": ["12"]},
    "AVCTORITAS": {"-": ["13"]},
    "AVCTORO": {"-": ["51"]},
    "AVCTVMNVS": {"1": ["12"]},
    "AVCTVS": {"1": ["14"], "2": ["21"]},
    "AVCVPIVM": {"-": ["12"]},
    "AVCVPO": {"-": ["51"]},
    "AVCVPOR": {"-": ["5J"]},
    "AVDACIA": {"-": ["11"]},
    "AVDACITER": {"-": ["60"]},
    "AVDAX": {"-": ["25"]},
    "AVDENS": {"-": ["25"]},
    "AVDENTER": {"-": ["6&"]},
    "AVDENTIA": {"-": ["11"]},
    "AVDEO": {"-": ["5S"]},
    "AVDIENS": {"-": ["25"]},
    "AVDIO": {"-": ["54"]},
    "AVDITIO": {"-": ["13"]},
    "AVDITOR": {"-": ["13"]},
    "AVDITORIVM": {"-": ["12"]},
    "AVDITVM": {"-": ["12"]},
    "AVDITVS": {"-": ["14"]},
    "AVE": {"-": ["90"]},
    "AVEHO": {"-": ["53"]},
    "AVELLO": {"-": ["53"]},
    "AVENA": {"-": ["11"]},
    "AVENTINVS": {"1N": ["12"], "2N": ["16"], "A": ["21"], "N": ["12"]},
    "AVEO": {"-": ["52"]},
    "AVERNVS": {"N": ["21"], "A": ["21"]},
    "AVERRO": {"-": ["53"]},
    "AVERRVNCO": {"-": ["51"]},
    "AVERSABILIS": {"-": ["24"]},
    "AVERSATIO": {"-": ["13"]},
    "AVERSIO": {"-": ["13"]},
    "AVERSOR": {"1": ["13"], "2": ["5J"]},
    "AVERSVM": {"-": ["12", "12 "]},
    "AVERSVS": {"-": ["21"]},
    "AVERTO": {"-": ["53"]},
    "AVFERO": {"-": ["56"]},
    "AVFVGIO": {"-": ["55"]},
    "AVGEO": {"-": ["52"]},
    "AVGESCO": {"-": ["53"]},
    "AVGMEN": {"-": ["13"]},
    "AVGVR": {"-": ["13"]},
    "AVGVRALE": {"-": ["13"]},
    "AVGVRALIS": {"-": ["24"]},
    "AVGVRATVS": {"-": ["14"]},
    "AVGVRIVM": {"-": ["12"]},
    "AVGVROR": {"-": ["5J"]},
    "AVGVSTA": {"1N": ["11"], "2N": ["11"], "N": ["11"]},
    "AVGVSTE": {"-": ["60"]},
    "AVGVSTVS": {"-": ["21"], "C": ["12"], "N": ["21"], "A": ["21"]},
    "AVIA": {"-": ["11"]},
    "AVIARIVM": {"-": ["12"]},
    "AVIDE": {"-": ["60"]},
    "AVIDITAS": {"-": ["13"]},
    "AVIDVS": {"-": ["21"]},
    "AVIS": {"-": ["13"]},
    "AVITVS": {"-": ["21"], "N": ["12"]},
    "AVIVM": {"-": ["12"]},
    "AVIVS": {"-": ["21"]},
    "AVLA": {"1": ["11"], "2": ["11"]},
    "AVLAEVM": {"-": ["12"]},
    "AVLVS": {"N": ["12"]},
    "AVOCAMENTVM": {"-": ["12"]},
    "AVOCATIO": {"-": ["13"]},
    "AVOCO": {"-": ["51"]},
    "AVOLO": {"2": ["51"]},
    "AVRA": {"-": ["11"], "N": ["11"]},
    "AVRARIA": {"-": ["11"]},
    "AVRATVS": {"-": ["21"]},
    "AVRELIVS": {"N": ["12"], "A": ["21"]},
    "AVREOLVS": {"-": ["21"]},
    "AVREVS": {"1": ["12"], "2": ["21"]},
    "AVRICILLA": {"-": ["11"]},
    "AVRICOMVS": {"-": ["21"]},
    "AVRICVLA": {"-": ["11"]},
    "AVRIFER": {"-": ["21"]},
    "AVRIFERVS": {"-": ["21"]},
    "AVRIFEX": {"-": ["13"]},
    "AVRIGA": {"-": ["11"]},
    "AVRION": {"-": ["60"]},
    "AVRIPIGMENTVM": {"-": ["12"]},
    "AVRIS": {"-": ["13"]},
    "AVRITVS": {"-": ["21"]},
    "AVRO": {"-": ["51"]},
    "AVRORA": {"-": ["11"], "N": ["11"]},
    "AVRVM": {"-": ["12"]},
    "AVRVNCI": {"N": ["12"]},
    "AVRVNCVS": {"N": ["21"], "A": ["21"]},
    "AVSCVLTATIO": {"-": ["13"]},
    "AVSCVLTO": {"-": ["51"]},
    "AVSPEX": {"-": ["13"], "N": ["13"]},
    "AVSPICATVS": {"1": ["14"], "2": ["21"], "-": ["14"]},
    "AVSPICIVM": {"-": ["12"]},
    "AVSPICO": {"-": ["51"]},
    "AVSPICOR": {"-": ["5J"]},
    "AVSTER": {"-": ["12"], "N": ["12"]},
    "AVSTERITAS": {"-": ["13"]},
    "AVSTERVS": {"-": ["21"]},
    "AVSTRALIS": {"-": ["24"]},
    "AVSTRINVS": {"-": ["21"]},
    "AVSVM": {"A": ["12"], "-": ["12"]},
    "AVT": {"-": ["81"]},
    "AVTEM": {"-": ["81"]},
    "AVTIONOR": {"-": ["5J"]},
    "AVTOMATOPOEETVS": {"-": ["21"]},
    "AVTVMNALIS": {"-": ["24"]},
    "AVTVMNITAS": {"-": ["13"]},
    "AVTVMNVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "AVTVMO": {"-": ["51"]},
    "AVVNCVLVS": {"-": ["12"]},
    "AVVS": {"-": ["12"]},
    "AVXILIARES": {"-": ["13"]},
    "AVXILIARII": {"-": ["12"]},
    "AVXILIARIS": {"-": ["24"]},
    "AVXILIARIVS": {"-": ["21"]},
    "AVXILIATOR": {"-": ["13"]},
    "AVXILIATVS": {"-": ["14"]},
    "AVXILIOR": {"-": ["5J"]},
    "AVXILIVM": {"-": ["12"]},
    "AXICVLVS": {"-": ["12"]},
    "AXIS": {"-": ["13"]},
    "AXON": {"-": ["17"]},
    "BABYLONICVM": {"-": ["12"], "N": ["12"]},
    "BACCA": {"-": ["11"]},
    "BACCAR": {"-": ["13"]},
    "BACCATVS": {"-": ["21"]},
    "BACCHA": {"-": ["17"], "N": ["11"]},
    "BACCHABVNDVS": {"-": ["21"]},
    "BACCHANAL": {"-": ["13"]},
    "BACCHANALIA": {"-": ["16"], "N": ["11"]},
    "BACCHATIO": {"-": ["13"]},
    "BACCHOR": {"-": ["5J"]},
    "BACCHVS": {"N": ["12"]},
    "BACCIFER": {"-": ["21"]},
    "BACILLVM": {"-": ["12"]},
    "BACVLVM": {"-": ["12"]},
    "BACVLVS": {"-": ["12"], "N": ["12"]},
    "BAIAE": {"N": ["11"]},
    "BALAENA": {"-": ["11"]},
    "BALANATVS": {"-": ["21"]},
    "BALANTES": {"-": ["13"]},
    "BALANVS": {"-": ["12"]},
    "BALATRO": {"-": ["13"], "N": ["13"]},
    "BALATVS": {"-": ["14"]},
    "BALBE": {"-": ["60"]},
    "BALBVS": {"-": ["21"], "N": ["12"]},
    "BALBVTIO": {"-": ["54"]},
    "BALEARES": {"N": ["13"]},
    "BALINEAE": {"-": ["11"]},
    "BALINEVM": {"-": ["12"]},
    "BALLISTA": {"-": ["11"]},
    "BALNEARIA": {"-": ["12"]},
    "BALNEARIVS": {"-": ["21"]},
    "BALNEATOR": {"-": ["13"]},
    "BALNEOLVM": {"-": ["12"]},
    "BALNEVM": {"-": ["16"]},
    "BALO": {"-": ["51"]},
    "BALSAMVM": {"-": ["12"]},
    "BALTEVM": {"-": ["12"]},
    "BALTEVS": {"-": ["12"]},
    "BAPTIZO": {"-": ["51"]},
    "BARATHRVM": {"-": ["12"]},
    "BARBA": {"-": ["11"], "N": ["11"]},
    "BARBARE": {"-": ["60"]},
    "BARBARIA": {"-": ["11"]},
    "BARBARICVS": {"-": ["21"]},
    "BARBARISMVS": {"-": ["12"]},
    "BARBARVS": {"1": ["12"], "2": ["21"]},
    "BARBATVS": {"2": ["21"]},
    "BARBIGER": {"-": ["21"]},
    "BARBITON": {"-": ["17"]},
    "BARBITOS": {"-": ["17"]},
    "BARDITVS": {"-": ["14"]},
    "BARIOBAL": {"N": ["17"]},
    "BARIS": {"-": ["17"]},
    "BARO": {"-": ["13"]},
    "BARRVS": {"-": ["12"], "N": ["12"]},
    "BARVLCOS": {"-": ["27"]},
    "BARYCEPHALVS": {"-": ["21"]},
    "BASCAVDA": {"-": ["11"]},
    "BASIATIO": {"-": ["13"]},
    "BASILICA": {"-": ["11"]},
    "BASILISCVS": {"N": ["12"]},
    "BASIO": {"-": ["51"]},
    "BASIS": {"-": ["17"]},
    "BASIVM": {"-": ["12"]},
    "BAVBOR": {"-": ["5J"]},
    "BEATE": {"-": ["60"]},
    "BEATVLVS": {"-": ["21"]},
    "BEATVM": {"-": ["12"]},
    "BEATVS": {"-": ["21"]},
    "BELLATOR": {"-": ["13"]},
    "BELLATRIX": {"-": ["13"]},
    "BELLE": {"-": ["60"]},
    "BELLICOSVS": {"-": ["21"]},
    "BELLICVM": {"-": ["12"]},
    "BELLICVS": {"-": ["21"]},
    "BELLIGER": {"-": ["21"]},
    "BELLIGERO": {"-": ["51"]},
    "BELLO": {"-": ["51"]},
    "BELLONA": {"N": ["11"]},
    "BELLVA": {"-": ["11"]},
    "BELLVM": {"-": ["12"], "N": ["12"]},
    "BELLVOSVS": {"-": ["21"]},
    "BELLVS": {"-": ["21"], "N": ["12"]},
    "BELVA": {"-": ["11"]},
    "BENE": {"-": ["60"]},
    "BENEDICO": {"2": ["53"]},
    "BENEDICTVM": {"-": ["12"]},
    "BENEFACIO": {"-": ["55"]},
    "BENEFACTVM": {"-": ["12"]},
    "BENEFICENTIA": {"-": ["11"]},
    "BENEFICIARIVS": {"1": ["12"], "2": ["21"]},
    "BENEFICIVM": {"-": ["12"]},
    "BENEFICVS": {"-": ["21"]},
    "BENEVENTANVS": {"N": ["21"], "A": ["21"]},
    "BENEVENTVM": {"N": ["12"]},
    "BENEVOLE": {"-": ["60"]},
    "BENEVOLENS": {"-": ["25"]},
    "BENEVOLENTIA": {"-": ["11"]},
    "BENEVOLVS": {"-": ["21"]},
    "BENIGNE": {"-": ["60"]},
    "BENIGNITAS": {"-": ["13"]},
    "BENIGNVS": {"-": ["21"], "N": ["12"]},
    "BEO": {"-": ["51"]},
    "BERYLLVS": {"-": ["12"]},
    "BES": {"-": ["16"]},
    "BESTIA": {"-": ["11"], "N": ["11"]},
    "BESTIARIVS": {"1": ["12"], "2": ["21"]},
    "BESTIOLA": {"-": ["11"]},
    "BETA": {"1": ["11"], "2": ["16"]},
    "BIBERARIVS": {"-": ["12"]},
    "BIBLIOTHECA": {"-": ["11"]},
    "BIBO": {"1": ["13"], "2": ["53"]},
    "BIBVLVS": {"-": ["21"], "N": ["12"]},
    "BICEPS": {"-": ["25"]},
    "BICOLOR": {"-": ["25"]},
    "BICORNES": {"-": ["13"]},
    "BICORNIS": {"-": ["24"]},
    "BIDENS": {"1": ["13"], "2": ["25"]},
    "BIDENTAL": {"-": ["13"]},
    "BIDINI": {"N": ["12"]},
    "BIDINVS": {"N": ["21"], "A": ["21"]},
    "BIDIS": {"N": ["13"]},
    "BIDVVM": {"-": ["12"]},
    "BIENNIS": {"-": ["24"]},
    "BIENNIVM": {"-": ["12"]},
    "BIFARIVS": {"-": ["21"]},
    "BIFER": {"-": ["21"]},
    "BIFORIS": {"-": ["24"]},
    "BIFORMIS": {"-": ["24"]},
    "BIGA": {"-": ["11"]},
    "BIGATVS": {"-": ["21"]},
    "BIIVGI": {"-": ["12"]},
    "BIIVGIS": {"-": ["24"]},
    "BIIVGVS": {"-": ["21"]},
    "BILIBRIS": {"-": ["24"]},
    "BILINGVIS": {"-": ["24"]},
    "BILIS": {"-": ["13"]},
    "BILVSTRIS": {"-": ["24"]},
    "BIMARIS": {"-": ["24"]},
    "BIMEMBRES": {"-": ["13"], "N": ["13"]},
    "BIMEMBRIS": {"-": ["24"]},
    "BIMESTRIS": {"-": ["24"]},
    "BIMVLVS": {"-": ["21"]},
    "BIMVS": {"-": ["21"]},
    "BINOCTIVM": {"-": ["12"]},
    "BINVS": {"-": ["33"]},
    "BIPALIVM": {"-": ["12"]},
    "BIPARTIOR": {"-": ["5M"]},
    "BIPARTITIO": {"-": ["13"]},
    "BIPARTITO": {"-": ["60"]},
    "BIPATENS": {"-": ["25"]},
    "BIPEDALIS": {"-": ["24"]},
    "BIPENNIS": {"1": ["13"], "2": ["24"]},
    "BIPES": {"1": ["13"], "2": ["25"]},
    "BIREMIS": {"1": ["13"], "2": ["24"]},
    "BIS": {"-": ["36"]},
    "BISON": {"-": ["17"]},
    "BISVLCVS": {"-": ["21"]},
    "BITHYNIA": {"N": ["11"]},
    "BITVMEN": {"-": ["13"]},
    "BITVMINOSVS": {"-": ["21"]},
    "BIVGVS": {"-": ["21"]},
    "BIVIVM": {"-": ["12"]},
    "BLAESVS": {"-": ["21"], "N": ["12"]},
    "BLANDE": {"-": ["60"]},
    "BLANDILOQVENTIA": {"-": ["11"]},
    "BLANDILOQVVS": {"-": ["21"]},
    "BLANDIMENTVM": {"-": ["12"]},
    "BLANDIOR": {"-": ["5M"]},
    "BLANDITIA": {"-": ["11"], "N": ["11"]},
    "BLANDVS": {"-": ["21"], "N": ["12"]},
    "BLATERO": {"-": ["51"]},
    "BLATTA": {"-": ["11"]},
    "BLATTARIVS": {"-": ["21"]},
    "BOARIVS": {"-": ["21"], "A": ["21"]},
    "BOEOTIA": {"N": ["11"]},
    "BOEOTII": {"N": ["12"]},
    "BOETHVS": {"N": ["12"]},
    "BOLETATIO": {"-": ["13"]},
    "BOLETVS": {"-": ["12"]},
    "BOMBVS": {"-": ["12"]},
    "BOMBYCINVS": {"-": ["21"]},
    "BOMBYX": {"-": ["13"]},
    "BONI": {"-": ["12"]},
    "BONITAS": {"-": ["13"]},
    "BONVM": {"-": ["12"]},
    "BONVS": {"-": ["21"], "N": ["21"], "A": ["21"]},
    "BOO": {"-": ["51"]},
    "BOREAS": {"-": ["17"], "N": ["17"]},
    "BOS": {"-": ["13"], "N": ["13"]},
    "BOTVLARIVS": {"-": ["12"]},
    "BRACA": {"-": ["11"]},
    "BRACATVS": {"-": ["21"]},
    "BRACHIOLVM": {"-": ["12"]},
    "BRACHIVM": {"-": ["12"]},
    "BRACTEA": {"-": ["11"]},
    "BRACTEATVS": {"-": ["21"]},
    "BRACTEOLA": {"-": ["11"]},
    "BRASSICA": {"-": ["11"]},
    "BREVE": {"1": ["13"], "2": ["60"]},
    "BREVI": {"-": ["60"]},
    "BREVIARIVM": {"-": ["12"]},
    "BREVIS": {"-": ["24"]},
    "BREVITAS": {"-": ["13"]},
    "BREVITER": {"-": ["60"]},
    "BRITANNIA": {"N": ["11"]},
    "BRITANNVS": {"N": ["21"], "A": ["21"]},
    "BROCCHVS": {"N": ["12"]},
    "BRVMA": {"-": ["11"]},
    "BRVMALIS": {"-": ["24"]},
    "BRVNDISIVM": {"N": ["12"]},
    "BRVTESCO": {"-": ["53"]},
    "BRVTVS": {"-": ["21"], "N": ["12"]},
    "BVBILE": {"-": ["13"]},
    "BVBO": {"-": ["13"]},
    "BVBVLA": {"-": ["11"]},
    "BVBVLCVS": {"-": ["12"]},
    "BVBVLVS": {"-": ["21"]},
    "BVCCA": {"-": ["11"]},
    "BVCCVLA": {"-": ["11"]},
    "BVCERIVS": {"-": ["21"]},
    "BVCERVS": {"-": ["21"]},
    "BVCINA": {"-": ["11"]},
    "BVCINATOR": {"-": ["13"]},
    "BVCVLA": {"-": ["11"]},
    "BVFO": {"-": ["13"]},
    "BVLBVS": {"-": ["12"], "N": ["12"]},
    "BVLLA": {"-": ["11"]},
    "BVLLATVS": {"-": ["21"]},
    "BVLLIO": {"-": ["54"]},
    "BVLLITVS": {"-": ["14"]},
    "BVLLO": {"-": ["51"]},
    "BVMASTVS": {"-": ["12"]},
    "BVRIS": {"-": ["13"]},
    "BVRRVS": {"N": ["12"]},
    "BVSIRIS": {"N": ["13"]},
    "BVSTVARIVS": {"1": ["12"], "2": ["21"]},
    "BVSTVM": {"-": ["12"]},
    "BVTHROTVM": {"N": ["17"]},
    "BVXIFER": {"-": ["21"]},
    "BVXVM": {"-": ["12"]},
    "BVXVS": {"-": ["12"]},
    "BYZANTII": {"N": ["12"]},
    "BYZANTIVM": {"N": ["12"]},
    "CABALLVS": {"-": ["12"]},
    "CACHINNO": {"1": ["13"], "2": ["51"]},
    "CACHINNVS": {"-": ["12"]},
    "CACO": {"-": ["51"]},
    "CACOETHES": {"-": ["13"]},
    "CACVMEN": {"-": ["13"]},
    "CACVRIVS": {"N": ["12"]},
    "CACVS": {"N": ["12"]},
    "CADAVER": {"-": ["13"]},
    "CADMEA": {"N": ["11"]},
    "CADMVS": {"-": ["12"], "N": ["12"]},
    "CADO": {"-": ["53"]},
    "CADVCEATOR": {"-": ["13"]},
    "CADVCEVM": {"-": ["12"]},
    "CADVCEVS": {"-": ["12"]},
    "CADVCIFER": {"-": ["21"], "N": ["12"]},
    "CADVCVM": {"-": ["12"]},
    "CADVCVS": {"-": ["21"]},
    "CADVRCVM": {"-": ["12"]},
    "CADVS": {"-": ["12"]},
    "CAECIGENVS": {"-": ["21"]},
    "CAECILIVS": {"N": ["12"], "A": ["21"]},
    "CAECITAS": {"-": ["13"]},
    "CAECO": {"-": ["51"]},
    "CAECVS": {"-": ["21"], "N": ["12"]},
    "CAEDES": {"-": ["13"]},
    "CAEDICIVS": {"N": ["12"]},
    "CAEDO": {"-": ["53"]},
    "CAEDVVS": {"-": ["21"]},
    "CAELATOR": {"-": ["13"]},
    "CAELATVRA": {"-": ["11"]},
    "CAELEBS": {"-": ["25"]},
    "CAELES": {"1": ["13"], "2": ["25"], "N": ["13"]},
    "CAELESTES": {"-": ["13"]},
    "CAELESTIA": {"-": ["13"]},
    "CAELESTIS": {"-": ["24"]},
    "CAELI": {"-": ["12"]},
    "CAELIBATVS": {"-": ["14"]},
    "CAELICOLA": {"-": ["11"]},
    "CAELIFER": {"-": ["21"]},
    "CAELITES": {"-": ["13"]},
    "CAELIVS": {"N": ["12"]},
    "CAELO": {"-": ["51"]},
    "CAELVM": {"1": ["12"], "2": ["12"]},
    "CAEMENTICIVS": {"-": ["21"]},
    "CAEMENTVM": {"-": ["12"]},
    "CAENINENSES": {"N": ["13"]},
    "CAENINVS": {"N": ["21"], "A": ["21"]},
    "CAENO": {"N": ["13"]},
    "CAENOSVS": {"-": ["21"]},
    "CAENVM": {"-": ["12"]},
    "CAEPA": {"-": ["17"]},
    "CAEPIO": {"N": ["13"]},
    "CAERE": {"N": ["16"]},
    "CAERIMONIA": {"-": ["11"]},
    "CAERVLA": {"-": ["12"]},
    "CAERVLEVM": {"-": ["12"]},
    "CAERVLEVS": {"-": ["21"]},
    "CAESAR": {"N": ["13"]},
    "CAESARIES": {"-": ["15"]},
    "CAESETIVS": {"N": ["12"]},
    "CAESIVS": {"-": ["21"], "N": ["12"], "A": ["21"]},
    "CAESO": {"N": ["13"]},
    "CAESONIVS": {"N": ["12"]},
    "CAESPES": {"-": ["13"]},
    "CAESTVS": {"-": ["14"]},
    "CALACTE": {"N": ["17"]},
    "CALACTINI": {"N": ["12"]},
    "CALACTINVS": {"N": ["21"], "A": ["21"]},
    "CALAMISTER": {"-": ["12"]},
    "CALAMITAS": {"-": ["13"]},
    "CALAMITOSVS": {"-": ["21"]},
    "CALAMVS": {"-": ["12"]},
    "CALATHISCVS": {"-": ["12"]},
    "CALATHVS": {"-": ["12"]},
    "CALCAR": {"-": ["13"]},
    "CALCARIVS": {"1": ["12"], "2": ["21"]},
    "CALCATVRA": {"-": ["11"]},
    "CALCEAMENTVM": {"-": ["12"]},
    "CALCEO": {"-": ["51"]},
    "CALCEVS": {"-": ["12"]},
    "CALCO": {"-": ["51"]},
    "CALCVLOSVS": {"-": ["21"]},
    "CALCVLVS": {"-": ["12"]},
    "CALDA": {"-": ["11"]},
    "CALDARIVM": {"-": ["12"]},
    "CALDARIVS": {"-": ["21"]},
    "CALEFACIO": {"-": ["55"]},
    "CALEFACTO": {"-": ["51"]},
    "CALEFIO": {"-": ["56"]},
    "CALENVS": {"N": ["12"], "A": ["21"]},
    "CALEO": {"-": ["52"]},
    "CALESCO": {"-": ["53"]},
    "CALICVLVS": {"-": ["12"]},
    "CALIDA": {"-": ["11"]},
    "CALIDIANVS": {"N": ["21"], "A": ["21"]},
    "CALIDIVS": {"N": ["12"]},
    "CALIDVS": {"-": ["21"]},
    "CALIENDRVM": {"-": ["12"]},
    "CALIGA": {"-": ["11"]},
    "CALIGATVS": {"-": ["21"]},
    "CALIGINOSVS": {"-": ["21"]},
    "CALIGO": {"1": ["13"], "2": ["51"]},
    "CALIX": {"-": ["13"]},
    "CALLEO": {"-": ["52"]},
    "CALLIDAMA": {"N": ["17"]},
    "CALLIDE": {"-": ["60"]},
    "CALLIDITAS": {"-": ["13"]},
    "CALLIDVS": {"-": ["21"]},
    "CALLIS": {"-": ["13"]},
    "CALLISCO": {"-": ["53"]},
    "CALLOSVS": {"-": ["21"]},
    "CALLVM": {"-": ["12"]},
    "CALO": {"1": ["13"], "2": ["51"]},
    "CALOR": {"-": ["13"]},
    "CALPVRNIVS": {"N": ["12"], "A": ["21"]},
    "CALTHA": {"-": ["11"]},
    "CALVARIA": {"1": ["11"], "2": ["11"]},
    "CALVITIVM": {"-": ["12"]},
    "CALVMNIA": {"-": ["11"]},
    "CALVMNIATOR": {"-": ["13"]},
    "CALVMNIOR": {"-": ["5J"]},
    "CALVVS": {"-": ["21"], "N": ["12"]},
    "CALX": {"1": ["13"], "2": ["13"]},
    "CAMELVS": {"-": ["12"]},
    "CAMENA": {"-": ["11"], "N": ["11"]},
    "CAMENAE": {"N": ["11"]},
    "CAMERA": {"-": ["11"]},
    "CAMERIA": {"N": ["11"]},
    "CAMINVS": {"-": ["12"]},
    "CAMMARVS": {"-": ["12"]},
    "CAMPANIA": {"N": ["11"]},
    "CAMPESTER": {"-": ["23"]},
    "CAMPESTRE": {"-": ["13"]},
    "CAMPESTRIA": {"-": ["13"]},
    "CAMPVS": {"1": ["12"], "2": ["12"], "N": ["12"]},
    "CAMVRVS": {"-": ["21"]},
    "CANALICVLVS": {"-": ["12"]},
    "CANALIS": {"1": ["13"], "2": ["24"]},
    "CANCELLVS": {"-": ["12"]},
    "CANCER": {"-": ["16"], "N": ["12"]},
    "CANDELA": {"-": ["11"]},
    "CANDELABRVM": {"-": ["12"]},
    "CANDENS": {"-": ["25", "22"]},
    "CANDENTIA": {"-": ["11"]},
    "CANDEO": {"-": ["52"]},
    "CANDESCO": {"-": ["53"]},
    "CANDIDATVS": {"1": ["12"], "2": ["21"]},
    "CANDIDVLVS": {"-": ["21"]},
    "CANDIDVS": {"-": ["21"], "N": ["12"]},
    "CANDOR": {"-": ["13"]},
    "CANEO": {"-": ["52"]},
    "CANEPHORA": {"N": ["17"]},
    "CANESCO": {"-": ["53"]},
    "CANI": {"-": ["12"]},
    "CANICVLA": {"-": ["11"], "N": ["11"]},
    "CANIDIA": {"-": ["11"], "N": ["11"]},
    "CANINVS": {"-": ["21"], "N": ["12"]},
    "CANIS": {"-": ["13"], "N": ["13"]},
    "CANISTRA": {"-": ["12"]},
    "CANISTRVM": {"-": ["12"]},
    "CANITIES": {"-": ["15"]},
    "CANNA": {"-": ["11"]},
    "CANNABIS": {"-": ["16"]},
    "CANNENSIS": {"N": ["24"], "A": ["24"]},
    "CANNVLA": {"-": ["11"]},
    "CANO": {"-": ["53"]},
    "CANON": {"-": ["17"]},
    "CANONICVS": {"1": ["12"], "2": ["21"]},
    "CANOR": {"-": ["13"]},
    "CANORVS": {"-": ["21"]},
    "CANTABRI": {"-": ["12"], "N": ["12"]},
    "CANTERIVS": {"-": ["12"]},
    "CANTHARVS": {"-": ["12"]},
    "CANTHVS": {"-": ["12"]},
    "CANTICVM": {"-": ["12"]},
    "CANTILENA": {"-": ["11"]},
    "CANTIO": {"-": ["13"]},
    "CANTITO": {"-": ["51"]},
    "CANTO": {"-": ["51"]},
    "CANTOR": {"-": ["13"]},
    "CANTVS": {"-": ["14"]},
    "CANVLEIANVS": {"N": ["21"], "A": ["21"]},
    "CANVLEIVS": {"N": ["12"]},
    "CANVS": {"-": ["21"], "N": ["12"]},
    "CAPACITAS": {"-": ["13"]},
    "CAPAX": {"-": ["25"], "N": ["13"]},
    "CAPEDVNCVLA": {"-": ["11"]},
    "CAPELLA": {"-": ["11"], "N": ["11"]},
    "CAPENA": {"N": ["11"]},
    "CAPENAS": {"A": ["25"]},
    "CAPER": {"-": ["12"]},
    "CAPESSO": {"-": ["53"]},
    "CAPETVS": {"N": ["12"]},
    "CAPILLAMENTVM": {"-": ["12"]},
    "CAPILLATVS": {"-": ["21"]},
    "CAPILLVS": {"-": ["12"]},
    "CAPIO": {"1": ["13"], "2": ["55"]},
    "CAPISTRO": {"-": ["51"]},
    "CAPISTRVM": {"-": ["12"]},
    "CAPITAL": {"-": ["13"]},
    "CAPITALIS": {"-": ["24"]},
    "CAPITATIO": {"-": ["13"]},
    "CAPITATVS": {"-": ["21"]},
    "CAPITINVS": {"N": ["21"], "A": ["21"]},
    "CAPITO": {"-": ["13"], "N": ["13"]},
    "CAPITOLINVS": {"A": ["21"], "N": ["21", "12"]},
    "CAPITOLIVM": {"N": ["12"]},
    "CAPITVLVM": {"-": ["12"]},
    "CAPPADOCIA": {"N": ["11"]},
    "CAPRA": {"-": ["11"], "N": ["11"]},
    "CAPREA": {"-": ["11"]},
    "CAPREOLVS": {"-": ["12"]},
    "CAPRIFICVS": {"-": ["12"]},
    "CAPRIGENVS": {"-": ["21"]},
    "CAPRILE": {"-": ["13"]},
    "CAPRIMVLGVS": {"-": ["12"]},
    "CAPRINVS": {"-": ["21"]},
    "CAPRIPES": {"-": ["25"]},
    "CAPSA": {"-": ["11"], "N": ["11"]},
    "CAPSVLA": {"-": ["11"]},
    "CAPSVS": {"-": ["12"]},
    "CAPTATIO": {"-": ["13"]},
    "CAPTATOR": {"-": ["13"]},
    "CAPTIO": {"-": ["13"]},
    "CAPTIOSVS": {"-": ["21"]},
    "CAPTIVA": {"-": ["11"]},
    "CAPTIVITAS": {"-": ["13"]},
    "CAPTIVVS": {"1": ["12"], "2": ["21"]},
    "CAPTO": {"-": ["51"]},
    "CAPTVRA": {"-": ["11"]},
    "CAPTVS": {"-": ["14"], "A": ["21"]},
    "CAPVLATOR": {"-": ["13"]},
    "CAPVLVS": {"-": ["12"]},
    "CAPVT": {"-": ["13"]},
    "CAPYS": {"N": ["17"]},
    "CARBASA": {"-": ["12"]},
    "CARBASEVS": {"-": ["21"]},
    "CARBASVS": {"1": ["12"], "2": ["21"]},
    "CARBO": {"-": ["13"], "N": ["13"]},
    "CARBVNCVLVS": {"-": ["12"]},
    "CARCER": {"-": ["13"]},
    "CARCHESIVM": {"-": ["12"]},
    "CARCINOMA": {"-": ["17"]},
    "CARDIACVS": {"1": ["12"], "2": ["21"]},
    "CARDINALIS": {"-": ["24"]},
    "CARDINATVS": {"-": ["21"]},
    "CARDO": {"-": ["13"]},
    "CARDVVS": {"-": ["12"]},
    "CARE": {"-": ["60"]},
    "CARECTVM": {"-": ["12"]},
    "CAREO": {"-": ["52"]},
    "CAREX": {"-": ["13"]},
    "CARICA": {"-": ["11"]},
    "CARIES": {"-": ["15"]},
    "CARINA": {"-": ["11"], "N": ["11"]},
    "CARIOSVS": {"-": ["21"]},
    "CARITAS": {"-": ["13"]},
    "CARMEN": {"1": ["13"], "2": ["13"]},
    "CARMENTALIS": {"N": ["24"], "A": ["24"]},
    "CARMENTIS": {"N": ["16"]},
    "CARMO": {"-": ["13"], "N": ["13"]},
    "CARNARIVM": {"-": ["12"]},
    "CARNIFEX": {"-": ["13"]},
    "CARNIFICINA": {"-": ["11"]},
    "CARNIFICO": {"-": ["51"]},
    "CARO": {"1": ["13"], "2": ["53"]},
    "CARPATINA": {"-": ["11"]},
    "CARPENTVM": {"-": ["12"]},
    "CARPINATIVS": {"N": ["12"]},
    "CARPINVS": {"-": ["12"]},
    "CARPO": {"-": ["53"]},
    "CARPTIM": {"-": ["60"]},
    "CARPTOR": {"-": ["13"]},
    "CARRVM": {"-": ["12"]},
    "CARRVS": {"-": ["12"]},
    "CARTHAGINIENSES": {"N": ["13"]},
    "CARTHAGINIENSIS": {"N": ["24"], "A": ["24"]},
    "CARTHAGO": {"N": ["13"]},
    "CARVS": {"-": ["21"], "N": ["12"]},
    "CARYATIS": {"-": ["17"]},
    "CASA": {"-": ["11"]},
    "CASEVM": {"-": ["12"]},
    "CASEVS": {"-": ["12"]},
    "CASIA": {"-": ["11"]},
    "CASSIANVS": {"N": ["21"], "A": ["21"]},
    "CASSIDA": {"-": ["11"]},
    "CASSIS": {"1": ["13"], "2": ["13"]},
    "CASSIVS": {"A": ["21"], "N": ["12"]},
    "CASSVM": {"-": ["12"]},
    "CASSVS": {"-": ["21"]},
    "CASTANEA": {"-": ["11"]},
    "CASTANEVS": {"-": ["21"]},
    "CASTE": {"-": ["60"]},
    "CASTELLANI": {"-": ["12"]},
    "CASTELLANVS": {"1": ["12"], "2": ["21"]},
    "CASTELLARIVS": {"-": ["12"]},
    "CASTELLVM": {"-": ["12"]},
    "CASTIFICVS": {"-": ["21"]},
    "CASTIGATIO": {"-": ["13"]},
    "CASTIGATOR": {"-": ["13"]},
    "CASTIGATVS": {"-": ["21"]},
    "CASTIGO": {"-": ["51"]},
    "CASTIMONIA": {"-": ["11"]},
    "CASTITAS": {"-": ["13"]},
    "CASTOR": {"-": ["17"], "N": ["17"]},
    "CASTOREVM": {"-": ["12"]},
    "CASTOREVS": {"N": ["21"], "A": ["21"]},
    "CASTRA": {"1": ["11"], "2": ["12"], "N": ["11"]},
    "CASTRENSIS": {"-": ["24"]},
    "CASTRICIVS": {"N": ["12"]},
    "CASTRO": {"-": ["51"]},
    "CASTRVM": {"-": ["12"], "N": ["12"]},
    "CASTVS": {"1": ["14"], "2": ["21"]},
    "CASV": {"-": ["60"]},
    "CASVLA": {"-": ["11"]},
    "CASVS": {"-": ["14"]},
    "CATAGRAPHVS": {"-": ["21"]},
    "CATAPHRACTVS": {"-": ["12"]},
    "CATAPLASMA": {"-": ["17"]},
    "CATAPLEXIS": {"-": ["13"]},
    "CATAPVLTA": {"-": ["11"]},
    "CATARACTA": {"-": ["11"]},
    "CATASCOPVS": {"-": ["12"]},
    "CATASTA": {"-": ["11"]},
    "CATATEXITECHNOS": {"-": ["27"]},
    "CATATONVS": {"-": ["21"]},
    "CATECHON": {"-": ["57"]},
    "CATELLA": {"-": ["11"], "2": ["11"], "1": ["12"]},
    "CATELLVS": {"-": ["12"]},
    "CATENA": {"-": ["11"]},
    "CATENARIVS": {"-": ["21"]},
    "CATENATIO": {"-": ["13"]},
    "CATENO": {"-": ["51"]},
    "CATERVA": {"-": ["11"]},
    "CATERVATIM": {"-": ["60"]},
    "CATHEDRA": {"-": ["11"]},
    "CATHEDRARIVS": {"-": ["21"]},
    "CATHETVS": {"-": ["17"]},
    "CATILINA": {"N": ["11"]},
    "CATILLVS": {"-": ["12"], "N": ["12"]},
    "CATINA": {"N": ["11"]},
    "CATINENSES": {"N": ["13"]},
    "CATINENSIS": {"N": ["24"], "A": ["24"]},
    "CATINVS": {"-": ["12"]},
    "CATLASTER": {"-": ["12"]},
    "CATO": {"N": ["13"]},
    "CATVLVS": {"-": ["12"], "N": ["12"]},
    "CATVS": {"-": ["21"], "N": ["12"]},
    "CAVATVS": {"-": ["21"]},
    "CAVDA": {"-": ["11"]},
    "CAVDEX": {"-": ["13"], "N": ["13"]},
    "CAVDICARIVS": {"-": ["21"]},
    "CAVEA": {"-": ["11"]},
    "CAVEO": {"-": ["52"]},
    "CAVERNA": {"-": ["11"]},
    "CAVILLATIO": {"-": ["13"]},
    "CAVILLATOR": {"-": ["13"]},
    "CAVILLOR": {"-": ["5J"]},
    "CAVLA": {"-": ["11"]},
    "CAVLICVLVS": {"-": ["12"]},
    "CAVLIS": {"-": ["13"]},
    "CAVO": {"1": ["51"], "2": ["53"]},
    "CAVPO": {"-": ["13"]},
    "CAVPONA": {"-": ["11"]},
    "CAVPONOR": {"-": ["5J"]},
    "CAVRVS": {"-": ["12"], "N": ["12"]},
    "CAVSA": {"-": ["11"]},
    "CAVSARIVS": {"-": ["21"]},
    "CAVSIDICVS": {"-": ["12"]},
    "CAVSOR": {"-": ["5J"]},
    "CAVSVLA": {"-": ["11"]},
    "CAVTE": {"-": ["60"]},
    "CAVTES": {"-": ["13"]},
    "CAVTIO": {"-": ["13"]},
    "CAVTVS": {"-": ["21"]},
    "CAVVM": {"-": ["12"]},
    "CAVVS": {"1": ["12"], "2": ["21"]},
    "CEDO": {"1": ["53"], "2": ["90"]},
    "CEDRIVM": {"-": ["12"]},
    "CEDRVS": {"-": ["12"]},
    "CELATVM": {"-": ["12"]},
    "CELEBER": {"-": ["23"]},
    "CELEBRATIO": {"-": ["13"]},
    "CELEBRATVS": {"-": ["21"]},
    "CELEBRITAS": {"-": ["13"]},
    "CELEBRITER": {"-": ["60"]},
    "CELEBRO": {"-": ["51"]},
    "CELER": {"-": ["23"], "N": ["12"]},
    "CELERIPES": {"-": ["25"]},
    "CELERITAS": {"-": ["13"]},
    "CELERITER": {"-": ["60"]},
    "CELERO": {"-": ["51"]},
    "CELLA": {"-": ["11"]},
    "CELLARIVS": {"1": ["12"], "2": ["21"]},
    "CELLO": {"-": ["53"]},
    "CELLVLA": {"-": ["11"]},
    "CELO": {"-": ["51"]},
    "CELSVS": {"-": ["21"], "N": ["12"]},
    "CENA": {"-": ["11"]},
    "CENACVLVM": {"-": ["12"]},
    "CENATIO": {"-": ["13"]},
    "CENDO": {"-": ["53"]},
    "CENO": {"-": ["51"]},
    "CENSEO": {"-": ["52"]},
    "CENSOR": {"-": ["13"]},
    "CENSORIVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "CENSVRA": {"-": ["11"]},
    "CENSVS": {"-": ["14"]},
    "CENTAVREVM": {"-": ["12"]},
    "CENTAVRVS": {"-": ["12"], "N": ["12"]},
    "CENTENARIVS": {"-": ["21"]},
    "CENTENVS": {"-": ["33"]},
    "CENTESIMA": {"-": ["11"]},
    "CENTESIMVS": {"-": ["32"]},
    "CENTICEPS": {"-": ["25"]},
    "CENTIES": {"-": ["36"]},
    "CENTIMANVS": {"-": ["21"]},
    "CENTO": {"-": ["13"], "N": ["13"]},
    "CENTRVM": {"-": ["12"]},
    "CENTVM": {"-": ["31"]},
    "CENTVMGEMINVS": {"-": ["21"]},
    "CENTVMPONDIVM": {"-": ["12"]},
    "CENTVMVIR": {"-": ["12"]},
    "CENTVMVIRALIS": {"-": ["24"]},
    "CENTVNCVLVS": {"-": ["12"]},
    "CENTVRIA": {"-": ["11"]},
    "CENTVRIATIM": {"-": ["60"]},
    "CENTVRIATVS": {"1": ["14"], "2": ["21"]},
    "CENTVRIO": {"1": ["13"], "2": ["51"]},
    "CENTVRIONATVS": {"-": ["14"]},
    "CENTVRIPINI": {"N": ["12"]},
    "CENTVRIPINVS": {"N": ["21"], "A": ["21"]},
    "CENTVSSIS": {"-": ["13"]},
    "CENVLA": {"-": ["11"]},
    "CEPHALOEDITANI": {"N": ["12"]},
    "CEPHALOEDITANVS": {"N": ["21"], "A": ["21"]},
    "CEPHALOEDIVM": {"N": ["12"]},
    "CEPTO": {"-": ["51"]},
    "CERA": {"-": ["11"]},
    "CERARIVM": {"-": ["12"]},
    "CERASTES": {"-": ["17"]},
    "CERASVM": {"-": ["12"]},
    "CERASVS": {"-": ["12"]},
    "CERBERVS": {"N": ["12"]},
    "CERCEI": {"N": ["12"]},
    "CERCOPITHECVS": {"-": ["12"]},
    "CERDO": {"-": ["13"], "N": ["13"]},
    "CEREALIS": {"-": ["24"], "A": ["24"]},
    "CEREBROSVS": {"-": ["21"]},
    "CEREBRVM": {"-": ["12"]},
    "CERES": {"N": ["13"], "-": ["23"]},
    "CEREVS": {"1": ["12"], "2": ["21"]},
    "CERINTHA": {"-": ["17"]},
    "CERNO": {"-": ["53"]},
    "CERNVLO": {"-": ["51"]},
    "CERO": {"-": ["51"]},
    "CEROMA": {"-": ["13"]},
    "CEROMATICVS": {"-": ["21"]},
    "CERPO": {"-": ["53"]},
    "CERRITVS": {"-": ["21"]},
    "CERRVS": {"-": ["12"]},
    "CERTAMEN": {"-": ["13"]},
    "CERTATIM": {"-": ["60"]},
    "CERTATIO": {"-": ["13"]},
    "CERTE": {"-": ["60"]},
    "CERTO": {"1": ["51"], "2": ["60"]},
    "CERTVM": {"1": ["12"], "2": ["60"]},
    "CERTVS": {"-": ["21"], "N": ["12"]},
    "CERVA": {"-": ["11"]},
    "CERVICAL": {"-": ["13"]},
    "CERVICVLA": {"-": ["11"]},
    "CERVINVS": {"-": ["21"]},
    "CERVIVS": {"N": ["12"]},
    "CERVIX": {"-": ["13"]},
    "CERVSSA": {"-": ["11"]},
    "CERVVS": {"-": ["12"]},
    "CERYX": {"-": ["13"]},
    "CESSATOR": {"-": ["13"]},
    "CESSIM": {"-": ["60"]},
    "CESSO": {"-": ["51"]},
    "CETARINI": {"N": ["12"]},
    "CETARIVM": {"-": ["12"]},
    "CETARIVS": {"1": ["12"], "2": ["21"], "-": ["21"]},
    "CETERA": {"1": ["12"], "2": ["60"]},
    "CETERI": {"-": ["12"]},
    "CETERVM": {"1": ["12"], "2": ["60"]},
    "CETERVS": {"-": ["21"]},
    "CETOS": {"-": ["17"]},
    "CETRA": {"-": ["11"]},
    "CETRATI": {"-": ["12"]},
    "CETRATVS": {"-": ["21"]},
    "CETVS": {"-": ["12"]},
    "CEV": {"1": ["66"], "2": ["82"]},
    "CEVEO": {"-": ["52"]},
    "CHAERE": {"-": ["90"]},
    "CHALCIDICVM": {"-": ["12"]},
    "CHALYBS": {"-": ["13"]},
    "CHAOS": {"-": ["17"], "N": ["12"]},
    "CHARA": {"-": ["11"]},
    "CHARACTERISMVS": {"-": ["17"]},
    "CHARIDEMVS": {"N": ["12"]},
    "CHARINVS": {"N": ["12"]},
    "CHARIS": {"-": ["17"], "N": ["16"]},
    "CHARTA": {"-": ["11"]},
    "CHARYBDIS": {"N": ["17"]},
    "CHEIROTOMETOS": {"-": ["27"]},
    "CHELE": {"-": ["17"], "N": ["17"]},
    "CHELIDON": {"-": ["17"], "N": ["17"]},
    "CHELONIVM": {"-": ["12"]},
    "CHELYDRVS": {"-": ["12"]},
    "CHELYS": {"-": ["17"]},
    "CHERAGRA": {"-": ["11"]},
    "CHERSONESVS": {"N": ["12"]},
    "CHILIARCHES": {"-": ["11"]},
    "CHILO": {"N": ["13"]},
    "CHIRAGRA": {"-": ["11"]},
    "CHIROGRAPHVM": {"-": ["12"]},
    "CHIRONOMON": {"-": ["17"]},
    "CHIVS": {"A": ["21"], "N": ["17"]},
    "CHLAMYDATVS": {"-": ["21"]},
    "CHLAMYS": {"-": ["17"]},
    "CHLORVS": {"N": ["12"]},
    "CHOO": {"-": ["51"]},
    "CHORAGIVM": {"-": ["12"]},
    "CHORAVLES": {"-": ["17"]},
    "CHORDA": {"-": ["11"]},
    "CHORDVS": {"-": ["21"]},
    "CHOREA": {"-": ["11"]},
    "CHOROBATES": {"-": ["17"]},
    "CHOROGRAPHIA": {"-": ["11"]},
    "CHORVS": {"-": ["12"]},
    "CHRIA": {"-": ["17"]},
    "CHRISTIANVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "CHROMA": {"-": ["17"]},
    "CHROMATICE": {"-": ["17"]},
    "CHROMATICOS": {"-": ["27"]},
    "CHRYSAS": {"N": ["17"]},
    "CHRYSOCOLLA": {"-": ["11"]},
    "CHRYSOGONVS": {"N": ["12"]},
    "CHRYSOLITHVS": {"-": ["12"]},
    "CIBARIVM": {"-": ["12"]},
    "CIBARIVS": {"-": ["21"]},
    "CIBATVS": {"-": ["14"]},
    "CIBORIVM": {"-": ["12"]},
    "CIBVS": {"-": ["12"]},
    "CIBYRA": {"N": ["11"]},
    "CIBYRATA": {"N": ["11"]},
    "CIBYRATICVS": {"N": ["21"], "A": ["21"]},
    "CICADA": {"-": ["11"]},
    "CICATRICOSVS": {"-": ["21"]},
    "CICATRIX": {"-": ["13"]},
    "CICER": {"-": ["13"]},
    "CICERO": {"N": ["13"]},
    "CICHORIVM": {"-": ["12"]},
    "CICONIA": {"-": ["11"]},
    "CICVR": {"-": ["25"]},
    "CICVTA": {"-": ["11"], "N": ["11"]},
    "CIDARIS": {"-": ["13"]},
    "CIDO": {"1": ["53"], "2": ["53"]},
    "CIEO": {"-": ["52"]},
    "CILICES": {"N": ["13"]},
    "CILICIA": {"N": ["11"]},
    "CILICIVM": {"-": ["12"]},
    "CILIO": {"-": ["51"]},
    "CIMEX": {"-": ["13"]},
    "CINAEDVS": {"1": ["12"], "2": ["21"]},
    "CINCINNVS": {"-": ["12"]},
    "CINEFACTVS": {"-": ["21"]},
    "CINERARIVS": {"1": ["12"], "2": ["21"]},
    "CINESIS": {"-": ["17"]},
    "CINGO": {"-": ["53"]},
    "CINGVLA": {"-": ["11"]},
    "CINGVLVM": {"-": ["12"], "N": ["12"]},
    "CINIFLO": {"-": ["13"]},
    "CINIS": {"-": ["13"]},
    "CINNA": {"N": ["11"]},
    "CINNAMVM": {"-": ["17"]},
    "CINNO": {"-": ["51"]},
    "CINNVS": {"-": ["12"]},
    "CINO": {"-": ["53"]},
    "CIO": {"-": ["54"]},
    "CIPIO": {"-": ["55"]},
    "CIPPVS": {"-": ["12"]},
    "CIRCA": {"1": ["60"], "2": ["70"]},
    "CIRCE": {"N": ["17"]},
    "CIRCENSES": {"-": ["13"]},
    "CIRCENSIS": {"-": ["24"], "A": ["24"]},
    "CIRCINATIO": {"-": ["13"]},
    "CIRCINO": {"-": ["51"]},
    "CIRCINVS": {"-": ["12"]},
    "CIRCITER": {"1": ["60"], "2": ["70"]},
    "CIRCITO": {"1": ["5J", "51"]},
    "CIRCVITIO": {"-": ["13"]},
    "CIRCVITVS": {"-": ["14"]},
    "CIRCVLARIS": {"-": ["24"]},
    "CIRCVLATOR": {"-": ["13"]},
    "CIRCVLOR": {"-": ["5J"]},
    "CIRCVLVS": {"-": ["12"]},
    "CIRCVM": {"1": ["60"], "2": ["70"]},
    "CIRCVMACTIO": {"-": ["13"]},
    "CIRCVMACTVS": {"-": ["14"]},
    "CIRCVMAGO": {"-": ["53"]},
    "CIRCVMAMPLECTOR": {"-": ["5L"]},
    "CIRCVMCAESVRA": {"-": ["11"]},
    "CIRCVMCIDANEVS": {"-": ["21"]},
    "CIRCVMCIDO": {"2": ["53"], "1": ["53"]},
    "CIRCVMCINGO": {"-": ["53"]},
    "CIRCVMCIRCA": {"-": ["60"]},
    "CIRCVMCISVS": {"C": ["21"], "-": ["21"]},
    "CIRCVMCLVDO": {"-": ["53"]},
    "CIRCVMCOLO": {"2": ["53"]},
    "CIRCVMCVRRO": {"-": ["53"]},
    "CIRCVMCVRSO": {"-": ["51"]},
    "CIRCVMDO": {"-": ["51"]},
    "CIRCVMDVCO": {"-": ["53"]},
    "CIRCVMDVCTIO": {"-": ["13"]},
    "CIRCVMEO": {"1": ["56"]},
    "CIRCVMFERO": {"-": ["56"]},
    "CIRCVMFIGO": {"-": ["53"]},
    "CIRCVMFLECTO": {"-": ["53"]},
    "CIRCVMFLO": {"-": ["51"]},
    "CIRCVMFLVO": {"-": ["53"]},
    "CIRCVMFLVVS": {"-": ["21"]},
    "CIRCVMFODIO": {"-": ["55"]},
    "CIRCVMFREMO": {"-": ["53"]},
    "CIRCVMFRICO": {"-": ["51"]},
    "CIRCVMFVNDO": {"2": ["53"]},
    "CIRCVMGEMO": {"-": ["53"]},
    "CIRCVMGREDIOR": {"-": ["5N"]},
    "CIRCVMIACEO": {"-": ["52"]},
    "CIRCVMICIO": {"-": ["55"]},
    "CIRCVMLATIO": {"-": ["13"]},
    "CIRCVMLATRO": {"-": ["51"]},
    "CIRCVMLEGO": {"2": ["53"]},
    "CIRCVMLEVIGO": {"-": ["51"]},
    "CIRCVMLEVO": {"1": ["51"], "-": ["51"]},
    "CIRCVMLIGO": {"2": ["51"]},
    "CIRCVMLINO": {"-": ["53"]},
    "CIRCVMLITIO": {"-": ["13"]},
    "CIRCVMLOQVOR": {"-": ["5L"]},
    "CIRCVMLVCENS": {"-": ["25"]},
    "CIRCVMMETIO": {"-": ["54"]},
    "CIRCVMMITTO": {"-": ["53"]},
    "CIRCVMMVGIO": {"-": ["54"]},
    "CIRCVMMVNIO": {"2": ["54"]},
    "CIRCVMMVNITIO": {"-": ["13"]},
    "CIRCVMPENDEO": {"-": ["52"]},
    "CIRCVMPLECTO": {"-": ["53"]},
    "CIRCVMPLECTOR": {"-": ["5L"]},
    "CIRCVMPLVMBO": {"-": ["51"]},
    "CIRCVMPONO": {"-": ["53"]},
    "CIRCVMQVAQVE": {"-": ["60"]},
    "CIRCVMRETIO": {"-": ["54"]},
    "CIRCVMRODO": {"-": ["53"]},
    "CIRCVMSAEPIO": {"-": ["54"]},
    "CIRCVMSCINDO": {"-": ["53"]},
    "CIRCVMSCRIBO": {"-": ["53"]},
    "CIRCVMSCRIPTIO": {"-": ["13"]},
    "CIRCVMSCRIPTOR": {"-": ["13"]},
    "CIRCVMSECO": {"-": ["51"]},
    "CIRCVMSEDEO": {"-": ["52"]},
    "CIRCVMSERO": {"3": ["53"]},
    "CIRCVMSESSIO": {"-": ["13"]},
    "CIRCVMSIDO": {"-": ["53"]},
    "CIRCVMSILIO": {"-": ["54"]},
    "CIRCVMSISTO": {"-": ["53"]},
    "CIRCVMSONO": {"1": ["51"]},
    "CIRCVMSONVS": {"-": ["21"]},
    "CIRCVMSPECTE": {"-": ["6&"]},
    "CIRCVMSPECTO": {"-": ["51"]},
    "CIRCVMSPECTVS": {"1": ["14"], "2": ["21"]},
    "CIRCVMSPICIO": {"-": ["55"]},
    "CIRCVMSTO": {"-": ["51"]},
    "CIRCVMSTREPO": {"-": ["53"]},
    "CIRCVMTENDO": {"-": ["53"]},
    "CIRCVMTENEO": {"-": ["52"]},
    "CIRCVMTERGEO": {"-": ["52"]},
    "CIRCVMTERO": {"-": ["53"]},
    "CIRCVMTEXO": {"-": ["53"]},
    "CIRCVMTONDEO": {"-": ["52"]},
    "CIRCVMTONO": {"-": ["51"]},
    "CIRCVMVADO": {"2": ["53"]},
    "CIRCVMVAGOR": {"2": ["5J"], "-": ["5J"]},
    "CIRCVMVAGVS": {"-": ["21"]},
    "CIRCVMVALLO": {"-": ["51"]},
    "CIRCVMVECTO": {"-": ["51"]},
    "CIRCVMVECTOR": {"-": ["5J"]},
    "CIRCVMVEHOR": {"-": ["5L"]},
    "CIRCVMVENIO": {"-": ["54"]},
    "CIRCVMVERRO": {"-": ["53"]},
    "CIRCVMVERSO": {"-": ["51"]},
    "CIRCVMVERTO": {"-": ["53"]},
    "CIRCVMVOLITO": {"-": ["51"]},
    "CIRCVMVOLO": {"2": ["51"]},
    "CIRCVMVOLVO": {"-": ["53"]},
    "CIRCVS": {"-": ["12"], "N": ["12"]},
    "CIRRATI": {"-": ["12"]},
    "CIRRVS": {"-": ["12"]},
    "CIS": {"-": ["70"]},
    "CISIVM": {"-": ["12"]},
    "CISTA": {"-": ["11"]},
    "CISTERNA": {"-": ["11"]},
    "CISTERNINVS": {"-": ["21"]},
    "CITATIM": {"-": ["60"]},
    "CITATVS": {"-": ["21"]},
    "CITE": {"-": ["6&"]},
    "CITER": {"-": ["21"]},
    "CITERIA": {"-": ["11"]},
    "CITERIOR": {"-": ["2&"], "A": ["2&"]},
    "CITHARA": {"-": ["11"]},
    "CITHARISTA": {"-": ["11"]},
    "CITHAROEDVS": {"-": ["12"]},
    "CITO": {"1": ["51"], "2": ["60"]},
    "CITRA": {"1": ["60"], "2": ["70"]},
    "CITREVS": {"-": ["21"]},
    "CITRO": {"-": ["60"]},
    "CITRVM": {"-": ["12"]},
    "CITRVS": {"-": ["12"]},
    "CITVS": {"-": ["21"]},
    "CIVICA": {"-": ["11"], "N": ["11"]},
    "CIVICVS": {"-": ["21"]},
    "CIVILIS": {"-": ["24"], "N": ["13"]},
    "CIVILITER": {"-": ["60"]},
    "CIVIS": {"-": ["13"]},
    "CIVITAS": {"-": ["13"]},
    "CLADES": {"-": ["13"]},
    "CLAM": {"1": ["60"], "2": ["70"]},
    "CLAMITO": {"-": ["51"]},
    "CLAMO": {"-": ["51"]},
    "CLAMOR": {"-": ["13"]},
    "CLAMOSVS": {"-": ["21"]},
    "CLANCVLVM": {"-": ["60"]},
    "CLANDESTINVS": {"-": ["21"]},
    "CLANGOR": {"-": ["13"]},
    "CLARE": {"-": ["60"]},
    "CLAREO": {"-": ["52"]},
    "CLARESCO": {"-": ["53"]},
    "CLARICITO": {"-": ["51"]},
    "CLARIGO": {"-": ["51"]},
    "CLARISONVS": {"-": ["21"]},
    "CLARITAS": {"-": ["13"]},
    "CLARITVDO": {"-": ["13"]},
    "CLARO": {"-": ["51"]},
    "CLARVS": {"-": ["21"], "N": ["12"]},
    "CLASSIARII": {"-": ["12"]},
    "CLASSIARIVS": {"-": ["21"]},
    "CLASSICI": {"-": ["12"]},
    "CLASSICVM": {"-": ["12"]},
    "CLASSICVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "CLASSIS": {"-": ["13"]},
    "CLATHRATVS": {"-": ["21"]},
    "CLATHRI": {"-": ["12"]},
    "CLAVA": {"-": ["11"]},
    "CLAVDICO": {"-": ["51"]},
    "CLAVDIVS": {"A": ["21"], "N": ["12"]},
    "CLAVDO": {"1": ["53"], "2": ["53"]},
    "CLAVDVS": {"-": ["21"]},
    "CLAVICVLA": {"-": ["11"]},
    "CLAVIS": {"-": ["13"]},
    "CLAVSTRVM": {"-": ["12"]},
    "CLAVSVLA": {"-": ["11"]},
    "CLAVSVM": {"-": ["12"]},
    "CLAVSVS": {"N": ["12"]},
    "CLAVVLVS": {"-": ["12"]},
    "CLAVVS": {"-": ["12"]},
    "CLEMENS": {"-": ["25"], "N": ["13"]},
    "CLEMENTER": {"-": ["60"]},
    "CLEMENTIA": {"-": ["11"]},
    "CLEOMENES": {"N": ["17"]},
    "CLEPO": {"-": ["53"]},
    "CLEPSYDRA": {"-": ["11"]},
    "CLIENS": {"-": ["13"]},
    "CLIENTA": {"-": ["11"]},
    "CLIENTELA": {"-": ["11"]},
    "CLIENTVLVS": {"-": ["12"]},
    "CLIMA": {"-": ["17"]},
    "CLIMACHIAS": {"N": ["17"]},
    "CLIMACIS": {"-": ["17"]},
    "CLINAMEN": {"-": ["13"]},
    "CLINO": {"-": ["51"]},
    "CLIPEATVS": {"-": ["12"], "2": ["12"], "1": ["12"]},
    "CLIPEO": {"-": ["51"]},
    "CLIPEVS": {"-": ["12"]},
    "CLITELLAE": {"-": ["11"]},
    "CLITELLARIVS": {"-": ["21"]},
    "CLIVA": {"-": ["12"]},
    "CLIVOSVS": {"-": ["21"]},
    "CLIVVS": {"-": ["12"]},
    "CLOACA": {"-": ["11"]},
    "CLODIVS": {"N": ["12"], "A": ["21"]},
    "CLOELIA": {"N": ["11"]},
    "CLOELIVS": {"A": ["21"], "N": ["12"]},
    "CLVDO": {"-": ["53"]},
    "CLVEO": {"-": ["52"]},
    "CLVNIS": {"-": ["13"]},
    "CLVPEA": {"N": ["16"]},
    "CLVSINVS": {"N": ["21"], "A": ["21"]},
    "CNEMIS": {"-": ["17"]},
    "CNIDII": {"N": ["12"]},
    "CNODAX": {"-": ["17"]},
    "COACERVATIO": {"-": ["13"]},
    "COACERVO": {"-": ["51"]},
    "COACTA": {"-": ["12"]},
    "COACTO": {"-": ["51"]},
    "COACTOR": {"-": ["13"]},
    "COACTVS": {"-": ["14"]},
    "COADDO": {"-": ["53"]},
    "COAEDIFICO": {"-": ["51"]},
    "COAEQVO": {"-": ["51"]},
    "COAGMENTATIO": {"-": ["13"]},
    "COAGMENTO": {"-": ["51"]},
    "COAGMENTVM": {"-": ["12"]},
    "COAGVLO": {"-": ["51"]},
    "COAGVLVM": {"-": ["12"]},
    "COALESCO": {"-": ["53"]},
    "COANGVSTO": {"-": ["51"]},
    "COARCTATIO": {"-": ["13"]},
    "COARCTO": {"-": ["51"]},
    "COAREO": {"-": ["52"]},
    "COARGVO": {"-": ["53"]},
    "COARTATIO": {"-": ["13"]},
    "COARTO": {"-": ["51"]},
    "COAXATIO": {"-": ["13"]},
    "COAXO": {"-": ["51"]},
    "COCCEIVS": {"N": ["12"]},
    "COCCINVS": {"-": ["21"]},
    "COCCVM": {"-": ["12"]},
    "COCHLEA": {"-": ["11"]},
    "COCLES": {"N": ["13"]},
    "COCTILIS": {"-": ["24"]},
    "COCTIO": {"-": ["13"]},
    "COCTVRA": {"-": ["11"]},
    "COCTVRNIX": {"-": ["13"]},
    "COCVLVM": {"-": ["12"]},
    "CODEX": {"-": ["13"]},
    "CODICILLVS": {"-": ["12"]},
    "COELIA": {"-": ["17"]},
    "COELIACVS": {"-": ["21"]},
    "COELIVS": {"N": ["12"]},
    "COEMO": {"-": ["53"]},
    "COEO": {"1": ["56"]},
    "COEPIO": {"-": ["55"]},
    "COEPTO": {"-": ["51"]},
    "COEPTVM": {"-": ["12"]},
    "COEPTVS": {"-": ["14"]},
    "COERCEO": {"-": ["52"]},
    "COERCITIO": {"-": ["13"]},
    "COETVS": {"-": ["14"]},
    "COGITABILIS": {"-": ["24"]},
    "COGITATIO": {"-": ["13"]},
    "COGITATVM": {"-": ["12"]},
    "COGITO": {"-": ["51"]},
    "COGNATIO": {"-": ["13"]},
    "COGNATVS": {"1": ["12"], "2": ["21"]},
    "COGNITIO": {"-": ["13"]},
    "COGNITOR": {"-": ["13"]},
    "COGNITVS": {"1": ["14"], "2": ["21"]},
    "COGNOBILIS": {"-": ["24"]},
    "COGNOMEN": {"-": ["13"]},
    "COGNOMENTVM": {"-": ["12"]},
    "COGNOSCO": {"-": ["53"]},
    "COGO": {"-": ["53"]},
    "COHAERENTIA": {"-": ["11"]},
    "COHAEREO": {"-": ["52"]},
    "COHAERESCO": {"-": ["53"]},
    "COHERES": {"-": ["13"]},
    "COHIBEO": {"-": ["52"]},
    "COHONESTO": {"-": ["51"]},
    "COHORS": {"-": ["13"]},
    "COHORTATIO": {"-": ["13"]},
    "COHORTOR": {"-": ["5J"]},
    "COI": {"N": ["12"]},
    "COICIO": {"-": ["55"]},
    "COINQVINO": {"-": ["51"]},
    "COITVS": {"-": ["14"]},
    "COLAPHVS": {"-": ["12"], "N": ["12"]},
    "COLLABEFACIO": {"-": ["55"]},
    "COLLABEFIO": {"-": ["56"]},
    "COLLABOR": {"2": ["5L"]},
    "COLLACTEVS": {"-": ["21"]},
    "COLLAEVO": {"-": ["51"]},
    "COLLATIA": {"N": ["11"]},
    "COLLATICIVS": {"-": ["21"]},
    "COLLATINI": {"N": ["12"]},
    "COLLATINVS": {"A": ["21"], "N": ["12"]},
    "COLLATIO": {"-": ["13"]},
    "COLLATRO": {"-": ["51"]},
    "COLLATVS": {"-": ["14"]},
    "COLLAVDO": {"-": ["51"]},
    "COLLAXO": {"-": ["51"]},
    "COLLECTIO": {"-": ["13"]},
    "COLLECTVS": {"1": ["14"], "2": ["21"]},
    "COLLEGA": {"-": ["11"], "N": ["11"]},
    "COLLEGIVM": {"-": ["12"]},
    "COLLEVO": {"2": ["51"]},
    "COLLIBERTVS": {"-": ["12"]},
    "COLLIBET": {"-": ["52"]},
    "COLLIBRO": {"-": ["51"]},
    "COLLICIAE": {"-": ["11"]},
    "COLLICIARIS": {"-": ["24"]},
    "COLLIDO": {"-": ["53"]},
    "COLLIGO": {"2": ["51"], "3": ["53"]},
    "COLLINO": {"-": ["53"]},
    "COLLINVS": {"N": ["21"], "A": ["21"]},
    "COLLIQVIARIA": {"-": ["12"]},
    "COLLIS": {"-": ["13"]},
    "COLLOCATIO": {"-": ["13"]},
    "COLLOCO": {"-": ["51"]},
    "COLLOQVIVM": {"-": ["12"]},
    "COLLOQVOR": {"-": ["5L"]},
    "COLLVCEO": {"-": ["52"]},
    "COLLVCO": {"-": ["51"]},
    "COLLVCTATIO": {"-": ["13"]},
    "COLLVCTOR": {"-": ["5J"]},
    "COLLVDO": {"-": ["53"]},
    "COLLVM": {"-": ["12"]},
    "COLLVO": {"-": ["53"]},
    "COLLVS": {"-": ["12"]},
    "COLLVSIO": {"-": ["13"]},
    "COLLVSOR": {"-": ["13"]},
    "COLLVSTRO": {"2": ["51"]},
    "COLLVVIES": {"-": ["15"]},
    "COLLVVIO": {"-": ["13"]},
    "COLLYBVS": {"-": ["12"]},
    "COLLYRIVM": {"-": ["12"]},
    "COLO": {"1": ["51"], "2": ["53"]},
    "COLOCASIVM": {"-": ["12"]},
    "COLOCYNTHA": {"-": ["11"]},
    "COLOEPHIVM": {"-": ["12"]},
    "COLONIA": {"-": ["11"], "N": ["11"]},
    "COLONVS": {"1": ["12"], "2": ["21"]},
    "COLOR": {"-": ["13"]},
    "COLORATVS": {"C": ["21"], "-": ["21"]},
    "COLORO": {"-": ["51"]},
    "COLOSSICVS": {"-": ["2G"]},
    "COLOSSVS": {"-": ["12"]},
    "COLVBER": {"-": ["12"]},
    "COLVBRA": {"-": ["11"]},
    "COLVM": {"-": ["12"]},
    "COLVMBA": {"-": ["11"]},
    "COLVMBARIVM": {"-": ["12"]},
    "COLVMBINVS": {"-": ["21"]},
    "COLVMBOR": {"-": ["5J"]},
    "COLVMBVS": {"-": ["12"]},
    "COLVMELLA": {"-": ["11"]},
    "COLVMEN": {"-": ["13"]},
    "COLVMNA": {"-": ["11"]},
    "COLVMNARIVM": {"-": ["12"]},
    "COLVMNATVS": {"-": ["21"]},
    "COLVRNVS": {"-": ["21"]},
    "COLVS": {"-": ["16"]},
    "COMA": {"-": ["11"]},
    "COMANS": {"-": ["25"]},
    "COMATVS": {"-": ["21"], "A": ["21"]},
    "COMBIBO": {"1": ["13"], "2": ["53"]},
    "COMBVRO": {"-": ["53"]},
    "COMEDO": {"1": ["13"], "2": ["56"]},
    "COMES": {"-": ["13"]},
    "COMETES": {"-": ["17"]},
    "COMICE": {"-": ["60"]},
    "COMICVS": {"1": ["12"], "2": ["21"]},
    "COMINIVS": {"N": ["12"]},
    "COMINVS": {"-": ["60"]},
    "COMIS": {"-": ["24"]},
    "COMISSABVNDVS": {"-": ["21"]},
    "COMISSATIO": {"-": ["13"]},
    "COMISSOR": {"-": ["5J"]},
    "COMITAS": {"-": ["13"]},
    "COMITATVS": {"1": ["14"], "2": ["21"]},
    "COMITER": {"-": ["60"]},
    "COMITIA": {"-": ["12"]},
    "COMITIALIS": {"1": ["13"], "2": ["24"]},
    "COMITIATVS": {"-": ["14"]},
    "COMITIO": {"-": ["51"]},
    "COMITIVM": {"-": ["12"]},
    "COMITO": {"-": ["51"]},
    "COMITOR": {"-": ["5J"]},
    "COMMACVLO": {"-": ["51"]},
    "COMMADEO": {"-": ["52"]},
    "COMMEATVS": {"-": ["14"]},
    "COMMEDITOR": {"-": ["5J"]},
    "COMMEMORATIO": {"-": ["13"]},
    "COMMEMORO": {"-": ["51"]},
    "COMMENDATICIVS": {"-": ["21"]},
    "COMMENDATIO": {"-": ["13"]},
    "COMMENDO": {"-": ["51"]},
    "COMMENSVS": {"-": ["14"]},
    "COMMENTARIVM": {"-": ["12"]},
    "COMMENTARIVS": {"1": ["12"], "2": ["21"]},
    "COMMENTICIVS": {"-": ["21"]},
    "COMMENTOR": {"1": ["13"], "2": ["5J"]},
    "COMMENTVM": {"-": ["12"]},
    "COMMEO": {"-": ["51"]},
    "COMMERCIVM": {"-": ["12"]},
    "COMMERCOR": {"-": ["5J"]},
    "COMMEREO": {"-": ["52"]},
    "COMMEREOR": {"-": ["5K"]},
    "COMMERITVM": {"-": ["12"]},
    "COMMIGRATIO": {"-": ["13"]},
    "COMMIGRO": {"-": ["51"]},
    "COMMILITIVM": {"-": ["12"]},
    "COMMILITO": {"1": ["13"], "2": ["51"]},
    "COMMINATIO": {"-": ["13"]},
    "COMMINGO": {"-": ["53"]},
    "CO": {"-": ["5L"]},
    "COMMINOR": {"-": ["5J"]},
    "COMMINVO": {"-": ["53"]},
    "COMMINVS": {"-": ["60"]},
    "COMMIS": {"-": ["13"]},
    "COMMISCEO": {"-": ["52"]},
    "COMMISSIO": {"-": ["13"]},
    "COMMISSVM": {"-": ["12"]},
    "COMMISSVRA": {"-": ["11"]},
    "COMMITTO": {"-": ["53"]},
    "COMMODE": {"-": ["60"]},
    "COMMODITAS": {"-": ["13"]},
    "COMMODO": {"1": ["51"], "2": ["60"]},
    "COMMODVLATIO": {"-": ["13"]},
    "COMMODVM": {"1": ["12"], "2": ["60"]},
    "COMMODVS": {"-": ["21"]},
    "COMMOLIOR": {"-": ["5M"]},
    "COMMONEFACIO": {"-": ["55"]},
    "COMMONEFIO": {"-": ["56"]},
    "COMMONEO": {"-": ["52"]},
    "COMMONSTRO": {"-": ["51"]},
    "COMMORDEO": {"-": ["52"]},
    "COMMORIOR": {"-": ["5N"]},
    "COMMOROR": {"1": ["5J"]},
    "COMMOTIO": {"-": ["13"]},
    "COMMOTVS": {"-": ["21"]},
    "COMMOVEO": {"-": ["52"]},
    "COMMVNE": {"C": ["13"], "-": ["13"]},
    "COMMVNICATIO": {"-": ["13"]},
    "COMMVNICO": {"-": ["51"]},
    "COMMVNIO": {"1": ["13"], "2": ["54"]},
    "COMMVNIS": {"-": ["24"]},
    "COMMVNITAS": {"-": ["13"]},
    "COMMVNITER": {"-": ["60"]},
    "COMMVNITIO": {"-": ["13"]},
    "COMMVTABILIS": {"-": ["24"]},
    "COMMVTATIO": {"-": ["13"]},
    "COMMVTATVS": {"-": ["14"]},
    "COMMVTO": {"2": ["51"]},
    "COMO": {"1": ["51"], "2": ["53"]},
    "COMOEDIA": {"-": ["11"]},
    "COMOEDVS": {"1": ["12"], "2": ["21"]},
    "COMPACTIO": {"-": ["13"]},
    "COMPACTVRA": {"-": ["11"]},
    "COMPACTVS": {"-": ["21"]},
    "COMPAGES": {"-": ["13"]},
    "COMPAGO": {"-": ["13"]},
    "COMPAR": {"1": ["13"], "2": ["25"]},
    "COMPARATIO": {"1": ["13"], "2": ["13"]},
    "COMPARATVS": {"-": ["14"]},
    "COMPAREO": {"-": ["52"]},
    "COMPARO": {"2": ["51"], "3": ["51"]},
    "COMPEDIO": {"-": ["54"]},
    "COMPELLO": {"1": ["51"], "2": ["53"]},
    "COMPENDIARIA": {"-": ["11"]},
    "COMPENDIARIVM": {"-": ["12"]},
    "COMPENDIARIVS": {"-": ["21"]},
    "COMPENDIVM": {"-": ["12"]},
    "COMPENDO": {"-": ["53"]},
    "COMPENSATIO": {"-": ["13"]},
    "COMPENSO": {"-": ["51"]},
    "COMPERENDINATIO": {"-": ["13"]},
    "COMPERENDINATVS": {"-": ["14"]},
    "COMPERENDINO": {"-": ["51"]},
    "COMPERIO": {"-": ["54"]},
    "COMPERTVS": {"-": ["21"]},
    "COMPESCO": {"-": ["53"]},
    "COMPETITOR": {"-": ["13"]},
    "COMPETO": {"-": ["53"]},
    "COMPILATIO": {"-": ["13"]},
    "COMPILO": {"1": ["51"], "2": ["51"]},
    "COMPINGO": {"1": ["53"], "2": ["53"]},
    "COMPITVM": {"-": ["12"]},
    "COMPLANO": {"-": ["51"]},
    "COMPLECTO": {"2": ["53"]},
    "COMPLECTOR": {"-": ["5L"]},
    "COMPLEMENTVM": {"-": ["12"]},
    "COMPLEO": {"-": ["52"]},
    "COMPLETVS": {"-": ["21"]},
    "COMPLEXIO": {"-": ["13"]},
    "COMPLEXVS": {"-": ["14"]},
    "COMPLICO": {"-": ["51"]},
    "COMPLODO": {"-": ["53"]},
    "COMPLORATIO": {"-": ["13"]},
    "COMPLORATVS": {"-": ["14"]},
    "COMPLORO": {"-": ["51"]},
    "COMPLVRES": {"1": ["13"], "2": ["22"]},
    "COMPLVRIES": {"-": ["60"]},
    "COMPLVVIVM": {"-": ["12"]},
    "COMPONO": {"-": ["53"]},
    "COMPORTATIO": {"-": ["13"]},
    "COMPORTO": {"-": ["51"]},
    "COMPOS": {"-": ["22"]},
    "COMPOSITE": {"-": ["60"]},
    "COMPOSITIO": {"-": ["13"]},
    "COMPOSITVM": {"-": ["12"]},
    "COMPOSITVRA": {"-": ["11"]},
    "COMPOSITVS": {"O": ["21"], "-": ["21"]},
    "COMPRECOR": {"-": ["5J"]},
    "COMPREHENDO": {"-": ["53"]},
    "COMPREHENSIBILIS": {"-": ["24"]},
    "COMPREHENSIO": {"-": ["13"]},
    "COMPRESSIO": {"-": ["13"]},
    "COMPRESSO": {"-": ["51"]},
    "COMPRESSVS": {"1": ["14"], "2": ["21"]},
    "COMPRIMO": {"-": ["53"]},
    "COMPROBO": {"-": ["51"]},
    "COMPROMISSVM": {"-": ["12"]},
    "COMPSANI": {"N": ["12"]},
    "COMPSANVS": {"N": ["21"], "A": ["21"]},
    "COMPTE": {"-": ["60"]},
    "COMPTVLVS": {"-": ["21"]},
    "COMPTVS": {"1": ["14"], "2": ["21"]},
    "COMPVNGO": {"-": ["53"]},
    "COMPVTATIO": {"-": ["13"]},
    "COMPVTATOR": {"-": ["13"]},
    "COMPVTO": {"-": ["51"]},
    "COMPVTRESCO": {"-": ["53"]},
    "CONAMEN": {"-": ["13"]},
    "CONARACHNE": {"-": ["17"]},
    "CONATA": {"-": ["12"]},
    "CONATVS": {"-": ["14"]},
    "CONCAEDES": {"-": ["13"]},
    "CONCALEFACIO": {"-": ["55"]},
    "CONCALEO": {"-": ["52"]},
    "CONCALESCO": {"-": ["53"]},
    "CONCALLEO": {"-": ["52"]},
    "CONCAMERATIO": {"-": ["13"]},
    "CONCAMERO": {"-": ["51"]},
    "CONCAVO": {"1": ["51"]},
    "CONCAVVS": {"-": ["21"]},
    "CONCEDO": {"1": ["53"]},
    "CONCELEBRO": {"-": ["51"]},
    "CONCENTIO": {"-": ["13"]},
    "CONCENTVS": {"-": ["14"]},
    "CONCEPTIO": {"-": ["13"]},
    "CONCEPTO": {"-": ["55"]},
    "CONCEPTVS": {"-": ["14"], "2": ["14"], "1": ["12"]},
    "CONCERNO": {"-": ["53"]},
    "CONCERTATOR": {"-": ["13"]},
    "CONCERTO": {"1": ["51"]},
    "CONCESSIO": {"-": ["13"]},
    "CONCESSVM": {"-": ["12"]},
    "CONCESSVS": {"-": ["14"]},
    "CONCHA": {"-": ["11"]},
    "CONCHIS": {"-": ["13"]},
    "CONCHYLIATVS": {"-": ["21"]},
    "CONCHYLIVM": {"-": ["12"]},
    "CONCIDO": {"1": ["53"], "2": ["53"]},
    "CONCIEO": {"-": ["52"]},
    "CONCILIABVLVM": {"-": ["12"]},
    "CONCILIATIO": {"-": ["13"]},
    "CONCILIATOR": {"-": ["13"]},
    "CONCILIATRIX": {"-": ["13"]},
    "CONCILIATVRA": {"-": ["11"]},
    "CONCILIATVS": {"-": ["14"]},
    "CONCILIO": {"-": ["51"]},
    "CONCILIVM": {"-": ["12"]},
    "CONCINNITAS": {"-": ["13"]},
    "CONCINNO": {"-": ["51"]},
    "CONCINNVS": {"-": ["21"]},
    "CONCINO": {"-": ["53"]},
    "CONCIO": {"-": ["54"]},
    "CONCIPIO": {"-": ["55"]},
    "CONCISVRA": {"-": ["11"]},
    "CONCISVS": {"-": ["21"]},
    "CONCITAMENTVM": {"-": ["12"]},
    "CONCITATE": {"-": ["60"]},
    "CONCITATIO": {"-": ["13"]},
    "CONCITATOR": {"-": ["13"]},
    "CONCITATVS": {"-": ["21"]},
    "CONCITO": {"1": ["51"]},
    "CONCITOR": {"-": ["13"]},
    "CONCLAMATIO": {"-": ["13"]},
    "CONCLAMO": {"-": ["51"]},
    "CONCLAVE": {"-": ["16"]},
    "CONCLVDO": {"-": ["53"]},
    "CONCLVSIO": {"-": ["13"]},
    "CONCLVSVRA": {"-": ["11"]},
    "CONCLVSVS": {"1": ["14"], "2": ["21"]},
    "CONCOLOR": {"-": ["25"]},
    "CONCOQVO": {"-": ["53"]},
    "CONCORDIA": {"-": ["11"], "N": ["11"]},
    "CONCORDITER": {"-": ["6-"]},
    "CONCORDO": {"-": ["51"]},
    "CONCORS": {"-": ["25"]},
    "CONCREDO": {"-": ["53"]},
    "CONCREMO": {"-": ["51"]},
    "CONCREPITO": {"-": ["51"]},
    "CONCREPO": {"-": ["51"]},
    "CONCRESCENTIA": {"-": ["11"]},
    "CONCRESCO": {"-": ["53"]},
    "CONCRETVS": {"1": ["14"], "2": ["21"]},
    "CONCRISPO": {"-": ["51"]},
    "CONCRVCIO": {"-": ["51"]},
    "CONCVBINA": {"-": ["11"]},
    "CONCVBINVS": {"-": ["12"]},
    "CONCVBITVS": {"-": ["14"]},
    "CONCVBIVS": {"-": ["21"]},
    "CONCVLCO": {"-": ["51"]},
    "CONCVMBO": {"-": ["53"]},
    "CONCVPIO": {"-": ["55"]},
    "CONCVPISCO": {"-": ["53"]},
    "CONCVPITOR": {"-": ["13"]},
    "CONCVRRO": {"-": ["53"]},
    "CONCVRSATIO": {"-": ["13"]},
    "CONCVRSATOR": {"-": ["13"]},
    "CONCVRSIO": {"-": ["13"]},
    "CONCVRSO": {"-": ["51"]},
    "CONCVRSVS": {"-": ["14"]},
    "CONCVSSVS": {"-": ["14"]},
    "CONCVTIO": {"-": ["55"]},
    "CONDECORO": {"-": ["51"]},
    "CONDELIQVESCO": {"-": ["53"]},
    "CONDEMNATOR": {"-": ["13"]},
    "CONDEMNO": {"-": ["51"]},
    "CONDENSEO": {"-": ["52"]},
    "CONDENSO": {"-": ["51"]},
    "CONDENSVS": {"-": ["21"]},
    "CONDEPSO": {"-": ["53"]},
    "CONDICIO": {"-": ["13"]},
    "CONDICO": {"2": ["53"]},
    "CONDIGNVS": {"-": ["21"]},
    "CONDIMENTVM": {"-": ["12"]},
    "CONDIO": {"-": ["54"]},
    "CONDISCIPVLVS": {"-": ["12"]},
    "CONDISCO": {"-": ["53"]},
    "CONDITIVVM": {"-": ["12"]},
    "CONDITIVVS": {"-": ["21"]},
    "CONDITOR": {"1": ["13"], "2": ["13"]},
    "CONDITVRA": {"1": ["11"], "2": ["11"]},
    "CONDITVS": {"1": ["14"], "2": ["14"], "3": ["21"], "4": ["21"]},
    "CONDO": {"-": ["53"]},
    "CONDOCEFACIO": {"-": ["55"]},
    "CONDOLEO": {"-": ["52"]},
    "CONDOLESCO": {"-": ["53"]},
    "CONDONATIO": {"-": ["13"]},
    "CONDONO": {"-": ["51"]},
    "CONDORMIO": {"-": ["54"]},
    "CONDVCO": {"-": ["53"]},
    "CONDVCTIO": {"-": ["13"]},
    "CONDVCTOR": {"-": ["13"]},
    "CONDVCTVM": {"-": ["12"]},
    "CONDVPLICO": {"-": ["51"]},
    "CONDVRO": {"-": ["51"]},
    "CONECTO": {"-": ["53"]},
    "CONFARREATIO": {"-": ["13"]},
    "CONFARREO": {"-": ["51"]},
    "CONFECTIO": {"-": ["13"]},
    "CONFECTOR": {"-": ["13"]},
    "CONFERCIO": {"-": ["54"]},
    "CONFERO": {"-": ["56"]},
    "CONFERTIM": {"-": ["60"]},
    "CONFERTVS": {"-": ["21"]},
    "CONFERVEFACIO": {"-": ["55"]},
    "CONFERVESCO": {"-": ["53"]},
    "CONFESSIO": {"-": ["13"]},
    "CONFESSVM": {"-": ["12"]},
    "CONFESSVS": {"-": ["21"]},
    "CONFESTIM": {"-": ["60"]},
    "CONFICIENS": {"-": ["25"]},
    "CONFICIO": {"-": ["55"]},
    "CONFIDENS": {"-": ["25"]},
    "CONFIDENTIA": {"-": ["11"]},
    "CONFIDO": {"-": ["5T"]},
    "CONFIGO": {"-": ["53"]},
    "CONFINE": {"-": ["13"]},
    "CONFINGO": {"-": ["53"]},
    "CONFINIS": {"1": ["13"], "2": ["24"]},
    "CONFINIVM": {"-": ["12"]},
    "CONFIO": {"-": ["56"]},
    "CONFIRMATIO": {"-": ["13"]},
    "CONFIRMATVS": {"-": ["21"]},
    "CONFIRMO": {"-": ["51"]},
    "CONFITEOR": {"-": ["5K"]},
    "CONFLAGRO": {"-": ["51"]},
    "CONFLICTO": {"-": ["51"]},
    "CONFLICTVS": {"-": ["14"]},
    "CONFLIGO": {"-": ["53"]},
    "CONFLO": {"-": ["51"]},
    "CONFLVENS": {"-": ["13"]},
    "CONFLVENTES": {"N": ["13"]},
    "CONFLVO": {"-": ["53"]},
    "CONFODIO": {"-": ["55"]},
    "CONFORMATIO": {"-": ["13"]},
    "CONFORMO": {"-": ["51"]},
    "CONFORNICATIO": {"-": ["13"]},
    "CONFORNICO": {"-": ["51"]},
    "CONFRAGOSVM": {"-": ["12"]},
    "CONFRAGOSVS": {"-": ["21"]},
    "CONFRICO": {"-": ["51"]},
    "CONFRINGO": {"-": ["53"]},
    "CONFRIO": {"-": ["51"]},
    "CONFVGIO": {"-": ["55"]},
    "CONFVLCIO": {"-": ["54"]},
    "CONFVNDO": {"2": ["53"]},
    "CONFVSE": {"-": ["60"]},
    "CONFVSIO": {"-": ["13"]},
    "CONFVSVS": {"-": ["21"]},
    "CONFVTO": {"-": ["51"]},
    "CONFVTVO": {"-": ["53"]},
    "CONGELO": {"-": ["51"]},
    "CONGEMO": {"-": ["53"]},
    "CONGERIES": {"-": ["15"]},
    "CONGERO": {"1": ["13"], "2": ["53"]},
    "CONGESTICIVS": {"-": ["21"]},
    "CONGESTIO": {"-": ["13"]},
    "CONGESTVS": {"1": ["14"], "2": ["21"]},
    "CONGIARIVM": {"-": ["12"]},
    "CONGIARIVS": {"-": ["21"]},
    "CONGIVS": {"-": ["12"]},
    "CONGLOBATIO": {"-": ["13"]},
    "CONGLOBO": {"-": ["51"]},
    "CONGLOMERO": {"-": ["51"]},
    "CONGLVTINO": {"-": ["51"]},
    "CONGREDIOR": {"-": ["5N"]},
    "CONGREGATIO": {"-": ["13"]},
    "CONGREGO": {"-": ["51"]},
    "CONGRESSVS": {"-": ["14"]},
    "CONGRVENS": {"-": ["25"]},
    "CONGRVO": {"-": ["53"]},
    "CONGRVVS": {"-": ["21"]},
    "CONICIO": {"-": ["55"]},
    "CONICVS": {"-": ["21"]},
    "CONIECTIO": {"-": ["13"]},
    "CONIECTO": {"-": ["51"]},
    "CONIECTVRA": {"-": ["11"]},
    "CONIECTVS": {"-": ["14"]},
    "CONIFER": {"-": ["21"]},
    "CONIGER": {"-": ["21"]},
    "CONISTERIVM": {"-": ["12"]},
    "CONIVGALIS": {"-": ["24"]},
    "CONIVGATOR": {"-": ["13"]},
    "CONIVGIVM": {"-": ["12"]},
    "CONIVGVLVS": {"-": ["21"]},
    "CONIVNCTE": {"-": ["6-"]},
    "CONIVNCTIM": {"-": ["60"]},
    "CONIVNCTIO": {"-": ["13"]},
    "CONIVNCTVM": {"-": ["12"]},
    "CONIVNCTVS": {"-": ["21"]},
    "CONIVNGO": {"-": ["53"]},
    "CONIVOLVS": {"-": ["21"]},
    "CONIVRATI": {"-": ["12"]},
    "CONIVRATIM": {"-": ["60"]},
    "CONIVRATIO": {"-": ["13"]},
    "CONIVRATOR": {"-": ["13"]},
    "CONIVRATVS": {"-": ["21"]},
    "CONIVRO": {"-": ["51"]},
    "CONIVX": {"-": ["13"]},
    "CONNECTO": {"-": ["53"]},
    "CONNEXVS": {"1": ["14"], "2": ["21"]},
    "CONNITOR": {"2": ["5L"]},
    "CONNIVEO": {"-": ["52"]},
    "CONNVBIVM": {"-": ["12"]},
    "CONOPEVM": {"-": ["12"]},
    "CONOPIVM": {"-": ["12"]},
    "CONOR": {"-": ["5J"]},
    "CONQVASSO": {"-": ["51"]},
    "CONQVEROR": {"-": ["5L"]},
    "CONQVESTIO": {"-": ["13"]},
    "CONQVIESCO": {"-": ["53"]},
    "CONQVIRO": {"-": ["53"]},
    "CONQVISITIO": {"-": ["13"]},
    "CONQVISITOR": {"-": ["13"]},
    "CONQVISITVS": {"-": ["21"]},
    "CONSAEPTVM": {"-": ["12"]},
    "CONSAEPTVS": {"1": ["14"], "2": ["21"]},
    "CONSALVTATIO": {"-": ["13"]},
    "CONSALVTO": {"-": ["51"]},
    "CONSANGVINEA": {"-": ["11"]},
    "CONSANGVINEI": {"-": ["12"]},
    "CONSANGVINEVS": {"1": ["12"], "2": ["21"]},
    "CONSANGVINITAS": {"-": ["13"]},
    "CONSARIO": {"-": ["54"]},
    "CONSCELERATVS": {"-": ["21"]},
    "CONSCELERO": {"-": ["51"]},
    "CONSCENDO": {"-": ["53"]},
    "CONSCIENTIA": {"-": ["11"]},
    "CONSCIO": {"-": ["54"]},
    "CONSCISCO": {"-": ["53"]},
    "CONSCIVS": {"-": ["21"]},
    "CONSCRIBILLO": {"-": ["51"]},
    "CONSCRIBO": {"-": ["53"]},
    "CONSCRIPTIO": {"-": ["13"]},
    "CONSECO": {"-": ["51"]},
    "CONSECRATIO": {"-": ["13"]},
    "CONSECRATVS": {"-": ["21"]},
    "CONSECRO": {"-": ["51"]},
    "CONSECTATIO": {"-": ["13"]},
    "CONSECTOR": {"2": ["5J"]},
    "CONSECVE": {"-": ["60"]},
    "CONSEDO": {"-": ["51"]},
    "CONSENESCO": {"-": ["53"]},
    "CONSENSIO": {"-": ["13"]},
    "CONSENSVS": {"-": ["14"]},
    "CONSENTANEVS": {"-": ["21"]},
    "CONSENTIENS": {"-": ["25"]},
    "CONSENTIO": {"-": ["54"]},
    "CONSEQVENS": {"1": ["13"], "2": ["25"]},
    "CONSEQVOR": {"-": ["5L"]},
    "CONSERO": {"2": ["53"], "3": ["53"]},
    "CONSERVATIO": {"-": ["13"]},
    "CONSERVATOR": {"-": ["13"], "N": ["13"]},
    "CONSERVO": {"-": ["51"]},
    "CONSERVVS": {"1": ["12"], "2": ["21"]},
    "CONSESSVS": {"-": ["14"]},
    "CONSIDEO": {"-": ["52"]},
    "CONSIDERANTIA": {"-": ["11"]},
    "CONSIDERATE": {"-": ["6&"]},
    "CONSIDERATIO": {"-": ["13"]},
    "CONSIDERATVS": {"-": ["21"]},
    "CONSIDERO": {"-": ["51"]},
    "CONSIDIVS": {"N": ["12"]},
    "CONSIDO": {"-": ["53"]},
    "CONSIGNO": {"-": ["51"]},
    "CONSILIARIVS": {"1": ["12"], "2": ["21"]},
    "CONSILIOR": {"-": ["5J"]},
    "CONSILIVM": {"-": ["12"]},
    "CONSIMILIS": {"-": ["24"]},
    "CONSIPIO": {"-": ["55"]},
    "CONSISTO": {"-": ["53"]},
    "CONSITOR": {"-": ["13"]},
    "CONSOBRINVS": {"-": ["12"]},
    "CONSOCIO": {"-": ["51"]},
    "CONSOLATIO": {"-": ["13"]},
    "CONSOLATOR": {"-": ["13"]},
    "CONSOLIDO": {"-": ["51"]},
    "CONSOLOR": {"-": ["5J"]},
    "CONSONANTER": {"-": ["6-"]},
    "CONSONANTIA": {"-": ["11"]},
    "CONSONO": {"1": ["51"]},
    "CONSONVS": {"-": ["21"]},
    "CONSOPIO": {"-": ["54"]},
    "CONSORS": {"-": ["25"]},
    "CONSORTIVM": {"-": ["12"]},
    "CONSPECTVS": {"1": ["14"], "2": ["21"]},
    "CONSPERGO": {"-": ["53"]},
    "CONSPICIO": {"-": ["55"]},
    "CONSPICOR": {"-": ["5J"]},
    "CONSPICVVM": {"-": ["12"]},
    "CONSPICVVS": {"-": ["21"]},
    "CONSPIRATIO": {"-": ["13"]},
    "CONSPIRO": {"1": ["51"], "2": ["51"]},
    "CONSPVO": {"-": ["53"]},
    "CONSPVRCO": {"-": ["51"]},
    "CONSTABILIO": {"-": ["54"]},
    "CONSTANS": {"-": ["25"]},
    "CONSTANTER": {"-": ["60"]},
    "CONSTANTIA": {"-": ["11"]},
    "CONSTERNATIO": {"-": ["13"]},
    "CONSTERNO": {"1": ["51"], "2": ["53"]},
    "CONSTIBILIS": {"-": ["24"]},
    "CONSTIPO": {"-": ["51"]},
    "CONSTITVO": {"-": ["53"]},
    "CONSTITVTIO": {"-": ["13"]},
    "CONSTITVTVM": {"-": ["12"]},
    "CONSTITVTVS": {"1": ["14"], "2": ["21"]},
    "CONSTO": {"-": ["51"]},
    "CONSTRATVS": {"-": ["21"]},
    "CONSTRICTVS": {"-": ["21"]},
    "CONSTRINGO": {"-": ["53"]},
    "CONSTRVCTIO": {"-": ["13"]},
    "CONSTRVO": {"-": ["53"]},
    "CONSTVPRO": {"-": ["51"]},
    "CONSVALIA": {"N": ["16"]},
    "CONSVDO": {"-": ["51"]},
    "CONSVEFACIO": {"-": ["55"]},
    "CONSVESCO": {"-": ["53"]},
    "CONSVETVDO": {"-": ["13"]},
    "CONSVETVS": {"-": ["21"]},
    "CONSVL": {"-": ["13"]},
    "CONSVLARIS": {"1": ["13"], "2": ["24"]},
    "CONSVLATVS": {"-": ["14"]},
    "CONSVLO": {"-": ["53"]},
    "CONSVLTATIO": {"-": ["13"]},
    "CONSVLTO": {"1": ["51"], "2": ["60"]},
    "CONSVLTOR": {"-": ["13"]},
    "CONSVLTVM": {"-": ["12"]},
    "CONSVMMABILIS": {"-": ["24"]},
    "CONSVMMATIO": {"-": ["13"]},
    "CONSVMMATVS": {"-": ["21"]},
    "CONSVMMO": {"-": ["51"]},
    "CONSVMO": {"-": ["53"]},
    "CONSVO": {"-": ["53"]},
    "CONSVRGO": {"-": ["53"]},
    "CONTABVLATIO": {"-": ["13"]},
    "CONTABVLO": {"-": ["51"]},
    "CONTACTVS": {"-": ["14"]},
    "CONTAGES": {"-": ["13"]},
    "CONTAGIO": {"-": ["13"]},
    "CONTAGIVM": {"-": ["12"]},
    "CONTAMINATVS": {"-": ["21"]},
    "CONTAMINO": {"-": ["51"]},
    "CONTEGO": {"-": ["53"]},
    "CONTEMERO": {"-": ["51"]},
    "CONTEMNENDVS": {"-": ["21"]},
    "CONTEMNO": {"-": ["53"]},
    "CONTEMPLATIO": {"-": ["13"]},
    "CONTEMPLATIVVS": {"-": ["21"]},
    "CONTEMPLATOR": {"-": ["13"]},
    "CONTEMPLOR": {"-": ["5J"]},
    "CONTEMPTE": {"-": ["6&"]},
    "CONTEMPTIM": {"-": ["60"]},
    "CONTEMPTIO": {"-": ["13"]},
    "CONTEMPTOR": {"-": ["13"]},
    "CONTEMPTRIX": {"-": ["13"]},
    "CONTEMPTVS": {"1": ["14"], "2": ["21"]},
    "CONTENDO": {"-": ["53"]},
    "CONTENTE": {"1": ["60"], "2": ["60"]},
    "CONTENTIO": {"1": ["13"], "2": ["13"]},
    "CONTENTVS": {"1": ["21"], "2": ["21"]},
    "CONTERMINVM": {"-": ["12"]},
    "CONTERMINVS": {"1": ["12"], "2": ["21"]},
    "CONTERO": {"-": ["53"]},
    "CONTERREO": {"-": ["52"]},
    "CONTESTATE": {"-": ["60"]},
    "CONTESTOR": {"-": ["5J"]},
    "CONTEXO": {"-": ["53"]},
    "CONTEXTVS": {"1": ["14"], "2": ["21"]},
    "CONTICESCO": {"-": ["53"]},
    "CONTIGNATIO": {"-": ["13"]},
    "CONTIGNO": {"-": ["51"]},
    "CONTIGVITAS": {"-": ["13"]},
    "CONTIGVVS": {"-": ["21"]},
    "CONTINENS": {"1": ["13"], "2": ["13"], "3": ["25"]},
    "CONTINENTER": {"-": ["60"]},
    "CONTINENTIA": {"-": ["11"]},
    "CONTINEO": {"-": ["52"]},
    "CONTINGO": {"1": ["53"], "2": ["53"]},
    "CONTINVATIO": {"-": ["13"]},
    "CONTINVATVS": {"2": ["21"], "-": ["21"]},
    "CONTINVO": {"1": ["51"], "2": ["60"]},
    "CONTINVVS": {"-": ["21"]},
    "CONTIO": {"-": ["13"]},
    "CONTIONABVNDVS": {"-": ["21"]},
    "CONTIONATOR": {"-": ["13"]},
    "CONTIONOR": {"-": ["5J"]},
    "CONTORQVEO": {"-": ["52"]},
    "CONTRA": {"1": ["60"], "2": ["70"]},
    "CONTRACTABILITER": {"-": ["60"]},
    "CONTRACTE": {"-": ["6&"]},
    "CONTRACTIO": {"-": ["13"]},
    "CONTRACTO": {"-": ["51"]},
    "CONTRACTVRA": {"-": ["11"]},
    "CONTRACTVS": {"1": ["14"], "2": ["21"]},
    "CONTRADICO": {"2": ["53"]},
    "CONTRADICTIO": {"-": ["13"]},
    "CONTRAHO": {"-": ["53"]},
    "CONTRALEGO": {"2": ["53"]},
    "CONTRARIE": {"-": ["60"]},
    "CONTRARIETAS": {"-": ["13"]},
    "CONTRARIVM": {"C": ["12"], "-": ["12"]},
    "CONTRARIVS": {"-": ["21"]},
    "CONTRAVERSVM": {"-": ["6O"]},
    "CONTRAVERTO": {"-": ["53"]},
    "CONTRECTO": {"-": ["51"]},
    "CONTREMISCO": {"-": ["53"]},
    "CONTREMO": {"-": ["53"]},
    "CONTRIBVO": {"-": ["53"]},
    "CONTRICO": {"-": ["51"]},
    "CONTRISTO": {"-": ["51"]},
    "CONTROVERSIA": {"-": ["11"]},
    "CONTROVERSIOSVS": {"-": ["21"]},
    "CONTROVERSVS": {"1": ["21"], "2": ["60"]},
    "CONTRVCIDO": {"-": ["51"]},
    "CONTRVDO": {"-": ["53"]},
    "CONTVBERNALIS": {"-": ["13"]},
    "CONTVBERNIVM": {"-": ["12"]},
    "CONTVEOR": {"-": ["5K"]},
    "CONTVITVS": {"-": ["14"]},
    "CONTVMACIA": {"-": ["11"]},
    "CONTVMACITER": {"-": ["60"]},
    "CONTVMAX": {"-": ["25"]},
    "CONTVMELIA": {"-": ["11"]},
    "CONTVMELIOSE": {"-": ["60"]},
    "CONTVMELIOSVS": {"-": ["21"]},
    "CONTVNDO": {"-": ["53"]},
    "CONTVOR": {"-": ["5L"]},
    "CONTVRBO": {"2": ["51"]},
    "CONTVS": {"-": ["12"]},
    "CONVALESCO": {"-": ["53"]},
    "CONVALLIS": {"-": ["13"]},
    "CONVECTO": {"-": ["51"]},
    "CONVECTOR": {"-": ["13"]},
    "CONVEHO": {"-": ["53"]},
    "CONVELLO": {"-": ["53"]},
    "CONVENA": {"-": ["11"]},
    "CONVENIENS": {"-": ["25"]},
    "CONVENIENTER": {"-": ["60"]},
    "CONVENIENTIA": {"-": ["11"]},
    "CONVENIO": {"-": ["54"]},
    "CONVENTICVLVM": {"-": ["12"]},
    "CONVENTIO": {"-": ["13"]},
    "CONVENTO": {"-": ["51"]},
    "CONVENTVM": {"-": ["12"]},
    "CONVENTVS": {"-": ["14"]},
    "CONVERBERO": {"2": ["51"]},
    "CONVERRO": {"-": ["53"]},
    "CONVERSATIO": {"-": ["13"]},
    "CONVERSIO": {"-": ["13"]},
    "CONVERSO": {"-": ["51"]},
    "CONVERSOR": {"-": ["5J"]},
    "CONVERTO": {"-": ["53"]},
    "CONVESTIO": {"-": ["54"]},
    "CONVEXVM": {"-": ["12"]},
    "CONVEXVS": {"-": ["21"]},
    "CONVICIATOR": {"-": ["13"]},
    "CONVICIO": {"-": ["51"]},
    "CONVICIOR": {"-": ["5J"]},
    "CONVICIVM": {"-": ["12"]},
    "CONVICTOR": {"-": ["13"]},
    "CONVICTVS": {"-": ["14"]},
    "CONVINCIO": {"-": ["54"]},
    "CONVINCO": {"-": ["53"]},
    "CONVISO": {"-": ["53"]},
    "CONVIVA": {"-": ["11"]},
    "CONVIVALIS": {"-": ["24"]},
    "CONVIVATOR": {"-": ["13"]},
    "CONVIVIVM": {"-": ["12"]},
    "CONVIVO": {"1": ["51"], "2": ["53"]},
    "CONVIVOR": {"-": ["5J"]},
    "CONVOCO": {"-": ["51"]},
    "CONVOLO": {"2": ["51"]},
    "CONVOLVO": {"-": ["53"]},
    "CONVOLVTO": {"-": ["51"]},
    "CONVOLVVLVS": {"-": ["12"]},
    "CONVOMO": {"-": ["53"]},
    "CONVS": {"-": ["12"]},
    "CONVVLNERO": {"-": ["51"]},
    "COOPERIO": {"-": ["54"]},
    "COOPTATIO": {"-": ["13"]},
    "COOPTO": {"-": ["51"]},
    "COORIOR": {"-": ["5M"]},
    "COORTVS": {"-": ["14"]},
    "COPHINVS": {"-": ["12"]},
    "COPIA": {"-": ["11"], "N": ["11"]},
    "COPIOSE": {"-": ["60"]},
    "COPIOSVS": {"-": ["21"]},
    "COPVLA": {"-": ["11"]},
    "COPVLO": {"-": ["51"]},
    "COPVLOR": {"-": ["5J"]},
    "COQVO": {"-": ["53"]},
    "COQVVS": {"-": ["12"]},
    "COR": {"-": ["13"]},
    "CORA": {"1N": ["11"], "N": ["11"]},
    "CORACINVS": {"-": ["21"]},
    "CORALIVM": {"-": ["12"]},
    "CORAM": {"1": ["60"], "2": ["70"]},
    "CORAX": {"-": ["13"], "N": ["13"]},
    "CORBIO": {"N": ["13"]},
    "CORBIS": {"-": ["13"]},
    "CORBVLA": {"-": ["11"]},
    "CORCYRA": {"N": ["11"]},
    "CORCYRAEI": {"N": ["12"]},
    "CORCYRAEVS": {"N": ["21"], "A": ["21"]},
    "CORDAX": {"-": ["13"]},
    "CORDVBA": {"N": ["11"]},
    "CORDVS": {"N": ["12"]},
    "CORIANDRVM": {"-": ["12"]},
    "CORINTHII": {"N": ["12"]},
    "CORINTHIVM": {"N": ["12"]},
    "CORINTHIVS": {"N": ["21"], "A": ["21"]},
    "CORINTHVS": {"N": ["12"]},
    "CORIOLANVS": {"A": ["21"], "N": ["12"]},
    "CORIOLI": {"N": ["12"]},
    "CORIVS": {"-": ["12"]},
    "CORNELIA": {"N": ["11"]},
    "CORNELIVS": {"A": ["21"], "N": ["12"]},
    "CORNEVS": {"1": ["21"], "2": ["21"]},
    "CORNICEN": {"-": ["13"]},
    "CORNICOR": {"-": ["5J"]},
    "CORNICVLA": {"-": ["11"]},
    "CORNICVLVM": {"N": ["12"]},
    "CORNIFICIVS": {"N": ["12"]},
    "CORNIGER": {"-": ["21"]},
    "CORNIPES": {"1": ["13"], "2": ["25"]},
    "CORNIX": {"-": ["13"]},
    "CORNV": {"-": ["14"]},
    "CORNVM": {"1": ["12"], "2": ["12"]},
    "CORNVS": {"-": ["16"]},
    "COROLLA": {"-": ["11"]},
    "COROLLARIVM": {"-": ["12"]},
    "CORONA": {"-": ["11"], "N": ["11"]},
    "CORONAMENTVM": {"-": ["12"]},
    "CORONARIVS": {"-": ["21"]},
    "CORONO": {"-": ["51"]},
    "CORPORALIS": {"-": ["24"]},
    "CORPORATVRA": {"-": ["11"]},
    "CORPOREVS": {"-": ["21"]},
    "CORPVS": {"-": ["13"]},
    "CORPVSCVLVM": {"-": ["12"]},
    "CORRADO": {"-": ["53"]},
    "CORRECTIO": {"-": ["13"]},
    "CORRECTOR": {"-": ["13"]},
    "CORREPO": {"-": ["53"]},
    "CORREPTIO": {"-": ["13"]},
    "CORRIDEO": {"-": ["52"]},
    "CORRIGO": {"-": ["53"]},
    "CORRIPIO": {"-": ["55"]},
    "CORROBORO": {"-": ["51"]},
    "CORRODO": {"-": ["53"]},
    "CORROGO": {"-": ["51"]},
    "CORROTVNDO": {"-": ["51"]},
    "CORRVDA": {"-": ["11"]},
    "CORRVGO": {"-": ["53"]},
    "CORRVMPO": {"-": ["53"]},
    "CORRVO": {"-": ["53"]},
    "CORRVPTELA": {"-": ["11"]},
    "CORRVPTIO": {"-": ["13"]},
    "CORRVPTOR": {"-": ["13"]},
    "CORRVPTVS": {"C": ["21"], "-": ["21"]},
    "CORSA": {"-": ["11"]},
    "CORTEX": {"-": ["13"]},
    "CORTINA": {"-": ["11"]},
    "CORVSCO": {"-": ["51"]},
    "CORVSCVS": {"-": ["21"]},
    "CORVVS": {"-": ["12"], "N": ["12"]},
    "CORYCEVM": {"-": ["12"]},
    "CORYLVS": {"-": ["12"]},
    "CORYMBVS": {"-": ["12"]},
    "CORYTVS": {"-": ["12"]},
    "COS": {"-": ["13"]},
    "COSMETA": {"-": ["11"]},
    "COSSVRA": {"N": ["17"]},
    "COSSVS": {"N": ["12"]},
    "COSSVTIVS": {"N": ["12"]},
    "COSTA": {"-": ["11"], "N": ["11"]},
    "COSTVM": {"-": ["17"]},
    "COTES": {"-": ["13"]},
    "COTHON": {"-": ["17"]},
    "COTHVRNATVS": {"-": ["21"]},
    "COTHVRNVS": {"-": ["12"]},
    "COTIDIANO": {"-": ["60"]},
    "COTIDIANVS": {"-": ["21"]},
    "COTIDIE": {"-": ["60"]},
    "COTONEVS": {"-": ["21"]},
    "COTTA": {"N": ["11"]},
    "COTTANA": {"-": ["12"]},
    "COTTIVS": {"N": ["12"], "A": ["21"]},
    "COTVRNIX": {"-": ["13"]},
    "COTYLA": {"-": ["11"], "N": ["11"]},
    "COTYTIA": {"-": ["12"]},
    "COVINNARIVS": {"-": ["12"]},
    "COVM": {"-": ["12"]},
    "COXA": {"-": ["11"]},
    "COXENDIX": {"-": ["13"]},
    "COXVS": {"-": ["21"]},
    "CRABRO": {"-": ["13"]},
    "CRAMBE": {"-": ["17"]},
    "CRAPVLA": {"-": ["11"]},
    "CRAS": {"1": ["16"], "2": ["60"]},
    "CRASSE": {"-": ["60"]},
    "CRASSITVDO": {"-": ["13"]},
    "CRASSVS": {"-": ["21"], "N": ["12"]},
    "CRASTINVM": {"-": ["12"]},
    "CRASTINVS": {"-": ["21"], "N": ["12"]},
    "CRATER": {"-": ["17"], "N": ["12"]},
    "CRATERA": {"-": ["11"]},
    "CRATES": {"-": ["13"], "N": ["13"]},
    "CRATICIVS": {"-": ["21"]},
    "CRATICVLA": {"-": ["11"]},
    "CRATIPPVS": {"N": ["12"]},
    "CREATOR": {"-": ["13"]},
    "CREATRIX": {"-": ["13"]},
    "CREBER": {"-": ["21"]},
    "CREBRA": {"-": ["60"]},
    "CREBRE": {"-": ["60"]},
    "CREBRESCO": {"-": ["53"]},
    "CREBRITAS": {"-": ["13"]},
    "CREBRO": {"1": ["51"], "2": ["60"]},
    "CREDIBILIS": {"-": ["24"]},
    "CREDITOR": {"-": ["13"]},
    "CREDITVM": {"-": ["12"]},
    "CREDO": {"-": ["53"]},
    "CREDVLITAS": {"-": ["13"]},
    "CREDVLVS": {"-": ["21"]},
    "CREMERA": {"N": ["11"]},
    "CREMO": {"-": ["51"]},
    "CREMOR": {"-": ["13"]},
    "CREMVTIVS": {"N": ["12"]},
    "CREO": {"-": ["51"]},
    "CREPAX": {"-": ["25"]},
    "CREPER": {"-": ["21"]},
    "CREPEREIVS": {"N": ["12"]},
    "CREPIDA": {"-": ["11"]},
    "CREPIDATVS": {"-": ["21"]},
    "CREPIDO": {"-": ["13"]},
    "CREPITACILLVM": {"-": ["12"]},
    "CREPITO": {"-": ["51"]},
    "CREPITVS": {"-": ["14"]},
    "CREPO": {"-": ["51"]},
    "CREPVNDIA": {"-": ["12"]},
    "CREPVSCVLVM": {"-": ["12"]},
    "CRESCENTIA": {"-": ["11"]},
    "CRESCO": {"-": ["53"]},
    "CRETA": {"-": ["11"], "N": ["11"]},
    "CRETATVS": {"-": ["21"]},
    "CRETENSES": {"N": ["13"]},
    "CRETEVS": {"-": ["21"]},
    "CRETICVS": {"-": ["21"], "A": ["21"], "N": ["12"]},
    "CRETIO": {"-": ["13"]},
    "CRETOSVS": {"-": ["21"]},
    "CRETVLA": {"-": ["11"]},
    "CREVSA": {"1N": ["11"], "2N": ["11"], "N": ["11"]},
    "CRIBRVM": {"-": ["12"]},
    "CRIMEN": {"-": ["13"]},
    "CRIMINALIS": {"-": ["24"]},
    "CRIMINATIO": {"-": ["13"]},
    "CRIMINATOR": {"-": ["13"]},
    "CRIMINO": {"-": ["51"]},
    "CRIMINOR": {"-": ["5J"]},
    "CRIMINOSE": {"-": ["60"]},
    "CRIMINOSVS": {"-": ["21"]},
    "CRINIO": {"-": ["54"]},
    "CRINIS": {"-": ["13"]},
    "CRINITVS": {"-": ["21"]},
    "CRIODOCIS": {"-": ["17"]},
    "CRISIS": {"-": ["17"]},
    "CRISPO": {"-": ["51"]},
    "CRISPVLVS": {"-": ["21"]},
    "CRISPVS": {"-": ["21"], "N": ["12"]},
    "CRISSO": {"-": ["51"]},
    "CRISTA": {"-": ["11"]},
    "CRISTATVS": {"-": ["21"]},
    "CRITICVS": {"1": ["12"]},
    "CRITOLAVS": {"N": ["12"]},
    "CROCEVS": {"-": ["21"]},
    "CROCINVM": {"-": ["12"]},
    "CROCINVS": {"-": ["21"]},
    "CROCODILVS": {"-": ["17"]},
    "CROCVM": {"-": ["12"]},
    "CROCVS": {"-": ["12"]},
    "CROTO": {"N": ["17"]},
    "CRVCIABILITER": {"-": ["60"]},
    "CRVCIATVS": {"-": ["14"]},
    "CRVCIO": {"-": ["51"]},
    "CRVDELIS": {"-": ["24"]},
    "CRVDELITAS": {"-": ["13"]},
    "CRVDELITER": {"-": ["60"]},
    "CRVDESCO": {"-": ["53"]},
    "CRVDITAS": {"-": ["13"]},
    "CRVDVS": {"-": ["21"]},
    "CRVENTE": {"-": ["6&"]},
    "CRVENTO": {"-": ["51"]},
    "CRVENTVS": {"-": ["21"]},
    "CRVMENA": {"-": ["11"]},
    "CRVOR": {"-": ["13"]},
    "CRVPPELLARII": {"-": ["12"]},
    "CRVS": {"-": ["13"]},
    "CRVSTA": {"-": ["11"]},
    "CRVSTO": {"-": ["51"]},
    "CRVSTVLARIVS": {"-": ["12"]},
    "CRVSTVLVM": {"-": ["12"]},
    "CRVSTVM": {"-": ["12"]},
    "CRVSTVMERIA": {"N": ["11"]},
    "CRVSTVMERIVM": {"N": ["12"]},
    "CRVSTVMINI": {"N": ["12"]},
    "CRVSTVMINVS": {"N": ["21"]},
    "CRVX": {"-": ["13"]},
    "CRYPTA": {"-": ["11"]},
    "CRYSTALLINA": {"-": ["12"]},
    "CRYSTALLINVS": {"-": ["21"]},
    "CVBICVLARIS": {"-": ["24"]},
    "CVBICVLARIVS": {"1": ["12"], "2": ["21"]},
    "CVBICVLATVS": {"-": ["21"]},
    "CVBICVLVM": {"-": ["12"]},
    "CVBICVS": {"-": ["21"]},
    "CVBILE": {"-": ["13"]},
    "CVBITAL": {"-": ["13"]},
    "CVBITALIS": {"-": ["24"]},
    "CVBITO": {"-": ["51"]},
    "CVBITVM": {"-": ["12"]},
    "CVBITVS": {"1": ["14"], "2": ["14"]},
    "CVBO": {"-": ["51"]},
    "CVBVS": {"-": ["12"]},
    "CVCVLLIO": {"-": ["13"]},
    "CVCVLLVS": {"-": ["12"]},
    "CVCVLVS": {"-": ["12"]},
    "CVCVMIS": {"-": ["13"]},
    "CVCVRBITA": {"-": ["11"]},
    "CVCVRBITIVVS": {"-": ["21"]},
    "CVDO": {"-": ["53"]},
    "CVICVIMODI": {"1": ["66"], "2": ["67"]},
    "CVIVS": {"1": ["46"], "2": ["47"]},
    "CVIVSMODI": {"1": ["66"], "2": ["67"]},
    "CVIVSQVEMODI": {"-": ["60"]},
    "CVLCITA": {"-": ["11"]},
    "CVLCO": {"-": ["51"]},
    "CVLEARIS": {"-": ["24"]},
    "CVLEVM": {"-": ["12"]},
    "CVLEVS": {"-": ["12"]},
    "CVLEX": {"-": ["13"]},
    "CVLIGNA": {"-": ["11"]},
    "CVLINA": {"-": ["11"]},
    "CVLMEN": {"-": ["13"]},
    "CVLMVS": {"-": ["12"]},
    "CVLO": {"-": ["53"]},
    "CVLPA": {"-": ["11"]},
    "CVLPABILIS": {"-": ["24"]},
    "CVLPATVS": {"-": ["21"]},
    "CVLPO": {"-": ["51"]},
    "CVLTA": {"-": ["12"]},
    "CVLTE": {"-": ["6&"]},
    "CVLTELLVS": {"-": ["12"]},
    "CVLTER": {"-": ["12"]},
    "CVLTIO": {"-": ["13"]},
    "CVLTO": {"-": ["51"]},
    "CVLTOR": {"-": ["13"]},
    "CVLTRIX": {"-": ["13"]},
    "CVLTVRA": {"-": ["11"]},
    "CVLTVS": {"1": ["14"], "2": ["21"]},
    "CVLVLLVS": {"-": ["12"]},
    "CVLVS": {"-": ["12"]},
    "CVM": {"1": ["60"], "2": ["70"], "3": ["82"]},
    "CVMAE": {"N": ["11"]},
    "CVMANVS": {"N": ["21"], "A": ["21"]},
    "CVMBO": {"-": ["53"]},
    "CVMCVMQVE": {"-": ["60"]},
    "CVMERA": {"1": ["11"], "-": ["11"]},
    "CVMINVM": {"-": ["12"]},
    "CVMPRIMIS": {"-": ["60"]},
    "CVMQVE": {"-": ["60"]},
    "CVMVLATE": {"-": ["60"]},
    "CVMVLATVS": {"-": ["21"]},
    "CVMVLO": {"-": ["51"]},
    "CVMVLVS": {"-": ["12"]},
    "CVNABVLA": {"-": ["12"]},
    "CVNAE": {"-": ["11"]},
    "CVNCTABVNDVS": {"-": ["21"]},
    "CVNCTANS": {"-": ["25"]},
    "CVNCTANTER": {"-": ["60"]},
    "CVNCTATIO": {"-": ["13"]},
    "CVNCTATOR": {"-": ["13"]},
    "CVNCTOR": {"-": ["5J"]},
    "CVNCTVS": {"-": ["21"]},
    "CVNEATIM": {"-": ["60"]},
    "CVNEO": {"-": ["51"]},
    "CVNEOLVS": {"-": ["12"]},
    "CVNEVS": {"-": ["12"]},
    "CVNICA": {"-": ["11"]},
    "CVNICVLOSVS": {"-": ["21"]},
    "CVNICVLVS": {"-": ["12"]},
    "CVNNVS": {"-": ["12"]},
    "CVPA": {"1": ["11"], "2": ["11"]},
    "CVPEDIA": {"-": ["11"]},
    "CVPEDO": {"-": ["13"]},
    "CVPIDE": {"-": ["60"]},
    "CVPIDITAS": {"-": ["13"]},
    "CVPIDO": {"-": ["13"], "N": ["13"]},
    "CVPIDVS": {"-": ["21"]},
    "CVPIENS": {"-": ["25"]},
    "CVPIO": {"-": ["55"]},
    "CVPITOR": {"-": ["13"]},
    "CVPITVM": {"-": ["12"]},
    "CVPRESSETVM": {"-": ["12"]},
    "CVPRESSEVS": {"-": ["21"]},
    "CVPRESSIFER": {"-": ["21"]},
    "CVPRESSVS": {"-": ["16"]},
    "CVPVLA": {"2": ["11"]},
    "CVR": {"1": ["66"], "2": ["67"], "-": ["6-"]},
    "CVRA": {"-": ["11"], "N": ["11"]},
    "CVRABILIS": {"-": ["24"]},
    "CVRATE": {"-": ["6&"]},
    "CVRATIO": {"-": ["13"]},
    "CVRATOR": {"-": ["13"]},
    "CVRATVS": {"1": ["14"], "2": ["21"]},
    "CVRCVLIO": {"-": ["13"], "N": ["13"]},
    "CVRES": {"N": ["13"]},
    "CVRIA": {"-": ["11"], "N": ["11"]},
    "CVRIALIS": {"-": ["24"]},
    "CVRIATIVS": {"N": ["12"]},
    "CVRIATVS": {"-": ["21"]},
    "CVRIDIVS": {"N": ["12"]},
    "CVRIO": {"N": ["13"], "-": ["22"]},
    "CVRIOSE": {"-": ["60"]},
    "CVRIOSVS": {"-": ["21"]},
    "CVRO": {"-": ["51"]},
    "CVRRICVLVM": {"-": ["12"]},
    "CVRRO": {"-": ["53"]},
    "CVRRVS": {"-": ["14"]},
    "CVRSIM": {"-": ["60"]},
    "CVRSITO": {"-": ["51"]},
    "CVRSO": {"-": ["51"]},
    "CVRSOR": {"-": ["13"], "N": ["13"]},
    "CVRSVS": {"-": ["14"]},
    "CVRTIVS": {"N": ["12"], "1N": ["12"], "2N": ["12"], "1": ["12"], "2": ["12"]},
    "CVRTO": {"-": ["51"]},
    "CVRTVS": {"-": ["21"]},
    "CVRVAMEN": {"-": ["13"]},
    "CVRVATVRA": {"-": ["11"]},
    "CVRVLIS": {"-": ["24"]},
    "CVRVO": {"-": ["51"]},
    "CVRVVS": {"-": ["21"]},
    "CVSO": {"-": ["51"]},
    "CVSPIS": {"-": ["13"]},
    "CVSTODIA": {"-": ["11"]},
    "CVSTODIO": {"-": ["54"]},
    "CVSTOS": {"-": ["13"], "N": ["13"]},
    "CVTICVLA": {"-": ["11"]},
    "CVTIO": {"-": ["55"]},
    "CVTIS": {"-": ["13"]},
    "CYATHVS": {"-": ["12"]},
    "CYBAEA": {"-": ["11"]},
    "CYCLADES": {"N": ["17", "13"]},
    "CYCLAS": {"-": ["13"], "N": ["17"]},
    "CYCLICOS": {"-": ["27"]},
    "CYCLOPS": {"N": ["17"]},
    "CYCLOTOS": {"-": ["27"]},
    "CYCNEVS": {"-": ["21"]},
    "CYCNVS": {"-": ["12"], "N": ["12"]},
    "CYLINDRVS": {"-": ["12"]},
    "CYMA": {"1": ["11"], "2": ["13"]},
    "CYMATIVM": {"-": ["12"]},
    "CYMBA": {"-": ["11"]},
    "CYMBALVM": {"-": ["17"]},
    "CYMBIVM": {"-": ["12"]},
    "CYNICE": {"-": ["17"]},
    "CYNICI": {"N": ["12"]},
    "CYNICVS": {"N": ["21"], "A": ["21"]},
    "CYPARISSVS": {"-": ["12"]},
    "CYPRIVS": {"-": ["21"], "N": ["21"], "A": ["21"]},
    "CYPRVS": {"N": ["17"]},
    "CYRVS": {"N": ["12"]},
    "CYTISVS": {"-": ["12"]},
    "CYZICENI": {"N": ["12"]},
    "CYZICVS": {"N": ["17"]},
    "DAEDALVS": {"-": ["21"], "N": ["12"]},
    "DAEMON": {"-": ["17"]},
    "DAMA": {"-": ["11"], "N": ["11"]},
    "DAMNAS": {"-": ["26"]},
    "DAMNATIO": {"-": ["13"]},
    "DAMNATORIVS": {"-": ["21"]},
    "DAMNATVS": {"1": ["14"], "2": ["21"]},
    "DAMNO": {"-": ["51"]},
    "DAMNOSE": {"-": ["60"]},
    "DAMNOSVS": {"-": ["21"]},
    "DAMNVM": {"-": ["12"]},
    "DAPALIS": {"-": ["24"]},
    "DAPS": {"-": ["13"]},
    "DAREVS": {"N": ["12"]},
    "DATIO": {"-": ["13"]},
    "DATO": {"-": ["51"]},
    "DATOR": {"-": ["13"]},
    "DATVM": {"-": ["12"]},
    "DE": {"-": ["70"]},
    "DEA": {"-": ["11"], "N": ["11"]},
    "DEACINO": {"-": ["51"]},
    "DEALBO": {"-": ["51"]},
    "DEAMBVLO": {"-": ["51"]},
    "DEBACCHOR": {"-": ["5J"]},
    "DEBELLO": {"-": ["51"]},
    "DEBENS": {"-": ["13"]},
    "DEBEO": {"-": ["52"]},
    "DEBILIS": {"-": ["24"]},
    "DEBILITAS": {"-": ["13"]},
    "DEBILITATIO": {"-": ["13"]},
    "DEBILITO": {"-": ["51"]},
    "DEBITOR": {"-": ["13"]},
    "DEBITVM": {"-": ["12"]},
    "DECANTO": {"-": ["51"]},
    "DECARPO": {"-": ["53"]},
    "DECASTYLVS": {"-": ["27"]},
    "DECEDO": {"1": ["53"]},
    "DECEM": {"-": ["31"]},
    "DECEMBER": {"A": ["23"], "N": ["13"]},
    "DECEMPEDA": {"-": ["11"]},
    "DECEMVIR": {"-": ["12"]},
    "DECEMVIRALIS": {"-": ["24"]},
    "DECENNIS": {"-": ["24"]},
    "DECENS": {"-": ["25"]},
    "DECENTER": {"-": ["60"]},
    "DECENTIA": {"-": ["11"]},
    "DECEPTOR": {"-": ["13"]},
    "DECEPTORIVS": {"-": ["21"]},
    "DECERNO": {"-": ["53"]},
    "DECERPO": {"-": ["53"]},
    "DECERTO": {"1": ["51"]},
    "DECESSIO": {"-": ["13"]},
    "DECESSOR": {"-": ["13"]},
    "DECESSVS": {"-": ["14"]},
    "DECET": {"-": ["52"]},
    "DECIDO": {"1": ["53"], "2": ["53"]},
    "DECIES": {"-": ["36"]},
    "DECIMA": {"-": ["11"]},
    "DECIMANI": {"-": ["12"]},
    "DECIMANVS": {"1": ["12"], "2": ["21"]},
    "DECIMVS": {"-": ["32"], "N": ["12"]},
    "DECIPIO": {"-": ["55"]},
    "DECISIO": {"-": ["13"]},
    "DECLAMATIO": {"-": ["13"]},
    "DECLAMATOR": {"-": ["13"]},
    "DECLAMATORIVS": {"-": ["21"]},
    "DECLAMO": {"-": ["51"]},
    "DECLARATOR": {"-": ["13"]},
    "DECLARO": {"-": ["51"]},
    "DECLINATIO": {"-": ["13"]},
    "DECLINO": {"-": ["51"]},
    "DECLIVE": {"-": ["13"]},
    "DECLIVIS": {"-": ["24"]},
    "DECLIVITAS": {"-": ["13"]},
    "DECLIVVS": {"-": ["21"]},
    "DECOCTA": {"-": ["11"]},
    "DECOCTIOR": {"-": ["2&"]},
    "DECOCTOR": {"-": ["13"]},
    "DECOCTVM": {"-": ["12"], "D\u00e9coction": ["12"]},
    "DECOCTVS": {"-": ["21"]},
    "DECOLLO": {"-": ["51"]},
    "DECOLO": {"1": ["51"], "-": ["51"]},
    "DECOLOR": {"-": ["25"]},
    "DECOLORO": {"-": ["51"]},
    "DECONDO": {"-": ["53"]},
    "DECOQVO": {"-": ["53"]},
    "DECOR": {"-": ["13"], "N": ["13"]},
    "DECORATVS": {"O": ["21"], "-": ["21"]},
    "DECORE": {"-": ["60"]},
    "DECORO": {"-": ["51"]},
    "DECORVM": {"C": ["12"], "-": ["12"]},
    "DECORVS": {"-": ["21"]},
    "DECREPITVS": {"-": ["21"]},
    "DECRESCENTIA": {"-": ["11"]},
    "DECRESCO": {"-": ["53"]},
    "DECRETORIVS": {"-": ["21"]},
    "DECRETVM": {"-": ["12"]},
    "DECVMATES": {"-": ["24"]},
    "DECVMBO": {"-": ["53"]},
    "DECVRIA": {"-": ["11"]},
    "DECVRIALIS": {"-": ["24"]},
    "DECVRIATIO": {"-": ["13"]},
    "DECVRIATVS": {"-": ["14"]},
    "DECVRIO": {"1": ["13"], "2": ["51"]},
    "DECVRIONALIS": {"-": ["24"]},
    "DECVRIONATVS": {"-": ["14"]},
    "DECVRRO": {"-": ["53"]},
    "DECVRSIO": {"-": ["13"]},
    "DECVRSVS": {"-": ["14"]},
    "DECVRTO": {"-": ["51"]},
    "DECVS": {"-": ["13"]},
    "DECVSSATIM": {"-": ["60"]},
    "DECVSSATIO": {"-": ["13"]},
    "DECVSSIS": {"-": ["16"]},
    "DECVTIO": {"-": ["55"]},
    "DEDECET": {"-": ["52"]},
    "DEDECORO": {"-": ["51"]},
    "DEDECORVS": {"-": ["21"]},
    "DEDECVS": {"-": ["13"]},
    "DEDICATIO": {"-": ["13"]},
    "DEDICATVS": {"-": ["21"]},
    "DEDICO": {"1": ["51"]},
    "DEDIGNATIO": {"-": ["13"]},
    "DEDIGNOR": {"-": ["5J"]},
    "DEDISCO": {"-": ["53"]},
    "DEDITICIVS": {"1": ["12"], "2": ["21"]},
    "DEDITIO": {"-": ["13"]},
    "DEDITVS": {"-": ["21"]},
    "DEDO": {"-": ["53"]},
    "DEDOCEO": {"-": ["52"]},
    "DEDOLEO": {"-": ["52"]},
    "DEDVCO": {"-": ["53"]},
    "DEDVCTIO": {"-": ["13"]},
    "DEDVCTVS": {"1": ["14"], "2": ["21"]},
    "DEERRO": {"2": ["51"]},
    "DEFAECATVS": {"-": ["21"]},
    "DEFATIGATIO": {"-": ["13"]},
    "DEFATIGO": {"-": ["51"]},
    "DEFECTIO": {"-": ["13"]},
    "DEFECTOR": {"-": ["13"]},
    "DEFECTVS": {"1": ["14"], "2": ["21"]},
    "DEFENDO": {"-": ["53"]},
    "DEFENSIO": {"-": ["13"]},
    "DEFENSO": {"-": ["51"]},
    "DEFENSOR": {"-": ["13"]},
    "DEFERO": {"-": ["56"]},
    "DEFERVEFACIO": {"-": ["55"]},
    "DEFERVEO": {"-": ["52"]},
    "DEFESSVS": {"-": ["21"]},
    "DEFETISCOR": {"-": ["5L"]},
    "DEFICIO": {"-": ["55"]},
    "DEFIGO": {"-": ["53"]},
    "DEFINGO": {"-": ["53"]},
    "DEFINIO": {"-": ["54"]},
    "DEFINITE": {"-": ["60"]},
    "DEFINITIO": {"-": ["13"]},
    "DEFINITIVVS": {"-": ["21"]},
    "DEFINITVS": {"-": ["21"]},
    "DEFIO": {"-": ["56"]},
    "DEFLAGRO": {"-": ["51"]},
    "DEFLECTO": {"-": ["53"]},
    "DEFLEO": {"-": ["52"]},
    "DEFLORESCO": {"-": ["53"]},
    "DEFLVO": {"-": ["53"]},
    "DEFODIO": {"-": ["55"]},
    "DEFORMATIO": {"1": ["13"], "2": ["13"]},
    "DEFORMATVS": {"-": ["21"]},
    "DEFORMIS": {"-": ["24"]},
    "DEFORMITAS": {"-": ["13"]},
    "DEFORMO": {"1": ["51"], "2": ["51"]},
    "DEFRICO": {"-": ["51"]},
    "DEFRINGO": {"-": ["53"]},
    "DEFRVDO": {"-": ["51"]},
    "DEFRVTO": {"-": ["51"]},
    "DEFRVTVM": {"-": ["12"]},
    "DEFVGIO": {"-": ["55"]},
    "DEFVNCTVS": {"-": ["21"]},
    "DEFVNDO": {"2": ["53"]},
    "DEFVNGOR": {"-": ["5L"]},
    "DEFVTVO": {"-": ["53"]},
    "DEGENER": {"-": ["22"]},
    "DEGENERO": {"-": ["51"]},
    "DEGERO": {"-": ["53"]},
    "DEGLVBO": {"-": ["53"]},
    "DEGO": {"-": ["53"]},
    "DEGRAVO": {"-": ["51"]},
    "DEGREDIOR": {"-": ["5N"]},
    "DEGVSTO": {"-": ["51"]},
    "DEHAVRIO": {"-": ["54"]},
    "DEHINC": {"-": ["60"]},
    "DEHISCO": {"-": ["53"]},
    "DEHONESTAMENTVM": {"-": ["12"]},
    "DEHONESTO": {"-": ["51"]},
    "DEHORTOR": {"-": ["5J"]},
    "DEICIO": {"-": ["55"]},
    "DEIECTIO": {"-": ["13"]},
    "DEIECTVS": {"1": ["14"], "2": ["21"]},
    "DEILLIVS": {"N": ["12"]},
    "DEINCEPS": {"-": ["60"]},
    "DEINDE": {"-": ["60"]},
    "DEIVNGO": {"-": ["53"]},
    "DELABOR": {"2": ["5L"]},
    "DELAPIDO": {"-": ["51"]},
    "DELASSO": {"-": ["51"]},
    "DELATIO": {"-": ["13"]},
    "DELATOR": {"-": ["13"]},
    "DELECTABILIS": {"-": ["24"]},
    "DELECTATIO": {"-": ["13"]},
    "DELECTO": {"-": ["51"]},
    "DELECTVS": {"-": ["14"]},
    "DELEGATIO": {"-": ["13"]},
    "DELEGATVM": {"-": ["12"]},
    "DELEGO": {"1": ["51"]},
    "DELENIMENTVM": {"-": ["12"]},
    "DELENIO": {"-": ["54"]},
    "DELEO": {"-": ["52"]},
    "DELIACVS": {"N": ["21"], "A": ["21"]},
    "DELIBERATIO": {"-": ["13"]},
    "DELIBERATVS": {"-": ["21"]},
    "DELIBERO": {"-": ["51"]},
    "DELIBO": {"-": ["51"]},
    "DELIBRO": {"-": ["51"]},
    "DELIBVO": {"-": ["53"]},
    "DELICATA": {"-": ["11"]},
    "DELICATE": {"-": ["60"]},
    "DELICATVS": {"1": ["12"], "2": ["21"]},
    "DELICIA": {"1": ["11"], "2": ["11"]},
    "DELICIO": {"-": ["55"]},
    "DELICIOLVM": {"-": ["12"]},
    "DELICIVM": {"-": ["12"]},
    "DELICTVM": {"-": ["12"]},
    "DELICVLVS": {"-": ["21"]},
    "DELIGO": {"2": ["51"], "3": ["53"]},
    "DELINQVO": {"-": ["53"]},
    "DELIQVESCO": {"-": ["53"]},
    "DELIQVIAE": {"-": ["11"]},
    "DELIRATIO": {"-": ["13"]},
    "DELIRO": {"-": ["51"]},
    "DELIRVS": {"-": ["21"]},
    "DELITEO": {"-": ["52"]},
    "DELITESCO": {"-": ["53"]},
    "DELIVS": {"N": ["21"], "A": ["21"]},
    "DELPHI": {"N": ["12"]},
    "DELPHICVS": {"N": ["21"], "A": ["21"]},
    "DELPHIN": {"-": ["17"], "N": ["13"]},
    "DELPHINVS": {"-": ["12"]},
    "DELVBRVM": {"-": ["12"]},
    "DELVDO": {"-": ["53"]},
    "DELVMBIS": {"-": ["24"]},
    "DELVMBO": {"-": ["51"]},
    "DELVS": {"N": ["17"]},
    "DELVTO": {"-": ["51"]},
    "DEMANO": {"-": ["51"]},
    "DEMARATVS": {"N": ["12"]},
    "DEMENS": {"-": ["25"]},
    "DEMENTER": {"-": ["60"]},
    "DEMENTIA": {"-": ["11"]},
    "DEMENTIO": {"-": ["54"]},
    "DEMEREO": {"-": ["52"]},
    "DEMEREOR": {"-": ["5K"]},
    "DEMERGO": {"-": ["53"]},
    "DEMETO": {"2": ["53"]},
    "DEMETRIVS": {"N": ["12"]},
    "DEMIGRO": {"-": ["51"]},
    "DEMINVO": {"-": ["53"]},
    "DEMINVTIO": {"-": ["13"]},
    "DEMINVTVS": {"-": ["21"]},
    "DEMIROR": {"-": ["5J"]},
    "DEMISSE": {"-": ["60"]},
    "DEMISSIO": {"-": ["13"]},
    "DEMISSVS": {"-": ["21"]},
    "DEMITTO": {"-": ["53"]},
    "DEMIVRGVS": {"-": ["12"]},
    "DEMNO": {"-": ["51"]},
    "DEMO": {"-": ["53"]},
    "DEMOCRITVS": {"N": ["12"]},
    "DEMOLIOR": {"-": ["5M"]},
    "DEMOLITIO": {"-": ["13"]},
    "DEMOLITOR": {"-": ["13"]},
    "DEMONSTRATIO": {"-": ["13"]},
    "DEMONSTRO": {"-": ["51"]},
    "DEMORDEO": {"-": ["52"]},
    "DEMORIOR": {"-": ["5N"]},
    "DEMOROR": {"1": ["5J"]},
    "DEMOVEO": {"-": ["52"]},
    "DEMVM": {"-": ["60"]},
    "DEMVTO": {"2": ["51"]},
    "DENARIVM": {"-": ["12"]},
    "DENARIVS": {"1": ["12"], "2": ["21"]},
    "DENARRO": {"-": ["51"]},
    "DENATO": {"-": ["51"]},
    "DENEGO": {"-": ["51"]},
    "DENI": {"-": ["33"]},
    "DENIGRO": {"-": ["51"]},
    "DENIQVE": {"-": ["60"]},
    "DENOMINO": {"-": ["51"]},
    "DENORMO": {"-": ["51"]},
    "DENOTATVS": {"1": ["14"], "2": ["21"]},
    "DENOTO": {"-": ["51"]},
    "DENS": {"-": ["13"]},
    "DENSE": {"-": ["6-"]},
    "DENSEO": {"-": ["52"]},
    "DENSO": {"-": ["51"]},
    "DENSVS": {"-": ["21"], "N": ["12"]},
    "DENTALE": {"-": ["13"]},
    "DENTATVS": {"-": ["21"], "N": ["12"]},
    "DENTICVLVS": {"-": ["12"]},
    "DENVBO": {"-": ["53"]},
    "DENVDO": {"-": ["51"]},
    "DENVNTIATIO": {"-": ["13"]},
    "DENVNTIO": {"-": ["51"]},
    "DENVO": {"1": ["51"], "2": ["53"]},
    "DEORSVM": {"-": ["60"]},
    "DEPACISCOR": {"-": ["5L"]},
    "DEPACO": {"-": ["51"]},
    "DEPANGO": {"-": ["53"]},
    "DEPASCO": {"-": ["53"]},
    "DEPASCOR": {"-": ["5L"]},
    "DEPECTO": {"-": ["53"]},
    "DEPECVLATOR": {"-": ["13"]},
    "DEPECVLOR": {"-": ["5J"]},
    "DEPELLO": {"-": ["53"]},
    "DEPENDEO": {"-": ["52"]},
    "DEPENDO": {"-": ["53"]},
    "DEPERDITVS": {"-": ["21"]},
    "DEPERDO": {"-": ["53"]},
    "DEPEREO": {"1": ["56"]},
    "DEPETIGO": {"-": ["13"]},
    "DEPILO": {"1": ["51"]},
    "DEPINGO": {"-": ["53"]},
    "DEPLANGO": {"-": ["53"]},
    "DEPLEO": {"-": ["52"]},
    "DEPLEXVS": {"-": ["21"]},
    "DEPLORATIO": {"-": ["13"]},
    "DEPLORO": {"-": ["51"]},
    "DEPLVO": {"-": ["53"]},
    "DEPONO": {"-": ["53"]},
    "DEPOPVLATIO": {"-": ["13"]},
    "DEPOPVLO": {"-": ["51"]},
    "DEPOPVLOR": {"-": ["5J"]},
    "DEPORTATIO": {"-": ["13"]},
    "DEPORTO": {"-": ["51"]},
    "DEPOSCO": {"-": ["53"]},
    "DEPOSITVM": {"-": ["12"]},
    "DEPOSTVLO": {"-": ["51"]},
    "DEPRAVATIO": {"-": ["13"]},
    "DEPRAVO": {"-": ["51"]},
    "DEPRECABVNDVS": {"-": ["21"]},
    "DEPRECATIO": {"-": ["13"]},
    "DEPRECATOR": {"-": ["13"]},
    "DEPRECATRIX": {"-": ["13"]},
    "DEPRECOR": {"-": ["5J"]},
    "DEPREHENDO": {"-": ["53"]},
    "DEPRESSE": {"-": ["6&"]},
    "DEPRESSIO": {"-": ["13"]},
    "DEPRESSVS": {"-": ["21"]},
    "DEPRIMO": {"-": ["53"]},
    "DEPROELIOR": {"-": ["5J"]},
    "DEPROMO": {"-": ["53"]},
    "DEPROPERO": {"-": ["51"]},
    "DEPSO": {"-": ["53"]},
    "DEPSTICIVS": {"-": ["21"]},
    "DEPVDET": {"-": ["52"]},
    "DEPVGIS": {"-": ["24"]},
    "DEPVGNATIO": {"-": ["13"]},
    "DEPVGNO": {"-": ["51"]},
    "DEPVNGO": {"-": ["53"]},
    "DEPVRGO": {"-": ["51"]},
    "DEPVTO": {"-": ["51"]},
    "DERADO": {"-": ["53"]},
    "DERELINQVO": {"-": ["53"]},
    "DEREPENTE": {"-": ["60"]},
    "DEREPO": {"-": ["53"]},
    "DERIDEO": {"-": ["52"]},
    "DERIDICVLVM": {"-": ["12"]},
    "DERIDICVLVS": {"-": ["21"]},
    "DERIGESCO": {"-": ["53"]},
    "DERIGO": {"-": ["53"]},
    "DERIPIO": {"-": ["55"]},
    "DERISOR": {"-": ["13"]},
    "DERISVS": {"-": ["14"]},
    "DERIVO": {"-": ["51"]},
    "DEROGO": {"-": ["51"]},
    "DERVMPO": {"-": ["53"]},
    "DERVPTA": {"-": ["12"]},
    "DERVPTVS": {"-": ["21"]},
    "DESAEVIO": {"-": ["54"]},
    "DESCENDO": {"-": ["53"]},
    "DESCENSVS": {"-": ["14"]},
    "DESCISCO": {"-": ["53"]},
    "DESCRIBO": {"-": ["53"]},
    "DESCRIPTA": {"-": ["12"]},
    "DESCRIPTIO": {"-": ["13"]},
    "DESECO": {"-": ["51"]},
    "DESERO": {"2": ["53"], "3": ["53"]},
    "DESERTA": {"-": ["12"]},
    "DESERTO": {"-": ["51"]},
    "DESERTOR": {"-": ["13"]},
    "DESERTVM": {"-": ["12"]},
    "DESERTVS": {"-": ["21"]},
    "DESERVIO": {"-": ["54"]},
    "DESES": {"-": ["25"]},
    "DESIDEO": {"-": ["52"]},
    "DESIDERABILIS": {"-": ["24"]},
    "DESIDERATIO": {"-": ["13"]},
    "DESIDERIVM": {"-": ["12"]},
    "DESIDERO": {"-": ["51"]},
    "DESIDIA": {"1": ["11"]},
    "DESIDIOSE": {"-": ["60"]},
    "DESIDIOSVS": {"-": ["21"]},
    "DESIDO": {"-": ["53"]},
    "DESIGNATIO": {"-": ["13"]},
    "DESIGNATOR": {"-": ["13"]},
    "DESIGNO": {"-": ["51"]},
    "DESILIO": {"-": ["54"]},
    "DESINO": {"-": ["53"]},
    "DESIPIENTIA": {"-": ["11"]},
    "DESIPIO": {"-": ["55"]},
    "DESISTO": {"-": ["53"]},
    "DESOLO": {"-": ["51"]},
    "DESPECTATIO": {"-": ["13"]},
    "DESPECTO": {"-": ["51"]},
    "DESPECTVS": {"1": ["14"], "2": ["21"]},
    "DESPERATIO": {"-": ["13"]},
    "DESPERATVS": {"-": ["21"]},
    "DESPERO": {"-": ["51"]},
    "DESPICATVS": {"1": ["14"], "2": ["21"]},
    "DESPICIENTIA": {"-": ["11"]},
    "DESPICIO": {"-": ["55"]},
    "DESPOLIO": {"-": ["51"]},
    "DESPONDEO": {"-": ["52"]},
    "DESPVMO": {"-": ["51"]},
    "DESPVO": {"-": ["53"]},
    "DESTERTO": {"-": ["53"]},
    "DESTILLATIO": {"-": ["13"]},
    "DESTILLO": {"-": ["51"]},
    "DESTINA": {"-": ["11"]},
    "DESTINATIO": {"-": ["13"]},
    "DESTINATVM": {"-": ["12"]},
    "DESTINATVS": {"-": ["21"]},
    "DESTINEO": {"-": ["52"]},
    "DESTINO": {"-": ["51"]},
    "DESTITVO": {"-": ["53"]},
    "DESTRICTVS": {"-": ["21"]},
    "DESTRINGO": {"-": ["53"]},
    "DESTRVO": {"-": ["53"]},
    "DESVBITO": {"-": ["60"]},
    "DESVDO": {"-": ["51"]},
    "DESVESCO": {"-": ["53"]},
    "DESVETVDO": {"-": ["13"]},
    "DESVLTOR": {"-": ["13"]},
    "DESVM": {"1": ["56"]},
    "DESVMO": {"-": ["53"]},
    "DESVPER": {"-": ["60"]},
    "DESVRGO": {"-": ["53"]},
    "DETECTVS": {"-": ["21"]},
    "DETEGO": {"-": ["53"]},
    "DETENDO": {"-": ["53"]},
    "DETENTO": {"-": ["51"]},
    "DETERGEO": {"-": ["52"]},
    "DETERGO": {"-": ["53"]},
    "DETERIOR": {"-": ["2&"]},
    "DETERIVS": {"-": ["6&"]},
    "DETERMINO": {"-": ["51"]},
    "DETERO": {"-": ["53"]},
    "DETERREO": {"-": ["52"]},
    "DETESTABILIS": {"-": ["24"]},
    "DETESTATIO": {"-": ["13"]},
    "DETESTOR": {"-": ["5J"]},
    "DETEXO": {"-": ["53"]},
    "DETINEO": {"-": ["52"]},
    "DETONDEO": {"-": ["52"]},
    "DETORQVEO": {"-": ["52"]},
    "DETRACTIO": {"-": ["13"]},
    "DETRACTO": {"-": ["51"]},
    "DETRACTOR": {"-": ["13"]},
    "DETRAHO": {"-": ["53"]},
    "DETRECTO": {"-": ["51"]},
    "DETRIMENTOSVS": {"-": ["21"]},
    "DETRIMENTVM": {"-": ["12"]},
    "DETRVDO": {"-": ["53"]},
    "DETRVNCO": {"-": ["51"]},
    "DETVRBO": {"2": ["51"]},
    "DEVECTO": {"-": ["51"]},
    "DEVEHO": {"-": ["53"]},
    "DEVELLO": {"-": ["53"]},
    "DEVENEROR": {"-": ["5J"]},
    "DEVENIO": {"-": ["54"]},
    "DEVERRO": {"-": ["53"]},
    "DEVERSOR": {"1": ["13"], "2": ["5J"]},
    "DEVERSORIVM": {"-": ["12"]},
    "DEVERTICVLVM": {"-": ["12"]},
    "DEVERTO": {"-": ["53"]},
    "DEVEXVM": {"-": ["12"]},
    "DEVEXVS": {"-": ["21"]},
    "DEVIA": {"-": ["12"]},
    "DEVINCIO": {"-": ["54"]},
    "DEVINCO": {"-": ["53"]},
    "DEVINCTVS": {"-": ["21"]},
    "DEVITO": {"-": ["51"]},
    "DEVIVS": {"-": ["21"]},
    "DEVNX": {"-": ["13"]},
    "DEVOCO": {"-": ["51"]},
    "DEVOLO": {"2": ["51"]},
    "DEVOLVO": {"-": ["53"]},
    "DEVORO": {"-": ["51"]},
    "DEVOTI": {"-": ["12"]},
    "DEVOTIO": {"-": ["13"]},
    "DEVOTVS": {"-": ["21"]},
    "DEVOVEO": {"-": ["52"]},
    "DEVRO": {"-": ["53"]},
    "DEVS": {"-": ["12"], "N": ["12"]},
    "DEXO": {"N": ["17"]},
    "DEXTANS": {"-": ["13"]},
    "DEXTER": {"-": ["21"], "N": ["12"]},
    "DEXTERA": {"-": ["11"]},
    "DEXTERE": {"-": ["60"]},
    "DEXTRORSVM": {"-": ["60"]},
    "DEXTRORSVS": {"-": ["60"]},
    "DIADEMA": {"-": ["17"]},
    "DIADVMENOS": {"-": ["17"]},
    "DIAGONIOS": {"-": ["27"]},
    "DIAGRAMMA": {"-": ["17"]},
    "DIALECTICA": {"-": ["11"]},
    "DIALECTICVS": {"1": ["12"], "2": ["21"]},
    "DIALIS": {"N": ["24"], "A": ["24"]},
    "DIALOGVS": {"-": ["12"]},
    "DIAMETROS": {"-": ["17"]},
    "DIANA": {"N": ["11"]},
    "DIANIVM": {"N": ["12"]},
    "DIARIVM": {"-": ["12"]},
    "DIASTYLOS": {"-": ["27"]},
    "DIATHESIS": {"-": ["17"]},
    "DIATHYRA": {"-": ["17"]},
    "DIATONOS": {"-": ["27"]},
    "DIAVLOS": {"-": ["17"]},
    "DIAZOMA": {"-": ["17"]},
    "DICA": {"-": ["11"]},
    "DICACITAS": {"-": ["13"]},
    "DICAX": {"-": ["25"]},
    "DICHALCVM": {"-": ["12"]},
    "DICIO": {"-": ["13"]},
    "DICIS": {"-": ["16"]},
    "DICO": {"1": ["51"], "2": ["53"]},
    "DICROTA": {"-": ["11"]},
    "DICTATOR": {"-": ["13"]},
    "DICTATORIVS": {"-": ["21"]},
    "DICTATVRA": {"-": ["11"]},
    "DICTIO": {"-": ["13"]},
    "DICTITO": {"-": ["51"]},
    "DICTO": {"-": ["51"]},
    "DICTVM": {"-": ["12"]},
    "DIDO": {"-": ["53"], "N": ["13"]},
    "DIDVCO": {"-": ["53"]},
    "DIEQVINTI": {"-": ["60"]},
    "DIES": {"-": ["15"], "N": ["13"]},
    "DIESIS": {"-": ["13"]},
    "DIEZEVGMENOS": {"-": ["27"]},
    "DIFFAMO": {"-": ["51"]},
    "DIFFERCIO": {"-": ["54"]},
    "DIFFERENTIA": {"-": ["11"]},
    "DIFFERITAS": {"-": ["13"]},
    "DIFFERO": {"-": ["56"]},
    "DIFFERTVS": {"-": ["21"]},
    "DIFFICILE": {"-": ["60"]},
    "DIFFICILIS": {"-": ["24"]},
    "DIFFICILITER": {"-": ["60"]},
    "DIFFICVLTAS": {"-": ["13"]},
    "DIFFICVLTER": {"-": ["60"]},
    "DIFFIDENTIA": {"-": ["11"]},
    "DIFFIDO": {"-": ["5T"]},
    "DIFFINDO": {"-": ["53"]},
    "DIFFINGO": {"-": ["53"]},
    "DIFFITEOR": {"-": ["5K"]},
    "DIFFLVO": {"-": ["53"]},
    "DIFFRINGO": {"-": ["53"]},
    "DIFFVGIO": {"-": ["55"]},
    "DIFFVNDO": {"2": ["53"]},
    "DIFFVSILIS": {"-": ["24"]},
    "DIFFVSIO": {"-": ["13"]},
    "DIFFVSVS": {"-": ["21"]},
    "DIFFVTVO": {"-": ["53"]},
    "DIGERO": {"-": ["53"]},
    "DIGITVLVS": {"-": ["12"]},
    "DIGITVS": {"-": ["12"]},
    "DIGNATIO": {"-": ["13"]},
    "DIGNE": {"-": ["60"]},
    "DIGNITAS": {"-": ["13"]},
    "DIGNO": {"-": ["51"]},
    "DIGNOR": {"-": ["5J"]},
    "DIGNOSCO": {"-": ["53"]},
    "DIGNVS": {"-": ["21"]},
    "DIGREDIOR": {"-": ["5N"]},
    "DIGRESSVS": {"-": ["14"]},
    "DIIVDICO": {"-": ["51"]},
    "DIIVNCTVS": {"-": ["21"]},
    "DIIVNGO": {"-": ["53"]},
    "DILABOR": {"2": ["5L"]},
    "DILACERO": {"-": ["51"]},
    "DILANIO": {"-": ["51"]},
    "DILARGIOR": {"-": ["5M"]},
    "DILATATIO": {"-": ["13"]},
    "DILATIO": {"-": ["13"]},
    "DILATO": {"-": ["51"]},
    "DILECTVS": {"1": ["14"], "2": ["21"]},
    "DILIGENS": {"-": ["25"]},
    "DILIGENTER": {"-": ["60"]},
    "DILIGENTIA": {"-": ["11"]},
    "DILIGO": {"3": ["53"]},
    "DILVCEO": {"-": ["52"]},
    "DILVCESCO": {"-": ["53"]},
    "DILVCIDE": {"-": ["60"]},
    "DILVCIDVS": {"-": ["21"]},
    "DILVDIVM": {"-": ["12"]},
    "DILVO": {"-": ["53"]},
    "DILVTVM": {"-": ["12"]},
    "DILVTVS": {"-": ["21"]},
    "DILVVIES": {"-": ["15"]},
    "DILVVIO": {"-": ["51"]},
    "DILVVIVM": {"-": ["12"]},
    "DIMACHAE": {"-": ["11"]},
    "DIMENSVS": {"-": ["21"]},
    "DIMETIOR": {"-": ["5M"]},
    "DIMICATIO": {"-": ["13"]},
    "DIMICO": {"-": ["51"]},
    "DIMIDIATVS": {"-": ["21"]},
    "DIMIDIO": {"-": ["51"]},
    "DIMIDIVM": {"-": ["12"]},
    "DIMIDIVS": {"-": ["21"]},
    "DIMINVTIO": {"-": ["13"]},
    "DIMISSIO": {"-": ["13"]},
    "DIMITTO": {"-": ["53"]},
    "DIMOEROS": {"-": ["27"]},
    "DIMOVEO": {"-": ["52"]},
    "DINOSCO": {"-": ["53"]},
    "DINVMERO": {"1": ["51"]},
    "DIO": {"N": ["17"]},
    "DIOCLES": {"N": ["17"]},
    "DIODORVS": {"N": ["12"]},
    "DIOGENES": {"N": ["17"]},
    "DIOGNETVS": {"N": ["12"]},
    "DIONYSIA": {"1N": ["11"], "N": ["11"]},
    "DIONYSIARCHVS": {"N": ["12"]},
    "DIONYSIVS": {"N": ["12"]},
    "DIONYSODORVS": {"N": ["12"]},
    "DIOPTRA": {"-": ["11"]},
    "DIOTA": {"-": ["11"]},
    "DIPECHYAEOS": {"-": ["27"]},
    "DIPISCOR": {"-": ["5L"]},
    "DIPLASION": {"-": ["27"]},
    "DIPLINTHIVS": {"-": ["21"]},
    "DIPONDIVM": {"-": ["12"]},
    "DIPONDIVS": {"-": ["12"]},
    "DIPTEROS": {"-": ["27"]},
    "DIRA": {"-": ["12"], "N": ["11"]},
    "DIRAE": {"-": ["11"]},
    "DIRECTE": {"-": ["60"]},
    "DIRECTIO": {"-": ["13"]},
    "DIRECTO": {"-": ["60"]},
    "DIRECTVM": {"1": ["12"], "2": ["60"]},
    "DIRECTVRA": {"-": ["11"]},
    "DIRECTVS": {"-": ["21"]},
    "DIREPTIO": {"-": ["13"]},
    "DIREPTOR": {"-": ["13"]},
    "DIRIGO": {"-": ["53"]},
    "DIRIMO": {"-": ["53"]},
    "DIRIPIO": {"-": ["55"]},
    "DIRVM": {"-": ["60"]},
    "DIRVMPO": {"-": ["53"]},
    "DIRVO": {"-": ["53"]},
    "DIRVS": {"-": ["21"]},
    "DIS": {"-": ["25"], "N": ["13"]},
    "DISCEDO": {"1": ["53"]},
    "DISCENS": {"-": ["13"]},
    "DISCEPTATIO": {"-": ["13"]},
    "DISCEPTATOR": {"-": ["13"]},
    "DISCEPTO": {"-": ["51"]},
    "DISCERNO": {"-": ["53"]},
    "DISCERPO": {"-": ["53"]},
    "DISCESSIO": {"-": ["13"]},
    "DISCESSVS": {"-": ["14"]},
    "DISCIDIVM": {"-": ["12"]},
    "DISCIDO": {"2": ["53"]},
    "DISCINCTVS": {"-": ["21"]},
    "DISCINDO": {"-": ["53"]},
    "DISCINGO": {"-": ["53"]},
    "DISCIPLINA": {"-": ["11"]},
    "DISCIPLINOSVS": {"-": ["21"]},
    "DISCIPVLA": {"-": ["11"]},
    "DISCIPVLVS": {"-": ["12"]},
    "DISCITO": {"1": ["51"]},
    "DISCLVDO": {"-": ["53"]},
    "DISCO": {"-": ["53"]},
    "DISCOLOR": {"-": ["25"]},
    "DISCONVENIO": {"-": ["54"]},
    "DISCOQVO": {"-": ["53"]},
    "DISCORDIA": {"-": ["11"], "N": ["11"]},
    "DISCORDIOSVS": {"-": ["21"]},
    "DISCORDO": {"-": ["51"]},
    "DISCORS": {"-": ["25"]},
    "DISCREPANTIA": {"-": ["11"]},
    "DISCREPITO": {"-": ["51"]},
    "DISCREPO": {"-": ["51"]},
    "DISCRESCO": {"-": ["53"]},
    "DISCRIBO": {"-": ["53"]},
    "DISCRIMEN": {"-": ["13"]},
    "DISCRIMINO": {"-": ["51"]},
    "DISCRVCIO": {"-": ["51"]},
    "DISCVMBO": {"-": ["53"]},
    "DISCVPIO": {"-": ["55"]},
    "DISCVRRO": {"-": ["53"]},
    "DISCVRSATIO": {"-": ["13"]},
    "DISCVRSO": {"-": ["51"]},
    "DISCVRSVS": {"-": ["14"]},
    "DISCVS": {"-": ["12"]},
    "DISCVTIO": {"-": ["55"]},
    "DISDIAPENTE": {"-": ["17"]},
    "DISDIATESSARON": {"-": ["17"]},
    "DISDO": {"-": ["53"]},
    "DISERTE": {"-": ["60"]},
    "DISERTVS": {"-": ["21"]},
    "DISHIASCO": {"-": ["53"]},
    "DISICIO": {"-": ["55"]},
    "DISIECTO": {"-": ["51"]},
    "DISIECTVS": {"-": ["14"]},
    "DISPANDO": {"2": ["53"]},
    "DISPAR": {"-": ["25"]},
    "DISPARATIO": {"-": ["13"]},
    "DISPARO": {"2": ["51"]},
    "DISPECTO": {"-": ["51"]},
    "DISPECTVS": {"-": ["14"]},
    "DISPELLO": {"-": ["53"]},
    "DISPENDIVM": {"-": ["12"]},
    "DISPENDO": {"-": ["53"]},
    "DISPENSATIO": {"-": ["13"]},
    "DISPENSATOR": {"-": ["13"]},
    "DISPENSO": {"-": ["51"]},
    "DISPERDO": {"-": ["53"]},
    "DISPEREO": {"1": ["56"]},
    "DISPERGO": {"-": ["53"]},
    "DISPERSE": {"-": ["60"]},
    "DISPERTIO": {"-": ["54"]},
    "DISPESCO": {"-": ["53"]},
    "DISPICIO": {"-": ["55"]},
    "DISPLICENTIA": {"-": ["11"]},
    "DISPLICEO": {"-": ["52"]},
    "DISPLODO": {"-": ["53"]},
    "DISPLVVIATVS": {"-": ["21"]},
    "DISPONO": {"-": ["53"]},
    "DISPOSITE": {"-": ["60"]},
    "DISPOSITIO": {"-": ["13"]},
    "DISPOSITVM": {"-": ["12"]},
    "DISPOSITVRA": {"-": ["11"]},
    "DISPOSITVS": {"-": ["21"], "2": ["21"], "1": ["12"]},
    "DISPVNGO": {"-": ["53"]},
    "DISPVTABILIS": {"-": ["24"]},
    "DISPVTATIO": {"-": ["13"]},
    "DISPVTATIVNCVLA": {"-": ["11"]},
    "DISPVTO": {"-": ["51"]},
    "DISQVIRO": {"-": ["53"]},
    "DISQVISITIO": {"-": ["13"]},
    "DISSAEPIO": {"-": ["54"]},
    "DISSAEPTIO": {"-": ["13"]},
    "DISSAEPTVM": {"-": ["12"]},
    "DISSEMINO": {"-": ["51"]},
    "DISSENSIO": {"-": ["13"]},
    "DISSENTIO": {"-": ["54"]},
    "DISSERO": {"1": ["51"], "2": ["53"], "3": ["53"]},
    "DISSERPO": {"-": ["53"]},
    "DISSERTATIO": {"-": ["13"]},
    "DISSERTIO": {"-": ["13"]},
    "DISSERTO": {"-": ["51"]},
    "DISSICIO": {"-": ["55"]},
    "DISSIDEO": {"-": ["52"]},
    "DISSIGNATOR": {"-": ["13"]},
    "DISSIGNO": {"-": ["51"]},
    "DISSILIO": {"-": ["54"]},
    "DISSIMILIS": {"-": ["24"]},
    "DISSIMILITER": {"-": ["60"]},
    "DISSIMILITVDO": {"-": ["13"]},
    "DISSIMVLANTER": {"-": ["60"]},
    "DISSIMVLATIO": {"-": ["13"]},
    "DISSIMVLATOR": {"-": ["13"]},
    "DISSIMVLO": {"-": ["51"]},
    "DISSIPABILIS": {"-": ["24"]},
    "DISSIPO": {"-": ["51"]},
    "DISSOCIABILIS": {"-": ["24"]},
    "DISSOCIATIO": {"-": ["13"]},
    "DISSOCIO": {"-": ["51"]},
    "DISSOLVBILIS": {"-": ["24"]},
    "DISSOLVO": {"-": ["53"]},
    "DISSOLVTE": {"-": ["60"]},
    "DISSOLVTIO": {"-": ["13"]},
    "DISSOLVTVS": {"-": ["21"]},
    "DISSONO": {"1": ["51"]},
    "DISSONVS": {"-": ["21"]},
    "DISSVADEO": {"-": ["52"]},
    "DISSVASIO": {"-": ["13"]},
    "DISSVASOR": {"-": ["13"]},
    "DISSVLTO": {"-": ["51"]},
    "DISSVO": {"-": ["53"]},
    "DISTABESCO": {"-": ["53"]},
    "DISTANTIA": {"-": ["11"]},
    "DISTENDO": {"-": ["53"]},
    "DISTENTO": {"-": ["51"]},
    "DISTENTVS": {"1": ["14"], "2": ["21"], "3": ["21"]},
    "DISTERMINO": {"-": ["51"]},
    "DISTERO": {"-": ["53"]},
    "DISTICHON": {"-": ["17"]},
    "DISTINCTE": {"-": ["60"]},
    "DISTINCTIO": {"-": ["13"]},
    "DISTINCTVS": {"1": ["14"], "2": ["21"]},
    "DISTINEO": {"-": ["52"]},
    "DISTINGVO": {"-": ["53"]},
    "DISTO": {"-": ["51"]},
    "DISTORQVEO": {"-": ["52"]},
    "DISTORTVS": {"-": ["21"]},
    "DISTRACTIO": {"-": ["13"]},
    "DISTRACTVS": {"-": ["21"]},
    "DISTRAHO": {"-": ["53"]},
    "DISTRIBVO": {"-": ["53"]},
    "DISTRIBVTIO": {"-": ["13"]},
    "DISTRICTIO": {"-": ["13"]},
    "DISTRICTVS": {"-": ["21"]},
    "DISTRINGO": {"-": ["53"]},
    "DISTVRBO": {"2": ["51"]},
    "DITESCO": {"-": ["53"]},
    "DITHYRAMBVS": {"-": ["12"]},
    "DITO": {"-": ["51"]},
    "DIV": {"-": ["60"]},
    "DIVA": {"-": ["11"], "N": ["11"]},
    "DIVARICO": {"-": ["51"]},
    "DIVDICO": {"-": ["51"]},
    "DIVELLO": {"-": ["53"]},
    "DIVENDO": {"-": ["53"]},
    "DIVERBERO": {"2": ["51"]},
    "DIVERSE": {"-": ["60"]},
    "DIVERSITAS": {"-": ["13"]},
    "DIVERSORIVM": {"-": ["12"]},
    "DIVERSVM": {"-": ["12"]},
    "DIVERSVS": {"-": ["21"]},
    "DIVERTO": {"-": ["53"]},
    "DIVES": {"-": ["22"]},
    "DIVIDO": {"-": ["53"]},
    "DIVIDVVS": {"-": ["21"]},
    "DIVINA": {"1": ["11"], "2": ["12"]},
    "DIVINATIO": {"-": ["13"]},
    "DIVINITAS": {"-": ["13"]},
    "DIVINITVS": {"-": ["60"]},
    "DIVINO": {"-": ["51"]},
    "DIVINVM": {"-": ["12"]},
    "DIVINVS": {"1": ["12"], "2": ["21"]},
    "DIVISIO": {"-": ["13"]},
    "DIVISOR": {"-": ["13"]},
    "DIVISVS": {"1": ["14"], "2": ["21"]},
    "DIVITIAE": {"-": ["11"]},
    "DIVORTIVM": {"-": ["12"]},
    "DIVRNVM": {"N": ["12"], "-": ["12"]},
    "DIVRNVS": {"-": ["21"]},
    "DIVS": {"-": ["21"], "A": ["21"]},
    "DIVTINVS": {"-": ["21"]},
    "DIVTVRNITAS": {"-": ["13"]},
    "DIVTVRNVS": {"-": ["21"]},
    "DIVVLGO": {"1": ["51"]},
    "DIVVLSIO": {"-": ["13"]},
    "DIVVM": {"-": ["12"]},
    "DIVVS": {"1": ["12"], "2": ["21"], "A": ["21"]},
    "DO": {"-": ["53", "51"], "1": ["51"], "2": ["53"]},
    "DOCEO": {"-": ["52"]},
    "DOCIBILIS": {"-": ["24"]},
    "DOCILIS": {"-": ["24"], "N": ["13"]},
    "DOCIMVS": {"N": ["12"]},
    "DOCIS": {"-": ["13"]},
    "DOCTE": {"-": ["60"]},
    "DOCTOR": {"-": ["13"]},
    "DOCTRINA": {"-": ["11"]},
    "DOCTVS": {"-": ["21"]},
    "DOCVMEN": {"-": ["13"]},
    "DOCVMENTVM": {"-": ["12"]},
    "DODRA": {"-": ["11"]},
    "DODRANS": {"-": ["13"]},
    "DOGMA": {"-": ["17"]},
    "DOLABELLA": {"N": ["11"]},
    "DOLABRA": {"-": ["11"]},
    "DOLENS": {"-": ["25"]},
    "DOLEO": {"-": ["52"]},
    "DOLESCO": {"-": ["53"]},
    "DOLIOLVM": {"-": ["12"]},
    "DOLITO": {"-": ["51"]},
    "DOLIVM": {"-": ["12"]},
    "DOLO": {"-": ["51"]},
    "DOLOR": {"-": ["13"], "N": ["13"]},
    "DOLOSE": {"-": ["60"]},
    "DOLOSVS": {"-": ["21"]},
    "DOLVS": {"-": ["12"]},
    "DOMABILIS": {"-": ["24"]},
    "DOMESTICVS": {"1": ["12"], "2": ["21"]},
    "DOMICILIVM": {"-": ["12"]},
    "DOMINA": {"-": ["11"]},
    "DOMINANS": {"1": ["13"], "2": ["25"]},
    "DOMINATIO": {"-": ["13"]},
    "DOMINATOR": {"-": ["13"]},
    "DOMINATRIX": {"-": ["13"]},
    "DOMINATVS": {"-": ["14"]},
    "DOMINICVS": {"-": ["21"]},
    "DOMINIVM": {"-": ["12"]},
    "DOMINOR": {"-": ["5J"]},
    "DOMINVS": {"-": ["12"]},
    "DOMITIVS": {"N": ["12"], "A": ["21"]},
    "DOMITO": {"-": ["51"]},
    "DOMITOR": {"-": ["13"]},
    "DOMITRIX": {"-": ["13"]},
    "DOMO": {"-": ["51"]},
    "DOMVNCVLA": {"-": ["11"]},
    "DOMVS": {"-": ["16"]},
    "DONARIA": {"-": ["12"]},
    "DONATICVS": {"-": ["21"]},
    "DONATIO": {"-": ["13"]},
    "DONATIVVM": {"-": ["12"]},
    "DONATOR": {"-": ["13"]},
    "DONEC": {"-": ["82"]},
    "DONO": {"-": ["51"]},
    "DONVM": {"-": ["12"]},
    "DORCAS": {"-": ["13"]},
    "DORICE": {"-": ["60"]},
    "DORMIO": {"-": ["54"]},
    "DORMITO": {"-": ["51"]},
    "DORON": {"-": ["17"]},
    "DOROTHEVS": {"N": ["12"]},
    "DORSVM": {"-": ["12"]},
    "DORYPHORA": {"-": ["11"]},
    "DORYPHOROS": {"-": ["17"]},
    "DOS": {"-": ["13"]},
    "DOTALIS": {"-": ["24"]},
    "DOTATVS": {"-": ["21"]},
    "DOTO": {"-": ["51"], "N": ["11"]},
    "DRACHMA": {"-": ["11"]},
    "DRACO": {"-": ["13"]},
    "DREPANITANVS": {"N": ["21"], "A": ["21"]},
    "DROMAS": {"-": ["13"]},
    "DROPAX": {"-": ["13"]},
    "DRVIDAE": {"-": ["13"]},
    "DRVIDES": {"-": ["13"]},
    "DRVSILLA": {"N": ["11"]},
    "DRVSVS": {"N": ["12"]},
    "DVBIE": {"-": ["60"]},
    "DVBITABILIS": {"-": ["24"]},
    "DVBITANTER": {"-": ["60"]},
    "DVBITATIM": {"-": ["60"]},
    "DVBITATIO": {"-": ["13"]},
    "DVBITATIVVS": {"-": ["21"]},
    "DVBITATOR": {"-": ["13"]},
    "DVBITO": {"-": ["51"]},
    "DVBIVM": {"-": ["12"]},
    "DVBIVS": {"-": ["21"], "N": ["12"]},
    "DVCENI": {"-": ["33"]},
    "DVCENTESIMA": {"-": ["11"]},
    "DVCENTESIMVS": {"-": ["32"]},
    "DVCENTI": {"-": ["31"]},
    "DVCENTIES": {"-": ["36"]},
    "DVCO": {"-": ["53"]},
    "DVCTARIVS": {"-": ["21"]},
    "DVCTIO": {"-": ["13"]},
    "DVCTO": {"-": ["51"]},
    "DVCTOR": {"-": ["13"]},
    "DVCTVS": {"-": ["14"]},
    "DVDVM": {"-": ["60"]},
    "DVELLVM": {"-": ["12"]},
    "DVILIVS": {"N": ["12"]},
    "DVLCE": {"1": ["13"], "2": ["60"]},
    "DVLCEDO": {"-": ["13"]},
    "DVLCESCO": {"-": ["53"]},
    "DVLCIS": {"-": ["24"]},
    "DVM": {"1": ["60"], "2": ["82"]},
    "DVMETVM": {"-": ["12"]},
    "DVMMODO": {"-": ["82"]},
    "DVMOSVS": {"-": ["21"]},
    "DVMTAXAT": {"-": ["60"]},
    "DVMVS": {"-": ["12"]},
    "DVO": {"-": ["31"]},
    "DVODECIM": {"-": ["31"]},
    "DVODECIMVS": {"-": ["32"]},
    "DVODENI": {"-": ["33"]},
    "DVODEQVADRAGINTA": {"-": ["31"]},
    "DVODETRICESIMVS": {"-": ["32"]},
    "DVODETRICIES": {"-": ["36"]},
    "DVODETRIGINTA": {"-": ["31"]},
    "DVODEVICESIMVS": {"-": ["32"]},
    "DVODEVIGINTI": {"-": ["31"]},
    "DVOETVICESIMVS": {"-": ["32"]},
    "DVPLA": {"-": ["11"]},
    "DVPLEX": {"-": ["34", "22"]},
    "DVPLICATIO": {"-": ["13"]},
    "DVPLICITER": {"-": ["60"]},
    "DVPLICO": {"-": ["51"]},
    "DVPLO": {"1": ["51"], "2": ["60"]},
    "DVPLVM": {"-": ["12"]},
    "DVPLVS": {"-": ["21"]},
    "DVRABILIS": {"-": ["24"]},
    "DVRACINVS": {"-": ["21"]},
    "DVRAMEN": {"-": ["13"]},
    "DVRAMENTVM": {"-": ["12"]},
    "DVRATEVS": {"-": ["21"]},
    "DVRE": {"-": ["60"]},
    "DVREO": {"-": ["52"]},
    "DVRESCO": {"-": ["53"]},
    "DVRITER": {"-": ["60"]},
    "DVRITIA": {"-": ["11"]},
    "DVRITIES": {"-": ["15"]},
    "DVRITVDO": {"-": ["13"]},
    "DVRO": {"-": ["51"]},
    "DVRVM": {"-": ["12"]},
    "DVRVS": {"-": ["21"], "N": ["12"]},
    "DVVMVIR": {"-": ["12"]},
    "DVX": {"-": ["13"]},
    "DYNASTES": {"-": ["17"]},
    "DYRRACHIVM": {"N": ["12"]},
    "DYSEROS": {"-": ["17"]},
    "DYSPEPSIA": {"-": ["11"]},
    "EA": {"-": ["60"]},
    "EADEM": {"-": ["60"]},
    "EATENVS": {"-": ["60"]},
    "EBENVS": {"-": ["12"]},
    "EBIBO": {"2": ["53"]},
    "EBLANDIOR": {"-": ["5M"]},
    "EBRIETAS": {"-": ["13"]},
    "EBRIOSVS": {"-": ["21"]},
    "EBRIVS": {"-": ["21"]},
    "EBVLLIO": {"-": ["54"]},
    "EBVLVM": {"-": ["12"]},
    "EBVR": {"-": ["13"]},
    "EBVRNEVS": {"-": ["21"]},
    "ECCE": {"-": ["90"]},
    "ECCLESIA": {"-": ["11"]},
    "ECCLESIASTERION": {"-": ["17"]},
    "ECETRANI": {"N": ["12"]},
    "ECHEA": {"-": ["17"]},
    "ECHINVS": {"-": ["12"]},
    "ECHO": {"-": ["17"], "N": ["16"]},
    "ECNVBO": {"-": ["53"]},
    "ECPHORA": {"-": ["17"]},
    "ECQVANDO": {"-": ["67"]},
    "ECQVI": {"-": ["67"]},
    "ECQVID": {"-": ["67"]},
    "ECQVIS": {"-": ["47"]},
    "ECQVISNAM": {"-": ["47"]},
    "ECQVO": {"-": ["67"]},
    "ECTYPVS": {"-": ["21"]},
    "EDAX": {"-": ["25"]},
    "EDICO": {"2": ["53"]},
    "EDICTVM": {"-": ["12"]},
    "EDISCO": {"-": ["53"]},
    "EDISSERO": {"2": ["53"]},
    "EDITA": {"-": ["12"]},
    "EDITIO": {"-": ["13"]},
    "EDITVM": {"-": ["12"]},
    "EDITVS": {"1": ["14"], "2": ["21"]},
    "EDO": {"1": ["53"], "2": ["56"]},
    "EDOCEO": {"-": ["52"]},
    "EDOMO": {"-": ["51"]},
    "EDORMIO": {"-": ["54"]},
    "EDVCATIO": {"-": ["13"]},
    "EDVCATOR": {"-": ["13"]},
    "EDVCO": {"1": ["51"], "2": ["53"]},
    "EDVCTIO": {"-": ["13"]},
    "EDVLIS": {"-": ["24"]},
    "EDVRO": {"-": ["51"]},
    "EDVRVS": {"-": ["21"]},
    "EFFARCIO": {"-": ["54"]},
    "EFFATVM": {"-": ["12"]},
    "EFFECTVM": {"-": ["12"]},
    "EFFECTVS": {"1": ["14"], "2": ["21"]},
    "EFFEMINATE": {"-": ["60"]},
    "EFFEMINATVS": {"-": ["21"]},
    "EFFEMINO": {"-": ["51"]},
    "EFFERATVS": {"-": ["21"]},
    "EFFERCIO": {"-": ["54"]},
    "EFFERO": {"1": ["51"], "2": ["56"]},
    "EFFERVEO": {"-": ["52"]},
    "EFFERVESCO": {"-": ["53"]},
    "EFFERVO": {"-": ["53"]},
    "EFFERVS": {"-": ["21"]},
    "EFFETVS": {"-": ["21"]},
    "EFFICACITER": {"-": ["6&"]},
    "EFFICAX": {"-": ["25"]},
    "EFFICIENS": {"1": ["13"], "2": ["25"]},
    "EFFICIENTIA": {"-": ["11"]},
    "EFFICIO": {"-": ["55"]},
    "EFFIGIA": {"-": ["11"]},
    "EFFIGIES": {"-": ["15"]},
    "EFFINGO": {"-": ["53"]},
    "EFFLAGITATIO": {"-": ["13"]},
    "EFFLAGITATVS": {"-": ["14"]},
    "EFFLAGITO": {"-": ["51"]},
    "EFFLIGO": {"-": ["53"]},
    "EFFLO": {"-": ["51"]},
    "EFFLORESCO": {"-": ["53"]},
    "EFFLVO": {"-": ["53"]},
    "EFFLVVIVM": {"-": ["12"]},
    "EFFODIO": {"-": ["55"]},
    "EFFOR": {"-": ["5J"]},
    "EFFRACTARIVS": {"-": ["12"]},
    "EFFRENATVS": {"-": ["21"]},
    "EFFRENO": {"-": ["51"]},
    "EFFRENVS": {"-": ["21"]},
    "EFFRICO": {"-": ["51"]},
    "EFFRINGO": {"-": ["53"]},
    "EFFVGIO": {"-": ["55"]},
    "EFFVGIVM": {"-": ["12"]},
    "EFFVLGEO": {"-": ["52"]},
    "EFFVNDO": {"2": ["53"]},
    "EFFVNSVS": {"-": ["21"]},
    "EFFVSE": {"-": ["60"]},
    "EFFVSIO": {"-": ["13"]},
    "EFFVSVS": {"-": ["21"]},
    "EFFVTIO": {"-": ["54"]},
    "EFFVTVO": {"-": ["53"]},
    "EGELIDVS": {"-": ["21"]},
    "EGENS": {"C": ["25"], "-": ["25"]},
    "EGENVS": {"-": ["21"]},
    "EGEO": {"-": ["52"]},
    "EGERIA": {"N": ["11"]},
    "EGERIVS": {"N": ["12"]},
    "EGERO": {"-": ["53"]},
    "EGESTAS": {"-": ["13"], "N": ["13"]},
    "EGIGNO": {"-": ["53"]},
    "EGNATIVS": {"N": ["12"]},
    "EGO": {"-": ["41"]},
    "EGREDIOR": {"-": ["5N"]},
    "EGREGIE": {"-": ["60"]},
    "EGREGIVM": {"-": ["12"]},
    "EGREGIVS": {"-": ["21"]},
    "EGRESSVS": {"-": ["14"]},
    "EHEV": {"-": ["90"]},
    "EI": {"-": ["90"]},
    "EIA": {"-": ["90"]},
    "EIACVLOR": {"-": ["5J"]},
    "EICIO": {"-": ["55"]},
    "EIECTAMENTVM": {"-": ["12"]},
    "EIECTIO": {"-": ["13"]},
    "EIECTO": {"-": ["51"]},
    "EIECTVS": {"-": ["14"]},
    "EIVLATIO": {"-": ["13"]},
    "EIVLATVS": {"-": ["14"]},
    "EIVLO": {"-": ["51"]},
    "EIVRATIO": {"-": ["13"]},
    "EIVRO": {"-": ["51"]},
    "EIVSMODI": {"-": ["60"]},
    "ELABOR": {"2": ["5L"]},
    "ELABORO": {"-": ["51"]},
    "ELAEOTHESIVM": {"-": ["12"]},
    "ELANGVESCO": {"-": ["53"]},
    "ELARGIOR": {"-": ["5M"]},
    "ELATE": {"1": ["17"], "2": ["60"]},
    "ELATIO": {"-": ["13"]},
    "ELATRO": {"2": ["51"]},
    "ELATVS": {"-": ["21"]},
    "ELAVO": {"-": ["51"]},
    "ELECTIO": {"-": ["13"]},
    "ELECTO": {"1": ["51"], "2": ["51"]},
    "ELECTRVM": {"-": ["12"]},
    "ELECTVS": {"1": ["14"], "2": ["21"]},
    "ELEGANS": {"-": ["25"]},
    "ELEGANTER": {"-": ["60"]},
    "ELEGANTIA": {"-": ["11"]},
    "ELEGEON": {"-": ["17"]},
    "ELEGI": {"-": ["12"]},
    "ELEGIDION": {"-": ["17"]},
    "ELEGO": {"1": ["51"]},
    "ELEMENTA": {"-": ["12"]},
    "ELEMENTARIVS": {"-": ["21"]},
    "ELEMENTVM": {"-": ["12"]},
    "ELENCHVS": {"-": ["12"]},
    "ELEPHANTVS": {"-": ["12"]},
    "ELEPHAS": {"-": ["13"]},
    "ELEVO": {"1": ["51"]},
    "ELICIO": {"-": ["55"]},
    "ELICIVS": {"N": ["12"]},
    "ELIDO": {"-": ["53"]},
    "ELIGO": {"3": ["53"]},
    "ELIMINO": {"-": ["51"]},
    "ELINGVIS": {"-": ["24"]},
    "ELIQVESCO": {"-": ["53"]},
    "ELIQVO": {"-": ["51"]},
    "ELIS": {"N": ["17"]},
    "ELISIO": {"-": ["13"]},
    "ELIXVS": {"-": ["21"]},
    "ELLIPSIS": {"-": ["17"]},
    "ELLYCHNIVM": {"-": ["12"]},
    "ELOCO": {"-": ["51"]},
    "ELOGIVM": {"-": ["12"]},
    "ELOQVENS": {"-": ["25"]},
    "ELOQVENTIA": {"-": ["11"]},
    "ELOQVIVM": {"-": ["12"]},
    "ELOQVOR": {"-": ["5L"]},
    "ELORINI": {"N": ["12"]},
    "ELORINVS": {"N": ["21"], "A": ["21"]},
    "ELORVS": {"N": ["12"]},
    "ELVACRVM": {"-": ["12"]},
    "ELVCEO": {"-": ["52"]},
    "ELVCTOR": {"-": ["5J"]},
    "ELVCVBRO": {"-": ["51"]},
    "ELVDO": {"-": ["53"]},
    "ELVMBIS": {"-": ["24"]},
    "ELVO": {"-": ["53"]},
    "ELVTVS": {"-": ["21"]},
    "ELVVIES": {"-": ["15"]},
    "EMACERATVS": {"-": ["21"]},
    "EMANCIPO": {"-": ["51"]},
    "EMANO": {"-": ["51"]},
    "EMARCEO": {"-": ["52"]},
    "EMARCESCO": {"-": ["53"]},
    "EMAX": {"-": ["25"]},
    "EMBATER": {"-": ["17"]},
    "EMBLEMA": {"-": ["17"]},
    "EMBOLIVM": {"-": ["12"]},
    "EMBOLVS": {"-": ["12"]},
    "EMENDABILIS": {"-": ["24"]},
    "EMENDATE": {"-": ["60"]},
    "EMENDATIO": {"-": ["13"]},
    "EMENDATOR": {"-": ["13"]},
    "EMENDATVS": {"-": ["21"]},
    "EMENDICATICIVS": {"-": ["21"]},
    "EMENDO": {"-": ["51"]},
    "EMENTIOR": {"-": ["5M"]},
    "EMERCOR": {"-": ["5J"]},
    "EMEREO": {"-": ["52"]},
    "EMEREOR": {"-": ["5K"]},
    "EMERGO": {"-": ["53"]},
    "EMERITVM": {"-": ["12"]},
    "EMERITVS": {"-": ["21"]},
    "EMERSVS": {"-": ["14"]},
    "EMETIOR": {"-": ["5M"]},
    "EMETO": {"2": ["53"]},
    "EMICO": {"-": ["51"]},
    "EMIGRO": {"-": ["51"]},
    "EMINENS": {"-": ["25"]},
    "EMINEO": {"-": ["52"]},
    "EMINVS": {"-": ["60"]},
    "EMIROR": {"-": ["5J"]},
    "EMISSARIVS": {"-": ["12"]},
    "EMISSVS": {"-": ["14"]},
    "EMITTO": {"-": ["53"]},
    "EMO": {"-": ["53"]},
    "EMODEROR": {"-": ["5J"]},
    "EMODVLOR": {"-": ["5J"]},
    "EMOLIOR": {"-": ["5M"]},
    "EMOLLIO": {"-": ["54"]},
    "EMOLO": {"-": ["53"]},
    "EMOLVMENTVM": {"-": ["12"]},
    "EMORIOR": {"-": ["5N"]},
    "EMOVEO": {"-": ["52"]},
    "EMPLASTRVM": {"-": ["12"]},
    "EMPLECTOS": {"-": ["27"]},
    "EMPORIVM": {"-": ["12"]},
    "EMPTIO": {"-": ["13"]},
    "EMPTITO": {"-": ["51"]},
    "EMPTOR": {"-": ["13"]},
    "EMVLGEO": {"-": ["52"]},
    "EMVNDO": {"-": ["51"]},
    "EMVNGO": {"-": ["53"]},
    "EMVNIO": {"2": ["54"]},
    "EN": {"-": ["90"]},
    "ENARRABILIS": {"-": ["24"]},
    "ENARRATIO": {"-": ["13"]},
    "ENARRO": {"-": ["51"]},
    "ENASCOR": {"-": ["5L"]},
    "ENATO": {"-": ["51"]},
    "ENAVIGO": {"-": ["51"]},
    "ENCARPI": {"-": ["12"]},
    "ENCELADOS": {"-": ["17"]},
    "ENCHYTVM": {"-": ["12"]},
    "ENCLIMA": {"-": ["17"]},
    "ENCYCLIOS": {"-": ["27"]},
    "ENDROMIS": {"-": ["17"]},
    "ENECO": {"-": ["51"]},
    "ENERVATVS": {"A": ["21"], "-": ["21"]},
    "ENERVIS": {"-": ["24"]},
    "ENERVO": {"-": ["51"]},
    "ENGVINI": {"N": ["12"]},
    "ENGVINVS": {"N": ["21"], "A": ["21"]},
    "ENIM": {"1": ["60"], "2": ["81"]},
    "ENIMVERO": {"-": ["60"]},
    "ENITEO": {"-": ["52"]},
    "ENITESCO": {"-": ["53"]},
    "ENITOR": {"2": ["5L"]},
    "ENIXE": {"-": ["60"]},
    "ENIXVS": {"1": ["14"], "2": ["21"]},
    "ENNIVS": {"N": ["12"]},
    "ENO": {"-": ["51"]},
    "ENODATIO": {"-": ["13"]},
    "ENODIS": {"-": ["24"]},
    "ENODO": {"-": ["51"]},
    "ENORMIS": {"-": ["24"]},
    "ENORMITAS": {"-": ["13"]},
    "ENOTESCO": {"-": ["53"]},
    "ENS": {"-": ["13"]},
    "ENSIGER": {"-": ["21"]},
    "ENSIS": {"-": ["13"]},
    "ENTASIS": {"-": ["17"]},
    "ENTELLINI": {"N": ["12"]},
    "ENTELLINVS": {"N": ["21"], "A": ["21"]},
    "ENTHEVS": {"-": ["21"]},
    "ENVCLEATVS": {"-": ["21"]},
    "ENVCLEO": {"-": ["51"]},
    "ENVMERATIO": {"-": ["13"]},
    "ENVMERO": {"1": ["51"]},
    "ENVNTIATIVVS": {"-": ["21"]},
    "ENVNTIATVM": {"-": ["12"]},
    "ENVNTIO": {"-": ["51"]},
    "ENVTRIO": {"-": ["54"]},
    "EO": {"1": ["56"], "3": ["60"]},
    "EODEM": {"-": ["60"]},
    "EOVS": {"N": ["21"], "A": ["21"]},
    "EPAEETIS": {"-": ["17"]},
    "EPAGON": {"-": ["57"]},
    "EPHEBEVM": {"-": ["12"]},
    "EPHEBVS": {"-": ["12"]},
    "EPHECTOS": {"-": ["27"]},
    "EPHEMERIDAE": {"-": ["11"]},
    "EPHEMERIS": {"-": ["17"]},
    "EPHESII": {"N": ["12"]},
    "EPHESIVS": {"N": ["21"], "A": ["21"]},
    "EPHESVS": {"N": ["17"]},
    "EPHIPPIATVS": {"-": ["21"]},
    "EPHIPPIVM": {"-": ["12"]},
    "EPIBATA": {"-": ["11"]},
    "EPIBATHRA": {"-": ["17"]},
    "EPICRATES": {"N": ["17"]},
    "EPICVRVS": {"N": ["12"]},
    "EPIDIMOEROS": {"-": ["27"]},
    "EPIDROMVS": {"-": ["12"]},
    "EPIGRAMMA": {"-": ["17"]},
    "EPIGRVS": {"-": ["12"]},
    "EPIMENIVS": {"-": ["21"]},
    "EPIPEMPTOS": {"-": ["27"]},
    "EPIRHEDIVM": {"-": ["12"]},
    "EPIRVS": {"N": ["17"]},
    "EPISCAENOS": {"-": ["17"]},
    "EPISCOPVS": {"-": ["12"]},
    "EPISTATES": {"-": ["17"]},
    "EPISTOLA": {"-": ["11"], "N": ["11"]},
    "EPISTOLIVM": {"-": ["12"]},
    "EPISTOMIVM": {"-": ["12"]},
    "EPISTYLIVM": {"-": ["12"]},
    "EPITONIVM": {"-": ["12"]},
    "EPITOXIS": {"-": ["17"]},
    "EPITRITOS": {"-": ["27"]},
    "EPITYRVM": {"-": ["12"]},
    "EPIZYGIS": {"-": ["17"]},
    "EPOS": {"-": ["17"]},
    "EPOTO": {"-": ["51"]},
    "EPVLAE": {"-": ["11"]},
    "EPVLOR": {"-": ["5J"]},
    "EPVLVM": {"-": ["12"]},
    "EQVA": {"-": ["11"]},
    "EQVES": {"-": ["13"]},
    "EQVESTER": {"-": ["23"], "A": ["13"]},
    "EQVESTRIA": {"Si\u00e8ges": ["13"], "-": ["13"]},
    "EQVIDEM": {"-": ["60"]},
    "EQVILE": {"-": ["13"]},
    "EQVINVS": {"-": ["21"]},
    "EQVITABILIS": {"-": ["24"]},
    "EQVITATVS": {"1": ["14"], "2": ["14"]},
    "EQVITO": {"-": ["51"]},
    "EQVOR": {"-": ["5L"]},
    "EQVVLEVS": {"-": ["12"]},
    "EQVVS": {"-": ["12"]},
    "ERADO": {"-": ["53"]},
    "ERCEO": {"-": ["52"]},
    "ERECTIO": {"-": ["13"]},
    "ERECTVS": {"-": ["21"]},
    "EREPO": {"-": ["53"]},
    "EREPTIO": {"-": ["13"]},
    "EREPTOR": {"-": ["13"]},
    "ERGA": {"-": ["70"]},
    "ERGASTVLVM": {"-": ["12"]},
    "ERGATA": {"-": ["11"]},
    "ERGO": {"1": ["70"], "2": ["81"]},
    "ERICIVS": {"-": ["12"]},
    "ERIGO": {"-": ["53"]},
    "ERIPHYLA": {"N": ["11"]},
    "ERIPIO": {"-": ["55"]},
    "ERISMA": {"-": ["16"]},
    "ERNEVM": {"-": ["12"]},
    "ERODO": {"-": ["53"]},
    "EROGATIO": {"-": ["13"]},
    "EROGO": {"-": ["51"]},
    "EROMENION": {"-": ["17"]},
    "EROS": {"2N": ["17"], "N": ["12"]},
    "ERRABVNDVS": {"-": ["21"]},
    "ERRATICVS": {"-": ["21"]},
    "ERRATIO": {"-": ["13"]},
    "ERRATVM": {"-": ["12"]},
    "ERRO": {"1": ["13"], "2": ["51"]},
    "ERROR": {"-": ["13"], "N": ["13"]},
    "ERVBEO": {"-": ["52"]},
    "ERVBESCO": {"-": ["53"]},
    "ERVCA": {"-": ["11"]},
    "ERVCTO": {"-": ["51"]},
    "ERVDERO": {"-": ["51"]},
    "ERVDIO": {"-": ["54"]},
    "ERVDITI": {"-": ["12"]},
    "ERVDITIO": {"-": ["13"]},
    "ERVDITVLVS": {"-": ["21"]},
    "ERVDITVS": {"-": ["21"]},
    "ERVMPO": {"-": ["53"]},
    "ERVNCO": {"-": ["51"]},
    "ERVO": {"-": ["53"]},
    "ERVPTIO": {"-": ["13"]},
    "ERVVM": {"-": ["12"]},
    "ERYCINVS": {"N": ["21"], "A": ["21"]},
    "ERYCVS": {"N": ["12"]},
    "ERYMANTHIVS": {"N": ["21"], "A": ["21"]},
    "ERYTHRAE": {"N": ["11"]},
    "ESCA": {"-": ["11"]},
    "ESCARIVS": {"-": ["21"]},
    "ESCENDO": {"-": ["53"]},
    "ESCENSIO": {"-": ["13"]},
    "ESCENSVS": {"-": ["14"]},
    "ESCHARA": {"-": ["17"]},
    "ESITO": {"-": ["51"]},
    "ESQVILIAE": {"N": ["11"]},
    "ESQVILINVS": {"N": ["21"], "A": ["21"]},
    "ESSEDA": {"-": ["11"]},
    "ESSEDARIVS": {"1": ["12"], "2": ["21"]},
    "ESSEDVM": {"-": ["12"]},
    "ESSENTIA": {"-": ["11"]},
    "ESVRIO": {"1": ["13"], "2": ["54"]},
    "ESVRITIO": {"-": ["13"]},
    "ET": {"1": ["60"], "2": ["81"]},
    "ETENIM": {"-": ["81"]},
    "ETESIAE": {"-": ["11"]},
    "ETESIVS": {"-": ["21"], "N": ["12"]},
    "ETHOLOGIA": {"-": ["17"]},
    "ETIAM": {"-": ["60"]},
    "ETIAMNVNC": {"-": ["60"]},
    "ETIAMSI": {"-": ["82"]},
    "ETIAMTVM": {"-": ["60"]},
    "ETIAMTVNC": {"-": ["60"]},
    "ETRVRIA": {"N": ["11"]},
    "ETRVSCI": {"N": ["12"]},
    "ETRVSCVS": {"N": ["21"], "A": ["21"]},
    "ETSI": {"1": ["81"], "2": ["82"]},
    "EV": {"-": ["90"]},
    "EVADO": {"2": ["53"]},
    "EVAGATIO": {"-": ["13"]},
    "EVAGOR": {"2": ["5J"]},
    "EVALESCO": {"-": ["53"]},
    "EVANDER": {"N": ["12"]},
    "EVANESCO": {"-": ["53"]},
    "EVANIDVS": {"-": ["21"]},
    "EVANS": {"-": ["13"]},
    "EVBVLIDA": {"N": ["11"]},
    "EVBVLIDAS": {"N": ["17"]},
    "EVECTIO": {"-": ["13"]},
    "EVECTO": {"-": ["51"]},
    "EVECTVS": {"-": ["14"]},
    "EVEHO": {"-": ["53"]},
    "EVELLO": {"-": ["53"]},
    "EVENIO": {"-": ["54"]},
    "EVENTVM": {"-": ["12"]},
    "EVENTVRA": {"-": ["12"]},
    "EVENTVS": {"-": ["14"]},
    "EVERBERO": {"2": ["51"]},
    "EVERRICVLVM": {"-": ["12"]},
    "EVERRO": {"-": ["53"]},
    "EVERSIO": {"-": ["13"]},
    "EVERSOR": {"-": ["13"]},
    "EVERTO": {"-": ["53"]},
    "EVGANEI": {"N": ["12"]},
    "EVGE": {"-": ["90"]},
    "EVGENEVS": {"-": ["21"]},
    "EVHYIVS": {"-": ["17"]},
    "EVIDENS": {"-": ["25"]},
    "EVIDENTIA": {"-": ["11"]},
    "EVIGILO": {"-": ["51"]},
    "EVINCIO": {"-": ["54"]},
    "EVINCO": {"-": ["53"]},
    "EVIRO": {"-": ["51"]},
    "EVITABILIS": {"-": ["24"]},
    "EVITO": {"-": ["51"]},
    "EVMENIDAS": {"N": ["17"]},
    "EVNVCHVS": {"-": ["12"]},
    "EVOCATI": {"-": ["12"]},
    "EVOCATIO": {"-": ["13"]},
    "EVOCATOR": {"-": ["13"]},
    "EVOCATVS": {"-": ["21"]},
    "EVOCO": {"-": ["51"]},
    "EVOE": {"-": ["90"]},
    "EVOLO": {"2": ["51"]},
    "EVOLVO": {"-": ["53"]},
    "EVOMO": {"-": ["53"]},
    "EVPHRANOR": {"-": ["13"], "N": ["13"]},
    "EVPOLEMVS": {"N": ["12"]},
    "EVRIPVS": {"-": ["12"], "N": ["12"]},
    "EVROPA": {"N": ["11"]},
    "EVROVS": {"-": ["21"]},
    "EVRVS": {"-": ["12"], "N": ["12"]},
    "EVRYTHMIA": {"-": ["11"]},
    "EVSTYLOS": {"-": ["27"]},
    "EVTHVS": {"-": ["27"]},
    "EVTHYGRAMMOS": {"-": ["27"]},
    "EVTHYMIA": {"-": ["17"]},
    "EVVLGO": {"1": ["51"]},
    "EVVLSIO": {"-": ["13"]},
    "EX": {"-": ["70"]},
    "EXACERBO": {"-": ["51"]},
    "EXACTIO": {"-": ["13"]},
    "EXACTOR": {"-": ["13"]},
    "EXACTVS": {"1": ["14"], "2": ["21"]},
    "EXACVO": {"-": ["53"]},
    "EXADVERSVM": {"1": ["60"], "2": ["70"]},
    "EXAEDIFICO": {"-": ["51"]},
    "EXAEQVATIO": {"-": ["13"]},
    "EXAEQVO": {"-": ["51"]},
    "EXAESTVO": {"-": ["51"]},
    "EXAGGERO": {"1": ["51"]},
    "EXAGITO": {"-": ["51"]},
    "EXAMEN": {"-": ["13"]},
    "EXAMINATIO": {"-": ["13"]},
    "EXAMINO": {"-": ["51"]},
    "EXANCLO": {"-": ["51"]},
    "EXANIMIS": {"-": ["24"]},
    "EXANIMO": {"-": ["51"]},
    "EXANIMVS": {"-": ["21"]},
    "EXARDEO": {"-": ["52"]},
    "EXARDESCO": {"-": ["53"]},
    "EXAREO": {"-": ["52"]},
    "EXARESCO": {"-": ["53"]},
    "EXARMO": {"-": ["51"]},
    "EXARO": {"-": ["51"]},
    "EXASPERO": {"-": ["51"]},
    "EXAVCTORO": {"-": ["51"]},
    "EXAVDIO": {"-": ["54"]},
    "EXAVGEO": {"-": ["52"]},
    "EXAVGVRO": {"-": ["51"]},
    "EXCAECO": {"-": ["51"]},
    "EXCALCEO": {"-": ["52"]},
    "EXCANDEO": {"-": ["52"]},
    "EXCANDESCO": {"-": ["53"]},
    "EXCANTO": {"-": ["51"]},
    "EXCARNIFICO": {"-": ["51"]},
    "EXCAVO": {"1": ["51"]},
    "EXCEDO": {"1": ["53"]},
    "EXCELLENS": {"-": ["25"]},
    "EXCELLENTIA": {"-": ["11"]},
    "EXCELLO": {"-": ["53"]},
    "EXCELSE": {"-": ["6&"]},
    "EXCELSITAS": {"-": ["13"]},
    "EXCELSVM": {"-": ["12"]},
    "EXCELSVS": {"-": ["21"]},
    "EXCEPTIO": {"-": ["13"]},
    "EXCEPTIVNCVLA": {"-": ["11"]},
    "EXCEPTO": {"-": ["51"]},
    "EXCERNO": {"-": ["53"]},
    "EXCERPO": {"-": ["53"]},
    "EXCESSVS": {"-": ["14"]},
    "EXCIDIVM": {"1": ["12"], "2": ["12"]},
    "EXCIDO": {"1": ["53"], "2": ["53"]},
    "EXCIEO": {"-": ["52"]},
    "EXCIO": {"-": ["54"]},
    "EXCIPIO": {"-": ["55"]},
    "EXCISIO": {"-": ["13"]},
    "EXCITATVS": {"-": ["21"]},
    "EXCITO": {"1": ["51"]},
    "EXCLAMATIO": {"-": ["13"]},
    "EXCLAMO": {"-": ["51"]},
    "EXCLARO": {"-": ["51"]},
    "EXCLVDO": {"-": ["53"]},
    "EXCLVSIO": {"-": ["13"]},
    "EXCLVSVS": {"-": ["21"]},
    "EXCOGITO": {"-": ["51"]},
    "EXCOLO": {"1": ["51"], "2": ["53"]},
    "EXCOQVO": {"-": ["53"]},
    "EXCORS": {"-": ["25"]},
    "EXCREMENTVM": {"-": ["12"]},
    "EXCREO": {"-": ["51"]},
    "EXCRESCO": {"-": ["53"]},
    "EXCRETVS": {"-": ["21"]},
    "EXCRVCIO": {"-": ["51"]},
    "EXCVBIAE": {"-": ["11"]},
    "EXCVBITOR": {"-": ["13"]},
    "EXCVBITVS": {"-": ["14"]},
    "EXCVBO": {"-": ["51"]},
    "EXCVDO": {"-": ["53"]},
    "EXCVLCO": {"-": ["51"]},
    "EXCVRRO": {"-": ["53"]},
    "EXCVRSIO": {"-": ["13"]},
    "EXCVRSOR": {"-": ["13"]},
    "EXCVRSVS": {"-": ["14"]},
    "EXCVSATE": {"-": ["6-"]},
    "EXCVSATIO": {"-": ["13"]},
    "EXCVSATVS": {"-": ["21"]},
    "EXCVSO": {"-": ["51"]},
    "EXCVSSE": {"-": ["60"]},
    "EXCVSSVS": {"-": ["21"]},
    "EXCVTIO": {"-": ["55"]},
    "EXEDO": {"2": ["56"]},
    "EXEDRA": {"-": ["11"]},
    "EXEMPLAR": {"-": ["13"]},
    "EXEMPLVM": {"-": ["12"]},
    "EXEMPTVS": {"-": ["14"]},
    "EXEO": {"1": ["56"]},
    "EXERCEO": {"2": ["52"], "-": ["52"]},
    "EXERCITATE": {"-": ["6&"]},
    "EXERCITATIO": {"-": ["13"]},
    "EXERCITATVS": {"-": ["21"]},
    "EXERCITIO": {"-": ["13"]},
    "EXERCITIVM": {"-": ["12"]},
    "EXERCITO": {"-": ["51"]},
    "EXERCITVS": {"1": ["14"], "2": ["21"]},
    "EXESOR": {"-": ["13"]},
    "EXFLORESCO": {"-": ["53"]},
    "EXHALO": {"-": ["51"]},
    "EXHAVRIO": {"-": ["54"]},
    "EXHEREDO": {"-": ["51"]},
    "EXHIBEO": {"-": ["52"]},
    "EXHILARO": {"-": ["51"]},
    "EXHORREO": {"-": ["52"]},
    "EXHORRESCO": {"-": ["53"]},
    "EXHORTATIO": {"-": ["13"]},
    "EXHORTOR": {"-": ["5J"]},
    "EXIGO": {"-": ["53"]},
    "EXIGVE": {"-": ["60"]},
    "EXIGVITAS": {"-": ["13"]},
    "EXIGVO": {"-": ["60"]},
    "EXIGVVM": {"1": ["12"], "2": ["60"]},
    "EXIGVVS": {"-": ["21"]},
    "EXILIS": {"-": ["24"]},
    "EXILITAS": {"-": ["13"]},
    "EXILITER": {"-": ["60"]},
    "EXIMIE": {"-": ["60"]},
    "EXIMIVS": {"-": ["21"]},
    "EXIMO": {"-": ["53"]},
    "EXINANIO": {"-": ["54"]},
    "EXINDE": {"-": ["60"]},
    "EXISTIMATIO": {"-": ["13"]},
    "EXISTIMATOR": {"-": ["13"]},
    "EXISTIMO": {"-": ["51"]},
    "EXITIABILIS": {"-": ["24"]},
    "EXITIALIS": {"-": ["24"]},
    "EXITIOSVS": {"-": ["21"]},
    "EXITIVM": {"-": ["12"]},
    "EXITVS": {"-": ["14"]},
    "EXODIVM": {"-": ["12"]},
    "EXOLEO": {"-": ["52"]},
    "EXOLESCO": {"-": ["53"]},
    "EXOLETI": {"-": ["12"]},
    "EXOLETVS": {"-": ["21"]},
    "EXONERO": {"-": ["51"]},
    "EXOPTATVS": {"-": ["21"]},
    "EXOPTO": {"-": ["51"]},
    "EXORABILIS": {"-": ["24"]},
    "EXORDIOR": {"-": ["5M"]},
    "EXORDIVM": {"-": ["12"]},
    "EXORIOR": {"-": ["5M"]},
    "EXORNO": {"-": ["51"]},
    "EXORO": {"-": ["51"]},
    "EXORSA": {"-": ["12"]},
    "EXORSVS": {"-": ["14"]},
    "EXORTVS": {"-": ["14"]},
    "EXOS": {"-": ["25"]},
    "EXOSCVLOR": {"-": ["5J"]},
    "EXOSSO": {"-": ["51"]},
    "EXOSVS": {"-": ["21"]},
    "EXPALLEO": {"-": ["52"]},
    "EXPALLESCO": {"-": ["53"]},
    "EXPANDO": {"2": ["53"]},
    "EXPATRO": {"-": ["51"]},
    "EXPAVEO": {"-": ["52"]},
    "EXPAVESCO": {"-": ["53"]},
    "EXPEDIO": {"-": ["54"]},
    "EXPEDITE": {"-": ["60"]},
    "EXPEDITIO": {"-": ["13"]},
    "EXPEDITO": {"-": ["54"]},
    "EXPEDITVM": {"-": ["12"]},
    "EXPEDITVS": {"-": ["21"]},
    "EXPELLO": {"-": ["53"]},
    "EXPENDO": {"-": ["53"]},
    "EXPENSO": {"-": ["51"]},
    "EXPENSVM": {"-": ["12"]},
    "EXPERGEFACIO": {"-": ["55"]},
    "EXPERGISCOR": {"-": ["5L"]},
    "EXPERGO": {"-": ["53"]},
    "EXPERIENS": {"-": ["25"]},
    "EXPERIENTIA": {"-": ["11"]},
    "EXPERIMENTVM": {"-": ["12"]},
    "EXPERIOR": {"-": ["5M"]},
    "EXPERISCOR": {"-": ["5L"]},
    "EXPERS": {"-": ["25"]},
    "EXPERTIO": {"-": ["13"]},
    "EXPERTVS": {"-": ["21"]},
    "EXPETIBILIS": {"-": ["24"]},
    "EXPETO": {"-": ["53"]},
    "EXPILATIO": {"-": ["13"]},
    "EXPILO": {"2": ["51"]},
    "EXPINSO": {"-": ["53"]},
    "EXPIO": {"-": ["51"]},
    "EXPLANABILIS": {"-": ["24"]},
    "EXPLANATIO": {"-": ["13"]},
    "EXPLANATVS": {"-": ["21"]},
    "EXPLANO": {"-": ["51"]},
    "EXPLAVDO": {"-": ["53"]},
    "EXPLEO": {"-": ["52"]},
    "EXPLETVS": {"-": ["21"]},
    "EXPLICABILIS": {"-": ["24"]},
    "EXPLICATIO": {"-": ["13"]},
    "EXPLICATVS": {"1": ["14"], "2": ["21"]},
    "EXPLICITVS": {"2": ["21"], "-": ["21"]},
    "EXPLICO": {"-": ["51"]},
    "EXPLODO": {"-": ["53"]},
    "EXPLORATIO": {"-": ["13"]},
    "EXPLORATOR": {"-": ["13"]},
    "EXPLORATVS": {"-": ["21"]},
    "EXPLORO": {"-": ["51"]},
    "EXPOLIO": {"2": ["54"]},
    "EXPOLITIO": {"-": ["13"]},
    "EXPOLITVS": {"-": ["21"]},
    "EXPONO": {"-": ["53"]},
    "EXPORRIGO": {"2": ["53"]},
    "EXPORTATIO": {"-": ["13"]},
    "EXPORTO": {"-": ["51"]},
    "EXPOSCO": {"-": ["53"]},
    "EXPOSITVS": {"-": ["21"]},
    "EXPOSTVLATIO": {"-": ["13"]},
    "EXPOSTVLO": {"-": ["51"]},
    "EXPRESSIO": {"-": ["13"]},
    "EXPRESSVS": {"1": ["14"], "2": ["21"]},
    "EXPRIMO": {"-": ["53"]},
    "EXPROBRATIO": {"-": ["13"]},
    "EXPROBRATOR": {"-": ["13"]},
    "EXPROBRATRIX": {"-": ["13"]},
    "EXPROBRO": {"-": ["51"]},
    "EXPROMO": {"-": ["53"]},
    "EXPVGNABILIS": {"-": ["24"]},
    "EXPVGNATIO": {"-": ["13"]},
    "EXPVGNATOR": {"-": ["13"]},
    "EXPVGNO": {"-": ["51"]},
    "EXPVLTRIX": {"-": ["13"]},
    "EXPVNGO": {"-": ["53"]},
    "EXPVRGO": {"-": ["51"]},
    "EXQVIRO": {"-": ["53"]},
    "EXQVISITE": {"-": ["6&"]},
    "EXQVISITVS": {"-": ["21"]},
    "EXSANGVINATVS": {"-": ["21"]},
    "EXSANGVIS": {"-": ["24"]},
    "EXSANIO": {"-": ["51"]},
    "EXSATIO": {"2": ["51"]},
    "EXSATVRABILIS": {"-": ["24"]},
    "EXSATVRO": {"-": ["51"]},
    "EXSCALPO": {"-": ["51"]},
    "EXSCINDO": {"-": ["53"]},
    "EXSCREO": {"-": ["51"]},
    "EXSCRIBO": {"-": ["53"]},
    "EXSCVLPO": {"-": ["53"]},
    "EXSECO": {"-": ["51"]},
    "EXSECRABILIS": {"-": ["24"]},
    "EXSECRANDVS": {"-": ["21"]},
    "EXSECRATIO": {"-": ["13"]},
    "EXSECRATVS": {"-": ["21"]},
    "EXSECROR": {"-": ["5J"]},
    "EXSECVTIO": {"-": ["13"]},
    "EXSEQVIAE": {"-": ["11"]},
    "EXSEQVOR": {"-": ["5L"]},
    "EXSERO": {"2": ["53"]},
    "EXSERTO": {"-": ["51"]},
    "EXSERTVS": {"-": ["21"]},
    "EXSIBILO": {"-": ["51"]},
    "EXSICCESCO": {"-": ["53"]},
    "EXSICCO": {"-": ["51"]},
    "EXSILIO": {"-": ["54"]},
    "EXSILIVM": {"-": ["12"]},
    "EXSISTO": {"-": ["53"]},
    "EXSOLESCO": {"-": ["53"]},
    "EXSOLVO": {"-": ["53"]},
    "EXSOLVTIO": {"-": ["13"]},
    "EXSOMNIS": {"-": ["24"]},
    "EXSORBEO": {"-": ["52"]},
    "EXSORS": {"-": ["25"]},
    "EXSPARGO": {"2": ["53"]},
    "EXSPATIOR": {"-": ["5J"]},
    "EXSPECTATIO": {"-": ["13"]},
    "EXSPECTATVM": {"-": ["12"]},
    "EXSPECTATVS": {"-": ["21"]},
    "EXSPECTO": {"-": ["51"]},
    "EXSPES": {"-": ["26"]},
    "EXSPIRO": {"-": ["51"]},
    "EXSPOLIO": {"-": ["51"]},
    "EXSPVO": {"-": ["53"]},
    "EXSTANS": {"-": ["25"]},
    "EXSTANTIA": {"-": ["11"]},
    "EXSTERNO": {"-": ["51"]},
    "EXSTIMVLATOR": {"-": ["13"]},
    "EXSTIMVLO": {"-": ["51"]},
    "EXSTINCTOR": {"-": ["13"]},
    "EXSTINGVO": {"-": ["53"]},
    "EXSTIRPO": {"-": ["51"]},
    "EXSTO": {"-": ["51"]},
    "EXSTRVCTIO": {"-": ["13"]},
    "EXSTRVCTVS": {"-": ["21"]},
    "EXSTRVO": {"-": ["53"]},
    "EXSVCATVS": {"-": ["21"]},
    "EXSVCTVS": {"-": ["21"]},
    "EXSVDO": {"-": ["51"]},
    "EXSVGO": {"-": ["53"]},
    "EXSVL": {"-": ["13"]},
    "EXSVLO": {"-": ["51"]},
    "EXSVLTATIO": {"-": ["13"]},
    "EXSVLTIM": {"-": ["60"]},
    "EXSVLTO": {"-": ["51"]},
    "EXSVPERABILIS": {"-": ["24"]},
    "EXSVPERATIO": {"-": ["13"]},
    "EXSVPERO": {"-": ["51"]},
    "EXSVRDO": {"-": ["51"]},
    "EXSVRGO": {"-": ["53"]},
    "EXTA": {"-": ["12"]},
    "EXTEMPLO": {"-": ["60"]},
    "EXTEMPORALIS": {"-": ["24"]},
    "EXTENDO": {"-": ["53"]},
    "EXTENTIO": {"-": ["13"]},
    "EXTENTO": {"-": ["51"]},
    "EXTENTVS": {"1": ["14"], "2": ["21"]},
    "EXTENVATIO": {"-": ["13"]},
    "EXTENVO": {"-": ["51"]},
    "EXTER": {"-": ["21"]},
    "EXTERGEO": {"-": ["52"]},
    "EXTERMINO": {"-": ["51"]},
    "EXTERNI": {"-": ["12"]},
    "EXTERNVM": {"-": ["12"]},
    "EXTERNVS": {"-": ["21"]},
    "EXTERO": {"-": ["53"]},
    "EXTERREO": {"-": ["52"]},
    "EXTERRITVS": {"-": ["21"]},
    "EXTIMESCO": {"-": ["53"]},
    "EXTOLLO": {"-": ["53"]},
    "EXTORQVEO": {"-": ["52"]},
    "EXTORRIS": {"-": ["24"]},
    "EXTRA": {"1": ["60"], "2": ["70"]},
    "EXTRAHO": {"-": ["53"]},
    "EXTRANEVS": {"-": ["21"]},
    "EXTRAORDINARIVS": {"-": ["21"]},
    "EXTRARIVS": {"-": ["21"]},
    "EXTREMA": {"-": ["12"]},
    "EXTREMITAS": {"-": ["13"]},
    "EXTREMVM": {"A": ["12"], "-": ["12"]},
    "EXTRICO": {"-": ["51"]},
    "EXTRINSECVS": {"-": ["60"]},
    "EXTRVDO": {"-": ["53"]},
    "EXTVNDO": {"-": ["53"]},
    "EXTVRBO": {"2": ["51"]},
    "EXVBERO": {"-": ["51"]},
    "EXVLCERATIO": {"-": ["13"]},
    "EXVLCERO": {"-": ["51"]},
    "EXVLVLO": {"-": ["51"]},
    "EXVNDATIO": {"-": ["13"]},
    "EXVNDO": {"-": ["51"]},
    "EXVO": {"-": ["53"]},
    "EXVRGEO": {"-": ["52"]},
    "EXVRO": {"-": ["53"]},
    "EXVVIAE": {"-": ["11"]},
    "FABA": {"-": ["11"], "N": ["11"]},
    "FABAGINVS": {"-": ["21"]},
    "FABALIA": {"-": ["13"]},
    "FABARIVS": {"-": ["21"]},
    "FABELLA": {"-": ["11"]},
    "FABER": {"1": ["12"], "2": ["21"]},
    "FABIANVS": {"A": ["21"], "N": ["12"]},
    "FABIVS": {"A": ["21"], "N": ["12"]},
    "FABRICA": {"-": ["11"]},
    "FABRICATIO": {"-": ["13"]},
    "FABRICATOR": {"-": ["13"]},
    "FABRICO": {"-": ["51"]},
    "FABRICOR": {"-": ["5J"]},
    "FABRILIS": {"-": ["24"]},
    "FABVLA": {"1": ["11"], "2": ["11"]},
    "FABVLATOR": {"-": ["13"]},
    "FABVLOR": {"-": ["5J"]},
    "FABVLOSVS": {"-": ["21"]},
    "FABVLVS": {"-": ["12"]},
    "FACESSO": {"-": ["53"]},
    "FACETE": {"-": ["60"]},
    "FACETIAE": {"-": ["11"]},
    "FACETVS": {"-": ["21"]},
    "FACIES": {"-": ["15"]},
    "FACILE": {"-": ["60"]},
    "FACILIS": {"-": ["24"]},
    "FACILITAS": {"-": ["13"]},
    "FACINOROSVS": {"-": ["21"]},
    "FACINVS": {"-": ["13"]},
    "FACIO": {"-": ["55"]},
    "FACTIO": {"-": ["13"]},
    "FACTIOSVS": {"-": ["21"]},
    "FACTITO": {"-": ["51"]},
    "FACTOR": {"-": ["13"]},
    "FACTVM": {"-": ["12"]},
    "FACTVS": {"1": ["14"], "2": ["21"]},
    "FACVLA": {"-": ["11"]},
    "FACVLTAS": {"-": ["13"]},
    "FACVNDE": {"-": ["60"]},
    "FACVNDIA": {"-": ["11"]},
    "FACVNDVS": {"-": ["21"]},
    "FAECARIVS": {"-": ["21"]},
    "FAECATVS": {"-": ["21"]},
    "FAECVLA": {"-": ["11"]},
    "FAENVS": {"-": ["13"]},
    "FAEX": {"-": ["13"]},
    "FAGINEVS": {"-": ["21"]},
    "FAGINVS": {"-": ["21"]},
    "FAGVS": {"-": ["16"]},
    "FALA": {"-": ["11"]},
    "FALARICA": {"-": ["11"]},
    "FALCARIVS": {"-": ["12"]},
    "FALCATVS": {"-": ["21"]},
    "FALCIFER": {"-": ["21"]},
    "FALCVLA": {"-": ["11"], "N": ["11"]},
    "FALLACIA": {"-": ["11"]},
    "FALLAX": {"-": ["25"]},
    "FALLO": {"-": ["53"]},
    "FALSARIVS": {"-": ["12"]},
    "FALSIPARENS": {"-": ["25"]},
    "FALSO": {"1": ["51"], "2": ["60"]},
    "FALSVM": {"-": ["12"]},
    "FALSVS": {"-": ["21"]},
    "FALX": {"-": ["13"]},
    "FAMA": {"-": ["11"], "N": ["11"]},
    "FAMELICVS": {"1": ["12"], "2": ["21"]},
    "FAMES": {"-": ["13"], "N": ["13"]},
    "FAMILIA": {"-": ["11"]},
    "FAMILIARICVS": {"-": ["21"]},
    "FAMILIARIS": {"1": ["13"], "2": ["24"]},
    "FAMILIARITAS": {"-": ["13"]},
    "FAMILIARITER": {"-": ["60"]},
    "FAMOSVS": {"-": ["21"]},
    "FAMVL": {"-": ["12"]},
    "FAMVLA": {"-": ["11"]},
    "FAMVLARIS": {"-": ["24"]},
    "FAMVLATVS": {"-": ["14"]},
    "FAMVLOR": {"-": ["5J"]},
    "FAMVLVS": {"1": ["12"], "2": ["21"]},
    "FANATICVS": {"-": ["21"]},
    "FANDVS": {"-": ["21"]},
    "FANNIVS": {"N": ["12"]},
    "FANOR": {"-": ["5J"]},
    "FANVM": {"-": ["12"], "N": ["12"]},
    "FAR": {"-": ["13"]},
    "FARCIO": {"-": ["54"]},
    "FARINA": {"-": ["11"]},
    "FARINARIVS": {"-": ["21"]},
    "FARNVS": {"-": ["12"]},
    "FARRAGO": {"-": ["13"]},
    "FARRARIVM": {"-": ["12"]},
    "FARRARIVS": {"-": ["21"]},
    "FARRATVS": {"-": ["21"]},
    "FARREO": {"-": ["51"]},
    "FARTOR": {"-": ["13"]},
    "FARTVRA": {"-": ["11"]},
    "FAS": {"-": ["16"], "N": ["16"]},
    "FASCIA": {"-": ["11"]},
    "FASCICVLVS": {"-": ["12"]},
    "FASCINA": {"-": ["11"]},
    "FASCINO": {"-": ["51"]},
    "FASCIOLA": {"-": ["11"]},
    "FASCIS": {"-": ["13"]},
    "FASTI": {"-": ["12"]},
    "FASTIDIO": {"-": ["54"]},
    "FASTIDIOSE": {"-": ["60"]},
    "FASTIDIOSVS": {"-": ["21"]},
    "FASTIDIVM": {"-": ["12"]},
    "FASTIGATE": {"-": ["60"]},
    "FASTIGATVS": {"-": ["21"]},
    "FASTIGIVM": {"-": ["12"]},
    "FASTIGO": {"-": ["51"]},
    "FASTVS": {"1": ["14"], "2": ["14"], "3": ["21"]},
    "FATALIS": {"-": ["24"]},
    "FATEOR": {"-": ["5K"]},
    "FATIDICVS": {"1": ["12"], "2": ["21"]},
    "FATIFER": {"-": ["21"]},
    "FATIGATIO": {"-": ["13"]},
    "FATIGO": {"-": ["51"]},
    "FATISCO": {"-": ["53"]},
    "FATISCOR": {"-": ["5L"]},
    "FATVA": {"-": ["11"]},
    "FATVM": {"-": ["12"], "N": ["12"]},
    "FATVVS": {"1": ["12"], "2": ["21"]},
    "FAVEO": {"-": ["52"]},
    "FAVILLA": {"-": ["11"]},
    "FAVNVS": {"-": ["12"], "N": ["12"]},
    "FAVONIVS": {"-": ["12"], "N": ["12"]},
    "FAVOR": {"-": ["13"]},
    "FAVORABILIS": {"-": ["24"]},
    "FAVSTVLVS": {"N": ["12"]},
    "FAVSTVS": {"-": ["21"], "N": ["12"]},
    "FAVTOR": {"-": ["13"]},
    "FAVVS": {"-": ["12"]},
    "FAVX": {"-": ["13"]},
    "FAX": {"-": ["13"]},
    "FEBRICITO": {"-": ["51"]},
    "FEBRICVLA": {"-": ["11"]},
    "FEBRICVLOSVS": {"-": ["21"]},
    "FEBRIS": {"-": ["13"], "N": ["13"]},
    "FEBRVARIVS": {"A": ["21"], "N": ["12"]},
    "FECTO": {"-": ["51"]},
    "FECVNDITAS": {"-": ["13"]},
    "FECVNDO": {"-": ["51"]},
    "FECVNDVS": {"-": ["21"]},
    "FEL": {"-": ["13"]},
    "FELES": {"-": ["13"]},
    "FELICITAS": {"-": ["13"], "N": ["13"]},
    "FELICITER": {"-": ["60"]},
    "FELIX": {"-": ["25"], "N": ["13"]},
    "FELLATOR": {"-": ["13"]},
    "FELLO": {"-": ["51"]},
    "FEMELLA": {"-": ["11"]},
    "FEMINA": {"-": ["11"]},
    "FEMINEVS": {"-": ["21"]},
    "FEMININVS": {"-": ["21"]},
    "FEMINO": {"-": ["51"]},
    "FEMVR": {"-": ["13"]},
    "FENARIVS": {"-": ["21"]},
    "FENDO": {"-": ["53"]},
    "FENEBRIS": {"-": ["24"]},
    "FENERATIO": {"-": ["13"]},
    "FENERATOR": {"-": ["13"]},
    "FENERO": {"-": ["51"]},
    "FENEROR": {"-": ["5J"]},
    "FENESTRA": {"-": ["11"]},
    "FENESTRO": {"-": ["51"]},
    "FENICVLVM": {"-": ["12"]},
    "FENILIA": {"-": ["13"]},
    "FENISECA": {"-": ["11"]},
    "FENVM": {"-": ["12"]},
    "FENVS": {"-": ["13"]},
    "FERA": {"-": ["11"]},
    "FERALIA": {"-": ["13"], "N": ["11"]},
    "FERALIS": {"-": ["24"]},
    "FERAX": {"-": ["25"]},
    "FERCIO": {"-": ["54"]},
    "FERCVLVM": {"-": ["12"]},
    "FERE": {"-": ["60"]},
    "FERENTARIVS": {"-": ["12"]},
    "FERENTINVS": {"N": ["21"], "A": ["21"]},
    "FERETRIVS": {"N": ["12"], "A": ["21"]},
    "FERETRVM": {"-": ["12"]},
    "FERIAE": {"-": ["11"]},
    "FERIATVS": {"-": ["21"]},
    "FERINA": {"-": ["11"]},
    "FERINVS": {"-": ["21"]},
    "FERIO": {"-": ["54"]},
    "FERITAS": {"-": ["13"]},
    "FERME": {"-": ["60"]},
    "FERMENTVM": {"-": ["12"]},
    "FERO": {"-": ["56"]},
    "FEROCIA": {"-": ["11"]},
    "FEROCITAS": {"-": ["13"]},
    "FEROCITER": {"-": ["60"]},
    "FEROCVLVS": {"-": ["21"]},
    "FERONIA": {"1N": ["11"], "2N": ["11"], "N": ["11"]},
    "FEROX": {"-": ["25"], "N": ["13"]},
    "FERRAMENTVM": {"-": ["12"]},
    "FERRARIA": {"1": ["11"], "2": ["11"]},
    "FERRARIVS": {"1": ["12"], "2": ["21"]},
    "FERRATVS": {"-": ["21"]},
    "FERREA": {"-": ["11"]},
    "FERREVS": {"-": ["21"]},
    "FERRVGINEVS": {"-": ["21"]},
    "FERRVGO": {"-": ["13"]},
    "FERRVM": {"-": ["12"]},
    "FERTILIS": {"-": ["24"]},
    "FERTILITAS": {"-": ["13"]},
    "FERTVM": {"-": ["12"]},
    "FERVEFACIO": {"-": ["55"]},
    "FERVENS": {"-": ["25"], "-": ["22"]},
    "FERVEO": {"-": ["52"]},
    "FERVESCO": {"-": ["53"]},
    "FERVIDVS": {"-": ["21"]},
    "FERVLA": {"-": ["11"]},
    "FERVO": {"-": ["53"]},
    "FERVOR": {"-": ["13"]},
    "FERVS": {"1": ["12"], "2": ["21"]},
    "FESCENNINI": {"-": ["12"]},
    "FESCENNINVS": {"1": ["12"], "2": ["21"], "A": ["21"]},
    "FESSVS": {"-": ["21"]},
    "FESTINANTER": {"-": ["6&"]},
    "FESTINATIO": {"-": ["13"]},
    "FESTINO": {"-": ["51"]},
    "FESTIVVS": {"-": ["21"]},
    "FESTVCA": {"-": ["11"]},
    "FESTVCVLA": {"2": ["11"], "-": ["11"]},
    "FESTVM": {"-": ["12"]},
    "FESTVS": {"-": ["21"], "N": ["12"]},
    "FETIALIS": {"1": ["13"], "2": ["24"]},
    "FETVRA": {"-": ["11"]},
    "FETVS": {"1": ["14"], "2": ["21"]},
    "FIBER": {"-": ["12"]},
    "FIBRA": {"-": ["11"]},
    "FIBVLA": {"-": ["11"]},
    "FIBVLATIO": {"-": ["13"]},
    "FICANA": {"N": ["11"]},
    "FICARIVS": {"-": ["21"]},
    "FICEDVLA": {"-": ["11"]},
    "FICIO": {"-": ["55"]},
    "FICTILE": {"-": ["13"]},
    "FICTILIS": {"-": ["24"]},
    "FICTRIX": {"-": ["13"]},
    "FICTVM": {"1": ["12"], "2": ["60"]},
    "FICTVS": {"-": ["21"]},
    "FICVLEA": {"N": ["11"]},
    "FICVLNEVS": {"-": ["21"]},
    "FICVLNVS": {"-": ["21"]},
    "FICVS": {"1": ["16"], "2": ["16"]},
    "FIDELIA": {"-": ["11"]},
    "FIDELIS": {"1": ["13"], "2": ["24"]},
    "FIDELITAS": {"-": ["13"]},
    "FIDELITER": {"-": ["60"]},
    "FIDENAE": {"N": ["11"]},
    "FIDENAS": {"N": ["25"]},
    "FIDENATES": {"N": ["13"]},
    "FIDENS": {"-": ["25"]},
    "FIDENTER": {"-": ["60"]},
    "FIDES": {
        "1": ["13"],
        "2": ["15"],
        "1N": ["13"],
        "2N": ["15"],
        "3N": ["13"],
        "N": ["13"],
    },
    "FIDICEN": {"-": ["13"], "N": ["13"]},
    "FIDICVLA": {"-": ["11"]},
    "FIDO": {"-": ["5T"]},
    "FIDVCIA": {"-": ["11"]},
    "FIDVCIARIVS": {"-": ["21"]},
    "FIDVS": {"-": ["21"], "N": ["12"]},
    "FIGLINVS": {"-": ["21"]},
    "FIGO": {"-": ["53"]},
    "FIGVLVS": {"-": ["12"], "N": ["12"]},
    "FIGVRA": {"-": ["11"]},
    "FIGVRATIO": {"-": ["13"]},
    "FIGVRO": {"-": ["51"]},
    "FILATIM": {"-": ["60"]},
    "FILIA": {"-": ["11"]},
    "FILICVLA": {"-": ["11"]},
    "FILIOLA": {"-": ["11"]},
    "FILIOLVS": {"-": ["12"]},
    "FILIVS": {"-": ["12"]},
    "FILIX": {"-": ["13"]},
    "FILVM": {"-": ["12"]},
    "FIMBRIA": {"N": ["11"], "-": ["11"]},
    "FIMVS": {"-": ["12"]},
    "FINDO": {"-": ["53"]},
    "FINE": {"-": ["70"]},
    "FINGO": {"-": ["53"]},
    "FINIO": {"-": ["54"]},
    "FINIS": {"-": ["13"]},
    "FINITIMI": {"-": ["12"]},
    "FINITIMVS": {"-": ["21"]},
    "FINITIO": {"-": ["13"]},
    "FINITOR": {"-": ["13"]},
    "FINITVS": {"-": ["21"]},
    "FIO": {"-": ["56"]},
    "FIRMAMEN": {"-": ["13"]},
    "FIRMAMENTVM": {"-": ["12"]},
    "FIRMATOR": {"-": ["13"]},
    "FIRME": {"-": ["60"]},
    "FIRMITAS": {"-": ["13"]},
    "FIRMITER": {"-": ["60"]},
    "FIRMITVDO": {"-": ["13"]},
    "FIRMO": {"-": ["51"]},
    "FIRMVS": {"-": ["21"], "N": ["12"]},
    "FISCELLA": {"-": ["11"]},
    "FISCINA": {"-": ["11"]},
    "FISCVS": {"-": ["12"]},
    "FISSILIS": {"-": ["24"]},
    "FISSVM": {"-": ["12"]},
    "FISSVRA": {"-": ["11"]},
    "FISTVCA": {"-": ["11"]},
    "FISTVCATIO": {"-": ["13"]},
    "FISTVCO": {"-": ["51"]},
    "FISTVLA": {"-": ["11"]},
    "FISTVLOSVS": {"-": ["21"]},
    "FITEOR": {"-": ["5K"]},
    "FITILLA": {"-": ["11"]},
    "FIXVS": {"-": ["21", "14"]},
    "FLABELLVM": {"-": ["12"]},
    "FLABRVM": {"-": ["12"]},
    "FLACCEO": {"-": ["52"]},
    "FLACCESCO": {"-": ["53"]},
    "FLACCIDVS": {"-": ["21"]},
    "FLACCVS": {"-": ["21"], "N": ["12"]},
    "FLAGELLO": {"-": ["51"]},
    "FLAGELLVM": {"-": ["12"]},
    "FLAGITATIO": {"-": ["13"]},
    "FLAGITATOR": {"-": ["13"]},
    "FLAGITIOSE": {"-": ["60"]},
    "FLAGITIOSVS": {"-": ["21"]},
    "FLAGITIVM": {"-": ["12"]},
    "FLAGITO": {"-": ["51"]},
    "FLAGRANS": {"-": ["25"]},
    "FLAGRANTER": {"-": ["6-"]},
    "FLAGRANTIA": {"-": ["11"]},
    "FLAGRO": {"-": ["51"]},
    "FLAGRVM": {"-": ["12"]},
    "FLAMEN": {"1": ["13"], "2": ["13"]},
    "FLAMINICA": {"-": ["11"]},
    "FLAMININVS": {"N": ["12"]},
    "FLAMINIVM": {"-": ["12"]},
    "FLAMINIVS": {"N": ["12"], "A": ["21"]},
    "FLAMMA": {"-": ["11"], "N": ["11"]},
    "FLAMMEOLVM": {"-": ["12"]},
    "FLAMMESCO": {"-": ["53"]},
    "FLAMMEVM": {"-": ["12"]},
    "FLAMMEVS": {"-": ["21"]},
    "FLAMMIFER": {"-": ["21"]},
    "FLAMMO": {"-": ["51"]},
    "FLATVRA": {"-": ["11"]},
    "FLATVS": {"-": ["14"]},
    "FLAVEO": {"-": ["52"]},
    "FLAVESCO": {"-": ["53"]},
    "FLAVIVS": {"N": ["12"], "A": ["21"]},
    "FLAVOLEIVS": {"N": ["12"]},
    "FLAVVS": {"-": ["21"], "N": ["12"]},
    "FLEBILE": {"-": ["60"]},
    "FLEBILIS": {"-": ["24"]},
    "FLEBILITER": {"-": ["60"]},
    "FLECTO": {"-": ["53"]},
    "FLEO": {"-": ["52"]},
    "FLETVS": {"-": ["14"]},
    "FLEXANIMVS": {"-": ["21"]},
    "FLEXIBILIS": {"-": ["24"]},
    "FLEXILIS": {"-": ["24"]},
    "FLEXO": {"-": ["51"]},
    "FLEXVOSVS": {"-": ["21"]},
    "FLEXVRA": {"-": ["11"]},
    "FLEXVS": {"-": ["14"]},
    "FLIGO": {"-": ["53"]},
    "FLO": {"-": ["51"]},
    "FLOCCVS": {"-": ["12"]},
    "FLORA": {"N": ["11"]},
    "FLORENS": {"-": ["25"]},
    "FLOREO": {"-": ["52"]},
    "FLORESCO": {"-": ["53"]},
    "FLOREVS": {"-": ["21"]},
    "FLORIDVLVS": {"-": ["21"]},
    "FLORIDVS": {"-": ["21"]},
    "FLORIFER": {"-": ["21"]},
    "FLOS": {"-": ["13"]},
    "FLOSCVLVS": {"-": ["12"]},
    "FLVCTIFRAGVS": {"-": ["21"]},
    "FLVCTISONVS": {"-": ["21"]},
    "FLVCTVATIO": {"-": ["13"]},
    "FLVCTVO": {"-": ["51"]},
    "FLVCTVOR": {"-": ["5J"]},
    "FLVCTVS": {"-": ["14"]},
    "FLVENS": {"-": ["25"]},
    "FLVENTER": {"-": ["60"]},
    "FLVENTISONVS": {"-": ["21"]},
    "FLVENTVM": {"-": ["12"]},
    "FLVIDVS": {"-": ["21"]},
    "FLVITO": {"-": ["51"]},
    "FLVMEN": {"-": ["13"]},
    "FLVMINEVS": {"-": ["21"]},
    "FLVO": {"-": ["53"]},
    "FLVTO": {"-": ["51"]},
    "FLVVIALIS": {"-": ["24"]},
    "FLVVIATICVS": {"-": ["21"]},
    "FLVVIATILIS": {"-": ["24"]},
    "FLVVIDVS": {"-": ["21"]},
    "FLVVIVS": {"-": ["12"]},
    "FLVXVS": {"1": ["14"], "2": ["21", "14"]},
    "FOCALE": {"-": ["13"]},
    "FOCILLO": {"-": ["51"]},
    "FOCO": {"-": ["51"]},
    "FOCVLVS": {"-": ["12"]},
    "FOCVS": {"-": ["12"]},
    "FODICO": {"-": ["51"]},
    "FODINA": {"-": ["11"]},
    "FODIO": {"-": ["55"]},
    "FOEDE": {"-": ["60"]},
    "FOEDERATVS": {"-": ["21"]},
    "FOEDITAS": {"-": ["13"]},
    "FOEDO": {"-": ["51"]},
    "FOEDVS": {"1": ["13"], "2": ["21"]},
    "FOETIDVS": {"-": ["21"]},
    "FOLIATVM": {"-": ["12"]},
    "FOLIATVRA": {"-": ["11"]},
    "FOLIVM": {"-": ["12"]},
    "FOLLICVLVS": {"-": ["12"]},
    "FOLLIS": {"-": ["13"]},
    "FOMENTVM": {"-": ["12"]},
    "FOMES": {"-": ["13"]},
    "FONS": {"-": ["13"]},
    "FONTANVS": {"-": ["21"]},
    "FONTICVLVS": {"-": ["12"]},
    "FOR": {"-": ["5J"]},
    "FORAMEN": {"-": ["13"]},
    "FORAS": {"-": ["60"]},
    "FORCEPS": {"-": ["13"]},
    "FORENSIS": {"-": ["24"]},
    "FORFEX": {"-": ["13"]},
    "FORICA": {"-": ["11"]},
    "FORIS": {"1": ["13"], "2": ["60"]},
    "FORMA": {"-": ["11"]},
    "FORMAMENTVM": {"-": ["12"]},
    "FORMATIO": {"-": ["13"]},
    "FORMATOR": {"-": ["13"]},
    "FORMATVRA": {"-": ["11"]},
    "FORMICA": {"-": ["11"]},
    "FORMIDABILIS": {"-": ["24"]},
    "FORMIDO": {"1": ["13"], "2": ["51"], "N": ["11"]},
    "FORMIDOLOSE": {"-": ["6&"]},
    "FORMIDOLOSVS": {"-": ["21"]},
    "FORMO": {"-": ["51"]},
    "FORMOSE": {"-": ["6&"]},
    "FORMOSVS": {"-": ["21"]},
    "FORMVLA": {"-": ["11"]},
    "FORNACVLA": {"-": ["11"]},
    "FORNAX": {"-": ["13"], "N": ["13"]},
    "FORNICATIO": {"-": ["13"]},
    "FORNIX": {"-": ["13"]},
    "FORO": {"-": ["51"]},
    "FORS": {"1": ["13"], "2": ["60"], "N": ["16"]},
    "FORSAN": {"-": ["60"]},
    "FORSIT": {"-": ["60"]},
    "FORSITAN": {"-": ["60"]},
    "FORTASSE": {"-": ["60"]},
    "FORTAX": {"-": ["13"]},
    "FORTE": {"-": ["60"]},
    "FORTIS": {"-": ["24"], "A": ["24"]},
    "FORTITER": {"-": ["60"]},
    "FORTITVDO": {"-": ["13"]},
    "FORTVITA": {"-": ["12"]},
    "FORTVITO": {"-": ["60"]},
    "FORTVITV": {"-": ["60"]},
    "FORTVITVS": {"-": ["21"]},
    "FORTVNA": {"-": ["11"], "N": ["11"]},
    "FORTVNO": {"-": ["51"]},
    "FORVLI": {"-": ["12"], "N": ["12"]},
    "FORVM": {"1": ["12"], "2": ["12"], "N": ["12"]},
    "FORVS": {"-": ["12"]},
    "FOSSA": {"-": ["11"]},
    "FOSSICIVS": {"-": ["21"]},
    "FOSSILIS": {"1": ["13"], "2": ["24"]},
    "FOSSIO": {"-": ["13"]},
    "FOSSOR": {"-": ["13"]},
    "FOSSVLA": {"-": ["11"]},
    "FOSSVRA": {"-": ["11"]},
    "FOVEA": {"-": ["11"]},
    "FOVEO": {"-": ["52"]},
    "FRACES": {"-": ["13"]},
    "FRACESCO": {"-": ["53"]},
    "FRACIDVS": {"-": ["21"]},
    "FRACTVRA": {"-": ["11"]},
    "FRACTVS": {"-": ["21"]},
    "FRAGA": {"-": ["12"]},
    "FRAGILIS": {"-": ["24"]},
    "FRAGILITAS": {"-": ["13"]},
    "FRAGMEN": {"-": ["13"]},
    "FRAGMENTVM": {"-": ["12"]},
    "FRAGOR": {"-": ["13"]},
    "FRAGOSVS": {"-": ["21"]},
    "FRAGRO": {"-": ["51"]},
    "FRAGVM": {"-": ["12"]},
    "FRAMEA": {"-": ["11"]},
    "FRANGO": {"-": ["53"]},
    "FRATER": {"-": ["13"]},
    "FRATERCVLVS": {"-": ["21", "12"]},
    "FRATERNITAS": {"-": ["13"]},
    "FRATERNVS": {"-": ["21"]},
    "FRAVDATIO": {"-": ["13"]},
    "FRAVDATOR": {"-": ["13"]},
    "FRAVDO": {"-": ["51"]},
    "FRAVDVLENTER": {"-": ["60"]},
    "FRAVDVLENTIA": {"-": ["11"]},
    "FRAVDVLENTVS": {"-": ["21"]},
    "FRAVS": {"-": ["13"]},
    "FRAXINEVS": {"-": ["21"]},
    "FRAXINVS": {"-": ["12"]},
    "FREMITVS": {"-": ["14"]},
    "FREMO": {"-": ["53"]},
    "FRENATOR": {"-": ["13"]},
    "FRENDO": {"-": ["53"]},
    "FRENO": {"-": ["51"]},
    "FRENVM": {"-": ["12"]},
    "FREQVENS": {"-": ["25"]},
    "FREQVENTATVS": {"-": ["21"]},
    "FREQVENTER": {"-": ["60"]},
    "FREQVENTIA": {"-": ["11"]},
    "FREQVENTO": {"-": ["51"]},
    "FRETVM": {"-": ["12"]},
    "FRETVS": {"1": ["14"], "2": ["21"]},
    "FRICATVRA": {"-": ["11"]},
    "FRICO": {"-": ["51"]},
    "FRIGEO": {"-": ["52"]},
    "FRIGERO": {"-": ["51"]},
    "FRIGESCO": {"-": ["53"]},
    "FRIGIDA": {"-": ["11"]},
    "FRIGIDARIVS": {"-": ["21"]},
    "FRIGIDVLVS": {"-": ["21"]},
    "FRIGIDVS": {"-": ["21"]},
    "FRIGO": {"1": ["53"]},
    "FRIGVS": {"-": ["13"]},
    "FRINGO": {"-": ["53"]},
    "FRIO": {"-": ["51"]},
    "FRITILLVS": {"-": ["12"]},
    "FRIVOLA": {"-": ["12"]},
    "FRIVOLVS": {"-": ["21"]},
    "FRONDATOR": {"-": ["13"]},
    "FRONDEO": {"-": ["52"]},
    "FRONDESCO": {"-": ["53"]},
    "FRONDEVS": {"-": ["21"]},
    "FRONDIFER": {"-": ["21"]},
    "FRONDOSVS": {"-": ["21"]},
    "FRONS": {"1": ["13"], "2": ["13"]},
    "FRONTATI": {"-": ["21"]},
    "FRVCTIFER": {"-": ["21"]},
    "FRVCTVOSVS": {"-": ["21"]},
    "FRVCTVS": {"-": ["14"]},
    "FRVGALIS": {"-": ["24"]},
    "FRVGALITAS": {"-": ["13"]},
    "FRVGALITER": {"-": ["60"]},
    "FRVGI": {"-": ["26"], "C": ["16"], "N": ["12"]},
    "FRVGIFER": {"-": ["21"]},
    "FRVGIFERENS": {"-": ["25"]},
    "FRVGIPARVS": {"-": ["21"]},
    "FRVMENTARIVS": {"1": ["12"], "2": ["21"]},
    "FRVMENTATIO": {"-": ["13"]},
    "FRVMENTOR": {"-": ["5J"]},
    "FRVMENTVM": {"-": ["12"]},
    "FRVOR": {"-": ["5L"]},
    "FRVSTRA": {"-": ["60"]},
    "FRVSTRAMEN": {"-": ["13"]},
    "FRVSTRATIO": {"-": ["13"]},
    "FRVSTROR": {"-": ["5J"]},
    "FRVSTVM": {"-": ["12"]},
    "FRVTEX": {"-": ["13"]},
    "FRVTICETVM": {"-": ["12"]},
    "FRVTICOR": {"-": ["5J"]},
    "FRVTICOSVS": {"-": ["21"]},
    "FRVX": {"-": ["13"]},
    "FVCATVS": {"-": ["21"]},
    "FVCO": {"-": ["51"]},
    "FVCVS": {"1": ["12"], "2": ["12"]},
    "FVFETIVS": {"N": ["12"]},
    "FVFICIVS": {"N": ["12"]},
    "FVGA": {"-": ["11"], "N": ["11"]},
    "FVGAX": {"-": ["25"]},
    "FVGELA": {"-": ["11"]},
    "FVGIO": {"-": ["55"]},
    "FVGITIVVS": {"1": ["12"], "2": ["21"]},
    "FVGITO": {"-": ["51"]},
    "FVGO": {"-": ["51"]},
    "FVLCIMENTVM": {"-": ["12"]},
    "FVLCIO": {"-": ["54"]},
    "FVLCRVM": {"-": ["12"]},
    "FVLGENS": {"-": ["25"]},
    "FVLGEO": {"-": ["52"]},
    "FVLGIDVS": {"-": ["21"]},
    "FVLGO": {"-": ["53"]},
    "FVLGOR": {"-": ["13"]},
    "FVLGVR": {"-": ["13"]},
    "FVLGVRATOR": {"-": ["13"]},
    "FVLGVRIO": {"-": ["54"]},
    "FVLGVRO": {"-": ["51"]},
    "FVLICA": {"-": ["11"]},
    "FVLIGO": {"-": ["13"]},
    "FVLLO": {"-": ["13"]},
    "FVLLONICA": {"-": ["11"]},
    "FVLLONICVS": {"-": ["21"]},
    "FVLLONIVS": {"-": ["21"]},
    "FVLMEN": {"-": ["13"]},
    "FVLMENTA": {"-": ["11"]},
    "FVLMENTVM": {"-": ["12"]},
    "FVLMINEVS": {"-": ["21"]},
    "FVLMINO": {"-": ["51"]},
    "FVLTVRA": {"-": ["11"]},
    "FVLVIVS": {"N": ["12"]},
    "FVLVVS": {"-": ["21"], "N": ["12"]},
    "FVMEVS": {"-": ["21"]},
    "FVMIDVS": {"-": ["21"]},
    "FVMO": {"-": ["51"]},
    "FVMOSVS": {"-": ["21"]},
    "FVMVS": {"-": ["12"]},
    "FVNALE": {"-": ["13"]},
    "FVNALIS": {"-": ["24"]},
    "FVNCTIO": {"-": ["13"]},
    "FVNCTVS": {"-": ["12"]},
    "FVNDA": {"-": ["11"]},
    "FVNDAMEN": {"-": ["13"]},
    "FVNDAMENTVM": {"-": ["12"]},
    "FVNDATIO": {"-": ["13"]},
    "FVNDATVS": {"-": ["21"]},
    "FVNDITOR": {"-": ["13"]},
    "FVNDITVS": {"-": ["60"]},
    "FVNDO": {"1": ["51"], "2": ["53"]},
    "FVNDVLVS": {"-": ["12"]},
    "FVNDVS": {"-": ["12"]},
    "FVNEBRIS": {"-": ["24"]},
    "FVNEREVS": {"-": ["21"]},
    "FVNERO": {"-": ["51"]},
    "FVNESTO": {"-": ["51"]},
    "FVNESTVS": {"-": ["21"]},
    "FVNGOR": {"-": ["5L"]},
    "FVNGVS": {"-": ["12"]},
    "FVNICVLVS": {"-": ["12"]},
    "FVNIS": {"-": ["13"]},
    "FVNVS": {"-": ["13"], "N": ["12"]},
    "FVR": {"-": ["13"]},
    "FVRCA": {"-": ["11"]},
    "FVRCIFER": {"-": ["12"]},
    "FVRCILLA": {"-": ["11"]},
    "FVRENS": {"-": ["25"]},
    "FVRIA": {"-": ["11"], "N": ["11"]},
    "FVRIALIS": {"-": ["24"]},
    "FVRIBVNDVS": {"-": ["21"]},
    "FVRIO": {"1": ["51"], "2": ["54"]},
    "FVRIOSE": {"-": ["60"]},
    "FVRIOSVS": {"-": ["21"]},
    "FVRIVS": {"A": ["21"], "N": ["12"]},
    "FVRNVS": {"-": ["12"]},
    "FVRO": {"-": ["53"]},
    "FVROR": {"1": ["13"], "2": ["5J", "13"], "N": ["13"]},
    "FVRTIM": {"-": ["60"]},
    "FVRTIVE": {"-": ["60"]},
    "FVRTIVVS": {"-": ["21"]},
    "FVRTVM": {"-": ["12"]},
    "FVRVVS": {"-": ["21"]},
    "FVSCINA": {"-": ["11"]},
    "FVSCO": {"-": ["51"]},
    "FVSCVS": {"-": ["21"], "N": ["12"]},
    "FVSILIS": {"-": ["24"]},
    "FVSIO": {"-": ["13"]},
    "FVSIVS": {"A": ["21"], "N": ["12"]},
    "FVSTERNA": {"-": ["11"]},
    "FVSTIS": {"-": ["13"]},
    "FVSVS": {"1": ["12"], "2": ["14"], "3": ["21"]},
    "FVTILE": {"1": ["13"], "2": ["60"]},
    "FVTILIS": {"-": ["24"]},
    "FVTO": {"-": ["51"]},
    "FVTVO": {"-": ["53"]},
    "FVTVRI": {"-": ["12"]},
    "FVTVRVM": {"-": ["12"]},
    "FVTVRVS": {"-": ["21"]},
    "FVTVTIO": {"-": ["13"]},
    "GABII": {"N": ["12"]},
    "GABINI": {"N": ["12"]},
    "GABINVS": {"N": ["21"], "A": ["21"]},
    "GADES": {"N": ["13"]},
    "GAESVM": {"-": ["12"]},
    "GAGATES": {"-": ["13"]},
    "GAIVS": {"N": ["12"]},
    "GALBA": {"N": ["11"]},
    "GALBANEVS": {"-": ["21"]},
    "GALBEVS": {"-": ["12"]},
    "GALBINVM": {"-": ["12"]},
    "GALEA": {"-": ["11"]},
    "GALEATVS": {"-": ["21"]},
    "GALEO": {"-": ["51"]},
    "GALERVS": {"-": ["12"]},
    "GALLA": {"-": ["11"], "N": ["11"]},
    "GALLI": {"N": ["12"]},
    "GALLIA": {"N": ["11"]},
    "GALLICVS": {"N": ["21"], "A": ["21"]},
    "GALLINA": {"-": ["11"], "N": ["11"]},
    "GALLINACEVS": {"-": ["21"]},
    "GALLVS": {"-": ["12"], "A": ["21"], "N": ["12"]},
    "GANEA": {"-": ["11"]},
    "GANEO": {"-": ["13"]},
    "GANEVM": {"-": ["12"]},
    "GANGABA": {"-": ["11"]},
    "GANNEO": {"-": ["13"]},
    "GANNIO": {"-": ["54"]},
    "GANNITVS": {"-": ["14"]},
    "GANOSIS": {"-": ["17"]},
    "GANYMEDES": {"N": ["17"]},
    "GARRIO": {"-": ["54"]},
    "GARRVLITAS": {"-": ["13"]},
    "GARRVLVS": {"-": ["21"]},
    "GARVM": {"-": ["12"]},
    "GAVDEO": {"-": ["5S"]},
    "GAVDIVM": {"-": ["12"], "N": ["12"]},
    "GAVIVS": {"N": ["12"]},
    "GAVSAPATVS": {"-": ["21"]},
    "GAVSAPVM": {"-": ["16"]},
    "GAZA": {"-": ["11"], "N": ["11"]},
    "GEGANIVS": {"N": ["12"]},
    "GELENSES": {"N": ["13"]},
    "GELICIDIVM": {"-": ["12"]},
    "GELIDVS": {"-": ["21"]},
    "GELLIVS": {"N": ["12"], "A": ["21"]},
    "GELO": {"-": ["51"]},
    "GELV": {"-": ["14"]},
    "GELVM": {"-": ["12"]},
    "GELVS": {"-": ["14"]},
    "GEMELLVS": {"1": ["12"], "2": ["21"]},
    "GEMINI": {"-": ["12"], "N": ["12"]},
    "GEMINO": {"-": ["51"]},
    "GEMINVS": {"-": ["21"], "N": ["12"], "A": ["21"]},
    "GEMISCO": {"-": ["53"]},
    "GEMITVS": {"-": ["14"]},
    "GEMMA": {"-": ["11"]},
    "GEMMATVS": {"-": ["21"]},
    "GEMMEVS": {"-": ["21"]},
    "GEMMIFER": {"-": ["21"]},
    "GEMMO": {"-": ["51"]},
    "GEMO": {"-": ["53"]},
    "GENA": {"-": ["11"]},
    "GENEALOGVS": {"-": ["12"]},
    "GENER": {"-": ["12"]},
    "GENERALIS": {"-": ["24"]},
    "GENERALITER": {"-": ["60"]},
    "GENERASCO": {"-": ["53"]},
    "GENERATIM": {"-": ["60"]},
    "GENERATOR": {"-": ["13"]},
    "GENERO": {"-": ["51"]},
    "GENEROSE": {"-": ["60"]},
    "GENEROSVS": {"-": ["21"]},
    "GENESIS": {"-": ["13"]},
    "GENETHLIOLOGIA": {"-": ["11"]},
    "GENETRIX": {"-": ["13"], "N": ["13"]},
    "GENIALIS": {"-": ["24"]},
    "GENIALITER": {"-": ["60"]},
    "GENICVLVS": {"-": ["12"]},
    "GENISTA": {"-": ["11"]},
    "GENITABILIS": {"-": ["24"]},
    "GENITALIA": {"-": ["12"]},
    "GENITALIS": {"-": ["24"], "A": ["24"]},
    "GENITALITER": {"-": ["60"]},
    "GENITOR": {"-": ["13"], "N": ["13"]},
    "GENIVS": {"-": ["12"], "N": ["12"]},
    "GENO": {"-": ["53"]},
    "GENS": {"-": ["13"]},
    "GENTICVS": {"-": ["21"]},
    "GENTILICIVS": {"-": ["21"]},
    "GENTILIS": {"1": ["13"], "2": ["24"]},
    "GENTILITAS": {"-": ["13"]},
    "GENV": {"-": ["14"]},
    "GENVALIA": {"-": ["13"]},
    "GENVCIVS": {"N": ["12"]},
    "GENVINVS": {"-": ["21"]},
    "GENVS": {"1": ["13"], "2": ["14"]},
    "GEOMETRES": {"-": ["17"]},
    "GEOMETRIA": {"-": ["11"]},
    "GEOMETRICA": {"-": ["12"]},
    "GEOMETRICE": {"1": ["17"], "2": ["60"]},
    "GEOMETRICVS": {"1": ["12"], "2": ["21"]},
    "GERMANA": {"-": ["11"]},
    "GERMANIA": {"N": ["11"]},
    "GERMANICVS": {"N": ["21"], "A": ["21"]},
    "GERMANVS": {"1": ["12"], "2": ["21"], "N": ["21"], "A": ["21"]},
    "GERMEN": {"-": ["13"]},
    "GERMINO": {"-": ["51"]},
    "GERO": {"-": ["53"]},
    "GERVLVS": {"1": ["12"], "2": ["21"]},
    "GERVSIA": {"-": ["11"], "N": ["11"]},
    "GESTA": {"-": ["12"]},
    "GESTAMEN": {"-": ["13"]},
    "GESTATIO": {"-": ["13"]},
    "GESTIO": {"1": ["13"], "2": ["54"]},
    "GESTO": {"-": ["51"]},
    "GESTVS": {"-": ["14"]},
    "GETA": {"C": ["11"], "N": ["11"]},
    "GETAE": {"-": ["11"], "N": ["11"]},
    "GETICVS": {"N": ["21"], "A": ["21"]},
    "GIBBER": {"1": ["13"], "2": ["21"]},
    "GIBBVS": {"-": ["12"]},
    "GIGAS": {"-": ["17"], "N": ["17"]},
    "GIGNO": {"-": ["53"]},
    "GILVVS": {"-": ["21"]},
    "GINGIVA": {"-": ["11"]},
    "GLABER": {"1": ["12"], "2": ["21"]},
    "GLABRIO": {"N": ["13"]},
    "GLACIALIS": {"-": ["24"]},
    "GLACIES": {"-": ["15"]},
    "GLACIO": {"-": ["51"]},
    "GLADIATOR": {"-": ["13"]},
    "GLADIATORIVS": {"-": ["21"]},
    "GLADIATVRA": {"-": ["11"]},
    "GLADIOLVS": {"-": ["12"]},
    "GLADIVS": {"-": ["12"]},
    "GLANDARIVS": {"-": ["21"]},
    "GLANDIFER": {"-": ["21"]},
    "GLANS": {"-": ["13"]},
    "GLAREA": {"-": ["11"]},
    "GLAREOSVS": {"-": ["21"]},
    "GLAVCIA": {"N": ["11"]},
    "GLAVCVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "GLEBA": {"-": ["11"]},
    "GLEBVLA": {"-": ["11"]},
    "GLESVM": {"-": ["12"]},
    "GLISCO": {"-": ["53"]},
    "GLOBO": {"-": ["51"]},
    "GLOBOSVS": {"-": ["21"]},
    "GLOBVS": {"-": ["12"]},
    "GLOMERAMEN": {"-": ["13"]},
    "GLOMERO": {"-": ["51"]},
    "GLOMVS": {"-": ["13"]},
    "GLORIA": {"-": ["11"], "N": ["11"]},
    "GLORIOR": {"-": ["5J"]},
    "GLORIOSE": {"-": ["60"]},
    "GLORIOSVS": {"-": ["21"]},
    "GLOSSA": {"-": ["11"]},
    "GLVBO": {"-": ["53"]},
    "GLVTEN": {"-": ["13"]},
    "GLVTINATOR": {"-": ["13"]},
    "GLVTINVM": {"-": ["12"]},
    "GLVTIO": {"-": ["54"]},
    "GLVTVS": {"1": ["14"], "2": ["21"]},
    "GNAEVS": {"N": ["12"]},
    "GNARVS": {"-": ["21"]},
    "GNOMON": {"-": ["17"]},
    "GNOMONICE": {"-": ["17"]},
    "GNOMONICVS": {"-": ["21"]},
    "GOBIO": {"-": ["13"]},
    "GORGO": {"N": ["17"]},
    "GRABATVS": {"-": ["12"]},
    "GRACCHVS": {"N": ["12"]},
    "GRACILIS": {"-": ["24"], "N": ["13"]},
    "GRACILITAS": {"-": ["13"]},
    "GRACVLVS": {"-": ["12"]},
    "GRADARIVS": {"-": ["21"]},
    "GRADATIM": {"-": ["60"]},
    "GRADATIO": {"-": ["13"]},
    "GRADIOR": {"-": ["5N"]},
    "GRADIVVS": {"N": ["12"]},
    "GRADVS": {"-": ["14"]},
    "GRAECE": {"-": ["60"]},
    "GRAECI": {"N": ["12"]},
    "GRAECIA": {"N": ["11"]},
    "GRAECOR": {"-": ["5J"]},
    "GRAECVLVS": {"N": ["21"], "A": ["21"]},
    "GRAECVS": {"N": ["21"], "A": ["21"]},
    "GRAII": {"N": ["12"]},
    "GRAIVS": {"N": ["21"], "A": ["21"]},
    "GRAMEN": {"-": ["13"]},
    "GRAMINEVS": {"-": ["21"]},
    "GRAMMATICA": {"1": ["12"], "2": ["17"]},
    "GRAMMATICVS": {"1": ["12"], "2": ["21"]},
    "GRAMMICVS": {"-": ["21"]},
    "GRANARIVM": {"-": ["12"]},
    "GRANATVS": {"-": ["14"]},
    "GRANDAEVVS": {"-": ["21"]},
    "GRANDE": {"-": ["60"]},
    "GRANDESCO": {"-": ["53"]},
    "GRANDIO": {"-": ["54"]},
    "GRANDIS": {"-": ["24"]},
    "GRANDISCAPIVS": {"-": ["21"]},
    "GRANDITER": {"-": ["6&"]},
    "GRANDO": {"-": ["13"]},
    "GRANEA": {"-": ["11"]},
    "GRANEVS": {"-": ["21"]},
    "GRANIFER": {"-": ["21"]},
    "GRANIVS": {"N": ["12"]},
    "GRANVM": {"-": ["12"]},
    "GRAPHICVS": {"-": ["2G", "21"]},
    "GRAPHIS": {"-": ["17"]},
    "GRAPHIVM": {"-": ["12"]},
    "GRASSATOR": {"-": ["13"]},
    "GRASSOR": {"-": ["5J"]},
    "GRATE": {"-": ["60"]},
    "GRATES": {"-": ["13"]},
    "GRATIA": {"-": ["11"], "N": ["11"]},
    "GRATIFICOR": {"-": ["5J"]},
    "GRATIOSVS": {"-": ["21"]},
    "GRATIS": {"-": ["60"]},
    "GRATOR": {"-": ["5J"]},
    "GRATVITO": {"-": ["60"]},
    "GRATVITVS": {"-": ["21"]},
    "GRATVLATIO": {"-": ["13"]},
    "GRATVLOR": {"-": ["5J"]},
    "GRATVS": {"-": ["21"], "N": ["12"]},
    "GRAVATIM": {"-": ["60"]},
    "GRAVE": {"-": ["60"]},
    "GRAVEDO": {"-": ["13"]},
    "GRAVEOLENS": {"-": ["25"]},
    "GRAVESCO": {"-": ["53"]},
    "GRAVIDVS": {"-": ["21"]},
    "GRAVIS": {"-": ["24"]},
    "GRAVITAS": {"-": ["13"]},
    "GRAVITER": {"-": ["60"]},
    "GRAVITVDO": {"-": ["13"]},
    "GRAVO": {"-": ["51"]},
    "GRAVOR": {"-": ["5J"]},
    "GREDIOR": {"-": ["5N"]},
    "GREGALES": {"-": ["13"]},
    "GREGALIS": {"-": ["24"]},
    "GREGARIVS": {"1": ["12"], "2": ["21"]},
    "GREGATIM": {"-": ["60"]},
    "GREGO": {"-": ["51"]},
    "GREMIVM": {"-": ["12"]},
    "GRESSVS": {"-": ["14"]},
    "GREX": {"-": ["13"]},
    "GROSPHVS": {"N": ["12"]},
    "GROSSVS": {"-": ["12"]},
    "GRVMVLVS": {"-": ["12"]},
    "GRVMVS": {"-": ["12"]},
    "GRVNNIO": {"-": ["54"]},
    "GRVO": {"-": ["53"]},
    "GRVS": {"-": ["13"]},
    "GRYPS": {"-": ["17"]},
    "GVBERNACVLVM": {"-": ["12"]},
    "GVBERNATIO": {"-": ["13"]},
    "GVBERNATOR": {"-": ["13"]},
    "GVBERNO": {"-": ["51"]},
    "GVLA": {"-": ["11"]},
    "GVLOSVS": {"-": ["21"]},
    "GVMMI": {"-": ["16"]},
    "GVRGES": {"-": ["13"], "N": ["13"]},
    "GVRGVLIO": {"-": ["13"]},
    "GVSTO": {"-": ["51"]},
    "GVSTVS": {"-": ["14"]},
    "GVTTA": {"-": ["11"], "N": ["11"]},
    "GVTTVLA": {"-": ["11"]},
    "GVTTVR": {"-": ["13"]},
    "GVTTVS": {"-": ["12"]},
    "GYMNASIARCHVS": {"-": ["12"]},
    "GYMNASIVM": {"-": ["12"]},
    "GYMNICVS": {"-": ["21"]},
    "GYNAECONITIS": {"-": ["17"]},
    "GYPSO": {"-": ["51"]},
    "GYPSVM": {"-": ["12"]},
    "GYRVS": {"-": ["12"]},
    "HABENA": {"-": ["11"]},
    "HABEO": {"-": ["52"]},
    "HABILIS": {"-": ["24"]},
    "HABILITAS": {"-": ["13"]},
    "HABITABILIS": {"-": ["24"]},
    "HABITANS": {"-": ["13"]},
    "HABITATIO": {"-": ["13"]},
    "HABITATOR": {"-": ["13"]},
    "HABITO": {"-": ["51"]},
    "HABITVS": {"1": ["14"], "2": ["21"], "N": ["12"]},
    "HABONIVS": {"N": ["12"]},
    "HAC": {"-": ["60"]},
    "HACTENVS": {"-": ["60"]},
    "HADRIANVS": {"N": ["12"], "A": ["21"]},
    "HADRIATICVS": {"N": ["21"], "A": ["21"]},
    "HADRVMENTVM": {"-": ["12"], "N": ["12"]},
    "HAEDILIA": {"-": ["11"]},
    "HAEDINVS": {"-": ["21"]},
    "HAEDVLVS": {"-": ["12"]},
    "HAEDVS": {"-": ["12"], "N": ["12"]},
    "HAEREO": {"-": ["52"]},
    "HAERESCO": {"-": ["53"]},
    "HAERESIS": {"-": ["17"]},
    "HAESITATIO": {"-": ["13"]},
    "HAESITO": {"-": ["51"]},
    "HALAESA": {"N": ["11"]},
    "HALAESINI": {"N": ["12"]},
    "HALAESINVS": {"N": ["21"], "A": ["21"]},
    "HALICARNASSVS": {"N": ["17"]},
    "HALICYENSES": {"N": ["13"]},
    "HALICYENSIS": {"N": ["24"], "A": ["24"]},
    "HALITVS": {"-": ["14"]},
    "HALO": {"-": ["51"]},
    "HAMA": {"-": ["11"]},
    "HAMATVS": {"-": ["21"]},
    "HAMAXOPODES": {"-": ["17"]},
    "HAMILCAR": {"N": ["13"]},
    "HAMVS": {"-": ["12"]},
    "HANNIBAL": {"N": ["13"]},
    "HAPHE": {"-": ["17"]},
    "HARA": {"-": ["11"]},
    "HARIOLVS": {"-": ["12"]},
    "HARMONIA": {"-": ["11"], "N": ["11"]},
    "HARMONICE": {"-": ["17"]},
    "HARPAGO": {"1": ["13"], "2": ["51"]},
    "HARVSPEX": {"-": ["13"]},
    "HARVSPICINA": {"-": ["11"]},
    "HARVSPICIVM": {"-": ["12"]},
    "HASDRVBAL": {"N": ["13"]},
    "HASTA": {"-": ["11"], "N": ["11"]},
    "HASTARIVS": {"1": ["12"], "2": ["21"]},
    "HASTATVS": {"1": ["12"], "2": ["21"]},
    "HASTILE": {"-": ["13"]},
    "HAVD": {"-": ["68"]},
    "HAVDDVM": {"-": ["68"]},
    "HAVDQVAQVAM": {"-": ["68"]},
    "HAVRIO": {"-": ["54"]},
    "HAVSTVS": {"-": ["14"]},
    "HEBEO": {"-": ["52"]},
    "HEBES": {"-": ["25"]},
    "HEBESCO": {"-": ["53"]},
    "HEBETATIO": {"-": ["13"]},
    "HEBETO": {"-": ["51"]},
    "HEBRVS": {"1N": ["12"], "2N": ["12"], "N": ["12"]},
    "HECATOMBE": {"-": ["17"]},
    "HEDERA": {"-": ["11"]},
    "HEDERACEVS": {"-": ["21"]},
    "HEDERIGER": {"-": ["21"]},
    "HEI": {"-": ["90"]},
    "HEIVS": {"N": ["12"]},
    "HELENA": {"N": ["11"]},
    "HELEPOLIS": {"-": ["17"]},
    "HELICON": {"-": ["17"], "N": ["17"]},
    "HELIX": {"-": ["13"]},
    "HELLEBORVM": {"-": ["12"]},
    "HELLEBORVS": {"-": ["12"]},
    "HELLESPONTVS": {"N": ["12"]},
    "HELO": {"-": ["51"]},
    "HELVOLVS": {"-": ["21"]},
    "HELVOR": {"-": ["5J"]},
    "HEM": {"-": ["90"]},
    "HEMICYCLIVM": {"-": ["12"]},
    "HEMINA": {"-": ["11"]},
    "HEMIOLIOS": {"-": ["27"]},
    "HEMISPHAERIVM": {"-": ["12"]},
    "HEMITONIVM": {"-": ["17"]},
    "HEMITRIGLYPHVS": {"-": ["12"]},
    "HENDECASYLLABVS": {"-": ["12"]},
    "HENNA": {"N": ["11"]},
    "HENNENSES": {"N": ["13"]},
    "HENNENSIS": {"N": ["24"], "A": ["24"]},
    "HERA": {"-": ["11"]},
    "HERACLEA": {"N": ["11"]},
    "HERACLEO": {"N": ["17"]},
    "HERACLEVS": {"N": ["12"], "A": ["21"]},
    "HERACLIENSES": {"N": ["13"]},
    "HERACLIENSIS": {"N": ["24"], "A": ["24"]},
    "HERACLIVS": {"N": ["12"]},
    "HERBA": {"-": ["11"]},
    "HERBIDVS": {"-": ["21"]},
    "HERBITA": {"N": ["11"]},
    "HERBITENSES": {"N": ["13"]},
    "HERBITENSIS": {"N": ["24"], "A": ["24"]},
    "HERBOSVS": {"-": ["21"]},
    "HERBVLA": {"-": ["11"]},
    "HERCISCO": {"-": ["53"]},
    "HERCVLES": {"-": ["90"], "N": ["16"]},
    "HEREDITARIVS": {"-": ["21"]},
    "HEREDITAS": {"-": ["13"]},
    "HEREDITO": {"-": ["51"]},
    "HERENNIVS": {"N": ["12"]},
    "HERES": {"-": ["13"]},
    "HERI": {"-": ["60"]},
    "HERIFVGA": {"-": ["21"]},
    "HERILIS": {"-": ["24"]},
    "HERMINIVS": {"N": ["12"]},
    "HERNICI": {"N": ["12"]},
    "HERNICVS": {"N": ["21"], "A": ["21"]},
    "HERODOTVS": {"N": ["12"]},
    "HEROICVS": {"-": ["21"]},
    "HEROINE": {"-": ["17"]},
    "HEROIS": {"-": ["17"]},
    "HEROS": {"-": ["17"]},
    "HEROVM": {"-": ["12"]},
    "HEROVS": {"-": ["21"]},
    "HERSILIA": {"N": ["11"]},
    "HERVS": {"-": ["12"]},
    "HESIODVS": {"N": ["12"]},
    "HESPERIA": {"N": ["11"]},
    "HESPERIVS": {"N": ["21"], "A": ["21"]},
    "HESTERNO": {"-": ["60"]},
    "HESTERNVS": {"-": ["21"]},
    "HEV": {"-": ["90"]},
    "HEVS": {"-": ["90"]},
    "HEXACHORDOS": {"-": ["27"]},
    "HEXAPHOROS": {"-": ["27"]},
    "HEXASTYLOS": {"-": ["27"]},
    "HIASCO": {"-": ["53"]},
    "HIATVS": {"-": ["14"]},
    "HIBEO": {"-": ["52"]},
    "HIBERNA": {"-": ["12"]},
    "HIBERNACVLVM": {"-": ["12"]},
    "HIBERNO": {"-": ["51"]},
    "HIBERNVM": {"-": ["12"]},
    "HIBERNVS": {"-": ["21"]},
    "HIBISCVM": {"-": ["12"]},
    "HIC": {"1": ["45"], "2": ["60"]},
    "HICCINE": {"-": ["47"]},
    "HIEMALIS": {"-": ["24"]},
    "HIEMO": {"-": ["51"]},
    "HIEMS": {"-": ["13"], "N": ["13"]},
    "HIERA": {"1": ["11"], "2": ["17"]},
    "HIERO": {"N": ["17"]},
    "HIERONICVS": {"N": ["21"], "A": ["21"]},
    "HILARE": {"-": ["60"]},
    "HILARICVLVS": {"-": ["21"]},
    "HILARIS": {"-": ["24"]},
    "HILARITAS": {"-": ["13"]},
    "HILARO": {"-": ["51"]},
    "HILARVS": {"-": ["21"], "N": ["12"]},
    "HILLA": {"-": ["11"]},
    "HILVM": {"-": ["12"]},
    "HIMERA": {"N": ["11"]},
    "HINC": {"-": ["60"]},
    "HINNIO": {"-": ["54"]},
    "HINNITVS": {"-": ["14"]},
    "HINNVLEVS": {"-": ["12"]},
    "HIO": {"-": ["51"]},
    "HIPPOCAMELVS": {"-": ["12"]},
    "HIPPOMANES": {"-": ["13"]},
    "HIPPOPERAE": {"-": ["11"]},
    "HIPPOPOTAMVS": {"-": ["12"]},
    "HIPPOTOTAE": {"-": ["11"]},
    "HIPPOTOXOTAE": {"-": ["11"]},
    "HIRCINVS": {"-": ["21"]},
    "HIRCOSVS": {"-": ["21"]},
    "HIRCVS": {"-": ["12"]},
    "HIRNEA": {"-": ["11"]},
    "HIRSVTVS": {"-": ["21"]},
    "HIRTVS": {"-": ["21"]},
    "HIRVNDO": {"-": ["13"]},
    "HISCO": {"-": ["53"]},
    "HISPANI": {"N": ["12"]},
    "HISPANIA": {"N": ["11"]},
    "HISPANVS": {"N": ["21"], "A": ["21"]},
    "HISPIDVS": {"-": ["21"]},
    "HISTORIA": {"-": ["11"], "N": ["11"]},
    "HISTORICVS": {"1": ["12"], "2": ["21"]},
    "HISTORVMENOS": {"-": ["57"]},
    "HISTRIO": {"-": ["13"]},
    "HISTRIONALIS": {"-": ["24"]},
    "HIVLCO": {"-": ["51"]},
    "HIVLCVS": {"-": ["21"]},
    "HODIE": {"-": ["60"]},
    "HODIERNVS": {"-": ["21"]},
    "HOMERVS": {"N": ["12"]},
    "HOMICIDA": {"-": ["11"]},
    "HOMICIDIVM": {"-": ["12"]},
    "HOMO": {"-": ["13"]},
    "HOMOEOMERIA": {"-": ["17"]},
    "HOMOTONVS": {"-": ["21"]},
    "HOMVLLVS": {"-": ["12"], "N": ["12"]},
    "HOMVNCIO": {"-": ["13"]},
    "HOMVNCVLVS": {"-": ["12"]},
    "HONESTAMENTVM": {"-": ["12"]},
    "HONESTAS": {"-": ["13"]},
    "HONESTE": {"-": ["60"]},
    "HONESTI": {"-": ["12"]},
    "HONESTO": {"-": ["51"]},
    "HONESTVM": {"-": ["12"]},
    "HONESTVS": {"-": ["21"]},
    "HONOR": {"-": ["13"], "N": ["13"]},
    "HONORARIVS": {"-": ["21"]},
    "HONORATVS": {"-": ["21"], "N": ["12"]},
    "HONORIFICE": {"-": ["60"]},
    "HONORIFICVS": {"-": ["21"]},
    "HONORO": {"-": ["51"]},
    "HONORVS": {"-": ["21"]},
    "HORA": {"-": ["11"], "N": ["11"]},
    "HORATIA": {"N": ["11"]},
    "HORATIVS": {"A": ["21"], "N": ["12"]},
    "HORDEACEVS": {"-": ["21"]},
    "HORDEVM": {"-": ["12"]},
    "HORIZON": {"-": ["17"]},
    "HORNOTINVS": {"-": ["21"]},
    "HORNVS": {"-": ["21"]},
    "HOROLOGIVM": {"-": ["12"]},
    "HOROSCOPVS": {"-": ["12"]},
    "HORRENDVM": {"-": ["60"]},
    "HORRENDVS": {"-": ["21"]},
    "HORRENS": {"-": ["25"]},
    "HORREO": {"-": ["52"]},
    "HORRESCO": {"-": ["53"]},
    "HORREVM": {"-": ["12"]},
    "HORRIBILIS": {"-": ["24"]},
    "HORRIDVLVS": {"-": ["21"]},
    "HORRIDVS": {"-": ["21"]},
    "HORRIFER": {"-": ["21"]},
    "HORRIFICE": {"-": ["60"]},
    "HORRIFICO": {"-": ["51"]},
    "HORRIFICVS": {"-": ["21"]},
    "HORRISONVS": {"-": ["21"]},
    "HORROR": {"-": ["13"], "N": ["13"]},
    "HORTAMEN": {"-": ["13"]},
    "HORTAMENTVM": {"-": ["12"]},
    "HORTATIO": {"-": ["13"]},
    "HORTATOR": {"-": ["13"]},
    "HORTATVS": {"-": ["14"]},
    "HORTENSIVS": {"N": ["12"]},
    "HORTENTIVS": {"N": ["12"]},
    "HORTOR": {"-": ["5J"]},
    "HORTVLVS": {"-": ["12"]},
    "HORTVS": {"-": ["12"]},
    "HOSPES": {"-": ["13"]},
    "HOSPITA": {"-": ["11"]},
    "HOSPITALIA": {"-": ["13"]},
    "HOSPITALIS": {"1": ["13"], "2": ["24"]},
    "HOSPITALITAS": {"-": ["13"]},
    "HOSPITALITER": {"-": ["60"]},
    "HOSPITIVM": {"-": ["12"]},
    "HOSPITOR": {"-": ["5J"]},
    "HOSPITVS": {"-": ["21"]},
    "HOSTES": {"-": ["13"]},
    "HOSTIA": {"-": ["11"]},
    "HOSTICVM": {"-": ["12"]},
    "HOSTICVS": {"-": ["21"]},
    "HOSTILE": {"-": ["13"]},
    "HOSTILIS": {"-": ["24"]},
    "HOSTILITER": {"-": ["60"]},
    "HOSTILIVS": {"A": ["21"], "N": ["12"]},
    "HOSTIS": {"-": ["13"]},
    "HOSTIVS": {"N": ["12"]},
    "HVC": {"-": ["60"]},
    "HVCCINE": {"-": ["67"]},
    "HVIVSMODI": {"-": ["60"]},
    "HVMANA": {"-": ["12"]},
    "HVMANE": {"-": ["60"]},
    "HVMANI": {"-": ["12"]},
    "HVMANITAS": {"-": ["13"]},
    "HVMANITER": {"-": ["60"]},
    "HVMANITVS": {"-": ["60"]},
    "HVMANVM": {"-": ["12"]},
    "HVMANVS": {"-": ["21"]},
    "HVMECTO": {"-": ["51"]},
    "HVMECTVS": {"-": ["21"]},
    "HVMEO": {"-": ["52"]},
    "HVMERVS": {"-": ["12"]},
    "HVMESCO": {"-": ["53"]},
    "HVMIDITAS": {"-": ["13"]},
    "HVMIDVLVS": {"-": ["21"]},
    "HVMIDVM": {"-": ["12"]},
    "HVMIDVS": {"-": ["21"]},
    "HVMILIS": {"-": ["24"]},
    "HVMILITAS": {"-": ["13"]},
    "HVMILITER": {"-": ["60"]},
    "HVMO": {"-": ["51"]},
    "HVMOR": {"-": ["13"]},
    "HVMVS": {"-": ["12"]},
    "HYACINTHINVS": {"-": ["21"]},
    "HYACINTHVS": {"-": ["12"]},
    "HYALVS": {"-": ["12"]},
    "HYBLENSES": {"N": ["13"]},
    "HYBRIDA": {"-": ["11"]},
    "HYDOR": {"-": ["17"]},
    "HYDRA": {"-": ["11"], "N": ["11"]},
    "HYDRALETES": {"-": ["17"]},
    "HYDRAVLICVS": {"-": ["21"]},
    "HYDRIA": {"-": ["11"]},
    "HYDRIAS": {"-": ["27"]},
    "HYDROPICVS": {"-": ["21"]},
    "HYDROPS": {"-": ["13"]},
    "HYDRVS": {"-": ["17"]},
    "HYMEN": {"-": ["13"], "N": ["13"]},
    "HYMENAEOS": {"-": ["17"]},
    "HYMENEIVS": {"-": ["21"]},
    "HYPAETROS": {"-": ["27"]},
    "HYPATE": {"-": ["17"]},
    "HYPATOS": {"-": ["27"]},
    "HYPERBOLAEOS": {"1": ["17"], "2": ["27"]},
    "HYPERBOLE": {"-": ["17"]},
    "HYPERTHYRVM": {"-": ["12"]},
    "HYPOCAVSIS": {"-": ["17"]},
    "HYPOGEVM": {"-": ["12"]},
    "HYPOMOCHLION": {"-": ["17"]},
    "HYPOTRACHELIVM": {"-": ["17"]},
    "HYSGINVM": {"-": ["12"]},
    "IACCHVS": {"N": ["12"]},
    "IACEO": {"-": ["52"]},
    "IACIO": {"-": ["55"]},
    "IACTANS": {"-": ["25"]},
    "IACTANTER": {"-": ["6&"]},
    "IACTANTIA": {"-": ["11"]},
    "IACTATIO": {"-": ["13"]},
    "IACTATVS": {"-": ["14"]},
    "IACTO": {"-": ["51"]},
    "IACTVRA": {"-": ["11"]},
    "IACTVS": {"-": ["14"]},
    "IACVLATOR": {"-": ["13"]},
    "IACVLATRIX": {"-": ["13"]},
    "IACVLOR": {"-": ["5J"]},
    "IACVLVM": {"-": ["12"]},
    "IACVLVS": {"1": ["12"], "2": ["21"]},
    "IALYSVS": {"N": ["12"]},
    "IAM": {"-": ["60"]},
    "IAMBVS": {"-": ["12"]},
    "IAMDIV": {"-": ["60"]},
    "IAMDVDVM": {"-": ["60"]},
    "IAMIAM": {"-": ["60"]},
    "IAMPRIDEM": {"-": ["60"]},
    "IANICVLVM": {"N": ["12"]},
    "IANITOR": {"-": ["13"], "N": ["13"]},
    "IANVA": {"-": ["11"]},
    "IANVARIVS": {"A": ["21"], "N": ["12"]},
    "IANVS": {"N": ["12"], "-": ["12"]},
    "IASPIS": {"-": ["17"]},
    "IBERVS": {"N": ["12"], "A": ["21"]},
    "IBI": {"-": ["60"]},
    "IBIDEM": {"-": ["60"]},
    "IBIS": {"-": ["17"], "N": ["13"]},
    "ICHNEVMON": {"-": ["17"]},
    "ICHNOGRAPHIA": {"-": ["11"]},
    "ICILIVS": {"N": ["12"]},
    "ICIO": {"-": ["55"]},
    "ICO": {"-": ["53"]},
    "ICONISMVS": {"-": ["12"]},
    "ICTERICVS": {"-": ["21"]},
    "ICTVS": {"-": ["14"]},
    "IDAEVS": {"A": ["21"], "N": ["12"]},
    "IDCIRCO": {"-": ["60"]},
    "IDEA": {"-": ["17"]},
    "IDEM": {"-": ["45"]},
    "IDENTIDEM": {"-": ["60"]},
    "IDEO": {"-": ["60"]},
    "IDIOTA": {"-": ["17"]},
    "IDONEVS": {"-": ["21"]},
    "IDOS": {"-": ["17"]},
    "IDVS": {"-": ["14"]},
    "IECTO": {"-": ["51"]},
    "IECVR": {"-": ["13"]},
    "IEIVNITAS": {"-": ["13"]},
    "IEIVNIVM": {"-": ["12"]},
    "IEIVNVS": {"-": ["21"]},
    "IETINI": {"N": ["12"]},
    "IGITVR": {"-": ["81"]},
    "IGNARVS": {"-": ["21"]},
    "IGNAVE": {"-": ["60"]},
    "IGNAVIA": {"-": ["11"]},
    "IGNAVITER": {"-": ["60"]},
    "IGNAVVS": {"1": ["12"], "2": ["21"]},
    "IGNESCO": {"-": ["53"]},
    "IGNEVS": {"-": ["21"]},
    "IGNIARIVM": {"-": ["12"]},
    "IGNICVLVS": {"-": ["12"]},
    "IGNIFER": {"-": ["21"]},
    "IGNIS": {"-": ["13"]},
    "IGNOBILIS": {"-": ["24"]},
    "IGNOBILITAS": {"-": ["13"]},
    "IGNOMINIA": {"-": ["11"]},
    "IGNOMINIOSVS": {"-": ["21"]},
    "IGNORANS": {"-": ["25"]},
    "IGNORANTIA": {"-": ["11"]},
    "IGNORATIO": {"-": ["13"]},
    "IGNORO": {"-": ["51"]},
    "IGNOSCO": {"-": ["53"]},
    "IGNOTITIA": {"-": ["11"]},
    "IGNOTVS": {"-": ["21"]},
    "IGO": {"-": ["53"]},
    "ILEX": {"-": ["13"]},
    "ILIA": {"-": ["16"], "N": ["11"]},
    "ILICET": {"-": ["60"]},
    "ILIGNEVS": {"-": ["21"]},
    "ILIGNVS": {"-": ["21"]},
    "ILIVM": {"-": ["12"], "N": ["12"]},
    "ILLA": {"-": ["60"]},
    "ILLABOR": {"2": ["5L"]},
    "ILLABORATVS": {"-": ["21"]},
    "ILLABORO": {"-": ["51"]},
    "ILLAC": {"-": ["60"]},
    "ILLACESSITVS": {"-": ["21"]},
    "ILLACRIMABILIS": {"-": ["24"]},
    "ILLACRIMO": {"-": ["51"]},
    "ILLACRIMOR": {"-": ["5J"]},
    "ILLAESVS": {"-": ["21"]},
    "ILLAETABILIS": {"-": ["24"]},
    "ILLAQVEO": {"2": ["51"]},
    "ILLARGIO": {"-": ["54"]},
    "ILLAVDATVS": {"-": ["21"]},
    "ILLAVTVS": {"-": ["21"]},
    "ILLE": {"-": ["45"]},
    "ILLECEBRA": {"-": ["11"]},
    "ILLECTO": {"-": ["51"]},
    "ILLECTVS": {"1": ["14"], "2": ["21"]},
    "ILLEPIDE": {"-": ["60"]},
    "ILLEPIDVS": {"-": ["21"]},
    "ILLI": {"-": ["60"]},
    "ILLIBATVS": {"-": ["21"]},
    "ILLIBERALIS": {"-": ["24"]},
    "ILLIC": {"1": ["45"], "2": ["60"]},
    "ILLICIO": {"-": ["55"]},
    "ILLICITE": {"1": ["60"], "2": ["60"]},
    "ILLICITVS": {"-": ["21"]},
    "ILLICO": {"-": ["60"]},
    "ILLIDO": {"-": ["53"]},
    "ILLIGO": {"2": ["51"]},
    "ILLIM": {"-": ["60"]},
    "ILLINC": {"-": ["60"]},
    "ILLINO": {"-": ["53"]},
    "ILLITTERATVS": {"-": ["21"]},
    "ILLIVSMODI": {"-": ["60"]},
    "ILLO": {"-": ["60"]},
    "ILLOC": {"-": ["60"]},
    "ILLOTVS": {"-": ["21"]},
    "ILLVC": {"-": ["60"]},
    "ILLVCEO": {"-": ["52"]},
    "ILLVCESCO": {"-": ["53"]},
    "ILLVDO": {"-": ["53"]},
    "ILLVMINO": {"-": ["51"]},
    "ILLVO": {"-": ["53"]},
    "ILLVSTRIS": {"-": ["24"]},
    "ILLVSTRO": {"2": ["51"]},
    "ILLVVIES": {"-": ["15"]},
    "ILLYRICVM": {"N": ["12"]},
    "IMACHARENSES": {"N": ["13"]},
    "IMACHARENSIS": {"N": ["24"], "A": ["24"]},
    "IMAGINARIVS": {"-": ["21"]},
    "IMAGINATIO": {"-": ["13"]},
    "IMAGINOR": {"-": ["5J"]},
    "IMAGINOSVS": {"-": ["21"]},
    "IMAGO": {"-": ["13"]},
    "IMBECILLI": {"-": ["12"]},
    "IMBECILLIS": {"-": ["24"]},
    "IMBECILLITAS": {"-": ["13"]},
    "IMBECILLVS": {"-": ["21"]},
    "IMBELLIS": {"-": ["24"]},
    "IMBER": {"-": ["13"]},
    "IMBERBIS": {"-": ["24"]},
    "IMBIBO": {"2": ["53"]},
    "IMBREX": {"-": ["13"]},
    "IMBRICATVS": {"-": ["21"]},
    "IMBRIFER": {"-": ["21"]},
    "IMBVO": {"-": ["53"]},
    "IMITABILIS": {"-": ["24"]},
    "IMITAMENTVM": {"-": ["12"]},
    "IMITATIO": {"-": ["13"]},
    "IMITATOR": {"-": ["13"]},
    "IMITATRIX": {"-": ["13"]},
    "IMITOR": {"-": ["5J"]},
    "IMMANE": {"-": ["60"]},
    "IMMANIS": {"-": ["24"]},
    "IMMANITAS": {"-": ["13"]},
    "IMMANSVETVS": {"-": ["21"]},
    "IMMATVRVS": {"-": ["21"]},
    "IMMEDIATVS": {"-": ["21"]},
    "IMMEIO": {"-": ["53"]},
    "IMMEMOR": {"-": ["26"]},
    "IMMEMORABILIS": {"-": ["24"]},
    "IMMENSVM": {"1": ["12"], "2": ["60"]},
    "IMMENSVS": {"-": ["21"]},
    "IMMERENS": {"-": ["25"]},
    "IMMERGO": {"-": ["53"]},
    "IMMERITO": {"-": ["60"]},
    "IMMERITVS": {"-": ["21"]},
    "IMMERSABILIS": {"-": ["24"]},
    "IMMETATVS": {"-": ["21"]},
    "IMMIGRO": {"-": ["51"]},
    "IMMINEO": {"-": ["52"]},
    "IMMINVO": {"-": ["53"]},
    "IMMISCEO": {"-": ["52"]},
    "IMMISERABILIS": {"-": ["24"]},
    "IMMISSARIVM": {"-": ["12"]},
    "IMMITIS": {"-": ["24"]},
    "IMMITTO": {"-": ["53"]},
    "IMMO": {"-": ["60"]},
    "IMMOBILIS": {"-": ["24"]},
    "IMMODERATE": {"-": ["60"]},
    "IMMODERATVS": {"-": ["21"]},
    "IMMODESTIA": {"-": ["11"]},
    "IMMODESTVS": {"-": ["21"]},
    "IMMODICE": {"-": ["60"]},
    "IMMODICVS": {"-": ["21"]},
    "IMMOLATIO": {"-": ["13"]},
    "IMMOLO": {"-": ["51"]},
    "IMMORDEO": {"-": ["52"]},
    "IMMORIOR": {"-": ["5N"]},
    "IMMOROR": {"-": ["5J"], "1": ["5J"]},
    "IMMORTALIS": {"-": ["24"]},
    "IMMORTALITAS": {"-": ["13"]},
    "IMMOTVS": {"-": ["21"]},
    "IMMVGIO": {"-": ["54"]},
    "IMMVNDVS": {"-": ["21"]},
    "IMMVNIO": {"2": ["54"]},
    "IMMVNIS": {"-": ["24"]},
    "IMMVNITAS": {"-": ["13"]},
    "IMMVNITVS": {"-": ["21"]},
    "IMMVRMVRO": {"-": ["51"]},
    "IMMVTABILIS": {"1": ["24"], "2": ["24"]},
    "IMMVTATIO": {"-": ["13"]},
    "IMMVTO": {"2": ["51"]},
    "IMO": {"-": ["53"]},
    "IMPACATVS": {"-": ["21"]},
    "IMPAGES": {"-": ["13"]},
    "IMPALLESCO": {"-": ["53"]},
    "IMPAR": {"-": ["25"]},
    "IMPARATVS": {"-": ["21"]},
    "IMPATIENS": {"-": ["25"]},
    "IMPATIENTER": {"-": ["60"]},
    "IMPATIENTIA": {"-": ["11"]},
    "IMPAVIDE": {"-": ["60"]},
    "IMPAVIDVS": {"-": ["21"]},
    "IMPEDIMENTVM": {"-": ["12"]},
    "IMPEDIO": {"-": ["54"]},
    "IMPEDITIO": {"-": ["13"]},
    "IMPEDITVM": {"-": ["12"]},
    "IMPEDITVS": {"-": ["21"]},
    "IMPELLO": {"-": ["53"]},
    "IMPENDEO": {"-": ["52"]},
    "IMPENDIVM": {"-": ["12"]},
    "IMPENDO": {"-": ["53"]},
    "IMPENETRABILIS": {"-": ["24"]},
    "IMPENSA": {"-": ["11"]},
    "IMPENSE": {"-": ["60"]},
    "IMPENSVS": {"-": ["21"]},
    "IMPERATOR": {"-": ["13"], "N": ["13"]},
    "IMPERATORIVS": {"-": ["21"]},
    "IMPERATVM": {"-": ["12"]},
    "IMPERCVSSVS": {"-": ["21"]},
    "IMPERFECTVS": {"-": ["21"]},
    "IMPERIOSVS": {"-": ["21"]},
    "IMPERITIA": {"-": ["11"]},
    "IMPERITO": {"-": ["51"]},
    "IMPERITVS": {"-": ["21"]},
    "IMPERIVM": {"-": ["12"]},
    "IMPERMISSVS": {"-": ["21"]},
    "IMPERO": {"-": ["51"]},
    "IMPERPETVVS": {"-": ["21"]},
    "IMPERTIO": {"-": ["54"]},
    "IMPERTVRBATVS": {"-": ["21"]},
    "IMPERVIVS": {"-": ["21"]},
    "IMPES": {"-": ["13"]},
    "IMPETRABILIS": {"-": ["24"]},
    "IMPETRO": {"-": ["51"]},
    "IMPETVS": {"-": ["14"]},
    "IMPEXVS": {"-": ["21"]},
    "IMPIE": {"-": ["60"]},
    "IMPIETAS": {"-": ["13"], "N": ["13"]},
    "IMPIGER": {"-": ["21"]},
    "IMPIGRE": {"-": ["60"]},
    "IMPINGO": {"-": ["53"]},
    "IMPIO": {"-": ["51"]},
    "IMPIVS": {"-": ["21"]},
    "IMPLACABILIS": {"-": ["24"]},
    "IMPLACABILITER": {"-": ["6&"]},
    "IMPLACATVS": {"-": ["21"]},
    "IMPLACIDVS": {"-": ["21"]},
    "IMPLECTO": {"2": ["53"]},
    "IMPLEO": {"-": ["52"]},
    "IMPLICATVS": {"-": ["21"]},
    "IMPLICITVS": {"-": ["21"]},
    "IMPLICO": {"-": ["51"]},
    "IMPLORATIO": {"-": ["13"]},
    "IMPLORO": {"-": ["51"]},
    "IMPLVMBO": {"-": ["51"]},
    "IMPLVMIS": {"-": ["24"]},
    "IMPLVO": {"-": ["53"]},
    "IMPLVVIVM": {"-": ["12"]},
    "IMPOLITVS": {"-": ["21"]},
    "IMPOLLVTVS": {"-": ["21"]},
    "IMPONO": {"-": ["53"]},
    "IMPORTATICIVS": {"-": ["21"]},
    "IMPORTO": {"-": ["51"]},
    "IMPORTVNITAS": {"-": ["13"]},
    "IMPORTVNVS": {"-": ["21"]},
    "IMPORTVOSVS": {"-": ["21"]},
    "IMPOS": {"-": ["22"]},
    "IMPOTENS": {"-": ["25"]},
    "IMPOTENTER": {"-": ["60"]},
    "IMPOTENTIA": {"-": ["11"]},
    "IMPRAESENTIARVM": {"-": ["60"]},
    "IMPRANSVS": {"-": ["21"]},
    "IMPRECATIO": {"-": ["13"]},
    "IMPRECOR": {"-": ["5J"]},
    "IMPRESSIO": {"-": ["13"]},
    "IMPRESSO": {"-": ["51"]},
    "IMPRIMIS": {"-": ["60"]},
    "IMPRIMO": {"-": ["53"]},
    "IMPROBABILIS": {"-": ["24"]},
    "IMPROBATIO": {"-": ["13"]},
    "IMPROBE": {"-": ["60"]},
    "IMPROBITAS": {"-": ["13"]},
    "IMPROBO": {"-": ["51"]},
    "IMPROBVLVS": {"-": ["21"]},
    "IMPROBVS": {"-": ["21"]},
    "IMPROCERVS": {"-": ["21"]},
    "IMPROMPTVS": {"-": ["21"]},
    "IMPROPERATVS": {"-": ["21"]},
    "IMPROPERO": {"-": ["51"]},
    "IMPROSPER": {"-": ["21"]},
    "IMPROSPERE": {"-": ["60"]},
    "IMPROVIDE": {"-": ["60"]},
    "IMPROVIDVS": {"-": ["21"]},
    "IMPROVISO": {"-": ["60"]},
    "IMPROVISVM": {"-": ["12"]},
    "IMPROVISVS": {"-": ["21"]},
    "IMPRVDENS": {"-": ["25"]},
    "IMPRVDENTIA": {"-": ["11"]},
    "IMPVBES": {"-": ["26"]},
    "IMPVDENS": {"-": ["25"]},
    "IMPVDENTER": {"-": ["60"]},
    "IMPVDENTIA": {"-": ["11"]},
    "IMPVDICITIA": {"-": ["11"]},
    "IMPVDICVS": {"-": ["21"]},
    "IMPVGNO": {"-": ["51"]},
    "IMPVLSOR": {"-": ["13"]},
    "IMPVLSVS": {"-": ["14"]},
    "IMPVNE": {"-": ["60"]},
    "IMPVNIS": {"-": ["24"]},
    "IMPVNITAS": {"-": ["13"]},
    "IMPVNITVS": {"-": ["21"]},
    "IMPVRVS": {"-": ["21"]},
    "IMPVTATOR": {"-": ["13"]},
    "IMPVTATVS": {"-": ["21"]},
    "IMPVTO": {"-": ["51"]},
    "IMVLVS": {"-": ["21"]},
    "IMVM": {"-": ["12"]},
    "IN": {"-": ["70"]},
    "INACCEDO": {"1": ["53"]},
    "INACCESSVS": {"-": ["21"]},
    "INACESCO": {"-": ["53"]},
    "INADVSTVS": {"-": ["21"]},
    "INAEDIFICO": {"-": ["51"]},
    "INAEQVABILIS": {"-": ["24"]},
    "INAEQVALIS": {"-": ["24"]},
    "INAEQVALITAS": {"-": ["13"]},
    "INAEQVALITER": {"-": ["60"]},
    "INAEQVATVS": {"-": ["21"]},
    "INAEQVO": {"-": ["51"]},
    "INAESTIMABILIS": {"-": ["24"]},
    "INAESTVO": {"-": ["51"]},
    "INAFFECTATVS": {"-": ["21"]},
    "INAFFECTO": {"-": ["51"]},
    "INAGITATVS": {"-": ["21"]},
    "INAMABILIS": {"-": ["24"]},
    "INAMARESCO": {"-": ["53"]},
    "INAMBVLATIO": {"-": ["13"]},
    "INAMBVLO": {"-": ["51"]},
    "INAMOENVS": {"-": ["21"]},
    "INANE": {"-": ["13"]},
    "INANIMALIS": {"-": ["24"]},
    "INANIMATVS": {"-": ["21"]},
    "INANIMIS": {"-": ["24"]},
    "INANIMVS": {"-": ["21"]},
    "INANIO": {"-": ["54"]},
    "INANIS": {"-": ["24"]},
    "INANITAS": {"-": ["13"]},
    "INANITER": {"-": ["60"]},
    "INARATVS": {"-": ["21"]},
    "INARDEO": {"-": ["52"]},
    "INARDESCO": {"-": ["53"]},
    "INARESCO": {"-": ["53"]},
    "INARO": {"-": ["51"]},
    "INASCENSVS": {"-": ["21"]},
    "INASSVETVS": {"-": ["21"]},
    "INAVDAX": {"-": ["25"]},
    "INAVDIO": {"-": ["54"]},
    "INAVDITVS": {"-": ["21"]},
    "INAVRATVS": {"-": ["21"]},
    "INAVRO": {"-": ["51"]},
    "INAVSPICATVS": {"-": ["21"]},
    "INAVSVS": {"-": ["21"]},
    "INCAEDVVS": {"-": ["21"]},
    "INCALESCO": {"-": ["53"]},
    "INCALLIDVS": {"-": ["21"]},
    "INCANDEO": {"-": ["52"]},
    "INCANDESCO": {"-": ["53"]},
    "INCANESCO": {"-": ["53"]},
    "INCANTO": {"-": ["51"]},
    "INCANVS": {"-": ["21"]},
    "INCASSVM": {"-": ["60"]},
    "INCASTIGO": {"-": ["51"]},
    "INCAVTE": {"-": ["60"]},
    "INCAVTVS": {"-": ["21"]},
    "INCEDO": {"1": ["53"]},
    "INCELEBRATVS": {"-": ["21"]},
    "INCENATVS": {"-": ["21"]},
    "INCENDIARIVS": {"1": ["12"], "2": ["21"]},
    "INCENDIVM": {"-": ["12"]},
    "INCENDO": {"-": ["53"]},
    "INCENSIO": {"-": ["13"]},
    "INCENSVM": {"-": ["12"]},
    "INCENSVS": {"1": ["21"], "2": ["21"]},
    "INCEPTVM": {"-": ["12"]},
    "INCEPTVS": {"1": ["14"], "2": ["21"]},
    "INCERNICVLVM": {"-": ["12"]},
    "INCERNO": {"-": ["53"]},
    "INCERO": {"-": ["51"]},
    "INCERTVM": {"-": ["12"]},
    "INCERTVS": {"-": ["21"]},
    "INCESSO": {"-": ["53"]},
    "INCESSVS": {"-": ["14"]},
    "INCESTE": {"-": ["60"]},
    "INCESTIFICVS": {"-": ["21"]},
    "INCESTO": {"-": ["51"]},
    "INCESTVM": {"-": ["12"]},
    "INCESTVS": {"1": ["14"], "2": ["21"]},
    "INCHOO": {"-": ["51"]},
    "INCIDENTIA": {"-": ["13", "11"]},
    "INCIDO": {"1": ["53"], "2": ["53"]},
    "INCIEO": {"-": ["52"]},
    "INCILE": {"-": ["13"]},
    "INCILIS": {"-": ["24"]},
    "INCILO": {"-": ["51"]},
    "INCINGO": {"-": ["53"]},
    "INCINO": {"-": ["53"]},
    "INCIO": {"-": ["54"]},
    "INCIPIO": {"-": ["55"]},
    "INCISVM": {"-": ["12"], "L": ["12"]},
    "INCITAMENTVM": {"-": ["12"]},
    "INCITATIO": {"-": ["13"]},
    "INCITATVS": {"1": ["14"], "2": ["21"]},
    "INCITO": {"1": ["51"]},
    "INCITVS": {"1": ["21"], "2": ["21"]},
    "INCLAMO": {"-": ["51"]},
    "INCLARESCO": {"-": ["53"]},
    "INCLEMENS": {"-": ["25"]},
    "INCLEMENTER": {"-": ["60"]},
    "INCLEMENTIA": {"-": ["11"]},
    "INCLINABILIS": {"-": ["24"]},
    "INCLINATIO": {"-": ["13"]},
    "INCLINATVS": {"1": ["14"], "2": ["21"]},
    "INCLINO": {"-": ["51"]},
    "INCLVDO": {"-": ["53"]},
    "INCLVTVS": {"-": ["21"]},
    "INCOACTVS": {"-": ["21"]},
    "INCOGITATVS": {"-": ["21"]},
    "INCOGITO": {"-": ["51"]},
    "INCOGNITVS": {"-": ["21"]},
    "INCOHIBEO": {"-": ["52"]},
    "INCOHO": {"-": ["51"]},
    "INCOLA": {"-": ["11"]},
    "INCOLO": {"1": ["51"], "2": ["53"]},
    "INCOLVMIS": {"-": ["24"]},
    "INCOLVMITAS": {"-": ["13"]},
    "INCOMITATVS": {"-": ["21"]},
    "INCOMMODE": {"-": ["60"]},
    "INCOMMODVM": {"-": ["12"]},
    "INCOMMODVS": {"-": ["21"]},
    "INCOMPERTVS": {"-": ["21"]},
    "INCOMPOSITVS": {"-": ["21"]},
    "INCOMPREHENSIBILIS": {"-": ["24"]},
    "INCOMPTVS": {"-": ["21"]},
    "INCONCESSVS": {"-": ["21"]},
    "INCONCINNVS": {"-": ["21"]},
    "INCONCVSSVS": {"-": ["21"]},
    "INCONDEMNATVS": {"-": ["21"]},
    "INCONDITVS": {"-": ["21"]},
    "INCONFVSVS": {"-": ["21"]},
    "INCONSIDERATVS": {"-": ["21"]},
    "INCONSTANS": {"-": ["25"]},
    "INCONSTANTER": {"-": ["60"]},
    "INCONSTANTIA": {"-": ["11"]},
    "INCONSVETVS": {"-": ["21"]},
    "INCONSVLTE": {"-": ["60"]},
    "INCONSVLTVS": {"1": ["14"], "2": ["21"]},
    "INCONTAMINATVS": {"-": ["21"]},
    "INCONTINENS": {"-": ["25"]},
    "INCONVENIENS": {"-": ["25"]},
    "INCOQVO": {"-": ["53"]},
    "INCORPORALIS": {"-": ["24"]},
    "INCORRVPTVS": {"-": ["21"]},
    "INCREBRESCO": {"-": ["53"]},
    "INCREDIBILIS": {"-": ["24"]},
    "INCREDIBILITER": {"-": ["60"]},
    "INCREDVLVS": {"-": ["21"]},
    "INCREMENTVM": {"-": ["12"]},
    "INCREPATVS": {"-": ["21"]},
    "INCREPITO": {"-": ["51"]},
    "INCREPITVS": {"1": ["14"], "2": ["21"]},
    "INCREPO": {"-": ["51"]},
    "INCRESCO": {"-": ["53"]},
    "INCRVENTVS": {"-": ["21"]},
    "INCRVSTO": {"-": ["51"]},
    "INCVBO": {"1": ["13"], "2": ["51"], "N": ["13"]},
    "INCVLCO": {"-": ["51"]},
    "INCVLTE": {"-": ["6&"]},
    "INCVLTVS": {"1": ["14"], "2": ["21"]},
    "INCVMBA": {"-": ["11"]},
    "INCVMBO": {"-": ["53"]},
    "INCVNABVLA": {"-": ["12"]},
    "INCVRIA": {"-": ["11"]},
    "INCVRIOSVS": {"-": ["21"]},
    "INCVRO": {"-": ["51"]},
    "INCVRRO": {"-": ["53"]},
    "INCVRSIO": {"-": ["13"]},
    "INCVRSITO": {"-": ["51"]},
    "INCVRSO": {"-": ["51"]},
    "INCVRSVS": {"-": ["14"]},
    "INCVRVO": {"-": ["51"]},
    "INCVRVVS": {"-": ["21"]},
    "INCVS": {"-": ["13"]},
    "INCVSATIO": {"-": ["13"]},
    "INCVSO": {"-": ["51"]},
    "INCVSSVS": {"-": ["14"]},
    "INCVSTODITVS": {"-": ["21"]},
    "INCVSVS": {"-": ["21"]},
    "INCVTIO": {"-": ["55"]},
    "INDAGATIO": {"-": ["13"]},
    "INDAGATOR": {"-": ["13"]},
    "INDAGATRIX": {"-": ["13"]},
    "INDAGO": {"1": ["13"], "2": ["51"]},
    "INDE": {"-": ["60"]},
    "INDEBITVS": {"-": ["21"]},
    "INDECENS": {"-": ["25"]},
    "INDECENTER": {"-": ["6&"]},
    "INDECENTIA": {"-": ["11"]},
    "INDECLINABILIS": {"-": ["24"]},
    "INDECORE": {"-": ["60"]},
    "INDECORO": {"-": ["51"]},
    "INDECORVS": {"-": ["21"]},
    "INDEFATIGABILIS": {"-": ["24"]},
    "INDEFATIGATVS": {"-": ["21"]},
    "INDEFENSVS": {"-": ["21"]},
    "INDEFESSVS": {"-": ["21"]},
    "INDEFLEXVS": {"-": ["21"]},
    "INDELEBILIS": {"-": ["24"]},
    "INDEMNIS": {"-": ["24"]},
    "INDEPRAVATVS": {"-": ["21"]},
    "INDEPREHENSVS": {"-": ["21"]},
    "INDESERTVS": {"-": ["21"]},
    "INDEX": {"-": ["13"]},
    "INDIA": {"N": ["11"]},
    "INDICIVM": {"-": ["12"]},
    "INDICO": {"1": ["51"], "2": ["53"]},
    "INDICTIO": {"-": ["13"]},
    "INDICTVS": {"-": ["21"]},
    "INDIDEM": {"-": ["60"]},
    "INDIFFERENS": {"-": ["25"]},
    "INDIGENA": {"-": ["11"]},
    "INDIGENS": {"1": ["13"], "2": ["25"]},
    "INDIGENTIA": {"-": ["11"]},
    "INDIGEO": {"-": ["52"]},
    "INDIGES": {"N": ["13"]},
    "INDIGESTVS": {"-": ["21"]},
    "INDIGNANDVS": {"-": ["21"]},
    "INDIGNATIO": {"-": ["13"]},
    "INDIGNE": {"-": ["60"]},
    "INDIGNITAS": {"-": ["13"]},
    "INDIGNOR": {"-": ["5J"]},
    "INDIGNVS": {"-": ["21"]},
    "INDIGVS": {"-": ["21"]},
    "INDILIGENS": {"-": ["25"]},
    "INDILIGENTER": {"-": ["6&"]},
    "INDILIGENTIA": {"-": ["11"]},
    "INDIPISCOR": {"-": ["5L"]},
    "INDISCRETVS": {"-": ["21"]},
    "INDISPOSITE": {"-": ["60"]},
    "INDISTINCTVS": {"-": ["21"]},
    "INDIVIDVVS": {"-": ["21"]},
    "INDIVISVS": {"-": ["21"]},
    "INDO": {"-": ["53"]},
    "INDOCILIS": {"-": ["24"]},
    "INDOCTVS": {"-": ["21"]},
    "INDOLENTIA": {"-": ["11"]},
    "INDOLES": {"-": ["13"]},
    "INDOLESCO": {"-": ["53"]},
    "INDOMITVS": {"-": ["21"]},
    "INDORMIO": {"-": ["54"]},
    "INDOTATVS": {"-": ["21"]},
    "INDVBITATVS": {"-": ["21"]},
    "INDVBIVS": {"-": ["21"]},
    "INDVCO": {"-": ["53"]},
    "INDVCTIO": {"-": ["13"]},
    "INDVCTVS": {"-": ["14"]},
    "INDVLGENS": {"-": ["25"]},
    "INDVLGENTER": {"-": ["6&"]},
    "INDVLGENTIA": {"-": ["11"]},
    "INDVLGEO": {"-": ["52"]},
    "INDVO": {"-": ["53"]},
    "INDVRATVS": {"-": ["21"]},
    "INDVRESCO": {"-": ["53"]},
    "INDVRO": {"-": ["51"]},
    "INDVS": {"A": ["21"], "N": ["12"]},
    "INDVSTRIA": {"-": ["11"]},
    "INDVSTRIE": {"-": ["60"]},
    "INDVSTRIOSE": {"-": ["60"]},
    "INDVSTRIVS": {"-": ["21"]},
    "INDVTIAE": {"-": ["11"]},
    "INDVTILIS": {"-": ["24"]},
    "INDVTVS": {"-": ["14"]},
    "INEBRIO": {"-": ["51"]},
    "INEDIA": {"-": ["11"]},
    "INEFFICAX": {"-": ["25"]},
    "INELABORATVS": {"-": ["21"]},
    "INELEGANS": {"-": ["25"]},
    "INELVCTABILIS": {"-": ["24"]},
    "INEMENDABILIS": {"-": ["24"]},
    "INEMORIOR": {"-": ["5J"]},
    "INEMPTVS": {"-": ["21"]},
    "INENSES": {"N": ["13"]},
    "INEO": {"1": ["56"]},
    "INEPTE": {"-": ["60"]},
    "INEPTIAE": {"-": ["11"]},
    "INEPTIO": {"-": ["54"]},
    "INEPTVS": {"-": ["21"]},
    "INERMIS": {"-": ["24"]},
    "INERMVS": {"-": ["21"]},
    "INERRANS": {"-": ["25"]},
    "INERRO": {"2": ["51"]},
    "INERS": {"-": ["25"]},
    "INERTIA": {"-": ["11"]},
    "INESCO": {"-": ["51"]},
    "INEVITABILIS": {"-": ["24"]},
    "INEXCITABILIS": {"-": ["24"]},
    "INEXCVSABILIS": {"-": ["24"]},
    "INEXERCITATVS": {"-": ["21"]},
    "INEXERCITO": {"-": ["51"]},
    "INEXHAVSTVS": {"-": ["21"]},
    "INEXORABILIS": {"-": ["24"]},
    "INEXPERTVS": {"-": ["21"]},
    "INEXPIABILIS": {"-": ["24"]},
    "INEXPLEBILIS": {"-": ["24"]},
    "INEXPLICABILIS": {"-": ["24"]},
    "INEXPLORATO": {"-": ["60"]},
    "INEXPLORATVS": {"-": ["21"]},
    "INEXPVGNABILIS": {"-": ["24"]},
    "INEXSPECTATVS": {"-": ["21"]},
    "INEXSVPERABILIS": {"-": ["24"]},
    "INEXTRICABILIS": {"-": ["24"]},
    "INFABRE": {"-": ["60"]},
    "INFABRICATVS": {"-": ["21"]},
    "INFACETIAE": {"-": ["11"]},
    "INFACETVS": {"-": ["21"]},
    "INFAMIA": {"-": ["11"]},
    "INFAMIS": {"-": ["24"]},
    "INFAMO": {"-": ["51"]},
    "INFANDVS": {"-": ["21"]},
    "INFANS": {"1": ["13"], "2": ["25"]},
    "INFANTIA": {"-": ["11"]},
    "INFATIGABILIS": {"-": ["24"]},
    "INFATVO": {"-": ["51"]},
    "INFAVSTVS": {"-": ["21"]},
    "INFECTIVVS": {"-": ["21"]},
    "INFECTVS": {"1": ["14"], "2": ["21"]},
    "INFECVNDITAS": {"-": ["13"]},
    "INFECVNDVS": {"-": ["21"]},
    "INFELICITAS": {"-": ["13"]},
    "INFELICITER": {"-": ["60"]},
    "INFELIX": {"-": ["25"]},
    "INFENDO": {"-": ["53"]},
    "INFENSE": {"-": ["6&"]},
    "INFENSO": {"-": ["51"]},
    "INFENSVS": {"-": ["21"]},
    "INFERI": {"-": ["12"], "N": ["12"]},
    "INFERIAE": {"-": ["11"]},
    "INFERIORA": {"-": ["13"]},
    "INFERIORES": {"-": ["13"]},
    "INFERIVS": {"-": ["21"]},
    "INFERNA": {"-": ["12"]},
    "INFERNAS": {"-": ["22"]},
    "INFERNE": {"-": ["60"]},
    "INFERNI": {"-": ["12"]},
    "INFERNVS": {"-": ["21"]},
    "INFERO": {"-": ["56"]},
    "INFERVEFACIO": {"-": ["55"]},
    "INFERVEO": {"-": ["52"]},
    "INFERVESCO": {"-": ["53"]},
    "INFERVS": {"-": ["21"]},
    "INFESTE": {"-": ["6&"]},
    "INFESTO": {"-": ["51"]},
    "INFESTVS": {"-": ["21"]},
    "INFEXIO": {"-": ["13"]},
    "INFICIO": {"-": ["55"]},
    "INFICISCO": {"-": ["53"]},
    "INFIDELIS": {"-": ["24"]},
    "INFIDELITAS": {"-": ["13"]},
    "INFIDVS": {"-": ["21"]},
    "INFIGO": {"-": ["53"]},
    "INFIMVM": {"-": ["12"]},
    "INFINDO": {"-": ["53"]},
    "INFINITVM": {"-": ["60"]},
    "INFINITVS": {"-": ["21"]},
    "INFIO": {"-": ["56"]},
    "INFIRMIS": {"-": ["24"]},
    "INFIRMITAS": {"-": ["13"]},
    "INFIRMO": {"-": ["51"]},
    "INFIRMVS": {"-": ["21"]},
    "INFITIAS": {"-": ["16"]},
    "INFITIATIO": {"-": ["13"]},
    "INFITIATOR": {"-": ["13"]},
    "INFITIOR": {"-": ["5J"]},
    "INFLAMMO": {"-": ["51"]},
    "INFLATE": {"-": ["6&"]},
    "INFLATIO": {"-": ["13"]},
    "INFLATVS": {"1": ["14"], "2": ["21"]},
    "INFLECTIO": {"-": ["13"]},
    "INFLECTO": {"-": ["53"]},
    "INFLEXIBILIS": {"-": ["24"]},
    "INFLEXILIS": {"-": ["24"]},
    "INFLEXIO": {"-": ["13"]},
    "INFLIGO": {"-": ["53"]},
    "INFLO": {"-": ["51"]},
    "INFLVO": {"-": ["53"]},
    "INFODIO": {"-": ["55"]},
    "INFORMIS": {"-": ["24"]},
    "INFORMO": {"-": ["51"]},
    "INFRA": {"1": ["60"], "2": ["70"]},
    "INFRACTVS": {"-": ["21"]},
    "INFRAGILIS": {"-": ["24"]},
    "INFREMO": {"-": ["53"]},
    "INFRENDEO": {"-": ["52"]},
    "INFRENVS": {"-": ["21"]},
    "INFREQVENS": {"-": ["25"]},
    "INFREQVENTIA": {"-": ["11"]},
    "INFRINGO": {"-": ["53"]},
    "INFRIO": {"-": ["51"]},
    "INFRVCTVOSVS": {"-": ["21"]},
    "INFRVNITVS": {"-": ["21"]},
    "INFVLA": {"-": ["11"]},
    "INFVLCIO": {"-": ["54"]},
    "INFVNDIBVLVM": {"-": ["12"]},
    "INFVNDO": {"2": ["53"]},
    "INFVSCO": {"-": ["51"]},
    "INGEMINO": {"-": ["51"]},
    "INGEMISCO": {"-": ["53"]},
    "INGEMO": {"-": ["53"]},
    "INGENERO": {"-": ["51"]},
    "INGENIOSE": {"-": ["60"]},
    "INGENIOSVS": {"-": ["21"]},
    "INGENITVS": {"1": ["21"], "2": ["21"]},
    "INGENIVM": {"-": ["12"]},
    "INGENS": {"-": ["25"]},
    "INGENVE": {"-": ["60"]},
    "INGENVITAS": {"-": ["13"]},
    "INGENVVS": {"-": ["21"]},
    "INGERO": {"-": ["53"]},
    "INGLORIVS": {"-": ["21"]},
    "INGLVVIES": {"-": ["15"]},
    "INGRATE": {"-": ["60"]},
    "INGRATIS": {"-": ["60"]},
    "INGRATVS": {"-": ["21"]},
    "INGRAVESCO": {"-": ["53"]},
    "INGREDIOR": {"-": ["5N"]},
    "INGRESSVS": {"-": ["14"]},
    "INGRVO": {"-": ["53"]},
    "INGVEN": {"-": ["13"]},
    "INGVSTATVS": {"-": ["21"]},
    "INHABILIS": {"-": ["24"]},
    "INHABITO": {"-": ["51"]},
    "INHAEREO": {"-": ["52"]},
    "INHIBEO": {"-": ["52"]},
    "INHIO": {"-": ["51"]},
    "INHONESTVS": {"-": ["21"]},
    "INHONORIFICVS": {"-": ["21"]},
    "INHORREO": {"-": ["52"]},
    "INHORRESCO": {"-": ["53"]},
    "INHOSPITALIS": {"-": ["24"]},
    "INHOSPITVS": {"-": ["21"]},
    "INHVMANE": {"-": ["60"]},
    "INHVMANITAS": {"-": ["13"]},
    "INHVMANITER": {"-": ["60"]},
    "INHVMANVS": {"-": ["21"]},
    "INHVMATVS": {"-": ["21"]},
    "INIBI": {"-": ["60"]},
    "INICIO": {"-": ["55"]},
    "INIECTIO": {"-": ["13"]},
    "INIECTVS": {"-": ["14"]},
    "INIMICITIA": {"-": ["11"]},
    "INIMICO": {"-": ["51"]},
    "INIMICVS": {"1": ["12"], "2": ["21"]},
    "INIMO": {"-": ["53"]},
    "INIQVE": {"-": ["60"]},
    "INIQVITAS": {"-": ["13"]},
    "INIQVVS": {"1": ["12"], "2": ["21"]},
    "INITIAMENTA": {"-": ["12"]},
    "INITIO": {"-": ["51"]},
    "INITIVM": {"-": ["12"]},
    "INITVS": {"-": ["14"]},
    "INIVCVNDVS": {"-": ["21"]},
    "INIVDICATVS": {"-": ["21"]},
    "INIVNGO": {"-": ["53"]},
    "INIVRATVS": {"-": ["21"]},
    "INIVRIA": {"-": ["11"]},
    "INIVRIOSE": {"-": ["6&"]},
    "INIVRIOSVS": {"-": ["21"]},
    "INIVROSVS": {"-": ["21"]},
    "INIVSSVS": {"1": ["14"], "2": ["21"]},
    "INIVSTE": {"-": ["60"]},
    "INIVSTITIA": {"-": ["11"]},
    "INIVSTVM": {"-": ["12"]},
    "INIVSTVS": {"-": ["21"]},
    "INLEXILIS": {"-": ["24"]},
    "INNABILIS": {"-": ["24"]},
    "INNASCOR": {"-": ["5L"]},
    "INNATO": {"-": ["51"]},
    "INNECTO": {"-": ["53"]},
    "INNITOR": {"2": ["5L"]},
    "INNO": {"-": ["51"]},
    "INNOCENS": {"-": ["25"]},
    "INNOCENTER": {"-": ["60"]},
    "INNOCENTIA": {"-": ["11"]},
    "INNOCVE": {"-": ["60"]},
    "INNOCVVS": {"-": ["21"]},
    "INNOTESCO": {"-": ["53"]},
    "INNOXIVS": {"-": ["21"]},
    "INNVBILVS": {"-": ["21"]},
    "INNVBIS": {"-": ["24"]},
    "INNVBVS": {"-": ["21"]},
    "INNVMERABILIS": {"-": ["24"]},
    "INNVMERABILITER": {"-": ["60"]},
    "INNVMERALIS": {"-": ["24"]},
    "INNVMERVS": {"-": ["21"]},
    "INNVO": {"-": ["53"]},
    "INNVPTA": {"-": ["11"]},
    "INNVPTVS": {"-": ["21"]},
    "INNVTRIO": {"-": ["54"]},
    "INOBSEQVENS": {"-": ["25"]},
    "INOBSERVABILIS": {"-": ["24"]},
    "INOBSERVATVS": {"-": ["21"]},
    "INODORVS": {"-": ["21"]},
    "INOFFENSE": {"-": ["60"]},
    "INOFFENSVS": {"-": ["21"]},
    "INOFFICIOSVS": {"-": ["21"]},
    "INOLENS": {"-": ["25"]},
    "INOLESCO": {"-": ["53"]},
    "INOMINATVS": {"-": ["21"]},
    "INOPERATVS": {"-": ["21"]},
    "INOPIA": {"-": ["11"]},
    "INOPINANS": {"-": ["25"]},
    "INOPINATVM": {"-": ["12"]},
    "INOPINATVS": {"-": ["21"]},
    "INOPINVS": {"-": ["21"]},
    "INOPS": {"-": ["26"]},
    "INORDINATVS": {"-": ["21"]},
    "INORNATVS": {"-": ["21"]},
    "INQVIES": {"1": ["13"], "2": ["25"]},
    "INQVIETO": {"-": ["51"]},
    "INQVIETVS": {"-": ["21"]},
    "INQVILINVS": {"-": ["12"]},
    "INQVINAMENTVM": {"-": ["12"]},
    "INQVINATVS": {"-": ["21"]},
    "INQVINO": {"-": ["51"]},
    "INQVIO": {"-": ["56"]},
    "INQVIRO": {"-": ["53"]},
    "INQVISITIO": {"-": ["13"]},
    "INQVISITOR": {"-": ["13"]},
    "INREGILLVM": {"N": ["12"]},
    "INSAEPTVS": {"1": ["21"], "2": ["21"]},
    "INSALVBRIS": {"-": ["24"]},
    "INSANABILIS": {"-": ["24"]},
    "INSANE": {"-": ["60"]},
    "INSANIA": {"-": ["11"]},
    "INSANIO": {"-": ["54"]},
    "INSANVM": {"-": ["60"]},
    "INSANVS": {"-": ["21"]},
    "INSAPIENS": {"-": ["25"]},
    "INSATIABILIS": {"-": ["24"]},
    "INSATIABILITER": {"-": ["60"]},
    "INSCENDO": {"-": ["53"]},
    "INSCIENS": {"1": ["25"], "2": ["25"]},
    "INSCIENTIA": {"-": ["11"]},
    "INSCITE": {"-": ["60"]},
    "INSCITIA": {"-": ["11"]},
    "INSCITVS": {"-": ["21"]},
    "INSCIVS": {"-": ["21"]},
    "INSCRIBO": {"-": ["53"]},
    "INSCRIPTIO": {"-": ["13"]},
    "INSCVLPO": {"-": ["53"]},
    "INSECABILIS": {"-": ["24"]},
    "INSECO": {"-": ["51"]},
    "INSECTATIO": {"-": ["13"]},
    "INSECTOR": {"2": ["5J"]},
    "INSEDABILITER": {"-": ["60"]},
    "INSEMINO": {"-": ["51"]},
    "INSENESCO": {"-": ["53"]},
    "INSENSIBILIS": {"-": ["24"]},
    "INSEPARABILIS": {"-": ["24"]},
    "INSEPVLTVS": {"1": ["21"], "2": ["21"]},
    "INSEQVOR": {"-": ["5L"]},
    "INSERO": {"2": ["53"], "3": ["53"]},
    "INSERTO": {"-": ["51"]},
    "INSERVIO": {"-": ["54"]},
    "INSIDEO": {"-": ["52"]},
    "INSIDIAE": {"-": ["11"], "N": ["11"]},
    "INSIDIATOR": {"-": ["13"]},
    "INSIDIOR": {"-": ["5J"]},
    "INSIDIOSE": {"-": ["6-"]},
    "INSIDIOSVS": {"-": ["21"]},
    "INSIDO": {"-": ["53"]},
    "INSIGNE": {"-": ["13"]},
    "INSIGNIO": {"-": ["54"]},
    "INSIGNIS": {"-": ["24"]},
    "INSIGNITA": {"-": ["12"]},
    "INSIGNITVS": {"-": ["21"]},
    "INSILIA": {"-": ["13"]},
    "INSILIO": {"-": ["54"]},
    "INSIMVLATIO": {"-": ["13"]},
    "INSIMVLO": {"-": ["51"]},
    "INSINCERVS": {"-": ["21"]},
    "INSINVO": {"-": ["51"]},
    "INSIPIENS": {"-": ["25"]},
    "INSIPIO": {"-": ["55"]},
    "INSISTO": {"-": ["53"]},
    "INSITICIVS": {"-": ["21"]},
    "INSITIO": {"-": ["13"]},
    "INSITIVVS": {"-": ["21"]},
    "INSITOR": {"-": ["13"]},
    "INSITVS": {"1": ["14"], "2": ["21"]},
    "INSOCIABILIS": {"-": ["24"]},
    "INSOLABILITER": {"-": ["60"]},
    "INSOLENS": {"-": ["25"]},
    "INSOLENTER": {"-": ["60"]},
    "INSOLENTIA": {"-": ["11"]},
    "INSOLESCO": {"-": ["53"]},
    "INSOLITVS": {"-": ["21"]},
    "INSOLVBILIS": {"-": ["24"]},
    "INSOMNIS": {"-": ["24"]},
    "INSOMNIVM": {"-": ["12"]},
    "INSONO": {"1": ["51"]},
    "INSONS": {"-": ["25"]},
    "INSOPOR": {"-": ["25"]},
    "INSPECTIO": {"-": ["13"]},
    "INSPECTO": {"-": ["51"]},
    "INSPECTOR": {"-": ["13"]},
    "INSPECTVS": {"-": ["14"]},
    "INSPERANS": {"-": ["25"]},
    "INSPERATVS": {"-": ["21"]},
    "INSPERGO": {"-": ["53"]},
    "INSPICIO": {"-": ["55"]},
    "INSPICO": {"-": ["51"]},
    "INSPIRO": {"-": ["51"]},
    "INSPVO": {"-": ["53"]},
    "INSPVRCO": {"-": ["51"]},
    "INSTABILIS": {"-": ["24"]},
    "INSTANS": {"-": ["25"]},
    "INSTANTER": {"-": ["6&"]},
    "INSTAR": {"-": ["16"]},
    "INSTAVRO": {"-": ["51"]},
    "INSTERNO": {"-": ["53"]},
    "INSTIGO": {"-": ["51"]},
    "INSTILLO": {"-": ["51"]},
    "INSTINCTVS": {"1": ["14"], "2": ["21"]},
    "INSTINGVO": {"-": ["53"]},
    "INSTIPO": {"-": ["51"]},
    "INSTITA": {"-": ["11"]},
    "INSTITOR": {"-": ["13"]},
    "INSTITVO": {"-": ["53"]},
    "INSTITVTIO": {"-": ["13"]},
    "INSTITVTVM": {"-": ["12"]},
    "INSTO": {"-": ["51"]},
    "INSTRAGVLVM": {"-": ["12"]},
    "INSTRATVM": {"-": ["12"]},
    "INSTREPO": {"-": ["53"]},
    "INSTRINGO": {"-": ["53"]},
    "INSTRVCTIO": {"-": ["13"]},
    "INSTRVCTVS": {"1": ["14"], "2": ["21"]},
    "INSTRVMENTVM": {"-": ["12"]},
    "INSTRVO": {"-": ["53"]},
    "INSVAVIS": {"-": ["24"]},
    "INSVDO": {"-": ["51"]},
    "INSVEFACTVS": {"-": ["21"]},
    "INSVESCO": {"-": ["53"]},
    "INSVETVS": {"-": ["21"]},
    "INSVLA": {"-": ["11"], "N": ["11"]},
    "INSVLANVS": {"-": ["12"]},
    "INSVLSVS": {"-": ["21"]},
    "INSVLTO": {"-": ["51"]},
    "INSVM": {"1": ["56"]},
    "INSVMO": {"-": ["53"]},
    "INSVO": {"-": ["53"]},
    "INSVPER": {"1": ["60"], "2": ["70"]},
    "INSVPERABILIS": {"-": ["24"]},
    "INSVRGO": {"-": ["53"]},
    "INSVSVRRO": {"-": ["51"]},
    "INTABESCO": {"-": ["53"]},
    "INTACTILIS": {"-": ["24"]},
    "INTACTVS": {"1": ["14"], "2": ["21"]},
    "INTAMINATVS": {"-": ["21"]},
    "INTECTVS": {"-": ["21"]},
    "INTEGELLVS": {"-": ["21"]},
    "INTEGER": {"-": ["21"]},
    "INTEGO": {"-": ["53"]},
    "INTEGRE": {"-": ["60"]},
    "INTEGRITAS": {"-": ["13"]},
    "INTEGRO": {"-": ["51"]},
    "INTELLECTIO": {"-": ["12"]},
    "INTELLECTVALIS": {"-": ["24"]},
    "INTELLECTVS": {"-": ["14"]},
    "INTELLIGENS": {"-": ["25"]},
    "INTELLIGENTIA": {"-": ["11"]},
    "INTELLIGIBILIS": {"-": ["24"]},
    "INTELLIGO": {"-": ["53"]},
    "INTEMERATVS": {"-": ["21"]},
    "INTEMPERANS": {"-": ["25"]},
    "INTEMPERANTER": {"-": ["60"]},
    "INTEMPERANTIA": {"-": ["11"]},
    "INTEMPERATVS": {"-": ["21"]},
    "INTEMPERIAE": {"-": ["11"]},
    "INTEMPERIES": {"-": ["15"]},
    "INTEMPESTIVE": {"-": ["60"]},
    "INTEMPESTIVVS": {"-": ["21"]},
    "INTEMPESTVS": {"-": ["21"]},
    "INTENDO": {"-": ["53"]},
    "INTENTATIO": {"-": ["13"]},
    "INTENTATVS": {"-": ["21"]},
    "INTENTE": {"-": ["6&"]},
    "INTENTIO": {"-": ["13"]},
    "INTENTO": {"-": ["51"]},
    "INTENTVS": {"1": ["14"], "2": ["21"]},
    "INTEPEO": {"-": ["52"]},
    "INTEPESCO": {"-": ["53"]},
    "INTER": {"-": ["70"]},
    "INTERARESCO": {"-": ["53"]},
    "INTERCALARIVS": {"-": ["21"]},
    "INTERCALO": {"-": ["51"]},
    "INTERCAPEDO": {"-": ["13"]},
    "INTERCARDINATVS": {"-": ["21"]},
    "INTERCEDO": {"1": ["53"]},
    "INTERCESSIO": {"-": ["13"]},
    "INTERCESSOR": {"-": ["13"]},
    "INTERCESSVS": {"-": ["14"]},
    "INTERCIDO": {"1": ["53"], "2": ["53"]},
    "INTERCIPIO": {"-": ["55"]},
    "INTERCLVDO": {"-": ["53"]},
    "INTERCOLVMNIVM": {"-": ["12"]},
    "INTERCVRRO": {"-": ["53"]},
    "INTERCVRSO": {"-": ["51"]},
    "INTERCVRSVS": {"-": ["14"]},
    "INTERCVS": {"-": ["22"]},
    "INTERDATVS": {"-": ["21"]},
    "INTERDICO": {"2": ["53"]},
    "INTERDICTVM": {"-": ["12"]},
    "INTERDIV": {"-": ["60"]},
    "INTERDO": {"-": ["51"]},
    "INTERDVM": {"-": ["60"]},
    "INTEREA": {"-": ["60"]},
    "INTEREMPTOR": {"-": ["13"]},
    "INTEREO": {"1": ["56"]},
    "INTEREQVITO": {"-": ["51"]},
    "INTERFECTOR": {"-": ["13"]},
    "INTERFECTRIX": {"-": ["13"]},
    "INTERFICIO": {"-": ["55"]},
    "INTERFIO": {"-": ["56"]},
    "INTERFLVO": {"-": ["53"]},
    "INTERFODIO": {"-": ["55"]},
    "INTERFOR": {"-": ["5J"]},
    "INTERFRINGO": {"-": ["53"]},
    "INTERFVGIO": {"-": ["55"]},
    "INTERFVNDO": {"2": ["53"]},
    "INTERFVSVS": {"-": ["21"]},
    "INTERIACEO": {"-": ["52"]},
    "INTERIACIO": {"-": ["55"]},
    "INTERICIO": {"-": ["55"]},
    "INTERIECTVS": {"-": ["14"]},
    "INTERIM": {"-": ["60"]},
    "INTERIMO": {"-": ["53"]},
    "INTERIOR": {"-": ["2&"], "2": ["2&"]},
    "INTERIORA": {"-": ["13"]},
    "INTERITIO": {"-": ["13"]},
    "INTERITVS": {"-": ["14"]},
    "INTERIVNGO": {"-": ["53"]},
    "INTERLABOR": {"2": ["5L"]},
    "INTERLEGO": {"2": ["53"]},
    "INTERLIGO": {"2": ["51"]},
    "INTERLINO": {"-": ["53"]},
    "INTERLOQVOR": {"-": ["5L"]},
    "INTERLVCEO": {"-": ["52"]},
    "INTERLVNIVM": {"-": ["12"]},
    "INTERLVO": {"-": ["53"]},
    "INTERMESTRIS": {"-": ["21"]},
    "INTERMINO": {"-": ["51"]},
    "INTERMISCEO": {"-": ["52"]},
    "INTERMISSIO": {"-": ["13"]},
    "INTERMITTO": {"-": ["53"]},
    "INTERMORIOR": {"-": ["5N"]},
    "INTERNASCOR": {"-": ["5L"]},
    "INTERNECIO": {"-": ["13"]},
    "INTERNECIVVS": {"-": ["21"]},
    "INTERNITEO": {"-": ["52"]},
    "INTERNODIVM": {"-": ["12"]},
    "INTERNOSCO": {"-": ["53"]},
    "INTERNVNTIA": {"-": ["11"]},
    "INTERNVNTIVS": {"-": ["12"]},
    "INTERNVS": {"-": ["21"]},
    "INTERO": {"-": ["53"]},
    "INTERPELLATIO": {"-": ["13"]},
    "INTERPELLATOR": {"-": ["13"]},
    "INTERPELLO": {"-": ["51"]},
    "INTERPENSIVA": {"-": ["12"]},
    "INTERPOLO": {"-": ["51"]},
    "INTERPONO": {"-": ["53"]},
    "INTERPOSITIO": {"-": ["13"]},
    "INTERPRES": {"-": ["13"]},
    "INTERPRETATIO": {"-": ["13"]},
    "INTERPRETOR": {"-": ["5J"]},
    "INTERPRIMO": {"-": ["53"]},
    "INTERPVNGO": {"-": ["53"]},
    "INTERPVTO": {"-": ["51"]},
    "INTERQVIESCO": {"-": ["53"]},
    "INTERRITVS": {"-": ["21"]},
    "INTERROGATIO": {"-": ["13"]},
    "INTERROGATIVNCVLA": {"-": ["11"]},
    "INTERROGO": {"-": ["51"]},
    "INTERRVMPO": {"-": ["53"]},
    "INTERSAEPIO": {"-": ["54"]},
    "INTERSCALMIVM": {"-": ["12"]},
    "INTERSCINDO": {"-": ["53"]},
    "INTERSECO": {"-": ["51"]},
    "INTERSECTIO": {"-": ["13"]},
    "INTERSERO": {"2": ["53"], "3": ["53"]},
    "INTERSPIRO": {"-": ["51"]},
    "INTERSTINCTVS": {"-": ["21"]},
    "INTERSTINGVO": {"-": ["53"]},
    "INTERSVM": {"1": ["56"]},
    "INTERTIGNIVM": {"-": ["12"]},
    "INTERTRIGO": {"-": ["13"]},
    "INTERTRIMENTVM": {"-": ["12"]},
    "INTERVALLVM": {"-": ["12"]},
    "INTERVECTVS": {"-": ["21"]},
    "INTERVELLO": {"-": ["53"]},
    "INTERVENIO": {"-": ["54"]},
    "INTERVENIVM": {"-": ["12"]},
    "INTERVENTVS": {"-": ["14"]},
    "INTERVERSOR": {"1": ["13"], "2": ["5J"], "-": ["5J"]},
    "INTERVERTO": {"-": ["53"]},
    "INTERVISO": {"-": ["53"]},
    "INTERVOMO": {"-": ["53"]},
    "INTERVTRASQVE": {"-": ["60"]},
    "INTESTABILIS": {"-": ["24"]},
    "INTESTATVS": {"-": ["21"]},
    "INTESTINVM": {"-": ["12"]},
    "INTESTINVS": {"-": ["21"]},
    "INTEXO": {"-": ["53"]},
    "INTEXTVS": {"-": ["14"]},
    "INTIME": {"-": ["6-"]},
    "INTIMVS": {"-": ["12"]},
    "INTINGO": {"-": ["53"]},
    "INTOLERABILIS": {"-": ["24"]},
    "INTOLERANDVS": {"-": ["21"]},
    "INTOLERANS": {"-": ["25"]},
    "INTOLERANTER": {"-": ["6&"]},
    "INTOLERANTIA": {"-": ["11"]},
    "INTONO": {"-": ["51"]},
    "INTONSVS": {"-": ["21"]},
    "INTORQVEO": {"-": ["52"]},
    "INTRA": {"1": ["60"], "2": ["70"]},
    "INTRACTABILIS": {"-": ["24"]},
    "INTRACTATVS": {"-": ["21"]},
    "INTREMO": {"-": ["53"]},
    "INTREPIDE": {"-": ["60"]},
    "INTREPIDVS": {"-": ["21"]},
    "INTRINSECVS": {"-": ["60"]},
    "INTRITVS": {"1": ["21"], "2": ["21"]},
    "INTRO": {"1": ["51"], "2": ["60"]},
    "INTROCEDO": {"1": ["53"]},
    "INTRODVCO": {"-": ["53"]},
    "INTRODVCTIO": {"-": ["13"]},
    "INTROEO": {"1": ["56"]},
    "INTROFERO": {"-": ["5F", "53"]},
    "INTROGREDIOR": {"-": ["5N"]},
    "INTROITVS": {"-": ["14"]},
    "INTROMITTO": {"-": ["53"]},
    "INTRORSVM": {"-": ["60"]},
    "INTRORVMPO": {"-": ["53"]},
    "INTROSPECTO": {"-": ["51"]},
    "INTROSPICIO": {"-": ["55"]},
    "INTROTRVDO": {"-": ["53"]},
    "INTROVERSVS": {"-": ["60"]},
    "INTVBVM": {"-": ["12"]},
    "INTVBVS": {"-": ["12"]},
    "INTVEOR": {"-": ["5K"]},
    "INTVMESCO": {"-": ["53"]},
    "INTVMVLATVS": {"-": ["21"]},
    "INTVOR": {"-": ["5L"]},
    "INTVRBIDVS": {"-": ["21"]},
    "INTVRBO": {"2": ["51"]},
    "INTVS": {"1": ["60"], "2": ["70"]},
    "INTVTVS": {"-": ["21"]},
    "INVADO": {"2": ["53"]},
    "INVALESCO": {"-": ["53"]},
    "INVALIDVS": {"-": ["21"]},
    "INVASIO": {"-": ["13"]},
    "INVECTICIVS": {"-": ["21"]},
    "INVECTOR": {"-": ["13"]},
    "INVECTVS": {"-": ["14"]},
    "INVEHO": {"-": ["53"]},
    "INVENIO": {"-": ["54"]},
    "INVENTIO": {"-": ["13"]},
    "INVENTOR": {"-": ["13"]},
    "INVENTRIX": {"-": ["13"]},
    "INVENTVM": {"-": ["12"]},
    "INVENTVS": {"-": ["14"]},
    "INVENVSTVS": {"-": ["21"]},
    "INVERECVNDE": {"-": ["60"]},
    "INVERECVNDVS": {"-": ["21"]},
    "INVERGO": {"-": ["53"]},
    "INVERSVRA": {"-": ["11"]},
    "INVERSVS": {"1": ["21"]},
    "INVERTO": {"-": ["53"]},
    "INVESPERASCO": {"-": ["53"]},
    "INVESTIGO": {"-": ["51"]},
    "INVESTIO": {"-": ["54"]},
    "INVETERASCO": {"-": ["53"]},
    "INVETERATVM": {"-": ["12"]},
    "INVETERATVS": {"-": ["21"]},
    "INVIA": {"-": ["12"]},
    "INVICEM": {"-": ["60"]},
    "INVICTVS": {"-": ["21"], "A": ["21"]},
    "INVIDEO": {"-": ["52"]},
    "INVIDIA": {"-": ["11"]},
    "INVIDIOSE": {"-": ["60"]},
    "INVIDIOSVS": {"-": ["21"]},
    "INVIDVS": {"1": ["12"], "2": ["21"]},
    "INVIGILO": {"-": ["51"]},
    "INVINCTVS": {"-": ["21"]},
    "INVIOLABILIS": {"-": ["24"]},
    "INVIOLATVS": {"-": ["21"]},
    "INVISITATVS": {"-": ["21"]},
    "INVISO": {"-": ["53"]},
    "INVISVS": {"1": ["21"], "2": ["21"]},
    "INVITAMENTVM": {"-": ["12"]},
    "INVITATIO": {"-": ["13"]},
    "INVITE": {"-": ["60"]},
    "INVITO": {"-": ["51"]},
    "INVITVS": {"-": ["21"]},
    "INVIVS": {"-": ["21"]},
    "INVLA": {"-": ["11"]},
    "INVLTVS": {"-": ["21"]},
    "INVMBRO": {"-": ["51"]},
    "INVNDATIO": {"-": ["13"]},
    "INVNDO": {"-": ["51"]},
    "INVNGO": {"-": ["53"]},
    "INVOCATVS": {"1": ["14"], "2": ["21"]},
    "INVOCO": {"-": ["51"]},
    "INVOLITO": {"-": ["51"]},
    "INVOLO": {"2": ["51"]},
    "INVOLVCRVM": {"-": ["12"]},
    "INVOLVO": {"-": ["53"]},
    "INVOLVTIO": {"-": ["13"]},
    "INVOLVTVS": {"-": ["21"]},
    "INVRBANE": {"-": ["60"]},
    "INVRGEO": {"-": ["52"]},
    "INVRO": {"-": ["53"]},
    "INVSITATE": {"-": ["60"]},
    "INVSITATVS": {"-": ["21"]},
    "INVTILIS": {"-": ["24"]},
    "INVTILITAS": {"-": ["13"]},
    "INVTILITER": {"-": ["60"]},
    "INVVLNERABILIS": {"-": ["24"]},
    "INVVS": {"N": ["12"]},
    "IO": {"-": ["90"], "N": ["13"]},
    "IOCATIO": {"-": ["13"]},
    "IOCOR": {"-": ["5J"]},
    "IOCOSE": {"-": ["60"]},
    "IOCOSVS": {"-": ["21"]},
    "IOCVLARIS": {"-": ["24"]},
    "IOCVS": {"-": ["12"], "N": ["12"]},
    "IONIA": {"N": ["11"]},
    "IONICE": {"-": ["60"]},
    "IPSE": {"-": ["45"]},
    "IRA": {"-": ["11"], "N": ["11"]},
    "IRACVNDIA": {"-": ["11"]},
    "IRACVNDVS": {"-": ["21"]},
    "IRASCOR": {"-": ["5L"]},
    "IRATVS": {"-": ["21"]},
    "IRIS": {"-": ["17"], "N": ["13"]},
    "IRONIA": {"-": ["11"]},
    "IRPEX": {"-": ["13"]},
    "IRRADO": {"-": ["53"]},
    "IRRATIONALIS": {"-": ["24"]},
    "IRRELIGATVS": {"-": ["21"]},
    "IRRELIGIOSE": {"-": ["60"]},
    "IRREMEABILIS": {"-": ["24"]},
    "IRREMEDIABILIS": {"-": ["24"]},
    "IRREPARABILIS": {"-": ["24"]},
    "IRREPERTVS": {"-": ["21"]},
    "IRREPO": {"-": ["53"]},
    "IRREQVIETVS": {"-": ["21"]},
    "IRRESECTVS": {"-": ["21"]},
    "IRRETIO": {"-": ["54"]},
    "IRRETORTVS": {"-": ["21"]},
    "IRREVERENTIA": {"-": ["11"]},
    "IRREVOCABILIS": {"-": ["24"]},
    "IRREVOCATVS": {"-": ["21"]},
    "IRRIDEO": {"-": ["52"]},
    "IRRIDICVLE": {"-": ["60"]},
    "IRRIGIVVS": {"-": ["21"]},
    "IRRIGO": {"-": ["51"]},
    "IRRIGVVS": {"-": ["21"]},
    "IRRISIO": {"-": ["13"]},
    "IRRISOR": {"-": ["13"]},
    "IRRISVS": {"-": ["14"]},
    "IRRITABILIS": {"-": ["24"]},
    "IRRITAMENTVM": {"-": ["12"]},
    "IRRITATIO": {"-": ["13"]},
    "IRRITATOR": {"-": ["13"]},
    "IRRITATVS": {"-": ["21"]},
    "IRRITO": {"-": ["51"]},
    "IRRITVM": {"-": ["12"]},
    "IRRITVS": {"-": ["21"]},
    "IRROGATIO": {"-": ["13"]},
    "IRROGO": {"-": ["51"]},
    "IRRORO": {"-": ["51"]},
    "IRRVMATIO": {"-": ["13"]},
    "IRRVMATOR": {"-": ["13"]},
    "IRRVMO": {"-": ["51"]},
    "IRRVMPO": {"-": ["53"]},
    "IRRVO": {"-": ["53"]},
    "IRRVPTIO": {"-": ["13"]},
    "IRRVPTVS": {"-": ["21"]},
    "IS": {"-": ["45"]},
    "ISATIS": {"-": ["17"]},
    "ISCHIACVS": {"-": ["21"]},
    "ISCHNOS": {"-": ["27"]},
    "ISIDORVS": {"N": ["12"]},
    "ISMENIAS": {"N": ["17"]},
    "ISODOMVS": {"-": ["21"]},
    "ISTAC": {"-": ["60"]},
    "ISTE": {"-": ["45"]},
    "ISTHMVS": {"-": ["17"], "N": ["12"]},
    "ISTI": {"-": ["60"]},
    "ISTIC": {"1": ["45"], "2": ["60"]},
    "ISTINC": {"-": ["60"]},
    "ISTIVSMODI": {"-": ["60"]},
    "ISTO": {"-": ["60"]},
    "ISTOC": {"-": ["60"]},
    "ISTVC": {"-": ["60"]},
    "ITA": {"-": ["60"]},
    "ITALIA": {"N": ["11"]},
    "ITALICVS": {"N": ["21"], "A": ["21"]},
    "ITALVS": {"A": ["21"], "N": ["12"]},
    "ITAQVE": {"-": ["60"]},
    "ITEM": {"-": ["60"]},
    "ITER": {"-": ["13"]},
    "ITERATIO": {"-": ["13"]},
    "ITERO": {"-": ["51"]},
    "ITERVM": {"-": ["60"]},
    "ITIDEM": {"-": ["60"]},
    "ITVS": {"-": ["14"]},
    "IVBA": {"-": ["11"], "N": ["11"]},
    "IVBAR": {"-": ["13"]},
    "IVBATVS": {"-": ["21"]},
    "IVBEO": {"-": ["52"]},
    "IVCVNDE": {"-": ["60"]},
    "IVCVNDITAS": {"-": ["13"]},
    "IVCVNDVS": {"-": ["21"]},
    "IVDEX": {"-": ["13"]},
    "IVDICATVM": {"-": ["12"]},
    "IVDICATVS": {"1": ["14"], "2": ["21"]},
    "IVDICIALIS": {"-": ["24"]},
    "IVDICIARIVS": {"-": ["21"]},
    "IVDICIVM": {"-": ["12"]},
    "IVDICO": {"-": ["51"]},
    "IVGALES": {"-": ["13"]},
    "IVGALIS": {"-": ["24"]},
    "IVGERVM": {"-": ["16"]},
    "IVGIS": {"-": ["24"]},
    "IVGLANS": {"-": ["13"]},
    "IVGO": {"-": ["51"]},
    "IVGOSVS": {"-": ["21"]},
    "IVGVLATIO": {"-": ["13"]},
    "IVGVLO": {"-": ["51"]},
    "IVGVLVM": {"-": ["12"]},
    "IVGVLVS": {"-": ["12"]},
    "IVGVM": {"-": ["12"]},
    "IVGVMENTO": {"-": ["51"]},
    "IVGVMENTVM": {"-": ["12"]},
    "IVGVS": {"-": ["21"]},
    "IVLIA": {"N": ["11"]},
    "IVLIVS": {"A": ["21"], "N": ["12"]},
    "IVLVS": {"N": ["12"]},
    "IVMENTVM": {"-": ["12"]},
    "IVNCTVRA": {"-": ["11"]},
    "IVNCTVS": {"-": ["21"]},
    "IVNCVS": {"-": ["12"], "N": ["12"]},
    "IVNGO": {"-": ["53"]},
    "IVNIANVS": {"N": ["21"], "A": ["21"]},
    "IVNIO": {"N": ["13"]},
    "IVNIOR": {"-": ["13"], "N": ["13"]},
    "IVNIPERVS": {"-": ["12"]},
    "IVNIVS": {"A": ["21"], "N": ["12"]},
    "IVNIX": {"-": ["13"]},
    "IVNO": {"N": ["13"]},
    "IVPPITER": {"N": ["13"]},
    "IVRAMENTVM": {"-": ["12"]},
    "IVRATVS": {"-": ["21"]},
    "IVRE": {"-": ["60"]},
    "IVRGIVM": {"-": ["12"]},
    "IVRGO": {"-": ["51"]},
    "IVRIDICVS": {"1": ["12"], "2": ["21"]},
    "IVRISCONSVLTVS": {"-": ["12"]},
    "IVRISDICTIO": {"-": ["13"]},
    "IVRO": {"-": ["51"]},
    "IVROR": {"-": ["5J"]},
    "IVS": {"1": ["13"], "2": ["13"]},
    "IVSCVLVM": {"-": ["12"]},
    "IVSIVRANDVM": {"-": ["13"]},
    "IVSSVM": {"-": ["12"]},
    "IVSSVS": {"-": ["14"]},
    "IVSTA": {"C": ["12"], "-": ["12"]},
    "IVSTE": {"-": ["60"]},
    "IVSTIFICVS": {"-": ["21"]},
    "IVSTITIA": {"-": ["11"], "N": ["11"]},
    "IVSTITIVM": {"-": ["12"]},
    "IVSTVM": {"-": ["12"]},
    "IVSTVS": {"-": ["21"], "N": ["12"]},
    "IVTO": {"-": ["51"]},
    "IVVAMEN": {"-": ["13"]},
    "IVVENALIS": {"-": ["24"], "N": ["13"], "A": ["21"]},
    "IVVENALITER": {"-": ["60"]},
    "IVVENCA": {"-": ["11"]},
    "IVVENCVS": {"1": ["12"], "2": ["21"]},
    "IVVENESCO": {"-": ["53"]},
    "IVVENILIS": {"-": ["24"]},
    "IVVENILITER": {"-": ["60"]},
    "IVVENIS": {"1": ["13"], "2": ["26"]},
    "IVVENTA": {"-": ["11"]},
    "IVVENTAS": {"-": ["13"], "N": ["13"]},
    "IVVENTVS": {"-": ["13"]},
    "IVVO": {"-": ["51"]},
    "IVXTA": {"1": ["60"], "2": ["70"]},
    "IVXTIM": {"-": ["60"]},
    "KALENDAE": {"-": ["11"]},
    "KALENDARIVM": {"-": ["12"]},
    "LABASCO": {"-": ["53"]},
    "LABASCOR": {"-": ["5J"]},
    "LABEFACIO": {"-": ["55"]},
    "LABEFACTO": {"-": ["51"]},
    "LABEFIO": {"-": ["56"]},
    "LABELLVM": {"1": ["12"], "2": ["12"]},
    "LABEOSVS": {"-": ["21"]},
    "LABES": {"-": ["13"]},
    "LABIA": {"-": ["11"]},
    "LABICI": {"N": ["12"]},
    "LABIDVS": {"-": ["21"]},
    "LABO": {"-": ["51"]},
    "LABOR": {"1": ["13"], "2": ["5L"], "N": ["13"]},
    "LABORIOSE": {"-": ["60"]},
    "LABORIOSVS": {"-": ["21"]},
    "LABORO": {"-": ["51"]},
    "LABRVM": {"1": ["12"], "2": ["12"]},
    "LABRVSCA": {"-": ["11"]},
    "LABYRINTHEVS": {"-": ["21"]},
    "LABYRINTHVS": {"-": ["12"]},
    "LAC": {"-": ["13"]},
    "LACEDAEMON": {"N": ["17"]},
    "LACEDAEMONII": {"N": ["12"]},
    "LACEDAEMONIVS": {"N": ["21"], "A": ["21"]},
    "LACER": {"-": ["21"]},
    "LACERATIO": {"-": ["13"]},
    "LACERNA": {"-": ["11"]},
    "LACERNATVS": {"-": ["21"]},
    "LACERO": {"-": ["51"]},
    "LACERTA": {"-": ["11"], "N": ["11"]},
    "LACERTOSVS": {"-": ["21"]},
    "LACERTVS": {"1": ["12"], "2": ["12"]},
    "LACESSO": {"-": ["53"]},
    "LACRIMA": {"-": ["11"]},
    "LACRIMABILIS": {"-": ["24"]},
    "LACRIMO": {"-": ["51"]},
    "LACRIMOSVS": {"-": ["21"]},
    "LACRIMVLA": {"-": ["11"]},
    "LACTEO": {"-": ["52"]},
    "LACTEOLVS": {"-": ["21"]},
    "LACTES": {"-": ["13"]},
    "LACTEVS": {"-": ["21"]},
    "LACTO": {"-": ["51"]},
    "LACTVCA": {"-": ["11"]},
    "LACVNA": {"-": ["11"]},
    "LACVNAR": {"-": ["13"]},
    "LACVNOSVS": {"-": ["21"]},
    "LACVS": {"-": ["14"]},
    "LAEDO": {"-": ["53"]},
    "LAELIVS": {"N": ["12"]},
    "LAENA": {"-": ["11"]},
    "LAESIO": {"-": ["13"]},
    "LAETABILIS": {"-": ["24"]},
    "LAETANS": {"-": ["25"]},
    "LAETE": {"-": ["6&"]},
    "LAETIFICO": {"-": ["51"]},
    "LAETIFICVS": {"-": ["21"]},
    "LAETILIVS": {"N": ["12"]},
    "LAETITIA": {"-": ["11"], "N": ["11"]},
    "LAETOR": {"-": ["5J"]},
    "LAETORIVS": {"N": ["12"]},
    "LAETVS": {"1": ["12"], "2": ["21"]},
    "LAEVA": {"1": ["11"], "2": ["12"], "3": ["60"]},
    "LAEVE": {"-": ["60"]},
    "LAEVINVS": {"N": ["12"]},
    "LAEVVM": {"1": ["12"], "2": ["60"], "-": ["60"]},
    "LAGANVM": {"-": ["12"]},
    "LAGENA": {"-": ["11"]},
    "LAGEVS": {"-": ["17"]},
    "LAGOIS": {"-": ["13"]},
    "LALLO": {"-": ["51"]},
    "LAMA": {"-": ["11"]},
    "LAMBO": {"-": ["53"]},
    "LAMELLA": {"-": ["11"]},
    "LAMENTA": {"-": ["12"]},
    "LAMENTABILIS": {"-": ["24"]},
    "LAMENTATIO": {"-": ["13"]},
    "LAMENTOR": {"-": ["5J"]},
    "LAMENTVM": {"-": ["12"]},
    "LAMIA": {"N": ["11"]},
    "LAMINA": {"-": ["11"]},
    "LAMPAS": {"-": ["17"]},
    "LAMPSACENI": {"N": ["12"]},
    "LAMPSACENVS": {"N": ["21"], "A": ["21"]},
    "LAMPSACVS": {"N": ["17"]},
    "LANA": {"-": ["11"]},
    "LANATVS": {"-": ["21"]},
    "LANCEA": {"-": ["11"]},
    "LANCINATIO": {"-": ["13"]},
    "LANCINO": {"-": ["51"]},
    "LANCVLA": {"-": ["11"]},
    "LANEVS": {"-": ["21"]},
    "LANGOR": {"-": ["13"]},
    "LANGVENS": {"-": ["25"]},
    "LANGVEO": {"-": ["52"]},
    "LANGVESCO": {"-": ["53"]},
    "LANGVIDE": {"-": ["60"]},
    "LANGVIDVLVS": {"-": ["21"]},
    "LANGVIDVS": {"-": ["21"]},
    "LANGVOR": {"-": ["13"]},
    "LANIATVS": {"-": ["14"]},
    "LANIFICVS": {"-": ["21"]},
    "LANIGER": {"1": ["12"], "2": ["21"]},
    "LANIGERA": {"-": ["11"]},
    "LANIO": {"-": ["51"], "2": ["13"], "1": ["21"]},
    "LANISTA": {"-": ["11"]},
    "LANITIVM": {"-": ["12"]},
    "LANVGO": {"-": ["13"]},
    "LANX": {"-": ["13"]},
    "LAODICIA": {"N": ["11"]},
    "LAPATHVM": {"-": ["12"]},
    "LAPATHVS": {"-": ["12"]},
    "LAPHIRO": {"N": ["13"]},
    "LAPICIDINAE": {"-": ["11"]},
    "LAPIDATIO": {"-": ["13"]},
    "LAPIDEVS": {"-": ["21"]},
    "LAPIDO": {"-": ["51"]},
    "LAPIDOSA": {"-": ["21"]},
    "LAPIDOSVS": {"-": ["21"]},
    "LAPILLVS": {"-": ["12"]},
    "LAPIS": {"-": ["13"]},
    "LAPPA": {"-": ["11"], "N": ["11"]},
    "LAPSO": {"-": ["51"]},
    "LAPSVS": {"-": ["14"]},
    "LAQVEAR": {"-": ["13"]},
    "LAQVEO": {"1": ["51"], "2": ["51"]},
    "LAQVEVS": {"-": ["12"]},
    "LAR": {"N": ["13"]},
    "LARCIVS": {"N": ["12"]},
    "LARDVM": {"-": ["12"]},
    "LARENTIA": {"N": ["11"]},
    "LARGE": {"-": ["60"]},
    "LARGIFICVS": {"-": ["21"]},
    "LARGIFLVVS": {"-": ["21"]},
    "LARGIOR": {"-": ["5M"]},
    "LARGITAS": {"-": ["13"]},
    "LARGITER": {"-": ["60"]},
    "LARGITIO": {"-": ["13"]},
    "LARGITOR": {"1": ["13"], "2": ["5J"], "-": ["5J"]},
    "LARGITVDO": {"-": ["13"]},
    "LARGIVS": {"N": ["12"]},
    "LARGO": {"-": ["60"]},
    "LARGVS": {"-": ["21"], "N": ["12"]},
    "LARIDVM": {"-": ["12"]},
    "LARIX": {"-": ["13"]},
    "LARVA": {"-": ["11"], "N": ["11"]},
    "LARVALIS": {"-": ["24"]},
    "LASANVM": {"-": ["12"]},
    "LASCIVE": {"-": ["60"]},
    "LASCIVIA": {"-": ["11"]},
    "LASCIVIO": {"-": ["54"]},
    "LASCIVVS": {"-": ["21"]},
    "LASER": {"-": ["13"]},
    "LASERPICIATVS": {"-": ["21"]},
    "LASERPICIVM": {"-": ["12"]},
    "LASERPITIFER": {"-": ["21"]},
    "LASSITVDO": {"-": ["13"]},
    "LASSO": {"-": ["51"]},
    "LASSVLVS": {"-": ["21"]},
    "LASSVS": {"-": ["21"]},
    "LATE": {"-": ["60"]},
    "LATEBRA": {"-": ["11"]},
    "LATEBROSVS": {"-": ["21"]},
    "LATENS": {"-": ["25"]},
    "LATEO": {"-": ["52"]},
    "LATER": {"-": ["13"]},
    "LATERAMEN": {"-": ["13"]},
    "LATERARIVS": {"-": ["21"]},
    "LATERCVLVS": {"-": ["12"]},
    "LATERICIVS": {"-": ["21"]},
    "LATERNA": {"-": ["11"]},
    "LATEX": {"-": ["13"]},
    "LATIBVLVM": {"-": ["12"]},
    "LATIFVNDIVM": {"-": ["12"]},
    "LATINE": {"-": ["60"]},
    "LATINI": {"N": ["12"]},
    "LATINIVS": {"N": ["12"]},
    "LATINVS": {"A": ["21"], "N": ["12"]},
    "LATIO": {"-": ["13"]},
    "LATITO": {"1": ["51"], "2": ["51"]},
    "LATITVDO": {"-": ["13"]},
    "LATIVM": {"N": ["12"]},
    "LATONA": {"N": ["11"]},
    "LATOR": {"-": ["13"]},
    "LATRATVS": {"-": ["14"]},
    "LATRO": {"1": ["13"], "2": ["51"]},
    "LATROCINIVM": {"-": ["12"]},
    "LATROCINOR": {"-": ["5J"]},
    "LATRVNCVLARIVS": {"-": ["21"]},
    "LATRVNCVLVS": {"-": ["12"]},
    "LATVS": {"1": ["13"], "2": ["21"]},
    "LATVSCVLVM": {"-": ["12"]},
    "LAVABRVM": {"-": ["12"]},
    "LAVACRVM": {"-": ["12"]},
    "LAVATIO": {"-": ["13"]},
    "LAVDABILIS": {"-": ["24"]},
    "LAVDABILITER": {"-": ["60"]},
    "LAVDATIO": {"-": ["13"]},
    "LAVDATOR": {"-": ["13"]},
    "LAVDATRIX": {"-": ["13"]},
    "LAVDATVS": {"-": ["21"]},
    "LAVDO": {"-": ["51"]},
    "LAVINIA": {"N": ["11"]},
    "LAVINIVM": {"N": ["12"]},
    "LAVO": {"1": ["51"], "2": ["53"]},
    "LAVREA": {"-": ["11"]},
    "LAVREATVS": {"-": ["21"]},
    "LAVRENS": {"N": ["25"], "A": ["25"]},
    "LAVRENTES": {"N": ["13"]},
    "LAVRENTINVS": {"N": ["21"], "A": ["21"]},
    "LAVREVS": {"-": ["21"]},
    "LAVRICOMVS": {"-": ["21"]},
    "LAVRIGER": {"-": ["21"]},
    "LAVRVS": {"-": ["16"]},
    "LAVS": {"-": ["13"]},
    "LAVTE": {"-": ["60"]},
    "LAVTITIA": {"-": ["11"]},
    "LAVTVMIAE": {"-": ["11"]},
    "LAVTVS": {"C": ["21"], "-": ["21"]},
    "LAXAMENTVM": {"-": ["12"]},
    "LAXATIO": {"-": ["13"]},
    "LAXATVS": {"-": ["21"]},
    "LAXE": {"-": ["60"]},
    "LAXITAS": {"-": ["13"]},
    "LAXO": {"-": ["51"]},
    "LAXVS": {"-": ["21"]},
    "LEA": {"-": ["11"]},
    "LEAENA": {"-": ["11"], "N": ["11"]},
    "LEBES": {"-": ["17"]},
    "LECTICA": {"-": ["11"]},
    "LECTICARIVS": {"-": ["12"]},
    "LECTICVLVS": {"-": ["12"]},
    "LECTIO": {"-": ["13"]},
    "LECTISTERNIVM": {"-": ["12"]},
    "LECTITO": {"-": ["51"]},
    "LECTO": {"-": ["51"]},
    "LECTOR": {"-": ["13"]},
    "LECTVLVS": {"-": ["12"]},
    "LECTVS": {"1": ["12"], "2": ["14"], "3": ["21"]},
    "LEGALIS": {"-": ["24"]},
    "LEGATIO": {"-": ["13"]},
    "LEGATVM": {"-": ["12"]},
    "LEGATVS": {"-": ["12"]},
    "LEGENS": {"-": ["13"]},
    "LEGIFER": {"-": ["21"]},
    "LEGIO": {"-": ["13"]},
    "LEGIONARIVS": {"-": ["21"]},
    "LEGITIME": {"-": ["60"]},
    "LEGITIMVS": {"-": ["21"]},
    "LEGO": {"1": ["51"], "2": ["53"]},
    "LEGVLVS": {"-": ["12"]},
    "LEGVMEN": {"-": ["13"]},
    "LEMBVS": {"-": ["12"]},
    "LEMVRES": {"-": ["13"]},
    "LENA": {"-": ["11"]},
    "LENE": {"1": ["13"], "2": ["60"]},
    "LENIMEN": {"-": ["13"]},
    "LENIO": {"-": ["54"]},
    "LENIS": {"1": ["13"], "2": ["24"]},
    "LENITAS": {"-": ["13"]},
    "LENITER": {"-": ["60"]},
    "LENITVDO": {"-": ["13"]},
    "LENO": {"1": ["13"], "2": ["51"]},
    "LENOCINIVM": {"-": ["12"]},
    "LENOCINOR": {"-": ["5J"]},
    "LENS": {"1": ["13"], "2": ["13"]},
    "LENTE": {"-": ["60"]},
    "LENTESCO": {"-": ["53"]},
    "LENTISCVS": {"-": ["12"]},
    "LENTITVDO": {"-": ["13"]},
    "LENTO": {"-": ["51"], "N": ["11"]},
    "LENTVLVS": {"N": ["12"]},
    "LENTVS": {"-": ["21"]},
    "LENVNCVLVS": {"-": ["12"]},
    "LEO": {"-": ["13"], "N": ["17"]},
    "LEONIDAS": {"N": ["17"]},
    "LEONINVS": {"-": ["21"]},
    "LEONTINI": {"1N": ["12"], "2N": ["12"], "N": ["12"]},
    "LEONTINVS": {"N": ["21"], "A": ["21"]},
    "LEPIDE": {"-": ["60"]},
    "LEPIDVS": {"-": ["21"], "N": ["12"]},
    "LEPOR": {"-": ["13"]},
    "LEPTIS": {"N": ["13"]},
    "LEPVS": {"-": ["13"]},
    "LEPVSCVLVS": {"-": ["12"]},
    "LETALIS": {"-": ["24"]},
    "LETALITER": {"-": ["60"]},
    "LETHARGICVS": {"-": ["12"]},
    "LETHARGVS": {"-": ["12"]},
    "LETIFER": {"-": ["21"]},
    "LETIFICVS": {"-": ["21"]},
    "LETO": {"-": ["51"]},
    "LETVM": {"-": ["12"], "N": ["12"]},
    "LEVAMEN": {"-": ["13"]},
    "LEVAMENTVM": {"-": ["12"]},
    "LEVATIO": {"-": ["13"]},
    "LEVATVS": {"-": ["21"]},
    "LEVCOPHAEVS": {"-": ["21"]},
    "LEVIGATIO": {"-": ["13"]},
    "LEVIGO": {"1": ["51"], "2": ["51"]},
    "LEVIS": {"1": ["24"], "2": ["24"]},
    "LEVISOMNVS": {"2": ["21"], "-": ["21"]},
    "LEVITAS": {"1": ["13"], "2": ["13"]},
    "LEVITER": {"-": ["60"]},
    "LEVO": {"1": ["51"], "2": ["51"]},
    "LEVOR": {"-": ["13"]},
    "LEX": {"-": ["13"]},
    "LIACVLVM": {"-": ["12"]},
    "LIBAMEN": {"-": ["13"]},
    "LIBAMENTVM": {"-": ["12"]},
    "LIBARIVS": {"-": ["12"]},
    "LIBELLA": {"-": ["11"]},
    "LIBELLVS": {"-": ["12"]},
    "LIBENS": {"-": ["25"]},
    "LIBENTER": {"-": ["60"]},
    "LIBER": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "LIBERA": {"N": ["11"]},
    "LIBERALIS": {"-": ["24"], "N": ["13"]},
    "LIBERALITAS": {"-": ["13"]},
    "LIBERALITER": {"-": ["60"]},
    "LIBERATIO": {"-": ["13"]},
    "LIBERATOR": {"-": ["13"]},
    "LIBERE": {"-": ["60"]},
    "LIBERI": {"-": ["12"]},
    "LIBERO": {"-": ["51"]},
    "LIBERTA": {"-": ["11"]},
    "LIBERTAS": {"-": ["13"], "N": ["13"]},
    "LIBERTINA": {"-": ["11"]},
    "LIBERTINVS": {"1": ["12"], "2": ["21"]},
    "LIBERTVS": {"-": ["12"]},
    "LIBET": {"-": ["52"]},
    "LIBIDINOSE": {"-": ["60"]},
    "LIBIDINOSVS": {"-": ["21"]},
    "LIBIDO": {"-": ["13"], "N": ["13"]},
    "LIBITA": {"-": ["12"]},
    "LIBITINARIVS": {"-": ["12"]},
    "LIBO": {"-": ["51"], "N": ["13"]},
    "LIBRA": {"-": ["11"], "N": ["11"]},
    "LIBRAMENTVM": {"-": ["12"]},
    "LIBRARIA": {"-": ["11"]},
    "LIBRARIVM": {"-": ["12"]},
    "LIBRARIVS": {"1": ["12"], "2": ["12"], "3": ["21"], "4": ["21"]},
    "LIBRATIO": {"-": ["13"]},
    "LIBRATOR": {"-": ["13"]},
    "LIBRILE": {"-": ["13"]},
    "LIBRO": {"-": ["51"]},
    "LIBVM": {"-": ["12"]},
    "LIBVRNA": {"-": ["11"]},
    "LIBVRNICA": {"-": ["11"], "N": ["11"]},
    "LICENS": {"-": ["25"]},
    "LICENTER": {"-": ["60"]},
    "LICENTIA": {"-": ["11"], "N": ["11"]},
    "LICEO": {"-": ["52"]},
    "LICEOR": {"-": ["5K"]},
    "LICET": {"1": ["52"], "2": ["60"], "3": ["82"]},
    "LICHANOS": {"-": ["17"]},
    "LICINIVS": {"N": ["12"], "A": ["21"]},
    "LICIO": {"-": ["55"]},
    "LICITATIO": {"-": ["13"]},
    "LICITVS": {"-": ["21"]},
    "LICIVM": {"-": ["12"]},
    "LICTOR": {"-": ["13"]},
    "LIDO": {"-": ["53"]},
    "LIEN": {"-": ["13"]},
    "LIENOSVS": {"-": ["21"]},
    "LIGAMEN": {"-": ["13"]},
    "LIGAMENTVM": {"-": ["12"]},
    "LIGELLVM": {"-": ["12"]},
    "LIGNATIO": {"-": ["13"]},
    "LIGNATOR": {"-": ["13"]},
    "LIGNEVS": {"-": ["21"]},
    "LIGNOR": {"-": ["5J"]},
    "LIGNVM": {"-": ["12"]},
    "LIGO": {"1": ["13"], "2": ["51"], "3": ["53"]},
    "LIGVLA": {"-": ["11"]},
    "LIGVR": {"C": ["13"], "N": ["25"], "A": ["25"]},
    "LIGVRES": {"N": ["13"]},
    "LIGVRRIO": {"-": ["54"]},
    "LIGVRRITOR": {"-": ["13"]},
    "LIGVSTRVM": {"-": ["12"]},
    "LILIVM": {"-": ["12"]},
    "LILYBAETANVS": {"N": ["21"], "A": ["21"]},
    "LILYBAEVM": {"N": ["17"]},
    "LIMA": {"-": ["11"]},
    "LIMATVRA": {"-": ["11"]},
    "LIMATVS": {"-": ["21"]},
    "LIMBVS": {"-": ["12"]},
    "LIMEN": {"-": ["13"]},
    "LIMES": {"-": ["13"]},
    "LIMNE": {"-": ["17"]},
    "LIMO": {"-": ["51"]},
    "LIMOSVS": {"-": ["21"]},
    "LIMPIDVS": {"-": ["21"]},
    "LIMVS": {"1": ["12"], "2": ["12"], "3": ["21"]},
    "LINEA": {"-": ["11"]},
    "LINEAMENTVM": {"-": ["12"]},
    "LINEATIO": {"-": ["13"]},
    "LINEO": {"-": ["51"]},
    "LINEVS": {"-": ["21"]},
    "LINGO": {"-": ["53"]},
    "LINGVA": {"-": ["11"]},
    "LINGVARIVM": {"-": ["12"]},
    "LINGVLA": {"-": ["11"]},
    "LINGVLATVS": {"-": ["21"]},
    "LINIGER": {"-": ["21"]},
    "LINIO": {"-": ["54"]},
    "LINO": {"-": ["53"]},
    "LINQVO": {"-": ["53"]},
    "LINTEATVS": {"-": ["21"]},
    "LINTER": {"-": ["13"]},
    "LINTEVM": {"-": ["12"]},
    "LINTEVS": {"-": ["21"]},
    "LINVM": {"-": ["12"]},
    "LIPARENSES": {"N": ["13"]},
    "LIPARENSIS": {"N": ["24"], "A": ["24"]},
    "LIPPIO": {"-": ["54"]},
    "LIPPITVDO": {"-": ["13"]},
    "LIPPVS": {"-": ["21"]},
    "LIQVEFACIO": {"-": ["55"]},
    "LIQVEFIO": {"-": ["56"]},
    "LIQVEO": {"-": ["52"]},
    "LIQVESCO": {"-": ["53"]},
    "LIQVIDO": {"-": ["60"]},
    "LIQVIDVM": {"1": ["12"], "2": ["60"]},
    "LIQVIDVS": {"-": ["21"]},
    "LIQVOR": {"1": ["13"], "2": ["5L"]},
    "LIS": {"-": ["13"]},
    "LITIGANS": {"-": ["13"]},
    "LITIGATOR": {"-": ["13"]},
    "LITIGIOSVS": {"-": ["21"]},
    "LITIGO": {"-": ["51"]},
    "LITO": {"-": ["51"]},
    "LITORALIS": {"-": ["24"]},
    "LITOREVS": {"-": ["21"]},
    "LITTERA": {"-": ["11"]},
    "LITTERARIVS": {"-": ["21"]},
    "LITTERATOR": {"-": ["13"]},
    "LITTERATVRA": {"-": ["11"]},
    "LITTERATVS": {"-": ["21"]},
    "LITTERVLA": {"-": ["11"]},
    "LITVRA": {"-": ["11"]},
    "LITVS": {"-": ["13"], "N": ["12"]},
    "LITVVS": {"-": ["12"]},
    "LIVENS": {"-": ["25"]},
    "LIVEO": {"-": ["52"]},
    "LIVESCO": {"-": ["53"]},
    "LIVIA": {"N": ["11"]},
    "LIVIDVLVS": {"-": ["21"]},
    "LIVIDVS": {"-": ["21"]},
    "LIVIVS": {"N": ["12"], "A": ["21"]},
    "LIVOR": {"-": ["13"], "N": ["13"]},
    "LIXA": {"1": ["11"], "-": ["11"]},
    "LIXIVVS": {"-": ["21"]},
    "LOCALIS": {"-": ["24"]},
    "LOCATIO": {"-": ["13"]},
    "LOCATOR": {"-": ["13"]},
    "LOCO": {"-": ["51"]},
    "LOCRENSES": {"N": ["13"]},
    "LOCVLAMENTVM": {"-": ["12"]},
    "LOCVLVS": {"-": ["12"]},
    "LOCVPLES": {"-": ["25"]},
    "LOCVPLETO": {"-": ["51"]},
    "LOCVS": {"-": ["12"]},
    "LOCVSTA": {"-": ["11"], "N": ["11"]},
    "LOCVTIO": {"-": ["13"]},
    "LODIX": {"-": ["13"]},
    "LOGEION": {"-": ["17"]},
    "LOGOS": {"-": ["17"]},
    "LOLIGO": {"-": ["13"]},
    "LOLIVM": {"-": ["12"]},
    "LOLLIVS": {"N": ["12"]},
    "LONGAEVVS": {"-": ["21"]},
    "LONGE": {"-": ["60"]},
    "LONGINQVITAS": {"-": ["13"]},
    "LONGINQVVM": {"1": ["12"], "2": ["60"]},
    "LONGINQVVS": {"-": ["21"]},
    "LONGITVDO": {"-": ["13"]},
    "LONGVLA": {"N": ["11"], "N12": ["11"]},
    "LONGVM": {"2": ["60", "6-"], "1": ["12"]},
    "LONGVRIVS": {"-": ["12"]},
    "LONGVS": {"-": ["21"], "A": ["21"], "N": ["12"]},
    "LOQVACITER": {"-": ["60"]},
    "LOQVACVLVS": {"-": ["21"]},
    "LOQVAX": {"-": ["25"]},
    "LOQVELA": {"-": ["11"]},
    "LOQVOR": {"-": ["5L"]},
    "LORA": {"-": ["11"]},
    "LOREVS": {"-": ["21"]},
    "LORICA": {"-": ["11"]},
    "LORICATIO": {"-": ["13"]},
    "LORICATVS": {"-": ["21"]},
    "LORICO": {"-": ["51"]},
    "LORICVLA": {"-": ["11"]},
    "LORIPES": {"-": ["13"]},
    "LORVM": {"-": ["12"]},
    "LOTIO": {"-": ["13"]},
    "LOTIVM": {"-": ["12"]},
    "LOTOS": {"-": ["17"]},
    "LVBRICO": {"-": ["51"]},
    "LVBRICVM": {"-": ["12"]},
    "LVBRICVS": {"-": ["21"]},
    "LVCA": {"-": ["11"], "N": ["11"], "A": ["21"]},
    "LVCAR": {"-": ["13"]},
    "LVCCEIVS": {"N": ["12"]},
    "LVCELLVM": {"-": ["12"]},
    "LVCENS": {"-": ["25"]},
    "LVCEO": {"-": ["52"]},
    "LVCERES": {"N": ["13"]},
    "LVCERNA": {"-": ["11"]},
    "LVCESCO": {"-": ["53"]},
    "LVCIDE": {"-": ["6&"]},
    "LVCIDVM": {"-": ["60"]},
    "LVCIDVS": {"-": ["21"]},
    "LVCIFER": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "LVCIFVGA": {"-": ["11"]},
    "LVCIFVGVS": {"-": ["21"]},
    "LVCILIVS": {"N": ["12"]},
    "LVCIVS": {"N": ["12"]},
    "LVCRETIA": {"N": ["11"]},
    "LVCRETIVS": {"N": ["12"]},
    "LVCRIFICO": {"-": ["51"]},
    "LVCRINVS": {"N": ["21"], "A": ["21"]},
    "LVCROR": {"-": ["5J"]},
    "LVCROSVS": {"-": ["21"]},
    "LVCRVM": {"-": ["12"]},
    "LVCTA": {"-": ["11"]},
    "LVCTATIO": {"-": ["13"]},
    "LVCTATOR": {"-": ["13"]},
    "LVCTIFER": {"-": ["21"]},
    "LVCTIFICABILIS": {"-": ["23"]},
    "LVCTIFICVS": {"-": ["21"]},
    "LVCTOR": {"-": ["5J"]},
    "LVCTVOSVS": {"-": ["21"]},
    "LVCTVS": {"-": ["14"], "N": ["12"]},
    "LVCVBRATIO": {"-": ["13"]},
    "LVCVBRO": {"-": ["51"]},
    "LVCVLENTVS": {"-": ["21"]},
    "LVCVLLVS": {"N": ["12"]},
    "LVCVMO": {"N": ["13"]},
    "LVCVS": {"-": ["12"], "A": ["21"], "N": ["12"]},
    "LVDIA": {"-": ["11"]},
    "LVDIBRIVM": {"-": ["12"]},
    "LVDIBVNDVS": {"-": ["21"]},
    "LVDICER": {"-": ["21"]},
    "LVDICRVM": {"-": ["12"]},
    "LVDICRVS": {"-": ["21"]},
    "LVDIFICATIO": {"-": ["13"]},
    "LVDIFICO": {"-": ["51"]},
    "LVDIFICOR": {"-": ["5J"]},
    "LVDIVS": {"-": ["12"]},
    "LVDO": {"-": ["53"]},
    "LVDVS": {"-": ["12"], "N": ["12"]},
    "LVELA": {"-": ["11"]},
    "LVES": {"-": ["13"], "N": ["13"]},
    "LVGEO": {"-": ["52"]},
    "LVGVBRE": {"-": ["60"]},
    "LVGVBRIA": {"-": ["13"]},
    "LVGVBRIS": {"-": ["24"]},
    "LVMBRICVS": {"-": ["12"]},
    "LVMBVS": {"-": ["12"]},
    "LVMEN": {"-": ["13"]},
    "LVMINARE": {"-": ["12"]},
    "LVMINO": {"-": ["51"]},
    "LVMINOSVS": {"-": ["21"]},
    "LVNA": {"-": ["11"], "N": ["11"]},
    "LVNARIS": {"-": ["24"], "A": ["24"]},
    "LVNATVS": {"-": ["21"]},
    "LVNO": {"-": ["51"]},
    "LVO": {"-": ["53"]},
    "LVPA": {"-": ["11"]},
    "LVPANAR": {"-": ["13"]},
    "LVPATA": {"-": ["12"]},
    "LVPATVS": {"-": ["21"]},
    "LVPERCAL": {"N": ["13"]},
    "LVPINARIVS": {"-": ["21"]},
    "LVPINVM": {"-": ["12"]},
    "LVPINVS": {"1": ["12"], "2": ["21"]},
    "LVPVS": {"-": ["12"], "N": ["12"]},
    "LVRIDVS": {"-": ["21"]},
    "LVROR": {"-": ["13"]},
    "LVSCINIA": {"-": ["11"]},
    "LVSCINIVS": {"-": ["12"]},
    "LVSCITIO": {"-": ["13"]},
    "LVSCITIOSVS": {"-": ["21"]},
    "LVSCVS": {"-": ["21"], "N": ["12"]},
    "LVSOR": {"-": ["13"]},
    "LVSORIAE": {"-": ["11"]},
    "LVSORIVM": {"-": ["12"]},
    "LVSORIVS": {"-": ["21"]},
    "LVSTRALIS": {"-": ["24"]},
    "LVSTRATIO": {"-": ["13"]},
    "LVSTRO": {"1": ["13"], "2": ["51"]},
    "LVSTRVM": {"1": ["12"], "2": ["12"]},
    "LVSVS": {"-": ["14"]},
    "LVTAMENTVM": {"-": ["12"]},
    "LVTATIVS": {"N": ["12"], "A": ["21"]},
    "LVTEOLVS": {"-": ["21"]},
    "LVTEVM": {"-": ["12"]},
    "LVTEVS": {"1": ["21"], "2": ["21"]},
    "LVTO": {"1": ["51"]},
    "LVTOSVS": {"-": ["21"]},
    "LVTRON": {"-": ["17"]},
    "LVTVLENTVS": {"-": ["21"]},
    "LVTVM": {"1": ["12"], "2": ["12"]},
    "LVX": {"-": ["13"], "N": ["13"]},
    "LVXO": {"-": ["51"]},
    "LVXVM": {"-": ["12"]},
    "LVXVRIA": {"-": ["11"], "N": ["11"]},
    "LVXVRIES": {"-": ["15"]},
    "LVXVRIO": {"-": ["51"]},
    "LVXVRIOR": {"-": ["5J"]},
    "LVXVRIOSE": {"-": ["60"]},
    "LVXVRIOSVS": {"-": ["21"]},
    "LVXVS": {"1": ["14"], "2": ["14"], "3": ["21"]},
    "LYCAEVS": {"A": ["21"], "N": ["12"]},
    "LYCHNOBIVS": {"-": ["12"]},
    "LYCHNVS": {"-": ["12"]},
    "LYCIA": {"N": ["11"]},
    "LYCII": {"N": ["12"]},
    "LYCISCVS": {"-": ["12"]},
    "LYMPHA": {"-": ["11"], "N": ["11"]},
    "LYMPHATICVS": {"1": ["12"], "2": ["21"]},
    "LYMPHATVS": {"1": ["14"], "2": ["21"]},
    "LYMPHO": {"-": ["51"]},
    "LYNX": {"-": ["17"]},
    "LYRA": {"-": ["11"], "N": ["11"]},
    "LYRICI": {"-": ["12"], "2": ["12"], "1": ["12"]},
    "LYRICVS": {"-": ["21"]},
    "LYSIMACHVS": {"N": ["12"]},
    "LYSIS": {"-": ["13"], "N": ["13"]},
    "LYSO": {"N": ["17"]},
    "MACEDONES": {"N": ["17"]},
    "MACEDONIA": {"N": ["11"]},
    "MACEDONICVS": {"N": ["21"], "A": ["21"]},
    "MACELLVM": {"-": ["12"]},
    "MACER": {"-": ["21"], "N": ["12"]},
    "MACERATIO": {"-": ["13"]},
    "MACERESCO": {"-": ["53"]},
    "MACERIA": {"-": ["11"]},
    "MACERO": {"-": ["51"]},
    "MACHAERA": {"-": ["11"], "N": ["11"]},
    "MACHINA": {"-": ["11"]},
    "MACHINAMENTVM": {"-": ["12"]},
    "MACHINATIO": {"-": ["13"]},
    "MACHINATOR": {"-": ["13"]},
    "MACHINATRIX": {"-": ["13"]},
    "MACHINOR": {"-": ["5J"]},
    "MACIES": {"-": ["15"], "N": ["13"]},
    "MACRESCO": {"-": ["53"]},
    "MACRITAS": {"-": ["13"]},
    "MACRO": {"N": ["13"]},
    "MACTABILIS": {"-": ["24"]},
    "MACTATOR": {"-": ["13"]},
    "MACTATVS": {"-": ["14"]},
    "MACTE": {"-": ["90"]},
    "MACTO": {"-": ["51"]},
    "MACTVS": {"-": ["21"]},
    "MACVLA": {"-": ["11"]},
    "MACVLO": {"-": ["51"]},
    "MACVLOSVS": {"-": ["21"]},
    "MADEFACIO": {"-": ["55"]},
    "MADEFIO": {"-": ["56"]},
    "MADEO": {"-": ["52"]},
    "MADESCO": {"-": ["53"]},
    "MADIDVS": {"-": ["21"]},
    "MAECENAS": {"N": ["13"]},
    "MAECILIVS": {"N": ["12"]},
    "MAENA": {"-": ["11"]},
    "MAENAS": {"-": ["17"], "N": ["17"]},
    "MAERENS": {"-": ["25"]},
    "MAEREO": {"-": ["52"]},
    "MAEROR": {"-": ["13"]},
    "MAESTITIA": {"-": ["11"]},
    "MAESTVS": {"-": ["21"]},
    "MAEVIVS": {"N": ["12"]},
    "MAGA": {"-": ["11"]},
    "MAGALIA": {"-": ["13"]},
    "MAGE": {"-": ["6&"]},
    "MAGICVS": {"-": ["21"]},
    "MAGIS": {"1": ["13"], "2": ["6&"]},
    "MAGISTER": {"-": ["12"]},
    "MAGISTERIVM": {"-": ["12"]},
    "MAGISTRA": {"-": ["11"]},
    "MAGISTRATVS": {"-": ["14"]},
    "MAGIVS": {"N": ["12"]},
    "MAGNANIMITAS": {"-": ["13"]},
    "MAGNANIMVS": {"-": ["21"]},
    "MAGNAS": {"-": ["13"]},
    "MAGNES": {"-": ["17"], "N": ["13"]},
    "MAGNETICVS": {"-": ["21"]},
    "MAGNIFICE": {"-": ["60"]},
    "MAGNIFICENS": {"-": ["25"]},
    "MAGNIFICENTER": {"-": ["60"]},
    "MAGNIFICENTIA": {"-": ["11"]},
    "MAGNIFICO": {"-": ["51"]},
    "MAGNIFICVS": {"-": ["21"]},
    "MAGNILOQVVS": {"-": ["21"]},
    "MAGNITVDO": {"-": ["13"]},
    "MAGNOPERE": {"-": ["60"]},
    "MAGNVM": {"1": ["12"], "2": ["60"]},
    "MAGNVS": {"-": ["21"], "A": ["21"], "N": ["12"]},
    "MAGVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "MAIESTAS": {"-": ["13"], "N": ["13"]},
    "MAIORES": {"-": ["13"]},
    "MAIVS": {"A": ["21"], "N": ["12"]},
    "MAIVSCVLVS": {"-": ["21"]},
    "MALA": {"-": ["11"]},
    "MALACIA": {"-": ["11"]},
    "MALAXO": {"-": ["51"]},
    "MALE": {"-": ["60"]},
    "MALEDICENS": {"-": ["25"]},
    "MALEDICO": {"2": ["53"]},
    "MALEDICTVM": {"-": ["12"]},
    "MALEDICVS": {"-": ["21"]},
    "MALEFACIO": {"-": ["55"]},
    "MALEFACTVM": {"-": ["12"]},
    "MALEFICIVM": {"-": ["12"]},
    "MALEFICVS": {"-": ["21"]},
    "MALESVADVS": {"-": ["21"]},
    "MALEVOLENTIA": {"-": ["11"]},
    "MALEVOLVS": {"-": ["21"]},
    "MALI": {"-": ["12"]},
    "MALIGNE": {"-": ["60"]},
    "MALIGNITAS": {"-": ["13"]},
    "MALIGNVS": {"-": ["21"]},
    "MALITIA": {"-": ["11"]},
    "MALITIOSA": {"N": ["11"]},
    "MALITIOSE": {"-": ["60"]},
    "MALLEOLVS": {"-": ["12"], "N": ["12"]},
    "MALLEVS": {"-": ["12"]},
    "MALLVS": {"-": ["12"], "N": ["12"]},
    "MALO": {"-": ["56"]},
    "MALOBATHRON": {"-": ["17"]},
    "MALVA": {"-": ["11"]},
    "MALVM": {"1": ["12"], "2": ["12"], "3": ["90"]},
    "MALVS": {"1": ["12"], "2": ["12"], "3": ["21"]},
    "MAMERTINI": {"N": ["12"]},
    "MAMERTINVS": {"A": ["21"]},
    "MAMILIVS": {"N": ["12"], "A": ["21"]},
    "MAMILLA": {"-": ["11"]},
    "MAMMA": {"-": ["11"]},
    "MAMMOSVS": {"-": ["21"]},
    "MANABILIS": {"-": ["24"]},
    "MANAEVS": {"-": ["21"]},
    "MANCEPS": {"-": ["13"]},
    "MANCIPIVM": {"-": ["12"]},
    "MANCIPO": {"-": ["51"]},
    "MANCVS": {"-": ["21"]},
    "MANDATVM": {"-": ["12"]},
    "MANDATVS": {"-": ["14"]},
    "MANDELA": {"-": ["11"]},
    "MANDO": {"1": ["13"], "2": ["51"], "3": ["53"]},
    "MANDRA": {"-": ["11"]},
    "MANDVCO": {"1": ["13"], "2": ["51"]},
    "MANE": {"1": ["16"], "2": ["60"]},
    "MANEO": {"-": ["52"]},
    "MANES": {"-": ["13"], "N": ["13"]},
    "MANGO": {"-": ["13"]},
    "MANICA": {"-": ["11"]},
    "MANICATVS": {"-": ["21"]},
    "MANICVLVS": {"-": ["12"]},
    "MANIFESTE": {"-": ["60"]},
    "MANIFESTO": {"1": ["51"], "2": ["60"]},
    "MANIFESTVS": {"-": ["21"]},
    "MANILIVS": {"N": ["12"], "A": ["21"]},
    "MANIPVLARIS": {"-": ["24"], "2": ["24"], "1": ["21"]},
    "MANIPVLVS": {"-": ["12"]},
    "MANIVS": {"N": ["12"]},
    "MANLIVS": {"N": ["12"], "A": ["21"]},
    "MANNVS": {"-": ["12"], "N": ["12"]},
    "MANO": {"-": ["51"]},
    "MANSITO": {"-": ["51"]},
    "MANSVEFACIO": {"-": ["55"]},
    "MANSVEFIO": {"-": ["56"]},
    "MANSVES": {"-": ["22"]},
    "MANSVESCO": {"-": ["53"]},
    "MANSVETE": {"-": ["60"]},
    "MANSVETVDO": {"-": ["13"]},
    "MANSVETVS": {"-": ["21"], "N": ["12"]},
    "MANTELE": {"-": ["13"]},
    "MANTICA": {"-": ["11"]},
    "MANVALIS": {"-": ["24"]},
    "MANVBIAE": {"-": ["11"]},
    "MANVBRIVM": {"-": ["12"]},
    "MANVCLA": {"-": ["11"]},
    "MANVLEATVS": {"-": ["21"]},
    "MANVMISSIO": {"-": ["13"]},
    "MANVMITTO": {"-": ["53"]},
    "MANVPRETIOSVS": {"-": ["21"]},
    "MANVPRETIVM": {"-": ["12"]},
    "MANVS": {"1": ["14"], "2": ["21"]},
    "MAPALIA": {"-": ["13"]},
    "MAPPA": {"-": ["11"]},
    "MARATHVS": {"-": ["12"], "N": ["12"]},
    "MARCELLIA": {"N": ["12"]},
    "MARCELLINVS": {"N": ["12"]},
    "MARCELLVS": {"N": ["12"]},
    "MARCENS": {"-": ["25"]},
    "MARCEO": {"-": ["52"]},
    "MARCESCO": {"-": ["53"]},
    "MARCIA": {"N": ["11"]},
    "MARCIDVS": {"-": ["21"]},
    "MARCIVS": {"N": ["12"], "A": ["21"]},
    "MARCOR": {"-": ["13"]},
    "MARCVS": {"N": ["12"]},
    "MARE": {"-": ["13"], "N": ["13"]},
    "MARGARITA": {"-": ["11"], "N": ["11"]},
    "MARGARITVM": {"-": ["12"]},
    "MARGO": {"-": ["13"]},
    "MARINVS": {"-": ["21"], "N": ["12"]},
    "MARISCA": {"-": ["11"]},
    "MARITA": {"-": ["11"]},
    "MARITALIS": {"-": ["24"]},
    "MARITIMI": {"-": ["12"]},
    "MARITIMVS": {"-": ["21"]},
    "MARITO": {"-": ["51"]},
    "MARITVS": {"1": ["12"], "2": ["21"]},
    "MARIVS": {"N": ["12"]},
    "MARMOR": {"-": ["13"]},
    "MARMORARIVS": {"1": ["12"], "2": ["21"]},
    "MARMOREVS": {"-": ["21"]},
    "MARRA": {"-": ["11"]},
    "MARS": {"N": ["13"]},
    "MARTIALIS": {"-": ["24"], "N": ["13"], "A": ["12"]},
    "MARTIVS": {"A": ["21"], "N": ["12"]},
    "MAS": {"-": ["13"]},
    "MASCVLINVS": {"-": ["21"]},
    "MASCVLVS": {"-": ["21"]},
    "MASINISSA": {"N": ["11"]},
    "MASSA": {"-": ["11"], "N": ["11"]},
    "MASSILIA": {"N": ["11"]},
    "MATARA": {"-": ["11"]},
    "MATELLA": {"-": ["11"]},
    "MATELLIO": {"-": ["13"]},
    "MATEOLA": {"-": ["11"]},
    "MATER": {"-": ["13"], "N": ["13"]},
    "MATERCVLA": {"-": ["11"]},
    "MATERIA": {"-": ["11"]},
    "MATERIALIS": {"-": ["24"]},
    "MATERIATIO": {"-": ["13"]},
    "MATERIATVRA": {"-": ["11"]},
    "MATERIES": {"-": ["15"]},
    "MATERINVS": {"-": ["21"]},
    "MATERIO": {"-": ["51"]},
    "MATERIOR": {"-": ["5J"]},
    "MATERNVS": {"-": ["21"], "N": ["12"]},
    "MATERTERA": {"-": ["11"]},
    "MATHEMATICA": {"-": ["17"]},
    "MATHEMATICVS": {"1": ["12"], "2": ["21"]},
    "MATRIMONIVM": {"-": ["12"]},
    "MATRIMVS": {"-": ["21"]},
    "MATRINIVS": {"N": ["12"]},
    "MATRONA": {"-": ["11"], "N": ["11"]},
    "MATRONALIA": {"-": ["13"]},
    "MATRONALIS": {"-": ["24"]},
    "MATVRE": {"-": ["60"]},
    "MATVRESCO": {"-": ["53"]},
    "MATVRITAS": {"-": ["13"]},
    "MATVRO": {"-": ["51"]},
    "MATVRVS": {"-": ["21"], "N": ["12"]},
    "MATVTINVM": {"-": ["12"]},
    "MATVTINVS": {"-": ["21"], "A": ["21"]},
    "MAVRETANIA": {"N": ["11"]},
    "MAXILLA": {"-": ["11"]},
    "MAXIMITAS": {"-": ["13"]},
    "MAXIMVM": {"-": ["6-"]},
    "MAXIMVS": {"N": ["12"]},
    "MAZONOMVS": {"-": ["12"]},
    "MEATVS": {"-": ["14"]},
    "MECHANICOS": {"-": ["60"]},
    "MEDEA": {"N": ["11"]},
    "MEDENS": {"-": ["13"]},
    "MEDEOR": {"-": ["5K"]},
    "MEDIANVS": {"-": ["21"]},
    "MEDIASTINVS": {"-": ["12"]},
    "MEDICA": {"1": ["11"], "2": ["11"]},
    "MEDICABILIS": {"-": ["24"]},
    "MEDICAMEN": {"-": ["13"]},
    "MEDICAMENTOSVS": {"-": ["21"]},
    "MEDICAMENTVM": {"-": ["12"]},
    "MEDICATVS": {"1": ["14"], "2": ["21"]},
    "MEDICINA": {"-": ["11"]},
    "MEDICO": {"-": ["51"]},
    "MEDICOR": {"-": ["5J"]},
    "MEDICVS": {"1": ["12"], "2": ["21"], "N": ["21"], "A": ["21"]},
    "MEDIETAS": {"-": ["13"]},
    "MEDIMNVM": {"-": ["12"]},
    "MEDIO": {"-": ["51"]},
    "MEDIOCRICVLVS": {"-": ["21"]},
    "MEDIOCRIS": {"-": ["24"]},
    "MEDIOCRITAS": {"-": ["13"]},
    "MEDIOCRITER": {"-": ["60"]},
    "MEDIPONTVS": {"-": ["12"]},
    "MEDITAMENTVM": {"-": ["12"]},
    "MEDITATE": {"-": ["60"]},
    "MEDITATIO": {"-": ["13"]},
    "MEDITERRANEVM": {"-": ["12"]},
    "MEDITERRANEVS": {"-": ["21"]},
    "MEDITOR": {"-": ["5J"]},
    "MEDIVM": {"-": ["12"]},
    "MEDIVS": {"-": ["21"]},
    "MEDIVSFIDIVS": {"-": ["90"]},
    "MEDVLLA": {"-": ["11"]},
    "MEDVLLIA": {"N": ["11"]},
    "MEDVSA": {"-": ["11"], "N": ["11"]},
    "MEGALENSIS": {"-": ["24"], "A": ["24"]},
    "MEGALOGRAPHIA": {"-": ["11"]},
    "MEGARIS": {"N": ["17"]},
    "MEGISTANES": {"-": ["17"]},
    "MEHERCVLE": {"-": ["90"]},
    "MEI": {"-": ["42"]},
    "MEIO": {"-": ["53"]},
    "MEL": {"-": ["13"]},
    "MELA": {"N": ["11"]},
    "MELANTHIVM": {"-": ["12"]},
    "MELICHROS": {"-": ["17"]},
    "MELICVS": {"1": ["12"], "2": ["21"]},
    "MELIMELVM": {"-": ["12"]},
    "MELIORO": {"-": ["51"]},
    "MELISPHYLLVM": {"-": ["12"]},
    "MELITA": {"N": ["17"]},
    "MELITENSES": {"N": ["13"]},
    "MELITENSIA": {"N": ["13"]},
    "MELITENSIS": {"N": ["24"], "A": ["24"]},
    "MELIVSCVLVS": {"-": ["21"]},
    "MELLITVS": {"-": ["21"]},
    "MELOGRAPHIA": {"-": ["11"]},
    "MELOS": {"-": ["17"]},
    "MEMBRANA": {"-": ["11"]},
    "MEMBRATIM": {"-": ["60"]},
    "MEMBRATVRA": {"-": ["11"]},
    "MEMBRVM": {"-": ["12"]},
    "MEMINI": {"-": ["53"]},
    "MEMOR": {"-": ["26"]},
    "MEMORABILIS": {"-": ["24"]},
    "MEMORANDVS": {"-": ["21"]},
    "MEMORATOR": {"-": ["13"]},
    "MEMORIA": {"-": ["11"]},
    "MEMORITER": {"-": ["60"]},
    "MEMORO": {"-": ["51"]},
    "MENAENI": {"N": ["12"]},
    "MENAENVS": {"N": ["21"], "A": ["21"]},
    "MENDA": {"-": ["11"]},
    "MENDACIVM": {"-": ["12"]},
    "MENDAX": {"-": ["25"]},
    "MENDICATIO": {"-": ["13"]},
    "MENDICE": {"-": ["60"]},
    "MENDICITAS": {"-": ["13"]},
    "MENDICO": {"-": ["51"]},
    "MENDICVS": {"-": ["21"], "2": ["12"], "1": ["12"]},
    "MENDO": {"-": ["51"]},
    "MENDOSE": {"-": ["60"]},
    "MENDOSVS": {"-": ["21"]},
    "MENDVM": {"-": ["12"]},
    "MENENIANVS": {"N": ["21"]},
    "MENENIVS": {"N": ["12"]},
    "MENISCVS": {"N": ["12"]},
    "MENS": {"-": ["13"], "N": ["13"]},
    "MENSA": {"-": ["11"]},
    "MENSIS": {"-": ["13"]},
    "MENSOR": {"-": ["13"]},
    "MENSTRVA": {"-": ["12"]},
    "MENSTRVALIS": {"-": ["24"]},
    "MENSTRVVM": {"-": ["12"]},
    "MENSTRVVS": {"-": ["21"]},
    "MENSVRA": {"-": ["11"]},
    "MENTA": {"-": ["11"]},
    "MENTIO": {"1": ["13"], "2": ["54"]},
    "MENTIOR": {"-": ["5M"]},
    "MENTOR": {"N": ["13"]},
    "MENTVLA": {"-": ["11"], "N": ["11"]},
    "MENTVM": {"-": ["12"]},
    "MEO": {"-": ["51"]},
    "MEPHITIS": {"-": ["13"]},
    "MERACE": {"-": ["6&"]},
    "MERACVS": {"-": ["21"]},
    "MERCABILIS": {"-": ["24"]},
    "MERCATOR": {"-": ["13"]},
    "MERCATVRA": {"-": ["11"]},
    "MERCATVS": {"-": ["14"]},
    "MERCEDVLA": {"-": ["11"]},
    "MERCENARIVS": {"1": ["12"], "2": ["21"]},
    "MERCES": {"-": ["13"]},
    "MERCIMONIVM": {"-": ["12"]},
    "MERCOR": {"-": ["5J"]},
    "MERCVRIALIS": {"-": ["21"], "A": ["24"]},
    "MERCVRIVS": {"N": ["12"]},
    "MERDA": {"-": ["11"]},
    "MERENS": {"-": ["25"]},
    "MEREO": {"-": ["52"]},
    "MEREOR": {"-": ["5K"]},
    "MERETRICIVS": {"-": ["21"]},
    "MERETRICVLA": {"-": ["11"]},
    "MERETRIX": {"-": ["13"]},
    "MEREVS": {"-": ["12"]},
    "MERGA": {"-": ["11"]},
    "MERGES": {"-": ["13"]},
    "MERGO": {"-": ["53"]},
    "MERGVS": {"-": ["12"], "N": ["12"]},
    "MERIDIANO": {"-": ["60"]},
    "MERIDIANVM": {"-": ["12"]},
    "MERIDIANVS": {"-": ["21"]},
    "MERIDIES": {"-": ["15"]},
    "MERIDIO": {"-": ["51"]},
    "MERITO": {"1": ["51"], "2": ["60"]},
    "MERITORIVM": {"-": ["12"]},
    "MERITORIVS": {"-": ["21"]},
    "MERITVM": {"-": ["12"]},
    "MERITVS": {"-": ["21"]},
    "MEROPS": {"-": ["13"], "N": ["13"]},
    "MEROS": {"-": ["17"], "N": ["17"]},
    "MERSO": {"-": ["51"]},
    "MERVLA": {"-": ["11"], "N": ["11"]},
    "MERVM": {"-": ["12"]},
    "MERVS": {"-": ["21"]},
    "MERX": {"-": ["13"]},
    "MESAVLOE": {"-": ["17"]},
    "MESE": {"-": ["17"]},
    "MESIA": {"N": ["11"]},
    "MESOLABIVM": {"-": ["12"]},
    "MESOS": {"-": ["27"]},
    "MESSALA": {"N": ["11"]},
    "MESSANA": {"N": ["11"]},
    "MESSIS": {"-": ["13"]},
    "MESSOR": {"-": ["13"]},
    "META": {"-": ["11"], "N": ["11"]},
    "METALLVM": {"-": ["12"]},
    "METAPONTVM": {"N": ["12"]},
    "METAXA": {"-": ["11"]},
    "METELLVS": {"N": ["12"]},
    "METHODVS": {"-": ["17"]},
    "METILIVS": {"N": ["12"]},
    "METIOR": {"-": ["5M"]},
    "METO": {"1": ["51"], "2": ["53"]},
    "METOPA": {"-": ["17"]},
    "METOR": {"-": ["5J"]},
    "METRETA": {"-": ["11"]},
    "METRODORVS": {"N": ["12"]},
    "METROPOLIS": {"N": ["17"]},
    "METROPOLITAE": {"N": ["17"]},
    "METRVM": {"-": ["12"]},
    "METTIVS": {"N": ["12"]},
    "METVENS": {"-": ["25"]},
    "METVO": {"-": ["53"]},
    "METVS": {"-": ["14"], "N": ["12"]},
    "MEVM": {"-": ["12"]},
    "MEVS": {"-": ["42"]},
    "MEZENTIVS": {"N": ["12"]},
    "MIA": {"-": ["11"]},
    "MICA": {"-": ["11"]},
    "MICANS": {"-": ["25"]},
    "MICO": {"-": ["51"]},
    "MICTVRIO": {"-": ["54"]},
    "MIGRATIO": {"-": ["13"]},
    "MIGRO": {"-": ["51"]},
    "MILES": {"-": ["13"]},
    "MILESII": {"N": ["12"]},
    "MILESIVS": {"N": ["21"], "A": ["21"]},
    "MILETVS": {"N": ["12"]},
    "MILIA": {"-": ["13"]},
    "MILIARIVS": {"-": ["21"]},
    "MILITARIS": {"-": ["24"]},
    "MILITIA": {"-": ["11"]},
    "MILITO": {"1": ["13"], "2": ["51"]},
    "MILIVM": {"-": ["12"]},
    "MILLE": {"-": ["31"]},
    "MILLESIMVS": {"-": ["32"]},
    "MILLIARIVM": {"-": ["12"]},
    "MILLIARIVS": {"-": ["21"]},
    "MILLIES": {"-": ["36"]},
    "MILVVS": {"-": ["12"], "N": ["12"]},
    "MILYAE": {"N": ["17"]},
    "MIMA": {"-": ["11"]},
    "MIMICE": {"-": ["60"]},
    "MIMICVS": {"-": ["21"]},
    "MIMVS": {"-": ["12"]},
    "MINA": {"-": ["11"]},
    "MINACITER": {"-": ["60"]},
    "MINAE": {"-": ["11"], "N": ["11"]},
    "MINANTER": {"-": ["60"]},
    "MINAX": {"-": ["25"]},
    "MINEO": {"-": ["52"]},
    "MINERVA": {"N": ["11"]},
    "MINGO": {"-": ["53"]},
    "MINIACEVS": {"-": ["21"]},
    "MINIMVM": {"1": ["12"], "2": ["6-", "12"]},
    "MINISCOR": {"-": ["5L"]},
    "MINISTER": {"1": ["12"], "2": ["21"]},
    "MINISTERIVM": {"-": ["12"]},
    "MINISTRA": {"-": ["11"]},
    "MINISTRATIO": {"-": ["13"]},
    "MINISTRATOR": {"-": ["13"]},
    "MINISTRO": {"-": ["51"]},
    "MINITABVNDVS": {"-": ["21"]},
    "MINITOR": {"-": ["5J"]},
    "MINIVM": {"-": ["12"]},
    "MINOR": {"-": ["5J"]},
    "MINVCIVS": {"A": ["21"], "N": ["12"]},
    "MINVO": {"-": ["53"]},
    "MINVS": {"-": ["12"]},
    "MINVSCVLVS": {"-": ["21"]},
    "MINVTAL": {"-": ["13"]},
    "MINVTATIM": {"-": ["60"]},
    "MINVTE": {"-": ["60"]},
    "MINVTIA": {"-": ["11"]},
    "MINVTIM": {"-": ["60"]},
    "MINVTVM": {"-": ["12"]},
    "MINVTVS": {"1": ["12"], "2": ["21"]},
    "MIRABILE": {"-": ["60"]},
    "MIRABILIS": {"-": ["24"]},
    "MIRABVNDVS": {"-": ["21"]},
    "MIRACVLVM": {"-": ["12"]},
    "MIRANDE": {"-": ["60"]},
    "MIRANDVS": {"-": ["21"]},
    "MIRATOR": {"-": ["13"]},
    "MIRATRIX": {"-": ["13"]},
    "MIRE": {"-": ["60"]},
    "MIRIFICE": {"-": ["60"]},
    "MIRIFICVS": {"-": ["21"]},
    "MIRMILLO": {"-": ["13"]},
    "MIROR": {"-": ["5J"]},
    "MIRVM": {"-": ["12"]},
    "MIRVS": {"-": ["21"]},
    "MISCELLANEA": {"-": ["12"]},
    "MISCELLVS": {"-": ["21"]},
    "MISCEO": {"-": ["52"]},
    "MISELLVS": {"-": ["21"]},
    "MISER": {"-": ["21"]},
    "MISERABILE": {"-": ["60"]},
    "MISERABILIS": {"-": ["24"]},
    "MISERABILITER": {"-": ["60"]},
    "MISERANDVS": {"-": ["21"]},
    "MISERATIO": {"-": ["13"]},
    "MISERE": {"-": ["60"]},
    "MISEREO": {"-": ["52"]},
    "MISEREOR": {"-": ["5K"]},
    "MISERESCO": {"-": ["53"]},
    "MISERIA": {"-": ["11"]},
    "MISERICORDIA": {"-": ["11"]},
    "MISERICORS": {"-": ["25"]},
    "MISERITER": {"-": ["60"]},
    "MISEROR": {"-": ["5J"]},
    "MISSILE": {"-": ["13"]},
    "MISSILIA": {"-": ["13"]},
    "MISSILIS": {"-": ["24"]},
    "MISSIO": {"-": ["13"]},
    "MISSITO": {"-": ["51"]},
    "MISSVS": {"-": ["14"]},
    "MITE": {"-": ["6&"]},
    "MITESCO": {"-": ["53"]},
    "MITHRIDATES": {"N": ["17"]},
    "MITIGO": {"-": ["51"]},
    "MITIS": {"-": ["24"]},
    "MITRA": {"-": ["11"]},
    "MITRATVS": {"-": ["21"]},
    "MITTO": {"-": ["53"]},
    "MITVLVS": {"-": ["12"]},
    "MITYLENAE": {"N": ["11"]},
    "MIXTIM": {"-": ["60"]},
    "MIXTIO": {"-": ["13"]},
    "MIXTVRA": {"-": ["11"]},
    "MIXTVS": {"-": ["21"]},
    "MNASISTRATVS": {"N": ["12"]},
    "MNEMOSYNON": {"-": ["17"]},
    "MOBILIS": {"-": ["24"]},
    "MOBILITAS": {"-": ["13"]},
    "MOBILITER": {"-": ["60"]},
    "MOBILITO": {"-": ["51"]},
    "MODERABILIS": {"-": ["24"]},
    "MODERAMEN": {"-": ["13"]},
    "MODERANTER": {"-": ["60"]},
    "MODERATE": {"-": ["60"]},
    "MODERATIM": {"-": ["60"]},
    "MODERATIO": {"-": ["13"]},
    "MODERATOR": {"-": ["13"]},
    "MODERATRIX": {"-": ["13"]},
    "MODERATVS": {"-": ["21"]},
    "MODEROR": {"-": ["5J"]},
    "MODESTE": {"-": ["60"]},
    "MODESTIA": {"-": ["11"]},
    "MODESTVS": {"-": ["21"], "N": ["12"]},
    "MODICE": {"-": ["60"]},
    "MODICVS": {"-": ["21"]},
    "MODIFICATIO": {"-": ["13"]},
    "MODIOLVS": {"-": ["12"]},
    "MODIVM": {"-": ["12"]},
    "MODIVS": {"-": ["12"], "N": ["12"]},
    "MODO": {"1": ["60"], "2": ["82"]},
    "MODVLATIO": {"-": ["13"]},
    "MODVLATOR": {"-": ["13"]},
    "MODVLATVS": {"1": ["14"], "2": ["21"]},
    "MODVLOR": {"-": ["5J"]},
    "MODVLVS": {"-": ["12"]},
    "MODVS": {"-": ["12"]},
    "MOECHA": {"-": ["11"]},
    "MOECHOR": {"-": ["5J"]},
    "MOECHVS": {"-": ["12"]},
    "MOENIA": {"-": ["13"]},
    "MOLA": {"-": ["11"]},
    "MOLARIS": {"1": ["13"], "2": ["24"]},
    "MOLARIVS": {"-": ["21"]},
    "MOLES": {"-": ["13"]},
    "MOLESTE": {"-": ["60"]},
    "MOLESTIA": {"-": ["11"]},
    "MOLESTO": {"-": ["51"]},
    "MOLESTVS": {"-": ["21"]},
    "MOLETRINA": {"-": ["11"]},
    "MOLILE": {"-": ["13"]},
    "MOLIMEN": {"-": ["13"]},
    "MOLIMENTVM": {"-": ["12"]},
    "MOLIOR": {"-": ["5M"]},
    "MOLITIO": {"-": ["13"], "1": ["13"]},
    "MOLITOR": {"-": ["13"]},
    "MOLLESCO": {"-": ["53"]},
    "MOLLICELLVS": {"-": ["21"]},
    "MOLLICVLVS": {"-": ["21"]},
    "MOLLIMENTVM": {"-": ["12"]},
    "MOLLIO": {"-": ["54"]},
    "MOLLIS": {"-": ["24"]},
    "MOLLITER": {"-": ["60"]},
    "MOLLITIA": {"-": ["11"]},
    "MOLLITIES": {"-": ["15"]},
    "MOLLITVDO": {"-": ["13"]},
    "MOLO": {"-": ["53"]},
    "MOMEN": {"-": ["13"]},
    "MOMENTVM": {"-": ["12"]},
    "MONACHA": {"-": ["11"]},
    "MONACHVS": {"-": ["12"]},
    "MONAS": {"-": ["17"]},
    "MONASTERIVM": {"-": ["12"]},
    "MONEO": {"-": ["52"]},
    "MONETA": {"-": ["11"], "N": ["11"]},
    "MONETALIS": {"-": ["24"]},
    "MONILE": {"-": ["13"]},
    "MONIMENTVM": {"-": ["12"]},
    "MONITIO": {"-": ["13"]},
    "MONITOR": {"-": ["13"]},
    "MONITVM": {"-": ["12"]},
    "MONITVS": {"-": ["14"]},
    "MONOPTEROS": {"-": ["27"]},
    "MONOTRIGLYPHVS": {"-": ["27"]},
    "MONS": {"-": ["13"], "N": ["13"]},
    "MONSTRATIO": {"-": ["13"]},
    "MONSTRATOR": {"-": ["13"]},
    "MONSTRO": {"-": ["51"]},
    "MONSTRVM": {"-": ["12"]},
    "MONSTRVOSVS": {"-": ["21"]},
    "MONTANI": {"-": ["12"]},
    "MONTANVS": {"-": ["21"], "N": ["12"]},
    "MONTIFER": {"-": ["21"]},
    "MONTIVAGVS": {"-": ["21"]},
    "MONTVOSVS": {"-": ["21"]},
    "MONVMENTVM": {"-": ["12"]},
    "MORA": {"-": ["11"]},
    "MORALIS": {"-": ["24"]},
    "MORATIO": {"-": ["13"]},
    "MORATOR": {"-": ["13"]},
    "MORATVS": {"-": ["21"]},
    "MORBIDVS": {"-": ["21"]},
    "MORBOSVS": {"-": ["21"]},
    "MORBVS": {"-": ["12"], "N": ["12"]},
    "MORDAX": {"-": ["25"]},
    "MORDEO": {"-": ["52"]},
    "MORDICVS": {"-": ["60"], "2": ["21"]},
    "MORIBVNDVS": {"-": ["21"]},
    "MORIGEROR": {"-": ["5J"]},
    "MORIGERVS": {"-": ["21"]},
    "MORIOR": {"-": ["5N"]},
    "MOROR": {"1": ["5J"], "2": ["5J"]},
    "MOROSVS": {"-": ["21"]},
    "MORS": {"-": ["13"], "N": ["13"]},
    "MORSVS": {"-": ["14"]},
    "MORTALIA": {"-": ["13"]},
    "MORTALIS": {"1": ["13"], "2": ["24"]},
    "MORTALITAS": {"-": ["13"]},
    "MORTARIVM": {"-": ["12"]},
    "MORTICINVS": {"-": ["21"]},
    "MORTIFER": {"-": ["21"]},
    "MORTVVS": {"1": ["12"], "2": ["21"]},
    "MORVM": {"-": ["12"]},
    "MORVS": {"-": ["12"]},
    "MOS": {"-": ["13"]},
    "MOTIO": {"-": ["13"]},
    "MOTIVNCVLA": {"-": ["11"]},
    "MOTO": {"-": ["51"]},
    "MOTVS": {"-": ["14"]},
    "MOVEO": {"-": ["52"]},
    "MOX": {"-": ["60"]},
    "MVCEO": {"-": ["52"]},
    "MVCIA": {"N": ["12"]},
    "MVCIDVS": {"-": ["21"]},
    "MVCIVS": {"A": ["21"], "N": ["12"]},
    "MVCRO": {"-": ["13"]},
    "MVCVS": {"-": ["12"]},
    "MVGILIS": {"-": ["13"]},
    "MVGILLA": {"N": ["11"]},
    "MVGIO": {"-": ["54"]},
    "MVGITVS": {"-": ["14"]},
    "MVLA": {"-": ["11"]},
    "MVLCEO": {"-": ["52"]},
    "MVLCO": {"-": ["51"]},
    "MVLCTRA": {"-": ["11"]},
    "MVLCTRARIVM": {"-": ["12"]},
    "MVLCTRVM": {"-": ["12"]},
    "MVLGEO": {"-": ["52"]},
    "MVLIEBRIA": {"-": ["13"]},
    "MVLIEBRIS": {"-": ["24"], "N": ["24"]},
    "MVLIEBRITER": {"-": ["60"]},
    "MVLIER": {"-": ["13"]},
    "MVLIERARIVS": {"1": ["12"], "2": ["21"]},
    "MVLIERCVLA": {"-": ["11"]},
    "MVLINVS": {"-": ["21"]},
    "MVLIO": {"-": ["13"]},
    "MVLLEVS": {"-": ["21"]},
    "MVLLVS": {"-": ["12"]},
    "MVLSVM": {"-": ["12"]},
    "MVLSVS": {"-": ["21"]},
    "MVLTA": {"1": ["11"], "2": ["60"]},
    "MVLTANGVLVS": {"-": ["21"]},
    "MVLTATIO": {"-": ["13"]},
    "MVLTESIMVS": {"-": ["21"]},
    "MVLTI": {"-": ["12"]},
    "MVLTICIA": {"-": ["12"]},
    "MVLTIFACIO": {"-": ["55"]},
    "MVLTIFARIAM": {"-": ["60"]},
    "MVLTIFIDVS": {"-": ["21"]},
    "MVLTIFORMIS": {"-": ["24"]},
    "MVLTIFORVS": {"-": ["21"]},
    "MVLTIGENVS": {"-": ["21"]},
    "MVLTIMODIS": {"-": ["60"]},
    "MVLTIMODVS": {"-": ["21"]},
    "MVLTIPLEX": {"-": ["25"]},
    "MVLTIPLICATIO": {"-": ["13"]},
    "MVLTIPLICO": {"-": ["51"]},
    "MVLTITO": {"-": ["51"]},
    "MVLTITVDO": {"-": ["13"]},
    "MVLTIVAGVS": {"-": ["21"]},
    "MVLTIVOLVS": {"-": ["21"]},
    "MVLTO": {"1": ["51"], "2": ["60"]},
    "MVLTVM": {"1": ["12"], "2": ["60"]},
    "MVLTVS": {"-": ["21"]},
    "MVLVS": {"-": ["12"]},
    "MVMMIVS": {"N": ["12"]},
    "MVNATIVS": {"-": ["12"], "N": ["12"]},
    "MVNDANVS": {"-": ["21"]},
    "MVNDE": {"-": ["60"]},
    "MVNDITIA": {"-": ["11"], "N": ["11"]},
    "MVNDITIES": {"-": ["15"]},
    "MVNDVS": {"1": ["12"], "2": ["21"]},
    "MVNERO": {"-": ["51"]},
    "MVNEROR": {"-": ["5J"]},
    "MVNIA": {"-": ["16"]},
    "MVNICEPS": {"-": ["13"]},
    "MVNICIPALIS": {"-": ["24"]},
    "MVNICIPIVM": {"-": ["12"]},
    "MVNIFICE": {"-": ["60"]},
    "MVNIFICENTIA": {"-": ["11"]},
    "MVNIFICO": {"-": ["51"]},
    "MVNIFICVS": {"-": ["21"]},
    "MVNIMEN": {"-": ["13"]},
    "MVNIMENTVM": {"-": ["12"]},
    "MVNIO": {"1": ["13"], "2": ["54"]},
    "MVNITIO": {"-": ["13"]},
    "MVNITOR": {"-": ["13"]},
    "MVNITVS": {"-": ["21"]},
    "MVNVS": {"-": ["13"]},
    "MVNVSCVLVM": {"-": ["12"]},
    "MVRAENA": {"-": ["11"]},
    "MVRALIS": {"-": ["24"]},
    "MVRCIA": {"N": ["11"]},
    "MVRENA": {"N": ["11"]},
    "MVREX": {"-": ["13"]},
    "MVRGANTINI": {"N": ["12"]},
    "MVRGANTINVS": {"N": ["21"], "A": ["21"]},
    "MVRIA": {"-": ["11"]},
    "MVRIES": {"-": ["15"]},
    "MVRMVR": {"-": ["13"]},
    "MVRMVRATIO": {"-": ["13"]},
    "MVRMVRO": {"-": ["51"]},
    "MVRVS": {"-": ["12"]},
    "MVS": {"-": ["13"], "N": ["12"]},
    "MVSA": {"-": ["11"], "N": ["11"]},
    "MVSAEVS": {"-": ["21"], "N": ["12"]},
    "MVSCA": {"-": ["11"]},
    "MVSCARIVS": {"-": ["21"]},
    "MVSCIPVLVM": {"-": ["12"]},
    "MVSCOSVS": {"-": ["21"]},
    "MVSCVLVS": {"-": ["12"]},
    "MVSCVS": {"-": ["12"]},
    "MVSICA": {"1": ["12", "11"], "2": ["17"]},
    "MVSICVS": {"1": ["12"], "2": ["21"]},
    "MVSIMO": {"-": ["13"]},
    "MVSSO": {"-": ["51"]},
    "MVSTACEVM": {"-": ["12"]},
    "MVSTACEVS": {"-": ["12"]},
    "MVSTARIVS": {"-": ["21"]},
    "MVSTELA": {"-": ["11"], "N": ["11"]},
    "MVSTEVS": {"-": ["21"]},
    "MVSTIVS": {"N": ["12"]},
    "MVSTVM": {"-": ["12"]},
    "MVSTVS": {"-": ["21"]},
    "MVTABILIS": {"-": ["24"]},
    "MVTABILITAS": {"-": ["13"]},
    "MVTATIO": {"-": ["13"]},
    "MVTILO": {"-": ["51"]},
    "MVTILVS": {"-": ["21"], "N": ["12"]},
    "MVTIO": {"-": ["54"]},
    "MVTO": {"1": ["13"], "2": ["51"]},
    "MVTVLVS": {"-": ["12"]},
    "MVTVO": {"1": ["51"], "2": ["60"]},
    "MVTVOR": {"-": ["5J"]},
    "MVTVS": {"-": ["21"], "N": ["12"], "A": ["21"]},
    "MVTVVM": {"-": ["12"]},
    "MVTVVS": {"-": ["21"]},
    "MVTYCENSIS": {"N": ["24"], "A": ["24"]},
    "MYNDVS": {"N": ["12"]},
    "MYOPARO": {"-": ["17"]},
    "MYRICE": {"-": ["17"]},
    "MYRO": {"N": ["17"]},
    "MYRRHA": {"-": ["11"], "N": ["11"]},
    "MYRRHEVS": {"-": ["21"]},
    "MYRRHINA": {"1": ["11"], "2": ["12"], "N": ["11"]},
    "MYRTA": {"-": ["11"]},
    "MYRTETVM": {"-": ["12"]},
    "MYRTEVS": {"-": ["21"]},
    "MYRTVM": {"-": ["12"]},
    "MYRTVS": {"-": ["16"]},
    "MYSTA": {"-": ["11"]},
    "MYSTAGOGVS": {"-": ["12"]},
    "MYSTERIVM": {"-": ["12"]},
    "MYSTICVS": {"-": ["21"]},
    "MYTILENAE": {"N": ["17"]},
    "NABLIA": {"-": ["16"]},
    "NAEVIVS": {"A": ["11"], "N": ["12"]},
    "NAEVVS": {"-": ["12"]},
    "NAM": {"-": ["81"]},
    "NAMQVE": {"-": ["81"]},
    "NANCISCOR": {"-": ["5L"]},
    "NANVS": {"-": ["12"]},
    "NAOS": {"-": ["17"]},
    "NARCISSVS": {"-": ["12"], "N": ["12"]},
    "NARDVM": {"-": ["12"]},
    "NARDVS": {"-": ["12"]},
    "NARIS": {"-": ["13"]},
    "NARRATIO": {"-": ["13"]},
    "NARRO": {"-": ["51"]},
    "NASCENTIA": {"-": ["11"]},
    "NASCOR": {"-": ["5L"]},
    "NASITERNA": {"-": ["11"]},
    "NASSA": {"-": ["11"]},
    "NASVS": {"-": ["12"]},
    "NASVTE": {"-": ["60"]},
    "NASVTVS": {"-": ["21"]},
    "NATA": {"-": ["11"]},
    "NATALES": {"-": ["13"]},
    "NATALICIVS": {"-": ["21"]},
    "NATALIS": {"1": ["13"], "2": ["24"], "N": ["13"]},
    "NATATOR": {"-": ["13"]},
    "NATI": {"-": ["12"]},
    "NATINOR": {"-": ["5J"]},
    "NATIO": {"1": ["13"], "2": ["13"]},
    "NATIS": {"-": ["13"]},
    "NATIVVS": {"-": ["21"]},
    "NATO": {"-": ["51"]},
    "NATRIX": {"-": ["13"]},
    "NATV": {"-": ["14"]},
    "NATVRA": {"-": ["11"], "N": ["11"]},
    "NATVRALIA": {"-": ["13"]},
    "NATVRALIS": {"-": ["24"]},
    "NATVRALITER": {"-": ["60"]},
    "NATVS": {"1": ["12"], "2": ["21"]},
    "NAVALE": {"-": ["13"]},
    "NAVALIA": {"-": ["13"]},
    "NAVALIS": {"-": ["24"], "A": ["24"]},
    "NAVARCHVS": {"-": ["12"]},
    "NAVE": {"-": ["60"]},
    "NAVFRAGIVM": {"-": ["12"]},
    "NAVFRAGVS": {"1": ["12"], "2": ["21"]},
    "NAVICVLA": {"-": ["11"]},
    "NAVICVLARIA": {"-": ["11"]},
    "NAVICVLARIVS": {"1": ["12"], "2": ["21"]},
    "NAVIFRAGVS": {"-": ["21"]},
    "NAVIGABILIS": {"-": ["24"]},
    "NAVIGANS": {"-": ["13"]},
    "NAVIGATIO": {"-": ["13"]},
    "NAVIGER": {"-": ["21"]},
    "NAVIGIOLVM": {"-": ["12"]},
    "NAVIGIVM": {"-": ["12"]},
    "NAVIGO": {"-": ["51"]},
    "NAVIS": {"-": ["13"]},
    "NAVITA": {"-": ["11"]},
    "NAVITER": {"-": ["60"]},
    "NAVIVS": {"N": ["12"]},
    "NAVLVM": {"-": ["12"]},
    "NAVMACHIA": {"-": ["11"]},
    "NAVO": {"-": ["51"]},
    "NAVPACTVS": {"N": ["17", "12"]},
    "NAVSEA": {"-": ["11"]},
    "NAVSEABVNDVS": {"-": ["21"]},
    "NAVSEATOR": {"-": ["13"]},
    "NAVSEO": {"-": ["51"]},
    "NAVTA": {"-": ["11"]},
    "NAVTICI": {"-": ["12"]},
    "NAVTICVS": {"-": ["21"]},
    "NAVTIVS": {"N": ["12"]},
    "NAVVS": {"-": ["21"]},
    "NE": {"1": ["60"], "2": ["67"], "3": ["68"], "4": ["82"]},
    "NEAPOLIS": {"N": ["17"]},
    "NEBVLA": {"-": ["11"]},
    "NEBVLO": {"-": ["13"]},
    "NEBVLOSVS": {"-": ["21"]},
    "NEC": {"1": ["68"], "2": ["81"]},
    "NECDVM": {"1": ["68"], "2": ["81"]},
    "NECESSARIA": {"-": ["11"]},
    "NECESSARIO": {"-": ["60"]},
    "NECESSARIVS": {"1": ["12"], "2": ["21"]},
    "NECESSE": {"-": ["26"]},
    "NECESSIS": {"-": ["13"]},
    "NECESSITAS": {"-": ["13"], "N": ["13"]},
    "NECESSITVDO": {"-": ["13"]},
    "NECNE": {"-": ["69", "6-"]},
    "NECNON": {"-": ["81"]},
    "NECO": {"-": ["51"]},
    "NECOPINANS": {"-": ["25"]},
    "NECOPINATO": {"-": ["60"]},
    "NECOPINATVS": {"-": ["21"]},
    "NECTAR": {"-": ["13"]},
    "NECTO": {"-": ["53"]},
    "NECVBI": {"-": ["82"]},
    "NEDVM": {"1": ["60"], "2": ["82"]},
    "NEFANDVS": {"-": ["21"]},
    "NEFARIE": {"-": ["60"]},
    "NEFARIVM": {"-": ["12"]},
    "NEFARIVS": {"-": ["21"]},
    "NEFAS": {"-": ["16"]},
    "NEFASTVM": {"-": ["12"]},
    "NEFASTVS": {"-": ["21"]},
    "NEGITO": {"-": ["51"]},
    "NEGLECTVS": {"1": ["14"], "2": ["21"]},
    "NEGLIGENS": {"-": ["25"]},
    "NEGLIGENTER": {"-": ["60"]},
    "NEGLIGENTIA": {"-": ["11"]},
    "NEGLIGO": {"-": ["53"]},
    "NEGO": {"-": ["51"]},
    "NEGOTIALIS": {"-": ["24"]},
    "NEGOTIATIO": {"-": ["13"]},
    "NEGOTIATOR": {"-": ["13"]},
    "NEGOTIOR": {"-": ["5J"]},
    "NEGOTIOSVS": {"-": ["21"]},
    "NEGOTIVM": {"-": ["12"]},
    "NEMO": {"-": ["48"]},
    "NEMORALIS": {"-": ["24"]},
    "NEMORIVAGVS": {"-": ["21"]},
    "NEMOROSVS": {"-": ["21"]},
    "N": {"-": ["60"]},
    "NEMVS": {"-": ["13"]},
    "NENIA": {"-": ["11"]},
    "NENVM": {"-": ["68"]},
    "NEO": {"-": ["52"]},
    "NEPOS": {"1": ["13"], "2": ["13"], "N": ["12"]},
    "NEPOTOR": {"-": ["5J"]},
    "NEPTIS": {"-": ["13"]},
    "NEPTVNVS": {"N": ["12"]},
    "NEQVAM": {"1": ["16"], "2": ["26"]},
    "NEQVAQVAM": {"-": ["68"]},
    "NEQVE": {"-": ["81"]},
    "NEQVEO": {"-": ["56"]},
    "NEQVIQVAM": {"-": ["60"]},
    "NEQVIS": {"-": ["48"]},
    "NEQVITER": {"-": ["60"]},
    "NEQVITIA": {"-": ["11"]},
    "NEQVITIES": {"-": ["15"]},
    "NEQVO": {"-": ["60"]},
    "NERO": {"N": ["13"]},
    "NERVICVS": {"-": ["21"], "A": ["21"]},
    "NERVOSVS": {"-": ["21"]},
    "NERVVLVS": {"-": ["12"]},
    "NERVVS": {"-": ["12"]},
    "NESCIO": {"-": ["54"]},
    "NESCIOQVA": {"-": ["60"]},
    "NESCIOQVANTVS": {"-": ["60"]},
    "NESCIOQVIS": {"-": ["48"]},
    "NESCIOQVOT": {"-": ["48"]},
    "NESCIVS": {"-": ["21"]},
    "NESTOR": {"-": ["17"], "N": ["13"]},
    "NETE": {"-": ["17"]},
    "NETINI": {"N": ["12"]},
    "NETVM": {"N": ["12"]},
    "NEVE": {"1": ["81"], "2": ["82"]},
    "NEVTER": {"-": ["48"]},
    "NEX": {"-": ["13"]},
    "NEXILIS": {"-": ["24"]},
    "NEXVS": {"-": ["14"]},
    "NI": {"1": ["68"], "2": ["82"]},
    "NICASIO": {"N": ["17"]},
    "NICE": {"N": ["17"]},
    "NICETERIVM": {"-": ["12"]},
    "NICO": {"N": ["13"]},
    "NICOMEDES": {"N": ["17"]},
    "NICTO": {"-": ["51"]},
    "NIDEO": {"-": ["52"]},
    "NIDESCO": {"-": ["53"]},
    "NIDIFICVS": {"-": ["21"]},
    "NIDOR": {"-": ["13"]},
    "NIDVS": {"-": ["12"]},
    "NIGER": {"-": ["21"], "N": ["12"]},
    "NIGRANS": {"-": ["25"]},
    "NIGRESCO": {"-": ["53"]},
    "NIGRO": {"-": ["51"]},
    "NIGROR": {"-": ["13"]},
    "NIHIL": {"-": ["48", "16"]},
    "NIHILDVM": {"-": ["60"]},
    "NIHILOMINVS": {"-": ["68"]},
    "NIHILOSETIVS": {"-": ["68"]},
    "NIHILVM": {"-": ["12"]},
    "NILVS": {"N": ["12"]},
    "NIMBOSVS": {"-": ["21"]},
    "NIMBVS": {"-": ["12"]},
    "NIMIO": {"-": ["60"]},
    "NIMIOPERE": {"-": ["60"]},
    "NIMIRVM": {"-": ["60"]},
    "NIMIS": {"-": ["60"]},
    "NIMIVM": {"1": ["12"], "2": ["60"]},
    "NIMIVS": {"-": ["21"]},
    "NINGO": {"-": ["53"]},
    "NINGVIS": {"-": ["13"]},
    "NISI": {"-": ["82"]},
    "NISVS": {"-": ["14"], "N": ["12"]},
    "NITENS": {"-": ["25"]},
    "NITEO": {"-": ["52"]},
    "NITESCO": {"-": ["53"]},
    "NITIDVS": {"-": ["21"]},
    "NITOR": {"1": ["13"], "2": ["5L"]},
    "NITROSVS": {"-": ["21"]},
    "NITRVM": {"-": ["12"]},
    "NIVALIS": {"-": ["24"]},
    "NIVENDIALIS": {"-": ["24"]},
    "NIVEVS": {"-": ["21"]},
    "NIVOSVS": {"-": ["21"]},
    "NIX": {"-": ["13"]},
    "NIXOR": {"-": ["5J"]},
    "NO": {"-": ["51"]},
    "NOBILIS": {"1": ["13"], "2": ["24"]},
    "NOBILITAS": {"-": ["13"]},
    "NOBILITER": {"-": ["60"]},
    "NOBILITO": {"-": ["51"]},
    "NOCENS": {"-": ["25"]},
    "NOCEO": {"-": ["52"]},
    "NOCTE": {"-": ["60"]},
    "NOCTIVAGVS": {"-": ["21"]},
    "NOCTV": {"-": ["60"]},
    "NOCTVA": {"-": ["11"]},
    "NOCTVRNVS": {"-": ["21"], "N": ["12"], "A": ["21"]},
    "NODATIO": {"-": ["13"]},
    "NODO": {"-": ["51"]},
    "NODOSVS": {"-": ["21"]},
    "NODVS": {"-": ["12"]},
    "NOLO": {"-": ["56"]},
    "NOMEN": {"-": ["13"]},
    "NOMENCLATIO": {"-": ["13"]},
    "NOMENCLATOR": {"-": ["13"]},
    "NOMENCLATVRA": {"-": ["11"]},
    "NOMENTVM": {"N": ["12"]},
    "NOMINATIM": {"-": ["60"]},
    "NOMINATIO": {"-": ["13"]},
    "NOMINATVS": {"1": ["14"], "2": ["21"], "N": ["12"]},
    "NOMINITO": {"-": ["51"]},
    "NOMINO": {"-": ["51"]},
    "NOMISMA": {"-": ["17"]},
    "NON": {"-": ["68"]},
    "NONAE": {"-": ["11"]},
    "NONAGENI": {"-": ["33"]},
    "NONAGESIMVS": {"-": ["32"]},
    "NONAGIES": {"-": ["36"]},
    "NONAGINTA": {"-": ["31"]},
    "NONANVS": {"-": ["21"]},
    "NONARIVS": {"-": ["21"]},
    "NONDVM": {"-": ["68"]},
    "NONGENTI": {"-": ["31"]},
    "NONNE": {"-": ["69", "6-"]},
    "NONNVLLVS": {"-": ["48"]},
    "NONNVMQVAM": {"-": ["60"]},
    "NONVS": {"-": ["32"]},
    "NORBA": {"N": ["11"]},
    "NORBANVS": {"N": ["12"]},
    "NORMA": {"-": ["11"]},
    "NOS": {"-": ["41"]},
    "NOSCITO": {"-": ["51"]},
    "NOSCO": {"-": ["53"]},
    "NOSTER": {"-": ["42"]},
    "NOSTRAS": {"-": ["25"]},
    "NOSTRI": {"-": ["42", "12"]},
    "NOTA": {"-": ["11"]},
    "NOTABILIS": {"-": ["24"]},
    "NOTATIO": {"-": ["13"]},
    "NOTESCO": {"-": ["53"]},
    "NOTHVS": {"-": ["21"], "N": ["12"]},
    "NOTI": {"-": ["12"], "N": ["12"]},
    "NOTIO": {"-": ["13"]},
    "NOTITIA": {"-": ["11"]},
    "NOTITIES": {"-": ["15"]},
    "NOTO": {"-": ["51"]},
    "NOTOR": {"-": ["13"]},
    "NOTVS": {"1": ["17"], "2": ["21"], "N": ["12"]},
    "NOVALE": {"-": ["13"]},
    "NOVALIS": {"-": ["24"]},
    "NOVATILLA": {"N": ["11"]},
    "NOVE": {"-": ["60"]},
    "NOVELLA": {"-": ["11"]},
    "NOVELLVS": {"-": ["21"], "N": ["12"]},
    "NOVEM": {"-": ["31"], "N": ["16"]},
    "NOVEMBER": {"A": ["23"], "N": ["13"]},
    "NOVENDIALIS": {"-": ["24"]},
    "NOVENVS": {"-": ["33"]},
    "NOVERCA": {"-": ["11"]},
    "NOVERCALIS": {"-": ["24"]},
    "NOVICIVS": {"-": ["21"]},
    "NOVIES": {"-": ["36"]},
    "NOVITAS": {"-": ["13"]},
    "NOVO": {"-": ["51"]},
    "NOVVMCOMVM": {"-": ["21"], "N": ["12"]},
    "NOVVS": {"-": ["21"], "A": ["21"]},
    "NOX": {"-": ["13"], "N": ["16"]},
    "NOXA": {"-": ["11"]},
    "NOXIA": {"-": ["11"]},
    "NOXIOSVS": {"-": ["21"]},
    "NOXIVS": {"-": ["21"]},
    "NVBES": {"-": ["13"]},
    "NVBIFER": {"-": ["21"]},
    "NVBILA": {"-": ["12"]},
    "NVBILIS": {"-": ["24"]},
    "NVBILO": {"-": ["51"]},
    "NVBILVM": {"-": ["12"]},
    "NVBILVS": {"-": ["21"]},
    "NVBO": {"-": ["53"]},
    "NVCEVS": {"-": ["21"]},
    "NVCLEVS": {"-": ["12"]},
    "NVDIVS": {"-": ["16"]},
    "NVDIVSTERTIVS": {"-": ["60"]},
    "NVDO": {"-": ["51"]},
    "NVDVS": {"-": ["21"]},
    "NVGAE": {"-": ["11"]},
    "NVGATOR": {"-": ["13"]},
    "NVGATORIVS": {"-": ["21"]},
    "NVGAX": {"-": ["25"]},
    "NVGOR": {"-": ["5J"]},
    "NVLLATENVS": {"-": ["68"]},
    "NVLLVS": {"-": ["48"]},
    "NVM": {"-": ["67"]},
    "NVMA": {"N": ["11"]},
    "NVMANTIA": {"N": ["11"]},
    "NVMARIVS": {"-": ["21"]},
    "NVMATVS": {"-": ["21"]},
    "NVMEN": {"-": ["13"]},
    "NVMENIVS": {"N": ["12"]},
    "NVMERATIO": {"-": ["13"]},
    "NVMERATVM": {"-": ["12"]},
    "NVMERATVS": {"C": ["21"], "-": ["21"]},
    "NVMERO": {"1": ["51"], "2": ["60"]},
    "NVMEROSVS": {"-": ["21"]},
    "NVMERVS": {"-": ["12"]},
    "NVMICIVS": {"N": ["12"]},
    "NVMICVS": {"N": ["12"]},
    "NVMIDICVS": {"C": ["12"], "A": ["21"], "N": ["12"]},
    "NVMITORIVS": {"N": ["12"]},
    "NVMNE": {"-": ["67"]},
    "NVMQVAM": {"-": ["68"]},
    "NVMQVANDO": {"-": ["67"]},
    "NVMQVI": {"-": ["67"]},
    "NVMQVID": {"-": ["67"]},
    "NVMQVIDNAM": {"-": ["67"]},
    "NVMQVIS": {"-": ["47"]},
    "NVMVLVS": {"-": ["12"]},
    "NVMVS": {"-": ["12"]},
    "NVNC": {"-": ["60"]},
    "NVNCVPATIO": {"-": ["13"]},
    "NVNCVPO": {"-": ["51"]},
    "NVNDINAE": {"-": ["11"]},
    "NVNDINATIO": {"-": ["13"]},
    "NVNDINOR": {"-": ["5J"]},
    "NVNTIA": {"-": ["11"]},
    "NVNTIO": {"-": ["51"]},
    "NVNTIVM": {"-": ["12"]},
    "NVNTIVS": {"1": ["12"], "2": ["21"]},
    "NVO": {"-": ["53"]},
    "NVPER": {"-": ["60"]},
    "NVPTA": {"-": ["11"]},
    "NVPTIAE": {"-": ["11"]},
    "NVPTIALIS": {"-": ["24"]},
    "NVRVS": {"-": ["14"]},
    "NVSQVAM": {"-": ["68"]},
    "NVTATIO": {"-": ["13"]},
    "NVTO": {"-": ["51"]},
    "NVTRICIA": {"1": ["11"], "2": ["12"]},
    "NVTRICIVM": {"-": ["12"]},
    "NVTRICIVS": {"1": ["12"], "2": ["21"]},
    "NVTRICVLA": {"-": ["11"]},
    "NVTRIMENTVM": {"-": ["12"]},
    "NVTRIO": {"-": ["54"]},
    "NVTRIOR": {"-": ["5M"]},
    "NVTRIX": {"-": ["13"]},
    "NVTVS": {"-": ["14"]},
    "NVX": {"-": ["13"]},
    "NYMPHA": {"-": ["11"], "N": ["11"]},
    "NYMPHAEVM": {"N": ["12"]},
    "NYMPHE": {"-": ["17"]},
    "NYMPHO": {"N": ["17"]},
    "NYMPHODORVS": {"N": ["12"]},
    "NYSA": {"-": ["17"], "N": ["11"]},
    "O": {"-": ["90"]},
    "OB": {"-": ["70"]},
    "OBAERATVS": {"1": ["12"], "2": ["21"]},
    "OBAMBVLO": {"-": ["51"]},
    "OBARMO": {"-": ["51"]},
    "OBBA": {"-": ["11"]},
    "OBBRVTESCO": {"-": ["53"]},
    "OBDO": {"-": ["53"]},
    "OBDVCO": {"-": ["53"]},
    "OBDVCTIO": {"-": ["13"]},
    "OBDVRESCO": {"-": ["53"]},
    "OBDVRO": {"-": ["51"]},
    "OBEDIENS": {"-": ["25"]},
    "OBEDIENTER": {"-": ["60"]},
    "OBEDIO": {"-": ["54"]},
    "OBEO": {"1": ["56"]},
    "OBEQVITO": {"-": ["51"]},
    "OBERRO": {"2": ["51"]},
    "OBESVS": {"-": ["21"]},
    "OBEX": {"-": ["13"]},
    "OBHAEREO": {"-": ["52"]},
    "OBIACEO": {"-": ["52"]},
    "OBICIO": {"-": ["55"]},
    "OBIECTA": {"-": ["12"]},
    "OBIECTATIO": {"-": ["13"]},
    "OBIECTO": {"-": ["51"]},
    "OBIECTVM": {"-": ["12"]},
    "OBIECTVS": {"-": ["14"]},
    "OBIRASCOR": {"-": ["5L"]},
    "OBITER": {"-": ["60"]},
    "OBITVS": {"-": ["14"]},
    "OBIVRGATIO": {"-": ["13"]},
    "OBIVRGATOR": {"-": ["13"]},
    "OBIVRGO": {"-": ["51"]},
    "OBLATRO": {"-": ["51"]},
    "OBLECTAMENTVM": {"-": ["12"]},
    "OBLECTATIO": {"-": ["13"]},
    "OBLECTO": {"-": ["51"]},
    "OBLENIO": {"-": ["54"]},
    "OBLIDO": {"-": ["53"]},
    "OBLIGATVS": {"-": ["21"]},
    "OBLIGO": {"2": ["51"]},
    "OBLIMO": {"-": ["51"]},
    "OBLINO": {"-": ["53"]},
    "OBLIQVE": {"-": ["60"]},
    "OBLIQVO": {"1": ["51"], "2": ["60"]},
    "OBLIQVVM": {"1": ["12"], "2": ["60"]},
    "OBLIQVVS": {"-": ["21"]},
    "OBLITESCO": {"-": ["53"]},
    "OBLITTERO": {"-": ["51"]},
    "OBLIVIO": {"-": ["13"], "N": ["13"]},
    "OBLIVIOSVS": {"-": ["21"]},
    "OBLIVISCOR": {"-": ["5L"]},
    "OBLIVIVM": {"-": ["12"]},
    "OBLONGVS": {"-": ["21"]},
    "OBLOQVOR": {"-": ["5L"]},
    "OBLVCTOR": {"-": ["5J"]},
    "OBMOLIOR": {"-": ["5M"]},
    "OBMOVEO": {"-": ["52"]},
    "OBMVRMVRO": {"-": ["51"]},
    "OBMVTESCO": {"-": ["53"]},
    "OBNITOR": {"2": ["5L"]},
    "OBNIXVM": {"-": ["60"]},
    "OBNIXVS": {"-": ["21"]},
    "OBNOXIVS": {"-": ["21"]},
    "OBNVBO": {"-": ["53"]},
    "OBOLVS": {"-": ["12"]},
    "OBORIOR": {"-": ["5M"]},
    "OBREPO": {"-": ["53"]},
    "OBRETIO": {"-": ["54"]},
    "OBRIGESCO": {"-": ["53"]},
    "OBRVO": {"-": ["53"]},
    "OBRVSSA": {"-": ["11"]},
    "OBSAEPIO": {"-": ["54"]},
    "OBSCENA": {"-": ["12"]},
    "OBSCENE": {"-": ["6&"]},
    "OBSCENVM": {"-": ["12"]},
    "OBSCENVS": {"-": ["21"]},
    "OBSCVRATIO": {"-": ["13"]},
    "OBSCVRE": {"-": ["60"]},
    "OBSCVRITAS": {"-": ["13"]},
    "OBSCVRO": {"-": ["51"]},
    "OBSCVRVM": {"1": ["12"], "2": ["60"]},
    "OBSCVRVS": {"-": ["21"]},
    "OBSECRO": {"-": ["51"]},
    "OBSECVNDO": {"-": ["51"]},
    "OBSEDEO": {"-": ["52"]},
    "OBSEQVENS": {"O": ["25"], "A": ["22"], "-": ["21"]},
    "OBSEQVENTIA": {"-": ["11"]},
    "OBSEQVIVM": {"-": ["12"]},
    "OBSEQVOR": {"-": ["5L"]},
    "OBSERO": {"1": ["51"], "3": ["53", "51"]},
    "OBSERVABILIS": {"-": ["24"]},
    "OBSERVATIO": {"-": ["13"]},
    "OBSERVATOR": {"-": ["13"]},
    "OBSERVO": {"-": ["51"]},
    "OBSES": {"-": ["13"]},
    "OBSESSIO": {"-": ["13"]},
    "OBSESSOR": {"-": ["13"]},
    "OBSIDEO": {"-": ["52"]},
    "OBSIDIO": {"-": ["13"]},
    "OBSIDIVM": {"1": ["12"], "2": ["12"]},
    "OBSIDO": {"-": ["53"]},
    "OBSIGNO": {"-": ["51"]},
    "OBSISTO": {"-": ["53"]},
    "OBSOLEFACIO": {"-": ["55"]},
    "OBSOLEFIO": {"-": ["56"]},
    "OBSOLESCO": {"-": ["53"]},
    "OBSOLETE": {"-": ["60"]},
    "OBSOLETVS": {"-": ["21"]},
    "OBSOLIDATVS": {"-": ["21"]},
    "OBSONATOR": {"-": ["13"]},
    "OBSONITO": {"-": ["51"]},
    "OBSONIVM": {"-": ["12"]},
    "OBSONO": {"1": ["51"]},
    "OBSTACVLVM": {"-": ["12"]},
    "OBSTANTIA": {"1": ["11"], "2": ["13"]},
    "OBSTETRIX": {"-": ["13"]},
    "OBSTINATE": {"-": ["60"]},
    "OBSTINATIO": {"-": ["13"]},
    "OBSTINATVS": {"-": ["21"]},
    "OBSTIPVS": {"-": ["21"]},
    "OBSTO": {"-": ["51"]},
    "OBSTREPO": {"-": ["53"]},
    "OBSTRIGILLO": {"-": ["51"]},
    "OBSTRINGO": {"-": ["53"]},
    "OBSTRVO": {"-": ["53"]},
    "OBSTVPEFACIO": {"-": ["55"]},
    "OBSTVPEFIO": {"-": ["56"]},
    "OBSTVPESCO": {"-": ["53"]},
    "OBSVM": {"1": ["56"]},
    "OBSVRDESCO": {"-": ["53"]},
    "OBTEGO": {"-": ["53"]},
    "OBTEMPERO": {"-": ["51"]},
    "OBTENDO": {"-": ["53"]},
    "OBTENTVS": {"-": ["14"]},
    "OBTERO": {"-": ["53"]},
    "OBTESTATIO": {"-": ["13"]},
    "OBTESTOR": {"-": ["5J"]},
    "OBTINEO": {"-": ["52"]},
    "OBTINGO": {"-": ["53"]},
    "OBTINGVO": {"-": ["53"]},
    "OBTORPESCO": {"-": ["53"]},
    "OBTORQVEO": {"-": ["52"]},
    "OBTRECTATIO": {"-": ["13"]},
    "OBTRECTATOR": {"-": ["13"]},
    "OBTRECTO": {"-": ["51"]},
    "OBTRVDO": {"-": ["53"]},
    "OBTRVNCO": {"-": ["51"]},
    "OBTVNDO": {"-": ["53"]},
    "OBTVRBO": {"2": ["51"]},
    "OBTVRGESCO": {"-": ["53"]},
    "OBTVRO": {"-": ["51"]},
    "OBTVSVS": {"-": ["21"]},
    "OBTVTVS": {"-": ["14"]},
    "OBVCVLA": {"-": ["11"], "N": ["11"]},
    "OBVENIO": {"-": ["54"]},
    "OBVERSO": {"-": ["5J"]},
    "OBVERSOR": {"-": ["5J"]},
    "OBVERSVS": {"-": ["21"]},
    "OBVERTO": {"-": ["53"]},
    "OBVIAM": {"-": ["60"]},
    "OBVIVS": {"-": ["21"]},
    "OBVMBRO": {"-": ["51"]},
    "OBVNCVS": {"-": ["21"]},
    "OBVOLVO": {"-": ["53"]},
    "OCCAECO": {"-": ["51"]},
    "OCCALLESCO": {"-": ["53"]},
    "OCCANO": {"-": ["53"]},
    "OCCASIO": {"-": ["13"]},
    "OCCASVS": {"-": ["14"]},
    "OCCEDO": {"1": ["53"]},
    "OCCIDENS": {"N": ["13"], "-": ["13"]},
    "OCCIDIO": {"-": ["13"]},
    "OCCIDO": {"1": ["53"], "2": ["53"]},
    "OCCIDVVS": {"-": ["21"]},
    "OCCIPIO": {"-": ["55"]},
    "OCCIPITIVM": {"-": ["12"]},
    "OCCIPVT": {"-": ["13"]},
    "OCCISIO": {"-": ["13"]},
    "OCCISVS": {"-": ["21"]},
    "OCCLVDO": {"-": ["53"]},
    "OCCO": {"-": ["51"]},
    "OCCVBO": {"-": ["51"]},
    "OCCVLCO": {"-": ["51"]},
    "OCCVLO": {"-": ["53"]},
    "OCCVLTATIO": {"-": ["13"]},
    "OCCVLTE": {"-": ["60"]},
    "OCCVLTO": {"-": ["51"]},
    "OCCVLTVM": {"-": ["12"]},
    "OCCVLTVS": {"-": ["21"]},
    "OCCVMBO": {"-": ["53"]},
    "OCCVPATIO": {"-": ["13"]},
    "OCCVPATVS": {"-": ["21"]},
    "OCCVPO": {"1": ["13"], "2": ["51"], "N": ["13"]},
    "OCCVRO": {"-": ["51"]},
    "OCCVRRO": {"-": ["53"]},
    "OCCVRSO": {"-": ["51"]},
    "OCCVRSVS": {"-": ["14"]},
    "OCEANVS": {"1": ["12"], "2": ["21"], "N": ["12"], "A": ["21"]},
    "OCELLVS": {"-": ["12"]},
    "OCHRA": {"-": ["17"]},
    "OCIMVM": {"-": ["12"]},
    "OCINVM": {"-": ["12"]},
    "OCIOR": {"-": ["2&"]},
    "OCITER": {"-": ["60"]},
    "OCLIFERIVS": {"-": ["21"]},
    "OCREA": {"-": ["11"], "N": ["11"]},
    "OCREATVS": {"-": ["21"]},
    "OCTACHORDOS": {"-": ["27"]},
    "OCTANS": {"-": ["13"]},
    "OCTASTYLOS": {"-": ["27"]},
    "OCTAVAE": {"-": ["11"]},
    "OCTAVIA": {"N": ["11"]},
    "OCTAVIANVS": {"C": ["12"], "N": ["21"], "A": ["21"]},
    "OCTAVIVS": {"N": ["12"]},
    "OCTAVVS": {"-": ["32"], "N": ["12"]},
    "OCTINGENTESIMVS": {"-": ["32"]},
    "OCTINGENTI": {"-": ["31"]},
    "OCTO": {"-": ["31"]},
    "OCTOBER": {"A": ["23"], "N": ["13"]},
    "OCTOGENARIVS": {"-": ["21"]},
    "OCTOGESIMVS": {"-": ["32"]},
    "OCTOGINTA": {"-": ["31"]},
    "OCTOGONOS": {"-": ["27"]},
    "OCTONI": {"-": ["33"]},
    "OCTOPHORON": {"-": ["17"]},
    "OCTVPLVM": {"-": ["12"]},
    "OCTVSSIS": {"-": ["13"]},
    "OCVLVS": {"-": ["12"]},
    "ODEVM": {"-": ["12"]},
    "ODI": {"-": ["53"]},
    "ODIOSVS": {"-": ["21"]},
    "ODIVM": {"-": ["12"]},
    "ODOR": {"-": ["13"]},
    "ODORATVS": {"1": ["14"], "2": ["21"]},
    "ODORIFER": {"-": ["21"]},
    "ODORO": {"-": ["51"]},
    "ODOROR": {"-": ["5J"]},
    "ODORVS": {"-": ["21"]},
    "ODYSSEA": {"N": ["11"]},
    "OEAX": {"-": ["17"]},
    "OECONOMIA": {"-": ["17"]},
    "OECVS": {"-": ["17"]},
    "OENOPHORVM": {"-": ["12"]},
    "OESTRVS": {"-": ["17"]},
    "OESYPVM": {"-": ["12"]},
    "OETA": {"N": ["17"]},
    "OFELLA": {"-": ["11"]},
    "OFFA": {"-": ["11"]},
    "OFFENDO": {"-": ["53"]},
    "OFFENSA": {"-": ["11"]},
    "OFFENSATIO": {"-": ["13"]},
    "OFFENSIO": {"-": ["13"]},
    "OFFENSO": {"-": ["51"]},
    "OFFENSVM": {"-": ["12"]},
    "OFFENSVS": {"1": ["14"], "2": ["21"]},
    "OFFERO": {"-": ["56"]},
    "OFFICINA": {"-": ["11"]},
    "OFFICINATOR": {"-": ["13"]},
    "OFFICIO": {"-": ["55"]},
    "OFFICIOSE": {"-": ["60"]},
    "OFFICIOSVS": {"1": ["12"], "2": ["21"]},
    "OFFICIVM": {"-": ["12"]},
    "OFFIGO": {"-": ["53"]},
    "OFFIRMO": {"-": ["51"]},
    "OFFOCO": {"-": ["51"]},
    "OFFVNDO": {"2": ["53"]},
    "OH": {"-": ["90"]},
    "OHE": {"-": ["90"]},
    "OLEA": {"-": ["11"]},
    "OLEAGINEVS": {"-": ["21"]},
    "OLEAGINVS": {"-": ["21"]},
    "OLEARIVS": {"1": ["12"], "2": ["21"]},
    "OLEASTER": {"-": ["12"]},
    "OLEITAS": {"-": ["13"]},
    "OLENS": {"-": ["25"]},
    "OLEO": {"-": ["52"]},
    "OLESCO": {"-": ["53"]},
    "OLETVM": {"1": ["12"], "2": ["12"]},
    "OLEVM": {"-": ["12"]},
    "OLFACIO": {"-": ["55"]},
    "OLFACTVS": {"-": ["14"]},
    "OLIDVS": {"-": ["21"]},
    "OLIM": {"-": ["60"]},
    "OLITIO": {"-": ["13"]},
    "OLITOR": {"-": ["13"]},
    "OLITORIVS": {"-": ["21"]},
    "OLIVA": {"-": ["11"]},
    "OLIVETVM": {"-": ["12"]},
    "OLIVVM": {"-": ["12"]},
    "OLOR": {"1": ["13"], "2": ["13"]},
    "OLORINVS": {"-": ["21"]},
    "OLVS": {"-": ["13"]},
    "OLVSCVLVM": {"-": ["12"]},
    "OLYMPICVS": {"-": ["21"], "A": ["21"]},
    "OLYMPIVS": {"A": ["21"]},
    "OLYMPVS": {"1N": ["12"], "2N": ["12"], "3N": ["12"], "N": ["12"]},
    "OMASVM": {"-": ["12"]},
    "OMEN": {"-": ["13"]},
    "OMENTVM": {"-": ["12"]},
    "OMINOR": {"-": ["5J"]},
    "OMITTO": {"-": ["53"]},
    "OMNIGENVS": {"1": ["21"], "2": ["60"]},
    "OMNIMODIS": {"-": ["60"]},
    "OMNIMODVS": {"-": ["21"]},
    "OMNINO": {"-": ["60"]},
    "OMNIPARENS": {"-": ["25"]},
    "OMNIPATENS": {"-": ["25"]},
    "OMNIPOTENS": {"-": ["25"], "A": ["22"]},
    "OMNIS": {"-": ["48"]},
    "OMNITVENS": {"-": ["25"]},
    "OMNIVOLVS": {"-": ["21"]},
    "ONAGER": {"-": ["12"]},
    "ONASVS": {"N": ["12"]},
    "ONERARIA": {"-": ["11"]},
    "ONERARIVS": {"-": ["21"]},
    "ONERATVS": {"-": ["21"]},
    "ONERO": {"-": ["51"]},
    "ONEROSVS": {"-": ["21"]},
    "ONVS": {"-": ["13"]},
    "ONVSTVS": {"-": ["21"]},
    "ONYX": {"-": ["17"]},
    "OPACITAS": {"-": ["13"]},
    "OPACO": {"-": ["51"]},
    "OPACVM": {"-": ["12"]},
    "OPACVS": {"-": ["21"]},
    "OPE": {"-": ["17"]},
    "OPELLA": {"-": ["11"]},
    "OPERA": {"-": ["11"]},
    "OPERANS": {"-": ["25"]},
    "OPERARIVS": {"1": ["12"], "2": ["12"]},
    "OPERATIO": {"-": ["13"]},
    "OPERATVS": {"-": ["21"]},
    "OPERCVLVM": {"-": ["12"]},
    "OPERIMENTVM": {"-": ["12"]},
    "OPERIO": {"-": ["54"]},
    "OPEROR": {"-": ["5J"]},
    "OPEROSE": {"-": ["60"]},
    "OPEROSVS": {"-": ["21"]},
    "OPERTORIVM": {"-": ["12"]},
    "OPERTVM": {"-": ["12"]},
    "OPERTVS": {"1": ["14"], "2": ["21"]},
    "OPICVS": {"-": ["21"]},
    "OPIFEX": {"-": ["13"]},
    "OPILIO": {"-": ["13"]},
    "OPIMIVS": {"N": ["12"]},
    "OPIMVS": {"-": ["21"]},
    "OPINATVS": {"-": ["14"]},
    "OPINIO": {"-": ["13"]},
    "OPINOR": {"-": ["5J"]},
    "OPITER": {"N": ["13"]},
    "OPITVLOR": {"-": ["5J"]},
    "OPOBALSAMVM": {"-": ["12"]},
    "OPORTET": {"-": ["52"]},
    "OPPEDO": {"-": ["53"]},
    "OPPERIOR": {"-": ["5M"]},
    "OPPETO": {"-": ["53"]},
    "OPPIA": {"N": ["11"]},
    "OPPICO": {"-": ["51"]},
    "OPPIDANI": {"-": ["12"]},
    "OPPIDANVS": {"1": ["12"], "2": ["21"]},
    "OPPIDO": {"-": ["60"]},
    "OPPIDVLVM": {"-": ["12"]},
    "OPPIDVM": {"-": ["12"]},
    "OPPIGNERO": {"-": ["51"]},
    "OPPILO": {"-": ["51"]},
    "OPPLEO": {"-": ["52"]},
    "OPPONO": {"-": ["53"]},
    "OPPORTVNE": {"-": ["60"]},
    "OPPORTVNITAS": {"-": ["13"]},
    "OPPORTVNVS": {"-": ["21"]},
    "OPPOSITVS": {"1": ["14"], "2": ["21"]},
    "OPPRESSIO": {"-": ["13"]},
    "OPPRESSVS": {"-": ["14"]},
    "OPPRIMO": {"-": ["53"]},
    "OPPROBRIVM": {"-": ["12"]},
    "OPPVGNATIO": {"-": ["13"]},
    "OPPVGNATOR": {"-": ["13"]},
    "OPPVGNATORIVS": {"-": ["21"]},
    "OPPVGNO": {"-": ["51"]},
    "OPS": {"-": ["13"], "N": ["13"]},
    "OPTABILIS": {"-": ["24"]},
    "OPTATVM": {"-": ["12"]},
    "OPTATVS": {"-": ["21"]},
    "OPTICE": {"-": ["17"]},
    "OPTICOS": {"-": ["27"]},
    "OPTICVS": {"-": ["21"]},
    "OPTIMAS": {"-": ["25"]},
    "OPTIMATES": {"-": ["13"]},
    "OPTIO": {"1": ["13"], "2": ["13"]},
    "OPTIONATVS": {"-": ["14"]},
    "OPTIVVS": {"-": ["21"]},
    "OPTO": {"-": ["51"]},
    "OPVLENS": {"-": ["25"]},
    "OPVLENTER": {"-": ["60"]},
    "OPVLENTIA": {"-": ["11"]},
    "OPVLENTO": {"-": ["51"]},
    "OPVLENTVS": {"-": ["21"]},
    "OPVNTIVS": {"N": ["21"], "A": ["21"]},
    "OPVS": {"1": ["13"], "2": ["16"]},
    "OPVSCVLVM": {"-": ["12"]},
    "ORA": {"-": ["11"], "N": ["11"]},
    "ORACVLVM": {"-": ["12"]},
    "ORATIO": {"-": ["13"]},
    "ORATIVNCVLA": {"-": ["11"]},
    "ORATOR": {"-": ["13"]},
    "ORATORIE": {"-": ["60"]},
    "ORATORIVS": {"-": ["21"]},
    "ORATVS": {"-": ["14"]},
    "ORBATIO": {"-": ["13"]},
    "ORBICVLVS": {"-": ["12"]},
    "ORBIS": {"-": ["13"]},
    "ORBITA": {"-": ["11"]},
    "ORBITAS": {"-": ["13"]},
    "ORBO": {"-": ["51"]},
    "ORBVS": {"-": ["21"]},
    "ORCA": {"-": ["11"]},
    "ORCHESTRA": {"-": ["11"]},
    "ORCHITIS": {"-": ["13"]},
    "ORCHOMENOS": {"N": ["17"]},
    "ORCVLA": {"-": ["11"]},
    "ORCVS": {"N": ["12"]},
    "ORDINARIVS": {"-": ["21"]},
    "ORDINATE": {"-": ["60"]},
    "ORDINATIM": {"-": ["60"]},
    "ORDINATIO": {"-": ["13"]},
    "ORDINATOR": {"-": ["13"]},
    "ORDINATVS": {"-": ["21"]},
    "ORDINO": {"-": ["51"]},
    "ORDIOR": {"-": ["5M"]},
    "ORDO": {"-": ["13"]},
    "OREAE": {"-": ["11"]},
    "OREXIS": {"-": ["13"]},
    "ORGANICOS": {"-": ["60"]},
    "ORGANICVS": {"1": ["12"], "2": ["21"]},
    "ORGANVM": {"-": ["12"]},
    "ORGIA": {"-": ["12"]},
    "ORIENS": {"N": ["13"], "-": ["22"]},
    "ORIFICIVM": {"-": ["12"]},
    "ORIGANITVS": {"-": ["21"]},
    "ORIGO": {"-": ["13"], "N": ["13"]},
    "ORIOR": {"-": ["5M"]},
    "ORIVNDVS": {"-": ["21"]},
    "ORNAMENTVM": {"-": ["12"]},
    "ORNATE": {"-": ["60"]},
    "ORNATIO": {"-": ["13"]},
    "ORNATRIX": {"-": ["13"]},
    "ORNATVS": {"1": ["14"], "2": ["21"]},
    "ORNO": {"-": ["51"]},
    "ORNVS": {"-": ["12"]},
    "ORO": {"-": ["51"]},
    "ORPHEVS": {"N": ["17"], "A": ["21"]},
    "ORSA": {"-": ["12"]},
    "ORTHOGONIVS": {"-": ["21"]},
    "ORTHOGRAPHIA": {"-": ["11"]},
    "ORTHOS": {"-": ["27"]},
    "ORTHOSTATA": {"-": ["11"]},
    "ORTONA": {"N": ["11"]},
    "ORTVS": {"-": ["14"]},
    "ORYX": {"-": ["13"]},
    "ORYZA": {"-": ["11"]},
    "OS": {"1": ["13"], "2": ["13"]},
    "OSCEN": {"-": ["13"]},
    "OSCILLVM": {"1": ["12"], "2": ["12"]},
    "OSCITANS": {"-": ["25"]},
    "OSCITATIO": {"-": ["13"]},
    "OSCITO": {"-": ["51"]},
    "OSCITOR": {"-": ["5J"]},
    "OSCVLATIO": {"-": ["13"]},
    "OSCVLOR": {"-": ["5J"]},
    "OSCVLVM": {"-": ["12"]},
    "OSSEVS": {"-": ["21"]},
    "OSSICVLVM": {"-": ["12"]},
    "OSSIFRAGA": {"-": ["11"]},
    "OSSIFRAGVS": {"-": ["21"]},
    "OSTENDO": {"-": ["53"]},
    "OSTENTATIO": {"-": ["13"]},
    "OSTENTATOR": {"-": ["13"]},
    "OSTENTO": {"-": ["51"]},
    "OSTENTVM": {"-": ["12"]},
    "OSTENTVS": {"-": ["14"]},
    "OSTIA": {"N": ["11"]},
    "OSTIARIVM": {"-": ["12"]},
    "OSTIARIVS": {"-": ["12"]},
    "OSTIATIM": {"-": ["60"]},
    "OSTIVM": {"-": ["12"]},
    "OSTREA": {"-": ["11"]},
    "OSTREOSVS": {"-": ["21"]},
    "OSTREVM": {"-": ["12"]},
    "OSTRIFER": {"-": ["21"]},
    "OSTRINVS": {"-": ["21"]},
    "OSTRVM": {"-": ["12"]},
    "OTIOR": {"-": ["5J"]},
    "OTIOSE": {"-": ["60"]},
    "OTIOSVS": {"-": ["21"]},
    "OTIVM": {"-": ["12"]},
    "OVATIO": {"-": ["13"]},
    "OVILE": {"-": ["13"]},
    "OVILLVS": {"-": ["21"]},
    "OVIS": {"-": ["13"]},
    "OVO": {"-": ["51"]},
    "OVVM": {"-": ["12"]},
    "OXYMELI": {"-": ["13"]},
    "PABVLATIO": {"-": ["13"]},
    "PABVLATOR": {"-": ["13"]},
    "PABVLOR": {"-": ["5J"]},
    "PABVLVM": {"-": ["12"]},
    "PACATOR": {"-": ["13"]},
    "PACATVS": {"-": ["21"]},
    "PACHYNVM": {"N": ["17"]},
    "PACIFICO": {"-": ["51"]},
    "PACILIVS": {"N": ["12"]},
    "PACISCOR": {"-": ["5L"]},
    "PACO": {"-": ["51"]},
    "PACTIO": {"-": ["13"]},
    "PACTVM": {"-": ["12"]},
    "PACTVS": {"1": ["14"], "2": ["21"]},
    "PAEAN": {"-": ["17"], "N": ["17"]},
    "PAEDAGOGIVM": {"-": ["12"]},
    "PAEDAGOGVS": {"-": ["12"]},
    "PAEDICO": {"1": ["13"], "2": ["51"]},
    "PAEDOR": {"-": ["13"]},
    "PAENE": {"-": ["60"]},
    "PAENITENTIA": {"-": ["11"]},
    "PAENITEO": {"-": ["52"]},
    "PAENVLA": {"-": ["11"]},
    "PAENVLATVS": {"-": ["21"]},
    "PAETVS": {"-": ["21"], "N": ["12"]},
    "PAGANVS": {"1": ["12"], "2": ["21"]},
    "PAGINA": {"-": ["11"]},
    "PAGVS": {"-": ["12"]},
    "PALA": {"-": ["11"]},
    "PALAESTRA": {"-": ["11"]},
    "PALAESTRICVS": {"-": ["21"]},
    "PALAESTRITA": {"-": ["11"]},
    "PALAM": {"1": ["60"], "2": ["70"]},
    "PALATINVS": {"N": ["21"], "A": ["21"]},
    "PALATIO": {"-": ["13"]},
    "PALATIVM": {"-": ["12"], "N": ["12"]},
    "PALATVM": {"-": ["12"]},
    "PALEA": {"-": ["11"]},
    "PALEAR": {"-": ["13"]},
    "PALICANVS": {"N": ["12"]},
    "PALIMPSESTVS": {"-": ["17"]},
    "PALIVRVS": {"-": ["12"]},
    "PALLA": {"-": ["11"], "N": ["11"]},
    "PALLADIVM": {"-": ["12"], "N": ["12"]},
    "PALLANTEVM": {"N": ["12"]},
    "PALLANTIVM": {"N": ["12"]},
    "PALLAS": {"-": ["17"], "O": ["13"], "N": ["21"]},
    "PALLENS": {"-": ["25"]},
    "PALLEO": {"-": ["52"]},
    "PALLESCO": {"-": ["53"]},
    "PALLIATVS": {"-": ["21"]},
    "PALLIDVLVS": {"-": ["21"]},
    "PALLIDVS": {"-": ["21"]},
    "PALLIOLVM": {"-": ["12"]},
    "PALLIVM": {"-": ["12"]},
    "PALLOR": {"-": ["13"], "N": ["13"]},
    "PALMA": {"-": ["11"]},
    "PALMES": {"-": ["13"]},
    "PALMETVM": {"-": ["12"]},
    "PALMEVS": {"-": ["21"]},
    "PALMIFER": {"-": ["21"]},
    "PALMOSVS": {"-": ["21"]},
    "PALMVLA": {"-": ["11"]},
    "PALMVS": {"-": ["12"], "N": ["12"]},
    "PALOR": {"-": ["5J"]},
    "PALPEBRAE": {"-": ["11"]},
    "PALPITATIO": {"-": ["13"]},
    "PALPITO": {"-": ["51"]},
    "PALPO": {"1": ["13"], "2": ["51"]},
    "PALPOR": {"-": ["5J"]},
    "PALPVM": {"-": ["12"]},
    "PALVDAMENTVM": {"-": ["12"]},
    "PALVDATVS": {"-": ["21"]},
    "PALVMBES": {"-": ["13"]},
    "PALVMBVS": {"-": ["12"]},
    "PALVS": {"1": ["12"], "2": ["13"]},
    "PALVSTER": {"-": ["23"]},
    "PAMPHILVS": {"N": ["12"]},
    "PAMPHYLIA": {"N": ["11"]},
    "PAMPINEVS": {"-": ["21"]},
    "PAMPINO": {"-": ["51"]},
    "PAMPINVS": {"-": ["12"]},
    "PAN": {"N": ["17"]},
    "PANACEA": {"-": ["11"]},
    "PANAX": {"-": ["13"]},
    "PANCHRESTVS": {"-": ["21"]},
    "PANCRATION": {"-": ["17"]},
    "PANDATIO": {"-": ["13"]},
    "PANDO": {"1": ["51"], "2": ["53"]},
    "PANDVS": {"-": ["21"]},
    "PANGO": {"-": ["53"]},
    "PANHORMITANVS": {"N": ["21"], "A": ["21"]},
    "PANHORMVS": {"N": ["12"]},
    "PANICVM": {"-": ["12"]},
    "PANIS": {"-": ["13"]},
    "PANNICVLVS": {"-": ["12"]},
    "PANNOSVS": {"-": ["21"]},
    "PANNVCIVS": {"-": ["21"]},
    "PANNVS": {"-": ["12"]},
    "PANTHERA": {"1": ["11"]},
    "PANTOMIMA": {"-": ["11"]},
    "PANTOMIMICVS": {"-": ["21"]},
    "PANTOMIMVS": {"-": ["12"]},
    "PANVS": {"-": ["12"]},
    "PAPAE": {"-": ["90"]},
    "PAPAS": {"-": ["17"]},
    "PAPAVER": {"-": ["13"]},
    "PAPHLAGONIA": {"N": ["11"]},
    "PAPILLA": {"-": ["11"]},
    "PAPINIVS": {"N": ["12"]},
    "PAPIRIVS": {"N": ["12"], "A": ["21"]},
    "PAPPO": {"-": ["51"]},
    "PAPPVS": {"-": ["12"]},
    "PAPVLA": {"-": ["11"]},
    "PAPYRVS": {"-": ["12"]},
    "PAR": {"1": ["13"], "2": ["25"]},
    "PARABILIS": {"-": ["24"]},
    "PARABOLA": {"-": ["17"]},
    "PARADOXVM": {"-": ["17"]},
    "PARADROMIS": {"-": ["17"]},
    "PARAENETICOS": {"-": ["27"]},
    "PARALVS": {"N": ["12"]},
    "PARALYSIS": {"-": ["17"]},
    "PARAMESE": {"-": ["17"]},
    "PARANETE": {"-": ["17"]},
    "PARAPEGMA": {"-": ["17"]},
    "PARARIVS": {"-": ["12"]},
    "PARASITA": {"-": ["11"]},
    "PARASITVS": {"-": ["12"]},
    "PARASTAS": {"-": ["17"]},
    "PARASTATA": {"-": ["11"]},
    "PARASTATICA": {"-": ["11"]},
    "PARATA": {"-": ["12"]},
    "PARATE": {"-": ["60"]},
    "PARATIO": {"-": ["13"]},
    "PARATVS": {"1": ["14"], "2": ["21"]},
    "PARCA": {"N": ["11"]},
    "PARCE": {"-": ["60"]},
    "PARCENS": {"-": ["25"]},
    "PARCITAS": {"-": ["13"]},
    "PARCO": {"-": ["53"]},
    "PARCVS": {"-": ["21"]},
    "PARDALIS": {"-": ["13"]},
    "PARDVS": {"-": ["12"]},
    "PARENS": {"1": ["13"], "2": ["25", "22"]},
    "PARENTES": {"-": ["13"]},
    "PARENTO": {"-": ["51"]},
    "PAREO": {"-": ["52"]},
    "PARERGON": {"-": ["17"]},
    "PARGO": {"-": ["53"]},
    "PARHYPATE": {"-": ["17"]},
    "PARIES": {"-": ["13"]},
    "PARILIS": {"-": ["24"], "A": ["24"]},
    "PARIO": {"1": ["51"], "2": ["55"]},
    "PARITER": {"-": ["60"]},
    "PARMA": {"-": ["11"], "N": ["11"]},
    "PARMVLA": {"-": ["11"]},
    "PARO": {"1": ["13"], "2": ["51"]},
    "PAROCHVS": {"-": ["12"]},
    "PAROPSIS": {"-": ["13"]},
    "PAROTIS": {"-": ["17"]},
    "PARRA": {"-": ["11"]},
    "PARRICIDA": {"-": ["11"]},
    "PARRICIDIVM": {"-": ["12"]},
    "PARS": {"-": ["13"]},
    "PARSIMONIA": {"-": ["11"]},
    "PARTHENICE": {"-": ["17"]},
    "PARTHICVS": {"N": ["21"], "A": ["21"]},
    "PARTHVS": {"N": ["21"], "A": ["21"]},
    "PARTIARIVS": {"-": ["21"]},
    "PARTICEPS": {"1": ["13"], "2": ["22"]},
    "PARTICIPO": {"-": ["51"]},
    "PARTICVLA": {"-": ["11"]},
    "PARTICVLARIS": {"-": ["24"]},
    "PARTICVLATIM": {"-": ["60"]},
    "PARTIM": {"-": ["60"]},
    "PARTIO": {"1": ["13"], "2": ["54"]},
    "PARTIOR": {"-": ["5M"]},
    "PARTITIO": {"-": ["13"]},
    "PARTVRIO": {"-": ["54"]},
    "PARTVS": {"1": ["14"], "2": ["21"]},
    "PARVE": {"-": ["60"]},
    "PARVICAX": {"-": ["25"]},
    "PARVM": {"1": ["12"], "2": ["60"]},
    "PARVMPER": {"-": ["60"]},
    "PARVVLVS": {"-": ["21"]},
    "PARVVM": {"-": ["12"]},
    "PARVVS": {"1": ["12"], "2": ["21"]},
    "PASCALIS": {"-": ["24"]},
    "PASCEOLVS": {"-": ["12"]},
    "PASCO": {"-": ["53"]},
    "PASCOR": {"-": ["5L"]},
    "PASCVVM": {"-": ["12"]},
    "PASCVVS": {"-": ["21"]},
    "PASSER": {"-": ["13"]},
    "PASSIM": {"-": ["60"]},
    "PASSVM": {"-": ["12"]},
    "PASSVS": {"1": ["14"], "2": ["21"]},
    "PASTAS": {"-": ["17"]},
    "PASTILLVS": {"-": ["12"]},
    "PASTIO": {"-": ["13"]},
    "PASTOR": {"-": ["13"], "N": ["13"]},
    "PASTVS": {"-": ["14"]},
    "PATAVIVM": {"N": ["12"]},
    "PATEFACIO": {"-": ["55"]},
    "PATEFIO": {"-": ["56"]},
    "PATELLA": {"-": ["11"]},
    "PATENS": {"-": ["25"]},
    "PATEO": {"-": ["52"]},
    "PATER": {"-": ["13"], "N": ["12"]},
    "PATERA": {"-": ["11"]},
    "PATERNVS": {"-": ["21"]},
    "PATESCO": {"-": ["53"]},
    "PATHICVS": {"-": ["21"]},
    "PATIBILIS": {"-": ["24"]},
    "PATIBVLVM": {"-": ["12"]},
    "PATIENS": {"-": ["25"]},
    "PATIENTER": {"-": ["60"]},
    "PATIENTIA": {"-": ["11"]},
    "PATINA": {"-": ["11"], "N": ["11"]},
    "PATIOR": {"-": ["5N"]},
    "PATRATOR": {"-": ["13"]},
    "PATRIA": {"-": ["11"], "N": ["11"]},
    "PATRICIVS": {"1": ["12"], "2": ["21"]},
    "PATRIMONIVM": {"-": ["12"]},
    "PATRIVS": {"-": ["21"]},
    "PATRO": {"-": ["51"]},
    "PATROCINIVM": {"-": ["12"]},
    "PATROCINOR": {"-": ["5J"]},
    "PATRONA": {"-": ["11"]},
    "PATRONVS": {"-": ["12"]},
    "PATRVELIS": {"1": ["13"], "2": ["24"]},
    "PATRVVS": {"1": ["12"], "2": ["21"]},
    "PATVLVS": {"-": ["21"]},
    "PAVA": {"-": ["11"]},
    "PAVCA": {"-": ["12"]},
    "PAVCI": {"-": ["12"]},
    "PAVCITAS": {"-": ["13"]},
    "PAVCVLVS": {"-": ["21"]},
    "PAVCVS": {"-": ["21"]},
    "PAVEFACIO": {"-": ["55"]},
    "PAVEO": {"-": ["52"]},
    "PAVESCO": {"-": ["53"]},
    "PAVICVLA": {"-": ["11"]},
    "PAVIDE": {"-": ["60"]},
    "PAVIDVS": {"-": ["21"]},
    "PAVIMENTATVS": {"-": ["21"]},
    "PAVIMENTVM": {"-": ["12"]},
    "PAVIO": {"-": ["54"]},
    "PAVITO": {"-": ["51"]},
    "PAVLATIM": {"-": ["60", "6-"]},
    "PAVLISPER": {"-": ["60"]},
    "PAVLO": {"-": ["60", "6-"]},
    "PAVLVLVM": {"1": ["12"], "2": ["60"]},
    "PAVLVLVS": {"-": ["21"]},
    "PAVLVM": {"1": ["12"], "2": ["60"]},
    "PAVLVS": {"-": ["21"], "N": ["12"]},
    "PAVO": {"-": ["13"]},
    "PAVOR": {"-": ["13"], "N": ["13"]},
    "PAVPER": {"-": ["22"]},
    "PAVPERCVLVS": {"-": ["21"]},
    "PAVPERIES": {"-": ["15"]},
    "PAVPERO": {"-": ["51"]},
    "PAVPERTAS": {"-": ["13"]},
    "PAVSA": {"-": ["11"]},
    "PAVSARIVS": {"-": ["12"]},
    "PAVSEA": {"-": ["11"]},
    "PAVVS": {"-": ["12"]},
    "PAVXILLVM": {"1": ["12"], "2": ["12"]},
    "PAVXILLVS": {"-": ["21"]},
    "PAX": {"-": ["13"], "N": ["13"]},
    "PECCANS": {"-": ["13"]},
    "PECCATVM": {"-": ["12"]},
    "PECCO": {"-": ["51"]},
    "PECTEN": {"-": ["13"]},
    "PECTINATIM": {"-": ["60"]},
    "PECTO": {"-": ["53"]},
    "PECTVS": {"-": ["13"]},
    "PECV": {"-": ["14"]},
    "PECVARIA": {"-": ["12"]},
    "PECVARIVS": {"1": ["12"], "2": ["21"]},
    "PECVLATVS": {"-": ["14"]},
    "PECVLIARIS": {"-": ["24"]},
    "PECVLIVM": {"-": ["12"]},
    "PECVNIA": {"-": ["11"], "N": ["11"]},
    "PECVNIARIVS": {"1": ["12"], "2": ["21"]},
    "PECVNIOSVS": {"-": ["21"]},
    "PECVS": {"1": ["13"], "2": ["13"]},
    "PEDALIS": {"-": ["24"]},
    "PEDARIVS": {"-": ["21"]},
    "PEDATVM": {"-": ["12"]},
    "PEDATVS": {"1": ["14"]},
    "PEDES": {"-": ["13"]},
    "PEDESTER": {"-": ["23"]},
    "PEDETENTIM": {"-": ["60"]},
    "PEDICA": {"-": ["11"]},
    "PEDICINVS": {"-": ["12"]},
    "PEDICVLVS": {"1": ["12"]},
    "PEDIO": {"-": ["54"]},
    "PEDITATVS": {"1": ["14"], "2": ["21"]},
    "PEDITO": {"-": ["51"]},
    "PEDITVM": {"-": ["12"]},
    "PEDO": {"1": ["13"], "2": ["51"], "3": ["53"], "N": ["13"]},
    "PEDVCAEANVS": {"N": ["21"], "A": ["21"]},
    "PEDVCAEVS": {"N": ["12"]},
    "PEDVM": {"-": ["12"], "N": ["12"]},
    "PEGMA": {"-": ["17"]},
    "PEIERO": {"-": ["51"]},
    "PELAGVS": {"-": ["17"]},
    "PELAMIS": {"-": ["13"]},
    "PELASTES": {"-": ["13"]},
    "PELECINVS": {"-": ["12"]},
    "PELLACIA": {"-": ["11"]},
    "PELLAX": {"-": ["25"]},
    "PELLEX": {"-": ["13"]},
    "PELLICIO": {"-": ["55"]},
    "PELLICVLA": {"-": ["11"]},
    "PELLIS": {"-": ["13"]},
    "PELLITVS": {"-": ["21"]},
    "PELLO": {"-": ["53"]},
    "PELLVCEO": {"-": ["52"]},
    "PELLVCIDVLVS": {"-": ["21"]},
    "PELLVCIDVS": {"-": ["21"]},
    "PELOPONNESVS": {"N": ["12"]},
    "PELORIS": {"-": ["17"], "N": ["13"]},
    "PELTA": {"-": ["11"]},
    "PELTATVS": {"-": ["21"]},
    "PELVIS": {"-": ["13"]},
    "PELVSIVM": {"N": ["12"]},
    "PENARIVS": {"-": ["21"]},
    "PENATES": {"-": ["13"], "N": ["13"]},
    "PENATOR": {"-": ["13"]},
    "PENDEO": {"-": ["52"]},
    "PENDO": {"-": ["53"]},
    "PENDVLVS": {"-": ["21"]},
    "PENES": {"-": ["70"]},
    "PENETRABILIS": {"-": ["24"]},
    "PENETRALE": {"-": ["13"]},
    "PENETRALIS": {"-": ["24"]},
    "PENETRO": {"-": ["51"]},
    "PENIS": {"-": ["13"]},
    "PENITE": {"-": ["60"]},
    "PENITVS": {"1": ["21"], "2": ["60"]},
    "PENNA": {"-": ["11"]},
    "PENNATVS": {"-": ["21"]},
    "PENNIPES": {"-": ["25"]},
    "PENNIPOTENTES": {"-": ["13"]},
    "PENSILIS": {"-": ["24"]},
    "PENSIO": {"-": ["13"]},
    "PENSITATIO": {"-": ["13"]},
    "PENSITO": {"-": ["51"]},
    "PENSO": {"-": ["51"]},
    "PENSVM": {"-": ["12"]},
    "PENSVS": {"-": ["21"]},
    "PENTADOROS": {"1": ["27"], "2": ["60"]},
    "PENTASPASTON": {"-": ["17"]},
    "PENTEMOEROS": {"-": ["27"]},
    "PENTEREMIS": {"-": ["13"]},
    "PENTERES": {"-": ["13"]},
    "PENVRIA": {"-": ["11"]},
    "PENVS": {"-": ["16"]},
    "PEPLVS": {"-": ["12"]},
    "PER": {"-": ["70"]},
    "PERA": {"-": ["11"]},
    "PERACVTVS": {"-": ["21"]},
    "PERADOLESCENS": {"-": ["25"]},
    "PERAEQVE": {"-": ["60"]},
    "PERAEQVO": {"-": ["51"]},
    "PERAGITO": {"-": ["51"]},
    "PERAGO": {"-": ["53"]},
    "PERAGRO": {"-": ["51"]},
    "PERAMBVLO": {"-": ["51"]},
    "PERAMOENVS": {"-": ["21"]},
    "PERAMPLVS": {"-": ["21"]},
    "PERANGVSTVS": {"-": ["21"]},
    "PERANTIQVVS": {"-": ["21"]},
    "PERARDVVS": {"-": ["21"]},
    "PERARIDVS": {"-": ["21"]},
    "PERARMO": {"-": ["51"]},
    "PERARO": {"-": ["51"]},
    "PERATTENTVS": {"-": ["21"]},
    "PERBIBO": {"2": ["53"]},
    "PERBONVS": {"-": ["21"]},
    "PERBREVI": {"-": ["60"]},
    "PERBREVIS": {"-": ["24"]},
    "PERCALEFACIO": {"-": ["55"]},
    "PERCALESCO": {"-": ["53"]},
    "PERCANDEFACIO": {"-": ["54"]},
    "PERCARVS": {"-": ["21"]},
    "PERCELEBRO": {"-": ["51"]},
    "PERCELLO": {"-": ["53"]},
    "PERCENNIVS": {"N": ["12"]},
    "PERCENSEO": {"-": ["52"]},
    "PERCEPTIO": {"-": ["13"]},
    "PERCIDO": {"2": ["53"]},
    "PERCIEO": {"-": ["52"]},
    "PERCIO": {"-": ["54"]},
    "PERCIPIO": {"-": ["55"]},
    "PERCITO": {"1": ["51"]},
    "PERCOLATIO": {"-": ["13"]},
    "PERCOLO": {"1": ["51"], "2": ["53"]},
    "PERCOMMODE": {"-": ["60"]},
    "PERCOMPERIO": {"-": ["54"]},
    "PERCONTATIO": {"-": ["13"]},
    "PERCONTATOR": {"-": ["13"]},
    "PERCOQVO": {"-": ["53"]},
    "PERCREBRESCO": {"-": ["53"]},
    "PERCREPO": {"-": ["51"]},
    "PERCRESCO": {"-": ["53"]},
    "PERCRVDVS": {"-": ["21"]},
    "PERCVLSVS": {"-": ["14"]},
    "PERCVNCTATIO": {"-": ["13"]},
    "PERCVNCTO": {"-": ["51"]},
    "PERCVNCTOR": {"-": ["5J"]},
    "PERCVRO": {"-": ["51"]},
    "PERCVRRO": {"-": ["53"]},
    "PERCVRSO": {"-": ["51"]},
    "PERCVSSIO": {"-": ["13"]},
    "PERCVSSOR": {"-": ["13"]},
    "PERCVSSVS": {"-": ["14"]},
    "PERCVTIO": {"-": ["55"]},
    "PERDELIRVS": {"-": ["21"]},
    "PERDEPSO": {"-": ["53"]},
    "PERDIFFICILIS": {"-": ["24"]},
    "PERDISCO": {"-": ["53"]},
    "PERDITE": {"-": ["60"]},
    "PERDITOR": {"-": ["13"]},
    "PERDITVS": {"-": ["21"]},
    "PERDIVES": {"-": ["22"]},
    "PERDIVTVRNVS": {"-": ["21"]},
    "PERDO": {"-": ["53"]},
    "PERDOCEO": {"-": ["52"]},
    "PERDOLO": {"-": ["51"]},
    "PERDOMO": {"-": ["51"]},
    "PERDVCO": {"-": ["53"]},
    "PERDVCTIO": {"-": ["13"]},
    "PERDVCTOR": {"-": ["13"]},
    "PERDVELLIO": {"-": ["13"]},
    "PERDVELLIS": {"-": ["13"]},
    "PERDVLCIS": {"-": ["24"]},
    "PERDVRO": {"-": ["51"]},
    "PEREDO": {"1": ["53"], "2": ["56"]},
    "PEREGRE": {"-": ["60"]},
    "PEREGRINATIO": {"-": ["13"]},
    "PEREGRINOR": {"-": ["5J"]},
    "PEREGRINVS": {"1": ["12"], "2": ["21"]},
    "PEREMPTOR": {"-": ["13"]},
    "PERENDINVS": {"-": ["21"]},
    "PERENNE": {"-": ["60"]},
    "PERENNIS": {"-": ["24"]},
    "PERENNO": {"-": ["51"]},
    "PEREO": {"1": ["56"]},
    "PEREQVITO": {"-": ["51"]},
    "PERERRO": {"2": ["51"]},
    "PEREXCELSVS": {"-": ["21"]},
    "PEREXIGVVS": {"-": ["21"]},
    "PERFACETE": {"-": ["60"]},
    "PERFACILE": {"-": ["60"]},
    "PERFACILIS": {"-": ["24"]},
    "PERFAMILIARIS": {"-": ["24"]},
    "PERFECTE": {"-": ["60"]},
    "PERFECTIO": {"-": ["13"]},
    "PERFECTVS": {"1": ["14"], "2": ["21"]},
    "PERFERO": {"-": ["56"]},
    "PERFICIO": {"-": ["55"]},
    "PERFICVS": {"-": ["21"]},
    "PERFIDE": {"-": ["60"]},
    "PERFIDIA": {"-": ["11"], "N": ["11"]},
    "PERFIDIOSVS": {"-": ["21"]},
    "PERFIDVM": {"-": ["60"]},
    "PERFIDVS": {"1": ["21"], "2": ["21"]},
    "PERFIGO": {"-": ["53"]},
    "PERFLATVS": {"-": ["14"]},
    "PERFLO": {"-": ["51"]},
    "PERFLVCTVO": {"-": ["51"]},
    "PERFLVO": {"-": ["53"]},
    "PERFODIO": {"-": ["55"]},
    "PERFORO": {"-": ["51"]},
    "PERFRICO": {"-": ["51"]},
    "PERFRIGESCO": {"-": ["53"]},
    "PERFRIGIDVS": {"-": ["21"]},
    "PERFRINGO": {"-": ["53"]},
    "PERFRVOR": {"-": ["5L"]},
    "PERFVGA": {"-": ["11"]},
    "PERFVGIO": {"-": ["55"]},
    "PERFVGIVM": {"-": ["12"]},
    "PERFVNCTIO": {"-": ["13"]},
    "PERFVNDO": {"2": ["53"]},
    "PERFVNGOR": {"-": ["5L"]},
    "PERFVRO": {"-": ["53"]},
    "PERFVSORIVS": {"-": ["21"]},
    "PERGA": {"N": ["11"]},
    "PERGAEVS": {"N": ["21"], "A": ["21"]},
    "PERGAMVS": {"N": ["17"]},
    "PERGO": {"-": ["53"]},
    "PERGRANDIS": {"-": ["24"]},
    "PERGRATVS": {"-": ["21"]},
    "PERGVLA": {"-": ["11"]},
    "PERHIBEO": {"-": ["52"]},
    "PERHILVM": {"-": ["60"]},
    "PERHONORIFICVS": {"-": ["21"]},
    "PERHORRESCO": {"-": ["53"]},
    "PERIACTOS": {"-": ["27"]},
    "PERICLES": {"N": ["17"]},
    "PERICLITOR": {"-": ["5J"]},
    "PERICVLOR": {"-": ["5J"]},
    "PERICVLOSE": {"-": ["60"]},
    "PERICVLOSVS": {"-": ["21"]},
    "PERICVLVM": {"-": ["12"]},
    "PERIDONEVS": {"-": ["21"]},
    "PERIECHON": {"-": ["57"]},
    "PERIMETROS": {"-": ["17"]},
    "PERIMO": {"-": ["53"]},
    "PERIMPEDITVS": {"-": ["21"]},
    "PERINDE": {"-": ["60"]},
    "PERINGRATVS": {"-": ["21"]},
    "PERINIQVVS": {"-": ["21"]},
    "PERINIVRIVS": {"-": ["21"]},
    "PERINTEMPESTIVVS": {"-": ["21"]},
    "PERINVALIDVS": {"-": ["21"]},
    "PERIPATETICI": {"N": ["12"]},
    "PERIPATETICVS": {"1": ["12"], "N": ["21"], "A": ["21"]},
    "PERIPETASMA": {"-": ["17"]},
    "PERIPTEROS": {"-": ["27"]},
    "PERISCELIS": {"-": ["13"]},
    "PERISTYLIVM": {"-": ["12"]},
    "PERISTYLVM": {"-": ["12"]},
    "PERITE": {"-": ["60"]},
    "PERITHECIOS": {"-": ["27"]},
    "PERITIA": {"-": ["11"]},
    "PERITRETOS": {"-": ["27"]},
    "PERITVS": {"-": ["21"]},
    "PERIVRIVM": {"-": ["12"]},
    "PERIVRO": {"-": ["51"]},
    "PERIVRVS": {"-": ["21"]},
    "PERLABOR": {"2": ["5L"]},
    "PERLATEO": {"-": ["52"]},
    "PERLEGO": {"2": ["53"]},
    "PERLIBRATIO": {"-": ["13"]},
    "PERLIBRO": {"-": ["51"]},
    "PERLIMO": {"-": ["51"]},
    "PERLVO": {"-": ["53"]},
    "PERLVSTRO": {"2": ["51"]},
    "PERMACERO": {"-": ["51"]},
    "PERMADESCO": {"-": ["53"]},
    "PERMAGNVS": {"-": ["21"]},
    "PERMANANTER": {"-": ["60"]},
    "PERMANEO": {"-": ["52"]},
    "PERMANO": {"-": ["51"]},
    "PERMATVRESCO": {"-": ["53"]},
    "PERMEO": {"-": ["51"]},
    "PERMEREO": {"-": ["52"]},
    "PERMETIOR": {"-": ["5M"]},
    "PERMINGO": {"-": ["53"]},
    "PERMISCEO": {"-": ["52"]},
    "PERMISSVS": {"1": ["14"], "2": ["21"]},
    "PERMITIALIS": {"-": ["24"]},
    "PERMITTO": {"-": ["53"]},
    "PERMIXTIO": {"-": ["13"]},
    "PERMIXTVS": {"-": ["21"]},
    "PERMODESTVS": {"-": ["21"]},
    "PERMOLESTE": {"-": ["60"]},
    "PERMOLO": {"-": ["53"]},
    "PERMOVEO": {"-": ["52"]},
    "PERMVLCEO": {"-": ["52"]},
    "PERMVLTO": {"-": ["60"]},
    "PERMVLTVM": {"-": ["60"]},
    "PERMVLTVS": {"-": ["21"]},
    "PERMVNIO": {"2": ["54"]},
    "PERMVTATIO": {"-": ["13"]},
    "PERMVTO": {"2": ["51"]},
    "PERNA": {"-": ["11"]},
    "PERNEGO": {"-": ["51"]},
    "PERNICIABILIS": {"-": ["24"]},
    "PERNICIES": {"-": ["15"]},
    "PERNICIOSVS": {"-": ["21"]},
    "PERNICITAS": {"-": ["13"]},
    "PERNICITER": {"-": ["60"]},
    "PERNIX": {"-": ["25"]},
    "PERNOBILIS": {"-": ["24"]},
    "PERNOCTO": {"-": ["51"]},
    "PERNOSCO": {"-": ["53"]},
    "PERNOTESCO": {"-": ["53"]},
    "PERNOX": {"-": ["25"]},
    "PERNVMERO": {"1": ["51"]},
    "PERO": {"-": ["13"], "N": ["13"]},
    "PEROLEO": {"-": ["52"]},
    "PERONATVS": {"-": ["21"]},
    "PEROPPORTVNE": {"-": ["60"]},
    "PEROPPORTVNVS": {"-": ["21"]},
    "PERORNO": {"-": ["51"]},
    "PERORO": {"-": ["51"]},
    "PEROSVS": {"-": ["21"]},
    "PERPARVVLVS": {"-": ["21"]},
    "PERPARVVS": {"-": ["21"]},
    "PERPAVCI": {"-": ["12"]},
    "PERPAVCVS": {"-": ["21"]},
    "PERPELLO": {"-": ["53"]},
    "PERPENDEO": {"-": ["52"]},
    "PERPENDICVLVM": {"-": ["12"]},
    "PERPENDO": {"-": ["53"]},
    "PERPENSO": {"-": ["51"]},
    "PERPERAM": {"-": ["60"]},
    "PERPERE": {"-": ["60"]},
    "PERPERNA": {"N": ["11"]},
    "PERPERVS": {"-": ["21"]},
    "PERPES": {"-": ["25"]},
    "PERPESSICIVS": {"-": ["21"]},
    "PERPESSIO": {"-": ["13"]},
    "PERPETIOR": {"-": ["5N"]},
    "PERPETO": {"-": ["53"]},
    "PERPETRO": {"-": ["51"]},
    "PERPETVE": {"-": ["6&"]},
    "PERPETVITAS": {"-": ["13"]},
    "PERPETVO": {"1": ["51"], "2": ["60"]},
    "PERPETVVM": {"-": ["60"]},
    "PERPETVVS": {"-": ["21"]},
    "PERPLEXVS": {"-": ["21"]},
    "PERPLICATVS": {"-": ["21"]},
    "PERPLVO": {"-": ["53"]},
    "PERPOPVLOR": {"-": ["5J"]},
    "PERPOTO": {"-": ["51"]},
    "PERPRIMO": {"-": ["53"]},
    "PERPVRGO": {"-": ["51"]},
    "PERQVADRATVS": {"-": ["21"]},
    "PERQVAM": {"-": ["60"]},
    "PERQVIRO": {"-": ["53"]},
    "PERRARO": {"-": ["60"]},
    "PERREPO": {"-": ["53"]},
    "PERRIDICVLE": {"-": ["60"]},
    "PERROGO": {"-": ["51"]},
    "PERRVMPO": {"-": ["53"]},
    "PERSA": {"N": ["11"]},
    "PERSAE": {"N": ["11"]},
    "PERSAEPE": {"-": ["60"]},
    "PERSALVTO": {"-": ["51"]},
    "PERSANO": {"-": ["51"]},
    "PERSANVS": {"-": ["21"]},
    "PERSCINDO": {"-": ["53"]},
    "PERSCRIBO": {"-": ["53"]},
    "PERSCRIPTOR": {"-": ["13"]},
    "PERSCRVTATIO": {"-": ["13"]},
    "PERSCRVTOR": {"-": ["5J"]},
    "PERSECO": {"-": ["51"]},
    "PERSECTOR": {"2": ["5J"]},
    "PERSEDEO": {"-": ["52"]},
    "PERSENTIO": {"-": ["54"]},
    "PERSENTISCO": {"-": ["53"]},
    "PERSEQVOR": {"-": ["5L"]},
    "PERSERO": {"2": ["53"], "3": ["53"]},
    "PERSES": {"N": ["17"], "A": ["25"]},
    "PERSEVERANS": {"-": ["25"]},
    "PERSEVERANTER": {"-": ["60"]},
    "PERSEVERANTIA": {"-": ["11"]},
    "PERSEVERO": {"-": ["51"]},
    "PERSEVERVS": {"-": ["21"]},
    "PERSIDO": {"-": ["53"]},
    "PERSIMPLEX": {"-": ["25"]},
    "PERSISTO": {"-": ["53"]},
    "PERSOLVO": {"-": ["53"]},
    "PERSONA": {"-": ["11"]},
    "PERSONATVS": {"-": ["21"]},
    "PERSONO": {"1": ["51"]},
    "PERSPECTO": {"-": ["51"]},
    "PERSPECTVS": {"-": ["21"]},
    "PERSPERGO": {"-": ["53"]},
    "PERSPICIO": {"-": ["55"]},
    "PERSPICVE": {"-": ["60"]},
    "PERSPICVITAS": {"-": ["13"]},
    "PERSPICVVS": {"-": ["21"]},
    "PERSPIRO": {"-": ["51"]},
    "PERSTERNO": {"-": ["53"]},
    "PERSTIMVLO": {"-": ["51"]},
    "PERSTO": {"-": ["51"]},
    "PERSTREPO": {"-": ["53"]},
    "PERSTRINGO": {"-": ["53"]},
    "PERSTRVO": {"-": ["53"]},
    "PERSTVDIOSVS": {"-": ["21"]},
    "PERSVADEO": {"-": ["52"]},
    "PERSVASIO": {"-": ["13"]},
    "PERSVBTILIS": {"-": ["24"]},
    "PERSVLTO": {"-": ["51"]},
    "PERTAEDESCO": {"-": ["53"]},
    "PERTAEDET": {"-": ["52"]},
    "PERTEGO": {"-": ["53"]},
    "PERTENDO": {"-": ["53"]},
    "PERTENTO": {"-": ["51"]},
    "PERTENVIS": {"-": ["24"]},
    "PERTEREBRO": {"-": ["51"]},
    "PERTERGEO": {"-": ["52"]},
    "PERTERREO": {"-": ["52"]},
    "PERTERRICREPVS": {"-": ["21"]},
    "PERTEXO": {"-": ["53"]},
    "PERTICA": {"-": ["11"]},
    "PERTIMEO": {"-": ["52"]},
    "PERTIMESCO": {"-": ["53"]},
    "PERTINACIA": {"-": ["11"]},
    "PERTINACITER": {"-": ["60"]},
    "PERTINAX": {"-": ["25"]},
    "PERTINEO": {"-": ["52"]},
    "PERTINGO": {"-": ["53"]},
    "PERTIO": {"-": ["54"]},
    "PERTOLERO": {"-": ["51"]},
    "PERTORQVEO": {"-": ["52"]},
    "PERTRACTO": {"-": ["51"]},
    "PERTRAHO": {"-": ["53"]},
    "PERTRITVS": {"-": ["21"]},
    "PERTVNDO": {"-": ["53"]},
    "PERTVRBATE": {"-": ["60"]},
    "PERTVRBATIO": {"-": ["13"]},
    "PERTVRBATVS": {"-": ["21"]},
    "PERTVRBO": {"2": ["51"]},
    "PERVADO": {"2": ["53"]},
    "PERVAGOR": {"2": ["5J"]},
    "PERVAGVS": {"-": ["21"]},
    "PERVALEO": {"-": ["52"]},
    "PERVASTO": {"-": ["51"]},
    "PERVECTOR": {"-": ["13"]},
    "PERVEHO": {"-": ["53"]},
    "PERVELLO": {"-": ["53"]},
    "PERVENIO": {"-": ["54"]},
    "PERVERSE": {"-": ["60"]},
    "PERVERSITAS": {"-": ["13"]},
    "PERVERSVM": {"-": ["12"]},
    "PERVERSVS": {"-": ["21"]},
    "PERVERTO": {"-": ["53"]},
    "PERVESTIGO": {"-": ["51"]},
    "PERVETVS": {"-": ["22"]},
    "PERVICACIA": {"-": ["11"]},
    "PERVICACITER": {"-": ["6&"]},
    "PERVICAX": {"-": ["25"]},
    "PERVIDEO": {"-": ["52"]},
    "PERVIGEO": {"-": ["52"]},
    "PERVIGIL": {"-": ["25"]},
    "PERVIGILIVM": {"-": ["12"]},
    "PERVIGILO": {"-": ["51"]},
    "PERVINCO": {"-": ["53"]},
    "PERVIVM": {"-": ["12"]},
    "PERVIVS": {"-": ["21"]},
    "PERVLA": {"-": ["11"]},
    "PERVNGO": {"-": ["53"]},
    "PERVOLITANTIA": {"-": ["11"]},
    "PERVOLITO": {"-": ["51"]},
    "PERVOLO": {"2": ["51"], "3": ["56"]},
    "PERVOLVO": {"-": ["53"]},
    "PERVRO": {"-": ["53"]},
    "PERVSINVS": {"N": ["21"], "A": ["21"]},
    "PERVVLGO": {"1": ["51"]},
    "PES": {"-": ["13"]},
    "PESSVM": {"1": ["12"], "2": ["60"]},
    "PESSVMDO": {"-": ["51"]},
    "PESTIFER": {"-": ["21"]},
    "PESTILENS": {"-": ["25"]},
    "PESTILENTIA": {"-": ["11"]},
    "PESTILITAS": {"-": ["13"]},
    "PESTIS": {"-": ["13"], "N": ["13"]},
    "PETASVNCVLVS": {"-": ["12"]},
    "PETAVRVM": {"-": ["12"]},
    "PETESSO": {"-": ["53"]},
    "PETILIVS": {"N": ["12"]},
    "PETIOR": {"-": ["5N"]},
    "PETITIO": {"-": ["13"]},
    "PETITOR": {"-": ["13"]},
    "PETITVS": {"-": ["14"]},
    "PETO": {"-": ["53"]},
    "PETORITVM": {"-": ["12"]},
    "PETRA": {"-": ["11"], "N": ["11"]},
    "PETRINI": {"N": ["12"]},
    "PETRO": {"-": ["51"]},
    "PETVLANS": {"-": ["25"]},
    "PETVLANTER": {"-": ["6&"]},
    "PETVLANTIA": {"-": ["11"]},
    "PETVLCVS": {"-": ["21"]},
    "PEXVS": {"-": ["21"]},
    "PHAECASIATVS": {"-": ["21"]},
    "PHAECASIVM": {"-": ["12"]},
    "PHALACRVS": {"N": ["12"]},
    "PHALANGA": {"-": ["11"]},
    "PHALANGARIVS": {"-": ["12"]},
    "PHALANX": {"-": ["13"]},
    "PHALARIS": {"N": ["17"]},
    "PHALERAE": {"-": ["11"]},
    "PHARETRA": {"-": ["11"]},
    "PHARETRATVS": {"-": ["21"]},
    "PHARMACOPOLA": {"-": ["11"]},
    "PHAROS": {"-": ["17"], "N": ["17"]},
    "PHARVS": {"-": ["17"], "N": ["12"]},
    "PHASELIS": {"N": ["13"]},
    "PHELLOS": {"-": ["17"]},
    "PHIALA": {"-": ["11"]},
    "PHILEMA": {"-": ["17"]},
    "PHILINVS": {"N": ["12"]},
    "PHILIPPVS": {"1N": ["12"], "2N": ["12"], "A": ["21"], "N": ["12"]},
    "PHILO": {"N": ["17"]},
    "PHILOCRATES": {"N": ["17"]},
    "PHILODAMVS": {"N": ["12"]},
    "PHILOLOGIA": {"-": ["11"]},
    "PHILOLOGVS": {"1": ["12"], "2": ["21"]},
    "PHILOMELIENSES": {"N": ["13"]},
    "PHILOMELIVM": {"N": ["12"]},
    "PHILOSOPHIA": {"-": ["11"]},
    "PHILOSOPHOR": {"-": ["5J"]},
    "PHILOSOPHVS": {"1": ["12"], "2": ["21"]},
    "PHILOTECHNVS": {"-": ["21"]},
    "PHILTRVM": {"-": ["12"]},
    "PHILYRA": {"-": ["11"]},
    "PHIMES": {"N": ["17"]},
    "PHIMVS": {"-": ["12"]},
    "PHINTIA": {"N": ["11"]},
    "PHOCA": {"-": ["11"]},
    "PHOCIS": {"N": ["17"]},
    "PHOEBVS": {"N": ["12"]},
    "PHOENICE": {"N": ["17"]},
    "PHOENICES": {"N": ["17", "13"]},
    "PHOENICOPTERVS": {"-": ["12"]},
    "PHOENIX": {"-": ["13"], "N": ["13"]},
    "PHONASCVS": {"-": ["12"]},
    "PHRENESIS": {"-": ["17"]},
    "PHRENETICVS": {"-": ["21"]},
    "PHRYGIA": {"N": ["11"]},
    "PHRYX": {"A": ["27"], "N": ["17"]},
    "PHTHISICVS": {"-": ["21"]},
    "PHTHISIS": {"-": ["17"]},
    "PHTHONGOS": {"-": ["17"]},
    "PHYLARCHVS": {"N": ["12"]},
    "PHYSETER": {"-": ["17"]},
    "PHYSICA": {"1": ["12"], "2": ["17"]},
    "PHYSICVS": {"1": ["12"], "2": ["21"]},
    "PHYSIOLOGIA": {"-": ["11"]},
    "PIACVLARIS": {"-": ["21"]},
    "PIACVLO": {"-": ["51"]},
    "PIACVLVM": {"-": ["12"]},
    "PIAMENTVM": {"-": ["12"]},
    "PICA": {"-": ["11"]},
    "PICEA": {"-": ["11"]},
    "PICEVS": {"-": ["21"]},
    "PICO": {"-": ["51"]},
    "PICTOR": {"-": ["13"], "N": ["13"]},
    "PICTVRA": {"-": ["11"]},
    "PICTVRATVS": {"-": ["21"]},
    "PICTVS": {"-": ["21"]},
    "PICVS": {"-": ["12"], "N": ["12"]},
    "PIE": {"-": ["60"]},
    "PIETAS": {"-": ["13"], "N": ["13"]},
    "PIGER": {"-": ["21"]},
    "PIGET": {"-": ["52"]},
    "PIGNERATOR": {"-": ["13"]},
    "PIGNERO": {"-": ["51"]},
    "PIGNVS": {"-": ["13"]},
    "PIGRE": {"-": ["60"]},
    "PIGRITIA": {"-": ["11"]},
    "PIGROR": {"-": ["5J"]},
    "PILA": {"1": ["11"], "2": ["11"], "3": ["11"]},
    "PILATIM": {"-": ["60"]},
    "PILATVS": {"-": ["21"], "N": ["12"]},
    "PILEATVS": {"-": ["21"]},
    "PILENTVM": {"-": ["12"]},
    "PILEOLVS": {"-": ["12"]},
    "PILEVM": {"-": ["12"]},
    "PILEVS": {"-": ["12"]},
    "PILICREPVS": {"-": ["12"]},
    "PILO": {"1": ["51"], "2": ["51"]},
    "PILOSVS": {"-": ["21"]},
    "PILVM": {"-": ["12"], "N": ["12"]},
    "PILVS": {"1": ["12"], "2": ["12"]},
    "PINACOTHECA": {"-": ["17"]},
    "PINARIVS": {"A": ["21"], "N": ["12"]},
    "PINAX": {"-": ["17"]},
    "PINETVM": {"-": ["12"]},
    "PINEVS": {"-": ["21"]},
    "PINGO": {"-": ["53"]},
    "PINGVE": {"-": ["13"]},
    "PINGVESCO": {"-": ["53"]},
    "PINGVIS": {"-": ["24"]},
    "PINGVITVDO": {"-": ["13"]},
    "PINIFER": {"-": ["21"]},
    "PINNA": {"-": ["11"]},
    "PINNIGER": {"-": ["21"]},
    "PINNIRAPVS": {"-": ["12"]},
    "PINSATIO": {"-": ["13"]},
    "PINSO": {"-": ["56"]},
    "PINVS": {"-": ["16"]},
    "PIO": {"-": ["51"]},
    "PIPA": {"N": ["11"]},
    "PIPER": {"-": ["13"]},
    "PIPIO": {"1": ["13"], "2": ["51"], "3": ["54"]},
    "PIRAEEVS": {"N": ["12"]},
    "PIRATA": {"-": ["11"]},
    "PIRATICA": {"-": ["11"]},
    "PIRATICVS": {"-": ["21"]},
    "PIRVM": {"-": ["12"]},
    "PIRVS": {"-": ["12"]},
    "PISATIO": {"-": ["13"]},
    "PISCARIVS": {"-": ["21"]},
    "PISCATOR": {"-": ["13"]},
    "PISCATORIVS": {"-": ["21"]},
    "PISCATVS": {"-": ["14"]},
    "PISCICVLVS": {"-": ["12"]},
    "PISCINA": {"-": ["11"]},
    "PISCIS": {"-": ["13"], "N": ["13"]},
    "PISCOR": {"-": ["5J"]},
    "PISCOSVS": {"-": ["21"]},
    "PISCVLENTVS": {"-": ["21"]},
    "PISIDIA": {"N": ["11"]},
    "PISO": {"N": ["13"]},
    "PISTOR": {"-": ["13"], "N": ["13"]},
    "PISTRINVM": {"-": ["12"]},
    "PISTRIS": {"-": ["13"]},
    "PISTRIX": {"-": ["13"]},
    "PITVITA": {"-": ["11"]},
    "PIVS": {"-": ["21"], "N": ["12"]},
    "PIX": {"-": ["13"], "2": ["13"], "1": ["12"]},
    "PLACABILIS": {"-": ["24"]},
    "PLACAMENTVM": {"-": ["12"]},
    "PLACATIO": {"-": ["13"]},
    "PLACATVS": {"-": ["21"]},
    "PLACENTA": {"-": ["11"]},
    "PLACENTINVS": {"N": ["21"], "A": ["21"]},
    "PLACEO": {"-": ["52"]},
    "PLACIDE": {"-": ["60"]},
    "PLACIDVS": {"-": ["21"], "N": ["12"]},
    "PLACITO": {"-": ["51"]},
    "PLACITVM": {"-": ["12"]},
    "PLACITVS": {"-": ["21"]},
    "PLACO": {"-": ["51"]},
    "PLAGA": {"1": ["11"], "2": ["11"], "3": ["11"]},
    "PLAGIARIVS": {"-": ["12"]},
    "PLAGOSVS": {"-": ["21"]},
    "PLANATVS": {"-": ["21"]},
    "PLANCTVS": {"-": ["14"]},
    "PLANE": {"-": ["60"]},
    "PLANGO": {"-": ["53"]},
    "PLANGOR": {"-": ["13"]},
    "PLANIPES": {"-": ["13"]},
    "PLANITAS": {"-": ["13"]},
    "PLANITIA": {"-": ["11"]},
    "PLANITIES": {"-": ["15"]},
    "PLANTA": {"-": ["11"], "N": ["11"]},
    "PLANTARIA": {"-": ["13"]},
    "PLANTARIS": {"-": ["24"]},
    "PLANTO": {"-": ["51"]},
    "PLANVM": {"-": ["12"]},
    "PLANVS": {"1": ["12"], "2": ["21"]},
    "PLASMA": {"-": ["17"]},
    "PLASTES": {"-": ["17"]},
    "PLASTICVS": {"-": ["21"]},
    "PLATANON": {"-": ["17"]},
    "PLATANVS": {"-": ["16"]},
    "PLATEA": {"1": ["11"]},
    "PLATO": {"N": ["17"]},
    "PLAVDO": {"-": ["53"]},
    "PLAVSIBILIS": {"-": ["24"]},
    "PLAVSOR": {"-": ["13"]},
    "PLAVSTELLVM": {"-": ["12"]},
    "PLAVSTRARIVS": {"2": ["21"]},
    "PLAVSTRVM": {"-": ["12"], "N": ["12"]},
    "PLAVSVS": {"-": ["14"]},
    "PLEBECVLA": {"-": ["11"]},
    "PLEBEIVS": {"1": ["12"], "2": ["21"]},
    "PLEBICOLA": {"-": ["11"]},
    "PLEBISCITVM": {"-": ["12"]},
    "PLEBITAS": {"-": ["13"]},
    "PLEBS": {"-": ["16"]},
    "PLECTO": {"1": ["53"], "2": ["53"]},
    "PLECTOR": {"-": ["5L"]},
    "PLECTRVM": {"-": ["12"]},
    "PLENE": {"-": ["60"]},
    "PLENITAS": {"-": ["13"]},
    "PLENVM": {"-": ["12"]},
    "PLENVS": {"-": ["21"]},
    "PLEO": {"-": ["52"]},
    "PLERAQVE": {"-": ["12"]},
    "PLERIQVE": {"-": ["12"]},
    "PLERVMQVE": {"-": ["60"]},
    "PLERVS": {"-": ["21"]},
    "PLERVSQVE": {"-": ["21"]},
    "PLEVRITIS": {"-": ["17"]},
    "PLICEO": {"-": ["52"]},
    "PLICO": {"-": ["51"]},
    "PLINTHIS": {"-": ["17"]},
    "PLINTHIVM": {"-": ["12"]},
    "PLINTHVS": {"-": ["17"]},
    "PLODO": {"-": ["53"]},
    "PLORABILIS": {"-": ["24"]},
    "PLORATVS": {"-": ["14"]},
    "PLORO": {"-": ["51"]},
    "PLOXEMVM": {"-": ["12"]},
    "PLVMA": {"-": ["11"]},
    "PLVMARIVS": {"-": ["12"]},
    "PLVMBARIVS": {"-": ["21"]},
    "PLVMBEVM": {"-": ["12"]},
    "PLVMBEVS": {"-": ["21"]},
    "PLVMBO": {"-": ["51"]},
    "PLVMBVM": {"-": ["12"]},
    "PLVMIPES": {"-": ["27"]},
    "PLVMO": {"-": ["51"]},
    "PLVMVLA": {"-": ["11"]},
    "PLVO": {"-": ["53"]},
    "PLVRIMVM": {"-": ["12"]},
    "PLVSCVLVM": {"1": ["12"], "2": ["60"]},
    "PLVTEVM": {"-": ["12"]},
    "PLVTEVS": {"-": ["12"]},
    "PLVVIA": {"-": ["11"]},
    "PLVVIALIS": {"-": ["24"]},
    "PLVVIVS": {"-": ["21"]},
    "PNEVMATICOS": {"-": ["27"]},
    "PNIGEVS": {"-": ["17"]},
    "POCILLVM": {"-": ["12"]},
    "POCVLVM": {"-": ["12"]},
    "PODAGRA": {"-": ["11"]},
    "PODAGRICVS": {"-": ["21"]},
    "PODEX": {"-": ["13"]},
    "PODIVM": {"-": ["12"]},
    "POEMA": {"-": ["16"]},
    "POENA": {"-": ["11"], "N": ["11"]},
    "POENI": {"N": ["12"]},
    "POENVS": {"N": ["21"], "A": ["21"]},
    "POETA": {"-": ["11"]},
    "POETICA": {"-": ["11"], "2": ["11"]},
    "POETICVS": {"-": ["21"]},
    "POETRIA": {"-": ["11"]},
    "POETRIS": {"-": ["17"]},
    "POL": {"-": ["90"]},
    "POLEAS": {"N": ["17"]},
    "POLEMARCHVS": {"N": ["12"]},
    "POLENTA": {"-": ["11"]},
    "POLIO": {"1": ["13"], "2": ["54"]},
    "POLITIO": {"-": ["13"]},
    "POLITOR": {"-": ["13"]},
    "POLITORIVM": {"N": ["12"]},
    "POLITVRA": {"-": ["11"]},
    "POLITVS": {"-": ["21"]},
    "POLLEN": {"-": ["13"]},
    "POLLENS": {"-": ["25"]},
    "POLLEO": {"-": ["52"]},
    "POLLEX": {"-": ["13"]},
    "POLLICEOR": {"-": ["5K"]},
    "POLLICITATIO": {"-": ["13"]},
    "POLLICITOR": {"-": ["5J"]},
    "POLLICITVM": {"-": ["12"]},
    "POLLINGO": {"-": ["53"]},
    "POLLIO": {"N": ["13"]},
    "POLLVCEO": {"-": ["52"]},
    "POLLVO": {"-": ["53"]},
    "POLLVTVS": {"-": ["21"]},
    "POLLVX": {"N": ["13"]},
    "POLVS": {"-": ["12"]},
    "POLVSCA": {"N": ["11"]},
    "POLYBIVS": {"N": ["12"]},
    "POLYCLITVS": {"N": ["12"]},
    "POLYGONIVS": {"-": ["21"]},
    "POLYPVS": {"-": ["12"]},
    "POLYSPASTON": {"-": ["17"]},
    "POMARIVM": {"-": ["12"]},
    "POMARIVS": {"1": ["12"], "2": ["21"]},
    "POMERIDIANVS": {"-": ["21"]},
    "POMERIVM": {"-": ["12"]},
    "POMETIA": {"N": ["11"]},
    "POMETINVS": {"N": ["21"], "A": ["21"]},
    "POMIFER": {"-": ["21"]},
    "POMOSVS": {"-": ["21"]},
    "POMPA": {"-": ["11"]},
    "POMPEIVS": {"A": ["21"], "N": ["12"]},
    "POMPILIVS": {"N": ["12"], "A": ["21"]},
    "POMPONIVS": {"N": ["12"]},
    "POMPTINVS": {"A": ["21"], "N": ["12"]},
    "POMVM": {"-": ["12"]},
    "POMVS": {"-": ["12"]},
    "PONDERATIO": {"-": ["13"]},
    "PONDERATVS": {"-": ["21"]},
    "PONDERO": {"-": ["51"]},
    "PONDEROSVS": {"-": ["21"]},
    "PONDO": {"-": ["16"]},
    "PONDVS": {"-": ["13"]},
    "PONE": {"1": ["60"], "2": ["70"]},
    "PONEVERSVS": {"-": ["70"]},
    "PONO": {"-": ["53"]},
    "PONS": {"-": ["13"], "N": ["13"]},
    "PONTICVLVS": {"-": ["12"]},
    "PONTICVS": {"N": ["21"], "A": ["21"]},
    "PONTIFEX": {"-": ["13"]},
    "PONTIFICALIS": {"-": ["24"]},
    "PONTIFICATVS": {"-": ["14"]},
    "PONTIFICIVS": {"-": ["21"], "N": ["12"]},
    "PONTO": {"-": ["13"]},
    "PONTVS": {"-": ["12"], "N": ["12"]},
    "POPA": {"-": ["11"]},
    "POPANVM": {"-": ["12"]},
    "POPELLVS": {"-": ["12"]},
    "POPILIVS": {"N": ["12"]},
    "POPINA": {"-": ["11"]},
    "POPINO": {"-": ["13"]},
    "POPLES": {"-": ["13"]},
    "POPVLARIS": {"1": ["13"], "2": ["24"]},
    "POPVLARITAS": {"-": ["13"]},
    "POPVLARITER": {"-": ["60"]},
    "POPVLATIO": {"-": ["13"]},
    "POPVLATOR": {"-": ["13"]},
    "POPVLEVS": {"-": ["21"]},
    "POPVLIFER": {"-": ["21"]},
    "POPVLNEVS": {"-": ["21"]},
    "POPVLO": {"-": ["51"]},
    "POPVLOR": {"-": ["5J"]},
    "POPVLVS": {"1": ["12"], "2": ["12"]},
    "PORCA": {"-": ["11"]},
    "PORCINVS": {"-": ["21"]},
    "PORCIVS": {"-": ["12"], "A": ["21"], "N": ["12"]},
    "PORCVLVS": {"-": ["12"]},
    "PORCVS": {"-": ["12"]},
    "PORRECTIO": {"-": ["13"]},
    "PORRECTVS": {"-": ["21"]},
    "PORRIGO": {"1": ["13"], "2": ["53"]},
    "PORRINA": {"-": ["11"]},
    "PORRO": {"-": ["60"]},
    "PORRVM": {"-": ["12"]},
    "PORRVS": {"-": ["12"]},
    "PORSENA": {"N": ["11"]},
    "PORTA": {"-": ["11"]},
    "PORTATIO": {"-": ["13"]},
    "PORTENDO": {"-": ["53"]},
    "PORTENTOSVS": {"-": ["21"]},
    "PORTENTVM": {"-": ["12"]},
    "PORTHMEVS": {"-": ["17"], "N": ["12"]},
    "PORTICVS": {"-": ["14"], "N": ["12"]},
    "PORTIO": {"-": ["13"]},
    "PORTISCVLVS": {"-": ["12"]},
    "PORTITOR": {"1": ["13"], "2": ["13"]},
    "PORTIVS": {"N": ["12"]},
    "PORTO": {"-": ["51"]},
    "PORTORIVM": {"-": ["12"]},
    "PORTVLA": {"-": ["11"]},
    "PORTVOSVS": {"-": ["21"]},
    "PORTVS": {"-": ["14"]},
    "PORVS": {"-": ["12"], "N": ["12"]},
    "POSCIMVS": {"-": ["53"]},
    "POSCO": {"-": ["53"]},
    "POSIDES": {"N": ["17"]},
    "POSIDORVS": {"N": ["12"]},
    "POSITIO": {"-": ["13"]},
    "POSITVRA": {"-": ["11"]},
    "POSITVS": {"-": ["14"]},
    "POSOTES": {"-": ["17"]},
    "POSSESSIO": {"-": ["13"]},
    "POSSESSOR": {"-": ["13"]},
    "POSSIDEO": {"-": ["52"]},
    "POSSIDO": {"-": ["53"]},
    "POSSVM": {"1": ["56"]},
    "POST": {"1": ["60"], "2": ["70"]},
    "POSTEA": {"-": ["60"]},
    "POSTEAQVAM": {"-": ["82"]},
    "POSTERGA": {"-": ["60"]},
    "POSTERGVM": {"-": ["60"]},
    "POSTERI": {"-": ["12"]},
    "POSTERITAS": {"-": ["13"]},
    "POSTERIVS": {"-": ["6&"]},
    "POSTERVS": {"-": ["21"]},
    "POSTFERO": {"-": ["56"]},
    "POSTGENITVS": {"-": ["21"]},
    "POSTHABEO": {"-": ["52"]},
    "POSTHAC": {"-": ["60"]},
    "POSTICVM": {"-": ["12"]},
    "POSTICVS": {"-": ["21"]},
    "POSTILLA": {"-": ["60"]},
    "POSTIS": {"-": ["13"]},
    "POSTMERIDIANVS": {"-": ["21"]},
    "POSTMODO": {"-": ["60"]},
    "POSTMODVM": {"-": ["60"]},
    "POSTPONO": {"-": ["53"]},
    "POSTQVAM": {"-": ["82"]},
    "POSTREMO": {"-": ["60"]},
    "POSTREMVM": {"-": ["6-"]},
    "POSTRIDIE": {"-": ["60"]},
    "POSTSCENIVM": {"-": ["12"]},
    "POSTSCRIBO": {"-": ["53"]},
    "POSTVLATICIVS": {"-": ["21"]},
    "POSTVLATIO": {"-": ["13"]},
    "POSTVLATVM": {"-": ["12"]},
    "POSTVLO": {"-": ["51"]},
    "POSTVMIVS": {"N": ["12"], "A": ["21"]},
    "POSTVMVS": {"N": ["12"]},
    "POTA": {"N": ["11"]},
    "POTAMO": {"N": ["13"]},
    "POTATIO": {"-": ["13"]},
    "POTENS": {"-": ["25"]},
    "POTENTATVS": {"-": ["14"]},
    "POTENTER": {"-": ["60"]},
    "POTENTIA": {"-": ["11"], "N": ["11"]},
    "POTESTAS": {"-": ["13"]},
    "POTIO": {"1": ["13"], "2": ["54"]},
    "POTIOR": {"-": ["5M"]},
    "POTIS": {"-": ["26"]},
    "POTISSIME": {"-": ["6-"]},
    "POTISSIMVM": {"-": ["6-"]},
    "POTITII": {"N": ["12"]},
    "POTIVS": {"-": ["6&"]},
    "POTO": {"-": ["51"]},
    "POTOR": {"-": ["13"]},
    "POTORO": {"-": ["51"]},
    "POTVS": {"1": ["12"], "2": ["14"]},
    "PRAE": {"1": ["60"], "2": ["70"]},
    "PRAEACVO": {"-": ["53"]},
    "PRAEALTVS": {"-": ["21"]},
    "PRAEBEO": {"-": ["52"]},
    "PRAEBITA": {"-": ["12"]},
    "PRAECALIDVS": {"-": ["21"]},
    "PRAECANVS": {"-": ["21"]},
    "PRAECAVEO": {"-": ["52"]},
    "PRAECEDO": {"1": ["53"]},
    "PRAECELLENS": {"-": ["25"]},
    "PRAECELLO": {"-": ["53"]},
    "PRAECELSVS": {"-": ["21"]},
    "PRAECEPS": {"1": ["13"], "2": ["25"], "3": ["60"]},
    "PRAECEPTIO": {"-": ["13"]},
    "PRAECEPTIVVS": {"-": ["21"]},
    "PRAECEPTOR": {"-": ["13"]},
    "PRAECEPTRIX": {"-": ["13"]},
    "PRAECEPTVM": {"-": ["12"]},
    "PRAECERPO": {"-": ["53"]},
    "PRAECIDANEVS": {"-": ["21"]},
    "PRAECIDO": {"2": ["53"]},
    "PRAECINCTIO": {"-": ["13"]},
    "PRAECINCTVRA": {"-": ["11"]},
    "PRAECINGO": {"-": ["53"]},
    "PRAECINO": {"-": ["53"]},
    "PRAECIPIO": {"-": ["55"]},
    "PRAECIPITANTER": {"-": ["60"]},
    "PRAECIPITATIO": {"-": ["13"]},
    "PRAECIPITO": {"-": ["51"]},
    "PRAECIPVA": {"-": ["12"]},
    "PRAECIPVE": {"-": ["60"]},
    "PRAECIPVVS": {"-": ["21"]},
    "PRAECISIO": {"-": ["13"]},
    "PRAECISVS": {"-": ["21"]},
    "PRAECLARE": {"-": ["60"]},
    "PRAECLARVS": {"-": ["21"]},
    "PRAECLVDO": {"-": ["53"]},
    "PRAECLVSIO": {"-": ["13"]},
    "PRAECO": {"-": ["13"]},
    "PRAECOGITO": {"-": ["51"]},
    "PRAECOLO": {"2": ["53"]},
    "PRAECONIVM": {"-": ["12"]},
    "PRAECOQVO": {"-": ["53"]},
    "PRAECORDIA": {"-": ["12"]},
    "PRAECOX": {"-": ["25"]},
    "PRAECVRRO": {"-": ["53"]},
    "PRAECVRSOR": {"-": ["13"]},
    "PRAECVTIO": {"-": ["55"]},
    "PRAEDA": {"-": ["11"]},
    "PRAEDABVNDVS": {"-": ["21"]},
    "PRAEDATIO": {"-": ["13"]},
    "PRAEDATOR": {"-": ["13"]},
    "PRAEDATORIVS": {"-": ["21"]},
    "PRAEDICABILIS": {"-": ["24"]},
    "PRAEDICATIO": {"-": ["13"]},
    "PRAEDICO": {"1": ["51"], "2": ["53"]},
    "PRAEDICTIO": {"-": ["13"]},
    "PRAEDICTVM": {"-": ["12"]},
    "PRAEDISCO": {"-": ["53"]},
    "PRAEDITVS": {"-": ["21"]},
    "PRAEDIVES": {"-": ["22"]},
    "PRAEDIVM": {"-": ["12"]},
    "PRAEDO": {"-": ["13"]},
    "PRAEDOCEO": {"-": ["52"]},
    "PRAEDOMO": {"-": ["51"]},
    "PRAEDONVLVS": {"-": ["12"]},
    "PRAEDOR": {"-": ["5J"]},
    "PRAEDVCO": {"-": ["53"]},
    "PRAEDVCTORIVS": {"-": ["21"]},
    "PRAEDVRVS": {"-": ["21"]},
    "PRAEEO": {"1": ["56"]},
    "PRAEFATIO": {"-": ["13"]},
    "PRAEFATVS": {"1": ["14"], "2": ["21"]},
    "PRAEFECTVRA": {"-": ["11"]},
    "PRAEFECTVS": {"-": ["12"]},
    "PRAEFERO": {"-": ["56"]},
    "PRAEFEROX": {"-": ["25"]},
    "PRAEFERRATVS": {"-": ["21"]},
    "PRAEFERVIDVS": {"-": ["21"]},
    "PRAEFESTINO": {"-": ["51"]},
    "PRAEFICIO": {"-": ["55"]},
    "PRAEFIGO": {"-": ["53"]},
    "PRAEFINIO": {"-": ["54"]},
    "PRAEFLORO": {"-": ["51"]},
    "PRAEFLVO": {"-": ["53"]},
    "PRAEFOR": {"-": ["5J"]},
    "PRAEFRACTVS": {"-": ["21"]},
    "PRAEFRINGO": {"-": ["53"]},
    "PRAEFVLGEO": {"-": ["52"]},
    "PRAEFVRNIVM": {"-": ["12"]},
    "PRAEGERO": {"-": ["53"]},
    "PRAEGESTIO": {"2": ["54"]},
    "PRAEGNANS": {"-": ["25"]},
    "PRAEGRACILIS": {"-": ["24"]},
    "PRAEGRANDIS": {"-": ["23"]},
    "PRAEGRAVIS": {"-": ["24"]},
    "PRAEGRAVO": {"-": ["51"]},
    "PRAEGREDIOR": {"-": ["5N"]},
    "PRAEGVSTO": {"-": ["51"]},
    "PRAEIACEO": {"-": ["52"]},
    "PRAEIVDICIVM": {"-": ["12"]},
    "PRAEIVDICO": {"-": ["51"]},
    "PRAELABOR": {"2": ["5L"]},
    "PRAELAMBO": {"-": ["53"]},
    "PRAELARGVS": {"-": ["21"]},
    "PRAELEGO": {"1": ["51"], "2": ["53"]},
    "PRAELIGANEVS": {"-": ["21"]},
    "PRAELIGO": {"2": ["51"]},
    "PRAELONGVS": {"-": ["21"]},
    "PRAELVCEO": {"-": ["52"]},
    "PRAEMATVRVS": {"-": ["21"]},
    "PRAEMEDICATVS": {"-": ["21"]},
    "PRAEMEDITATVS": {"-": ["21"]},
    "PRAEMEDITOR": {"-": ["5J"]},
    "PRAEMETVENTER": {"-": ["60"]},
    "PRAEMETVO": {"-": ["53"]},
    "PRAEMINEO": {"-": ["52"]},
    "PRAEMIOSVS": {"-": ["21"]},
    "PRAEMITTO": {"-": ["53"]},
    "PRAEMIVM": {"-": ["12"]},
    "PRAEMONEO": {"-": ["52"]},
    "PRAEMONSTRO": {"-": ["51"]},
    "PRAEMORDEO": {"-": ["52"]},
    "PRAEMORIOR": {"-": ["5N"]},
    "PRAEMVNIO": {"2": ["54"]},
    "PRAENATO": {"-": ["51"]},
    "PRAENAVIGO": {"-": ["51"]},
    "PRAENESTE": {"N": ["13"]},
    "PRAENITEO": {"-": ["52"]},
    "PRAENOMEN": {"-": ["13"]},
    "PRAENOSCO": {"-": ["53"]},
    "PRAENOTIO": {"-": ["13"]},
    "PRAENVBILVS": {"-": ["21"]},
    "PRAENVNTIO": {"-": ["51"]},
    "PRAENVNTIVS": {"1": ["12"], "2": ["21"]},
    "PRAEOBTVRO": {"-": ["51"]},
    "PRAEOCCVPO": {"2": ["51"]},
    "PRAEOPTO": {"-": ["51"]},
    "PRAEPANDO": {"2": ["53"]},
    "PRAEPARATIO": {"-": ["13"]},
    "PRAEPARATVS": {"-": ["21"]},
    "PRAEPARO": {"2": ["51"]},
    "PRAEPEDIO": {"-": ["54"]},
    "PRAEPENDEO": {"-": ["52"]},
    "PRAEPES": {"1": ["13"], "2": ["25"]},
    "PRAEPETO": {"-": ["53"]},
    "PRAEPILATVS": {"1": ["21"]},
    "PRAEPINGVIS": {"-": ["24"]},
    "PRAEPOLLENS": {"-": ["25"]},
    "PRAEPOLLEO": {"-": ["52"]},
    "PRAEPONDERO": {"-": ["51"]},
    "PRAEPONO": {"-": ["53"]},
    "PRAEPORTO": {"-": ["51"]},
    "PRAEPOSITIO": {"-": ["13"]},
    "PRAEPOSITVS": {"1": ["12"]},
    "PRAEPOSTERO": {"1": ["51"], "2": ["60"]},
    "PRAEPOSTERVS": {"-": ["21"]},
    "PRAEPOTENS": {"-": ["25"]},
    "PRAEPOTENTES": {"-": ["13"]},
    "PRAEPROPERANTER": {"-": ["60"]},
    "PRAEPROPERE": {"-": ["60"]},
    "PRAEPVTIVM": {"-": ["12"]},
    "PRAERADIO": {"-": ["51"]},
    "PRAERAPIDVS": {"-": ["21"]},
    "PRAERIGEO": {"-": ["52"]},
    "PRAERIPIO": {"-": ["55"]},
    "PRAERODO": {"-": ["53"]},
    "PRAEROGATIVA": {"-": ["11"]},
    "PRAEROGATIVVS": {"-": ["21"]},
    "PRAERVMPO": {"-": ["53"]},
    "PRAERVPTA": {"-": ["12"]},
    "PRAERVPTVM": {"-": ["12"]},
    "PRAERVPTVS": {"-": ["21"]},
    "PRAES": {"1": ["13"], "2": ["60"]},
    "PRAESAEPE": {"-": ["13"]},
    "PRAESAEPES": {"-": ["13"]},
    "PRAESAEPIO": {"-": ["54"]},
    "PRAESAGIO": {"-": ["54"]},
    "PRAESAGIVM": {"-": ["12"]},
    "PRAESAGVS": {"-": ["21"]},
    "PRAESCISCO": {"-": ["53"]},
    "PRAESCIVS": {"-": ["21"]},
    "PRAESCRIBO": {"-": ["53"]},
    "PRAESCRIPTIO": {"-": ["13"]},
    "PRAESCRIPTVM": {"-": ["12"]},
    "PRAESECO": {"-": ["51"]},
    "PRAESEMINATIO": {"-": ["13"]},
    "PRAESENS": {"-": ["25"]},
    "PRAESENSIO": {"-": ["13"]},
    "PRAESENTANEVS": {"-": ["21"]},
    "PRAESENTIA": {"1": ["11"], "2": ["13"]},
    "PRAESENTIO": {"-": ["54"]},
    "PRAESENTO": {"-": ["51"]},
    "PRAESERO": {"3": ["53"]},
    "PRAESERTIM": {"-": ["60"]},
    "PRAESES": {"-": ["13"]},
    "PRAESIDENS": {"-": ["13"]},
    "PRAESIDEO": {"-": ["52"]},
    "PRAESIDIVM": {"-": ["12"]},
    "PRAESIGNIS": {"-": ["24"]},
    "PRAESONO": {"1": ["51"]},
    "PRAESPARGO": {"2": ["53"]},
    "PRAESPICIO": {"-": ["55"]},
    "PRAESTABILIS": {"-": ["24"]},
    "PRAESTANS": {"-": ["25"]},
    "PRAESTANTIA": {"-": ["11"]},
    "PRAESTATIO": {"-": ["13"]},
    "PRAESTERNO": {"-": ["53"]},
    "PRAESTIGIAE": {"-": ["11"]},
    "PRAESTIGIATOR": {"-": ["13"]},
    "PRAESTITVO": {"-": ["53"]},
    "PRAESTO": {"1": ["51"], "2": ["60"]},
    "PRAESTOLOR": {"-": ["5J"]},
    "PRAESTRIGIAE": {"-": ["11"]},
    "PRAESTRINGO": {"-": ["53"]},
    "PRAESTRVO": {"-": ["53"]},
    "PRAESVM": {"1": ["56"]},
    "PRAESVMO": {"-": ["53"]},
    "PRAESVMPTIO": {"-": ["13"]},
    "PRAESVMPTIOSVS": {"-": ["21"]},
    "PRAESVMPTOR": {"-": ["13"]},
    "PRAESVMPTVS": {"-": ["21"]},
    "PRAETEGO": {"-": ["53"]},
    "PRAETEMPTO": {"-": ["51"]},
    "PRAETENDO": {"-": ["53"]},
    "PRAETENTO": {"-": ["51"]},
    "PRAETEPESCO": {"-": ["53"]},
    "PRAETER": {"1": ["60"], "2": ["70"]},
    "PRAETERAGO": {"-": ["53"]},
    "PRAETEREA": {"-": ["60"]},
    "PRAETEREO": {"1": ["56"]},
    "PRAETERFERO": {"-": ["56"]},
    "PRAETERFLVO": {"-": ["53"]},
    "PRAETERGREDIOR": {"-": ["5N"]},
    "PRAETERITA": {"-": ["12"]},
    "PRAETERITVM": {"-": ["12"]},
    "PRAETERITVS": {"-": ["21"]},
    "PRAETERLABOR": {"2": ["5L"]},
    "PRAETERMEO": {"-": ["51"]},
    "PRAETERMITTO": {"-": ["53"]},
    "PRAETERO": {"-": ["53"]},
    "PRAETERQVAM": {"-": ["60"]},
    "PRAETERVECTIO": {"-": ["13"]},
    "PRAETERVEHOR": {"-": ["5L"]},
    "PRAETERVERTO": {"-": ["53"]},
    "PRAETERVOLO": {"2": ["51"]},
    "PRAETEXO": {"-": ["53"]},
    "PRAETEXTA": {"-": ["11"]},
    "PRAETEXTATVS": {"-": ["21"]},
    "PRAETEXTVM": {"-": ["12"]},
    "PRAETEXTVS": {"1": ["14"], "2": ["21"]},
    "PRAETIMEO": {"-": ["52"]},
    "PRAETOR": {"-": ["13"]},
    "PRAETORIANVS": {"-": ["21"]},
    "PRAETORIVM": {"-": ["12"]},
    "PRAETORIVS": {"1": ["12"], "2": ["21"]},
    "PRAETRACTO": {"-": ["51"]},
    "PRAETREPIDO": {"-": ["51"]},
    "PRAETREPIDVS": {"-": ["21"]},
    "PRAETVRA": {"-": ["11"]},
    "PRAEVALENS": {"-": ["25"]},
    "PRAEVALEO": {"-": ["52"]},
    "PRAEVALIDVS": {"-": ["21"]},
    "PRAEVALLO": {"-": ["51"]},
    "PRAEVARICATIO": {"-": ["13"]},
    "PRAEVARICOR": {"-": ["5J"]},
    "PRAEVEHOR": {"-": ["5L"]},
    "PRAEVENIO": {"-": ["54"]},
    "PRAEVERRO": {"-": ["53"]},
    "PRAEVERTO": {"-": ["53"]},
    "PRAEVERTOR": {"-": ["5L"]},
    "PRAEVIDEO": {"-": ["52"]},
    "PRAEVINCO": {"-": ["53"]},
    "PRAEVIVS": {"-": ["21"]},
    "PRAEVMBRO": {"-": ["51"]},
    "PRAEVRO": {"-": ["53"]},
    "PRAGMATICVS": {"1": ["12"], "2": ["21"]},
    "PRANDEO": {"-": ["52"]},
    "PRANDIVM": {"-": ["12"]},
    "PRANSITO": {"-": ["51"]},
    "PRATENSIS": {"-": ["24"]},
    "PRATVM": {"-": ["12"]},
    "PRAVE": {"-": ["60"]},
    "PRAVITAS": {"-": ["13"]},
    "PRAVVS": {"-": ["21"]},
    "PRAXITELES": {"N": ["17"]},
    "PRECARIO": {"-": ["60"]},
    "PRECARIVS": {"-": ["21"]},
    "PRECATIO": {"-": ["13"]},
    "PRECIAE": {"-": ["11"]},
    "PRECOR": {"-": ["5J"]},
    "PREHENDO": {"-": ["53"]},
    "PREHENSO": {"-": ["51"]},
    "PRELVM": {"-": ["12"]},
    "PREMO": {"-": ["53"]},
    "PRESBYTER": {"-": ["12"]},
    "PRESBYTERIVM": {"-": ["12"]},
    "PRESSE": {"-": ["60"]},
    "PRESSIO": {"-": ["13"]},
    "PRESSO": {"-": ["51"]},
    "PRESSVS": {"1": ["14"], "2": ["21"]},
    "PRESTER": {"-": ["17"]},
    "PRETIOSE": {"-": ["6&"]},
    "PRETIOSVS": {"-": ["21"]},
    "PRETIVM": {"-": ["12"]},
    "PREX": {"-": ["13"]},
    "PRIAMVS": {"N": ["12"]},
    "PRIAPVS": {"-": ["12"], "N": ["12"]},
    "PRIDEM": {"-": ["60"]},
    "PRIDIE": {"-": ["60"]},
    "PRIMA": {"1": ["12"], "2": ["60"], "N": ["11"]},
    "PRIMAEVVS": {"-": ["21"]},
    "PRIMANI": {"-": ["12"]},
    "PRIMARIVS": {"-": ["21"]},
    "PRIMIGENVS": {"-": ["21"]},
    "PRIMIPILARIS": {"-": ["13"]},
    "PRIMIPILVS": {"-": ["12"]},
    "PRIMITVS": {"-": ["60"]},
    "PRIMO": {"-": ["53", "60"], "1": ["3N"], "2": ["53"]},
    "PRIMORDIVM": {"-": ["12"]},
    "PRIMORES": {"-": ["13"]},
    "PRIMORIS": {"-": ["26"]},
    "PRIMVM": {"-": ["3N", "6-"]},
    "PRIMVS": {"-": ["3K", "13"], "N": ["12"]},
    "PRINCEPS": {"1": ["13"], "2": ["22"]},
    "PRINCIPALIS": {"-": ["24"]},
    "PRINCIPALITER": {"-": ["60"]},
    "PRINCIPATVS": {"-": ["14"]},
    "PRINCIPIALIS": {"-": ["24"]},
    "PRINCIPIVM": {"-": ["12"]},
    "PRIOR": {"-": ["3B", "2&"]},
    "PRIORES": {"-": ["13"]},
    "PRISCI": {"-": ["12"], "N": ["12"]},
    "PRISCVS": {"-": ["21"], "N": ["12"]},
    "PRISTINVS": {"-": ["21"]},
    "PRIVATIM": {"-": ["60"]},
    "PRIVATO": {"-": ["60"]},
    "PRIVATVS": {"-": ["21"]},
    "PRIVIENVS": {"-": ["12"]},
    "PRIVIGNA": {"-": ["11"]},
    "PRIVIGNVS": {"-": ["12"]},
    "PRIVILEGIVM": {"-": ["12"]},
    "PRIVO": {"-": ["51"]},
    "PRIVS": {"-": ["3E", "6-"]},
    "PRIVSQVAM": {"-": ["82"]},
    "PRIVVS": {"-": ["21"]},
    "PRO": {"1": ["70"], "2": ["90"], "3": ["70"]},
    "PROAGORVS": {"-": ["12"]},
    "PROAVIA": {"-": ["11"]},
    "PROAVVS": {"-": ["12"]},
    "PROBABILIS": {"-": ["24"]},
    "PROBABILITAS": {"-": ["13"]},
    "PROBATIO": {"-": ["13"]},
    "PROBATOR": {"-": ["13"]},
    "PROBATVS": {"-": ["21"]},
    "PROBE": {"-": ["60"]},
    "PROBITAS": {"-": ["13"]},
    "PROBO": {"-": ["51"]},
    "PROBOSCIS": {"-": ["17"]},
    "PROBRO": {"-": ["51"]},
    "PROBROSVS": {"-": ["21"]},
    "PROBRVM": {"-": ["12"]},
    "PROBVS": {"-": ["21"], "N": ["12"]},
    "PROCACITER": {"-": ["6&"]},
    "PROCAS": {"N": ["17"]},
    "PROCAX": {"-": ["25"]},
    "PROCEDO": {"1": ["53"]},
    "PROCELLA": {"-": ["11"]},
    "PROCELLO": {"-": ["53"]},
    "PROCELLOSVS": {"-": ["21"]},
    "PROCER": {"-": ["13"]},
    "PROCERES": {"-": ["13"]},
    "PROCERITAS": {"-": ["13"]},
    "PROCERVS": {"-": ["21"]},
    "PROCESSIO": {"-": ["13"]},
    "PROCESSVS": {"-": ["14"]},
    "PROCIDO": {"1": ["53"]},
    "PROCINCTVS": {"1": ["14"], "2": ["21"]},
    "PROCLAMO": {"-": ["51"]},
    "PROCLINATIO": {"-": ["13"]},
    "PROCLINO": {"-": ["51"]},
    "PROCLIVE": {"1": ["13"], "2": ["60"]},
    "PROCLIVIS": {"-": ["24"]},
    "PROCLIVITAS": {"-": ["13"]},
    "PROCLIVITER": {"-": ["60"]},
    "PROCLIVVS": {"-": ["21"]},
    "PROCONSVL": {"-": ["13"]},
    "PROCONSVLARIS": {"-": ["24"]},
    "PROCONSVLATVS": {"-": ["14"]},
    "PROCRASTINO": {"-": ["51"]},
    "PROCREATIO": {"-": ["13"]},
    "PROCREO": {"-": ["51"]},
    "PROCRESCO": {"-": ["53"]},
    "PROCVBO": {"-": ["51"]},
    "PROCVDO": {"-": ["53"]},
    "PROCVL": {"-": ["60"]},
    "PROCVLCATIO": {"-": ["13"]},
    "PROCVLCO": {"-": ["51"]},
    "PROCVLVS": {"N": ["12"]},
    "PROCVMBO": {"-": ["53"]},
    "PROCVRATIO": {"-": ["13"]},
    "PROCVRATIVNCVLA": {"-": ["11"]},
    "PROCVRATOR": {"-": ["13"]},
    "PROCVRO": {"-": ["51"]},
    "PROCVRRO": {"-": ["53"]},
    "PROCVRSVS": {"-": ["14"]},
    "PROCVRVVS": {"-": ["21"]},
    "PROCVS": {"-": ["12"]},
    "PRODEO": {"1": ["56"]},
    "PRODIGE": {"-": ["60"]},
    "PRODIGENTIA": {"-": ["11"]},
    "PRODIGIOSVS": {"-": ["21"]},
    "PRODIGIVM": {"-": ["12"]},
    "PRODIGO": {"-": ["53"]},
    "PRODIGVS": {"-": ["21"]},
    "PRODITIO": {"1": ["13"], "2": ["13"]},
    "PRODITOR": {"-": ["13"]},
    "PRODO": {"-": ["53"]},
    "PRODOCEO": {"-": ["52"]},
    "PRODVCO": {"-": ["53"]},
    "PRODVCTA": {"-": ["12"]},
    "PRODVCTIO": {"-": ["13"]},
    "PRODVCTVS": {"-": ["21"]},
    "PROELIATOR": {"-": ["13"]},
    "PROELIOR": {"-": ["5J"]},
    "PROELIVM": {"-": ["12"]},
    "PROFANO": {"1": ["51"], "2": ["51"]},
    "PROFANVS": {"-": ["21"]},
    "PROFECTIO": {"-": ["13"]},
    "PROFECTO": {"-": ["60"]},
    "PROFECTVS": {"-": ["14"]},
    "PROFERO": {"-": ["56"]},
    "PROFESSIO": {"-": ["13"]},
    "PROFESSOR": {"-": ["13"]},
    "PROFESSORIVS": {"-": ["21"]},
    "PROFESSVS": {"-": ["21"]},
    "PROFESTVS": {"-": ["21"]},
    "PROFICIO": {"-": ["55"]},
    "PROFICISCOR": {"-": ["5L"]},
    "PROFITEOR": {"-": ["5K"]},
    "PROFLIGATOR": {"-": ["13"]},
    "PROFLIGATVS": {"-": ["21"]},
    "PROFLIGO": {"-": ["51"]},
    "PROFLO": {"-": ["51"]},
    "PROFLVENS": {"-": ["25"]},
    "PROFLVENTER": {"-": ["60"]},
    "PROFLVO": {"-": ["53"]},
    "PROFLVVIVM": {"-": ["12"]},
    "PROFLVVIVS": {"-": ["21"]},
    "PROFOR": {"-": ["5J"]},
    "PROFVGIO": {"-": ["55"]},
    "PROFVGVS": {"1": ["12"], "2": ["21"]},
    "PROFVNDO": {"2": ["53"]},
    "PROFVNDVM": {"-": ["12"]},
    "PROFVNDVS": {"-": ["21"]},
    "PROFVSE": {"-": ["6&"]},
    "PROFVSIO": {"-": ["13"]},
    "PROFVSVS": {"-": ["21"]},
    "PROGENER": {"-": ["12"]},
    "PROGENERO": {"-": ["51"]},
    "PROGENIES": {"-": ["15"]},
    "PROGERO": {"-": ["53"]},
    "PROGIGNO": {"-": ["53"]},
    "PROGNATVS": {"-": ["21"]},
    "PROGNE": {"-": ["17"], "N": ["11"]},
    "PROGREDIOR": {"-": ["5N"]},
    "PROGRESSIO": {"-": ["13"]},
    "PROGRESSVS": {"1": ["14"], "2": ["21"]},
    "PROGYMNASTES": {"-": ["17"]},
    "PROHIBEO": {"-": ["52"]},
    "PROHIBITA": {"-": ["12"]},
    "PROHIBITIO": {"-": ["13"]},
    "PROICIO": {"-": ["55"]},
    "PROIECTVRA": {"-": ["11"]},
    "PROIECTVS": {"1": ["14"], "2": ["21"]},
    "PROINDE": {"-": ["60"]},
    "PROLABOR": {"2": ["5L"]},
    "PROLATIO": {"-": ["13"]},
    "PROLATO": {"-": ["51"]},
    "PROLECTO": {"-": ["51"]},
    "PROLES": {"-": ["13"]},
    "PROLETARIVS": {"-": ["21"]},
    "PROLEVO": {"1": ["51"]},
    "PROLICEO": {"-": ["52"]},
    "PROLICIO": {"-": ["55"]},
    "PROLIXE": {"-": ["60"]},
    "PROLIXVS": {"-": ["21"]},
    "PROLOGVS": {"-": ["12"]},
    "PROLONGO": {"-": ["51"]},
    "PROLOQVOR": {"-": ["5L"]},
    "PROLVBIVM": {"-": ["12"]},
    "PROLVDO": {"-": ["53"]},
    "PROLVO": {"-": ["53"]},
    "PROLVVIES": {"-": ["15"]},
    "PROMEREO": {"-": ["52"]},
    "PROMEREOR": {"-": ["5K"]},
    "PROMERITVM": {"-": ["12"]},
    "PROMETHEVS": {"N": ["17"], "A": ["21"]},
    "PROMINENS": {"1": ["13"], "2": ["25"]},
    "PROMINENTIA": {"-": ["11"]},
    "PROMINEO": {"-": ["52"]},
    "PROMISCVE": {"-": ["60"]},
    "PROMISCVVM": {"-": ["12"]},
    "PROMISCVVS": {"-": ["21"]},
    "PROMISSVM": {"-": ["12"]},
    "PROMISSVS": {"-": ["21"]},
    "PROMITTO": {"-": ["53"]},
    "PROMO": {"-": ["53"]},
    "PROMONTORIVM": {"-": ["12"]},
    "PROMOVEO": {"-": ["52"]},
    "PROMPTE": {"-": ["60"]},
    "PROMPTVARIVS": {"-": ["21"]},
    "PROMPTVS": {"1": ["14"], "2": ["21"]},
    "PROMVLGO": {"-": ["51"]},
    "PROMVS": {"-": ["12"]},
    "PROMVTVVS": {"-": ["21"]},
    "PRONAVS": {"-": ["17"]},
    "PRONE": {"-": ["60"]},
    "PRONEPOS": {"-": ["13"]},
    "PRONEPTIS": {"-": ["13"]},
    "PRONVBA": {"-": ["11"]},
    "PRONVBVS": {"-": ["21"]},
    "PRONVNTIATIO": {"-": ["13"]},
    "PRONVNTIO": {"-": ["51"]},
    "PRONVRVS": {"-": ["14"]},
    "PRONVS": {"-": ["21"]},
    "PROOEMIVM": {"-": ["12"]},
    "PROPAGATIO": {"-": ["13"]},
    "PROPAGATOR": {"-": ["13"]},
    "PROPAGO": {"1": ["13"], "2": ["51"]},
    "PROPALAM": {"-": ["60"]},
    "PROPATVLVM": {"-": ["12"]},
    "PROPATVLVS": {"-": ["21"]},
    "PROPE": {"1": ["60"], "2": ["70"]},
    "PROPEDIEM": {"-": ["60"]},
    "PROPELLO": {"-": ["53"]},
    "PROPEMODVM": {"-": ["60"]},
    "PROPENDEO": {"-": ["52"]},
    "PROPENSIO": {"-": ["13"]},
    "PROPENSVS": {"-": ["21"]},
    "PROPERANS": {"-": ["25"]},
    "PROPERANTER": {"-": ["60"]},
    "PROPERANTIA": {"-": ["11"]},
    "PROPERATIO": {"-": ["13"]},
    "PROPERATO": {"-": ["60"]},
    "PROPERATVS": {"-": ["21"]},
    "PROPERE": {"-": ["60"]},
    "PROPERIPES": {"-": ["25"]},
    "PROPERO": {"-": ["51"]},
    "PROPERVS": {"-": ["21"]},
    "PROPINATIO": {"-": ["13"]},
    "PROPINO": {"-": ["51"]},
    "PROPINQVI": {"-": ["12"]},
    "PROPINQVITAS": {"-": ["13"]},
    "PROPINQVO": {"-": ["51"]},
    "PROPINQVVM": {"-": ["12"]},
    "PROPINQVVS": {"-": ["21"], "N": ["12"]},
    "PROPIOR": {"-": ["2&"]},
    "PROPITIO": {"-": ["51"]},
    "PROPITIVS": {"-": ["21"]},
    "PROPIVS": {"-": ["70"]},
    "PROPNIGEVM": {"-": ["17"]},
    "PROPOLA": {"-": ["11"]},
    "PROPOLLVO": {"-": ["53"]},
    "PROPONO": {"-": ["53"]},
    "PROPORRO": {"-": ["60"]},
    "PROPORTIO": {"-": ["13"]},
    "PROPOSITIO": {"-": ["13"]},
    "PROPOSITVM": {"-": ["12"]},
    "PROPRAETOR": {"-": ["13"]},
    "PROPRIATIM": {"-": ["60"]},
    "PROPRIE": {"-": ["60"]},
    "PROPRIETAS": {"-": ["13"]},
    "PROPRITIM": {"-": ["60"]},
    "PROPRIVM": {"-": ["12"]},
    "PROPRIVS": {"-": ["21"]},
    "PROPTER": {"1": ["60"], "2": ["70"]},
    "PROPTEREA": {"-": ["60"]},
    "PROPVGNACVLVM": {"-": ["12"]},
    "PROPVGNATOR": {"-": ["13"]},
    "PROPVGNO": {"-": ["51"]},
    "PROPVLSO": {"-": ["51"]},
    "PROQVAESTOR": {"-": ["13"]},
    "PROQVAM": {"-": ["66"]},
    "PRORA": {"-": ["11"]},
    "PROREPO": {"-": ["53"]},
    "PRORIPIO": {"-": ["55"]},
    "PRORIS": {"-": ["13"]},
    "PRORITO": {"-": ["51"]},
    "PROROGO": {"-": ["51"]},
    "PRORSVM": {"-": ["60"]},
    "PRORSVS": {"1": ["21"], "2": ["60"]},
    "PRORVMPO": {"-": ["53"]},
    "PRORVO": {"-": ["53"]},
    "PROSAPIA": {"-": ["11"]},
    "PROSCAENIVM": {"-": ["12"]},
    "PROSCENIVM": {"-": ["12"]},
    "PROSCINDO": {"-": ["53"]},
    "PROSCRIBO": {"-": ["53"]},
    "PROSCRIPTIO": {"-": ["13"]},
    "PROSECO": {"-": ["51"]},
    "PROSEMINO": {"-": ["51"]},
    "PROSEQVOR": {"-": ["5L"]},
    "PROSERO": {"2": ["53"], "3": ["53"]},
    "PROSERPINA": {"N": ["11"]},
    "PROSEVCHA": {"-": ["11"]},
    "PROSILIO": {"-": ["54"]},
    "PROSITVS": {"-": ["21"]},
    "PROSLAMBANOMENOS": {"-": ["17"]},
    "PROSOCER": {"-": ["12"]},
    "PROSPECTO": {"-": ["51"]},
    "PROSPECTVS": {"-": ["14"]},
    "PROSPECVLOR": {"-": ["5J"]},
    "PROSPER": {"-": ["21"]},
    "PROSPERA": {"-": ["12"]},
    "PROSPERE": {"-": ["60"]},
    "PROSPERITAS": {"-": ["13"]},
    "PROSPERO": {"-": ["51"]},
    "PROSPICIO": {"-": ["55"]},
    "PROSTAS": {"-": ["17"]},
    "PROSTERNO": {"-": ["53"]},
    "PROSTITVO": {"-": ["53"]},
    "PROSTITVTA": {"-": ["11"]},
    "PROSTITVTVS": {"-": ["21"]},
    "PROSTO": {"-": ["51"]},
    "PROSTYLOS": {"-": ["27"]},
    "PROSVBIGO": {"-": ["51"]},
    "PROSVM": {"1": ["56"]},
    "PROTECTVRA": {"-": ["11"]},
    "PROTEGO": {"-": ["53"]},
    "PROTELVM": {"-": ["12"]},
    "PROTENDO": {"-": ["53"]},
    "PROTENTVS": {"-": ["21"]},
    "PROTERO": {"-": ["53"]},
    "PROTERREO": {"-": ["52"]},
    "PROTERVE": {"-": ["60"]},
    "PROTERVITAS": {"-": ["13"]},
    "PROTERVVS": {"-": ["21"]},
    "PROTHYRVM": {"-": ["17"]},
    "PROTINVS": {"-": ["60"]},
    "PROTOLLO": {"-": ["53"]},
    "PROTRAHO": {"-": ["53"]},
    "PROTRITVS": {"-": ["21"]},
    "PROTROPVM": {"-": ["12"]},
    "PROTRVDO": {"-": ["53"]},
    "PROTVRBO": {"2": ["51"]},
    "PROVECTVS": {"1": ["14"], "2": ["21"]},
    "PROVEHO": {"-": ["53"]},
    "PROVENIO": {"-": ["54"]},
    "PROVENTVS": {"-": ["14"]},
    "PROVERBIVM": {"-": ["12"]},
    "PROVERTO": {"-": ["53"]},
    "PROVIDENS": {"-": ["25"]},
    "PROVIDENTER": {"-": ["60"]},
    "PROVIDENTIA": {"-": ["11"]},
    "PROVIDEO": {"-": ["52"]},
    "PROVIDVS": {"-": ["21"]},
    "PROVINCIA": {"-": ["11"]},
    "PROVINCIALIS": {"1": ["13"], "2": ["24"]},
    "PROVINCO": {"-": ["53"]},
    "PROVISIO": {"-": ["13"]},
    "PROVISO": {"1": ["53"], "2": ["60"]},
    "PROVISOR": {"-": ["13"]},
    "PROVISVS": {"-": ["14"]},
    "PROVIVO": {"-": ["53"]},
    "PROVOCATIO": {"-": ["13"]},
    "PROVOCO": {"-": ["51"]},
    "PROVOLO": {"2": ["51"]},
    "PROVOLVO": {"-": ["53"]},
    "PROVOMO": {"-": ["53"]},
    "PROVT": {"-": ["66"]},
    "PROXENETA": {"-": ["11"]},
    "PROXIME": {"-": ["70"]},
    "PROXIMITAS": {"-": ["13"]},
    "PROXIMVM": {"-": ["12"]},
    "PRVDENS": {"-": ["25"]},
    "PRVDENTER": {"-": ["60"]},
    "PRVDENTIA": {"-": ["11"]},
    "PRVINA": {"-": ["11"]},
    "PRVINOSVS": {"-": ["21"]},
    "PRVNA": {"-": ["11"]},
    "PRVNVM": {"-": ["12"]},
    "PRVNVS": {"-": ["12"]},
    "PRVRIGO": {"-": ["13"]},
    "PRVRIO": {"-": ["54"]},
    "PRYTANEVM": {"-": ["12"]},
    "PRYTANIS": {"-": ["13"], "N": ["13"]},
    "PSALLO": {"-": ["53"]},
    "PSALTRIA": {"-": ["11"]},
    "PSEVDOTHYRVM": {"-": ["12"]},
    "PSITTACVS": {"-": ["12"]},
    "PSYLLVS": {"-": ["12"]},
    "PSYTHIA": {"-": ["11"]},
    "PTEROMA": {"-": ["17"]},
    "PTERYGOMA": {"-": ["17"]},
    "PTISANARIVM": {"-": ["12"]},
    "PTOLEMAEVS": {"N": ["12"]},
    "PVBENS": {"-": ["25"]},
    "PVBERTAS": {"-": ["13"]},
    "PVBES": {"1": ["13"], "2": ["26"]},
    "PVBESCO": {"-": ["53"]},
    "PVBLICA": {"-": ["11"]},
    "PVBLICANVS": {"1": ["12"], "2": ["21"]},
    "PVBLICATIO": {"-": ["13"]},
    "PVBLICE": {"-": ["60"]},
    "PVBLICO": {"-": ["51"]},
    "PVBLICOLA": {"N": ["11"]},
    "PVBLICVM": {"-": ["12"]},
    "PVBLICVS": {"1": ["12"], "2": ["21"], "N": ["12"], "A": ["21"]},
    "PVBLILIVS": {"N": ["12"]},
    "PVBLIVS": {"N": ["12"]},
    "PVDENDA": {"-": ["12"]},
    "PVDENDVM": {"-": ["90"]},
    "PVDENDVS": {"-": ["21"]},
    "PVDENS": {"-": ["25"], "N": ["13"]},
    "PVDENTER": {"-": ["60"]},
    "PVDEO": {"-": ["52"]},
    "PVDIBVNDVS": {"-": ["21"]},
    "PVDICE": {"-": ["60"]},
    "PVDICITIA": {"-": ["11"], "N": ["11"]},
    "PVDICVS": {"-": ["21"]},
    "PVDOR": {"-": ["13"], "N": ["13"]},
    "PVELLA": {"-": ["11"], "N": ["11"]},
    "PVELLARIS": {"-": ["24"]},
    "PVELLVLA": {"-": ["11"]},
    "PVELLVS": {"-": ["12"]},
    "PVER": {"-": ["12"]},
    "PVERILIS": {"-": ["24"]},
    "PVERILITAS": {"-": ["13"]},
    "PVERILITER": {"-": ["60"]},
    "PVERITIA": {"-": ["11"]},
    "PVERPERA": {"-": ["11"]},
    "PVERPERIVM": {"-": ["12"]},
    "PVERPERVS": {"-": ["21"]},
    "PVERVLVS": {"-": ["12"]},
    "PVGA": {"-": ["11"]},
    "PVGIL": {"-": ["13"]},
    "PVGILLAR": {"-": ["13"]},
    "PVGILLARES": {"-": ["13"]},
    "PVGILLARIS": {"-": ["24"]},
    "PVGILLVS": {"-": ["12"]},
    "PVGIO": {"-": ["13"]},
    "PVGNA": {"-": ["11"]},
    "PVGNACITAS": {"-": ["13"]},
    "PVGNACITER": {"-": ["6&"]},
    "PVGNANS": {"-": ["25"]},
    "PVGNAX": {"-": ["25"]},
    "PVGNO": {"-": ["51"]},
    "PVGNVS": {"-": ["12"]},
    "PVLCHER": {"-": ["21"], "C": ["12"], "N": ["12"]},
    "PVLCHRE": {"-": ["60"]},
    "PVLCHRITVDO": {"-": ["13"]},
    "PVLEX": {"-": ["13"]},
    "PVLLARIVS": {"1": ["12"], "2": ["21"]},
    "PVLLATVS": {"-": ["21"]},
    "PVLLVLO": {"-": ["51"]},
    "PVLLVM": {"2": ["12"], "-": ["12"]},
    "PVLLVS": {"1": ["12"], "2": ["21"]},
    "PVLMENTARIVM": {"-": ["12"]},
    "PVLMENTVM": {"-": ["12"]},
    "PVLMO": {"-": ["13"]},
    "PVLPA": {"-": ["11"]},
    "PVLPITVM": {"-": ["12"]},
    "PVLS": {"-": ["13"]},
    "PVLSO": {"-": ["51"]},
    "PVLSVS": {"-": ["14"]},
    "PVLTO": {"-": ["51"]},
    "PVLVEREVS": {"-": ["21"]},
    "PVLVERVLENTVS": {"-": ["21"]},
    "PVLVILLVS": {"N": ["12"], "-": ["12"]},
    "PVLVINAR": {"-": ["13"]},
    "PVLVINATVS": {"-": ["21"]},
    "PVLVINVS": {"-": ["12"]},
    "PVLVIS": {"-": ["13"]},
    "PVLVISCVLVS": {"-": ["12"]},
    "PVMEX": {"-": ["13"]},
    "PVMICO": {"-": ["51"]},
    "PVMICOSVS": {"-": ["21"]},
    "PVMILIO": {"-": ["13"]},
    "PVNCTIVNCVLA": {"-": ["11"]},
    "PVNCTVM": {"-": ["12"]},
    "PVNCTVS": {"1": ["14"], "2": ["21"]},
    "PVNGO": {"-": ["53"]},
    "PVNICEVS": {"-": ["21"], "A": ["21"]},
    "PVNICIVS": {"-": ["21"]},
    "PVNICVM": {"-": ["12"]},
    "PVNICVS": {"-": ["21"], "N": ["21"], "A": ["21"]},
    "PVNIO": {"-": ["54"]},
    "PVNIOR": {"-": ["5M"]},
    "PVPA": {"-": ["11"]},
    "PVPILLA": {"-": ["11"]},
    "PVPILLARIS": {"-": ["24"]},
    "PVPILLVS": {"-": ["12"]},
    "PVPPIS": {"-": ["13"]},
    "PVPVLA": {"-": ["11"]},
    "PVPVLVS": {"-": ["12"]},
    "PVRE": {"-": ["60"]},
    "PVRGAMENTVM": {"-": ["12"]},
    "PVRGATIO": {"-": ["13"]},
    "PVRGATVS": {"-": ["21"]},
    "PVRGO": {"-": ["51"]},
    "PVRITER": {"-": ["60"]},
    "PVRPVRA": {"-": ["11"]},
    "PVRPVRATVS": {"1": ["12"], "2": ["21"]},
    "PVRPVREVS": {"-": ["21"]},
    "PVRVLENTVS": {"-": ["21"]},
    "PVRVS": {"-": ["21"]},
    "PVS": {"-": ["13"]},
    "PVSILLVM": {"1": ["12"], "2": ["60"]},
    "PVSILLVS": {"-": ["21"]},
    "PVSIO": {"-": ["13"], "N": ["13"]},
    "PVSTVLA": {"-": ["11"]},
    "PVSVLA": {"-": ["11"]},
    "PVTAMEN": {"-": ["13"]},
    "PVTATOR": {"-": ["13"]},
    "PVTEAL": {"-": ["13"], "N": ["13"]},
    "PVTEALIS": {"-": ["24"]},
    "PVTEO": {"-": ["52"]},
    "PVTEOLI": {"N": ["12"]},
    "PVTESCO": {"-": ["53"]},
    "PVTEVS": {"-": ["12"]},
    "PVTIDE": {"-": ["60"]},
    "PVTIDVS": {"-": ["21"]},
    "PVTO": {"-": ["51"]},
    "PVTOR": {"-": ["13"]},
    "PVTREFACIO": {"-": ["55"]},
    "PVTREO": {"-": ["52"]},
    "PVTRESCO": {"-": ["53"]},
    "PVTRIDVS": {"-": ["21"]},
    "PVTRIS": {"-": ["24"]},
    "PYCNOSTYLOS": {"-": ["27"]},
    "PYGARGVS": {"-": ["12"]},
    "PYLAEMENES": {"N": ["13"]},
    "PYRA": {"-": ["11"]},
    "PYRAGRO": {"N": ["17"]},
    "PYRAMIS": {"-": ["17"], "N": ["13"]},
    "PYRENAEVS": {"N": ["21"], "A": ["21"]},
    "PYRETHRVM": {"-": ["12"]},
    "PYROPVS": {"-": ["12"]},
    "PYTHAGORAS": {"N": ["17"]},
    "PYTHAVLA": {"-": ["17"]},
    "PYTHIA": {"N": ["12", "11"], "O": ["11"]},
    "PYTHICVS": {"N": ["21"], "A": ["21"]},
    "PYTISMA": {"-": ["17"]},
    "PYXIS": {"-": ["17"]},
    "QVA": {"1": ["66"], "2": ["67"], "3": ["60"]},
    "QVACVMQVE": {"1": ["66"], "2": ["60"]},
    "QVADRA": {"-": ["11"]},
    "QVADRAGENARIVS": {"-": ["21"]},
    "QVADRAGENI": {"-": ["33"]},
    "QVADRAGESIMA": {"-": ["11"]},
    "QVADRAGESIMVS": {"-": ["32"]},
    "QVADRAGIES": {"-": ["36"]},
    "QVADRAGINTA": {"-": ["31"]},
    "QVADRANS": {"-": ["13"]},
    "QVADRANTAL": {"-": ["13"]},
    "QVADRANTARIVS": {"-": ["21"]},
    "QVADRATIO": {"-": ["13"]},
    "QVADRATVM": {"-": ["12"]},
    "QVADRATVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "QVADRIANGVLVS": {"-": ["21"]},
    "QVADRIDENS": {"-": ["25"]},
    "QVADRIDVVM": {"-": ["12"]},
    "QVADRIENNIVM": {"-": ["12"]},
    "QVADRIFARIAM": {"-": ["60"]},
    "QVADRIFIDVS": {"-": ["21"]},
    "QVADRIFLVVIVM": {"-": ["12"]},
    "QVADRIGA": {"-": ["11"]},
    "QVADRIIVGI": {"-": ["12"]},
    "QVADRIIVGIS": {"-": ["24"]},
    "QVADRIIVGVS": {"-": ["21"]},
    "QVADRIMVS": {"-": ["21"]},
    "QVADRINGENI": {"-": ["34", "21"]},
    "QVADRINGENTESIMVS": {"-": ["32"]},
    "QVADRINGENTI": {"-": ["31"]},
    "QVADRINGENTIES": {"-": ["36"]},
    "QVADRINI": {"-": ["33"]},
    "QVADRIPARTITVS": {"-": ["21"]},
    "QVADRIREMIS": {"1": ["13"], "2": ["24"]},
    "QVADRIVIVM": {"-": ["12"]},
    "QVADRO": {"-": ["51"]},
    "QVADRVPEDVS": {"-": ["21"]},
    "QVADRVPES": {"1": ["13"], "2": ["25"]},
    "QVADRVPLATOR": {"-": ["13"]},
    "QVADRVPLEX": {"-": ["34", "13"]},
    "QVADRVPLO": {"1": ["51"], "2": ["60"]},
    "QVADRVPLVM": {"-": ["12"]},
    "QVAERITO": {"-": ["51"]},
    "QVAERO": {"-": ["53"]},
    "QVAESITOR": {"-": ["13"]},
    "QVAESITVM": {"-": ["12"]},
    "QVAESITVS": {"1": ["14"], "2": ["21"]},
    "QVAESO": {"-": ["56"]},
    "QVAESTIO": {"-": ["13"]},
    "QVAESTIVNCVLA": {"-": ["11"]},
    "QVAESTOR": {"-": ["13"]},
    "QVAESTORIVM": {"-": ["12"]},
    "QVAESTORIVS": {"1": ["12"], "2": ["21"]},
    "QVAESTVARIA": {"-": ["11"]},
    "QVAESTVOSE": {"-": ["6-"]},
    "QVAESTVOSVS": {"-": ["21"]},
    "QVAESTVRA": {"-": ["11"]},
    "QVAESTVS": {"-": ["14"]},
    "QVALIBET": {"-": ["60"]},
    "QVALIS": {"1": ["46"], "2": ["47"]},
    "QVALISCVMQVE": {"1": ["46"], "2": ["48"]},
    "QVALITAS": {"-": ["13"]},
    "QVALITER": {"1": ["66"], "2": ["67"]},
    "QVALVM": {"-": ["12"]},
    "QVALVS": {"-": ["12"]},
    "QVAM": {"1": ["66"], "2": ["67"]},
    "QVAMDIV": {"1": ["66"], "2": ["67"]},
    "QVAMDVDVM": {"-": ["67"]},
    "QVAMLIBET": {"1": ["66"], "2": ["60"]},
    "QVAMOBREM": {"1": ["66"], "2": ["67"]},
    "QVAMPLVRES": {"-": ["22"]},
    "QVAMPRIMVM": {"-": ["60"]},
    "QVAMQVAM": {"1": ["66"], "2": ["82"], "3": ["60"]},
    "QVAMVIS": {"1": ["66"], "2": ["82"]},
    "QVANAM": {"-": ["67"]},
    "QVANDANTENVS": {"-": ["60"]},
    "QVANDO": {"1": ["66"], "2": ["67"], "3": ["82"], "4": ["60"]},
    "QVANDOCVMQVE": {"1": ["66"], "2": ["60"]},
    "QVANDOQVE": {"1": ["60"], "2": ["82"]},
    "QVANDOQVIDEM": {"-": ["82"]},
    "QVANTITAS": {"-": ["13"]},
    "QVANTO": {"1": ["66"], "2": ["67"]},
    "QVANTOPERE": {"1": ["66"], "2": ["67"]},
    "QVANTVLVM": {"1": ["46"], "2": ["47"], "3": ["66"], "4": ["67"]},
    "QVANTVLVS": {"1": ["46"], "2": ["47"]},
    "QVANTVLVSCVMQVE": {"1": ["46"], "2": ["48"]},
    "QVANTVLVSLIBET": {"1": ["46"], "2": ["48"]},
    "QVANTVM": {"1": ["46"], "2": ["47"], "3": ["66"], "4": ["67"]},
    "QVANTVMCVMQVE": {"-": ["66"]},
    "QVANTVMLIBET": {"1": ["48"], "2": ["60"]},
    "QVANTVMVIS": {"1": ["60"], "2": ["82"]},
    "QVANTVS": {"1": ["46"], "2": ["47"]},
    "QVANTVSCVMQVE": {"1": ["46"], "2": ["48"]},
    "QVANTVSLIBET": {"-": ["48"]},
    "QVANTVSVIS": {"-": ["48"]},
    "QVAPROPTER": {"1": ["66", "6-"], "2": ["67"]},
    "QVAQVAVERSVM": {"-": ["60"]},
    "QVAQVE": {"-": ["60"]},
    "QVARE": {"1": ["66"], "2": ["67"]},
    "QVARTADECIMANI": {"-": ["12"]},
    "QVARTANA": {"-": ["11"]},
    "QVARTANVS": {"-": ["21"]},
    "QVARTARIVS": {"-": ["12"]},
    "QVARTO": {"-": ["35"]},
    "QVARTVM": {"1": ["12"], "2": ["36"]},
    "QVARTVS": {"-": ["32"]},
    "QVASI": {"1": ["66"], "2": ["82"]},
    "QVASILLVS": {"-": ["12"]},
    "QVASSATIO": {"-": ["13"]},
    "QVASSO": {"-": ["51"]},
    "QVASSVS": {"1": ["14"], "2": ["21"]},
    "QVATENVS": {"1": ["66"], "2": ["67"], "3": ["82"]},
    "QVATER": {"-": ["36"]},
    "QVATERNI": {"-": ["33"]},
    "QVATIO": {"-": ["55"]},
    "QVATRIDVVM": {"-": ["12"]},
    "QVATVOR": {"-": ["31"]},
    "QVATVORDECIM": {"-": ["31"]},
    "QVATVORVIR": {"-": ["12"]},
    "QVAVIS": {"-": ["60"]},
    "QVE": {"-": ["81"]},
    "QVEMADMODVM": {"1": ["66"], "2": ["67"]},
    "QVEO": {"-": ["56"]},
    "QVERCANEVS": {"-": ["21"]},
    "QVERCEVS": {"-": ["21"]},
    "QVERCVS": {"-": ["16"]},
    "QVERELA": {"-": ["11"]},
    "QVERIMONIA": {"-": ["11"]},
    "QVERITOR": {"-": ["5J"]},
    "QVERNEVS": {"-": ["21"]},
    "QVERNVS": {"-": ["21"]},
    "QVEROR": {"-": ["5L"]},
    "QVERQVETVM": {"-": ["12"]},
    "QVERVLVS": {"-": ["21"]},
    "QVESTVS": {"-": ["14"]},
    "QVI": {"1": ["46"], "2": ["66"], "3": ["67"], "4": ["60"]},
    "QVIA": {"-": ["82"]},
    "QVIANAM": {"-": ["67"]},
    "QVICVMQVE": {"1": ["46"], "2": ["48"]},
    "QVIDAM": {"-": ["48"]},
    "QVIDEM": {"-": ["60"]},
    "QVIDNI": {"-": ["69", "6-"]},
    "QVIES": {"-": ["13"]},
    "QVIESCO": {"-": ["53"]},
    "QVIETE": {"-": ["60"]},
    "QVIETO": {"-": ["51"]},
    "QVIETVM": {"-": ["12"]},
    "QVIETVS": {"-": ["21"], "N": ["12"]},
    "QVILIBET": {"-": ["48"]},
    "QVIN": {"1": ["60"], "2": ["82"], "3": ["69", "6-"]},
    "QVINARIVS": {"-": ["21"]},
    "QVINCVNX": {"-": ["13"]},
    "QVINDECIES": {"-": ["36"]},
    "QVINDECIM": {"-": ["31"]},
    "QVINDECIMVIRALIS": {"-": ["24"]},
    "QVINDECIMVIRI": {"-": ["12"]},
    "QVINGENARIVS": {"-": ["21"]},
    "QVINGENI": {"-": ["33"]},
    "QVINGENTESIMVS": {"-": ["32"]},
    "QVINGENTI": {"-": ["31"]},
    "QVINI": {"-": ["33"]},
    "QVINIDENI": {"-": ["33"]},
    "QVINQVAGENARIVS": {"-": ["21"]},
    "QVINQVAGENI": {"-": ["33"]},
    "QVINQVAGESIMA": {"-": ["11"]},
    "QVINQVAGESIMVS": {"-": ["32"]},
    "QVINQVAGINTA": {"-": ["31"]},
    "QVINQVATRIA": {"-": ["13"]},
    "QVINQVE": {"-": ["31"]},
    "QVINQVENNALIS": {"-": ["24"]},
    "QVINQVENNIS": {"-": ["24"]},
    "QVINQVENNIVM": {"-": ["12"]},
    "QVINQVEREMIS": {"1": ["13"], "2": ["24"]},
    "QVINQVEVIR": {"-": ["12"]},
    "QVINQVIES": {"-": ["36"]},
    "QVINQVIPLICO": {"-": ["51"]},
    "QVINTADECIMANI": {"-": ["12"]},
    "QVINTANI": {"-": ["12"]},
    "QVINTARIVS": {"-": ["21"]},
    "QVINTILIS": {"A": ["24"], "N": ["13"]},
    "QVINTIVS": {"N": ["12"]},
    "QVINTVM": {"-": ["36"]},
    "QVINTVS": {"-": ["32"], "N": ["12"]},
    "QVIPPE": {"1": ["60"], "2": ["81"]},
    "QVIQVI": {"-": ["48"]},
    "QVIRINALIS": {"N": ["24"], "A": ["24"]},
    "QVIRINVS": {"N": ["12"], "A": ["21"]},
    "QVIRITES": {"N": ["13"]},
    "QVIRITO": {"-": ["51"]},
    "QVIRO": {"-": ["53"]},
    "QVIS": {"1": ["47"], "2": ["48"]},
    "QVISNAM": {"-": ["47"]},
    "QVISPIAM": {"-": ["48"]},
    "QVISQVAM": {"-": ["48"]},
    "QVISQVE": {"1": ["46"], "2": ["48"]},
    "QVISQVIS": {"1": ["46"], "2": ["48"]},
    "QVIVIS": {"-": ["48"]},
    "QVIVISCVMQVE": {"-": ["48"]},
    "QVO": {"1": ["66"], "2": ["66"], "3": ["67"], "4": ["60"], "5": ["82", "6-"]},
    "QVOAD": {"1": ["66"], "2": ["67"]},
    "QVOCIRCA": {"-": ["66"]},
    "QVOCVMQVE": {"1": ["66"], "2": ["60"]},
    "QVOD": {"1": ["81"], "2": ["82"]},
    "QVODAMMODO": {"-": ["60"]},
    "QVOLIBET": {"-": ["60"]},
    "QVOMINVS": {"-": ["82"]},
    "QVOMODO": {"1": ["66"], "2": ["67"]},
    "QVOMODOCVMQVE": {"1": ["66"], "2": ["60"]},
    "QVONAM": {"-": ["67"]},
    "QVONDAM": {"-": ["60"]},
    "QVONIAM": {"-": ["82"]},
    "QVOQVAM": {"-": ["60"]},
    "QVOQVE": {"-": ["60"]},
    "QVOQVO": {"-": ["66"]},
    "QVOQVOVERSVS": {"-": ["60"]},
    "QVORSVM": {"-": ["67"]},
    "QVOT": {"1": ["46"], "2": ["47"], "3": ["48"]},
    "QVOTANNIS": {"-": ["60"]},
    "QVOTCVMQVE": {"-": ["46"]},
    "QVOTIENS": {"1": ["66"], "2": ["67"]},
    "QVOTIENSCVMQVE": {"-": ["66"]},
    "QVOTQVOT": {"1": ["46"], "2": ["48"]},
    "QVOTVIS": {"-": ["48"]},
    "QVOTVS": {"-": ["47"]},
    "QVOTVSCVMQVE": {"-": ["46"]},
    "QVOTVSQVISQVE": {"-": ["47"]},
    "QVOVIS": {"-": ["66"]},
    "QVOVSQVE": {"-": ["67"], "2": ["6-"]},
    "R": {"-": ["16"]},
    "RABIDE": {"-": ["60"]},
    "RABIDVS": {"-": ["21"]},
    "RABIES": {"-": ["15"]},
    "RABIOSVS": {"-": ["21"]},
    "RABO": {"2": ["53"]},
    "RACEMVS": {"-": ["12"]},
    "RACILIVS": {"N": ["12"]},
    "RADIANS": {"-": ["25"]},
    "RADIATVS": {"-": ["21"]},
    "RADICESCO": {"-": ["53"]},
    "RADICITVS": {"-": ["60"]},
    "RADICVLA": {"-": ["11"]},
    "RADIO": {"-": ["51"]},
    "RADIVS": {"-": ["12"]},
    "RADIX": {"-": ["13"]},
    "RADO": {"-": ["53"]},
    "RAECIVS": {"N": ["12"]},
    "RAEDA": {"-": ["11"]},
    "RAMALE": {"-": ["13"]},
    "RAMENTVM": {"-": ["12"]},
    "RAMEVS": {"-": ["21"]},
    "RAMEX": {"-": ["13"]},
    "RAMNENSES": {"N": ["13"]},
    "RAMNES": {"N": ["13"]},
    "RAMOSVS": {"-": ["21"]},
    "RAMVLVS": {"-": ["12"]},
    "RAMVS": {"-": ["12"]},
    "RANA": {"-": ["11"]},
    "RANCENS": {"-": ["25"]},
    "RANCIDVLVS": {"-": ["21"]},
    "RANCIDVS": {"-": ["21"]},
    "RAPACITAS": {"-": ["13"]},
    "RAPAX": {"-": ["25"], "A": ["13"]},
    "RAPHANVS": {"-": ["12"]},
    "RAPICIVS": {"-": ["21"]},
    "RAPIDE": {"-": ["60"]},
    "RAPIDITAS": {"-": ["13"]},
    "RAPIDVS": {"-": ["21"]},
    "RAPINA": {"1": ["11"], "2": ["11"]},
    "RAPIO": {"-": ["55"]},
    "RAPSO": {"-": ["51"]},
    "RAPTIM": {"-": ["60"]},
    "RAPTO": {"-": ["51"]},
    "RAPTOR": {"-": ["13"]},
    "RAPTVM": {"-": ["12"]},
    "RAPTVS": {"-": ["14"]},
    "RAPVLVM": {"-": ["12"]},
    "RAPVM": {"-": ["12"]},
    "RARE": {"-": ["60"]},
    "RAREFACIO": {"-": ["55"]},
    "RAREFIO": {"-": ["56"]},
    "RARENTER": {"-": ["60"]},
    "RARESCO": {"-": ["53"]},
    "RARITAS": {"-": ["13"]},
    "RARO": {"-": ["60"]},
    "RARVS": {"-": ["21"]},
    "RASILIS": {"-": ["24"]},
    "RASTER": {"-": ["12"]},
    "RASTRI": {"-": ["12"]},
    "RASTRVM": {"-": ["12"]},
    "RATIO": {"-": ["13"]},
    "RATIOCINATIO": {"-": ["13"]},
    "RATIOCINATOR": {"-": ["13"]},
    "RATIOCINO": {"-": ["51"]},
    "RATIOCINOR": {"-": ["5J"]},
    "RATIONALIS": {"1": ["13"], "2": ["24"]},
    "RATIONALITER": {"-": ["60"]},
    "RATIS": {"-": ["13"]},
    "RATIVNCVLA": {"-": ["11"]},
    "RATVS": {"-": ["21"]},
    "RAVCISONVS": {"-": ["21"]},
    "RAVCVS": {"-": ["21"]},
    "RAVVS": {"-": ["21"]},
    "REA": {"-": ["11"]},
    "REALIS": {"-": ["24"]},
    "REAPSE": {"-": ["60"]},
    "REBELLATIO": {"-": ["13"]},
    "REBELLIO": {"-": ["13"]},
    "REBELLIS": {"-": ["24"]},
    "REBELLO": {"-": ["51"]},
    "REBOO": {"-": ["51"]},
    "RECALCITRO": {"-": ["51"]},
    "RECALESCO": {"-": ["53"]},
    "RECALFACIO": {"-": ["55"]},
    "RECANTO": {"-": ["51"]},
    "RECEDO": {"1": ["53"]},
    "RECELLO": {"-": ["53"]},
    "RECENS": {"1": ["25"], "2": ["60"]},
    "RECENSEO": {"-": ["52"]},
    "RECEPIO": {"-": ["55"]},
    "RECEPTACVLVM": {"-": ["12"]},
    "RECEPTICIVS": {"-": ["21"]},
    "RECEPTO": {"-": ["51"]},
    "RECEPTOR": {"-": ["13"]},
    "RECEPTRIX": {"-": ["13"]},
    "RECEPTVM": {"-": ["12"]},
    "RECEPTVS": {"1": ["14"], "2": ["21"], "N": ["12"]},
    "RECESSIO": {"-": ["13"]},
    "RECESSVS": {"1": ["14"], "2": ["21"]},
    "RECHAMVS": {"-": ["12"]},
    "RECIDIVVS": {"-": ["21"]},
    "RECIDO": {"1": ["53"], "2": ["53"]},
    "RECINGO": {"-": ["53"]},
    "RECINO": {"-": ["53"]},
    "RECIPIO": {"-": ["55"]},
    "RECIPROCO": {"-": ["51"]},
    "RECIPROCVS": {"-": ["21"]},
    "RECISVS": {"-": ["21"]},
    "RECITATIO": {"-": ["13"]},
    "RECITATOR": {"-": ["13"]},
    "RECITO": {"1": ["51"]},
    "RECLAMO": {"-": ["51"]},
    "RECLINIS": {"-": ["24"]},
    "RECLINO": {"-": ["51"]},
    "RECLVDO": {"-": ["53"]},
    "RECOGITO": {"-": ["51"]},
    "RECOGNITIO": {"-": ["13"]},
    "RECOGNOSCO": {"-": ["53"]},
    "RECOLLIGO": {"-": ["53"], "3": ["53"]},
    "RECOLO": {"1": ["51"], "2": ["53"]},
    "RECOMPONO": {"-": ["53"]},
    "RECONCILIATIO": {"-": ["13"]},
    "RECONCILIO": {"-": ["51"]},
    "RECONDITVS": {"-": ["21"]},
    "RECONDO": {"-": ["53"]},
    "RECONFLO": {"-": ["51"]},
    "RECOQVO": {"-": ["53"]},
    "RECORDATIO": {"-": ["13"]},
    "RECORDOR": {"-": ["5J"]},
    "RECORRIGO": {"-": ["53"]},
    "RECREO": {"-": ["51"]},
    "RECREPO": {"-": ["51"]},
    "RECRESCO": {"-": ["53"]},
    "RECRVDESCO": {"-": ["53"]},
    "RECTA": {"-": ["60"]},
    "RECTE": {"-": ["60"]},
    "RECTOR": {"-": ["13"]},
    "RECTRIX": {"-": ["13"]},
    "RECTVM": {"-": ["12"]},
    "RECTVS": {"-": ["21"]},
    "RECVBO": {"-": ["51"]},
    "RECVMBO": {"-": ["53"]},
    "RECVPERATOR": {"-": ["13"]},
    "RECVPERATORIVS": {"-": ["21"]},
    "RECVPERO": {"-": ["51"]},
    "RECVRO": {"-": ["51"]},
    "RECVRRO": {"-": ["53"]},
    "RECVRSO": {"-": ["51"]},
    "RECVRSVS": {"-": ["14"]},
    "RECVRVO": {"-": ["51"]},
    "RECVRVVS": {"-": ["21"]},
    "RECVSATIO": {"-": ["13"]},
    "RECVSO": {"-": ["51"]},
    "RECVTIO": {"-": ["55"]},
    "RECVTITVS": {"-": ["21"]},
    "REDAMO": {"-": ["51"]},
    "REDARDESCO": {"-": ["53"]},
    "REDARGVO": {"-": ["53"]},
    "REDDO": {"-": ["53"]},
    "REDDVCO": {"-": ["53"]},
    "REDEMPTIO": {"-": ["13"]},
    "REDEMPTITO": {"-": ["51"]},
    "REDEMPTOR": {"-": ["13"]},
    "REDEO": {"1": ["56"]},
    "REDHALO": {"-": ["51"]},
    "REDIGO": {"-": ["53"]},
    "REDIMICVLVM": {"-": ["12"]},
    "REDIMIO": {"-": ["54"]},
    "REDIMO": {"-": ["53"]},
    "REDINTEGRO": {"-": ["51"]},
    "REDITIO": {"-": ["13"]},
    "REDITVS": {"-": ["14"]},
    "REDIVIVVS": {"-": ["21"]},
    "REDOLEO": {"-": ["52"]},
    "REDONO": {"-": ["51"]},
    "REDVCO": {"-": ["53"]},
    "REDVCTIO": {"-": ["13"]},
    "REDVCTVS": {"-": ["21"]},
    "REDVNDANTIA": {"-": ["11"]},
    "REDVNDATIO": {"-": ["13"]},
    "REDVNDO": {"-": ["51"]},
    "REDVRESCO": {"-": ["53"]},
    "REDVX": {"-": ["25"]},
    "REFECTIO": {"-": ["13"]},
    "REFELLO": {"-": ["53"]},
    "REFERCIO": {"-": ["54"]},
    "REFERIO": {"-": ["54"]},
    "REFERO": {"-": ["56"]},
    "REFERT": {"-": ["56"]},
    "REFERTVS": {"-": ["21"]},
    "REFICIO": {"-": ["55"]},
    "REFIGO": {"-": ["53"]},
    "REFINGO": {"-": ["53"]},
    "REFLAGITO": {"-": ["51"]},
    "REFLECTO": {"-": ["53"]},
    "REFLO": {"-": ["51"]},
    "REFLVO": {"-": ["53"]},
    "REFLVVS": {"-": ["21"]},
    "REFORMATIO": {"-": ["13"]},
    "REFORMIDO": {"2": ["51"]},
    "REFORMO": {"-": ["51"]},
    "REFOVEO": {"-": ["52"]},
    "REFRACTARIVS": {"-": ["21"]},
    "REFRAGOR": {"-": ["5J"]},
    "REFRENATIO": {"-": ["13"]},
    "REFRENO": {"-": ["51"]},
    "REFRICO": {"-": ["51"]},
    "REFRIGERATIO": {"-": ["13"]},
    "REFRIGERO": {"-": ["51"]},
    "REFRIGESCO": {"-": ["53"]},
    "REFRINGO": {"-": ["53"]},
    "REFVGIO": {"-": ["55"]},
    "REFVGVS": {"1": ["12"], "2": ["21"]},
    "REFVLGEO": {"-": ["52"]},
    "REFVNDO": {"2": ["53"]},
    "REFVTATVS": {"-": ["14"]},
    "REFVTO": {"-": ["51"]},
    "REGALIS": {"-": ["24"]},
    "REGELO": {"-": ["51"]},
    "REGENS": {"-": ["13"]},
    "REGERO": {"-": ["53"]},
    "REGIA": {"-": ["11"], "N": ["11"]},
    "REGIE": {"-": ["60"]},
    "REGIFICVS": {"-": ["21"]},
    "REGIGNO": {"-": ["53"]},
    "REGILLVS": {"1N": ["12"], "2N": ["12"], "1": ["12"], "2": ["12"]},
    "REGIMEN": {"-": ["13"]},
    "REGINA": {"-": ["11"], "N": ["11"]},
    "REGINI": {"N": ["12"]},
    "REGIO": {"-": ["13"]},
    "REGIVM": {"N": ["12"]},
    "REGIVS": {"-": ["21"], "N": ["12"]},
    "REGLVTINO": {"-": ["51"]},
    "REGNATOR": {"-": ["13"]},
    "REGNATRIX": {"-": ["25"]},
    "REGNO": {"-": ["51"]},
    "REGNVM": {"-": ["12"]},
    "REGO": {"-": ["53"]},
    "REGREDIOR": {"-": ["5N"]},
    "REGRESSVS": {"-": ["14"]},
    "REGVLA": {"-": ["11"]},
    "REGVLVS": {"N": ["12"], "-": ["12"]},
    "REGVSTO": {"-": ["51"]},
    "REICIO": {"-": ["55"]},
    "REICVLVS": {"-": ["21"]},
    "REIECTIO": {"-": ["13"]},
    "REIECTO": {"-": ["51"]},
    "RELABOR": {"2": ["5L"]},
    "RELANGVESCO": {"-": ["53"]},
    "RELATIO": {"-": ["13"]},
    "RELATVS": {"-": ["14"]},
    "RELAXO": {"-": ["51"]},
    "RELEGATIO": {"-": ["13"]},
    "RELEGO": {"1": ["51"], "2": ["53"]},
    "RELENTESCO": {"-": ["53"]},
    "RELEVATVS": {"C": ["21"], "-": ["21"]},
    "RELEVO": {"1": ["51"]},
    "RELICTIO": {"-": ["13"]},
    "RELICTVM": {"-": ["12"]},
    "RELICTVS": {"1": ["14"], "2": ["21"]},
    "RELIGIO": {"-": ["13"]},
    "RELIGIOSE": {"-": ["60"]},
    "RELIGIOSVS": {"-": ["21"]},
    "RELIGO": {"2": ["51"]},
    "RELINO": {"-": ["53"]},
    "RELINQVO": {"-": ["53"]},
    "RELIQVA": {"-": ["12"]},
    "RELIQVI": {"-": ["12"]},
    "RELIQVIAE": {"-": ["11"]},
    "RELIQVVM": {"-": ["12"]},
    "RELIQVVS": {"-": ["21"]},
    "RELVCEO": {"-": ["52"]},
    "RELVCTOR": {"-": ["5J"]},
    "REMANEO": {"-": ["52"]},
    "REMANO": {"-": ["51"]},
    "REMEDIABILIS": {"-": ["24"]},
    "REMEDIVM": {"-": ["12"]},
    "REMEO": {"-": ["51"]},
    "REMETIOR": {"-": ["5M"]},
    "REMEX": {"-": ["13"]},
    "REMIGIVM": {"-": ["12"]},
    "REMIGO": {"-": ["51"]},
    "REMIGRO": {"-": ["51"]},
    "REMINISCO": {"-": ["53"]},
    "REMINISCOR": {"-": ["5L"]},
    "REMISCEO": {"-": ["52"]},
    "REMISSARIVS": {"-": ["21"]},
    "REMISSE": {"-": ["6&"]},
    "REMISSIO": {"-": ["13"]},
    "REMISSVS": {"-": ["21"]},
    "REMITTO": {"-": ["53"]},
    "REMOLIOR": {"-": ["5M"]},
    "REMOLLESCO": {"-": ["53"]},
    "REMOLLIO": {"-": ["54"]},
    "REMORDEO": {"-": ["52"]},
    "REMOROR": {"1": ["5J"]},
    "REMOTA": {"-": ["12"]},
    "REMOTVS": {"-": ["21"]},
    "REMOVEO": {"-": ["52"]},
    "REMVGIO": {"-": ["54"]},
    "REMVLCVM": {"-": ["12"]},
    "REMVNERATIO": {"-": ["13"]},
    "REMVNERO": {"-": ["51"]},
    "REMVNEROR": {"-": ["5J"]},
    "REMVS": {"-": ["12"], "N": ["12"], "A": ["21"]},
    "RENARRO": {"-": ["51"]},
    "RENASCOR": {"-": ["5L"]},
    "RENAVIGO": {"-": ["51"]},
    "RENES": {"-": ["13"]},
    "RENIDEO": {"-": ["52"]},
    "RENIDESCO": {"-": ["53"]},
    "RENO": {"-": ["51"]},
    "RENODO": {"-": ["51"]},
    "RENOVO": {"-": ["51"]},
    "RENVMERO": {"1": ["51"]},
    "RENVNTIATIO": {"-": ["13"]},
    "RENVNTIO": {"-": ["51"]},
    "RENVO": {"-": ["53"]},
    "RENVTO": {"-": ["51"]},
    "REOR": {"-": ["5K"]},
    "REPAGVLA": {"-": ["12"]},
    "REPARABILIS": {"-": ["24"]},
    "REPARCO": {"-": ["53"]},
    "REPARO": {"2": ["51"]},
    "REPASTINO": {"-": ["51"]},
    "REPECTO": {"-": ["53"]},
    "REPELLO": {"-": ["53"]},
    "REPENDO": {"-": ["53"]},
    "REPENS": {"1": ["25"], "2": ["60"]},
    "REPENSO": {"-": ["51"]},
    "REPENTE": {"-": ["60"]},
    "REPENTINO": {"-": ["60"]},
    "REPENTINVS": {"-": ["21"], "N": ["12"]},
    "REPERCVSSVS": {"-": ["14"]},
    "REPERCVTIO": {"-": ["55"]},
    "REPERIO": {"-": ["54"]},
    "REPERTOR": {"-": ["13"]},
    "REPERTVM": {"-": ["12"]},
    "REPERTVS": {"1": ["14"], "2": ["21"]},
    "REPETENTIA": {"-": ["11"]},
    "REPETITIO": {"-": ["13"]},
    "REPETITOR": {"-": ["13"]},
    "REPETO": {"-": ["53"]},
    "REPETVNDAE": {"-": ["11"]},
    "REPLEO": {"-": ["52"]},
    "REPLETVS": {"-": ["21"]},
    "REPLICO": {"-": ["51"]},
    "REPLVM": {"-": ["12"]},
    "REPO": {"-": ["53"]},
    "REPONO": {"-": ["53"]},
    "REPORTO": {"-": ["51"]},
    "REPOSCO": {"-": ["53"]},
    "REPOSITORIVM": {"-": ["12"]},
    "REPOSITVS": {"-": ["21"]},
    "REPOTIA": {"-": ["12"]},
    "REPRAESENTATIO": {"-": ["13"]},
    "REPRAESENTO": {"-": ["51"]},
    "REPREHENDO": {"-": ["53"]},
    "REPREHENSIO": {"-": ["13"]},
    "REPREHENSOR": {"-": ["13"]},
    "REPRIMO": {"-": ["53"]},
    "REPTABVNDVS": {"-": ["21"]},
    "REPTO": {"-": ["51"]},
    "REPVDIATIO": {"-": ["13"]},
    "REPVDIO": {"-": ["51"]},
    "REPVDIVM": {"-": ["12"]},
    "REPVGNANS": {"-": ["25"]},
    "REPVGNANTER": {"-": ["60"]},
    "REPVGNATORIVS": {"-": ["21"]},
    "REPVGNO": {"-": ["51"]},
    "REPVLLVLO": {"-": ["51"]},
    "REPVLSA": {"-": ["11"]},
    "REPVLSO": {"-": ["51"]},
    "REPVLSVS": {"1": ["14"], "2": ["21"]},
    "REPVRGO": {"-": ["51"]},
    "REPVTATIO": {"-": ["13"]},
    "REPVTO": {"-": ["51"]},
    "REQVIES": {"-": ["16"]},
    "REQVIESCO": {"-": ["53"]},
    "REQVIRO": {"-": ["53"]},
    "RES": {"-": ["15"]},
    "RESALVTO": {"-": ["51"]},
    "RESANESCO": {"-": ["53"]},
    "RESARCIO": {"-": ["54"]},
    "RESCINDO": {"-": ["53"]},
    "RESCIO": {"-": ["54"]},
    "RESCISCO": {"-": ["53"]},
    "RESCRIBO": {"-": ["53"]},
    "RESCRIPTVM": {"-": ["12"]},
    "RESECO": {"-": ["51"]},
    "RESEQVOR": {"-": ["5L"]},
    "RESERO": {"1": ["51"], "3": ["53"]},
    "RESERVO": {"-": ["51"]},
    "RESES": {"-": ["25"]},
    "RESIDEO": {"-": ["52"]},
    "RESIDO": {"-": ["53"]},
    "RESIDVVS": {"-": ["21"]},
    "RESIGNO": {"-": ["51"]},
    "RESILIO": {"-": ["54"]},
    "RESINA": {"-": ["11"]},
    "RESINATVS": {"-": ["21"]},
    "RESIPISCO": {"-": ["53"]},
    "RESISTO": {"-": ["53"]},
    "RESOLVO": {"-": ["53"]},
    "RESOLVTVS": {"-": ["21"]},
    "RESONABILIS": {"-": ["24"]},
    "RESONANTIA": {"-": ["11"]},
    "RESONO": {"1": ["51"]},
    "RESORBEO": {"-": ["52"]},
    "RESPECTO": {"-": ["51"]},
    "RESPECTVS": {"-": ["14"]},
    "RESPERGO": {"-": ["53"]},
    "RESPICIO": {"-": ["55"]},
    "RESPIRATIO": {"-": ["13"]},
    "RESPIRO": {"-": ["51"]},
    "RESPLENDEO": {"-": ["52"]},
    "RESPONDEO": {"-": ["52"]},
    "RESPONSIO": {"-": ["13"]},
    "RESPONSO": {"-": ["51"]},
    "RESPONSVM": {"-": ["12"]},
    "RESPONSVS": {"-": ["14"]},
    "RESPVBLICA": {"-": ["15"]},
    "RESPVO": {"-": ["53"]},
    "RESTAGNO": {"1": ["51"]},
    "RESTAVRO": {"-": ["51"]},
    "RESTIBILIS": {"-": ["24"]},
    "RESTICVLA": {"-": ["11"]},
    "RESTINGVO": {"-": ["53"]},
    "RESTIS": {"-": ["13"]},
    "RESTITVO": {"-": ["53"]},
    "RESTO": {"-": ["51"]},
    "RESTRICTE": {"-": ["60"]},
    "RESTRICTVS": {"-": ["21"]},
    "RESTRINGO": {"-": ["53"]},
    "RESVDO": {"-": ["51"]},
    "RESVLTO": {"-": ["51"]},
    "RESVMO": {"-": ["53"]},
    "RESVPINO": {"-": ["51"]},
    "RESVPINVS": {"-": ["21"]},
    "RESVRGO": {"-": ["53"]},
    "RETARDATIO": {"-": ["13"]},
    "RETARDO": {"-": ["51"]},
    "RETE": {"-": ["13"]},
    "RETEGO": {"-": ["53"]},
    "RETENDO": {"-": ["53"]},
    "RETENTIO": {"-": ["13"]},
    "RETENTO": {"1": ["51"], "2": ["51"]},
    "RETERO": {"-": ["53"]},
    "RETEXO": {"-": ["53"]},
    "RETICENTIA": {"-": ["11"]},
    "RETICEO": {"-": ["52"]},
    "RETICESCO": {"-": ["53"]},
    "RETICVLATVS": {"-": ["21"]},
    "RETICVLVM": {"-": ["12"]},
    "RETINA": {"-": ["11"]},
    "RETINACVLVM": {"-": ["12"]},
    "RETINENS": {"-": ["25"]},
    "RETINENTIA": {"-": ["11"]},
    "RETINEO": {"-": ["52"]},
    "RETINGO": {"1": ["53"], "2": ["53"]},
    "RETIO": {"-": ["54"]},
    "RETONO": {"-": ["51"]},
    "RETORQVEO": {"-": ["52"]},
    "RETORRIDVS": {"-": ["21"]},
    "RETRACTATIO": {"-": ["13"]},
    "RETRACTIO": {"-": ["13"]},
    "RETRACTO": {"-": ["51"]},
    "RETRAHO": {"-": ["53"]},
    "RETRIBVO": {"-": ["53"]},
    "RETRO": {"1": ["60"], "2": ["70"]},
    "RETROCEDO": {"1": ["53"]},
    "RETRODVCO": {"-": ["53"]},
    "RETROFLECTO": {"-": ["53"]},
    "RETROLEGO": {"2": ["53"]},
    "RETRORSVM": {"-": ["60"]},
    "RETROVERTO": {"-": ["53"]},
    "RETRVDO": {"-": ["53"]},
    "RETVNDO": {"-": ["53"]},
    "RETVSVS": {"-": ["21"]},
    "REVALESCO": {"-": ["53"]},
    "REVECTO": {"-": ["51"]},
    "REVEHO": {"-": ["53"]},
    "REVELLO": {"-": ["53"]},
    "REVELO": {"-": ["51"]},
    "REVENIO": {"-": ["54"]},
    "REVERA": {"-": ["60"]},
    "REVERBERO": {"2": ["51"]},
    "REVERENS": {"-": ["25"]},
    "REVERENTER": {"-": ["60"]},
    "REVERENTIA": {"-": ["11"], "N": ["11"]},
    "REVEREOR": {"-": ["5K"]},
    "REVERSIO": {"-": ["13"]},
    "REVERSO": {"-": ["51"]},
    "REVERTO": {"-": ["53"]},
    "REVERTOR": {"-": ["5L"]},
    "REVILESCO": {"-": ["53"]},
    "REVINCIO": {"-": ["54"]},
    "REVINCO": {"-": ["53"]},
    "REVISO": {"-": ["53"]},
    "REVIVISCO": {"-": ["53"]},
    "REVIVO": {"-": ["53"]},
    "REVOCABILIS": {"-": ["24"]},
    "REVOCAMEN": {"-": ["13"]},
    "REVOCATIO": {"-": ["13"]},
    "REVOCO": {"-": ["51"]},
    "REVOLO": {"2": ["51"]},
    "REVOLVO": {"-": ["53"]},
    "REVOMO": {"-": ["53"]},
    "REVS": {"-": ["12"]},
    "REX": {"-": ["13"], "N": ["13"]},
    "RHADINE": {"-": ["17"]},
    "RHEA": {"N": ["11"]},
    "RHEDA": {"-": ["11"]},
    "RHENO": {"-": ["13"]},
    "RHETOR": {"-": ["13"]},
    "RHETORICVS": {"-": ["21"]},
    "RHINOCEROS": {"-": ["17"]},
    "RHODII": {"N": ["12"]},
    "RHODIVS": {"N": ["21"], "A": ["21"]},
    "RHODOS": {"N": ["17"]},
    "RHOMBOS": {"-": ["17"]},
    "RHYTHMVS": {"-": ["12"]},
    "RICINVS": {"-": ["12"]},
    "RICTVM": {"-": ["12"]},
    "RICTVS": {"-": ["14"]},
    "RIDENDVS": {"-": ["21"]},
    "RIDEO": {"-": ["52"]},
    "RIDIBVNDVS": {"-": ["21"]},
    "RIDICA": {"-": ["11"]},
    "RIDICVLARIA": {"-": ["12"]},
    "RIDICVLE": {"-": ["60"]},
    "RIDICVLVM": {"-": ["12"]},
    "RIDICVLVS": {"1": ["12"], "2": ["21"]},
    "RIGENS": {"-": ["25"]},
    "RIGEO": {"-": ["52"]},
    "RIGESCO": {"-": ["53"]},
    "RIGIDA": {"-": ["11"]},
    "RIGIDE": {"-": ["60"]},
    "RIGIDITAS": {"-": ["13"]},
    "RIGIDO": {"-": ["51"]},
    "RIGIDVS": {"-": ["21"]},
    "RIGO": {"-": ["53", "51"]},
    "RIGOR": {"-": ["13"]},
    "RIGVVS": {"-": ["21"]},
    "RIMA": {"-": ["11"]},
    "RIMOR": {"-": ["5J"]},
    "RIMOSVS": {"-": ["21"]},
    "RIMVLA": {"-": ["11"]},
    "RINGOR": {"-": ["5L"]},
    "RIPA": {"-": ["11"]},
    "RIPIO": {"-": ["55"]},
    "RISVS": {"-": ["14"]},
    "RITE": {"-": ["60"]},
    "RITVS": {"-": ["14"]},
    "RIVALIS": {"1": ["13"], "2": ["24"]},
    "RIVO": {"-": ["53"], "1": ["51"], "2": ["53"]},
    "RIVVS": {"-": ["12"]},
    "RIXA": {"-": ["11"]},
    "RIXOR": {"-": ["5J"]},
    "ROBIGALIA": {"-": ["13"]},
    "ROBIGO": {"-": ["13"], "N": ["13"]},
    "ROBORATVS": {"-": ["21"]},
    "ROBORO": {"-": ["51"]},
    "ROBVR": {"-": ["13"]},
    "ROBVS": {"-": ["21"]},
    "ROBVSTEVS": {"-": ["21"]},
    "ROBVSTVS": {"-": ["21"], "N": ["12"]},
    "RODEVS": {"-": ["21"]},
    "RODO": {"-": ["53"]},
    "ROGALIS": {"-": ["24"]},
    "ROGATIO": {"-": ["13"]},
    "ROGATVS": {"-": ["14"]},
    "ROGITO": {"-": ["51"]},
    "ROGO": {"-": ["51"]},
    "ROGVS": {"-": ["12"]},
    "ROMA": {"N": ["11"]},
    "ROMANI": {"N": ["12"]},
    "ROMANVS": {"N": ["21"], "A": ["21"]},
    "ROMVLARIS": {"N": ["24"]},
    "ROMVLEVS": {"N": ["21"], "A": ["21"]},
    "ROMVLVS": {"N": ["12"], "A": ["21"]},
    "RORIDVS": {"-": ["21"]},
    "RORIFER": {"-": ["21"]},
    "RORO": {"-": ["51"]},
    "RORVLENTVS": {"-": ["21"]},
    "ROS": {"-": ["13"], "N": ["13"]},
    "ROSA": {"-": ["11"]},
    "ROSARIVM": {"-": ["12"]},
    "ROSCIDVS": {"-": ["21"]},
    "ROSETVM": {"-": ["12"]},
    "ROSEVS": {"-": ["21"], "A": ["21"]},
    "ROSTRATVS": {"-": ["21"]},
    "ROSTRVM": {"-": ["12"]},
    "ROTA": {"-": ["11"]},
    "ROTATIO": {"-": ["13"]},
    "ROTO": {"-": ["51"]},
    "ROTVLA": {"-": ["11"]},
    "ROTVNDATIO": {"-": ["13"]},
    "ROTVNDITAS": {"-": ["13"]},
    "ROTVNDO": {"-": ["51"]},
    "ROTVNDVS": {"-": ["21"]},
    "RVBELLVS": {"-": ["21"]},
    "RVBENS": {"-": ["25"]},
    "RVBEO": {"-": ["52"]},
    "RVBER": {"-": ["21"], "N": ["21"], "A": ["23"]},
    "RVBESCO": {"-": ["53"]},
    "RVBETA": {"-": ["11"]},
    "RVBETVM": {"-": ["12"]},
    "RVBEVS": {"-": ["21"]},
    "RVBIA": {"-": ["11"]},
    "RVBICVNDVLVS": {"-": ["21"]},
    "RVBICVNDVS": {"-": ["21"]},
    "RVBOR": {"-": ["13"]},
    "RVBRICA": {"-": ["11"]},
    "RVBRICOSVS": {"-": ["21"]},
    "RVBRIVS": {"N": ["12"]},
    "RVBVS": {"-": ["12"]},
    "RVCTABVNDVS": {"-": ["21"]},
    "RVCTO": {"-": ["51"]},
    "RVCTVS": {"-": ["14"]},
    "RVDECTVS": {"-": ["21"]},
    "RVDENS": {"-": ["13"]},
    "RVDERATIO": {"-": ["13"]},
    "RVDICVLA": {"-": ["11"]},
    "RVDIMENTVM": {"-": ["12"]},
    "RVDIS": {"1": ["13"], "2": ["24"]},
    "RVDO": {"-": ["53"]},
    "RVDVS": {"-": ["13"]},
    "RVFVS": {"-": ["21"], "N": ["21"]},
    "RVGA": {"-": ["11"]},
    "RVGOSVS": {"-": ["21"]},
    "RVINA": {"-": ["11"]},
    "RVINOSVS": {"-": ["21"]},
    "RVMINALIS": {"N": ["24"], "A": ["24"]},
    "RVMINO": {"-": ["51"]},
    "RVMOR": {"-": ["13"], "N": ["13"]},
    "RVMPO": {"-": ["53"]},
    "RVNCO": {"-": ["51"]},
    "RVO": {"-": ["53"]},
    "RVPES": {"-": ["13"]},
    "RVPILIVS": {"A": ["21"], "N": ["12"]},
    "RVPTOR": {"-": ["13"]},
    "RVRICOLA": {"-": ["11"]},
    "RVRSVS": {"-": ["60"]},
    "RVS": {"-": ["13"]},
    "RVSCEVS": {"-": ["21"]},
    "RVSCVM": {"-": ["12"]},
    "RVSCVS": {"-": ["12"]},
    "RVSSATVS": {"-": ["21"]},
    "RVSSEVS": {"-": ["21"]},
    "RVSSVS": {"-": ["21"]},
    "RVSTARIVS": {"-": ["21"]},
    "RVSTICA": {"-": ["11"]},
    "RVSTICANVS": {"1": ["12"], "2": ["21"]},
    "RVSTICE": {"-": ["6&"]},
    "RVSTICITAS": {"-": ["13"]},
    "RVSTICVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "RVTA": {"-": ["11"]},
    "RVTABVLVM": {"-": ["12"]},
    "RVTILANS": {"-": ["25"]},
    "RVTILIA": {"N": ["11"]},
    "RVTILIVS": {"N": ["12"]},
    "RVTILO": {"-": ["51"]},
    "RVTILVS": {"-": ["21"], "N": ["12"]},
    "RVTRVM": {"-": ["12"]},
    "RVTVLI": {"N": ["12"]},
    "SABBATVM": {"-": ["12"], "N": ["12"]},
    "SABINI": {"N": ["12"]},
    "SABINVS": {"N": ["21"], "A": ["21"]},
    "SABVLO": {"-": ["13"]},
    "SABVLOSVS": {"-": ["21"]},
    "SABVLVM": {"-": ["12"]},
    "SABVRRA": {"-": ["11"], "N": ["11"]},
    "SACCATVS": {"-": ["21"]},
    "SACCO": {"-": ["51"]},
    "SACCVLVS": {"-": ["12"]},
    "SACCVS": {"-": ["12"]},
    "SACELLVM": {"-": ["12"]},
    "SACER": {"-": ["26"], "N": ["21"], "A": ["23"]},
    "SACERDOS": {"-": ["13"], "N": ["13"]},
    "SACERDOTIVM": {"-": ["12"]},
    "SACRAMENTALIS": {"-": ["24"]},
    "SACRAMENTVM": {"-": ["12"]},
    "SACRARIVM": {"-": ["12"]},
    "SACRATIVIR": {"N": ["12"]},
    "SACRATVS": {"-": ["21"]},
    "SACRIFICALIS": {"-": ["24"]},
    "SACRIFICIVM": {"-": ["12"]},
    "SACRIFICO": {"-": ["51"]},
    "SACRIFICVS": {"-": ["21"]},
    "SACRILEGA": {"-": ["11"]},
    "SACRILEGIVM": {"-": ["12"]},
    "SACRILEGVS": {"1": ["12"], "2": ["21"]},
    "SACRO": {"-": ["51"]},
    "SACROSANCTVS": {"-": ["21"]},
    "SACRVM": {"-": ["12"]},
    "SADALA": {"N": ["11"]},
    "SAECVLARIS": {"-": ["24"]},
    "SAECVLVM": {"-": ["12"]},
    "SAEPE": {"-": ["60"]},
    "SAEPENVMERO": {"-": ["60"]},
    "SAEPES": {"-": ["13"]},
    "SAEPIA": {"-": ["11"]},
    "SAEPIO": {"-": ["54"]},
    "SAEPTA": {"-": ["12"], "L": ["12"]},
    "SAEPTIO": {"-": ["13"]},
    "SAEPTVM": {"-": ["12"]},
    "SAETA": {"-": ["11"]},
    "SAETIGER": {"1": ["12"], "2": ["21"]},
    "SAETOSVS": {"-": ["21"]},
    "SAEVE": {"-": ["6&"]},
    "SAEVIO": {"-": ["54"]},
    "SAEVITIA": {"-": ["11"]},
    "SAEVVS": {"-": ["21"]},
    "SAGA": {"1": ["11"], "2": ["11"]},
    "SAGACITAS": {"-": ["13"]},
    "SAGACITER": {"-": ["6&"]},
    "SAGAX": {"-": ["25"]},
    "SAGINA": {"-": ["11"]},
    "SAGINATVS": {"-": ["21"]},
    "SAGINO": {"-": ["51"]},
    "SAGIO": {"-": ["54"]},
    "SAGITTA": {"-": ["11"], "N": ["11"]},
    "SAGITTARIVS": {"1": ["12"], "2": ["21"]},
    "SAGITTIFER": {"1": ["12"], "2": ["21"]},
    "SAGVLVM": {"-": ["12"]},
    "SAGVM": {"-": ["12"]},
    "SAGVNTINI": {"-": ["12"], "N": ["12"]},
    "SAGVS": {"1": ["12"], "2": ["21"]},
    "SAL": {"-": ["13"]},
    "SALAPVTIVM": {"-": ["12"]},
    "SALARIVM": {"-": ["12"]},
    "SALAX": {"-": ["25"]},
    "SALEBRA": {"-": ["11"]},
    "SALEBROSVS": {"-": ["21"]},
    "SALGAMA": {"-": ["12"]},
    "SALIARIS": {"1": ["24"], "2": ["24"], "A": ["24"]},
    "SALICTARIVS": {"1": ["12"]},
    "SALICTVM": {"-": ["12"]},
    "SALIFODINA": {"-": ["11"]},
    "SALIGNEVS": {"-": ["21"]},
    "SALIGNVS": {"-": ["21"]},
    "SALII": {"-": ["12"], "N": ["12"]},
    "SALILLVM": {"-": ["12"]},
    "SALINAE": {"-": ["11"]},
    "SALINATOR": {"-": ["13"], "N": ["13"]},
    "SALINVM": {"-": ["12"]},
    "SALIO": {"1": ["54"], "2": ["54"]},
    "SALIVA": {"-": ["11"]},
    "SALIVNCA": {"-": ["11"]},
    "SALIX": {"-": ["13"]},
    "SALLVSTIVS": {"N": ["12"]},
    "SALSAMENTVM": {"-": ["12"]},
    "SALSITVDO": {"-": ["13"]},
    "SALSVGO": {"-": ["13"]},
    "SALSVM": {"-": ["12"], "N": ["12"]},
    "SALSVS": {"-": ["21"]},
    "SALTATIO": {"-": ["13"]},
    "SALTATOR": {"-": ["13"]},
    "SALTATVS": {"-": ["14"]},
    "SALTEM": {"-": ["60"]},
    "SALTO": {"-": ["51"]},
    "SALTVOSVS": {"-": ["21"]},
    "SALTVS": {"1": ["14"], "2": ["14"]},
    "SALVBER": {"-": ["23"]},
    "SALVBRITAS": {"-": ["13"]},
    "SALVBRITER": {"-": ["60"]},
    "SALVEO": {"-": ["52"]},
    "SALVIDIENVS": {"N": ["12"]},
    "SALVM": {"-": ["12"]},
    "SALVS": {"-": ["13"], "N": ["13"]},
    "SALVTARIS": {"-": ["24"]},
    "SALVTARITER": {"-": ["60"]},
    "SALVTATIO": {"-": ["13"]},
    "SALVTATOR": {"-": ["13"]},
    "SALVTATRIX": {"-": ["13"]},
    "SALVTIFER": {"-": ["21"]},
    "SALVTO": {"-": ["51"]},
    "SALVVS": {"-": ["21"], "N": ["12"]},
    "SAMBVCA": {"-": ["17"]},
    "SAMII": {"N": ["12"]},
    "SAMIVS": {"N": ["21"], "A": ["21"]},
    "SAMNIS": {"N": ["25"], "A": ["24"]},
    "SAMOS": {"N": ["17"]},
    "SANABILIS": {"-": ["24"]},
    "SANCIO": {"-": ["54"]},
    "SANCTE": {"-": ["60"]},
    "SANCTIMONIA": {"-": ["11"]},
    "SANCTIO": {"-": ["13"]},
    "SANCTITAS": {"-": ["13"]},
    "SANCTOR": {"-": ["13"]},
    "SANCTVM": {"-": ["12"]},
    "SANCTVS": {"-": ["21"], "A": ["21"], "N": ["12"]},
    "SANCVS": {"-": ["16"], "N": ["12"]},
    "SANDAPILA": {"-": ["11"]},
    "SANDARACA": {"-": ["11"]},
    "SANDYX": {"-": ["13"]},
    "SANE": {"-": ["60"]},
    "SANGVINARIVS": {"-": ["21"]},
    "SANGVINEVS": {"-": ["21"]},
    "SANGVINO": {"-": ["51"]},
    "SANGVINOLENTVS": {"-": ["21"]},
    "SANGVIS": {"-": ["13"]},
    "SANIES": {"-": ["15"]},
    "SANITAS": {"-": ["13"]},
    "SANNA": {"-": ["11"]},
    "SANO": {"-": ["51"]},
    "SANVS": {"-": ["21"]},
    "SAPA": {"-": ["11"]},
    "SAPERDA": {"-": ["11"]},
    "SAPIENS": {"1": ["13"], "2": ["25"], "C": ["13"], "N": ["13"]},
    "SAPIENTER": {"-": ["60"]},
    "SAPIENTIA": {"-": ["11"]},
    "SAPINEVS": {"-": ["21"]},
    "SAPINVS": {"-": ["12"]},
    "SAPIO": {"-": ["55"]},
    "SAPOR": {"-": ["13"]},
    "SAPPHO": {"N": ["17"]},
    "SARCINA": {"-": ["11"]},
    "SARCINARIVS": {"-": ["21"]},
    "SARCINVLA": {"-": ["11"]},
    "SARCIO": {"-": ["54"]},
    "SARCOPHAGVS": {"-": ["12"]},
    "SARCVLVM": {"-": ["12"]},
    "SARDA": {"-": ["11"]},
    "SARDINIA": {"N": ["11"]},
    "SARDONYX": {"-": ["17"]},
    "SARIO": {"-": ["54"]},
    "SARISA": {"-": ["11"]},
    "SARISOPHORVS": {"-": ["12"]},
    "SARMENTVM": {"-": ["12"]},
    "SARRACVM": {"-": ["12"]},
    "SARTAGO": {"-": ["13"]},
    "SARTVRA": {"-": ["11"]},
    "SARTVS": {"-": ["21"]},
    "SATA": {"-": ["12"]},
    "SATAGEVS": {"-": ["21"]},
    "SATAGO": {"-": ["53"]},
    "SATARIVS": {"-": ["21"]},
    "SATELLES": {"-": ["13"]},
    "SATIAS": {"-": ["13"]},
    "SATIETAS": {"-": ["13"]},
    "SATIO": {"1": ["13"], "2": ["51"]},
    "SATIS": {"1": ["26"], "2": ["60"]},
    "SATISACCIPIO": {"-": ["55"]},
    "SATISDO": {"-": ["51"]},
    "SATISFACIO": {"-": ["55"]},
    "SATISFACTIO": {"-": ["13"]},
    "SATISFIO": {"-": ["56"]},
    "SATOR": {"-": ["13"], "N": ["13"]},
    "SATRAPEA": {"-": ["11"]},
    "SATRAPES": {"-": ["16"]},
    "SATRICVM": {"N": ["12"]},
    "SATRIVS": {"N": ["12"]},
    "SATVR": {"-": ["21"]},
    "SATVRA": {"-": ["11"], "N": ["11"]},
    "SATVRATVS": {"-": ["21"]},
    "SATVREIVM": {"-": ["12"]},
    "SATVRITAS": {"-": ["13"], "N": ["13"]},
    "SATVRNALIA": {"N": ["13"]},
    "SATVRNINVS": {"N": ["12"]},
    "SATVRNVS": {"N": ["12"]},
    "SATVRO": {"-": ["51"]},
    "SATVS": {"1": ["14"], "2": ["21"]},
    "SATYRICVS": {"-": ["21"]},
    "SATYRVS": {"-": ["12"], "N": ["12"]},
    "SAVCIATIO": {"-": ["13"]},
    "SAVCIO": {"-": ["51"]},
    "SAVCIVS": {"-": ["21"]},
    "SAVILLVM": {"-": ["12"]},
    "SAVIOR": {"-": ["5J"]},
    "SAVIVM": {"-": ["12"]},
    "SAXEVS": {"-": ["21"]},
    "SAXIFICVS": {"-": ["21"]},
    "SAXOSA": {"-": ["12"]},
    "SAXOSVS": {"-": ["21"]},
    "SAXVM": {"-": ["12"], "N": ["12"]},
    "SCABELLVM": {"-": ["12"]},
    "SCABER": {"-": ["21"]},
    "SCABIES": {"-": ["15"]},
    "SCABIOSVS": {"-": ["21"]},
    "SCABO": {"-": ["53"]},
    "SCAEVOLA": {"N": ["11"]},
    "SCAEVVS": {"-": ["21"]},
    "SCALAE": {"-": ["11"]},
    "SCALMVS": {"-": ["12"]},
    "SCALPELLVM": {"-": ["12"]},
    "SCALPO": {"-": ["53"]},
    "SCALPRVM": {"-": ["12"]},
    "SCALPTVRA": {"-": ["11"]},
    "SCAMILLVS": {"-": ["12"]},
    "SCAMMONIVM": {"-": ["12"]},
    "SCAMNVM": {"-": ["12"]},
    "SCANDENS": {"-": ["25"]},
    "SCANDILIANVS": {"N": ["21"], "A": ["21"]},
    "SCANDILIVS": {"N": ["12"]},
    "SCANDO": {"-": ["53"]},
    "SCANDVLA": {"-": ["11"]},
    "SCANSIO": {"-": ["13"]},
    "SCANSOR": {"-": ["13"]},
    "SCANSORIVS": {"-": ["21"]},
    "SCAPHE": {"-": ["17"]},
    "SCAPHIVM": {"-": ["12"]},
    "SCAPVLA": {"-": ["11"], "N": ["11"]},
    "SCAPVS": {"-": ["12"]},
    "SCARABEVS": {"-": ["12"]},
    "SCARVS": {"-": ["12"]},
    "SCATEBRA": {"-": ["11"]},
    "SCATEO": {"-": ["52"]},
    "SCATO": {"-": ["53"], "N": ["13"]},
    "SCATVRIGO": {"-": ["13"]},
    "SCAVRVS": {"-": ["21"], "N": ["12"]},
    "SCELERATE": {"-": ["60"]},
    "SCELERATI": {"-": ["12"]},
    "SCELERATVS": {"-": ["21"], "N": ["21"], "A": ["21"]},
    "SCELERO": {"1": ["13"], "2": ["51"]},
    "SCELEROSVS": {"-": ["21"]},
    "SCELESTVS": {"-": ["21"]},
    "SCELVS": {"-": ["13"], "N": ["12"]},
    "SCENA": {"-": ["11"]},
    "SCENICA": {"-": ["11"]},
    "SCENICVS": {"1": ["12"], "2": ["21"]},
    "SCEPTRIFER": {"-": ["21"]},
    "SCEPTRVM": {"-": ["12"]},
    "SCEPTVCHVS": {"-": ["12"]},
    "SCHEDA": {"-": ["11"]},
    "SCHEMA": {"-": ["16"]},
    "SCHERINI": {"N": ["12"]},
    "SCHIDIA": {"-": ["11"]},
    "SCHOENOBATES": {"-": ["17"]},
    "SCHOENVS": {"-": ["12"]},
    "SCHOLA": {"-": ["11"], "N": ["11"]},
    "SCHOLASTICA": {"-": ["12"]},
    "SCHOLASTICVS": {"1": ["12"], "2": ["21"]},
    "SCIATHVS": {"N": ["12"]},
    "SCIENS": {"-": ["25"]},
    "SCIENTER": {"-": ["60"]},
    "SCIENTIA": {"-": ["11"]},
    "SCILICET": {"-": ["60"]},
    "SCILLA": {"-": ["11"]},
    "SCINDO": {"-": ["53"]},
    "SCINTILLA": {"-": ["11"], "N": ["11"]},
    "SCINTILLO": {"-": ["51"]},
    "SCIO": {"-": ["54"]},
    "SCIOTHERES": {"-": ["17"]},
    "SCIPIO": {"-": ["13"], "N": ["13"]},
    "SCIRPEA": {"-": ["11"]},
    "SCIRPICVLVS": {"1": ["12"], "2": ["21"]},
    "SCISCITOR": {"-": ["5J"]},
    "SCISCO": {"-": ["53"]},
    "SCISSVRA": {"-": ["11"]},
    "SCITE": {"-": ["60"]},
    "SCITOR": {"-": ["5J"]},
    "SCITVM": {"-": ["12"]},
    "SCITVS": {"1": ["14"], "2": ["21"]},
    "SCOMBER": {"-": ["12"]},
    "SCOPAE": {"-": ["11"]},
    "SCOPIO": {"-": ["13"]},
    "SCOPVLA": {"-": ["11"]},
    "SCOPVLVS": {"-": ["12"]},
    "SCORDALVS": {"-": ["12"]},
    "SCORPIO": {"-": ["13"]},
    "SCORPIVS": {"-": ["17"], "N": ["12"]},
    "SCORTATOR": {"-": ["13"]},
    "SCORTILLVM": {"-": ["12"]},
    "SCORTVM": {"-": ["12"], "N": ["12"]},
    "SCOTEINOS": {"-": ["27"]},
    "SCOTIA": {"-": ["11"]},
    "SCREO": {"-": ["51"]},
    "SCRIBA": {"-": ["11"]},
    "SCRIBLITA": {"-": ["11"]},
    "SCRIBO": {"-": ["53"]},
    "SCRINIVM": {"-": ["12"]},
    "SCRIPTA": {"-": ["12"]},
    "SCRIPTOR": {"-": ["13"]},
    "SCRIPTVLVM": {"-": ["12"]},
    "SCRIPTVM": {"-": ["12"]},
    "SCRIPTVRA": {"-": ["11"]},
    "SCRIPVLVM": {"-": ["12"]},
    "SCROBIS": {"-": ["13"]},
    "SCROFA": {"-": ["11"]},
    "SCRVPEVS": {"-": ["21"]},
    "SCRVPOSVS": {"-": ["21"]},
    "SCRVPVLOSVS": {"-": ["21"]},
    "SCRVPVLVS": {"-": ["12"]},
    "SCRVTA": {"-": ["12"]},
    "SCRVTANS": {"-": ["25"]},
    "SCRVTATIO": {"-": ["13"]},
    "SCRVTOR": {"-": ["5J"]},
    "SCVLPO": {"-": ["53"]},
    "SCVLPONEAE": {"-": ["11"]},
    "SCVLPTVRA": {"-": ["11"]},
    "SCVRRA": {"-": ["11"]},
    "SCVRRILIS": {"-": ["24"]},
    "SCVRRILITAS": {"-": ["13"]},
    "SCVTATVS": {"-": ["21"]},
    "SCVTICA": {"-": ["11"]},
    "SCVTRA": {"-": ["11"]},
    "SCVTRISCVM": {"-": ["12"]},
    "SCVTVLA": {"-": ["11"]},
    "SCVTVLATA": {"-": ["12"]},
    "SCVTVM": {"-": ["12"]},
    "SCYLLA": {"N": ["11"]},
    "SCYMNVS": {"-": ["12"]},
    "SCYPHVS": {"-": ["12"]},
    "SCYTHAE": {"N": ["11"]},
    "SCYTHIA": {"N": ["11"]},
    "SEBVM": {"-": ["12"]},
    "SECEDO": {"1": ["53"]},
    "SECENTI": {"-": ["31"]},
    "SECERNO": {"-": ["53"]},
    "SECESSIO": {"-": ["13"]},
    "SECESSVS": {"-": ["14"]},
    "SECLVDO": {"-": ["53"]},
    "SECLVSVS": {"-": ["21"]},
    "SECO": {"-": ["51"]},
    "SECRETE": {"-": ["60"]},
    "SECRETO": {"-": ["60"]},
    "SECRETVM": {"-": ["12"]},
    "SECRETVS": {"-": ["21"]},
    "SECRO": {"-": ["51"]},
    "SECROR": {"-": ["5J"]},
    "SECTA": {"-": ["11"]},
    "SECTATOR": {"-": ["13"]},
    "SECTILIS": {"-": ["24"]},
    "SECTIO": {"-": ["13"]},
    "SECTIVVS": {"-": ["21"]},
    "SECTOR": {"1": ["13"], "2": ["5J"]},
    "SECTVRA": {"-": ["11"]},
    "SECVBITVS": {"-": ["14"]},
    "SECVBO": {"-": ["51"]},
    "SECVNDAE": {"-": ["11"]},
    "SECVNDANI": {"-": ["12"]},
    "SECVNDANVS": {"-": ["21"]},
    "SECVNDARIVS": {"-": ["21"]},
    "SECVNDE": {"-": ["60"]},
    "SECVNDO": {"1": ["35"], "2": ["51"]},
    "SECVNDVM": {"1": ["60"], "2": ["70"]},
    "SECVNDVS": {"1": ["21"], "2": ["32"], "N": ["12"]},
    "SECVRE": {"-": ["60"]},
    "SECVRICLATVS": {"-": ["21"]},
    "SECVRICVLA": {"-": ["11"]},
    "SECVRIGER": {"-": ["21"]},
    "SECVRIS": {"-": ["13"]},
    "SECVRITAS": {"-": ["13"]},
    "SECVRVS": {"-": ["21"]},
    "SECVS": {"1": ["16"], "2": ["60"], "3": ["70"]},
    "SECVTOR": {"-": ["13"]},
    "SED": {"-": ["81"]},
    "SEDAMEN": {"-": ["13"]},
    "SEDATIO": {"-": ["13"]},
    "SEDATVS": {"-": ["21"]},
    "SEDECIM": {"-": ["31"]},
    "SEDENTARIVS": {"-": ["21"]},
    "SEDEO": {"-": ["52"]},
    "SEDES": {"-": ["13"]},
    "SEDILE": {"-": ["13"]},
    "SEDITIO": {"-": ["13"]},
    "SEDITIOSE": {"-": ["60"]},
    "SEDITIOSVS": {"-": ["21"]},
    "SEDO": {"-": ["51"]},
    "SEDVCO": {"-": ["53"]},
    "SEDVCTVS": {"-": ["21"]},
    "SEDVLITAS": {"-": ["13"]},
    "SEDVLO": {"-": ["60"]},
    "SEDVLVS": {"-": ["21"]},
    "SEGES": {"-": ["13"]},
    "SEGESTA": {"N": ["11"]},
    "SEGESTANI": {"N": ["12"]},
    "SEGESTANVS": {"N": ["21"], "A": ["21"]},
    "SEGMEN": {"-": ["13"]},
    "SEGMENTATVS": {"-": ["21"]},
    "SEGMENTVM": {"-": ["12"]},
    "SEGNE": {"-": ["60"]},
    "SEGNIPES": {"-": ["25"]},
    "SEGNIS": {"-": ["24"]},
    "SEGNITER": {"-": ["60"]},
    "SEGNITIA": {"-": ["11"]},
    "SEGNITIES": {"-": ["15"]},
    "SEGREGO": {"-": ["51"]},
    "SEGREX": {"-": ["25"]},
    "SEIANIANVS": {"N": ["21"], "A": ["21"]},
    "SEIANVS": {"N": ["12"]},
    "SEIVNCTIM": {"-": ["60"]},
    "SEIVNCTVS": {"-": ["21"]},
    "SEIVNGO": {"-": ["53"]},
    "SELECTVS": {"C": ["21"], "-": ["21"]},
    "SELENE": {"N": ["17"]},
    "SELIBRA": {"-": ["11"]},
    "SELIGO": {"3": ["53"]},
    "SELLA": {"-": ["11"]},
    "SELLARIVS": {"-": ["12"]},
    "SELLISTERNIVM": {"-": ["12"]},
    "SEMEL": {"-": ["36"]},
    "SEMEN": {"1": ["13"], "2": ["16"]},
    "SEMENTIS": {"-": ["13"]},
    "SEMENTIVVS": {"-": ["21"], "A": ["21"]},
    "SEMESTRIS": {"-": ["24"]},
    "SEMESVS": {"-": ["21"]},
    "SEMIADAPERTVS": {"-": ["21"]},
    "SEMIANIMIS": {"-": ["24"]},
    "SEMIANIMVS": {"-": ["21"]},
    "SEMIBOS": {"-": ["25"]},
    "SEMICANALICVLVS": {"-": ["12"]},
    "SEMIDEA": {"-": ["11"]},
    "SEMIDEVS": {"1": ["12"], "2": ["21"]},
    "SEMIERMIS": {"-": ["24"]},
    "SEMIERMVS": {"-": ["21"]},
    "SEMIFACTVS": {"-": ["21"]},
    "SEMIFASTIGIVM": {"-": ["12"]},
    "SEMIFER": {"-": ["21"]},
    "SEMIFVNIVM": {"-": ["12"]},
    "SEMIGERMANVS": {"-": ["21"]},
    "SEMIGRAVIS": {"-": ["24"]},
    "SEMIHIANS": {"-": ["25"]},
    "SEMIHOMO": {"-": ["13"]},
    "SEMIHORA": {"-": ["11"]},
    "SEMILATER": {"-": ["16"]},
    "SEMILAVTVS": {"-": ["21"]},
    "SEMILIBER": {"-": ["21"]},
    "SEMIMARINVS": {"-": ["21"]},
    "SEMIMAS": {"-": ["25"]},
    "SEMIMETOPIA": {"-": ["12"]},
    "SEMIMORTVVS": {"-": ["21"]},
    "SEMINARIVM": {"-": ["12"]},
    "SEMINARIVS": {"-": ["21"]},
    "SEMINATOR": {"-": ["13"]},
    "SEMINEX": {"-": ["25"]},
    "SEMINIVM": {"-": ["12"]},
    "SEMINO": {"-": ["51"]},
    "SEMINVDVS": {"-": ["21"]},
    "SEMIPAGANVS": {"-": ["21"]},
    "SEMIPES": {"-": ["13"]},
    "SEMIPLENVS": {"-": ["21"]},
    "SEMIPVTATVS": {"-": ["21"]},
    "SEMIRASVS": {"-": ["21"]},
    "SEMIREDVCTVS": {"-": ["21"]},
    "SEMIREFECTVS": {"-": ["21"]},
    "SEMIRVTVS": {"-": ["21"]},
    "SEMIS": {"-": ["16"], "2": ["13"], "1": ["21"]},
    "SEMISEPVLTVS": {"-": ["21"]},
    "SEMISOMNIS": {"-": ["24"]},
    "SEMISOMNVS": {"-": ["21"]},
    "SEMISVPINVS": {"-": ["21"]},
    "SEMITA": {"-": ["11"]},
    "SEMITARIVS": {"-": ["21"]},
    "SEMIVIR": {"-": ["12"]},
    "SEMIVIVVS": {"-": ["21"]},
    "SEMIVSTVS": {"-": ["21"]},
    "SEMODIALIS": {"-": ["24"]},
    "SEMODIVS": {"-": ["12"]},
    "SEMOTVS": {"-": ["21"]},
    "SEMOVEO": {"-": ["52"]},
    "SEMPER": {"-": ["60"]},
    "SEMPITERNO": {"-": ["60"]},
    "SEMPITERNVM": {"-": ["60"]},
    "SEMPITERNVS": {"-": ["21"]},
    "SEMPRONIVS": {"A": ["21"], "N": ["12"]},
    "SEMVNCIA": {"-": ["11"]},
    "SEMVSTVLATVS": {"-": ["21"]},
    "SENARIOLVS": {"-": ["12"]},
    "SENATOR": {"-": ["13"]},
    "SENATORIVS": {"-": ["21"]},
    "SENATVS": {"-": ["14"]},
    "SENECA": {"N": ["11"]},
    "SENECTA": {"-": ["11"]},
    "SENECTVS": {"1": ["13"], "2": ["21"], "N": ["12"]},
    "SENEO": {"-": ["52"]},
    "SENESCO": {"-": ["53"]},
    "SENEX": {"1": ["13"], "2": ["26"]},
    "SENI": {"-": ["33"]},
    "SENILIS": {"-": ["24"]},
    "SENIO": {"-": ["13"]},
    "SENIOR": {"-": ["13"]},
    "SENIOSVS": {"-": ["21"]},
    "SENIVM": {"-": ["12"]},
    "SENSA": {"-": ["12"]},
    "SENSATIO": {"-": ["13"]},
    "SENSIBILIS": {"-": ["24"]},
    "SENSIFER": {"-": ["21"]},
    "SENSILIS": {"-": ["24"]},
    "SENSIM": {"-": ["60"]},
    "SENSVS": {"-": ["14"]},
    "SENTENTIA": {"-": ["11"]},
    "SENTINA": {"-": ["11"]},
    "SENTINOSVS": {"-": ["21"]},
    "SENTIO": {"-": ["54"]},
    "SENTIOR": {"-": ["5M"]},
    "SENTIS": {"-": ["13"]},
    "SENTISCO": {"-": ["53"]},
    "SENTIVS": {"N": ["12"]},
    "SENTVS": {"-": ["21"]},
    "SEORSVM": {"1": ["60"], "2": ["70"]},
    "SEPARATE": {"-": ["60"]},
    "SEPARATIM": {"-": ["60"]},
    "SEPARATVM": {"-": ["12"]},
    "SEPARATVS": {"1": ["14"], "2": ["21"]},
    "SEPARO": {"2": ["51"]},
    "SEPELIO": {"-": ["54"]},
    "SEPES": {"-": ["13"]},
    "SEPONO": {"-": ["53"]},
    "SEPOSITVS": {"-": ["21"]},
    "SEPSE": {"-": ["43"]},
    "SEPTEM": {"-": ["31"], "N": ["16"]},
    "SEPTEMBER": {"A": ["23"], "N": ["13"]},
    "SEPTEMGEMINVS": {"-": ["21"]},
    "SEPTEMPLEX": {"-": ["34", "22"]},
    "SEPTEMREMIS": {"-": ["25"]},
    "SEPTEMVIR": {"-": ["12"]},
    "SEPTENDECIM": {"-": ["31"]},
    "SEPTENDECIMVS": {"-": ["32"]},
    "SEPTENTRIO": {"-": ["13"], "N": ["13"]},
    "SEPTENTRIONALIA": {"-": ["13"]},
    "SEPTENTRIONALIS": {"-": ["24"]},
    "SEPTENVS": {"-": ["33"]},
    "SEPTICIVS": {"N": ["12"]},
    "SEPTIMIVS": {"N": ["12"]},
    "SEPTIMVM": {"-": ["36"]},
    "SEPTIMVS": {"-": ["32"]},
    "SEPTINGENTESIMVS": {"-": ["32"]},
    "SEPTINGENTI": {"-": ["31"]},
    "SEPTVAGESIMVS": {"-": ["32"]},
    "SEPTVAGINTA": {"-": ["31"]},
    "SEPVLCRALIS": {"-": ["24"]},
    "SEPVLCRETVM": {"-": ["12"]},
    "SEPVLCRVM": {"-": ["12"]},
    "SEPVLTVRA": {"-": ["11"]},
    "SEQVAX": {"-": ["25"]},
    "SEQVENS": {"-": ["25"]},
    "SEQVESTER": {"1": ["16"], "2": ["26"]},
    "SEQVESTRO": {"2": ["60"]},
    "SEQVOR": {"-": ["5L"]},
    "SERA": {"1": ["11"], "2": ["60"]},
    "SERAPIS": {"N": ["17"]},
    "SERARIVS": {"-": ["21"]},
    "SERENITAS": {"-": ["13"]},
    "SERENO": {"-": ["51"]},
    "SERENVM": {"-": ["12"]},
    "SERENVS": {"-": ["21"], "N": ["12"]},
    "SERESCO": {"1": ["53"]},
    "SERGIVS": {"N": ["12"], "A": ["21"]},
    "SERIA": {"-": ["11"]},
    "SERICA": {"-": ["12"]},
    "SERICVS": {"-": ["21"], "A": ["21"]},
    "SERIES": {"-": ["15"]},
    "SERIO": {"-": ["60"]},
    "SERIOLA": {"-": ["11"]},
    "SERIPHVS": {"N": ["12"]},
    "SERIVM": {"-": ["12"]},
    "SERIVS": {"-": ["21"]},
    "SERMO": {"-": ["13"], "N": ["13"]},
    "SERMOCINOR": {"-": ["5J"]},
    "SERO": {"1": ["51"], "2": ["53"], "3": ["53"], "4": ["60"]},
    "SERPENS": {"-": ["13"]},
    "SERPO": {"-": ["53"]},
    "SERPYLLVM": {"-": ["12"]},
    "SERRA": {"-": ["11"]},
    "SERRARIVS": {"-": ["12"]},
    "SERRATIM": {"-": ["60"]},
    "SERRATVS": {"-": ["21"]},
    "SERTA": {"-": ["11"]},
    "SERTIVS": {"N": ["12"]},
    "SERTORIANVS": {"N": ["21"], "A": ["21"]},
    "SERTORIVS": {"N": ["12"]},
    "SERTVM": {"-": ["12"]},
    "SERVA": {"-": ["11"]},
    "SERVANS": {"-": ["25"]},
    "SERVATOR": {"-": ["13"]},
    "SERVIENS": {"-": ["13"]},
    "SERVILIA": {"N": ["11"]},
    "SERVILIS": {"-": ["24"]},
    "SERVILIVS": {"N": ["12"], "A": ["21"]},
    "SERVIO": {"-": ["54"]},
    "SERVITIVM": {"-": ["12"]},
    "SERVITVS": {"-": ["13"], "N": ["12"]},
    "SERVIVS": {"N": ["12"]},
    "SERVM": {"1": ["12"], "2": ["12"]},
    "SERVO": {"-": ["51"]},
    "SERVS": {"-": ["21"]},
    "SERVVLVS": {"-": ["12"]},
    "SERVVS": {"1": ["12"], "2": ["21"]},
    "SESAMA": {"-": ["11"]},
    "SESCENI": {"-": ["33"]},
    "SESCENTESIMVS": {"-": ["32"]},
    "SESCENTI": {"-": ["31"]},
    "SESCENTIES": {"-": ["36"]},
    "SESQVIALTER": {"-": ["48"]},
    "SESQVIDIGITVS": {"-": ["12"]},
    "SESQVILIBRA": {"-": ["11"]},
    "SESQVIMODIVS": {"-": ["12"]},
    "SESQVIPEDALIS": {"-": ["24"]},
    "SESQVIPES": {"-": ["13"]},
    "SESQVIPLAGA": {"-": ["11"]},
    "SESSILIS": {"-": ["24"]},
    "SESSIO": {"-": ["13"]},
    "SESSOR": {"-": ["13"]},
    "SESTERTIVM": {"-": ["12"]},
    "SESTERTIVS": {"1": ["12"], "2": ["21"]},
    "SEV": {"1": ["81"], "2": ["82"]},
    "SEVEHO": {"-": ["53"]},
    "SEVERA": {"-": ["12"]},
    "SEVERE": {"-": ["60"]},
    "SEVERI": {"-": ["12"]},
    "SEVERITAS": {"-": ["13"]},
    "SEVERO": {"-": ["51"]},
    "SEVERVS": {"-": ["21"], "N": ["12"]},
    "SEVOCO": {"-": ["51"]},
    "SEX": {"-": ["31"]},
    "SEXAGENI": {"-": ["33"]},
    "SEXAGESIMVS": {"-": ["32"]},
    "SEXAGINTA": {"-": ["31"]},
    "SEXCENARIVS": {"-": ["21"]},
    "SEXENNIS": {"-": ["24"]},
    "SEXIES": {"-": ["36"]},
    "SEXIS": {"-": ["13"]},
    "SEXTANS": {"-": ["13"]},
    "SEXTARIVS": {"-": ["12"]},
    "SEXTILIS": {"-": ["13"], "A": ["24"], "N": ["13"]},
    "SEXTIVS": {"N": ["12"], "A": ["21"]},
    "SEXTVLA": {"-": ["11"]},
    "SEXTVM": {"-": ["35"]},
    "SEXTVS": {"-": ["32"], "N": ["12"]},
    "SEXVS": {"-": ["14"]},
    "SI": {"1": ["67"], "2": ["82"]},
    "SIBILO": {"-": ["51"]},
    "SIBILVM": {"-": ["12"]},
    "SIBILVS": {"1": ["12"], "2": ["21"]},
    "SIBYLLA": {"N": ["11"]},
    "SIBYLLINVS": {"-": ["21"], "A": ["21"]},
    "SIC": {"-": ["60"]},
    "SICA": {"-": ["11"]},
    "SICARIVS": {"-": ["12"]},
    "SICCESCO": {"-": ["51"]},
    "SICCITAS": {"-": ["13"]},
    "SICCIVS": {"N": ["12"]},
    "SICCO": {"-": ["51"]},
    "SICCVM": {"-": ["12"]},
    "SICCVS": {"-": ["21"]},
    "SICILIA": {"N": ["11"]},
    "SICILICVS": {"-": ["12"]},
    "SICILIENSIS": {"N": ["24"], "A": ["24"]},
    "SICILIMENTVM": {"-": ["12"]},
    "SICINE": {"-": ["67"]},
    "SICINIVS": {"N": ["12"]},
    "SICVBI": {"-": ["82"]},
    "SICVLA": {"-": ["11"]},
    "SICVLI": {"N": ["12"]},
    "SICVLVS": {"N": ["21"], "A": ["21"]},
    "SICVT": {"1": ["66"], "2": ["82"]},
    "SICYONIVS": {"N": ["21"], "A": ["21"]},
    "SIDEO": {"-": ["52"]},
    "SIDEREVS": {"-": ["21"]},
    "SIDERO": {"-": ["51"]},
    "SIDO": {"-": ["53"], "N": ["13"]},
    "SIDVS": {"-": ["13"]},
    "SIGILLARIA": {"-": ["16"]},
    "SIGILLATVS": {"-": ["21"]},
    "SIGILLVM": {"-": ["12"]},
    "SIGNATOR": {"-": ["13"]},
    "SIGNATVS": {"-": ["21"]},
    "SIGNIA": {"N": ["11"]},
    "SIGNIFER": {"1": ["12"], "2": ["21"]},
    "SIGNIFICANS": {"-": ["25"]},
    "SIGNIFICATIO": {"-": ["13"]},
    "SIGNIFICATVM": {"-": ["12"]},
    "SIGNIFICATVS": {"-": ["14"]},
    "SIGNIFICO": {"-": ["51"]},
    "SIGNO": {"-": ["51"]},
    "SIGNVM": {"-": ["12"]},
    "SIL": {"-": ["13"]},
    "SILACEVS": {"-": ["21"]},
    "SILANIO": {"N": ["13"]},
    "SILANVS": {"N": ["12"], "-": ["12"]},
    "SILENS": {"-": ["25"]},
    "SILENTIVM": {"-": ["12"]},
    "SILEO": {"-": ["52"]},
    "SILER": {"-": ["13"]},
    "SILESCO": {"-": ["53"]},
    "SILEX": {"-": ["13"]},
    "SILICEVS": {"-": ["21"]},
    "SILICVLVS": {"-": ["12"]},
    "SILIGINEVS": {"1": ["12"], "2": ["21"]},
    "SILIGO": {"-": ["13"]},
    "SILIO": {"-": ["54"]},
    "SILIQVA": {"-": ["11"]},
    "SILPHIVM": {"-": ["12"]},
    "SILVA": {"-": ["11"]},
    "SILVATICVS": {"-": ["21"]},
    "SILVESTER": {"-": ["23"]},
    "SILVESTRIA": {"-": ["13"]},
    "SILVESTRIS": {"-": ["24"]},
    "SILVIA": {"N": ["11"]},
    "SILVICOLA": {"-": ["11"]},
    "SILVICVLTRIX": {"-": ["13"]},
    "SILVIFRAGVS": {"-": ["21"]},
    "SILVIVS": {"A": ["21"], "N": ["12"]},
    "SILVOSVS": {"-": ["21"]},
    "SILVRVS": {"-": ["12"]},
    "SIMA": {"-": ["11"]},
    "SIMIA": {"-": ["11"]},
    "SIMILAGO": {"-": ["13"]},
    "SIMILE": {"1": ["13"], "2": ["60"]},
    "SIMILIA": {"-": ["13"]},
    "SIMILIS": {"-": ["24"]},
    "SIMILITAS": {"-": ["13"]},
    "SIMILITER": {"-": ["60"]},
    "SIMILITVDO": {"-": ["13"]},
    "SIMIVS": {"-": ["12"]},
    "SIMO": {"-": ["51"], "N": ["11"]},
    "SIMPLEX": {"-": ["34", "22"], "N": ["13"]},
    "SIMPLICITAS": {"-": ["13"]},
    "SIMPLICITER": {"-": ["60"]},
    "SIMPLVM": {"-": ["12"]},
    "SIMPVLVM": {"-": ["12"]},
    "SIMPVVIVM": {"-": ["12"]},
    "SIMVL": {"1": ["60"], "2": ["70"], "3": ["82"]},
    "SIMVLAC": {"1": ["60"], "2": ["82"], "-": ["6-"]},
    "SIMVLACRVM": {"-": ["12"]},
    "SIMVLAMEN": {"-": ["13"]},
    "SIMVLANS": {"-": ["25"], "N": ["13"]},
    "SIMVLATIO": {"-": ["13"]},
    "SIMVLATOR": {"-": ["13"]},
    "SIMVLATQVE": {"1": ["60"], "2": ["82"], "-": ["6-"]},
    "SIMVLO": {"-": ["51"]},
    "SIMVLTAS": {"-": ["13"]},
    "SIMVLVS": {"-": ["21"]},
    "SIMVS": {"-": ["21"]},
    "SIN": {"-": ["82"]},
    "SINCERE": {"-": ["60"]},
    "SINCERITAS": {"-": ["13"]},
    "SINCERVS": {"-": ["21"]},
    "SINCIPVT": {"-": ["13"]},
    "SINE": {"-": ["70"]},
    "SINGILLARITER": {"-": ["60"]},
    "SINGILLATIM": {"-": ["60"]},
    "SINGVLA": {"1": ["11"], "2": ["12"]},
    "SINGVLARES": {"-": ["13"], "N": ["13"]},
    "SINGVLARIS": {"-": ["24"]},
    "SINGVLARITER": {"-": ["60"]},
    "SINGVLATIM": {"-": ["60"]},
    "SINGVLTIM": {"-": ["60"]},
    "SINGVLTIO": {"-": ["54"]},
    "SINGVLTVS": {"-": ["14"]},
    "SINGVLVM": {"-": ["12"]},
    "SINGVLVS": {"-": ["33"]},
    "SINISTER": {"-": ["21"]},
    "SINISTRA": {"1": ["11"], "2": ["12"]},
    "SINISTRE": {"-": ["60"]},
    "SINISTRORSVM": {"-": ["60"]},
    "SINISTRORSVS": {"-": ["60"]},
    "SINISTRVM": {"-": ["12"]},
    "SINO": {"-": ["53"]},
    "SINOPE": {"N": ["17"]},
    "SINVO": {"-": ["51"]},
    "SINVOSVS": {"-": ["21"]},
    "SINVS": {"-": ["14"]},
    "SIPARIVM": {"-": ["12"]},
    "SIPHO": {"-": ["13"]},
    "SIPIO": {"-": ["55"]},
    "SIQVIDEM": {"-": ["82"]},
    "SIREMPS": {"-": ["48"]},
    "SIRPICIVM": {"-": ["12"]},
    "SIRVS": {"-": ["12"]},
    "SISENNA": {"N": ["11"]},
    "SISER": {"-": ["13"]},
    "SISTO": {"-": ["53"]},
    "SISTRVM": {"-": ["12"]},
    "SISYPHVS": {"N": ["17"]},
    "SITICVLOSVS": {"-": ["21"]},
    "SITIO": {"-": ["54"]},
    "SITIS": {"-": ["13"]},
    "SITVLVS": {"-": ["12"]},
    "SITVS": {"1": ["14"], "2": ["21"]},
    "SIVE": {"1": ["81", "82"], "2": ["82"]},
    "SMARAGDVS": {"-": ["12"]},
    "SMYRNA": {"-": ["11"], "N": ["11"]},
    "SMYRNION": {"-": ["12"]},
    "SOBRIETAS": {"-": ["13"]},
    "SOBRINA": {"-": ["11"]},
    "SOBRIVS": {"-": ["21"]},
    "SOCCATVS": {"-": ["21"]},
    "SOCCVLVS": {"-": ["12"]},
    "SOCCVS": {"-": ["12"]},
    "SOCER": {"-": ["12"]},
    "SOCIA": {"-": ["11"]},
    "SOCIABILIS": {"-": ["24"]},
    "SOCIALIS": {"-": ["24"]},
    "SOCIALITAS": {"-": ["13"]},
    "SOCIETAS": {"-": ["13"]},
    "SOCIO": {"-": ["51"]},
    "SOCIVS": {"1": ["12"], "2": ["21"]},
    "SOCORDIA": {"-": ["11"]},
    "SOCORDITER": {"-": ["6&"]},
    "SOCORS": {"-": ["25"]},
    "SOCRATES": {"N": ["17"]},
    "SOCRVS": {"-": ["14"]},
    "SODALICIVM": {"-": ["12"]},
    "SODALIS": {"1": ["13"], "2": ["24"]},
    "SODALITAS": {"-": ["13"]},
    "SODALITIVM": {"-": ["12"]},
    "SODES": {"-": ["60"]},
    "SOL": {"-": ["13"], "N": ["13"]},
    "SOLACIOLVM": {"-": ["12"]},
    "SOLACIVM": {"-": ["12"]},
    "SOLAMEN": {"-": ["13"]},
    "SOLATOR": {"-": ["13"]},
    "SOLDVRII": {"-": ["12"]},
    "SOLEA": {"-": ["11"]},
    "SOLEATVS": {"-": ["21"]},
    "SOLENNE": {"-": ["13"]},
    "SOLENS": {"-": ["25"]},
    "SOLEO": {"-": ["5S"]},
    "SOLICISMVS": {"-": ["12"]},
    "SOLIDATIO": {"-": ["13"]},
    "SOLIDESCO": {"-": ["53"]},
    "SOLIDITAS": {"-": ["13"]},
    "SOLIDO": {"-": ["51"]},
    "SOLIDVM": {"1": ["12"], "2": ["60"]},
    "SOLIDVS": {"1": ["12"], "2": ["21"]},
    "SOLIFER": {"-": ["21"]},
    "SOLITARIVS": {"-": ["21"]},
    "SOLITVDO": {"-": ["13"]},
    "SOLITVM": {"-": ["12"]},
    "SOLITVS": {"-": ["21"]},
    "SOLIVAGVS": {"-": ["21"]},
    "SOLIVM": {"-": ["12"]},
    "SOLLEMNE": {"-": ["13"]},
    "SOLLEMNIS": {"-": ["24"]},
    "SOLLERS": {"-": ["25"], "N": ["13"]},
    "SOLLERTER": {"-": ["60"]},
    "SOLLERTIA": {"-": ["11"]},
    "SOLLICITATIO": {"-": ["13"]},
    "SOLLICITE": {"-": ["60"]},
    "SOLLICITO": {"-": ["51"]},
    "SOLLICITVDO": {"-": ["13"]},
    "SOLLICITVS": {"-": ["21"]},
    "SOLO": {"1": ["51"], "2": ["51"]},
    "SOLOECISMVS": {"-": ["12"]},
    "SOLOR": {"-": ["5J"]},
    "SOLSTITIALIS": {"-": ["24"]},
    "SOLSTITIVM": {"-": ["12"]},
    "SOLVM": {"1": ["12"], "2": ["60"]},
    "SOLVNTINI": {"N": ["12"]},
    "SOLVNTINVS": {"N": ["21"], "A": ["21"]},
    "SOLVO": {"-": ["53"]},
    "SOLVS": {"-": ["48"]},
    "SOLVTE": {"-": ["60"]},
    "SOLVTIO": {"-": ["13"]},
    "SOLVTVM": {"-": ["12"]},
    "SOLVTVS": {"-": ["21"]},
    "SOMNIO": {"-": ["51"]},
    "SOMNIVM": {"-": ["12"], "N": ["12"]},
    "SOMNVS": {"-": ["12"], "N": ["12"]},
    "SONANS": {"-": ["25"]},
    "SONAX": {"-": ["25"]},
    "SONIPES": {"1": ["13"], "2": ["25"]},
    "SONITVS": {"-": ["14"]},
    "SONIVIVS": {"-": ["21"]},
    "SONO": {"1": ["51"], "2": ["53"]},
    "SONOR": {"-": ["13"]},
    "SONORVS": {"-": ["21"]},
    "SONS": {"1": ["13"], "2": ["25"]},
    "SONTICVS": {"-": ["21"]},
    "SONVS": {"-": ["16"]},
    "SOPATER": {"N": ["12"]},
    "SOPHIA": {"-": ["11"]},
    "SOPHISMA": {"-": ["17"]},
    "SOPHISTES": {"-": ["17"]},
    "SOPIO": {"-": ["54"]},
    "SOPITIO": {"-": ["13"]},
    "SOPOR": {"-": ["13"], "N": ["13"]},
    "SOPORATVS": {"-": ["21"]},
    "SOPORIFER": {"-": ["21"]},
    "SOPORO": {"-": ["51"]},
    "SOPORVS": {"-": ["21"]},
    "SORBEO": {"-": ["52"]},
    "SORBITIO": {"-": ["13"]},
    "SORBVM": {"-": ["12"]},
    "SORBVS": {"-": ["12"]},
    "SORDEO": {"-": ["52"]},
    "SORDES": {"-": ["13"]},
    "SORDESCO": {"-": ["53"]},
    "SORDIDATVS": {"-": ["21"]},
    "SORDIDE": {"-": ["60"]},
    "SORDIDVLVS": {"-": ["21"]},
    "SORDIDVS": {"-": ["21"]},
    "SORITES": {"-": ["17"]},
    "SOROR": {"-": ["13"], "N": ["13"]},
    "SORS": {"-": ["13"]},
    "SORTIOR": {"-": ["5M"]},
    "SORTITIO": {"-": ["13"]},
    "SORTITO": {"-": ["60"]},
    "SORTITOR": {"-": ["13"]},
    "SORTITVS": {"-": ["14"]},
    "SOSIPPVS": {"N": ["12"]},
    "SOSITHEVS": {"N": ["12"]},
    "SOSPES": {"-": ["22"]},
    "SOSPITO": {"-": ["51"]},
    "SOSTRATVS": {"N": ["12"]},
    "SPADIX": {"-": ["13"]},
    "SPADO": {"-": ["13"]},
    "SPARGO": {"1": ["13"], "2": ["53"]},
    "SPARSVS": {"-": ["21"]},
    "SPARTA": {"N": ["17"]},
    "SPARTEVS": {"-": ["21"]},
    "SPARTVM": {"-": ["12"]},
    "SPARVS": {"-": ["12"]},
    "SPATHA": {"-": ["11"]},
    "SPATIATOR": {"-": ["13"]},
    "SPATIOR": {"-": ["5J"]},
    "SPATIOSE": {"-": ["60"]},
    "SPATIOSVM": {"-": ["12"]},
    "SPATIOSVS": {"-": ["21"]},
    "SPATIVM": {"-": ["12"]},
    "SPECIALIS": {"-": ["24"]},
    "SPECIATIM": {"-": ["60"]},
    "SPECIES": {"-": ["15"]},
    "SPECILLVM": {"-": ["12"]},
    "SPECIMEN": {"-": ["13"]},
    "SPECIO": {"-": ["55"]},
    "SPECIOSE": {"-": ["60"]},
    "SPECIOSVS": {"-": ["21"]},
    "SPECTABILIS": {"-": ["24"]},
    "SPECTACVLVM": {"-": ["12"]},
    "SPECTATIO": {"-": ["13"]},
    "SPECTATOR": {"-": ["13"]},
    "SPECTATRIX": {"-": ["13"]},
    "SPECTATVS": {"-": ["21"]},
    "SPECTO": {"-": ["51"]},
    "SPECVLA": {"1": ["11"], "2": ["11"]},
    "SPECVLARE": {"-": ["13"]},
    "SPECVLARIA": {"-": ["16"]},
    "SPECVLARIS": {"1": ["24"], "2": ["24"], "-": ["24"]},
    "SPECVLATOR": {"-": ["13"]},
    "SPECVLATORIVS": {"-": ["21"]},
    "SPECVLATRIX": {"-": ["13"]},
    "SPECVLOR": {"-": ["5J"]},
    "SPECVLVM": {"-": ["12"]},
    "SPECVS": {"-": ["14"]},
    "SPELAEVM": {"-": ["12"]},
    "SPELVNCA": {"-": ["11"], "N": ["11"]},
    "SPERATA": {"-": ["11"]},
    "SPERATVM": {"-": ["12"]},
    "SPERATVS": {"1": ["12"], "2": ["21"]},
    "SPERGO": {"-": ["53"]},
    "SPERNENDVS": {"-": ["21"]},
    "SPERNENS": {"-": ["25"]},
    "SPERNO": {"-": ["53"]},
    "SPERNOR": {"-": ["5J"]},
    "SPERO": {"-": ["51"]},
    "SPES": {"-": ["15"], "N": ["15"]},
    "SPHAERA": {"-": ["11"]},
    "SPHAERITA": {"-": ["11"]},
    "SPHAEROIDES": {"-": ["27"]},
    "SPHAEROMACHIA": {"-": ["17"]},
    "SPICA": {"-": ["11"]},
    "SPICESCO": {"-": ["53"]},
    "SPICEVS": {"-": ["21"]},
    "SPICIFER": {"-": ["21"]},
    "SPICIO": {"-": ["55"]},
    "SPICO": {"-": ["51"]},
    "SPICVLVM": {"-": ["12"]},
    "SPINA": {"-": ["11"]},
    "SPINETVM": {"-": ["12"]},
    "SPINOSVS": {"-": ["21"]},
    "SPINTRIA": {"-": ["11"]},
    "SPINVS": {"-": ["12"]},
    "SPIRA": {"-": ["11"]},
    "SPIRABILIS": {"-": ["24"]},
    "SPIRACVLVM": {"-": ["12"]},
    "SPIRAMEN": {"-": ["13"]},
    "SPIRAMENTVM": {"-": ["12"]},
    "SPIRITVS": {"-": ["14"]},
    "SPIRO": {"-": ["51"]},
    "SPISSAMENTVM": {"-": ["12"]},
    "SPISSE": {"-": ["60"]},
    "SPISSITAS": {"-": ["13"]},
    "SPISSO": {"-": ["51"]},
    "SPISSVS": {"-": ["21"]},
    "SPLEN": {"-": ["17"]},
    "SPLENDENS": {"-": ["25"]},
    "SPLENDEO": {"-": ["52"]},
    "SPLENDESCO": {"-": ["53"]},
    "SPLENDIDE": {"-": ["60"]},
    "SPLENDIDVS": {"-": ["21"]},
    "SPLENDOR": {"-": ["13"]},
    "SPOLIARIVM": {"-": ["12"]},
    "SPOLIATIO": {"-": ["13"]},
    "SPOLIATOR": {"-": ["13"]},
    "SPOLIATVS": {"1": ["21"], "2": ["21"]},
    "SPOLIO": {"-": ["51"]},
    "SPOLIVM": {"-": ["12"]},
    "SPONDA": {"-": ["11"]},
    "SPONDEO": {"-": ["52"]},
    "SPONDYLVS": {"-": ["12"]},
    "SPONGIA": {"-": ["11"]},
    "SPONS": {"-": ["13"]},
    "SPONSA": {"-": ["11"]},
    "SPONSALIA": {"-": ["16"]},
    "SPONSIO": {"-": ["13"]},
    "SPONSOR": {"-": ["13"]},
    "SPONSVM": {"-": ["12"]},
    "SPONSVS": {"1": ["12"], "2": ["14"], "N": ["12"]},
    "SPONTE": {"-": ["60"]},
    "SPORTA": {"-": ["11"]},
    "SPORTVLA": {"-": ["11"]},
    "SPRETOR": {"-": ["13"]},
    "SPRETVS": {"1": ["14"], "2": ["21"]},
    "SPVMA": {"-": ["11"]},
    "SPVMANS": {"-": ["25"]},
    "SPVMESCO": {"-": ["53"]},
    "SPVMEVS": {"-": ["21"]},
    "SPVMIGER": {"-": ["21"]},
    "SPVMO": {"-": ["51"]},
    "SPVMOSVS": {"-": ["21"]},
    "SPVO": {"-": ["53"]},
    "SPVRCE": {"-": ["6&"]},
    "SPVRCITIA": {"-": ["11"]},
    "SPVRCITIES": {"-": ["15"]},
    "SPVRCO": {"-": ["51"]},
    "SPVRCVS": {"-": ["21"]},
    "SPVRIVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "SPVTVM": {"-": ["12"]},
    "SPVTVS": {"-": ["14"]},
    "SQVALENS": {"-": ["25"]},
    "SQVALEO": {"-": ["52"]},
    "SQVALIDVS": {"-": ["21"]},
    "SQVALOR": {"-": ["13"]},
    "SQVAMA": {"-": ["11"]},
    "SQVAMEVS": {"-": ["21"]},
    "SQVAMIFER": {"-": ["21"]},
    "SQVAMIGER": {"-": ["21"]},
    "SQVAMOSVS": {"-": ["21"]},
    "SQVILLA": {"-": ["11"]},
    "STABILIA": {"-": ["13"]},
    "STABILIMEN": {"-": ["13"]},
    "STABILIO": {"-": ["54"]},
    "STABILIS": {"-": ["24"]},
    "STABILITAS": {"-": ["13"]},
    "STABILITER": {"-": ["60"]},
    "STABILITOR": {"-": ["13"]},
    "STABVLARIVS": {"-": ["12"]},
    "STABVLO": {"-": ["51"]},
    "STABVLOR": {"-": ["5J"]},
    "STABVLVM": {"-": ["12"]},
    "STACTA": {"-": ["11"]},
    "STADIATVS": {"-": ["21"]},
    "STADIVM": {"-": ["12"]},
    "STAGNANS": {"-": ["25"]},
    "STAGNO": {"1": ["51"], "2": ["51"]},
    "STAGNVM": {"1": ["12"], "2": ["12"]},
    "STAIENVS": {"N": ["12"]},
    "STAMEN": {"-": ["13"]},
    "STAMINEVS": {"-": ["21"]},
    "STATERA": {"-": ["11"]},
    "STATICVLVS": {"-": ["12"]},
    "STATIM": {"-": ["60"]},
    "STATIO": {"-": ["13"]},
    "STATIVA": {"-": ["12"]},
    "STATIVS": {"N": ["12"]},
    "STATIVVS": {"-": ["21"]},
    "STATOR": {"-": ["13"], "N": ["13"]},
    "STATVA": {"-": ["11"]},
    "STATVARIVS": {"1": ["12"], "2": ["21"]},
    "STATVMEN": {"-": ["13"]},
    "STATVMINATIO": {"-": ["13"]},
    "STATVMINO": {"-": ["51"]},
    "STATVO": {"-": ["53"]},
    "STATVRA": {"-": ["11"]},
    "STATVS": {"1": ["14"], "2": ["21"]},
    "STATVTIO": {"-": ["13"]},
    "STATVTVM": {"-": ["12"]},
    "STATVTVS": {"-": ["21"]},
    "STELLA": {"-": ["11"], "N": ["11"]},
    "STELLIFER": {"-": ["21"]},
    "STELLIGER": {"-": ["21"]},
    "STELLIO": {"-": ["13"]},
    "STELLO": {"-": ["51"]},
    "STEMMA": {"-": ["17"]},
    "STERCORARIVS": {"-": ["21"]},
    "STERCORO": {"-": ["51"]},
    "STERCOROSVS": {"-": ["21"]},
    "STERCVLINVM": {"-": ["12"]},
    "STERCVS": {"-": ["13"]},
    "STEREOBATES": {"-": ["17"]},
    "STERILIS": {"-": ["24"]},
    "STERILITAS": {"-": ["13"]},
    "STERILVS": {"-": ["21"]},
    "STERNO": {"-": ["53"]},
    "STERNVMENTVM": {"-": ["12"]},
    "STERNVO": {"-": ["53"]},
    "STERNVTAMENTVM": {"-": ["12"]},
    "STERTO": {"-": ["53"]},
    "STESICHORVS": {"N": ["12"]},
    "STHENIANVS": {"N": ["21"], "A": ["21"]},
    "STHENIVS": {"N": ["12"]},
    "STIGMA": {"-": ["17"]},
    "STILLA": {"-": ["11"]},
    "STILLARIVM": {"-": ["12"]},
    "STILLICIDIVM": {"-": ["12"]},
    "STILLO": {"-": ["51"]},
    "STILVS": {"-": ["12"]},
    "STIMVLO": {"-": ["51"]},
    "STIMVLVS": {"-": ["12"]},
    "STINGVO": {"-": ["53"]},
    "STIPATOR": {"-": ["13"]},
    "STIPATVS": {"-": ["21"]},
    "STIPENDIALIS": {"-": ["24"]},
    "STIPENDIARIVS": {"1": ["12"], "2": ["21"]},
    "STIPENDIOR": {"-": ["5J"]},
    "STIPENDIOSVS": {"-": ["21"]},
    "STIPENDIVM": {"-": ["12"]},
    "STIPES": {"-": ["13"]},
    "STIPO": {"-": ["51"]},
    "STIPS": {"-": ["13"]},
    "STIPVLA": {"-": ["11"]},
    "STIPVLATIO": {"-": ["13"]},
    "STIPVLOR": {"-": ["5J"]},
    "STIRIA": {"-": ["11"]},
    "STIRPS": {"-": ["13"]},
    "STITVO": {"-": ["53"]},
    "STIVA": {"-": ["11"]},
    "STLATARIVS": {"-": ["12"]},
    "STLOPPVS": {"-": ["12"]},
    "STO": {"-": ["51"]},
    "STOECHION": {"-": ["17"]},
    "STOICA": {"1": ["12"], "2": ["17"]},
    "STOICI": {"N": ["12"]},
    "STOICVS": {"N": ["21"], "A": ["21"]},
    "STOLA": {"-": ["11"]},
    "STOLATVS": {"-": ["21"]},
    "STOLIDE": {"-": ["60"]},
    "STOLIDVS": {"-": ["21"]},
    "STOMACHICVS": {"-": ["12"]},
    "STOMACHOR": {"-": ["5J"]},
    "STOMACHOSVS": {"-": ["21"]},
    "STOMACHVS": {"-": ["12"]},
    "STOREA": {"-": ["11"]},
    "STRABO": {"-": ["13"], "N": ["13"]},
    "STRAGES": {"-": ["13"]},
    "STRAGVLA": {"-": ["11"]},
    "STRAGVLVM": {"-": ["12"]},
    "STRAGVLVS": {"-": ["21"]},
    "STRAMEN": {"-": ["13"]},
    "STRAMENTARIVS": {"-": ["21"]},
    "STRAMENTICIVS": {"-": ["21"]},
    "STRAMENTVM": {"-": ["12"]},
    "STRAMINEVS": {"-": ["21"]},
    "STRANGVLO": {"-": ["51"]},
    "STRANGVRIA": {"-": ["11"]},
    "STRATVM": {"-": ["12"]},
    "STRENVE": {"-": ["60"]},
    "STRENVVS": {"-": ["21"]},
    "STREPITO": {"-": ["51"]},
    "STREPITVS": {"-": ["14"]},
    "STREPO": {"-": ["53"]},
    "STRIA": {"-": ["11"]},
    "STRICTIM": {"-": ["60"]},
    "STRICTIVVS": {"-": ["21"]},
    "STRICTOR": {"-": ["13"]},
    "STRICTVS": {"-": ["21"]},
    "STRIDENS": {"-": ["25"]},
    "STRIDEO": {"-": ["52"]},
    "STRIDO": {"-": ["53"]},
    "STRIDOR": {"-": ["13"]},
    "STRIDVLVS": {"-": ["21"]},
    "STRIGILIS": {"-": ["13"]},
    "STRIGO": {"-": ["51"]},
    "STRINGO": {"-": ["53"]},
    "STRINGOR": {"-": ["13"]},
    "STRIO": {"-": ["51"]},
    "STRIX": {"-": ["13"]},
    "STROPHA": {"-": ["17"]},
    "STROPHIVM": {"-": ["12"]},
    "STROPHVS": {"-": ["12"]},
    "STRVCTILIS": {"-": ["21"]},
    "STRVCTOR": {"-": ["13"]},
    "STRVCTVRA": {"-": ["11"]},
    "STRVES": {"-": ["13"]},
    "STRVMA": {"-": ["11"]},
    "STRVMOSVS": {"-": ["21"]},
    "STRVO": {"-": ["53"]},
    "STRVTHEVS": {"-": ["21"]},
    "STRVTHOCAMELVS": {"-": ["12"]},
    "STVDENS": {"-": ["25"]},
    "STVDEO": {"-": ["52"]},
    "STVDIOSE": {"-": ["60"]},
    "STVDIOSVS": {"-": ["21"]},
    "STVDIVM": {"-": ["12"]},
    "STVLTE": {"-": ["60"]},
    "STVLTI": {"-": ["12"], "N": ["12"]},
    "STVLTITIA": {"-": ["11"]},
    "STVLTVS": {"-": ["21"]},
    "STVPEFACIO": {"-": ["55"]},
    "STVPEFIO": {"-": ["56"]},
    "STVPEO": {"-": ["52"]},
    "STVPESCO": {"-": ["53"]},
    "STVPIDVS": {"-": ["21"]},
    "STVPOR": {"-": ["13"]},
    "STVPPA": {"-": ["11"]},
    "STVPPEVS": {"-": ["21"]},
    "STVPRATOR": {"-": ["13"]},
    "STVPRO": {"-": ["51"]},
    "STVPRVM": {"-": ["12"]},
    "STYGIVS": {"-": ["21"], "A": ["21"]},
    "STYLOBATA": {"-": ["17"]},
    "STYX": {"N": ["17"]},
    "SUPERINICIO": {"-": ["55"]},
    "SVADEO": {"-": ["52"]},
    "SVASIO": {"-": ["13"]},
    "SVASOR": {"-": ["13"]},
    "SVASORIA": {"-": ["11"]},
    "SVASVS": {"-": ["14"]},
    "SVAVE": {"-": ["60"]},
    "SVAVIA": {"-": ["13"]},
    "SVAVIDICVS": {"-": ["21"]},
    "SVAVILOQVENS": {"-": ["25"]},
    "SVAVIOLVM": {"-": ["12"]},
    "SVAVIOR": {"-": ["5J"]},
    "SVAVIS": {"-": ["24"]},
    "SVAVITAS": {"-": ["13"]},
    "SVAVITER": {"-": ["60"]},
    "SVAVIVM": {"-": ["12"]},
    "SVB": {"-": ["70"]},
    "SVBACIDVS": {"-": ["21"]},
    "SVBACTIO": {"-": ["13"]},
    "SVBAERATVS": {"-": ["21"]},
    "SVBARESCO": {"-": ["53"]},
    "SVBCAVVS": {"-": ["21"]},
    "SVBCERNO": {"-": ["53"]},
    "SVBCRISPVS": {"-": ["21"]},
    "SVBCRVDVS": {"-": ["21"]},
    "SVBDEFICIO": {"-": ["55"]},
    "SVBDIFFICILIS": {"-": ["24"]},
    "SVBDITIVVS": {"-": ["21"]},
    "SVBDIV": {"-": ["60"]},
    "SVBDO": {"-": ["53"]},
    "SVBDOLE": {"-": ["60"]},
    "SVBDOLVS": {"-": ["21"]},
    "SVBDVCO": {"-": ["53"]},
    "SVBDVCTARIVS": {"-": ["21"]},
    "SVBDVCTIO": {"-": ["13"]},
    "SVBEDO": {"2": ["56"]},
    "SVBEO": {"1": ["56"], "-": ["52"]},
    "SVBER": {"-": ["13"]},
    "SVBFVSCVS": {"-": ["21"]},
    "SVBIACEO": {"-": ["52"]},
    "SVBICIO": {"-": ["55"]},
    "SVBIECTE": {"-": ["6-"]},
    "SVBIECTIO": {"-": ["13"]},
    "SVBIECTO": {"-": ["51"]},
    "SVBIECTOR": {"-": ["13"]},
    "SVBIECTVM": {"-": ["12"]},
    "SVBIECTVS": {"1": ["12"], "2": ["14"], "3": ["21"]},
    "SVBIGO": {"-": ["53"]},
    "SVBINDE": {"-": ["60"]},
    "SVBITARIVS": {"-": ["21"]},
    "SVBITO": {"-": ["60"]},
    "SVBITVM": {"-": ["12"]},
    "SVBITVS": {"-": ["21"]},
    "SVBIVGIVS": {"-": ["21"]},
    "SVBIVNGO": {"-": ["53"]},
    "SVBLABOR": {"2": ["5L"]},
    "SVBLECTO": {"-": ["51"]},
    "SVBLEGO": {"2": ["53"]},
    "SVBLEVO": {"1": ["51"]},
    "SVBLICA": {"-": ["11"]},
    "SVBLICIVS": {"-": ["21"], "N": ["21"], "A": ["21"]},
    "SVBLIDO": {"-": ["53"]},
    "SVBLIGAR": {"-": ["13"]},
    "SVBLIGO": {"2": ["51"], "3": ["53"]},
    "SVBLIME": {"1": ["13"], "2": ["60"]},
    "SVBLIMIS": {"-": ["24"]},
    "SVBLIMITAS": {"-": ["13"]},
    "SVBLIMITER": {"-": ["60"]},
    "SVBLIMO": {"-": ["51"]},
    "SVBLIMVS": {"1": ["21"]},
    "SVBLINO": {"-": ["53"]},
    "SVBLVCEO": {"-": ["52"]},
    "SVBLVO": {"-": ["53"]},
    "SVBLVSTRIS": {"-": ["24"]},
    "SVBMANO": {"-": ["51"]},
    "SVBMERGO": {"-": ["53"]},
    "SVBMINISTRATOR": {"-": ["13"]},
    "SVBMINISTRO": {"-": ["51"]},
    "SVBMISSE": {"-": ["60"]},
    "SVBMISSVS": {"-": ["21"]},
    "SVBMITTO": {"-": ["53"]},
    "SVBMOVEO": {"-": ["52"]},
    "SVBMVGIO": {"-": ["54"]},
    "SVBNASCOR": {"-": ["5L"]},
    "SVBNECTO": {"-": ["53"]},
    "SVBNITOR": {"2": ["5L"], "-": ["5L"]},
    "SVBNIXVS": {"-": ["21"]},
    "SVBNOTO": {"-": ["51"]},
    "SVBNVBA": {"-": ["11"]},
    "SVBNVBILVS": {"-": ["21"]},
    "SVBO": {"-": ["51"]},
    "SVBOLEO": {"-": ["52"]},
    "SVBOLES": {"-": ["13"]},
    "SVBORIOR": {"-": ["5M"]},
    "SVBORNO": {"-": ["51"]},
    "SVBORTVS": {"-": ["14"]},
    "SVBRADO": {"-": ["53"]},
    "SVBRECTITO": {"-": ["51"]},
    "SVBRECTVS": {"-": ["21"]},
    "SVBREPO": {"-": ["53"]},
    "SVBRIDEO": {"-": ["52"]},
    "SVBRIGO": {"-": ["53"]},
    "SVBRIPIO": {"-": ["55"]},
    "SVBROGO": {"-": ["51"]},
    "SVBROTATVS": {"-": ["21"]},
    "SVBRVBEO": {"-": ["52"]},
    "SVBRVBICVNDVS": {"-": ["21"]},
    "SVBRVFVS": {"-": ["21"]},
    "SVBRVO": {"-": ["53"]},
    "SVBSCRIBO": {"-": ["53"]},
    "SVBSCRIPTIO": {"-": ["13"]},
    "SVBSCVS": {"-": ["13"]},
    "SVBSECIVVS": {"-": ["21"]},
    "SVBSECO": {"-": ["51"]},
    "SVBSELLIVM": {"-": ["12"]},
    "SVBSEQVOR": {"-": ["5L"]},
    "SVBSERO": {"2": ["53"], "3": ["53"]},
    "SVBSESSOR": {"-": ["13"]},
    "SVBSIDENTIA": {"-": ["11"]},
    "SVBSIDEO": {"-": ["52"]},
    "SVBSIDIARII": {"-": ["12"]},
    "SVBSIDIARIVS": {"-": ["21"]},
    "SVBSIDIOR": {"-": ["5J"]},
    "SVBSIDIVM": {"-": ["12"]},
    "SVBSIDO": {"-": ["53"]},
    "SVBSILIO": {"-": ["54"]},
    "SVBSILVS": {"-": ["21"]},
    "SVBSISTO": {"-": ["53"]},
    "SVBSORTIOR": {"-": ["5M"]},
    "SVBSORTITIO": {"-": ["13"]},
    "SVBSTANTIA": {"-": ["11"]},
    "SVBSTERNO": {"-": ["53"]},
    "SVBSTILLVS": {"-": ["21"]},
    "SVBSTITVO": {"-": ["53"]},
    "SVBSTO": {"-": ["51"]},
    "SVBSTRINGO": {"-": ["53"]},
    "SVBSTRVCTIO": {"-": ["13"]},
    "SVBSTRVCTVM": {"-": ["12"]},
    "SVBSTRVO": {"-": ["53"]},
    "SVBSVM": {"1": ["56"]},
    "SVBSVO": {"-": ["53"]},
    "SVBTEGO": {"-": ["53"]},
    "SVBTEMEN": {"-": ["13"]},
    "SVBTENDO": {"-": ["53"]},
    "SVBTER": {"1": ["60"], "2": ["70"]},
    "SVBTERFLVO": {"-": ["53"]},
    "SVBTERFVGIO": {"-": ["55"]},
    "SVBTERLABOR": {"2": ["5L"]},
    "SVBTERO": {"-": ["53"]},
    "SVBTERRANEVM": {"-": ["12"]},
    "SVBTERRANEVS": {"-": ["21"]},
    "SVBTEXO": {"-": ["53"]},
    "SVBTILE": {"-": ["13"]},
    "SVBTILIS": {"-": ["24"]},
    "SVBTILITAS": {"-": ["13"]},
    "SVBTILITER": {"-": ["60"]},
    "SVBTRAHO": {"-": ["53"]},
    "SVBTVS": {"1": ["60"], "2": ["70"]},
    "SVBTVSVS": {"-": ["21"]},
    "SVBVCVLA": {"-": ["11"]},
    "SVBVECTIO": {"-": ["13"]},
    "SVBVECTO": {"-": ["51"]},
    "SVBVECTOR": {"-": ["13"]},
    "SVBVECTVS": {"-": ["14"]},
    "SVBVEHO": {"-": ["53"]},
    "SVBVENIO": {"-": ["54"]},
    "SVBVERSO": {"-": ["51"]},
    "SVBVERSOR": {"-": ["13"]},
    "SVBVERTO": {"-": ["53"]},
    "SVBVLA": {"-": ["11"]},
    "SVBVLCVS": {"-": ["12"]},
    "SVBVOLVO": {"-": ["53"]},
    "SVBVRBANI": {"-": ["12"]},
    "SVBVRBANITAS": {"-": ["13"]},
    "SVBVRBANVM": {"-": ["12"]},
    "SVBVRBANVS": {"-": ["21"]},
    "SVBVRGEO": {"-": ["52"]},
    "SVCCEDENS": {"-": ["13"]},
    "SVCCEDO": {"1": ["53"]},
    "SVCCENDO": {"-": ["53"]},
    "SVCCENSEO": {"-": ["52"]},
    "SVCCESSIO": {"-": ["13"]},
    "SVCCESSOR": {"-": ["13"]},
    "SVCCESSVS": {"-": ["14"]},
    "SVCCIDIA": {"-": ["11"]},
    "SVCCIDO": {"1": ["53"], "2": ["53"]},
    "SVCCIDVVS": {"-": ["21"]},
    "SVCCINCTVS": {"-": ["21"]},
    "SVCCINGO": {"-": ["53"]},
    "SVCCINO": {"-": ["53"]},
    "SVCCIPIO": {"-": ["55"]},
    "SVCCRESCO": {"-": ["53"]},
    "SVCCVBO": {"2": ["51"]},
    "SVCCVMBO": {"-": ["53"]},
    "SVCCVRRO": {"-": ["53"]},
    "SVCCVTIO": {"-": ["55"]},
    "SVCIDVS": {"-": ["21"]},
    "SVCINVM": {"-": ["12"]},
    "SVCINVS": {"1": ["12"], "2": ["21"]},
    "SVCVLA": {"-": ["11"]},
    "SVCVS": {"-": ["12"]},
    "SVDARIVM": {"-": ["12"]},
    "SVDATIO": {"-": ["13"]},
    "SVDATORIVM": {"-": ["12"]},
    "SVDIS": {"-": ["13"]},
    "SVDO": {"-": ["51"]},
    "SVDOR": {"-": ["13"]},
    "SVDVS": {"-": ["21"]},
    "SVESCO": {"-": ["53"]},
    "SVESSA": {"N": ["11"]},
    "SVETTIVS": {"N": ["12"]},
    "SVETVS": {"-": ["21"]},
    "SVFES": {"-": ["13"]},
    "SVFFERO": {"-": ["56"]},
    "SVFFICIENS": {"-": ["25"]},
    "SVFFICIO": {"-": ["55"]},
    "SVFFIGO": {"-": ["53"]},
    "SVFFIO": {"-": ["54"]},
    "SVFFLAMEN": {"-": ["13"]},
    "SVFFLO": {"-": ["51"]},
    "SVFFOCATIO": {"-": ["13"]},
    "SVFFOCO": {"1": ["51"], "2": ["51"]},
    "SVFFODIO": {"-": ["55"]},
    "SVFFOSSIO": {"-": ["13"]},
    "SVFFRAGATIO": {"-": ["13"]},
    "SVFFRAGATOR": {"-": ["13"]},
    "SVFFRAGIVM": {"-": ["12"]},
    "SVFFRAGOR": {"-": ["5J"]},
    "SVFFVGIO": {"-": ["55"]},
    "SVFFVGIVM": {"-": ["12"]},
    "SVFFVLCIO": {"-": ["54"]},
    "SVFFVNDO": {"1": ["51"], "2": ["53"]},
    "SVFFVSIO": {"-": ["13"]},
    "SVGGERO": {"-": ["53"]},
    "SVGGESTVM": {"-": ["12"]},
    "SVGGESTVS": {"-": ["14"]},
    "SVGGREDIOR": {"-": ["5N"]},
    "SVGGRVNDA": {"-": ["11"]},
    "SVGGRVNDATIO": {"-": ["13"]},
    "SVGGRVNDIVM": {"-": ["12"]},
    "SVGILLATIO": {"-": ["13"]},
    "SVGILLO": {"-": ["51"]},
    "SVGO": {"-": ["53"]},
    "SVI": {"1": ["43"], "2": ["44"]},
    "SVILLVS": {"-": ["21"]},
    "SVLCO": {"-": ["51"]},
    "SVLCVS": {"-": ["12"]},
    "SVLFVR": {"-": ["13"]},
    "SVLFVRATVS": {"-": ["21"]},
    "SVLFVREVS": {"-": ["21"]},
    "SVLFVROSVS": {"-": ["21"]},
    "SVLLA": {"N": ["11"]},
    "SVLPICIVS": {"N": ["12"], "A": ["21"]},
    "SVLTO": {"-": ["51"]},
    "SVM": {"1": ["56"], "2": ["E6"]},
    "SVMEN": {"-": ["13"]},
    "SVMMA": {"-": ["11"]},
    "SVMMARIVM": {"-": ["12"]},
    "SVMMARIVS": {"1": ["12"], "2": ["21"]},
    "SVMMATIM": {"-": ["60"]},
    "SVMMATVS": {"-": ["14"]},
    "SVMME": {"-": ["60"]},
    "SVMMOPERE": {"-": ["60"]},
    "SVMMVLA": {"-": ["11"]},
    "SVMMVM": {"1": ["12"], "2": ["60"]},
    "SVMO": {"-": ["53"]},
    "SVMPTIO": {"-": ["13"]},
    "SVMPTVARIVS": {"-": ["21"]},
    "SVMPTVOSE": {"-": ["60"]},
    "SVMPTVOSVS": {"-": ["21"]},
    "SVMPTVS": {"1": ["12"], "2": ["14"]},
    "SVO": {"-": ["53"]},
    "SVOVETAVRILIA": {"-": ["13"]},
    "SVPELLEX": {"-": ["13"]},
    "SVPER": {"1": ["60"], "2": ["70"]},
    "SVPERA": {"1": ["60"], "2": ["70"]},
    "SVPERABILIS": {"-": ["24"]},
    "SVPERADDO": {"-": ["53"]},
    "SVPERANS": {"-": ["25"]},
    "SVPERATIO": {"-": ["13"]},
    "SVPERATOR": {"-": ["13"]},
    "SVPERBE": {"-": ["60"]},
    "SVPERBIA": {"-": ["11"]},
    "SVPERBIFICVS": {"-": ["21"]},
    "SVPERBIO": {"-": ["54"]},
    "SVPERBVM": {"-": ["12"]},
    "SVPERBVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "SVPERCADO": {"-": ["53"]},
    "SVPERCERNO": {"-": ["53"]},
    "SVPERCILIOSVS": {"-": ["21"]},
    "SVPERCILIVM": {"-": ["12"]},
    "SVPERCRESCO": {"-": ["53"]},
    "SVPERCVRRO": {"-": ["53"]},
    "SVPEREMINEO": {"-": ["52"]},
    "SVPERFICIARIVS": {"1": ["12"], "2": ["21"]},
    "SVPERFIO": {"-": ["56"]},
    "SVPERFLVENS": {"-": ["25"]},
    "SVPERFLVO": {"1": ["53"], "2": ["60"]},
    "SVPERFVNDO": {"2": ["53"]},
    "SVPERGREDIOR": {"-": ["5N"]},
    "SVPERI": {"-": ["12"]},
    "SVPERIACIO": {"-": ["55"]},
    "SVPERIMPONO": {"-": ["53"]},
    "SVPERINCVMBO": {"-": ["53"]},
    "SVPERIOR": {"-": ["13", "2&"]},
    "SVPERIORA": {"-": ["13"]},
    "SVPERIVS": {"-": ["6&"]},
    "SVPERLABOR": {"2": ["5L"]},
    "SVPERLIGO": {"2": ["51"], "-": ["51"]},
    "SVPERNAS": {"2": ["22"]},
    "SVPERNE": {"-": ["60"]},
    "SVPERNVM": {"-": ["12"]},
    "SVPERNVS": {"-": ["21"]},
    "SVPERO": {"-": ["51"]},
    "SVPERPENDENS": {"-": ["25"]},
    "SVPERPONO": {"-": ["53"]},
    "SVPERSCANDO": {"-": ["53"]},
    "SVPERSEDEO": {"-": ["52"]},
    "SVPERSERO": {"3": ["53"]},
    "SVPERSIDO": {"-": ["53"]},
    "SVPERSPICIO": {"-": ["55"]},
    "SVPERSTAGNO": {"-": ["51"]},
    "SVPERSTES": {"-": ["22"]},
    "SVPERSTITIO": {"-": ["13"]},
    "SVPERSTITIOSE": {"-": ["60"]},
    "SVPERSTO": {"-": ["51"]},
    "SVPERSTRVO": {"-": ["53"]},
    "SVPERSVM": {"1": ["56"]},
    "SVPERTEGO": {"-": ["53"]},
    "SVPERVACANEVS": {"-": ["21"]},
    "SVPERVACVVM": {"-": ["12"]},
    "SVPERVACVVS": {"-": ["21"]},
    "SVPERVADO": {"2": ["53"]},
    "SVPERVECTO": {"-": ["51"]},
    "SVPERVEHO": {"-": ["53"]},
    "SVPERVEHOR": {"-": ["5L"]},
    "SVPERVENIO": {"-": ["54"]},
    "SVPERVINCO": {"-": ["53"]},
    "SVPERVIVO": {"-": ["53"]},
    "SVPERVM": {"-": ["12"]},
    "SVPERVOLO": {"2": ["51"]},
    "SVPERVRGEO": {"-": ["52"]},
    "SVPERVS": {"-": ["21"]},
    "SVPINE": {"-": ["60"]},
    "SVPINO": {"-": ["51"]},
    "SVPINVS": {"-": ["21"]},
    "SVPPARVM": {"-": ["12"]},
    "SVPPEDITO": {"-": ["51"]},
    "SVPPERNATVS": {"-": ["21"]},
    "SVPPETIAE": {"-": ["11"]},
    "SVPPETO": {"-": ["53"]},
    "SVPPILO": {"2": ["51"]},
    "SVPPLANTO": {"-": ["51"]},
    "SVPPLAVDO": {"-": ["53"]},
    "SVPPLEMENTVM": {"-": ["12"]},
    "SVPPLEO": {"-": ["52"]},
    "SVPPLEX": {"1": ["13"], "2": ["25"]},
    "SVPPLICATIO": {"-": ["13"]},
    "SVPPLICITER": {"-": ["60"]},
    "SVPPLICIVM": {"-": ["12"]},
    "SVPPLICO": {"-": ["51"]},
    "SVPPONO": {"-": ["53"]},
    "SVPPORTO": {"-": ["51"]},
    "SVPPRESSIO": {"-": ["13"]},
    "SVPPRESSVS": {"-": ["21"]},
    "SVPPRIMO": {"-": ["53"]},
    "SVPPVRATIO": {"-": ["13"]},
    "SVPPVRO": {"-": ["51"]},
    "SVPPVS": {"-": ["21"]},
    "SVPPVTO": {"-": ["51"]},
    "SVPRA": {"1": ["60"], "2": ["70"]},
    "SVPRADICO": {"2": ["53"]},
    "SVPRASCANDO": {"-": ["53"]},
    "SVPRAVIVO": {"-": ["53"]},
    "SVPREMO": {"-": ["6-"], "1": ["6-"]},
    "SVPREMVM": {"1": ["12"], "2": ["6-"]},
    "SVRA": {"-": ["11"], "N": ["11"]},
    "SVRCVLVS": {"-": ["12"]},
    "SVRDVS": {"-": ["21"]},
    "SVRGO": {"-": ["53"]},
    "SVRRIPIO": {"-": ["55"]},
    "SVRSVM": {"-": ["60"]},
    "SVRSVMVORSVM": {"-": ["60"]},
    "SVS": {"-": ["13"]},
    "SVSCENSEO": {"-": ["52"]},
    "SVSCEPTIO": {"-": ["13"]},
    "SVSCIPIO": {"-": ["55"]},
    "SVSCITO": {"1": ["51"]},
    "SVSPECTO": {"1": ["51"], "2": ["60"]},
    "SVSPECTVS": {"1": ["14"], "2": ["21"]},
    "SVSPENDIVM": {"-": ["12"]},
    "SVSPENDO": {"-": ["53"]},
    "SVSPENSIO": {"-": ["13"]},
    "SVSPENSVM": {"-": ["12"]},
    "SVSPENSVRA": {"-": ["11"]},
    "SVSPENSVS": {"-": ["21"]},
    "SVSPICAX": {"-": ["25"]},
    "SVSPICIO": {"1": ["13"], "2": ["55"]},
    "SVSPICIOSVS": {"-": ["21"]},
    "SVSPICOR": {"-": ["5J"]},
    "SVSPIRIVM": {"-": ["12"]},
    "SVSPIRO": {"-": ["51"]},
    "SVSTENTO": {"-": ["51"]},
    "SVSTINEO": {"-": ["52"]},
    "SVSTOLLO": {"-": ["53"]},
    "SVSVRRO": {"-": ["51"]},
    "SVSVRRVS": {"-": ["12"]},
    "SVTILIS": {"-": ["24"]},
    "SVTOR": {"-": ["13"]},
    "SVTORIVS": {"-": ["21"]},
    "SVTRINA": {"-": ["11"]},
    "SVTRINVM": {"-": ["12"]},
    "SVTRINVS": {"-": ["21"]},
    "SVVM": {"-": ["12"]},
    "SVVS": {"-": ["44"]},
    "SYLLA": {"N": ["11"]},
    "SYLLABA": {"-": ["11"]},
    "SYLLANVS": {"N": ["21"], "A": ["21"]},
    "SYLLOGISMVS": {"-": ["12"]},
    "SYMBOLVS": {"-": ["12"]},
    "SYMMACHVS": {"N": ["12"]},
    "SYMMETRIA": {"-": ["11"]},
    "SYMMETROS": {"-": ["27"]},
    "SYMPATHIA": {"-": ["11"]},
    "SYMPHONIA": {"-": ["11"]},
    "SYMPHONIACVS": {"1": ["12"], "2": ["21"]},
    "SYNEMMENOS": {"-": ["27"]},
    "SYNGRAPHA": {"-": ["17"]},
    "SYNGRAPHE": {"-": ["17"]},
    "SYNHECHON": {"-": ["57"]},
    "SYRACVSAE": {"N": ["11"]},
    "SYRACVSANI": {"N": ["12"]},
    "SYRACVSANVS": {"N": ["21"], "A": ["21"]},
    "SYRI": {"N": ["12"]},
    "SYRIA": {"N": ["11"]},
    "SYRINX": {"-": ["17"]},
    "SYRMA": {"-": ["17"]},
    "SYRTIS": {"N": ["17"], "-": ["13"]},
    "SYSTYLOS": {"-": ["27"]},
    "TABELLA": {"-": ["11"]},
    "TABELLARIVS": {"1": ["12"], "2": ["21"]},
    "TABEO": {"-": ["52"]},
    "TABERNA": {"-": ["11"]},
    "TABERNACVLVM": {"-": ["12"]},
    "TABERNARIVS": {"1": ["12"], "2": ["21"]},
    "TABES": {"-": ["13"], "N": ["13"]},
    "TABESCO": {"-": ["53"]},
    "TABIDVS": {"-": ["21"]},
    "TABIFICVS": {"-": ["21"]},
    "TABVLA": {"-": ["11"]},
    "TABVLARIVM": {"-": ["12"]},
    "TABVLARIVS": {"1": ["12"], "2": ["21"]},
    "TABVLATIO": {"-": ["13"]},
    "TABVLATVM": {"-": ["12"]},
    "TABVLINVM": {"-": ["12"]},
    "TABVM": {"-": ["12"]},
    "TACEO": {"-": ["52"]},
    "TACITE": {"-": ["60"]},
    "TACITVM": {"-": ["12"]},
    "TACITVRNITAS": {"-": ["13"]},
    "TACITVRNVS": {"-": ["21"]},
    "TACITVS": {"-": ["21"], "N": ["12"], "A": ["21"]},
    "TACTILIS": {"-": ["24"]},
    "TACTIO": {"-": ["13"]},
    "TACTVS": {"-": ["14"]},
    "TADIVS": {"N": ["12"]},
    "TAEDA": {"-": ["11"]},
    "TAEDET": {"-": ["52"]},
    "TAEDIFER": {"-": ["21"]},
    "TAEDIVM": {"-": ["12"]},
    "TAENIA": {"-": ["11"]},
    "TALARIA": {"-": ["13"]},
    "TALARIS": {"-": ["24"]},
    "TALEA": {"-": ["11"]},
    "TALENTVM": {"-": ["12"]},
    "TALIO": {"-": ["13"]},
    "TALIS": {"-": ["45"]},
    "TALPA": {"-": ["11"]},
    "TALVS": {"-": ["12"], "N": ["12"]},
    "TAM": {"-": ["60"]},
    "TAMDIV": {"-": ["60"]},
    "TAMEN": {"-": ["60"]},
    "TAMENETSI": {"-": ["82"]},
    "TAMETSI": {"1": ["81"], "2": ["82"]},
    "TAMINO": {"-": ["51"]},
    "TAMQVAM": {"1": ["66"], "2": ["82"]},
    "TAMTVMDEM": {"1": ["45"]},
    "TANAQVIL": {"N": ["13"]},
    "TANDEM": {"-": ["60"]},
    "TANGO": {"-": ["53"]},
    "TANTILLVM": {"1": ["45"], "2": ["60"]},
    "TANTILLVS": {"-": ["45"]},
    "TANTISPER": {"-": ["60"]},
    "TANTO": {"-": ["60"]},
    "TANTOPERE": {"-": ["60"]},
    "TANTVLVM": {"1": ["45"], "2": ["60"]},
    "TANTVLVS": {"-": ["45"]},
    "TANTVM": {"1": ["45"], "2": ["60"]},
    "TANTVMDEM": {"1": ["45"], "2": ["60"]},
    "TANTVMMODO": {"-": ["60"]},
    "TANTVS": {"-": ["45"]},
    "TANTVSDEM": {"-": ["45"]},
    "TAPETVM": {"-": ["12"]},
    "TARDE": {"-": ["60"]},
    "TARDESCO": {"-": ["53"]},
    "TARDILOQVVS": {"-": ["21"]},
    "TARDIPES": {"-": ["25"]},
    "TARDITAS": {"-": ["13"]},
    "TARDO": {"-": ["51"]},
    "TARDVS": {"-": ["21"]},
    "TARENTINI": {"N": ["12"]},
    "TARIVS": {"N": ["12"]},
    "TARPEIVS": {"A": ["21"], "N": ["12"]},
    "TARQVINIA": {"N": ["11"]},
    "TARQVINIANVS": {"N": ["21"]},
    "TARQVINIENSES": {"N": ["13"]},
    "TARQVINIENSIS": {"N": ["24"], "A": ["24"]},
    "TARQVINII": {"N": ["12"]},
    "TARQVINIVS": {"A": ["21"], "N": ["12"]},
    "TARTARA": {"N": ["12"]},
    "TARTAREVS": {"-": ["21"], "A": ["21"]},
    "TARTARVS": {"N": ["12"]},
    "TATIVS": {"N": ["12"], "A": ["21"]},
    "TAVREA": {"-": ["11"], "N": ["11"]},
    "TAVREVS": {"-": ["21"]},
    "TAVRIFORMIS": {"-": ["24"]},
    "TAVRINVS": {"-": ["21"], "A": ["21"]},
    "TAVRIS": {"N": ["13"]},
    "TAVROBOLIVM": {"-": ["12"]},
    "TAVROMENITANI": {"N": ["12"]},
    "TAVROMENITANVS": {"N": ["21"], "A": ["21"]},
    "TAVRVS": {"-": ["12"], "N": ["12"]},
    "TAXATIO": {"-": ["13"]},
    "TAXILLVS": {"-": ["12"]},
    "TAXIS": {"-": ["17"]},
    "TAXO": {"-": ["51"]},
    "TAXVS": {"-": ["12"]},
    "TECTE": {"-": ["6&"]},
    "TECTINATVS": {"-": ["21"]},
    "TECTOR": {"-": ["13"]},
    "TECTORIVM": {"-": ["12"]},
    "TECTORIVS": {"-": ["21"]},
    "TECTVM": {"-": ["12"]},
    "TECTVS": {"-": ["21"], "A": ["21"]},
    "TEGES": {"-": ["13"]},
    "TEGIMEN": {"-": ["13"]},
    "TEGIMENTVM": {"-": ["12"]},
    "TEGO": {"-": ["53"]},
    "TEGVLA": {"-": ["11"]},
    "TELA": {"-": ["11"]},
    "TELEOS": {"-": ["27"]},
    "TELLENA": {"N": ["12"]},
    "TELLVS": {"-": ["13"], "N": ["13"]},
    "TELVM": {"-": ["12"]},
    "TEMENITES": {"N": ["17"]},
    "TEMERARIVS": {"-": ["21"]},
    "TEMERE": {"-": ["60"]},
    "TEMERITAS": {"-": ["13"]},
    "TEMERO": {"-": ["51"]},
    "TEMESA": {"N": ["17"]},
    "TEMETVM": {"-": ["12"]},
    "TEMNO": {"-": ["53"]},
    "TEMO": {"-": ["13"]},
    "TEMPE": {"-": ["17"], "N": ["11"]},
    "TEMPERAMENTVM": {"-": ["12"]},
    "TEMPERANS": {"-": ["25"]},
    "TEMPERANTER": {"-": ["60"]},
    "TEMPERANTIA": {"-": ["11"]},
    "TEMPERATE": {"-": ["60"]},
    "TEMPERATIO": {"-": ["13"]},
    "TEMPERATOR": {"-": ["13"]},
    "TEMPERATVRA": {"-": ["11"]},
    "TEMPERATVS": {"1": ["14"], "2": ["21"]},
    "TEMPERIES": {"-": ["15"]},
    "TEMPERO": {"-": ["51"]},
    "TEMPESTAS": {"-": ["13"], "N": ["13"]},
    "TEMPESTIVE": {"-": ["60"]},
    "TEMPESTIVO": {"-": ["60"]},
    "TEMPESTIVVS": {"-": ["21"]},
    "TEMPLVM": {"-": ["12"]},
    "TEMPORARIVM": {"-": ["12"]},
    "TEMPORARIVS": {"-": ["21"]},
    "TEMPSANVS": {"N": ["21"], "A": ["21"]},
    "TEMPTATOR": {"-": ["13"]},
    "TEMPVS": {"1": ["13"], "2": ["13"]},
    "TEMVLENTVS": {"1": ["12"], "2": ["21"]},
    "TENACITER": {"-": ["60"]},
    "TENAX": {"-": ["25"]},
    "TENDICVLA": {"-": ["11"]},
    "TENDO": {"-": ["53"]},
    "TENEBRAE": {"-": ["11"]},
    "TENEBRICOSVS": {"-": ["21"]},
    "TENEBROSVS": {"-": ["21"]},
    "TENEDII": {"N": ["12"]},
    "TENEDVS": {"N": ["17"]},
    "TENELLVLVS": {"-": ["21"]},
    "TENEO": {"-": ["52"]},
    "TENER": {"-": ["21"]},
    "TENERASCO": {"-": ["53"]},
    "TENERE": {"-": ["60"]},
    "TENERITAS": {"-": ["13"]},
    "TENERVM": {"-": ["12"]},
    "TENES": {"N": ["17"]},
    "TENOR": {"-": ["13"]},
    "TENSA": {"-": ["11"]},
    "TENSVS": {"-": ["21"]},
    "TENTABVNDVS": {"-": ["21"]},
    "TENTIGO": {"-": ["13"]},
    "TENTIO": {"-": ["13"]},
    "TENTO": {"-": ["51"]},
    "TENTORIOLVM": {"-": ["12"]},
    "TENTORIVM": {"-": ["12"]},
    "TENTVM": {"-": ["12"]},
    "TENVE": {"-": ["13"]},
    "TENVIS": {"-": ["24"]},
    "TENVITAS": {"-": ["13"]},
    "TENVITER": {"-": ["60"]},
    "TENVO": {"-": ["51"]},
    "TENVS": {"1": ["16"], "2": ["70"], "N": ["12"]},
    "TEPEFACIO": {"-": ["55"]},
    "TEPEFACTO": {"-": ["51"]},
    "TEPEFIO": {"-": ["56"]},
    "TEPEO": {"-": ["52"]},
    "TEPESCO": {"-": ["53"]},
    "TEPIDARIVS": {"-": ["21"]},
    "TEPIDVS": {"-": ["21"]},
    "TEPOR": {"-": ["13"]},
    "TER": {"-": ["36"]},
    "TERCENTVM": {"-": ["31"]},
    "TERDECIES": {"-": ["36"]},
    "TEREBINTHVS": {"-": ["12"]},
    "TEREBRA": {"-": ["11"]},
    "TEREBRATIO": {"-": ["13"]},
    "TEREBRO": {"-": ["51"]},
    "TEREDO": {"-": ["13"]},
    "TERENTIVS": {"A": ["21"], "N": ["12"]},
    "TERES": {"-": ["25"]},
    "TERGEMINVS": {"-": ["21"]},
    "TERGEO": {"-": ["52"]},
    "TERGIVERSATIO": {"-": ["13"]},
    "TERGIVERSOR": {"-": ["5J"]},
    "TERGO": {"-": ["53"]},
    "TERGVM": {"-": ["12"]},
    "TERGVS": {"-": ["13"]},
    "TERMES": {"-": ["13"]},
    "TERMINALIA": {"-": ["13"], "N": ["11"]},
    "TERMINALIS": {"-": ["24"]},
    "TERMINATIO": {"-": ["13"]},
    "TERMINO": {"-": ["51"]},
    "TERMINVS": {"-": ["12"], "N": ["12"]},
    "TERNI": {"-": ["33"]},
    "TERO": {"-": ["53"]},
    "TERRA": {"-": ["11"], "N": ["11"]},
    "TERRENA": {"-": ["12"]},
    "TERRENI": {"-": ["12"]},
    "TERRENVM": {"-": ["12"]},
    "TERRENVS": {"-": ["21"]},
    "TERREO": {"-": ["52"]},
    "TERRESTRIS": {"-": ["24"]},
    "TERREVS": {"-": ["21"]},
    "TERRIBILIS": {"-": ["24"]},
    "TERRIFICO": {"-": ["51"]},
    "TERRIFICVS": {"-": ["21"]},
    "TERRIGENA": {"-": ["11"]},
    "TERRILOQVVS": {"-": ["21"]},
    "TERRITO": {"-": ["51"]},
    "TERRITORIVM": {"-": ["12"]},
    "TERROR": {"-": ["13"]},
    "TERROSVS": {"-": ["21"]},
    "TERSVS": {"1": ["14"], "2": ["21"]},
    "TERTIA": {"N": ["11"]},
    "TERTIANVS": {"1": ["12"], "2": ["21"]},
    "TERTIARIVS": {"1": ["12"], "2": ["21"]},
    "TERTIO": {"-": ["35"]},
    "TERTIVM": {"-": ["35"]},
    "TERTIVS": {"-": ["32"]},
    "TERTIVSDECIMVS": {"-": ["32"]},
    "TESCA": {"-": ["12"]},
    "TESSELA": {"-": ["11"]},
    "TESSERA": {"-": ["11"]},
    "TESSERVLA": {"-": ["11"]},
    "TESTA": {"-": ["11"]},
    "TESTACEVS": {"-": ["21"]},
    "TESTAMENTARIVS": {"2": ["21"], "1": ["12"]},
    "TESTAMENTVM": {"-": ["12"]},
    "TESTATVS": {"-": ["21"]},
    "TESTICVLVS": {"-": ["12"]},
    "TESTIFICATIO": {"-": ["13"]},
    "TESTIFICO": {"-": ["51"]},
    "TESTIFICOR": {"-": ["5J"]},
    "TESTIMONIVM": {"-": ["12"]},
    "TESTIS": {"1": ["13"], "2": ["13"]},
    "TESTO": {"-": ["51"]},
    "TESTOR": {"-": ["5J"]},
    "TESTV": {"-": ["16"]},
    "TESTVDINEATVS": {"-": ["21"]},
    "TESTVDINEVS": {"-": ["21"]},
    "TESTVDO": {"-": ["13"]},
    "TESTVM": {"-": ["12"]},
    "TETER": {"-": ["21"]},
    "TETRADOROS": {"1": ["27"], "2": ["60"]},
    "TETRANS": {"-": ["13"]},
    "TETRARCHES": {"-": ["17"], "N": ["13"]},
    "TETRASTYLVS": {"-": ["27"]},
    "TETRE": {"-": ["60"]},
    "TETRICVS": {"-": ["21"]},
    "TETTIVS": {"N": ["12"]},
    "TEXO": {"-": ["53"]},
    "TEXTILE": {"-": ["13"]},
    "TEXTILIS": {"-": ["24"]},
    "TEXTOR": {"-": ["13"]},
    "TEXTORIVS": {"-": ["21"]},
    "TEXTRINVM": {"-": ["12"]},
    "TEXTRIX": {"-": ["13"]},
    "TEXTVM": {"-": ["12"]},
    "TEXTVRA": {"-": ["11"]},
    "TEXTVS": {"-": ["14"]},
    "THALAMVS": {"-": ["12"]},
    "THALASSINVS": {"-": ["21"]},
    "THALASSIVS": {"N": ["12"]},
    "THAPSVS": {"-": ["12"], "N": ["12"]},
    "THARSVS": {"N": ["17"]},
    "THEATRALIS": {"-": ["24"]},
    "THEATRVM": {"-": ["12"]},
    "THEBANI": {"N": ["12"]},
    "THEBANVS": {"N": ["21"], "A": ["21"]},
    "THECA": {"-": ["11"]},
    "THEMATISMOS": {"-": ["17"]},
    "THEMISTAGORAS": {"N": ["17"]},
    "THEODORVS": {"N": ["12"]},
    "THEOLOGVS": {"-": ["12"]},
    "THEOMNASTVS": {"N": ["12"]},
    "THEOPANES": {"N": ["17"]},
    "THEORACTVS": {"N": ["12"]},
    "THERICLEVS": {"N": ["21"], "A": ["21"]},
    "THERMAE": {"-": ["11"]},
    "THERMITANI": {"N": ["12"]},
    "THERMITANVS": {"N": ["21"], "A": ["21"]},
    "THESAVRVS": {"-": ["12"]},
    "THESPIADES": {"N": ["13"]},
    "THESPIAE": {"N": ["11"]},
    "THESPIENSES": {"N": ["13"]},
    "THESSALVS": {"N": ["12"], "A": ["21"]},
    "THETA": {"-": ["17"]},
    "THIASVS": {"-": ["12"]},
    "THOLVS": {"-": ["12"]},
    "THOMIX": {"-": ["13"]},
    "THORAX": {"-": ["17"]},
    "THRACES": {"N": ["17"]},
    "THRACIA": {"N": ["11"]},
    "THRACIVS": {"N": ["21"], "A": ["21"]},
    "THRASO": {"N": ["17"]},
    "THRAX": {"N": ["27"], "A": ["25"]},
    "THVRINVS": {"N": ["21"], "A": ["21"]},
    "THYAS": {"-": ["17"], "N": ["17"]},
    "THYIVS": {"-": ["21"]},
    "THYMBRA": {"-": ["11"]},
    "THYMELICVS": {"-": ["21"]},
    "THYMVM": {"-": ["12"]},
    "THYNNVS": {"-": ["12"]},
    "THYROMA": {"-": ["17"]},
    "THYRORON": {"-": ["17"]},
    "THYRSIGER": {"-": ["21"]},
    "THYRSVS": {"-": ["12"]},
    "TIARA": {"-": ["11"]},
    "TIARAS": {"-": ["17"]},
    "TIBERINVS": {"N": ["12"], "A": ["21"]},
    "TIBERIS": {"N": ["13"]},
    "TIBERIVS": {"N": ["12"]},
    "TIBIA": {"-": ["11"]},
    "TIBICEN": {"-": ["13"]},
    "TIBICINA": {"-": ["11"]},
    "TICESCO": {"-": ["53"]},
    "TIGILLVM": {"-": ["12"]},
    "TIGNARIVS": {"-": ["21"]},
    "TIGNVM": {"-": ["12"]},
    "TIGRIS": {"-": ["17"], "N": ["13"]},
    "TILIA": {"-": ["11"]},
    "TIMARCHIDES": {"N": ["17"]},
    "TIMEFACIO": {"-": ["55"]},
    "TIMENS": {"-": ["25"]},
    "TIMEO": {"-": ["52"]},
    "TIMESCO": {"-": ["53"]},
    "TIMIDE": {"-": ["60"]},
    "TIMIDITAS": {"-": ["13"]},
    "TIMIDVS": {"-": ["21"]},
    "TIMOR": {"-": ["13"], "N": ["13"]},
    "TINEA": {"-": ["11"]},
    "TINEO": {"-": ["52"]},
    "TINGO": {"-": ["53"]},
    "TINNIO": {"-": ["54"]},
    "TINNITVS": {"-": ["14"]},
    "TINNVLVS": {"-": ["21"]},
    "TINTINNABVLVM": {"-": ["12"]},
    "TINTINNO": {"-": ["51"]},
    "TIRO": {"-": ["13"], "N": ["13"]},
    "TIROCINIVM": {"-": ["12"]},
    "TIRVNCVLVS": {"-": ["12"]},
    "TISSENSES": {"N": ["13"]},
    "TITIENSES": {"N": ["13"]},
    "TITILLATIO": {"-": ["13"]},
    "TITILLO": {"-": ["51"]},
    "TITINIVS": {"N": ["12"]},
    "TITIVS": {"N": ["12"], "A": ["21"]},
    "TITVBATIO": {"-": ["13"]},
    "TITVBO": {"-": ["51"]},
    "TITVLVS": {"-": ["12"]},
    "TITVS": {"N": ["12"], "-": ["12"]},
    "TLEPOLEMVS": {"N": ["12"]},
    "TOFVS": {"-": ["12"]},
    "TOGA": {"-": ["11"]},
    "TOGATA": {"-": ["11"]},
    "TOGATI": {"-": ["12"]},
    "TOGATVS": {"-": ["21"]},
    "TOLERABILIS": {"-": ["24"]},
    "TOLERABILITER": {"-": ["60"]},
    "TOLERANDVS": {"-": ["21"]},
    "TOLERANS": {"-": ["25"]},
    "TOLERANTIA": {"-": ["11"]},
    "TOLERATVS": {"1": ["14"], "2": ["21"]},
    "TOLERO": {"-": ["51"]},
    "TOLLO": {"-": ["53"]},
    "TOLVTARIVS": {"-": ["21"]},
    "TOMACVLVM": {"-": ["12"]},
    "TOMENTVM": {"-": ["12"]},
    "TONANS": {"-": ["25"], "A": ["22"], "N": ["13"]},
    "TONDEO": {"-": ["52"]},
    "TONITRALIS": {"-": ["24"]},
    "TONITRVS": {"-": ["14"]},
    "TONITRVVM": {"-": ["12"]},
    "TONO": {"-": ["51"]},
    "TONSA": {"1": ["11"], "2": ["11"]},
    "TONSOR": {"-": ["13"]},
    "TONSTRICVLA": {"-": ["11"]},
    "TONSVRA": {"-": ["11"]},
    "TONVS": {"-": ["12"]},
    "TOPEODVS": {"-": ["17"]},
    "TOPIA": {"-": ["12"]},
    "TORAL": {"-": ["13"]},
    "TORCVLAR": {"-": ["13"]},
    "TORCVLARIVM": {"-": ["12"]},
    "TORCVLVS": {"-": ["21"]},
    "TOREVMA": {"-": ["17"]},
    "TORMENTVM": {"-": ["12"]},
    "TORMINA": {"-": ["13"]},
    "TORNATVS": {"-": ["21"]},
    "TORNO": {"-": ["51"]},
    "TORNVS": {"-": ["12"]},
    "TOROSVS": {"-": ["21"]},
    "TORPEDO": {"-": ["13"]},
    "TORPEO": {"-": ["52"]},
    "TORPESCO": {"-": ["53"]},
    "TORPIDVS": {"-": ["21"]},
    "TORPOR": {"-": ["13"]},
    "TORQVATVS": {"-": ["21"], "N": ["12"]},
    "TORQVEO": {"-": ["52"]},
    "TORQVIS": {"-": ["13"]},
    "TORRENS": {"1": ["13"], "2": ["25"]},
    "TORREO": {"-": ["52"]},
    "TORRESCO": {"-": ["53"]},
    "TORRIDVS": {"-": ["21"]},
    "TORRO": {"-": ["53"]},
    "TORSIO": {"-": ["13"]},
    "TORTE": {"-": ["60"]},
    "TORTILIS": {"-": ["24"]},
    "TORTIO": {"-": ["13"]},
    "TORTIVVS": {"-": ["21"]},
    "TORTO": {"-": ["51"]},
    "TORTOR": {"-": ["13"]},
    "TORTVOSVS": {"-": ["21"]},
    "TORTVS": {"-": ["14"]},
    "TORVA": {"-": ["60"]},
    "TORVITAS": {"-": ["13"]},
    "TORVLVS": {"-": ["12"]},
    "TORVS": {"-": ["12"]},
    "TORVVM": {"-": ["60"]},
    "TORVVS": {"-": ["21"]},
    "TOT": {"-": ["45"]},
    "TOTIDEM": {"1": ["45"], "2": ["60"]},
    "TOTIENS": {"-": ["60"]},
    "TOTVM": {"-": ["12"]},
    "TOTVS": {"-": ["48"]},
    "TOXICVM": {"-": ["12"]},
    "TRABALIS": {"-": ["24"]},
    "TRABEA": {"-": ["11"]},
    "TRABEATVS": {"-": ["21"]},
    "TRABECVLA": {"-": ["11"]},
    "TRABS": {"-": ["13"]},
    "TRACTA": {"-": ["12"]},
    "TRACTABILIS": {"-": ["24"]},
    "TRACTABILITAS": {"-": ["13"]},
    "TRACTATIO": {"-": ["13"]},
    "TRACTATOR": {"-": ["13"]},
    "TRACTATVS": {"-": ["14"]},
    "TRACTIM": {"-": ["60"]},
    "TRACTIO": {"-": ["13"]},
    "TRACTO": {"-": ["51"]},
    "TRACTORIVS": {"-": ["21"]},
    "TRACTVS": {"-": ["14"]},
    "TRADITIO": {"-": ["13"]},
    "TRADO": {"-": ["53"]},
    "TRADVCO": {"-": ["53"]},
    "TRADVCTIO": {"-": ["13"]},
    "TRAGICE": {"-": ["60"]},
    "TRAGICVS": {"1": ["12"], "2": ["21"]},
    "TRAGOEDIA": {"-": ["11"], "N": ["11"]},
    "TRAGOEDVS": {"-": ["12"]},
    "TRAGVLA": {"-": ["11"]},
    "TRAHA": {"-": ["11"]},
    "TRAHO": {"-": ["53"]},
    "TRAICIO": {"-": ["55"]},
    "TRAIECTIO": {"-": ["13"]},
    "TRAIECTVRA": {"-": ["11"]},
    "TRAIECTVS": {"-": ["14"]},
    "TRAMA": {"-": ["11"]},
    "TRAMES": {"-": ["13"]},
    "TRANGRESSVS": {"-": ["14"]},
    "TRANO": {"-": ["51"]},
    "TRANQVILLE": {"-": ["6&"]},
    "TRANQVILLITAS": {"-": ["13"]},
    "TRANQVILLO": {"-": ["51"]},
    "TRANQVILLVM": {"-": ["12"]},
    "TRANQVILLVS": {"-": ["21"], "N": ["12"]},
    "TRANS": {"1": ["60"], "2": ["70"]},
    "TRANSACTOR": {"-": ["13"]},
    "TRANSALPINVS": {"N": ["21"], "A": ["21"]},
    "TRANSCENDO": {"-": ["53"]},
    "TRANSCOLO": {"1": ["51"]},
    "TRANSCRIBO": {"-": ["53"]},
    "TRANSCVRRO": {"-": ["53"]},
    "TRANSCVRSVS": {"-": ["14"]},
    "TRANSDVCO": {"-": ["53"]},
    "TRANSEO": {"1": ["56"]},
    "TRANSERO": {"2": ["53"]},
    "TRANSFERO": {"-": ["56"]},
    "TRANSFIGO": {"-": ["53"]},
    "TRANSFIGVRO": {"-": ["51"]},
    "TRANSFODIO": {"-": ["55"]},
    "TRANSFORMO": {"-": ["51"]},
    "TRANSFORO": {"-": ["51"]},
    "TRANSFVGA": {"-": ["11"]},
    "TRANSFVGIO": {"-": ["55"]},
    "TRANSFVGIVM": {"-": ["12"]},
    "TRANSFVNDO": {"2": ["53"]},
    "TRANSGERO": {"-": ["53"]},
    "TRANSGREDIOR": {"-": ["5N"]},
    "TRANSGRESSIO": {"-": ["13"]},
    "TRANSGRESSVS": {"-": ["14"]},
    "TRANSIGO": {"-": ["53"]},
    "TRANSILIO": {"-": ["54"]},
    "TRANSITIO": {"-": ["13"]},
    "TRANSITVS": {"-": ["14"]},
    "TRANSLATICIVS": {"-": ["21"]},
    "TRANSLATIO": {"-": ["13"]},
    "TRANSLATOR": {"-": ["13"]},
    "TRANSLATVS": {"-": ["14"]},
    "TRANSLEGO": {"1": ["51"], "2": ["53"]},
    "TRANSLVCEO": {"-": ["52"]},
    "TRANSMARINVS": {"-": ["21"]},
    "TRANSMEO": {"-": ["51"]},
    "TRANSMIGRO": {"-": ["51"]},
    "TRANSMISSVS": {"-": ["14"]},
    "TRANSMITTO": {"-": ["53"]},
    "TRANSMOVEO": {"-": ["52"]},
    "TRANSMVTO": {"2": ["51"]},
    "TRANSNO": {"-": ["51"]},
    "TRANSPARENS": {"-": ["25"]},
    "TRANSPECTVS": {"-": ["14"]},
    "TRANSPICIO": {"-": ["55"]},
    "TRANSPONO": {"-": ["53"]},
    "TRANSPORTATIO": {"-": ["13"]},
    "TRANSPORTO": {"-": ["51"]},
    "TRANSSCRIBO": {"-": ["53"]},
    "TRANSTRVM": {"-": ["12"]},
    "TRANSVECTIO": {"-": ["13"]},
    "TRANSVECTO": {"-": ["51"]},
    "TRANSVEHO": {"-": ["53"]},
    "TRANSVERBERO": {"2": ["51"]},
    "TRANSVERSE": {"-": ["60"]},
    "TRANSVERSO": {"-": ["51"]},
    "TRANSVERSVM": {"1": ["12"], "2": ["60"]},
    "TRANSVERSVS": {"-": ["21"]},
    "TRANSVERTO": {"-": ["53"]},
    "TRANSVIO": {"-": ["51"]},
    "TRANSVO": {"-": ["53"]},
    "TRANSVOLITO": {"-": ["51"]},
    "TRANSVOLO": {"2": ["51"]},
    "TRAPES": {"-": ["13"]},
    "TRAPETVM": {"-": ["12"]},
    "TRAPETVS": {"-": ["12"]},
    "TRAVLIZO": {"-": ["53"]},
    "TREBIVM": {"N": ["12"]},
    "TREBONIVS": {"N": ["12"]},
    "TRECENI": {"-": ["33"]},
    "TRECENTENI": {"-": ["33"]},
    "TRECENTESIMVS": {"-": ["32"]},
    "TRECENTI": {"-": ["31"], "N": ["12"]},
    "TRECENTIES": {"-": ["36"]},
    "TRECHEDIPNVM": {"-": ["12"]},
    "TRECTO": {"-": ["51"]},
    "TREDECIM": {"-": ["31"]},
    "TREMEBVNDVS": {"-": ["21"]},
    "TREMEFACIO": {"-": ["55"]},
    "TREMELLIVS": {"N": ["12"]},
    "TREMENDVS": {"-": ["21"]},
    "TREMESCO": {"-": ["53"]},
    "TREMO": {"-": ["53"]},
    "TREMOR": {"-": ["13"]},
    "TREMVLVS": {"1": ["12"], "2": ["12"], "3": ["21"], "N": ["12"]},
    "TREPIDANTER": {"-": ["6&"]},
    "TREPIDATIO": {"-": ["13"]},
    "TREPIDE": {"-": ["60"]},
    "TREPIDO": {"-": ["51"]},
    "TREPIDVS": {"-": ["21"]},
    "TRES": {"-": ["31"], "N": ["13"]},
    "TRESSIS": {"-": ["13"]},
    "TRESVIRI": {"-": ["12"]},
    "TRIARIVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "TRIBON": {"-": ["17"]},
    "TRIBVLIS": {"-": ["24"]},
    "TRIBVLO": {"-": ["51"]},
    "TRIBVLVM": {"-": ["12"]},
    "TRIBVLVS": {"-": ["12"]},
    "TRIBVNAL": {"-": ["13"]},
    "TRIBVNATVS": {"-": ["14"]},
    "TRIBVNICIVS": {"1": ["12"], "2": ["21"]},
    "TRIBVNVS": {"-": ["12"]},
    "TRIBVO": {"-": ["53"]},
    "TRIBVS": {"-": ["14"]},
    "TRIBVTARIVS": {"-": ["21"]},
    "TRIBVTIM": {"-": ["60"]},
    "TRIBVTVM": {"-": ["12"]},
    "TRIBVTVS": {"1": ["14"], "2": ["21"]},
    "TRICENARIVS": {"-": ["21"]},
    "TRICENI": {"-": ["33"]},
    "TRICEPS": {"-": ["25"]},
    "TRICESIMVS": {"-": ["32"]},
    "TRICHALCVM": {"-": ["12"]},
    "TRICHILA": {"-": ["11"]},
    "TRICHO": {"N": ["13"]},
    "TRICIENS": {"-": ["36"]},
    "TRICIPITINVS": {"N": ["12"]},
    "TRICLINIVM": {"-": ["12"]},
    "TRICOR": {"-": ["5J"]},
    "TRICORPOR": {"-": ["25"]},
    "TRIDENS": {"1": ["13"], "2": ["25"]},
    "TRIDVVM": {"-": ["12"]},
    "TRIENNIVM": {"-": ["12"]},
    "TRIERARCHVS": {"-": ["12"]},
    "TRIERIS": {"2": ["24"]},
    "TRIETERICA": {"-": ["12"]},
    "TRIETERICVS": {"-": ["21"]},
    "TRIFARIAM": {"-": ["60"]},
    "TRIFAVX": {"-": ["25"]},
    "TRIFIDVS": {"-": ["21"]},
    "TRIFISSILIS": {"-": ["24"]},
    "TRIFORMIS": {"-": ["24"]},
    "TRIGINTA": {"-": ["31"]},
    "TRIGLYPHVS": {"-": ["12"]},
    "TRIGON": {"-": ["13"]},
    "TRIGONIVM": {"-": ["12"]},
    "TRIGONOS": {"-": ["27"]},
    "TRIGONVM": {"-": ["12"]},
    "TRILIBRIS": {"-": ["24"]},
    "TRILINGVIS": {"-": ["24"]},
    "TRILIX": {"-": ["25"]},
    "TRIMESTRIS": {"-": ["24"]},
    "TRIMVS": {"-": ["21"]},
    "TRINODIS": {"-": ["24"]},
    "TRINVS": {"-": ["33"]},
    "TRIOBOLVS": {"-": ["12"]},
    "TRIOCALINVS": {"N": ["21"], "A": ["21"]},
    "TRIONES": {"-": ["13"]},
    "TRIPARTIO": {"-": ["54", "13"]},
    "TRIPARTITO": {"-": ["60", "6-"]},
    "TRIPECTORVS": {"-": ["21"]},
    "TRIPEDANEVS": {"-": ["21"]},
    "TRIPERTITO": {"-": ["60"]},
    "TRIPES": {"1": ["13"], "2": ["25"]},
    "TRIPLEX": {"-": ["34", "13"]},
    "TRIPLINTHIVS": {"-": ["21"]},
    "TRIPODO": {"-": ["51"]},
    "TRIPTOLEMVS": {"N": ["12"]},
    "TRIPVDIO": {"-": ["51"]},
    "TRIPVDIVM": {"-": ["12"]},
    "TRIPVS": {"1": ["17"], "2": ["25"]},
    "TRIQVETRVS": {"-": ["21"], "A": ["21"]},
    "TRIREMIS": {"1": ["13"], "2": ["24"]},
    "TRISCVRRIVM": {"-": ["12"]},
    "TRISPASTOS": {"-": ["27"]},
    "TRISTE": {"1": ["13"], "2": ["60"]},
    "TRISTIMONIA": {"-": ["11"]},
    "TRISTIS": {"-": ["24"]},
    "TRISTITIA": {"-": ["11"]},
    "TRISTOR": {"-": ["5J"]},
    "TRISVLCVS": {"-": ["21"]},
    "TRITE": {"-": ["17"]},
    "TRITICEVS": {"-": ["21"]},
    "TRITICVM": {"-": ["12"]},
    "TRITVRA": {"-": ["11"]},
    "TRITVS": {"1": ["14"], "2": ["21"]},
    "TRIVIALIS": {"-": ["24"]},
    "TRIVIVM": {"-": ["12"]},
    "TRIVIVS": {"-": ["21"]},
    "TRIVMPHALIA": {"-": ["13"]},
    "TRIVMPHALIS": {"1": ["13"], "2": ["24"]},
    "TRIVMPHO": {"-": ["51"]},
    "TRIVMPHVS": {"-": ["12"], "N": ["12"]},
    "TRIVMVIR": {"-": ["12"]},
    "TRIVMVIRALIS": {"-": ["24"]},
    "TRIVMVIRATVS": {"-": ["14"]},
    "TROCHILVS": {"-": ["17"]},
    "TROCHLEA": {"-": ["11"]},
    "TROCHVS": {"-": ["12"]},
    "TROIA": {"N": ["11"]},
    "TROIANI": {"N": ["12"]},
    "TROIANVS": {"A": ["21"], "N": ["12"]},
    "TROPAEVM": {"-": ["12"]},
    "TROSSVLVS": {"-": ["12"]},
    "TRVCIDATIO": {"-": ["13"]},
    "TRVCIDO": {"-": ["51"]},
    "TRVCVLENTIA": {"-": ["11"]},
    "TRVCVLENTVS": {"-": ["21"], "N": ["12"]},
    "TRVDIS": {"-": ["13"]},
    "TRVDO": {"-": ["53"]},
    "TRVLLA": {"-": ["11"]},
    "TRVLLEVS": {"-": ["12"]},
    "TRVLLISSATIO": {"-": ["13"]},
    "TRVLLISSO": {"-": ["51"]},
    "TRVNCO": {"-": ["51"]},
    "TRVNCVS": {"1": ["12"], "2": ["21"]},
    "TRVSATILIS": {"-": ["24"]},
    "TRVSO": {"-": ["51"]},
    "TRVTINA": {"-": ["11"]},
    "TRVTINOR": {"-": ["5J"]},
    "TRVX": {"-": ["25"]},
    "TV": {"-": ["41"]},
    "TVBA": {"-": ["11"]},
    "TVBER": {"-": ["13"]},
    "TVBICEN": {"-": ["13"]},
    "TVBVLA": {"-": ["11"]},
    "TVBVLVS": {"-": ["12"], "N": ["12"]},
    "TVBVS": {"-": ["12"]},
    "TVCCETVM": {"-": ["12"]},
    "TVDITO": {"-": ["51"]},
    "TVEOR": {"-": ["5K"]},
    "TVGVRIVM": {"-": ["12"]},
    "TVI": {"-": ["42", "12"]},
    "TVLLIA": {"N": ["11"]},
    "TVLLIVS": {"N": ["12"]},
    "TVLLVS": {"N": ["12"]},
    "TVM": {"-": ["60"]},
    "TVMEFACIO": {"-": ["55"]},
    "TVMEO": {"-": ["52"]},
    "TVMESCO": {"-": ["53"]},
    "TVMIDVS": {"-": ["21"]},
    "TVMOR": {"-": ["13"]},
    "TVMVLO": {"-": ["51"]},
    "TVMVLOSVS": {"-": ["21"]},
    "TVMVLTVARIVS": {"-": ["21"]},
    "TVMVLTVOR": {"-": ["5J"]},
    "TVMVLTVOSE": {"-": ["6&"]},
    "TVMVLTVOSVS": {"-": ["21"]},
    "TVMVLTVS": {"-": ["14"]},
    "TVMVLVS": {"-": ["12"]},
    "TVNC": {"-": ["60"]},
    "TVNDO": {"-": ["53"]},
    "TVNICA": {"-": ["11"]},
    "TVNICATVS": {"-": ["21"]},
    "TVOR": {"-": ["5L"]},
    "TVRBA": {"-": ["11"]},
    "TVRBATE": {"-": ["60"]},
    "TVRBATOR": {"-": ["13"]},
    "TVRBATVS": {"-": ["21"]},
    "TVRBEN": {"-": ["13"]},
    "TVRBIDE": {"-": ["60"]},
    "TVRBIDVM": {"-": ["60"]},
    "TVRBIDVS": {"-": ["21"]},
    "TVRBO": {"1": ["13"], "2": ["51"], "N": ["11"]},
    "TVRBVLENTVS": {"-": ["21"]},
    "TVRDA": {"-": ["11"]},
    "TVRDVS": {"-": ["12"]},
    "TVREVS": {"-": ["21"]},
    "TVRGEO": {"-": ["52"]},
    "TVRGESCO": {"-": ["53"]},
    "TVRGIDVLVS": {"-": ["21"]},
    "TVRGIDVS": {"-": ["21"]},
    "TVRIBVLVM": {"-": ["12"]},
    "TVRICREMVS": {"-": ["21"]},
    "TVRIFER": {"-": ["21"]},
    "TVRMA": {"-": ["11"]},
    "TVRMATIM": {"-": ["60"]},
    "TVRNVS": {"N": ["12"]},
    "TVRPE": {"1": ["13"], "2": ["60"]},
    "TVRPICVLVS": {"-": ["21"]},
    "TVRPIO": {"N": ["13"]},
    "TVRPIS": {"-": ["24"]},
    "TVRPITER": {"-": ["60"]},
    "TVRPITVDO": {"-": ["13"]},
    "TVRPO": {"-": ["51"]},
    "TVRRICVLA": {"-": ["11"]},
    "TVRRIGER": {"-": ["21"]},
    "TVRRIS": {"-": ["13"]},
    "TVRRITVS": {"-": ["21"]},
    "TVRTVR": {"-": ["13"]},
    "TVRTVRILLA": {"-": ["11"]},
    "TVRVNDA": {"-": ["11"]},
    "TVS": {"-": ["13"]},
    "TVSCI": {"N": ["12"]},
    "TVSCVLANVM": {"N": ["12"]},
    "TVSCVLANVS": {"N": ["21"], "A": ["21"]},
    "TVSCVLVM": {"N": ["12"], "-": ["12"]},
    "TVSCVS": {"N": ["21"], "A": ["21"]},
    "TVSSIO": {"-": ["54"]},
    "TVSSIS": {"-": ["13"]},
    "TVTAMEN": {"-": ["13"]},
    "TVTAMENTVM": {"-": ["12"]},
    "TVTE": {"-": ["60"]},
    "TVTELA": {"-": ["11"], "N": ["11"]},
    "TVTO": {"1": ["51"], "2": ["60"], "-": ["6-"]},
    "TVTOR": {"1": ["13"], "2": ["5J"], "N": ["11"]},
    "TVTVM": {"-": ["12"]},
    "TVTVS": {"-": ["21"]},
    "TVVM": {"-": ["12"]},
    "TVVS": {"-": ["42"]},
    "TYCHA": {"N": ["11"]},
    "TYMPANVM": {"-": ["12"]},
    "TYNDARIS": {"N": ["17"]},
    "TYNDARITANI": {"N": ["12"]},
    "TYNDARITANVS": {"N": ["21"], "A": ["21"]},
    "TYRACINVS": {"N": ["12"]},
    "TYRANNICE": {"-": ["60"]},
    "TYRANNICIDA": {"-": ["11"]},
    "TYRANNICVS": {"-": ["21"]},
    "TYRANNIS": {"-": ["17"]},
    "TYRANNVS": {"-": ["12"]},
    "TYRIVS": {"N": ["21"], "A": ["21"]},
    "TYROS": {"N": ["17"]},
    "VACATIO": {"-": ["13"]},
    "VACCA": {"-": ["11"], "N": ["11"]},
    "VACCINIVM": {"-": ["12"]},
    "VACEFIO": {"-": ["56"]},
    "VACILLO": {"-": ["51"]},
    "VACO": {"-": ["51"]},
    "VACVEFACIO": {"-": ["55"]},
    "VACVITAS": {"-": ["13"]},
    "VACVO": {"1": ["51"], "2": ["60"]},
    "VACVVM": {"-": ["12"]},
    "VACVVS": {"-": ["21"]},
    "VADIMONIVM": {"-": ["12"]},
    "VADO": {"1": ["51"], "2": ["53"]},
    "VADOR": {"-": ["5J"]},
    "VADOSVS": {"-": ["21"]},
    "VADVM": {"-": ["12"], "N": ["12"]},
    "VAE": {"-": ["90"]},
    "VAFER": {"-": ["21"]},
    "VAFRE": {"-": ["60"]},
    "VAFRITIA": {"-": ["11"]},
    "VAGABVNDVS": {"-": ["21"]},
    "VAGATIO": {"-": ["13"]},
    "VAGE": {"-": ["60"]},
    "VAGINA": {"-": ["11"]},
    "VAGIO": {"-": ["54"]},
    "VAGITVS": {"-": ["14"]},
    "VAGOR": {"1": ["13"], "2": ["5J"]},
    "VAGVS": {"-": ["21"]},
    "VALDE": {"-": ["60"]},
    "VALEDICO": {"2": ["53"], "-": ["53"]},
    "VALENS": {"-": ["25"], "N": ["13"]},
    "VALENTINI": {"N": ["12"]},
    "VALENTIVS": {"N": ["12"]},
    "VALEO": {"-": ["52"]},
    "VALERIVS": {"N": ["12"], "A": ["21"]},
    "VALESCO": {"-": ["53"]},
    "VALETVDINARIVM": {"-": ["12"]},
    "VALETVDINARIVS": {"1": ["12"], "2": ["21"]},
    "VALETVDO": {"-": ["13"]},
    "VALIDE": {"-": ["6&"]},
    "VALIDVS": {"-": ["21"]},
    "VALLIS": {"-": ["13"]},
    "VALLO": {"-": ["51"]},
    "VALLVM": {"-": ["12"]},
    "VALLVS": {"1": ["12"], "2": ["12"]},
    "VALOR": {"-": ["13"]},
    "VALVA": {"-": ["11"]},
    "VALVAE": {"-": ["11"]},
    "VALVATVS": {"-": ["21"]},
    "VANA": {"-": ["12"]},
    "VANE": {"-": ["60"]},
    "VANESCO": {"-": ["53"]},
    "VANILOQVENTIA": {"-": ["11"]},
    "VANILOQVVS": {"-": ["21"]},
    "VANITAS": {"-": ["13"]},
    "VANNVS": {"-": ["12"]},
    "VANVM": {"-": ["12"]},
    "VANVS": {"-": ["21"]},
    "VAPIDVS": {"-": ["21"]},
    "VAPOR": {"-": ["13"]},
    "VAPORO": {"-": ["51"]},
    "VAPPA": {"-": ["11"]},
    "VAPVLO": {"-": ["51"]},
    "VARA": {"-": ["11"]},
    "VARIANTIA": {"-": ["11"]},
    "VARIATVS": {"-": ["21"]},
    "VARICOSVS": {"-": ["21"]},
    "VARICVS": {"-": ["21"]},
    "VARIE": {"-": ["60"]},
    "VARIETAS": {"-": ["13"]},
    "VARIO": {"-": ["51"]},
    "VARIVM": {"-": ["12"]},
    "VARIVS": {"-": ["21"], "N": ["12"], "A": ["21"]},
    "VARIX": {"-": ["13"]},
    "VARRO": {"N": ["13"]},
    "VARVS": {"1": ["12"], "2": ["21"], "N": ["12"]},
    "VAS": {"1": ["13"], "2": ["16"]},
    "VASARIVM": {"-": ["12"]},
    "VASCVLARIVS": {"-": ["12"]},
    "VASCVLVM": {"-": ["12"]},
    "VASCVS": {"-": ["21"]},
    "VASTATIO": {"-": ["13"]},
    "VASTATOR": {"-": ["13"]},
    "VASTATRIX": {"-": ["13"]},
    "VASTE": {"-": ["60"]},
    "VASTITAS": {"-": ["13"]},
    "VASTITVDO": {"-": ["13"]},
    "VASTO": {"-": ["51"]},
    "VASTVS": {"-": ["21"]},
    "VATES": {"-": ["13"]},
    "VATICINATIO": {"-": ["13"]},
    "VATICINOR": {"-": ["5J"]},
    "VATILLVM": {"-": ["12"]},
    "VBER": {"1": ["13"], "2": ["26"]},
    "VBERE": {"-": ["60"]},
    "VBERO": {"-": ["51"]},
    "VBERTAS": {"-": ["13"]},
    "VBERTIM": {"-": ["60"]},
    "VBI": {"1": ["66"], "2": ["67"], "3": ["82"]},
    "VBICVMQVE": {"1": ["66"], "2": ["60"]},
    "VBILIBET": {"-": ["60"]},
    "VBINAM": {"1": ["66"], "2": ["67"]},
    "VBIQVE": {"-": ["60"]},
    "VBIVIS": {"-": ["60"]},
    "VDVS": {"-": ["21"]},
    "VE": {"-": ["81"]},
    "VECORDIA": {"-": ["11"]},
    "VECORS": {"-": ["25"]},
    "VECTATIO": {"-": ["13"]},
    "VECTIARIVS": {"-": ["12"]},
    "VECTICVLARIVS": {"-": ["21"]},
    "VECTIGAL": {"-": ["13"]},
    "VECTIGALIS": {"-": ["24"]},
    "VECTIS": {"-": ["13"]},
    "VECTITO": {"-": ["51"]},
    "VECTO": {"-": ["51"]},
    "VECTOR": {"1": ["13"], "2": ["5J", "13"]},
    "VECTORIVS": {"-": ["21"]},
    "VECTVRA": {"-": ["11"]},
    "VEDIVS": {"N": ["12"]},
    "VEGEO": {"-": ["52"]},
    "VEGETO": {"-": ["51"]},
    "VEGETVS": {"-": ["21"]},
    "VEGRANDIS": {"-": ["24"]},
    "VEHEMENS": {"-": ["25"]},
    "VEHEMENTER": {"-": ["60"]},
    "VEHEMENTIA": {"-": ["11"]},
    "VEHES": {"-": ["13"]},
    "VEHICVLVM": {"-": ["12"]},
    "VEHO": {"-": ["53"]},
    "VEHOR": {"-": ["5L"]},
    "VEIENS": {"N": ["25"], "A": ["25"]},
    "VEIENTES": {"N": ["13"]},
    "VEII": {"N": ["12"]},
    "VEL": {"1": ["60"], "2": ["81"]},
    "VELAMEN": {"-": ["13"]},
    "VELAMENTVM": {"-": ["12"]},
    "VELARIVM": {"-": ["12"]},
    "VELES": {"-": ["13"]},
    "VELIA": {"N": ["11"]},
    "VELIFER": {"-": ["21"]},
    "VELIFICO": {"-": ["51"]},
    "VELIFICOR": {"-": ["5J"]},
    "VELITARIS": {"-": ["24"]},
    "VELITERNVS": {"N": ["21"]},
    "VELITRAE": {"N": ["11"]},
    "VELIVOLVS": {"-": ["21"]},
    "VELLICATIO": {"-": ["13"]},
    "VELLICO": {"-": ["51"]},
    "VELLO": {"-": ["53"]},
    "VELLVS": {"-": ["13"]},
    "VELO": {"-": ["51"]},
    "VELOCITAS": {"-": ["13"]},
    "VELOCITER": {"-": ["60"]},
    "VELOX": {"-": ["25"]},
    "VELVM": {"-": ["12"]},
    "VELVT": {"1": ["66"], "2": ["82"]},
    "VELVTSI": {"-": ["82"]},
    "VENA": {"-": ["11"]},
    "VENABVLVM": {"-": ["12"]},
    "VENALES": {"-": ["13"]},
    "VENALIA": {"-": ["13"]},
    "VENALIS": {"-": ["24"]},
    "VENANS": {"-": ["13"]},
    "VENATICVS": {"-": ["21"]},
    "VENATIO": {"-": ["13"]},
    "VENATOR": {"-": ["13"]},
    "VENATORIVS": {"-": ["21"]},
    "VENATRIX": {"-": ["13"]},
    "VENATVS": {"-": ["14"]},
    "VENDAX": {"-": ["25"]},
    "VENDIBILIS": {"-": ["24"]},
    "VENDITATIO": {"-": ["13"]},
    "VENDITIO": {"-": ["13"]},
    "VENDITO": {"-": ["51"]},
    "VENDITOR": {"-": ["13"]},
    "VENDITVM": {"-": ["12"]},
    "VENDO": {"-": ["53"]},
    "VENEFICA": {"-": ["11"]},
    "VENEFICIVM": {"-": ["12"]},
    "VENEFICVS": {"1": ["12"], "2": ["21"]},
    "VENENATVM": {"-": ["12"]},
    "VENENATVS": {"-": ["21"]},
    "VENENO": {"-": ["51"]},
    "VENENVM": {"-": ["12"]},
    "VENEO": {"-": ["56"]},
    "VENERABILIS": {"-": ["24"]},
    "VENERABVNDVS": {"-": ["21"]},
    "VENERANDVS": {"-": ["21"]},
    "VENERATIO": {"-": ["13"]},
    "VENEREVS": {"N": ["21"], "A": ["21"], "-": ["21"]},
    "VENERIA": {"-": ["11"]},
    "VENEROR": {"-": ["5J"]},
    "VENETI": {"1N": ["12"], "N": ["12"]},
    "VENIA": {"-": ["11"]},
    "VENIO": {"-": ["54"]},
    "VENNONIVS": {"N": ["12"]},
    "VENOR": {"-": ["5J"]},
    "VENOSVS": {"-": ["21"]},
    "VENTER": {"-": ["13"]},
    "VENTILO": {"-": ["51"]},
    "VENTITO": {"-": ["51"]},
    "VENTO": {"-": ["51"]},
    "VENTOSVS": {"-": ["21"]},
    "VENTRICVLVM": {"-": ["12"]},
    "VENTRICVLVS": {"-": ["12"]},
    "VENTVRVM": {"-": ["12"]},
    "VENTVS": {"-": ["12"]},
    "VENVCVLVS": {"-": ["21"]},
    "VENVLEIVS": {"N": ["12"]},
    "VENVM": {"-": ["12"]},
    "VENVNDO": {"-": ["51"]},
    "VENVS": {"1": ["13"], "2": ["14"], "N": ["13"]},
    "VENVSTAS": {"-": ["13"], "N": ["13"]},
    "VENVSTE": {"-": ["60"]},
    "VENVSTVS": {"-": ["21"]},
    "VEPALLIDVS": {"-": ["21"]},
    "VEPRES": {"-": ["13"]},
    "VER": {"-": ["13"]},
    "VERATRVM": {"-": ["12"]},
    "VERAX": {"-": ["25"], "N": ["13"]},
    "VERBENA": {"-": ["11"]},
    "VERBER": {"-": ["13"]},
    "VERBERATVS": {"-": ["14"]},
    "VERBERO": {"1": ["13"], "2": ["51"]},
    "VERBOSVS": {"-": ["21"]},
    "VERBVM": {"-": ["12"]},
    "VERE": {"-": ["60"]},
    "VERECVNDE": {"-": ["60"]},
    "VERECVNDIA": {"-": ["11"]},
    "VERECVNDVS": {"-": ["21"]},
    "VERENDVS": {"-": ["21"]},
    "VEREOR": {"-": ["5K"]},
    "VERGENS": {"-": ["25"]},
    "VERGO": {"-": ["53"]},
    "VERGOBRETVS": {"-": ["12"]},
    "VERIDICVS": {"-": ["21"]},
    "VERISIMILIS": {"-": ["24"]},
    "VERISIMILITVDO": {"-": ["13"]},
    "VERITAS": {"-": ["13"], "N": ["13"]},
    "VERMICVLVS": {"-": ["12"]},
    "VERMINA": {"-": ["13"]},
    "VERMINATIO": {"-": ["13"]},
    "VERMINO": {"-": ["51"]},
    "VERMIS": {"-": ["13"]},
    "VERNA": {"-": ["11"]},
    "VERNACVLA": {"-": ["11"]},
    "VERNACVLVS": {"1": ["12"], "2": ["21"]},
    "VERNILITAS": {"-": ["13"]},
    "VERNILITER": {"-": ["60"]},
    "VERNO": {"1": ["51"], "2": ["60"]},
    "VERNVLA": {"-": ["11"]},
    "VERNVM": {"-": ["12"]},
    "VERNVS": {"-": ["21"]},
    "VERO": {"1": ["13"], "2": ["51"], "3": ["60"], "4": ["81"]},
    "VERPA": {"-": ["11"]},
    "VERPVS": {"-": ["12"]},
    "VERRES": {"-": ["13"], "N": ["13"]},
    "VERRIA": {"N": ["12"]},
    "VERRINVS": {"N": ["21"], "-": ["21"], "A": ["21"]},
    "VERRIVS": {"A": ["21"]},
    "VERRO": {"-": ["53"]},
    "VERRVCA": {"-": ["11"]},
    "VERRVCIVS": {"N": ["12"]},
    "VERRVCOSVS": {"-": ["21"], "N": ["12"]},
    "VERSABILIS": {"-": ["24"]},
    "VERSABVNDVS": {"-": ["21"]},
    "VERSATILIS": {"-": ["24"]},
    "VERSATIO": {"-": ["13"]},
    "VERSATVS": {"-": ["21"]},
    "VERSICOLOR": {"-": ["25"]},
    "VERSICVLVS": {"-": ["12"]},
    "VERSO": {"-": ["51"]},
    "VERSOR": {"-": ["5J"]},
    "VERSVRA": {"-": ["11"]},
    "VERSVS": {"1": ["14"], "2": ["21"], "3": ["60"], "4": ["70"]},
    "VERSVTVS": {"-": ["21"]},
    "VERTEBRA": {"-": ["11"]},
    "VERTENS": {"-": ["25"]},
    "VERTEX": {"-": ["13"]},
    "VERTICVLA": {"-": ["11"]},
    "VERTIGO": {"-": ["13"]},
    "VERTO": {"-": ["53"]},
    "VERTVMNVS": {"N": ["12"]},
    "VERV": {"-": ["14"]},
    "VERVACTVM": {"-": ["12"]},
    "VERVEX": {"-": ["13"]},
    "VERVM": {"1": ["12"], "2": ["12"], "3": ["60"], "4": ["81"], "-": ["12"]},
    "VERVMTAMEN": {"-": ["60"]},
    "VERVMVERO": {"-": ["60"]},
    "VERVS": {"-": ["21"], "N": ["12"]},
    "VERVTVM": {"-": ["12"]},
    "VERVTVS": {"-": ["21"]},
    "VESANIA": {"-": ["11"]},
    "VESANIO": {"-": ["54"]},
    "VESANVS": {"-": ["21"]},
    "VESCOR": {"-": ["5L"]},
    "VESCVS": {"-": ["21"]},
    "VESICA": {"-": ["11"]},
    "VESICVLA": {"-": ["11"]},
    "VESPER": {"-": ["16"], "N": ["12"]},
    "VESPERA": {"-": ["11"]},
    "VESPERASCO": {"-": ["53"]},
    "VESPERTINVS": {"-": ["21"]},
    "VESTA": {"N": ["11"]},
    "VESTALIA": {"-": ["13"]},
    "VESTALIS": {"1": ["13"], "2": ["24"], "A": ["24"], "N": ["13"]},
    "VESTER": {"-": ["42"]},
    "VESTIARIVM": {"-": ["12"]},
    "VESTIARIVS": {"-": ["21"], "2": ["21"], "1": ["12"]},
    "VESTIBVLVM": {"-": ["12"]},
    "VESTIGATOR": {"-": ["13"]},
    "VESTIGIVM": {"-": ["12"]},
    "VESTIGO": {"-": ["51"]},
    "VESTIMENTVM": {"-": ["12"]},
    "VESTIO": {"-": ["54"]},
    "VESTIS": {"-": ["13"]},
    "VESTITVS": {"1": ["14"], "2": ["21"]},
    "VESTRI": {"-": ["42"]},
    "VETELIA": {"N": ["11"]},
    "VETERA": {"-": ["13"], "N": ["11"]},
    "VETERANVS": {"1": ["12"], "2": ["21"]},
    "VETERARIVM": {"-": ["12"]},
    "VETERATOR": {"-": ["13"]},
    "VETERATORIVS": {"-": ["21"]},
    "VETERES": {"-": ["13"]},
    "VETERINVS": {"-": ["21"]},
    "VETERNOSVS": {"-": ["21"]},
    "VETERNVS": {"1": ["12"], "2": ["21"]},
    "VETITVM": {"-": ["12"]},
    "VETO": {"-": ["51"]},
    "VETTIVS": {"N": ["12"]},
    "VETVLA": {"-": ["11"]},
    "VETVLVS": {"1": ["12"], "2": ["21"]},
    "VETVRIA": {"N": ["11"]},
    "VETVRIVS": {"N": ["12"]},
    "VETVS": {"-": ["22"], "N": ["22"], "A": ["21"]},
    "VETVSTAS": {"-": ["13"]},
    "VETVSTVS": {"-": ["21"]},
    "VEXAMEN": {"-": ["13"]},
    "VEXATIO": {"-": ["13"]},
    "VEXATOR": {"-": ["13"]},
    "VEXILLARIVS": {"-": ["12"]},
    "VEXILLVM": {"-": ["12"]},
    "VEXO": {"-": ["51"]},
    "VIA": {"-": ["11"], "N": ["11"]},
    "VIATICVM": {"-": ["12"]},
    "VIATOR": {"-": ["13"]},
    "VIATORIVS": {"-": ["21"]},
    "VIBEX": {"-": ["13"]},
    "VIBIVS": {"N": ["12"]},
    "VIBO": {"N": ["13"]},
    "VIBRANS": {"-": ["25", "35"], "V\u00e9h\u00e9ment": ["25"]},
    "VIBRATVS": {"1": ["14"], "2": ["21"]},
    "VIBRO": {"-": ["51"]},
    "VIBVRNVM": {"-": ["12"]},
    "VICA": {"N": ["11"]},
    "VICARIVS": {"1": ["12"], "2": ["21"]},
    "VICATIM": {"-": ["60"]},
    "VICECOMES": {"-": ["13"]},
    "VICENARIVS": {"-": ["21"]},
    "VICENVS": {"-": ["33"]},
    "VICESIMA": {"-": ["11"]},
    "VICESIMANVS": {"-": ["12"]},
    "VICESIMVS": {"-": ["32"]},
    "VICIA": {"1": ["11"], "2": ["11"]},
    "VICIES": {"-": ["36"]},
    "VICINA": {"-": ["11"]},
    "VICINALIS": {"-": ["24"]},
    "VICINIA": {"-": ["11"]},
    "VICINITAS": {"-": ["13"]},
    "VICINVM": {"-": ["12"]},
    "VICINVS": {"1": ["12"], "2": ["21"]},
    "VICIS": {"-": ["13"]},
    "VICISSIM": {"-": ["60"]},
    "VICISSITVDO": {"-": ["13"]},
    "VICTIMA": {"-": ["11"]},
    "VICTOR": {"-": ["13"], "N": ["13"]},
    "VICTORIA": {"-": ["11"], "N": ["11"]},
    "VICTORIATVS": {"-": ["12"]},
    "VICTRIX": {"-": ["13"], "2": ["22"], "1": ["21"]},
    "VICTVS": {"-": ["14"]},
    "VICVLVS": {"-": ["12"]},
    "VICVS": {"-": ["12"], "N": ["12"]},
    "VIDELICET": {"-": ["60"]},
    "VIDENS": {"-": ["25"]},
    "VIDEO": {"-": ["52"]},
    "VIDVA": {"-": ["11"]},
    "VIDVERTAS": {"-": ["13"]},
    "VIDVITAS": {"-": ["13"]},
    "VIDVO": {"-": ["51"]},
    "VIDVVS": {"-": ["21"]},
    "VIEO": {"-": ["52"]},
    "VIESCO": {"-": ["53"]},
    "VIETVS": {"-": ["21"]},
    "VIGENS": {"-": ["25"]},
    "VIGEO": {"-": ["52"]},
    "VIGESCO": {"-": ["53"]},
    "VIGIL": {"-": ["25"], "2": ["13"], "1": ["12"]},
    "VIGILANS": {"-": ["25"]},
    "VIGILANTER": {"-": ["60"]},
    "VIGILANTIA": {"-": ["11"]},
    "VIGILES": {"-": ["13"]},
    "VIGILIA": {"-": ["11"]},
    "VIGILIARIVM": {"-": ["12"]},
    "VIGILO": {"-": ["51"]},
    "VIGINTI": {"-": ["31"]},
    "VIGINTIVIRATVS": {"-": ["14"]},
    "VIGOR": {"-": ["13"]},
    "VILIS": {"-": ["24"]},
    "VILITAS": {"-": ["13"]},
    "VILLA": {"-": ["11"]},
    "VILLICA": {"-": ["11"]},
    "VILLICO": {"2": ["51"]},
    "VILLICVS": {"1": ["12"], "2": ["21"]},
    "VILLOSVS": {"-": ["21"]},
    "VILLVLA": {"-": ["11"]},
    "VILLVS": {"-": ["12"]},
    "VIMEN": {"-": ["13"]},
    "VIMENTVM": {"-": ["12"]},
    "VIMINALIS": {"-": ["24"], "N": ["24"]},
    "VIMINEVS": {"-": ["21"]},
    "VINACEA": {"-": ["11"]},
    "VINACEVS": {"-": ["12"]},
    "VINARIVM": {"-": ["12"]},
    "VINARIVS": {"2": ["21"], "1": ["12"]},
    "VINCIO": {"-": ["54"]},
    "VINCO": {"-": ["53"]},
    "VINCTVS": {"-": ["12"]},
    "VINCVLVM": {"-": ["12"]},
    "VINDEMIA": {"-": ["11"]},
    "VINDEMIATOR": {"-": ["13"]},
    "VINDEX": {"-": ["13"], "N": ["13"]},
    "VINDICIAE": {"-": ["11"]},
    "VINDICIVS": {"N": ["12"]},
    "VINDICO": {"-": ["51"]},
    "VINDICTA": {"-": ["11"]},
    "VINEA": {"-": ["11"]},
    "VINEATICVS": {"-": ["21"]},
    "VINETVM": {"-": ["12"]},
    "VINITOR": {"-": ["13"]},
    "VINOLENTIA": {"-": ["11"]},
    "VINOLENTVS": {"1": ["12"], "2": ["21"]},
    "VINOSVS": {"-": ["21"]},
    "VINVM": {"-": ["12"]},
    "VIOLA": {"-": ["11"]},
    "VIOLABILIS": {"-": ["24"]},
    "VIOLACEVS": {"-": ["21"]},
    "VIOLARIVM": {"-": ["12"]},
    "VIOLATIO": {"-": ["13"]},
    "VIOLATOR": {"-": ["13"]},
    "VIOLENS": {"-": ["25"]},
    "VIOLENTER": {"-": ["60"]},
    "VIOLENTIA": {"-": ["11"]},
    "VIOLENTVS": {"-": ["21"]},
    "VIOLO": {"-": ["51"]},
    "VIPERA": {"-": ["11"]},
    "VIPEREVS": {"-": ["21"]},
    "VIPERINVS": {"-": ["21"]},
    "VIR": {"-": ["12"]},
    "VIRAGO": {"-": ["13"]},
    "VIRENS": {"-": ["25"]},
    "VIRENTIA": {"-": ["13"]},
    "VIREO": {"1": ["13"], "2": ["52"]},
    "VIRESCO": {"1": ["53"], "2": ["53"], "-": ["53"]},
    "VIRETVM": {"-": ["12"]},
    "VIRGA": {"-": ["11"]},
    "VIRGATVS": {"-": ["21"]},
    "VIRGEVS": {"-": ["21"]},
    "VIRGILIAE": {"-": ["11"]},
    "VIRGILIVS": {"N": ["12"]},
    "VIRGINALIS": {"-": ["24"]},
    "VIRGINEVS": {"-": ["21"], "A": ["21"]},
    "VIRGINITAS": {"-": ["13"]},
    "VIRGINIVS": {"N": ["12"]},
    "VIRGO": {"-": ["13"], "N": ["13"]},
    "VIRGVLA": {"-": ["11"]},
    "VIRGVLTVM": {"-": ["12"]},
    "VIRGVNCVLA": {"-": ["11"]},
    "VIRIDANS": {"-": ["25"]},
    "VIRIDE": {"1": ["13"], "2": ["60"], "-": ["6-"]},
    "VIRIDIS": {"-": ["24"]},
    "VIRIDITAS": {"-": ["13"]},
    "VIRIDO": {"-": ["51"]},
    "VIRILIS": {"-": ["24"], "A": ["24"]},
    "VIRILITAS": {"-": ["13"]},
    "VIRILITER": {"-": ["60"]},
    "VIRITIM": {"-": ["60"]},
    "VIROSVS": {"-": ["21"]},
    "VIRTVS": {"-": ["13"], "N": ["13"]},
    "VIRVS": {"-": ["16"]},
    "VIS": {"-": ["13"]},
    "VISCERATIO": {"-": ["13"]},
    "VISCO": {"-": ["51"]},
    "VISCVM": {"-": ["12"]},
    "VISCVS": {"1": ["12"], "2": ["13"], "N": ["12"]},
    "VISITATIO": {"-": ["13"]},
    "VISITO": {"-": ["51"]},
    "VISO": {"-": ["53"]},
    "VISVM": {"-": ["12"]},
    "VISVS": {"-": ["14"]},
    "VITA": {"-": ["11"]},
    "VITABVNDVS": {"-": ["21"]},
    "VITALIA": {"-": ["13"]},
    "VITALIS": {"-": ["24"]},
    "VITALITER": {"-": ["60"]},
    "VITELLIVS": {"N": ["12"]},
    "VITELLVS": {"-": ["12"]},
    "VITEVS": {"-": ["21"]},
    "VITEX": {"-": ["13"]},
    "VITIARIVM": {"-": ["12"]},
    "VITICVLA": {"-": ["11"]},
    "VITIGENVS": {"-": ["21"]},
    "VITIGINEVS": {"-": ["21"]},
    "VITILIS": {"-": ["24"]},
    "VITILITIGO": {"-": ["51"]},
    "VITIO": {"1": ["51"], "2": ["54"]},
    "VITIOSITAS": {"-": ["13"]},
    "VITIOSVS": {"-": ["21"]},
    "VITIS": {"-": ["13"]},
    "VITIVM": {"-": ["12"]},
    "VITO": {"-": ["51"]},
    "VITRARIVS": {"1": ["12"], "2": ["21"]},
    "VITREVM": {"-": ["12"]},
    "VITREVS": {"-": ["21"]},
    "VITRICVS": {"-": ["12"]},
    "VITRVM": {"-": ["12"]},
    "VITTA": {"-": ["11"]},
    "VITTATVS": {"-": ["21"]},
    "VITVLA": {"-": ["11"]},
    "VITVLVS": {"-": ["12"]},
    "VITVPERATIO": {"-": ["13"]},
    "VITVPERIVM": {"-": ["12"]},
    "VITVPERO": {"1": ["13"], "2": ["51"]},
    "VIVARIVM": {"-": ["12"]},
    "VIVATVS": {"-": ["21"]},
    "VIVAX": {"-": ["25"]},
    "VIVENS": {"-": ["13"]},
    "VIVESCO": {"-": ["53"]},
    "VIVIDVS": {"-": ["21"]},
    "VIVIRADIX": {"-": ["13"]},
    "VIVO": {"-": ["53"]},
    "VIVVM": {"-": ["12"]},
    "VIVVS": {"1": ["12"], "2": ["21"]},
    "VIX": {"-": ["60"]},
    "VIXDVM": {"-": ["60"]},
    "VLCERATIO": {"-": ["13"]},
    "VLCERO": {"-": ["51"]},
    "VLCEROSVS": {"-": ["21"]},
    "VLCISCOR": {"-": ["5L"]},
    "VLCVS": {"-": ["13"]},
    "VLCVSCVLVM": {"-": ["12"]},
    "VLIGO": {"-": ["13"]},
    "VLIXES": {"N": ["17"]},
    "VLIXEVS": {"N": ["21"]},
    "VLLVS": {"-": ["48"]},
    "VLMEVS": {"-": ["21"]},
    "VLMVS": {"-": ["12"]},
    "VLNA": {"-": ["11"]},
    "VLPICVM": {"-": ["12"]},
    "VLTERIOR": {"-": ["2&"], "A": ["2&"]},
    "VLTERIORA": {"-": ["13"]},
    "VLTIMVM": {"1": ["12"], "2": ["6-", "12"]},
    "VLTIMVS": {"-": ["2-", "21"]},
    "VLTIO": {"-": ["13"], "N": ["13"]},
    "VLTOR": {"-": ["13"], "N": ["13"]},
    "VLTRA": {"1": ["60"], "2": ["70"]},
    "VLTRIX": {"-": ["13"]},
    "VLTRO": {"-": ["60"]},
    "VLVA": {"-": ["11"]},
    "VLVLA": {"-": ["11"]},
    "VLVLATVS": {"-": ["14"]},
    "VLVLO": {"-": ["51"]},
    "VMBELLA": {"-": ["11"]},
    "VMBILICVS": {"-": ["12"]},
    "VMBO": {"-": ["13"]},
    "VMBRA": {"-": ["11"]},
    "VMBRACVLVM": {"-": ["12"]},
    "VMBRATICVS": {"-": ["21"]},
    "VMBRIFER": {"-": ["21"]},
    "VMBRO": {"-": ["51"], "N": ["11"]},
    "VMBROSA": {"-": ["12"]},
    "VMBROSVS": {"-": ["21"]},
    "VMQVAM": {"-": ["60"]},
    "VNA": {"-": ["60"]},
    "VNANIMVS": {"-": ["21"]},
    "VNCIA": {"-": ["11"]},
    "VNCIARIVS": {"-": ["21"]},
    "VNCINVS": {"-": ["12"]},
    "VNCIOLA": {"-": ["11"]},
    "VNCTIO": {"-": ["13"]},
    "VNCTOR": {"-": ["13"]},
    "VNCTVM": {"-": ["12"]},
    "VNCTVS": {"1": ["14"], "2": ["21"]},
    "VNCVS": {"1": ["12"], "2": ["21"]},
    "VNDA": {"-": ["11"]},
    "VNDE": {"1": ["66"], "2": ["67"]},
    "VNDECIM": {"-": ["31"]},
    "VNDECIMVS": {"-": ["32"]},
    "VNDECVMQVE": {"1": ["66"], "2": ["60"]},
    "VNDENI": {"-": ["33"]},
    "VNDEOCTOGINTA": {"-": ["31"]},
    "VNDEQVINQVAGESIMVS": {"-": ["31"]},
    "VNDEQVINQVAGINTA": {"-": ["31"]},
    "VNDETRICESIMVS": {"-": ["32"]},
    "VNDETRIGINTA": {"-": ["31"]},
    "VNDEVICESIMVS": {"-": ["32"]},
    "VNDEVIGINTI": {"-": ["31"]},
    "VNDIQVE": {"-": ["60"]},
    "VNDISONVS": {"-": ["21"]},
    "VNDO": {"-": ["51"]},
    "VNDOSVS": {"-": ["21"]},
    "VNETVICESIMANI": {"-": ["12"]},
    "VNETVICESIMVS": {"-": ["32"]},
    "VNGO": {"-": ["53"]},
    "VNGVEN": {"-": ["13"]},
    "VNGVENTARIA": {"-": ["11"]},
    "VNGVENTARIVM": {"-": ["12"]},
    "VNGVENTARIVS": {"1": ["12"], "2": ["21"]},
    "VNGVENTO": {"-": ["51"]},
    "VNGVENTVM": {"-": ["12"]},
    "VNGVICVLVS": {"-": ["12"]},
    "VNGVIS": {"-": ["13"]},
    "VNGVITO": {"-": ["51"]},
    "VNGVLA": {"-": ["11"]},
    "VNICE": {"-": ["60"]},
    "VNICVS": {"-": ["21"]},
    "VNIFORMIS": {"-": ["24"]},
    "VNIGENA": {"-": ["21"]},
    "VNIO": {"1": ["13"], "2": ["54"]},
    "VNITAS": {"-": ["13"]},
    "VNITER": {"-": ["60"]},
    "VNIVERSALIS": {"-": ["24"]},
    "VNIVERSE": {"-": ["60"]},
    "VNIVERSI": {"-": ["12"]},
    "VNIVERSITAS": {"-": ["13"]},
    "VNIVERSVM": {"-": ["12"]},
    "VNIVERSVS": {"-": ["21"]},
    "VNVM": {"-": ["12"]},
    "VNVS": {"-": ["31"]},
    "VNVSQVISQVE": {"-": ["48"]},
    "VNVSQVISQVIS": {"-": ["48"]},
    "VOCABVLVM": {"-": ["12"]},
    "VOCALES": {"-": ["13"]},
    "VOCALIS": {"1": ["13"], "2": ["24"]},
    "VOCAMEN": {"-": ["13"]},
    "VOCATIO": {"-": ["13"]},
    "VOCATOR": {"-": ["13"]},
    "VOCIFERATIO": {"-": ["13"]},
    "VOCIFEROR": {"-": ["5J"]},
    "VOCITO": {"-": ["51"]},
    "VOCO": {"-": ["51"]},
    "VOCONIVS": {"A": ["21"], "N": ["12"]},
    "VOCVLA": {"-": ["11"], "N": ["11"]},
    "VOLANS": {"-": ["13"]},
    "VOLANTES": {"-": ["13"]},
    "VOLATICVS": {"-": ["21"]},
    "VOLATILIS": {"-": ["24"]},
    "VOLATVS": {"-": ["14"]},
    "VOLEMVS": {"-": ["21"]},
    "VOLENS": {"-": ["25"]},
    "VOLERO": {"N": ["13"]},
    "VOLESVS": {"N": ["12"]},
    "VOLITO": {"-": ["51"]},
    "VOLO": {"1": ["13"], "2": ["51"], "3": ["56"]},
    "VOLSCI": {"N": ["12"]},
    "VOLSCVS": {"N": ["21"], "A": ["21"]},
    "VOLTEIVS": {"N": ["12"]},
    "VOLVBILIS": {"-": ["24"]},
    "VOLVBILITAS": {"-": ["13"]},
    "VOLVCER": {"-": ["23"]},
    "VOLVCRIS": {"-": ["13"]},
    "VOLVMEN": {"-": ["13"]},
    "VOLVMNIA": {"N": ["11"]},
    "VOLVNTARII": {"-": ["12"]},
    "VOLVNTARIVS": {"-": ["21"]},
    "VOLVNTAS": {"-": ["13"]},
    "VOLVO": {"-": ["53"]},
    "VOLVPTARIVS": {"-": ["21"]},
    "VOLVPTAS": {"-": ["13"], "N": ["13"]},
    "VOLVSIVS": {"N": ["12"]},
    "VOLVTA": {"-": ["11"]},
    "VOLVTABRVM": {"-": ["12"]},
    "VOLVTATIO": {"-": ["13"]},
    "VOLVTO": {"-": ["51"]},
    "VOMER": {"-": ["13"]},
    "VOMICA": {"-": ["11"]},
    "VOMIS": {"-": ["13"]},
    "VOMITIO": {"-": ["13"]},
    "VOMITO": {"-": ["51"]},
    "VOMITOR": {"-": ["13"]},
    "VOMITORIVM": {"-": ["12"]},
    "VOMITORIVS": {"-": ["21"]},
    "VOMITVS": {"-": ["14"]},
    "VOMO": {"-": ["53"]},
    "VORAGINOSVS": {"-": ["21"]},
    "VORAGO": {"-": ["13"]},
    "VORAX": {"-": ["25"]},
    "VORO": {"-": ["51"]},
    "VOS": {"-": ["41"]},
    "VOTIVVM": {"-": ["12"]},
    "VOTIVVS": {"-": ["21"]},
    "VOTO": {"-": ["51"]},
    "VOTVM": {"-": ["12"]},
    "VOVEO": {"-": ["52"]},
    "VOX": {"-": ["13"]},
    "VRBANI": {"-": ["12"]},
    "VRBANITAS": {"-": ["13"]},
    "VRBANVS": {"-": ["21"]},
    "VRBIVS": {"N": ["21"]},
    "VRBS": {"-": ["13"], "N": ["13"]},
    "VRCEOLVS": {"-": ["12"]},
    "VRCEVS": {"-": ["12"]},
    "VREDO": {"-": ["13"]},
    "VRGENS": {"-": ["25"]},
    "VRGEO": {"-": ["52"]},
    "VRINA": {"-": ["11"]},
    "VRIVS": {"N": ["17"]},
    "VRNA": {"-": ["11"]},
    "VRNALIS": {"-": ["24"]},
    "VRO": {"-": ["53"]},
    "VRSA": {"-": ["11"], "N": ["11"]},
    "VRSVS": {"-": ["12"]},
    "VRTICA": {"-": ["11"]},
    "VRVCA": {"-": ["11"]},
    "VRVS": {"-": ["12"]},
    "VSIO": {"-": ["13"]},
    "VSITATVS": {"-": ["21"]},
    "VSPIAM": {"-": ["60"]},
    "VSQVAM": {"-": ["60"]},
    "VSQVE": {"-": ["60"]},
    "VSQVEQVAQVE": {"-": ["60"]},
    "VSTA": {"-": ["11"]},
    "VSTOR": {"-": ["13"]},
    "VSTVLO": {"-": ["51"]},
    "VSVALIS": {"-": ["24"]},
    "VSVCAPIO": {"1": ["13"], "2": ["55"]},
    "VSVRA": {"-": ["11"]},
    "VSVRPATIO": {"-": ["13"]},
    "VSVRPO": {"-": ["51"]},
    "VSVS": {"-": ["14"]},
    "VSVSFRVCTVS": {"-": ["14"]},
    "VSVVENIO": {"-": ["54"]},
    "VT": {"1": ["66"], "2": ["67"], "3": ["60"], "4": ["82"]},
    "VTCVMQVE": {"1": ["66"], "2": ["60"]},
    "VTENS": {"-": ["25"]},
    "VTENSILIA": {"-": ["13"]},
    "VTENSILIS": {"-": ["24"]},
    "VTER": {"1": ["13"], "2": ["46"], "3": ["47"], "4": ["48"]},
    "VTERCVMQVE": {"1": ["46"], "2": ["48"]},
    "VTERLIBET": {"-": ["48"]},
    "VTERQVE": {"-": ["48"]},
    "VTERVIS": {"-": ["48"]},
    "VTERVS": {"-": ["12"]},
    "VTICA": {"N": ["11"]},
    "VTICENSIS": {"N": ["24"], "A": ["24"]},
    "VTILE": {"-": ["13"]},
    "VTILIS": {"-": ["24"]},
    "VTILITAS": {"-": ["13"]},
    "VTILITER": {"-": ["60"]},
    "VTINAM": {"-": ["60"]},
    "VTIQVE": {"-": ["60"]},
    "VTOR": {"-": ["5L"]},
    "VTPOTE": {"-": ["60"]},
    "VTQVI": {"-": ["82"]},
    "VTRAQVE": {"-": ["60"]},
    "VTRICVS": {"-": ["12"]},
    "VTRIMQVE": {"-": ["60"]},
    "VTRIMQVESECVS": {"-": ["60"]},
    "VTRINDE": {"-": ["60"]},
    "VTRO": {"1": ["60"], "2": ["67"]},
    "VTROBI": {"-": ["60"]},
    "VTROBIQVE": {"-": ["60"]},
    "VTROQVE": {"-": ["60"]},
    "VTRVM": {"-": ["67"]},
    "VTRVMNE": {"-": ["67"]},
    "VVA": {"-": ["11"]},
    "VVESCO": {"-": ["53"]},
    "VVIDVLVS": {"-": ["21"]},
    "VVIDVS": {"-": ["21"]},
    "VVLCANVS": {"N": ["12"]},
    "VVLCATIVS": {"N": ["12"]},
    "VVLGARIA": {"-": ["13"]},
    "VVLGARIS": {"-": ["24"]},
    "VVLGATOR": {"-": ["13"]},
    "VVLGATVS": {"-": ["21"]},
    "VVLGIVAGVS": {"-": ["21"]},
    "VVLGO": {"1": ["51"], "2": ["60"]},
    "VVLGVS": {"-": ["12"]},
    "VVLNERABILIS": {"-": ["24"]},
    "VVLNERATIO": {"-": ["13"]},
    "VVLNERO": {"-": ["51"]},
    "VVLNIFICVS": {"-": ["21"]},
    "VVLNVS": {"-": ["13"]},
    "VVLPECVLA": {"-": ["11"]},
    "VVLPES": {"-": ["13"]},
    "VVLTVR": {"-": ["13"], "N": ["13"]},
    "VVLTVRIVS": {"-": ["12"]},
    "VVLTVRNVS": {"-": ["12"], "N": ["12"]},
    "VVLTVS": {"-": ["14"]},
    "VVLVA": {"-": ["11"]},
    "VXOR": {"-": ["13"]},
    "VXORIVS": {"-": ["21"]},
    "XENIVM": {"-": ["12"]},
    "XENO": {"N": ["17"]},
    "XENOCRATES": {"N": ["17"]},
    "XERAMPELINVS": {"-": ["21"]},
    "XYSTVS": {"-": ["17"]},
    "ZAMA": {"N": ["11"]},
    "ZELOTYPVS": {"-": ["21"]},
    "ZELVS": {"-": ["12"]},
    "ZENO": {"N": ["17"]},
    "ZEPHYRVS": {"N": ["12"], "-": ["12"]},
    "ZMARAGDVS": {"-": ["12"]},
    "ZONA": {"-": ["11"]},
    "ZONVLA": {"-": ["11"]},
    "ZOPHORVS": {"-": ["17"]},
    "ZOSIPPVS": {"N": ["12"]},
    "ZYGIOS": {"-": ["25"]},
    "ZYGON": {"-": ["17"]},
    "ALTOR": {"-": ["13"]},
    "AMFRACTVS": {"1": ["14"], "2": ["21"]},
    "BVTHROTOS": {"N": ["17"]},
    "CHERSONENSVS": {"N": ["12"]},
    "COMMINGO5L": {"COMMINISCOR": ["53"]},
    "CONSVLTVS": {"-": ["21"]},
    "CORIVM": {"-": ["12"]},
    "DISIVNCTVS": {"-": ["21"]},
    "DIVNGO": {"-": ["53"]},
    "DISIVNGO": {"-": ["53"]},
    "FVFIVS": {"A": ["21"], "N": ["12"]},
    "GERMANA11": {"GERMANIA": ["11"]},
    "HOSTVS": {"-": ["12"]},
    "LAEVVS": {"-": ["221"]},
    "MVSCVS": {"-": ["12"]},
    "NEMPE": {"-": ["60"]},
    "PAVLLO": {"-": ["60"]},
    "PETREIVS": {"N": ["12"]},
    "PROMVNTVRIVM": {"-": ["12"]},
    "RECERNO": {"-": ["53"]},
    "RVSTICATIO": {"-": ["213"]},
    "SCOBIS": {"-": ["13"]},
    "SVPERINICIO": {"-": ["55"]},
    "_SVM": {"1": ["56"], "2": ["E6"]},
    "VENERIVS": {"N": ["21"], "-": ["21"]},
    "#": {"-": ["--"]},
    "BABYLONICVS": {"A": ["21"]},
    "MAMMIVS": {"N": ["12"]},
    "MICIPSA": {"N": ["11"]},
    "SVCRONENSIS": {"A": ["24"]},
    "CANDAVIA": {"N": ["11"]},
    "CECROPIA": {"N": ["11"]},
    "LVCCIVS": {"N": ["12"]},
    "SEGIMERVS": {"N": ["12"]},
    "BEDRIACVM": {"N": ["12"]},
    "PYLIVS": {"A": ["21"]},
    "PERILLEVS": {"A": ["21"]},
    "REMMIVS": {"A": ["21"], "N": ["12"]},
    "ICARIVM": {"N": ["12"]},
    "LATIARIS": {"A": ["24"], "N": ["16"]},
    "ACHILLEVS": {"A": ["21"]},
    "LIBANVS": {"N": ["12"]},
    "APAMEA": {"N": ["11"]},
    "CRISPINILLA": {"N": ["11"]},
    "LOLLVS": {"N": ["12"]},
    "CAIVS": {"N": ["12"]},
    "MARCIANVS": {"N": ["12"], "A": ["21"]},
    "SABELLVS": {"A": ["21"]},
    "ZOSIMVS": {"N": ["12"]},
    "ILLYRIVS": {"A": ["21"]},
    "FVCINVS": {"N": ["12"]},
    "ARGYRIPPVS": {"N": ["12"]},
    "PRIAMEIVS": {"A": ["21"]},
    "ORTYGIVS": {"A": ["21"], "N": ["12"]},
    "VALA": {"N": ["11"]},
    "THYBRIS": {"A": ["24"], "N": ["16"]},
    "AVERNA": {"N": ["11"]},
    "SCAEA": {"N": ["11"]},
    "PAPIVS": {"N": ["12"], "A": ["21"]},
    "CRVSTVMIVS": {"A": ["21"]},
    "SVLMONENSIS": {"A": ["24"]},
    "EPYTVS": {"N": ["12"]},
    "RAVOLA": {"N": ["11"]},
    "SALAMINIVS": {"A": ["21"]},
    "APOPHORETA": {"2": ["12"], "1": ["12"]},
    "VESTRICIVS": {"N": ["12"]},
    "CEPHEIVS": {"A": ["21"]},
    "ACVTIA": {"N": ["11"]},
    "CATVLLA": {"N": ["11"]},
    "COLCHVS": {"A": ["21"]},
    "CVPIENNIVS": {"N": ["12"]},
    "PHEMIVS": {"A": ["21"]},
    "ASCRA": {"N": ["11"]},
    "DVCENIVS": {"N": ["12"]},
    "CARVILIVS": {"N": ["12"]},
    "CHERVSCVS": {"A": ["21"]},
    "TESSERARIVS": {"1": ["12"]},
    "LEVCTRA": {"N": ["11"]},
    "SAENIVS": {"A": ["21"], "N": ["12"]},
    "OBARITVS": {"N": ["12"]},
    "VIBVLENVS": {"N": ["12"]},
    "TARRACINENSIS": {"A": ["24"], "N": ["16"]},
    "HIBERVS": {"A": ["21"], "N": ["12"]},
    "ORCHIVIVS": {"N": ["12"]},
    "TALAVS": {"N": ["12"]},
    "CLONIVS": {"N": ["12"]},
    "TIMOTHEVS": {"N": ["12"]},
    "LATERANVS": {"N": ["12"]},
    "STATILIVS": {"N": ["12"]},
    "GABINIVS": {"N": ["12"], "A": ["21"]},
    "FLAVIANVS": {"A": ["21"], "N": ["12"]},
    "HICETAONIVS": {"A": ["21"]},
    "CLARIVS": {"N": ["12"], "A": ["21"]},
    "ACILIANVS": {"N": ["12"]},
    "CINNAMVS": {"N": ["12"]},
    "COLMINIANVS": {"A": ["21"]},
    "CARATACVS": {"N": ["12"]},
    "VOPISCVS": {"N": ["12"]},
    "ACTAEVS": {"A": ["21"]},
    "SIMBRVINVS": {"A": ["21"]},
    "ARIOVISTVS": {"N": ["12"]},
    "SOCRATICVS": {"A": ["21"]},
    "BIBVLA": {"N": ["11"]},
    "OEENSIS": {"A": ["24"]},
    "MEGAREA": {"N": ["11"]},
    "SOPHOCLEVS": {"A": ["21"]},
    "ALBINTIMILIVM": {"N": ["12"]},
    "HIPPARCHVS": {"N": ["12"]},
    "SILANA": {"N": ["11"]},
    "AMALTHEA": {"N": ["11"]},
    "CASTALIVS": {"A": ["21"]},
    "CAPHAREVS": {"A": ["21"], "N": ["12"]},
    "ELYSIVS": {"A": ["21"]},
    "OTOGESA": {"N": ["11"]},
    "THAMYRVS": {"N": ["12"]},
    "NERITIVS": {"A": ["21"]},
    "ALCATHOVS": {"N": ["12"]},
    "TOGONIVS": {"N": ["12"]},
    "VOLTVRCIVS": {"N": ["12"]},
    "SALENTINVS": {"A": ["21"]},
    "TIRYNTHIVS": {"N": ["12"], "A": ["21"]},
    "FVSCINVS": {"N": ["12"]},
    "PHENEVM": {"N": ["12"]},
    "TOLVMNIVS": {"N": ["12"]},
    "LATIVS": {"A": ["21"]},
    "SEMIPLACENTINVS": {"A": ["21"]},
    "AMORGVS": {"N": ["12"]},
    "PASITHEA": {"N": ["11"]},
    "VOLVMNIVS": {"N": ["12"]},
    "ANCHARIVS": {"N": ["12"]},
    "PHALEREVS": {"A": ["21"]},
    "SORICARIA": {"N": ["11"]},
    "BAEBIVS": {"N": ["12"]},
    "MODIA": {"N": ["11"]},
    "LYCVS": {"N": ["12"]},
    "MASSICVM": {"N": ["12"]},
    "CAVARILLVS": {"N": ["12"]},
    "LABDACVS": {"N": ["12"]},
    "PALINVRVS": {"N": ["12"]},
    "LARONIA": {"N": ["11"]},
    "VTRICVLVS": {"1": ["12"]},
    "TRAIANVS": {"A": ["21"], "N": ["12"]},
    "GABOLVS": {"N": ["12"]},
    "SVMMANVS": {"N": ["12"]},
    "HYRIEVS": {"A": ["21"], "N": ["12"]},
    "STYMPHALA": {"N": ["11"]},
    "ILIONEVS": {"N": ["12"]},
    "CEPHEVS": {"N": ["12"], "A": ["21"]},
    "PEDIVS": {"N": ["12"]},
    "CAPVA": {"N": ["11"]},
    "VOLSINIVS": {"A": ["21"]},
    "LAESTRYGONIVS": {"A": ["21"]},
    "AETHIOPIA": {"N": ["11"]},
    "CAMPANVS": {"A": ["21"], "N": ["12"]},
    "AMPSANCTVS": {"N": ["12"]},
    "HORVS": {"N": ["12"]},
    "TRAVSIVS": {"N": ["12"]},
    "VITELLIA": {"N": ["11"]},
    "PATVLCIVS": {"N": ["12"]},
    "CVRTILIVS": {"N": ["12"]},
    "SCANTIVS": {"A": ["21"]},
    "MEMNONIVS": {"A": ["21"]},
    "SATYRA": {"N": ["11"]},
    "PERSICVS": {"N": ["12"], "A": ["21"]},
    "TERPOLIVS": {"N": ["12"]},
    "POMPONIA": {"N": ["11"]},
    "CAMVLOGENVS": {"N": ["12"]},
    "AGNALIA": {"N": ["11"]},
    "COLONEVS": {"A": ["21"]},
    "SVADELA": {"N": ["11"]},
    "TRYPHAENA": {"N": ["11"]},
    "SARDANAPALVS": {"N": ["12"]},
    "IASIVS": {"N": ["12"]},
    "DECRIVS": {"N": ["12"]},
    "PENTHESILEA": {"N": ["11"]},
    "CESTIVS": {"N": ["12"]},
    "TRINACRIVS": {"A": ["21"]},
    "ADIATVANVS": {"N": ["12"]},
    "MENA": {"N": ["11"]},
    "FIRMINVS": {"N": ["12"]},
    "CLITVMNVS": {"N": ["12"]},
    "TIBVRTVS": {"N": ["12"]},
    "DAMASIPPVS": {"N": ["12"]},
    "PEGASEVS": {"A": ["21"]},
    "VOSEGVS": {"N": ["12"]},
    "CHLOREVS": {"N": ["12"]},
    "NARNIA": {"N": ["11"]},
    "RHOETVS": {"N": ["12"]},
    "LYCAONIVS": {"A": ["21"]},
    "CHAONIVS": {"A": ["21"]},
    "PEDANVS": {"A": ["21"]},
    "TRIFOLINVS": {"A": ["21"]},
    "TARENTVM": {"N": ["12"]},
    "CYLLARVS": {"N": ["12"]},
    "CLARANVS": {"N": ["12"]},
    "SALIVS": {"A": ["21"], "N": ["12"]},
    "LYSIPPVS": {"N": ["12"]},
    "OFONIVS": {"N": ["12"]},
    "GYGAEVS": {"A": ["21"]},
    "HYPERBOREVS": {"A": ["21"]},
    "ORESTILLA": {"N": ["11"]},
    "POLYPHEMVS": {"N": ["12"]},
    "ATRIVS": {"N": ["12"]},
    "LANVVIVM": {"N": ["12"]},
    "GALLOGRAECIA": {"N": ["11"]},
    "ARQVITIVS": {"N": ["12"]},
    "DELIA": {"N": ["11"]},
    "VBIVS": {"A": ["21"]},
    "NONACRIVS": {"A": ["21"]},
    "NERVLLINVS": {"N": ["12"]},
    "QVADRATILLA": {"N": ["11"]},
    "IVLIANVS": {"A": ["21"], "N": ["12"]},
    "ANAGNIA": {"N": ["11"]},
    "ZALACVS": {"N": ["12"]},
    "MIMALLONEVS": {"A": ["21"]},
    "ARGIVVS": {"A": ["21"]},
    "TARRACINA": {"N": ["11"]},
    "VIBELLIVS": {"N": ["12"]},
    "TISAMENVS": {"N": ["12"]},
    "TANTALEVS": {"A": ["21"]},
    "BIONEVS": {"A": ["21"]},
    "CYDNVS": {"N": ["12"]},
    "LARINA": {"N": ["11"]},
    "CAMARINA": {"N": ["11"]},
    "SEBOSIANVS": {"A": ["21"]},
    "NERTHVS": {"N": ["12"]},
    "CALVINIVS": {"N": ["12"]},
    "CLEOPATRA": {"N": ["11"]},
    "PISANVS": {"N": ["12"], "A": ["21"]},
    "RVLLVS": {"N": ["12"]},
    "SILARVS": {"N": ["12"]},
    "BVTA": {"N": ["11"]},
    "BAETICA": {"N": ["11"]},
    "FORTVNATVS": {"-": ["21"], "N": ["12"]},
    "HYPSIPYLAEVS": {"A": ["21"]},
    "ATHENAEVS": {"A": ["21"]},
    "FADIVS": {"N": ["12"]},
    "CLAVDIALIS": {"A": ["24"]},
    "MEGABOCCVS": {"N": ["12"]},
    "CINCIVS": {"A": ["21"]},
    "ATELLA": {"N": ["11"]},
    "PERGAMVM": {"N": ["12"]},
    "ANXVRVS": {"A": ["21"]},
    "MASTANESOSVS": {"N": ["12"]},
    "SVETONIVS": {"N": ["12"]},
    "CINARA": {"N": ["11"]},
    "CATONIANVS": {"A": ["21"]},
    "ITHACA": {"N": ["11"]},
    "ZACYNTHVS": {"N": ["12"]},
    "TILLIVS": {"N": ["12"]},
    "STICHVS": {"N": ["12"]},
    "NEMETOCENNA": {"N": ["11"]},
    "ALEXANDRINVS": {"A": ["21"]},
    "GRAECINA": {"N": ["11"]},
    "HERODVS": {"N": ["12"]},
    "CAMVRIVS": {"N": ["12"]},
    "MARTIGENA": {"N": ["11"]},
    "HECTOREVS": {"A": ["21"]},
    "LVGDVNENSIS": {"A": ["24"]},
    "CORELLIA": {"N": ["11"]},
    "TVRANNIVS": {"N": ["12"]},
    "HERMIPPVS": {"N": ["12"]},
    "LILYBAEA": {"N": ["11"]},
    "ENTELLVS": {"N": ["12"]},
    "PHOLVS": {"N": ["12"]},
    "TROILVS": {"N": ["12"]},
    "VENETVS": {"A": ["21"], "N": ["12"]},
    "PHTHIVS": {"A": ["21"]},
    "OPPIDIVS": {"N": ["12"]},
    "DACICVS": {"N": ["12"], "A": ["21"]},
    "BALEARICVS": {"N": ["12"]},
    "HELVIDIVS": {"A": ["21"], "N": ["12"]},
    "SILVS": {"N": ["12"]},
    "NISEVS": {"A": ["21"]},
    "THESSALIS": {"A": ["24"]},
    "ACRAGANTINVS": {"A": ["21"]},
    "IANALIS": {"A": ["24"]},
    "LVCTERIVS": {"N": ["12"]},
    "LITERNINVS": {"A": ["21"]},
    "PHERONACTVS": {"N": ["12"]},
    "IVNONICOLA": {"N": ["11"]},
    "MEDIA": {"N": ["11"]},
    "PARAETONICVS": {"A": ["21"]},
    "MINIVS": {"N": ["12"]},
    "SALONINVS": {"N": ["12"]},
    "CARPATHIVS": {"A": ["21"]},
    "CAENINA": {"N": ["11"]},
    "THYONIANVS": {"A": ["21"]},
    "LENNIVS": {"N": ["12"]},
    "CAECILIANVS": {"N": ["12"], "A": ["21"]},
    "BEDRIACVS": {"A": ["21"]},
    "PANDATERIA": {"N": ["11"]},
    "TARCONDARIVS": {"N": ["12"]},
    "CANOPIVS": {"A": ["21"]},
    "SALERNVM": {"N": ["12"]},
    "BATAVVS": {"A": ["21"]},
    "CVRTILLVS": {"N": ["12"]},
    "QVARTILLA": {"N": ["11"]},
    "SVEBVS": {"A": ["21"]},
    "PACENVS": {"N": ["12"]},
    "IAPETVS": {"N": ["12"]},
    "NASIDIENVS": {"N": ["12"]},
    "FAVSTA": {"N": ["11"]},
    "MOESIA": {"N": ["11"]},
    "HARPAGVS": {"N": ["12"]},
    "FAESIDIVS": {"N": ["12"]},
    "HYBLA": {"N": ["11"]},
    "MESSALLA": {"N": ["11"]},
    "MESSIVS": {"N": ["12"]},
    "ILISSVS": {"N": ["12"]},
    "AMBIVIVS": {"N": ["12"]},
    "NAVPACTOVS": {"A": ["21"]},
    "LAECA": {"N": ["11"]},
    "TYPHOEVS": {"A": ["21"], "N": ["12"]},
    "IPHITVS": {"N": ["12"]},
    "ARGYNNVS": {"N": ["12"]},
    "BAIVS": {"A": ["21"], "N": ["12"]},
    "HYPSAEVS": {"N": ["12"]},
    "HARPALVS": {"N": ["12"]},
    "DECIANVS": {"N": ["12"]},
    "ANDROGEVS": {"N": ["12"]},
    "LYCOMEDIVS": {"N": ["12"]},
    "PHOCVS": {"N": ["12"]},
    "BETACEVS": {"1": ["12"]},
    "SATRACHVS": {"N": ["12"]},
    "TVRIENSIS": {"A": ["24"]},
    "CROTONIENSIS": {"A": ["24"]},
    "PHIDIPPVS": {"N": ["12"]},
    "MAELIVS": {"N": ["12"]},
    "OEBALVS": {"N": ["12"]},
    "PAVLLVS": {"N": ["12"]},
    "LAVREOLVS": {"N": ["12"]},
    "TAENARIVS": {"A": ["21"]},
    "LICINVS": {"N": ["12"]},
    "OENOTRIVS": {"A": ["21"]},
    "ANICIVS": {"N": ["12"]},
    "LVSTRICIVS": {"N": ["12"]},
    "VEIA": {"N": ["11"]},
    "VACCENSIS": {"A": ["24"]},
    "DINIA": {"N": ["11"]},
    "METHYMNAEVS": {"A": ["21"]},
    "PAGYDA": {"N": ["11"]},
    "LAMVS": {"N": ["12"]},
    "PAGASA": {"N": ["11"]},
    "VOLAGINIVS": {"N": ["12"]},
    "REATINVS": {"A": ["21"]},
    "MELISSA": {"N": ["11"]},
    "FLEVVM": {"N": ["12"]},
    "DECIBALVS": {"N": ["12"]},
    "SINOPA": {"N": ["11"]},
    "GORDYAEVS": {"A": ["21"]},
    "RABOCENTVS": {"N": ["12"]},
    "LIBYA": {"N": ["11"]},
    "CORDOBA": {"N": ["11"]},
    "ANTEIVS": {"N": ["12"]},
    "PARIVM": {"N": ["12"]},
    "SALVIANVS": {"N": ["12"]},
    "NEAPOLITANVS": {"A": ["21"]},
    "TRITONIS": {"A": ["24"], "N": ["13"]},
    "CRINISVS": {"N": ["12"]},
    "MACCVS": {"N": ["12"]},
    "CLAVDIA": {"N": ["11"]},
    "SCYLACEVM": {"N": ["12"]},
    "THEMISCYRA": {"N": ["11"]},
    "AGERMVS": {"N": ["12"]},
    "ACYLLA": {"N": ["11"]},
    "OECHALIA": {"N": ["11"]},
    "VIRGINIA": {"N": ["11"]},
    "TITIDIVS": {"N": ["12"]},
    "ILIONA": {"N": ["11"]},
    "SOPHRONISCVS": {"N": ["12"]},
    "NEAERA": {"N": ["11"]},
    "TRANSPADANVS": {"A": ["21"]},
    "HIMELLA": {"N": ["11"]},
    "PARRHASIS": {"A": ["24"]},
    "APSVS": {"N": ["12"]},
    "CAPRICORNVS": {"N": ["12"], "-": ["12"]},
    "PVLCHRA": {"N": ["11"]},
    "PACHYNVS": {"N": ["12"]},
    "COSYRA": {"N": ["11"]},
    "LVGDVNVM": {"N": ["12"]},
    "ACADEMICA": {"N": ["11"]},
    "ZOILVS": {"N": ["12"]},
    "ORATA": {"N": ["11"]},
    "PATROBIVS": {"N": ["12"]},
    "MARSICVS": {"A": ["21"]},
    "ONAGVS": {"N": ["12"]},
    "SIMONIDEVS": {"A": ["21"]},
    "CADIVS": {"N": ["12"]},
    "CLVVIVS": {"N": ["12"]},
    "TRASIMENVS": {"N": ["12"], "A": ["21"]},
    "SVLGIVS": {"N": ["12"]},
    "THEOPHRASTVS": {"N": ["12"]},
    "PESSINVNTIVS": {"A": ["21"]},
    "CHALYBEIVS": {"A": ["21"]},
    "PSEVDOPHILIPPVS": {"N": ["12"]},
    "HIRTIVS": {"A": ["21"], "N": ["12"]},
    "CORFIDIVS": {"N": ["12"]},
    "ARRIA": {"N": ["11"]},
    "MECVLONIVS": {"N": ["12"]},
    "MARSAEVS": {"N": ["12"]},
    "FALANIVS": {"N": ["12"]},
    "VICTIVS": {"N": ["12"]},
    "ORTINVS": {"A": ["21"]},
    "LOLLIANVS": {"A": ["21"]},
    "PANNONICVS": {"A": ["21"]},
    "THRAESSA": {"N": ["11"]},
    "CONTEREBROMNIVS": {"A": ["21"]},
    "MYSIA": {"N": ["11"]},
    "OLYNTHIVS": {"A": ["21"]},
    "DARDANIVS": {"A": ["21"]},
    "MONOBAZVS": {"N": ["12"]},
    "EVTHYNICVS": {"N": ["12"]},
    "ARATVS": {"N": ["12"]},
    "CINNANVS": {"A": ["21"]},
    "EVANDRIVS": {"A": ["21"]},
    "VORTVMNVS": {"N": ["12"]},
    "GRAECANICVS": {"A": ["21"]},
    "CYRENAEVS": {"A": ["21"]},
    "MELIBOEVS": {"A": ["21"], "N": ["12"]},
    "FVNDANIVS": {"N": ["12"]},
    "CATIVS": {"N": ["12"]},
    "VOLVSVS": {"N": ["12"]},
    "TELEVS": {"N": ["12"]},
    "PRISCA": {"N": ["11"]},
    "METABVS": {"N": ["12"]},
    "LIGVRINVS": {"N": ["12"]},
    "VERVLAMIVM": {"N": ["12"]},
    "OGVLNIA": {"N": ["11"]},
    "VINIVS": {"N": ["12"]},
    "SIGNINVS": {"A": ["21"]},
    "BACTRA": {"N": ["11"]},
    "GONOESSA": {"N": ["11"]},
    "LEGERDA": {"N": ["11"]},
    "NEMEAEVS": {"A": ["21"]},
    "NYCTEVS": {"N": ["12"]},
    "MAGNESIVS": {"A": ["21"]},
    "RVSPINA": {"N": ["11"]},
    "LIBYCVS": {"A": ["21"]},
    "PHARNABAZVS": {"N": ["12"]},
    "AVRVNCVLEIVS": {"N": ["12"]},
    "BERECYNTIA": {"N": ["11"]},
    "PHERAEVS": {"A": ["21"]},
    "HELYMVS": {"N": ["12"]},
    "AMBRACIVS": {"A": ["21"]},
    "TIBERIOLVS": {"N": ["12"]},
    "METISCVS": {"N": ["12"]},
    "PHILADELPHVS": {"N": ["12"]},
    "PRIVERNVS": {"N": ["12"]},
    "AESONIVS": {"A": ["21"]},
    "CLYTAEMESTRA": {"N": ["11"]},
    "DINDYMENA": {"N": ["11"]},
    "ACCIVS": {"N": ["12"]},
    "DANAVS": {"A": ["21"], "N": ["12"]},
    "BOIOCALVS": {"N": ["12"]},
    "AQVILIA": {"N": ["11"]},
    "CASINVM": {"N": ["12"]},
    "VICESIMARIVS": {"1": ["12"]},
    "ALFIVS": {"N": ["12"]},
    "ARIADNAEVS": {"A": ["21"]},
    "VELLAVNODVNVM": {"N": ["12"]},
    "SPOLETINVS": {"A": ["21"]},
    "POLYHYMNIA": {"N": ["11"]},
    "POMEIA": {"N": ["11"]},
    "CAMERINVM": {"N": ["12"]},
    "NVCERINVS": {"N": ["12"], "A": ["21"]},
    "LAPITHA": {"N": ["11"]},
    "PAVLINVS": {"N": ["12"]},
    "TINGITANVS": {"A": ["21"]},
    "THABENA": {"N": ["11"]},
    "BLVCIVM": {"N": ["12"]},
    "HERMARCHVS": {"N": ["12"]},
    "AESCHYLEVS": {"A": ["21"]},
    "PELETHRONIVM": {"N": ["12"]},
    "CORREVS": {"N": ["12"]},
    "BADVHENNA": {"N": ["11"]},
    "COEVS": {"N": ["12"]},
    "CALLISTRATVS": {"N": ["12"]},
    "CAENEVS": {"N": ["12"]},
    "LEDAEVS": {"A": ["21"]},
    "APICATA": {"N": ["11"]},
    "CERCINA": {"N": ["11"]},
    "QVINTILIVS": {"N": ["12"]},
    "CARYSTIVS": {"A": ["21"]},
    "MALTINVS": {"N": ["12"]},
    "MAZIPPA": {"N": ["11"]},
    "DARDANIS": {"A": ["24"]},
    "BITHYNICVS": {"A": ["21"], "N": ["12"]},
    "GARGARA": {"N": ["11"]},
    "ELIDENSIS": {"A": ["24"]},
    "STRATONICA": {"N": ["11"]},
    "HELVETICVS": {"A": ["21"]},
    "IVNONIVS": {"A": ["21"]},
    "LAVRENTIVS": {"A": ["21"]},
    "CHALDAEVS": {"N": ["12"], "A": ["21"]},
    "CROESVS": {"N": ["12"]},
    "AQVILIANVS": {"A": ["21"]},
    "SCIRONIVS": {"A": ["21"]},
    "CARTIMANDVA": {"N": ["11"]},
    "ARCESILAVS": {"N": ["12"]},
    "VANNIVS": {"N": ["12"]},
    "PLINIVS": {"N": ["12"]},
    "ERGENNA": {"N": ["11"]},
    "NIOBEVS": {"A": ["21"]},
    "ARIMIENSIS": {"A": ["24"]},
    "SVTRIVM": {"N": ["12"]},
    "ARVPINVS": {"A": ["21"]},
    "MOCHVS": {"N": ["12"]},
    "CERBEREVS": {"A": ["21"]},
    "IBYCVS": {"N": ["12"]},
    "NABDALSA": {"N": ["11"]},
    "SORANVS": {"N": ["12"], "A": ["21"]},
    "AMPHOTERVS": {"N": ["12"]},
    "VENVSTINVS": {"A": ["21"]},
    "PALAEPHARSALVS": {"N": ["12"]},
    "QVADRIGARIVS": {"1": ["12"], "N": ["12"]},
    "VINNIVS": {"N": ["12"]},
    "MAGIA": {"N": ["11"]},
    "COMMIVS": {"N": ["12"]},
    "THERAEVS": {"A": ["21"]},
    "PERIBOMIVS": {"N": ["12"]},
    "HESVS": {"N": ["12"]},
    "CYTHEREA": {"N": ["11"]},
    "LARGA": {"N": ["11"]},
    "VRBICVS": {"N": ["12"]},
    "CALABRIA": {"N": ["11"]},
    "LAVSVS": {"N": ["12"]},
    "IVVENALIA": {"N": ["11"]},
    "HIPPOMENEIVS": {"A": ["21"]},
    "AMPHAEVS": {"N": ["12"]},
    "DAVNVS": {"N": ["12"]},
    "ASYLVS": {"N": ["12"]},
    "FAVENTINVS": {"N": ["12"]},
    "DVRIVS": {"N": ["12"]},
    "GESSIVS": {"N": ["12"]},
    "ILITHYIA": {"N": ["11"]},
    "PLAGVLEIVS": {"N": ["12"]},
    "OCELVM": {"N": ["12"]},
    "CEIVS": {"N": ["12"]},
    "SABELLICVS": {"A": ["21"]},
    "HECATAEVS": {"N": ["12"]},
    "GANGETIS": {"A": ["24"]},
    "RVFVLVS": {"-": ["21"], "N": ["12"]},
    "SEIVS": {"N": ["12"]},
    "SINVESSA": {"N": ["11"]},
    "HORMVS": {"N": ["12"]},
    "VERCASSIVELLAVNVS": {"N": ["12"]},
    "MAECIA": {"N": ["11"]},
    "EPIDAVRVS": {"N": ["12"]},
    "LVCILIANVS": {"A": ["21"]},
    "PHILIPPEVS": {"A": ["21"]},
    "CALATINVS": {"N": ["12"]},
    "TREVIRVS": {"A": ["21"]},
    "CHALINVS": {"N": ["12"]},
    "NICOMEDIA": {"N": ["11"]},
    "DACIA": {"N": ["11"]},
    "DAMASCVS": {"N": ["12"]},
    "ATACINVS": {"A": ["21"]},
    "LAERTIVS": {"A": ["21"], "N": ["12"]},
    "CEBALINVS": {"N": ["12"]},
    "ABVDIVS": {"N": ["12"]},
    "PLOTINA": {"N": ["11"]},
    "AQVINVS": {"N": ["12"]},
    "MVNIVS": {"N": ["12"]},
    "ANDRVS": {"N": ["12"]},
    "HEBRAEVS": {"A": ["21"]},
    "VENVSINVS": {"A": ["21"]},
    "CLEOBVLA": {"N": ["11"]},
    "PANATHENAICVS": {"N": ["12"]},
    "ARELLIVS": {"N": ["12"]},
    "CHIA": {"N": ["11"]},
    "FONTEIANVS": {"A": ["21"]},
    "QVIRINIVS": {"N": ["12"]},
    "TVTICANVS": {"N": ["12"]},
    "BABYLONIVS": {"A": ["21"]},
    "DIDIVS": {"N": ["12"], "A": ["21"]},
    "FABRICIVS": {"A": ["21"], "N": ["12"]},
    "MAEOTIS": {"A": ["24"], "N": ["16"]},
    "IDYIA": {"N": ["11"]},
    "OSTIENSIS": {"A": ["24"]},
    "QVINQVATRVS": {"N": ["12"]},
    "LACAENA": {"N": ["11"]},
    "APONIVS": {"N": ["12"]},
    "PRAECONINVS": {"N": ["12"]},
    "AVGAEVS": {"N": ["12"]},
    "MAEONIA": {"N": ["11"]},
    "LILYBEIVS": {"A": ["21"]},
    "AMISVS": {"N": ["12"]},
    "CASMILLA": {"N": ["11"]},
    "ASVVIVS": {"N": ["12"]},
    "SALARIVS": {"N": ["12"], "A": ["21"]},
    "ABYDENVS": {"A": ["21"]},
    "CAVDIVM": {"N": ["12"]},
    "HYMETTVS": {"N": ["12"]},
    "PARAPANISVS": {"N": ["12"]},
    "PORTICANVS": {"N": ["12"]},
    "DIANIVS": {"A": ["21"]},
    "ICCIVS": {"N": ["12"]},
    "MARICA": {"N": ["11"]},
    "LVSCIVS": {"N": ["12"]},
    "BROGITARVS": {"N": ["12"]},
    "TAMPIVS": {"N": ["12"]},
    "ADIABENVS": {"A": ["21"]},
    "CLVENTIVS": {"N": ["12"]},
    "PLAVTIVS": {"N": ["12"], "A": ["21"]},
    "TIFERNIVM": {"N": ["12"]},
    "BONONIA": {"N": ["11"]},
    "MAEONIVS": {"A": ["21"]},
    "NERIVS": {"N": ["12"]},
    "CHAERIPPVS": {"N": ["12"]},
    "ABDERA": {"N": ["11"]},
    "METTVS": {"N": ["12"]},
    "ZALEVCVS": {"N": ["12"]},
    "ELEGIA": {"N": ["11"]},
    "TYNDAREVS": {"N": ["12"]},
    "COMETA": {"N": ["11"]},
    "SEDVLLVS": {"N": ["12"]},
    "TANTALVS": {"N": ["12"]},
    "SCYLLAEVS": {"A": ["21"]},
    "BITHYNIVS": {"A": ["21"]},
    "VELEDA": {"N": ["11"]},
    "HELIODORVS": {"N": ["12"]},
    "SCAEVA": {"2": ["11"], "N": ["12"]},
    "ORICVM": {"N": ["12"]},
    "PANAETIVS": {"N": ["12"]},
    "HISPVLLA": {"N": ["11"]},
    "TMARVS": {"N": ["12"]},
    "THERAPNAEVS": {"A": ["21"]},
    "CLVSIVM": {"N": ["12"]},
    "APPVLEIVS": {"A": ["21"], "N": ["12"]},
    "GRATIANVS": {"N": ["12"]},
    "ERYTHEIS": {"A": ["24"]},
    "TVDICIVS": {"N": ["12"]},
    "EPHYREIACVS": {"A": ["21"]},
    "CASTRICIANVS": {"A": ["21"]},
    "ATALANTA": {"N": ["11"]},
    "BOSPORANVS": {"A": ["21"]},
    "TELESPHORVS": {"N": ["12"]},
    "PLANESIVM": {"N": ["12"]},
    "AMISSVS": {"N": ["12"]},
    "CRITOGNATVS": {"N": ["12"]},
    "PRVSA": {"N": ["11"]},
    "LITERNVM": {"N": ["12"]},
    "OSSA": {"N": ["11"]},
    "LETOVS": {"A": ["21"]},
    "TARENTINVS": {"A": ["21"]},
    "IVGVRTHA": {"N": ["11"]},
    "CYNOSVRA": {"N": ["11"]},
    "NASIDIANVS": {"A": ["21"]},
    "FENESTELLA": {"N": ["11"]},
    "CREMERENSIS": {"A": ["24"]},
    "CASCA": {"N": ["11"]},
    "CLYMENVS": {"N": ["12"]},
    "STAIVS": {"N": ["12"]},
    "GIGANTEVS": {"A": ["21"]},
    "DIPHILVS": {"N": ["12"]},
    "PYTHIVS": {"N": ["12"], "A": ["21"]},
    "GORGONEVS": {"A": ["21"]},
    "BESTIVS": {"N": ["12"]},
    "MALCHVS": {"N": ["12"]},
    "HIERONYMVS": {"N": ["12"]},
    "PYTAGOREVS": {"A": ["21"]},
    "LONDINIVM": {"N": ["12"]},
    "VINALIA": {"N": ["11"]},
    "ALFENVS": {"N": ["12"]},
    "TRYPHERVS": {"N": ["12"]},
    "MEGARVS": {"A": ["21"]},
    "PARTHENOPAEVS": {"N": ["12"]},
    "LECHAEVM": {"N": ["12"]},
    "HERIVS": {"N": ["12"]},
    "OCNVS": {"N": ["12"]},
    "NEREIVS": {"A": ["21"]},
    "SARNVS": {"N": ["12"]},
    "BAZAIRA": {"N": ["11"]},
    "THASVS": {"N": ["12"]},
    "ROMILIVS": {"N": ["12"]},
    "FORMIANVM": {"N": ["12"]},
    "TRISANTONA": {"N": ["11"]},
    "EPIMACHVS": {"N": ["12"]},
    "CIRTA": {"N": ["11"]},
    "ANDROMACHA": {"N": ["11"]},
    "CYRENENSIS": {"A": ["24"]},
    "VERANIVS": {"N": ["12"]},
    "LARINVM": {"N": ["12"]},
    "GANNASCVS": {"N": ["12"]},
    "ABNOBA": {"N": ["11"]},
    "SERGESTVS": {"N": ["12"]},
    "EPPONINA": {"N": ["11"]},
    "EVRYSTHEVS": {"N": ["12"]},
    "CYTHERIACVS": {"A": ["21"]},
    "CALLIOPEA": {"N": ["11"]},
    "MARVS": {"N": ["12"]},
    "MARRVCINVS": {"A": ["21"]},
    "MYTILENENSIS": {"A": ["24"]},
    "ARISTIPPVS": {"N": ["12"]},
    "NOREIA": {"N": ["11"]},
    "TRANSRHENANVS": {"A": ["21"]},
    "STATILIA": {"N": ["11"]},
    "OCELLA": {"N": ["11"]},
    "SARDONIVS": {"A": ["21"]},
    "TITARESVS": {"N": ["12"]},
    "FAENIVS": {"N": ["12"]},
    "OENOTRVS": {"A": ["21"]},
    "COSSINIVS": {"N": ["12"]},
    "NEMORENSIS": {"A": ["24"]},
    "VIRBIVS": {"N": ["12"]},
    "PANSA": {"N": ["11"]},
    "FORTVNATA": {"N": ["11"]},
    "SISYPHIVS": {"A": ["21"]},
    "HATERIVS": {"N": ["12"]},
    "VMBRIA": {"N": ["11"]},
    "GAVDA": {"N": ["11"]},
    "PROTESILAEVS": {"A": ["21"]},
    "CYTHEREIVS": {"A": ["21"]},
    "LAECANIVS": {"N": ["12"]},
    "RECENTORICVS": {"A": ["21"]},
    "CONSVS": {"N": ["12"]},
    "ARGANTHONIVS": {"N": ["12"]},
    "CAVILLONVM": {"N": ["12"]},
    "LESBIVS": {"N": ["12"], "A": ["21"]},
    "CLEONAEVS": {"A": ["21"]},
    "DAVVS": {"N": ["12"]},
    "HASTENSIS": {"A": ["24"]},
    "COMPSA": {"N": ["11"]},
    "PYLVS": {"N": ["12"]},
    "DARDANIA": {"N": ["11"]},
    "THERMODONTIVS": {"A": ["21"]},
    "GALLINARIVS": {"A": ["21"]},
    "CAESIANVS": {"N": ["12"]},
    "ARISTOBVLVS": {"N": ["12"]},
    "ADRASTVS": {"N": ["12"]},
    "GALATIA": {"N": ["11"]},
    "CLOTA": {"N": ["11"]},
    "QVINTIANVS": {"A": ["21"], "N": ["12"]},
    "VESTRIVS": {"N": ["12"]},
    "PAVSIACVS": {"A": ["21"]},
    "ARABVS": {"A": ["21"], "N": ["12"]},
    "APRILIS": {"N": ["13"], "A": ["12"]},
    "TVSCVLVS": {"A": ["21"], "N": ["12"]},
    "PATISIVS": {"N": ["12"]},
    "CAMILLVS": {"N": ["12"]},
    "GRAVPIVS": {"N": ["12"]},
    "ACERRONIA": {"N": ["11"]},
    "SOSPITA": {"N": ["11"]},
    "LESBIS": {"A": ["24"]},
    "EPIDAMNVS": {"N": ["12"]},
    "THESSANDRVS": {"N": ["12"]},
    "TORONAEVS": {"A": ["21"]},
    "BELVS": {"N": ["12"]},
    "TELEGONVS": {"N": ["12"]},
    "TEMNVS": {"N": ["12"]},
    "BAGRADA": {"N": ["11"]},
    "TELAMONIVS": {"N": ["12"]},
    "STABERIVS": {"N": ["12"]},
    "CYTORIVS": {"A": ["21"]},
    "MINOTAVRVS": {"N": ["12"]},
    "ANGITIA": {"N": ["11"]},
    "CRETANVS": {"N": ["12"]},
    "MAEOTICVS": {"A": ["21"]},
    "ANAPVS": {"N": ["12"]},
    "ITYRAEVS": {"A": ["21"]},
    "TARRACINIENSIS": {"A": ["24"]},
    "SAMAXVS": {"N": ["12"]},
    "LEMANNVS": {"N": ["12"]},
    "EVODVS": {"N": ["12"]},
    "DONVSA": {"N": ["11"]},
    "NILIACVS": {"A": ["21"]},
    "DVRATIVS": {"N": ["12"]},
    "FLORALIS": {"A": ["24"]},
    "ZONARIVS": {"1": ["12"]},
    "EPRIVS": {"N": ["12"]},
    "FABVLLA": {"N": ["11"]},
    "COTTIANVS": {"A": ["21"]},
    "NIREVS": {"N": ["12"]},
    "ARDEATINVM": {"N": ["12"]},
    "SILA": {"N": ["11"]},
    "TYRRHENVS": {"N": ["12"], "A": ["21"]},
    "MENELAVS": {"N": ["12"]},
    "MALIVS": {"A": ["21"]},
    "VERATIANVS": {"A": ["21"]},
    "GYTHEVM": {"N": ["12"]},
    "NEAPOLITANVM": {"N": ["12"]},
    "ZELA": {"N": ["11"]},
    "ATLANTEVS": {"A": ["21"]},
    "CARRVCA": {"N": ["11"]},
    "LYGDAMVS": {"N": ["12"]},
    "SAEVINVS": {"N": ["12"]},
    "AREOPAGVS": {"N": ["12"]},
    "DEMODOCVS": {"N": ["12"]},
    "NIGRINVS": {"N": ["12"]},
    "MAROBODVVS": {"N": ["12"]},
    "VELABRA": {"N": ["11"]},
    "ALTINVS": {"N": ["12"]},
    "HIPPONOVS": {"N": ["12"]},
    "PHILOPOLEMVS": {"N": ["12"]},
    "LAVINVS": {"A": ["21"]},
    "VENILIA": {"N": ["11"]},
    "VFENTINA": {"N": ["11"]},
    "LICYMNIA": {"N": ["11"]},
    "MARONEVS": {"A": ["21"]},
    "SIDICINVS": {"A": ["21"]},
    "PRIMIGENIVS": {"N": ["12"]},
    "AMATHVSIVS": {"A": ["21"]},
    "BASSARICVS": {"A": ["21"]},
    "LEVCADIVS": {"A": ["21"]},
    "LYCTIVS": {"A": ["21"]},
    "LYNCEVS": {"N": ["12"], "A": ["21"]},
    "TORQVATA": {"N": ["11"]},
    "ELATEA": {"N": ["11"]},
    "VENTIDIVS": {"N": ["12"]},
    "THRIASIVS": {"A": ["21"]},
    "NICEPHORIVM": {"N": ["12"]},
    "ERGASILVS": {"N": ["12"]},
    "SAPPHICVS": {"A": ["21"]},
    "GALLICANVS": {"A": ["21"]},
    "PHOENISSVS": {"A": ["21"]},
    "THIA": {"N": ["11"]},
    "ASTYPALAEA": {"N": ["11"]},
    "VENAFRVM": {"N": ["12"]},
    "DAVCIVS": {"A": ["21"]},
    "PASSIENVS": {"N": ["12"]},
    "CABERA": {"N": ["11"]},
    "TROVCILLVS": {"N": ["12"]},
    "IAPYDIA": {"N": ["11"]},
    "DRVSIANVS": {"A": ["21"]},
    "STYMPHALIVS": {"A": ["21"]},
    "SERAPA": {"N": ["11"]},
    "CREMONA": {"N": ["11"]},
    "TONGILIVS": {"N": ["12"]},
    "PACARIVS": {"N": ["12"]},
    "LYCINNA": {"N": ["11"]},
    "SATTIA": {"N": ["11"]},
    "TELESINVS": {"N": ["12"]},
    "SACROVIRIANVS": {"A": ["21"]},
    "CERAVNVS": {"A": ["21"]},
    "SCAEVINVS": {"N": ["12"]},
    "ADAMASTVS": {"N": ["12"]},
    "COCYTIVS": {"A": ["21"]},
    "CARFINIA": {"N": ["11"]},
    "PIPLEIVS": {"A": ["21"]},
    "HYMETTIVS": {"A": ["21"]},
    "BIBACVLVS": {"N": ["12"]},
    "SEQVANA": {"N": ["11"]},
    "SVBVRANVS": {"A": ["21"], "N": ["12"]},
    "EPPIA": {"N": ["11"]},
    "ERETVM": {"N": ["12"]},
    "SVEVVS": {"A": ["21"]},
    "PAELIGNVS": {"A": ["21"], "N": ["12"]},
    "SVFFICIVS": {"N": ["12"]},
    "CORYCIVS": {"A": ["21"]},
    "HYRTACVS": {"N": ["12"]},
    "LATONIVS": {"A": ["21"]},
    "CAMERIVM": {"N": ["12"]},
    "GORTYNIS": {"A": ["24"]},
    "SICYONIA": {"N": ["11"]},
    "LYCAONIA": {"N": ["11"]},
    "BAREA": {"N": ["11"]},
    "PAGASAEVS": {"A": ["21"]},
    "CHARIOVALDA": {"N": ["11"]},
    "MOLORCHVS": {"N": ["12"]},
    "VELINVS": {"A": ["21"], "N": ["12"]},
    "AVTOBVLVS": {"N": ["12"]},
    "PANDARVS": {"N": ["12"]},
    "CATHEGVS": {"N": ["12"]},
    "TERMESTINVS": {"A": ["21"]},
    "PYRRHVS": {"N": ["12"]},
    "NERITVS": {"N": ["12"]},
    "ILIACVS": {"A": ["21"]},
    "OPHIVCHVS": {"N": ["12"]},
    "ERYMANTHVS": {"N": ["12"]},
    "ARTEMONA": {"N": ["11"]},
    "ALPHESIBOEA": {"N": ["11"]},
    "PAMPHILA": {"N": ["11"]},
    "GALERIVS": {"N": ["12"]},
    "ARVLENVS": {"N": ["12"]},
    "TERTVLLINVS": {"N": ["12"]},
    "OLYMPISCVS": {"N": ["12"]},
    "HESIONA": {"N": ["11"]},
    "DAEDALEVS": {"A": ["21"]},
    "ARCHESTRATA": {"N": ["11"]},
    "GARVMNA": {"N": ["11"]},
    "MATINVS": {"N": ["12"], "A": ["21"]},
    "MNESTHEVS": {"N": ["12"]},
    "MATVTA": {"N": ["11"]},
    "PHTHIOTICVS": {"A": ["21"]},
    "CATIENA": {"N": ["11"]},
    "MIMNERMVS": {"N": ["12"]},
    "SVILLIVS": {"N": ["12"]},
    "COMMAGENVS": {"A": ["21"]},
    "AVENIENSIS": {"A": ["24"]},
    "ANTONIA": {"N": ["11"]},
    "ARPINVM": {"N": ["12"]},
    "PVNICANVS": {"A": ["21"]},
    "SVRRENTINVS": {"A": ["21"]},
    "TERENTIANVS": {"A": ["21"]},
    "ANCONITANVS": {"A": ["21"]},
    "VLIA": {"N": ["11"]},
    "THAPSACVS": {"N": ["12"]},
    "MAENIVS": {"A": ["21"], "N": ["12"]},
    "VACVNALIS": {"A": ["24"]},
    "AENEIVS": {"A": ["21"]},
    "EVRYLOCHVS": {"N": ["12"]},
    "ASSYRIA": {"N": ["11"]},
    "TMOLIVS": {"A": ["21"]},
    "TEGEAEVS": {"A": ["21"]},
    "VELABRVM": {"N": ["12"]},
    "RVFRIVS": {"N": ["12"]},
    "COERANVS": {"N": ["12"]},
    "CLEVSTRATA": {"N": ["11"]},
    "PHILOCTETA": {"N": ["11"]},
    "NABALIA": {"N": ["11"]},
    "CHRISTVS": {"N": ["12"]},
    "PILVMNVS": {"N": ["12"]},
    "SANQVINIVS": {"N": ["12"]},
    "ANDANIA": {"N": ["11"]},
    "HERBESVS": {"N": ["12"]},
    "CRISPINVS": {"N": ["12"]},
    "ARCTVS": {"N": ["12"]},
    "TYRTAEVS": {"N": ["12"]},
    "CARYSTVS": {"N": ["12"]},
    "VERVLANA": {"N": ["11"]},
    "MAVORTIVS": {"A": ["21"]},
    "LESBOVS": {"A": ["21"]},
    "ICARIA": {"N": ["11"]},
    "BOLA": {"N": ["11"]},
    "LVCANIA": {"N": ["11"]},
    "CANTIVM": {"N": ["12"]},
    "HIBERINA": {"N": ["11"]},
    "PERIDIA": {"N": ["11"]},
    "EPHORVS": {"-": ["12"], "N": ["12"]},
    "PROTHOVS": {"N": ["12"]},
    "THEONINVS": {"A": ["21"]},
    "CABVRVS": {"N": ["12"]},
    "DICAEARCHVS": {"N": ["12"]},
    "PETVSIVS": {"N": ["12"]},
    "NYCTELIVS": {"A": ["21"], "N": ["12"]},
    "ANTAEVS": {"N": ["12"]},
    "TEGEA": {"N": ["11"]},
    "NYMPHIDIVS": {"N": ["12"]},
    "ATTIANVS": {"A": ["21"], "N": ["12"]},
    "IAERA": {"N": ["11"]},
    "POSIDONIVS": {"N": ["12"]},
    "BISTONIS": {"A": ["24"]},
    "PALILIS": {"A": ["24"]},
    "THYNVS": {"A": ["21"]},
    "PHOCEVS": {"A": ["21"], "N": ["12"]},
    "DALMATICVS": {"A": ["21"]},
    "AVGVSTALIS": {"A": ["24"]},
    "ARENACVM": {"N": ["12"]},
    "ORCHOMENVS": {"N": ["12"]},
    "LEDA": {"N": ["11"]},
    "ARGVS": {"A": ["21"], "N": ["12"]},
    "NAXIVS": {"A": ["21"]},
    "CORNVTVS": {"N": ["12"]},
    "OLYMPIACVS": {"A": ["21"]},
    "AMPHILOCHIA": {"N": ["11"]},
    "SAETABVS": {"A": ["21"]},
    "GABINIANVS": {"A": ["21"], "N": ["12"]},
    "ARPINIVS": {"N": ["12"]},
    "SEXTILIVS": {"N": ["12"]},
    "REGINVS": {"N": ["12"], "A": ["21"]},
    "EVRYALVS": {"N": ["12"]},
    "PESSINVS": {"N": ["12"]},
    "SVEBIA": {"N": ["11"]},
    "DALMATIA": {"N": ["11"]},
    "PISISTRATVS": {"N": ["12"]},
    "PICENVM": {"N": ["12"]},
    "ATINA": {"N": ["11"]},
    "DINDYMVS": {"A": ["21"], "N": ["12"]},
    "PICENTIVS": {"A": ["21"]},
    "THESEVS": {"A": ["21"], "N": ["12"]},
    "NORENSIS": {"A": ["24"]},
    "DYMNVS": {"N": ["12"]},
    "BATAVODVRVM": {"N": ["12"]},
    "FALERNVM": {"N": ["12"]},
    "ICARIVS": {"A": ["21"], "N": ["12"]},
    "PETELIA": {"N": ["11"]},
    "CATVLLVS": {"N": ["12"]},
    "RVBRENVS": {"N": ["12"]},
    "SARMENTVS": {"N": ["12"]},
    "PROXVMVS": {"N": ["12"]},
    "VITIA": {"N": ["11"]},
    "CRESSIVS": {"A": ["21"]},
    "ACADEMVS": {"N": ["12"]},
    "NERONEVS": {"A": ["21"]},
    "PATAREVS": {"A": ["21"]},
    "NVMISIVS": {"N": ["12"]},
    "DVELLIVS": {"N": ["12"]},
    "OPTIMVS": {"N": ["12"]},
    "ZENOBIA": {"N": ["11"]},
    "PARILIA": {"N": ["11"]},
    "OPPIVS": {"N": ["12"], "A": ["21"]},
    "BVLLATIVS": {"N": ["12"]},
    "SILLIVS": {"N": ["12"]},
    "OLLIVS": {"N": ["12"]},
    "CANVTIVS": {"N": ["12"]},
    "MAVSOLEVS": {"A": ["21"]},
    "CLASSICIANVS": {"N": ["12"]},
    "LATINIENSIS": {"A": ["24"], "N": ["16"]},
    "FRONTINVS": {"N": ["12"]},
    "GAETVLVS": {"A": ["21"]},
    "ASPARAGIVM": {"N": ["12"]},
    "AMATHVSIA": {"N": ["11"]},
    "MVRCVS": {"N": ["12"]},
    "LYDIVS": {"A": ["21"]},
    "CVTIVS": {"N": ["12"]},
    "OLEARVS": {"N": ["12"]},
    "NEARCHVS": {"N": ["12"]},
    "NASAMONIVS": {"A": ["21"]},
    "TYPHOIS": {"A": ["24"]},
    "BISTONIVS": {"A": ["21"]},
    "PLOCAMVS": {"N": ["12"]},
    "CEPHALVS": {"N": ["12"]},
    "PHORCVS": {"N": ["12"]},
    "SCANTIANVS": {"A": ["21"]},
    "ANAXARCHVS": {"N": ["12"]},
    "LVCAGVS": {"N": ["12"]},
    "TIMONEVS": {"A": ["21"]},
    "POSTVMIANVS": {"A": ["21"]},
    "BOVIANVM": {"N": ["12"]},
    "ROVCILLVS": {"N": ["12"]},
    "FOLIA": {"N": ["11"]},
    "TVCCIVS": {"N": ["12"]},
    "CAECVBVS": {"A": ["21"]},
    "DANAEIVS": {"A": ["21"]},
    "CRASSICIVS": {"N": ["12"]},
    "APONIANA": {"N": ["11"]},
    "HERCVLEIVS": {"N": ["12"]},
    "BRENNVS": {"N": ["12"]},
    "FVRFANIVS": {"N": ["12"]},
    "ADONEVS": {"N": ["12"]},
    "BRVNDISINVS": {"A": ["21"]},
    "CENTONARIVS": {"1": ["12"]},
    "TEVTHRANTIVS": {"A": ["21"]},
    "MOPSOPIA": {"N": ["11"]},
    "PLATENIVS": {"A": ["21"]},
    "CALIGVLA": {"N": ["11"]},
    "ENIPEVS": {"N": ["12"]},
    "PERMESSVS": {"N": ["12"]},
    "THESSALA": {"N": ["11"]},
    "ACMONENSIS": {"A": ["24"]},
    "STHENEBOEA": {"N": ["11"]},
    "TIRESIA": {"N": ["11"]},
    "SCYTHICVS": {"A": ["21"]},
    "SCANTIA": {"N": ["11"]},
    "BOCVLVS": {"N": ["12"]},
    "DIOMEDEVS": {"A": ["21"]},
    "PETRINVM": {"N": ["12"]},
    "ATIVS": {"N": ["12"]},
    "ILVA": {"N": ["11"]},
    "DOSSENVS": {"N": ["12"]},
    "SQVILLVS": {"N": ["12"]},
    "TARSVS": {"N": ["12"]},
    "CVRETIS": {"A": ["24"]},
    "GORTYNIVS": {"A": ["21"]},
    "ACCA": {"N": ["11"]},
    "CECROPIS": {"A": ["24"]},
    "BELLIENVS": {"N": ["12"]},
    "TECMESSA": {"N": ["11"]},
    "PHERECLEVS": {"A": ["21"]},
    "SAMOTHRACIA": {"N": ["11"]},
    "MAENALIS": {"A": ["24"]},
    "PAPHIVS": {"A": ["21"]},
    "LVCETIVS": {"N": ["12"]},
    "DEXITHEA": {"N": ["11"]},
    "MEMPHITIS": {"A": ["24"]},
    "PHARIVS": {"N": ["12"], "A": ["21"]},
    "ISTHMIVS": {"A": ["21"]},
    "SVBRIVS": {"N": ["12"]},
    "MEGALESIACVS": {"A": ["21"]},
    "SALIENVS": {"N": ["12"]},
    "ALBESIVS": {"N": ["12"]},
    "XENOPHILVS": {"N": ["12"]},
    "DAEDALA": {"N": ["11"]},
    "ALBIVS": {"N": ["12"]},
    "ACVTIVS": {"N": ["12"]},
    "MANNICIVS": {"N": ["12"]},
    "MAECILIA": {"N": ["11"]},
    "BYZANTIVS": {"A": ["21"]},
    "VAGELLIVS": {"N": ["12"]},
    "ALLOBROGICVS": {"N": ["12"]},
    "CORNICVLANVS": {"A": ["21"]},
    "ARISBA": {"N": ["11"]},
    "PROCILLA": {"N": ["11"]},
    "TYDEVS": {"N": ["12"]},
    "BATTVS": {"N": ["12"]},
    "MEDVS": {"N": ["12"], "A": ["21"]},
    "LEONIDA": {"N": ["11"]},
    "MARDVS": {"A": ["21"]},
    "AEQVICVLVS": {"A": ["21"]},
    "PEDANIVS": {"N": ["12"]},
    "AEGIMVRVS": {"N": ["12"]},
    "ACTIVM": {"N": ["12"]},
    "SERGIANVS": {"A": ["21"]},
    "CRETAEVS": {"A": ["21"]},
    "PADVSA": {"N": ["11"]},
    "GORGOBINA": {"N": ["11"]},
    "ENNIA": {"N": ["11"]},
    "TEMESAEVS": {"A": ["21"]},
    "CASSIVELLAVNVS": {"N": ["12"]},
    "PVPINIA": {"N": ["11"]},
    "NARBONENSIS": {"A": ["24"]},
    "APPENNINVS": {"N": ["12"]},
    "VIRDOMARVS": {"N": ["12"]},
    "CONGVS": {"N": ["12"]},
    "CAESELLIVS": {"N": ["12"]},
    "OSINIVS": {"N": ["12"]},
    "SCRIBONIANVS": {"N": ["12"]},
    "DIONYSVS": {"N": ["12"]},
    "RVMINVS": {"A": ["21"]},
    "THEMISTA": {"N": ["11"]},
    "HYMENAEVS": {"-": ["12"], "N": ["12"]},
    "NASIDIVS": {"N": ["12"]},
    "CAICVS": {"N": ["12"]},
    "DVLICHIA": {"N": ["11"]},
    "ECBOLIMA": {"N": ["11"]},
    "AEGEVS": {"N": ["12"]},
    "THEODAMANTEVS": {"A": ["21"]},
    "OPITERGIVM": {"N": ["12"]},
    "VESPASIANVS": {"N": ["12"]},
    "AVIENVS": {"N": ["12"]},
    "PISTORIENSIS": {"A": ["24"]},
    "VALENTINVS": {"N": ["12"]},
    "THRAECIVS": {"A": ["21"]},
    "IONIS": {"A": ["24"]},
    "FADVS": {"N": ["12"]},
    "VERVLANVS": {"N": ["12"]},
    "VESVVIVS": {"N": ["12"]},
    "LABERIANVS": {"A": ["21"]},
    "CLEOMBROTVS": {"N": ["12"]},
    "TVLLINVS": {"N": ["12"]},
    "ALINDIENSIS": {"A": ["24"]},
    "DANVVIVS": {"N": ["12"]},
    "CASTICVS": {"N": ["12"]},
    "AVRVNCVLEIA": {"N": ["11"]},
    "ADRANA": {"N": ["11"]},
    "CALPVRNIANVS": {"N": ["12"]},
    "CERVARIVS": {"N": ["12"]},
    "ZETA": {"N": ["11"]},
    "GALBIANVS": {"A": ["21"]},
    "ITHACVS": {"A": ["21"], "N": ["12"]},
    "ARPINVS": {"A": ["21"]},
    "ACHAEMENIVS": {"A": ["21"]},
    "PANGAEVS": {"N": ["12"]},
    "AEGIENSIS": {"A": ["24"]},
    "CLYTIVS": {"N": ["12"]},
    "THREISSA": {"N": ["11"]},
    "NORICA": {"N": ["11"]},
    "COVS": {"A": ["21"], "N": ["12"]},
    "FALISCVS": {"A": ["21"]},
    "HYPSAEA": {"N": ["11"]},
    "ALLIARIA": {"N": ["11"]},
    "ANICETVS": {"N": ["12"]},
    "MORITASGVS": {"N": ["12"]},
    "FONTEIVS": {"A": ["21"], "N": ["12"]},
    "OLELVS": {"N": ["12"]},
    "GRATILLA": {"N": ["11"]},
    "COROEBVS": {"N": ["12"]},
    "CANTHARA": {"N": ["11"]},
    "HADRIVS": {"A": ["21"]},
    "PORRIMA": {"N": ["11"]},
    "EREBEVS": {"A": ["21"]},
    "PARNETHVS": {"N": ["12"]},
    "NICOMEDENSIS": {"A": ["24"]},
    "LEONNATVS": {"N": ["12"]},
    "PSEVDOLVS": {"N": ["12"]},
    "COPTVS": {"N": ["12"]},
    "OFELLVS": {"N": ["12"]},
    "PARENTALIS": {"A": ["24"]},
    "HIPPONENSIS": {"A": ["24"]},
    "BELLOVACVS": {"A": ["21"]},
    "DREPANVM": {"N": ["12"]},
    "ACRISIONEVS": {"A": ["21"]},
    "RVTVBA": {"N": ["11"]},
    "LIBITINA": {"N": ["11"]},
    "LYSIDICVS": {"N": ["12"]},
    "ASCONIVS": {"N": ["12"]},
    "PHILOMELA": {"N": ["11"]},
    "VESCVLARIVS": {"N": ["12"]},
    "SOLYMVS": {"A": ["21"]},
    "TARQVITVS": {"N": ["12"]},
    "AMPHIONIVS": {"A": ["21"]},
    "MARGIANA": {"N": ["11"]},
    "DVMNACVS": {"N": ["12"]},
    "PYGMAEVS": {"A": ["21"]},
    "SOZA": {"N": ["11"]},
    "CAESENNIVS": {"N": ["12"]},
    "MOSELLA": {"N": ["11"]},
    "PONTIVS": {"N": ["12"]},
    "CISSEVS": {"N": ["12"]},
    "PHOEBEVS": {"A": ["21"]},
    "SATICVLVS": {"N": ["12"]},
    "APENNINVS": {"N": ["12"]},
    "VELLEIVS": {"N": ["12"]},
    "ANEMVRIENSIS": {"A": ["24"]},
    "NEPTVNIVS": {"A": ["21"]},
    "EGNATIA": {"N": ["11"]},
    "SCISSA": {"N": ["11"]},
    "TEVTONICVS": {"A": ["21"]},
    "ISMARA": {"N": ["11"]},
    "AMVLLIVS": {"N": ["12"]},
    "LVCERIA": {"N": ["11"]},
    "SELINVS": {"N": ["12"]},
    "LIBERALIA": {"N": ["11"]},
    "MALLIVS": {"N": ["12"]},
    "TROXOBORVS": {"N": ["12"]},
    "CORFINIENSIS": {"A": ["24"]},
    "TMARIVS": {"A": ["21"]},
    "HIRRVS": {"N": ["12"]},
    "MINTVRNENSIS": {"A": ["24"]},
    "FABATVS": {"N": ["12"]},
    "FLACCILLA": {"N": ["11"]},
    "PANTHVS": {"N": ["12"]},
    "BACCHEIVS": {"A": ["21"]},
    "ANDRIA": {"N": ["11"]},
    "LAMYRVS": {"N": ["12"]},
    "AVARICENSIS": {"A": ["24"]},
    "AERIA": {"N": ["11"]},
    "IDRVS": {"N": ["12"]},
    "MILICHVS": {"N": ["12"]},
    "DIVODVRVM": {"N": ["12"]},
    "CAPHARIS": {"A": ["24"]},
    "CORYBANTIVS": {"A": ["21"]},
    "FIMBRIANVS": {"A": ["21"]},
    "LEMNIVS": {"A": ["21"]},
    "SELEVCVS": {"N": ["12"]},
    "ARGOLICVS": {"A": ["21"], "N": ["12"]},
    "ALBINA": {"N": ["11"]},
    "POLYPLVSIVS": {"A": ["21"]},
    "AMATHVS": {"N": ["12"]},
    "SPEVSIPPVS": {"N": ["12"]},
    "PIERIVS": {"A": ["21"]},
    "CORVNCANIVS": {"N": ["12"]},
    "QVINTILIANVS": {"N": ["12"]},
    "CLVVIDIENVS": {"N": ["12"]},
    "LEVCIPPVS": {"N": ["12"]},
    "POMPEIANVS": {"N": ["12"], "A": ["21"]},
    "AGRIVS": {"N": ["12"]},
    "FVLVIA": {"N": ["11"]},
    "DORIVS": {"A": ["21"]},
    "THRASYLLVS": {"N": ["12"]},
    "DEIANIRA": {"N": ["11"]},
    "SABRINA": {"N": ["11"]},
    "BAIANVS": {"A": ["21"]},
    "ARTORIVS": {"N": ["12"]},
    "TRALLIANVS": {"A": ["21"]},
    "MAVRICVS": {"N": ["12"]},
    "NERATIVS": {"N": ["12"]},
    "BERYTVS": {"N": ["12"]},
    "POPVLIANA": {"N": ["11"]},
    "TORANIVS": {"N": ["12"]},
    "CAMPANICVS": {"A": ["21"]},
    "LVGIVS": {"A": ["21"]},
    "CHAERESTRATVS": {"N": ["12"]},
    "TANFANA": {"N": ["11"]},
    "CATILVS": {"N": ["12"]},
    "DODONAEVS": {"A": ["21"]},
    "DIVITIACVS": {"N": ["12"]},
    "MARSVS": {"N": ["12"], "A": ["21"]},
    "BLOSSIVS": {"N": ["12"]},
    "THERMVS": {"N": ["12"]},
    "ALBANA": {"N": ["11"]},
    "GALATEA": {"N": ["11"]},
    "TABRACA": {"N": ["11"]},
    "LAODICEA": {"N": ["11"]},
    "SYRACOSIVS": {"A": ["21"]},
    "SYMAETHIVS": {"A": ["21"]},
    "AEACIDINVS": {"A": ["21"]},
    "PROCILLVS": {"N": ["12"]},
    "ANICIANVS": {"A": ["21"]},
    "AQVITANVS": {"A": ["21"], "N": ["12"]},
    "CIMBERIVS": {"N": ["12"]},
    "MOLORCHAEVS": {"A": ["21"]},
    "CISPIANVS": {"A": ["21"]},
    "ADGANDESTRIVS": {"N": ["12"]},
    "PARDALISCA": {"N": ["11"]},
    "CARPVS": {"N": ["12"]},
    "PLANCINA": {"N": ["11"]},
    "BERECYNTIVS": {"A": ["21"]},
    "CETHEGVS": {"N": ["12"]},
    "STROBILVS": {"N": ["12"]},
    "DINDYMIS": {"A": ["24"]},
    "ILLVRGAVONENSIS": {"A": ["24"]},
    "SOHAEMVS": {"N": ["12"]},
    "STROPHIVS": {"N": ["12"]},
    "FVNDANVS": {"N": ["12"], "A": ["21"]},
    "MEGARA": {"N": ["11"], "O": ["11"]},
    "AEOLVS": {"N": ["12"]},
    "LIBVRNICVS": {"A": ["21"]},
    "MEGALOBVLVS": {"N": ["12"]},
    "ALLIENVS": {"N": ["12"]},
    "THARSIMACHVS": {"N": ["12"]},
    "NONACRINVS": {"A": ["21"]},
    "AGYIEVS": {"N": ["12"]},
    "RIPHAEVS": {"A": ["21"]},
    "MEDOBRIGA": {"N": ["11"]},
    "CAESARIANVS": {"A": ["21"]},
    "MACRINVS": {"N": ["12"]},
    "BELIVS": {"N": ["12"]},
    "VOLVSENVS": {"N": ["12"]},
    "SAFINIVS": {"N": ["12"]},
    "HIBERIA": {"N": ["11"]},
    "PAPVS": {"N": ["12"]},
    "ILVS": {"N": ["12"]},
    "GNOSIS": {"A": ["24"], "N": ["16"]},
    "AGYLLINVS": {"A": ["21"]},
    "VERGILIVS": {"N": ["12"]},
    "RAETIA": {"N": ["11"]},
    "ELAPTONIVS": {"N": ["12"]},
    "LASERPICIARIVS": {"A": ["21"]},
    "SABAEA": {"N": ["11"]},
    "RVTVPINVS": {"N": ["12"]},
    "PISONIANVS": {"A": ["21"]},
    "SAXA": {"N": ["11"]},
    "THALIARCHVS": {"N": ["12"]},
    "AGRIPPINA": {"N": ["11"]},
    "TVCCIA": {"N": ["11"]},
    "APONIANVS": {"N": ["12"]},
    "PAXAEA": {"N": ["11"]},
    "PATROCLVS": {"N": ["12"]},
    "BROMIVS": {"N": ["12"]},
    "CARTEINENSIS": {"A": ["24"]},
    "CINGA": {"N": ["11"]},
    "BALBINVS": {"N": ["12"]},
    "AEGISTHVS": {"N": ["12"]},
    "ELISSA": {"N": ["11"]},
    "PERRHAEBIA": {"N": ["11"]},
    "OEOBAZVS": {"N": ["12"]},
    "SAGVNTINVS": {"A": ["21"]},
    "ATHAMANVS": {"A": ["21"]},
    "MELANIPPA": {"N": ["11"]},
    "STYMPHALIS": {"A": ["24"], "N": ["16"]},
    "IVNIA": {"N": ["11"]},
    "VENAFRANVS": {"A": ["21"]},
    "SYCHAEVS": {"A": ["21"], "N": ["12"]},
    "HOMERICVS": {"A": ["21"]},
    "ALSIENSIS": {"A": ["24"]},
    "ILLYRICVS": {"A": ["21"]},
    "APHOBETVS": {"N": ["12"]},
    "CAVCASEVS": {"A": ["21"]},
    "ONESICRITVS": {"N": ["12"]},
    "ACADEMIA": {"N": ["11"]},
    "VALERIA": {"N": ["11"]},
    "ARTABAZVS": {"N": ["12"]},
    "VNOMAMMIVS": {"A": ["21"]},
    "POLYCLETVS": {"N": ["12"]},
    "HIBERNIA": {"N": ["11"]},
    "FLAVINIVS": {"A": ["21"]},
    "PENELOPEVS": {"A": ["21"]},
    "CVRENSIS": {"A": ["24"]},
    "TERETINA": {"N": ["11"]},
    "COMANA": {"N": ["11"]},
    "HYACINTVS": {"N": ["12"]},
    "AMISIA": {"N": ["11"]},
    "CALLIMACHVS": {"N": ["12"]},
    "VISTILIVS": {"N": ["12"]},
    "CYNTHIVS": {"A": ["21"], "N": ["12"]},
    "PEVCOLAVS": {"N": ["12"]},
    "SOTERICVS": {"N": ["12"]},
    "FVNISVLANVS": {"N": ["12"]},
    "BITHVS": {"N": ["12"]},
    "AMINNEVS": {"A": ["21"]},
    "PHENEVS": {"N": ["12"]},
    "FIDENA": {"N": ["11"]},
    "MYRTILVS": {"N": ["12"]},
    "SVRVS": {"N": ["12"]},
    "ASVDIVS": {"N": ["12"]},
    "ANAGNINVS": {"A": ["21"]},
    "CADMEIVS": {"A": ["21"]},
    "CYDONIVM": {"N": ["12"]},
    "VALGIVS": {"N": ["12"]},
    "LAIVS": {"N": ["12"]},
    "SYRVS": {"N": ["12"], "A": ["21"]},
    "LYCIVS": {"N": ["12"], "A": ["21"]},
    "NOVARIA": {"N": ["11"]},
    "MINCIVS": {"N": ["12"]},
    "PACIDEIANVS": {"N": ["12"]},
    "THESSALONICA": {"N": ["11"]},
    "AFRICANA": {"N": ["11"]},
    "MASSICVS": {"A": ["21"], "N": ["12"]},
    "PYTHAGOREVS": {"A": ["21"]},
    "ASIVS": {"A": ["21"], "N": ["12"]},
    "CARMANIA": {"N": ["11"]},
    "MANTVA": {"N": ["11"]},
    "GARVTIANVS": {"N": ["12"]},
    "ITVRIVS": {"N": ["12"]},
    "BRIAREVS": {"N": ["12"]},
    "BRVCTERVS": {"A": ["21"]},
    "TITIANVS": {"N": ["12"]},
    "TREBELLIVS": {"N": ["12"]},
    "PACVVIVS": {"N": ["12"]},
    "BVTRA": {"N": ["11"]},
    "CHOBVS": {"N": ["12"]},
    "VINDONISSA": {"N": ["11"]},
    "MENNIVS": {"N": ["12"]},
    "AEGIALVS": {"N": ["12"]},
    "ACESTA": {"N": ["11"]},
    "DYRRACHINVS": {"A": ["21"]},
    "LYCA": {"N": ["11"]},
    "PAPIA": {"N": ["11"]},
    "RHESVS": {"N": ["12"]},
    "ORFITVS": {"N": ["12"]},
    "CISALPINVS": {"A": ["21"]},
    "GRANICVS": {"N": ["12"]},
    "COSA": {"N": ["11"]},
    "RHOETEVM": {"N": ["12"]},
    "VATICANVS": {"A": ["21"], "N": ["12"]},
    "FIRMANVS": {"A": ["21"]},
    "CADMEVS": {"A": ["21"]},
    "CHARAXVS": {"N": ["12"]},
    "FERENTINVM": {"N": ["12"]},
    "MINICIANVS": {"N": ["12"]},
    "SISOCOSTVS": {"N": ["12"]},
    "TREBELLENVS": {"N": ["12"]},
    "GAVRANVS": {"A": ["21"]},
    "PLANCIVS": {"N": ["12"]},
    "FRISIVS": {"A": ["21"]},
    "APERENSIS": {"A": ["24"]},
    "EVMOLPVS": {"N": ["12"]},
    "NAVPLIVS": {"N": ["12"]},
    "FADIA": {"N": ["11"]},
    "ASCLA": {"N": ["11"]},
    "SITHONIS": {"A": ["24"]},
    "LICINIA": {"N": ["11"]},
    "PIRITHOVS": {"N": ["12"]},
    "ARRIANVS": {"N": ["12"]},
    "TVRSELIVS": {"N": ["12"]},
    "CORMA": {"N": ["11"]},
    "SOGDIANVS": {"A": ["21"]},
    "VIRGILIANVS": {"N": ["12"]},
    "VOLCACIVS": {"N": ["12"]},
    "NAXVS": {"N": ["12"]},
    "MNESILOCHVS": {"N": ["12"]},
    "CINYPHIVS": {"A": ["21"]},
    "ANTEIA": {"N": ["11"]},
    "PACONIVS": {"N": ["12"]},
    "ACHERVSIVS": {"A": ["21"]},
    "VINICIVS": {"N": ["12"]},
    "MARVLLVS": {"N": ["12"]},
    "ENCOLPIVS": {"N": ["12"]},
    "PICENVS": {"A": ["21"]},
    "POENINVS": {"A": ["21"]},
    "BRIXIA": {"N": ["11"]},
    "OSCVS": {"A": ["21"]},
    "MVLVIVS": {"A": ["21"], "N": ["12"]},
    "PALPELLIVS": {"N": ["12"]},
    "NOVATVS": {"N": ["12"]},
    "SYBILLA": {"N": ["11"]},
    "REBILVS": {"N": ["12"]},
    "FVRVNCVLVS": {"1": ["12"]},
    "CARPOPHORVS": {"N": ["12"]},
    "ARIADNEVS": {"A": ["21"]},
    "TVRVLLIVS": {"N": ["12"]},
    "SESTVS": {"N": ["12"]},
    "AEOLIVS": {"A": ["21"]},
    "TVRRANIVS": {"N": ["12"]},
    "OTACILIVS": {"N": ["12"]},
    "OGYGIVS": {"A": ["21"]},
    "ELIVS": {"N": ["12"]},
    "SETINVM": {"N": ["12"]},
    "VXELLODVNVM": {"N": ["12"]},
    "RHADAMANTHVS": {"N": ["12"]},
    "BYRSA": {"N": ["11"]},
    "HAEMONIVS": {"A": ["21"]},
    "PANVRGVS": {"N": ["12"]},
    "ATELLANA": {"N": ["11"]},
    "EXAERAMBVS": {"N": ["12"]},
    "SAFINIANVS": {"A": ["21"]},
    "SVESSANVS": {"A": ["21"]},
    "APINIVS": {"N": ["12"]},
    "NVMITORIA": {"N": ["11"]},
    "PARADA": {"N": ["11"]},
    "PERIMEDEVS": {"A": ["21"]},
    "CHRYSALVS": {"N": ["12"]},
    "SIBYRTIVS": {"N": ["12"]},
    "SOSIBIVS": {"N": ["12"]},
    "SEQVANVS": {"A": ["21"]},
    "ITHACENSIS": {"A": ["24"]},
    "TIMAVVS": {"N": ["12"]},
    "CIRRHAEVS": {"A": ["21"]},
    "STALAGMVS": {"N": ["12"]},
    "ENNOSIGAEVS": {"N": ["12"]},
    "PARTHIA": {"N": ["11"]},
    "COSCONIVS": {"N": ["12"]},
    "SCIRONIS": {"A": ["24"]},
    "APPIANVS": {"A": ["21"], "N": ["12"]},
    "AGONALIS": {"A": ["24"]},
    "COA": {"N": ["11"]},
    "VICETIA": {"N": ["11"]},
    "ATHENA": {"N": ["11"]},
    "PATALIA": {"N": ["11"]},
    "AVENTICVM": {"N": ["12"]},
    "CALVENTIVS": {"A": ["21"], "N": ["12"]},
    "TEANVM": {"N": ["12"]},
    "BRIXIANVS": {"A": ["21"]},
    "PHALANTHVS": {"N": ["12"]},
    "CYPRIA": {"N": ["11"]},
    "MYTILENAEVS": {"A": ["21"]},
    "IVDA": {"N": ["11"]},
    "CODRVS": {"N": ["12"]},
    "TITYRVS": {"N": ["12"]},
    "MOLOSSVS": {"N": ["12"], "A": ["21"]},
    "BRISAEVS": {"N": ["12"]},
    "DODONA": {"N": ["11"]},
    "FVRNIVS": {"N": ["12"]},
    "ATTICINVS": {"N": ["12"]},
    "AVTRONIVS": {"N": ["12"]},
    "ROSIVS": {"N": ["12"]},
    "BALBVTIVS": {"N": ["12"]},
    "VRBINIA": {"N": ["11"]},
    "AVERNALIS": {"A": ["24"]},
    "ALBANVM": {"N": ["12"]},
    "SYRIACVS": {"A": ["21"]},
    "CINYRA": {"N": ["11"]},
    "PINARVS": {"N": ["12"]},
    "GAZABA": {"N": ["11"]},
    "INOVS": {"A": ["21"]},
    "PAETINA": {"N": ["11"]},
    "CHAONIA": {"N": ["11"]},
    "MARIANVS": {"A": ["21"]},
    "TANAGRAEVS": {"A": ["21"]},
    "PARTHENIVS": {"N": ["12"], "A": ["21"]},
    "ERYTHRVS": {"N": ["12"]},
    "CONCANVS": {"A": ["21"]},
    "BESSICVS": {"A": ["21"]},
    "APPVLEIA": {"N": ["11"]},
    "MAMERCVS": {"N": ["12"]},
    "BARBIVS": {"N": ["12"]},
    "MAVRVSIVS": {"A": ["21"]},
    "MILONIVS": {"N": ["12"]},
    "ACTVMERVS": {"N": ["12"]},
    "TIBERINIS": {"A": ["24"]},
    "CAYCVS": {"N": ["12"]},
    "MALEA": {"N": ["11"]},
    "ISIACVS": {"A": ["21"]},
    "CENTAVREVS": {"A": ["21"]},
    "VORANVS": {"N": ["12"]},
    "MVTVSCA": {"N": ["11"]},
    "PRVSIADA": {"N": ["11"]},
    "APVLEIVS": {"N": ["12"]},
    "OLENNIVS": {"N": ["12"]},
    "ALPINVS": {"N": ["12"], "A": ["21"]},
    "GORDIVS": {"N": ["12"]},
    "ARCETIVS": {"N": ["12"]},
    "LVSITANVS": {"A": ["21"], "N": ["12"]},
    "ALSVS": {"N": ["12"]},
    "ANIENVS": {"A": ["21"]},
    "PHLEGRAEVS": {"A": ["21"]},
    "ALBINGAVNVM": {"N": ["12"]},
    "GALVS": {"N": ["12"]},
    "COTVATVS": {"N": ["12"]},
    "ARVERNVS": {"A": ["21"]},
    "AETHALVS": {"N": ["12"]},
    "TANAGRA": {"N": ["11"]},
    "DINDYMA": {"N": ["11"]},
    "IVGVRTHINVS": {"A": ["21"]},
    "VOTIENVS": {"N": ["12"]},
    "SEMPRONIA": {"N": ["11"]},
    "PVLFENIVS": {"N": ["12"]},
    "STRYMONIVS": {"A": ["21"], "N": ["12"]},
    "STLOGA": {"N": ["11"]},
    "PLANCVS": {"N": ["12"]},
    "HYDAEPA": {"N": ["11"]},
    "HERACLITVS": {"N": ["12"]},
    "ACRATVS": {"N": ["12"]},
    "REMVRIA": {"N": ["11"]},
    "DEMOLEVS": {"N": ["12"]},
    "ARCHIPPVS": {"N": ["12"]},
    "SARDOVS": {"A": ["21"]},
    "LEVCOPETRA": {"N": ["11"]},
    "CAMPANEVS": {"A": ["21"]},
    "CYBELVS": {"N": ["12"]},
    "NABATHAEVS": {"A": ["21"]},
    "SCETANIVS": {"N": ["12"]},
    "CATIA": {"N": ["11"]},
    "PYTHAGORICVS": {"A": ["21"]},
    "CHOERILVS": {"N": ["12"]},
    "PVLLIA": {"N": ["11"]},
    "CERINTHVS": {"N": ["12"]},
    "TYNDARVS": {"N": ["12"]},
    "PAEONIVS": {"A": ["21"]},
    "RHAEBVS": {"N": ["12"]},
    "PELOPEIVS": {"A": ["21"]},
    "POLYTIMETVS": {"N": ["12"]},
    "LEBEDVS": {"N": ["12"]},
    "CETRIVS": {"N": ["12"]},
    "ARVSEIVS": {"N": ["12"]},
    "NIGIDIVS": {"N": ["12"]},
    "CHAONIS": {"A": ["24"]},
    "GERGOVIA": {"N": ["11"]},
    "MOSA": {"N": ["11"]},
    "CENTAVROMACHIA": {"N": ["11"]},
    "LVCANIVS": {"A": ["21"], "N": ["12"]},
    "AMANVS": {"N": ["12"]},
    "LERNA": {"N": ["11"]},
    "EVRYTHMVS": {"N": ["12"]},
    "MARMARICVS": {"A": ["21"]},
    "MATTIVM": {"N": ["12"]},
    "HELLESPONTIVS": {"A": ["21"]},
    "CORINTHIA": {"N": ["11"]},
    "SELEVCEA": {"N": ["11"]},
    "DECIVS": {"N": ["12"]},
    "AEGALEVS": {"N": ["12"]},
    "ANCONA": {"N": ["11"]},
    "PITVANIVS": {"N": ["12"]},
    "GALLONIVS": {"N": ["12"]},
    "SYBYLLA": {"N": ["11"]},
    "ANTEMNA": {"N": ["11"]},
    "RAEDARIVS": {"1": ["12"]},
    "EVBOEA": {"N": ["11"]},
    "ACIDALIA": {"N": ["11"]},
    "HYRCANVS": {"A": ["21"]},
    "IDALIVS": {"A": ["21"]},
    "ASPAVIA": {"N": ["11"]},
    "PHOCAEVS": {"A": ["21"]},
    "HERCVLANEVS": {"N": ["12"], "A": ["21"]},
    "FIDVLIVS": {"N": ["12"]},
    "TVSCILIVS": {"N": ["12"]},
    "ANA": {"N": ["11"]},
    "CLVSIVS": {"N": ["12"]},
    "PERGAMA": {"N": ["11"]},
    "AEQVIMAELIVM": {"N": ["12"]},
    "APENNINICOLA": {"N": ["11"]},
    "SORA": {"N": ["11"]},
    "IOLAVS": {"N": ["12"]},
    "RADAMISTVS": {"N": ["12"]},
    "INSTEIVS": {"N": ["12"]},
    "TANVSIVS": {"N": ["12"]},
    "VEHILIVS": {"N": ["12"]},
    "MACAREVS": {"N": ["12"]},
    "COMPITALIA": {"-": ["11"], "N": ["12"]},
    "CAESERNIVS": {"N": ["12"]},
    "PANCHAIA": {"N": ["11"]},
    "CARSIDIVS": {"N": ["12"]},
    "EVNOMIA": {"N": ["11"]},
    "MEMMIVS": {"N": ["12"]},
    "HECVBA": {"N": ["11"]},
    "FERENTVM": {"N": ["12"]},
    "CLVENTIA": {"N": ["11"]},
    "PYRRHA": {"N": ["11"]},
    "COLCHIS": {"N": ["13"], "A": ["21"]},
    "HYRCANIA": {"N": ["11"]},
    "METLOSEDVM": {"N": ["12"]},
    "PANOPEA": {"N": ["11"]},
    "IDALIA": {"N": ["11"]},
    "HEGELOCHVS": {"N": ["12"]},
    "PELOPONNESIACVS": {"A": ["21"]},
    "ALBVNEA": {"N": ["11"]},
    "IASONIVS": {"A": ["21"]},
    "VISELLIVS": {"N": ["12"]},
    "CRODVNVM": {"N": ["12"]},
    "QVINTILIA": {"N": ["11"]},
    "PERGAMENVS": {"A": ["21"]},
    "DOMITIANVS": {"N": ["12"], "A": ["21"]},
    "MARACANDA": {"N": ["11"]},
    "AECVS": {"N": ["12"]},
    "AVRVNCA": {"N": ["11"]},
    "EPPIVS": {"N": ["12"]},
    "PARRHASIVS": {"A": ["21"], "N": ["12"]},
    "DEMIANVS": {"N": ["12"]},
    "ARRETINVS": {"A": ["21"]},
    "TIBVRNVS": {"A": ["21"], "N": ["12"]},
    "AEMILIA": {"N": ["11"]},
    "FORNACALIA": {"N": ["11"]},
    "CLINIA": {"N": ["11"]},
    "NVMIDVS": {"A": ["21"]},
    "LAVINIVS": {"A": ["21"]},
    "VRGVLANIA": {"N": ["11"]},
    "PHRIXVS": {"N": ["12"]},
    "VIBIDIA": {"N": ["11"]},
    "LVPIA": {"N": ["11"]},
    "ALBIANVS": {"A": ["21"]},
    "BACTRVS": {"N": ["12"]},
    "TOLOSA": {"N": ["11"]},
    "TAMPILVS": {"N": ["12"]},
    "AMPHRYSVS": {"N": ["12"]},
    "GLOBVLVS": {"-": ["12"], "N": ["12"]},
    "FIDICVLANIVS": {"N": ["12"]},
    "MESOPOTAMIA": {"N": ["11"]},
    "QVINTA": {"N": ["11"]},
    "ANIENSIS": {"A": ["24"]},
    "BACCHIVS": {"N": ["12"]},
    "ELEGEIA": {"N": ["11"]},
    "SIDONICVS": {"A": ["21"]},
    "LABIENVS": {"N": ["12"]},
    "THERODAMANTEVS": {"A": ["21"]},
    "LAPPIVS": {"N": ["12"]},
    "BODOTRIA": {"N": ["11"]},
    "ROMANIENSIS": {"A": ["24"]},
    "CAECILIA": {"N": ["11"]},
    "ARIONIVS": {"A": ["21"]},
    "THRONIVS": {"N": ["12"]},
    "DAVNIVS": {"A": ["21"]},
    "BALBILLVS": {"N": ["12"]},
    "STHENELVS": {"N": ["12"]},
    "MINOVS": {"A": ["21"]},
    "TREVIRICVS": {"A": ["21"]},
    "PHINEIVS": {"A": ["21"]},
    "SESCENTOPLAGVS": {"N": ["12"]},
    "QVERQVETVLANVS": {"N": ["12"]},
    "VECELLINVS": {"N": ["12"]},
    "TARRACONENSIS": {"A": ["24"]},
    "COCCEIANVS": {"N": ["12"]},
    "AEFVLA": {"N": ["11"]},
    "CARNVTIS": {"A": ["24"]},
    "SALLVSTIANVS": {"A": ["21"]},
    "PODALIRIVS": {"N": ["12"]},
    "INDVTIOMARVS": {"N": ["12"]},
    "SANGARIVS": {"N": ["12"]},
    "MOENVS": {"N": ["12"]},
    "LIPARAEVS": {"A": ["21"]},
    "INTERAMNA": {"N": ["11"]},
    "CAESENNIA": {"N": ["11"]},
    "ARTEMITA": {"N": ["11"]},
    "LVCANVS": {"A": ["21"], "N": ["12"]},
    "ALEXANDREA": {"N": ["11"]},
    "CASTALIA": {"N": ["11"]},
    "ERYTHRAEVS": {"A": ["21"]},
    "RVFRIVM": {"N": ["12"]},
    "THERMOPOLIVM": {"N": ["12"]},
    "SIRPICVS": {"N": ["12"]},
    "ERECHTHEVS": {"N": ["12"]},
    "APOLLONIDENSIS": {"A": ["24"]},
    "TEIVS": {"A": ["21"]},
    "ILLYRIA": {"N": ["11"]},
    "CYNTHVS": {"N": ["12"]},
    "MVSONIVS": {"N": ["12"]},
    "PACTOLVS": {"N": ["12"]},
    "AQVILEIENSIS": {"A": ["24"]},
    "SVSIS": {"A": ["24"]},
    "TROICA": {"N": ["11"]},
    "HELLESPONTIACVS": {"A": ["21"]},
    "TIGELLIVS": {"N": ["12"]},
    "RHEGINVS": {"A": ["21"]},
    "CISPIVS": {"N": ["12"]},
    "TERENTIA": {"N": ["11"]},
    "SICANVS": {"A": ["21"]},
    "IARBA": {"N": ["11"]},
    "COMVM": {"N": ["12"]},
    "PHARMACVS": {"N": ["12"]},
    "ANDRONICVS": {"N": ["12"]},
    "NELEIVS": {"A": ["21"]},
    "ABYDVS": {"N": ["12"]},
    "RIGODVLVM": {"N": ["12"]},
    "METHYMNA": {"N": ["11"]},
    "PACIVS": {"N": ["12"]},
    "GYARVS": {"N": ["12"]},
    "BOVILLANVS": {"A": ["21"]},
    "CERAVNIA": {"N": ["11"]},
    "IONICVS": {"A": ["21"]},
    "GARGONIVS": {"N": ["12"]},
    "NICAEA": {"N": ["11"]},
    "CENCHREVS": {"N": ["12"]},
    "CYLLENA": {"N": ["11"]},
    "IVDAEVS": {"A": ["21"]},
    "HEGESIMACHVS": {"N": ["12"]},
    "TYCHIVS": {"N": ["12"]},
    "SENTICVS": {"A": ["21"]},
    "CELADVS": {"N": ["12"]},
    "TARPA": {"N": ["11"]},
    "HELVINVS": {"A": ["21"]},
    "BABYLONIA": {"N": ["11"]},
    "ZEVGMA": {"N": ["11"]},
    "VESVLVS": {"N": ["12"]},
    "VRSIDIVS": {"N": ["12"]},
    "PLAETORIVS": {"A": ["21"], "N": ["12"]},
    "HAEMVS": {"N": ["12"]},
    "RHENVS": {"N": ["12"]},
    "XENIPPA": {"N": ["11"]},
    "VENALICIVS": {"1": ["12"]},
    "AMASTRVS": {"N": ["12"]},
    "HISTRIA": {"N": ["11"]},
    "MACRINA": {"N": ["11"]},
    "LICINIANVS": {"A": ["21"], "N": ["12"]},
    "COENVS": {"N": ["12"]},
    "ASINARIA": {"N": ["11"]},
    "VALERIANVS": {"A": ["21"]},
    "IGVIVM": {"N": ["12"]},
    "HELENVS": {"N": ["12"]},
    "CVRIVS": {"N": ["12"]},
    "SICELIS": {"A": ["24"]},
    "ALEIVS": {"A": ["21"]},
    "PENTHEVS": {"N": ["12"]},
    "AHALA": {"N": ["11"]},
    "ERILLVS": {"N": ["12"]},
    "PASSENNVS": {"N": ["12"]},
    "CENABENSIS": {"A": ["24"]},
    "DIOXENVS": {"N": ["12"]},
    "POSTVERTA": {"N": ["11"]},
    "CISRHENANVS": {"A": ["21"]},
    "SANGA": {"N": ["11"]},
    "NORTIA": {"N": ["11"]},
    "THEOPHILVS": {"N": ["12"]},
    "ELECTRVS": {"N": ["12"]},
    "BITHYNVS": {"A": ["21"]},
    "VENETIA": {"N": ["11"]},
    "PALLADIVS": {"A": ["21"]},
    "RHINOCOLVRA": {"N": ["11"]},
    "PERSEVS": {"N": ["12"], "A": ["21"]},
    "SCYTHA": {"N": ["11"]},
    "PROTESILAVS": {"N": ["12"]},
    "BABA": {"N": ["11"]},
    "DIA": {"N": ["11"]},
    "GRATIDIVS": {"N": ["12"]},
    "OROANDICVS": {"A": ["21"]},
    "PAPHLAGONA": {"N": ["11"]},
    "PRISCINVS": {"N": ["12"]},
    "BELGICVS": {"A": ["21"]},
    "TREBVLANVS": {"A": ["21"]},
    "STERTINIVS": {"N": ["12"], "A": ["21"]},
    "DECETIA": {"N": ["11"]},
    "LYDVS": {"A": ["21"], "N": ["12"]},
    "PALILIA": {"N": ["11"]},
    "BIRRIVS": {"N": ["12"]},
    "PHRYGIVS": {"A": ["21"]},
    "AMEANA": {"N": ["11"]},
    "LYGDVS": {"N": ["12"]},
    "FANNIA": {"N": ["11"]},
    "ERIGONEIVS": {"A": ["21"]},
    "COPONIVS": {"N": ["12"]},
    "VIRIATVS": {"N": ["12"]},
    "EVNAEVS": {"N": ["12"]},
    "ARGOVS": {"A": ["21"]},
    "SARRANVS": {"A": ["21"]},
    "FALCIDIANVS": {"A": ["21"]},
    "ABASCANTVS": {"N": ["12"]},
    "ANDROGEONEVS": {"A": ["21"]},
    "HAMILLVS": {"N": ["12"]},
    "LENAEVS": {"A": ["21"], "N": ["12"]},
    "HERMVS": {"N": ["12"]},
    "CAEPASIVS": {"N": ["12"]},
    "DARDANVS": {"A": ["21"], "N": ["12"]},
    "VARIA": {"N": ["11"]},
    "ARCADIVS": {"A": ["21"]},
    "BETILIENVS": {"N": ["12"]},
    "HIRPINVS": {"A": ["21"], "N": ["12"]},
    "CATVVOLCVS": {"N": ["12"]},
    "DEIDAMIA": {"N": ["11"]},
    "SACRANVS": {"A": ["21"]},
    "CLOACINA": {"N": ["11"]},
    "RIPAEVS": {"A": ["21"]},
    "RVFA": {"N": ["11"]},
    "VIBIENVS": {"N": ["12"]},
    "MAZAEVS": {"N": ["12"]},
    "CIRRHA": {"N": ["11"]},
    "TAVROMENVM": {"N": ["12"]},
    "PERRHAEBVS": {"A": ["21"]},
    "MEGADORVS": {"N": ["12"]},
    "VIBIA": {"N": ["11"]},
    "EPEVS": {"N": ["12"]},
    "PHIDIACVS": {"A": ["21"]},
    "ECBATANA": {"N": ["11"]},
    "PETRAEVS": {"N": ["12"]},
    "MEMORATVS": {"1": ["12"]},
    "SASERNA": {"N": ["11"]},
    "CRVSTVMINVM": {"N": ["12"]},
    "DORIS": {"N": ["13"], "A": ["12"]},
    "VERANIA": {"N": ["11"]},
    "ICELVS": {"N": ["12"]},
    "MAENALA": {"N": ["11"]},
    "CLITARCHVS": {"N": ["12"]},
    "GERYONACEVS": {"A": ["21"]},
    "ALCESIMVS": {"N": ["12"]},
    "SILIANVS": {"A": ["21"]},
    "PARNASSIVS": {"A": ["21"]},
    "PICENTINVS": {"A": ["21"]},
    "BONNA": {"N": ["11"]},
    "MENOPHILA": {"N": ["11"]},
    "MONOECVS": {"N": ["12"]},
    "PISTOCLERVS": {"N": ["12"]},
    "NAVTACA": {"N": ["11"]},
    "MAMVRRA": {"N": ["11"]},
    "CONCONNETODVMNVS": {"N": ["12"]},
    "ANDROMACHVS": {"N": ["12"]},
    "VOLTINIENSIS": {"A": ["24"]},
    "ITIVS": {"N": ["12"]},
    "CLEONICVS": {"N": ["12"]},
    "SITTIVS": {"N": ["12"]},
    "VARILLVS": {"N": ["12"]},
    "CNEVS": {"N": ["12"]},
    "APOLLINEVS": {"A": ["21"]},
    "DEIPHOBVS": {"N": ["12"]},
    "HERMOBIVS": {"N": ["12"]},
    "FREGELLANVS": {"A": ["21"], "N": ["12"]},
    "PHAEACIVS": {"A": ["21"]},
    "ITYLVS": {"N": ["12"]},
    "CVPENCVS": {"N": ["12"]},
    "ORBIVS": {"N": ["12"]},
    "VARGVNTEIVS": {"N": ["12"]},
    "COSANVM": {"N": ["12"]},
    "CRISPINA": {"N": ["11"]},
    "CYBELEIVS": {"A": ["21"]},
    "PYRAGRVS": {"N": ["12"]},
    "TAXIMAGVLVS": {"N": ["12"]},
    "ASSACANVS": {"N": ["12"]},
    "SESTIANVS": {"A": ["21"]},
    "PVTEOLANVS": {"A": ["21"]},
    "GEMELLINVS": {"N": ["12"]},
    "GLAVCIPPVS": {"N": ["12"]},
    "ASIANVS": {"A": ["21"]},
    "DELLIVS": {"N": ["12"]},
    "VOLOGAESVS": {"N": ["12"]},
    "TRICCA": {"N": ["11"]},
    "LOLLIA": {"N": ["11"]},
    "CEA": {"N": ["11"]},
    "TIBVRTINVS": {"A": ["21"]},
    "HISPANIENSIS": {"A": ["24"]},
    "EDESSA": {"N": ["11"]},
    "TREBVLA": {"N": ["11"]},
    "PACTVMEVS": {"N": ["12"]},
    "SYMAETHEVS": {"A": ["21"]},
    "SYMPOSIVM": {"N": ["12"]},
    "AQVILEIA": {"N": ["11"]},
    "ACHERVNTIA": {"N": ["11"]},
    "MARICCVS": {"N": ["12"]},
    "OXVS": {"N": ["12"]},
    "CALVINVS": {"N": ["12"]},
    "MYRTOVS": {"A": ["21"]},
    "TELANVS": {"A": ["21"]},
    "MOSCHVS": {"N": ["12"]},
    "HOMERISTA": {"N": ["11"]},
    "LYRNESIVS": {"A": ["21"]},
    "PLATAGIDORVS": {"N": ["12"]},
    "IGILIVM": {"N": ["12"]},
    "SCYRIVS": {"A": ["21"]},
    "DORYLENSIS": {"A": ["24"]},
    "TAVNVS": {"N": ["12"]},
    "CRATINVS": {"N": ["12"]},
    "SERVAEVS": {"N": ["12"]},
    "HYLLEVS": {"A": ["21"]},
    "PLISTHENIVS": {"A": ["21"]},
    "LETHAEVS": {"A": ["21"]},
    "SVEBICVS": {"A": ["21"]},
    "ORICIVS": {"A": ["21"]},
    "CAECINA": {"N": ["11"]},
    "THRASEA": {"N": ["11"]},
    "ERICHTHONIVS": {"N": ["12"], "A": ["21"]},
    "PLACENTIA": {"N": ["11"]},
    "SAMVS": {"N": ["12"]},
    "VERONENSIS": {"A": ["24"]},
    "VICANVS": {"1": ["12"]},
    "CAMVRTIVS": {"N": ["12"]},
    "NVMANTINA": {"N": ["11"]},
    "IRVS": {"N": ["12"]},
    "METANIRA": {"N": ["11"]},
    "DEIOTARVS": {"N": ["12"]},
    "HAGNA": {"N": ["11"]},
    "PEGASVS": {"N": ["12"]},
    "AQVITANIA": {"N": ["11"]},
    "VIENNA": {"N": ["11"]},
    "HYDROCHOEVS": {"N": ["12"]},
    "DESVLTORIVS": {"1": ["12"]},
    "OBVLTRONIVS": {"N": ["12"]},
    "NASVA": {"N": ["11"]},
    "GLAPHYRVS": {"N": ["12"]},
    "TAYGETVS": {"N": ["12"]},
    "TRIPHALLVS": {"N": ["12"]},
    "ALCAEVS": {"N": ["12"]},
    "PETRONIA": {"N": ["11"]},
    "ORCYNIA": {"N": ["11"]},
    "PETRIANVS": {"A": ["21"]},
    "FORMIANVS": {"A": ["21"]},
    "THREICIVS": {"A": ["21"]},
    "CAELIANVS": {"N": ["12"], "A": ["21"]},
    "TROIVS": {"A": ["21"]},
    "EVROPAEVS": {"A": ["21"]},
    "VATIA": {"N": ["11"]},
    "PELIVS": {"A": ["21"]},
    "SEGOVIA": {"N": ["11"]},
    "FALCIDIVS": {"N": ["12"]},
    "RVFILLA": {"N": ["11"]},
    "SYRTICVS": {"A": ["21"]},
    "NASICA": {"N": ["11"]},
    "LEVCTRICVS": {"A": ["21"]},
    "ORITHYIA": {"N": ["11"]},
    "BRATVSPANTIVM": {"N": ["12"]},
    "CHIONIVS": {"A": ["21"]},
    "TENTYRA": {"N": ["11"]},
    "PYRRICHVS": {"N": ["12"]},
    "GELOVS": {"A": ["21"]},
    "THESPROTVS": {"N": ["12"]},
    "OETAEVS": {"A": ["21"]},
    "CAEPARIVS": {"N": ["12"]},
    "AVARICVM": {"N": ["12"]},
    "ORICVS": {"N": ["12"]},
    "THYNIA": {"N": ["11"]},
    "GALERIA": {"N": ["11"]},
    "MEGALOPOLITANVS": {"A": ["21"]},
    "AMPIVS": {"N": ["12"]},
    "AVFIDIVS": {"N": ["12"]},
    "THYREATIS": {"A": ["24"]},
    "ARIONVS": {"A": ["21"]},
    "PANDIONIVS": {"A": ["21"]},
    "LIBENTIA": {"N": ["11"]},
    "CORVINVS": {"N": ["12"]},
    "CENTAVRICVS": {"A": ["21"]},
    "MAMVRRIVS": {"N": ["12"]},
    "GNOSIVS": {"A": ["21"]},
    "GRYPVS": {"N": ["12"]},
    "MYCONVS": {"N": ["12"]},
    "ANTIVS": {"N": ["12"]},
    "MVLVCHA": {"N": ["11"]},
    "PTOLOMAEVS": {"N": ["12"]},
    "ACIDINVS": {"N": ["12"]},
    "ALPINIVS": {"N": ["12"]},
    "SANTONICVS": {"A": ["21"]},
    "ASINA": {"N": ["11"]},
    "AXONA": {"N": ["11"]},
    "TIGELLINVS": {"N": ["12"]},
    "TETRARCHIA": {"-": ["11"], "N": ["12"]},
    "NOTARIVS": {"1": ["12"]},
    "PVSILLA": {"N": ["11"]},
    "NVMERIVS": {"N": ["12"]},
    "MAEOTIVS": {"A": ["21"]},
    "PERILAVS": {"N": ["12"]},
    "HELVETIVS": {"A": ["21"]},
    "MOPSOPIVS": {"A": ["21"]},
    "VENVTIVS": {"N": ["12"]},
    "HYLAEVS": {"N": ["12"], "A": ["21"]},
    "CAMVLODVNVM": {"N": ["12"]},
    "LAELIANVS": {"A": ["21"]},
    "SVLLANVS": {"A": ["21"]},
    "HALOTVS": {"N": ["12"]},
    "SALONINA": {"N": ["11"]},
    "INTIMILIVM": {"N": ["12"]},
    "CNIDVS": {"N": ["12"]},
    "ROSIANVS": {"N": ["12"]},
    "OLENVS": {"N": ["12"]},
    "BOVDICCA": {"N": ["11"]},
    "DORYCLVS": {"N": ["12"]},
    "GRVMENTVM": {"N": ["12"]},
    "HYBLAEVS": {"A": ["21"]},
    "CYLLENIVS": {"A": ["21"], "N": ["12"]},
    "OEBALIA": {"N": ["11"]},
    "HIPPOLYTA": {"N": ["11"]},
    "ARSAMOSATA": {"N": ["11"]},
    "MAECIVS": {"N": ["12"]},
    "LONGARENVS": {"N": ["12"]},
    "LONGINVS": {"N": ["12"]},
    "CARA": {"N": ["11"]},
    "STHENELEIVS": {"A": ["21"]},
    "MASSICA": {"N": ["11"]},
    "GRYLLVS": {"N": ["12"]},
    "DORVS": {"N": ["12"], "A": ["21"]},
    "VISIDIVS": {"N": ["12"]},
    "IONIVS": {"A": ["21"]},
    "RVSTIANVS": {"N": ["12"]},
    "HERCEVS": {"A": ["21"]},
    "PROCHYTA": {"N": ["11"]},
    "AVSONIA": {"N": ["11"]},
    "ELYSIVM": {"N": ["12"]},
    "SEGIMVNDVS": {"N": ["12"]},
    "EPOREDIA": {"N": ["11"]},
    "CORANVS": {"N": ["12"], "A": ["21"]},
    "PANNONIVS": {"A": ["21"]},
    "AMITERNVS": {"A": ["21"]},
    "LAODAMIA": {"N": ["11"]},
    "CAYSTRVS": {"N": ["12"]},
    "LEPIDA": {"N": ["11"]},
    "NYSIVS": {"N": ["12"]},
    "NOVAESIVM": {"N": ["12"]},
    "THEODOTVS": {"N": ["12"]},
    "DIADEMATVS": {"N": ["12"]},
    "LENTIDIVS": {"N": ["12"]},
    "FISCELLVS": {"N": ["12"]},
    "IPHIANASSA": {"N": ["11"]},
    "ARCTVRVS": {"N": ["12"]},
    "BRVTTIDIVS": {"N": ["12"]},
    "COTVS": {"N": ["12"]},
    "LIMETANVS": {"N": ["12"]},
    "PELVSIACVS": {"A": ["21"]},
    "AXIA": {"N": ["11"]},
    "PTERELA": {"N": ["11"]},
    "CHAERIBVLVS": {"N": ["12"]},
    "COBIAMACHVS": {"N": ["12"]},
    "TAENARVS": {"N": ["12"]},
    "GENVSVS": {"N": ["12"]},
    "BLESAMIVS": {"N": ["12"]},
    "LYBIA": {"N": ["11"]},
    "AVIOLA": {"N": ["11"]},
    "CARYSTEVS": {"A": ["21"]},
    "MISENVS": {"N": ["12"], "A": ["21"]},
    "ACARNANIA": {"N": ["11"]},
    "PAVLINA": {"N": ["11"]},
    "CYRSILVS": {"N": ["12"]},
    "OCTAVIANIVS": {"A": ["21"]},
    "QVINCTIVS": {"N": ["12"]},
    "CLVSTVMINA": {"N": ["11"]},
    "ARPVS": {"N": ["12"]},
    "DILLIVS": {"N": ["12"]},
    "EPIGONVS": {"N": ["12"]},
    "PERGAMEVS": {"A": ["21"]},
    "STATIRA": {"N": ["11"]},
    "ARARICVS": {"N": ["12"]},
    "HERMOLAVS": {"N": ["12"]},
    "AMBRACIA": {"N": ["11"]},
    "MENTOREVS": {"A": ["21"]},
    "CRESSVS": {"A": ["21"]},
    "ISMARIVS": {"A": ["21"]},
    "HORTALVS": {"N": ["12"]},
    "ISMENVS": {"N": ["12"]},
    "AVGVSTODVNVM": {"N": ["12"]},
    "PALLANTIVS": {"A": ["21"]},
    "AEDVVS": {"A": ["21"]},
    "MVTINA": {"N": ["11"]},
    "EVRYTVS": {"N": ["12"]},
    "SVADA": {"N": ["11"]},
    "IVLLVS": {"N": ["12"]},
    "VIRIOLA": {"N": ["11"]},
    "AVFIDIENVS": {"N": ["12"]},
    "DORYPHORVS": {"N": ["12"]},
    "CAESONINVS": {"N": ["12"]},
    "LYAEVS": {"N": ["12"]},
    "ACONTIVS": {"N": ["12"]},
    "CALENVM": {"N": ["12"]},
    "SASSIA": {"N": ["11"]},
    "MESSALINA": {"N": ["11"]},
    "PETREVS": {"N": ["12"]},
    "CELEMNA": {"N": ["11"]},
    "AELIVS": {"A": ["21"], "N": ["12"]},
    "TOLBIACVM": {"N": ["12"]},
    "VETILIVS": {"N": ["12"]},
    "CAVSINIVS": {"N": ["12"]},
    "PINDARVS": {"N": ["12"]},
    "CORINTHIENSIS": {"A": ["24"]},
    "PHYLLEIS": {"A": ["24"]},
    "CASTA": {"N": ["11"]},
    "IOLCIACVS": {"A": ["21"]},
    "LIGVSTICVS": {"A": ["21"]},
    "LABIENIANVS": {"A": ["21"]},
    "ANTINOVS": {"N": ["12"]},
    "RVSTICVLVS": {"1": ["12"]},
    "PHARSALIA": {"N": ["11"]},
    "MENOECEVS": {"N": ["12"]},
    "CINYRVS": {"N": ["12"]},
    "OECONOMICVS": {"N": ["12"]},
    "HARMODIVS": {"N": ["12"]},
    "SVNIVM": {"N": ["12"]},
    "ITVRAEVS": {"A": ["21"]},
    "IONIACVS": {"A": ["21"]},
    "AGANIPPAEVS": {"A": ["21"]},
    "NEOPTOLEMVS": {"N": ["12"]},
    "CEPHALIVS": {"A": ["21"]},
    "IVVENTIVS": {"N": ["12"], "A": ["21"]},
    "DONATIVS": {"N": ["12"]},
    "BOSPORVS": {"N": ["12"]},
    "LATONIA": {"N": ["11"]},
    "CHALCIDICVS": {"A": ["21"]},
    "GANGETICVS": {"A": ["21"]},
    "CANINIVS": {"N": ["12"]},
    "POMPONIANVS": {"N": ["12"], "A": ["21"]},
    "TANTALICVS": {"A": ["21"]},
    "LABERIVS": {"N": ["12"]},
    "HELIVS": {"N": ["12"]},
    "BOLANVS": {"N": ["12"]},
    "AVRIVS": {"N": ["12"]},
    "MYGDONIVS": {"A": ["21"]},
    "CAECVBVM": {"N": ["12"]},
    "PRIVERNVM": {"N": ["12"]},
    "CABALLINVS": {"A": ["21"]},
    "ALTINVM": {"N": ["12"]},
    "CARENVS": {"A": ["21"]},
    "GAIANVS": {"A": ["21"]},
    "THRASIVS": {"N": ["12"]},
    "EVRYMACHVS": {"N": ["12"]},
    "CASINVS": {"N": ["12"]},
    "NINVS": {"N": ["12"]},
    "SENIENSIS": {"A": ["24"]},
    "BOIHAEMVM": {"N": ["12"]},
    "PAEGNIVM": {"N": ["12"]},
    "ARGEVS": {"A": ["21"]},
    "XANTHIPPVS": {"N": ["12"]},
    "VELANIVS": {"N": ["12"]},
    "CAECVLVS": {"N": ["12"]},
    "HERCVLEVS": {"A": ["21"]},
    "AMPHRYSIVS": {"A": ["21"]},
    "CARFVLENVS": {"N": ["12"]},
    "CATAMITVS": {"N": ["12"], "-": ["12"]},
    "AMPELVS": {"N": ["12"]},
    "CORYNAEVS": {"N": ["12"]},
    "LACONICVS": {"A": ["21"]},
    "ALBVCIVS": {"N": ["12"]},
    "LAOMEDONTEVS": {"A": ["21"]},
    "AQVICVLVS": {"N": ["12"]},
    "CALYDONIVS": {"A": ["21"]},
    "POPVLONIA": {"N": ["11"]},
    "MATRALIA": {"N": ["11"]},
    "IOLLA": {"N": ["11"]},
    "PHILIPPENSIS": {"A": ["24"]},
    "PHILODEMVS": {"N": ["12"]},
    "EVDOXVS": {"N": ["12"]},
    "GRYNEVS": {"A": ["21"]},
    "HESPERVS": {"N": ["12"]},
    "BERENICEVS": {"A": ["21"]},
    "LACETANIA": {"N": ["11"]},
    "OPHIONIVS": {"A": ["21"]},
    "ANCYRA": {"N": ["11"]},
    "CREMONENSIS": {"A": ["24"]},
    "ATABVLVS": {"N": ["12"]},
    "NICEPHORIVS": {"N": ["12"]},
    "CYTHERA": {"N": ["11"]},
    "ARTACANA": {"N": ["11"]},
    "ANCHEMOLVS": {"N": ["12"]},
    "SVLPICIA": {"N": ["11"]},
    "TYRRHVS": {"N": ["12"]},
    "MENVLLA": {"N": ["11"]},
    "ISAVRICVS": {"N": ["12"]},
    "BARIVM": {"N": ["12"]},
    "METELLA": {"N": ["11"]},
    "PHAEDRIA": {"N": ["11"]},
    "ANTISTIA": {"N": ["11"]},
    "ARMILLATVS": {"-": ["21"], "N": ["12"]},
    "ANDROMEDA": {"N": ["11"]},
    "PISEVS": {"A": ["21"]},
    "VARISIDIVS": {"N": ["12"]},
    "LINGONVS": {"N": ["12"]},
    "ARNIENSIS": {"A": ["24"]},
    "AGRIPPINVS": {"N": ["12"]},
    "MEGALESIA": {"N": ["11"]},
    "EQVIRRIA": {"N": ["11"]},
    "LANARIVS": {"1": ["12"], "N": ["12"]},
    "GRATTIVS": {"N": ["12"]},
    "TOXEVS": {"N": ["12"]},
    "ARECTEVS": {"A": ["21"]},
    "MVTINENSIS": {"A": ["24"]},
    "OEAGRVS": {"N": ["12"]},
    "VSTICA": {"N": ["11"]},
    "CALVISIVS": {"N": ["12"]},
    "ORFIDIVS": {"N": ["12"]},
    "CEDROSIA": {"N": ["11"]},
    "PTOLEMAEEVS": {"A": ["21"]},
    "TITANIVS": {"A": ["21"]},
    "EPIDAVRIVS": {"A": ["21"]},
    "NIPHAEVS": {"N": ["12"]},
    "FABRICIANVS": {"A": ["21"]},
    "MEGYLLA": {"N": ["11"]},
    "EXECESTVS": {"N": ["12"]},
    "TICIDA": {"N": ["11"]},
    "SATVREIANVS": {"A": ["21"]},
    "CANIVS": {"N": ["12"]},
    "VOLANERIVS": {"N": ["12"]},
    "LARIVS": {"N": ["12"]},
    "CHARVS": {"N": ["12"]},
    "MESSAPVS": {"N": ["12"]},
    "PANTILIVS": {"N": ["12"]},
    "PHLEGRA": {"N": ["11"]},
    "VACCVS": {"N": ["12"]},
    "TALLVS": {"N": ["12"]},
    "HERCVLIVS": {"A": ["21"]},
    "PROPONTIACVS": {"A": ["21"]},
    "CORFINIVM": {"N": ["12"]},
    "PLANASIA": {"N": ["11"]},
    "CALVSIDIVS": {"N": ["12"]},
    "VITELLIANVS": {"A": ["21"]},
    "ATRATINVS": {"N": ["12"]},
    "MAEANDRIVS": {"N": ["12"], "A": ["21"]},
    "EPIDICVS": {"N": ["12"]},
    "CARSEOLANVS": {"A": ["21"]},
    "MEDEIS": {"A": ["24"]},
    "OCRICVLANVS": {"A": ["21"]},
    "RESTITVTVS": {"N": ["12"]},
    "PADVA": {"N": ["11"]},
    "VIRDIVS": {"N": ["12"]},
    "BRVTTEDIVS": {"N": ["12"]},
    "ANIVS": {"N": ["12"]},
    "AVNVS": {"N": ["12"]},
    "PHRIXEVS": {"A": ["21"]},
    "CLAVDIANVS": {"A": ["21"]},
    "ENNIANVS": {"A": ["21"]},
    "QVINTIA": {"N": ["11"]},
    "CRATERVS": {"N": ["12"]},
    "ATIA": {"N": ["11"]},
    "APVLVS": {"A": ["21"]},
    "ASELLIVS": {"N": ["12"]},
    "ILIPA": {"N": ["11"]},
    "BARBATIVS": {"N": ["12"]},
    "GERELLANVS": {"N": ["12"]},
    "ASTVRICVS": {"N": ["12"]},
    "IMBRASVS": {"N": ["12"]},
    "DVLICHIVS": {"A": ["21"]},
    "ATTIVS": {"N": ["12"]},
    "CADVRCVS": {"A": ["21"]},
    "CIMINVS": {"N": ["12"]},
    "TABVRNVS": {"N": ["12"]},
    "VIBENNA": {"N": ["11"]},
    "HAEMONIA": {"N": ["11"]},
    "PHTHIA": {"N": ["11"]},
    "CRAGVS": {"N": ["12"]},
    "MVLLINVS": {"N": ["12"]},
    "CRESIVS": {"A": ["21"]},
    "VERBIGENVS": {"N": ["12"]},
    "PANDA": {"N": ["11"]},
    "ONOMASTVS": {"N": ["12"]},
    "THEOTIMVS": {"N": ["12"]},
    "OPIMIANVM": {"N": ["12"]},
    "BOXVS": {"N": ["12"]},
    "AEACVS": {"N": ["12"]},
    "EVRYPYLVS": {"N": ["12"]},
    "DOMNILAVS": {"N": ["12"]},
    "ANCAEVS": {"N": ["12"]},
    "FAESVLANVS": {"A": ["21"]},
    "HYPERMESTRA": {"N": ["11"]},
    "CARTEIA": {"N": ["11"]},
    "ELEAZARVS": {"N": ["12"]},
    "PHIGALESIVS": {"A": ["21"]},
    "SIRIVS": {"N": ["12"], "A": ["21"]},
    "CEREALIA": {"N": ["11"]},
    "RHIPEVS": {"N": ["12"]},
    "LESBIVM": {"N": ["12"]},
    "PHARSALIVS": {"A": ["21"]},
    "ASTERIA": {"N": ["11"]},
    "VARENVS": {"N": ["12"]},
    "PROPERTIVS": {"N": ["12"]},
    "SICCA": {"N": ["11"]},
    "ATREVS": {"N": ["12"]},
    "ACHARNEVS": {"N": ["12"]},
    "VOLCANVS": {"N": ["12"]},
    "EVCAERVS": {"N": ["12"]},
    "PAEONIA": {"N": ["11"]},
    "ROMVLIVS": {"N": ["12"]},
    "ACILIA": {"N": ["11"]},
    "SOLIMVS": {"N": ["12"]},
    "PRAETEXTATA": {"N": ["11"]},
    "PLAVTINVS": {"A": ["21"]},
    "NESSEVS": {"A": ["21"]},
    "GRAECINVS": {"N": ["12"]},
    "DORICVS": {"A": ["21"]},
    "DIRCAEVS": {"A": ["21"]},
    "FALERNVS": {"A": ["21"]},
    "BOEOTIVS": {"A": ["21"]},
    "GAVILLA": {"N": ["11"]},
    "PERILLVS": {"N": ["12"]},
    "BESSA": {"N": ["11"]},
    "ACRISIVS": {"N": ["12"]},
    "XANTHVS": {"N": ["12"]},
    "MAREOTICVM": {"N": ["12"]},
    "MINOIS": {"N": ["13"], "A": ["12"]},
    "BVPALVS": {"N": ["12"]},
    "SIGEVM": {"N": ["12"]},
    "NOVIA": {"N": ["11"]},
    "VOLTINIA": {"N": ["11"]},
    "RVTVLVS": {"A": ["21"]},
    "POSTVMINA": {"N": ["11"]},
    "CARCHEDONIVS": {"A": ["21"]},
    "GRATIDIANVS": {"N": ["12"]},
    "PAESTVM": {"N": ["12"]},
    "HAEDVVS": {"A": ["21"]},
    "ARGOLIS": {"A": ["24"]},
    "SILENVS": {"N": ["12"]},
    "SICANIVS": {"A": ["21"]},
    "SEMVRIVM": {"N": ["12"]},
    "PHAEACIA": {"N": ["11"]},
    "POLYXENIVS": {"A": ["21"]},
    "THASIVS": {"A": ["21"]},
    "TRINACRIA": {"N": ["11"]},
    "SCANTINIVS": {"A": ["21"]},
    "TEDIA": {"N": ["11"]},
    "SIGEVS": {"A": ["21"]},
    "TVRPILIANVS": {"N": ["12"]},
    "BANTINVS": {"A": ["21"]},
    "IVDAICVS": {"A": ["21"]},
    "BALVENTIVS": {"N": ["12"]},
    "MAENALIVS": {"A": ["21"]},
    "GARAMANTIS": {"A": ["24"]},
    "IAVOLENVS": {"N": ["12"]},
    "PHAEACVS": {"A": ["21"]},
    "ACHELOVS": {"N": ["12"]},
    "LITAVICCVS": {"N": ["12"]},
    "SASSVRA": {"N": ["11"]},
    "THIRMIDA": {"N": ["11"]},
    "EPICHARNVS": {"N": ["12"]},
    "DANDARICA": {"N": ["11"]},
    "MAGNESSVS": {"A": ["21"]},
    "PHARSALICVS": {"A": ["21"]},
    "SELEVCENSIS": {"A": ["24"]},
    "FIRMIDIVS": {"N": ["12"]},
    "THYSDRA": {"N": ["11"]},
    "ASCYLTVS": {"N": ["12"]},
    "PISAEVS": {"A": ["21"]},
    "SPARTANVS": {"A": ["21"]},
    "ASCLEPIODOTVS": {"N": ["12"]},
    "TROICVS": {"A": ["21"]},
    "MASSIVA": {"N": ["11"]},
    "SERVIANVS": {"N": ["12"]},
    "VEIANIVS": {"N": ["12"]},
    "ASCIBVRGIVM": {"N": ["12"]},
    "ERVCIVS": {"N": ["12"]},
    "VOLANDVM": {"N": ["12"]},
    "SAVREA": {"N": ["11"]},
    "ACROCERAVNIA": {"N": ["11"]},
    "ATLANTICVS": {"A": ["21"]},
    "INACHIA": {"N": ["11"]},
    "MENELAEVS": {"A": ["21"]},
    "SVBVRA": {"N": ["11"]},
    "ANNA": {"N": ["11"]},
    "POLYXENA": {"N": ["11"]},
    "BOEOTVS": {"A": ["21"]},
    "CVRICTA": {"N": ["11"]},
    "CORSICVS": {"A": ["21"]},
    "TIBVRTINVM": {"N": ["12"]},
    "METIVS": {"N": ["12"]},
    "ASCVRVM": {"N": ["12"]},
    "CILLA": {"N": ["11"]},
    "SORRENTVM": {"N": ["12"]},
    "ERASINVS": {"N": ["12"]},
    "ORBILIVS": {"N": ["12"]},
    "VCVBENSIS": {"A": ["24"]},
    "FVLGINIVS": {"N": ["12"]},
    "ARABIVS": {"A": ["21"]},
    "PHYLLEIVS": {"A": ["21"]},
    "THYMBRAEVS": {"A": ["21"], "N": ["12"]},
    "CLEOPHANTVS": {"N": ["12"]},
    "ARRETIVM": {"N": ["12"]},
    "CHAEREA": {"N": ["11"]},
    "NYSAEVS": {"A": ["21"]},
    "ARRHIDAEVS": {"N": ["12"]},
    "VENETICVS": {"A": ["21"]},
    "PHAEDROMVS": {"N": ["12"]},
    "MESSALINVS": {"N": ["12"]},
    "TIMAEVS": {"N": ["12"]},
    "PERINTHVS": {"N": ["12"]},
    "GEDVSANVS": {"A": ["21"]},
    "SEXTIA": {"N": ["11"]},
    "NEREVS": {"N": ["12"]},
    "RAVIDVS": {"N": ["12"]},
    "AEOLIA": {"N": ["11"]},
    "PHILODORVS": {"N": ["12"]},
    "VIRIDOMARVS": {"N": ["12"]},
    "PHOSPHORVS": {"N": ["12"]},
    "CAMERTINVS": {"A": ["21"]},
    "EVHENVS": {"N": ["12"]},
    "MANDVBRACIVS": {"N": ["12"]},
    "NOCTILVCA": {"N": ["11"]},
    "ERYCINA": {"N": ["11"]},
    "PELOPEIS": {"A": ["24"]},
    "PANGAEA": {"N": ["11"]},
    "CLODIANVS": {"A": ["21"], "N": ["12"]},
    "IMBRVS": {"N": ["12"]},
    "BASILICVS": {"2": ["21"], "1": ["12"]},
    "CRVCIARIVS": {"1": ["12"]},
    "NICOSTRATVS": {"N": ["12"]},
    "PLOTIVS": {"A": ["21"], "N": ["12"]},
    "NVRSIA": {"N": ["11"]},
    "THERSIPPVS": {"N": ["12"]},
    "CARISTIA": {"N": ["11"]},
    "SITHONIVS": {"A": ["21"]},
    "ROSCIVS": {"A": ["21"], "N": ["12"]},
    "ALESIA": {"N": ["11"]},
    "CELTIBERIA": {"N": ["11"]},
    "SPVRINNA": {"N": ["11"]},
    "MATTIACVS": {"A": ["21"]},
    "VELIENSIS": {"A": ["24"]},
    "FVLCINIVS": {"N": ["12"]},
    "ARCHILOCHVS": {"N": ["12"]},
    "EVGANEVS": {"A": ["21"]},
    "SMYRNAEVS": {"A": ["21"]},
    "ASCVLVM": {"N": ["12"]},
    "ARMENIVS": {"N": ["12"], "A": ["21"]},
    "QVINTILLA": {"N": ["11"]},
    "ARISTONVS": {"N": ["12"]},
    "SATVRNALICIVS": {"A": ["21"]},
    "PELIACVS": {"A": ["21"]},
    "TIBVRTIVS": {"A": ["21"], "N": ["12"]},
    "ARCHEMORVS": {"N": ["12"]},
    "GADITANVS": {"A": ["21"]},
    "PROTEVS": {"N": ["12"]},
    "TITHONVS": {"N": ["12"]},
    "FOROIVLIENSIS": {"A": ["24"]},
    "PANCHAEVS": {"A": ["21"]},
    "POMPVLENVS": {"N": ["12"]},
    "ALCMAEVS": {"N": ["12"]},
    "CALPETVS": {"N": ["12"]},
    "MOGONTIACVM": {"N": ["12"]},
    "CILICENSIS": {"A": ["24"]},
    "ZETHVS": {"N": ["12"]},
    "CASPERIA": {"N": ["11"]},
    "BRVTTIANVS": {"N": ["12"]},
    "APVLA": {"N": ["11"]},
    "SESTIVS": {"N": ["12"]},
    "VERTISCVS": {"N": ["12"]},
    "FABVLLVS": {"N": ["12"]},
    "PLATONICVS": {"A": ["21"]},
    "VADA": {"N": ["11"]},
    "MVRRANVS": {"N": ["12"]},
    "ASOPVS": {"N": ["12"]},
    "VERGINIVS": {"N": ["12"]},
    "NESTOREVS": {"A": ["21"]},
    "FVFIDIVS": {"N": ["12"]},
    "POLYBVS": {"N": ["12"]},
    "SALTIVS": {"N": ["12"]},
    "VERRITVS": {"N": ["12"]},
    "ATRIA": {"N": ["11"]},
    "TRAPEZVS": {"N": ["12"]},
    "THERSILOCHVS": {"N": ["12"]},
    "PAVCVVIVS": {"N": ["12"]},
    "TREBIVS": {"N": ["12"]},
    "AMASTRIACVS": {"A": ["21"]},
    "PAPIRVS": {"N": ["12"]},
    "MEDIOLANVM": {"N": ["12"]},
    "VELIVS": {"N": ["12"]},
    "BROCHVBELVS": {"N": ["12"]},
    "CAELICVLVS": {"N": ["12"]},
    "ISTHMIA": {"N": ["11"]},
    "LARENTALIA": {"N": ["11"]},
    "PROCVLEIVS": {"N": ["12"]},
    "NVCERIA": {"N": ["11"]},
    "SINCAIVS": {"N": ["12"]},
    "MINICIVS": {"N": ["12"]},
    "AVZEA": {"N": ["11"]},
    "THYESTVS": {"A": ["21"]},
    "ALBVRNVS": {"N": ["12"]},
    "METAVRVM": {"N": ["12"]},
    "SATVRNIA": {"N": ["11"]},
    "VENERARIVS": {"2": ["21"], "1": ["12"]},
    "PRODICVS": {"N": ["12"]},
    "CATIENVS": {"N": ["12"]},
    "HORDEONIVS": {"N": ["12"]},
    "TRAVLVS": {"N": ["12"]},
    "LAPIDARIVS": {"2": ["21"], "1": ["12"]},
    "PARNASVS": {"N": ["12"]},
    "CALVINA": {"N": ["11"]},
    "LABICANVS": {"A": ["21"]},
    "MVCIANVS": {"N": ["12"]},
    "ANTEGVA": {"N": ["11"]},
    "SCRIBONIVS": {"N": ["12"]},
    "CANVSIVM": {"N": ["12"]},
    "DIODOTVS": {"N": ["12"]},
    "VRANIA": {"N": ["11"]},
    "DELTA": {"N": ["11"]},
    "HADRIA": {"N": ["11"]},
    "CEBENNA": {"N": ["11"]},
    "MEVIVS": {"N": ["12"]},
    "DECIDIANA": {"N": ["11"]},
    "AMPHIARAEVS": {"A": ["21"]},
    "VETTIDIVS": {"N": ["12"]},
    "CAESONIA": {"N": ["11"]},
    "ATHAMANIA": {"N": ["11"]},
    "MAXIMILLA": {"N": ["11"]},
    "ISELASTICVS": {"2": ["21"], "1": ["12"]},
    "GELA": {"N": ["11"]},
    "CERCIVS": {"N": ["12"]},
    "IDA": {"N": ["11"]},
    "POPPAEANVS": {"A": ["21"]},
    "ERYTHEA": {"N": ["11"]},
    "MANLIANVS": {"A": ["21"]},
    "AFRANIVS": {"N": ["12"]},
    "DEXAMENVS": {"N": ["12"]},
    "CASTABALVS": {"N": ["12"]},
    "CITA": {"N": ["11"]},
    "CAMERINVS": {"N": ["12"], "A": ["21"]},
    "VARINIVS": {"N": ["12"]},
    "CLOANTHVS": {"N": ["12"]},
    "APPVLEIANVS": {"A": ["21"]},
    "TEVTOBVRGIENSIS": {"A": ["24"]},
    "BONONIENSIS": {"A": ["24"]},
    "ALPHESIBOEVS": {"N": ["12"]},
    "BENACVS": {"N": ["12"]},
    "RAVENNA": {"N": ["11"]},
    "VOLATERRANVS": {"A": ["21"]},
    "ITALICENSIS": {"A": ["24"]},
    "FABRATERIA": {"N": ["11"]},
    "CLYTAEMNESTRA": {"N": ["11"]},
    "COCYTVS": {"N": ["12"]},
    "HIEROSOLYMVS": {"N": ["12"]},
    "ELEVSIVM": {"N": ["12"]},
    "VETTONIANVS": {"N": ["12"]},
    "POPPAEVS": {"N": ["12"], "A": ["21"]},
    "PIRAEVS": {"N": ["12"]},
    "PHINEVS": {"A": ["21"], "N": ["12"]},
    "CAVLIVS": {"N": ["12"]},
    "BRVTTIVS": {"A": ["21"]},
    "POLYSTRATVS": {"N": ["12"]},
    "THOANTEVS": {"A": ["21"]},
    "GLITIVS": {"N": ["12"]},
    "LVCINA": {"N": ["11"]},
    "ABGARVS": {"N": ["12"]},
    "SCYTHVS": {"A": ["21"]},
    "PYRAMVS": {"N": ["12"]},
    "VORENVS": {"N": ["12"]},
    "SARDANAPALLVS": {"N": ["12"]},
    "AQVINIVS": {"N": ["12"]},
    "HELICONIVS": {"A": ["21"]},
    "NONIANVS": {"N": ["12"]},
    "OCCIA": {"N": ["11"]},
    "VINICIANVS": {"N": ["12"]},
    "CECROPIVS": {"A": ["21"]},
    "ALLIA": {"N": ["11"]},
    "ILERDA": {"N": ["11"]},
    "RVDINVS": {"A": ["21"]},
    "BALEARIS": {"A": ["24"]},
    "TERTVLLVS": {"N": ["12"]},
    "LYRNESVS": {"N": ["12"]},
    "SERESTVS": {"N": ["12"]},
    "SEXTILIA": {"N": ["11"]},
    "PHILETAEVS": {"A": ["21"]},
    "ASTRAEA": {"N": ["11"]},
    "NINNIVS": {"N": ["12"]},
    "ATRACIVS": {"A": ["21"]},
    "LATVM": {"N": ["12"]},
    "AESOPVS": {"N": ["12"]},
    "DICTYNNA": {"N": ["11"]},
    "ICARVS": {"N": ["12"]},
    "AVSONIVS": {"A": ["21"]},
    "CALLISTVS": {"N": ["12"]},
    "VERANIOLVS": {"N": ["12"]},
    "SERVILIANVS": {"A": ["21"]},
    "NAVPORTVM": {"N": ["12"]},
    "ITONVS": {"N": ["12"]},
    "CVMAEVS": {"A": ["21"]},
    "BATHYLLVS": {"N": ["12"]},
    "CYRRVS": {"N": ["12"]},
    "CLVTORIVS": {"N": ["12"]},
    "CHRYSANTHVS": {"N": ["12"]},
    "MASSVGRADA": {"N": ["11"]},
    "TARTESIVS": {"A": ["21"]},
    "LARISAEVS": {"A": ["21"]},
    "AMERIA": {"N": ["11"]},
    "OLENIVS": {"A": ["21"]},
    "FIDIVS": {"N": ["12"]},
    "ABELLA": {"N": ["11"]},
    "PONTIA": {"N": ["11"]},
    "NERVA": {"N": ["11"]},
    "SINOPENSIS": {"A": ["24"]},
    "AMAZONIVS": {"A": ["21"]},
    "ECHIONIVS": {"A": ["21"]},
    "ZANCLAEVS": {"A": ["21"]},
    "ASTRAEVS": {"N": ["12"]},
    "AFRANIANVS": {"A": ["21"]},
    "POTITIVS": {"N": ["12"]},
    "THRAECIDICA": {"N": ["11"]},
    "VESTINVS": {"A": ["21"], "N": ["12"]},
    "MELITENA": {"N": ["11"]},
    "APPIA": {"N": ["11"]},
    "CASSANDREVS": {"N": ["12"]},
    "ATEIVS": {"N": ["12"]},
    "VIPSTANVS": {"N": ["12"]},
    "PHASMA": {"N": ["11"]},
    "DACVS": {"A": ["21"]},
    "TITANIS": {"A": ["24"], "N": ["16"]},
    "ALPHEVS": {"N": ["12"], "A": ["21"]},
    "CARMELVS": {"N": ["12"]},
    "PALAESTINVS": {"A": ["21"]},
    "TVRONVS": {"A": ["21"]},
    "SEPTIMILLVS": {"N": ["12"]},
    "MAIA": {"N": ["11"]},
    "OPPIANICVS": {"N": ["12"]},
    "TELEMVS": {"N": ["12"]},
    "DERCENNVS": {"N": ["12"]},
    "CVRTISIVS": {"N": ["12"]},
    "BEDRIACENSIS": {"A": ["24"]},
    "ARRECINVS": {"N": ["12"]},
    "NATTA": {"N": ["11"]},
    "NASTA": {"N": ["11"]},
    "PINDARICVS": {"A": ["21"]},
    "AQVILONIVS": {"A": ["21"]},
    "PRILIVS": {"N": ["12"]},
    "BOCCHVS": {"N": ["12"]},
    "PALLACINVS": {"A": ["21"]},
    "LEMONIA": {"N": ["11"]},
    "MAVRA": {"N": ["11"]},
    "FVFIANVS": {"A": ["21"]},
    "MISENENSIS": {"A": ["24"]},
    "XENOPHANTVS": {"N": ["12"]},
    "PROETVS": {"N": ["12"]},
    "CVSVS": {"N": ["12"]},
    "PARSAGADA": {"N": ["11"]},
    "CAPANEVS": {"N": ["12"]},
    "SVEDIVS": {"N": ["12"]},
    "LEARCHVS": {"N": ["12"]},
    "GRACCHANVS": {"A": ["21"]},
    "ARCHIDEMVS": {"N": ["12"]},
    "TVNNIVS": {"N": ["12"]},
    "TOLENVM": {"N": ["12"]},
    "AVFIDVS": {"N": ["12"]},
    "DICTAEVS": {"A": ["21"]},
    "BRIGANTICVS": {"N": ["12"]},
    "BELLICA": {"N": ["11"]},
    "LAENIVS": {"N": ["12"]},
    "MOPSVS": {"N": ["12"]},
    "CLATERNA": {"N": ["11"]},
    "ALERNVM": {"N": ["12"]},
    "MENSARIVS": {"1": ["12"]},
    "CATVLINVS": {"N": ["12"]},
    "ABDVCILLVS": {"N": ["12"]},
    "VATINIVS": {"N": ["12"]},
    "PITTHEIVS": {"A": ["21"]},
    "SYRIVS": {"A": ["21"]},
    "GARGILIVS": {"N": ["12"]},
    "BARDAICVS": {"A": ["21"]},
    "VSSETA": {"N": ["11"]},
    "VIPSANIVS": {"A": ["21"], "N": ["12"]},
    "ANTANDRVS": {"N": ["12"]},
    "ISAEVS": {"N": ["12"]},
    "GAVRVS": {"N": ["12"]},
    "AMYCLAEVS": {"A": ["21"]},
    "AREMORICVS": {"A": ["21"]},
    "EPICVREVS": {"A": ["21"]},
    "PACONIANVS": {"N": ["12"]},
    "HEGESARETVS": {"N": ["12"]},
    "ANTICYRA": {"N": ["11"]},
    "BOSPHORVS": {"N": ["12"]},
    "DORYLAVS": {"N": ["12"]},
    "PADAEVS": {"A": ["21"]},
    "TVLLA": {"N": ["11"]},
    "CASPERIVS": {"N": ["12"]},
    "AEETAEVS": {"A": ["21"]},
    "ABDALONYMVS": {"N": ["12"]},
    "CANTABRICVS": {"A": ["21"]},
    "AELIANVS": {"N": ["12"]},
    "QVARINVS": {"N": ["12"]},
    "SALEIVS": {"N": ["12"]},
    "SIPYLVS": {"N": ["12"]},
    "SAVFEIA": {"N": ["11"]},
    "ARGYRIPA": {"N": ["11"]},
    "AVXIMVM": {"N": ["12"]},
    "PHAEDRA": {"N": ["11"]},
    "SAMOTHRACIVS": {"A": ["21"]},
    "STOICIDA": {"N": ["11"]},
    "SILVANVS": {"N": ["12"]},
    "CASSANDRA": {"N": ["11"]},
    "SELEVCIA": {"N": ["11"]},
    "THEAETETVS": {"N": ["12"]},
    "OSTORIVS": {"N": ["12"]},
    "THALLVS": {"N": ["12"]},
    "VOLSINIENSIS": {"A": ["24"]},
    "PENTHEIVS": {"A": ["21"]},
    "SARMATIA": {"N": ["11"]},
    "CALGACVS": {"N": ["12"]},
    "ATELLANICVS": {"A": ["21"]},
    "MAVRVS": {"A": ["21"]},
    "SALMONEVS": {"N": ["12"]},
    "MELANTHIVS": {"N": ["12"]},
    "PHANAEVS": {"A": ["21"]},
    "TEIA": {"N": ["11"]},
    "VISTILIA": {"N": ["11"]},
    "VOLOGESVS": {"N": ["12"]},
    "IXIONIVS": {"A": ["21"]},
    "PELOPIVS": {"A": ["21"]},
    "MASSYLVS": {"A": ["21"]},
    "BETVVS": {"N": ["12"]},
    "ALLIENSIS": {"A": ["24"]},
    "CIMBRICVS": {"A": ["21"]},
    "CIMMERIVS": {"A": ["21"]},
    "RHOETEVS": {"A": ["21"], "N": ["12"]},
    "CORDALVS": {"N": ["12"]},
    "ERICTHONIVS": {"N": ["12"]},
    "LEMONVM": {"N": ["12"]},
    "FERENTIVM": {"N": ["12"]},
    "ARCHIDAMVS": {"N": ["12"]},
    "LAGVS": {"N": ["12"]},
    "PHOENISSA": {"N": ["11"]},
    "TRIVICVM": {"N": ["12"]},
    "APHIDNA": {"N": ["11"]},
    "ISMARVS": {"N": ["12"]},
    "TRITONIA": {"N": ["11"]},
    "LIBVRNVS": {"A": ["21"]},
    "IDOMENEVS": {"N": ["12"]},
    "PHASIACVS": {"A": ["21"]},
    "CAVARINVS": {"N": ["12"]},
    "FLORALIA": {"N": ["11"]},
    "PALLANTEVS": {"A": ["21"]},
    "NAEVA": {"N": ["11"]},
    "DEMAENETVS": {"N": ["12"]},
    "SAVFEIVS": {"N": ["12"]},
    "MANCIA": {"N": ["11"]},
    "LEBINTHVS": {"N": ["12"]},
    "THVBVRSICVS": {"A": ["21"]},
    "GENAVA": {"N": ["11"]},
    "NVCVLA": {"N": ["11"]},
    "ARETHVSIS": {"A": ["24"]},
    "PERENNA": {"N": ["11"]},
    "TALAIONIVS": {"A": ["21"]},
    "PARMENSIS": {"A": ["24"]},
    "TITYVS": {"N": ["12"]},
    "CILICIVS": {"A": ["21"]},
    "CORYTHVS": {"N": ["12"]},
    "CELEVS": {"N": ["12"]},
    "SERIPHIVS": {"A": ["21"]},
    "ARACINTHVS": {"N": ["12"]},
    "MARPESIVS": {"A": ["21"]},
    "HELORIVS": {"A": ["21"]},
    "EVDEMVS": {"N": ["12"]},
    "SILIA": {"N": ["11"]},
    "DINAEA": {"N": ["11"]},
    "SPARTACVS": {"N": ["12"]},
    "OVIVS": {"N": ["12"]},
    "OEBALIVS": {"A": ["21"]},
    "ISAVRVS": {"A": ["21"]},
    "SAGANA": {"N": ["11"]},
    "VETTIANVS": {"A": ["21"]},
    "IPHIGENIA": {"N": ["11"]},
    "PELOPEA": {"N": ["11"]},
    "ZMINTHEVS": {"A": ["21"]},
    "ENNA": {"N": ["11"]},
    "MEGAERA": {"N": ["11"]},
    "MYTILENA": {"N": ["11"]},
    "VALETIACVS": {"N": ["12"]},
    "MASVRIVS": {"N": ["12"]},
    "NOVIODVNVM": {"N": ["12"]},
    "ILIVS": {"A": ["21"]},
    "PAPHVS": {"N": ["12"]},
    "PANTOLABVS": {"N": ["12"]},
    "CYDONEVS": {"A": ["21"]},
    "BVRRIENVS": {"N": ["12"]},
    "PLVTONIVS": {"A": ["21"]},
    "LATONIGENA": {"N": ["11"]},
    "CORSICA": {"N": ["11"]},
    "CRETENSIS": {"A": ["24"]},
    "ADVATVCA": {"N": ["11"]},
    "ARMINIVS": {"N": ["12"]},
    "ARIVSIVS": {"A": ["21"]},
    "RHODOPEIVS": {"A": ["21"]},
    "THRACA": {"N": ["11"]},
    "SVSIANVS": {"A": ["21"]},
    "IBERICVS": {"A": ["21"]},
    "AMATA": {"N": ["11"]},
    "VIBIDIVS": {"N": ["12"]},
    "SARIOLENVS": {"N": ["12"]},
    "THRAESSVS": {"A": ["21"]},
    "CHIMAERA": {"N": ["11"]},
    "BALACRVS": {"N": ["12"]},
    "TVBILVSTRIA": {"N": ["11"]},
    "EVBOICVS": {"A": ["21"]},
    "BOVMELVS": {"N": ["12"]},
    "AEOLIS": {"A": ["24"], "N": ["13"]},
    "PADVS": {"N": ["12"]},
    "DAVARA": {"N": ["11"]},
    "IVDAEA": {"N": ["11"]},
    "TELEMACHVS": {"N": ["12"]},
    "EVTRAPELVS": {"N": ["12"]},
    "MAMMAEA": {"N": ["11"]},
    "CAVCASVS": {"N": ["12"]},
    "APOLLINARIS": {"N": ["13"], "A": ["12"]},
    "MYRINA": {"N": ["11"]},
    "STILONIVS": {"N": ["12"]},
    "TRIGEMINA": {"N": ["11"]},
    "VOCETIVS": {"N": ["12"]},
    "VVLCANIVS": {"A": ["21"]},
    "BASSVS": {"N": ["12"]},
    "CHAERONEA": {"N": ["11"]},
    "SATELLIVS": {"N": ["12"]},
    "AEAEVS": {"A": ["21"]},
    "GAETVLICVS": {"N": ["12"]},
    "RVFILLVS": {"N": ["12"]},
    "NORICVS": {"A": ["21"]},
    "ANYTVS": {"N": ["12"]},
    "LIGVRIA": {"N": ["11"]},
    "CANNVTIVS": {"N": ["12"]},
    "ARGVETIVS": {"N": ["12"]},
    "BASSAREVS": {"N": ["12"]},
    "TEVCRIA": {"N": ["11"]},
    "SEMELEIVS": {"A": ["21"]},
    "SERRANA": {"N": ["11"]},
    "MARDONIVS": {"N": ["12"]},
    "CAESAREVS": {"A": ["21"]},
    "ALBINOVANVS": {"N": ["12"]},
    "CLODIA": {"N": ["11"]},
    "CAMERINA": {"N": ["11"]},
    "FIBVLVS": {"N": ["12"]},
    "IDISTAVISVS": {"N": ["12"]},
    "APIDIVS": {"N": ["12"]},
    "CALEDONIA": {"N": ["11"]},
    "CAESAREA": {"N": ["11"]},
    "CLEARETA": {"N": ["11"]},
    "ADRAMYTTEVM": {"N": ["12"]},
    "CERIALIA": {"N": ["11"]},
    "MITHRIDATICVS": {"A": ["21"]},
    "SERRANVS": {"N": ["12"]},
    "CORELLIVS": {"N": ["12"]},
    "GNOSIACVS": {"A": ["21"]},
    "TARVLA": {"N": ["11"]},
    "CAMERIVS": {"N": ["12"]},
    "NARYCIVS": {"A": ["21"]},
    "PARVS": {"N": ["12"]},
    "ISEVM": {"N": ["12"]},
    "DECIMIVS": {"N": ["12"]},
    "PELASGVS": {"N": ["12"], "A": ["21"]},
    "LVSIVS": {"N": ["12"]},
    "EXTITIVS": {"N": ["12"]},
    "ATIMETVS": {"N": ["12"]},
    "CLEANTHEVS": {"A": ["21"]},
    "BELICA": {"N": ["11"]},
    "DIGENTIA": {"N": ["11"]},
    "NOVIVS": {"N": ["12"]},
    "PACALIS": {"A": ["24"]},
    "POSTVMIA": {"N": ["11"]},
    "ORONTEVS": {"A": ["21"]},
    "AVFILENVS": {"N": ["12"]},
    "AMMIVS": {"N": ["12"]},
    "CYTHEREVS": {"A": ["21"]},
    "PHILOXENVS": {"N": ["12"]},
    "ISINDIVS": {"A": ["21"]},
    "VARIANVS": {"A": ["21"]},
    "SVRENA": {"N": ["11"]},
    "SINVESSANVS": {"A": ["21"]},
    "RAMNVSIVS": {"A": ["21"]},
    "MONTANIVS": {"N": ["12"]},
    "ROMANICVS": {"A": ["21"]},
    "RVBELLIVS": {"N": ["12"]},
    "MOLOSSICVS": {"A": ["21"]},
    "CAVDINVS": {"N": ["12"], "A": ["21"]},
    "RHODVS": {"N": ["12"]},
    "SCRIBONIA": {"N": ["11"]},
    "NEMAEVS": {"A": ["21"]},
    "PAGASVS": {"N": ["12"]},
    "TVDITANVS": {"N": ["12"]},
    "IDALIVM": {"N": ["12"]},
    "PERSIVS": {"N": ["12"]},
    "OCTODVRVS": {"N": ["12"]},
    "ALBRVNA": {"N": ["11"]},
    "MENEDEMVS": {"N": ["12"]},
    "EVENVS": {"N": ["12"]},
    "ARRIONILLA": {"N": ["11"]},
    "PALICVS": {"N": ["12"]},
    "GABBA": {"N": ["11"]},
    "CLONVS": {"N": ["12"]},
    "ERSILAVS": {"N": ["12"]},
    "SVEBA": {"N": ["11"]},
    "LACAENVS": {"A": ["21"]},
    "MATIVS": {"N": ["12"]},
    "ARIADNA": {"N": ["11"]},
    "CAESILIANVS": {"N": ["12"]},
    "ACADIRA": {"N": ["11"]},
    "DORYLAEVM": {"N": ["12"]},
    "PARNASIVS": {"A": ["21"]},
    "IPHICLVS": {"N": ["12"]},
    "GELONVS": {"N": ["12"]},
    "LISCVS": {"N": ["12"]},
    "ARDA": {"N": ["11"]},
    "ALCINOVS": {"N": ["12"]},
    "CAESARIENSIS": {"A": ["24"]},
    "MELANTHEVS": {"A": ["21"]},
    "SOSIVS": {"N": ["12"]},
    "ORSILOCHVS": {"N": ["12"]},
    "THESPIVS": {"A": ["21"]},
    "PATRVITVS": {"N": ["12"]},
    "MENARCHVS": {"N": ["12"]},
    "TARPEIA": {"N": ["11"]},
    "SABINA": {"N": ["11"]},
    "BONNENSIS": {"A": ["24"]},
    "SVBLAQVEVM": {"N": ["12"]},
    "VALENTIA": {"N": ["11"]},
    "OLYNTHVS": {"N": ["12"]},
    "SAMAROBRIVA": {"N": ["11"]},
    "SCIRTVS": {"N": ["12"]},
    "GALLIVS": {"N": ["12"]},
    "ADRAMYTTENVS": {"A": ["21"]},
    "HADRVMETVM": {"N": ["12"]},
    "TVRTA": {"N": ["11"]},
    "CERCYONEVS": {"A": ["21"]},
    "CATONIVS": {"N": ["12"]},
    "MARTINA": {"N": ["11"]},
    "POEANTIVS": {"A": ["21"]},
    "LAMPADIVM": {"N": ["12"]},
    "MEGARICVS": {"A": ["21"]},
    "PHILVS": {"N": ["12"]},
    "BACTRIANVS": {"A": ["21"]},
    "TEREVS": {"N": ["12"]},
    "CICIRRVS": {"N": ["12"]},
    "MAREOTIS": {"A": ["24"], "N": ["16"]},
    "DIOXIPPVS": {"N": ["12"]},
    "LIBYSTINVS": {"A": ["21"]},
    "VIBILIVS": {"N": ["12"]},
    "EPASNACTVS": {"N": ["12"]},
    "EVMELVS": {"N": ["12"]},
    "LEMNVS": {"N": ["12"]},
    "AVIDIENVS": {"N": ["12"]},
    "OENOTHEA": {"N": ["11"]},
    "MISENA": {"N": ["11"]},
    "PITTACVS": {"N": ["12"]},
    "PVBLICIVS": {"A": ["21"], "N": ["12"]},
    "RVTILA": {"N": ["11"]},
    "AHENOBARBVS": {"N": ["12"]},
    "ALLIVS": {"N": ["12"]},
    "AMPHILOCHVS": {"N": ["12"]},
    "PERSEIS": {"A": ["24"], "N": ["16"]},
    "DEIOPEA": {"N": ["11"]},
    "HERMVNDVRVS": {"A": ["21"]},
    "BAETICVS": {"A": ["21"]},
    "AVIDIVS": {"N": ["12"]},
    "ODRYSIVS": {"A": ["21"]},
    "PHLYVS": {"N": ["12"]},
    "PIRENIDVS": {"A": ["21"]},
    "CLVVIA": {"N": ["11"]},
    "NAMMEIVS": {"N": ["12"]},
    "CYNTHIA": {"N": ["11"]},
    "FONTEIA": {"N": ["11"]},
    "ANQVILLARIA": {"N": ["11"]},
    "THALA": {"N": ["11"]},
    "TRACHALVS": {"N": ["12"]},
    "SVRRENTVM": {"N": ["12"]},
    "VMBRENVS": {"N": ["12"]},
    "ADMETVS": {"N": ["12"]},
    "DOMITIA": {"N": ["11"]},
    "HIRTVLEIVS": {"N": ["12"]},
    "PALFVRIVS": {"N": ["12"]},
    "ALEXA": {"N": ["11"]},
    "COGIDVMNVS": {"N": ["12"]},
    "ARTEMISIVS": {"N": ["12"]},
    "DANVBIVS": {"N": ["12"]},
    "SILIVS": {"N": ["12"]},
    "MARCODVRVM": {"N": ["12"]},
    "EVTYCHVS": {"N": ["12"]},
    "ENCELADVS": {"N": ["12"]},
    "CNIDIVS": {"A": ["21"]},
    "LAVRENTINVM": {"N": ["12"]},
    "PINTHIA": {"N": ["11"]},
    "MYSVS": {"A": ["21"]},
    "NONIVS": {"N": ["12"]},
    "AGONIA": {"-": ["12"], "N": ["12"]},
    "CASILINVM": {"N": ["12"]},
    "NESSVS": {"N": ["12"]},
    "MYCENA": {"N": ["11"]},
    "CAPRIVS": {"N": ["12"]},
    "ANTONINVS": {"N": ["12"]},
    "NVMANVS": {"N": ["12"]},
    "RVFINVS": {"N": ["12"]},
    "LYCVRGVS": {"N": ["12"]},
    "PETROSIDIVS": {"N": ["12"]},
    "ANTHEVS": {"N": ["12"]},
    "CELTILLVS": {"N": ["12"]},
    "PERSIS": {"A": ["24"], "N": ["16"]},
    "SEPLASIA": {"N": ["11"]},
    "BODVOGNATVS": {"N": ["12"]},
    "OLYMPIA": {"O": ["12"], "N": ["12"]},
    "PHAEDRVS": {"N": ["12"]},
    "TENCTERVS": {"A": ["21"]},
    "CEVS": {"A": ["21"]},
    "AVRIANVS": {"A": ["21"]},
    "PORSENNA": {"N": ["11"]},
    "TANAVS": {"N": ["12"]},
    "LIBYSTIS": {"A": ["24"]},
    "LVPERCALIA": {"N": ["11"]},
    "IONIVM": {"N": ["12"]},
    "LVTETIA": {"N": ["11"]},
    "NOMENTANVM": {"N": ["12"]},
    "MANILIA": {"N": ["11"]},
    "CORINNA": {"N": ["11"]},
    "ARTORIA": {"N": ["11"]},
    "ERIGYIVS": {"N": ["12"]},
    "PASIPPVS": {"N": ["12"]},
    "MAECENATIANVS": {"N": ["12"]},
    "CAMILLIANVS": {"A": ["21"]},
    "NERONIANVS": {"A": ["21"]},
    "PANDVSA": {"N": ["11"]},
    "OPSIVS": {"N": ["12"]},
    "VERVCLOETIVS": {"N": ["12"]},
    "ASVVIANVS": {"A": ["21"]},
    "TAVRIANVS": {"A": ["21"]},
    "CENABVM": {"N": ["12"]},
    "CRETHEVS": {"N": ["12"]},
    "CRABRA": {"N": ["11"]},
    "MASSILIENSIS": {"A": ["24"]},
    "PERELLIVS": {"N": ["12"]},
    "OCRESIA": {"N": ["11"]},
    "CINGONIVS": {"N": ["12"]},
    "SALVIVS": {"N": ["12"]},
    "CRESSA": {"N": ["11"], "A": ["21"]},
    "SAMNIVM": {"N": ["12"]},
    "THYESTEVS": {"A": ["21"]},
    "HELVIVS": {"N": ["12"]},
    "HADRIACVS": {"A": ["21"]},
    "PERVSIA": {"N": ["11"]},
    "DEMOPHILVS": {"N": ["12"]},
    "INDICVS": {"A": ["21"]},
    "SVFFENVS": {"N": ["12"]},
    "POLLITTA": {"N": ["11"]},
    "ALTHAEA": {"N": ["11"]},
    "AVGVRINVS": {"N": ["12"]},
    "ELECTRA": {"N": ["11"]},
    "GALERIANVS": {"N": ["12"]},
    "EVCHADIA": {"N": ["11"]},
    "LINVS": {"N": ["12"]},
    "ANDECVMBORIVS": {"N": ["12"]},
    "CASCVS": {"N": ["12"]},
    "TROIVGENA": {"N": ["11"]},
    "TASGETIVS": {"N": ["12"]},
    "THERMAEVS": {"A": ["21"]},
    "LARA": {"N": ["11"]},
    "SARDVS": {"A": ["21"]},
    "EVHIVS": {"-": ["21"], "N": ["12"]},
    "GALESVS": {"N": ["12"]},
    "PLEMVRIVM": {"N": ["12"]},
    "TECTA": {"N": ["11"]},
    "PHILARGYRVS": {"N": ["12"]},
    "AMYCVS": {"N": ["12"]},
    "BACTRIANA": {"N": ["11"]},
    "ASCVLANVS": {"A": ["21"]},
    "CVSINIVS": {"N": ["12"]},
    "SAMIARIVS": {"N": ["12"]},
    "IVNONALIS": {"A": ["24"]},
    "ORMENVS": {"N": ["12"]},
    "ERIDANVS": {"N": ["12"]},
    "FORNACALIS": {"A": ["24"]},
    "BEBRYCIVS": {"A": ["21"]},
    "ARISTONICVS": {"N": ["12"]},
    "ARMENIA": {"N": ["11"]},
    "TORIVS": {"N": ["12"]},
    "TAGVS": {"N": ["12"]},
    "GALLOGRAECVS": {"A": ["21"]},
    "VMMIDIVS": {"N": ["12"]},
    "GLYCERA": {"N": ["11"]},
    "AMYTHAONIVS": {"A": ["21"]},
    "MEVANIA": {"N": ["11"]},
    "PHILAENIVM": {"N": ["12"]},
    "DIABOLVS": {"N": ["12"]},
    "GOMPHENSIS": {"A": ["24"]},
    "CASCELLIVS": {"N": ["12"]},
    "AETIDEMVS": {"N": ["12"]},
    "MAEVIA": {"N": ["11"]},
    "PHILIPPICVS": {"A": ["21"]},
    "GRAPTVS": {"N": ["12"]},
    "VERGILIANVS": {"A": ["21"]},
    "GELDVBA": {"N": ["11"]},
    "FIRMIVS": {"N": ["12"]},
    "CIRCEVS": {"A": ["21"]},
    "NAVA": {"N": ["11"]},
    "TYRVS": {"N": ["12"]},
    "BINGIVM": {"N": ["12"]},
    "PANNONIA": {"N": ["11"]},
    "LAOMEDONTIVS": {"A": ["21"]},
    "CESTILIVS": {"N": ["12"]},
    "GORDIVM": {"N": ["12"]},
    "ALCMENA": {"N": ["11"]},
    "ANTILOCHVS": {"N": ["12"]},
    "PACIDEIVS": {"N": ["12"]},
    "HIPPOLYTVS": {"N": ["12"]},
    "DVLICHIVM": {"N": ["12"]},
    "APIDANVS": {"N": ["12"]},
    "PHOEBIGENA": {"N": ["11"]},
    "CANVSINVS": {"A": ["21"]},
    "GVLVSA": {"N": ["11"]},
    "CILNIVS": {"N": ["12"]},
    "EVXINVS": {"N": ["12"]},
    "TIGRANOCERTA": {"N": ["11"]},
    "APELLEVS": {"A": ["21"]},
    "CAVCILIVS": {"N": ["12"]},
    "THYBRINVS": {"A": ["21"]},
    "XVTHVS": {"N": ["12"]},
    "PHEGEVS": {"N": ["12"]},
    "SARMATA": {"N": ["11"]},
    "PHRYXIANVS": {"A": ["21"]},
    "OEAGRIVS": {"A": ["21"]},
    "CALLIPHANA": {"N": ["11"]},
    "APVLIA": {"N": ["11"]},
    "GAIA": {"N": ["11"]},
    "TRIVIA": {"N": ["11"]},
    "CAELIMONTANVS": {"A": ["21"]},
    "TIBERIANVS": {"A": ["21"]},
    "SAMBVS": {"N": ["12"]},
    "GEMINIVS": {"N": ["12"]},
    "ALCMAEONIVS": {"A": ["21"]},
    "PETILLIVS": {"N": ["12"]},
    "MAMVRIVS": {"N": ["12"]},
    "IDVMAEVS": {"A": ["21"]},
    "PENEIVS": {"A": ["21"]},
    "VZITTA": {"N": ["11"]},
    "AVLERCVS": {"A": ["21"]},
    "MEDIOLANENSIS": {"A": ["24"]},
    "CLITVS": {"N": ["12"]},
    "TIMARCHVS": {"N": ["12"]},
    "SINGILIENSIS": {"A": ["24"]},
    "ITALIS": {"A": ["24"]},
    "MELLA": {"N": ["11"]},
    "PEREDIA": {"N": ["11"]},
    "NVMMVLARIVS": {"1": ["12"]},
    "VIBENNIVS": {"N": ["12"]},
    "INTERAMNIVM": {"N": ["12"]},
    "ACASTVS": {"N": ["12"]},
    "ALLEDIVS": {"N": ["12"]},
    "INACHIVS": {"A": ["21"]},
    "SICANIA": {"N": ["11"]},
    "TITHONIVS": {"A": ["21"]},
    "BRITANNICVS": {"A": ["21"], "N": ["12"]},
    "THERMODONTIACVS": {"A": ["21"]},
    "ABELLANVS": {"A": ["21"]},
    "NVMIDA": {"N": ["11"]},
    "TALASSIVS": {"N": ["12"]},
    "PANCHARIA": {"N": ["11"]},
    "AMANTIA": {"N": ["11"]},
    "CYCNEIVS": {"A": ["21"]},
    "HYLLVS": {"N": ["12"]},
    "STAPHYLA": {"N": ["11"]},
    "ACHERONTICVS": {"A": ["21"]},
    "OENEVS": {"N": ["12"]},
    "EMATHIA": {"N": ["11"]},
    "HOSTILIA": {"N": ["11"]},
    "MENESTHEVS": {"N": ["12"]},
    "CINCINNATVS": {"-": ["21"], "N": ["12"]},
    "AGENOREVS": {"A": ["21"]},
    "THYONEVS": {"N": ["12"]},
    "RHODIA": {"N": ["11"]},
    "CLEOMACHVS": {"N": ["12"]},
    "CYTHNVS": {"N": ["12"]},
    "REMVLVS": {"N": ["12"]},
    "HADRIATICVM": {"N": ["12"]},
    "SEPTIMINVS": {"N": ["12"]},
    "PROMOLVS": {"N": ["12"]},
    "BAVIVS": {"N": ["12"]},
    "LATAGVS": {"N": ["12"]},
    "CALPVRNIA": {"N": ["11"]},
    "MINOIVS": {"A": ["21"]},
    "CYCLOPIVS": {"A": ["21"]},
    "CATVALDA": {"N": ["11"]},
    "BOIVS": {"A": ["21"]},
    "VACVNA": {"N": ["11"]},
    "DOMNOTAVRVS": {"N": ["12"]},
    "MVTILIA": {"N": ["11"]},
    "PETTIVS": {"N": ["12"]},
    "ERYLVS": {"N": ["12"]},
    "LEARCHEVS": {"A": ["21"]},
    "THESSALIA": {"N": ["11"]},
    "RAETICVS": {"A": ["21"]},
    "EPONA": {"N": ["11"]},
    "LERNAEVS": {"A": ["21"]},
    "SYGAMBRA": {"N": ["11"]},
    "COSSVTIANVS": {"N": ["12"]},
    "TRIARIA": {"N": ["11"]},
    "LIVINEIVS": {"N": ["12"]},
    "DOSSENNVS": {"N": ["12"]},
    "LVSCINVS": {"N": ["12"]},
    "NERVIVS": {"A": ["21"]},
    "HARPYIA": {"N": ["11"]},
    "NERINVS": {"A": ["21"]},
    "SABAEVS": {"A": ["21"]},
    "PHALAECEVS": {"A": ["21"]},
    "LYRNESIS": {"A": ["24"]},
    "TALTHYBIVS": {"N": ["12"]},
    "TARSA": {"N": ["11"]},
    "BEIRA": {"N": ["11"]},
    "HELLANICVS": {"N": ["12"]},
    "CASINA": {"N": ["11"]},
    "FLAMINIA": {"N": ["11"]},
    "STIMVLA": {"N": ["11"]},
    "MELETEVS": {"A": ["21"]},
    "AELIA": {"N": ["11"]},
    "ARISTIVS": {"N": ["12"]},
    "SCAPTENSVLA": {"N": ["11"]},
    "PRASVTAGVS": {"N": ["12"]},
    "AEGAEVM": {"N": ["12"]},
    "PERPENNA": {"N": ["11"]},
    "PERVSINVM": {"N": ["12"]},
    "SALONA": {"N": ["11"]},
    "FVLCINIANVS": {"A": ["21"]},
    "BELGIVM": {"N": ["12"]},
    "MAEONIS": {"A": ["24"], "N": ["16"]},
    "MEMPHITICVS": {"A": ["21"]},
    "CASTVLONENSIS": {"A": ["24"]},
    "POPPAEA": {"N": ["11"]},
    "ARBELA": {"N": ["11"]},
    "PARIVS": {"A": ["21"]},
    "IVRA": {"N": ["11"]},
    "CAEDICVS": {"N": ["12"]},
    "LANVVINVS": {"A": ["21"]},
    "BATVLVM": {"N": ["12"]},
    "HILAIRA": {"N": ["11"]},
    "CORDVBENSIS": {"A": ["24"]},
    "OILEVS": {"N": ["12"]},
    "TARBELLVS": {"A": ["21"]},
    "POLYDORVS": {"N": ["12"]},
    "MONA": {"N": ["11"]},
    "PHILIPPA": {"N": ["11"]},
    "PELOPEVS": {"A": ["21"]},
    "COLCHICVS": {"A": ["21"]},
    "TMOLVS": {"N": ["12"]},
    "POLLENTIA": {"-": ["11"], "N": ["12"]},
    "DIONAEVS": {"A": ["21"]},
    "HALAESVS": {"N": ["12"]},
    "VARILLA": {"N": ["11"]},
    "VIPSANIA": {"N": ["11"]},
    "AEGINA": {"N": ["11"]},
    "MOESICVS": {"A": ["21"]},
    "LYCAMBEVS": {"A": ["21"]},
    "TROGVS": {"N": ["12"]},
    "AENOBARBVS": {"N": ["12"]},
    "MANCINVS": {"N": ["12"]},
    "TITVRIVS": {"N": ["12"]},
    "VENVLVS": {"N": ["12"]},
    "PACIDEVS": {"N": ["12"]},
    "MEVVLANVS": {"N": ["12"]},
    "BROMIA": {"N": ["11"]},
    "HARPALYCVS": {"N": ["12"]},
    "CENTVMALVS": {"N": ["12"]},
    "CHRYSIPPVS": {"N": ["12"]},
    "MYCENAEVS": {"A": ["21"]},
    "PACORVS": {"N": ["12"]},
    "MARRVVIVS": {"A": ["21"]},
    "CAETRONIVS": {"N": ["12"]},
    "AVRELIA": {"N": ["11"]},
    "CALAVIVS": {"N": ["12"]},
    "TETRICA": {"N": ["11"]},
    "AVSONIS": {"A": ["24"]},
    "OCRICVLVM": {"N": ["12"]},
    "BRIXELLVM": {"N": ["12"]},
    "PIRAEA": {"N": ["11"]},
    "ADIMANTVS": {"N": ["12"]},
    "APPVS": {"N": ["12"]},
    "OVIDIVS": {"N": ["12"]},
    "OEDIPVS": {"N": ["12"]},
    "HALIVS": {"N": ["12"]},
    "ELEVSINVS": {"A": ["21"]},
    "PENEVS": {"N": ["12"]},
    "PYLADVS": {"N": ["12"]},
    "NORICVM": {"N": ["12"]},
    "LVSITANIA": {"N": ["11"]},
    "PENELEVS": {"N": ["12"]},
    "AGRIANVS": {"A": ["21"]},
    "AGRIPPINENSIS": {"A": ["24"]},
    "MELINVS": {"N": ["12"]},
    "TARQVITIVS": {"N": ["12"]},
    "CALLAICVS": {"A": ["21"]},
    "HADRIACVM": {"N": ["12"]},
    "HIEROCAESARIA": {"N": ["11"]},
    "VIBVLLIVS": {"N": ["12"]},
    "LEVCADIA": {"N": ["11"]},
    "SANCIA": {"N": ["11"]},
    "PHRVGIA": {"N": ["11"]},
    "MONYCHVS": {"N": ["12"]},
    "SATRIA": {"N": ["11"]},
    "ATEGVA": {"N": ["11"]},
    "ISSA": {"N": ["11"]},
    "CENAEVS": {"A": ["21"]},
    "TRIMERVS": {"N": ["12"]},
    "NOMENTANVS": {"N": ["12"]},
    "CARBONARIVS": {"1": ["12"]},
    "PERBIBESIA": {"N": ["11"]},
    "HALVS": {"N": ["12"]},
    "SIPONTINVS": {"A": ["21"]},
    "CAETRA": {"N": ["11"]},
    "PELLAEVS": {"A": ["21"]},
    "VELLOCATVS": {"N": ["12"]},
    "NVMANTINVS": {"A": ["21"]},
    "MAGNETIS": {"A": ["24"]},
    "AMYCLANVS": {"A": ["21"]},
    "PHILYREIVS": {"A": ["21"]},
    "FLORVS": {"N": ["12"]},
    "CVRIANVS": {"N": ["12"]},
    "DVROCORTORVM": {"N": ["12"]},
    "ARADVS": {"N": ["12"]},
    "APELLA": {"N": ["11"]},
    "LAVERNA": {"N": ["11"]},
    "OTHONIANVS": {"A": ["21"]},
    "HYRCANIVS": {"A": ["21"]},
    "MEDVLLINA": {"N": ["11"]},
    "SAGVNTVM": {"N": ["12"]},
    "ISMENIS": {"A": ["24"]},
    "VLLEXODVNVM": {"N": ["12"]},
    "ASCLEPIODORVS": {"N": ["12"]},
    "BAETVRIA": {"N": ["11"]},
    "ESQVILIVS": {"A": ["21"]},
    "LATINA": {"N": ["11"]},
    "CARNA": {"N": ["11"]},
    "LEVCOTHEA": {"N": ["11"]},
    "LYSANIA": {"N": ["11"]},
    "TERRASIDIVS": {"N": ["12"]},
    "SANGVNNIVS": {"N": ["12"]},
    "POLYAENVS": {"N": ["12"]},
    "IVTVRNA": {"N": ["11"]},
    "GRAIVGENA": {"N": ["11"]},
    "ATHENODORVS": {"N": ["12"]},
    "ARTABANVS": {"N": ["12"]},
    "ELEVS": {"A": ["21"]},
    "SACRATA": {"N": ["11"]},
    "PRAENESTINVS": {"A": ["21"]},
    "ANCHARIANVS": {"A": ["21"]},
    "ARNVS": {"N": ["12"]},
    "GYLIPPVS": {"N": ["12"]},
    "ASCRAEVS": {"A": ["21"]},
    "LIGEA": {"N": ["11"]},
    "ALBVCILLA": {"N": ["11"]},
    "LENTICVLA": {"N": ["11"]},
    "BELLEROPHONTEVS": {"A": ["21"]},
    "NICOBVLVS": {"N": ["12"]},
    "SIDONIVS": {"A": ["21"]},
    "AVFILENA": {"N": ["11"]},
    "POTITVS": {"N": ["12"]},
    "VESAEVVS": {"N": ["12"]},
    "AEGIVM": {"N": ["12"]},
    "CARSVLANVM": {"N": ["12"]},
    "IOLCVS": {"N": ["12"]},
    "CVRIONIANVS": {"A": ["21"]},
    "PHOTINVS": {"N": ["12"]},
    "ALLIFANVS": {"A": ["21"]},
    "CYTORVS": {"N": ["12"]},
    "SCYRVS": {"N": ["12"]},
    "HIPPODAMIA": {"N": ["11"]},
    "CVPIDINEVS": {"A": ["21"]},
    "EPAPHRODITVS": {"N": ["12"]},
    "TVTILIVS": {"N": ["12"]},
    "PHILOSITVS": {"N": ["12"]},
    "RHODANVS": {"N": ["12"]},
    "AMENANVS": {"A": ["21"]},
    "PLAVTVS": {"N": ["12"]},
    "AMOEBEVS": {"N": ["12"]},
    "CALLIDROMVS": {"N": ["12"]},
    "INTERAMNANVS": {"A": ["21"]},
    "TEGEEVS": {"A": ["21"]},
    "SATVRNIVS": {"A": ["21"]},
    "STROLVS": {"N": ["12"]},
    "ERECTHEVS": {"A": ["21"]},
    "HERCYNIVS": {"A": ["21"]},
    "CAEDICIA": {"N": ["11"]},
    "BARCANVS": {"A": ["21"]},
    "CADRA": {"N": ["11"]},
    "LISSVS": {"N": ["12"]},
    "SOSIA": {"N": ["11"]},
    "PARAETONIVM": {"N": ["12"]},
    "RECTINA": {"N": ["11"]},
    "TITHONIA": {"N": ["11"]},
    "BITTIVS": {"N": ["12"]},
    "AGESILAVS": {"N": ["12"]},
    "VALERVS": {"N": ["12"]},
    "ARABIA": {"N": ["11"]},
    "CLANIVS": {"N": ["12"]},
    "LACINIVS": {"A": ["21"]},
    "POENIVS": {"N": ["12"]},
    "ACHELOIVS": {"A": ["21"]},
    "CYMAEVS": {"A": ["21"]},
    "AGAMEMNONIVS": {"A": ["21"]},
    "QVIRINIANVS": {"A": ["21"]},
    "VILLIVS": {"N": ["12"]},
    "IOCASTA": {"N": ["11"]},
    "THEODOSIVS": {"N": ["12"]},
    "BEROEA": {"N": ["11"]},
    "CAIETA": {"N": ["11"]},
    "VMMIDIA": {"N": ["11"]},
    "TVBERTVS": {"N": ["12"]},
    "SETINVS": {"A": ["21"]},
    "MARSYA": {"N": ["11"]},
    "LVCVLLIANVS": {"A": ["21"]},
    "ITALICA": {"N": ["11"]},
    "ARVIRAGVS": {"N": ["12"]},
    "ARCHIBVLVS": {"N": ["12"]},
    "SATVRIVS": {"N": ["12"]},
    "AVILLIVS": {"N": ["12"]},
    "CIRCAEVS": {"A": ["21"]},
    "LATMIVS": {"A": ["21"]},
    "VAGA": {"N": ["11"]},
    "EDONVS": {"A": ["21"]},
    "POSTVMINVS": {"N": ["12"]},
    "AFINIVS": {"N": ["12"]},
    "AVTOLYCVS": {"N": ["12"]},
    "DINOMACHVS": {"N": ["12"]},
    "VELLEVS": {"N": ["12"]},
    "TIGVRINVS": {"A": ["21"]},
    "ANTIOPA": {"N": ["11"]},
    "SIGEIVS": {"A": ["21"]},
    "NVMIDIA": {"N": ["11"]},
    "SERGIOLVS": {"N": ["12"]},
    "ORTYGIA": {"N": ["11"]},
    "BVCCILLVS": {"N": ["12"]},
    "CALESTRIVS": {"N": ["12"]},
    "LENTINVS": {"N": ["12"]},
    "LVPERCVS": {"N": ["12"]},
    "BVCEPHALA": {"N": ["11"]},
    "INGVIOMERVS": {"N": ["12"]},
    "LYCISCA": {"N": ["11"]},
    "ASSARACVS": {"N": ["12"]},
    "EREBVS": {"N": ["12"]},
    "MELAMPVS": {"N": ["12"]},
    "ALABANDA": {"N": ["11"]},
    "CORYPHAEVS": {"N": ["12"]},
    "LATOVS": {"N": ["12"]},
    "LAELIA": {"N": ["11"]},
    "ETHYMANTVS": {"N": ["12"]},
    "CYDONIVS": {"A": ["21"]},
    "PELORVS": {"N": ["12"]},
    "RABIRIVS": {"N": ["12"]},
    "PENNVS": {"N": ["12"]},
    "EPHYRAEVS": {"A": ["21"]},
    "APRONIA": {"N": ["11"]},
    "SVLPICIANVS": {"A": ["21"]},
    "ARTAXATA": {"N": ["11"]},
    "AGEDINCVM": {"N": ["12"]},
    "PROCVLA": {"N": ["11"]},
    "GALAESVS": {"N": ["12"]},
    "CASPIVS": {"A": ["21"]},
    "MENANDREVS": {"A": ["21"]},
    "VASIVS": {"N": ["12"]},
    "SVRRENTINVM": {"N": ["12"]},
    "EBVSVS": {"N": ["12"]},
    "MISENVM": {"N": ["12"]},
    "ANCHISEVS": {"A": ["21"]},
    "POMPEIA": {"N": ["11"]},
    "CEPARIVS": {"N": ["12"]},
    "CAMILLA": {"N": ["11"]},
    "PACCIVS": {"N": ["12"]},
    "CYRNEVS": {"A": ["21"]},
    "MEDVSAEVS": {"A": ["21"]},
    "AMASENVS": {"N": ["12"]},
    "GAVIANVS": {"N": ["12"]},
    "CRVCISALVS": {"N": ["12"]},
    "TVRPILIVS": {"N": ["12"]},
    "PEDIATIA": {"N": ["11"]},
    "TISIDIVM": {"N": ["12"]},
    "DIDYMVS": {"N": ["12"]},
    "TAVRICVS": {"A": ["21"]},
    "PYRRIA": {"N": ["11"]},
    "NEMEA": {"N": ["11"]},
    "BIVIVS": {"-": ["21"], "N": ["12"]},
    "THESSALICVS": {"A": ["21"]},
    "CRITOBVLVS": {"N": ["12"]},
    "PINDVS": {"N": ["12"]},
    "TRITONIVS": {"A": ["21"]},
    "IVLEVS": {"A": ["21"]},
    "PACIAECVS": {"N": ["12"]},
    "NAEVOLVS": {"N": ["12"]},
    "CAPENVS": {"A": ["21"]},
    "COSMVS": {"N": ["12"]},
    "APHIDNVS": {"N": ["12"]},
    "MVNDENSIS": {"A": ["24"]},
    "THALIA": {"N": ["11"]},
    "BELGICA": {"N": ["11"]},
    "BANDVSIA": {"N": ["11"]},
    "BLITIVS": {"N": ["12"]},
    "NEAEREA": {"N": ["11"]},
    "SABINVM": {"N": ["12"]},
    "PVPIVS": {"N": ["12"]},
    "MAGETOBRIGA": {"N": ["11"]},
    "PISA": {"N": ["11"]},
    "VESCINVS": {"A": ["21"]},
    "VRBINVM": {"N": ["12"]},
    "MNASYLVS": {"N": ["12"]},
    "MAZACA": {"N": ["11"]},
    "ANTIMACHVS": {"N": ["12"]},
    "TREBATIVS": {"N": ["12"]},
    "NOLANVS": {"A": ["21"]},
    "IPSITHILLA": {"N": ["11"]},
    "ARRVNTIVS": {"N": ["12"]},
    "CARANVS": {"N": ["12"]},
    "EGNATVLEIVS": {"N": ["12"]},
    "NICOMACHVS": {"N": ["12"]},
    "TVLLIANVM": {"N": ["12"]},
    "BASILVS": {"N": ["12"]},
    "PORTVNVS": {"N": ["12"]},
    "CORSVS": {"A": ["21"]},
    "TVRIVS": {"N": ["12"]},
    "THERAPONTIGONVS": {"N": ["12"]},
    "ASICIVS": {"N": ["12"]},
    "RHIDAGNVS": {"N": ["12"]},
    "GYARA": {"N": ["11"]},
    "THEOPOMPVS": {"N": ["12"]},
    "SVSA": {"N": ["11"]},
    "CLVENTIANVS": {"A": ["21"]},
    "VERONA": {"N": ["11"]},
    "CYLLENEVS": {"A": ["21"]},
    "SVSAGVS": {"N": ["12"]},
    "VEIENTANVS": {"A": ["21"]},
    "IBERIA": {"N": ["11"]},
    "ACERRONIVS": {"N": ["12"]},
    "EVPHOLIVS": {"N": ["12"]},
    "PELEVS": {"N": ["12"]},
    "PETRONIVS": {"N": ["12"]},
    "ATTA": {"N": ["11"]},
    "SIPHNVS": {"N": ["12"]},
    "LARISA": {"N": ["11"]},
    "MALACA": {"N": ["11"]},
    "AQVINVM": {"N": ["12"]},
    "SARMATICVS": {"A": ["21"]},
    "LESBIA": {"N": ["11"]},
    "ABDVS": {"N": ["12"]},
    "SILENA": {"N": ["11"]},
    "TIBVLLVS": {"N": ["12"]},
    "PYTHODORVS": {"N": ["12"]},
    "ANTIOCHINVS": {"A": ["21"]},
    "PANTVLEIVS": {"N": ["12"]},
    "MYLASA": {"N": ["11"]},
    "THRESSVS": {"A": ["21"]},
    "ARABICVS": {"A": ["21"]},
    "SENIVS": {"A": ["21"]},
    "GNATIA": {"N": ["11"]},
    "ACONTEVS": {"N": ["12"]},
    "INACHVS": {"N": ["12"]},
    "VATINIANVS": {"A": ["21"]},
    "CAVCASIVS": {"A": ["21"]},
    "PLEVRATVS": {"N": ["12"]},
    "CARIA": {"N": ["11"]},
    "ARDVENNA": {"N": ["11"]},
    "TEVTOMATVS": {"N": ["12"]},
    "SMINTHEVS": {"N": ["12"]},
    "ASSYRIVS": {"A": ["21"]},
    "LIGARIVS": {"N": ["12"]},
    "STRATVS": {"N": ["12"]},
    "GNOSIA": {"N": ["11"]},
    "NOLA": {"N": ["11"]},
    "CANOPVS": {"N": ["12"]},
    "TELEPHVS": {"N": ["12"]},
    "TICINVM": {"N": ["12"]},
    "AMPHIARAVS": {"N": ["12"]},
    "EPIDAVRVM": {"N": ["12"]},
    "VACALVS": {"N": ["12"]},
    "CYMODOCEA": {"N": ["11"]},
    "BESSVS": {"N": ["12"]},
    "GARGANVS": {"N": ["12"]},
    "OCRICVLANVM": {"N": ["12"]},
    "SOSIANVS": {"N": ["12"]},
    "CENSORINVS": {"N": ["12"]},
    "DECIDIVS": {"N": ["12"]},
    "ATTIA": {"N": ["11"]},
    "AVRIA": {"N": ["11"]},
    "MACHAONIVS": {"A": ["21"]},
    "AMERINVS": {"A": ["21"]},
    "ARGIVS": {"N": ["12"]},
    "VMBRICIVS": {"N": ["12"]},
    "CLVVIENVS": {"N": ["12"]},
    "ARISTARCHVS": {"N": ["12"]},
    "MALLOVENDVS": {"N": ["12"]},
    "MVNDA": {"N": ["11"]},
    "ARCHIACVS": {"A": ["21"]},
    "CRETIS": {"A": ["24"]},
    "LYDIA": {"N": ["11"]},
    "ARGANTHVS": {"N": ["12"]},
    "MAENALVS": {"N": ["12"]},
    "AONIVS": {"A": ["21"]},
    "LEMVRIA": {"N": ["11"]},
    "CELTAE": {"N": ["11"]},
    "NVMIDAE": {"N": ["11"]},
    "FORMIAE": {"N": ["11"]},
    "SVIONAE": {"N": ["11"]},
    "VOLCAE": {"N": ["11"]},
    "ARSACIDAE": {"N": ["11"]},
    "VERGILIAE": {"N": ["11"]},
    "CYRENAE": {"N": ["11"]},
    "SAVROMATAE": {"N": ["11"]},
    "RVFRAE": {"N": ["11"]},
    "PHARITAE": {"N": ["11"]},
    "ARVAE": {"N": ["11"]},
    "LAPITHAE": {"N": ["11"]},
    "SALONAE": {"N": ["11"]},
    "NERSAE": {"N": ["11"]},
    "CARSVLAE": {"N": ["11"]},
    "CLVVIAE": {"N": ["11"]},
    "CYANEAE": {"N": ["11"]},
    "AENEADAE": {"N": ["11"]},
    "SYBARITAE": {"N": ["11"]},
    "STABIAE": {"N": ["11"]},
    "AEGEAE": {"N": ["11"]},
    "ORNYTVS": {"N": ["12"]},
    "CARINAE": {"N": ["11"]},
    "DAHAE": {"N": ["11"]},
    "BORYSTHENIDAE": {"N": ["11"]},
    "GORTVAE": {"N": ["11"]},
    "NAPAEAE": {"N": ["11"]},
    "BOVILLAE": {"N": ["11"]},
    "ANTENORIDAE": {"N": ["11"]},
    "IVGVLAE": {"N": ["11"]},
    "MINYAE": {"N": ["11"]},
    "MYCENAE": {"N": ["11"]},
    "VLVBRAE": {"N": ["11"]},
    "BISALTAE": {"N": ["11"]},
    "PHASELITAE": {"N": ["11"]},
    "BASTARNAE": {"N": ["11"]},
    "LYNCESTAE": {"N": ["11"]},
    "CELAENAE": {"N": ["11"]},
    "CIETAE": {"N": ["11"]},
    "AVSONIDAE": {"N": ["11"]},
    "PARAPAMISADAE": {"N": ["11"]},
    "SARMATAE": {"N": ["11"]},
    "BRANCHIDAE": {"N": ["11"]},
    "ORESTAE": {"N": ["11"]},
    "LIBVRNAE": {"N": ["11"]},
    "FAESVLAE": {"N": ["11"]},
    "BELITAE": {"N": ["11"]},
    "THESIDAE": {"N": ["11"]},
    "VOLATERRAE": {"N": ["11"]},
    "TEGEATAE": {"N": ["11"]},
    "MAZAGAE": {"N": ["11"]},
    "EVMOLPIDAE": {"N": ["11"]},
    "BELGAE": {"N": ["11"]},
    "ARGINVSAE": {"N": ["11"]},
    "OXIONAE": {"N": ["11"]},
    "OCHVS": {"N": ["12"]},
    "CLVPEAE": {"N": ["11"]},
    "TELEBOAE": {"N": ["11"]},
    "THERMOPYLAE": {"N": ["11"]},
    "CLEONAE": {"N": ["11"]},
    "DANAE": {"N": ["11"]},
    "ARCHELAVS": {"N": ["12"]},
    "SVDRACAE": {"N": ["11"]},
    "PLATAEAE": {"N": ["11"]},
    "PYLAE": {"N": ["11"]},
    "SABARCAE": {"N": ["11"]},
    "CANNAE": {"N": ["11"]},
    "SAMARITAE": {"N": ["11"]},
    "ODRYSAE": {"N": ["11"]},
    "PISAVRVM": {"N": ["12"]},
    "EVERGETAE": {"N": ["11"]},
    "PIRVSTAE": {"N": ["11"]},
    "AEGINETAE": {"N": ["11"]},
    "TYRRHIDAE": {"N": ["11"]},
    "MINTVRNAE": {"N": ["11"]},
    "PHOENISSAE": {"N": ["11"]},
    "SCAEAE": {"N": ["11"]},
    "GORNEAE": {"N": ["11"]},
    "DALMATAE": {"N": ["11"]},
    "OEBALIDAE": {"N": ["11"]},
    "PETREIANVS": {"A": ["21"]},
    "ARGONAVTAE": {"N": ["11"]},
    "THEBAE": {"N": ["11"]},
    "HELVECONAE": {"N": ["11"]},
    "HERACLEOTAE": {"N": ["11"]},
    "CECROPIDAE": {"N": ["11"]},
    "ANTEMNAE": {"N": ["11"]},
    "TABAE": {"N": ["11"]},
    "DENSELETAE": {"N": ["11"]},
    "SABINAE": {"N": ["11"]},
    "ALOIDAE": {"N": ["11"]},
    "BAPTAE": {"N": ["11"]},
    "GRAVISCAE": {"N": ["11"]},
    "MYLAE": {"N": ["11"]},
    "CERCETAE": {"N": ["11"]},
    "CRANAE": {"N": ["11"]},
    "ALAVDAE": {"N": ["11"]},
    "MASSAGETAE": {"N": ["11"]},
    "LVRIVS": {"N": ["12"]},
    "GEMONIAE": {"N": ["11"]},
    "ACERRAE": {"N": ["11"]},
    "HORITAE": {"N": ["11"]},
    "ARABITAE": {"N": ["11"]},
    "CENTVMCELLAE": {"N": ["11"]},
    "CALYDNAE": {"N": ["11"]},
    "GANGARIDAE": {"N": ["11"]},
    "PISAE": {"N": ["11"]},
    "APHIDNAE": {"N": ["11"]},
    "COELALETAE": {"N": ["11"]},
    "TYNDARIDAE": {"N": ["11"]},
    "VERCELLAE": {"N": ["11"]},
    "DRANGAE": {"N": ["11"]},
    "ROMVLIDAE": {"N": ["11"]},
    "BIAE": {"N": ["11"]},
    "CANOPITAE": {"N": ["11"]},
    "SACAE": {"N": ["11"]},
    "GALATAE": {"N": ["11"]},
    "CASPIAE": {"N": ["11"]},
    "CAPREAE": {"N": ["11"]},
    "CLAZOMENAE": {"N": ["11"]},
    "ONCHAE": {"N": ["11"]},
    "PASIPHAE": {"N": ["11"]},
    "PISIDAE": {"N": ["11"]},
    "MAEONIDAE": {"N": ["11"]},
    "AMYCLAE": {"N": ["11"]},
    "DANDARIDAE": {"N": ["11"]},
    "DANAIDAE": {"N": ["11"]},
    "PHOCAEI": {"N": ["12"]},
    "TRIBALLI": {"N": ["12"]},
    "SYCAMBRI": {"N": ["12"]},
    "VBII": {"N": ["12"]},
    "TERMESTINI": {"N": ["12"]},
    "ALPINI": {"N": ["12"]},
    "HIBERI": {"N": ["12"]},
    "FAVNI": {"N": ["12"]},
    "ARGOLICI": {"N": ["12"]},
    "TIANI": {"N": ["12"]},
    "CHAMAVI": {"N": ["12"]},
    "ARECOMICI": {"N": ["12"]},
    "HERMVNDVRI": {"N": ["12"]},
    "BRITANNI": {"N": ["12"]},
    "POMPEII": {"N": ["12"]},
    "DII": {"N": ["12"]},
    "AVSCI": {"N": ["12"]},
    "COMMAGENI": {"N": ["12"]},
    "THVRII": {"N": ["12"]},
    "MELDI": {"N": ["12"]},
    "SEDOCHEZI": {"N": ["12"]},
    "CALETI": {"N": ["12"]},
    "EPICVREI": {"N": ["12"]},
    "GRAI": {"N": ["12"]},
    "HAMMONII": {"N": ["12"]},
    "SAPAEI": {"N": ["12"]},
    "AGATHYRSI": {"N": ["12"]},
    "SABELLI": {"N": ["12"]},
    "PARTHIENI": {"N": ["12"]},
    "CINITHII": {"N": ["12"]},
    "VXII": {"N": ["12"]},
    "NOLANI": {"N": ["12"]},
    "HELVII": {"N": ["12"]},
    "BRVNDISINI": {"N": ["12"]},
    "ADVATICI": {"N": ["12"]},
    "MALLI": {"N": ["12"]},
    "TAVRINI": {"N": ["12"]},
    "OSISMI": {"N": ["12"]},
    "MATTIACI": {"N": ["12"]},
    "TVRONI": {"N": ["12"]},
    "SEGVSIAVI": {"N": ["12"]},
    "AVLERCI": {"N": ["12"]},
    "SVLCITANI": {"N": ["12"]},
    "TVLINGI": {"N": ["12"]},
    "VELLAVII": {"N": ["12"]},
    "SITHONII": {"N": ["12"]},
    "DARDANI": {"N": ["12"]},
    "PERGAMENI": {"N": ["12"]},
    "PRAESTI": {"N": ["12"]},
    "DECANGI": {"N": ["12"]},
    "MARSACI": {"N": ["12"]},
    "CARALITANI": {"N": ["12"]},
    "TVNGRI": {"N": ["12"]},
    "BOTTIAEI": {"N": ["12"]},
    "TEVTONI": {"N": ["12"]},
    "MAGI": {"N": ["12"]},
    "RIPAEI": {"N": ["12"]},
    "PHILADELPHENI": {"N": ["12"]},
    "ACYLLITANI": {"N": ["12"]},
    "PARTHINI": {"N": ["12"]},
    "AFRANIANI": {"N": ["12"]},
    "CADVRCI": {"N": ["12"]},
    "SVEVI": {"N": ["12"]},
    "GRAECVLI": {"N": ["12"]},
    "ARACHOSII": {"N": ["12"]},
    "CHASVARII": {"N": ["12"]},
    "VSIPI": {"N": ["12"]},
    "MARSI": {"N": ["12"]},
    "MARRVCINI": {"N": ["12"]},
    "TELEBOI": {"N": ["12"]},
    "VATINIANI": {"N": ["12"]},
    "SANTONI": {"N": ["12"]},
    "THERAEI": {"N": ["12"]},
    "GELONI": {"N": ["12"]},
    "MAGYNI": {"N": ["12"]},
    "CASSI": {"N": ["12"]},
    "REATINI": {"N": ["12"]},
    "MENAPII": {"N": ["12"]},
    "FABRATERNI": {"N": ["12"]},
    "CHALDEI": {"N": ["12"]},
    "TERGESTINI": {"N": ["12"]},
    "MAVRI": {"N": ["12"]},
    "MARSIGNI": {"N": ["12"]},
    "CENOMANI": {"N": ["12"]},
    "PTIANII": {"N": ["12"]},
    "TREVIRI": {"N": ["12"]},
    "ARIMASPI": {"N": ["12"]},
    "MOSCHI": {"N": ["12"]},
    "LEMOVII": {"N": ["12"]},
    "POMETII": {"N": ["12"]},
    "SYRII": {"N": ["12"]},
    "MEGARICI": {"N": ["12"]},
    "PYRRHONEI": {"N": ["12"]},
    "ELEVTETI": {"N": ["12"]},
    "ANGRIVARII": {"N": ["12"]},
    "PRAESTVTIANI": {"N": ["12"]},
    "MANII": {"N": ["12"]},
    "LARISAEI": {"N": ["12"]},
    "ASSYRII": {"N": ["12"]},
    "VINDELICI": {"N": ["12"]},
    "SARDIANI": {"N": ["12"]},
    "VELIOCASSI": {"N": ["12"]},
    "LANGOBARDI": {"N": ["12"]},
    "HAEDVI": {"N": ["12"]},
    "ALEXANDRINI": {"N": ["12"]},
    "GEIDVMNI": {"N": ["12"]},
    "ARVERNI": {"N": ["12"]},
    "AEGYPTII": {"N": ["12"]},
    "CHALCEDONII": {"N": ["12"]},
    "GALLOGRAECI": {"N": ["12"]},
    "ILII": {"N": ["12"]},
    "MEDIOMATRICI": {"N": ["12"]},
    "HESPERII": {"N": ["12"]},
    "DACI": {"N": ["12"]},
    "AVSETANI": {"N": ["12"]},
    "PELASGI": {"N": ["12"]},
    "GORDYAEI": {"N": ["12"]},
    "COLOPHONII": {"N": ["12"]},
    "PLATONICI": {"N": ["12"]},
    "PETELINI": {"N": ["12"]},
    "VERAGRI": {"N": ["12"]},
    "AMANTINI": {"N": ["12"]},
    "SIRACI": {"N": ["12"]},
    "CAMPANI": {"N": ["12"]},
    "ITALI": {"N": ["12"]},
    "OPICI": {"N": ["12"]},
    "MARDI": {"N": ["12"]},
    "MOSSYNI": {"N": ["12"]},
    "VOLATERRANI": {"N": ["12"]},
    "SABAEI": {"N": ["12"]},
    "NVMANTINI": {"N": ["12"]},
    "BOEOTI": {"N": ["12"]},
    "LIBVRNI": {"N": ["12"]},
    "ITVRAEI": {"N": ["12"]},
    "FENNI": {"N": ["12"]},
    "FIRMANI": {"N": ["12"]},
    "OLYMPENI": {"N": ["12"]},
    "CONDRVSI": {"N": ["12"]},
    "CIRCEII": {"N": ["12"]},
    "CLAVDIOPOLITANI": {"N": ["12"]},
    "TIGVRINI": {"N": ["12"]},
    "ESSVBII": {"N": ["12"]},
    "COLCHI": {"N": ["12"]},
    "ANNICERII": {"N": ["12"]},
    "SALAPINI": {"N": ["12"]},
    "ALLOBROGICI": {"N": ["12"]},
    "CEDROSII": {"N": ["12"]},
    "SVEBI": {"N": ["12"]},
    "OSI": {"N": ["12"]},
    "MORINI": {"N": ["12"]},
    "BOII": {"N": ["12"]},
    "PANNONII": {"N": ["12"]},
    "BAETICI": {"N": ["12"]},
    "BESSI": {"N": ["12"]},
    "CVGERNI": {"N": ["12"]},
    "LVSITANI": {"N": ["12"]},
    "AFRI": {"N": ["12"]},
    "GABALI": {"N": ["12"]},
    "SEDVNI": {"N": ["12"]},
    "CORSI": {"N": ["12"]},
    "LEVCOSYRI": {"N": ["12"]},
    "LANVVINI": {"N": ["12"]},
    "INDI": {"N": ["12"]},
    "MOSTENI": {"N": ["12"]},
    "CAEROSI": {"N": ["12"]},
    "PARENTANI": {"N": ["12"]},
    "SOLONI": {"N": ["12"]},
    "IVDAEI": {"N": ["12"]},
    "SOLYMI": {"N": ["12"]},
    "ANAGNINI": {"N": ["12"]},
    "NABATAEI": {"N": ["12"]},
    "CERAVNII": {"N": ["12"]},
    "SVSIANI": {"N": ["12"]},
    "SICANI": {"N": ["12"]},
    "SOGDIANI": {"N": ["12"]},
    "TYRII": {"N": ["12"]},
    "VRII": {"N": ["12"]},
    "ACHAICI": {"N": ["12"]},
    "LEVCI": {"N": ["12"]},
    "LVGII": {"N": ["12"]},
    "REMI": {"N": ["12"]},
    "IBERI": {"N": ["12"]},
    "LEPTITANI": {"N": ["12"]},
    "CHATTI": {"N": ["12"]},
    "CIMBRI": {"N": ["12"]},
    "GARVMNI": {"N": ["12"]},
    "THESVNTI": {"N": ["12"]},
    "IADESTINI": {"N": ["12"]},
    "SOLI": {"N": ["12"]},
    "PECOLI": {"N": ["12"]},
    "FRENTRANI": {"N": ["12"]},
    "APAMENI": {"N": ["12"]},
    "SARDI": {"N": ["12"]},
    "HYPAEPENI": {"N": ["12"]},
    "SIMVI": {"N": ["12"]},
    "TEMNII": {"N": ["12"]},
    "SIBI": {"N": ["12"]},
    "AEDVI": {"N": ["12"]},
    "NERVII": {"N": ["12"]},
    "CALAGVRRITANI": {"N": ["12"]},
    "SALENTINI": {"N": ["12"]},
    "CENIMAGNI": {"N": ["12"]},
    "IVLIOPOLITANI": {"N": ["12"]},
    "GAETVLI": {"N": ["12"]},
    "HELISII": {"N": ["12"]},
    "QVADI": {"N": ["12"]},
    "CAEMANI": {"N": ["12"]},
    "AMBARRI": {"N": ["12"]},
    "PVTEOLANI": {"N": ["12"]},
    "BRVTTII": {"N": ["12"]},
    "SEDVSII": {"N": ["12"]},
    "PHILIPPI": {"N": ["12"]},
    "AMBIVARETI": {"N": ["12"]},
    "LIBVI": {"N": ["12"]},
    "CHAVCI": {"N": ["12"]},
    "SALAMINII": {"N": ["12"]},
    "AMBIBARII": {"N": ["12"]},
    "MACROBII": {"N": ["12"]},
    "SEQVANI": {"N": ["12"]},
    "RVGII": {"N": ["12"]},
    "LATOBRIGI": {"N": ["12"]},
    "MOESICI": {"N": ["12"]},
    "RHOXOLANI": {"N": ["12"]},
    "ATII": {"N": ["12"]},
    "DOMITIANI": {"N": ["12"]},
    "MYNDII": {"N": ["12"]},
    "OSCI": {"N": ["12"]},
    "TENCTHERI": {"N": ["12"]},
    "VENETHI": {"N": ["12"]},
    "BELLOVACI": {"N": ["12"]},
    "MVSICANI": {"N": ["12"]},
    "GOMPHI": {"N": ["12"]},
    "CYRENAICI": {"N": ["12"]},
    "CERCINITANI": {"N": ["12"]},
    "PLEVMOXII": {"N": ["12"]},
    "STYMPHALIDAE": {"N": ["11"]},
    "TAVRI": {"N": ["12"]},
    "PYTHAGOREI": {"N": ["12"]},
    "VOLSINII": {"N": ["12"]},
    "FERENTANI": {"N": ["12"]},
    "ADIABENI": {"N": ["12"]},
    "VNELLI": {"N": ["12"]},
    "LEXOVII": {"N": ["12"]},
    "TAPVRI": {"N": ["12"]},
    "LEPIDANI": {"N": ["12"]},
    "GALILAEI": {"N": ["12"]},
    "IDAEI": {"N": ["12"]},
    "SVNVCI": {"N": ["12"]},
    "DELIACI": {"N": ["12"]},
    "CRVSTVMERI": {"N": ["12"]},
    "GABINIANI": {"N": ["12"]},
    "ANGLII": {"N": ["12"]},
    "ARGIVI": {"N": ["12"]},
    "IVLIANI": {"N": ["12"]},
    "LOCRI": {"N": ["12"]},
    "CHERVSCI": {"N": ["12"]},
    "CASSIANI": {"N": ["12"]},
    "AETOLI": {"N": ["12"]},
    "AVSONII": {"N": ["12"]},
    "GOLGI": {"N": ["12"]},
    "GAMBRIVII": {"N": ["12"]},
    "TRANSRHENANI": {"N": ["12"]},
    "COSSAEI": {"N": ["12"]},
    "HYRCANI": {"N": ["12"]},
    "AMERINI": {"N": ["12"]},
    "PANHORMITANI": {"N": ["12"]},
    "TYRRHENI": {"N": ["12"]},
    "POMEIANI": {"N": ["12"]},
    "ADELPHI": {"N": ["12"]},
    "COTINI": {"N": ["12"]},
    "IGVINI": {"N": ["12"]},
    "MEDI": {"N": ["12"]},
    "MANIMI": {"N": ["12"]},
    "TVSCVLANI": {"N": ["12"]},
    "CAESARIANI": {"N": ["12"]},
    "VANDILII": {"N": ["12"]},
    "VARINI": {"N": ["12"]},
    "SMYRNAEI": {"N": ["12"]},
    "ICENI": {"N": ["12"]},
    "GEMELLI": {"N": ["12"]},
    "PANNONICI": {"N": ["12"]},
    "TENII": {"N": ["12"]},
    "TAPSITANI": {"N": ["12"]},
    "CIMMERII": {"N": ["12"]},
    "HELVETII": {"N": ["12"]},
    "AFRICANI": {"N": ["12"]},
    "SOSII": {"N": ["12"]},
    "CELTIBERI": {"N": ["12"]},
    "CARMANI": {"N": ["12"]},
    "AVERNI": {"N": ["12"]},
    "VOCONTII": {"N": ["12"]},
    "CHORASMII": {"N": ["12"]},
    "ANDECAVI": {"N": ["12"]},
    "HELLVSII": {"N": ["12"]},
    "SIDICINI": {"N": ["12"]},
    "EOI": {"N": ["12"]},
    "TRANSPADANI": {"N": ["12"]},
    "BRVCTERI": {"N": ["12"]},
    "MASSYLI": {"N": ["12"]},
    "NARISTI": {"N": ["12"]},
    "MEMACENI": {"N": ["12"]},
    "POMPEI": {"N": ["12"]},
    "PEVCINI": {"N": ["12"]},
    "TARBELLI": {"N": ["12"]},
    "DANAI": {"N": ["12"]},
    "MYTILENAEI": {"N": ["12"]},
    "CAVNII": {"N": ["12"]},
    "GADITANI": {"N": ["12"]},
    "PAELIGNI": {"N": ["12"]},
    "CVMANI": {"N": ["12"]},
    "BREVNI": {"N": ["12"]},
    "CADVSII": {"N": ["12"]},
    "MARCOMANI": {"N": ["12"]},
    "PHILAENI": {"N": ["12"]},
    "MAEDI": {"N": ["12"]},
    "LAODICENI": {"N": ["12"]},
    "PATAVINI": {"N": ["12"]},
    "HADRVMETINI": {"N": ["12"]},
    "PRASII": {"N": ["12"]},
    "AQVITANI": {"N": ["12"]},
    "TRALLIANI": {"N": ["12"]},
    "HALICARNASSII": {"N": ["12"]},
    "DYRRACHINI": {"N": ["12"]},
    "MESSENII": {"N": ["12"]},
    "VIROMANDVI": {"N": ["12"]},
    "SEBAGNINI": {"N": ["12"]},
    "AESTII": {"N": ["12"]},
    "AMPSIVARII": {"N": ["12"]},
    "BABYLONII": {"N": ["12"]},
    "AORSI": {"N": ["12"]},
    "LVCANI": {"N": ["12"]},
    "SYNEPHEBI": {"N": ["12"]},
    "ELYMAEI": {"N": ["12"]},
    "EDONI": {"N": ["12"]},
    "BAETASII": {"N": ["12"]},
    "HYPERBOREI": {"N": ["12"]},
    "SEGNI": {"N": ["12"]},
    "BARCAEI": {"N": ["12"]},
    "HENIOCHI": {"N": ["12"]},
    "ARGI": {"N": ["12"]},
    "EPIGONI": {"N": ["12"]},
    "VMBRI": {"N": ["12"]},
    "ASCVLANI": {"N": ["12"]},
    "OMBI": {"N": ["12"]},
    "AMASTRIANI": {"N": ["12"]},
    "LOTOPHAGI": {"N": ["12"]},
    "ARAVISCI": {"N": ["12"]},
    "PELOPONNESII": {"N": ["12"]},
    "PHOCII": {"N": ["12"]},
    "VACCAEI": {"N": ["12"]},
    "RAETI": {"N": ["12"]},
    "ALANI": {"N": ["12"]},
    "THYSDRITANI": {"N": ["12"]},
    "MEGALOPOLITANI": {"N": ["12"]},
    "SEGONTIACI": {"N": ["12"]},
    "ERETRICI": {"N": ["12"]},
    "HENETI": {"N": ["12"]},
    "ARMENII": {"N": ["12"]},
    "ABII": {"N": ["12"]},
    "CARSEOLI": {"N": ["12"]},
    "FRISII": {"N": ["12"]},
    "PYRGI": {"N": ["12"]},
    "GRVDII": {"N": ["12"]},
    "FVNDI": {"N": ["12"]},
    "DVLICHII": {"N": ["12"]},
    "RVBI": {"N": ["12"]},
    "NAHANARVALI": {"N": ["12"]},
    "RVTENI": {"N": ["12"]},
    "TRIBOCI": {"N": ["12"]},
    "ARRETINI": {"N": ["12"]},
    "MANDVBII": {"N": ["12"]},
    "AMISENI": {"N": ["12"]},
    "MVSVLAMII": {"N": ["12"]},
    "MYSI": {"N": ["12"]},
    "FOSI": {"N": ["12"]},
    "FALISCI": {"N": ["12"]},
    "BITHYNI": {"N": ["12"]},
    "BIBROCI": {"N": ["12"]},
    "ACHAEI": {"N": ["12"]},
    "VICETINI": {"N": ["12"]},
    "REVDIGNI": {"N": ["12"]},
    "PLACENTINI": {"N": ["12"]},
    "SPARTANI": {"N": ["12"]},
    "PANICI": {"N": ["12"]},
    "MOESI": {"N": ["12"]},
    "TENCTERI": {"N": ["12"]},
    "POMPEIANI": {"N": ["12"]},
    "MINYI": {"N": ["12"]},
    "TEVCRI": {"N": ["12"]},
    "BAVLI": {"N": ["12"]},
    "ATTICI": {"N": ["12"]},
    "CASPII": {"N": ["12"]},
    "LEVACI": {"N": ["12"]},
    "HARII": {"N": ["12"]},
    "TVRDETANI": {"N": ["12"]},
    "CHII": {"N": ["12"]},
    "THESSALI": {"N": ["12"]},
    "ELESIODVLI": {"N": ["12"]},
    "AMBILIATI": {"N": ["12"]},
    "METHYMNAEI": {"N": ["12"]},
    "RAVRACI": {"N": ["12"]},
    "OCTAVIANI": {"N": ["12"]},
    "AVGVSTIANI": {"N": ["12"]},
    "ALBICI": {"N": ["12"]},
    "GERMANI": {"N": ["12"]},
    "TROGODYTAE": {"N": ["11"]},
    "NEAPOLITANI": {"N": ["12"]},
    "NORICI": {"N": ["12"]},
    "AGRIANI": {"N": ["12"]},
    "BOSPORANI": {"N": ["12"]},
    "TARTESSII": {"N": ["12"]},
    "FLORENTINI": {"N": ["12"]},
    "ILLYRII": {"N": ["12"]},
    "ORDOVICES": {"N": ["13"]},
    "LAERTES": {"N": ["13"]},
    "ACESINES": {"N": ["13"]},
    "HEDYMELES": {"N": ["13"]},
    "MITHRES": {"N": ["13"]},
    "NICARCHIDES": {"N": ["13"]},
    "EPIPHANES": {"N": ["13"]},
    "CINYRADES": {"N": ["13"]},
    "CORIOSOLITES": {"N": ["13"]},
    "PRAEXASPES": {"N": ["13"]},
    "AMPHICTYONES": {"N": ["13"]},
    "ARSACIDES": {"N": ["13"]},
    "TERIOLTES": {"N": ["13"]},
    "LEMNIADES": {"N": ["13"]},
    "ONOMASTORIDES": {"N": ["13"]},
    "MENETES": {"N": ["13"]},
    "CALES": {"N": ["13"]},
    "NARNIENSES": {"N": ["13"]},
    "TANTALIDES": {"N": ["13"]},
    "LADES": {"N": ["13"]},
    "COCOSATES": {"N": ["13"]},
    "THAPSITANI": {"N": ["12"]},
    "MYRMIDONES": {"N": ["13"]},
    "SOPHITES": {"N": ["13"]},
    "THESSALONICENSES": {"N": ["13"]},
    "CIRTENSES": {"N": ["13"]},
    "SVLMONENSES": {"N": ["13"]},
    "CAMERTES": {"N": ["13"]},
    "CARMONENSES": {"N": ["13"]},
    "GOTONES": {"N": ["13"]},
    "ACMONENSES": {"N": ["13"]},
    "HERACLIDES": {"N": ["13"]},
    "DEMOCRATES": {"N": ["13"]},
    "PONTICI": {"N": ["12"]},
    "VRSAONENSES": {"N": ["13"]},
    "PHERETIADES": {"N": ["13"]},
    "METHYMNIADES": {"N": ["13"]},
    "MEDATES": {"N": ["13"]},
    "PHTHIOTES": {"N": ["13"]},
    "ATRIDES": {"N": ["13"]},
    "AONIDES": {"N": ["13"]},
    "PARRACES": {"N": ["13"]},
    "IXIONIDES": {"N": ["13"]},
    "ANTICLES": {"N": ["13"]},
    "DRYADES": {"N": ["13"]},
    "GARAMANTES": {"N": ["13"]},
    "DROPIDES": {"N": ["13"]},
    "ATTALENSES": {"N": ["13"]},
    "ZAMENSES": {"N": ["13"]},
    "ORSINES": {"N": ["13"]},
    "PYRRHIADES": {"N": ["13"]},
    "LYCVRGIDES": {"N": ["13"]},
    "ANCHISIADES": {"N": ["13"]},
    "INTERAMNATES": {"N": ["13"]},
    "ARII": {"N": ["12"]},
    "ERINDES": {"N": ["13"]},
    "EVNONES": {"N": ["13"]},
    "MINTVRNENSES": {"N": ["13"]},
    "LEPTINES": {"N": ["13"]},
    "POLYBOETES": {"N": ["13"]},
    "DRANCES": {"N": ["13"]},
    "ARSAMES": {"N": ["13"]},
    "EVANTHES": {"N": ["13"]},
    "LAOMEDONTIADES": {"N": ["13"]},
    "BRONTES": {"N": ["13"]},
    "NAMNETES": {"N": ["13"]},
    "OILIADES": {"N": ["13"]},
    "NAVSIPHANES": {"N": ["13"]},
    "HYDASPES": {"N": ["13"]},
    "ERINNYES": {"N": ["13"]},
    "EBVRONES": {"N": ["13"]},
    "SAMNITES": {"N": ["13"]},
    "GATES": {"N": ["13"]},
    "IACETANI": {"N": ["12"]},
    "SOTIATES": {"N": ["13"]},
    "AEOLIDES": {"N": ["13"]},
    "HIPPOMENES": {"N": ["13"]},
    "SAVROMATES": {"N": ["13"]},
    "ARPI": {"N": ["12"]},
    "ORODES": {"N": ["13"]},
    "PISCES": {"N": ["13"]},
    "MERIONES": {"N": ["13"]},
    "BORYSTHENES": {"N": ["13"]},
    "OAXES": {"N": ["13"]},
    "MALIENSES": {"N": ["13"]},
    "PHRADATES": {"N": ["13"]},
    "CATVRIGES": {"N": ["13"]},
    "AENIDES": {"N": ["13"]},
    "HELICONIDES": {"N": ["13"]},
    "AESCHINES": {"N": ["13"]},
    "LELEGES": {"N": ["13"]},
    "PLEIADES": {"N": ["13"]},
    "ECHECRATIDES": {"N": ["13"]},
    "ASBYTES": {"N": ["13"]},
    "CORDVBENSES": {"N": ["13"]},
    "BVLIDENSES": {"N": ["13"]},
    "TAVRINITES": {"N": ["13"]},
    "TREVERI": {"N": ["12"]},
    "ARTAVASDES": {"N": ["13"]},
    "SABACES": {"N": ["13"]},
    "HAMADRYADES": {"N": ["13"]},
    "TYDIDES": {"N": ["13"]},
    "RHOSACES": {"N": ["13"]},
    "PARMENIDES": {"N": ["13"]},
    "AMBRACIOTES": {"N": ["13"]},
    "MEHERDATES": {"N": ["13"]},
    "NAVTES": {"N": ["13"]},
    "ILIENSES": {"N": ["13"]},
    "IGVVINATES": {"N": ["13"]},
    "GOMPHENSES": {"N": ["13"]},
    "BATAVI": {"N": ["12"]},
    "ORESTES": {"N": ["13"]},
    "LACIADES": {"N": ["13"]},
    "DRYOPES": {"N": ["13"]},
    "SPITAMENES": {"N": ["13"]},
    "HERMOGENES": {"N": ["13"]},
    "THYESTES": {"N": ["13"]},
    "SINNACES": {"N": ["13"]},
    "REDONES": {"N": ["13"]},
    "CATANES": {"N": ["13"]},
    "MOYSES": {"N": ["13"]},
    "VELIOCASSES": {"N": ["13"]},
    "BARZAENTES": {"N": ["13"]},
    "AGRIPPINENSES": {"N": ["13"]},
    "PANTHOIDES": {"N": ["13"]},
    "HARVDES": {"N": ["13"]},
    "ACHAEMENIDES": {"N": ["13"]},
    "TYRIOTES": {"N": ["13"]},
    "IORDANES": {"N": ["13"]},
    "AEGIDES": {"N": ["13"]},
    "PRVSENSES": {"N": ["13"]},
    "PENEIDES": {"N": ["13"]},
    "THEOPHANES": {"N": ["13"]},
    "ARCHIGENES": {"N": ["13"]},
    "APOLLONIDENSES": {"N": ["13"]},
    "VOCATES": {"N": ["13"]},
    "CHOASPES": {"N": ["13"]},
    "CEPHALLENES": {"N": ["13"]},
    "TROADES": {"N": ["13"]},
    "STROPHADES": {"N": ["13"]},
    "ASTASPES": {"N": ["13"]},
    "AQVINATES": {"N": ["13"]},
    "BRVTTIANI": {"N": ["12"]},
    "ARGOLIDES": {"N": ["13"]},
    "PROETIDES": {"N": ["13"]},
    "OENIDES": {"N": ["13"]},
    "ZORSINES": {"N": ["13"]},
    "IAZYGES": {"N": ["13"]},
    "HESPERIDES": {"N": ["13"]},
    "CHARICLES": {"N": ["13"]},
    "TRINOBANTES": {"N": ["13"]},
    "THESIDES": {"N": ["13"]},
    "NASAMONES": {"N": ["13"]},
    "APOECIDES": {"N": ["13"]},
    "IMBRASIDES": {"N": ["13"]},
    "TROIADES": {"N": ["13"]},
    "LAERTIADES": {"N": ["13"]},
    "BELIDES": {"N": ["13"]},
    "SELEVCENSES": {"N": ["13"]},
    "CROTOPIADES": {"N": ["13"]},
    "ABISARES": {"N": ["13"]},
    "CROTONIATES": {"N": ["13"]},
    "THABENENSES": {"N": ["13"]},
    "SITALCES": {"N": ["13"]},
    "CVRETES": {"N": ["13"]},
    "ACARNANES": {"N": ["13"]},
    "BERONES": {"N": ["13"]},
    "INDIGETES": {"N": ["13"]},
    "ARAXES": {"N": ["13"]},
    "STEROPES": {"N": ["13"]},
    "PITTHIDES": {"N": ["13"]},
    "PHAETHONTIADES": {"N": ["13"]},
    "LEOPREPIDES": {"N": ["13"]},
    "TMOLITES": {"N": ["13"]},
    "SOPHOCLES": {"N": ["13"]},
    "AMYNTIADES": {"N": ["13"]},
    "ACOETES": {"N": ["13"]},
    "ELVSATES": {"N": ["13"]},
    "MINDYRIDES": {"N": ["13"]},
    "NEALCES": {"N": ["13"]},
    "GYNDES": {"N": ["13"]},
    "ORNOSPADES": {"N": ["13"]},
    "OXYARTES": {"N": ["13"]},
    "BITVRIGES": {"N": ["13"]},
    "NAVCRATES": {"N": ["13"]},
    "ATARNITES": {"N": ["13"]},
    "EVMEDES": {"N": ["13"]},
    "HARMERIDES": {"N": ["13"]},
    "SYRTES": {"N": ["13"]},
    "EVRYMIDES": {"N": ["13"]},
    "TIMOCLES": {"N": ["13"]},
    "ORIGINES": {"N": ["13"]},
    "MITHRENES": {"N": ["13"]},
    "BATTIADES": {"N": ["13"]},
    "ALYATTES": {"N": ["13"]},
    "ALETRINATES": {"N": ["13"]},
    "TARVSATES": {"N": ["13"]},
    "PIERIDES": {"N": ["13"]},
    "CEVTRONES": {"N": ["13"]},
    "MENOETIADES": {"N": ["13"]},
    "BOOTES": {"N": ["13"]},
    "SALMACIDES": {"N": ["13"]},
    "AGYLLINI": {"N": ["12"]},
    "ASTACIDES": {"N": ["13"]},
    "NAVPLIADES": {"N": ["13"]},
    "DATAPHERNES": {"N": ["13"]},
    "LIBETHRIDES": {"N": ["13"]},
    "DORYLENSES": {"N": ["13"]},
    "GERYONES": {"N": ["13"]},
    "MEDOBRIGENSES": {"N": ["13"]},
    "NORENSES": {"N": ["13"]},
    "ANTISTHENES": {"N": ["13"]},
    "HYSTASPES": {"N": ["13"]},
    "PETROCORII": {"N": ["12"]},
    "CHREMES": {"N": ["13"]},
    "VONONES": {"N": ["13"]},
    "SEVTHES": {"N": ["13"]},
    "PHRAATES": {"N": ["13"]},
    "STOECHADES": {"N": ["13"]},
    "EPIMENES": {"N": ["13"]},
    "MASSILIENSES": {"N": ["13"]},
    "ADRYADES": {"N": ["13"]},
    "RHEOMITHRES": {"N": ["13"]},
    "CARES": {"N": ["13"]},
    "CHALYBES": {"N": ["13"]},
    "CATAMANTALOES": {"N": ["13"]},
    "NIPHATES": {"N": ["13"]},
    "VSPENSES": {"N": ["13"]},
    "XERXES": {"N": ["13"]},
    "EVPHRATES": {"N": ["13"]},
    "PEVCESTES": {"N": ["13"]},
    "ELYSII": {"N": ["12"]},
    "PHAEACES": {"N": ["13"]},
    "GENAVNI": {"N": ["12"]},
    "CYCLOPES": {"N": ["13"]},
    "EVDOSES": {"N": ["13"]},
    "VTICENSES": {"N": ["13"]},
    "TEMNITES": {"N": ["13"]},
    "SEMNONES": {"N": ["13"]},
    "IAPYDES": {"N": ["13"]},
    "PEGASIDES": {"N": ["13"]},
    "PARMENSES": {"N": ["13"]},
    "IASIDES": {"N": ["13"]},
    "TROES": {"N": ["13"]},
    "MAZACES": {"N": ["13"]},
    "BRIGANTES": {"N": ["13"]},
    "CAMBYSES": {"N": ["13"]},
    "MASSAGETES": {"N": ["13"]},
    "LVPERCI": {"N": ["12"]},
    "THEODOROMEDES": {"N": ["13"]},
    "IOANNES": {"N": ["13"]},
    "PERIPHANES": {"N": ["13"]},
    "MONAESES": {"N": ["13"]},
    "ARIMAZES": {"N": ["13"]},
    "ONITES": {"N": ["13"]},
    "HIPPOCRATES": {"N": ["13"]},
    "PISTORENSES": {"N": ["13"]},
    "ACHILLES": {"N": ["13"]},
    "THALES": {"N": ["13"]},
    "INSVBRES": {"N": ["13"]},
    "CEPHENES": {"N": ["13"]},
    "IPHICRATES": {"N": ["13"]},
    "AMEDINES": {"N": ["13"]},
    "GRAIOCELI": {"N": ["12"]},
    "OTHRYADES": {"N": ["13"]},
    "ACHATES": {"N": ["13"]},
    "PICTONES": {"N": ["13"]},
    "PHILIPPENSES": {"N": ["13"]},
    "BVRI": {"N": ["12"]},
    "LIBONES": {"N": ["13"]},
    "ATIZYES": {"N": ["13"]},
    "HISPELLATES": {"N": ["13"]},
    "TERPSICHORES": {"N": ["13"]},
    "PENTHELIDES": {"N": ["13"]},
    "ACESTES": {"N": ["13"]},
    "SANTONES": {"N": ["13"]},
    "SISYPHIDES": {"N": ["13"]},
    "DITAMENES": {"N": ["13"]},
    "VELIENSES": {"N": ["13"]},
    "VIENNENSES": {"N": ["13"]},
    "PHYLACIDES": {"N": ["13"]},
    "PANES": {"N": ["13"]},
    "VELLIOCASSES": {"N": ["13"]},
    "NICOCLES": {"N": ["13"]},
    "PLISTHENES": {"N": ["13"]},
    "ANTIPHATES": {"N": ["13"]},
    "ATINATES": {"N": ["13"]},
    "AETHIOPES": {"N": ["13"]},
    "ANDROMENES": {"N": ["13"]},
    "EBVROVICES": {"N": ["13"]},
    "SVGAMBRI": {"N": ["12"]},
    "HERODES": {"N": ["13"]},
    "SARDES": {"N": ["13"]},
    "AVXIMATES": {"N": ["13"]},
    "NICOMEDENSES": {"N": ["13"]},
    "SALLVES": {"N": ["13"]},
    "DEMOSTHENES": {"N": ["13"]},
    "CLINIADES": {"N": ["13"]},
    "OZINES": {"N": ["13"]},
    "AMYNTORIDES": {"N": ["13"]},
    "TIRIDATES": {"N": ["13"]},
    "THVCYDIDES": {"N": ["13"]},
    "HYRTACIDES": {"N": ["13"]},
    "ABORIGINES": {"N": ["13"]},
    "APOLLONIATES": {"N": ["13"]},
    "HARPOCRATES": {"N": ["13"]},
    "POTNIADES": {"N": ["13"]},
    "VSIPETES": {"N": ["13"]},
    "SVARDONES": {"N": ["13"]},
    "SORTIENTES": {"N": ["13"]},
    "SIDONII": {"N": ["12"]},
    "ACHAEMENES": {"N": ["13"]},
    "PHILYRIDES": {"N": ["13"]},
    "FOROIVLIENSES": {"N": ["13"]},
    "RHOEMETALCES": {"N": ["13"]},
    "HELICONIADES": {"N": ["13"]},
    "EMERITENSES": {"N": ["13"]},
    "ARGYRASPIDES": {"N": ["13"]},
    "NITIOBROGES": {"N": ["13"]},
    "TVBANTES": {"N": ["13"]},
    "HISPALENSES": {"N": ["13"]},
    "ETEOCLES": {"N": ["13"]},
    "ANARTES": {"N": ["13"]},
    "PETRAITES": {"N": ["13"]},
    "APHRODISIENSES": {"N": ["13"]},
    "CAPSENSES": {"N": ["13"]},
    "SERES": {"N": ["13"]},
    "TARRACINENSES": {"N": ["13"]},
    "AEGEATES": {"N": ["13"]},
    "ACMONIDES": {"N": ["13"]},
    "SARRASTES": {"N": ["13"]},
    "BASILIDES": {"N": ["13"]},
    "PHINIDES": {"N": ["13"]},
    "ERATOSTHENES": {"N": ["13"]},
    "ANTORES": {"N": ["13"]},
    "LEVCIPPIDES": {"N": ["13"]},
    "DARDANIDES": {"N": ["13"], "A": ["25"]},
    "GRINNES": {"N": ["13"]},
    "CRETES": {"N": ["13"]},
    "AREOPAGITES": {"N": ["13"]},
    "AGGRAMMES": {"N": ["13"]},
    "CENABENSES": {"N": ["13"]},
    "ORCADES": {"N": ["13"]},
    "CHALDAEI": {"N": ["12"]},
    "PARTHYAEI": {"N": ["12"]},
    "HERMES": {"N": ["13"]},
    "ARETES": {"N": ["13"]},
    "TECTOSAGES": {"N": ["13"]},
    "BORESTI": {"N": ["12"]},
    "MITHRACENES": {"N": ["13"]},
    "APOLLONIDES": {"N": ["13"]},
    "MENOETES": {"N": ["13"]},
    "ATREBATES": {"N": ["13"]},
    "LYSIADES": {"N": ["13"]},
    "BAGISTANES": {"N": ["13"]},
    "PHILOCTETES": {"N": ["13"]},
    "NVITONES": {"N": ["13"]},
    "GYGES": {"N": ["13"]},
    "TEVTONES": {"N": ["13"]},
    "MYSTES": {"N": ["13"]},
    "ARISTOPHONTES": {"N": ["13"]},
    "CORONIDES": {"N": ["13"]},
    "ANTIGENES": {"N": ["13"]},
    "GANGARIDES": {"N": ["13"]},
    "NICETES": {"N": ["13"]},
    "PALES": {"N": ["13"]},
    "STRATOCLES": {"N": ["13"]},
    "ELEI": {"N": ["12"]},
    "ALETES": {"N": ["13"]},
    "THYMOETES": {"N": ["13"]},
    "BARGATES": {"N": ["13"]},
    "ABANTIADES": {"N": ["13"]},
    "IDARNES": {"N": ["13"]},
    "CARTEIENSES": {"N": ["13"]},
    "CALLICRATES": {"N": ["13"]},
    "SVIONES": {"N": ["13"]},
    "SISENES": {"N": ["13"]},
    "LYNCESTES": {"N": ["13"]},
    "DEMADES": {"N": ["13"]},
    "PHRYGES": {"N": ["13"]},
    "LEMOVICES": {"N": ["13"]},
    "AMBIANI": {"N": ["12"]},
    "ASPRENATES": {"N": ["13"]},
    "CALETES": {"N": ["13"]},
    "DARES": {"N": ["13"]},
    "PRIAMIDES": {"N": ["13"]},
    "PALAMEDES": {"N": ["13"]},
    "THIMODES": {"N": ["13"]},
    "VASACES": {"N": ["13"]},
    "CAPPADOCES": {"N": ["13"]},
    "SICCENSES": {"N": ["13"]},
    "ORONTES": {"N": ["13"]},
    "MEMMIADES": {"N": ["13"]},
    "ABVLITES": {"N": ["13"]},
    "ERICHAETES": {"N": ["13"]},
    "HAVSTANES": {"N": ["13"]},
    "TRINOVANTES": {"N": ["13"]},
    "BAGOPHANES": {"N": ["13"]},
    "CREMONENSES": {"N": ["13"]},
    "CARNVTES": {"N": ["13"]},
    "HELIADES": {"N": ["13"]},
    "MILTIADES": {"N": ["13"]},
    "SATROPATES": {"N": ["13"]},
    "AVLESTES": {"N": ["13"]},
    "LAESTRYGONES": {"N": ["13"]},
    "SISIMITHRES": {"N": ["13"]},
    "SEGESTES": {"N": ["13"]},
    "PYTHOCLES": {"N": ["13"]},
    "BVRSAVONENSES": {"N": ["13"]},
    "ARTAXERXES": {"N": ["13"]},
    "TAXILES": {"N": ["13"]},
    "SINDES": {"N": ["13"]},
    "ARISTOMEDES": {"N": ["13"]},
    "BYLLIDENSES": {"N": ["13"]},
    "IMMORTALES": {"N": ["13"]},
    "ABDAGAESES": {"N": ["13"]},
    "PHARNACES": {"N": ["13"]},
    "MYGDONIDES": {"N": ["13"]},
    "EPIDAPHNES": {"N": ["13"]},
    "BACTRIANI": {"N": ["12"]},
    "BERECYNTIADES": {"N": ["13"]},
    "EPIRENSES": {"N": ["13"]},
    "VETTONES": {"N": ["13"]},
    "LYCAMBES": {"N": ["13"]},
    "STRATIPPOCLES": {"N": ["13"]},
    "OECLIDES": {"N": ["13"]},
    "CYRENENSES": {"N": ["13"]},
    "AGRIANES": {"N": ["13"]},
    "NABATHES": {"N": ["13"]},
    "SATIBARZANES": {"N": ["13"]},
    "VARDANES": {"N": ["13"]},
    "PAPHLAGONES": {"N": ["13"]},
    "CATAONES": {"N": ["13"]},
    "LEVCATES": {"N": ["13"]},
    "ARIARATHES": {"N": ["13"]},
    "NEREIDES": {"N": ["13"]},
    "RHODIENSES": {"N": ["13"]},
    "LARES": {"N": ["13"]},
    "POLEMOCRATES": {"N": ["13"]},
    "ISOCRATES": {"N": ["13"]},
    "AMPHIAREIADES": {"N": ["13"]},
    "IONES": {"N": ["13"]},
    "SAMOTHRACES": {"N": ["13"]},
    "PHILLYRIDES": {"N": ["13"]},
    "CARALES": {"N": ["13"]},
    "ORONTOBATES": {"N": ["13"]},
    "ZETES": {"N": ["13"]},
    "MIMALLONIDES": {"N": ["13"]},
    "LABEATES": {"N": ["13"]},
    "TIGRANES": {"N": ["13"]},
    "ALCIBIADES": {"N": ["13"]},
    "NVCERINI": {"N": ["12"]},
    "HYPERIDES": {"N": ["13"]},
    "MVNDENSES": {"N": ["13"]},
    "STRATONICENSES": {"N": ["13"]},
    "ZALACES": {"N": ["13"]},
    "AMPHITRYONIADES": {"N": ["13"]},
    "TIBVRTES": {"N": ["13"]},
    "ABISTAMENES": {"N": ["13"]},
    "NICAEENSES": {"N": ["13"]},
    "ANCALITES": {"N": ["13"]},
    "ILIADES": {"N": ["13"], "O": ["17"]},
    "MELICERTES": {"N": ["13"]},
    "SINOPENSES": {"N": ["13"]},
    "HYADES": {"N": ["13"]},
    "LACONES": {"N": ["13"]},
    "NABARZANES": {"N": ["13"]},
    "CYPRII": {"N": ["12"]},
    "OLYMPIONICES": {"N": ["13"]},
    "ERICES": {"N": ["13"]},
    "AEETES": {"N": ["13"]},
    "PANOPES": {"N": ["13"]},
    "RAMESSES": {"N": ["13"]},
    "EVMENIDES": {"N": ["13"]},
    "PICENTES": {"N": ["13"]},
    "PELOPONNENSES": {"N": ["13"]},
    "DEMOCHARES": {"N": ["13"]},
    "CANNINEFATES": {"N": ["13"]},
    "SITHONES": {"N": ["13"]},
    "AVGVSTALES": {"N": ["13"]},
    "SCIPIADES": {"N": ["13"]},
    "EMPEDOCLES": {"N": ["13"]},
    "GOBARES": {"N": ["13"]},
    "CHARES": {"N": ["13"]},
    "DIORES": {"N": ["13"]},
    "BVTES": {"N": ["13"]},
    "TIMAGENES": {"N": ["13"]},
    "SYMPLEGADES": {"N": ["13"]},
    "APHELIOTES": {"N": ["13"]},
    "GANGES": {"N": ["13"]},
    "PARTHI": {"N": ["12"]},
    "PERIPLANES": {"N": ["13"]},
    "NEMETES": {"N": ["13"]},
    "PAMMENES": {"N": ["13"]},
    "POLIORCETES": {"N": ["13"]},
    "LVGDVNENSES": {"N": ["13"]},
    "MAEONIDES": {"N": ["13"]},
    "CALLICRATIDES": {"N": ["13"]},
    "PROTOGENES": {"N": ["13"]},
    "COBARES": {"N": ["13"]},
    "DRAPPES": {"N": ["13"]},
    "OXYDATES": {"N": ["13"]},
    "LINGONES": {"N": ["13"]},
    "LARINATES": {"N": ["13"]},
    "TYRES": {"N": ["13"]},
    "ELELEIDES": {"N": ["13"]},
    "HOMONADENSES": {"N": ["13"]},
    "AGAMEMNONIDES": {"N": ["13"]},
    "ARISTOPHANES": {"N": ["13"]},
    "ACTORIDES": {"N": ["13"]},
    "EVMENES": {"N": ["13"]},
    "HYPSIDES": {"N": ["13"]},
    "DIABLINTES": {"N": ["13"]},
    "ANTIPHANES": {"N": ["13"]},
    "ALTINATES": {"N": ["13"]},
    "NANTVATES": {"N": ["13"]},
    "THEMISTOCLES": {"N": ["13"]},
    "OEENSES": {"N": ["13"]},
    "CICONES": {"N": ["13"]},
    "THYESTIADES": {"N": ["13"]},
    "ISTAEVONES": {"N": ["13"]},
    "PHRATAPHERNES": {"N": ["13"]},
    "OPHELTES": {"N": ["13"]},
    "SVESSIONES": {"N": ["13"]},
    "LYCONIDES": {"N": ["13"]},
    "CLEANTHES": {"N": ["13"]},
    "LYDI": {"N": ["12"]},
    "SOCRATICI": {"N": ["12"]},
    "IZATES": {"N": ["13"]},
    "FICEDVLENSES": {"N": ["13"]},
    "ARSACES": {"N": ["13"]},
    "APAMENSES": {"N": ["13"]},
    "VERONENSES": {"N": ["13"]},
    "ITALICENSES": {"N": ["13"]},
    "ILLVRGAVONENSES": {"N": ["13"]},
    "INACHIDES": {"N": ["13"]},
    "DISCORIDES": {"N": ["13"]},
    "ARISTOMENES": {"N": ["13"]},
    "TITIES": {"N": ["13"]},
    "AVIONES": {"N": ["13"]},
    "ATLANTIDES": {"N": ["13"]},
    "ANDES": {"N": ["13"]},
    "ARABES": {"N": ["13"]},
    "DRYANTIADES": {"N": ["13"]},
    "MARTIALES": {"N": ["13"]},
    "DOLOPES": {"N": ["13"]},
    "PAEONES": {"N": ["13"]},
    "VACCENSES": {"N": ["13"]},
    "SENIENSES": {"N": ["13"]},
    "ORSES": {"N": ["13"]},
    "ARCHIDEMIDES": {"N": ["13"]},
    "ATHAMANES": {"N": ["13"]},
    "BISTONES": {"N": ["13"]},
    "GAVDIMONIVM": {"-": ["12"]},
    "VNISVBSELLIVM": {"-": ["12"]},
    "PIPVLVM": {"-": ["12"]},
    "LVMBIFRAGIVM": {"-": ["12"]},
    "PVGNVM": {"-": ["12"]},
    "PROPYLAEVM": {"-": ["12"]},
    "PRAEDIOLVM": {"-": ["12"]},
    "CONSTRATVM": {"-": ["12"]},
    "PERMAGNVM": {"-": ["12"]},
    "PVLPAMENTVM": {"-": ["12"]},
    "HARPASTES": {"N": ["13"]},
    "HAEMONIDES": {"N": ["13"]},
    "CLOSTELLVM": {"-": ["12"]},
    "PATARACINVM": {"-": ["12"]},
    "AVRICHALCVM": {"-": ["12"]},
    "IDOLVM": {"-": ["12"]},
    "GVRGVSTIVM": {"-": ["12"]},
    "TRIOBOLVM": {"-": ["12"]},
    "EPYTIDES": {"N": ["13"]},
    "AVTOMATVM": {"-": ["12"]},
    "PERPAVXILLVM": {"-": ["12"]},
    "BARBARVM": {"-": ["12"]},
    "BICLINIVM": {"-": ["12"]},
    "STIBADIVM": {"-": ["12"]},
    "LYCOMEDES": {"N": ["13"]},
    "CORYLETVM": {"-": ["12"]},
    "PARRHESIASTES": {"N": ["13"]},
    "SILVRES": {"N": ["13"]},
    "DVLGVBINI": {"N": ["12"]},
    "CONTIGNATVM": {"-": ["12"]},
    "PORCINVM": {"-": ["12"]},
    "ALIORSVM": {"-": ["12"]},
    "LINTEOLVM": {"-": ["12"]},
    "TORCVLVM": {"-": ["12"]},
    "CARNEADES": {"N": ["13"]},
    "PATROCLES": {"N": ["13"]},
    "SYNOECIVM": {"-": ["12"]},
    "BAPTISTERIVM": {"-": ["12"]},
    "PSALTERIVM": {"-": ["12"]},
    "POLITES": {"N": ["13"]},
    "ACHILLIDES": {"N": ["13"]},
    "PELIDES": {"N": ["13"]},
    "CATINVM": {"-": ["12"]},
    "MVSCARIVM": {"-": ["12"]},
    "INGAEVONES": {"N": ["13"]},
    "INTERPRETAMENTVM": {"-": ["12"]},
    "CLAVARIVM": {"-": ["12"]},
    "OEBALIDES": {"N": ["13"]},
    "VIRIDARIVM": {"-": ["12"]},
    "SIBVLATES": {"N": ["13"]},
    "NEXVM": {"-": ["12"]},
    "PENICILLVM": {"-": ["12"]},
    "SIMONIDES": {"N": ["13"]},
    "DEPRAESENTIARVM": {"-": ["12"]},
    "COMPACTVM": {"-": ["12"]},
    "DELIRAMENTVM": {"-": ["12"]},
    "DELECTAMENTVM": {"-": ["12"]},
    "PHERES": {"N": ["13"]},
    "PERSICVM": {"-": ["12"]},
    "HAVSTRVM": {"-": ["12"]},
    "CAVAEDIVM": {"-": ["12"]},
    "REFVGIVM": {"-": ["12"]},
    "CORCILLVM": {"-": ["12"]},
    "AONES": {"N": ["13"]},
    "PRATVLVM": {"-": ["12"]},
    "TRISTIMONIVM": {"-": ["12"]},
    "GALLINICIVM": {"-": ["12"]},
    "FRIGIDVM": {"-": ["12"]},
    "PIGMENTVM": {"-": ["12"]},
    "PANARIVM": {"-": ["12"]},
    "SPHAERISTERIVM": {"-": ["12"]},
    "SVDVM": {"-": ["12"]},
    "PARTVM": {"-": ["12"]},
    "CHIRAMAXIVM": {"-": ["12"]},
    "OSTIOLVM": {"-": ["12"]},
    "SPITACES": {"N": ["13"]},
    "DESIDIABVLVM": {"-": ["12"]},
    "SEXENNIVM": {"-": ["12"]},
    "BELLEROPHONTES": {"N": ["13"]},
    "PISVM": {"-": ["12"]},
    "STERQVILINVM": {"-": ["12"]},
    "METHODIVM": {"-": ["12"]},
    "TERMENTVM": {"-": ["12"]},
    "POSTLIMINIVM": {"-": ["12"]},
    "ARCISELLIVM": {"-": ["12"]},
    "VANGIONES": {"N": ["13"]},
    "VVLTVM": {"-": ["12"]},
    "ALCIDES": {"N": ["13"]},
    "CATILLVM": {"-": ["12"]},
    "SESAMVM": {"-": ["12"]},
    "QVASILLVM": {"-": ["12"]},
    "ALIQVOVORSVM": {"-": ["12"]},
    "MAGNETES": {"N": ["13"]},
    "SEMICINCTIVM": {"-": ["12"]},
    "EVRIPIDES": {"N": ["13"]},
    "LEPONTII": {"N": ["12"]},
    "PYLADES": {"N": ["13"]},
    "OSCENSES": {"N": ["13"]},
    "DIVERBIVM": {"-": ["12"]},
    "TRALLES": {"N": ["13"]},
    "TVRBAMENTVM": {"-": ["12"]},
    "AEACIDES": {"N": ["13"]},
    "CINGILLVM": {"-": ["12"]},
    "IMMERITVM": {"-": ["12"]},
    "LARIDES": {"N": ["13"]},
    "HORAEVM": {"-": ["12"]},
    "DIFFVGIVM": {"-": ["12"]},
    "SVBLIGACVLVM": {"-": ["12"]},
    "NASVM": {"-": ["12"]},
    "SENONES": {"N": ["13"]},
    "BIFVRCVM": {"-": ["12"]},
    "SEPLASIVM": {"-": ["12"]},
    "BLITEVM": {"-": ["12"]},
    "OXATHRES": {"N": ["13"]},
    "CORCVLVM": {"-": ["12"]},
    "VESTICONTVBERNIVM": {"-": ["12"]},
    "LENIMENTVM": {"-": ["12"]},
    "PITTACIVM": {"-": ["12"]},
    "VIRECTVM": {"-": ["12"]},
    "SAXETVM": {"-": ["12"]},
    "NAVCVM": {"-": ["12"]},
    "BASILICVM": {"-": ["12"]},
    "PROPVDIVM": {"-": ["12"]},
    "VTERVM": {"-": ["12"]},
    "CONTICINIVM": {"-": ["12"]},
    "MANTELVM": {"-": ["12"]},
    "CONSIDIVM": {"-": ["12"]},
    "EXVVIVM": {"-": ["12"]},
    "VTROQVEVERSVM": {"-": ["12"]},
    "RHAMNES": {"N": ["13"]},
    "VNCTORIVM": {"-": ["12"]},
    "LITIGIVM": {"-": ["12"]},
    "PISTILLVM": {"-": ["12"]},
    "CLIVVM": {"-": ["12"]},
    "FVSTVARIVM": {"-": ["12"]},
    "LATINIENSES": {"N": ["13"]},
    "CLIPEVM": {"-": ["12"]},
    "ORICALCHVM": {"-": ["12"]},
    "EVRYTIDES": {"N": ["13"]},
    "IENTACVLVM": {"-": ["12"]},
    "STABILIMENTVM": {"-": ["12"]},
    "HIPPOTADES": {"N": ["13"]},
    "PRAEIVDICATVM": {"-": ["12"]},
    "GLANDIVM": {"-": ["12"]},
    "OENOPOLIVM": {"-": ["12"]},
    "VERIVERBIVM": {"-": ["12"]},
    "BVBACES": {"N": ["13"]},
    "CALLISTHENES": {"N": ["13"]},
    "TOLOSATES": {"N": ["13"]},
    "LESBIADES": {"N": ["13"]},
    "CORFINIENSES": {"N": ["13"]},
    "ARIOBARZANES": {"N": ["13"]},
    "CALAMISTRVM": {"-": ["12"]},
    "POEMATIVM": {"-": ["12"]},
    "CVPEDIVM": {"-": ["12"]},
    "DENTIFRANGIBVLVM": {"-": ["12"]},
    "CECROPIDES": {"N": ["13"]},
    "GVSTATORIVM": {"-": ["12"]},
    "EXPLEMENTVM": {"-": ["12"]},
    "MYROPOLIVM": {"-": ["12"]},
    "NASTVRTIVM": {"-": ["12"]},
    "INTEGVMENTVM": {"-": ["12"]},
    "THESTIADES": {"N": ["13"]},
    "IRRIDICVLVM": {"-": ["12"]},
    "ODARIVM": {"-": ["12"]},
    "HOSTIMENTVM": {"-": ["12"]},
    "CRYSTALLVM": {"-": ["12"]},
    "NVCIFRANGIBVLVM": {"-": ["12"]},
    "ASCLEPIADES": {"N": ["13"]},
    "TRIFOLIVM": {"-": ["12"]},
    "MARSVPIVM": {"-": ["12"]},
    "THESAVRVM": {"-": ["12"]},
    "EMBOLVM": {"-": ["12"]},
    "BACCIBALLVM": {"-": ["12"]},
    "PROCRVSTES": {"N": ["13"]},
    "PROMONTVRIVM": {"-": ["12"]},
    "NOMADES": {"N": ["13"]},
    "AMPHITEATRVM": {"-": ["12"]},
    "CORYBANTES": {"N": ["13"]},
    "ILLORSVM": {"-": ["12"]},
    "BISTONIDES": {"N": ["13"]},
    "SVBVRBIVM": {"-": ["12"]},
    "MENDICABVLVM": {"-": ["12"]},
    "SPECVM": {"-": ["12"]},
    "MALEICORIVM": {"-": ["12"]},
    "EVRYBATES": {"N": ["13"]},
    "TELINVM": {"-": ["12"]},
    "GOTARZES": {"N": ["13"]},
    "VASCONES": {"N": ["13"]},
    "LABDACIDES": {"N": ["13"]},
    "CLEOCHARES": {"N": ["13"]},
    "BRANNOVICES": {"N": ["13"]},
    "IDYLLIVM": {"-": ["12"]},
    "GYNAECEVM": {"-": ["12"]},
    "CONSERVITIVM": {"-": ["12"]},
    "DILVCVLVM": {"-": ["12"]},
    "OREADES": {"N": ["13"]},
    "MELCVLVM": {"-": ["12"]},
    "MANVCIOLVM": {"-": ["12"]},
    "STATVNCVLVM": {"-": ["12"]},
    "SISYMBRIVM": {"-": ["12"]},
    "BASIOLVM": {"-": ["12"]},
    "ZARIASPES": {"N": ["13"]},
    "CHARITES": {"N": ["13"]},
    "THERSITES": {"N": ["13"]},
    "AESONIDES": {"N": ["13"]},
    "COLIPHIVM": {"-": ["12"]},
    "OFFENDICVLVM": {"-": ["12"]},
    "ARABARCHES": {"N": ["13"]},
    "MELLICVLVM": {"-": ["12"]},
    "CRVCIAMENTVM": {"-": ["12"]},
    "ALTROVORSVM": {"-": ["12"]},
    "VENALICIVM": {"-": ["12"]},
    "MILIARIVM": {"-": ["12"]},
    "BIGERRIONES": {"N": ["13"]},
    "APODYTERIVM": {"-": ["12"]},
    "CONDITORIVM": {"-": ["12"]},
    "LATERICIVM": {"-": ["12"]},
    "PROSTIBVLVM": {"-": ["12"]},
    "TENTAMENTVM": {"-": ["12"]},
    "OXARTES": {"N": ["13"]},
    "SVBNIMIVM": {"-": ["12"]},
    "FASCINVM": {"-": ["12"]},
    "OXYCOMINVM": {"-": ["12"]},
    "EXOTICVM": {"-": ["12"]},
    "DIOMEDES": {"N": ["13"]},
    "CALIDVM": {"-": ["12"]},
    "BLANDVM": {"-": ["12"]},
    "LIBYES": {"N": ["13"]},
    "CHRYSES": {"N": ["13"]},
    "LESBIDES": {"N": ["13"]},
    "CORYMBIVM": {"-": ["12"]},
    "HERMINONES": {"N": ["13"]},
    "DIARDINES": {"N": ["13"]},
    "MORETVM": {"-": ["12"]},
    "GASTRVM": {"-": ["12"]},
    "TONSTRINVM": {"-": ["12"]},
    "ARPINATES": {"N": ["13"]},
    "PARISII": {"N": ["12"]},
    "INFORTVNIVM": {"-": ["12"]},
    "TRINVNDINVM": {"-": ["12"]},
    "CEREBELLVM": {"-": ["12"]},
    "ELLVM": {"-": ["12"]},
    "FEBRVVM": {"-": ["12"]},
    "LIBRVM": {"-": ["12"]},
    "PHARASMANES": {"N": ["13"]},
    "COPHES": {"N": ["13"]},
    "SENTICETVM": {"-": ["12"]},
    "OFFVLGEO": {"-": ["52"]},
    "SACES": {"N": ["13"]},
    "CAERACATES": {"N": ["13"]},
    "ARGESTES": {"N": ["13"]},
    "DIRIBEO": {"-": ["52"]},
    "SESSORIVM": {"-": ["12"]},
    "MACEO": {"-": ["52"]},
    "PRAEHIBEO": {"-": ["52"]},
    "VERCVLVM": {"-": ["12"]},
    "TITTIBILICIVM": {"-": ["12"]},
    "EXSVGEO": {"-": ["52"]},
    "ALLEVAMENTVM": {"-": ["12"]},
    "ATTORQVEO": {"-": ["52"]},
    "THVCIDIDES": {"N": ["13"]},
    "SPICVM": {"-": ["12"]},
    "ISELASTICVM": {"-": ["12"]},
    "OBOLEO": {"-": ["52"]},
    "AFFLEO": {"-": ["52"]},
    "ORICHALCVM": {"-": ["12"]},
    "FRENDEO": {"-": ["52"]},
    "REFERVEO": {"-": ["52"]},
    "SPLENIVM": {"-": ["12"]},
    "CONSVADEO": {"-": ["52"]},
    "ADDENSEO": {"-": ["52"]},
    "LINTEO": {"-": ["52"]},
    "FOETEO": {"-": ["52"]},
    "RECALEO": {"-": ["52"]},
    "TYRIDATES": {"N": ["13"]},
    "CONFVLGEO": {"-": ["52"]},
    "CIRCVMSIDEO": {"-": ["52"]},
    "SVBTIMEO": {"-": ["52"]},
    "IMMVLGEO": {"-": ["52"]},
    "SVPERIMMINEO": {"-": ["52"]},
    "SCHEDIVM": {"-": ["12"]},
    "INSCRIPTVM": {"-": ["12"]},
    "COMPLACEO": {"-": ["52"]},
    "REDHIBEO": {"-": ["52"]},
    "REMVLCEO": {"-": ["52"]},
    "OBSORBEO": {"-": ["52"]},
    "SEPARATIO": {"-": ["13"]},
    "PERPOTATIO": {"-": ["13"]},
    "PERSALVTATIO": {"-": ["13"]},
    "PARASITATIO": {"-": ["13"]},
    "COLLAVDATIO": {"-": ["13"]},
    "LIBATIO": {"-": ["13"]},
    "DISTVRBATIO": {"-": ["13"]},
    "CONSOCIATIO": {"-": ["13"]},
    "PELLICVLATIO": {"-": ["13"]},
    "CONCERTATIO": {"-": ["13"]},
    "OCCVRSATIO": {"-": ["13"]},
    "OCCATIO": {"-": ["13"]},
    "EGESTIO": {"-": ["13"]},
    "RECEPTIO": {"-": ["13"]},
    "SPECTIO": {"-": ["13"]},
    "OSCILLATIO": {"-": ["13"]},
    "PERSCRIPTIO": {"-": ["13"]},
    "IMMODERATIO": {"-": ["13"]},
    "OBSECRATIO": {"-": ["13"]},
    "ASSECTATIO": {"-": ["13"]},
    "PLACABILITAS": {"-": ["13"]},
    "INSTAVRATIO": {"-": ["13"]},
    "RESTITVTIO": {"-": ["13"]},
    "MVTITIO": {"-": ["13"]},
    "NEGATIO": {"-": ["13"]},
    "BLATIO": {"-": ["13"]},
    "EXANIMATIO": {"-": ["13"]},
    "SINISTERITAS": {"-": ["13"]},
    "VERBIVELITATIO": {"-": ["13"]},
    "COEMPTIO": {"-": ["13"]},
    "CONDEMNATIO": {"-": ["13"]},
    "AEQVABILITAS": {"-": ["13"]},
    "INVECTIO": {"-": ["13"]},
    "DERELICTIO": {"-": ["13"]},
    "EMANCIPATIO": {"-": ["13"]},
    "PERCVRSATIO": {"-": ["13"]},
    "VILICATIO": {"-": ["13"]},
    "IVGATIO": {"-": ["13"]},
    "DIIVNCTIO": {"-": ["13"]},
    "AFFABILITAS": {"-": ["13"]},
    "EMBASICOETAS": {"-": ["13"]},
    "AMANDATIO": {"-": ["13"]},
    "STIMVLATIO": {"-": ["13"]},
    "PVLSATIO": {"-": ["13"]},
    "AMATIO": {"-": ["13"]},
    "LOQVACITAS": {"-": ["13"]},
    "OBSTRVCTIO": {"-": ["13"]},
    "EXPVRGATIO": {"-": ["13"]},
    "EMINATIO": {"-": ["13"]},
    "CONSORTIO": {"-": ["13"]},
    "INFLAMMATIO": {"-": ["13"]},
    "CONVOCATIO": {"-": ["13"]},
    "INVESTIGATIO": {"-": ["13"]},
    "PRAECENTIO": {"-": ["13"]},
    "OBROGATIO": {"-": ["13"]},
    "COLLIGATIO": {"-": ["13"]},
    "ADMIRABILITAS": {"-": ["13"]},
    "DOCILITAS": {"-": ["13"]},
    "PENSATIO": {"-": ["13"]},
    "FRIGVTIO": {"-": ["13"]},
    "PROCACITAS": {"-": ["13"]},
    "SORTIO": {"-": ["13"]},
    "DECLARATIO": {"-": ["13"]},
    "FESTIVITAS": {"-": ["13"]},
    "INTERFATIO": {"-": ["13"]},
    "GVSTATIO": {"-": ["13"]},
    "DEFLAGRATIO": {"-": ["13"]},
    "ANNOTATIO": {"-": ["13"]},
    "COMPELLATIO": {"-": ["13"]},
    "MALEDICTIO": {"-": ["13"]},
    "INCOMMODITAS": {"-": ["13"]},
    "CONGLVTINATIO": {"-": ["13"]},
    "CONSITIO": {"-": ["13"]},
    "EXPLICEO": {"-": ["52"]},
    "DISCRIPTIO": {"-": ["13"]},
    "ADMIXTIO": {"-": ["13"]},
    "COITIO": {"-": ["13"]},
    "GRANDITAS": {"-": ["13"]},
    "RELIGATIO": {"-": ["13"]},
    "NVNTIATIO": {"-": ["13"]},
    "PERACTIO": {"-": ["13"]},
    "IACVLATIO": {"-": ["13"]},
    "DIRITAS": {"-": ["13"]},
    "EFFRENATIO": {"-": ["13"]},
    "APPROPINQVATIO": {"-": ["13"]},
    "CELLATIO": {"-": ["13"]},
    "DESTITVTIO": {"-": ["13"]},
    "OBSCENITAS": {"-": ["13"]},
    "RECVPERATIO": {"-": ["13"]},
    "DICATIO": {"-": ["13"]},
    "INTERPVNCTIO": {"-": ["13"]},
    "INTERCEPTIO": {"-": ["13"]},
    "VIVACITAS": {"-": ["13"]},
    "STIPATIO": {"-": ["13"]},
    "QVAESITIO": {"-": ["13"]},
    "VITATIO": {"-": ["13"]},
    "EMACITAS": {"-": ["13"]},
    "INFIRMATIO": {"-": ["13"]},
    "EXPIATIO": {"-": ["13"]},
    "SVBIGITATIO": {"-": ["13"]},
    "APRICATIO": {"-": ["13"]},
    "CONFICTIO": {"-": ["13"]},
    "IRRIGATIO": {"-": ["13"]},
    "STVPIDITAS": {"-": ["13"]},
    "COMPOTATIO": {"-": ["13"]},
    "SCISCITATIO": {"-": ["13"]},
    "HELVATIO": {"-": ["13"]},
    "GERMANITAS": {"-": ["13"]},
    "ABIECTIO": {"-": ["13"]},
    "PVTATIO": {"-": ["13"]},
    "SEDVCTIO": {"-": ["13"]},
    "CONCENATIO": {"-": ["13"]},
    "INCOMITIO": {"-": ["13"]},
    "PRAEROGATIO": {"-": ["13"]},
    "SERVATIO": {"-": ["13"]},
    "DECERTATIO": {"-": ["13"]},
    "EXSECTIO": {"-": ["13"]},
    "PERAGRATIO": {"-": ["13"]},
    "RESTIPVLATIO": {"-": ["13"]},
    "EDICTIO": {"-": ["13"]},
    "AMPVTATIO": {"-": ["13"]},
    "SCRIPTIO": {"-": ["13"]},
    "SVPPOSITIO": {"-": ["13"]},
    "SATISDATIO": {"-": ["13"]},
    "PROPVLSATIO": {"-": ["13"]},
    "OPIMITAS": {"-": ["13"]},
    "INSVLSITAS": {"-": ["13"]},
    "PROCRASTINATIO": {"-": ["13"]},
    "DISPERTITIO": {"-": ["13"]},
    "DIRIBITIO": {"-": ["13"]},
    "NEGLECTIO": {"-": ["13"]},
    "PROPVGNATIO": {"-": ["13"]},
    "ILLIBERALITAS": {"-": ["13"]},
    "TEMPESTIVITAS": {"-": ["13"]},
    "OBNVNTIO": {"-": ["13"]},
    "CONSVETIO": {"-": ["13"]},
    "EPVLATIO": {"-": ["13"]},
    "IMPVRITAS": {"-": ["13"]},
    "RECLAMATIO": {"-": ["13"]},
    "MVTVATIO": {"-": ["13"]},
    "AEQVATIO": {"-": ["13"]},
    "FORMOSITAS": {"-": ["13"]},
    "VINDICATIO": {"-": ["13"]},
    "GRATIFICATIO": {"-": ["13"]},
    "DIVITATIO": {"-": ["13"]},
    "REPASTINATIO": {"-": ["13"]},
    "DEVEXITAS": {"-": ["13"]},
    "COLLOCVTIO": {"-": ["13"]},
    "MOROSITAS": {"-": ["13"]},
    "DISPERCVTIO": {"-": ["13"]},
    "DISSIPATIO": {"-": ["13"]},
    "INSATIETAS": {"-": ["13"]},
    "IMMATVRITAS": {"-": ["13"]},
    "DETRECTATIO": {"-": ["13"]},
    "COACTIO": {"-": ["13"]},
    "ASSIGNATIO": {"-": ["13"]},
    "PROMVLGATIO": {"-": ["13"]},
    "IMPLICATIO": {"-": ["13"]},
    "ASSERTIO": {"-": ["13"]},
    "INTERDICTIO": {"-": ["13"]},
    "DERIVATIO": {"-": ["13"]},
    "DEBITIO": {"-": ["13"]},
    "OPTATIO": {"-": ["13"]},
    "OBLIQVITAS": {"-": ["13"]},
    "ELOCVTIO": {"-": ["13"]},
    "INSPECTATIO": {"-": ["13"]},
    "ROGITATIO": {"-": ["13"]},
    "CONSVRRECTIO": {"-": ["13"]},
    "CALVIA": {"N": ["11"]},
    "GALLITTA": {"N": ["11"]},
    "IARBITAS": {"N": ["13"]},
    "AMYNTAS": {"N": ["13"]},
    "DAMOETAS": {"N": ["13"]},
    "LYCOTAS": {"N": ["13"]},
    "GORGATAS": {"N": ["13"]},
    "PHILETAS": {"N": ["13"]},
    "HORRATAS": {"N": ["13"]},
    "ARCHYTAS": {"N": ["13"]},
    "EVROTAS": {"N": ["13"]},
    "FAVSTITAS": {"N": ["13"]},
    "PHILOTAS": {"N": ["13"]},
    "AERIFER": {"-": ["12"]},
    "DIFFIDENTER": {"-": ["12"]},
    "MELEAGER": {"N": ["12"]},
    "FATALITER": {"-": ["12"]},
    "INSIPIENTER": {"-": ["12"]},
    "GRANDIFER": {"-": ["12"]},
    "ELOQVENTER": {"-": ["12"]},
    "TVRRIFER": {"-": ["12"]},
    "SCAMANDER": {"N": ["12"]},
    "INCONTINENTER": {"-": ["12"]},
    "SOTER": {"N": ["12"]},
    "IRREVERENTER": {"-": ["12"]},
    "THYMBER": {"N": ["12"]},
    "HISTER": {"N": ["12"]},
    "AFER": {"N": ["12"], "A": ["23"]},
    "RACEMIFER": {"-": ["12"]},
    "CAPITALITER": {"-": ["12"]},
    "AMPHITHEATER": {"-": ["12"]},
    "NEGLEGENTER": {"-": ["12"]},
    "FVMIFER": {"-": ["12"]},
    "CONFIDENTER": {"-": ["12"]},
    "FVRIALITER": {"-": ["12"]},
    "FVRACITER": {"-": ["12"]},
    "INSIGNITER": {"-": ["12"]},
    "PVELLARITER": {"-": ["12"]},
    "LATENTER": {"-": ["12"]},
    "DENTER": {"N": ["12"]},
    "TEVCER": {"N": ["12"], "A": ["23"]},
    "PERSAPIENTER": {"-": ["12"]},
    "CIMBER": {"N": ["12"], "A": ["23"]},
    "SOCIALITER": {"-": ["12"]},
    "ASSIMILITER": {"-": ["12"]},
    "INTELLIGENTER": {"-": ["12"]},
    "DVLCITER": {"-": ["12"]},
    "VILITER": {"-": ["12"]},
    "ELAVER": {"N": ["12"]},
    "LYSANDER": {"N": ["12"]},
    "OLIVIFER": {"-": ["12"]},
    "TANAGER": {"N": ["12"]},
    "MONSTRIFER": {"-": ["12"]},
    "NOTABILITER": {"-": ["12"]},
    "PRODIGIALITER": {"-": ["12"]},
    "TITVBANTER": {"-": ["12"]},
    "TRISTITER": {"-": ["12"]},
    "EXCELLENTER": {"-": ["12"]},
    "ALCANDER": {"N": ["12"]},
    "OBSEQVENTER": {"-": ["12"]},
    "PINIGER": {"-": ["12"]},
    "PERBREVITER": {"-": ["12"]},
    "PERCELER": {"-": ["12"]},
    "ANTONIASTER": {"N": ["12"]},
    "SPHINTER": {"N": ["12"]},
    "AMPLITER": {"-": ["12"]},
    "BLANDITER": {"-": ["12"]},
    "CONGER": {"-": ["12"]},
    "EXSVLTANTER": {"-": ["12"]},
    "ALABASTER": {"-": ["12"]},
    "ASANDER": {"N": ["12"]},
    "SACRIFER": {"-": ["12"]},
    "SPINTHER": {"N": ["12"]},
    "ENSIFER": {"-": ["12"]},
    "DOLENTER": {"-": ["12"]},
    "CREDIBILITER": {"-": ["12"]},
    "CASSANDER": {"N": ["12"]},
    "ANTIPATER": {"N": ["12"]},
    "SERVILITER": {"-": ["12"]},
    "SOMNIFER": {"-": ["12"]},
    "MILITARITER": {"-": ["12"]},
    "MVLCIBER": {"N": ["12"]},
    "CLEANDER": {"N": ["12"]},
    "HERBIFER": {"-": ["12"]},
    "CORYMBIFER": {"-": ["12"]},
    "ARISTANDER": {"N": ["12"]},
    "SCVRRILITER": {"-": ["12"]},
    "IMPERABILITER": {"-": ["12"]},
    "PISANDER": {"N": ["12"]},
    "VNGVLATER": {"-": ["12"]},
    "LEANDER": {"N": ["12"]},
    "FALLACITER": {"-": ["12"]},
    "PACIFER": {"-": ["12"]},
    "PRAESTANTER": {"-": ["12"]},
    "SEMICAPER": {"-": ["12"]},
    "TRVCVLENTER": {"-": ["12"]},
    "DIESPITER": {"N": ["12"]},
    "TVDER": {"N": ["12"]},
    "REDVNDANTER": {"-": ["12"]},
    "IMPARITER": {"-": ["12"]},
    "PHAGER": {"-": ["12"]},
    "CORPORALITER": {"-": ["12"]},
    "MNESTER": {"N": ["12"]},
    "TRIFVRCIFER": {"-": ["12"]},
    "MAEANDER": {"N": ["12"], "-": ["12"]},
    "APPETENTER": {"-": ["12"]},
    "BICORNIGER": {"N": ["12"]},
    "MALIFER": {"-": ["12"]},
    "ARVNDIFER": {"-": ["12"]},
    "MENANDER": {"N": ["12"]},
    "ANTHEMVSIAS": {"N": ["17"]},
    "ARSANIAS": {"N": ["17"]},
    "ARCHIAS": {"N": ["17"]},
    "ADARRHIAS": {"N": ["17"]},
    "ATARRHIAS": {"N": ["17"]},
    "HARPOCRAS": {"N": ["17"]},
    "HABINNAS": {"N": ["17"]},
    "TIRESIAS": {"N": ["17"]},
    "ATTINAS": {"N": ["17"]},
    "PNYTAGORAS": {"N": ["17"]},
    "SCIPIADAS": {"N": ["17"]},
    "MIMAS": {"N": ["17"]},
    "CASPIAS": {"N": ["17"]},
    "ONOMAS": {"N": ["17"]},
    "MIDAS": {"N": ["17"]},
    "DERDAS": {"N": ["17"]},
    "LYCORMAS": {"N": ["17"]},
    "DRYAS": {"N": ["17"]},
    "GNOSIAS": {"N": ["17"], "A": ["25"]},
    "LYSIAS": {"N": ["17"]},
    "PHIDIAS": {"N": ["17"]},
    "ERYMAS": {"N": ["17"]},
    "TROAS": {"N": ["17"], "A": ["25"]},
    "NAIAS": {"N": ["17"]},
    "SYMPLEGAS": {"N": ["17"]},
    "IOLLAS": {"N": ["17"]},
    "LADAS": {"N": ["17"]},
    "CYDAS": {"N": ["17"]},
    "ANAXAGORAS": {"N": ["17"]},
    "SIMIAS": {"N": ["17"]},
    "ABAS": {"N": ["17"]},
    "ARCAS": {"N": ["17"], "A": ["25"]},
    "PYTHIAS": {"N": ["17"]},
    "ATHENAGORAS": {"N": ["17"]},
    "DYMAS": {"N": ["17"]},
    "SCOPAS": {"N": ["17"]},
    "HYLAS": {"N": ["17"]},
    "ACHILLAS": {"N": ["17"]},
    "THESTIAS": {"N": ["17"]},
    "PERDICCAS": {"N": ["17"]},
    "HIERAS": {"N": ["17"]},
    "ATYANAS": {"N": ["17"]},
    "CALLICRATIDAS": {"N": ["17"]},
    "PTERELAS": {"N": ["17"]},
    "THOAS": {"N": ["17"]},
    "PRVSIAS": {"N": ["17"]},
    "GYAS": {"N": ["17"]},
    "CORYBAS": {"N": ["17"]},
    "ASILAS": {"N": ["17"]},
    "EVHYAS": {"N": ["17"]},
    "PAVSANIAS": {"N": ["17"]},
    "MENIDAS": {"N": ["17"]},
    "PERIPHAS": {"N": ["17"]},
    "PHORBAS": {"N": ["17"]},
    "ATHARRIAS": {"N": ["17"]},
    "AVGEAS": {"N": ["17"]},
    "THAVMANTIAS": {"N": ["17"]},
    "PELIAS": {"N": ["17"], "A": ["25"]},
    "PANTAGIAS": {"N": ["17"]},
    "GELAS": {"N": ["17"]},
    "MAENIDAS": {"N": ["17"]},
    "THAMYRAS": {"N": ["17"]},
    "IARBAS": {"N": ["17"]},
    "PHINTIAS": {"N": ["17"]},
    "ORCAS": {"N": ["17"]},
    "ELIAS": {"N": ["17"]},
    "PALLANTIAS": {"N": ["17"]},
    "LICHAS": {"N": ["17"]},
    "PHOEBAS": {"N": ["17"]},
    "PLEIAS": {"N": ["17"]},
    "ACAMAS": {"N": ["17"]},
    "ATHANAGORAS": {"N": ["17"]},
    "BELIAS": {"N": ["17"]},
    "PHTHIAS": {"N": ["17"]},
    "PROTAGORAS": {"N": ["17"]},
    "ALEVAS": {"N": ["17"]},
    "AERIAS": {"N": ["17"]},
    "HYAS": {"N": ["17"]},
    "CINEAS": {"N": ["17"]},
    "CINYRAS": {"N": ["17"]},
    "PELORIAS": {"N": ["17"]},
    "CARRINAS": {"N": ["17"]},
    "TACFARINAS": {"N": ["17"]},
    "ARTAXIAS": {"N": ["17"]},
    "ARPINAS": {"N": ["17"], "A": ["25"]},
    "THOANTIAS": {"N": ["17"]},
    "DIPSAS": {"N": ["17"]},
    "THIODAMAS": {"N": ["17"]},
    "LYCIDAS": {"N": ["17"]},
    "APPIAS": {"N": ["17"]},
    "EVRYDAMAS": {"N": ["17"]},
    "BITIAS": {"N": ["17"]},
    "PASAS": {"N": ["17"]},
    "IOPAS": {"N": ["17"]},
    "LAENAS": {"N": ["17"]},
    "PHASIAS": {"N": ["17"]},
    "BROTEAS": {"N": ["17"]},
    "NOMAS": {"N": ["17"]},
    "MENALCAS": {"N": ["17"]},
    "MARSYAS": {"N": ["17"]},
    "HELLAS": {"N": ["17"]},
    "CORAS": {"N": ["17"]},
    "CLVSINAS": {"N": ["17"]},
    "POEAS": {"N": ["17"]},
    "PLIAS": {"N": ["17"]},
    "PSECAS": {"N": ["17"]},
    "BIAS": {"N": ["17"]},
    "CALCHAS": {"N": ["17"]},
    "TEVTHRAS": {"N": ["17"]},
    "ATHAMAS": {"N": ["17"]},
    "PHLEGYAS": {"N": ["17"]},
    "IPHIAS": {"N": ["17"]},
    "BAGOAS": {"N": ["17"]},
    "TAMIRAS": {"N": ["17"]},
    "THEMILLAS": {"N": ["17"]},
    "PERLIBERALITER": {"-": ["12"]},
    "PRAETERPROPTER": {"-": ["12"]},
    "ARELAS": {"N": ["17"]},
    "IDAS": {"N": ["17"]},
    "FELGINAS": {"N": ["17"]},
    "ATLAS": {"N": ["17"]},
    "THYIAS": {"N": ["17"]},
    "LYCORIAS": {"N": ["17"]},
    "GORGIAS": {"N": ["17"]},
    "OLYMPIAS": {"N": ["17"]},
    "EPAMINONDAS": {"N": ["17"]},
    "HIPPIAS": {"N": ["17"]},
    "ACRAGAS": {"N": ["17"]},
    "XANTHIAS": {"N": ["17"]},
    "BVCEPHALAS": {"N": ["17"]},
    "ILIAS": {"N": ["17"]},
    "HERMAGORAS": {"N": ["17"]},
    "ATHENION": {"N": ["17"]},
    "LYCON": {"N": ["17"]},
    "STIMICHON": {"N": ["17"]},
    "EMATHION": {"N": ["17"]},
    "OROMEDON": {"N": ["17"]},
    "SYNODON": {"-": ["17"]},
    "MEDON": {"N": ["17"]},
    "CALCHEDON": {"N": ["17"]},
    "NICASION": {"N": ["17"]},
    "PORPHYRION": {"N": ["17"]},
    "AVEON": {"N": ["17"]},
    "MARATHON": {"N": ["17"]},
    "LYCAON": {"N": ["17"]},
    "HAEMON": {"N": ["17"]},
    "LEVCON": {"N": ["17"]},
    "AVTOMEDON": {"N": ["17"]},
    "PELAGON": {"N": ["17"]},
    "BITON": {"N": ["17"]},
    "PYGMALION": {"N": ["17"]},
    "CHIRON": {"N": ["17"]},
    "TIMON": {"N": ["17"]},
    "IXION": {"N": ["17"]},
    "SASON": {"N": ["17"]},
    "HYPERION": {"N": ["17"]},
    "TAVRON": {"N": ["17"]},
    "BABYLON": {"N": ["17"]},
    "AGATHON": {"N": ["17"]},
    "CALYDON": {"N": ["17"]},
    "PLEVRON": {"N": ["17"]},
    "XENOPHON": {"N": ["17"]},
    "ALCIMEDON": {"N": ["17"]},
    "OLYMPION": {"N": ["17"]},
    "PALAEMON": {"N": ["17"]},
    "AMAZON": {"N": ["17"]},
    "PHITOLEON": {"N": ["17"]},
    "DEVCALION": {"N": ["17"]},
    "CREON": {"N": ["17"]},
    "MAEON": {"N": ["17"]},
    "ENDYMION": {"N": ["17"]},
    "IASON": {"N": ["17"]},
    "AGAMEMNON": {"N": ["17"]},
    "POTAMON": {"N": ["17"]},
    "POLEMON": {"N": ["17"]},
    "PATRON": {"N": ["17"]},
    "ARISTION": {"N": ["17"]},
    "AMYDON": {"N": ["17"]},
    "PROCOETON": {"-": ["17"]},
    "HECATON": {"N": ["17"]},
    "ARTAMON": {"N": ["17"]},
    "COLOPHON": {"N": ["17"]},
    "EVRYTION": {"N": ["17"]},
    "ELEGIDARION": {"-": ["17"]},
    "SILANION": {"N": ["17"]},
    "HERACLEON": {"N": ["17"]},
    "APION": {"N": ["17"]},
    "ZENON": {"N": ["17"]},
    "GERYON": {"N": ["17"]},
    "THEON": {"N": ["17"]},
    "LEON": {"N": ["17"]},
    "XENON": {"N": ["17"]},
    "DEMOPHOON": {"N": ["17"]},
    "PARMENON": {"N": ["17"]},
    "TELAMON": {"N": ["17"]},
    "MACHAON": {"N": ["17"]},
    "HIPPON": {"N": ["17"]},
    "LYCMON": {"N": ["17"]},
    "HEGION": {"N": ["17"]},
    "BOLON": {"N": ["17"]},
    "CLEON": {"N": ["17"]},
    "LADON": {"N": ["17"]},
    "METRON": {"N": ["17"]},
    "DEMOPHON": {"N": ["17"]},
    "CRANNON": {"N": ["17"]},
    "HAMMON": {"N": ["17"]},
    "HISBON": {"N": ["17"]},
    "PHORMION": {"N": ["17"]},
    "GITON": {"N": ["17"]},
    "PYRRHON": {"N": ["17"]},
    "ALISON": {"N": ["17"]},
    "VRION": {"N": ["17"]},
    "STRYMON": {"N": ["17"]},
    "PHAON": {"N": ["17"]},
    "NICON": {"N": ["17"]},
    "CTESIPHON": {"N": ["17"]},
    "CITHAERON": {"N": ["17"]},
    "THERON": {"N": ["17"]},
    "PHAEDON": {"N": ["17"]},
    "AMMON": {"N": ["17"]},
    "POLYPEMON": {"N": ["17"]},
    "TELON": {"N": ["17"]},
    "DIDYMEON": {"N": ["17"]},
    "ACHERON": {"N": ["17"]},
    "LACON": {"N": ["17"], "A": ["25"]},
    "HIERON": {"N": ["17"]},
    "CONON": {"N": ["17"]},
    "PYTHON": {"N": ["17"]},
    "MATHON": {"N": ["17"]},
    "LYSON": {"N": ["17"]},
    "SIMON": {"N": ["17"]},
    "ANACREON": {"N": ["17"]},
    "MELON": {"N": ["17"]},
    "EVDAEMON": {"N": ["17"]},
    "LYCOPHRON": {"N": ["17"]},
    "AVLON": {"N": ["17"]},
    "DRACON": {"N": ["17"]},
    "LEOMEDON": {"N": ["17"]},
    "DAMASICTHON": {"N": ["17"]},
    "SCIRON": {"N": ["17"]},
    "PLVTON": {"N": ["17"]},
    "DAMON": {"N": ["17"]},
    "NYMPHON": {"N": ["17"]},
    "TARCHON": {"N": ["17"]},
    "BLEPHARON": {"N": ["17"]},
    "VCALEGON": {"N": ["17"]},
    "DIDYMAON": {"N": ["17"]},
    "ACRON": {"N": ["17"]},
    "THRASON": {"N": ["17"]},
    "ECHION": {"N": ["17"]},
    "PHARON": {"N": ["17"]},
    "CROTON": {"N": ["17"]},
    "MENON": {"N": ["17"]},
    "CHAON": {"N": ["17"]},
    "SERAPION": {"N": ["17"]},
    "DOLICHAON": {"N": ["17"]},
    "HERACON": {"N": ["17"]},
    "ANDROGEON": {"N": ["17"]},
    "CAVLON": {"N": ["17"]},
    "GNATHON": {"N": ["17"]},
    "PROCYON": {"N": ["17"]},
    "HEPHAESTION": {"N": ["17"]},
    "EVCTEMON": {"N": ["17"]},
    "PHAETHON": {"N": ["17"]},
    "HIPPOCOON": {"N": ["17"]},
    "AMPHION": {"N": ["17"]},
    "LAOMEDON": {"N": ["17"]},
    "ORION": {"N": ["17"]},
    "OARION": {"N": ["17"]},
    "SOCRATION": {"N": ["17"]},
    "POLYPERCON": {"N": ["17"]},
    "ARISTON": {"N": ["17"]},
    "SOLON": {"N": ["17"]},
    "THAEMON": {"N": ["17"]},
    "IMAON": {"N": ["17"]},
    "STRABON": {"N": ["17"]},
    "NOEMON": {"N": ["17"]},
    "SOTION": {"N": ["17"]},
    "PANDION": {"N": ["17"]},
    "MILANION": {"N": ["17"]},
    "ARION": {"N": ["17"]},
    "AEGON": {"N": ["17"]},
    "ATHLON": {"-": ["17"]},
    "DAPHNON": {"-": ["17"]},
    "ARTEMON": {"N": ["17"]},
    "CORYDON": {"N": ["17"]},
    "HYPERBATON": {"-": ["17"]},
    "AESCHRION": {"N": ["17"]},
    "MICON": {"N": ["17"]},
    "MYRON": {"N": ["17"]},
    "ACTAEON": {"N": ["17"]},
    "MILON": {"N": ["17"]},
    "PARMENION": {"N": ["17"]},
    "SARPEDON": {"N": ["17"]},
    "CHARON": {"N": ["17"]},
    "STRATON": {"N": ["17"]},
    "IDMON": {"N": ["17"]},
    "DION": {"N": ["17"]},
    "PHILON": {"N": ["17"]},
    "MYOPARON": {"-": ["17"]},
    "AMPHITRYON": {"N": ["17"]},
    "PYRACMON": {"N": ["17"]},
    "CYDON": {"N": ["17"]},
    "SINON": {"N": ["17"]},
    "THERMODON": {"N": ["17"]},
    "SICYON": {"N": ["17"]},
    "SVNION": {"N": ["17"]},
    "HYPOCAVSTON": {"-": ["17"]},
    "CIMON": {"N": ["17"]},
    "CALLIPHON": {"N": ["17"]},
    "SATYRION": {"-": ["17"]},
    "AETHON": {"N": ["17"]},
    "ACMON": {"N": ["17"]},
    "THESPRION": {"N": ["17"]},
    "MEMNON": {"N": ["17"]},
    "PLATON": {"N": ["17"]},
    "THEMISON": {"N": ["17"]},
    "EETION": {"N": ["17"]},
    "ALIACMON": {"N": ["17"]},
    "EVCLION": {"N": ["17"]},
    "PELION": {"N": ["17"]},
    "ALCON": {"N": ["17"]},
    "TETRASTYLON": {"-": ["17"]},
    "STILBON": {"N": ["17"]},
    "GLYCON": {"N": ["17"]},
    "DOLON": {"-": ["13"], "N": ["21"]},
    "TRICHON": {"N": ["17"]},
    "PRASINVS": {"-": ["12"]},
    "PRISCE": {"-": ["6-"]},
    "TRICENVS": {"-": ["21"]},
    "AVSVS": {"-": ["21"]},
    "SEPTIENS": {"-": ["6-"]},
    "SVLMO": {"N": ["13"]},
    "ADBITO": {"-": ["53"]},
    "MALITIOSVS": {"-": ["21"]},
    "PHAECASIA": {"-": ["11"]},
    "HECTOR": {"N": ["13"]},
    "FVSTITVDINVS": {"-": ["21"]},
    "PETAVRISTARIVS": {"-": ["12"]},
    "PROMISSOR": {"-": ["13"]},
    "ENTHYMEMA": {"-": ["13"]},
    "HAESITABVNDVS": {"-": ["21"]},
    "CANTAMEN": {"-": ["13"]},
    "DISPOLIO": {"-": ["51"]},
    "CONGREGABILIS": {"-": ["24"]},
    "EVRYNOME": {"N": ["11"]},
    "LYDE": {"N": ["11"]},
    "VLTIMO": {"2": ["51"]},
    "EXAVSPICO": {"-": ["51"]},
    "HALOPHANTA": {"-": ["11"]},
    "DELIBERATOR": {"-": ["13"]},
    "ARVNDINEVS": {"-": ["21"]},
    "PHYLACA": {"-": ["11"]},
    "LECTICVLA": {"-": ["11"]},
    "SVBIGITO": {"-": ["51"]},
    "INTESTINA": {"-": ["12"]},
    "PERFIDIOSE": {"-": ["6-"]},
    "CATEIA": {"-": ["11"]},
    "STEGA": {"-": ["11"]},
    "CAVTOR": {"-": ["13"]},
    "IMPASTVS": {"-": ["21"]},
    "SORORIVS": {"-": ["12"]},
    "SPATALOCINAEDVS": {"-": ["12"]},
    "POSTILIO": {"-": ["13"]},
    "FESTINVS": {"-": ["21"]},
    "INDIGNATIVNCVLA": {"-": ["11"]},
    "PRAETERBITO": {"-": ["53"]},
    "INANIAE": {"-": ["11"]},
    "AETATVLA": {"-": ["11"]},
    "PRODIGIALIS": {"-": ["24"]},
    "INFORO": {"-": ["51"]},
    "CROTALISTRIA": {"-": ["11"]},
    "STIGMOSVS": {"-": ["21"]},
    "TONSORIVS": {"-": ["21"]},
    "SVPERVOLITO": {"-": ["51"]},
    "LAMELLVLA": {"-": ["11"]},
    "RIDEOR": {"-": ["5K"]},
    "CORNEOLVS": {"-": ["21"]},
    "AEQVIPARABILIS": {"-": ["24"]},
    "CAVSIFICOR": {"-": ["13"]},
    "DELIQVVS": {"-": ["21"]},
    "DEPVLSOR": {"-": ["13"]},
    "PASTORICIVS": {"-": ["21"]},
    "OPTEMPERO": {"-": ["51"]},
    "CIRCVMSPECTATRIX": {"-": ["13"]},
    "CRITICA": {"-": ["12"]},
    "LAVDICENVS": {"-": ["12"]},
    "LATERNARIVS": {"-": ["12"]},
    "OBSCAEVO": {"-": ["51"]},
    "ELEVSIN": {"N": ["13"]},
    "SISAPO": {"N": ["13"]},
    "NEREIS": {"N": ["13"]},
    "SCYLAX": {"N": ["13"]},
    "ROGATIVNCVLA": {"-": ["11"]},
    "BELLIPOTENS": {"-": ["22"]},
    "MERCATORIVS": {"-": ["21"]},
    "LATERENSIS": {"N": ["13"]},
    "DVELLATOR": {"-": ["13"]},
    "REPVGNANTIA": {"-": ["13"]},
    "COPIS": {"-": ["24"]},
    "HAEDILLVS": {"-": ["12"]},
    "MISCIX": {"-": ["22"]},
    "PASTORALIS": {"-": ["24"]},
    "LVCRETILIS": {"N": ["13"]},
    "CENATORIA": {"-": ["12"]},
    "PROPINQVE": {"-": ["6-"]},
    "SVPPINGO": {"-": ["53"]},
    "POLYDAMAS": {"N": ["13"]},
    "NEVTIQVAM": {"-": ["6-"]},
    "VASVS": {"-": ["12"]},
    "INTERNECO": {"-": ["51"]},
    "AVTOPYROS": {"-": ["17"]},
    "QVATERNVS": {"-": ["21"]},
    "ARIETINVS": {"-": ["21"]},
    "LENONIVS": {"-": ["21"]},
    "MORIGERO": {"-": ["51"]},
    "EMINOR": {"-": ["5J"]},
    "TAPES": {"-": ["13"]},
    "COMPITALICIVS": {"-": ["21"]},
    "SOLARIVM": {"2": ["12"]},
    "BALISCVS": {"-": ["12"]},
    "VIVE": {"-": ["6-"]},
    "INDVSIATVS": {"-": ["21"]},
    "PALMVLVS": {"-": ["12"]},
    "IMPERATRIX": {"-": ["13"]},
    "SEPTIFLVVS": {"-": ["21"]},
    "ABITO": {"-": ["53"]},
    "INSECTO": {"-": ["51"]},
    "FIDICINA": {"-": ["11"]},
    "PEREGRI": {"-": ["6-"]},
    "LIMAX": {"-": ["13"]},
    "CROCOTVLA": {"-": ["11"]},
    "LINGVLACA": {"-": ["11"]},
    "CENTIENS": {"-": ["6-"]},
    "CISTVLA": {"-": ["11"]},
    "PACTA": {"-": ["11"]},
    "LIBIDINOR": {"-": ["5J"]},
    "STRIGA": {"2": ["11"]},
    "AMASIS": {"N": ["13"]},
    "DAPHNIS": {"N": ["13"]},
    "LVSITO": {"-": ["51"]},
    "LIMONE": {"N": ["11"]},
    "DEVIRGINO": {"-": ["51"]},
    "RVSTICOR": {"-": ["5J"]},
    "INSPVTO": {"-": ["51"]},
    "CANTVRIO": {"-": ["54"]},
    "COQVINO": {"-": ["51"]},
    "APPINGO": {"-": ["53"]},
    "SALAMIS": {"N": ["13"]},
    "SENVS": {"-": ["21"]},
    "PVGNATOR": {"-": ["13"]},
    "ARIETILLVS": {"-": ["21"]},
    "DVODECIENS": {"-": ["6-"]},
    "PARENTALIA": {"-": ["13"]},
    "TRITON": {"N": ["13"]},
    "PROPONTIS": {"N": ["13"]},
    "ATYS": {"N": ["13"]},
    "FALSE": {"-": ["6-"]},
    "PERFVNCTORIE": {"-": ["6-"]},
    "INSTRVCTOR": {"-": ["13"]},
    "SIMITV": {"-": ["6-"]},
    "ECLOGA": {"-": ["11"]},
    "PARTHENOPE": {"N": ["11"]},
    "VESERIS": {"N": ["13"]},
    "FLAMMARIVS": {"-": ["12"]},
    "ARBITRARIVS": {"-": ["21"]},
    "PANNVCIA": {"-": ["11"]},
    "INFIRME": {"-": ["6-"]},
    "CONTVMVLO": {"-": ["51"]},
    "DIFFLAGITO": {"-": ["51"]},
    "MYTILENE": {"N": ["11"]},
    "PATINARIVS": {"-": ["21"]},
    "CASTIGATORIVS": {"-": ["21"]},
    "CELAENO": {"N": ["13"]},
    "ZEPHYRITIS": {"N": ["13"]},
    "OENONE": {"N": ["11"]},
    "SCVRROR": {"-": ["5J"]},
    "CONIVGO": {"-": ["51"]},
    "STERNAX": {"-": ["22"]},
    "CAPO": {"-": ["13"]},
    "HIPPVRVS": {"-": ["12"]},
    "HOSTIO": {"1": ["54"]},
    "INSIPIENTIA": {"-": ["11"]},
    "ORACVLARIVS": {"-": ["21"]},
    "FVNDATOR": {"1": ["13"]},
    "SCALPVRRIO": {"-": ["54"]},
    "IRREVERENS": {"-": ["22"]},
    "INDIPISCO": {"-": ["53"]},
    "SITIENS": {"-": ["2&"]},
    "CONGRAECO": {"-": ["51"]},
    "DIARIA": {"-": ["12"]},
    "ASSESSVS": {"-": ["14"]},
    "MELILOTOS": {"-": ["17"]},
    "COACESCO": {"-": ["53"]},
    "AVELLANA": {"-": ["11"]},
    "EXPETESSO": {"-": ["53"]},
    "OBSVO": {"-": ["53"]},
    "COTYS": {"N": ["13"]},
    "MEROBIBVS": {"-": ["21"]},
    "BEATITVDO": {"-": ["13"]},
    "INTERSCRIBO": {"-": ["53"]},
    "CONSVLTE": {"-": ["6-"]},
    "GAVSAPE": {"-": ["13"]},
    "HEREDIPETA": {"-": ["11"]},
    "PERNECESSARIVS": {"-": ["21"]},
    "HECALE": {"N": ["11"]},
    "BVLEVTA": {"-": ["11"]},
    "DEAMO": {"-": ["51"]},
    "DESOMNIS": {"-": ["24"]},
    "HYDRAVLES": {"-": ["11"]},
    "VIATICVS": {"-": ["12"]},
    "ACLYS": {"-": ["13"]},
    "VITIOSE": {"-": ["6-"]},
    "SEDECIES": {"-": ["6-"]},
    "EXSVSCITO": {"1": ["51"]},
    "CILO": {"N": ["13"]},
    "MVLTIPOTENS": {"-": ["22"]},
    "NOCTIVIGILVS": {"-": ["21"]},
    "ZOTHECA": {"-": ["11"]},
    "QVINDECIENS": {"-": ["6-"]},
    "PRAEDVLCIS": {"-": ["24"]},
    "TARRACO": {"N": ["13"]},
    "COLLABEFACTO": {"-": ["51"]},
    "COCIO": {"-": ["13"]},
    "INSALVTATVS": {"-": ["21"]},
    "DEFAECO": {"-": ["51"]},
    "TITAN": {"N": ["13"]},
    "ALARIS": {"-": ["24"]},
    "DEONERO": {"-": ["51"]},
    "LAGVNCVLA": {"-": ["11"]},
    "PERSVASVS": {"-": ["14"]},
    "BAIVLO": {"-": ["51"]},
    "ERADICO": {"-": ["51"]},
    "APOCVLO": {"-": ["51"]},
    "INTERCEPTOR": {"-": ["13"]},
    "SERMONOR": {"-": ["5J"]},
    "AQVILEX": {"-": ["13"]},
    "COMVLA": {"-": ["11"]},
    "SVCOSVS": {"-": ["21"]},
    "PERFLAGITIOSVS": {"-": ["21"]},
    "ORESTIS": {"N": ["13"]},
    "DIGNITOSVS": {"-": ["21"]},
    "SVPEREGREDIOR": {"-": ["5N"]},
    "EXPVLSIO": {"-": ["13"]},
    "PERPLEXOR": {"-": ["5J"]},
    "SYBARIS": {"N": ["13"]},
    "OPPESSVLATVS": {"-": ["21"]},
    "DISSENSVS": {"-": ["14"]},
    "REDIVIA": {"-": ["11"]},
    "BYLLIS": {"N": ["13"]},
    "CENATICVS": {"-": ["21"]},
    "CATAPIRATERIA": {"-": ["11"]},
    "HELLE": {"N": ["11"]},
    "AGRICVLTVRA": {"-": ["11"]},
    "INDELECTATVS": {"-": ["21"]},
    "CONSAEPIO": {"-": ["54"]},
    "TRACHIN": {"N": ["13"]},
    "NOVACVLA": {"-": ["11"]},
    "CONTRVNCO": {"-": ["51"]},
    "INDEPLORATVS": {"-": ["21"]},
    "INCONTINENTIA": {"-": ["11"]},
    "CARINVS": {"-": ["21"]},
    "DAMMA": {"-": ["11"]},
    "CIRCVMVISO": {"-": ["53"]},
    "OTHRYS": {"N": ["13"]},
    "SENTENTIOLA": {"-": ["11"]},
    "OGGANNIO": {"-": ["54"]},
    "INEXSTINCTVS": {"-": ["21"]},
    "MANNVLVS": {"-": ["12"]},
    "HIPPODROMVS": {"-": ["12"]},
    "HERMIONE": {"N": ["11"]},
    "BVRDVBASTA": {"-": ["11"]},
    "PARALYTICVS": {"-": ["21"]},
    "CONFLICTOR": {"2": ["5J"]},
    "DEOSCVLOR": {"-": ["5J"]},
    "TOLVTIM": {"-": ["6-"]},
    "MALEDICE": {"-": ["6-"]},
    "PLECTILIS": {"-": ["24"]},
    "VIOLARIVS": {"-": ["12"]},
    "REVIRESCO": {"-": ["53"]},
    "EVISCERO": {"-": ["51"]},
    "EBVRATVS": {"-": ["21"]},
    "FALSIMONIA": {"-": ["11"]},
    "PERISTROMA": {"-": ["13"]},
    "MODERO": {"-": ["51"]},
    "IMPERTERRITVS": {"-": ["21"]},
    "SEMIGRO": {"-": ["51"]},
    "CHORAVLA": {"-": ["11"]},
    "VIDVLVS": {"-": ["12"]},
    "LVSIO": {"-": ["13"]},
    "EXTERSVS": {"-": ["21"]},
    "FABARIS": {"N": ["13"]},
    "POLYMITVS": {"-": ["21"]},
    "SERMVNCVLVS": {"-": ["12"]},
    "OPINO": {"-": ["51"]},
    "DELACERO": {"-": ["51"]},
    "NATALICIA": {"-": ["11"]},
    "CERASINVS": {"-": ["21"]},
    "EXPRESSE": {"-": ["6-"]},
    "ALMO": {"N": ["13"]},
    "AMBESTRIX": {"-": ["13"]},
    "QVINQVEVIRATVS": {"-": ["14"]},
    "EPVLARIS": {"-": ["24"]},
    "DEVERSITOR": {"-": ["13"]},
    "SEXCENTIENS": {"-": ["6-"]},
    "SCROFIPASCVS": {"-": ["21"]},
    "INCERTO": {"1": ["6-"]},
    "ANXVR": {"N": ["13"]},
    "SYCOPHANTIA": {"-": ["11"]},
    "FVLIGINEVS": {"-": ["21"]},
    "POETICE": {"2": ["17"], "-": ["12"], "1": ["12"]},
    "PROMPTO": {"-": ["51"]},
    "PRISTIS": {"-": ["13"], "N": ["13"]},
    "BINOMINIS": {"-": ["24"]},
    "QVISQVILIA": {"-": ["12"]},
    "SVPRASCRIBO": {"-": ["53"]},
    "COMMENDATVS": {"-": ["21"]},
    "SPERABILIS": {"-": ["24"]},
    "BVCINVS": {"-": ["12"]},
    "PRANSVS": {"2": ["14"]},
    "TRANSFVSIO": {"-": ["13"]},
    "PERELEGANS": {"-": ["22"]},
    "VANITVDO": {"-": ["13"]},
    "PRAESERVIO": {"-": ["54"]},
    "HORDEIA": {"-": ["11"]},
    "SVBAVSCVLTO": {"-": ["51"]},
    "THYONE": {"N": ["11"]},
    "EDICTO": {"-": ["51"]},
    "GORGE": {"N": ["11"]},
    "DEMANDO": {"2": ["51"]},
    "ERATO": {"N": ["13"]},
    "INVIOLATE": {"-": ["6-"]},
    "DIVEXO": {"-": ["51"]},
    "VENERO": {"-": ["51"]},
    "EXSONO": {"1": ["51"]},
    "CORONIS": {"N": ["13"]},
    "IMPVRE": {"-": ["6-"]},
    "PVGILICE": {"-": ["6-"]},
    "MANTISCINOR": {"-": ["5J"]},
    "HELOPS": {"-": ["13"]},
    "MALEDICTOR": {"-": ["13"]},
    "SVCRO": {"N": ["13"]},
    "CVRSVRA": {"-": ["11"]},
    "MANTO": {"N": ["13"]},
    "EXHERES": {"-": ["13"]},
    "PERTRACTATE": {"-": ["6-"]},
    "SALTATRIX": {"-": ["13"]},
    "BARBVLA": {"-": ["11"]},
    "AFFIRMATE": {"-": ["6-"]},
    "AMASIVNCVLVS": {"-": ["12"]},
    "RESTITO": {"-": ["51"]},
    "IRRELIGIOSVS": {"-": ["21"]},
    "ARMAMENTA": {"-": ["12"]},
    "PERCOPIOSVS": {"-": ["21"]},
    "EXHORTO": {"-": ["51"]},
    "MATVLA": {"-": ["11"]},
    "CALCITROSVS": {"-": ["21"]},
    "NERVIA": {"-": ["12"]},
    "RECENSIO": {"-": ["13"]},
    "EXOSTRA": {"-": ["11"]},
    "AMPLEXO": {"-": ["51"]},
    "PERLIBET": {"-": ["52"]},
    "TRAPEZITA": {"-": ["11"]},
    "AVLOEDVS": {"-": ["12"]},
    "SAEPICVLE": {"-": ["6-"]},
    "CHRYSOPHRYS": {"-": ["17"]},
    "INCLVSIO": {"-": ["13"]},
    "INSTIMVLO": {"-": ["51"]},
    "SPONDEVS": {"-": ["21"]},
    "ACTIVNCVLA": {"-": ["11"]},
    "OPERARIA": {"-": ["11"]},
    "STENTOR": {"N": ["13"]},
    "ABSTINAX": {"-": ["22"]},
    "OBSEQVELA": {"-": ["11"]},
    "OSOR": {"-": ["13"]},
    "CARYOTA": {"-": ["11"]},
    "POLLINCTOR": {"-": ["13"]},
    "MEDVLLVLA": {"-": ["11"]},
    "DISTAEDET": {"-": ["52"]},
    "IAMIAMQVE": {"-": ["6-"]},
    "PERSAPIENS": {"-": ["22"]},
    "DEBLATERO": {"-": ["51"]},
    "CENSIO": {"-": ["13"]},
    "PYRENE": {"N": ["11"]},
    "VENTRIOSVS": {"-": ["21"]},
    "AMOENE": {"-": ["6-"]},
    "LEVATOR": {"-": ["13"]},
    "HISPALIS": {"N": ["13"]},
    "TARPEZITA": {"-": ["11"]},
    "HERBEVS": {"-": ["21"]},
    "PERSPICIENTIA": {"-": ["11"]},
    "CVBITORIVS": {"-": ["21"]},
    "PERPLEXE": {"-": ["6-"]},
    "REDAMBVLO": {"-": ["51"]},
    "NVTRICO": {"-": ["51"]},
    "PANOPE": {"N": ["11"]},
    "REPOSTOR": {"-": ["13"]},
    "CRINALIS": {"-": ["24"]},
    "MANIPVLATIM": {"-": ["6-"]},
    "ACCIPITRINA": {"-": ["11"]},
    "COLLARE": {"-": ["13"]},
    "ANAGNOSTES": {"-": ["17"]},
    "BVXEVS": {"-": ["21"]},
    "OCTOGIENS": {"-": ["6-"]},
    "ARISTO": {"N": ["13"]},
    "BARBATORIA": {"-": ["11"]},
    "GEMELLIPARVS": {"-": ["21"]},
    "FAVSTE": {"-": ["6-"]},
    "QVANTILLVS": {"-": ["21"]},
    "NVMVLARIOLVS": {"-": ["12"]},
    "PATAGIATVS": {"-": ["21"]},
    "MORTIFERE": {"-": ["6-"]},
    "HELIOPOLIS": {"N": ["13"]},
    "CANTHARIS": {"-": ["13"]},
    "ATHLETICE": {"-": ["6-"]},
    "INDVSIARIVS": {"-": ["12"]},
    "CIRRATVS": {"-": ["21"]},
    "MARO": {"N": ["13"]},
    "SVPPLOSIO": {"-": ["13"]},
    "ISTACTENVS": {"-": ["6-"]},
    "CORPVLENTVS": {"-": ["21"]},
    "PERVTILIS": {"-": ["24"]},
    "NEMESIS": {"N": ["13"]},
    "NAR": {"N": ["13"]},
    "LANISTICIVS": {"-": ["21"]},
    "INDVCTOR": {"-": ["13"]},
    "IMPLVVIATVS": {"-": ["21"]},
    "ARABS": {"N": ["13"], "A": ["21"]},
    "PENELOPE": {"N": ["11"]},
    "DECEMPEDATOR": {"-": ["13"]},
    "LVNVLA": {"-": ["11"]},
    "CIRCVMSCRIPTVS": {"-": ["21"]},
    "SEMICREMATVS": {"-": ["21"]},
    "ALCIS": {"N": ["13"]},
    "TONSTRINVS": {"-": ["21"]},
    "SOBRIE": {"-": ["6-"]},
    "NIVATVS": {"-": ["21"]},
    "CASTERIA": {"-": ["11"]},
    "ADONIS": {"N": ["13"]},
    "PHALERATVS": {"-": ["21"]},
    "RETROVERSVS": {"-": ["21"]},
    "ALLACRIMO": {"-": ["51"]},
    "COENO": {"-": ["51"]},
    "PERSVASTRIX": {"-": ["13"]},
    "VIRGATOR": {"-": ["13"]},
    "BLENNVS": {"-": ["21"]},
    "DVCENVS": {"-": ["21"]},
    "PORCINA": {"-": ["11"]},
    "GOBIVS": {"-": ["12"]},
    "BABAECALVS": {"-": ["21"]},
    "PATAGIARIVS": {"-": ["12"]},
    "CALVOR": {"-": ["5L"]},
    "PROTOPRAXIA": {"-": ["11"]},
    "APPOTVS": {"-": ["21"]},
    "INGRATIFICVS": {"-": ["21"]},
    "PERINVISVS": {"-": ["21"]},
    "INEXCITVS": {"-": ["21"]},
    "CINCTVTVS": {"-": ["21"]},
    "RAPACIDA": {"-": ["11"]},
    "MELANVRVS": {"-": ["12"]},
    "TRITICEIVS": {"-": ["21"]},
    "SCOPVLOSVS": {"-": ["21"]},
    "SIRPICVLVS": {"-": ["21"]},
    "SOPOLIS": {"N": ["13"]},
    "AMYNTOR": {"N": ["13"]},
    "PRAEGRESSVS": {"-": ["21"]},
    "LAPIDATOR": {"-": ["13"]},
    "PERSONVS": {"-": ["21"]},
    "INTERCINO": {"-": ["53"]},
    "INAVGVRO": {"-": ["51"]},
    "LATRATOR": {"-": ["13"]},
    "PEDALE": {"-": ["13"]},
    "TONSITO": {"-": ["51"]},
    "CAPSELLA": {"-": ["11"]},
    "TVBVRCINABVNDVS": {"-": ["21"]},
    "CRATHIS": {"N": ["13"]},
    "SVBRANCIDVS": {"-": ["21"]},
    "VOLVPTVOSVS": {"-": ["21"]},
    "COMPLVRIENS": {"-": ["6-"]},
    "LATRINA": {"-": ["11"]},
    "SEXIENS": {"-": ["6-"]},
    "FOETOR": {"-": ["13"]},
    "DODRANTARIVS": {"-": ["21"]},
    "HYPASIS": {"N": ["13"]},
    "ISIS": {"N": ["13"]},
    "PETOSIRIS": {"N": ["13"]},
    "APPOSCO": {"-": ["53"]},
    "XIPHIAS": {"-": ["17"]},
    "CVRATVRA": {"-": ["11"]},
    "SVBINVISVS": {"-": ["21"]},
    "LABECVLA": {"-": ["11"]},
    "SIRMIO": {"N": ["13"]},
    "NERVOSE": {"-": ["6-"]},
    "SOLIDE": {"-": ["6-"]},
    "PACIFICVS": {"-": ["21"]},
    "MISSICVLO": {"-": ["51"]},
    "MENSVLA": {"-": ["11"]},
    "ELPENOR": {"N": ["13"]},
    "PARENTICIDA": {"-": ["11"]},
    "LATVRA": {"-": ["11"]},
    "POTITO": {"-": ["51"]},
    "PELOPS": {"N": ["13"]},
    "DESTIMVLO": {"-": ["51"]},
    "NARES": {"-": ["13"]},
    "RISOR": {"-": ["13"]},
    "SITELLA": {"-": ["11"]},
    "ASTITVO": {"-": ["12"]},
    "MVSSITO": {"-": ["51"]},
    "POLITE": {"-": ["6-"]},
    "IATRALIPTA": {"-": ["11"]},
    "COLLATIVVS": {"-": ["21"]},
    "SIREN": {"N": ["13"]},
    "CLIO": {"N": ["13"]},
    "OMPHALE": {"N": ["11"]},
    "PATRICE": {"-": ["6-"]},
    "NITEDVLA": {"-": ["11"]},
    "CONNVBIALIS": {"-": ["24"]},
    "NESIS": {"N": ["13"]},
    "QVERIBVNDVS": {"-": ["21"]},
    "CONCVRO": {"-": ["51"]},
    "SEVIRATVS": {"-": ["14"]},
    "SINGVLARIVS": {"-": ["21"]},
    "ECHIDNA": {"-": ["11"]},
    "OBRODO": {"-": ["53"]},
    "AVSTERE": {"-": ["6-"]},
    "APOPHORETVS": {"-": ["21"]},
    "PERSEQVENS": {"-": ["22"]},
    "PIPERATVS": {"-": ["21"]},
    "LITTERATE": {"-": ["6-"]},
    "SVTVRA": {"-": ["11"]},
    "AHA": {"-": ["11"]},
    "CARCERARIVS": {"2": ["12"]},
    "AVLIS": {"N": ["13"]},
    "CYMATILIS": {"-": ["24"]},
    "HERO": {"N": ["13"]},
    "SVBREMIGO": {"-": ["51"]},
    "SALTVARIVS": {"-": ["12"]},
    "TERDECIENS": {"-": ["13"]},
    "COMMVRMVRO": {"-": ["51"]},
    "PERTVRPIS": {"-": ["24"]},
    "VBIVBI": {"-": ["6-"]},
    "PERSOLLA": {"-": ["11"]},
    "GRACILIPES": {"-": ["22"]},
    "SOMNICVLOSVS": {"-": ["21"]},
    "CISTOPHORVS": {"-": ["12"]},
    "FERRICREPINVS": {"-": ["21"]},
    "ADIACENS": {"-": ["13"]},
    "CANDIDE": {"-": ["6-"]},
    "QVINQVIENS": {"-": ["6-"]},
    "AGRICVLTOR": {"-": ["13"]},
    "FIDELE": {"-": ["6-"]},
    "DENVS": {"-": ["21"]},
    "SOBRINVS": {"-": ["12"]},
    "LOQVITOR": {"-": ["5J"]},
    "ECHENEIS": {"-": ["13"]},
    "LVDIFICABILIS": {"-": ["24"]},
    "COCTOR": {"-": ["13"]},
    "MESSORIVS": {"-": ["21"]},
    "LATICLAVIVS": {"2": ["21"]},
    "HERMEROS": {"N": ["12"]},
    "CYME": {"N": ["11"]},
    "AGGEMO": {"-": ["13"]},
    "INVENVSTE": {"-": ["6-"]},
    "PERREPTO": {"-": ["51"]},
    "NVMELLA": {"-": ["11"]},
    "HISTORICE": {"-": ["17"]},
    "MVGIL": {"-": ["13"]},
    "PRAENVNTIA": {"-": ["11"]},
    "THAIS": {"N": ["13"]},
    "OBVIGILO": {"-": ["51"]},
    "TERNVS": {"-": ["21"]},
    "COTISO": {"N": ["13"]},
    "NINIVE": {"N": ["11"]},
    "CAPREIDA": {"-": ["11"]},
    "PERCITVS": {"-": ["21"]},
    "BILYCHNIS": {"-": ["24"]},
    "INCOMMODO": {"1": ["51"]},
    "PRAEFIDENS": {"-": ["22"]},
    "MONSTRABILIS": {"-": ["24"]},
    "PERLITTERATVS": {"-": ["21"]},
    "INSTINCTOR": {"-": ["13"]},
    "SERVATVS": {"-": ["21"]},
    "TONSTRINA": {"-": ["11"]},
    "CALCEOLARIVS": {"-": ["12"]},
    "CHIONE": {"N": ["11"]},
    "SPONSIVNCVLA": {"-": ["11"]},
    "QVOQVOMODO": {"1": ["6-"], "2": ["21"]},
    "QVADRAGENVS": {"-": ["21"]},
    "ALTARE": {"-": ["13"]},
    "OCCVLTATOR": {"-": ["13"]},
    "SEPTEMPEDALIS": {"-": ["24"]},
    "QVADRIMVLVS": {"-": ["21"]},
    "PERBACCHOR": {"-": ["5J"]},
    "BIMARITVS": {"-": ["21"]},
    "COMISSATOR": {"-": ["13"]},
    "BIFRONS": {"-": ["22"]},
    "SPECVLABVNDVS": {"-": ["21"]},
    "MVRICIDVS": {"-": ["21"]},
    "CAPERO": {"-": ["51"]},
    "COMPACISCOR": {"-": ["5L"]},
    "PERLEPIDE": {"-": ["6-"]},
    "EROGITO": {"-": ["51"]},
    "ARCTOPHYLAX": {"N": ["13"]},
    "DELECTOR": {"-": ["5J"]},
    "SVFFIMEN": {"-": ["13"]},
    "DORMITORIVS": {"-": ["21"]},
    "PRECATOR": {"-": ["13"]},
    "XANTHIPPE": {"N": ["11"]},
    "STRICTVRA": {"-": ["11"]},
    "CALLISTO": {"N": ["13"]},
    "PERIVCVNDVS": {"-": ["21"]},
    "SCAZON": {"-": ["13"]},
    "ANACHARSIS": {"N": ["13"]},
    "VMBER": {"A": ["23"]},
    "ACACIA": {"-": ["11"]},
    "FORDA": {"-": ["11"]},
    "CONDVCTICIVS": {"-": ["21"]},
    "PORCELLVS": {"-": ["12"]},
    "INEXPLETVS": {"-": ["21"]},
    "GESTITO": {"-": ["51"]},
    "INFECTO": {"-": ["51"]},
    "DESQVAMO": {"-": ["51"]},
    "VRBANE": {"-": ["6-"]},
    "PSAMATHE": {"N": ["11"]},
    "PERSEPHONE": {"N": ["11"]},
    "INTERTEXO": {"-": ["53"]},
    "MASTRVCA": {"-": ["11"]},
    "EPILOGVS": {"-": ["12"]},
    "SALIATVS": {"-": ["14"]},
    "BARDVS": {"2": ["21"]},
    "LVCTAMEN": {"-": ["13"]},
    "MAIALIS": {"-": ["13"]},
    "CATAPHRACTA": {"-": ["11"]},
    "COMMANIPVLARIS": {"-": ["24"]},
    "AGAVE": {"N": ["11"]},
    "TRIPOLIS": {"N": ["13"]},
    "CAPIS": {"-": ["13"]},
    "HAESITATOR": {"-": ["13"]},
    "FVCOSVS": {"-": ["21"]},
    "ECCERE": {"-": ["6-"]},
    "BVLE": {"-": ["17"]},
    "GRAVATE": {"-": ["6-"]},
    "REPROMITTO": {"-": ["53"]},
    "NEPHELE": {"N": ["11"]},
    "FATVS": {"-": ["21"]},
    "SOMNICVLOSE": {"-": ["6-"]},
    "INCONCILIO": {"-": ["51"]},
    "PERMIRVS": {"-": ["21"]},
    "PAVONINVS": {"-": ["21"]},
    "SPOLIATRIX": {"-": ["13"]},
    "VIRILIA": {"-": ["13"]},
    "IAPYX": {"A": ["22"], "N": ["21"]},
    "OMINOSVS": {"-": ["21"]},
    "GALBINVS": {"-": ["21"]},
    "INEQVITABILIS": {"-": ["24"]},
    "SERISAPIA": {"-": ["11"]},
    "AVGE": {"N": ["11"]},
    "OBSIGNATOR": {"-": ["13"]},
    "EXPISCOR": {"-": ["5J"]},
    "COEMPTIONALIS": {"-": ["24"]},
    "FALSIDICVS": {"-": ["21"]},
    "REPRESSOR": {"-": ["13"]},
    "SAME": {"N": ["11"]},
    "CICARO": {"-": ["13"]},
    "LANIENA": {"-": ["11"]},
    "MVLIONIVS": {"-": ["21"]},
    "FIMBRIAE": {"-": ["11"]},
    "GRAVASTELLVS": {"-": ["21"]},
    "ANAS": {"1": ["13"]},
    "PRAESTIGIATRIX": {"-": ["13"]},
    "CONFOSSVS": {"-": ["21"]},
    "RALLVS": {"-": ["21"]},
    "ATTHIS": {"N": ["13"]},
    "EXPVLSOR": {"-": ["13"]},
    "SCALDIS": {"N": ["13"]},
    "COMINCOMMODVS": {"-": ["21"]},
    "CHARONDAS": {"N": ["13"]},
    "HALYS": {"N": ["13"]},
    "DELITIGO": {"-": ["51"]},
    "REPROMISSIO": {"-": ["13"]},
    "NICEROS": {"N": ["12"]},
    "INIMICE": {"-": ["6-"]},
    "CATVLA": {"-": ["11"]},
    "CHLORIS": {"N": ["13"]},
    "VOLVPE": {"-": ["13"]},
    "EBRIOLVS": {"-": ["21"]},
    "TRANSFORMIS": {"-": ["24"]},
    "AMPHITRITE": {"N": ["17"]},
    "SORORICIDA": {"-": ["11"]},
    "VRCEATIM": {"-": ["6-"]},
    "EPVLO": {"1": ["51"], "N": ["11"], "-": ["12"]},
    "IVNCEVS": {"-": ["21"]},
    "INTERNECTO": {"-": ["53"]},
    "DESVDASCO": {"-": ["53"]},
    "CALIGARIVS": {"-": ["21"]},
    "PEDISEQVA": {"-": ["11"]},
    "PARAETACENE": {"N": ["11"]},
    "PRIMITIAE": {"-": ["11"]},
    "OBTVEOR": {"-": ["5K"]},
    "FLEXIO": {"-": ["13"]},
    "INCOMPOSITE": {"-": ["6-"]},
    "DESPICOR": {"-": ["5J"]},
    "MIMIAMBI": {"-": ["12"]},
    "PACENSIS": {"N": ["13"]},
    "NOMENCVLATOR": {"-": ["13"]},
    "SVBLATVS": {"-": ["21"]},
    "MYCALE": {"N": ["11"]},
    "ACOR": {"-": ["13"]},
    "ILLVSTRE": {"-": ["6-"]},
    "PERNICIOSE": {"-": ["6-"]},
    "SELLVLA": {"-": ["11"]},
    "MECASTOR": {"-": ["6-"]},
    "FVRCIFERA": {"-": ["11"]},
    "SVBBASILICANVS": {"-": ["12"]},
    "SCRIBILITA": {"-": ["11"]},
    "OCTIPES": {"-": ["22"]},
    "CRANTOR": {"N": ["13"]},
    "IMPETRIO": {"-": ["54"]},
    "NISIBIS": {"N": ["13"]},
    "QVADRVPEDANS": {"1": ["13"], "2": ["12"]},
    "EXLEX": {"-": ["22"]},
    "DETRACTATOR": {"-": ["13"]},
    "PERCONTOR": {"-": ["5J"]},
    "DIAETA": {"-": ["11"]},
    "LABEO": {"N": ["13"]},
    "RECESSIM": {"-": ["6-"]},
    "FICTOR": {"-": ["13"]},
    "COLLIQVEFACIO": {"-": ["55"]},
    "AGEMA": {"-": ["13"]},
    "SANGVEN": {"-": ["13"]},
    "ROGATOR": {"-": ["13"]},
    "SEXTADECIMANI": {"-": ["12"]},
    "CVMBVLA": {"-": ["11"]},
    "PROMINVLVS": {"-": ["21"]},
    "ASPELLO": {"-": ["53"]},
    "PARASITICVS": {"-": ["21"]},
    "INSPECIOSVS": {"-": ["21"]},
    "LIMVLVS": {"-": ["21"]},
    "PHOLOE": {"N": ["11"]},
    "PHLEGETHON": {"N": ["13"]},
    "BIBLIOPOLA": {"-": ["11"]},
    "CONCINNE": {"-": ["6-"]},
    "PLVMOSVS": {"-": ["21"]},
    "NARO": {"N": ["13"]},
    "EPIDIPNIS": {"-": ["13"]},
    "ABSTO": {"-": ["51"]},
    "RICA": {"-": ["11"]},
    "MARMORO": {"-": ["51"]},
    "COMARCHVS": {"-": ["12"]},
    "DEDVCTOR": {"-": ["13"]},
    "SAXATILIS": {"-": ["24"]},
    "PRAEFODIO": {"-": ["55"]},
    "RHETORICA": {"-": ["11"]},
    "THYMELE": {"N": ["11"]},
    "OCTONVS": {"-": ["21"]},
    "FOLLITVS": {"-": ["21"]},
    "LOLIGVNCVLA": {"-": ["11"]},
    "CLVO": {"-": ["53"]},
    "ANIMADVERSOR": {"-": ["13"]},
    "HIPPOLYTE": {"N": ["11"]},
    "AVTONOE": {"N": ["11"]},
    "DIVINE": {"-": ["6-"]},
    "APOLACTIZO": {"-": ["51"]},
    "VERBEREVS": {"-": ["21"]},
    "PRAECONIVS": {"-": ["21"]},
    "NOCTVINVS": {"-": ["21"]},
    "COMMENDATOR": {"-": ["13"]},
    "INGIGNO": {"-": ["53"]},
    "PVBLICITVS": {"-": ["6-"]},
    "INSTIMVLATOR": {"-": ["13"]},
    "DVCENARIVS": {"-": ["12"]},
    "BABAE": {"-": ["11"]},
    "PHANTASIA": {"-": ["11"]},
    "EXENTERO": {"-": ["51"]},
    "HORRIDE": {"-": ["6-"]},
    "PYGISIACVS": {"-": ["21"]},
    "MESOCHORVS": {"-": ["12"]},
    "ACCLINO": {"-": ["51"]},
    "SMARIS": {"-": ["13"]},
    "BELLATVLVS": {"-": ["21"]},
    "EXEMPLARES": {"-": ["13"]},
    "RVBIDVS": {"-": ["21"]},
    "PERDECORVS": {"-": ["21"]},
    "TVBEROSVS": {"-": ["21"]},
    "VOLVPTABILIS": {"-": ["24"]},
    "INDEMNATVS": {"-": ["21"]},
    "HELVO": {"-": ["13"]},
    "CALYPSO": {"N": ["13"]},
    "CALLIOPE": {"N": ["11"]},
    "TISIPHONE": {"N": ["11"]},
    "CONSVASOR": {"-": ["13"]},
    "OCVLISSIMVS": {"-": ["21"]},
    "COLVMBVLVS": {"-": ["12"]},
    "SALVTIGERVLVS": {"-": ["21"]},
    "LESBOS": {"N": ["12"]},
    "ALICVLA": {"-": ["11"]},
    "QVADRINGENTIENS": {"-": ["6-"]},
    "PERSIMILIS": {"-": ["24"]},
    "ANCORARIVS": {"-": ["12"]},
    "PRAESTINO": {"-": ["51"]},
    "HECATE": {"N": ["11"]},
    "CYLLENE": {"N": ["11"]},
    "MENDICVLVS": {"-": ["21"]},
    "PARTITVDO": {"-": ["13"]},
    "ODIOSE": {"-": ["6-"]},
    "DECLAMITO": {"-": ["51"]},
    "CALDICEREBRIVS": {"-": ["12"]},
    "CIRCVMFORANEVS": {"-": ["21"]},
    "SCABITVDO": {"-": ["13"]},
    "ANTEGREDIOR": {"-": ["5N"]},
    "DIPONDIARIVS": {"-": ["21"]},
    "MISCELLANEVS": {"-": ["21"]},
    "AGORANOMVS": {"-": ["12"]},
    "METICVLOSVS": {"-": ["21"]},
    "TRANSMISSIO": {"-": ["13"]},
    "IGNIPOTENS": {"-": ["22"]},
    "OFFENSIVNCVLA": {"-": ["11"]},
    "HEPATIA": {"-": ["12"]},
    "GENICVLATVS": {"-": ["21"]},
    "REMVRMVRO": {"-": ["51"]},
    "TRADITOR": {"-": ["13"]},
    "ERANVS": {"-": ["12"]},
    "ILLICITATOR": {"-": ["13"]},
    "PALAESTE": {"N": ["11"]},
    "BERENICE": {"N": ["11"]},
    "DEFENSITO": {"-": ["51"]},
    "DEGRANDINAT": {"-": ["51"]},
    "PANCRATICE": {"-": ["6-"]},
    "MANTISSA": {"-": ["11"]},
    "DVCTIM": {"-": ["6-"]},
    "AVGVRIVS": {"-": ["21"]},
    "SPHINX": {"N": ["13"]},
    "DRYOPE": {"N": ["11"]},
    "SCORTOR": {"-": ["5J"]},
    "INTERTVRBO": {"2": ["51"]},
    "BOTVLVS": {"-": ["12"]},
    "TOPIARIVS": {"-": ["12"]},
    "QVIRITATVS": {"-": ["14"]},
    "LIMBOLARIVS": {"-": ["12"]},
    "POLENTARIVS": {"-": ["21"]},
    "PROCRIS": {"N": ["13"]},
    "SVSQVEDEQVE": {"-": ["6-"]},
    "ADVENTOR": {"-": ["13"]},
    "ATESTE": {"N": ["11"]},
    "QVIRIS": {"N": ["13"]},
    "VIGILAX": {"-": ["13"]},
    "APICVLA": {"-": ["11"]},
    "RVCTOR": {"-": ["5J"]},
    "SAEVITVDO": {"-": ["13"]},
    "HELICE": {"N": ["17"]},
    "QVINVS": {"-": ["21"]},
    "NEPA": {"-": ["11"]},
    "PRAEPOSTERE": {"-": ["6-"]},
    "PRAEBITOR": {"-": ["13"]},
    "TRECENTIENS": {"-": ["6-"]},
    "SVFFLATVS": {"-": ["14"]},
    "MVLTIBIBVS": {"-": ["21"]},
    "COPTOPLACENTA": {"-": ["11"]},
    "GELIDE": {"-": ["6-"]},
    "PRAETERVEHO": {"-": ["53"]},
    "HETAERIA": {"-": ["11"]},
    "TRIENS": {"-": ["22"]},
    "VTIBILIS": {"-": ["24"]},
    "DETONO": {"-": ["51"]},
    "DESTRICTE": {"-": ["6-"]},
    "PROSPICIENTIA": {"-": ["11"]},
    "PIRENE": {"N": ["11"]},
    "COMMEMORABILIS": {"-": ["24"]},
    "PAVXILLATIM": {"-": ["6-"]},
    "DIGLADIOR": {"-": ["5J"]},
    "MANIFESTARIVS": {"-": ["21"]},
    "DEFVNCTORIVS": {"-": ["21"]},
    "NEBRIS": {"-": ["13"]},
    "PELAGIVS": {"-": ["21"]},
    "INHONORVS": {"-": ["21"]},
    "CALVITIES": {"-": ["15"]},
    "MOECHISSO": {"-": ["51"]},
    "INDOMABILIS": {"-": ["24"]},
    "SCRIPTITO": {"-": ["51"]},
    "CONIECTOR": {"-": ["13"]},
    "CONDORMISCO": {"-": ["53"]},
    "LEVCAS": {"N": ["13"]},
    "ANTHRAX": {"N": ["13"]},
    "TRICLINIARCHES": {"-": ["11"]},
    "CYCLICVS": {"-": ["21"]},
    "PERBASIO": {"-": ["51"]},
    "EXDORSVO": {"-": ["51"]},
    "EFFVLTVS": {"-": ["21"]},
    "PERPLEXABILIS": {"-": ["24"]},
    "CROCOTARIVS": {"-": ["21"]},
    "IMBERBVS": {"-": ["21"]},
    "MEMPHIS": {"N": ["13"]},
    "BESSALIS": {"-": ["24"]},
    "BONATVS": {"-": ["21"]},
    "HISTRIONIA": {"-": ["11"]},
    "VERBOSE": {"-": ["6-"]},
    "OBSONATVS": {"-": ["14"]},
    "FVLCIPEDIA": {"-": ["11"]},
    "AMATRIX": {"-": ["22"]},
    "OBNOXIOSE": {"-": ["6-"]},
    "NESAPIVS": {"-": ["21"]},
    "INTERDIVS": {"-": ["6-"]},
    "RETRICES": {"-": ["13"]},
    "DENTIFRANGIBVLVS": {"-": ["21"]},
    "PANTHERINVS": {"-": ["21"]},
    "INGRESSIO": {"-": ["13"]},
    "PERISTASIS": {"-": ["13"]},
    "PERINDVLGENS": {"-": ["22"]},
    "QVADRILIBRIS": {"-": ["24"]},
    "NIDVLVS": {"-": ["12"]},
    "PERCALLESCO": {"-": ["53"]},
    "PELLENE": {"N": ["11"]},
    "CANTABVNDVS": {"-": ["21"]},
    "NAVTEA": {"-": ["11"]},
    "DAVLIS": {"N": ["13"]},
    "PRIMVLVS": {"-": ["21"]},
    "VICTITO": {"-": ["51"]},
    "INFECTOR": {"-": ["13"]},
    "PRAEVARICATOR": {"-": ["13"]},
    "GESTATOR": {"-": ["13"]},
    "ESCVLENTVS": {"-": ["21"]},
    "BARCALA": {"-": ["11"]},
    "APPROMITTO": {"-": ["53"]},
    "CALAMISTRATVS": {"-": ["21"]},
    "MELINA": {"-": ["11"]},
    "ACTE": {"N": ["17"]},
    "PILANVS": {"-": ["12"]},
    "DENASO": {"-": ["51"]},
    "COAEQVALIS": {"-": ["24"]},
    "PHILOPATOR": {"N": ["13"]},
    "INFRENO": {"-": ["51"]},
    "APELLO": {"-": ["53"]},
    "CALYMNE": {"N": ["11"]},
    "PERTVSVS": {"-": ["21"]},
    "CVRIS": {"-": ["13"]},
    "PVLVINARIS": {"-": ["24"]},
    "DIPLOMA": {"-": ["13"]},
    "IMPENDIOSVS": {"-": ["21"]},
    "DISPALESCO": {"-": ["53"]},
    "BILIX": {"-": ["22"]},
    "CRVRALIS": {"-": ["24"]},
    "FABRE": {"-": ["6-"]},
    "AFFORMIDO": {"2": ["51"]},
    "TVSSICVLA": {"-": ["11"]},
    "SCRVPVS": {"-": ["12"]},
    "PERDOCTVS": {"-": ["21"]},
    "CELOX": {"-": ["13"]},
    "CONCVSTODIO": {"-": ["54"]},
    "PAVXILLVLVM": {"2": ["12"]},
    "LEPOS": {"-": ["13"], "N": ["13"]},
    "ADMOLIOR": {"-": ["5M"]},
    "SCISSOR": {"-": ["13"]},
    "LODICVLA": {"-": ["11"]},
    "DECIMO": {"-": ["6-"]},
    "SACERDOTALIS": {"-": ["13"]},
    "VNOCVLVS": {"-": ["12"]},
    "NIXVS": {"-": ["21"]},
    "VICIENS": {"-": ["6-"]},
    "CLYMENE": {"N": ["11"]},
    "PROFATVS": {"-": ["21"]},
    "QVARTANI": {"-": ["12"]},
    "FRATRICIDA": {"-": ["11"]},
    "DESCENSIO": {"-": ["13"]},
    "PHANTASMA": {"-": ["13"]},
    "TAPANTA": {"-": ["16"]},
    "HIPPODAME": {"N": ["11"]},
    "INCORRVPTE": {"-": ["6-"]},
    "TEPIDE": {"-": ["6-"]},
    "AGNELLVS": {"-": ["12"]},
    "ARCESSITOR": {"-": ["13"]},
    "INTRIBVO": {"-": ["53"]},
    "SPVRCATVS": {"-": ["21"]},
    "CENATVS": {"-": ["21"]},
    "TOGVLA": {"-": ["11"]},
    "SIMVLATE": {"-": ["6-"]},
    "CALAIS": {"N": ["13"]},
    "ASTIPVLATOR": {"-": ["13"]},
    "HIPPOCRENE": {"N": ["11"]},
    "PRASINATVS": {"-": ["21"]},
    "RVBICO": {"N": ["13"]},
    "BALAENACEVS": {"-": ["21"]},
    "IMMVTATVS": {"-": ["21"]},
    "SCALPTOR": {"-": ["13"]},
    "TVRBVLENTE": {"-": ["6-"]},
    "SVBBLANDIOR": {"-": ["5M"]},
    "DIRCE": {"N": ["11"]},
    "INTERIBI": {"-": ["6-"]},
    "CONCESSO": {"-": ["51"]},
    "CONVADOR": {"-": ["5J"]},
    "TVNICVLA": {"-": ["11"]},
    "TVBERO": {"N": ["13"]},
    "PHOEBE": {"N": ["11"]},
    "AVRARIVS": {"2": ["21"]},
    "SVBDOMO": {"-": ["51"]},
    "PRAEQVAM": {"-": ["6-"]},
    "HEPATARIVS": {"-": ["21"]},
    "CINCTICVLVS": {"-": ["12"]},
    "BOVILE": {"-": ["13"]},
    "CVIATIS": {"-": ["2&"]},
    "ANTIPOLIS": {"N": ["13"]},
    "RENVTVS": {"-": ["14"]},
    "HAESITANTIA": {"-": ["11"]},
    "CVSTODITE": {"-": ["6-"]},
    "INTESTATO": {"-": ["6-"]},
    "MALEDICAX": {"-": ["22"]},
    "PERFACETVS": {"-": ["21"]},
    "RESECRO": {"-": ["51"]},
    "OFFRENATVS": {"-": ["21"]},
    "SOCIENNVS": {"-": ["12"]},
    "DERVNCINO": {"-": ["51"]},
    "STROPHIARIVS": {"-": ["12"]},
    "RESTIPVLOR": {"-": ["5J"]},
    "MILLESIMA": {"-": ["6-"]},
    "BERBEX": {"-": ["13"]},
    "ASSVO": {"-": ["53"]},
    "GILLO": {"N": ["13"]},
    "SCVTARIVS": {"-": ["21"]},
    "REPVERASCO": {"-": ["53"]},
    "CAVILLA": {"-": ["11"]},
    "RVSCINO": {"N": ["13"]},
    "BENEMORIVS": {"-": ["21"]},
    "CONCILIATRICVLA": {"-": ["11"]},
    "LIBRARIOLVS": {"-": ["12"]},
    "INSATVRABILIS": {"-": ["24"]},
    "INCIPISSO": {"-": ["53"]},
    "PACIFICATORIVS": {"-": ["21"]},
    "PRAETRVNCO": {"-": ["51"]},
    "DEPVLSIO": {"-": ["13"]},
    "GVSTATVS": {"-": ["21"]},
    "MELICA": {"-": ["11"]},
    "VEPRECVLA": {"-": ["11"]},
    "PERSPICAX": {"-": ["22"]},
    "MYROPOLA": {"-": ["11"]},
    "PERCVPIO": {"-": ["55"]},
    "FIGVLARIS": {"-": ["24"]},
    "CARINARIVS": {"-": ["12"]},
    "ANNOTATOR": {"-": ["13"]},
    "INCERTE": {"-": ["6-"]},
    "LIQVO": {"-": ["51"]},
    "PROTINAM": {"-": ["6-"]},
    "COCHLEARE": {"-": ["13"]},
    "PERCRVCIO": {"-": ["51"]},
    "IMPERSPICVVS": {"-": ["21"]},
    "PERPETVARIVS": {"-": ["12"]},
    "QVISQVILIAE": {"-": ["11"]},
    "TECHNA": {"-": ["11"]},
    "AMASTRIS": {"N": ["13"]},
    "OBSERVANS": {"-": ["22"]},
    "ACERBO": {"-": ["51"]},
    "PROOEMIOR": {"-": ["5J"]},
    "PHILIPPOPOLIS": {"N": ["13"]},
    "INCONSIDERATE": {"-": ["6-"]},
    "VERENDA": {"-": ["11"]},
    "DIOECETES": {"-": ["17"]},
    "QVADRAGIENS": {"-": ["6-"]},
    "DEFLOCCO": {"-": ["51"]},
    "CYMOTHOE": {"N": ["11"]},
    "POSTIDEA": {"-": ["6-"]},
    "CORYCVS": {"-": ["12"]},
    "NVGIGERVLVS": {"-": ["12"]},
    "MILVA": {"-": ["11"]},
    "IAMBEVS": {"-": ["21"]},
    "MALVGINENSIS": {"N": ["13"]},
    "DEBELLATOR": {"-": ["13"]},
    "DISSORS": {"-": ["22"]},
    "PLVSSCIVS": {"-": ["21"]},
    "FVLLONIA": {"-": ["11"]},
    "MILO": {"N": ["13"]},
    "PELLICATVS": {"-": ["14"]},
    "FLEMINA": {"-": ["13"]},
    "PRAECANTO": {"-": ["51"]},
    "PRAETERMISSIO": {"-": ["13"]},
    "SVBSICIVVS": {"-": ["21"]},
    "REVOLVBILIS": {"-": ["24"]},
    "LAMENTARIVS": {"-": ["21"]},
    "TROEZEN": {"N": ["13"]},
    "ACIDE": {"-": ["6-"]},
    "PVRPVRISSATVS": {"-": ["21"]},
    "OTHO": {"N": ["13"]},
    "DAPSILIS": {"-": ["24"]},
    "PERIODVS": {"-": ["12"]},
    "ADAEQVE": {"-": ["6-"]},
    "TIBVR": {"N": ["13"]},
    "TRIMETER": {"1": ["12"]},
    "INSTIGATOR": {"-": ["13"]},
    "VXORCVLA": {"-": ["11"]},
    "NVMERABILIS": {"-": ["24"]},
    "QVIPPINI": {"-": ["6-"]},
    "OBSEQVIOSVS": {"-": ["21"]},
    "MALACISSO": {"-": ["51"]},
    "VENVSTVLVS": {"-": ["21"]},
    "OBTEXO": {"-": ["53"]},
    "SVBLESTVS": {"-": ["21"]},
    "ALIQVANTILLVS": {"-": ["21"]},
    "IATRALIPTES": {"-": ["17"]},
    "PSEVDOMENOS": {"-": ["17"]},
    "EFFRENATE": {"-": ["6-"]},
    "TEMPORI": {"-": ["6-"]},
    "CIRCVMMINGO": {"-": ["6-"]},
    "CVIAS": {"-": ["13"]},
    "CLIBANVS": {"-": ["12"]},
    "VITVLINA": {"-": ["11"]},
    "VINVS": {"-": ["12"]},
    "BOSTAR": {"N": ["13"]},
    "LVRCHINABVNDVS": {"-": ["21"]},
    "PATRIMVS": {"-": ["21"]},
    "TRANSNATO": {"-": ["51"]},
    "VASA": {"-": ["12"]},
    "TERVNCIVS": {"-": ["12"]},
    "SARCINATOR": {"-": ["13"]},
    "CROCOTA": {"-": ["11"]},
    "MACRITVDO": {"-": ["13"]},
    "OBSTINO": {"-": ["51"]},
    "CVNCTO": {"-": ["51"]},
    "RVMVSCVLVS": {"-": ["12"]},
    "PROMVLSIS": {"-": ["13"]},
    "ALEC": {"-": ["13"]},
    "ABNVTO": {"-": ["51"]},
    "ACIS": {"N": ["13"]},
    "NOVIENS": {"-": ["6-"]},
    "INEPTIA": {"-": ["11"]},
    "ILLOCABILIS": {"-": ["24"]},
    "SPVTO": {"-": ["51"]},
    "SESQVIHORA": {"-": ["11"]},
    "SEPIOLA": {"-": ["11"]},
    "QVAQVA": {"-": ["6-"]},
    "PIAMEN": {"-": ["13"]},
    "EXOBSECRO": {"-": ["51"]},
    "VPVPA": {"-": ["11"]},
    "DOMVSIO": {"-": ["13"]},
    "DEPECVLATVS": {"-": ["21"]},
    "PERPOLIO": {"-": ["54"], "2": ["21"]},
    "INTERSTRINGO": {"-": ["53"]},
    "EVAN": {"N": ["16"]},
    "ZAMIA": {"-": ["11"]},
    "AVDIENTIA": {"-": ["11"]},
    "ANNEXVS": {"-": ["14"]},
    "STERILICVLA": {"-": ["11"]},
    "MORMYR": {"-": ["13"]},
    "FERICVLVS": {"-": ["21"]},
    "LARIFVGA": {"-": ["11"]},
    "PASTORIVS": {"-": ["21"]},
    "ANATHYMIASIS": {"-": ["13"]},
    "PRAELOQVOR": {"-": ["5L"]},
    "ELVVIO": {"-": ["13"]},
    "ANTIDHAC": {"-": ["6-"]},
    "IMMEDICABILIS": {"-": ["24"]},
    "FENEVS": {"-": ["21"]},
    "TALARIVS": {"-": ["21"]},
    "CATILLO": {"2": ["13"]},
    "DANISTA": {"-": ["11"]},
    "SPARVLVS": {"-": ["12"]},
    "EFFLICTIM": {"-": ["6-"]},
    "IEIVNIOSVS": {"-": ["21"]},
    "INCVRIOSE": {"-": ["6-"]},
    "ABORTIO": {"1": ["54"]},
    "BALLIO": {"N": ["13"]},
    "HOMOLE": {"N": ["11"]},
    "COMMENDATRIX": {"-": ["13"]},
    "TETHYS": {"N": ["13"]},
    "SAGARIS": {"N": ["13"]},
    "VINNVLVS": {"-": ["21"]},
    "IMPLICISCO": {"-": ["53"]},
    "PRAEFRACTE": {"-": ["6-"]},
    "LORVS": {"-": ["12"]},
    "PARRICIDALIS": {"-": ["24"]},
    "DILATOR": {"-": ["51"]},
    "NVPERVS": {"-": ["21"]},
    "METATOR": {"-": ["13"]},
    "INDISPOSITVS": {"-": ["21"]},
    "SYNGRAPHVS": {"-": ["12"]},
    "FRENI": {"-": ["12"]},
    "RHODOPE": {"N": ["11"]},
    "GIGERIA": {"-": ["12"]},
    "LEMNISCATVS": {"-": ["21"]},
    "ADOLESCENTVLA": {"-": ["11"]},
    "EXCETRA": {"-": ["11"]},
    "CONSERVA": {"-": ["11"]},
    "IMPERDITVS": {"-": ["21"]},
    "BION": {"N": ["13"]},
    "MELLILLA": {"-": ["11"]},
    "INVVLNERATVS": {"-": ["21"]},
    "THESAVRARIVS": {"2": ["12"]},
    "MASCARPIO": {"-": ["13"]},
    "OCVLEVS": {"-": ["21"]},
    "COPS": {"-": ["22"]},
    "HISPO": {"N": ["13"]},
    "MALEFACTOR": {"-": ["13"]},
    "PRAEDICATOR": {"-": ["13"]},
    "COLLARIS": {"-": ["24"]},
    "PRAXIS": {"-": ["13"]},
    "RETRACTVS": {"-": ["14"]},
    "TRIFVR": {"-": ["13"]},
    "LAODICE": {"N": ["11"]},
    "DIIVNCTE": {"-": ["6-"]},
    "CATAPHAGAS": {"-": ["17"]},
    "DIERECTVS": {"-": ["21"]},
    "SVTELA": {"-": ["11"]},
    "INSVLARIVS": {"-": ["12"]},
    "CONTEMPLO": {"-": ["51"]},
    "FRONTO": {"N": ["13"]},
    "ANTESCHOLANVS": {"-": ["12"]},
    "STERNVTO": {"-": ["51"]},
    "PLVMATILIS": {"-": ["24"]},
    "SECVTVLEIVS": {"-": ["21"]},
    "CORBITA": {"-": ["11"]},
    "CLAVIGER": {"2": ["23"]},
    "OCCALESCO": {"-": ["53"]},
    "VOCATVS": {"-": ["14"]},
    "DISPVTATOR": {"-": ["13"]},
    "RECLAMITO": {"-": ["51"]},
    "GAVSAPA": {"-": ["12"]},
    "PROSERPO": {"-": ["53"]},
    "CISTELLVLA": {"-": ["11"]},
    "APPENDICVLA": {"-": ["11"]},
    "POPPYSMA": {"-": ["13"]},
    "NICANOR": {"N": ["13"]},
    "CIVITATVLA": {"-": ["11"]},
    "MINITO": {"-": ["51"]},
    "ALCITHOE": {"N": ["17"]},
    "INFRENIS": {"-": ["24"]},
    "SACRICOLA": {"-": ["11"]},
    "DEPOPVLATOR": {"-": ["13"]},
    "ALCIMEDE": {"N": ["17"]},
    "COLEVS": {"-": ["12"]},
    "COLLATOR": {"-": ["13"]},
    "LOCELLVS": {"-": ["12"]},
    "ELEPHANTINE": {"N": ["11"]},
    "SEVIR": {"-": ["12"]},
    "LOQVENTIA": {"-": ["11"]},
    "RVBRICATVS": {"-": ["21"]},
    "MINOS": {"N": ["12"]},
    "DEMARCHVS": {"-": ["12"]},
    "ARENARIA": {"-": ["11"]},
    "PASSERCVLVS": {"-": ["12"]},
    "PHALANGAE": {"-": ["11"]},
    "CONCVSSIO": {"-": ["13"]},
    "CONTONAT": {"-": ["51"]},
    "EXPOSITICIVS": {"-": ["21"]},
    "LACTICVLOSVS": {"-": ["12"]},
    "LAMPETIE": {"N": ["11"]},
    "SYROPHOENIX": {"N": ["13"]},
    "TENELLVS": {"-": ["21"]},
    "PRAESEPIS": {"-": ["13"]},
    "PRAEVOLO": {"2": ["51"]},
    "STELA": {"-": ["11"]},
    "CHRYSIS": {"N": ["13"]},
    "DENVMERO": {"1": ["51"]},
    "PROVOCATOR": {"-": ["13"]},
    "ADVERSARIA": {"-": ["12"]},
    "MVROBATHARIVS": {"-": ["12"]},
    "PSEPHISMA": {"-": ["13"]},
    "AREOLA": {"-": ["11"]},
    "HIRA": {"-": ["11"]},
    "TERTIANA": {"-": ["11"]},
    "INTERBIBO": {"2": ["53"]},
    "SVBLATE": {"-": ["6-"]},
    "PRAESTES": {"A": ["13"]},
    "QVINTVSDECIMVS": {"-": ["21"]},
    "TVATIM": {"-": ["6-"]},
    "PERLONGINQVVS": {"-": ["21"]},
    "CONTRITVS": {"-": ["21"]},
    "PALMARIS": {"2": ["24"]},
    "OBSONOR": {"-": ["5J"]},
    "EXPVTRESCO": {"-": ["53"]},
    "INGRAVO": {"-": ["51"]},
    "CALAMITOSE": {"-": ["6-"]},
    "MELANIPPE": {"N": ["11"]},
    "RVGO": {"-": ["51"]},
    "COALO": {"-": ["53"]},
    "REDORMIO": {"-": ["54"]},
    "AMPVLLA": {"-": ["11"]},
    "AGINO": {"-": ["51"]},
    "MAVORS": {"N": ["13"]},
    "CONDICTOR": {"-": ["13"]},
    "OLLA": {"-": ["11"]},
    "SINGVLTO": {"-": ["51"]},
    "FABALIS": {"-": ["24"]},
    "AFFLICTOR": {"-": ["13"]},
    "MEROE": {"N": ["11"]},
    "IMBITO": {"-": ["53"]},
    "BATVO": {"-": ["53"]},
    "LATERICVLVS": {"-": ["12"]},
    "ASCRIPTOR": {"-": ["13"]},
    "SACCELLVS": {"-": ["12"]},
    "ANAPAESTVS": {"-": ["12"]},
    "SYENE": {"N": ["11"]},
    "CANACE": {"N": ["11"]},
    "SVBREPTICIVS": {"-": ["21"]},
    "TRIBVARIVS": {"-": ["21"]},
    "CERNVVS": {"2": ["21"]},
    "OFFLA": {"-": ["11"]},
    "GERVLIFIGVLVS": {"-": ["12"]},
    "PERACESCO": {"-": ["53"]},
    "CAVPONVLA": {"-": ["11"]},
    "INTERTRAHO": {"-": ["53"]},
    "INVESTIGATOR": {"-": ["13"]},
    "FORENSIA": {"-": ["13"]},
    "AVGVRO": {"-": ["51"]},
    "EXPALLIATVS": {"-": ["21"]},
    "BVCCO": {"-": ["13"]},
    "PVMICEVS": {"-": ["21"]},
    "GINGILIPHVS": {"-": ["12"]},
    "RECVLA": {"-": ["11"]},
    "IVLIS": {"-": ["13"]},
    "INFLETVS": {"-": ["21"]},
    "AVDACVLVS": {"-": ["21"]},
    "PVRGAMEN": {"-": ["13"]},
    "CATAPVLTARIVS": {"-": ["21"]},
    "DETVMESCO": {"-": ["53"]},
    "MVNITO": {"-": ["51"]},
    "ELINGVO": {"-": ["51"]},
    "LVCVLENTE": {"-": ["6-"]},
    "MOLOCHINARIVS": {"-": ["12"]},
    "PHYLLIS": {"N": ["13"]},
    "HONORABILIS": {"-": ["24"]},
    "SPECTANS": {"-": ["22"]},
    "PECVLATOR": {"-": ["13"]},
    "TONSVS": {"-": ["21"]},
    "MALACVS": {"-": ["21"]},
    "NOCVVS": {"-": ["21"]},
    "COPVLATVS": {"-": ["14"]},
    "VRSINA": {"-": ["11"]},
    "IVRATOR": {"-": ["13"]},
    "SYCOPHANTA": {"-": ["11"]},
    "PETASVS": {"-": ["12"]},
    "DEFRAVDO": {"-": ["51"]},
    "PVGIVNCVLVS": {"-": ["12"]},
    "PERSEPOLIS": {"N": ["13"]},
    "AGANIPPE": {"N": ["11"]},
    "ZEVXIS": {"N": ["13"]},
    "PARONYCHIA": {"-": ["11"]},
    "COCCINEVS": {"-": ["21"]},
    "LACHESIS": {"N": ["13"]},
    "PSYCHROLVTA": {"-": ["11"]},
    "EXCATARISSO": {"-": ["51"]},
    "LVPATRIA": {"-": ["11"]},
    "INARIME": {"N": ["11"]},
    "LONGIVSCVLVS": {"-": ["21"]},
    "VACIVVS": {"-": ["21"]},
    "PLEIONE": {"N": ["11"]},
    "VITISATOR": {"-": ["13"]},
    "EDITICIVS": {"-": ["21"]},
    "OBTICESCO": {"-": ["53"]},
    "MELIE": {"N": ["11"]},
    "FVRFVR": {"-": ["13"]},
    "PINNVLA": {"-": ["11"]},
    "COMATORIVS": {"-": ["21"]},
    "COMPOTOR": {"-": ["13"]},
    "DELIQVIO": {"-": ["13"]},
    "ARCVLARIVS": {"-": ["12"]},
    "ILLECEBROSVS": {"-": ["21"]},
    "SCAPHA": {"-": ["11"]},
    "INCOMMODESTICVS": {"-": ["21"]},
    "PVGNEVS": {"-": ["21"]},
    "FORTVNATE": {"-": ["6-"]},
    "SARGVS": {"-": ["12"]},
    "LEMMA": {"-": ["13"]},
    "FVGIENS": {"2": ["22"]},
    "COLONICVS": {"-": ["21"]},
    "ARIS": {"N": ["13"]},
    "PERNECESSE": {"-": ["6-"]},
    "COGNOMINIS": {"-": ["24"]},
    "CRYPTOPORTICVS": {"-": ["14"]},
    "EXTEXO": {"-": ["53"]},
    "PERBENE": {"-": ["6-"]},
    "PROPEXVS": {"-": ["21"]},
    "PROSECTA": {"-": ["11"]},
    "FASTOSVS": {"-": ["21"]},
    "IMPERCO": {"-": ["53"]},
    "VINEVS": {"-": ["21"]},
    "FVRAX": {"-": ["22"]},
    "FREMOR": {"-": ["13"]},
    "DEFERVESCO": {"-": ["53"]},
    "FRATERNE": {"-": ["6-"]},
    "OCCLAMITO": {"-": ["51"]},
    "HECATOMPYLOS": {"N": ["17"]},
    "VERSIPELLIS": {"-": ["13"]},
    "DESVEFACIO": {"-": ["55"]},
    "EBOREVS": {"-": ["21"]},
    "OCCENTO": {"-": ["51"]},
    "SORICINVS": {"-": ["21"]},
    "EVILESCO": {"-": ["53"]},
    "ABSVMEDO": {"-": ["13"]},
    "ELECEBRA": {"-": ["11"]},
    "NVNDINALIS": {"-": ["24"]},
    "ADMVTILO": {"-": ["51"]},
    "ANTICATO": {"N": ["13"]},
    "ESVRIALIS": {"-": ["24"]},
    "AMASIVS": {"-": ["12"]},
    "THEMIS": {"N": ["13"]},
    "VENDITATOR": {"-": ["13"]},
    "TEREBINTHINVS": {"-": ["21"]},
    "GYPSATVS": {"-": ["21"]},
    "ARTVTVS": {"-": ["21"]},
    "DVOETVICESIMANI": {"-": ["12"]},
    "DELETRIX": {"-": ["22"]},
    "IMITAMEN": {"-": ["13"]},
    "NVBECVLA": {"-": ["11"]},
    "MATTVS": {"-": ["21"]},
    "MOERIS": {"N": ["13"]},
    "DECEMVIRATVS": {"-": ["14"]},
    "PACISCO": {"-": ["53"]},
    "QVINQVAGIENS": {"-": ["6-"]},
    "CATOMIDIO": {"-": ["51"]},
    "PAVPERVS": {"-": ["21"]},
    "TERVENEFICVS": {"-": ["12"]},
    "AMOENA": {"-": ["12"]},
    "SVPERSTITIOSVS": {"-": ["21"]},
    "TVBVLATVS": {"-": ["21"]},
    "SVFFLAMINO": {"-": ["51"]},
    "INGRATIA": {"-": ["11"]},
    "VALENTVLVS": {"-": ["21"]},
    "VTVT": {"-": ["6-"]},
    "BOGVD": {"N": ["13"]},
    "SVAVITVDO": {"-": ["13"]},
    "TAVRVLVS": {"-": ["12"]},
    "ORARIVS": {"-": ["21"]},
    "DEPREHENSIO": {"-": ["13"]},
    "MASTIGIA": {"-": ["11"]},
    "PISTORIVS": {"-": ["21"]},
    "TVGIO": {"N": ["13"]},
    "MACILENTVS": {"-": ["21"]},
    "SCORDALIA": {"-": ["11"]},
    "MONEDVLA": {"-": ["11"]},
    "PVLCHRALIA": {"-": ["13"]},
    "PERVAGATVS": {"-": ["21"]},
    "AGAGA": {"-": ["11"]},
    "AMATORIE": {"-": ["6-"]},
    "INDECOR": {"-": ["22"]},
    "OPTATO": {"-": ["6-"]},
    "DEBATVO": {"-": ["53"]},
    "CERCYRVS": {"-": ["12"]},
    "TYRO": {"N": ["13"]},
    "DILVTE": {"-": ["6-"]},
    "OVILLA": {"-": ["11"]},
    "OPIPARE": {"-": ["6-"]},
    "EDENTVLVS": {"-": ["21"]},
    "SVFFRINGO": {"-": ["53"]},
    "SEMIDOCTVS": {"-": ["21"]},
    "VSVRARIVS": {"-": ["21"]},
    "CRYSTALLVS": {"-": ["12"]},
    "IVVENOR": {"-": ["5J"]},
    "REFORMATOR": {"-": ["13"]},
    "ROBOREVS": {"-": ["21"]},
    "RESTITVTOR": {"-": ["13"]},
    "ALOGVS": {"-": ["21"]},
    "CLIMACTERICVS": {"-": ["21"]},
    "ADLVO": {"-": ["53"]},
    "TROPICA": {"-": ["12"]},
    "SVBMISSIO": {"-": ["13"]},
    "CYMBALISTRIA": {"-": ["11"]},
    "SEVERITVDO": {"-": ["13"]},
    "LYRICA": {"-": ["12"]},
    "DISCOLORIVS": {"-": ["21"]},
    "ADMISSARIVS": {"-": ["21"]},
    "SOMNIOR": {"-": ["5J"]},
    "CONTINVE": {"-": ["6-"]},
    "GERRAE": {"-": ["11"]},
    "PVNITOR": {"-": ["13"]},
    "MEROPE": {"N": ["11"]},
    "INCRETO": {"-": ["51"]},
    "EPASTVS": {"-": ["21"]},
    "LITICEN": {"-": ["13"]},
    "MORIO": {"-": ["13"]},
    "INCALLIDE": {"-": ["6-"]},
    "STEROPE": {"N": ["11"]},
    "BAETIS": {"N": ["13"]},
    "DIRIBITOR": {"-": ["13"]},
    "CENATIVNCVLA": {"-": ["11"]},
    "FESTIVE": {"-": ["6-"]},
    "FEMEN": {"-": ["13"]},
    "MIMVLA": {"-": ["11"]},
    "EMISSICIVS": {"-": ["21"]},
    "SVBSCRIPTOR": {"-": ["13"]},
    "CAESICIVS": {"-": ["21"]},
    "REBITO": {"-": ["53"]},
    "EFFERTVS": {"-": ["21"]},
    "OBSERVANTIA": {"-": ["11"]},
    "TRANSENNA": {"-": ["11"]},
    "MATTEA": {"-": ["11"]},
    "SEPTEMVIRI": {"-": ["12"]},
    "REDEMPTO": {"-": ["51"]},
    "ELECTA": {"-": ["12"]},
    "SEXAGIENS": {"-": ["6-"]},
    "POSITOR": {"-": ["13"]},
    "MAGMENTARIVS": {"-": ["21"]},
    "CECROPS": {"N": ["13"]},
    "OBVALLO": {"-": ["51"]},
    "RHOMBVS": {"-": ["12"]},
    "PIETATICVLTRIX": {"-": ["13"]},
    "SECTARIVS": {"-": ["21"]},
    "SEPTEMVIRATVS": {"-": ["14"]},
    "CYRENE": {"N": ["11"]},
    "PERDOLESCO": {"-": ["53"]},
    "ITYS": {"N": ["13"]},
    "INIVRIVS": {"-": ["21"]},
    "ANATICVLA": {"-": ["11"]},
    "OLO": {"-": ["53"]},
    "ARMENTALIS": {"-": ["24"]},
    "CELERE": {"-": ["6-"]},
    "CVCVMVLA": {"-": ["11"]},
    "SVBALBVS": {"-": ["21"]},
    "ENYO": {"N": ["13"]},
    "ALCYONE": {"N": ["17"]},
    "DVCTITO": {"-": ["51"]},
    "TVRPIFICATVS": {"-": ["21"]},
    "DISCORDABILIS": {"-": ["24"]},
    "BYBLOS": {"N": ["17"]},
    "PHASIS": {"N": ["13"]},
    "EXAMBVLO": {"-": ["51"]},
    "DRVIDA": {"-": ["11"]},
    "DEIERO": {"-": ["51"]},
    "CYDIPPE": {"N": ["11"]},
    "CONSECTATRIX": {"-": ["13"]},
    "CLOACALIS": {"-": ["24"]},
    "CONCACO": {"-": ["51"]},
    "VAVATO": {"-": ["13"]},
    "SVBTERDVCO": {"-": ["53"]},
    "FRVSTILLATIM": {"-": ["6-"]},
    "ADAMBVLO": {"-": ["51"]},
    "EDISSERTO": {"-": ["51"]},
    "RHOEA": {"-": ["11"]},
    "CHANE": {"-": ["17"]},
    "DELINGO": {"-": ["53"]},
    "CONGREDIO": {"-": ["54"]},
    "PLVMEVS": {"-": ["21"]},
    "OBVSTVS": {"-": ["21"]},
    "ASSVLATIM": {"-": ["6-"]},
    "COMMISEROR": {"-": ["5J"]},
    "ENOTO": {"-": ["51"]},
    "FERRVMEN": {"-": ["13"]},
    "CORBVLO": {"N": ["13"]},
    "HARIOLOR": {"-": ["5J"]},
    "COMPRANSOR": {"-": ["13"]},
    "OCVLATVS": {"-": ["21"]},
    "OBEDIENTIA": {"-": ["11"]},
    "SALAMANDRA": {"-": ["11"]},
    "VERVINA": {"-": ["11"]},
    "CONTOLLO": {"-": ["53"]},
    "CAPPARIS": {"-": ["13"]},
    "DVVMVIRATVS": {"-": ["14"]},
    "DESIPIENS": {"-": ["22"]},
    "REDVCTOR": {"-": ["13"]},
    "COLLACRIMO": {"-": ["51"]},
    "IMPVRATVS": {"-": ["21"]},
    "MVFRIVS": {"-": ["12"]},
    "OPIME": {"-": ["6-"]},
    "PVTILLVS": {"-": ["21"]},
    "AEGLE": {"N": ["17"]},
    "CIRCITOR": {"-": ["13"]},
    "LVSCINIOLA": {"-": ["11"]},
    "SITVLA": {"-": ["11"]},
    "MODICO": {"-": ["6-"]},
    "BALITANS": {"-": ["22"]},
    "INCLITVS": {"-": ["21"]},
    "AXICIA": {"-": ["11"]},
    "ERVILIA": {"-": ["11"]},
    "INVOLVCRE": {"-": ["13"]},
    "CACCABVS": {"-": ["12"]},
    "NOCTIFER": {"A": ["12"]},
    "SVSPICIOSE": {"-": ["6-"]},
    "CIRCVMLVO": {"-": ["53"]},
    "TVRBELLAE": {"-": ["11"]},
    "AMYMONE": {"N": ["11"]},
    "STIGMATIAS": {"-": ["11"]},
    "DRAPETA": {"-": ["11"]},
    "NARRATIVNCVLA": {"-": ["11"]},
    "SALTATORIVS": {"-": ["21"]},
    "SVBOLFACIO": {"-": ["55"]},
    "CARIS": {"-": ["13"]},
    "PERTRECTO": {"-": ["51"]},
    "APICATVS": {"-": ["21"]},
    "DISPVDET": {"-": ["52"]},
    "LAOCOON": {"N": ["13"]},
    "ALOGIA": {"-": ["11"]},
    "EVRYDICE": {"N": ["11"]},
    "SVBSVLTO": {"-": ["51"]},
    "LARVO": {"-": ["51"]},
    "LYMPHOR": {"-": ["13"]},
    "INCONGRVENS": {"-": ["22"]},
    "ZOTHECVLA": {"-": ["11"]},
    "CONCERPO": {"-": ["53"]},
    "DEVOTO": {"-": ["51"]},
    "INERVDITVS": {"-": ["21"]},
    "PERTRIBVO": {"-": ["53"]},
    "BARCE": {"N": ["11"]},
    "BAVCIS": {"N": ["13"]},
    "PROLVSIO": {"-": ["13"]},
    "APODIXIS": {"-": ["13"]},
    "SVBSIGNO": {"-": ["51"]},
    "COMPETITRIX": {"-": ["13"]},
    "AVSPICATO": {"-": ["6-"]},
    "SEMIVSTVLO": {"-": ["51"]},
    "SVBFODIO": {"-": ["55"]},
    "CHROMIS": {"-": ["13"], "N": ["13"]},
    "MACEDO": {"N": ["13"], "A": ["25"]},
    "PRAESEGMEN": {"-": ["13"]},
    "DIONE": {"N": ["11"]},
    "OCIVS": {"-": ["6-"]},
    "POTIVNCVLA": {"-": ["11"]},
    "CVCVMA": {"-": ["11"]},
    "INSPOLIATVS": {"-": ["21"]},
    "PERMAESTVS": {"-": ["21"]},
    "PVELLARIVS": {"-": ["12"]},
    "SEXCENTI": {"-": ["21"]},
    "IMPERITE": {"-": ["6-"]},
    "INFACVNDVS": {"-": ["21"]},
    "PRAECVRSORIVS": {"-": ["21"]},
    "AGIS": {"N": ["13"]},
    "PRAEGVSTATOR": {"-": ["13"]},
    "EBVRNVS": {"-": ["21"]},
    "REDAVSPICO": {"-": ["51"]},
    "ERYTHINVS": {"-": ["12"]},
    "LETHE": {"N": ["11"]},
    "MAESTITVDO": {"-": ["13"]},
    "INFORTVNATVS": {"-": ["21"]},
    "SORTILEGVS": {"2": ["12"]},
    "ANSERINVS": {"-": ["21"]},
    "COMPES": {"-": ["22"]},
    "CALAMIS": {"N": ["13"]},
    "SVSPICO": {"-": ["51"]},
    "INSTIGATRIX": {"-": ["13"]},
    "IEIVNE": {"-": ["6-"]},
    "CONGIALIS": {"-": ["24"]},
    "CONDECET": {"-": ["52"]},
    "CERINVS": {"-": ["21"]},
    "LOTIS": {"N": ["13"]},
    "CATAPLVS": {"-": ["12"]},
    "EXCVRATVS": {"-": ["21"]},
    "SOPHOS": {"-": ["17"]},
    "PERGRAVIS": {"-": ["24"]},
    "SYMBOLA": {"-": ["11"]},
    "VISCATVS": {"-": ["21"]},
    "SAGATVS": {"-": ["21"]},
    "FRVSTRO": {"-": ["51"]},
    "CORDATVS": {"-": ["21"]},
    "COGITATE": {"-": ["6-"]},
    "SITICEN": {"-": ["13"]},
    "PROLAPSIO": {"-": ["13"]},
    "DECLINIS": {"-": ["24"]},
    "SERRVLA": {"-": ["11"]},
    "PESSVLVS": {"-": ["12"]},
    "EXASCIATVS": {"-": ["21"]},
    "OPPANGO": {"-": ["53"]},
    "INTERMINOR": {"-": ["5J"]},
    "BELLVLVS": {"-": ["21"]},
    "MORDEX": {"-": ["22"]},
    "TRAGVS": {"-": ["12"]},
    "SARDIS": {"N": ["13"]},
    "LIRIS": {"N": ["13"]},
    "AVTHEPSA": {"-": ["11"]},
    "CAPREAGINVS": {"-": ["21"]},
    "SARTOR": {"2": ["13"]},
    "SVBSIGNANVS": {"-": ["21"]},
    "NAPE": {"N": ["11"]},
    "PERCVRIOSVS": {"-": ["21"]},
    "PERACERBVS": {"-": ["21"]},
    "EOS": {"N": ["12"]},
    "SIPHVNCVLVS": {"-": ["12"]},
    "CALTHVLA": {"-": ["11"]},
    "TRAGICOMOEDIA": {"-": ["11"]},
    "SEPTEMVIRALIS": {"2": ["24"], "1": ["12"]},
    "ADRADO": {"-": ["53"]},
    "THETIS": {"N": ["13"]},
    "VERBERITO": {"-": ["51"]},
    "PECOROSVS": {"-": ["21"]},
    "FVNDITO": {"-": ["51"]},
    "PERLEVIS": {"-": ["24"]},
    "SINAPE": {"-": ["13"]},
    "FRVSTVLENTVS": {"-": ["21"]},
    "ECDICVS": {"-": ["12"]},
    "SOLITO": {"-": ["51"]},
    "PERCOLOPO": {"-": ["51"]},
    "ASSIDVO": {"2": ["51"]},
    "PALVDOSVS": {"-": ["21"]},
    "ANTHIAS": {"-": ["17"]},
    "RVRO": {"-": ["51"]},
    "COMMETO": {"1": ["51"]},
    "SEMIRAMIS": {"N": ["13"]},
    "SESTERTIARIVS": {"-": ["21"]},
    "LACINIA": {"-": ["11"]},
    "ADVESPERASCIT": {"-": ["53"]},
    "HEDEROSVS": {"-": ["21"]},
    "IMPRODICTVS": {"-": ["21"]},
    "ZAPLVTVS": {"-": ["21"]},
    "COMMENTARIOLVS": {"-": ["12"]},
    "OFFVCIA": {"-": ["11"]},
    "PERATVS": {"-": ["12"]},
    "INTROVOCO": {"-": ["51"]},
    "METVLA": {"-": ["11"]},
    "CLANDESTINO": {"-": ["6-"]},
    "MACESCO": {"-": ["53"]},
    "MERACVLVS": {"-": ["21"]},
    "CYPEROS": {"-": ["17"]},
    "FATVOR": {"1": ["5J"]},
    "INFELICO": {"-": ["51"]},
    "VINDEMITOR": {"-": ["13"], "N": ["13"]},
    "NAVFRAGO": {"-": ["51"]},
    "LVSTROR": {"-": ["5J"]},
    "DEPLORABVNDVS": {"-": ["21"]},
    "PEDICELLVS": {"-": ["12"]},
    "VENTVLVS": {"-": ["12"]},
    "ARQVATVS": {"-": ["12"]},
    "ANIMVLVS": {"-": ["12"]},
    "ATTACTVS": {"-": ["14"]},
    "DAPINO": {"-": ["51"]},
    "AMASIVNCVLA": {"-": ["11"]},
    "BEROE": {"N": ["11"]},
    "BIFARIAM": {"-": ["6-"]},
    "FOEDIFRAGVS": {"-": ["21"]},
    "BACALVSIAE": {"-": ["11"]},
    "ALCESTIS": {"N": ["13"]},
    "OCCILLO": {"-": ["51"]},
    "ANTIQVO": {"-": ["51"]},
    "PEDISEQVVS": {"-": ["21"]},
    "VERNILIS": {"-": ["24"]},
    "OBROGO": {"-": ["51"]},
    "SALVE": {"-": ["13"]},
    "VNAETVICESIMANI": {"-": ["12"]},
    "REMELIGO": {"-": ["13"]},
    "MILVINVS": {"-": ["21"]},
    "PRAEFISCINE": {"-": ["6-"]},
    "MVNDO": {"-": ["51"]},
    "ASTYANAX": {"N": ["13"]},
    "ACCVSATRIX": {"-": ["13"]},
    "PRAEDIATORIVS": {"-": ["21"]},
    "AGENOR": {"N": ["13"]},
    "INCONDITE": {"-": ["6-"]},
    "FVRTIFICVS": {"-": ["21"]},
    "CORRVPTE": {"-": ["6-"]},
    "HIRVDO": {"-": ["13"]},
    "PERTRISTIS": {"-": ["24"]},
    "LANIVS": {"-": ["12"]},
    "FAVTRIX": {"-": ["13"]},
    "PVGILATVS": {"-": ["14"]},
    "STRATEGVS": {"-": ["12"]},
    "MANVLEARIVS": {"-": ["12"]},
    "INDVCVLA": {"-": ["11"]},
    "SOLEARIVS": {"-": ["12"]},
    "FLICTVS": {"-": ["14"]},
    "POMPILVS": {"-": ["12"]},
    "TORRIS": {"-": ["13"]},
    "PHASELVS": {"-": ["12"]},
    "APPARATE": {"-": ["6-"]},
    "EGVRGITO": {"-": ["51"]},
    "RETICVLVS": {"-": ["12"]},
    "ARVINA": {"-": ["11"]},
    "IRACVNDE": {"-": ["6-"]},
    "PROMISSIO": {"-": ["13"]},
    "CANDELABRVS": {"-": ["12"]},
    "OFFIRMATVS": {"-": ["21"]},
    "DVODETRICIENS": {"-": ["6-"]},
    "PERATTENTE": {"-": ["6-"]},
    "PRAESAGIOR": {"-": ["5M"]},
    "CONSILIATOR": {"-": ["13"]},
    "PACATE": {"-": ["6-"]},
    "FVLCIMEN": {"-": ["13"]},
    "RAPA": {"-": ["11"]},
    "PIGNEROR": {"-": ["5J"]},
    "POSTILENA": {"-": ["11"]},
    "CALCITRO": {"1": ["13"], "2": ["12"]},
    "SILVESCO": {"-": ["53"]},
    "EXANIMALIS": {"-": ["24"]},
    "GALERITVS": {"-": ["12"]},
    "PERLECEBRA": {"-": ["11"]},
    "PRAEPROPERVS": {"-": ["21"]},
    "MALEVOLENS": {"-": ["13"]},
    "OMINATOR": {"-": ["13"]},
    "DECIENS": {"-": ["6-"]},
    "PERSCRVTO": {"-": ["51"]},
    "CHALCIOPE": {"N": ["11"]},
    "SVPELLECTICARIVS": {"-": ["21"]},
    "CAMELLA": {"-": ["11"]},
    "LEPAS": {"-": ["13"]},
    "MODIALIS": {"-": ["24"]},
    "SORBILIS": {"-": ["24"]},
    "INCLYTVS": {"-": ["21"]},
    "ANATINA": {"-": ["11"]},
    "DEARTVO": {"-": ["51"]},
    "REPORRIGO": {"2": ["53"]},
    "LINGVOSVS": {"-": ["21"]},
    "MORTVALIA": {"-": ["13"]},
    "TANAIS": {"N": ["13"]},
    "PERIVCVNDE": {"-": ["6-"]},
    "SEMELE": {"N": ["11"]},
    "ERIGONE": {"N": ["11"]},
    "FRVNISCOR": {"-": ["5L"]},
    "NEVTRO": {"-": ["6-"]},
    "ADVENTICIA": {"-": ["11"]},
    "NAVCLERICVS": {"-": ["21"]},
    "FERRVMINO": {"-": ["51"]},
    "RAMENTA": {"-": ["11"]},
    "CALPE": {"N": ["11"]},
    "PROELIARIS": {"-": ["24"]},
    "PIABILIS": {"-": ["24"]},
    "ARGVTOR": {"-": ["5J"]},
    "SCORTEVS": {"-": ["21"]},
    "PROMVLSIDARE": {"-": ["13"]},
    "ANTIGONE": {"N": ["17"]},
    "GERENS": {"-": ["22"]},
    "SVPPARASITOR": {"-": ["5J"]},
    "TRYGONVS": {"-": ["12"]},
    "EXAGGERATVS": {"-": ["21"]},
    "OCLOPETA": {"-": ["11"]},
    "CONDVCIBILIS": {"-": ["24"]},
    "PRAECVRSIO": {"-": ["13"]},
    "INDIREPTVS": {"-": ["21"]},
    "SPORTELLA": {"-": ["11"]},
    "PEROPTATVS": {"-": ["21"]},
    "EMPTICIVS": {"-": ["21"]},
    "ILLEX": {"1": ["13"]},
    "TRICAE": {"-": ["11"]},
    "BASILICE": {"-": ["17"]},
    "DVRIVSCVLVS": {"-": ["21"]},
    "LYRISTES": {"-": ["17"]},
    "PVLTARIVS": {"-": ["12"]},
    "INTERAESTVO": {"-": ["51"]},
    "FALSILOQVVS": {"-": ["12"]},
    "PEDIS": {"-": ["13"]},
    "QVADRIPARTITO": {"-": ["6-"]},
    "NONAGIENS": {"-": ["6-"]},
    "SVBACCVSO": {"-": ["51"]},
    "SCELESTE": {"-": ["6-"]},
    "MICARIVS": {"-": ["21"]},
    "SEXTVSDECIMVS": {"-": ["21"]},
    "OPHTHALMIA": {"-": ["11"]},
    "INDVBITO": {"-": ["51"]},
    "INSTANTIA": {"-": ["11"]},
    "STRABONVS": {"-": ["21"]},
    "NVPTVS": {"-": ["21"]},
    "INTERCALARIS": {"-": ["24"]},
    "PRAEFOCO": {"-": ["51"]},
    "OBDORMIO": {"-": ["54"]},
    "SEPTIMANI": {"-": ["12"]},
    "REMANSIO": {"-": ["13"]},
    "PERENDIE": {"-": ["6-"]},
    "GRADATVS": {"2": ["14"]},
    "CONSESSOR": {"-": ["13"]},
    "DVELLICVS": {"-": ["21"]},
    "CALAVTICA": {"-": ["11"]},
    "PERCA": {"-": ["11"]},
    "SVBRVMPO": {"-": ["53"]},
    "LACO": {"N": ["13"]},
    "LICITATOR": {"-": ["13"]},
    "DIABATHRARIVS": {"-": ["12"]},
    "IMPERIVRATVS": {"-": ["21"]},
    "STAMINATVS": {"-": ["21"]},
    "MASTRVCATVS": {"-": ["21"]},
    "PALLENE": {"N": ["11"]},
    "MVLTIGENERIS": {"-": ["24"]},
    "CYANE": {"N": ["11"]},
    "TABVLARIS": {"-": ["24"]},
    "INCONSVMPTVS": {"-": ["21"]},
    "PAPAVEREVS": {"-": ["21"]},
    "TRICLINIA": {"-": ["11"]},
    "AMICVLA": {"-": ["11"]},
    "SVFFARCINO": {"-": ["51"]},
    "IANITRIX": {"-": ["13"]},
    "TROS": {"N": ["12"], "A": ["25"]},
    "INHALO": {"-": ["51"]},
    "EXAMVSSIM": {"-": ["6-"]},
    "OPIPARVS": {"-": ["21"]},
    "ALCEDONIA": {"-": ["12"]},
    "DENTILEGVS": {"-": ["21"]},
    "MENDICOR": {"-": ["5J"]},
    "COMBARDVS": {"-": ["21"]},
    "INVRBANVS": {"-": ["21"]},
    "ERVDITE": {"-": ["6-"]},
    "PYTHO": {"N": ["13"]},
    "CATASTROPHA": {"-": ["11"]},
    "ERYX": {"N": ["13"]},
    "BOLVS": {"-": ["12"]},
    "PRAEDIATOR": {"-": ["13"]},
    "NEVTRVBI": {"-": ["6-"]},
    "TRANSADIGO": {"-": ["53"]},
    "PARIS": {"N": ["13"]},
    "INSIGNITE": {"-": ["6-"]},
    "CROCIO": {"-": ["53"]},
    "PRAEMERCOR": {"-": ["5J"]},
    "PERBLANDVS": {"-": ["21"]},
    "CASSIOPE": {"N": ["11"]},
    "DATATIM": {"-": ["6-"]},
    "APOSPHRAGISMA": {"-": ["13"]},
    "SIDON": {"N": ["13"]},
    "INGVRGITO": {"-": ["51"]},
    "INORATVS": {"-": ["21"]},
    "INCEPTO": {"-": ["51"]},
    "NITIDE": {"-": ["6-"]},
    "VOLSELLA": {"-": ["11"]},
    "COMMARITVS": {"-": ["12"]},
    "HYPSIPYLE": {"N": ["11"]},
    "OCCATOR": {"-": ["13"]},
    "SCIRPEVS": {"-": ["21"]},
    "LEVCOTHOE": {"N": ["11"]},
    "CARDVELIS": {"-": ["13"]},
    "SVBHORRIDVS": {"-": ["21"]},
    "PINSITO": {"-": ["51"]},
    "BOIA": {"-": ["11"]},
    "DOMEFACTVS": {"-": ["21"]},
    "TRADVX": {"-": ["13"]},
    "CONSOBRINA": {"-": ["11"]},
    "AENEOLVS": {"-": ["21"]},
    "MATRICIDA": {"-": ["11"]},
    "CONDIGNE": {"-": ["6-"]},
    "MASO": {"N": ["13"]},
    "ALEATORIVS": {"-": ["12"]},
    "HOSTIFICVS": {"-": ["21"]},
    "COMMINISCOR": {"-": ["5L"]},
    "HERBESCO": {"-": ["53"]},
    "IMMISSIO": {"-": ["13"]},
    "PRAELVSIO": {"-": ["13"]},
    "CAERVLVS": {"-": ["21"]},
    "CYPASSIS": {"N": ["13"]},
    "COMMEMINI": {"-": ["53"]},
    "SVCCEDANEVS": {"-": ["12"]},
    "PORCINARIVS": {"-": ["12"]},
    "TRANSVERSARIVS": {"-": ["21"]},
    "INCLAMITO": {"-": ["51"]},
    "IOLE": {"N": ["11"]},
    "ANTECESSIO": {"-": ["13"]},
    "CONCASTIGO": {"-": ["51"]},
    "RAVIS": {"-": ["13"]},
    "NOTATVS": {"-": ["21"]},
    "NARBO": {"N": ["13"]},
    "PERGRAECOR": {"-": ["5J"]},
    "FENERATO": {"-": ["6-"]},
    "PERPROPINQVVS": {"-": ["21"]},
    "HELIOCAMINVS": {"-": ["12"]},
    "NYCTIMENE": {"N": ["11"]},
    "SALPA": {"-": ["11"]},
    "GESTICVLOR": {"-": ["5J"]},
    "SALSVRA": {"-": ["11"]},
    "FRVMENTATOR": {"-": ["51"]},
    "EVADNE": {"N": ["11"]},
    "DELOS": {"N": ["12"]},
    "EXCIDIO": {"-": ["13"]},
    "INTERREX": {"-": ["13"]},
    "PERLONGVS": {"-": ["21"]},
    "SVBAVRATVS": {"-": ["21"]},
    "QVARTVSDECIMVS": {"-": ["21"]},
    "OBDORMISCO": {"-": ["53"]},
    "CONGEMINO": {"-": ["51"]},
    "BELLATORIVS": {"-": ["21"]},
    "FEBRVA": {"-": ["12"]},
    "GLIS": {"-": ["13"]},
    "PRODICO": {"2": ["53"]},
    "CALIDE": {"-": ["6-"]},
    "CONTENTIOSVS": {"-": ["21"]},
    "SCVTIGERVLVS": {"-": ["12"]},
    "RIVVLVS": {"-": ["12"]},
    "CONGRESSIO": {"-": ["13"]},
    "POESIS": {"-": ["13"]},
    "DICACVLVS": {"-": ["21"]},
    "TERTIADECIMANI": {"-": ["12"]},
    "SEPTVENNIS": {"-": ["24"]},
    "ORPHVS": {"-": ["12"]},
    "SPVRCIDICVS": {"-": ["21"]},
    "CROTALIA": {"-": ["12"]},
    "DVCENTIENS": {"-": ["6-"]},
    "PERVIVO": {"-": ["53"]},
    "VEIENTO": {"N": ["13"]},
    "FICTE": {"-": ["6-"]},
    "VSVFACIO": {"-": ["55"]},
    "SVPERVENTVS": {"-": ["14"]},
    "BADIZO": {"-": ["51"]},
    "CONGRVENTIA": {"-": ["13"]},
    "CHALCIS": {"N": ["13"]},
    "EDORMISCO": {"-": ["53"]},
    "MYS": {"N": ["13"]},
    "FRVSINO": {"N": ["13"]},
    "EXPVDORATVS": {"-": ["21"]},
    "CONFIRMATOR": {"-": ["13"]},
    "BETO": {"-": ["53"]},
    "DICTAMNVS": {"-": ["12"]},
    "TVRILEGVS": {"-": ["21"]},
    "VFENS": {"N": ["13"]},
    "DROMO": {"N": ["13"]},
    "HALOSIS": {"N": ["13"]},
    "VRANIE": {"N": ["17"]},
    "TYPHON": {"N": ["13"]},
    "RAPO": {"N": ["13"]},
    "INACHIS": {"N": ["13"]},
    "SENECIO": {"N": ["13"]},
    "LVCRO": {"N": ["11"]},
    "TRIO": {"N": ["13"]},
    "CEYX": {"N": ["13"]},
    "LVRCO": {"N": ["13"]},
    "PHALERIS": {"N": ["13"]},
    "SACRATOR": {"N": ["13"]},
    "BETIS": {"N": ["13"]},
    "LEVCASPIS": {"N": ["13"]},
    "ACANTHIS": {"N": ["13"]},
    "AEGOCEROS": {"N": ["17"]},
    "PELLIO": {"N": ["13"]},
    "MINIO": {"N": ["11"]},
    "SVAVISAVIATIO": {"N": ["13"]},
    "SILO": {"N": ["13"]},
    "POLLIS": {"N": ["13"]},
    "INO": {"N": ["13"]},
    "PVLLO": {"N": ["11"]},
    "MEFITIS": {"N": ["13"]},
    "PROSELENOS": {"N": ["17"]},
    "ARGITIS": {"N": ["13"]},
    "VESPERVGO": {"N": ["13"]},
    "VESPILLO": {"N": ["13"]},
    "DAPHNE": {"N": ["17"]},
    "METAMORPHOSIS": {"N": ["13"]},
    "MVTTO": {"N": ["13"]},
    "BVRDO": {"N": ["13"]},
    "LATOIS": {"N": ["16"]},
    "DAVNIAS": {"A": ["25"]},
    "ASOPIS": {"N": ["16"]},
    "DODONIS": {"N": ["16"]},
    "OMPHIS": {"N": ["16"]},
    "LARINAS": {"A": ["25"]},
    "ILIBERRIS": {"N": ["16"]},
    "BACCHIS": {"N": ["16"]},
    "AMARYLLIS": {"N": ["16"]},
    "PITTHEIS": {"N": ["16"]},
    "ZIOBETIS": {"N": ["16"]},
    "TOMYRIS": {"N": ["16"]},
    "THYRSIS": {"N": ["16"]},
    "GARAMAS": {"A": ["25"]},
    "STRATONICIS": {"N": ["16"]},
    "LYCORIS": {"N": ["16"]},
    "IGVVINAS": {"A": ["25"]},
    "BARDYLIS": {"N": ["16"]},
    "SABIS": {"N": ["16"]},
    "POTNIAS": {"A": ["25"]},
    "CADMEIS": {"N": ["16"]},
    "ABARIS": {"N": ["16"]},
    "SAGARITIS": {"N": ["16"]},
    "HIAROTIS": {"N": ["16"]},
    "ATREBAS": {"A": ["25"]},
    "ALETRINAS": {"A": ["25"]},
    "EPICHARIS": {"N": ["16"]},
    "VENAFER": {"A": ["23"]},
    "AQVINAS": {"A": ["25"]},
    "PHEGIS": {"N": ["16"]},
    "TRINACRIS": {"N": ["16"]},
    "AMAZONIS": {"N": ["16"]},
    "ATLANTIS": {"N": ["16"]},
    "CEPHEIS": {"N": ["16"]},
    "LIMNATIS": {"N": ["16"]},
    "CERIALIS": {"N": ["16"]},
    "BASSARIS": {"N": ["16"]},
    "BOEBEIS": {"N": ["16"]},
    "ACHAIAS": {"A": ["25"]},
    "INSVBER": {"A": ["23"]},
    "VADIMONIS": {"N": ["16"]},
    "PALLANTIS": {"N": ["16"]},
    "ERECHTHIS": {"N": ["16"]},
    "AGANIPPIS": {"N": ["16"]},
    "CISSEIS": {"N": ["16"]},
    "RHESCVPORIS": {"N": ["16"]},
    "TAENARIS": {"N": ["16"]},
    "THERMVTIS": {"N": ["16"]},
    "BYBLIS": {"N": ["16"]},
    "SIDONIS": {"N": ["16"]},
    "LOCRIS": {"N": ["16"]},
    "NISIAS": {"A": ["25"]},
    "THYNIAS": {"A": ["25"]},
    "SESOSIS": {"N": ["16"]},
    "CARTHASIS": {"N": ["16"]},
    "POMPEIOPOLIS": {"N": ["16"]},
    "DANAIS": {"N": ["16"]},
    "BELIS": {"N": ["16"]},
    "STELLAS": {"A": ["25"]},
    "OSIRIS": {"N": ["16"]},
    "ATRACIS": {"N": ["16"]},
    "PANNYCHIS": {"N": ["16"]},
    "SCYRIAS": {"A": ["25"]},
    "SALMONIS": {"N": ["16"]},
    "THEBAIS": {"N": ["16"]},
    "AETOLIS": {"N": ["16"]},
    "AZANIS": {"N": ["16"]},
    "ATHESIS": {"N": ["16"]},
    "OENEIS": {"N": ["16"]},
    "ACTIAS": {"A": ["25"]},
    "SEBETHIS": {"N": ["16"]},
    "ARARIS": {"N": ["16"]},
    "THYMBRIS": {"N": ["16"]},
    "TRITONIDIS": {"N": ["16"]},
    "CANTABER": {"A": ["23"]},
    "CYTAEIS": {"N": ["16"]},
    "PASITIGRIS": {"N": ["16"]},
    "THALESTRIS": {"N": ["16"]},
    "VRBINAS": {"A": ["25"]},
    "TAMESIS": {"N": ["16"]},
    "ICARIS": {"N": ["16"]},
    "SOTERIS": {"N": ["16"]},
    "MANAPIS": {"N": ["16"]},
    "VEIOVIS": {"N": ["16"]},
    "DENTHALIAS": {"A": ["25"]},
    "SISIGAMBIS": {"N": ["16"]},
    "CALABER": {"A": ["23"]},
    "LAIS": {"N": ["16"]},
    "LYRNESSIS": {"N": ["16"]},
    "RAVENNAS": {"A": ["25"]},
    "CANINEFAS": {"A": ["25"]},
    "ICARIOTIS": {"N": ["16"]},
    "EVRYTIS": {"N": ["16"]},
    "APOLLONIS": {"N": ["16"]},
    "DICARCHIS": {"N": ["16"]},
    "NYSIAS": {"A": ["25"]},
    "NAIS": {"N": ["16"]},
    "CRAVGIS": {"N": ["16"]},
    "INACHIOTIS": {"N": ["16"]},
    "SINIS": {"N": ["16"]},
    "SCHOENEIS": {"N": ["16"]},
    "SICORIS": {"N": ["16"]},
    "ARCHEPOLIS": {"N": ["16"]},
    "MYCENIS": {"N": ["16"]},
    "AORNIS": {"N": ["16"]},
    "ANTIAS": {"A": ["25"]},
    "BOCCHORIS": {"N": ["16"]},
    "HAEMONIS": {"N": ["16"]},
    "ARDEAS": {"A": ["25"]},
    "PRIVERNAS": {"A": ["25"]},
    "CHRYSEIS": {"N": ["16"]},
    "ALEXIS": {"N": ["16"]},
    "CYROPOLIS": {"N": ["16"]},
    "IBER": {"A": ["23"]},
    "MENECRATIS": {"N": ["16"]},
    "DAMALIS": {"N": ["16"]},
    "CELENDRIS": {"N": ["16"]},
    "VCVBIS": {"N": ["16"]},
    "DINIS": {"N": ["16"]},
    "CLEOPHIS": {"N": ["16"]},
    "SIMOIS": {"N": ["16"]},
    "SVGAMBER": {"A": ["23"]},
    "DITIS": {"N": ["16"]},
    "TRVCCVLENSIS": {"N": ["16"]},
    "BITHYNIS": {"N": ["16"]},
    "PHARIS": {"N": ["16"]},
    "PTOLEMAIS": {"N": ["16"]},
    "TANAITIS": {"N": ["16"]},
    "AGONIS": {"N": ["16"]},
    "MENNIS": {"N": ["16"]},
    "SAMIRAMIS": {"N": ["16"]},
    "OPIS": {"N": ["16"]},
    "PELOPEIAS": {"A": ["25"]},
    "PIMPLEIS": {"N": ["16"]},
    "ATINAS": {"A": ["25"], "N": ["13"]},
    "PHORONIS": {"N": ["16"]},
    "THESPIS": {"N": ["16"]},
    "NYCTEIS": {"N": ["16"]},
    "LEVCIPPIS": {"N": ["16"]},
    "EDONIS": {"N": ["16"]},
    "CELTIBER": {"A": ["23"]},
    "ANVBIS": {"N": ["16"]},
    "PELASGIS": {"N": ["16"]},
    "AMANICAE": {"A": ["25"]},
    "BACENIS": {"N": ["16"]},
    "LATRIS": {"N": ["16"]},
    "PEDASIS": {"N": ["16"]},
    "IASIS": {"N": ["16"]},
    "SITHON": {"A": ["25"]},
    "TELESTIS": {"N": ["16"]},
    "CAPPADOX": {"A": ["25"]},
    "HYPANIS": {"N": ["16"]},
    "TANTALIS": {"N": ["16"]},
    "CASINAS": {"A": ["25"]},
    "DYSPARIS": {"N": ["16"]},
    "ALBIS": {"N": ["16"]},
    "TRALLIS": {"N": ["16"]},
    "CAR": {"A": ["25"]},
    "ACARNAN": {"A": ["25"]},
    "CLANIS": {"N": ["16"]},
    "TVNGER": {"A": ["23"]},
    "CILIX": {"A": ["25"]},
    "LYCAONIS": {"N": ["16"]},
    "CANNINEFAS": {"A": ["25"]},
    "CRES": {"A": ["25"]},
    "SCYDROTHEMIS": {"N": ["16"]},
    "FVLGINAS": {"A": ["25"]},
    "THESTYLIS": {"N": ["16"]},
    "ASIS": {"N": ["16"]},
    "OEBALIS": {"N": ["16"]},
    "CILISSA": {"A": ["21"]},
    "DAVLIAS": {"A": ["25"]},
    "BRISEIS": {"N": ["16"]},
    "RHACOTIS": {"N": ["16"]},
    "NYSIGENA": {"A": ["21"]},
    "PELASGIAS": {"A": ["25"]},
    "THESEIS": {"N": ["16"]},
    "CALIS": {"N": ["16"]},
    "VISVRGIS": {"N": ["16"]},
    "CONVICTOLITAVIS": {"N": ["16"]},
    "SCYTHES": {"A": ["25"]},
    "SVDANS": {"A": ["25"]},
    "ATHAMANTIS": {"N": ["16"]},
    "DVBIS": {"N": ["16"]},
    "LIBYS": {"A": ["25"]},
    "PRIAMEIS": {"N": ["16"]},
    "RHASCYPOLIS": {"N": ["16"]},
    "CAERES": {"A": ["25"]},
    "PICENS": {"A": ["25"], "N": ["13"]},
    "TIBVRS": {"A": ["25"]},
    "EVPOLIS": {"N": ["16"]},
    "SPALIS": {"N": ["16"]},
    "OCEANITIS": {"N": ["16"]},
    "TVRESIS": {"N": ["16"]},
    "THRAEX": {"A": ["25"], "N": ["13"]},
    "AETHIOPS": {"A": ["25"]},
    "NISEIS": {"N": ["16"]},
    "PHORCIS": {"N": ["16"]},
    "PIERIS": {"N": ["16"]},
    "PHAEAX": {"A": ["25"]},
    "MEMPHITES": {"A": ["25"]},
    "ELEATES": {"A": ["25"]},
    "ACROPOLISTIS": {"N": ["16"]},
    "IAPYS": {"A": ["25"]},
    "CAMERS": {"A": ["25"], "N": ["13"]},
    "THESPIAS": {"A": ["25"]},
    "ALPIS": {"N": ["16"]},
    "NICOPOLIS": {"N": ["16"]},
    "STRYMONIS": {"N": ["16"]},
    "PHYLLODOCE": {"N": ["17"]},
    "BVBACENE": {"N": ["17"]},
    "BIBRACTE": {"N": ["17"]},
    "HELENE": {"N": ["17"]},
    "EVANTHE": {"N": ["17"]},
    "BARINE": {"N": ["17"]},
    "PHIDYLE": {"N": ["17"]},
    "TRICRENE": {"N": ["17"]},
    "ARTACIE": {"N": ["17"]},
    "PROCNE": {"N": ["17"]},
    "CHLOE": {"N": ["17"]},
    "PSYCHE": {"N": ["17"]},
    "ALCIPPE": {"N": ["17"]},
    "ACME": {"N": ["17"]},
    "ANDROMEDE": {"N": ["17"]},
    "NEPTVNINE": {"N": ["17"]},
    "LEVCOPHRYNE": {"N": ["17"]},
    "CEYCE": {"N": ["17"]},
    "ALSIENSE": {"N": ["17"]},
    "CAMERE": {"N": ["17"]},
    "CHLIDE": {"N": ["17"]},
    "SOPHENE": {"N": ["17"]},
    "EVTERPE": {"N": ["17"]},
    "COELE": {"N": ["17"]},
    "THEBE": {"N": ["17"]},
    "VSPE": {"N": ["17"]},
    "HELLANICE": {"N": ["17"]},
    "CARENE": {"N": ["17"]},
    "ALCMENE": {"N": ["17"]},
    "RHODE": {"N": ["17"]},
    "PHIALE": {"N": ["17"]},
    "EPHYRE": {"N": ["17"]},
    "SITTACENE": {"N": ["17"]},
    "SORACTE": {"N": ["17"]},
    "CYMODOCE": {"N": ["17"]},
    "AMYCLE": {"N": ["17"]},
    "MYRTALE": {"N": ["17"]},
    "LIPARE": {"N": ["17"]},
    "DIDYME": {"N": ["17"]},
    "ORTYGIE": {"N": ["17"]},
    "HARPALYCE": {"N": ["17"]},
    "LIBYE": {"N": ["17"]},
    "CYBELE": {"N": ["17"]},
    "HEBE": {"N": ["17"]},
    "DEIPHOBE": {"N": ["17"]},
    "NIOBE": {"N": ["17"]},
    "ILIONE": {"N": ["17"]},
    "EVROPE": {"N": ["17"]},
    "ATALANTE": {"N": ["17"]},
    "NEOBVLE": {"N": ["17"]},
    "LEVCONOE": {"N": ["17"]},
    "ROXANE": {"N": ["17"]},
    "ISCHOMACHE": {"N": ["17"]},
    "BARSINE": {"N": ["17"]},
    "CALYBE": {"N": ["17"]},
    "MOTHONE": {"N": ["17"]},
    "NISAEE": {"N": ["17"]},
    "IDE": {"N": ["17"]},
    "CALLIRHOE": {"N": ["17"]},
    "MELPOMENE": {"N": ["17"]},
    "PARTHIENE": {"N": ["17"]},
    "PARTHENIE": {"N": ["17"]},
    "THVLE": {"N": ["17"]},
    "PEGE": {"N": ["17"]},
    "SCARPHE": {"N": ["17"]},
    "HYPERMESTRE": {"N": ["17"]},
    "DINOMACHE": {"N": ["17"]},
    "HEDYLE": {"N": ["17"]},
    "ARELATE": {"N": ["17"]},
    "ANTIOPE": {"N": ["17"]},
    "NESAEE": {"N": ["17"]},
    "PALLACINE": {"N": ["17"]},
    "PHRYNE": {"N": ["17"]},
    "HERCVLANENSE": {"N": ["17"]},
    "PETALE": {"N": ["17"]},
    "CYBEBE": {"N": ["17"]},
    "HEROPHILE": {"N": ["17"]},
    "AEETINE": {"N": ["17"]},
    "CYTAEINE": {"N": ["17"]},
    "EVPHROSYNE": {"N": ["17"]},
    "ERIPHYLE": {"N": ["17"]},
    "PANTAGIE": {"N": ["17"]},
    "TAYGETE": {"N": ["17"]},
    "ANDROMACHE": {"N": ["17"]},
    "LALAGE": {"N": ["17"]},
    "ENISPE": {"N": ["17"]},
    "ANACTORIE": {"N": ["17"]},
    "LEDE": {"N": ["17"]},
    "THRACE": {"N": ["17"]},
    "CATABATHMOS": {"N": ["17"]},
    "ORSILOS": {"N": ["17"]},
    "ATHOS": {"N": ["17"]},
    "ILIOS": {"N": ["17"]},
    "CROCOS": {"N": ["17"]},
    "PHILEROS": {"N": ["17"]},
    "ISSOS": {"N": ["17"]},
    "SANBVLOS": {"N": ["17"]},
    "PEPARETHOS": {"N": ["17"]},
    "ARGOS": {"N": ["17"]},
    "SPERCHIOS": {"N": ["17"]},
    "PALAETYROS": {"N": ["17"]},
    "EVXENIPPOS": {"N": ["17"]},
    "OROPS": {"N": ["13"]},
    "PYRGO": {"N": ["13"]},
    "CONGRIO": {"N": ["13"]},
    "ALCANOR": {"N": ["13"]},
    "ARISTOGITON": {"N": ["13"]},
    "MACHAERIO": {"N": ["13"]},
    "ARGO": {"N": ["13"]},
    "CLOTHO": {"N": ["13"]},
    "EPOREDORIX": {"N": ["13"]},
    "ASPRENAS": {"N": ["13"]},
    "METRONAX": {"N": ["13"]},
    "HYLAX": {"N": ["13"]},
    "NVMITOR": {"N": ["13"]},
    "CVPAVO": {"N": ["13"]},
    "BRIMO": {"N": ["13"]},
    "THESAVROCHRYSONICOCH.": {"N": ["13"]},
    "RVFIO": {"N": ["13"]},
    "SER": {"N": ["13"]},
    "ASPAR": {"N": ["13"]},
    "CINGETORIX": {"N": ["13"]},
    "VOLVX": {"N": ["13"]},
    "TREVIR": {"N": ["13"]},
    "BOCCAR": {"N": ["13"]},
    "ATHO": {"N": ["13"]},
    "LVGOTORIX": {"N": ["13"]},
    "ARCENS": {"N": ["13"]},
    "AGGAR": {"N": ["13"]},
    "VANGIO": {"N": ["13"]},
    "AMBIORIX": {"N": ["13"]},
    "VOLCENS": {"N": ["13"]},
    "DIVICO": {"N": ["13"]},
    "DRYOPS": {"N": ["13"]},
    "BVRSAVO": {"N": ["13"]},
    "MASTANABAL": {"N": ["13"]},
    "ADHERBAL": {"N": ["13"]},
    "VRSAO": {"N": ["13"]},
    "GOBANNITIO": {"N": ["13"]},
    "HIEROCAESARIEN.": {"N": ["13"]},
    "CRVPTORIX": {"N": ["13"]},
    "DVMNORIX": {"N": ["13"]},
    "TIMO": {"N": ["13"]},
    "SACROVIR": {"N": ["13"]},
    "OLLOVICO": {"N": ["13"]},
    "SENO": {"N": ["13"]},
    "SEGOVAX": {"N": ["13"]},
    "STASANOR": {"N": ["13"]},
    "TAVROENS": {"N": ["13"]},
    "PVLEIO": {"N": ["13"]},
    "SPIO": {"N": ["13"]},
    "ZOPYRIO": {"N": ["13"]},
    "ANIEN": {"N": ["13"]},
    "TAPPO": {"N": ["13"]},
    "SVTHVL": {"N": ["13"]},
    "BRITTO": {"N": ["13"]},
    "VOCCIO": {"N": ["13"]},
    "POLYMESTOR": {"N": ["13"]},
    "CRESCENS": {"N": ["13"]},
    "COTYTO": {"N": ["13"]},
    "XANTHO": {"N": ["13"]},
    "BIANOR": {"N": ["13"]},
    "CARIO": {"N": ["13"]},
    "COTYLO": {"N": ["13"]},
    "DEMONAX": {"N": ["13"]},
    "RVSO": {"N": ["13"]},
    "MALORIX": {"N": ["13"]},
    "MATISCO": {"N": ["13"]},
    "PLATOR": {"N": ["13"]},
    "VIRIDOVIX": {"N": ["13"]},
    "IERVSALEM": {"N": ["13"]},
    "TIPHYS": {"N": ["13"]},
    "AGATHO": {"N": ["13"]},
    "NOBILIOR": {"N": ["13"]},
    "DRYMO": {"N": ["13"]},
    "CALAS": {"N": ["13"]},
    "MVTHVL": {"N": ["13"]},
    "TVISTO": {"N": ["13"]},
    "VERGILIO": {"N": ["13"]},
    "ARAR": {"N": ["13"]},
    "ORGETORIX": {"N": ["13"]},
    "ERINNYS": {"N": ["13"]},
    "VVLCHALO": {"N": ["13"]},
    "DOLOPS": {"N": ["13"]},
    "LATO": {"N": ["13"]},
    "BRINNO": {"N": ["13"]},
    "PITHON": {"N": ["13"]},
    "MACELO": {"N": ["13"]},
    "CAFO": {"N": ["13"]},
    "TVLLIO": {"N": ["13"]},
    "THEANO": {"N": ["13"]},
    "SYPHAX": {"N": ["13"]},
    "VENTIPO": {"N": ["13"]},
    "BAMBALIO": {"N": ["13"]},
    "GRAECVLIO": {"N": ["13"]},
    "PITHO": {"N": ["13"]},
    "VESONTIO": {"N": ["13"]},
    "TYPHO": {"N": ["13"]},
    "MERCELLO": {"N": ["13"]},
    "QVERCENS": {"N": ["13"]},
    "POETOVIO": {"N": ["13"]},
    "LIBAS": {"N": ["13"]},
    "LIGER": {"N": ["13"]},
    "ISION": {"N": ["13"]},
    "LEPISO": {"N": ["13"]},
    "HIEMPSAL": {"N": ["13"]},
    "CYDRO": {"N": ["13"]},
    "VERCINGETORIX": {"N": ["13"]},
    "NASO": {"N": ["13"]},
    "HELENOR": {"N": ["13"]},
    "ACCO": {"N": ["13"]},
    "CARCASO": {"N": ["13"]},
    "BOMILCAR": {"N": ["13"]},
    "BIBRAX": {"N": ["13"]},
    "ATAX": {"N": ["13"]},
    "DABAR": {"N": ["13"]},
    "TRIMALCHIO": {"N": ["13"]},
    "CORDALIO": {"N": ["13"]},
    "GALLIO": {"N": ["13"]},
    "SEMO": {"N": ["13"]},
    "VERTICO": {"N": ["13"]},
    "ASTYR": {"N": ["13"]},
    "FELICIO": {"N": ["13"]},
    "VIRRO": {"N": ["13"]},
    "COCOCOCO": {"-": ["--"]},
    "EXPRETVS": {"-": ["21"]},
    "DISMARITVS": {"-": ["21"]},
    "NONAGENVS": {"-": ["32"]},
    "MILITATVRA": {"-": ["11"]},
    "OPTESTOR": {"-": ["5J"]},
    "EM": {"-": ["90"]},
    "PTHISICVS": {"-": ["21"]},
    "STOICE": {"-": ["6&"]},
    "ODIOSSICVS": {"-": ["21"]},
    "NVBIGENA": {"-": ["27"]},
    "THYIADES": {"-": ["17"]},
    "QVADRIGENTI": {"-": ["31"]},
    "PERMEDITOR": {"-": ["5J"]},
    "LARS": {"-": ["13"]},
    "FIDVCARIVS": {"-": ["21"]},
    "INEXSTINGVO": {"-": ["53"]},
    "ANTECIPIO": {"-": ["55"]},
    "CIRCVMVEHO": {"-": ["53"]},
    "BAT": {"-": ["--"]},
    "EDEPOL": {"-": ["90"]},
    "ABSENTIVVS": {"-": ["21"]},
    "VNDENVS": {"-": ["32"]},
    "EVGEPAE": {"-": ["90"]},
    "DVODENVS": {"-": ["32"]},
    "APAGE": {"-": ["90"]},
    "EHEM": {"-": ["90"]},
    "VAH": {"-": ["90"]},
    "ARRADO": {"-": ["53"]},
    "ECASTOR": {"-": ["90"]},
    "QVINQVAGENVS": {"-": ["32"]},
    "EHO": {"-": ["90"]},
    "SESCENTIENS": {"-": ["33"]},
    "CONCILIOR": {"-": ["5J"]},
    "IMMODVLOR": {"-": ["5J"]},
    "TRICENTIENS": {"-": ["33"]},
    "SEXCENVS": {"-": ["31"]},
    "BALINEVS": {"-": ["21"]},
    "DRACONIGENVS": {"-": ["21"]},
    "INVERENCVNDVS": {"-": ["21"]},
    "_EO": {"2": ["56"]},
    "QVASSILARIA": {"-": ["11"]},
    "SVDES": {"-": ["13"]},
    "BENEDICE": {"-": ["6&"]},
    "QVADRINVS": {"-": ["32"]},
    "QVINGENVS": {"-": ["31"]},
    "PHYRGIO": {"-": ["13"]},
    "AVERTOR": {"-": ["5L"]},
    "ST": {"-": ["--"]},
    "MOSCILLVS": {"-": ["12"]},
    "HEDIA": {"-": ["11"]},
    "VVLCANALIS": {"-": ["24"]},
    "AVCTORATI": {"-": ["12"]},
    "SESCENVS": {"-": ["32"]},
    "VERBERABILIS": {"-": ["24"]},
    "MV": {"-": ["--"]},
    "MA": {"-": ["--"]},
    "VNVSETVICESIMVS": {"-": ["32"]},
    "MILLIENS": {"-": ["33"]},
    "FVI": {"-": ["56"]},
    "ATTATAE": {"-": ["90"]},
    "INCALEFACIO": {"-": ["55"]},
    "OCCLVSVS": {"-": ["21"]},
    "TRECENVS": {"-": ["31"]},
    "PRAEPOLLVO": {"-": ["53"]},
    "REGILLA": {"-": ["11"]},
    "SOTADICVS": {"-": ["21"]},
    "TARRACONENSES": {"-": ["13"]},
    "TRIVENEFICVS": {"-": ["21"]},
    "TRECENTENVS": {"-": ["31"]},
    "SEXAGENVS": {"-": ["32"]},
    "SEPTVAGIENS": {"-": ["33"]},
    "SVMMANIS": {"-": ["24"]},
    "APOCALO": {"-": ["51"]},
    "PRAECOMPONO": {"-": ["53"]},
    "EXIVRO": {"-": ["51"]},
    "THYLACISTA": {"-": ["11"]},
    "INDEMNO": {"-": ["51"]},
    "RVMIFERO": {"-": ["51"]},
    "HAMADRYAS": {"-": ["17"]},
    "VENERIAE": {"-": ["11"]},
    "ACIA": {"-": ["11"]},
    "THIMIAMAE": {"-": ["11"]},
    "OFFERRO": {"-": ["53"]},
    "COMPEDES": {"-": ["25"]},
    "INNOXIE": {"-": ["6&"]},
    "PVNCTATORIOLA": {"-": ["11"]},
    "ILLVNIS": {"-": ["34"]},
    "PLAGIPATIDA": {"-": ["11"]},
    "ANDROGYNOS": {"-": ["17"]},
    "PRASINIANVS": {"-": ["21"]},
    "OCCENDO": {"-": ["53"]},
    "PERNVLA": {"-": ["11"]},
    "EXCELLENTIS": {"-": ["24"]},
    "STRIGOR": {"-": ["13"]},
    "SESQVEOPVS": {"-": ["13"]},
    "CINIPES": {"-": ["13"]},
    "INGVRGITOR": {"-": ["5J"]},
    "CAVNEA": {"-": ["11"]},
    "INEXSPECTO": {"-": ["51"]},
    "LOQVO": {"-": ["53"]},
    "IMMEMORO": {"-": ["51"]},
    "CALVAE": {"-": ["11"]},
    "NESCIOQVID": {"-": ["48"]},
    "NOMVS": {"-": ["12"]},
    "DVETVICESIMVS": {"-": ["32"]},
    "LOGVS": {"-": ["12"]},
    "GRAMA": {"-": ["11"]},
    "CLATHRA": {"-": ["12"]},
    "VETVO": {"-": ["51"]},
    "SANGVNCVLVS": {"-": ["12"]},
    "BISSACIVS": {"-": ["12"]},
    "HIPPODROMOS": {"-": ["17"]},
    "INTERVIAS": {"-": ["26"]},
    "LAMYROS": {"-": ["17"]},
    "CLASSIA": {"-": ["11"]},
    "DIVIDIA": {"-": ["11"]},
    "CHIRAGRICI": {"-": ["12"]},
    "FIVO": {"-": ["53"]},
    "CHRYSOS": {"-": ["17"]},
    "PRAEVT": {"-": ["69"]},
    "THEBAICAE": {"-": ["11"]},
    "PENTHIACVS": {"-": ["12"]},
    "APSIS": {"-": ["17"]},
    "VESTES": {"-": ["13"]},
    "COMMISSVS": {"-": ["21"]},
    "D": {"-": ["--"]},
    "IMPOLLITVS": {"-": ["21"]},
    "VELANDA": {"-": ["12"]},
    "QVAQVAM": {"-": ["68"]},
    "AV": {"-": ["90"]},
    "STERTEIVS": {"-": ["21"]},
    "CAELVS": {"-": ["12"]},
    "VLS": {"-": ["7-"]},
    "POTICIVS": {"-": ["12"]},
    "OPISTOGRAPHVS": {"-": ["21"]},
    "MARTIOLVS": {"-": ["12"]},
    "STOLATA": {"-": ["11"]},
    "ASSVDASCO": {"-": ["53"]},
    "INDECET": {"-": ["52"]},
    "CEDRA": {"-": ["11"]},
    "NONNIHIL": {"-": ["26"]},
    "EVAX": {"-": ["90"]},
    "TAMIAS": {"-": ["13"]},
    "NVMNAM": {"-": ["67"]},
    "LASANVS": {"-": ["12"]},
    "NVMQVO": {"-": ["67"]},
    "NVMQVISNAM": {"-": ["47"]},
    "ADVSQVE": {"2": ["70"]},
    "CANALE": {"-": ["13"]},
    "NESCIOQVOMODO": {"-": ["67"]},
    "PORGO": {"-": ["53"]},
    "SERE": {"-": ["60"]},
    "EXOPINISSO": {"-": ["21"]},
}
mapping = {
    "AB": {
        "-": {
            "morpho": ["p---------"], "uri": ["a0001"],
        },
    },
    "ABACTVS": {
        "-": {
            "morpho": ["n-s---mn4-"],
            "uri": ["a9982"],
            },
            },
    "ABACVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0006"]}},
    "ABALIENO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0015"]}},
    "ABATOS": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "ABAVVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0026"]}},
    "ABBAS": {"-": {"morpho": ["n-s---mn3-"], "uri": ["41588"]}},
    "ABDICO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a0034"]}},
    "ABDITVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["46918"]}},
    "ABDITVS": {"C": {"morpho": ["aps---mn1-"], "uri": ["34180"]}, "-": {"morpho": ["aps---mn1-"], "uri": ["34180"]}},
    "ABDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0038"]}},
    "ABDOMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a0039"]}},
    "ABDVCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0040"]}},
    "ABEO": {"1": {"morpho": ["v1spia--4-"], "uri": ["a0043"]}},
    "ABERRO": {"2": {"morpho": ["v1spia--1-"], "uri": ["a0046"]}},
    "ABHINC": {"-": {"morpho": ["rp--------"], "uri": ["a0049"]}},
    "ABHORREO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0050"]}},
    "ABHORRIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0051"]}},
    "ABICIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a0053"]}},
    "ABIECTE": {"-": {"morpho": ["rp--------", "rp--------"], "uri": ["44049", "44049"]}},
    "ABIECTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["43027"]}},
    "ABIEGNEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["51153"]}},
    "ABIEGNVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0055"]}},
    "ABIES": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0057"]}},
    "ABIGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0063"]}},
    "ABITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a9984"]}},
    "ABIVDICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0067"]}},
    "ABIVNGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0069"]}},
    "ABIVRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0073"]}},
    "ABLAQVEO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a0077"]}},
    "ABLEGO": {"1": {"morpho": ["v1spia--1-"], "uri": ["$0149"]}},
    "ABLEVO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a0085"]}},
    "ABLVDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0091"]}},
    "ABLVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a9091"]}},
    "ABNEGO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0101"]}},
    "ABNEPOS": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0102"]}},
    "ABNOCTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0104"]}},
    "ABNORMIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0106"]}},
    "ABNVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0111"]}},
    "ABOLEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0116"]}},
    "ABOLESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1752"]}},
    "ABOLITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0117"]}},
    "ABOLLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0119"]}},
    "ABOMINOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a0122"]}},
    "ABORIOR": {"-": {"morpho": ["v1spid--4-"], "uri": ["a0125"]}},
    "ABORISCOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a1874"]}},
    "ABORTIVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0129"]}},
    "ABORTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a0128"]}},
    "ABRADO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0133"]}},
    "ABRIPIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a0140"]}},
    "ABROGO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0143"]}},
    "ABROTONVM": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "ABROTONVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["49773"]}},
    "ABRVMPO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0145"]}},
    "ABRVPTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["46275"]}},
    "ABRVPTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["54093"]}},
    "ABSCEDO": {"1": {"morpho": ["v1spia--3-"], "uri": ["a0147"]}},
    "ABSCESSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a0160"]}},
    "ABSCIDO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a0149"]}},
    "ABSCINDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0150"]}},
    "ABSCISIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0151"]}},
    "ABSCISVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["29057"]}}, "ABSCONDITVS": {
        "C": {"morpho": ["aps---mn1-"], "uri": ["57127"]}, "-": {"morpho": ["aps---mn1-"], "uri": ["57127"]}
    }, "ABSCONDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0155"]}},
    "ABSCONSVS": {"C": {"morpho": ["aps---mn1-"], "uri": ["47827"]}, "-": {"morpho": ["aps---mn1-"], "uri": ["47827"]}},
    "ABSENS": {"-": {"morpho": ["aps---an3i"], "uri": ["97491"]}},
    "ABSENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0184"]}},
    "ABSILIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a0165"]}},
    "ABSIMILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0166"]}},
    "ABSINTHIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["19167"]}},
    "ABSISTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0169"]}},
    "ABSOLVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0178"]}},
    "ABSOLVTE": {"-": {"morpho": ["rp--------"], "uri": ["19170"]}},
    "ABSOLVTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0174"]}},
    "ABSOLVTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19172"]}},
    "ABSONVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0180"]}},
    "ABSORBEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0181"]}},
    "ABSQVE": {"1": {"morpho": ["rp--------"], "uri": []}, "2": {"morpho": ["p---------"], "uri": ["a0183"]}},
    "ABSTANTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0186"]}},
    "ABSTEMIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0185"]}},
    "ABSTERGEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0187"]}},
    "ABSTERREO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0188"]}},
    "ABSTINENS": {"-": {"morpho": ["aps---an3i"], "uri": ["97494"]}},
    "ABSTINENTER": {"-": {"morpho": ["rp--------"], "uri": ["54689"]}},
    "ABSTINENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0190"]}},
    "ABSTINEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0191"]}},
    "ABSTRAHO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0195"]}},
    "ABSTRVDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0196"]}},
    "ABSTRVSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19184"]}},
    "ABSVM": {"1": {"morpho": ["v1spia--2-"], "uri": ["a0200"]}},
    "ABSVMO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0202"]}},
    "ABSVRDE": {"-": {"morpho": ["rp--------"], "uri": ["19187"]}},
    "ABSVRDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0205"]}},
    "ABVNDANS": {"1": {"morpho": ["n-s----n3-"], "uri": []}, "2": {"morpho": ["aps---an3i"], "uri": ["19189"]}},
    "ABVNDANTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0210"]}},
    "ABVNDE": {"-": {"morpho": ["rp--------"], "uri": ["19193"]}},
    "ABVNDO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0212"]}},
    "ABVNDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0213"]}},
    "ABVSQVE": {"-": {"morpho": ["p---------"], "uri": []}},
    "ABVTOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a0217"]}},
    "AC": {"1": {"morpho": ["----------"], "uri": []}, "2": {"morpho": ["----------"], "uri": []}},
    "ACADEMICI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ACADEMICVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ACALANTHIS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0219"]}},
    "ACANTHVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0225"]}},
    "ACCEDO": {"1": {"morpho": ["v1spia--3-"], "uri": ["a0268"]}},
    "ACCELERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0243"]}},
    "ACCENDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0245"]}},
    "ACCENSEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0247"]}},
    "ACCENSI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["54755"]}},
    "ACCENSVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a8916"]}, "2": {"morpho": ["n-s---mn4-"], "uri": ["a0277"]}},
    "ACCEPTATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["31591"]}},
    "ACCEPTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0256"]}},
    "ACCEPTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0258"]}},
    "ACCEPTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a8917"]}},
    "ACCEPTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19213"]}},
    "ACCESSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0265"]}},
    "ACCESSITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0266"]}},
    "ACCESSO": {"-": {"morpho": ["v1spia--1-"], "uri": []}},
    "ACCESSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a0239"]}},
    "ACCIDENS": {"-": {"morpho": ["n-s---nn3-"], "uri": ["19218"]}}, "ACCIDO": {
        "1": {"morpho": ["v1spia--3-"], "uri": ["a0269", "a0268"]},
        "2": {"morpho": ["v1spia--3-"], "uri": ["a0269", "a0268"]}
    }, "ACCIEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a9270"]}},
    "ACCINGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0272"]}},
    "ACCIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a0274"]}},
    "ACCIPIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a0275"]}},
    "ACCIPITER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0276"]}},
    "ACCITO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a0281"]}},
    "ACCITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a9989"]}},
    "ACCLAMATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0282"]}},
    "ACCLAMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0283"]}},
    "ACCLINIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0287"]}},
    "ACCLIVIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0289"]}},
    "ACCLIVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0290"]}},
    "ACCLIVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0289"]}},
    "ACCOGNOSCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0291"]}},
    "ACCOLA": {"-": {"morpho": ["n-s---cn1-"], "uri": ["a0292"]}},
    "ACCOLO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a0294"]}},
    "ACCOMMODATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0295"]}},
    "ACCOMMODATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["32391"]}},
    "ACCOMMODO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0298"]}},
    "ACCOMMODVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0299"]}},
    "ACCREDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0301"]}},
    "ACCREMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0302"]}},
    "ACCRESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0303"]}},
    "ACCVBITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0308"]}},
    "ACCVBO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0311"]}},
    "ACCVMBO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0314"]}},
    "ACCVMVLATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0316"]}},
    "ACCVMVLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0317"]}},
    "ACCVRATE": {"-": {"morpho": ["rp--------"], "uri": ["49145"]}},
    "ACCVRATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19234"]}},
    "ACCVRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0320"]}},
    "ACCVRRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0321"]}},
    "ACCVRSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a0364"]}},
    "ACCVSATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0323"]}},
    "ACCVSATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0325"]}},
    "ACCVSATORIE": {"-": {"morpho": ["rp--------"], "uri": ["41463"]}},
    "ACCVSATORIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0326"]}},
    "ACCVSO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0329"]}},
    "ACEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0333"]}},
    "ACER": {"1": {"morpho": ["n-s---nn3-"], "uri": ["a0335"]}, "2": {"morpho": ["aps---mn3-"], "uri": []}},
    "ACERBE": {"-": {"morpho": ["rp--------"], "uri": ["19242"]}},
    "ACERBITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0340"]}},
    "ACERBVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["43823"]}},
    "ACERBVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0343"]}},
    "ACERNVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0344"]}},
    "ACERRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0346"]}},
    "ACERSECOMES": {"-": {"morpho": ["n-s---mn1g"], "uri": ["a0347"]}},
    "ACERVATIM": {"-": {"morpho": ["rp--------"], "uri": ["a0350"]}},
    "ACERVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0352"]}},
    "ACERVVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0353"]}},
    "ACESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0333"]}},
    "ACESTENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "ACETABVLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0356"]}},
    "ACETVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0360"]}},
    "ACHAEVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ACHAIA": {"N": {"morpho": ["n-s---fn1-"], "uri": ["101203"]}},
    "ACHAICVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ACHIVI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ACHIVVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ACHRADINA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "ACIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0377"]}},
    "ACIES": {"-": {"morpho": ["n-s---fn5-"], "uri": ["a0379"]}},
    "ACILIVS": {"A": {"morpho": ["aps---mn1-"], "uri": ["97498"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ACINA": {"1": {"morpho": ["n-s---fn1-"], "uri": ["a0385"]}, "-": {"morpho": ["n-s---fn1-"], "uri": ["a0385"]}},
    "ACINACES": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "ACINVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0385"]}},
    "ACINVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0385"]}},
    "ACIPENSER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0386"]}},
    "ACOENONOETVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0393"]}},
    "ACONITVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0398"]}},
    "ACOSMOS": {"-": {"morpho": ["n-s----n-g"], "uri": []}},
    "ACQVIESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0406"]}},
    "ACQVIRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0407"]}},
    "ACQVISITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0408"]}},
    "ACRIMONIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0417"]}},
    "ACRITER": {"-": {"morpho": ["rp--------"], "uri": ["30866"]}},
    "ACRITVDO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0418"]}},
    "ACROAMA": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a0422"]}},
    "ACROASIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a0423"]}},
    "ACROBATICOS": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "ACROLITHVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["39093"]}},
    "ACROTERIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0435"]}},
    "ACTA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0436"]}},
    "ACTIACVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ACTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0441"]}},
    "ACTITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0445"]}},
    "ACTIVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a9446"]}},
    "ACTIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["97502"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["97502"]}},
    "ACTIVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0447"]}},
    "ACTOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0448"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "ACTVARIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0453"]}},
    "ACTVARIVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a0453"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a0453"]}},
    "ACTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a9991"]}},
    "ACTVOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0454"]}}, "ACTVS": {
        "1": {"morpho": ["n-s---mn4-"], "uri": ["a9991"]}, "2": {"morpho": ["aps---mn1-"], "uri": []},
        "A": {"morpho": ["aps---mn1-"], "uri": []}
    }, "ACTVTVM": {"-": {"morpho": ["rp--------"], "uri": ["a0455"]}},
    "ACVLEATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0459"]}},
    "ACVLEVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0460"]}},
    "ACVMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a0462"]}},
    "ACVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0464"]}}, "ACVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a0468"]}, "2": {"morpho": ["n-s---nn3-"], "uri": ["a0468"]},
        "3": {"morpho": ["n-s---fn4-"], "uri": ["a0468"]}
    }, "ACVTE": {"-": {"morpho": ["rp--------"], "uri": ["19282"]}},
    "ACVTVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0476"]}},
    "ACVTVM": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "ACVTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19283"]}},
    "AD": {"1": {"morpho": ["rp--------"], "uri": []}, "2": {"morpho": ["p---------"], "uri": ["a0479"]}},
    "ADACTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a9993"]}},
    "ADAEQVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0483"]}},
    "ADAGGERO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a0487"]}},
    "ADAMANTINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0494"]}},
    "ADAMAS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["19289"]}},
    "ADAMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0500"]}},
    "ADAPERIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a0502"]}},
    "ADAREO": {"-": {"morpho": ["v1spia--2-"], "uri": []}},
    "ADAVCTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a0477"]}},
    "ADAVGEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0512"]}},
    "ADAVGESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0513"]}},
    "ADAVGMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a0514"]}},
    "ADBIBO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a0516"]}},
    "ADDECET": {"-": {"morpho": ["v3spia--2-"], "uri": ["a0521"]}},
    "ADDICO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a0527"]}},
    "ADDICTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0528"]}},
    "ADDISCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0530"]}},
    "ADDITAMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0531"]}},
    "ADDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0535"]}},
    "ADDOCEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0536"]}},
    "ADDVBITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0541"]}},
    "ADDVCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0542"]}},
    "ADDVCTE": {"-": {"morpho": ["rp--------"], "uri": ["37971"]}},
    "ADDVCTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["30018"]}},
    "ADEDO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a0546"]}}, "ADEMO": {"-": {"morpho": ["v1spia--3-"], "uri": []}},
    "ADEMPTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0550"]}},
    "ADEO": {"1": {"morpho": ["v1spia--2-"], "uri": ["a0553"]}, "2": {"morpho": ["rp--------"], "uri": ["a0552"]}},
    "ADEPS": {"-": {"morpho": ["n-s---mn3i"], "uri": []}},
    "ADEQVITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0556"]}},
    "ADHAEREO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0561"]}},
    "ADHAERESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1878"]}},
    "ADHAESVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a9994"]}},
    "ADHIBEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0565"]}},
    "ADHINNIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a0567"]}},
    "ADHORTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0569"]}},
    "ADHORTATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0571"]}},
    "ADHORTOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a0573"]}},
    "ADHVC": {"-": {"morpho": ["rp--------"], "uri": ["a0575"]}},
    "ADIACEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0577"]}},
    "ADICIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a0586"]}},
    "ADICO": {"2": {"morpho": ["v1spia--3-"], "uri": []}},
    "ADIECTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0584"]}},
    "ADIECTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a9996"]}},
    "ADIGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0585"]}},
    "ADIMO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0588"]}},
    "ADINDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0593"]}},
    "ADIPATVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0605"]}},
    "ADIPES": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "ADIPISCOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a0607"]}},
    "ADITIALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0611"]}},
    "ADITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a9997"]}},
    "ADIVDICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0617"]}},
    "ADIVMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0619"]}},
    "ADIVNCTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0620"]}},
    "ADIVNCTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a8919"]}},
    "ADIVNCTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["43482"]}},
    "ADIVNGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0623"]}},
    "ADIVRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0628"]}},
    "ADIVTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0630"]}},
    "ADIVTOR": {"1": {"morpho": ["n-s---mn3-"], "uri": ["a0631"]}, "2": {"morpho": ["v1spid--1-"], "uri": []}},
    "ADIVTORIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0632"]}},
    "ADIVTRIX": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0633"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "ADIVVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0639"]}},
    "ADMATVRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0643"]}},
    "ADMETIOR": {"-": {"morpho": ["v1spid--4-"], "uri": ["a0647"]}},
    "ADMINICVLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0857"]}},
    "ADMINISTER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["a0655"]}},
    "ADMINISTRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0655"]}},
    "ADMINISTRATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0657"]}},
    "ADMINISTRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0662"]}},
    "ADMIRABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0663"]}},
    "ADMIRANDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19402"]}},
    "ADMIRATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0665"]}},
    "ADMIRATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0667"]}},
    "ADMIROR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a0668"]}},
    "ADMISCEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0669"]}},
    "ADMISSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0671"]}},
    "ADMISSVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a9998"]}},
    "ADMITTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0676"]}},
    "ADMODERATE": {"-": {"morpho": ["rp--------"], "uri": ["a0678"]}},
    "ADMODVM": {"-": {"morpho": ["rp--------"], "uri": ["a0681"]}},
    "ADMONEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0684"]}},
    "ADMONITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0685"]}},
    "ADMONITOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0687"]}},
    "ADMONITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a0711"]}},
    "ADMORDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0690"]}},
    "ADMOVEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0692"]}},
    "ADMVGIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a0693"]}},
    "ADMVRMVRATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0696"]}},
    "ADMVRMVRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0697"]}},
    "ADOLEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a1875"]}}, "ADOLESCENS": {
        "1": {"morpho": ["n-s---mn3i"], "uri": ["97520"]}, "2": {"morpho": ["aps---an3-"], "uri": ["38568"]}
    }, "ADOLESCENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["45968"]}},
    "ADOLESCENTVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["42136"]}},
    "ADOLESCO": {"1": {"morpho": ["v1spia--3-"], "uri": ["a0727"]}, "2": {"morpho": ["v1spia--3-"], "uri": ["a0727"]}},
    "ADOPERIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a0729"]}},
    "ADOPINOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a0732"]}},
    "ADOPTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0736"]}},
    "ADOPTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0738"]}},
    "ADOPTIVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0739"]}},
    "ADOPTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0740"]}}, "ADOR": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "ADOREA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["41906"]}},
    "ADOREVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2263"]}},
    "ADORIOR": {"-": {"morpho": ["v1spid--4-"], "uri": ["a0751"]}},
    "ADORNO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0752"]}},
    "ADORO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0753"]}},
    "ADVECTICIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0762"]}},
    "ADVECTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0764"]}},
    "ADVECTOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0765"]}},
    "ADVECTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8885"]}},
    "ADVEHO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0766"]}},
    "ADVEHOR": {"-": {"morpho": ["v1spid--3-"], "uri": []}},
    "ADVELO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0768"]}},
    "ADVENA": {"-": {"morpho": ["n-s---cn1-"], "uri": ["a0769"]}},
    "ADVENIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a0772"]}},
    "ADVENTICIVM": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "ADVENTICIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0773"]}},
    "ADVENTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0774"]}},
    "ADVENTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8886"]}},
    "ADVERRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0781"]}}, "ADVERSARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a0964"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a0749"]}
    }, "ADVERSO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0792"]}},
    "ADVERSOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a0793"]}},
    "ADVERSVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2245"]}}, "ADVERSVS": {
        "1": {"morpho": ["aps---mn1-"], "uri": ["19548"]}, "2": {"morpho": ["rp--------"], "uri": ["a0795"]},
        "-": {"morpho": ["p---------"], "uri": []}, "3": {"morpho": ["p---------"], "uri": []}
    }, "ADVERTO": {"-": {"morpho": ["rp--------"], "uri": []}},
    "ADVESPERASCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["97540"]}},
    "ADVIGILO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0799"]}},
    "ADVIVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0800"]}},
    "ADVLATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0802"]}},
    "ADVLATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0803"]}},
    "ADVLATORIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0804"]}},
    "ADVLESCENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a9973"]}},
    "ADVLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0810"]}},
    "ADVLOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["132213"]}},
    "ADVLTER": {"1": {"morpho": ["n-s---mn2r"], "uri": ["a0812"]}, "2": {"morpho": ["aps---mn1r"], "uri": ["a0812"]}},
    "ADVLTERA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0812"]}},
    "ADVLTERINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0816"]}},
    "ADVLTERIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0819"]}},
    "ADVLTERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0820"]}},
    "ADVLTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["37013"]}},
    "ADVMBRATIM": {"-": {"morpho": ["rp--------"], "uri": ["a0821"]}},
    "ADVMBRATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a9821"]}},
    "ADVMBRATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19526"]}},
    "ADVMBRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0822"]}},
    "ADVNCVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0826"]}},
    "ADVOCATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0828"]}},
    "ADVOCATVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0794"]}},
    "ADVOCO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0829"]}},
    "ADVOLO": {"2": {"morpho": ["v1spia--1-"], "uri": ["a0832"]}},
    "ADVOLVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0833"]}},
    "ADVRGEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0835"]}},
    "ADVRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0836"]}},
    "ADVSTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19532"]}},
    "ADYTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0839"]}},
    "AEBVTIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": ["97543"]}},
    "AEDES": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a0842"]}},
    "AEDICVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0843"]}},
    "AEDIFICATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0845"]}},
    "AEDIFICATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0847"]}},
    "AEDIFICIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0851"]}},
    "AEDIFICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0852"]}}, "AEDILICIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a2143"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a2095"]}
    }, "AEDILIS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["a0855"]}},
    "AEDILITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0856"]}},
    "AEDITVENS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["a0859"]}},
    "AEDITVMVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0861"]}},
    "AEDITVVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a0863"]}},
    "AEDON": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0864"]}},
    "AEGAEVS": {"-": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AEGER": {"1": {"morpho": ["n-s----n2-"], "uri": []}, "2": {"morpho": ["aps---mn1r"], "uri": ["a0866"]}},
    "AEGINIVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "AEGIS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0870"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "AEGRE": {"-": {"morpho": ["rp--------"], "uri": ["19565"]}},
    "AEGREO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a0880"]}},
    "AEGRESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a0880"]}},
    "AEGRIMONIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0881"]}},
    "AEGRITOMARVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AEGRITVDO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0885"]}},
    "AEGROR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0886"]}},
    "AEGROTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0888"]}},
    "AEGROTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0890"]}},
    "AEGROTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0891"]}},
    "AEGRVM": {"C": {"morpho": ["n-s----n2-"], "uri": []}, "-": {"morpho": ["n-s----n2-"], "uri": []}},
    "AEGYPTIVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AEGYPTVS": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "AELINOS": {"-": {"morpho": ["n-s---mn2g"], "uri": ["97548"]}},
    "AELVRVS": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "AEMILIANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AEMILIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": ["97549"]}},
    "AEMVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0901"]}},
    "AEMVLATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0897"]}},
    "AEMVLATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0898"]}},
    "AEMVLATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8887"]}},
    "AEMVLOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a0900"]}},
    "AEMVLVS": {"1": {"morpho": ["n-s----n2-"], "uri": []}, "2": {"morpho": ["aps---mn1-"], "uri": ["a0901"]}},
    "AENEAS": {"N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "AENEATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0902"]}},
    "AENEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2271"]}},
    "AENEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2286"]}},
    "AENIGMA": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a0905"]}},
    "AENIPES": {"-": {"morpho": ["aps---an3-"], "uri": ["a0908"]}},
    "AENVM": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "AENVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["32574"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["37635"]}},
    "AEQVABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0912"]}},
    "AEQVABILITER": {"-": {"morpho": ["rp--------"], "uri": ["19582"]}},
    "AEQVAEVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0914"]}},
    "AEQVALIS": {"1": {"morpho": ["n-s----n3-"], "uri": []}, "2": {"morpho": ["aps---cn3i"], "uri": ["a0915"]}},
    "AEQVALITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0916"]}},
    "AEQVALITER": {"-": {"morpho": ["rp--------"], "uri": ["19585"]}},
    "AEQVANIMITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0920"]}},
    "AEQVE": {"-": {"morpho": ["rp--------"], "uri": ["19589"]}}, "AEQVI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "AEQVICOLI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "AEQVICVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AEQVILATATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0933"]}},
    "AEQVINOCTIALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0942"]}},
    "AEQVINOCTIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0943"]}},
    "AEQVIPARO": {"-": {"morpho": ["v1spia--1-"], "uri": ["56187"]}},
    "AEQVIPOLLENS": {"-": {"morpho": ["aps---an3i"], "uri": ["a0953"]}},
    "AEQVIPONDIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0954"]}},
    "AEQVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0957"]}},
    "AEQVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a0961"]}},
    "AEQVOR": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a0962"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "AEQVOREVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0963"]}},
    "AEQVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0868"]}}, "AEQVVS": {
        "-": {"morpho": ["aps---mn1-"], "uri": ["a0867"]}, "N": {"morpho": ["aps---mn1-"], "uri": ["a0867"]},
        "A": {"morpho": ["aps---mn1-"], "uri": ["a0867"]}
    }, "AER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0965"]}},
    "AERACEVS": {"-": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AERARIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0970"]}},
    "AERARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0970"]}},
    "AERARIVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a0970"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a0970"]}},
    "AERATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0972"]}},
    "AEREVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a0974"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a0974"]}},
    "AERIPES": {"-": {"morpho": ["aps---an3-"], "uri": ["a0979"]}},
    "AERIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0981"]}},
    "AERO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0986"]}},
    "AERVCA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0992"]}},
    "AERVGINOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0993"]}},
    "AERVGO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a0995"]}},
    "AERVMNA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0997"]}},
    "AERVMNABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a0998"]}},
    "AERVMNOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0999"]}},
    "AES": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a1004"]}},
    "AESCHRIO": {"N": {"morpho": ["n-s----n-g"], "uri": []}}, "AESCHYLVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AESCVLAPIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AESCVLETVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1009"]}},
    "AESCVLINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1012"]}},
    "AESCVLVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["a1013"]}},
    "AESERNINVS": {"C": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AESOPEVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AESTAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1017"]}},
    "AESTIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["a1018"]}},
    "AESTIMATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1020"]}},
    "AESTIMATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1021"]}},
    "AESTIMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1025"]}},
    "AESTIVA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["a1030"]}},
    "AESTIVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1030"]}},
    "AESTVARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1032"]}},
    "AESTVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1034"]}},
    "AESTVOSE": {"-": {"morpho": ["rp--------"], "uri": ["57061"]}},
    "AESTVOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1035"]}},
    "AESTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a1036"]}},
    "AETAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1037"]}},
    "AETERNITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1041"]}},
    "AETERNO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a1042"]}, "2": {"morpho": ["rp--------"], "uri": ["19627"]}},
    "AETERNVM": {"1": {"morpho": ["n-s----n2-"], "uri": []}, "2": {"morpho": ["rp--------"], "uri": ["19628"]}},
    "AETERNVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1043"]}},
    "AETHER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1046"]}},
    "AETHERIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1047"]}},
    "AETHRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1049"]}, "N": {"morpho": ["n-s---fn1-"], "uri": ["a1049"]}},
    "AETIOLOGIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1050"]}}, "AETNA": {
        "1N": {"morpho": ["n-s----n1-"], "uri": []}, "2N": {"morpho": ["n-s----n1-"], "uri": []},
        "N": {"morpho": ["n-s---fn1-"], "uri": []}
    }, "AETNAEVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AETNENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "AETNENSIS": {"N": {"morpho": ["aps---cn3i"], "uri": []}, "A": {"morpho": ["aps---cn3i"], "uri": []}},
    "AETOLIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "AETOLICVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AETOLVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AEVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1057"]}},
    "AEVVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["28365"]}},
    "AFFABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1061"]}},
    "AFFABRE": {"-": {"morpho": ["rp--------"], "uri": ["55422"]}},
    "AFFATIM": {"-": {"morpho": ["rp--------"], "uri": ["a3705"]}},
    "AFFATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8888"]}},
    "AFFECTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1069"]}},
    "AFFECTATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1070"]}},
    "AFFECTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1072"]}},
    "AFFECTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1076"]}},
    "AFFECTVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a1055"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["37417"]}},
    "AFFERO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1079"]}},
    "AFFICIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a1081"]}},
    "AFFIGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1084"]}},
    "AFFINGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1087"]}},
    "AFFINIS": {"1": {"morpho": ["n-s----n3-"], "uri": []}, "2": {"morpho": ["aps---cn3i"], "uri": ["a1088"]}},
    "AFFINITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1089"]}},
    "AFFIRMATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1090"]}},
    "AFFIRMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1093"]}},
    "AFFLATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8889"]}},
    "AFFLICTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1100"]}},
    "AFFLICTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1101"]}},
    "AFFLICTVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a8891"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["36969"]}},
    "AFFLIGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1104"]}},
    "AFFLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1105"]}},
    "AFFLVENS": {"-": {"morpho": ["aps---an3-"], "uri": ["31487"]}},
    "AFFLVENTER": {"-": {"morpho": ["rp--------"], "uri": ["54931"]}},
    "AFFLVENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1106"]}},
    "AFFLVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1108"]}},
    "AFFOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a1112"]}},
    "AFFRICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1116"]}},
    "AFFVLGEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a1118"]}},
    "AFFVNDO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a1119"]}},
    "AFRICA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "AFRICANVS": {
        "C": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["aps---mn1-"], "uri": []},
        "A": {"morpho": ["aps---mn1-"], "uri": []}
    }, "AFRICVS": {"A": {"morpho": ["aps---mn1-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AGASO": {"-": {"morpho": ["n-s----n3-"], "uri": []}}, "AGATHINVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AGATHOCLES": {"N": {"morpho": ["n-s---mn3-"], "uri": []}}, "AGE": {"-": {"morpho": ["----------"], "uri": []}},
    "AGEDVM": {"-": {"morpho": ["----------"], "uri": []}},
    "AGELLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1135"]}},
    "AGENS": {"1": {"morpho": ["n-s---mn3i"], "uri": ["97562"]}, "2": {"morpho": ["aps---an3i"], "uri": ["102047"]}},
    "AGER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["a1137"]}, "N": {"morpho": ["n-s----n2-"], "uri": []}},
    "AGGER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1145"]}},
    "AGGERATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1147"]}},
    "AGGERO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a1150"]}, "2": {"morpho": ["v1spia--3-"], "uri": ["a1149"]}},
    "AGGESTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8893"]}},
    "AGGLOMERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1152"]}},
    "AGGLVTINO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1154"]}},
    "AGGRAVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1157"]}},
    "AGGREDIOR": {"-": {"morpho": ["v1spid--3i"], "uri": ["a1158"]}},
    "AGGREGO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a9967"]}},
    "AGILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1165"]}},
    "AGILITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1166"]}},
    "AGITABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1171"]}},
    "AGITATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1172"]}},
    "AGITATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1173"]}},
    "AGITATVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a1191"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["19653"]}},
    "AGITE": {"-": {"morpho": ["----------"], "uri": []}}, "AGITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1176"]}},
    "AGMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a1179"]}},
    "AGNA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1207"]}},
    "AGNASCOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a1183"]}},
    "AGNATVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1163"]}},
    "AGNINA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1194"]}},
    "AGNINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1194"]}},
    "AGNOSCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1204"]}},
    "AGNVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1207"]}},
    "AGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1208"]}}, "AGON": {"-": {"morpho": ["n-s----n-g"], "uri": []}},
    "AGRAMMATVS": {
        "-": {"morpho": ["aps---mn1-"], "uri": ["a1223"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["a1223"]}
    }, "AGRARII": {"-": {"morpho": ["n-p---mn2-"], "uri": ["a1226"]}},
    "AGRARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1226"]}}, "AGRESTIS": {
        "1": {"morpho": ["n-s---mn3i"], "uri": ["97566"]}, "2": {"morpho": ["aps---cn3i"], "uri": ["a1230"]},
        "N": {"morpho": ["n-s----n3-"], "uri": []}
    }, "AGRICOLA": {"-": {"morpho": ["n-s---mn1-"], "uri": ["a1233"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "AGRIGENTINI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "AGRIGENTINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AGRIGENTVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}}, "AGRIPPA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "AGRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["102102"]}},
    "AGYRINENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "AGYRINENSIS": {"N": {"morpho": ["aps---cn3i"], "uri": []}, "A": {"morpho": ["aps---cn3i"], "uri": []}},
    "AGYRIVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}}, "AH": {"-": {"morpho": ["----------"], "uri": []}},
    "AIAX": {"N": {"morpho": ["n-s----n3-"], "uri": []}}, "AIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a1250"]}},
    "ALA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1256"]}},
    "ALACER": {"-": {"morpho": ["aps---mn3-"], "uri": ["a1262"]}},
    "ALACRITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1264"]}},
    "ALACRITER": {"-": {"morpho": ["rp--------"], "uri": ["33446"]}},
    "ALAPA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1265"]}},
    "ALARII": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "ALARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1268"]}},
    "ALATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1272"]}}, "ALBA": {
        "1N": {"morpho": ["n-s---fn1-"], "uri": ["a1311"]}, "2N": {"morpho": ["n-s---fn1-"], "uri": ["a1311"]},
        "N": {"morpho": ["n-s---fn1-"], "uri": ["a1311"]}
    }, "ALBANI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ALBANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ALBARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1277"]}},
    "ALBATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1280"]}},
    "ALBEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a9041"]}},
    "ALBESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1282"]}},
    "ALBICERIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a2294"]}},
    "ALBICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1290"]}},
    "ALBIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1294"]}},
    "ALBINVS": {"N": {"morpho": ["n-s---mn2-"], "uri": ["a1296"]}},
    "ALBVLA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "ALBVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1306"]}},
    "ALBVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1311"]}},
    "ALBVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1311"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["a1311"]}},
    "ALCES": {"-": {"morpho": ["n-s---fn3i"], "uri": ["30109"]}},
    "ALCYON": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1319"]}},
    "ALCYONEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2253"]}},
    "ALEA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1324"]}},
    "ALEATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1327"]}},
    "ALEBRIA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["102142"]}},
    "ALEO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1331"]}}, "ALES": {
        "1": {"morpho": ["n-s---cn3-"], "uri": ["a1332"]}, "2": {"morpho": ["aps---an3-"], "uri": ["19691"]},
        "N": {"morpho": ["n-s---mn1g"], "uri": []}
    }, "ALESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1333"]}},
    "ALEXANDER": {"N": {"morpho": ["n-s----n-g"], "uri": []}},
    "ALEXANDRIA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "ALGA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1339"]}},
    "ALGEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a1337"]}}, "ALGIDVS": {
        "-": {"morpho": ["aps---mn1-"], "uri": ["a1340"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []},
        "A": {"morpho": ["aps---mn1-"], "uri": ["a1340"]}
    }, "ALGOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1343"]}},
    "ALGVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a1345"]}},
    "ALIA": {"-": {"morpho": ["rp--------"], "uri": ["44793"]}},
    "ALIAS": {"-": {"morpho": ["rp--------"], "uri": ["19698"]}},
    "ALIBI": {"-": {"morpho": ["rp--------"], "uri": ["a1348"]}},
    "ALICA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1350"]}},
    "ALICVBI": {"-": {"morpho": ["rp--------"], "uri": ["a1354"]}},
    "ALICVNDE": {"-": {"morpho": ["rp--------"], "uri": ["a1355"]}},
    "ALIENATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1356"]}},
    "ALIENIGENA": {"-": {"morpho": ["n-s----n1-"], "uri": []}},
    "ALIENIGENVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1358"]}},
    "ALIENO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1362"]}},
    "ALIENVM": {"C": {"morpho": ["n-s---nn2-"], "uri": ["a1363"]}, "-": {"morpho": ["n-s---nn2-"], "uri": ["a1363"]}},
    "ALIENVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a1363"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a1363"]},
        "N": {"morpho": ["n-s---mn2-"], "uri": ["a1363"]}
    }, "ALIGER": {"-": {"morpho": ["aps---mn1r"], "uri": ["a1365"]}},
    "ALIMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1367"]}},
    "ALIMONIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1370"]}},
    "ALIO": {"-": {"morpho": ["rp--------"], "uri": ["19709"]}},
    "ALIOQVI": {"-": {"morpho": ["rp--------"], "uri": ["a1371"]}},
    "ALIPES": {"-": {"morpho": ["aps---an3-"], "uri": ["a9956", "a1374"]}},
    "ALIPILVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1376"]}},
    "ALIPTES": {"-": {"morpho": ["n-s---mn1g"], "uri": ["97577"]}},
    "ALIQVA": {"-": {"morpho": ["rp--------"], "uri": ["45057"]}},
    "ALIQVAM": {"-": {"morpho": ["rp--------"], "uri": ["49199"]}},
    "ALIQVAMDIV": {"-": {"morpho": ["rp--------"], "uri": ["a9205"]}},
    "ALIQVAMMVLTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a9212"]}},
    "ALIQVANDO": {"-": {"morpho": ["rp--------"], "uri": ["a9206"]}},
    "ALIQVANTO": {"-": {"morpho": ["rp--------"], "uri": ["19713"]}},
    "ALIQVANTVLVM": {"1": {"morpho": ["----------"], "uri": []}, "2": {"morpho": ["rp--------"], "uri": ["54977"]}},
    "ALIQVANTVM": {"1": {"morpho": ["----------"], "uri": []}, "2": {"morpho": ["rp--------"], "uri": ["19714"]}},
    "ALIQVANTVS": {"-": {"morpho": ["----------"], "uri": []}},
    "ALIQVATENVS": {"-": {"morpho": ["rp--------"], "uri": ["a9210"]}},
    "ALIQVIS": {"-": {"morpho": ["----------"], "uri": []}},
    "ALIQVO": {"-": {"morpho": ["rp--------"], "uri": ["29384"]}},
    "ALIQVOT": {"-": {"morpho": ["----------"], "uri": []}},
    "ALIQVOTFARIAM": {"-": {"morpho": ["rp--------"], "uri": ["33662"]}},
    "ALIQVOTIENS": {"-": {"morpho": ["rp--------"], "uri": ["a9211"]}},
    "ALIS": {"-": {"morpho": ["----------"], "uri": []}}, "ALITER": {"-": {"morpho": ["rp--------"], "uri": ["19715"]}},
    "ALIVBI": {"-": {"morpho": ["rp--------"], "uri": ["a9214"]}},
    "ALIVNDE": {"-": {"morpho": ["rp--------"], "uri": ["a9213"]}},
    "ALIVS": {"-": {"morpho": ["----------"], "uri": []}},
    "ALIVSMODI": {"-": {"morpho": ["rp--------"], "uri": ["a8504"]}},
    "ALLABOR": {"2": {"morpho": ["v1spid--3-"], "uri": ["a1384"]}},
    "ALLABORO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1385"]}},
    "ALLAPSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a1401"]}},
    "ALLAVDABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1391"]}},
    "ALLEC": {"-": {"morpho": ["n-s---nn3-"], "uri": ["31101"]}},
    "ALLECTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1398"]}, "N": {"morpho": ["n-s----n1-"], "uri": []}},
    "ALLEGATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1402"]}},
    "ALLEGO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a1403"]}, "2": {"morpho": ["v1spia--3-"], "uri": ["a1404"]}},
    "ALLEVO": {
        "1": {"morpho": ["v1spia--1-"], "uri": ["a1415", "a1378"]},
        "2": {"morpho": ["v1spia--1-"], "uri": ["a1415", "a1378"]}
    }, "ALLEX": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a9937"]}},
    "ALLICEFACIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a1416"]}},
    "ALLICIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a1417"]}},
    "ALLIDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1418"]}},
    "ALLIGATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1421"]}},
    "ALLIGO": {"2": {"morpho": ["v1spia--1-"], "uri": ["a1424"]}},
    "ALLINO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1426"]}},
    "ALLIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["57803"]}},
    "ALLOBROGES": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ALLOBROX": {"N": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---an3-"], "uri": []}},
    "ALLOCVTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1429"]}},
    "ALLOQVIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1433"]}},
    "ALLOQVOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a1434"]}},
    "ALLVCEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a1437"]}},
    "ALLVDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1441"]}},
    "ALLVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1442"]}},
    "ALMVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1454"]}},
    "ALNEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1455"]}},
    "ALNVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["a1456"]}},
    "ALO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1457"]}},
    "ALOE": {"-": {"morpho": ["n-s---fn1g"], "uri": ["a1459"]}}, "ALPES": {"N": {"morpho": ["n-p---fn3i"], "uri": []}},
    "ALPHA": {"-": {"morpho": ["n-s---nn--"], "uri": ["a1468"]}},
    "ALSIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1477"]}},
    "ALTAR": {"-": {"morpho": ["n-s---nn3i"], "uri": ["97583"]}},
    "ALTE": {"-": {"morpho": ["rp--------"], "uri": ["19730"]}}, "ALTER": {"-": {"morpho": ["----------"], "uri": []}},
    "ALTERAS": {"-": {"morpho": ["rp--------"], "uri": ["44525"]}},
    "ALTERCATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1485"]}},
    "ALTERCOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a1486"]}},
    "ALTERNIS": {"-": {"morpho": ["rp--------"], "uri": ["19738"]}},
    "ALTERNO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1493"]}},
    "ALTERNVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1494"]}},
    "ALTERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1495"]}},
    "ALTERVTER": {"-": {"morpho": ["----------"], "uri": []}},
    "ALTILIS": {"1": {"morpho": ["n-s----n3-"], "uri": []}, "2": {"morpho": ["aps---cn3i"], "uri": ["a1513"]}},
    "ALTISONVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1519"]}},
    "ALTITONANS": {"-": {"morpho": ["aps---an3i"], "uri": ["a1498"]}},
    "ALTITVDO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1522"]}},
    "ALTIVOLANS": {"-": {"morpho": ["aps---an3i"], "uri": ["a1525"]}},
    "A": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "ALTRINSECVS": {"-": {"morpho": ["rp--------"], "uri": ["a1529"]}},
    "ALTRIX": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1530"]}},
    "ALTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a8929"]}},
    "ALTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19744"]}},
    "ALVARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1534"]}},
    "ALVCINATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1535"]}},
    "ALVEATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1541"]}},
    "ALVEOLATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1542"]}},
    "ALVEOLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1543"]}},
    "ALVEVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1544"]}},
    "ALVMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a1548"]}},
    "ALVMINOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1551"]}},
    "ALVMNA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1554"]}},
    "ALVMNVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a1554"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a1554"]}},
    "ALVNTINI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ALVNTINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ALVNTIVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "ALVTA": {"1": {"morpho": ["n-s----n1-"], "uri": []}, "2": {"morpho": ["n-s----n1-"], "uri": []}},
    "ALVVS": {"-": {"morpho": ["n-s---cn2-"], "uri": ["a1546"]}},
    "AM": {"-": {"morpho": ["p---------"], "uri": ["a1564"]}},
    "AMABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1565"]}},
    "AMABILITER": {"-": {"morpho": ["rp--------"], "uri": ["48420"]}},
    "AMANDO": {"2": {"morpho": ["v1spia--1-"], "uri": ["a1570"]}, "-": {"morpho": ["v1spia--1-"], "uri": ["a1570"]}},
    "AMANS": {"1": {"morpho": ["n-s---mn3i"], "uri": ["97589"]}, "2": {"morpho": ["aps---an3i"], "uri": ["19755"]}},
    "AMANTER": {"-": {"morpho": ["rp--------"], "uri": ["97590"]}},
    "AMARA": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "AMARACINVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1573"]}},
    "AMARACVS": {"-": {"morpho": ["n-s---cn2-"], "uri": ["a1574"]}},
    "AMARANTVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1575"]}},
    "AMARE": {"-": {"morpho": ["rp--------"], "uri": ["43810"]}},
    "AMARITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1580"]}},
    "AMARITIES": {"-": {"morpho": ["n-s---fn5-"], "uri": ["a1581"]}},
    "AMARITVDO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1584"]}},
    "AMAROR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1587"]}},
    "AMARVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1589"]}},
    "AMATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1596"]}},
    "AMATORIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1598"]}},
    "AMATORIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1598"]}},
    "AMBACTVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1603"]}},
    "AMBAGES": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1605"]}},
    "AMBARVALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1608"]}},
    "AMBEDO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a1612"]}},
    "AMBEO": {"1": {"morpho": ["v1spia--3-"], "uri": []}},
    "AMBIGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1622"]}},
    "AMBIGVE": {"-": {"morpho": ["rp--------"], "uri": ["19768"]}},
    "AMBIGVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1623"]}},
    "AMBIGVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1624"]}},
    "AMBIGVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1624"]}},
    "AMBIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a1626"]}},
    "AMBITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1627"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "AMBITIOSE": {"-": {"morpho": ["rp--------"], "uri": ["56671"]}},
    "AMBITIOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1628"]}},
    "AMBITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8896"]}},
    "AMBO": {"2": {"morpho": ["----------"], "uri": []}},
    "AMBROSIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2297"]}},
    "AMBROSIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2370"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["97593"]}},
    "AMBVBAIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1640"]}},
    "AMBVLATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1644"]}},
    "AMBVLATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1647"]}},
    "AMBVLATORIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1648"]}},
    "AMBVLATRIX": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1649"]}},
    "AMBVLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1650"]}},
    "AMBVRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1653"]}},
    "AMELLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1657"]}},
    "AMENS": {"-": {"morpho": ["aps---an3i"], "uri": ["a1660"]}},
    "AMENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1662"]}},
    "AMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["58292"]}},
    "AMERIOLA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "AMES": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1664"]}},
    "AMESTRATINI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "AMESTRATINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AMESTRATVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AMETHYSTINA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["a1666"]}},
    "AMETHYSTVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["a1668"]}},
    "AMICA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a0808"]}},
    "AMICE": {"-": {"morpho": ["rp--------"], "uri": ["40694"]}},
    "AMICIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a1679"]}},
    "AMICITIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2373"]}},
    "AMICITIES": {"-": {"morpho": ["n-s---fn5-"], "uri": ["a2383"]}},
    "AMICTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8898"]}},
    "AMICVLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1654"]}},
    "AMICVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1685"]}},
    "AMICVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a1687"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a0783"]}},
    "AMISSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1689"]}},
    "AMITA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1690"]}},
    "AMITTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1723"]}},
    "AMMONIACVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["34825"]}},
    "AMNIS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["a1704"]}},
    "AMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1705"]}},
    "AMOENITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1708"]}},
    "AMOENVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1710"]}},
    "AMOLIOR": {"-": {"morpho": ["v1spid--4-"], "uri": ["a1712"]}},
    "AMOMON": {"-": {"morpho": ["n-s---nn2g"], "uri": ["102364"]}},
    "AMOMVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1715"]}},
    "AMOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1716"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "AMOVEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a1724"]}},
    "AMPHERES": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "AMPHILOCHI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}}, "AMPHIO": {"-": {"morpho": ["n-s----n-g"], "uri": []}},
    "AMPHIPOLIS": {"N": {"morpho": ["n-s---fn3-"], "uri": []}},
    "AMPHIPROSTYLOS": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "AMPHITHALAMVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["102388"]}},
    "AMPHITHEATRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": []}},
    "AMPHORA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1757"]}},
    "AMPLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1761"]}},
    "AMPLE": {"-": {"morpho": ["rp--------"], "uri": ["36310"]}},
    "AMPLECTOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a1762"]}},
    "AMPLEXOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a1766"]}},
    "AMPLEXVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a1780"]}},
    "AMPLIFICATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1769"]}},
    "AMPLIFICATOR": {"-": {"morpho": ["n-s---fn3-"], "uri": ["43416"]}},
    "AMPLIFICE": {"-": {"morpho": ["rp--------"], "uri": ["19819"]}},
    "AMPLIFICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1772"]}},
    "AMPLIO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1774"]}},
    "AMPLITVDO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1775"]}},
    "AMPLIVS": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "AMPLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1778"]}},
    "AMPVLLOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a1786"]}},
    "AMPVTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a1792"]}},
    "AMVLIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AMVRCA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1797"]}},
    "AMVRCARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1798"]}},
    "AMVSOS": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "AMVSSIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1803"]}},
    "AMYGDALVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1806"]}},
    "AMYLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1813"]}},
    "AMYSTIS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1815"]}},
    "AN": {"-": {"morpho": ["rp--------"], "uri": ["45604"]}},
    "ANABATHRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1821"]}},
    "ANADEMA": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a1841"]}},
    "ANALECTA": {"-": {"morpho": ["n-s---mn1-"], "uri": ["a1859"]}},
    "ANALEMMA": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a1861"]}},
    "ANALEMPTRIS": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "ANALOGIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1864"]}},
    "ANAPHORICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a8879"]}},
    "ANATONVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1905"]}},
    "ANCEPS": {"1": {"morpho": ["n-s----n3-"], "uri": []}, "2": {"morpho": ["aps---an3-"], "uri": ["a1912"]}},
    "ANCHISES": {"N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "ANCILE": {"-": {"morpho": ["n-s---nn3i"], "uri": ["a1915"]}},
    "ANCILLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1916"]}},
    "ANCILLARIOLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1917"]}},
    "ANCILLARIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1918"]}},
    "ANCILLVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1920"]}},
    "ANCISVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1923"]}},
    "ANCON": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1926"]}, "N": {"morpho": ["n-s----n-g"], "uri": []}},
    "ANCORA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1927"]}},
    "ANCVS": {"N": {"morpho": ["n-s---mn2-"], "uri": ["30758"]}}, "ANDRO": {"N": {"morpho": ["n-s----n-g"], "uri": []}},
    "ANDRON": {"-": {"morpho": ["n-s----n-g"], "uri": []}, "N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ANDRONITIS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a1947"]}},
    "ANDROSTHENES": {"N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "ANELLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1955"]}},
    "ANETHVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1961"]}},
    "ANFRACTVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a1968"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a1968"]}},
    "ANGELLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1977"]}},
    "ANGELVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a1979"]}},
    "ANGINA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1980"]}},
    "ANGIPORTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1981"]}},
    "ANGIPORTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a1981"]}},
    "ANGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1983"]}},
    "ANGOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1985"]}},
    "ANGVBATAE": {"-": {"morpho": ["n-s----n1-"], "uri": []}},
    "ANGVICOMVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1987"]}},
    "ANGVIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["a1989"]}},
    "ANGVILLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1991"]}},
    "ANGVIMANVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["44511"]}},
    "ANGVINEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1993"]}},
    "ANGVINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1994"]}},
    "ANGVIS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["a1996"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "ANGVLARIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a1997"]}},
    "ANGVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2004"]}},
    "ANGVSTE": {"-": {"morpho": ["rp--------"], "uri": ["39562"]}},
    "ANGVSTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2006"]}},
    "ANGVSTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2009"]}},
    "ANGVSTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2010"]}},
    "ANGVSTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2010"]}},
    "ANHELITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a2014"]}},
    "ANHELO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2015"]}},
    "ANHELVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2017"]}},
    "ANIATROLOGETVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2019"]}},
    "ANICVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2022"]}},
    "ANILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a2024"]}},
    "ANILITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2025"]}},
    "ANILITER": {"-": {"morpho": ["rp--------"], "uri": ["46414"]}},
    "ANIMA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2028"]}},
    "ANIMADVERSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2030"]}},
    "ANIMADVERTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a1944"]}},
    "ANIMAL": {"-": {"morpho": ["n-s---nn3i"], "uri": ["a2034"]}},
    "ANIMALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a2034"]}},
    "ANIMANS": {"1": {"morpho": ["n-s---mn3i"], "uri": ["97619"]}, "2": {"morpho": ["aps---an3-"], "uri": ["42505"]}},
    "ANIMATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2036"]}},
    "ANIMATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19882"]}},
    "ANIMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2041"]}},
    "ANIMOSE": {"-": {"morpho": ["rp--------"], "uri": ["19885"]}},
    "ANIMOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2043"]}},
    "ANIMVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2046"]}}, "ANIO": {"N": {"morpho": ["n-s----n--"], "uri": []}},
    "ANISOCYCLI": {"-": {"morpho": ["n-s----n-g"], "uri": []}},
    "ANISVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["51473"]}},
    "ANNAEA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}}, "ANNAEVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ANNALIS": {
        "1": {"morpho": ["n-s---mn3i"], "uri": ["102569"]}, "2": {"morpho": ["aps---cn3i"], "uri": ["a2050"]},
        "N": {"morpho": ["n-s----n3-"], "uri": []}
    }, "ANNATO": {"-": {"morpho": ["v1spia--1-"], "uri": ["47472"]}},
    "ANNE": {"-": {"morpho": ["rp--------"], "uri": ["97622"]}},
    "ANNECTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["33632"]}},
    "ANNIA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}}, "ANNIANVS": {"A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ANNICVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2054"]}},
    "ANNITOR": {"2": {"morpho": ["v1spid--3-"], "uri": ["132243"]}},
    "ANNIVERSARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2061"]}},
    "ANNIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ANNO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a7061"]}, "2": {"morpho": ["v1spia--1-"], "uri": ["a7061"]}},
    "ANNONA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2063"]}},
    "ANNOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2067"]}},
    "ANNOTINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2068"]}},
    "ANNOTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["29083"]}},
    "ANNVMERO": {"1": {"morpho": ["v1spia--1-"], "uri": ["43132"]}},
    "ANNVNTIO": {"-": {"morpho": ["v1spia--1-"], "uri": ["44727"]}},
    "ANNVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["29188"]}},
    "ANNVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2077"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["a2077"]}},
    "ANNVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a0854"]}},
    "ANNVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2080"]}},
    "ANQVIRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2090"]}},
    "ANSA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2092"]}},
    "ANSER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a0919"]}, "N": {"morpho": ["n-s----n2-"], "uri": []}},
    "ANTAE": {"-": {"morpho": ["n-p---fn1-"], "uri": ["a2102"]}},
    "ANTARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2677"]}},
    "ANTE": {"1": {"morpho": ["rp--------"], "uri": ["19899"]}, "2": {"morpho": ["p---------"], "uri": ["a2109"]}},
    "ANTEA": {"-": {"morpho": ["rp--------"], "uri": ["a2110"]}},
    "ANTEACTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a7110"]}},
    "ANTEAGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["48884"]}},
    "ANTEAQVAM": {"-": {"morpho": ["----------"], "uri": []}},
    "ANTECAPIO": {"2": {"morpho": ["v1spia--3i"], "uri": ["a2117"]}},
    "ANTECEDENS": {"-": {"morpho": ["aps---an3i"], "uri": ["19903"]}},
    "ANTECEDENTIA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["102622"]}},
    "ANTECEDO": {"1": {"morpho": ["v1spia--3-"], "uri": ["a2118"]}},
    "ANTECELLO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a7118"]}},
    "ANTECESSOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2122"]}},
    "ANTECHON": {"-": {"morpho": ["----------"], "uri": []}},
    "ANTECVRRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2124"]}},
    "ANTECVRSOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2125"]}},
    "ANTEEO": {"1": {"morpho": ["v1spia--4-"], "uri": ["a1533"]}},
    "ANTEFERO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2128"]}},
    "ANTEFIXVM": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "ANTEFIXVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2129"]}},
    "ANTEHABEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a2135"]}},
    "ANTEHAC": {"-": {"morpho": ["rp--------"], "uri": ["a2136"]}},
    "ANTELVCANVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a0952"]}},
    "ANTEMERIDIANVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2148"]}},
    "ANTEMNATES": {"N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "ANTENNA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["43966"]}},
    "ANTENOR": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ANTEPAGMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2158"]}},
    "ANTEPONO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2165"]}},
    "ANTEQVAM": {"-": {"morpho": ["----------"], "uri": []}, "1": {"morpho": ["rp--------"], "uri": ["32023"]}},
    "ANTERIS": {"-": {"morpho": ["n-s----n-g"], "uri": []}},
    "ANTES": {"-": {"morpho": ["n-s---mn3i"], "uri": ["a2172"]}}, "ANTESIGNANVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a2175"]}, "2": {"morpho": ["aps---mn1-"], "uri": []},
        "A": {"morpho": ["aps---mn1-"], "uri": []}
    }, "ANTESTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2176"]}},
    "ANTESTOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a2177"]}},
    "ANTEVENIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a2180"]}},
    "ANTEVERTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2183"]}},
    "ANTHROPINVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ANTIATES": {"N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "ANTIBASIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a2216"]}},
    "ANTIBOREVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2217"]}},
    "ANTICIPO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2225"]}},
    "ANTIDOTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2229"]}},
    "ANTIGONVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ANTIOCHENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "ANTIOCHIA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "ANTIOCHVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ANTIPODES": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2244"]}},
    "ANTIQVARIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2247"]}}, "ANTIQVARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a2247"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a2247"]}
    }, "ANTIQVE": {"-": {"morpho": ["rp--------"], "uri": ["45736"]}},
    "ANTIQVI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["a1063"]}},
    "ANTIQVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2250"]}},
    "ANTIQVITVS": {"-": {"morpho": ["rp--------"], "uri": ["a2251"]}},
    "ANTIQVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1228"]}},
    "ANTIQVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1056"]}},
    "ANTISTES": {"-": {"morpho": ["n-s---cn3-"], "uri": ["a2264"]}},
    "ANTISTITA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2265"]}},
    "ANTISTIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ANTITHETON": {"-": {"morpho": ["n-s---nn2g"], "uri": ["a2273"]}},
    "ANTIVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}}, "ANTONIANI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ANTONIANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ANTONIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": ["97633"]}},
    "ANTRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2282"]}},
    "ANVLARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2979"]}},
    "ANVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2290"]}},
    "ANVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a2292"]}, "2": {"morpho": ["n-s---fn4-"], "uri": ["a2293"]}},
    "ANXIE": {"-": {"morpho": ["rp--------"], "uri": ["19940"]}},
    "ANXIETAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2296"]}},
    "ANXIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2983"]}},
    "AONIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "APELLES": {"N": {"morpho": ["n-s---mn3i"], "uri": []}},
    "APER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["a2320"]}, "N": {"morpho": ["n-s----n2-"], "uri": []}},
    "APERIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a2322"]}},
    "APERTE": {"-": {"morpho": ["rp--------"], "uri": ["19947"]}},
    "APERTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2349"]}},
    "APERTVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2330"]}},
    "APERTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["19951"]}},
    "APEX": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2331"]}},
    "APIACVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2350"]}},
    "APICIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2359"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["97637"]}},
    "APIOLAE": {"N": {"morpho": ["n-p---fn1-"], "uri": []}},
    "APIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a2369"]}, "N": {"morpho": ["n-s---mn3g"], "uri": []}},
    "APISCOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a1899"]}},
    "APIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2371"]}},
    "APLVSTRE": {"-": {"morpho": ["n-s---nn3i"], "uri": ["a2375"]}},
    "APOLLO": {"N": {"morpho": ["n-s---mn3-"], "uri": []}}, "APOLLODORVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "APOLLONIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "APOLLONIATAE": {"N": {"morpho": ["n-p---fn1g"], "uri": []}},
    "APOLLONIENSIS": {"N": {"morpho": ["aps---cn3i"], "uri": []}, "A": {"morpho": ["aps---cn3i"], "uri": []}},
    "APOLLONIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "APOLOGO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2412"]}},
    "APOPHYSIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a2419"]}},
    "APOTHECA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2451"]}},
    "APOTHESIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a2455"]}},
    "APPAGINECVLI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["102861"]}},
    "APPARATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2463"]}},
    "APPARATVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a2502"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["19985"]}},
    "APPAREO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a2468"]}},
    "APPARIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a2469"]}},
    "APPARITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2470"]}},
    "APPARITOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2471"]}},
    "APPARO": {"2": {"morpho": ["v1spia--1-"], "uri": ["a2473"]}},
    "APPELLATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a7476"]}},
    "APPELLATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2478"]}},
    "APPELLITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2480"]}},
    "APPELLO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a2481"]}, "2": {"morpho": ["v1spia--3-"], "uri": ["a2482"]}},
    "APPENDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a2485"]}},
    "APPENDIX": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2487"]}},
    "APPENDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2488"]}},
    "APPERIOR": {"-": {"morpho": ["v1spid--4-"], "uri": []}},
    "APPETENS": {"-": {"morpho": ["aps---an3i"], "uri": ["97648"]}},
    "APPETITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2495"]}},
    "APPETITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8901"]}},
    "APPETO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2498"]}},
    "APPIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": ["97651"]}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "APPLAVDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2504"]}},
    "APPLICATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2507"]}},
    "APPLICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2508"]}},
    "APPLORO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2510"]}},
    "APPONO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2513"]}},
    "APPORTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2515"]}},
    "APPORTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2516"]}},
    "APPOSITVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20007"]}},
    "APPRECOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a2522"]}},
    "APPREHENDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2523"]}},
    "APPREHENSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2525"]}},
    "APPRIME": {"-": {"morpho": ["rp--------"], "uri": ["42618"]}},
    "APPRIMO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2528"]}},
    "APPRIMVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2529"]}},
    "APPROBATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2532"]}},
    "APPROBO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2535"]}},
    "APPROPERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2540"]}},
    "APPROPINQVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2542"]}},
    "APPVGNO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2547"]}},
    "APPVLSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8903"]}},
    "APRICVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2555"]}},
    "APRICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2555"]}},
    "APRONIANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "APRONIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "APTE": {"-": {"morpho": ["rp--------"], "uri": ["49254"]}},
    "APTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2575"]}},
    "APTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20021"]}},
    "APVD": {"-": {"morpho": ["p---------"], "uri": ["a2579"]}},
    "APYRENVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1287"]}},
    "AQVA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2585"]}},
    "AQVALICVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2589"]}},
    "AQVALIS": {"2": {"morpho": ["aps---cn3i"], "uri": ["a2590"]}, "1": {"morpho": ["aps---cn3i"], "uri": ["a2590"]}},
    "AQVARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1316"]}}, "AQVARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a1309"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a1289"]},
        "N": {"morpho": ["n-s---mn2-"], "uri": ["a1309"]}
    }, "AQVATICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2594"]}},
    "AQVATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2596"]}},
    "AQVATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2597"]}},
    "AQVATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2598"]}},
    "AQVIFOLIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1326"]}},
    "AQVILA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1531"]}, "N": {"morpho": ["n-s---fn1-"], "uri": ["a1531"]}},
    "AQVILIFER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["a2609"]}},
    "AQVILIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": ["a2611"]}},
    "AQVILO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2612"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "AQVOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a2622"]}},
    "AQVOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2624"]}},
    "AQVVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["43253"]}},
    "ARA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2625"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "ARACHNE": {"-": {"morpho": ["n-s---fn1g"], "uri": ["a2630"]}, "N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "ARAEOSTYLOS": {"-": {"morpho": ["aps---mn1g"], "uri": ["a2633"]}},
    "ARANEA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2639"]}},
    "ARANEOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2638"]}},
    "ARANEVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a2639"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a2639"]}},
    "ARATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2641"]}},
    "ARATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2644"]}},
    "ARATRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2647"]}},
    "ARBITER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["a2649"]}},
    "ARBITRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2649"]}},
    "ARBITRATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8904"]}},
    "ARBITRIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2657"]}},
    "ARBITROR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a2658"]}},
    "ARBOR": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2659"]}},
    "ARBORARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2661"]}},
    "ARBOREVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2664"]}},
    "ARBVSCVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2665"]}, "N": {"morpho": ["n-s---fn1-"], "uri": ["a2665"]}},
    "ARBVSTIVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2668"]}},
    "ARBVSTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2671"]}},
    "ARBVTEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2672"]}},
    "ARBVTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2673"]}},
    "ARBVTVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["48234"]}},
    "ARCA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2674"]}},
    "ARCADES": {"N": {"morpho": ["n-s---mn1g"], "uri": []}}, "ARCADIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "ARCADICVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ARCANO": {"-": {"morpho": ["rp--------"], "uri": ["41000"]}},
    "ARCANVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1321"]}},
    "ARCANVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2675"]}}, "ARCATVS": {
        "1": {"morpho": ["n-s----n2-"], "uri": []}, "2": {"morpho": ["aps---mn1-"], "uri": []},
        "A": {"morpho": ["aps---mn1-"], "uri": []}
    }, "ARCEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a2683"]}}, "ARCESSITVS": {
        "1": {"morpho": ["n-s---mn4-"], "uri": ["a2676"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["97663"]}
    }, "ARCESSO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2688"]}},
    "ARCHAGATHVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ARCHETYPVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2696"]}},
    "ARCHIMAGIRVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2712"]}},
    "ARCHIMEDES": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ARCHIPIRATA": {"-": {"morpho": ["n-s---mn1-"], "uri": ["a2717"]}},
    "ARCHITECTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2724"]}},
    "ARCHITECTONICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2726"]}}, "ARCHITECTOR": {
        "1": {"morpho": ["n-s---mn3-"], "uri": ["a2728"]}, "2": {"morpho": ["v1spid--1-"], "uri": ["132255"]},
        "-": {"morpho": ["v1spid--1-"], "uri": ["132255"]}
    }, "ARCHITECTVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2730"]}},
    "ARCHITECTVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2733"]}},
    "ARCHONIDAS": {"N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "ARCITENENS": {"1": {"morpho": ["n-s----n3-"], "uri": []}, "2": {"morpho": ["aps---an3i"], "uri": ["a2905"]}},
    "ARCTOS": {"-": {"morpho": ["n-s---fn2g"], "uri": ["a8930"]}},
    "ARCTOVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20063"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["20063"]}},
    "ARCVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2759"]}},
    "ARCVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a2764"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ARDEA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2766"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "ARDENS": {"-": {"morpho": ["aps---an3i"], "uri": ["20068"]}},
    "ARDENTER": {"-": {"morpho": ["rp--------"], "uri": ["20069"]}},
    "ARDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a2767"]}},
    "ARDESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2270"]}},
    "ARDOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2771"]}},
    "ARDVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2774"]}},
    "ARDVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2774"]}},
    "AREA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2775"]}},
    "AREFACIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a3715"]}},
    "ARENA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["47288"]}},
    "ARENARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["34368"]}},
    "ARENATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["46929"]}},
    "ARENATVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["54448"]}},
    "ARENATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["37199"]}},
    "ARENOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["56634"]}},
    "AREO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a2778"]}},
    "ARESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2778"]}},
    "ARETALOGVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a7779"]}},
    "ARETHVSA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "AREVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": ["97670"]}},
    "ARGEI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ARGENTARIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2788"]}}, "ARGENTARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a2788"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a2788"]}
    },
    "ARGENTEVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a2390"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a2578"]}},
    "ARGENTIFODINA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3713"]}},
    "ARGENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2795"]}},
    "ARGILETVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "ARGILLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2797"]}},
    "ARGVMENTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2802"]}},
    "ARGVMENTOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a2806"]}},
    "ARGVMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2808"]}},
    "ARGVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2809"]}},
    "ARGVTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a2810"]}},
    "ARGVTE": {"-": {"morpho": ["rp--------"], "uri": ["20089"]}},
    "ARGVTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2813"]}},
    "ARGVTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2816"]}},
    "ARGVTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20092"]}},
    "ARICIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "ARICINI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ARICINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ARIDA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a1568"]}},
    "ARIDVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2826"]}},
    "ARIDVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1594"]}},
    "ARIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1540"]}},
    "ARIES": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2829"]}, "N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "ARIETARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2830"]}},
    "ARIETO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2834"]}},
    "ARIMINENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "ARIMINVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "ARISTA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2845"]}},
    "ARISTAEVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}}, "ARISTEVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ARISTIDES": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ARISTODAMVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ARISTODEMVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ARISTOTELES": {"N": {"morpho": ["n-p---mn3-"], "uri": []}},
    "ARISTVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ARITHMETICA": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "ARMA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["a2855"]}},
    "ARMAMAXA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2856"]}},
    "ARMAMENTARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1618"]}},
    "ARMAMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2857"]}},
    "ARMARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2860"]}},
    "ARMATI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["97677"]}},
    "ARMATVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2864"]}},
    "ARMATVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a2909"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["45112"]}},
    "ARMENTARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a2866"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a2866"]}
    }, "ARMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2870"]}},
    "ARMIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["a2876"]}},
    "ARMIGER": {"1": {"morpho": ["n-s---mn2r"], "uri": ["a2877"]}, "2": {"morpho": ["aps---mn1r"], "uri": ["a2877"]}},
    "ARMILLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2879"]}},
    "ARMIPOTENS": {"-": {"morpho": ["aps---an3i"], "uri": ["a2885"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "ARMISONVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2886"]}},
    "ARMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2888"]}},
    "ARMVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2891"]}},
    "ARO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2895"]}},
    "ARRECTARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2911"]}},
    "ARRECTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["34104"]}},
    "ARREPO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2912"]}},
    "ARRIDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a2918"]}},
    "ARRIGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2919"]}},
    "ARRIPIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a2921"]}},
    "ARRISOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2923"]}},
    "ARRIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}}, "ARRODO": {"-": {"morpho": ["v1spia--3-"], "uri": []}},
    "ARROGANS": {"A": {"morpho": ["aps---an3-"], "uri": ["32325"]}, "-": {"morpho": ["aps---an3-"], "uri": ["32325"]}},
    "ARROGANTER": {"-": {"morpho": ["rp--------"], "uri": ["57933"]}},
    "ARROGANTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2925"]}},
    "ARROGO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2928"]}},
    "ARROSOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a2930"]}},
    "ARRVNS": {"N": {"morpho": ["n-s----n3-"], "uri": []}},
    "ARS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a2934"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "ARSENICON": {"-": {"morpho": ["n-s---nn2-"], "uri": ["40294"]}},
    "ARSIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "ARSINEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2937"]}},
    "ARSINOE": {"N": {"morpho": ["n-s---fn1g"], "uri": []}},
    "ARTE": {"-": {"morpho": ["rp--------"], "uri": ["43992"]}},
    "ARTEMIDORVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ARTEMO": {"-": {"morpho": ["n-s----n3-"], "uri": []}, "N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ARTERIA": {"-": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ARTERIACE": {"-": {"morpho": ["n-s---fn1g"], "uri": ["a2944"]}},
    "ARTICVLARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2952"]}},
    "ARTICVLATIM": {"-": {"morpho": ["rp--------"], "uri": ["a2953"]}},
    "ARTICVLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2956"]}},
    "ARTICVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a2958"]}},
    "ARTIFEX": {"-": {"morpho": ["n-s---cn3-"], "uri": ["a2959"]}},
    "ARTIFICIOSE": {"-": {"morpho": ["rp--------"], "uri": ["52957"]}},
    "ARTIFICIOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2963"]}},
    "ARTIFICIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a2964"]}},
    "ARTIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a2967"]}},
    "ARTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a2971"]}},
    "ARTOCREAS": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a2973"]}},
    "ARTOPTA": {"-": {"morpho": ["n-s---mn1-"], "uri": ["a2975"]}},
    "ARTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a1973"]}},
    "ARTVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a1967"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a1891"]}},
    "ARVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a2992"]}},
    "ARVNDINETVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["97689"]}},
    "ARVNDINOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["97690"]}},
    "ARVNDO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["45524"]}},
    "ARVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3006"]}},
    "ARX": {"-": {"morpho": ["n-s---fn3-"], "uri": ["20128"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "AS": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3009"]}}, "ASCANIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ASCENDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3021"]}},
    "ASCENSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3023"]}},
    "ASCENSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a2969"]}},
    "ASCIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3026"]}},
    "ASCIO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a2285"]}, "2": {"morpho": ["v1spia--1-"], "uri": ["a2285"]}},
    "ASCISCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3028"]}},
    "ASCITVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["55283"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["33308"]}},
    "ASCRIBO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3034"]}},
    "ASCRIPTICIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3035"]}},
    "ASCRIPTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3036"]}},
    "ASELLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3045"]}},
    "ASELLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3045"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["a3045"]}},
    "ASIA": {"N": {"morpho": ["n-s---fn1-"], "uri": ["a3049"]}},
    "ASIATICVS": {"N": {"morpho": ["aps---mn1-"], "uri": ["97696"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["97696"]}},
    "ASILVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3051"]}},
    "ASINARIVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a3055"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a3055"]}},
    "ASININVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3057"]}},
    "ASINIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ASINVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3052"]}},
    "ASOTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3063"]}},
    "ASPARAGVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3065"]}},
    "ASPECTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3068"]}},
    "ASPECTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a3106"]}},
    "ASPENDIVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ASPENDVS": {"N": {"morpho": ["n-s----n-g"], "uri": []}},
    "ASPER": {"-": {"morpho": ["aps---mn1r"], "uri": ["a3072"]}, "N": {"morpho": ["n-s----n2-"], "uri": []}},
    "ASPERA": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "ASPERE": {"-": {"morpho": ["rp--------"], "uri": ["20152"]}},
    "ASPERGO": {"1": {"morpho": ["n-s---fn3-"], "uri": ["a3076"]}, "2": {"morpho": ["v1spia--3-"], "uri": ["a3075"]}},
    "ASPERITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3077"]}},
    "ASPERNATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3081"]}},
    "ASPERNOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a3083"]}},
    "ASPERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3084"]}},
    "ASPERVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["56941"]}},
    "ASPHALTITIS": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "ASPICIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a3091"]}},
    "ASPIRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3096"]}},
    "ASPIS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3097"]}},
    "ASPLENOS": {"-": {"morpho": ["n-s---mn2g"], "uri": ["a3099"]}},
    "ASPORTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3100"]}},
    "ASPORTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3101"]}},
    "ASSARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3113"]}},
    "ASSECLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["34815"]}},
    "ASSECTATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3115"]}},
    "ASSECTOR": {"2": {"morpho": ["v1spid--1-"], "uri": ["a3116"]}},
    "ASSENSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3125"]}},
    "ASSENSOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3126"]}},
    "ASSENSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a3151"]}},
    "ASSENTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3128"]}},
    "ASSENTATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3130"]}},
    "ASSENTIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a3134"]}},
    "ASSENTIOR": {"-": {"morpho": ["v1spid--4-"], "uri": ["132261"]}},
    "ASSENTOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a3135"]}},
    "ASSEQVOR": {"-": {"morpho": ["v1spid--3-"], "uri": ["a3137"]}},
    "ASSER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3138"]}},
    "ASSERCVLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["36947"]}},
    "ASSERO": {"2": {"morpho": ["v1spia--3-"], "uri": ["a3141"]}, "3": {"morpho": ["v1spia--3-"], "uri": ["a3141"]}},
    "ASSERTOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a6142"]}},
    "ASSERVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3147"]}},
    "ASSESSOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3149"]}},
    "ASSEVERATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3153"]}},
    "ASSEVERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3154"]}},
    "ASSICCO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3157"]}},
    "ASSIDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3159"]}},
    "ASSIDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3160"]}},
    "ASSIDVE": {"-": {"morpho": ["rp--------"], "uri": ["40859"]}},
    "ASSIDVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3161"]}},
    "ASSIDVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3163"]}},
    "ASSIGNO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3169"]}},
    "ASSILIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a3171"]}},
    "ASSIMILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a3173"]}},
    "ASSIMVLATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3172"]}},
    "ASSIMVLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3175"]}},
    "ASSISTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a2418"]}},
    "ASSOLEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3186"]}},
    "ASSONO": {"1": {"morpho": ["v1spia--1-"], "uri": ["a3190"]}},
    "ASSORINI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "ASSORINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ASSORVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "ASSVEFACIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a3193"]}},
    "ASSVESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3194"]}},
    "ASSVETVDO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3195"]}},
    "ASSVETVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["35174"]}},
    "ASSVGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3196"]}},
    "ASSVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3197"]}},
    "ASSVLTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3201"]}},
    "ASSVLTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a8907"]}},
    "ASSVM": {"1": {"morpho": ["v1spia--1-"], "uri": ["36160"]}, "2": {"morpho": ["n-s---nn2-"], "uri": ["a3108"]}},
    "ASSVMO": {"-": {"morpho": ["v1spia--3-"], "uri": []}},
    "ASSVRGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3210"]}},
    "ASSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3108"]}},
    "ASTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3239"]}},
    "ASTRAGALVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3245"]}},
    "ASTREPO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3249"]}},
    "ASTRICTE": {"-": {"morpho": ["rp--------"], "uri": ["56905"]}},
    "ASTRICTVS": {"A": {"morpho": ["aps---mn1-"], "uri": ["43555"]}, "-": {"morpho": ["aps---mn1-"], "uri": ["43555"]}},
    "ASTRINGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3259"]}},
    "ASTROLOGIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3265"]}},
    "ASTROLOGVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3267"]}},
    "ASTRONOMIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3268"]}},
    "ASTRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3275"]}},
    "ASTRVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3276"]}},
    "ASTV": {"-": {"morpho": ["n-s---nn--"], "uri": ["a3277"]}},
    "ASTVPEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3280"]}},
    "ASTVRCO": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "ASTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a3282"]}},
    "ASTVTE": {"-": {"morpho": ["rp--------"], "uri": ["20187"]}},
    "ASTVTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3283"]}},
    "ASTVTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3285"]}},
    "ASYLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3290"]}},
    "AT": {"1": {"morpho": ["rp--------"], "uri": ["46556"]}, "2": {"morpho": ["----------"], "uri": []}},
    "ATAT": {"-": {"morpho": ["----------"], "uri": []}}, "ATAVI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["97704"]}},
    "ATAVVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3298"]}},
    "ATER": {"-": {"morpho": ["aps---mn1r"], "uri": ["a6288"]}},
    "ATHENAE": {"N": {"morpho": ["n-p---fn1-"], "uri": []}},
    "ATHENIENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "ATHENIENSIS": {"N": {"morpho": ["aps---cn3-"], "uri": []}, "A": {"morpho": ["aps---cn3-"], "uri": []}},
    "ATHENIO": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ATHLETA": {"-": {"morpho": ["n-s---mn1-"], "uri": ["a6295"]}},
    "ATIDIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}}, "ATILIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ATINIVS": {"A": {"morpho": ["aps---mn1-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ATLANTES": {"-": {"morpho": ["n-s----n-g"], "uri": []}},
    "ATOMVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["a6300"]}},
    "ATQVE": {"1": {"morpho": ["----------"], "uri": []}, "2": {"morpho": ["----------"], "uri": []}},
    "ATQVI": {"-": {"morpho": ["----------"], "uri": []}},
    "ATRAMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3305"]}},
    "ATRATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3307"]}},
    "ATRIENSIS": {"1": {"morpho": ["n-s---mn3i"], "uri": ["a3312"]}, "2": {"morpho": ["aps---cn3i"], "uri": ["97709"]}},
    "ATRIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3318"]}, "N": {"morpho": ["n-s---nn2-"], "uri": ["a3318"]}},
    "ATROCITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3319"]}},
    "ATROCITER": {"-": {"morpho": ["rp--------"], "uri": ["20206"]}},
    "ATROX": {"-": {"morpho": ["aps---an3i"], "uri": ["a3323"]}},
    "ATTAGEN": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a1984"]}},
    "ATTALICVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "ATTALVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "ATTAMEN": {"-": {"morpho": ["rp--------"], "uri": ["28550"]}},
    "ATTEGIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3333"]}},
    "ATTEMPERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3337"]}},
    "ATTENDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3339"]}},
    "ATTENTE": {"-": {"morpho": ["rp--------"], "uri": ["20216"]}},
    "ATTENTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["20218"]}},
    "ATTENTVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a3327"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["20219"]}},
    "ATTENVATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20221"]}},
    "ATTENVO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3344"]}},
    "ATTERO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3346"]}},
    "ATTEXO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3352"]}},
    "ATTICA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "ATTICVS": {
        "C": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []},
        "A": {"morpho": ["aps---mn1-"], "uri": []}
    }, "ATTINEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3361"]}},
    "ATTINGO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3400"]}},
    "ATTIS": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "ATTOLLO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3368"]}},
    "ATTONDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3369"]}},
    "ATTONITVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a8910"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["54232"]}},
    "ATTONO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3370"]}},
    "ATTRACTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3374"]}},
    "ATTRACTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["51775"]}},
    "ATTRAHO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3376"]}},
    "ATTRECTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3378"]}},
    "ATTRIBVO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3381"]}},
    "ATTRITVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a8913"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["20243"]}},
    "ATTVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}}, "AVARA": {"-": {"morpho": ["n-s----n1-"], "uri": []}},
    "AVARE": {"-": {"morpho": ["rp--------"], "uri": ["20320"]}},
    "AVARITER": {"-": {"morpho": ["rp--------"], "uri": ["20321"]}},
    "AVARITIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3395"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "AVARITIES": {"-": {"morpho": ["n-s---fn5-"], "uri": ["a3395"]}},
    "AVARVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["97737"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a3396"]}},
    "AVCEPS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["a3401"]}},
    "AVCTE": {"-": {"morpho": ["rp--------"], "uri": ["39877"]}},
    "AVCTIFICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3407"]}},
    "AVCTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3408"]}},
    "AVCTIONARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3410"]}},
    "AVCTIONOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a3412"]}},
    "AVCTITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3413"]}},
    "AVCTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3416"]}},
    "AVCTOR": {"-": {"morpho": ["n-s---cn3-"], "uri": ["a3417"]}},
    "AVCTORAMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3419"]}},
    "AVCTORITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3422"]}},
    "AVCTORO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3423"]}},
    "AVCTVMNVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["33721"]}},
    "AVCTVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["a3499"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["97715"]}},
    "AVCVPIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3431"]}},
    "AVCVPO": {"-": {"morpho": ["v1spia--1-"], "uri": ["97716"]}},
    "AVCVPOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a3432"]}},
    "AVDACIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3433"]}},
    "AVDACITER": {"-": {"morpho": ["rp--------"], "uri": ["20257"]}},
    "AVDAX": {"-": {"morpho": ["aps---an3i"], "uri": ["a3434"]}},
    "AVDENS": {"-": {"morpho": ["aps---an3i"], "uri": ["20260"]}},
    "AVDENTER": {"-": {"morpho": ["rp--------"], "uri": ["46574"]}},
    "AVDENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3403"]}},
    "AVDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3437"]}},
    "AVDIENS": {"-": {"morpho": ["aps---an3i"], "uri": ["97718"]}},
    "AVDIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["a3440"]}},
    "AVDITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3441"]}},
    "AVDITOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3444"]}},
    "AVDITORIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3446"]}},
    "AVDITVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["97719"]}},
    "AVDITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a3449"]}}, "AVE": {"-": {"morpho": ["----------"], "uri": []}},
    "AVEHO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3450"]}},
    "AVELLO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3451"]}},
    "AVENA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3452"]}}, "AVENTINVS": {
        "1N": {"morpho": ["n-s----n2-"], "uri": []}, "2N": {"morpho": ["n-s----n--"], "uri": []},
        "A": {"morpho": ["aps---mn1-"], "uri": ["97739"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}
    }, "AVEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3465"]}},
    "AVERNVS": {"N": {"morpho": ["aps---mn1-"], "uri": ["97740"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["97740"]}},
    "AVERRO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3456"]}},
    "AVERRVNCO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3457"]}},
    "AVERSABILIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a3459"]}},
    "AVERSATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3460"]}},
    "AVERSIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3462"]}},
    "AVERSOR": {"1": {"morpho": ["n-s---mn3-"], "uri": ["a3464"]}, "2": {"morpho": ["v1spid--1-"], "uri": ["a3463"]}},
    "AVERSVM": {"-": {"morpho": ["n-s---nn2-", "n-s---nn2-"], "uri": ["47031", "47031"]}},
    "AVERSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["52297"]}},
    "AVERTO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3468"]}},
    "AVFERO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a3469"]}},
    "AVFVGIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["a3470"]}},
    "AVGEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["a3471"]}},
    "AVGESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["a6471"]}},
    "AVGMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["a3475"]}},
    "AVGVR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3480"]}},
    "AVGVRALE": {"-": {"morpho": ["n-s---nn3i"], "uri": ["a3482"]}},
    "AVGVRALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a3482"]}},
    "AVGVRATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a3519"]}},
    "AVGVRIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3489"]}},
    "AVGVROR": {"-": {"morpho": ["v1spid--1-"], "uri": ["132271"]}}, "AVGVSTA": {
        "1N": {"morpho": ["n-s----n1-"], "uri": []}, "2N": {"morpho": ["n-s----n1-"], "uri": []},
        "N": {"morpho": ["n-s---fn2-"], "uri": []}
    }, "AVGVSTE": {"-": {"morpho": ["rp--------"], "uri": ["20280"]}}, "AVGVSTVS": {
        "-": {"morpho": ["aps---mn1-"], "uri": ["a3492"]}, "C": {"morpho": ["n-s----n2-"], "uri": []},
        "N": {"morpho": ["aps---mn1-"], "uri": ["a3492"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["a3492"]}
    }, "AVIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3628"]}},
    "AVIARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3494"]}},
    "AVIDE": {"-": {"morpho": ["rp--------"], "uri": ["20331"]}},
    "AVIDITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3498"]}},
    "AVIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a6490"]}},
    "AVIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a3502"]}},
    "AVITVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3504"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["97743"]}},
    "AVIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3505"]}},
    "AVIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3505"]}},
    "AVLA": {"1": {"morpho": ["n-s---fn1-"], "uri": ["a3672"]}, "2": {"morpho": ["n-s---fn1-"], "uri": ["a3672"]}},
    "AVLAEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3507"]}},
    "AVLVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "AVOCAMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3521"]}},
    "AVOCATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3522"]}},
    "AVOCO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3525"]}},
    "AVOLO": {"2": {"morpho": ["v1spia--1-"], "uri": ["a3527"]}},
    "AVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3529"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "AVRARIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3531"]}},
    "AVRATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["$0164"]}},
    "AVRELIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": ["97723"]}},
    "AVREOLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a2035"]}},
    "AVREVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["a3541"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a3541"]}},
    "AVRICILLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["33303"]}},
    "AVRICOMVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3546"]}},
    "AVRICVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3547"]}},
    "AVRIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["a3550"]}},
    "AVRIFERVS": {"-": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AVRIFEX": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3551"]}},
    "AVRIGA": {"-": {"morpho": ["n-s---mn1-"], "uri": ["a3556"]}},
    "AVRION": {"-": {"morpho": ["rp--------"], "uri": []}},
    "AVRIPIGMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3568"]}},
    "AVRIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["a3569"]}},
    "AVRITVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3572"]}},
    "AVRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3574"]}},
    "AVRORA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["a3576"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "AVRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3585"]}},
    "AVRVNCI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "AVRVNCVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "AVSCVLTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3586"]}},
    "AVSCVLTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3588"]}},
    "AVSPEX": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3589"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "AVSPICATVS": {
        "1": {"morpho": ["n-s---mn4-"], "uri": ["97727"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["20300"]},
        "-": {"morpho": ["n-s---mn4-"], "uri": ["97727"]}
    }, "AVSPICIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3592"]}},
    "AVSPICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["97728"]}},
    "AVSPICOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a3593"]}},
    "AVSTER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["a3595"]}, "N": {"morpho": ["n-s---mn2r"], "uri": ["a3595"]}},
    "AVSTERITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3597"]}},
    "AVSTERVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3599"]}},
    "AVSTRALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a3600"]}},
    "AVSTRINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a3602"]}},
    "AVSVM": {"A": {"morpho": ["n-s---nn2-"], "uri": ["a3436"]}, "-": {"morpho": ["n-s---nn2-"], "uri": ["a3436"]}},
    "AVT": {"-": {"morpho": ["----------"], "uri": []}}, "AVTEM": {"-": {"morpho": ["----------"], "uri": []}},
    "AVTIONOR": {"-": {"morpho": ["v1spid--1-"], "uri": []}},
    "AVTOMATOPOEETVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["103429"]}},
    "AVTVMNALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a3623"]}},
    "AVTVMNITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["a3625"]}}, "AVTVMNVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["a3668"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["a3667"]},
        "N": {"morpho": ["n-s---mn2-"], "uri": ["a3668"]}
    }, "AVTVMO": {"-": {"morpho": ["v1spia--1-"], "uri": ["a3630"]}},
    "AVVNCVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3633"]}},
    "AVVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3634"]}},
    "AVXILIARES": {"-": {"morpho": ["n-p---mn3i"], "uri": ["a3637"]}},
    "AVXILIARII": {"-": {"morpho": ["n-p---mn2-"], "uri": ["a3638"]}},
    "AVXILIARIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["a3637"]}},
    "AVXILIARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["a1733"]}},
    "AVXILIATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3640"]}},
    "AVXILIATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["a3629"]}},
    "AVXILIOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["a3642"]}},
    "AVXILIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["a3643"]}},
    "AXICVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["a3659"]}},
    "AXIS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["a3666"]}},
    "AXON": {"-": {"morpho": ["n-s---mn3-"], "uri": ["a3658"]}},
    "BABYLONICVM": {"-": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "BACCA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0038"]}},
    "BACCAR": {"-": {"morpho": ["n-s---nn3i"], "uri": ["b0014"]}},
    "BACCATVS": {"-": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "BACCHA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["103458"]}, "N": {"morpho": ["n-s---fn3-"], "uri": []}},
    "BACCHABVNDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0016"]}},
    "BACCHANAL": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "BACCHANALIA": {"-": {"morpho": ["n-s----n--"], "uri": []}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "BACCHATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["b0018"]}},
    "BACCHOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["b0022"]}},
    "BACCHVS": {"N": {"morpho": ["n-s---mn2-"], "uri": ["b0023"]}},
    "BACCIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["33914"]}},
    "BACILLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0028"]}},
    "BACVLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0034"]}},
    "BACVLVS": {"-": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "BAIAE": {"N": {"morpho": ["n-p---fn1-"], "uri": []}},
    "BALAENA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["50968"]}},
    "BALANATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0049"]}},
    "BALANTES": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "BALANVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["b0053"]}},
    "BALATRO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["b0054"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "BALATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["b0060"]}},
    "BALBE": {"-": {"morpho": ["rp--------"], "uri": ["20350"]}},
    "BALBVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0056"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["54747"]}},
    "BALBVTIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["b0058"]}},
    "BALEARES": {"N": {"morpho": ["n-s---mn1g"], "uri": []}}, "BALINEAE": {"-": {"morpho": ["n-s----n1-"], "uri": []}},
    "BALINEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": []}},
    "BALLISTA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0070"]}},
    "BALNEARIA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["b0333"]}},
    "BALNEARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0286"]}},
    "BALNEATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["b0078"]}},
    "BALNEOLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0084"]}},
    "BALNEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0085"]}},
    "BALO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0086"]}},
    "BALSAMVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0092"]}},
    "BALTEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["97750"]}},
    "BALTEVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0096"]}},
    "BAPTIZO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0112"]}},
    "BARATHRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0113"]}},
    "BARBA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0114"]}, "N": {"morpho": ["n-s---fn1-"], "uri": ["b0114"]}},
    "BARBARE": {"-": {"morpho": ["rp--------"], "uri": ["46789"]}},
    "BARBARIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0115"]}},
    "BARBARICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0117"]}},
    "BARBARISMVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0118"]}},
    "BARBARVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["b0121"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["b0121"]}},
    "BARBATVS": {"2": {"morpho": ["aps---mn1-"], "uri": ["b0125"]}},
    "BARBIGER": {"-": {"morpho": ["aps---mn1r"], "uri": ["b0127"]}},
    "BARBITON": {"-": {"morpho": ["n-s---nn2-"], "uri": ["36859"]}},
    "BARBITOS": {"-": {"morpho": ["n-s---mn2g"], "uri": ["b0131"]}},
    "BARDITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["103531"]}},
    "BARIOBAL": {"N": {"morpho": ["n-s----n-g"], "uri": []}},
    "BARIS": {"-": {"morpho": ["n-s---fn3g"], "uri": ["b0143"]}},
    "BARO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["b0445"]}},
    "BARRVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0148"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["b0148"]}},
    "BARVLCOS": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "BARYCEPHALVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0149"]}},
    "BASCAVDA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0152"]}},
    "BASIATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["b0154"]}},
    "BASILICA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0157"]}},
    "BASILISCVS": {"N": {"morpho": ["n-s---mn2-"], "uri": ["b0165"]}},
    "BASIO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0168"]}},
    "BASIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["b0170"]}},
    "BASIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0171"]}},
    "BAVBOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["b0193"]}},
    "BEATE": {"-": {"morpho": ["rp--------"], "uri": ["53889"]}},
    "BEATVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0207"]}},
    "BEATVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["41561"]}},
    "BEATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20387"]}},
    "BELLATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["b0220"]}},
    "BELLATRIX": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "BELLE": {"-": {"morpho": ["rp--------"], "uri": ["20393"]}},
    "BELLICOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0227"]}},
    "BELLICVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0082"]}},
    "BELLICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0015"]}},
    "BELLIGER": {"-": {"morpho": ["aps---mn1r"], "uri": ["b0232"]}},
    "BELLIGERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0234"]}},
    "BELLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0241"]}},
    "BELLONA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "BELLVA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["20399"]}},
    "BELLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0244"]}, "N": {"morpho": ["n-s---nn2-"], "uri": ["b0244"]}},
    "BELLVOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["103587"]}},
    "BELLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0245"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "BELVA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0164"]}},
    "BENE": {"-": {"morpho": ["rp--------"], "uri": ["b0256"]}},
    "BENEDICO": {"2": {"morpho": ["v1spia--3-"], "uri": ["b9256"]}},
    "BENEDICTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["103597"]}},
    "BENEFACIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["b9257"]}},
    "BENEFACTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b9259"]}},
    "BENEFICENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0262"]}}, "BENEFICIARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["b0216"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["b0198"]}
    }, "BENEFICIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0266"]}},
    "BENEFICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0268"]}},
    "BENEVENTANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "BENEVENTVM": {"N": {"morpho": ["n-s---nn3-"], "uri": []}},
    "BENEVOLE": {"-": {"morpho": ["rp--------"], "uri": ["32459"]}},
    "BENEVOLENS": {"-": {"morpho": ["aps---an3i"], "uri": ["b0272"]}},
    "BENEVOLENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0795"]}},
    "BENEVOLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0905"]}},
    "BENIGNE": {"-": {"morpho": ["rp--------"], "uri": ["20416"]}},
    "BENIGNITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["b0274"]}},
    "BENIGNVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0275"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "BEO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0278"]}},
    "BERYLLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0280"]}},
    "BES": {"-": {"morpho": ["n-s---mn3i"], "uri": ["b0281"]}},
    "BESTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0491"]}, "N": {"morpho": ["n-s---fn1-"], "uri": ["b0491"]}},
    "BESTIARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["b0285"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["b0285"]}
    }, "BESTIOLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0254"]}},
    "BETA": {"1": {"morpho": ["n-s---fn1-"], "uri": ["b0287"]}, "2": {"morpho": ["n-s---fn1-"], "uri": ["b0287"]}},
    "BIBERARIVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0300"]}},
    "BIBLIOTHECA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0307"]}},
    "BIBO": {"1": {"morpho": ["n-s---mn3-"], "uri": ["b0312"]}, "2": {"morpho": ["v1spia--3-"], "uri": ["b0311"]}},
    "BIBVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0315"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["34019"]}},
    "BICEPS": {"-": {"morpho": ["aps---an3-"], "uri": ["b0317"]}},
    "BICOLOR": {"-": {"morpho": ["aps---an3-"], "uri": ["b0322"]}},
    "BICORNES": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "BICORNIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0325"]}},
    "BIDENS": {"1": {"morpho": ["n-s---mn3i"], "uri": ["b0332"]}, "2": {"morpho": ["aps---an3i"], "uri": ["b0332"]}},
    "BIDENTAL": {"-": {"morpho": ["n-s---nn3i"], "uri": ["b0341"]}},
    "BIDINI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "BIDINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "BIDIS": {"N": {"morpho": ["n-s---fn3-"], "uri": []}},
    "BIDVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0338"]}},
    "BIENNIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0342"]}},
    "BIENNIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0342"]}},
    "BIFARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0346"]}},
    "BIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["b0344"]}},
    "BIFORIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0351"]}},
    "BIFORMIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0353"]}},
    "BIGA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0356"]}},
    "BIGATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0359"]}},
    "BIIVGI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["97766"]}},
    "BIIVGIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0798"]}},
    "BIIVGVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0797"]}},
    "BILIBRIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0370"]}},
    "BILINGVIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0371"]}},
    "BILIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["b0373"]}},
    "BILVSTRIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0376"]}},
    "BIMARIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0379"]}},
    "BIMEMBRES": {"-": {"morpho": ["n-p---mn3i"], "uri": ["97773"]}, "N": {"morpho": ["n-s---mn1g"], "uri": []}},
    "BIMEMBRIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0383"]}},
    "BIMESTRIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["103680"]}},
    "BIMVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0387"]}},
    "BIMVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0388"]}},
    "BINOCTIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0393"]}},
    "BINVS": {"-": {"morpho": ["rp--------"], "uri": []}},
    "BIPALIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0400"]}},
    "BIPARTIOR": {"-": {"morpho": ["v1spid--4-"], "uri": []}},
    "BIPARTITIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["36619"]}},
    "BIPARTITO": {"-": {"morpho": ["rp--------"], "uri": ["40021"]}},
    "BIPATENS": {"-": {"morpho": ["aps---an3i"], "uri": ["b0403"]}},
    "BIPEDALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0404"]}},
    "BIPENNIS": {"1": {"morpho": ["n-s---fn3i"], "uri": ["58143"]}, "2": {"morpho": ["aps---cn3i"], "uri": ["b0410"]}},
    "BIPES": {"1": {"morpho": ["n-s---mn3-"], "uri": ["97780"]}, "2": {"morpho": ["aps---an3-"], "uri": ["b0413"]}},
    "BIREMIS": {"1": {"morpho": ["n-s---fn3i"], "uri": ["97782"]}, "2": {"morpho": ["aps---cn3i"], "uri": ["b0415"]}},
    "BIS": {"-": {"morpho": ["rp--------"], "uri": ["b0418"]}},
    "BISON": {"-": {"morpho": ["n-s---mn3i"], "uri": ["b0428"]}},
    "BISVLCVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0433"]}},
    "BITHYNIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "BITVMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["b0438"]}},
    "BITVMINOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0441"]}},
    "BIVGVS": {"-": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "BIVIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0424"]}},
    "BLAESVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0450"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["47928"]}},
    "BLANDE": {"-": {"morpho": ["rp--------"], "uri": ["37042"]}},
    "BLANDILOQVENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0457"]}},
    "BLANDILOQVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0460"]}},
    "BLANDIMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0461"]}},
    "BLANDIOR": {"-": {"morpho": ["v1spid--4-"], "uri": ["b0462"]}},
    "BLANDITIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0463"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "BLANDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0466"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "BLATERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0477"]}},
    "BLATTA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0496"]}},
    "BLATTARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0482"]}},
    "BOARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0666"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["b0666"]}},
    "BOEOTIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "BOEOTII": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "BOETHVS": {"N": {"morpho": ["n-s---mn2-"], "uri": ["b0498"]}},
    "BOLETATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["b0504"]}},
    "BOLETVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0505"]}},
    "BOMBVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0522"]}},
    "BOMBYCINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0524"]}},
    "BOMBYX": {"-": {"morpho": ["n-s---cn3-"], "uri": ["b0527"]}},
    "BONI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["97788"]}},
    "BONITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["b0533"]}},
    "BONVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0144"]}}, "BONVS": {
        "-": {"morpho": ["aps---mn1-"], "uri": ["b0537"]}, "N": {"morpho": ["aps---mn1-"], "uri": ["b0537"]},
        "A": {"morpho": ["aps---mn1-"], "uri": ["b0537"]}
    }, "BOO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0539"]}},
    "BOREAS": {"-": {"morpho": ["n-s----n-g"], "uri": []}, "N": {"morpho": ["n-s---mn1g"], "uri": ["b0611"]}},
    "BOS": {"-": {"morpho": ["n-s---cn3-"], "uri": ["b0551"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "BOTVLARIVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0568"]}},
    "BRACA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0582"]}},
    "BRACATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0584"]}},
    "BRACHIOLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["40790"]}},
    "BRACHIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["49386"]}},
    "BRACTEA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["36625"]}},
    "BRACTEATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["55710"]}},
    "BRACTEOLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["35033"]}},
    "BRASSICA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0607"]}},
    "BREVE": {"1": {"morpho": ["n-s---nn3i"], "uri": ["97793"]}, "2": {"morpho": ["rp--------"], "uri": []}},
    "BREVI": {"-": {"morpho": ["rp--------"], "uri": ["51970"]}},
    "BREVIARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0621"]}},
    "BREVIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0630"]}},
    "BREVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["b0631"]}},
    "BREVITER": {"-": {"morpho": ["rp--------"], "uri": ["20483"]}},
    "BRITANNIA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "BRITANNVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "BROCCHVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "BRVMA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0646"]}},
    "BRVMALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["b0647"]}},
    "BRVNDISIVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "BRVTESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["b0653"]}},
    "BRVTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0651"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "BVBILE": {"-": {"morpho": ["n-s---nn3i"], "uri": ["30069"]}},
    "BVBO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["b0664"]}},
    "BVBVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0703"]}},
    "BVBVLCVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0670"]}},
    "BVBVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0697"]}},
    "BVCCA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0677"]}},
    "BVCCVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0687"]}},
    "BVCERIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["34960"]}},
    "BVCERVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0691"]}},
    "BVCINA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0804"]}},
    "BVCINATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["b0695"]}},
    "BVCVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0707"]}},
    "BVFO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["b0710"]}},
    "BVLBVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["b0720"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["b0720"]}},
    "BVLLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["b0728"]}},
    "BVLLATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["b0729"]}},
    "BVLLIO": {"-": {"morpho": ["v1spia--4-"], "uri": ["b0731"]}},
    "BVLLITVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["b0688"]}},
    "BVLLO": {"-": {"morpho": ["v1spia--1-"], "uri": ["b0733"]}},
    "BVMASTVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["b0736"]}},
    "BVRIS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["b0744"]}},
    "BVRRVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}}, "BVSIRIS": {"N": {"morpho": ["n-s----n3-"], "uri": []}},
    "BVSTVARIVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["b0768"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["b0768"]}
    }, "BVSTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0769"]}},
    "BVTHROTVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "BVXIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["b0785"]}},
    "BVXVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["b0788"]}},
    "BVXVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["b0788"]}},
    "BYZANTII": {"N": {"morpho": ["n-p---mn2-"], "uri": []}}, "BYZANTIVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "CABALLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0002"]}},
    "CACHINNO": {"1": {"morpho": ["n-s---mn3-"], "uri": ["c0030"]}, "2": {"morpho": ["v1spia--1-"], "uri": ["c0029"]}},
    "CACHINNVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0032"]}},
    "CACO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0036"]}},
    "CACOETHES": {"-": {"morpho": ["n-s---nn3-"], "uri": ["c0040"]}},
    "CACVMEN": {"-": {"morpho": ["n-s---nn3-"], "uri": ["c0050"]}},
    "CACVRIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CACVS": {"N": {"morpho": ["n-s---mn2-"], "uri": ["103996"]}},
    "CADAVER": {"-": {"morpho": ["n-s---nn3-"], "uri": ["c0052"]}},
    "CADMEA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "CADMVS": {"-": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CADO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c0061"]}},
    "CADVCEATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0063"]}},
    "CADVCEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0065"]}},
    "CADVCEVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["43410"]}},
    "CADVCIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["97799"]}, "N": {"morpho": ["n-s----n2-"], "uri": []}},
    "CADVCVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0068"]}},
    "CADVCVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0068"]}},
    "CADVRCVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0069"]}},
    "CADVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0070"]}},
    "CAECIGENVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0074"]}},
    "CAECILIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CAECITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0076"]}},
    "CAECO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0078"]}},
    "CAECVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0080"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["c0080"]}},
    "CAEDES": {"-": {"morpho": ["n-s---fn3i"], "uri": ["c0082"]}},
    "CAEDICIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CAEDO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c0083"]}},
    "CAEDVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0084"]}},
    "CAELATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0086"]}},
    "CAELATVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0130"]}},
    "CAELEBS": {"-": {"morpho": ["aps---an3-"], "uri": ["c0087"]}}, "CAELES": {
        "1": {"morpho": ["n-s----n3-"], "uri": []}, "2": {"morpho": ["aps---an3-"], "uri": ["c0088"]},
        "N": {"morpho": ["n-s---mn1g"], "uri": []}
    }, "CAELESTES": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "CAELESTIA": {"-": {"morpho": ["n-p---nn3i"], "uri": ["97801"]}},
    "CAELESTIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["c0090"]}},
    "CAELI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["104013"]}},
    "CAELIBATVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["c4698"]}},
    "CAELICOLA": {"-": {"morpho": ["n-s---cn1-"], "uri": ["c0095"]}},
    "CAELIFER": {"-": {"morpho": ["aps---mn1r"], "uri": ["c0097"]}},
    "CAELITES": {"-": {"morpho": ["n-p---mn3-"], "uri": ["48504"]}},
    "CAELIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CAELO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0104"]}}, "CAELVM": {
        "1": {"morpho": ["n-s---nn2-"], "uri": ["c9105", "c0105"]},
        "2": {"morpho": ["n-s---nn2-"], "uri": ["c9105", "c0105"]}
    }, "CAEMENTICIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0107"]}},
    "CAEMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0149"]}},
    "CAENINENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "CAENINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CAENO": {"N": {"morpho": ["n-s----n3-"], "uri": []}},
    "CAENOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0110"]}},
    "CAENVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0112"]}},
    "CAEPA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0317"]}},
    "CAEPIO": {"N": {"morpho": ["n-s----n3-"], "uri": []}}, "CAERE": {"N": {"morpho": ["n-s----n--"], "uri": []}},
    "CAERIMONIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0000"]}},
    "CAERVLA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["97805"]}},
    "CAERVLEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0726"]}},
    "CAERVLEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0626"]}},
    "CAESAR": {"N": {"morpho": ["n-s---mn3-"], "uri": ["104039"]}},
    "CAESARIES": {"-": {"morpho": ["n-s---fn5-"], "uri": ["c0132"]}},
    "CAESETIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}}, "CAESIVS": {
        "-": {"morpho": ["aps---mn1-"], "uri": ["c0137"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["38093"]},
        "A": {"morpho": ["aps---mn1-"], "uri": ["c0137"]}
    }, "CAESO": {"N": {"morpho": ["n-s---mn3-"], "uri": []}}, "CAESONIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CAESPES": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0139"]}},
    "CAESTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["c0145"]}},
    "CALACTE": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "CALACTINI": {"N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "CALACTINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CALAMISTER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["c0162"]}},
    "CALAMITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0164"]}},
    "CALAMITOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0166"]}},
    "CALAMVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0168"]}},
    "CALATHISCVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0170"]}},
    "CALATHVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0172"]}},
    "CALCAR": {"-": {"morpho": ["n-s---nn3i"], "uri": ["c0178"]}},
    "CALCARIVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["c0182"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c0182"]}},
    "CALCATVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0213"]}},
    "CALCEAMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0190"]}},
    "CALCEO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0193"]}},
    "CALCEVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0198"]}},
    "CALCO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0203"]}},
    "CALCVLOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0211"]}},
    "CALCVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0212"]}},
    "CALDA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["54763"]}},
    "CALDARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["41332"]}},
    "CALDARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["45236"]}},
    "CALEFACIO": {"-": {"morpho": ["v1spia--3i"], "uri": ["c0214"]}},
    "CALEFACTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0218"]}},
    "CALEFIO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0215"]}},
    "CALENVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CALEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["c0222"]}},
    "CALESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c1101"]}},
    "CALICVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0228"]}},
    "CALIDA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c2516"]}},
    "CALIDIANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CALIDIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CALIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c2491"]}},
    "CALIENDRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["36566"]}},
    "CALIGA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0232"]}},
    "CALIGATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c9235"]}},
    "CALIGINOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0238"]}},
    "CALIGO": {"1": {"morpho": ["n-s---fn3-"], "uri": ["c0239"]}, "2": {"morpho": ["v1spia--1-"], "uri": ["c0240"]}},
    "CALIX": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0245"]}},
    "CALLEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["c0250"]}},
    "CALLIDAMA": {"N": {"morpho": ["n-s----n-g"], "uri": []}},
    "CALLIDE": {"-": {"morpho": ["rp--------"], "uri": ["45369"]}},
    "CALLIDITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0253"]}},
    "CALLIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0255"]}},
    "CALLIS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["c0263"]}},
    "CALLISCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c9704"]}},
    "CALLOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0269"]}},
    "CALLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0270"]}},
    "CALO": {"1": {"morpho": ["n-s---mn3-"], "uri": ["c1152"]}, "2": {"morpho": ["v1spia--1-"], "uri": ["c0272"]}},
    "CALOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0276"]}},
    "CALPVRNIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CALTHA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0281"]}},
    "CALVARIA": {"1": {"morpho": ["n-s---fn1-"], "uri": ["c9284"]}, "2": {"morpho": ["n-s---fn1-"], "uri": ["c9284"]}},
    "CALVITIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0290"]}},
    "CALVMNIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0292"]}},
    "CALVMNIATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0294"]}},
    "CALVMNIOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["c0296"]}},
    "CALVVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c2611"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["40642"]}},
    "CALX": {
        "1": {"morpho": ["n-s---fn3-"], "uri": ["c0291", "c0302"]},
        "2": {"morpho": ["n-s---fn3-"], "uri": ["c0291", "c0302"]}
    }, "CAMELVS": {"-": {"morpho": ["n-s---cn2-"], "uri": ["c3037"]}},
    "CAMENA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["34882"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "CAMENAE": {"N": {"morpho": ["n-p---fn1-"], "uri": []}},
    "CAMERA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["$0155"]}},
    "CAMERIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "CAMINVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0325"]}},
    "CAMMARVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0327"]}},
    "CAMPANIA": {"N": {"morpho": ["n-s---fn1-"], "uri": ["104154"]}},
    "CAMPESTER": {"-": {"morpho": ["aps---mn3-"], "uri": ["51934"]}},
    "CAMPESTRE": {"-": {"morpho": ["n-s---nn3i"], "uri": ["97815"]}},
    "CAMPESTRIA": {"-": {"morpho": ["n-p---nn3i"], "uri": ["c0334"]}}, "CAMPVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["c1593"]}, "2": {"morpho": ["n-s---mn2-"], "uri": ["c1593"]},
        "N": {"morpho": ["n-s---mn2-"], "uri": ["c1593"]}
    }, "CAMVRVS": {"-": {"morpho": ["aps---mn1-"], "uri": []}},
    "CANALICVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0361"]}},
    "CANALIS": {"1": {"morpho": ["n-s---fn3i"], "uri": []}, "2": {"morpho": ["aps---cn3i"], "uri": ["104171"]}},
    "CANCELLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0373"]}},
    "CANCER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["c0375"]}, "N": {"morpho": ["n-s----n2-"], "uri": []}},
    "CANDELA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0383"]}},
    "CANDELABRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0385"]}},
    "CANDENS": {"-": {"morpho": ["aps---an3i", "aps---an3i"], "uri": ["97819", "97819"]}},
    "CANDENTIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c9917"]}},
    "CANDEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["c0386"]}},
    "CANDESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c1245"]}}, "CANDIDATVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["c9393"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c9722"]}
    }, "CANDIDVLVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0393"]}},
    "CANDIDVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0394"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CANDOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0397"]}},
    "CANEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["c0400"]}},
    "CANEPHORA": {"N": {"morpho": ["n-s----n-g"], "uri": []}},
    "CANESCO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c0400"]}},
    "CANI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["c0472"]}},
    "CANICVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0401"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "CANIDIA": {"-": {"morpho": ["n-s----n1-"], "uri": []}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "CANINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0407"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["41544"]}},
    "CANIS": {"-": {"morpho": ["n-s---mn3-"], "uri": ["20611"]}, "N": {"morpho": ["n-s---fn3-"], "uri": []}},
    "CANISTRA": {"-": {"morpho": ["n-p---nn2-"], "uri": ["97822"]}},
    "CANISTRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["41731"]}},
    "CANITIES": {"-": {"morpho": ["n-s---fn5-"], "uri": ["c0415"]}},
    "CANNA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c2412"]}},
    "CANNABIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["c0421"]}},
    "CANNENSIS": {"N": {"morpho": ["aps---cn3i"], "uri": []}, "A": {"morpho": ["aps---cn3i"], "uri": []}},
    "CANNVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0427"]}},
    "CANO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c0428"]}},
    "CANON": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0429"]}},
    "CANONICVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["c0431"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c0431"]}},
    "CANOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0433"]}},
    "CANORVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0434"]}},
    "CANTABRI": {"-": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["n-p---mn2-"], "uri": []}},
    "CANTERIVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["42432"]}},
    "CANTHARVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0451"]}},
    "CANTHVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0456"]}},
    "CANTICVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0458"]}},
    "CANTILENA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0460"]}},
    "CANTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0463"]}},
    "CANTITO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0464"]}},
    "CANTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0466"]}},
    "CANTOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0467"]}},
    "CANTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["c9983"]}},
    "CANVLEIANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CANVLEIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CANVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0472"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CAPACITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0476"]}},
    "CAPAX": {"-": {"morpho": ["aps---an3i"], "uri": ["c0477"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "CAPEDVNCVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0479"]}},
    "CAPELLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0480"]}, "N": {"morpho": ["n-p---fn1-"], "uri": []}},
    "CAPENA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "CAPENAS": {"A": {"morpho": ["aps---an3-"], "uri": []}},
    "CAPER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["c0483"]}},
    "CAPESSO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c0485"]}},
    "CAPETVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CAPILLAMENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0490"]}},
    "CAPILLATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0495"]}},
    "CAPILLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0499"]}},
    "CAPIO": {"1": {"morpho": ["n-s---fn3-"], "uri": ["c0501"]}, "2": {"morpho": ["v1spia--3i"], "uri": ["c0500"]}},
    "CAPISTRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0505"]}},
    "CAPISTRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0506"]}},
    "CAPITAL": {"-": {"morpho": ["n-s---nn3i"], "uri": ["c0507"]}},
    "CAPITALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["c0507"]}},
    "CAPITATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0511"]}},
    "CAPITATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0512"]}},
    "CAPITINVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CAPITO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0516"]}, "N": {"morpho": ["n-s---mn3-"], "uri": ["c0516"]}},
    "CAPITOLINVS": {
        "A": {"morpho": ["aps---mn1-"], "uri": []}, "N": {"morpho": ["aps---mn1-", "n-s---mn2-"], "uri": []}
    }, "CAPITOLIVM": {"N": {"morpho": ["n-s---nn2-"], "uri": []}},
    "CAPITVLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0522"]}},
    "CAPPADOCIA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}},
    "CAPRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0483"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "CAPREA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0537"]}},
    "CAPREOLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0518"]}},
    "CAPRIFICVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["c0546"]}},
    "CAPRIGENVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0547"]}},
    "CAPRILE": {"-": {"morpho": ["n-s---nn3i"], "uri": ["c0548"]}},
    "CAPRIMVLGVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0549"]}},
    "CAPRINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0551"]}},
    "CAPRIPES": {"-": {"morpho": ["aps---an3-"], "uri": ["c0553"]}},
    "CAPSA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0557"]}, "N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "CAPSVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0563"]}},
    "CAPSVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0564"]}},
    "CAPTATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0566"]}},
    "CAPTATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0567"]}},
    "CAPTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0571"]}},
    "CAPTIOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0572"]}},
    "CAPTIVA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0581"]}},
    "CAPTIVITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0574"]}},
    "CAPTIVVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["c0581"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c0581"]}},
    "CAPTO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0582"]}},
    "CAPTVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c9918"]}},
    "CAPTVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["c9984"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["104282"]}},
    "CAPVLATOR": {"-": {"morpho": ["n-s----n3-"], "uri": []}},
    "CAPVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0590"]}},
    "CAPVT": {"-": {"morpho": ["n-s---nn3-"], "uri": ["c0592"]}}, "CAPYS": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "CARBASA": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "CARBASEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c9597"]}}, "CARBASVS": {
        "1": {"morpho": ["n-s---fn2-"], "uri": []}, "2": {"morpho": ["aps---mn1-"], "uri": ["c0598"]},
        "A": {"morpho": ["aps---mn1-"], "uri": ["c0598"]}
    }, "CARBO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0599"]}, "N": {"morpho": ["n-s---mn3-"], "uri": ["c0599"]}},
    "CARBVNCVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0606"]}},
    "CARCER": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0608"]}},
    "CARCHESIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0614"]}},
    "CARCINOMA": {"-": {"morpho": ["n-s---nn3-"], "uri": ["c0617"]}},
    "CARDIACVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["28973"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c0519"]}},
    "CARDINALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["c0629"]}},
    "CARDINATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["$0170"]}},
    "CARDO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0633"]}},
    "CARDVVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0635"]}},
    "CARE": {"-": {"morpho": ["rp--------"], "uri": ["44377"]}},
    "CARECTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c9636"]}},
    "CAREO": {"-": {"morpho": ["v1spia--2-"], "uri": ["c0637"]}},
    "CAREX": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0639"]}},
    "CARICA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c9921"]}},
    "CARIES": {"-": {"morpho": ["n-s---fn5-"], "uri": ["c0642"]}},
    "CARINA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0643"]}, "N": {"morpho": ["n-s---fn1-"], "uri": ["c0643"]}},
    "CARIOSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0649"]}},
    "CARITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0655"]}},
    "CARMEN": {"1": {"morpho": ["n-s---nn3-"], "uri": ["c4060"]}, "2": {"morpho": ["n-s---nn3-"], "uri": ["c4060"]}},
    "CARMENTALIS": {"N": {"morpho": ["aps---cn3i"], "uri": []}, "A": {"morpho": ["aps---cn3i"], "uri": []}},
    "CARMENTIS": {"N": {"morpho": ["n-s----n--"], "uri": []}},
    "CARMO": {"-": {"morpho": ["n-s----n3-"], "uri": []}, "N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "CARNARIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0663"]}},
    "CARNIFEX": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0670"]}},
    "CARNIFICINA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0673"]}},
    "CARNIFICO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0675"]}},
    "CARO": {"1": {"morpho": ["n-s---fn3-"], "uri": ["c0681"]}, "2": {"morpho": ["v1spia--3-"], "uri": ["c0680"]}},
    "CARPATINA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["97836"]}},
    "CARPENTVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0692"]}},
    "CARPINATIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CARPINVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["c0696"]}},
    "CARPO": {"-": {"morpho": ["v1spia--3-"], "uri": ["c0698"]}},
    "CARPTIM": {"-": {"morpho": ["rp--------"], "uri": ["c9698"]}},
    "CARPTOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0700"]}},
    "CARRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["53849"]}},
    "CARRVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0713"]}},
    "CARTHAGINIENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "CARTHAGINIENSIS": {"N": {"morpho": ["aps---cn3-"], "uri": []}, "A": {"morpho": ["aps---cn3-"], "uri": []}},
    "CARTHAGO": {"N": {"morpho": ["n-s---fn3-"], "uri": []}},
    "CARVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0721"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CARYATIS": {"-": {"morpho": ["n-s----n-g"], "uri": []}},
    "CASA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0728"]}},
    "CASEVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["31950"]}},
    "CASEVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0740"]}},
    "CASIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0741"]}},
    "CASSIANVS": {"N": {"morpho": ["aps---mn1-"], "uri": []}, "A": {"morpho": ["aps---mn1-"], "uri": []}},
    "CASSIDA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0749"]}},
    "CASSIS": {"1": {"morpho": ["n-s---mn3i"], "uri": ["20712"]}, "2": {"morpho": ["n-s---mn3i"], "uri": ["20712"]}},
    "CASSIVS": {"A": {"morpho": ["aps---mn1-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CASSVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0760"]}},
    "CASSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0760"]}},
    "CASTANEA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0761"]}},
    "CASTANEVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["97843"]}},
    "CASTE": {"-": {"morpho": ["rp--------"], "uri": ["20718"]}},
    "CASTELLANI": {"-": {"morpho": ["n-p---mn2-"], "uri": ["c0764"]}}, "CASTELLANVS": {
        "1": {"morpho": ["n-s---mn2-"], "uri": ["47581"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c0764"]}
    }, "CASTELLARIVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0765"]}},
    "CASTELLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0767"]}},
    "CASTIFICVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0771"]}},
    "CASTIGATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0773"]}},
    "CASTIGATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0774"]}},
    "CASTIGATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["56329"]}},
    "CASTIGO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0776"]}},
    "CASTIMONIA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0778"]}},
    "CASTITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0781"]}},
    "CASTOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0783"]}, "N": {"morpho": ["n-s---mn3-"], "uri": ["c0783"]}},
    "CASTOREVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0784"]}},
    "CASTOREVS": {"N": {"morpho": ["aps---mn1-"], "uri": ["c0784"]}, "A": {"morpho": ["aps---mn1-"], "uri": ["c0784"]}},
    "CASTRA": {
        "1": {"morpho": ["n-s---fn1-"], "uri": ["c0787"]}, "2": {"morpho": ["n-s---fn1-"], "uri": ["c0787"]},
        "N": {"morpho": ["n-s---fn2-"], "uri": []}
    }, "CASTRENSIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["c0793"]}},
    "CASTRICIVS": {"N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CASTRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0796"]}},
    "CASTRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c0787"]}, "N": {"morpho": ["n-s---nn2-"], "uri": ["c0787"]}},
    "CASTVS": {"1": {"morpho": ["n-s---mn4-"], "uri": ["c0798"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c0798"]}},
    "CASV": {"-": {"morpho": ["rp--------"], "uri": ["45470"]}},
    "CASVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0800"]}},
    "CASVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["c9985"]}},
    "CATAGRAPHVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0820"]}},
    "CATAPHRACTVS": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "CATAPLASMA": {"-": {"morpho": ["n-s----n2-"], "uri": []}},
    "CATAPLEXIS": {"-": {"morpho": ["n-s---fn3i"], "uri": ["c0838"]}},
    "CATAPVLTA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0841"]}},
    "CATARACTA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0843"]}},
    "CATASCOPVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c9850"]}},
    "CATASTA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0851"]}},
    "CATATEXITECHNOS": {"-": {"morpho": ["aps---mn1g"], "uri": []}, "A": {"morpho": ["aps---mn1g"], "uri": []}},
    "CATATONVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0859"]}},
    "CATECHON": {"-": {"morpho": ["----------"], "uri": []}}, "CATELLA": {
        "-": {"morpho": ["n-s---fn1-"], "uri": ["c0871"]}, "2": {"morpho": ["n-s---fn1-"], "uri": ["c0871"]},
        "1": {"morpho": ["n-s---fn1-"], "uri": ["c0871"]}
    }, "CATELLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0871"]}},
    "CATENA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0872"]}},
    "CATENARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0873"]}},
    "CATENATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0874"]}},
    "CATENO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c0877"]}},
    "CATERVA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0880"]}},
    "CATERVATIM": {"-": {"morpho": ["rp--------"], "uri": ["c0882"]}},
    "CATHEDRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0885"]}},
    "CATHEDRARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0888"]}},
    "CATHETVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["c0893"]}},
    "CATILINA": {"N": {"morpho": ["n-s---fn2-"], "uri": []}},
    "CATILLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0900"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["c0900"]}},
    "CATINA": {"N": {"morpho": ["n-s---fn1-"], "uri": []}}, "CATINENSES": {"N": {"morpho": ["n-p---cn3i"], "uri": []}},
    "CATINENSIS": {"N": {"morpho": ["aps---cn3i"], "uri": []}, "A": {"morpho": ["aps---cn3i"], "uri": []}},
    "CATINVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0902"]}},
    "CATLASTER": {"-": {"morpho": ["n-s---mn2r"], "uri": ["c0903"]}},
    "CATO": {"N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "CATVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0923"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["c0923"]}},
    "CATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0925"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["97850"]}},
    "CAVATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["46262"]}},
    "CAVDA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0935"]}},
    "CAVDEX": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0937"]}, "N": {"morpho": ["n-s----n3-"], "uri": []}},
    "CAVDICARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0940"]}},
    "CAVEA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0945"]}},
    "CAVEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["c0949"]}},
    "CAVERNA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0952"]}},
    "CAVILLATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c0961"]}},
    "CAVILLATOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0962"]}},
    "CAVILLOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["c0966"]}},
    "CAVLA": {"-": {"morpho": ["n-s----n1-"], "uri": []}},
    "CAVLICVLVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0974"]}},
    "CAVLIS": {"-": {"morpho": ["n-s---mn3i"], "uri": ["c0976"]}},
    "CAVO": {"1": {"morpho": ["v1spia--1-"], "uri": ["c0980"]}, "2": {"morpho": ["v1spia--1-"], "uri": ["c0980"]}},
    "CAVPO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c0982"]}},
    "CAVPONA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0983"]}},
    "CAVPONOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["c0987"]}},
    "CAVRVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c0994"]}, "N": {"morpho": ["n-s---mn2-"], "uri": ["c0994"]}},
    "CAVSA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c0995"]}},
    "CAVSARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c0999"]}},
    "CAVSIDICVS": {"-": {"morpho": ["n-s---mn2-"], "uri": ["c1008"]}},
    "CAVSOR": {"-": {"morpho": ["v1spid--1-"], "uri": ["c1012"]}},
    "CAVSVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c1014"]}},
    "CAVTE": {"-": {"morpho": ["rp--------"], "uri": ["20765"]}},
    "CAVTES": {"-": {"morpho": ["n-s---fn3i"], "uri": ["c1020"]}},
    "CAVTIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c1022"]}},
    "CAVTVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20769"]}},
    "CAVVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c1028"]}},
    "CAVVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["20779"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c1028"]}},
    "CEDO": {"1": {"morpho": ["v1spia--3-"], "uri": ["c1031"]}, "2": {"morpho": ["----------"], "uri": []}},
    "CEDRIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c1037"]}},
    "CEDRVS": {"-": {"morpho": ["n-s---fn2-"], "uri": ["c1039"]}},
    "CELATVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["57522"]}},
    "CELEBER": {"-": {"morpho": ["aps---mn3-"], "uri": ["c1043"]}},
    "CELEBRATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c1045"]}},
    "CELEBRATVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["20785"]}},
    "CELEBRITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c1047"]}},
    "CELEBRITER": {"-": {"morpho": ["rp--------"], "uri": ["31070"]}},
    "CELEBRO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c1049"]}},
    "CELER": {"-": {"morpho": ["aps---an3-"], "uri": ["c1051"]}, "N": {"morpho": ["n-s---mn3-"], "uri": []}},
    "CELERIPES": {"-": {"morpho": ["aps---an3-"], "uri": ["c1054"]}},
    "CELERITAS": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c1055"]}},
    "CELERITER": {"-": {"morpho": ["rp--------"], "uri": ["20792"]}},
    "CELERO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c1059"]}},
    "CELLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c1067"]}},
    "CELLARIVS": {"1": {"morpho": ["n-s---mn2-"], "uri": ["c1073"]}, "2": {"morpho": ["aps---mn1-"], "uri": ["c1073"]}},
    "CELLO": {"-": {"morpho": ["v1spia--3-"], "uri": ["104533"]}},
    "CELLVLA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c1076"]}},
    "CELO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c1078"]}},
    "CELSVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c1085"]}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CENA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c1089"]}},
    "CENACVLVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c1094"]}},
    "CENATIO": {"-": {"morpho": ["n-s---fn3-"], "uri": ["c1098"]}},
    "CENDO": {"-": {"morpho": ["v1spia--3-"], "uri": []}}, "CENO": {"-": {"morpho": ["v1spia--1-"], "uri": ["c1108"]}},
    "CENSEO": {"-": {"morpho": ["v1spia--2-"], "uri": ["c1119"]}},
    "CENSOR": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c1123"]}}, "CENSORIVS": {
        "1": {"morpho": ["n-s----n2-"], "uri": []}, "2": {"morpho": ["aps---mn1-"], "uri": ["c1125"]},
        "N": {"morpho": ["n-s---mn2-"], "uri": []}
    }, "CENSVRA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c1118"]}},
    "CENSVS": {"-": {"morpho": ["n-s---mn4-"], "uri": ["c1130"]}},
    "CENTAVREVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c3981"]}},
    "CENTAVRVS": {"-": {"morpho": ["n-s----n2-"], "uri": []}, "N": {"morpho": ["n-s---mn2-"], "uri": []}},
    "CENTENARIVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c1129"]}},
    "CENTENVS": {"-": {"morpho": ["rp--------"], "uri": []}},
    "CENTESIMA": {"-": {"morpho": ["n-s---fn1-"], "uri": ["c3987"]}},
    "CENTESIMVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c3985"]}},
    "CENTICEPS": {"-": {"morpho": ["aps---an3-"], "uri": ["c1136"]}},
    "CENTIES": {"-": {"morpho": ["rp--------"], "uri": []}},
    "CENTIMANVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c1142"]}},
    "CENTO": {"-": {"morpho": ["n-s---mn3-"], "uri": ["c1150"]}, "N": {"morpho": ["n-s---mn3-"], "uri": ["c1150"]}},
    "CENTRVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c1158"]}},
    "CENTVM": {"-": {"morpho": ["----------"], "uri": []}},
    "CENTVMGEMINVS": {"-": {"morpho": ["aps---mn1-"], "uri": ["c1160"]}},
    "CENTVMPONDIVM": {"-": {"morpho": ["n-s---nn2-"], "uri": ["c8163"]}},
    "CENTVMVIR": {"-": {"morpho": ["n-s---mn2r"], "uri": ["35152"]}},
    "CENTVMVIRALIS": {"-": {"morpho": ["aps---cn3i"], "uri": ["c1164"]}},
    "CENTVNCVLVS": {"-": {"morpho": ["n-s---mn2-"], 