"""
This module defines the usage for the analog functionality
of the RPiSoC.
"""

__author__ = 'Brian Bradley'
__version__ = '1.2.4'

from rpisoc import *
from time import sleep

class CapSense(object):
    """
    :Class:

        Provides functionality for use of CapSense buttons on the RPiSoC

    :Example:

        Define a CapSense object in the following way::

            CapSense(0)
    """
    def __init__(self,PIN, THRESHOLD = 0):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Initializes a Capsense object for use as a touch sensor                                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **list**   *PIN*                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *Optional*        |* **list**   *THRESHOLD*                                                                                                                                                                                       |
        | *Parameters*      |                                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *PIN*             |The capsense pin number. PIN n directly corresponds to the pin which you assign to Capsense_BTN_N in your rpisoc firmware                                                                                      |
        |                   |   * The default version (V1.2) comes with 8 CapSense pins, and so 0-7 are valid arguments                                                                                                                     |
        |                   |   * The default version has CapSense pins 0-5 on Port 4, pins 0-5. *PIN* 6 is on P0[5] and *PIN* 7 is on P0[6]                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *THRESHOLD*       |The number of counts, between 0 and 255, which will be added to your calibrated baseline value to determine if a button has been touched                                                                       |
        |                   |   * By default, this argument is 0, which means there is no *THRESHOLD* being used                                                                                                                            |
        |                   |   * 0 means that the baseline value itself will be used as a comparison value. Generally, baseline + THRESHOLD is used                                                                                        |
        |                   |   * The threshold may need to be high if the button data is noisy. Try printing the results of :meth:`~readRaw` to get an understanding of the output behavior as you touch the button, this will help        |
        |                   |     determine a suitable value                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Notes**         |* *THRESHOLD* is only needed if using :meth:`~isTouched` to read sensors, which is the suggested method. :meth:`~Read` is only valid when SmartSense is used in your PSoC Creator firmware                     |
        |                   |   - The default version uses no tuning method, because the calibration is manually handled seperately                                                                                                         |
        |                   |* If SmartSense is used as a tuning method in your PSoC Creator software, for instance if you want to implement sliding capability, you will need to connect a 2.2 nF capacitor to P4[6]                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if RPiSoC.CAPSENSE_SENSOR_NUM is 0:
            raise ValueError('Error: No CapSense pins found. Verify your schematic on the RPiSoC is correct, or try resetting the RPiSoC and trying again.')
        elif PIN not in range(RPiSoC.CAPSENSE_SENSOR_NUM):
            raise ValueError('Invalid CapSense sensor number chosen: Valid range is between 0 and %d'%(RPiSoC.CAPSENSE_SENSOR_NUM - 1))
        else:
            self.number = PIN
            self.address = RPiSoC.CAPSENSE_REGISTER
        self.threshold = THRESHOLD

        self.__running = False

    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |* Initializes registers and enables active mode power template bits of the subcomponents used within CapSense                                                                                                  |
        |                   |* Establishes a CapSense baseline, which is used as a comparison value in :meth:`~isTouched`                                                                                                                   |
        |                   |* The button should not be touched when this method is called, otherwise the baseline value will be calibrated with a touched value. It will only set the baseline if it sees two identical readings in a row  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        cmd = 0x00
        RPiSoC.commChannel.sendData((self.address,cmd))
        while True:
            self.baseline = self.readRaw()
            if RPiSoC.DEBUG:
                print("Calibrating Sensor %d->Raw data:%d"%(self.number,self.baseline))
            if self.baseline == self.readRaw():
                if RPiSoC.DEBUG:
                    print("Sensor %d calibrated for raw values of %d"%(self.number,self.baseline))
                break
        self.__running = True

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the CapSense component and all subcomponents are currently operational                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if CapSense is active, or *False* if it is not                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return self.__running

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Disables component interrupts, and calls CapSense_ClearSensors() to reset all sensors to an inactive state                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Side Effects**  |This will affect *all* CapSense sensors, use it only if this is your intention                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        cmd = 0x01
        RPiSoC.commChannel.sendData((self.address,cmd))
        self.__running = False

    def Sleep(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Sleep                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Prepares the component for the device entering a low-power mode. Disables Active mode power template bits of the sub components used within CapSense, saves nonretention registers, and resets all sensors to  |
        |                   |an inactive state                                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Side Effects**  |This will affect *all* CapSense sensors, use it only if this is your intention                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x02
        RPiSoC.commChannel.sendData((self.address,cmd))

    def Wakeup(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Wakeup                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Restores CapSense configuration and nonretention register values after the device wake from a low power mode sleep mode                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Side Effects**  |This will affect *all* CapSense sensors, use it only if this is your intention                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x03
        RPiSoC.commChannel.sendData((self.address,cmd))

    def Read(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Read                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |* Determines the state of the capsense button, as determined by PSoC SmartSensing. This will need to be calibrated in PSoC Creator for optimal results                                                         |
        |                   |* Use :meth:`isTouched` for more general application                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |**Parameters**     |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *state*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |*state*            |A True or False value when the output is determined to be touched or not touched, respectively                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x18
        val = self.number
        return bool(RPiSoC.commChannel.receiveData((self.address,cmd, val),delay = 0.03))

    def readRaw(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | readRaw                                                                                                                                                                                                       |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |* Gives an 8-bit raw value from the capsense raw data array, which represents capacitance detection and is thus correlated with touch data                                                                     |
        |                   |* This is used to generate a baseline in :meth:`~Start`                                                                                                                                                        |
        |                   |* Use this to calibrate touch buttons by deciding an approriate *THRESHOLD* as described in :meth:`~__init__`                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |**Parameters**     |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *val*                                                                                                                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |*state*            |The raw value on the capsense button, which can be used to determine a touch event                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x0F
        val = self.number
        return RPiSoC.commChannel.receiveData((self.address, cmd, val))

    def isTouched(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isTouched                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Uses the calibrated baseline and provided threshold value to determine if the requested CapSense button is being touched. This is the suggested method for touch detection.                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *state*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *state*           |A True or False value when the output is determined to be touched or not touched, respectively                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if self.readRaw()>(self.baseline + self.threshold):
            return True
        else:
            return False

class analogPin(object):
    """
    :Class:

        Provides functionality for use of the general purpose analog input pins exposed on the RPiSoC

    :Example:

        Define an analogPin object in the following way::

            My_analog_pin = analogPin(PIN)
    """
    def __init__(self, PIN):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Constructs and initializes an analogPin object for general purpose analog input                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *PIN*                                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *PIN*             |The analog pin number - This directly corresponds to the pins connected to your sequenced SAR ADC, where 0 is the top pin                                                                                      |
        |                   |   * The default version (V1.2) comes with 8 CapSense pins, and so 0-7 are valid arguments                                                                                                                     |
        |                   |   * For the default version, *PIN* *0-7* are relative to Port 3. So *PIN* = 0 an analog pin on Port 3 Pin 0 (P3[0]), and *PIN* = 7 is on Port 3 Pin 7 (P3[7])                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        if int(PIN) not in range(RPiSoC.ANALOG_IN_NUM):
            raise ValueError('Invalid Pin for Analog input specified. Only %d analog inputs available.' %RPiSoC.ANALOG_IN_NUM)
        else:
            self.pin = PIN

        self.address = RPiSoC.ANALOG_IN_REGISTER
        self.__resolution = 12 #this needs to be updated so that it is read in during init...
        self.__max_counts = pow(2,self.__resolution)
        self.__offset = 0

    def Read(self):
        """

        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Read                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Reads the calculated digital value after analog to digital conversion                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |**Parameters**     |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *counts*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *counts*          |* Digital value after an A/D conversion                                                                                                                                                                        |
        |                   |* The voltage which is represented by this value will depend on the resolution set by :meth:`~SetResolution`                                                                                                   |
        |                   |* Default resolution is 12, so :math:`V = 5.0*\\frac{counts}{2^{12} - 1}`                                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        cmd = 0x00
        counts = int(RPiSoC.commChannel.receiveData((self.address, cmd, self.pin)))
        return counts
    def ReadVolts(self, PRECISION = 2, COUNTS = None):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | ReadVolts                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Reads the specified analog pin and converts that digital reading to a voltage in Volts, or converts a given count value to Volts                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *Optional*        |* **int**    *PRECISION*                                                                                                                                                                                       |
        | *Parameters*      |* **int**    *COUNTS*                                                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *PRECISION*     |The number of decimal points to be included in the returned result                                                                                                                                             |
        |                   |   * Defaults to 2 if no parameter is given                                                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *COUNTS*        |The count value which should be converted to an analog voltage                                                                                                                                                 |
        |                   |   * Only provide this parameter if you only want to convert the given count value, but not get a new one. No data request will be sent to the RPiSoC if this argument is provided                             |
        |                   |   * If no parameter is provided, this method will get a digital value from the RPiSoC, and then convert it                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** volts                                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *volts*           |The converted digital value into a representative analog voltage.                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x01
        if COUNTS is None:
            return round((float(((RPiSoC.commChannel.receiveData((self.address,cmd, self.pin)))/1000000.0))), PRECISION)
        else:
            return round(float(float(COUNTS - self.__offset)/self.__max_counts)*5.0, PRECISION)

    def SetOffset(self,counts):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetOffset                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets an offset on the ADC which is used for the voltage conversion. It will subtract the given offset before making the conversion                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *counts*                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *counts*          |Digital value which will be subtracted from any digital result before its conversion to a voltage takes place                                                                                                  |
        |                   |   * For example, if a digital value of 255 is to be converted to a voltage, and the offset has been provided by this method at any point as 10, the result will be the conversion of 245 instead of 255       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x02
        self.__offset = counts
        RPiSoC.commChannel.sendData((self.address, cmd, counts))

    def SetResolution(self, resolution):
        """
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetResolution                                                                                                                                                                                                                 |
        +===================+===============================================================================================================================================================================================================================+
        | **Description**   |* This function sets the resolution of ALL analogPin objects.                                                                                                                                                                  |
        |                   |* The resolution is defaulted to 12, so this method is only needed if this needs to be changed                                                                                                                                 |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *resolution*                                                                                                                                                                                                         |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *resolution*      | An integer value, in bits, that represents the new resolution of the SAR ADC                                                                                                                                                  |
        |                   |   * Valid arguments are *8, 10, or 12* only                                                                                                                                                                                   |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                                      |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        if resolution not in [8, 10, 12]:
            raise ValueError('Invalid resolution specified: valid entries are 8, 10, or 12')
        self.__resolution = resolution
        self.__max_counts = pow(2,self.__resolution)

        cmd = 0x03
        RPiSoC.commChannel.sendData((self.address, cmd, resolution))




class ADC(object):
    """
    :Class:

        The ADC class provides functionality for using the Delta Sigma ADC and two
        Succesive Approximation ADC's on the RPiSoC.

    :Example:

        Define ADC objects in the following way::

                My_DELSIG    = ADC('DELSIG')
                My_SAR       = ADC('SAR0')
                My_other_SAR = ADC('SAR1')
    """

    def __init__(self, c_type):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Constructs and initializes an ADC object for use of a SAR or DELSIG ADC                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**string**   *c_type*                                                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *c_type*          |The type of ADC to be used. Valid arguments are:                                                                                                                                                               |
        |                   |   * 'DELSIG'- This is for the Delta sigma ADC, which has resolution up to 20 bits, but a slower conversion rate                                                                                               |
        |                   |   * 'SAR0' or 'SAR1' - These are for the two Successive Approximation ADCs, which has resolution of 8, 10, or 12 bits, but with a very fast conversion rate                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Note**          |None of the ADCs are included by default in V1.2, so this class will be unavailable. You are able to add them by simply placing them in your schematic on your RPiSoC program and reprogramming the board      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        self.type = c_type

        if self.type == 'DELSIG':
            if not RPiSoC.DELSIG:
                raise ValueError('Delta-Sigma ADC not found on PSoC Creator program, verify that it is in your schematic and named correctly')
            self.address = RPiSoC.DELSIG_ADC_CONTROL
        elif self.type == 'SAR0':
            if not RPiSoC.SAR0:
                raise ValueError('SAR0 ADC not found on PSoC Creator program, verify that it is in your schematic and named correctly')
            self.address = RPiSoC.SAR_ADC0_CONTROL
        elif self.type == 'SAR1':
            if not RPiSoC.SAR1:
                raise ValueError('SAR1 ADC not found on PSoC Creator program, verify that it is in your schematic and named correctly')
            self.address = RPiSoC.SAR_ADC1_CONTROL
        else:
            raise ValueError('Invalid ADC type: Choose "DELSIG" "SAR0" or "SAR1"')
        self.polarity = 1

        if RPiSoC.DEBUG:
            if self.address in RPiSoC.REGISTERS_IN_USE:
                print('WARNING: Attempting to initialize object at register %d which is already in use.' %self.address)
        RPiSoC.REGISTERS_IN_USE.append(self.address)

        self.__running = False

    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets the initVar variable on the RPiSoC, calls the ADC_Init() function, and then calls the ADC_Enable() function. This function configures and powers up the ADC, but does not start conversions               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x00
        data = (self.address, cmd)
        RPiSoC.commChannel.sendData(data)
        self.__running =True

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Disables and powers down the ADC                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x01
        data = (self.address, cmd)
        RPiSoC.commChannel.sendData(data)
        self.__runnning = False

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the ADC component and all subcomponents are currently operational                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if the ADC is active, or *False* if it is not                                                                                                            `|
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        return self.__running

    def SetBufferGain(self, gain):
        """
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetBufferGain                                                                                                                                                                                                                 |
        +===================+===============================================================================================================================================================================================================================+
        | **Description**   |Sets the input buffer gain                                                                                                                                                                                                     |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *gain*                                                                                                                                                                                                               |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *gain*            |The number by which the ADC output will be multiplied after conversion                                                                                                                                                         |
        |                   |   * Valid arguments are *1, 2, 4,* or *8*                                                                                                                                                                                     |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                                      |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Side Effects**  |Increasing the gain will lower the buffer bandwidth                                                                                                                                                                            |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        if self.type != 'DELSIG':
            raise ValueError('Setting Buffer Gain only supported by DELSIG ADC')

        cmd = 0x02

        if gain not in [1,2,4,8]:
            raise ValueError('Invalid Gain Value for DELSIG ADC: Accepts 1, 2, 4, or 8')

        data = (self.address, cmd, gain)
        RPiSoC.commChannel.sendData(data)

    def SetResolution(self, resolution):
        """

        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetResolution                                                                                                                                                                                                                 |
        +===================+===============================================================================================================================================================================================================================+
        | **Description**   |* This function sets the resolution of the SAR ADCs only. This method is unavailable for the DELSIG ADC                                                                                                                        |
        |                   |* The resolution is defaulted to 12, so this method is only needed if this needs to be changed                                                                                                                                 |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *resolution*                                                                                                                                                                                                         |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *resolution*      | An integer value, in bits, that represents the new resolution of the SAR ADC                                                                                                                                                  |
        |                   |   * Valid arguments are *8, 10, or 12* only                                                                                                                                                                                   |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                                      |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Side Effects**  |* The ADC resolution cannot be changed during a conversion cycle. The recommended bestpractice is to stop conversions with :meth:`~Stop` before changint the resolution, and then restarting with :meth:`~Start`               |
        |                   |* If you call :meth:`~SetResolution` during a conversion, the resolution will not change until the current conversion is complete, and data will not be available in the new resolution for another 6+resolution clock cycles  |
        +-------------------+-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        if self.type == 'DELSIG':
            raise ValueError('Invalid operation for the DELSIG ADC; SetResolution is only used by the SAR ADC')

        if resolution not in [8, 10, 12]:
            raise ValueError('Invalid resolution specified: valid entries are 8, 10, or 12')

        cmd = 0x03
        RPiSoC.commChannel.sendData((self.address, cmd, resolution))

    def StartConvert(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | StartConvert                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Forces the ADC to initialize a conversion. This is handled internally for the Delsig ADC, using the :meth:`~Read` function, but it is kept seperate for the SAR. However, It can also be used for the Delsig   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if self.type == "SAR0" or self.type == "SAR1":
            cmd = 0x04
        elif self.type == 'DELSIG':
            cmd = 0x03

        RPiSoC.commChannel.sendData((self.address, cmd))

    def StopConvert(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | StopConvert                                                                                                                                                                                                   |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Forces the ADC to end conversion. This is handled internally for the Delsig ADC, using the :meth:`~Read` function, but it is kept seperate for the SAR. However, It can also be used for the Delsig            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if self.type == "SAR0" or self.type == "SAR1":
            cmd = 0x05
        elif self.type == 'DELSIG':
            cmd = 0x04

        RPiSoC.commChannel.sendData((self.address,cmd))

    def GetResult(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GetResult                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Gets the result of a conversion which has been completed. :meth:`~StartConvert` must be called prior to this function, and :meth:`~StopConvert` must be called after                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |**Parameters**     |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *counts*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *counts*          |Digital value which represents the result of the most recent AD conversion                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        cmd = 0x0A
        counts = int(RPiSoC.commChannel.receiveData((self.address, cmd)))

        if counts<0:
            self.polarity = -1
        else:
            self.polarity = 1

        return counts


    def Read(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Read                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Simplifies the reading process by starting conversion, waiting for conversion to complete, stopping conversion, and returing the result, when called. It is only valid for the *DELSIG*                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |**Parameters**     |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *counts*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *counts*          |Digital value which represents the result of the most recent AD conversion                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if self.type != 'DELSIG':
            raise ValueError('Read function is only valid for the DELSIG ADC')

        cmd = 0x0D

        counts = int(RPiSoC.commChannel.receiveData((self.address, cmd)))

        if counts<0:
            self.polarity = -1
        else:
            self.polarity = 1

        return counts

    def IsEndConversion(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | IsEndConversion                                                                                                                                                                                               |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the ADC is done converting                                                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |**Parameters**     |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *counts*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *counts*          |* Digital value which represents the result of the most recent AD conversion, if it is complete                                                                                                                |
        |                   |* *0*  if conversions are not complete                                                                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if self.type == 'DELSIG':
            cmd = 0x07
        elif self.type == "SAR0" or self.type == "SAR1":
            cmd = 0x08

        return int(RPiSoC.commChannel.receiveData((self.address,cmd)))

    def SetOffset(self, offset):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetOffset                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets an offset on the ADC which is used for the voltage conversion. It will subtract the given offset before making the conversion                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *counts*                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *counts*          |Digital value which will be subtracted from any digital result before its conversion to a voltage takes place                                                                                                  |
        |                   |   * For example, if a digital value of 255 is to be converted to a voltage, and the offset has been provided by this method at any point as 10, the result will be the conversion of 245 instead of 255       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        if self.type == 'DELSIG':
            cmd = 0x0E
        elif self.type == "SAR0" or self.type == "SAR1":
            cmd = 0x0B

        RPiSoC.commChannel.sendData((self.address,cmd, offset))

    def SetGain(self, gain):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetGain                                                                                                                                                                                                       |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets the ADC *gain* in counts per volt, which will be applied before voltage conversion                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *gain*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *gain*            |Output gain in counts per volt to be applied before conversion                                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Warning**       |*gain* is set by default by the reference and input range settings. It should only be used to further calibrate the ADC with a known input or if an external reference is used                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        if self.type == 'DELSIG':
            cmd = 0x0F
        elif self.type == "SAR0" or self.type == "SAR1":
            gain*=10 #SAR takes gain in counts per 10 volts
            cmd = 0x0C

        RPiSoC.commChannel.sendData((self.address,cmd, gain))

    def CountsTo_Volts(self, counts):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | CountsTo_Volts                                                                                                                                                                                                |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Converts the ADC output to a Voltage, taking into account any set gains or offsets                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *counts*                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *counts*          |Digital value to be converted to it's analog equivalent                                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** volts                                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *volts*           |The resulting voltage after conversion of *counts*                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if self.type == 'DELSIG':
            cmd = 0x10
        elif self.type == "SAR0" or self.type == "SAR1":
            cmd = 0x0D

        counts*=self.polarity
        counts = int(counts)
        return self.polarity*(float(((RPiSoC.commChannel.receiveData((self.address,cmd, counts)))/1000000.0)))

    def Sleep(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Sleep                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the component is enabled, then it stops the ADC and saves the current configuration for later use                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        if self.type == 'DELSIG':
            cmd = 0x14
        elif self.type == "SAR0" or self.type == "SAR1":
            cmd = 0x10

        RPiSoC.commChannel.sendData((self.address,cmd))

    def Wakeup(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Wakeup                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Restores and enables the most recently saved configuration of the ADC                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        if self.type == 'DELSIG':
            cmd = 0x15
        elif self.type == "SAR0" or self.type == "SAR1":
            cmd = 0x11

        RPiSoC.commChannel.sendData((self.address,cmd))

class IDAC(object):
    """
    :Class:

        The IDAC class provides functionality for using the IDAC's available on the RPiSoC.

    :Example:

        Define IDAC objects in the following way::

            My_IDAC       = IDAC(0)
            My_other_IDAC = IDAC(1)

    """
    def __init__(self, channel):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Constructs and initializes an IDAC object                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**      *channel*                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *channel*         |Determines which IDAC is to be utilized                                                                                                            |
        |                   |   * *0* for the first IDAC; output is on P0[7] by default                                                                                         |
        |                   |   * *1* for the second IDAC; not available by default                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        self.channel = channel

        if self.channel == 0:
            if not RPiSoC.IDAC0:
                raise ValueError('IDAC0 not found on PSoC Creator program, verify that it is in your schematic and named correctly')
            self.address = RPiSoC.IDAC0_CONTROL
            self.full_range = RPiSoC.IDAC0_RANGE
       	elif self.channel == 1:
            if not RPiSoC.IDAC1:
                raise ValueError('IDAC1 not found on PSoC Creator program, verify that it is in your schematic and named correctly')
            self.address = RPiSoC.IDAC1_CONTROL
            self.full_range = RPiSoC.IDAC1_RANGE
        else:
            raise ValueError('Invalid channel: Only two IDACs available; choose 0 or 1.')

        if RPiSoC.DEBUG:
            if self.address in RPiSoC.REGISTERS_IN_USE:
                print('WARNING: Attempting to initialize object at register %d which is already in use.' %self.address)
        RPiSoC.REGISTERS_IN_USE.append(self.address)
        self.__running = False

    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets the initVar variable on the RPiSoC, calls the IDAC8_Init() function and then calls the IDAC8_Enable() function. Enables and powers up the IDAC                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x00
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = True

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Powers down IDAC to lowest power state and disables output                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x01
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = False

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the DAC component and all subcomponents are currently operational                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if the DAC is active, or *False* if it is not                                                                                                            `|
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        return self.__running

    def SetSpeed(self,speed):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetSpeed                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets the DAC speed, and consequently the power level                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**string**      *speed*                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *speed*           |*HIGH* or *LOW* for high speed or low speed, respectively                                                                                          |
        |                   |   * *HIGH* speed puts the DAC in high power mode                                                                                                  |
        |                   |   * *LOW* speed puts the DAC in low power mode                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x02
        if speed =='LOW':
            val = 0
        elif speed =='HIGH':
            val = 1
        else:
            raise ValueError('Invalid Speed: Input "HIGH" or "LOW"')

        RPiSoC.commChannel.sendData((self.address, cmd, val))

    def SetPolarity(self,polarity):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetPolarity                                                                                                                                       |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets the DAC output polarity                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**string**      *polarity*                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *polarity*        |*SOURCE* or *SINK*                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        cmd = 0x03
        if polarity =='SOURCE':
            val = 0
        elif polarity =='SINK':
            val = 1
        else:
            raise ValueError('Invalid Polarity: Input "SOURCE" or "SINK"')

        RPiSoC.commChannel.sendData((self.address, cmd, val))


    def SetRange(self, mode):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetRange                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets the full scale range for the IDAC                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**      *mode*                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *mode*            |Choices are *0, 1,* or *2*                                                                                                                         |
        |                   |   * *0*: Sets full scale range to **31.875 uA**                                                                                                   |
        |                   |   * *1*: Sets full scale range to **255 uA**                                                                                                      |
        |                   |   * *2*: Sets full scale range to **2.04 mA**                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x04
        if mode not in range(3):
            raise ValueError('Invalid Mode Specified:\n"0" for 32uA range\n"1" for 255uA range\n"2" for 2mA range ')
        if mode == 0:
            self.full_range = .031875
        elif mode == 1:
            self.full_range = .255
        elif mode == 2:
            self.full_range = 2.04

        RPiSoC.commChannel.sendData((self.address, cmd, mode))


    def SetValue(self, value):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetValue                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets a digital value to be converted into an analog current and output by the IDAC                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**      *value*                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *value*           |Digital number between *0* and *255*                                                                                                               |
        |                   |   * A value of 0 will correspond to an output of the lowest possible current output                                                               |
        |                   |   * A value of 255 will output the full scale value, as selected with :meth:`~SetRange`                                                           |
        |                   |   * All values between *0* and *255* are linearized according to the full scale range                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x05
        if value not in range(256):
            raise ValueError('Invalid IDAC Value: Input integer values between 0(min) and 255(max)')

        RPiSoC.commChannel.sendData((self.address, cmd, value))

    def SetCurrent(self, amps):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetCurrent                                                                                                                                        |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets a current in milliamps to be output on the specified IDAC                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float**      *amps*                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *amps*            |A number between 0 and the full scale range, as selected by :meth:`~SetRange`. This will be converted to an analog current and output.             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        if amps <= self.full_range and amps>=0:
            digital_val = int((float(amps)/self.full_range) * 255 + 0.5)
        else:
            digital_val = int(255*(amps>self.full_range))

        self.SetValue(digital_val)


    def Sleep(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Sleep                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the component is enabled, then it stops the DAC and saves the current configuration for later use                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x06
        RPiSoC.commChannel.sendData((self.address, cmd))

    def Wakeup(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Wakeup                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Restores and enables the most recently saved configuration of the DAC                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x07
        RPiSoC.commChannel.sendData((self.address, cmd))

class VDAC(object):
    """
    :Class:

        The VDAC class provides functionality for using the VDAC's available on the RPiSoC.

    :Example:

        Define VDAC objects in the following way::

            My_VDAC       = VDAC(0)
            My_other_VDAC = VDAC(1)
    """
    def __init__(self, channel):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Constructs and initializes a VDAC object                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**      *channel*                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *channel*         |Determines which IDAC is to be utilized                                                                                                            |
        |                   |   * *0* for the first VDAC; output is on P0[1] by default                                                                                         |
        |                   |   * *1* for the second VDAC; not available by default                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        self.channel = channel

        if self.channel == 0:
            if not RPiSoC.VDAC0:
                raise ValueError('VDAC0 not found on PSoC Creator program, verify that it is in your schematic and named correctly')
            self.address = RPiSoC.VDAC0_CONTROL
            self.full_range = RPiSoC.VDAC0_RANGE
        elif self.channel == 1:
            if not RPiSoC.VDAC1:
                raise ValueError('VDAC1 not found on PSoC Creator program, verify that it is in your schematic and named correctly')
            self.address = RPiSoC.VDAC1_CONTROL
            self.full_range = RPiSoC.VDAC1_RANGE
        else:
            raise ValueError('Invalid channel: Only two VDACs available; choose 0 or 1')

        if RPiSoC.DEBUG:
            if self.address in RPiSoC.REGISTERS_IN_USE:
                print('WARNING: Attempting to initialize object at register %d which is already in use.' %self.address)
        RPiSoC.REGISTERS_IN_USE.append(self.address)
        self.__running = False

    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   | Enables and powers up the VDAC                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x00
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = True

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Powers down VDAC to lowest power state and disables output                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x01
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = False

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the DAC component and all subcomponents are currently operational                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if the DAC is active, or *False* if it is not                                                                                                            `|
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        return self.__running

    def SetSpeed(self,speed):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetSpeed                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets the DAC speed, and consequently the power level                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**string**      *speed*                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *speed*           |*HIGH* or *LOW* for high speed or low speed, respectively                                                                                          |
        |                   |   * *HIGH* speed puts the DAC in high power mode                                                                                                  |
        |                   |   * *LOW* speed puts the DAC in low power mode                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x02
        if speed =='LOW':
            val = 0
        elif speed =='HIGH':
            val = 1
        else:
            raise ValueError('Invalid Speed: Input "HIGH" or "LOW"')

        RPiSoC.commChannel.sendData((self.address, cmd, val))

    def SetRange(self,Range):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetRange                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets the full scale range for the VDAC                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**string**      *Range*                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *Range*           |Choices are *HIGH* or *LOW*                                                                                                                        |
        |                   |   * *HIGH*: Sets full scale range to **4.080V**                                                                                                   |
        |                   |   * *LOW*: Sets full scale range to **1.020V**                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x03
        if Range =='LOW': #Range of 1.020 V
            val = 0
            self.full_range = 1.02
        elif Range =='HIGH': #Range of 4V
            val = 1
            self.full_range = 4.08
        else:
            raise ValueError('Invalid Range: Input "HIGH" for 4.080V range or "LOW" for 1.020V range')

        RPiSoC.commChannel.sendData((self.address, cmd, val))

    def SetVoltage(self, volts):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetVoltage                                                                                                                                        |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets a voltage in Volts to be output on the specified VDAC                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float**      *volts*                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *amps*            |A number between 0 and the full scale range, as selected by  :meth:`~SetRange`. This will be converted to an analog voltage and output.            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """


        if volts <= self.full_range and volts>=0:
            digital_val = int((float(volts)/self.full_range) * 255 + 0.5)
        else:
            digital_val = int(255*(volts>self.full_range))
        self.SetValue(digital_val)

    def SetValue(self, value):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetValue                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets a digital value to be converted into an analog voltage and output by the VDAC                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**      *value*                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *value*           |Digital number between *0* and *255*                                                                                                               |
        |                   |   * A value of 0 will correspond to an output of the lowest possible voltage output                                                               |
        |                   |   * A value of 255 will output the full scale value, as selected with :meth:`~SetRange`                                                           |
        |                   |   * All values between *0* and *255* are linearized according to the full scale range                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x04
        if value not in range(256):
            raise ValueError('Invalid VDAC Value: Input values between 0(min) and 255(max)')

        RPiSoC.commChannel.sendData((self.address, cmd, value))


    def Sleep(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Sleep                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the component is enabled, then it stops the VDAC and saves the current configuration for later use                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x05
        RPiSoC.commChannel.sendData((self.address, cmd))

    def Wakeup(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Wakeup                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Restores and enables the most recently saved configuration of the VDAC                                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        cmd = 0x06
        RPiSoC.commChannel.sendData((self.address, cmd))

class WaveDAC(object):
    """
    :Class:

        The WaveDAC class provides functionality for using the WaveDAC available on the RPiSoC.

    :Example:

        Define WaveDAC objects in the following way::

            My_Wave = WaveDAC()
    """

    def __init__(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Constructs and initializes an WaveDAC object                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        if not RPiSoC.WAVEDAC:
                raise ValueError('Wave DAC not found on PSoC Creator program, verify that it is in your schematic and named correctly')
        self.master_clk = 3000000 #might want to automate....
        self.frequency = 250000
        self.divider_value = 12
        self.amplitude = 255
        self.dcBias = 0
        self.waveType = 'SINE'
        self.address = RPiSoC.WAVEDAC_CONTROL

        if RPiSoC.DEBUG:
            if self.address in RPiSoC.REGISTERS_IN_USE:
                print('WARNING: Attempting to initialize object at register %d which is already in use.' %self.address)
        RPiSoC.REGISTERS_IN_USE.append(self.address)
        self.__running = False

    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Performs all of the required initialization for the component and enables power to the block                                                                                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x00
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = True

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Powers down the WaveDAC to its lowest power state and disables output                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x01
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = False

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the DAC component and all subcomponents are currently operational                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if the DAC is active, or *False* if it is not                                                                                                            `|
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        return self.__running

    def SetSpeed(self, speed):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetSpeed                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets the DAC speed to one of the defined settings                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**string**      *speed*                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *speed*           |*HIGH* or *LOW* for high speed or low speed, respectively                                                                                          |
        |                   |   * *HIGH:* Highest power and fastest slew rate                                                                                                   |
        |                   |   * *LOW:* Lowest active power and slowest slew rate                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x07

        if speed == 'HIGH':
            val = 1
        elif speed == 'LOW':
            val = 0
        else:
            raise ValueError('Invalid Speed: Choose "HIGH" or "LOW"')


        RPiSoC.commChannel.sendData((self.address, cmd, val))

    def GenerateWave(self, waveType):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GenerateWave                                                                                                                                      |
        +===================+===================================================================================================================================================+
        | **Description**   |* Provides functionality for generating a specific waveform                                                                                        |
        |                   |* It uses the class attributes *amplitude* and *dcB,* which define the waveforms peak voltage and the DC Bias, respectively                        |
        |                   |* *amplitude* and *dcB* are defaulted to 4V amplitude and 0V DC Bias, unless modifed by using :meth:`~setAmplitude` and :meth:`~setdcBias`         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**string**      *waveType*                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *waveType*        |The output behavior of the WaveDAC, Valid choices are:                                                                                             |
        |                   |   * *SINE, SQUARE, TRIANGLE,* or *SAWTOOTH*                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Note**          |*V1.3* intends to introduce inputting custom waveforms                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        cmd = 0x04
        self.waveType = waveType

        if waveType == 'SINE':
            val = 0x01
        elif waveType =='SQUARE':
            val = 0x03
        elif waveType == 'TRIANGLE':
            val = 0x05
        elif waveType == 'SAWTOOTH':
            val = 0x07
        else:
            raise ValueError('Invalid waveType: Choose "SINE" "SQUARE" "TRIANGLE" or "SAWTOOTH" ')

        RPiSoC.commChannel.sendData((self.address, cmd))
        RPiSoC.commChannel.sendData((val, self.amplitude, self.dcBias))


    def SetValue(self, val):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetValue                                                                                                                                          |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets a digital value to be converted into a voltage and output by the WaveDAC.                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**      *val*                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *val*             |Digital number between *0* and *255*                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Warning**       |The WaveDAC should be stopped before using this function                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x09
        if val not in range(256):
            raise ValueError('Invalid WaveDAC Value: Input integer between 0(min) and 255(max)')
        else:
            RPiSoC.commChannel.sendData((self.address,cmd,val))

    def Sleep(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Sleep                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the component is enabled, then it stops the DAC and saves the current configuration for later use                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x0B
        RPiSoC.commChannel.sendData((self.address, cmd))

    def Wakeup(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Wakeup                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Restores and enables the most recently saved configuration of the DAC                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x0C
        RPiSoC.commChannel.sendData((self.address, cmd))


    def SetFrequency(self, frequency):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetFrequency                                                                                                                                      |
        +===================+===================================================================================================================================================+
        | **Description**   |Sets an approximate frequency by using an appropriate clock divider so that the resulting wave is as close to the desired frequency as possible    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float**      *frequency*                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *frequency*       |A frequency in Hz that will be representative of the output wave rate                                                                              |
        |                   |   * Valid range is between 0.46 and 2500                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0xFF

        if frequency > 2500:
            print("WARNING: Wave Frequency cannot be greater than 2.5kHz \nWave frequency will be set to 2.5kHz to prevent runtime errors.")
            frequency = 2500
        elif frequency<0.46:
            print('WARNING: Wave Frequency must not be less than 0.46 Hz because no valid clock divider will sufficiently achieve such a frequency. Frequency will be set to 0.46 Hz to prevent runtime errors ')
            frequency = 0.46

        divider = int((self.master_clk/(frequency*100.0)) + 0.5)
        self.divider_value = (RPiSoC.commChannel.receiveData((self.address, cmd, divider))) + 1

    def GetFrequency(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GetFrequency                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates the actual wave frequency based on the most recently confirmed clock divider value                                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float**  *freq*                                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *freq*            |The *actual* frequency of the wave, *not* a requested frequency                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        return int(((self.master_clk/self.divider_value)/100.0))

    def StartClock(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | StartClock                                                                                                                                                                                                    |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Restarts the WaveDAC clock after it has been stopped                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0xFD
        RPiSoC.commChannel.sendData((self.address,cmd))

    def StopClock(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | StopClock                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Stops the WaveDAC clock so that a value can be set without interference by the clock                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0xFE
        RPiSoC.commChannel.sendData((self.address,cmd))

    def setAmplitude(self, amplitude):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | setAmplitude                                                                                                                                      |
        +===================+===================================================================================================================================================+
        | **Description**   |Modifies the class attribute associated with peak voltage of a generated wave                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float**      *amplitude*                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *amplitude*       |A peak amplitude, in Volts, between *0* and *4.08*                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Warning**       |This method will become deprecated in *V1.3.* There will be an alternate way of accomplishing this                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        self.amplitude = int(255/4.0*float(amplitude))

    def setdcBias(self, dcBias):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | setdcBias                                                                                                                                         |
        +===================+===================================================================================================================================================+
        | **Description**   |Modifies the class attribute associated with DC Bias of a generated wave                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float**      *dcBias*                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | *dcBias*          |A DC bias, in Volts, between *0* and *4.08*                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Warning**       |This method will become deprecated in *V1.3.* There will be an alternate way of accomplishing this                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        self.dcBias = int(255/4.0*float(dcBias))


