"""
mfswr module.  Contains the ModflowSwr1 class. Note that the user can access
the ModflowSwr1 class as `flopy.modflow.ModflowSwr1`.

Additional information for this MODFLOW process can be found at the `Online
MODFLOW Guide
<http://water.usgs.gov/ogw/modflow-nwt/MODFLOW-NWT-Guide/swr.htm>`_.

"""
import sys
from flopy.mbase import Package


class ModflowSwr1(Package):
    """
    MODFLOW Surface-Water Routing Class.

    Parameters
    ----------
    model : model object
        The model object (of type :class:`flopy.modflow.mf.Modflow`) to which
        this package will be added.

    extension : string
        Filename extension (default is 'swr')

    unitnumber : int
        File unit number (default is 36).


    Attributes
    ----------

    Methods
    -------


    See Also
    --------

    Notes
    -----
    SWR1 Class is only used to write SWR1 filename to name file. Full functionality still
    needs to be implemented.

    Examples
    --------

    >>> import flopy
    >>> m = flopy.modflow.Modflow()
    >>> swr = flopy.modflow.ModflowSwr1(m)

    """

    def __init__(self, model, extension='swr', unitnumber=36):
        """
        Package constructor.

        """
        Package.__init__(self, model, extension, 'SWR',
                         unitnumber)  # Call ancestor's init to set self.parent, extension, name and unit number
        self.heading = '# SWR1 process for MODFLOW, generated by Flopy.'
        self.url = 'swr1.htm'

        self.parent.add_package(self)

    def __repr__(self):
        return 'SRW1 class'


    def write_file(self):
        """
        Write the package input file.

        """
        print('SWR1 write method not implemented yet')
        # f_swr = open(self.fn_path, 'w')
        #f_swr.write('{0}\n'.format(self.heading))
        #f_swr.close()


    @staticmethod
    def load(f, model, ext_unit_dict=None):
        """
        Load an existing package.

        Parameters
        ----------
        f : filename or file handle
            File to load.
        model : model object
            The model object (of type: class:`flopy.modflow.mf.Modflow`)
            to which this package will be added.
        ext_unit_dict : dictionary, optional
            If the arrays in the file are specified using EXTERNAL,
            or older style array control records, then `f` should be a file
            handle.  In this case ext_unit_dict is required, which can be
            constructed using the function
            :class:`flopy.utils.mfreadnam.parsenamefile`.

        Returns
        -------
        hfb : ModflowHfb object
            ModflowHfb object (of type :class:`flopy.modflow.mfbas.ModflowHfb`)

        Notes
        -----
        Load method still needs to be implemented.

        Examples
        --------

        >>> import flopy
        >>> m = flopy.modflow.Modflow()
        >>> swr = flopy.modflow.ModflowSwr1.load('test.swr', m)

        """

        if model.verbose:
            sys.stdout.write('loading swr1 process file...\n')

        # todo: everything
        if not hasattr(f, 'read'):
            filename = f
            f = open(filename, 'r')

        print('   Warning: load method not completed. default swr1 object created.')

        # close open file
        f.close()

        # create swr1 object instance
        swr1 = ModflowSwr1(model)

        # return swr object
        return swr1
