import numpy as np
from ..pakbase import Package
from ..utils import Util2d, Util3d


class ModflowSwi(Package):
    """
    Salt Water Intrusion (SWI) package class

    Parameters
    ----------
    model : model object
        The model object (of type :class:`flopy.modflow.mf.Modflow`) to which
        this package will be added.
    npln : int
        number of active surfaces (interfaces). This equals the number of zones
        minus one. (default is 1).
    istrat : int
        flag indicating the density distribution. (default is 1).
    iswizt : int
        unit number for zeta output. (default is 53).
    nprn : int
        Number of steps between ZETA recordings; ZETA is recorded every NPRN
        steps. (default is 1)
    toeslope : float
        Maximum slope of toe cells. (default is 0.05)
    tipslope : float
        Maximum slope of tip cells. (default is 0.05)
    zetamin : float
        Minimum elevation of a plane before it is removed from a cell.
        (default is 0.005)
    delzeta : float
        Elevation for a plane when it is moved into an adjacent empty cell.
        (default is 0.05)
    nu : float or array of floats
        if istart = 1, density of each zone (nsrf + 1 values). if istrat = 0,
        density along top of layer, each surface, and bottom of layer
        (nsrf + 2 values). (default is 0.025)
    zeta : list of floats or list of array of floats
        [(nlay, nrow, ncol), (nlay, nrow, ncol)]
        initial elevations of the active surfaces. (default is 0.)
    ssz : float or array of floats (nlay, nrow, ncol)
        effective porosity. (default is 0.25)
    isource : integer or array of integers (nlay, nrow, ncol)
        Source type of any external sources or sinks, specified with any
        outside package.
        (i.e. WEL Package, RCH Package, GHB Package). (default is 0).
        If ISOURCE > 0 sources and sinks have the same fluid density as the
        zone ISOURCE. If such a zone is not present in the cell, sources and
        sinks have the same fluid density as the active zone at the top of
        the aquifer. If ISOURCE = 0 sources and sinks have the same fluid
        density as the active zone at the top of the aquifer. If ISOURCE < 0
        sources have the same fluid density as the zone with a number equal
        to the absolute value of ISOURCE. Sinks have the same fluid density
        as the active zone at the top of the aquifer. This option is useful
        for the modeling of the ocean bottom where infiltrating water is
        salt, yet exfiltrating water is of the same type as the water at the
        top of the aquifer.
    extension : str
        Filename extension (default is 'swi')
    fname_output : str

    Returns
    -------

    """

    def __init__(self, model, npln=1, istrat=1, iswizt=53, nprn=1,
                 toeslope=0.05, tipslope=0.05, zetamin=0.005, delzeta=0.05,
                 nu=0.025, zeta=[], ssz=[], isource=0, extension='swi',
                 fname_output='swi.zta'):
        """
        Package constructor.

        """
        Package.__init__(self, model)  # Call ancestor's init to set self.parent
        nrow, ncol, nlay, nper = self.parent.nrow_ncol_nlay_nper
        self.unit_number = [29, 53]
        self.extension = extension
        self.file_name = [self.parent.name + '.' + self.extension, fname_output]
        self.name = ['SWI', 'DATA(BINARY)']
        self.heading = '# Salt Water Intrusion package file for MODFLOW-2000, generated by Flopy.'
        self.npln = npln
        self.istrat = istrat
        self.iswizt = iswizt
        self.nprn = nprn
        self.toeslope = toeslope
        self.tipslope = tipslope
        self.zetamin = zetamin
        self.delzeta = delzeta
        # Create arrays so that they have the correct size
        if self.istrat == 1:
            self.nu = Util2d(model, (self.npln + 1,), np.float32, nu, name='nu')
        else:
            self.nu = Util2d(model, (self.npln + 2,), np.float32, nu, name='nu')
        self.zeta = []
        for i in range(nlay):
            # self.zeta.append( empty((nrow, ncol, self.npln)) )
            self.zeta
        for i in range(nlay):
            self.zeta.append(Util2d(model, (self.npln, nrow, ncol), np.float32, zeta[i], name='zeta_' + str(i + 1)))
        self.ssz = Util3d(model, (nlay, nrow, ncol), np.float32, ssz, name='ssz')
        self.isource = Util3d(model, (nlay, nrow, ncol), np.int, isource, name='isource')
        self.parent.add_package(self)

    def write_file(self):
        """
        Write the package file.

        Returns
        -------
        None

        """

        nrow, ncol, nlay, nper = self.parent.nrow_ncol_nlay_nper
        # Open file for writing
        f_swi = open(self.fn_path, 'w')
        # First line: heading
        # f_swi.write('%s\n' % self.heading)  # Writing heading not allowed in SWI???
        f_swi.write('%10d%10d%10d%10d\n' % (self.npln, self.istrat, self.iswizt, self.nprn))
        f_swi.write('%10f%10f%10f%10f\n' % (self.toeslope, self.tipslope, self.zetamin, self.delzeta))
        self.parent.write_array_old(f_swi, self.nu, self.unit_number[0], True, 13, 20)
        for isur in range(self.npln):
            # for ilay in range(nlay):
            # self.parent.write_array_old( f_swi, self.zeta[ilay][:,:,isur], self.unit_number[0], True, 13, ncol )
            f_swi.write(self.zeta[isur].get_file_entry())
        # for ilay in range(nlay):
        #        self.parent.write_array_old( f_swi, self.ssz[:,:,ilay], self.unit_number[0], True, 13, ncol )
        f_swi.write(self.ssz.get_file_entry())
        # for ilay in range(nlay):
        #        self.parent.write_array_old( f_swi, self.isource[:,:,ilay], self.unit_number[0], True, 13, ncol )
        f_swi.write(self.isource.get_file_entry())

        # Close file
        f_swi.close()

    @staticmethod
    def load(f, model, ext_unit_dict=None):
        """
        Load an existing package.

        Parameters
        ----------
        f : filename or file handle
            File to load.
        model : model object
            The model object (of type :class:`flopy.modflow.mf.Modflow`) to
            which this package will be added.
        ext_unit_dict : dictionary, optional
            If the arrays in the file are specified using EXTERNAL,
            or older style array control records, then `f` should be a file
            handle.  In this case ext_unit_dict is required, which can be
            constructed using the function
            :class:`flopy.utils.mfreadnam.parsenamefile`.

        Returns
        -------
        swi : ModflowSwi object

        Examples
        --------

        >>> import flopy
        >>> m = flopy.modflow.Modflow()
        >>> swi = flopy.modflow.ModflowSwi.load('test.swi', m)

        """

        if model.verbose:
            sys.stdout.write('loading swi package file...\n')

        if not hasattr(f, 'read'):
            filename = f
            f = open(filename, 'r')
        # dataset 0 -- header
        while True:
            line = f.readline()
            if line[0] != '#':
                break
        # determine problem dimensions
        nrow, ncol, nlay, nper = model.get_nrow_ncol_nlay_nper()

        print('   Warning: load method not completed. default swi object created.')

        # close the open file
        f.close()

        swi = ModflowSwi(model)
        return swi
