# -*- coding: utf-8 -*-
#
# Copyright (c) 2007 - 2009 -- Lars Heuer - Semagia <http://www.semagia.com/>.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials provided
#       with the distribution.
#
#     * Neither the name 'Semagia' nor the name 'Mappa' nor the names of the
#       contributors may be used to endorse or promote products derived from 
#       this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
"""\
This module provides a ``TopicMapWriter`` which writes 
`XML Topic Maps (XTM) 1.0 <http://www.topicmaps.org/xtm/1.0/>`_

:author:       Lars Heuer (heuer[at]semagia.com)
:organization: Semagia - http://www.semagia.com/
:version:      $Rev: 160 $ - $Date: 2009-06-15 14:13:12 +0200 (Mo, 15 Jun 2009) $
:license:      BSD license
"""
from mappa import XSD
from mappa.writer import AbstractTopicMapWriter
from mappa.writer.xmlutils import XMLWriter

from mappa import voc
_NS_XLINK = voc.XLINK
_NS_XTM_10 = voc.XTM_10
del voc

class XTM10TopicMapWriter(AbstractTopicMapWriter):
    """\
    The XTM 1.0 writer.
    """
    def __init__(self, out, base, encoding='utf-8'):
        super(XTM10TopicMapWriter, self).__init__(base)
        if not out:
            raise TypeError('"out" is not specified')
        self._writer = None    # XMLWriter instance to serialize a topic map
        self._out = out
        self._encoding = encoding
        self.export_iids = True
        self.prettify = False

    def write(self, topicmap):
        """\
        Serializes the specified ``topicmap``.
        """
        self._writer = XMLWriter(self._out, self._encoding)
        writer = self._writer
        writer.prettify = self.prettify
        writer.startDocument()
        attrs = {'xmlns': _NS_XTM_10, 'xmlns:xlink': _NS_XLINK}
        self._add_id(attrs, topicmap)
        writer.startElement('topicMap', attrs)
        write_topic = self._write_topic
        for topic in topicmap.topics:
            write_topic(topic)
        write_assoc = self._write_association
        for assoc in topicmap.associations:
            write_assoc(assoc)
        writer.endElement('topicMap')
        writer.endDocument()

    def _write_topic(self, topic):
        """\
        Serializes a topic and its characteristics.
        
        `topic`
            The topic to serialize
        """
        self._writer.startElement('topic', {'id': self._id(topic)})
        self._write_identities(topic)
        write_name = self._write_name
        for name in topic.names:
            write_name(name)
        write_occurrence = self._write_occurrence
        for occ in topic.occurrences:
            write_occurrence(occ)
        self._writer.endElement('topic')

    def _write_occurrence(self, occ):
        """\
        Serializes the occurrence.

        `occurrence`
            An occurrence.
        """
        self._writer.startElement('occurrence', self._reifier(occ))
        self._write_type(occ)
        self._write_scope(occ)
        self._write_data(occ)
        self._writer.endElement('occurrence')

    def _write_name(self, name):
        """\
        Serializes the topic name and its variants.

        `name`
            A name.
        """
        self._writer.startElement('baseName', self._reifier(name))
        self._write_scope(name)
        self._writer.dataElement('baseNameString', name.value)
        write_variant = self._write_variant
        for variant in name.variants:
            write_variant(variant)
        self._writer.endElement('baseName')

    def _write_variant(self, variant):
        """\
        Serializes a variant.
        """
        startElement, endElement = self._writer.startElement, self._writer.endElement
        startElement('variant', self._reifier(variant))
        startElement('parameters')
        write_topic_ref = self._write_topic_ref
        for theme in variant.scope:
            write_topic_ref(theme)
        endElement('parameters')
        startElement('variantName')
        self._write_data(variant)
        endElement('variantName')
        endElement('variant')

    def _write_association(self, assoc):
        """\
        Serializes an association.

        `association`
            An association.
        """
        roles = tuple(assoc.roles)
        if not roles:
            #TODO: Warning
            return
        self._writer.startElement('association', self._reifier(assoc))
        self._write_type(assoc)
        self._write_scope(assoc)
        write_role = self._write_role
        for role in roles:
            write_role(role)
        self._writer.endElement('association')

    def _write_role(self, role):
        """\
        Serializes a role.
        """
        startElement, endElement = self._writer.startElement, self._writer.endElement
        write_topic_ref = self._write_topic_ref
        startElement('member', self._reifier(role))
        startElement('roleSpec')
        write_topic_ref(role.type)
        endElement('roleSpec')
        write_topic_ref(role.player)
        endElement('member')

    def _write_data(self, datatyped):
        """\
        Serializes the ``value`` property of an occurrence or variant.

        `datatyped`
            Either an occurrence instance or a variant instance.
        """
        dt = datatyped.datatype
        if dt == XSD.anyURI:
            self._writer.emptyElement('resourceRef', self._href(datatyped.value))
        else:
            if dt != XSD.string:
                #TODO: Warning
                pass
            self._writer.dataElement('resourceData', datatyped.value)

    def _write_locators(self, name, locs):
        emptyElement = self._writer.emptyElement
        href = self._href
        for loc in locs:
            emptyElement(name, href(loc))

    def _write_type(self, typed):
        """\
        Serizalizes the ``type`` of a typed Topic Maps construct.

        `typed`
            A typed Topic Maps construct (association, role, occurrence, name).
        """
        self._writer.startElement('instanceOf')
        self._write_topic_ref(typed.type)
        self._writer.endElement('instanceOf')

    def _write_scope(self, scoped):
        """\
        Serizalizes the ``scope`` of the `scoped` Topic Maps construct.

        `scoped`
            The scoped Topic Maps construct (association, occurrence, name, variant)
        """
        write_topic_ref = self._write_topic_ref
        written = False
        for i, theme in enumerate(scoped.scope):
            if i == 0:
                self._writer.startElement('scope')
                written = True
            write_topic_ref(theme)
        if written:
            self._writer.endElement('scope')

    def _write_topic_ref(self, topic):
        """\
        Writes a ``<topicRef xlink:href="#topic-ref"/>`` element.
        """
        self._writer.emptyElement('topicRef', self._href('#%s' % self._id(topic)))

    def _write_identities(self, topic):
        """\
        
        """
        sids, slos = tuple(topic.sids), tuple(topic.slos)
        reifiable = topic.reified
        if not reifiable and not sids and not slos:
            return
        writer = self._writer
        href = self._href
        writer.startElement('subjectIdentity')
        if slos:
            if len(slos) > 1:
                #LOG.warning("The topic " + topic.id + " has more than one subject locator, exporting just one")
                pass
            # Choose one subject locator
            writer.emptyElement('resourceRef', href(slos[0]))
        for sid in sids:
            writer.emptyElement('subjectIndicatorRef', href(sid))
        if reifiable:
            writer.emptyElement("subjectIndicatorRef", href('#%s' % self._reifiable_id(reifiable)))
        writer.endElement('subjectIdentity')

    def _reifier(self, reifiable):
        attrs = {}
        self._add_id(attrs, reifiable)
        return attrs

    def _reifiable_id(self, reifiable):
        return 'reifier-%s' % self._id(reifiable.reifier)

    def _add_id(self, attrs, reifiable):
        """\
        Adds an "id" attribute to `attrs` iff ``reifiable`` is reified.
        """
        if reifiable.reifier:
            attrs.update({'id': self._reifiable_id(reifiable)})

    def _href(self, iri):
        """\
        Returns a ``{'xlink:href': iri}`` `dict`.
        """
        return {'xlink:href': iri}
